//********************************************************************************
//* File       : Keymap.hpp                                                      *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 03-Apr-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: Class definition and miscellaneous constant definitions         *
//*              for the Keymap application.                                     *
//*                                                                              *
//*                                                                              *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//*         Full text of the GPL License may be found in the Texinfo             *
//*         documentation for this program under 'Copyright Notice'.             *
//********************************************************************************

//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"   //* NcDialog API family of classes
#include <fstream>         //* C++ file I/O
#include <limits.h>        //* for realpath() and PATH_MAX
#include <dirent.h>        //* Read directory entries
#include <sys/stat.h>      //* Define stat(), lstat(), etc.

//***************
//* Definitions *
//***************

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
typedef struct stat64 FileStats ;
typedef struct dirent64 deStats ;
typedef struct tm Tm ;
//typedef struct passwd PwdInfo ;
//typedef struct group GrpInfo ;
//typedef struct utimbuf uTime ;


//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const char* const AppVersion = "0.0.06" ;
const char* const crYears = "2019-2025" ;
const char* const AppTitle = "Keymap - v:" ;
const char* const titleTemplate = "%s%s (c)%s The Software Samurai  " ;
const char* const hdrName = "NCursesKeyDef_auto.hpp" ;
const char* const tt2Name = "TransTable2.hpp" ;
const char* const logName = "Keymap.log" ;

const short MAX_PATH  = PATH_MAX ;     //* Size of buffer to hold path/filename
const short MIN_ROWS = 25 ;            //* Minimum lines needed for dialog window
const short MIN_COLS = 132 ;           //* Minimum columns needed for dialog window
const short VERBUF   = 32 ;            //* Size of various version-number buffers

//* Root directory path. (during processing, this is often a special case) *
const wchar_t* const ROOT_PATH = L"/" ;
const wchar_t fSLASH = L'/' ;
const wchar_t wNEWLINE = L'\n' ;

const short DFLT_TIMEOUT = 10 ;        //* Default user-input timeout (ten seconds)

//* Gather and process command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->appPath    = NULLCHAR ;
      this->keyTimeout  = ZERO ; // none specified
      this->diagPause = ZERO ;   // no pause
      this->autoFlag = this->briefFlag = this->mouseFlag = 
      this->verFlag = this->helpFlag = false ;
   }
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'
   char     appPath[MAX_PATH] ;  // directory of application executable
   short    keyTimeout ;         // delay before timeout waiting for user response
   short    diagPause ;          // -p option: >0 == pause after start-up diagnostics
                                 // (undocumented) >1 == display verbose start-up diagnostics
   bool     autoFlag ;           // -g option: 'true' if automatically generate default header file
   bool     ttab2Flag ;          // -l option: 'true' if automatically generate TransTable2[]
   bool     briefFlag ;          // -b option: 'true' if brief output
   bool     mouseFlag ;          // -m option: 'true' if enable mouse support
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // -H option: 'true' if display command-line help
} ;


const short kdDESC_LEN = 48 ;    // max characters in keycode description
const short kdNAME_LEN = 16 ;    // max characters in keycode name
const short kdKSEQ_LEN = 24 ;    // max characters in formatted escape sequence
class keyDesc
{
   public:
   ~keyDesc ( void ) {}          // destructor
   keyDesc ( void )              // default constructor
   {
      this->reset() ;
      this->code  = ZERO ;
      *this->name = NULLCHAR ;
      *this->desc = NULLCHAR ;
      this->kcode = ZERO ;
      *this->kseq = NULLCHAR ;
      this->match = true ;
   }
   //* Initialization constructor *
   keyDesc ( wchar_t c, const wchar_t* n, const wchar_t* s, const wchar_t* d )
   {
      this->reset() ;
      this->code = c ;
      wcsncpy ( this->name, n, kdNAME_LEN ) ;
      wcsncpy ( this->seq,  s, kdKSEQ_LEN ) ;
      wcsncpy ( this->desc, d, kdDESC_LEN ) ;
   }
   void reset ( void )
   {
      this->code  = ZERO ;
      *this->name = NULLCHAR ;
      *this->desc = NULLCHAR ;
      this->kcode = ZERO ;
      *this->kseq = NULLCHAR ;
      this->match = true ;
   }

   void operator= ( const keyDesc& src )
   {
      this->reset() ;      // not necessary, but comforting
      this->code  = src.code ;
      wcsncpy ( this->name, src.name, kdNAME_LEN ) ;
      wcsncpy ( this->desc, src.desc, kdDESC_LEN ) ;
      wcsncpy ( this->seq,  src.seq,  kdKSEQ_LEN ) ;
      this->kcode = src.kcode ;
      wcsncpy ( this->kseq,  src.kseq, kdKSEQ_LEN ) ;
      this->match = src.match ;
   }

   wchar_t  code ;               // defined keycode
   wchar_t  name[kdNAME_LEN] ;   // name of keycode
   wchar_t  desc[kdDESC_LEN] ;   // keycode description
   wchar_t  seq[kdKSEQ_LEN] ;    // default escape sequence
   wchar_t  kcode ;              // captured keycode
   wchar_t  kseq[kdKSEQ_LEN] ;   // captured escape sequence
   bool     match ;              // if 'true', defined 'code' and captured 'kcode' match
} ;


//********************************
//* Application class definition *
//********************************

class Keymap
{
   public:

   virtual ~Keymap ( void ) ;             //* destructor
   Keymap ( commArgs& clArgs ) ;          //* constructor

   void UserInterface ( void ) ;          //* Interact with the user.

   void  AllKeyGroups ( void ) ;          //* Test all keycode groups sequentially
   void  NavigationKeys ( bool hdr = true ) ; //* Navigation-keys test
   void  FunctionKeys ( bool hdr = true ) ;   //* Function-keys test
   void  KeypadKeys ( bool hdr = true, bool all = true ) ; //* Keypad-keys and 'special-keys' test
   void  SequenceTest ( short kIndex, short kCount ) ; //* Execute the test sequence
   void  stInsertDefault ( short kIndex, bool msg = true ) ;  //* Insert default keycode into table
   bool  AppendEscSequence ( gString& gsOut, short kIndex ) ; //* Append raw escape sequence
   void  SingleKey ( void ) ;             //* Test a single user-specified key combination
   void  SummaryReport ( short recCount, short recTested, //* Testing summary report
                         short recGood, short recBad, bool pause ) ;
   bool  GenerateHeaderFile ( bool genDflt = false ) ; //* Generate a C++ header file to be #include'd
   bool  GenerateTransTable ( bool dflt = false ) ;    //* Generate the TransTable2[] translation table

   void  PrintInstructions ( void ) ;     //* Display operation menu
   void  InstructionPrompt ( bool nl = false ) ; //* Display a brief menu prompt
   void  ModeMsg ( void ) ;               //* Display 'standard mode' instructions
   void  ViewLogFile ( void ) ;           //* View (and optionally save) the log file
   void  SaveLogFile ( void ) ;           //* Save the log file
   void  CreateQueryTable ( void ) ;      //* Create lookup table
   void  ApplicationHelp ( void ) ;       //* Application Help

   //* Output methods *
   void WriteParagraph ( const gString& msg, attr_t attr, bool toLog = false ) ;
   void WriteParagraph ( const wchar_t *msg, attr_t attr, bool toLog = false ) ;
   void WriteParagraph ( const char    *msg, attr_t attr, bool toLog = false ) ;
   void SetCursor ( const winPos new_wp ) ;

   //* Utility methods *
   bool RenameAsBackup ( const gString& fName ) ;
   bool RenameFile ( const gString& srcPath, const gString& trgPath ) ;
   bool TargetExists ( const gString& trgPath ) ;
   bool DeleteFile ( const gString& trgPath ) ;
   bool CreateTemppath ( gString& basePath ) ;
   bool CreateTempname ( gString& tmpPath ) ;
   void DeleteTemppath ( void ) ;
   bool GetTempdirPath ( gString& tdPath ) ;
   void FindAppPath ( commArgs& ca ) ;
   void GetVersions ( void ) ;
   short GetLine ( std::ifstream& ifs, gString& gsIn ) ;
   bool GetCommandLineArgs ( commArgs& ca ) ;
   void ClearWindow ( void ) ;
   void DiagMsg ( const char* msg, attr_t color, bool newLine = true ) ;
   void DisplayVersion ( void ) ;
   void DisplayHelp ( void ) ;

   //****************
   //* Data Members *
   //****************
   private:
   NcDialog *dPtr ;           //* Pointer to Keymap application dialog
   char appPath[MAX_PATH] ;   //* Application path (current working directory)
   char tfPath[MAX_PATH] ;    //* Path to system's temp file directory
   gString logPath ;          //* Filespec of temp log file
   keyDesc* keyLookup ;       //* Pointer to lookup table
   short   klCount ;          //* Number of entries in keyLookup
   winPos  yxPos ;            //* Text insertion point
   winPos  suPos ;            //* Cursor position for stat-up messages
   ofstream ofsLog ;          //* Operation logfile stream
   wchar_t minKey ;           //* Minimum ncurses keycode value
   wchar_t maxKey ;           //* Maximum ncurses keycode value
   attr_t  menuAttr ;         //* Color attribute for menu, prompt, help
   attr_t  okAttr ;           //* Color attribute for matching responses
   attr_t  errAttr ;          //* Color attribute for mismatched responses
   attr_t  normAttr ;         //* Generic color attribute
   short   nvIndex ;          //* Offset into keyLookup[] for Navigation_key-test records
   short   nvCount ;          //* Number of Navigation_key-test records
   short   fnIndex ;          //* Offset into keyLookup[] for Function_key-test records
   short   fnCount ;          //* Number of Function_key-test records
   short   kpIndex ;          //* Offset into keyLookup[] for Keypad_key-test records
   short   kpCount ;          //* Number of Keypad_key-test records
   short   kpnIndex ;         //* Offset into keyLookup[] for Keypad_key-test records if no numeric keypad
   short   kpnCount ;         //* Number of Keypad_key-test records if no numeric keypad
   short   goodTotal ;        //* Cumulative total of matching records
   short   badTotal ;         //* Cumulative total of mismatched records
   short   goodRecord ;       //* Count of tests that match the record in keyLookup[]
   short   badRecord ;        //* Count of tests that DO NOT match keyLookup[] record
   short   keyTimeout ;       //* Timeout (in seconds) to wait for user response
   bool    autoGen ;          //* If 'true', automatically generate a default header file
   bool    autoTab ;          //* If 'true', automatically generate TransTable2[] for Dialog2
   bool    briefOut ;         //* If 'true', generate brief output (no esc sequences)
   bool    hdrGenerated ;     //* 'true' if C++ header file generated
   bool    logSave ;          //* 'true' if log file to be saved on exit

   //* Size of terminal window (set when NCurses Engine starts)           *
   //* Modified when terminal window is resized.                          *
   short termRows,
         termCols,
         actualRows,
         actualCols ;

   const char** tt1 ;         //* Translation Table #1 ( ASCII control codes )
   short tt1Min ;
   short tt1Max ;
   const char** tt2 ;         //* Translation Table #2 ( troublemakers )
   short tt2Min ;
   short tt2Max ;
   const char** tt3 ;         //* Translation Table #3 ( Alt keys )
   short tt3Count ;

} ;