//********************************************************************************
//* File       : Keymap.cpp                                                      *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in Keymap.hpp              *
//* Date       : 22-Mar-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description:                                                                 *
//* The Keymap utility interactively maps the keycode associated with each       *
//* key on the keyboard. The application is based on the NcDialog API and is     *
//* designed to analyze the keycode definitions and key escape sequences visible *
//* within the terminal window environment.                                      *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.06 01-Jun-2024                                                        *
//*   -- Changes to the Wayland compositor have contributed to many keycodes     *
//*      associated with the raw escape sequences have changed.                  *
//*      Factors beyond the Wayland re-definitions may also have had an effect.  *
//*      -- A new table of default keycode definitions was generated for         *
//*         integration into the NCursesKeyDef.hpp source file.                  *
//*      -- Expand the TransTable2 array generator from 0x024B to 0x0251.        *
//*         This makes room for the recently-redefined keycodes for:             *
//*         nckpMINUS (0x024C).                                                  *
//*      -- A new TransTable2 was generated for the Dialog2 utility              *
//*         (test #5, report keycodes). This table is integrated into the        *
//*         NcKey.cpp source module.                                             *
//*   -- Add a new method to query the system for version numbers for:           *
//*      kernel-release, gnome-terminal, KDE Konsole, XTerm                      *
//*   -- Update documentation to reflect the changes.                            *
//*                                                                              *
//* v: 0.0.05 04-Dec-2023                                                        *
//*   -- When interactively generating the keymap for "all" keycodes, ask the    *
//*      user whether the keyboard includes a numeric keypad. If it does, then   *
//*      interactively generate the keypad map. Otherwise, use the default       *
//*      keycode mapping for the numeric keypad group.                           *
//*   -- When interactively generating the keymap(s), if the indicated key is    *
//*      captured by the terminal/system such that no keycode is seen by the     *
//*      application, then user may skip that test Ctrl+S. When this happens the *
//*      default value for that key combination will be inserted into the table. *
//*      See stInsertDefault() method.                                           *
//*      -- Formerly, the skipped test was considered as a bad test and assigned *
//*         an invalid keycode i.e. 0x0000.                                      *
//*   -- Update documentation to reflect the changes.                            *
//*                                                                              *
//* v: 0.0.04 27-Jun-2023                                                        *
//*   -- The recent update of ncurses to v:6.4 broke the keycodes for cursor key *
//*      combinations and keypad (AGAIN)!                                        *
//*      The ncurses version is 6.4-3.20230114.fc37 but it didn't get updated    *
//*      in the dnf/apt-get mirror sites until 2023-06-06.                       *
//*      -- A new table of default keycode definitions was generated for         *
//*         integration into the NCursesKeyDef.hpp source file.                  *
//*      -- Expand the TransTable2 array generator from 0x0246 to 0x024B.        *
//*         This makes room for the recently-redefined keycodes for:             *
//*         nckpDIV (0x0247), nckpMULT (0x0249) and nckpMINUS (0x024A).          *
//*      -- A new TransTable2 was generated for the Dialog2 utility              *
//*         (test #5, report keycodes). This table is integrated into the        *
//*         NcKey.cpp source module.                                             *
//*   -- Bug Fix: When saving the interactively-generated keymap, the default    *
//*      keycodes for the numeric keypad were being saved instead of the         *
//*      generated keycodes. Bad programmer! Go to your room!                    *
//*                                                                              *
//* v: 0.0.03 07-May-2020                                                        *
//*   -- The transition from Fedora30 to Fedora32 has broken a large number of   *
//*      keycodes. We blame this on Wayland because experience tells us that     *
//*      the GNOME/Wayland desktop is responsible for almost everything that     *
//*      goes wrong. The broken key combinations generate the same escape        *
//*      sequences as previously; however, the keycodes assigned to them have    *
//*      been arbitrarily changed.                                               *
//*      -- Note that the affected key combinations are not defined within       *
//*         the terminfo database, but seem to have been pulled out of some      *
//*         developer's uhh... dream. Are the newly assigned codes more or       *
//*         less logical than the previous layout? That is a judgement call,     *
//*         and to be honest, we tend to agree with the maintainer's decision.   *
//*         But it's annoying when stable code must be modified without actual   *
//*         design benefit.                                                      *
//*      -- Note also that the right ALT key is configured as the "AltGr" key    *
//*         by default, meaning that it cannot be relied upon to perform the     *
//*         same function as the left ALT key.                                   *
//*   -- Bug Fix: The "Total" reported by SummaryReport() was incorrect for      *
//*      individual keycode groups. In the previous version, the reported        *
//*      total was correct only in AllKeyGroups().                               *
//*   -- Bug Fix: If a test was skipped via CTRL+S command, the skipped test     *
//*      was not being included in the testing total. A skipped test is now      *
//*      included in the 'badRecord' accumulator.                                *
//*   -- Bug Fix: In SingleKey() test, there was an off-by-one error which       *
//*      prevented the user from selecting the last item in the list.            *
//*   -- Documentation update.                                                   *
//*                                                                              *
//* v: 0.0.02 19-Oct-2019                                                        *
//*   -- Konsole (v:19.04.2) reports a slightly different keymap by default.     *
//*      There are three keycode profiles: "XFree 4" (default), Linux console,   *
//*      and Solaris console.                                                    *
//*      Although there is a general agreement between gnome-terminal and        *
//*      Konsole(XFree4), Konsole directly filters certain key combination and   *
//*      substitutes the base keycode. For instance UpArrow, Alt+UpArrow,        *
//*      Ctrl+UpArrow and Alt+Ctrl+UpArrow all report the same keycode,          *
//*      0x0103(UpArrow).                                                        *
//*      In our not-at-all-humble opinion, this is bat-shit crazy. Why deny      *
//*      perfectly good keycodes to console applications? It stinks of           *
//*      Redmond, WA. and the Microsloth "we know what's good for you"           *
//*      philosophy of design. (Yes, we understand that you want to support      *
//*      console emacs and vim, but come on, really!?!)                          *
//*      -- /usr/share/konsole/default.keytab                                    *
//*      -- /usr/share/konsole/linux.keytab                                      *
//*      -- /usr/share/konsole/solaris.keytab                                    *
//*      Editing these profiles or creating a custom profile is beyond the       *
//*      scope of this application, but it is not too difficult. See:            *
//*      https://docs.kde.org/stable5/en/applications/konsole/key-bindings.html  *
//*      -- Customized .keytab files are located in:                             *
//*         ~/.local/share/konsole                                               *
//*                                                                              *
//* v: 0.0.01 05-Oct-2019                                                        *
//*   -- First effort. Design visual style and user interface.                   *
//*   -- Create static lookup table based on key output from ncurses v:6.1       *
//*      under Gnome terminal v:3.32.2                                           *
//*   -- Auto-generate NCursesKeyDef_auto.hpp containing the keycode             *
//*      definitions in the range of (0x0102 - 0x0246) for the target group      *
//*      of 146 key combinations                                                 *
//*      -- The default definitions are embedded in the NCursesKeyDef.hpp        *
//*         module of the NcDialogAPI. This required a general restructuring     *
//*         of NCursesKeyDef.hpp so that the generated file with customized      *
//*         definitions can optionally be "#included'd" in it.                   *
//*   -- Auto-generate the lookup table used by the NcDialogAPI Dialog2          *
//*      test-and-demonstration app for interactive keycode testing.             *
//*      -- This table must be manually copied into the indicated position in    *
//*         the NcKey.cpp module of the NcDialogAPI.                             *
//*      -- Note that generating this table requires a hack that makes some      *
//*         assumptions about which keypad key combinations duplicate the        *
//*         keycodes of the corresponding keys on the main keyboard.             *
//*         Keep an eye on this!                                                 *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"
#include "Keymap.hpp"            //* Application-class definition

//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************

//**********
//*  Data  *
//**********

const char* const Prompt1 = 
   "\nEnter the specified key combinations: "
   "(Ctrl+R to repeat a test, Ctrl+S to skip a test, Ctrl+C to abort)\n" ;
const char* const Prompt2 = 
   "Note that some key combinations may not be available "
   "in the terminal window.\n" ;


//*************************
//*         main          *
//*************************
//********************************************************************************
//* Program entry point.                                                         *
//*                                                                              *
//* Command-line Usage:  See "GetCommandLineArgs() method.                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* User may have specified one or more command-line arguments.           *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object and interact with the user.       *
   Keymap* kmptr = new Keymap( clArgs ) ;

   //* Before returning to the system, delete the Keymap object.     *
   //* While the class object and all its resources will be released *
   //* by the system on exit, we want to force execution of the      *
   //* destructor for an orderly termination.                        *
   delete kmptr ;

   exit ( ZERO ) ;

}  //* End main() *

//*************************
//*        ~Keymap        *
//*************************
//********************************************************************************
//* Destructor. Return all resources to the system.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

Keymap::~Keymap ( void )
{
   //* Release the lookup table (not currently used) *
   //if ( this->keyLookup != NULL )
   //   delete [] this->keyLookup ;

   //* Delete an temporary files created by the application.*
   // NOTE: The exit-to-new-CWD temp file (if any) IS NOT deleted here.
   this->DeleteTemppath () ;

}  //* End ~Keymap() *

//*************************
//*        Keymap         *
//*************************
//********************************************************************************
//* Constructor.                                                                 *
//*                                                                              *
//*                                                                              *
//* Input  : commArgs class object (by reference)                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

Keymap::Keymap ( commArgs& clArgs )
{
   //* Initialize our data members *
   this->dPtr         = NULL ;
   *this->appPath     = NULLCHAR ;
   *this->tfPath      = NULLCHAR ;
   this->keyLookup    = NULL ;
   this->klCount      = ZERO ;
   this->termRows     = ZERO ;
   this->termCols     = ZERO ;
   this->actualRows   = MIN_ROWS ;
   this->actualCols   = MIN_COLS ;
   this->yxPos = { 1, 0 } ;
   this->suPos = { 1, 0 } ;
   this->minKey = KEY_CODE_YES ;    // ncurses min-key definition (0400 octal)
   this->maxKey = KEY_MAX ;         // ncurses max-key definition (0777 octal)
   this->nvIndex = this->nvCount =  // this group initialized below
   this->fnIndex = this->fnCount = 
   this->kpIndex = this->kpCount = 
   this->kpnIndex = this->kpnCount = ZERO ;
   this->goodTotal = this->badTotal = this->goodRecord = badRecord = ZERO ;
   this->keyTimeout = DFLT_TIMEOUT ;
   this->autoGen  = false ;         // no auto-generate of header file
   this->autoTab  = false ;         // no auto-generate of translation table
   this->briefOut = false ;         // standard output
   this->hdrGenerated = false ;     // header file not yet created
   this->logSave  = false ;         // logfile save not specified

   //* See what the user has to say *
   this->GetCommandLineArgs ( clArgs ) ;

   gString gsOut ;
   gsOut = clArgs.appPath ;                  // application path
   gsOut.copy( this->appPath, MAX_PATH ) ;

   if ( clArgs.keyTimeout > ZERO )           // user-input timeout
      this->keyTimeout = clArgs.keyTimeout ;
   this->autoGen  = clArgs.autoFlag ;        // 'true' if auto-gen of header file
   this->autoTab  = clArgs.ttab2Flag ;       // 'true' if auto-gen of TransTable2[]
   this->briefOut = clArgs.briefFlag ;       // standard/brief output formatting

   //* Establish temp-file directory *
   bool badTmpDir = false ;
   if ( (this->CreateTemppath ( gsOut )) != false )
   {
      gsOut.copy( this->tfPath, MAX_PATH ) ;
      this->CreateTempname ( this->logPath ) ;
   }
   else  // We cannot continue without a valid temp-file directory.
      badTmpDir = true ;

   //***  Initialize the NCurses Engine  ***
   if( (!(clArgs.helpFlag || clArgs.verFlag || badTmpDir)) && 
       ((nc.StartNCursesEngine ()) == OK) )
   {
      //* Get the size of our playground *
      nc.ScreenDimensions ( this->termRows, this->termCols ) ;

      //* Verify that the current locale supports UTF-8 character encoding.    *
      gString envLocale( nc.GetLocale() ) ;  // name of locale from environment
      short localeSetOk = nc.VerifyLocale ();// env locale supports UTF-8?
      //nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetCursorState ( nccvNORMAL ) ;     // normal block cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed
      gsOut.compose( titleTemplate, AppTitle, AppVersion, crYears ) ;
      nc.WriteString ( ZERO, ZERO, gsOut.gstr(), nc.bwU ) ; // draw app title line to console

      this->DiagMsg ( "NCurses engine initialized.", nc.bw ) ;

      //* Determine whether the hardware supports display of color text. *
      if ( nc.ColorText_Available () )
      {
         this->DiagMsg ( "Multi-color screen output supported.", nc.gr ) ;

         //* Color Engine successfully initialized? *
         if ( nc.ColorText_Initialized () )
         {
            termColorInfo tci ;
            nc.TerminalColorSupport ( tci ) ;
            gsOut.compose( L"Color Engine started. (%hd RGB Registers and "
                            "%hd fgnd/bkgnd Color Pairs)", 
                           &tci.rgb_regs, &tci.fgbg_pairs ) ;
            this->DiagMsg ( gsOut.ustr(), nc.gr ) ;
         }
         else
         {
            this->DiagMsg ( "Unable to start Color Engine.", nc.bw ) ;
         }
      }
      else
      {
         this->DiagMsg ( "Terminal does not support color output.", nc.bw ) ;
         this->DiagMsg ( " Starting in monochrome mode.", nc.bw ) ;
      }

      //* Report the results of locale settings for character-encoding *
      this->DiagMsg ( "Locale from environment: ", nc.gr, false ) ;
      gsOut.compose( L"\"%S\"", envLocale.gstr() ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bw, false ) ;
      if ( localeSetOk == OK )
         this->DiagMsg ( "  UTF-8 encoding support verified.", nc.gr ) ;
      else
         this->DiagMsg ( "  may not support UTF-8 encoding.", nc.reB ) ;

      //* Report terminal-window dimensions *
      gsOut.compose( "Terminal window dimensions: (%hd x %hd)",
                     &this->termRows, &this->termCols ) ;
      this->DiagMsg ( gsOut.ustr(), nc.gr ) ;

      //* Report path of log file *
      this->DiagMsg ( "Logfile Path: ", nc.gr, false ) ;
      this->DiagMsg ( this->logPath.ustr(), nc.bw ) ;

      #if 0    // DEBUG ONLY
      //* Report remaining command-line arguments *
      this->DiagMsg ( "Prompt-to-Input Timeout: ", nc.gr, false ) ;
      gsOut.compose( "%hd", &this->keyTimeout ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bw ) ;
      gsOut.compose( "autoGen:%hhd", &this->autoGen ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bw ) ;
      gsOut.compose( "autoTab:%hhd", &this->autoTab ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bw ) ;
      gsOut.compose( "briefOut:%hhd", &this->briefOut ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bw ) ;
      #endif   // DEBUG ONLY

      //* If specified on the command-line  *
      //* wait for user to read diagnostics.*
      if ( clArgs.diagPause > ZERO )
      {
         if ( clArgs.diagPause == 1 )
         {
            chrono::duration<short>aWhile( 4 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
         else
         {
            this->DiagMsg ( " * Press Any Key to Continue *", nc.bl ) ;
            nckPause();
         }
      }

      //* Set application color attributes *
      this->menuAttr = nc.bl ;
      this->okAttr   = nc.gr ;
      this->errAttr  = nc.reB ;
      this->normAttr = nc.bw ;

      nc.OutputStreamScrolling ( true ) ;    // enable auto-scrolling window

      //* Rearrange the table into the order keycodes will be queried.*
      //* This call initializes data members: keyLookup, klCount,     *
      //* tt1, tt1Min, tt1Max,* tt2, tt2Min, tt2Max, tt3, tt3Count,   *
      //* nvIndex, nvCount, fnIndex, fnCount, kpIndex, kpCount,       *
      //* kpnIndex, kpnCount.                                         *
      this->CreateQueryTable () ;


      //***************************
      //* Interact with the user. *
      //* (Watch out! They bite!) *
      //***************************
      this->UserInterface () ;


      nc.OutputStreamScrolling ( false ) ;   // disable auto-scrolling window
      nc.RestoreCursorState () ;             // make cursor visible
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

#if 0    // DISABLE
      //* Remind user if header file created *
      if ( this->hdrGenerated )
         wcout << L"Header file: \"" << hdrName << "\" created in current directory." << endl ;
#endif   // DISABLE
   }
   else
   {
      if ( clArgs.verFlag )
         this->DisplayVersion () ;
      else if ( clArgs.helpFlag )
         this->DisplayHelp () ;
      else if ( badTmpDir )
         wcout << "SYSTEM ERROR: Unable to establish path for temporary files." << endl ;
      else
         wcout << "\n ERROR! Unable to initialize NCurses engine.\n" << endl ;
   }

}  //* End Keymap() *

//*************************
//*     UserInterface     *
//*************************
//********************************************************************************
//* Interact with the user.                                                      *
//*                                                                              *
//* Input  : none                                                                *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::UserInterface ( void )
{
   gString gsOut ;               // output formatting
   wkeyCode wk ;                 // user input
   bool done = false ;           // loop control

   //* Construct and display the main application title line *
   this->ClearWindow () ;
   gsOut.compose( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   this->WriteParagraph ( gsOut, (this->menuAttr | ncuATTR), true ) ;

   //* Complain if terminal window is too small
   if ( (this->termRows < MIN_ROWS) || (this->termCols < MIN_COLS) )
   {
      gsOut.compose( "Terminal window is too small: (%2hd x %3hd)               \n"
                     "Please resize window to at least %hd rows by %hd columns\n"
                     "Press any key to exit.                                 \n\n",
                     &this->termRows, &this->termCols, &MIN_ROWS, &MIN_COLS ) ;
      this->WriteParagraph ( gsOut, this->errAttr ) ;
      nckPause() ;
      return ;       // NOTE THE EARLY RETURN
   }

   //* If user specified to automatically generate a default header file *
   if ( this->autoGen )
   {
      this->GenerateHeaderFile ( true ) ;
      if ( ! this->autoTab )
      {
         this->WriteParagraph ( L" Press any key to exit...", this->menuAttr ) ;
         nckPause();
      }
      done = true ;
   }
   //* If user specified to automatically generate the TransTable2[]   *
   //* translation table which is used in the Dialog2 utility, Test #5.*
   if ( this->autoTab )
   {
      this->GenerateTransTable ( true ) ;
      this->WriteParagraph ( L" Press any key to exit...", this->menuAttr ) ;
      nckPause();
      done = true ;
   }
   else
   {
      //* Open the operational log file *
      this->ofsLog.open( this->logPath.ustr(), (ofstream::out | ofstream::trunc) ) ;
   }

   this->PrintInstructions () ;

   while ( ! done )
   {
      if ( (nc.GetKeyInput ( wk )) == wktPRINT )
      {
         switch ( wk.key )
         {
            case L'a':              // Test all key groups
            case L'A':
               this->AllKeyGroups () ;
               break ;
            case L'n':              // Navigation key group
            case L'N':
               this->NavigationKeys () ;
               break ;
            case L'f':              // Function key group
            case L'F':
               this->FunctionKeys () ;
               break ;
            case L'k':              // Keypad key group
            case L'K':
               this->KeypadKeys () ;
               break ;
            case L's':              // Test a single key combination
            case L'S':
               this->SingleKey () ;
               break ;
            case L'i':              // Menu
            case L'I':
               this->PrintInstructions () ;
               break ;
            case L'g':              // Generate the C++ header file
            case L'G':
               this->GenerateHeaderFile () ;
               if ( this->autoTab )
                  this->GenerateTransTable () ;
               this->WriteParagraph ( L" ", this->menuAttr ) ;
               break ;
            case L'v':              // View/Save log file
            case L'V':
               this->ViewLogFile () ;
               break ;
            case L'h':              // Application Help
            case L'H':
               this->ApplicationHelp () ;
               break ;
            case L'c':              // Clear the display
            case L'C':
               this->ClearWindow () ;
               break ;
            case L'x':              // quit
            case L'X':
            case L'q':
               if ( this->logSave )
                  this->SaveLogFile () ;
               done = true ;
               break ;
            default:                // call user a dumb-guy
               nc.UserAlert () ;
               this->InstructionPrompt () ;
               break ;
         }
      }
      else
      {
         nc.UserAlert () ;
         this->ClearWindow () ;
         this->PrintInstructions () ;
      }
   }

   //* If log file is open, close it now *
   if ( this->ofsLog.is_open() )
      this->ofsLog.close() ;

}  //* End UserInterface() *

//*************************
//*     AllKeyGroups      *
//*************************
//********************************************************************************
//* Test all keycode groups in sequence and generate a summary report.           *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::AllKeyGroups ( void )
{
   this->ClearWindow () ;
   this->WriteParagraph ( "For each section enter the specified key combinations: "
                          "(Ctrl+R to repeat a test, Ctrl+C to abort)\n", 
                          (this->menuAttr | ncuATTR), true ) ;
   this->WriteParagraph ( "Note that some keys may not be accessible "
                          "in the terminal window.\n", this->menuAttr ) ;
   this->ModeMsg () ;
   this->WriteParagraph ( L"\n", this->menuAttr ) ;

   //* Reset the accumulators *
   this->goodTotal = this->badTotal = ZERO ;

   //* Navigation Keys test sequence *
   this->NavigationKeys ( false ) ;
   this->goodTotal += this->goodRecord ;
   this->badTotal += this->badRecord ;

   if ( (this->goodRecord + this->badRecord) == this->nvCount )
   {
      //* Function Keys test sequence *
      this->FunctionKeys ( false ) ;
      this->goodTotal += this->goodRecord ;
      this->badTotal += this->badRecord ;

      if ( (this->goodRecord + this->badRecord) == this->fnCount )
      {
         //* Ask user whether keyboard has a numeric keypad. *
         wkeyCode wk ;
         bool     allKeypadKeys = false ;
         this->WriteParagraph ( "\nDoes your keyboard have a numeric keypad? Y/n: ",
                                (this->menuAttr | ncuATTR), true ) ;
         nc.GetKeyInput ( wk ) ;
         if ( ((wk.type == wktFUNKEY) && (wk.key == nckENTER)) ||
              ((wk.type == wktPRINT) && (wk.key == L'y' || wk.key == L'Y')) )
         { allKeypadKeys = true ; }
         this->WriteParagraph ( L"\n", this->menuAttr ) ;

         //* Keypad test sequence *
         this->KeypadKeys ( false, allKeypadKeys ) ;
         this->goodTotal += this->goodRecord ;
         this->badTotal += this->badRecord ;
         if ( ! allKeypadKeys )
            this->WriteParagraph ( " * Default keycodes generated for remainder of keypad group.\n",
                                   this->menuAttr, true ) ;
      }
   }

   //* Summary report *
   short tested = this->goodTotal + this->badTotal ;
   this->SummaryReport ( this->klCount, tested, this->goodTotal, this->badTotal, false ) ;

   //* If all keycodes tested, generate the header file. *
   //* Note that the TransTable2 file is automatically   *
   //* generated regardless of whether user requested it.*
   //* This is done so if the user forgot to specify the *
   //* TransTable2 file, he/she will not have to test all*
   //* the keycodes again.                               *
   //tested = this->klCount ;   // FOR DEBUGGING ONLY
   if ( tested == this->klCount )
   {
      this->GenerateHeaderFile () ;
      this->GenerateTransTable () ;
   }
   else
      this->WriteParagraph ( L" Testing incomplete. Header file not created.\n",
                             this->errAttr, true ) ;
   this->WriteParagraph ( L" Press any key to continue...", this->menuAttr ) ;
   nckPause();

   //* Return to main menu *
   this->ClearWindow () ;
   this->PrintInstructions () ;

}  //* End AllKeyGroups() *

//*************************
//*    NavigationKeys     *
//*************************
//********************************************************************************
//* Interactively test the navigation keys (Arrows, Home, End, etc.).            *
//*                                                                              *
//* Input  : hdr:: (optional, 'true' by default)                                 *
//*                'true' : display header                                       *
//*                'false': do not display header                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::NavigationKeys ( bool hdr )
{
   if ( hdr != false )
   {
      this->WriteParagraph ( Prompt1, (this->menuAttr | ncuATTR), true ) ;
      this->WriteParagraph ( Prompt2, this->menuAttr ) ;
      this->ModeMsg () ;
      this->WriteParagraph ( L"\n", this->menuAttr ) ;
   }

   //* Run the test sequence *
   this->SequenceTest ( this->nvIndex, this->nvCount ) ;

   if ( hdr )
   {
      //* Summary report *
      short tested = this->goodRecord + this->badRecord ;
      this->SummaryReport ( this->nvCount, tested, this->goodRecord, this->badRecord, true ) ;

      //* Return to main menu *
      this->ClearWindow () ;
      this->PrintInstructions () ;
   }

}  //* End NavigationKeys() *

//*************************
//*     FunctionKeys      *
//*************************
//********************************************************************************
//* Interactively test the function keys (F01, Shift+F01, etc.).                 *
//*                                                                              *
//* Input  : hdr:: (optional, 'true' by default)                                 *
//*                'true' : display header                                       *
//*                'false': do not display header                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* F11 : toggles full-screen view for terminal window. Disable in Preferences.  *
//*                                                                              *
//********************************************************************************

void Keymap::FunctionKeys ( bool hdr )
{
   if ( hdr != false )
   {
      this->WriteParagraph ( Prompt1, (this->menuAttr | ncuATTR), true ) ;
      this->WriteParagraph ( Prompt2, this->menuAttr ) ;
      this->ModeMsg () ;
      this->WriteParagraph ( L"\n", this->menuAttr ) ;
   }

   //* Run the test sequence *
   this->SequenceTest ( this->fnIndex, this->fnCount ) ;

   if ( hdr )
   {
      //* Summary report *
      short tested = this->goodRecord + this->badRecord ;
      this->SummaryReport ( this->fnCount, tested, this->goodRecord, this->badRecord, true ) ;

      //* Return to main menu *
      this->ClearWindow () ;
      this->PrintInstructions () ;
   }

}  //* End FunctionKeys() *

//*************************
//*      KeypadKeys       *
//*************************
//********************************************************************************
//* Interactively test the numeric keypad keys.                                  *
//*                                                                              *
//* Input  : hdr: (optional, 'true' by default)                                  *
//*               'true' : display header                                        *
//*               'false': do not display header                                 *
//*          all: (optional, 'true' by default)                                  *
//*               'true'  : test all member of the keypad group                  *
//*               'false' : Keyboard has no numeric keypad. Test only the        *
//*                         non-keypad members of the group, and use default     *
//*                         values for keypad keycodes (see note below)          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* The system GUI captures a large number of keycodes for its own purposes.     *
//* Most of these can be disabled through the Settings->Keyboard->Shortcuts.     *
//* Note: At least two keycodes captured by the GUI system cannot be disabled    *
//*       through the "Settings" GUI.                                            *
//*       nckACUP   == Wayland "workspace-up"   - disable using dconf-editor     *
//*       nckACDOWN == Wayland "workspace-down" - disable using dconf-editor     *
//* ----- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ----- *
//* Certain keycodes of the keypad group are not keypad keys.                    *
//* Therefore, When testing _only_ the non-keypad members of the keypad group,   *
//* the following will be tested in the usual way:                               *
//*   a) nckDELETE, nckSDEL, nckADEL, nckASDEL, nckCDEL, nckCSDEL                *
//*   b) nckINSert, nckAINSERT                                                   *
//*   c) nckBKSP, nckSTAB                                                        *
//********************************************************************************

void Keymap::KeypadKeys ( bool hdr, bool all )
{
   if ( hdr != false )
   {
      this->WriteParagraph ( Prompt1, (this->menuAttr | ncuATTR), true ) ;
      this->WriteParagraph ( Prompt2, this->menuAttr ) ;
      this->ModeMsg () ;
      this->WriteParagraph ( L"\n", this->menuAttr ) ;
   }

   //* Run the test sequence *
   if ( all != false )
      this->SequenceTest ( this->kpIndex, this->kpCount ) ;
   else
   {
      //* Fill in the untested records using default values *
      this->SequenceTest ( this->kpnIndex, this->kpnCount ) ;
      short kIndex = this->kpIndex ;
      for ( short i = ZERO ; i < this->kpCount ; ++i, ++kIndex )
      {
         if ( (kIndex < this->kpnIndex) || (i == (this->kpCount - 1)) )
            this->stInsertDefault ( kIndex, false ) ;
      }
   }

   if ( hdr )
   {
      //* Summary report *
      short tested = this->goodRecord + this->badRecord ;
      this->SummaryReport ( this->kpCount, tested, this->goodRecord, this->badRecord, true ) ;

      //* Return to main menu *
      this->ClearWindow () ;
      this->PrintInstructions () ;
   }

}  //* End KeypadKeys() *

//*************************
//*     SequenceTest      *
//*************************
//********************************************************************************
//* Execute the keycode test for the specified sequence of records.              *
//*   1) Get the keycode assigned to the target key combination.                 *
//*   2) For non-printing keycodes, get the raw escape sequence.                 *
//* If keycode received matches keycode expected, _and_ if escape sequence       *
//* captured successfully (or is one of the special ASCII codes), then mark the  *
//* record as a match. Else save the captured keycode and escape sequence data   *
//* and mark record as a mismatch.                                               *
//*                                                                              *
//* Input  : kIndex  : index of first record to test                             *
//*          kCount  : number of records to test                                 *
//*                                                                              *
//* Returns: nothing                                                             *
//*          The 'goodRecord' and 'badRecord' members report test results.       *
//********************************************************************************

void Keymap::SequenceTest ( short kIndex, short kCount )
{
   const short COL2 = 25,                 // offset to column 2
               COL3 = 45 ;                // offset to column 3
   gString  gsOut ;                       // output formatting
   wkeyCode wk ;                          // key input
   attr_t outAttr ;                       // color attribute for output
   short padCols ;                        // align output by column
   short *gbPtr ;                         // point to good/bad record count
   this->goodRecord = this->badRecord = ZERO ;  // reset counters

   //* Walk through the sequence of records. *
   for ( short k = ZERO ; k < kCount ; ++k, ++kIndex )
   {
      gsOut.compose( " %S: ", this->keyLookup[kIndex].desc ) ;
      this->WriteParagraph ( gsOut, this->menuAttr, true ) ;
      padCols = gsOut.gscols() ;          // move to column 2
      gsOut.clear() ;
      gsOut.padCols( COL2 - padCols ) ;
      nc.GetKeyInput ( wk ) ;             // get user response
      this->WriteParagraph ( gsOut, this->menuAttr, true ) ;

      if ( wk.type == wktFUNKEY )
      {
         if ( (wk.key >= this->tt2Min) && (wk.key <= this->tt2Max) )
         {
            this->keyLookup[kIndex].kcode = wk.key ;
            if ( wk.key == this->keyLookup[kIndex].code )
            {
               this->keyLookup[kIndex].match = true ;
               gbPtr = &this->goodRecord ;
               ++*gbPtr ;
               outAttr = this->okAttr ;
            }
            else
            {
               this->keyLookup[kIndex].match = false ;
               gbPtr = &this->badRecord ;
               ++*gbPtr ;
               outAttr = this->errAttr ;
            }

            gsOut.compose( "0x%04X  %S", &wk.key, this->keyLookup[kIndex].name ) ;
            if ( !this->briefOut )
            {
               gsOut.append( L' ' ) ;
               gsOut.padCols( (COL3 - this->yxPos.xpos), L'-' ) ; // move to column 3
               if ( !(this->AppendEscSequence ( gsOut, kIndex )) )
                  outAttr = this->errAttr ;  // (bad esc seq)
            }
         // if ( this->keyLookup[kIndex].match == false) 
         // { gsOut.append( "\tcode:0x%04X", &this->keyLookup[kIndex].code ) ; }
            gsOut.append( L"\n" ) ;
            this->WriteParagraph ( gsOut, outAttr, true ) ;
         }

         else if ( (wk.key >= this->tt1Min) && (wk.key <= this->tt1Max) )
         {
            if ( wk.key == nckC_C )
            {
               this->WriteParagraph ( 
                     L"Operation Paused:\n"
                      "Press Ctrl+r to Retest previous key entry.\n"
                      "Press Ctrl+s to Skip current key test.\n"
                      "Press Ctrl+c again to abort test and return to main menu "
                      "(or Enter to continue). ", this->menuAttr ) ;
               nc.GetKeyInput ( wk ) ;
               this->WriteParagraph ( L"\n", this->menuAttr ) ;
               if ( wk.type == wktFUNKEY )
               {
                  if ( wk.key == nckC_R )       // step back to previous record
                  {
                     kIndex -= 2 ;
                     k -= 2 ;
                     --*gbPtr ;
                     this->WriteParagraph ( L"\n", this->menuAttr, true ) ;
                  }
                  else if ( wk.key == nckC_S )
                  {
                     //* Re-display the test prompt *
                     gsOut.compose( " %S: ", this->keyLookup[kIndex].desc ) ;
                     gsOut.padCols( COL2 ) ;
                     this->WriteParagraph ( gsOut, this->menuAttr, false ) ;

                     //* Substitute the default value for the skipped keycode *
                     this->stInsertDefault ( kIndex ) ;
                  }
                  else if ( wk.key == nckC_C )  // abort the test sequence
                  {
                     break ;
                  }
                  else  // ignore the interruption and index the previou prompt
                  {
                     --kIndex ;
                     --k ;
                  }
               }
            }
            else if ( wk.key == nckC_R )
            {
               gsOut.clear() ;
               for ( short i = this->yxPos.xpos ; i > ZERO ; --i )
                  gsOut.append( L"\b" ) ;
               this->WriteParagraph ( gsOut, this->menuAttr ) ;
               kIndex -= 2 ;
               k -= 2 ;
               --*gbPtr ;
            }
            else if ( wk.key == nckC_S )
            {
               //* Substitute the default value for the skipped keycode *
               this->stInsertDefault ( kIndex ) ;
            }
            else
            {
               //* This test is primarily for Konsole and Xterm because they *
               //* return ASCII newline i.e. nckENTER for keypad Enter.      *
               if ( this->keyLookup[kIndex].code == nckpENTER )
               {
                  gsOut.compose( "0x%04X  %s\n", &wk.key, this->tt1[wk.key] ) ;
                  this->WriteParagraph ( gsOut, this->normAttr, true ) ;
               }
               else
               {
                  gsOut.append( "0x%04X  %s\n", &wk.key, this->tt1[wk.key] ) ;
                  this->WriteParagraph ( gsOut, this->normAttr, true ) ;
                  --kIndex ;
                  --k ;
               }
            }
         }
         else
         {  // A wktFUNKEY keycode not in range of either table.
            // Although it will work in the header file, it will 
            // not be reported by the Dialog2 test code.
            gsOut.compose( "0x%04X  (outside test range) - ", &wk.key ) ;
            this->keyLookup[kIndex].kcode = wk.key ;
            this->keyLookup[kIndex].match = false ;
            gbPtr = &this->badRecord ;
            ++*gbPtr ;
            outAttr = this->errAttr ;
            if ( !this->briefOut )
            {
               if ( !(this->AppendEscSequence ( gsOut, kIndex )) )
                  outAttr = this->errAttr ;  // (bad esc seq)
            }
            gsOut.append( L"\n" ) ;
            this->WriteParagraph ( gsOut, outAttr, true ) ;
         }
      }
      else if ( wk.type == wktEXTEND )
      {
         gsOut.compose( "0x%04X  %s\n", &wk.key, this->tt3[wk.key] ) ;
         this->WriteParagraph ( gsOut, this->normAttr, true ) ;
         --kIndex ;
         --k ;
      }
      else if ( wk.type == wktPRINT )
      {
         //* Note: Konsole _incorrectly_ reports keypad +-*/ as the equivalent *
         //*       ASCII values. This test compensates for Konsole's stupidity.*
         if ( ((wk.key == L'+') || (wk.key == L'-') || 
               (wk.key == L'*') || (wk.key == L'/')) && 
              ((this->keyLookup[kIndex].code == nckpPLUS)  ||
               (this->keyLookup[kIndex].code == nckpMINUS) ||
               (this->keyLookup[kIndex].code == nckpMULT)  ||
               (this->keyLookup[kIndex].code == nckpDIV)) )
         {
            gsOut.compose( "0x%04X '%C'\n", &wk.key, &wk.key ) ;
            this->WriteParagraph ( gsOut, this->normAttr, true ) ;
         }
         else
         {
            gsOut.compose( "0x%04X '%C'\n", &wk.key, &wk.key ) ;
            this->WriteParagraph ( gsOut, this->normAttr, true ) ;
            --kIndex ;
            --k ;
         }
      }
      else if ( wk.type == wktESCSEQ )
      {
         gsOut.compose( "Untranslated Escape Sequence:" ) ;
         short ccnt ;
         const wchar_t* wPtr = nc.GetEscapeSequence ( ccnt ) ;
         for ( short i = ZERO ; wPtr[i] != NULLCHAR ; ++i )
            gsOut.append( " %02X", &wPtr[i] ) ;
         gsOut.append( L"\n" ) ;
         this->WriteParagraph ( gsOut, this->errAttr, true ) ;
      }
      else if ( wk.type == wktMOUSE )
      {  // This should not happen because the NCurses mouse is not active.
         this->WriteParagraph ( L"Mouse input ignored.\n", this->normAttr, true ) ;
      }
      else  // (wk.type == wktERR)
      {  // This should not happen because we perform 
         // a blocking read on the input stream.
         this->WriteParagraph ( L"Error! Unable to capture the keycode.\n", 
                                this->normAttr, true ) ;
      }
   }     // for(;;)

}  //* End SequenceTest() *

//*************************
//*    stInsertDefault    *
//*************************
//********************************************************************************
//* Insert default keycode into table at specified index.                        *
//* -- Called by SequenceTest() when user skips a test.                          *
//* -- Called by KeypadKeys() when user has specified that there is no keypad.   *
//*                                                                              *
//* Input  : kIndex : index into 'keyLookup' table                               *
//*          msg    : (optional, 'true' by default) if 'true', display message   *
//*                   in both terminal window and in log file.                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::stInsertDefault ( short kIndex, bool msg )
{
   gString gs( this->keyLookup[kIndex].seq ) ;
   gs.copy( this->keyLookup[kIndex].kseq, kdKSEQ_LEN ) ;
   this->keyLookup[kIndex].kcode = this->keyLookup[kIndex].code ;
   this->keyLookup[kIndex].match = true ;
   ++this->goodRecord ;       // default value assumed to be a good value

   if ( msg )
      this->WriteParagraph ( "not tested, default keycode used\n", 
                             this->menuAttr, true ) ;

}  //* End stInsertDefault() *

//*************************
//*   AppendEscSequence   *
//*************************
//********************************************************************************
//* Get the raw (untranslated) escape sequence associated with the specified     *
//* keycode. Format the sequence and append it to caller's display string.       *
//*                                                                              *
//* Input  : gsOut : formatted display data for keycode test                     *
//*          kIndex: index of record under test                                  *
//*                  on success, the record's kseq[] member is initialized       *
//*                                                                              *
//* Returns: 'true'  if successful capture, else 'false'                         *
//********************************************************************************
//* Programmer's Note: The raw escape sequences associated with each key are     *
//* fairly well standardized and are unlikely to deviate from the sequences      *
//* stored in the keyGroup[] array defined above. If the captured sequence is    *
//* different from the sequence stored in keyGroup[],                            *
//* Comparison is not performed, and the captured sequence will silently         *
//* overwrite the original sequence.                                             *
//* If this becomes a problem in future iterations, we can perform a comparison  *
//* and set a flag if mismatch occurs.                                           *
//********************************************************************************

bool Keymap::AppendEscSequence ( gString& gsOut, short kIndex )
{
   bool success = true ;      // return value

   //* Disable keycode translation *
   nc.SetKeyProcState ( nckpNO_TRANSLATION ) ;

   //* Get the keycode (s/b type=wktESCSEQ, key=0x0000) *
   wkeyCode wk ;
   nc.GetKeyInput ( wk ) ;
   gString gsx ;

   //* Get pointer to the escape sequence, format *
   //* it and append it to caller's data          *
   short ccnt ;
   const wchar_t* wPtr = NULL ;
   if ( (wk.type == wktESCSEQ) && (wk.key == 0x0000) &&
      ((wPtr = nc.GetEscapeSequence ( ccnt )) != NULL) && (ccnt > ZERO) )
   {
      for ( short i = ZERO ; wPtr[i] != NULLCHAR ; ++i )
         gsx.append( " %02X", &wPtr[i] ) ;
      gsOut.append( gsx.gstr() ) ;
      gsx.shiftChars( -1 ) ;
      gsx.copy( this->keyLookup[kIndex].kseq, kdKSEQ_LEN ) ;
   }
   else
   {
      //* For nckpPLUS, nckpMINUS, nckpMULT, nckpDIV escape sequences may not be *
      //* returned. For these keys, the ASCII '+', '-', '*' and '/' are returned.*
      //* For Backspace _or_ Delete, ASCII Delete (0x7F) may be returned.        *
      if ( (wk.type == wktPRINT) && 
           ((wk.key == L'+') || (wk.key == L'-') || 
            (wk.key == L'*') || (wk.key == L'/') || (0x007F)) )
      {
         if ( wk.key == 0x007F )
            gsx.compose( " '0x%02X' (ASCII Delete)", &wk.key ) ;
         else
            gsx.compose( " '%C'", &wk.key ) ;
         gsOut.append( gsx.gstr() ) ;
         gsx.shiftChars( -1 ) ;
         gsx.copy( this->keyLookup[kIndex].kseq, kdKSEQ_LEN ) ;
      }
      else if ( (wk.type == wktFUNKEY) && (wk.key == nckENTER) )
      {
         gsx.compose( " '0x%02X' (ASCII Enter)", &wk.key ) ;
         gsOut.append( gsx.gstr() ) ;
         gsx.shiftChars( -1 ) ;
         gsx.copy( this->keyLookup[kIndex].kseq, kdKSEQ_LEN ) ;
      }
      else     // (should never happen)
      {
         gsx = "(escape sequence unknown)" ;
         gsOut.append( gsx.gstr() ) ;
         gsx.copy( this->keyLookup[kIndex].kseq, kdKSEQ_LEN ) ;
         success = false ;
      }
   }

   //* Re-enable keycode translation *
   nc.SetKeyProcState ( nckpRAW ) ;

   return success ;

}  //* End AppendEscSequence() *

//*************************
//*       SingleKey       *
//*************************
//********************************************************************************
//* Test a single user-specified key combination.                                *
//* a) Prompt the user for a key combination.                                    *
//* b) Scan the lookup table for that combination, and if found,                 *
//*    prompt user to enter that combination and enter the results into the      *
//*    table.                                                                    *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::SingleKey ( void )
{
   const short colHeight = 18,
               col2      = (colHeight * 2),
               col2X     = 30,
               col3X     = (col2X * 2) ;
   gString gsOut ;            // text formatting
   wkeyCode wk ;              // user response
   short grpIndex = ZERO,     // base index of group
         trgIndex = ZERO,     // index of target test
         kIndex   = ZERO,     // index within selected group
         kCount   = ZERO,     // number of records in selected group
         goodTest = ZERO ;    // number of successful tests performed
   bool done  = false ;       // loop control


   while ( ! done )
   {
      this->WriteParagraph ( L"\n"
          "\n Test a single keycode. Which keycode group contains the keycode to be tested?"
          "\n n) Navigation key group" 
          "\n f) Function key group" 
          "\n k) Keypad key group"
          "\n or Enter to cancel the test."
          "\n >>> ", 
          this->menuAttr, false ) ;
      nc.GetKeyInput ( wk ) ;
      if ( wk.type == wktPRINT )
      {
         if ( (wk.key == 'n') || (wk.key == 'N') )
         {
            gsOut.compose( L"%C\n ", &wk.key ) ;
            grpIndex = kIndex = this->nvIndex ;
            kCount = this->nvCount ;
         }
         else if ( (wk.key == 'f') || (wk.key == 'F') )
         {
            gsOut.compose( L"%C\n ", &wk.key ) ;
            grpIndex = kIndex = this->fnIndex ;
            kCount = this->fnCount ;
         }
         else if ( (wk.key == 'k') || (wk.key == 'K') )
         {
            gsOut.compose( L"%C\n ", &wk.key ) ;
            grpIndex = kIndex = this->kpIndex ;
            kCount = this->kpCount ;
         }
         else
         {
            gsOut.clear() ;
            done = true ;
         }
         this->WriteParagraph ( gsOut, this->menuAttr ) ;
      }
      else
         done = true ;
      if ( ! done )
      {
         this->ClearWindow () ;
         this->WriteParagraph ( L"Select the keycode to be tested.\n\n", 
                                (this->menuAttr | ncuATTR), false ) ;
         winPos wp = this->yxPos ;     // base cursor position
         short y = wp.ypos,            // base Y cursor position
               i ;                     // record number (1-based)
         kIndex = grpIndex ;
         for ( short k = ZERO ; k < kCount ; ++k, ++kIndex )
         {
            if ( k < colHeight )
               wp = { this->yxPos.ypos, 1 } ;
            else if ( (k >= colHeight) && (k < col2) )
               wp = { short((k == colHeight) ? y : this->yxPos.ypos), col2X } ;
            else if ( k >= col2 )
               wp = { short((k == (colHeight * 2)) ? y : this->yxPos.ypos), col3X } ;
            this->SetCursor ( wp ) ;
            i = k + 1 ;
            gsOut.compose( "%2hd %S\n", &i, this->keyLookup[kIndex].desc ) ;
            this->WriteParagraph ( gsOut, this->menuAttr ) ;
         }
         wp = { short(y + colHeight), ZERO } ;
         this->SetCursor ( wp ) ;
         this->WriteParagraph ( L"\n Select a test by number"
                                 "\n or Enter to cancel the test."
                                 "\n >>> ", this->menuAttr ) ;
         for ( i = 3 ; i > ZERO ; --i )
         {
            nc.GetKeyInput ( wk ) ;
            if ( wk.type == wktPRINT )
            {
               if ( (wk.key >= L'0') && (wk.key <= L'9') )
               {
                  gsOut.compose( "%C", &wk.key ) ;
                  this->WriteParagraph ( gsOut, this->menuAttr ) ;
                  if ( trgIndex == ZERO )
                     trgIndex = short(wk.key - L'0') ;
                  else
                  {
                     trgIndex *= 10 ;
                     trgIndex += short(wk.key - L'0') ;
                     break ;
                  }
               }
            }
            else if ( (trgIndex > ZERO) &&      // single=digit selection
                      (wk.type == wktFUNKEY) && (wk.key == nckENTER) )
            { break ; }
            else
            { done = true ; break ; }
         }
         if ( ! done )        // perform the selected test
         {
            this->briefOut = false ;   // no brief mode for single-test
            if ( goodTest == ZERO )    // display instructions
            {
               this->WriteParagraph ( L"\n     ", this->menuAttr ) ;
               this->ModeMsg () ;
            }
            this->WriteParagraph ( L"\n    ", this->menuAttr ) ;
            if ( (trgIndex > ZERO) && (trgIndex <= kCount) )
            {
               trgIndex += grpIndex - 1 ;
               this->SequenceTest ( trgIndex, 1 ) ;
               ++goodTest ;
            }
            else
               this->WriteParagraph ( L"Out-of-range!\n", this->errAttr ) ;
            trgIndex = ZERO ;          // reset the index
            this->WriteParagraph ( L" Another Test? (y/n): ", this->menuAttr ) ;
            nc.GetKeyInput ( wk ) ;
            if ( !(wk.type == wktPRINT) || !((wk.key == L'y') || (wk.key == L'Y')) )
               done = true ;
         }
      }
   }  // while()

   if ( goodTest == ZERO )
   {
      this->WriteParagraph ( L" test cancelled.\n", this->menuAttr ) ;
      chrono::duration<short>aMoment( 1 ) ;
      this_thread::sleep_for( aMoment ) ;
      this->ClearWindow () ;
      this->PrintInstructions () ;
   }
   else
      this->InstructionPrompt ( true ) ;

}  //* End SingleKey() *

//*************************
//*     SummaryReport     *
//*************************
//********************************************************************************
//* Produce a summary report of testing using the provided data.                 *
//*                                                                              *
//* Input  : recCount : total records in test group                              *
//*          recTested: number of records tested                                 *
//*          recGood  : number of good tests                                     *
//*          recBad   : number of bad tests                                      *
//*          pause    : if 'true', pause for user input                          *
//*                     if 'false', return immediately                           *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::SummaryReport ( short recCount, short recTested, 
                             short recGood, short recBad, bool pause )
{
   if ( this->yxPos.xpos != ZERO )
      this->WriteParagraph ( L"\n", this->menuAttr, true ) ;
   this->WriteParagraph ( L" ", this->menuAttr, true ) ;
   gString gsOut( "** Total:%hd  Tested:%hd  Good:%hd  Mismatch:%hd **\n", 
                  &recCount, &recTested, &recGood, &recBad ) ;
   this->WriteParagraph ( gsOut, (this->menuAttr | ncrATTR), true ) ;
   if ( pause )
   {
      this->WriteParagraph ( L" Press any key to continue...", this->menuAttr ) ;
      nckPause();
   }

}  //* End SummaryReport() *

//*************************
//*   CreateQueryTable    *
//*************************
//********************************************************************************
//* Arrange the NcDialog library's keycode lookup table according to functional  *
//* grouping and within each group arrange the combination in a logical and      *
//* consistent sequence. This is the order in which the user will be queried.    *
//*                                                                              *
//* In future, we may dynamically create this table; however, it is currently    *
//* impractical and unnecessary to do so, and we hard-code the table.            *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* -- Keycodes in the range 0x0000 - 0x001F are ASCII control codes and are     *
//*    not handled by this application.                                          *
//* -- The ASCII range 0x0020 - 0x007E are the standard printing characters      *
//*    which are not handled by this application.                                * 
//*    -- Note that the ncurses library includes 0x007F (ASCII DEL) as part      *
//*       of the ASCII group.                                                    *
//* -- The so-called extended-ASCII range (0x0080 - 0x00FF) is not used in       *
//*    UTF-8 encoding, and will "never" be seen in a GNU-Linux environment.      *
//* -- For the ncurses C-language library, the range of valid keycodes which     *
//*    are locally available for definition is 0x100 - 0x1FF (0400-0777 octal).  *
//*    -- Of these, 0x0100 and 0x0101 are reserved.                              *
//*    -- Not all keycodes in the range are defined within the ncurses library.  *
//* -- The largest keycode currently reported on any of our systems is 0x024A    *
//*    (keypad Minus), but we want to leave some room for growth, so the keycode *
//*    translation table, "TransTable2[]" embedded within the NcDialogAPI is     *
//*    designed to hold keycodes between 0x0102 and 0x024B.                      *
//*    Of course, not all of these keycodes are defined for any give system,     *
//*    and some of those keycodes will only be seen on old DEC, DG, Wang or      *
//*    AS/400 systems. These systems will seldom be see outside a museum (or in  *
//*    the back of our garage).                                                  *
//* -- Wintel harware dominates the modern workstation world, and on Wintel      *
//*    machines, there are approximately 150 defined keycodes in the             *
//*    0x0102-0x024B  range. While we make no attemt to create a comprehensive   *
//*    map, we identify the key combinations in this range that might be of      *
//*    practical use for an application's user interface.                        *
//*                                                                              *
//********************************************************************************

void Keymap::CreateQueryTable ( void )
{
//********************************************************************************
//* Static key lookup table.                                                     *
//* ========================                                                     *
//* The table is based on the keycode definitions generated under:               *
//* Fedora Linux v:39                                                            *
//* GNOME terminal v:3.50.1 (verified with Konsole and Xterm)                    *
//* ncurses      v:6.4.20230520                                                  *
//*                                                                              *
//* -- Marked as "(untrans)" if key combination generates an                     *
//*    untranslated escape sequence.                                             *
//* -- Marked as "(undef)" if no response to key combination.                    *
//* -- 'DUP' signifies that the keypad key returns the same escape sequence as   *
//*          the corresponding key on the main keyboard.                         *
//********************************************************************************
static const short keygroupCOUNT = 154 ;
static keyDesc keyGroup[keygroupCOUNT] = 
{
   //** Navigation Key Group (52 records) **
   { nckUP,       L"nckUP",      L"1B 5B 41",             L"UpArrow"                 },
   { nckSUP,      L"nckSUP",     L"1B 5B 31 3B 32 41",    L"Shift+UpArrow"           },
   { nckAUP,      L"nckAUP",     L"1B 5B 31 3B 33 41",    L"Alt+UpArrow"             },
   { nckASUP,     L"nckASUP",    L"1B 5B 31 3B 34 41",    L"Alt+Shift+UpArrow"       },
   { nckCUP,      L"nckCUP",     L"1B 5B 31 3B 35 41",    L"Ctrl+UpArrow"            },
   { nckCSUP,     L"nckCSUP",    L"1B 5B 31 3B 36 41",    L"Ctrl+Shift+UpArrow"      },
   { nckACUP,     L"nckACUP",    L"1B 5B 31 3B 37 41",    L"Alt+Ctrl+UpArrow"        },

   { nckDOWN,     L"nckDOWN",    L"1B 5B 42",             L"DownArrow"               },
   { nckSDOWN,    L"nckSDOWN",   L"1B 5B 31 3B 32 42",    L"Shift+DownArrow"         },
   { nckADOWN,    L"nckADOWN",   L"1B 5B 31 3B 33 42",    L"Alt+DownArrow"           },
   { nckASDOWN,   L"nckASDOWN",  L"1B 5B 31 3B 34 42",    L"Alt+Shift+DownArrow"     },
   { nckCDOWN,    L"nckCDOWN",   L"1B 5B 31 3B 35 42",    L"Ctrl+DownArrow"          },
   { nckCSDOWN,   L"nckCSDOWN",  L"1B 5B 31 3B 36 42",    L"Ctrl+Shift+DownArrow"    },
   { nckACDOWN,   L"nckACDOWN",  L"1B 5B 31 3B 37 42",    L"Alt+Ctrl+DownArrow"      },

   { nckLEFT,     L"nckLEFT",    L"1B 5B 44",             L"LeftArrow"               },
   { nckSLEFT,    L"nckSLEFT",   L"1B 5B 31 3B 32 44",    L"Shift+LeftArrow"         },
   { nckALEFT,    L"nckALEFT",   L"1B 5B 31 3B 33 44",    L"Alt+LeftArrow"           },
   { nckASLEFT,   L"nckASLEFT",  L"1B 5B 31 3B 34 44",    L"Alt+Shift+LeftArrow"     },
   { nckCLEFT,    L"nckCLEFT",   L"1B 5B 31 3B 35 44",    L"Ctrl+LeftArrow"          },
   { nckCSLEFT,   L"nckCSLEFT",  L"1B 5B 31 3B 36 44",    L"Ctrl+Shift+LeftArrow"    },
   { nckACLEFT,   L"nckACLEFT",  L"1B 5B 31 3B 37 44",    L"Alt+Ctrl+LeftArrow"      },

   { nckRIGHT,    L"nckRIGHT",   L"1B 5B 43",             L"RightArrow"              },
   { nckSRIGHT,   L"nckSRIGHT",  L"1B 5B 31 3B 32 43",    L"Shift+RightArrow"        },
   { nckARIGHT,   L"nckARIGHT",  L"1B 5B 31 3B 33 43",    L"Alt+RightArrow"          },
   { nckASRIGHT,  L"nckASRIGHT", L"1B 5B 31 3B 34 43",    L"Alt+Shift+RightArrow"    },
   { nckCRIGHT,   L"nckCRIGHT",  L"1B 5B 31 3B 35 43",    L"Ctrl+RightArrow"         },
   { nckCSRIGHT,  L"nckCSRIGHT", L"1B 5B 31 3B 36 43",    L"Ctrl+Shift+RightArrow"   },
   { nckACRIGHT,  L"nckACRIGHT", L"1B 5B 31 3B 37 43",    L"Alt+Ctrl+RightArrow"     },

   { nckPGUP,     L"nckPGUP",    L"1B 5B 35 7E",          L"PageUp"                  },
   { nckSPGUP,    L"nckSPGUP",   L"1B 5B 35 3B 32 7E",    L"Shift+PageUp"            },
   { nckAPGUP,    L"nckAPGUP",   L"1B 5B 35 3B 33 7E",    L"Alt+PageUp"              },
   { nckASPGUP,   L"nckASPGUP",  L"1B 5B 35 3B 34 7E",    L"Alt+Shift+PageUp"        },
   { nckACPGUP,   L"nckACPGUP",  L"1B 5B 35 3B 37 7E",    L"Alt+Ctrl+PageUp"         },

   { nckPGDOWN,   L"nckPGDOWN",  L"1B 5B 36 7E",          L"PageDown"                },
   { nckSPGDN,    L"nckSPGDN",   L"1B 5B 36 3B 32 7E",    L"Shift+PageDown"          },
   { nckAPGDN,    L"nckAPGDN",   L"1B 5B 36 3B 33 7E",    L"Alt+PageDown"            },
   { nckASPGDN,   L"nckASPGDN",  L"1B 5B 36 3B 34 7E",    L"Alt+Shift+PageDown"      },
   { nckACPGDN,   L"nckACPGDN",  L"1B 5B 36 3B 37 7E",    L"Alt+Ctrl+PageDown"       },

   { nckHOME,     L"nckHOME",    L"1B 5B 48",             L"Home"                    },
   { nckSHOME,    L"nckSHOME",   L"1B 5B 31 3B 32 48",    L"Shift+Home"              },
   { nckAHOME,    L"nckAHOME",   L"1B 5B 31 3B 33 48",    L"Alt+Home"                },
   { nckASHOME,   L"nckASHOME",  L"1B 5B 31 3B 34 48",    L"Alt+Shift+Home"          },
   { nckCHOME,    L"nckCHOME",   L"1B 5B 31 3B 35 48",    L"Ctrl+Home"               },
   { nckCSHOME,   L"nckCSHOME",  L"1B 5B 31 3B 36 48",    L"Ctrl+Shift+Home"         },
   { nckACHOME,   L"nckACHOME",  L"1B 5B 31 3B 37 48",    L"Alt+Ctrl+Home"           },

   { nckEND,      L"nckEND",     L"1B 5B 46",             L"End"                     },
   { nckSEND,     L"nckSEND",    L"1B 5B 31 3B 32 46",    L"Shift+End"               },
   { nckAEND,     L"nckAEND",    L"1B 5B 31 3B 33 46",    L"Alt+End"                 },
   { nckASEND,    L"nckASEND",   L"1B 5B 31 3B 34 46",    L"Alt+Shift+End"           },
   { nckCEND,     L"nckCEND",    L"1B 5B 31 3B 35 46",    L"Ctrl+End"                },
   { nckCSEND,    L"nckCSEND",   L"1B 5B 31 3B 36 46",    L"Ctrl+Shift+End"          },
   { nckACEND,    L"nckACEND",   L"1B 5B 31 3B 37 46",    L"Alt+Ctrl+End"            },

   //** Function-key Group (48 records)
   { nckF01,      L"nckF01",     L"1B 4F 50",             L"F01"                     },
   { nckF02,      L"nckF02",     L"1B 4F 51",             L"F02"                     },
   { nckF03,      L"nckF03",     L"1B 4F 52",             L"F03"                     },
   { nckF04,      L"nckF04",     L"1B 4F 53",             L"F04"                     },
   { nckF05,      L"nckF05",     L"1B 5B 31 35 7E",       L"F05"                     },
   { nckF06,      L"nckF06",     L"1B 5B 31 37 7E",       L"F06"                     },
   { nckF07,      L"nckF07",     L"1B 5B 31 38 7E",       L"F07"                     },
   { nckF08,      L"nckF08",     L"1B 5B 31 39 7E",       L"F08"                     },
   { nckF09,      L"nckF09",     L"1B 5B 32 30 7E",       L"F09"                     },
   { nckF10,      L"nckF10",     L"1B 5B 32 31 7E",       L"F10"                     },
   { nckF11,      L"nckF11",     L"1B 5B 32 33 7E",       L"F11"                     },
   { nckF12,      L"nckF12",     L"1B 5B 32 34 7E",       L"F12"                     },

   { nckSF01,     L"nckSF01",    L"1B 5B 31 3B 32 50",    L"Shift+F01"               },
   { nckSF02,     L"nckSF02",    L"1B 5B 31 3B 32 51",    L"Shift+F02"               },
   { nckSF03,     L"nckSF03",    L"1B 5B 31 3B 32 52",    L"Shift+F03"               },
   { nckSF04,     L"nckSF04",    L"1B 5B 31 3B 32 53",    L"Shift+F04"               },
   { nckSF05,     L"nckSF05",    L"1B 5B 31 35 3B 32 7E", L"Shift+F05"               },
   { nckSF06,     L"nckSF06",    L"1B 5B 31 37 3B 32 7E", L"Shift+F06"               },
   { nckSF07,     L"nckSF07",    L"1B 5B 31 38 3B 32 7E", L"Shift+F07"               },
   { nckSF08,     L"nckSF08",    L"1B 5B 31 39 3B 32 7E", L"Shift+F08"               },
   { nckSF09,     L"nckSF09",    L"1B 5B 32 30 3B 32 7E", L"Shift+F09"               },
   { nckSF10,     L"nckSF10",    L"1B 5B 32 31 3B 32 7E", L"Shift+F10"               },
   { nckSF11,     L"nckSF11",    L"1B 5B 32 33 3B 32 7E", L"Shift+F11"               },
   { nckSF12,     L"nckSF12",    L"1B 5B 32 34 3B 32 7E", L"Shift+F12"               },

   { nckCF01,     L"nckCF01",    L"1B 5B 31 3B 35 50",    L"Ctrl+F01"                },
   { nckCF02,     L"nckCF02",    L"1B 5B 31 3B 35 51",    L"Ctrl+F02"                },
   { nckCF03,     L"nckCF03",    L"1B 5B 31 3B 35 52",    L"Ctrl+F03"                },
   { nckCF04,     L"nckCF04",    L"1B 5B 31 3B 35 53",    L"Ctrl+F04"                },
   { nckCF05,     L"nckCF05",    L"1B 5B 31 35 3B 35 7E", L"Ctrl+F05"                },
   { nckCF06,     L"nckCF06",    L"1B 5B 31 37 3B 35 7E", L"Ctrl+F06"                },
   { nckCF07,     L"nckCF07",    L"1B 5B 31 38 3B 35 7E", L"Ctrl+F07"                },
   { nckCF08,     L"nckCF08",    L"1B 5B 31 39 3B 35 7E", L"Ctrl+F08"                },
   { nckCF09,     L"nckCF09",    L"1B 5B 32 30 3B 35 7E", L"Ctrl+F09"                },
   { nckCF10,     L"nckCF10",    L"1B 5B 32 31 3B 35 7E", L"Ctrl+F10"                },
   { nckCF11,     L"nckCF11",    L"1B 5B 32 33 3B 35 7E", L"Ctrl+F11"                },
   { nckCF12,     L"nckCF12",    L"1B 5B 32 34 3B 35 7E", L"Ctrl+F12"                },

   { nckCSF01,    L"nckCSF01",   L"1B 5B 31 3B 36 50",    L"Ctrl+Shift+F01"          },
   { nckCSF02,    L"nckCSF02",   L"1B 5B 31 3B 36 51",    L"Ctrl+Shift+F02"          },
   { nckCSF03,    L"nckCSF03",   L"1B 5B 31 3B 36 52",    L"Ctrl+Shift+F03"          },
   { nckCSF04,    L"nckCSF04",   L"1B 5B 31 3B 36 53",    L"Ctrl+Shift+F04"          },
   { nckCSF05,    L"nckCSF05",   L"1B 5B 31 35 3B 36 7E", L"Ctrl+Shift+F05"          },
   { nckCSF06,    L"nckCSF06",   L"1B 5B 31 37 3B 36 7E", L"Ctrl+Shift+F06"          },
   { nckCSF07,    L"nckCSF07",   L"1B 5B 31 38 3B 36 7E", L"Ctrl+Shift+F07"          },
   { nckCSF08,    L"nckCSF08",   L"1B 5B 31 39 3B 36 7E", L"Ctrl+Shift+F08"          },
   { nckCSF09,    L"nckCSF09",   L"1B 5B 32 30 3B 36 7E", L"Ctrl+Shift+F09"          },
   { nckCSF10,    L"nckCSF10",   L"1B 5B 32 31 3B 36 7E", L"Ctrl+Shift+F10"          },
   { nckCSF11,    L"nckCSF11",   L"1B 5B 32 33 3B 36 7E", L"Ctrl+Shift+F11"          },
   { nckCSF12,    L"nckCSF12",   L"1B 5B 32 34 3B 36 7E", L"Ctrl+Shift+F12"          },

   //** Keypad Group (54 records) **
   { nckpCENTER,  L"nckpCENTER", L"1B 5B 45",             L"keypad Center"           },
// { nckpCENTER,  L"(untrans)",  L"", L"keypad Alt+Center"       },
// { nckpCENTER,  L"(untrans)",  L"", L"keypad Ctrl+Center"      },
// { nckpCENTER,  L"(untrans)",  L"", L"keypad Alt+Ctrl+Center"  },

   { nckpUP,      L"nckpUP",     L"1B 5B 41",             L"keypad UpArrow"          },/*DUP*/
   { nckpAUP,     L"nckpAUP",    L"1B 5B 31 3B 33 41",    L"keypad Alt+UpArrow"      },/*DUP*/
   { nckpCUP,     L"nckpCUP",    L"1B 5B 31 3B 35 41",    L"keypad Ctrl+UpArrow"     },/*DUP*/
   { nckpACUP,    L"nckpACUP",   L"1B 5B 31 3B 37 41",    L"keypad Alt+Ctrl+Up"      },/*DUP*/

   { nckpDOWN,    L"nckpDOWN",   L"1B 5B 42",             L"keypad DownArrow"        },/*DUP*/
   { nckpADOWN,   L"nckpADOWN",  L"1B 5B 31 3B 33 42",    L"keypad Alt+DownArrow"    },/*DUP*/
   { nckpCDOWN,   L"nckpCDOWN",  L"1B 5B 31 3B 35 42",    L"keypad Ctrl+DownArrow"   },/*DUP*/
   { nckpACDOWN,  L"nckpACDOWN", L"1B 5B 31 3B 37 42",    L"keypad Alt+Ctrl+Down"    },/*DUP*/

   { nckpLEFT,    L"nckpLEFT",   L"1B 5B 44",             L"keypad LeftArrow"        },/*DUP*/
   { nckpALEFT,   L"nckpALEFT",  L"1B 5B 31 3B 33 44",    L"keypad Alt+LeftArrow"    },/*DUP*/
   { nckpCLEFT,   L"nckpCLEFT",  L"1B 5B 31 3B 35 44",    L"keypad Ctrl+LeftArrow"   },/*DUP*/
   { nckpACLEFT,  L"nckpACLEFT", L"1B 5B 31 3B 37 44",    L"keypad Alt+Ctrl+Left"    },/*DUP*/

   { nckpRIGHT,   L"nckpRIGHT",  L"1B 5B 43",             L"keypad RightArrow"       },/*DUP*/
   { nckpARIGHT,  L"nckpARIGHT", L"1B 5B 31 3B 33 43",    L"keypad Alt+RightArrow"   },/*DUP*/
   { nckpCRIGHT,  L"nckpCRIGHT", L"1B 5B 31 3B 35 43",    L"keypad Ctrl+RightArrow"  },/*DUP*/
   { nckpACRIGHT, L"nckpACRIGHT",L"1B 5B 31 3B 37 43",    L"keypad Alt+Ctrl+Right"   },/*DUP*/

   { nckpINSERT,  L"nckpINSERT", L"1B 5B 32 7E",          L"keypad Insert"           },/*DUP*/
   { nckpAINSERT, L"nckpAINSERT",L"1B 5B 32 3B 33 7E",    L"keypad Alt+Insert"       },/*DUP*/
// { nckpCINS,    L"(undef) ",   L"                 ",    L"keypad Ctrl+Delete"      },

   { nckpDELETE,  L"nckpDELETE", L"1B 5B 33 7E",          L"keypad Delete"           },/*DUP*/
   { nckpADEL,    L"nckpADEL",   L"1B 5B 33 3B 33 7E",    L"keypad Alt+Delete"       },/*DUP*/
   { nckpCDEL,    L"nckpCDEL",   L"1B 5B 33 3B 35 7E",    L"keypad Ctrl+Delete"      },/*DUP*/
   { nckpACDEL,   L"nckpACDEL",  L"1B 5B 33 3B 37 7E",    L"keypad Alt+Ctrl+Delete"  },

   { nckpPGUP,    L"nckpPGUP",   L"1B 5B 35 7E",          L"keypad PageUp"           },/*DUP*/
   { nckpAPGUP,   L"nckpAPGUP",  L"1B 5B 35 3B 33 7E",    L"keypad Alt+PageUp"       },/*DUP*/
   { nckpCPGUP,   L"nckpCPGUP",  L"1B 5B 35 3B 35 7E",    L"keypad Ctrl+PageUp"      },/*DUP*/
   { nckpACPGUP,  L"nckpACPGUP", L"1B 5B 35 3B 37 7E",    L"keypad Alt+Ctrl+PageUp"  },/*DUP*/
   { nckpPGDN,    L"nckpPGDN",   L"1B 5B 36 7E",          L"keypad PageDown"         },/*DUP*/
   { nckpAPGDN,   L"nckpAPGDN",  L"1B 5B 36 3B 33 7E",    L"keypad Alt+PageDown"     },/*DUP*/
   { nckpCPGDN,   L"nckpCPGDN",  L"1B 5B 36 3B 35 7E",    L"keypad Ctrl+PageDown"    },/*DUP*/
   { nckpACPGDN,  L"nckpACPGDN", L"1B 5B 36 3B 37 7E",    L"keypad Alt+Ctrl+PageDown"},/*DUP*/

   { nckpHOME,    L"nckpHOME",   L"1B 5B 48",             L"keypad Home"             },/*DUP*/
   { nckpAHOME,   L"nckpAHOME",  L"1B 5B 31 3B 33 48",    L"keypad Alt+Home"         },/*DUP*/
   { nckpCHOME,   L"nckpCHOME",  L"1B 5B 31 3B 35 48",    L"keypad Ctrl+Home"        },/*DUP*/
   { nckpACHOME,  L"nckpACHOME", L"1B 5B 31 3B 37 48",    L"keypad Alt+Ctrl+Home"    },/*DUP*/

   { nckpEND,     L"nckpEND",    L"1B 5B 46",             L"keypad End"              },/*DUP*/
   { nckpAEND,    L"nckpAEND",   L"1B 5B 31 3B 33 46",    L"keypad Alt+End"          },/*DUP*/
   { nckpCEND,    L"nckpCEND",   L"1B 5B 31 3B 35 46",    L"keypad Ctrl+End"         },/*DUP*/
   { nckpACEND,   L"nckpACEND",  L"1B 5B 31 3B 37 46",    L"keypad Alt+Ctrl+End"     },/*DUP*/

   { nckpPLUS,    L"nckpPLUS",   L"'+'",                  L"keypad Plus"             },
// { nckAPLUS,    L"(untrans)",  L"", L"keypad Alt+Plus"         },

   { nckpMINUS,   L"nckpMINUS",  L"'-'",                  L"keypad Minus"            },
// { nckpMINUS,   L"(untrans)",  L"", L"keypad Alt+Minus"        },

   { nckpMULT,    L"nckpMULT",   L"'*'",                  L"keypad Multiply"         },
// { nckpMULT,    L"(untrans)",  L"", L"keypad Alt+Multiply"     },
// { nckpMULT,    L"(untrans)",  L"", L"keypad Alt+Ctrl+Multiply"},

   { nckpDIV,     L"nckpDIV",    L"'/'",                  L"keypad Divide"           },
// { nckpADIV,    L"(undef)",    L"", L"keypad Alt+Divide"       },
// { nckpACDIV,   L"(undef)",    L"", L"keypad Alt+Ctrl+Divide"  },

   { nckINSERT,   L"nckINSERT",  L"1B 5B 32 7E",          L"Insert key (main keybrd)"},
   { nckAINSERT,  L"nckAINSERT", L"1B 5B 32 3B 33 7E",    L"Alt+Insert"              },
// { nckSINSERT,  L"(cutbuff)",  L"", L"Shift+Insert"            },
// { nckCINSERT,  L"(undef)",    L"", L"Ctrl+Insert"             },

   { nckDELETE,   L"nckDELETE",  L"1B 5B 33 7E      ",    L"Delete key (main keybrd)"},
   { nckSDEL,     L"nckSDEL",    L"1B 5B 33 3B 32 7E",    L"Shift+Delete"            },
   { nckADEL,     L"nckADEL",    L"1B 5B 33 3B 33 7E",    L"Alt+Delete"              },
   { nckASDEL,    L"nckASDEL",   L"1B 5B 33 3B 34 7E",    L"Alt+Shift+Delete"        },
   { nckCDEL,     L"nckCDEL",    L"1B 5B 33 3B 35 7E",    L"Ctrl+Delete"             },
   { nckCSDEL,    L"nckCSDEL",   L"1B 5B 33 3B 36 7E",    L"Ctrl+Shift+Delete"       },
// { nckACSDEL,   L"(untrans)",  L"", L"Alt+Ctrl+Shift+Delete"   },

   { nckBKSP,     L"nckBKSP",    L"'0x7F' (ASCII Delete)", L"Backspace"             },
   { nckSTAB,     L"nckSTAB",    L"1B 5B 5A",             L"Shift+Tab"               },

   { nckpENTER,   L"nckpENTER",  L"'0x0A' (ASCII Enter)", L"keypad Enter"           },
// { nckpAENT,    L"(undef)",    L"", L"keypad Alt+Enter"        },
// { nckpACENT,   L"(undef)",    L"", L"keypad Alt+Ctrl+Enter"   },
// { nckpCENT,    L"(undef)",    L"", L"keypad Ctrl+Enter"       },
} ;

   #define DEBUG_CQT (0)

   gString gs ;         // format and copy

   //* Access the human-readable keycode names *
   //* as defined in the NcDialog library.     *
   nc.GetTransTable1 ( this->tt1, this->tt1Min, this->tt1Max ) ;
   nc.GetTransTable2 ( this->tt2, this->tt2Min, this->tt2Max ) ;
   nc.GetTransTable3 ( this->tt3, this->tt3Count ) ;

   //* Position and number of keycodes in each test group *
   //* within our local lookup table.                     *
   this->keyLookup = keyGroup ;
   this->klCount = keygroupCOUNT ;
   this->nvIndex = ZERO ;
   for ( short indx = ZERO ; indx < this->klCount ; ++indx )
   {
      //* If first record in Function Key group *
      if ( this->keyLookup[indx].code == nckF01 )
         this->nvCount = this->fnIndex = indx ;
      //* If first record in Keypad Key group *
      else if ( this->keyLookup[indx].code == nckpCENTER )
      {
         this->fnCount = indx - this->nvCount ;
         this->kpIndex = indx ;
      }
      //* If last record in Keypad Key group *
      else if ( this->keyLookup[indx].code == nckpENTER )
         this->kpCount = ++indx - (this->nvCount + this->fnCount) ;
   }

   //* Initialization for keyboards without a numeric keypad *
   for ( short indx = this->kpIndex ; indx < this->klCount ; ++indx )
   {
      if ( keyLookup[indx].code == nckINSERT )
         this->kpnIndex = indx ;
      else if ( keyLookup[indx].code == nckpENTER )
         this->kpnCount = indx - this->kpnIndex ;
   }

   #if DEBUG_CQT != 0   // FOR DEBUGGIN ONLY
   gs.compose( "\nkeyLookup[%hd]:\n"
               "  nvIndex:  %02hd nvCount: %02hd\n"
               "  fnIndex:  %02hd fnCount: %02hd\n"
               "  kpIndex: %02hd kpCount: %02hd\n"
               "  kpnIndex:%02hd kpnCount:%02hd\n",
               &this->klCount, 
               &this->nvIndex,  &this->nvCount,
               &this->fnIndex,  &this->fnCount,
               &this->kpIndex,  &this->kpCount,
               &this->kpnIndex, &this->kpnCount ) ;
   this->WriteParagraph ( gs, nc.reB ) ;
   nckPause();
   #endif   // DEBUG_CQT
   #undef DEBUG_CQU

}  //* End CreateQueryTable() *

//*************************
//*   PrintInstructions   *
//*************************
//********************************************************************************
//* Print user-interface instructions at the current window position.            *
//* (not written to log file)                                                    *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::PrintInstructions ( void )
{
   this->WriteParagraph (
      L"Select an operation:\n"
       " 'a' test All key groups.           "" 'i' to display this Instruction menu.\n"
       " 'n' Navigation key group test.     "" 'c' to Clear display.\n"
       " 'f' Function Key group test.       "" 'g' Generate C++ header file\n"
       " 'k' Keypad key group test.         "" 'v' View/save log file.\n"
       " 's' test a Single key combo.       "" 'h' to display application Help.\n"
       " 'x' (or 'q') to eXit.\n"
       "  ", this->menuAttr, false ) ;
   
}  //* End PrintInstructions() *

//*************************
//*   InstructionPrompt   *
//*************************
//********************************************************************************
//* Display a brief instruction prompt.                                          *
//*                                                                              *
//* Input  : nl  : (optional, 'false' by default)                                *
//*                if 'true', print the message on a new line                    *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::InstructionPrompt ( bool nl )
{
   gString gsOut( L"Press 'i' for Instructions.\n" ) ;
   if ( nl )
      gsOut.insert( L'\n' ) ;
   this->WriteParagraph ( gsOut, this->menuAttr ) ;

}  //* End InstructionPrompt() *

//*************************
//*        ModeMsg        *
//*************************
//********************************************************************************
//* Display 'standard mode' or 'brief mode' instructions.                        *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

void Keymap::ModeMsg ( void )
{
   if ( this->briefOut )
      this->WriteParagraph ( "Brief Mode", (this->menuAttr | ncuATTR) ) ;
   else
      this->WriteParagraph ( "Standard Mode", (this->menuAttr | ncuATTR) ) ;
   this->WriteParagraph ( " is active. Enter each key-combination ", 
                          this->menuAttr ) ;
   if ( this->briefOut )
      this->WriteParagraph ( " ONCE ", this->menuAttr | ncrATTR ) ;
   else
      this->WriteParagraph ( " TWICE ", this->menuAttr | ncrATTR ) ;
   this->WriteParagraph ( ".\n", this->menuAttr ) ;

}  //* End ModeMsg() *

//*************************
//*    WriteParagraph     *
//*************************
//********************************************************************************
//* Write the specified data to the display at the current cursor position.      *
//*                                                                              *
//* Input  : msg  : (gString&, wchar_t* or char*) text data to be written        *
//*          attr : color attribute                                              *
//*          toLog: (optional, 'false' by default)                               *
//*                 'true'  copy 'msg' to the log file                           *
//*                 'false' do not update log file                               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::WriteParagraph ( const gString& msg, attr_t attr, bool toLog )
{
   const char *startMenuAttr  = "\033[0;34m",   // blue foreground
              *startNormAttr  = "\033[0;30m",   // black foreground
              *startOkAttr    = "\033[0;32m",   // green foreground
              *startErrAttr   = "\033[1;31m" ;  // bold red foreground
   const char *attrPtr ;

   this->yxPos = nc.WriteStream ( msg.gstr(), attr ) ;

   //* Copy the data to log file *
   if ( toLog && (this->ofsLog.is_open()) )
   {
      //* Apply ANSI color attribute to data output. *
      //* Note that the ANSI sequence is in effect   *
      //* only until a NEWLINE is encountered, so    *
      //* the color attribute must be inserted into  *
      //* the stream at the start of each line.      *
      if ( (attr == this->menuAttr) || 
           (attr == (this->menuAttr | ncuATTR)) ||
           (attr == (this->menuAttr | ncrATTR)) )
         attrPtr = startMenuAttr ;
      else if ( (attr == this->okAttr) || (attr == (this->okAttr | ncrATTR)) )
         attrPtr = startOkAttr ;
      else if ( (attr == this->errAttr) || (attr == (this->errAttr | ncrATTR)) )
         attrPtr = startErrAttr ;
      else
         attrPtr = startNormAttr ;
      short nlindx = msg.find( wNEWLINE ) ;
      if ( (nlindx < ZERO) || (nlindx == ((msg.gschars()) - 2)) )
      {
         this->ofsLog << attrPtr ;
         this->ofsLog << msg.ustr() ;
      }
      else
      {
         short x = (msg.utfbytes() - 1) ;
         char utfBuff[gsDFLTBYTES] ;
         msg.copy( utfBuff, gsDFLTBYTES ) ;
         if ( utfBuff[ZERO] != NEWLINE )
            this->ofsLog << attrPtr ;
         for ( short i = ZERO ; i < x ; ++i )
         {
            if ( utfBuff[i] == NEWLINE )
               this->ofsLog << utfBuff[i] << attrPtr ;
            else
               this->ofsLog << utfBuff[i] ;
         }
      }
      this->ofsLog.flush() ;
   }

}  //* End WriteParagraph() *

void Keymap::WriteParagraph ( const wchar_t *msg, attr_t attr, bool toLog )
{

   gString gs( msg ) ;
   this->WriteParagraph ( gs, attr, toLog ) ;

}  //* End WriteParagraph() *

void Keymap::WriteParagraph ( const char *msg, attr_t attr, bool toLog )
{

   gString gs( msg ) ;
   this->WriteParagraph ( gs, attr, toLog ) ;

}  //* End WriteParagraph() *

//*************************
//*       SetCursor       *
//*************************
//********************************************************************************
//* Set the output cursor position.                                              *
//*                                                                              *
//* Input  : new_wp : new cursor position                                        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::SetCursor ( const winPos new_wp )
{

   nc.SetCursor ( new_wp ) ;

}  //* End SetCursor() * ;

//*************************
//*    CreateTemppath     *
//*************************
//********************************************************************************
//* Create a unique directory in the system's temp-file directory to contain     *
//* the application's temporary files. Resulting path is returned to caller.     *
//*                                                                              *
//* This method calls 'mkdtemp' to create the unique directory name.             *
//* All temporary files created by the application will live in this directory,  *
//* so clean-up on exit will be easy.                                            *
//*                                                                              *
//* Note that the temp file for exit-to-CWD is not stored in this directory      *
//* because the invocation shell script needs it _after_ exit.                   *
//*                                                                              *
//* Input  : basePath: (by reference) receives the path/dirname                  *
//*                    for creation of temporary files                           *
//*                                                                              *
//* Returns: 'true'  if path/dirname created successfully                        *
//*          'false' if library call failed                                      *
//********************************************************************************
//* Programmer's Note: We do this in the base application code rather than in    *
//* the FileDlg class because it is done only once for the application, AND      *
//* because the base code needs access to the temp-file directory.               *
//********************************************************************************

bool Keymap::CreateTemppath ( gString& basePath )
{
   char tn[gsDFLTBYTES] ;              // receives the particularized filespec
   bool  status = false ;              // return value

   //* Get path of temp directory*
   if ( (this->GetTempdirPath ( basePath )) != false )
   {
      basePath.append( "/KMAP_XXXXXX" ) ;
      basePath.copy( tn, gsDFLTBYTES ) ;
      if ( (mkdtemp ( tn )) != NULL )
      {
         basePath = tn ;
         status = true ;
      }
      else
         basePath.clear() ;
   }
   return status ;

}  //* End CreateTemppath() *

//*************************
//*    CreateTempname     *
//*************************
//********************************************************************************
//* Create a unique path/filename for a temporary file within the previously     *
//* established application temporary directory. (See CreateTemppath())          *
//*                                                                              *
//* Input  : tmpPath: (by reference) receives the path/filename                  *
//*                                                                              *
//* Returns: 'true'  if path/filename created successfully                       *
//*          'false' if library call failed                                      *
//********************************************************************************

bool Keymap::CreateTempname ( gString& tmpPath )
{
   bool  status = false ;

   char tn[gsDFLTBYTES] ;
   gString gstmp( "%s/KMAP_XXXXXX", this->tfPath ) ;
   gstmp.copy( tn, gsDFLTBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpPath = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpPath.clear() ;

   return status ;

}  //* End CreateTempname() *

//*************************
//*    DeleteTemppath     *
//*************************
//********************************************************************************
//* Delete the directory containing the application's temporary files.           *
//*                                                                              *
//* Called by the destructor. At this point, the directory _should be_ empty,    *
//* but in case it is not, we first delete the contents of the directory, and    *
//* then delete the directory itself.                                            *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//*          (There is no point in complaining if we fail.)                      *
//********************************************************************************
//* Programmer's Note: Fortunately, we will see only non-special filenames       *
//* here in the temp directory; however, please see the FMgr-class for notes     *
//* on using 'rmdir' and 'unlink' on filenames which contain "special"           *
//* characters.                                                                  *
//********************************************************************************

void Keymap::DeleteTemppath ( void )
{
   //* First scan the target directory for its contents *
   //* and delete each remaining temp file (if any).    *
   DIR* dirPtr = NULL ;
   if ( (dirPtr = opendir ( this->tfPath )) != NULL )
   {
      gString   tspec ;             // target filespec
      deStats*  destat ;            // directory-entry record
      bool      showFile ;          // 'true' if file is to be copied

      //* Read each item in the directory *
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names. *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            //* Build full path/filename for target, then delete the file.*
            // Programmer's Note: The filetype test uses the compiler's 
            // primitive: DT_DIR: (unsigned char)(4).
            tspec.compose( "%s/%s", this->tfPath, destat->d_name ) ;
            if ( destat->d_type == DT_DIR )
            {
               // Programmer's Note: At this time the application does not 
               // create any subdirectories in our tempfile directory, so this 
               // is an unnecessary step.
               rmdir ( tspec.ustr() ) ;   // remove the directory
            }
            else
               unlink ( tspec.ustr() ) ;            
         }
      }
      closedir ( dirPtr ) ;      // Close the directory

      //* Delete the directory itself. *
      rmdir ( this->tfPath ) ;
   }

}  //* End DeleteTemppath() *

//*************************
//*    GetTempdirPath     *
//*************************
//********************************************************************************
//* This is a stub function which provides access to the C++17 function:         *
//*                       temp_directory_path()                                  *
//*                                                                              *
//* Because most programmer's will not reach C++17 for at least several          *
//* months, we isolate the call here and implement what we expect is the way     *
//* the library function will be implemented.                                    *
//*         (This WILL NOT work under Windoze--and we don't care.)               *
//*         (Screw Windoze, and the horse it rode in on.         )               *
//* -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --*
//* From <http://en.cppreference.com>                                            *
//* ---------------------------------                                            *
//* These functions are available in C++17 (released Jan. 2018):                 *
//* const char* fs::temp_directory_path ( void ) ;                               *
//* const char* fs::temp_directory_path ( std::error_code& ec ) ;                *
//*                                                                              *
//* Returns A directory suitable for temporary files. The path is guaranteed     *
//* to exist and to be a directory. The overload that takes error_code&          *
//* argument returns an empty path on error.                                     *
//*                                                                              *
//* Exceptions: The overload that does not take a std::error_code& parameter     *
//* throws filesystem_error on underlying OS API errors, constructed with path   *
//* to be returned as the first argument and the OS error code as the error      *
//* code argument. std::bad_alloc may be thrown if memory allocation fails.      *
//* The overload taking a std::error_code& parameter sets it to the OS API       *
//* error code if an OS API call fails, and executes ec.clear() if no errors     *
//* occur.                                                                       *
//*                                                                              *
//* On POSIX systems, the path may be the one specified in the environment       *
//* variables TMPDIR, TMP, TEMP, TEMPDIR, and, if none of them are specified,    *
//* the path "/tmp" is returned.                                                 *
//*                                                                              *
//* Input  : tdPath : (by reference) receives the path string                    *
//*                                                                              *
//* Returns: 'true'  if successful, 'false' if system error                      *
//********************************************************************************
//* Programmer's Note: As of C++17 the library provides a method for acquiring   *
//* the temp directory path.                                                     *
//*   std::filesystem::path p = std::filesystem::temp_directory_path () ;        *
//*   gString tdPath( p.c_str() ) ;                                              *
//*                                                                              *
//* As of NcDialogAPI v:0.0.37, however, only C++11 is required.                 *
//* We may advance to C++17 in a future release.                                 *
//********************************************************************************

bool Keymap::GetTempdirPath ( gString& tdPath )
{
   //* Default path of temp directory on GNU/Linux systems. *
   //* (Used only if environment variable not set.)         *
   const char* const dfltPath = "/tmp" ;

   const char* envPath ;            // returned by getenv()
   FileStats   fStats ;             // target filestats
   bool        status = false ;     // return value

   if ( (envPath = std::getenv ( "TMPDIR" )) == NULL )
      if ( (envPath = std::getenv ( "TMP" )) == NULL )
         if ( (envPath = std::getenv ( "TEMP" )) == NULL )
            if ( (envPath = std::getenv ( "TEMPDIR" )) == NULL )
               envPath = dfltPath ;
   tdPath = envPath ;
   if ( (stat64 ( tdPath.ustr(), &fStats )) == OK )
   {
      if ( ((S_ISDIR(fStats.st_mode)) != false) &&
           ((access ( tdPath.ustr(), R_OK )) == ZERO) && 
           ((access ( tdPath.ustr(), W_OK )) == ZERO) )
         status = true ;
   }
   return status ;

}  //* End GetTempdirPath()

//*************************
//*        GetLine        *
//*************************
//********************************************************************************
//* Generic ifs.getline() method.                                                *
//* 1) Read the raw data line (discarding the newline character).                *
//* 2) If a Windoze carriage-return ('\r') character at end of line, delete it.  *
//*                                                                              *
//*                                                                              *
//* Input  :  ifs     : (by reference) input data stream                         *
//*           gsIn    : (by reference) receives the stream data                  *
//*                                                                              *
//* Returns: number of bytes read from the stream                                *
//*          (incl. CR if present, not incl. discarded newline)                  *
//********************************************************************************

short Keymap::GetLine ( std::ifstream& ifs, gString& gsIn )
{
   char  inBuff[gsDFLTBYTES] ;      // input buffer
   short inBytes ;                  // return value

   ifs.getline( inBuff, gsDFLTBYTES, NEWLINE ) ;
   if ( (inBytes = ifs.gcount()) > ZERO )
   {
      short indx = (inBytes - 2) ;  // protect against Windoze CR/LF combo
      if ( (indx >= ZERO) && (inBuff[indx] == '\r') )
         inBuff[indx] = NULLCHAR ;
      gsIn = inBuff ;
   }
   return ( inBytes ) ;

}  //* End GetLine() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//********************************************************************************
//* Capture user's command-line arguments.                                       *
//* Valid Arguments: see DisplayHelp() method.                                   *
//*                                                                              *
//* Input  : commArgs class object (by reference)                                *
//*                                                                              *
//* Returns: 'false' if normal startup                                           *
//*          'true'  if request for help or invalid command-line argument        *
//********************************************************************************
//* Programmer's Note: Even though it is not documented, we accept both          *
//* lowercase and uppercase option and sub-option characters where possible.     *
//* This allows us to add new, case-sensitive options at a later date.           *
//********************************************************************************

bool Keymap::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA (0)
//   const char EQUAL = '=' ;

   //* Get the application executable's directory path.        *
   //* Save the actual, absolute path specification, replacing *
   //* symbolic links and relative path specifications.        *
   if ( (realpath ( ca.argList[0], ca.appPath )) != NULL )
   {
      gString gs( ca.appPath ) ;
      gs.limitChars( (gs.findlast( L'/' )) ) ;
      gs.copy( ca.appPath, MAX_PATH ) ;
   }

   //* If user provided command-line arguments *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
//      const char* argPtr ;       // pointer to option argument
      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            #if DEBUG_GCA != 0  // debugging only
            wcout << L"TOKEN:" << i << L" '" << &ca.argList[i][j] << L"'" << endl ;
            #endif   // DEBUG_GCA

            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //* Request for application version number *
               if ( (gs.compare( L"--version" )) == ZERO )
                  ca.helpFlag = ca.verFlag = true ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  ca.helpFlag = true ;

               else  // invalid argument
               { ca.helpFlag = true ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            char argLetter = ca.argList[i][j] ;

            //* Generate a default C++ header file (no user interaction) *
            if ( argLetter == 'g' || argLetter == 'G' )
            {
               ca.autoFlag = true ;
            }

            //* Generate the TransTable2[] lookup table (no user interaction) *
            else if ( argLetter == 't' || argLetter == 'T' )
            {
               ca.ttab2Flag = true ;
            }

            //* Specify brief output *
            else if ( argLetter == 'b' || argLetter == 'B' )
            {
               ca.briefFlag = true ;
            }

#if 0    // DISABLE - NOT CURRENTLY USED
            //* Delay between the display of a "enter key" prompt *
            //* and the timeout while waiting for a response.     *
            else if ( argLetter == 'd' || argLetter == 'D' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument

               //* Capture the delay (in seconds) and convert. *
               short delay ;
               if ( (sscanf ( argPtr, "%hd", &delay )) == 1 )
               {
                  if ( delay < ZERO )
                     delay = -(delay) ;
                  if ( delay > 30 )
                     delay = 30 ;
                  ca.keyTimeout = delay ;
               }
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               continue ;     // finished with this argument
            }

            //* Enable Mouse support. *
            else if ( argLetter == 'm' || argLetter == 'M' )
            {
               ca.mouseFlag = true ;
            }
#endif   // DISABLE - NOT CURRENTLY USED

            //* Pause after start-up sequence so user can see diagnostics *
            else if ( argLetter == 'p' || argLetter == 'P' )
            {
               char subchar = tolower ( ca.argList[i][j + 1] ) ;
               if ( subchar == 'v' )
               {
                  ca.diagPause = 2 ;
                  ++j ;
               }
               else
                  ca.diagPause = 1 ;
            }

            //* Else, is either 'h', 'H' or an invalid argument.*
            //* Either way, invoke help.                        *
            else
            {
               ca.helpFlag = true ;
               if ( argLetter != 'h' && argLetter != 'H' )
                  break ;
            }

            //* If separate tokens have been concatenated *
            if ( ca.argList[i][j + 1] != nckNULLCHAR )
               multiarg = true ;
         }
         else  // invalid argument, token does not begin with a DASH character
         { ca.helpFlag = true ; break ; }

      }     // for(;;)

      #if DEBUG_GCA != 0  // debugging only
      wcout << L"commArgs Settings" << endl ;
      gs.compose( L"%hd", &ca.argCount ) ;
      wcout << L"argCount   : " << gs.gstr() << endl ;
      for ( short i = ZERO ; i < ca.argCount ; i++ )
      {
         gs.compose( L"argList[%hd] : '%s'", &i, ca.argList[i] ) ;
         wcout << gs.gstr() << endl ;
      }
      gs.compose( L"appPath    : '%s'", ca.appPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"autoFlag: %hhd", &ca.autoFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"ttab2Flag: %hhd", &ca.ttab2Flag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"briefFlag: %hhd", &ca.briefFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"diagPause  : %hd", &ca.diagPause ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"mouseFlag  : %hhd", &ca.mouseFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"verFlag    : %hhd", &ca.verFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"helpFlag   : %hhd", &ca.helpFlag ) ;
      wcout << gs.gstr() << endl ;
      wcout << L"\n Press Enter" << endl ;
      getchar () ;
      #endif   // DEBUG_GCA
   }
   return ca.helpFlag ;

   #undef DEBUG_GCA
}  //* End GetCommandLineArgs() *

//*************************
//*      ClearWindow      *
//*************************
//********************************************************************************
//* Clear the display and set cursor position to origin (0,0).                   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::ClearWindow ( void )
{

   nc.ClearScreen () ;
   this->yxPos = { ZERO, ZERO } ;

}  //* End ClearWindow() *

//*************************
//*       DiagMsg         *
//*************************
//********************************************************************************
//* Display start-up diagnostic message in console window.                       *
//* DO NOT call this method after dialog window has opened.                      *
//*                                                                              *
//* Input  : msg    : message to be displayed                                    *
//*          color  : color attribute for message                                *
//*          newLine: (optional, 'true' by default)                              *
//*                   if 'true'  move cursor to next message position            *
//*                   if 'false' leave cursor at end of message                  *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::DiagMsg ( const char* msg, attr_t color, bool newLine )
{
   static short xCol = ZERO, xColBaseY = 11 ;

   this->suPos = nc.WriteString ( this->suPos.ypos, this->suPos.xpos, msg, color ) ;
   if ( newLine )
   {
      this->suPos = { short(this->suPos.ypos + 1), xCol  } ;
      if ( xCol == ZERO && (this->suPos.ypos >= this->termRows) )
      {
         xCol = this->termCols / 2 + 1 ;
         this->suPos = { short(xColBaseY + 1), xCol } ;
      }
   }

}  //* End DiagMsg() *

//*************************
//*    DisplayVersion     *
//*************************
//********************************************************************************
//* Print application version number and copyright notice to tty (bypassing      *
//* NCurses output).                                                             *
//* The NCurses engine has been shut down (or didn't start) so we use simple     *
//* console I/O to display the version/copyright text.                           *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::DisplayVersion ( void )
{
   const short fsWidth = 68 ;
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) + fsWidth), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << wNEWLINE << freeSoftware << endl ;

}  //* End DisplayVersion() *

//*************************
//*    DisplayHelp        *
//*************************
//********************************************************************************
//* Print command-line help to tty (bypassing NCurses output).                   *
//* The NCurses engine has been shut down (or didn't start) so we use simple     *
//* console I/O to display the help text.                                        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Keymap::DisplayHelp ( void )
{
   static const char* Help = 
    //12345678901234567890123456789012345678901234567890123456789012345678901234567890
   "\nUSAGE: keymap [OPTIONS]   Example: keymap --version"
   "\n       Options may be specified in any order. If an option is not specified,"
   "\n       then its default value will be used."
   "\n -g   Generate a C++ header file using default keycode values."
   "\n -t   Generate a lookup table sorted numerically by keycode."
   "\n -b   Brief output which will report only the captured keycodes, but"
   "\n      not the underlying escape sequences."
   "\n -p   Pause to display startup diagnostics before entering main program."
   "\n --version        Display application version number, then exit."
   "\n --help, -h, -?   Help for command line options"
   "\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) * 2), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << Help << endl ;

}  //* End DisplayHelp() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

