//********************************************************************************
//* File       : Wayclip.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice located below                       *
//* Date       : 19-Mar-2025                                                     *
//* Version    : (see AppVersion string below)                                   *
//*                                                                              *
//* Description:                                                                 *
//* This application replaces the previous 'Dialogx' application which was       *
//* designed to interface with the X11 clipboard. That application was           *
//* functional in a rudimentary way, but X is on its way out. The most likely    *
//* replacement for X is Wayland. While Wayland is certainly a work-in-progress  *
//* it has become relatively stable as of early 2019.                            *
//*                                                                              *
//* This application is designed as a simple test of access to the               *
//* GNOME/Wayland compositor's two clipboards: "Primary" and "Regular".          *
//*                                                                              *
//* From inside the terminal window, access to the Wayland clipboard is          *
//* through the wl-clipboard utilities, "wl-copy" and "wl-paste".                *
//* These utilities provide a straightforward method for accessing the system    *
//* clipboard from shell scripts, Perl scripts, etc. For console applications,   *
//* however, the opportunities for communication errors make direct access       *
//* rather dangerous.                                                            *
//* The WaylandCB class provides a uniform (and relatively bug-free) way to      *
//* access the system clipboard without subjecting application programmers to    *
//* the tedious communications protocols necessary for interacting with other    *
//* console utilities.                                                           *
//*                                                                              *
//* The WaylandCB class handles all direct access to the clipboards, while       *
//* the Wayclip application simply acts as a test suite for that class.          *
//*                        ---  ---  ---  ---                                    *
//*                                                                              *
//* Note: An NcDialogAPI version of the Wayclip application, "Dialogx" is        *
//* available. Both applications communicate with the system clipboard through   *
//* the WaylandCB class. The console version of the application will be          *
//* distributed with the stand-alone WaylandCB package, while the Dialogx        *
//* version of the application is distributed as part of the NcDialogAPI.        *
//*                                                                              *
//* Development Tools:                                                           *
//*   Fedora Linux 39, GNOME terminal 3.50.1                                     *
//*   Wintel (Lenovo) hardware.                                                  *
//*   GNU G++ (Gcc v:13.3.1)                                                     *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//*         Full text of the GPL License may be found in the Texinfo             *
//*         documentation for this program under 'Copyright Notice'.             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.05 19-Mar-2025                                                        *
//*   -- Update to integrate new release of gString class (v:0.0.37).            *
//*   -- Update some comments.                                                   *
//*   -- No change in functionality.                                             *
//*   -- Documentation update.                                                   *
//*                                                                              *
//* v: 0.0.04 08-Jun-2024                                                        *
//*   -- Implement the internal text transfer option used by the application.    *
//*      This is a test of communications between the application and the        *
//*      WaylandCB class.                                                        *
//*      -- This includes a stress test for buffer overflow.                     *
//*         The WaylandCB class arbitrarily sets the maximum transfer buffer     *
//*         to 16KBytes (16,384 UTF-8 bytes or 4,096 UTF-32 codepoints).         *
//*         The test stresses the protocol for preventing buffer overrun.        *
//*   -- Specify I/O locale on startup. This supports formatting of formatted    *
//*      numeric output.                                                         *
//*   -- Documentation update. No change in functionality.                       *
//*                                                                              *
//* v: 0.0.03 25-Apr-2020                                                        *
//*   -- Update some tests to reflect changes in return value of some            *
//*      WaylandCB-class methods.                                                *
//*                                                                              *
//* v: 0.0.02 10-Jan-2020                                                        *
//*   -- Code from v:0.0.01 moved to Dialogx application.                        *
//*   -- Begin conversion to a pure console application.                         *
//*      -- Instead of writing status info to the Billboard control, write       *
//*         to wcout using ANSI color sequencing.                                *
//*      -- NOTE: We assume here that 'true' == 1 and 'false' == 0.              *
//*               The 'wcbPrimary' member is used both as a flag and as an       *
//*               index into CbNames[]. Technically, we shouldn't do that        *
//*               because 'true' is define as != 0. It works here because we     *
//*               (tentativlly) trust the G++ compiler not to change it.         *
//*                                                                              *
//* v: 0.0.01 27-Dec-2019                                                        *
//*   -- Complete rewrite:                                                       *
//*      -- Test command-line access to the Wayland clipboard using the          *
//*         wl-clipboard pair of utilities: "wl-copy" and "wl-paste".            *
//*         These are simple (possibly too simple) utilities that communicate    *
//*         the terminal and the Wayland clipboards. See the WaylandCB class     *
//*         for research results.                                                *
//*         Note that these utilities may not be installed by default, even      *
//*         on a GNOME/Wayland desktop. If invoked on a system where they are    *
//*         not installed, yum/dnf offers to install them. (/usr/bin)            *
//*      -- Access to the clipboard is built into the GTK+ development suite;    *
//*         however, we are philosophically opposed to third-party libraries     *
//*         for two reasons: 1) it complicates development, 2) it reduces the    *
//*         chance for fun/education. (Code warriors are all about the fun :-)   *
//*   -- Basic application structure is derived from "Dialogx" which was         *
//*      written to exercise access to the X11 clipboard. If there is a call     *
//*      for continuing support of X, the similaries between these applications  *
//*      will allow for semi-painless re-integration of the X functionality      *
//*      after the WaylandCB class is stable.                                    *
//*   -- Escaping of "special" characters in clipboard data is borrowed from     *
//*      the FileMangler algorithm used to escape characters in filespecs.       *
//*      This escaping is necessary for passing commands to the shell program.   *
//*   -- This release is the first draft of the Wayland clipboard access code,   *
//*      so it is likely that bugs will hatch. Please send a note about          *
//*      your experiences.                                                       *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
#include "Wayclip.hpp"     //* Application-class definition

//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************

//**********
//*  Data  *
//**********

static const char* const CbNames[] = 
{
   "Regular",
   "Primary"
} ;

static const short TD_ITEMS = 7 ;
static const char* const TestData[TD_ITEMS] = 
{
   // 1)
   "A child of five would understand this.\n"
   "  Send someone to fetch a child of five. -- Groucho Marx",
   // 2)
   "Always remember that you are absolutely unique.\n"
   "  Just like everyone else. -- Margaret Mead",
   // 3)
   "Behind every great man is a woman rolling her eyes. -- Jim Carrey",
   // 4)
   "Roses are red, violets are blue,\n"
   "  I'm schizophrenic, and so am I. -- Oscar Levant",
   // 5)
   "No matter what, your parents are going to worry about you.\n"
   "  I had a tour bus, and my mother still thought I was broke.\n"
   "  Remember: It's your life, not theirs. Just because your parents sent you\n"
   "  to college doesn't mean they bought the rest of your life. -- Lewis Black",
   // 6)
   "Outside of a dog, a book is a man's best friend.\n"
   "  Inside of a dog it's too dark to read. -- Groucho Marx",
   // 7)
   "We'll love you just the way you are if you're perfect. -- Alanis Morissette",
} ;

//*************************
//*         main          *
//*************************
//********************************************************************************
//* Program entry point.                                                         *
//*                                                                              *
//* Command-line Usage:  See "GetCommandLineArgs() method.                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* User may have specified one or more command-line arguments.*
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object and interact with the user.*
   Wayclip* wcptr = new Wayclip( clArgs ) ;

   //* Before returning to the system, delete the Wayclip object.    *
   //* While the class object and all its resources will be released *
   //* by the system on exit, we want to force execution of the      *
   //* destructor for an orderly termination.                        *
   delete wcptr ;

   exit ( ZERO ) ;

}  //* End main() *

//*************************
//*       ~Wayclip        *
//*************************
//********************************************************************************
//* Destructor. Return all resources to the system.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

Wayclip::~Wayclip ( void )
{

   //* If locale specified, delete the object *
   if ( this->ioLocale != NULL )
      delete ( this->ioLocale ) ;

   //* Delete the connection to the Wayland clipboard.*
   delete this->wclip ;

}  //* End ~Wayclip() *

//*************************
//*        Wayclip        *
//*************************
//********************************************************************************
//* Constructor.                                                                 *
//*                                                                              *
//*                                                                              *
//* Input  : commArgs class object (by reference)                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

Wayclip::Wayclip ( commArgs& clArgs )
{
   //* Initialize our data members *
   *this->appPath    = NULLCHAR ;
   this->testData    = TestData[0] ;// default test data
   this->wclip       = NULL ;       // not yet instantiated
   this->wcbConnect  = false ;      // not yet connected
   this->wcbPrimary  = false ;      // target clipboard == 'Regular'
   this->txtfmt      = txtfmtUTF8 ; // default internal communications format
   this->ioLocale    = NULL ;       // locale not yet specified
   this->apnColor = attrBK ;        // application normal text color
   this->aphColor = attrBR ;        // application highlight text color
   this->aprColor = attrBRr ;       // application highlight reversed
   this->apeColor = attrRE ;        // application error-message color
   this->stnColor = attrBL ;        // status normal color
   this->sthColor = attrGR ;        // status highlight color

   //* Set the "locale" (character encoding and text-formatting) *
   //* Locale name is taken from terminal environment, and then  *
   //* is given "global" i.e. application/process scope.         *
   this->ioLocale = new locale("") ;// get default locale from environment
   this->ioLocale->global( *this->ioLocale ) ;

   //* See what the user has to say *
   this->GetCommandLineArgs ( clArgs ) ;

   gString gsOut ;
   gsOut = clArgs.appPath ;                  // application path
   gsOut.copy( this->appPath, MAX_PATH ) ;

   //* Establish the connection between the application *
   //* and the Wayland clipboard. If connection cannot  *
   //* be established, it will be reported when the     *
   //* dialog opens.                                    *
   this->wclip = new WaylandCB() ;
   this->wcbConnect = this->wclip->wcbIsConnected() ;

   //** Request for version/copyright info overrides all.**
   if ( clArgs.verFlag )
      this->DisplayVersion () ;

   //** Request for help overrides main loop.**
   else if ( clArgs.helpFlag )
      this->DisplayHelp () ;

   //** Interact with the user.**
   else if ( this->wcbConnect )
   {
      //* Capture and discard the signal, "SIGINT" i.e. Ctrl+C,     *
      //* otherwise user would be accidentally aborting the session.*
      signal ( SIGINT, SIG_IGN ) ;

      this->UserInterface () ;
   }

   //* Unable to connect with Wayland clipboard.*
   else
   {
      this->WheresWayland ( true ) ;
   }

}  //* End Wayclip() *

//*************************
//*     UserInterface     *
//*************************
//********************************************************************************
//* Interact with the user.                                                      *
//*                                                                              *
//* Input  : none                                                                *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::UserInterface ( void )
{
   //* Display the application title and main menu.*
   this->DisplayTitle ( true, true ) ;

   wchar_t  wch ;
   bool done = false ;
   while ( ! done )
   {
      this->WriteParagraph ( L">> ", this->aphColor, this->apnColor ) ;
      wcin >> wch ;

      if ( wch == L'a' )               // set active clipboard (Regular or Primary)
      {
         this->SpecifyActiveCb () ;
      }
      else if ( wch == L'c' )          // copy test data to active clipboard
      {
         this->SetClipboard ( this->wcbPrimary ) ;
      }
      else if ( wch == L'C' )          // copy test data to INACTIVE clipboard
      {
         this->SetClipboard ( ! this->wcbPrimary ) ;
      }
      else if ( wch == L'p' )          // paste data from active clipboard
      {
         this->GetClipboard ( this->wcbPrimary ) ;
      }
      else if ( wch == L'P' )          // paste data from INACTIVE clipboard
      {
         this->GetClipboard ( ! this->wcbPrimary ) ;
      }
      else if ( wch == L'x' )          // clear active clipboard
      {
         this->ClearClipboard ( this->wcbPrimary ) ;
      }
      else if ( wch == L'X' )          // clear INACTIVE clipboard
      {
         this->ClearClipboard ( ! this->wcbPrimary ) ;
      }
      else if ( wch == L's' )          // specifiy text to be copied to clipboard
      {
         this->SpecifyTestText () ;
      }
      else if ( wch == L'S' )          // specifiy internal text-data format
      {
         this->SpecifyInternalFormat () ;
      }
      else if ( wch == L'r' )          // report available MIME types, CB bytes, version
      {
         this->ReportFormats () ;
      }
      else if ( wch == L'R' )          // reset/reinitialize clipboard connection
      {
         this->ReinitWaylandCB () ;
      }
      else if ( wch == L't' )          // test WaylandCB-to-clipboard connection
      {
         this->TestConnection () ;
      }
      else if ( wch == L'T' )          // Stress test communication buffer
      {
         this->TestBufferingLimits () ;
      }
      else if ( wch == L'm' )          // display main menu
      {
         this->DisplayMenu () ;
      }
      else if ( wch == L'w' )          // clear the terminal window and display main menu
      {
         this->DisplayTitle ( true, true ) ;
      }
      else if ( wch == L'q' || wch == L'Q' ) // quit (exit)
      {
         this->WriteParagraph ( L"bye!\n\n", this->apnColor, this->apnColor ) ;
         done = true ;
      }
   }

}  //* End UserInterface() *

//*************************
//*     SetClipboard      *
//*************************
//********************************************************************************
//* Copy the selected test data to the active clipboard.                         *
//*                                                                              *
//* Note that all the sample text passing through this method for communication  *
//* with the WaylandCB class is less than or equal to MAX_CB_UTF8 bytes, or      *
//* MAX_CB_UTF32 wchar_t characters.                                             *
//* data).                                                                       *
//*                                                                              *
//* Input  : primary : specifies target clipboard                                *
//*                    if 'false', send data to Regular clipboard                *
//*                    if 'true',  send data to Primary clipboard                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::SetClipboard ( bool primary )
{
   const wchar_t* const trgCbDisp[2] = 
   {
      L" (to Wayland \"Regular\" clipboard)",
      L" (to Wayland \"Primary\" clipboard)",
   } ;
   gString gs, gscb ;

   this->WriteParagraph ( L"'Copy Selected'\n", this->sthColor, this->apnColor ) ;

   //* If connected to the system clipboard *
   if ( this->wcbConnect )
   {
      short setStatus ;
      if ( this->txtfmt == txtfmtUTF8 )
      {
         char buff[gsALLOCMED] ;
         this->testData.copy( buff, gsALLOCMED ) ;
         setStatus = this->wclip->wcbSet( buff, -1, primary ) ;
      }
      else if ( this->txtfmt == txtfmtUTF32 )
      {
         wchar_t buff[gsALLOCDFLT] ;
         this->testData.copy( buff, gsALLOCDFLT ) ;
         setStatus = this->wclip->wcbSet( buff, -1, primary ) ;
      }
      else  // (this->txtfmt == txtfmtGSTRING)
      {
         setStatus = this->wclip->wcbSet( this->testData, -1, primary ) ;
      }

      //* Display the target clipboard name and operation status.*
      gs.compose( "%S [%S]\n", trgCbDisp[primary], 
                  ((setStatus >= ZERO) ? L"OK" : 
                   (setStatus == -1 ? L"not connected" : L"system error")) ) ;
      this->WriteParagraph ( gs.gstr(), this->stnColor, this->apnColor ) ;
   }     // (connected)
   else
      this->WheresWayland () ;

}  //* End SetCLipboard() *

//*************************
//*     GetClipboard      *
//*************************
//********************************************************************************
//* Retrieve a copy of the active clipboard's contents.                          *
//*                                                                              *
//* Note that all the sample text passing through this method for communication  *
//* with the WaylandCB class is less or equal to MAX_CB_UTF8 bytes, or           *
//* MAX_CB_UTF32 wchar_ characters.                                              *
//*                                                                              *
//* Input  : primary : specifies source clipboard                                *
//*                    if 'false', retrieve data from Regular clipboard          *
//*                    if 'true',  retrieve data from Primary clipboard          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::GetClipboard ( bool primary )
{
   const wchar_t* const srcCbDisp[2] = 
   {
      L" (from 'Regular' clipboard)",
      L" (from 'Primary' clipboard)",
   } ;
   gString gs, gscb ;
   short cb_chars = -2 ;

   this->WriteParagraph ( L"'Paste'\n", this->sthColor, this->apnColor ) ;

   //* If connected to the system clipboard *
   if ( this->wcbConnect )
   {
      if ( this->txtfmt == txtfmtUTF8 )
      {
         char buff[MAX_CB_UTF8] ;
         cb_chars = this->wclip->wcbGet( buff, MAX_CB_UTF8, primary ) ;
         gscb = buff ;
      }
      else if ( this->txtfmt == txtfmtUTF32 )
      {
         wchar_t buff[MAX_CB_UTF32] ;
         cb_chars = this->wclip->wcbGet( buff, MAX_CB_UTF32, primary ) ;
         gscb = buff ;
      }
      else  // (this->txtfmt == txtfmtGSTRING)
      {
         cb_chars = this->wclip->wcbGet( gscb, gsALLOCDFLT, primary ) ;
      }

      //* Display the source of the pasted data.*
      gs.compose( "%S [%S]\n", srcCbDisp[primary],
                  cb_chars > 1 ? L"OK" : 
                  cb_chars == 1 ? L"clipboard empty" : L"system error" ) ;
      this->WriteParagraph ( gs.gstr(), this->stnColor, this->apnColor ) ;

      //* Display retrieved data *
      if ( (gscb.gschars()) > 1 )
      {
/* TEMP - CZONE - Possible assignment error in gString */ gs.reAlloc( gsALLOCMAX ) ;
         gs.compose( " \"%S\"\n", gscb.gstr() ) ;
         this->WriteParagraph ( gs.gstr(), this->apnColor, this->apnColor ) ;
      }
   }     // (connected)
   else
      this->WheresWayland () ;

}  //* End GetClipboard() *

//*************************
//*    ClearClipboard     *
//*************************
//********************************************************************************
//* Clear the active clipboard. That is, set contents of the clipboard to        *
//* an empty string. Write an informational message.                             *
//*                                                                              *
//* Input  : primary : specifies clipboard to be cleared                         *
//*                    if 'false', clear Regular clipboard                       *
//*                    if 'true',  clear Primary clipboard                       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::ClearClipboard ( bool primary )
{
   if ( this->wcbConnect )
   {
      this->wclip->wcbClear( primary ) ;

      // The empty string SHOULD BE returned *
      gString gscb ;
      this->wclip->wcbGet ( gscb, gsALLOCDFLT, primary ) ;

      gString gsOut( "Clear %S Clipboard (\"%S\")\n", 
                     (primary == this->wcbPrimary ? L"Active" : L"Inactive"), 
                     gscb.gstr() ) ;
      this->WriteParagraph ( gsOut.gstr(), this->sthColor, this->apnColor ) ;
   }
   else
      this->WheresWayland () ;

}  //* End ClearClipboard() *

//*************************
//*      GetCbBytes       *
//*************************
//********************************************************************************
//* Report the number of data bytes stored on the specified clipboard.           *
//*                                                                              *
//* Input  : primary : specifies which clipboard to query:                       *
//*                    'false' == "Regular" clipboard                            *
//*                    'true'  == "Primary" clipboard                            *
//*                                                                              *
//* Returns: number of data bytes                                                *
//********************************************************************************

short Wayclip::GetCbBytes ( bool primary )
{

   return ( (this->wclip->wcbBytes ( primary )) ) ;

}  //* End GetCbBytes() *

short Wayclip::GetCbChars ( bool primary )
{

   return ( (this->wclip->wcbChars ( primary )) ) ;

}  //* End GetCbChars() *

//*************************
//*    SpecifyActiveCb    *
//*************************
//********************************************************************************
//* Present user with options on test data to be used for clipboard testing.     *
//* Alternatively, allow user to enter text data to be used.                     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::SpecifyActiveCb ( void )
{
   gString gsOut( "Specify the clipboard to use for copy/paste operations:\n"
                  " r : %s\n"
                  " p : %s\n", CbNames[0], CbNames[1] ) ;
   this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor ) ;

   wchar_t  wch ;
   short    deadman = 3 ;
   do
   {
      this->WriteParagraph ( L" >> ", this->aphColor, this->apnColor ) ;
      wcin >> wch ;
      if ( wch == L'p' || wch == L'P' )
      { this->wcbPrimary = true ; break ; }
      if ( wch == L'r' || wch == L'R' )
      { this->wcbPrimary = false ; break ; }
   }
   while ( --deadman > ZERO ) ;

   if ( deadman <= ZERO )     // if user could not decide
      this->wcbPrimary = false ;

   gsOut.compose( " Target clipboard set to \"%s\" clipboard\n", 
                  CbNames[this->wcbPrimary] ) ;
   this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor ) ;
   chrono::duration<short, std::milli>aMoment( 900 ) ;
   this_thread::sleep_for( aMoment ) ;  // give user time to read the message
   this->DisplayTitle ( true, true ) ;

}  //* End SpecifyActiveCb() *

//*************************
//*    SpecifyTestText    *
//*************************
//********************************************************************************
//* Present user with options on test data to be used for clipboard testing.     *
//* Alternatively, allow user to enter text data to be used.                     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::SpecifyTestText ( void )
{
   gString gsOut ;
   wchar_t  wch, wchMax = TD_ITEMS + 1 + 0x030 ;
   short indx = 1 ;
   this->WriteParagraph ( L"Select test data, or enter your own data.\n", 
                          this->aphColor, this->apnColor ) ;

   for ( short i = ZERO ; i < TD_ITEMS ; ++i, ++indx )
   {
      gsOut.compose( "%hd) %s\n", &indx, TestData[i] ) ;
      this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor ) ;
   }
   gsOut.compose( "%hd) Write your own test data (1024 characters max)\n", &indx ) ;
   this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor ) ;

   short    deadman = 3 ;
   do
   {
      this->WriteParagraph ( L" >> ", this->aphColor, this->apnColor ) ;
      wcin >> wch ;
      if ( wch >= L'1' && wch <= wchMax )
      {
         indx = wch - 0x031 ; // convert ASCII to numeric index
         break ;
      }
   }
   while ( --deadman > ZERO ) ;

   if ( deadman <= ZERO )        // if user could not decide
      indx = ZERO ;
   if ( indx < TD_ITEMS )        // preformatted text selected
      this->testData = TestData[indx] ;
   else  // (indx == TD_ITEMS)   // allow user to enter free-form text
   {
      this->WriteParagraph ( L"Write up to 1000 character and press Enter:\n >> ", 
                          this->aphColor, this->apnColor ) ;
      wchar_t buff[gsALLOCDFLT] ;
      do
      { wcin.getline( buff, gsALLOCDFLT, L'\n' ) ; }
      while ( wcin.gcount() <= 1 ) ;
      this->testData = buff ;
   }
   this->WriteParagraph ( L"Test data specified. Press 'c' to copy it to clipboard.\n\n", 
                       this->aphColor, this->apnColor ) ;

}  //* End SpecifyTestText() *

//*************************
//* SpecifyInternalFormat *
//*************************
//********************************************************************************
//* Present user with the internal text formatting options. The specified option *
//* will be used for communications between the application and the WaylandCB    *
//* class. Communications between the WaylandCB class and the system clipboard   *
//* uses UTF-8, MIME type: "text/plain;charset=utf8" exclusively.                *
//*                                                                              *
//* Available Options:                                                           *
//*  c) char*                                                                    *
//*  w) wchar_t*                                                                 *
//*  g) gString                                                                  *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::SpecifyInternalFormat ( void )
{
   gString gsOut( "Specify text format for communications with WaylandCB:\n"
                  " c : char*    (UTF-8) (default)\n"
                  " w : wchar_t* (UTF-32)\n"
                  " g : gString  (dual-encoded object)\n" ) ;
   this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor ) ;

   wchar_t  wch ;
   this->WriteParagraph ( L" >> ", this->aphColor, this->apnColor ) ;
   wcin >> wch ;
   if ( wch == L'c' || wch == L'C' )
      this->txtfmt = txtfmtUTF8 ;
   else if ( wch == L'w' || wch == L'W' )
      this->txtfmt = txtfmtUTF32 ;
   else if ( wch == L'g' || wch == L'G' )
      this->txtfmt = txtfmtGSTRING ;

   gsOut.compose( " Communications format set to %s.\n",
                  ((this->txtfmt == txtfmtUTF8)  ? "UTF-8" :
                   (this->txtfmt == txtfmtUTF32) ? "UTF-32" : "dual-encoded") ) ;
   this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor ) ;

   chrono::duration<short, std::milli>aMoment( 900 ) ;
   this_thread::sleep_for( aMoment ) ;  // give user time to read the message
   this->DisplayTitle ( true, true ) ;

}  //* End SpecifyInternalFormat() *

//*************************
//*    TestConnection     *
//*************************
//********************************************************************************
//* Send a short test message to the Wayland clipboard.                          *
//* The called method in WaylandCB class will compare the data sent to the       *
//* clipboard with the data retrieved from the clipboard.                        *
//*                                                                              *
//* Note that this method ignores the 'wcbConnect' member and tries the          *
//* connection regardless.                                                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::TestConnection ( void )
{
   const char* testData = "Is anyone home?" ;
   gString gsOut ;

   if ( (this->wclip->wcbTest( testData )) == wcbsACTIVE )
      gsOut = "Clipboard connection verified.\n" ;
   else
      gsOut = "Communication test failed.\n" ;

   this->WriteParagraph ( gsOut.gstr(), this->sthColor, this->apnColor ) ;

}  //* End TestConnection() *

//*************************
//*  TestBufferingLimits  *
//*************************
//********************************************************************************
//* Test the maximum data size for communications.                               *
//* The maximum buffer size for WaylandCB class is MAX_CB_UTF8 for UTF-8 data,   *
//* or MAX_CB_UTF32 for UTF-32 data.                                             *
//*                                                                              *
//* The following tests are performed for both UTF-8 and UTF-32 data sets.       *
//* 1) Attempt to overflow the buffer and verify that the WaylandCB class denies *
//*    the attempt.                                                              *
//* 2) Using data that are just under the maximum data size, write to the        *
//*    clipboard; then retrieve the data from the clipboard and verify that the  *
//*    data written exactly matches the data reteieved.                          *
//*                                                                              *
//* -- The data used is a small Lorem Ipsum sentence, duplicated as needed to    *
//*    reach the required data size.                                             *
//* -- Note that because Lorem Ipsum is ASCII data the word count would equal    *
//*    the byte count by default. For this reason, we insert the Chinse word     *
//*    "Ma" (horse) into the test data at various points. This exercises the     *
//*    various byte vs. character calculations used during encoding operations.  *
//* -- If communication errors occur during this test, enable the delay option   *
//*    (set the 'wait4it' flag == true). The most likely sources of errors are   *
//*    a VERY SLOW CPU or a slow hard drive where temporary files are stored.    *
//*    Note that our 15-year-old Lenovo laptop computer is able to handle these  *
//*    tests smoothly without inserting any additional inter-operation delay.    *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::TestBufferingLimits ( void )
{
   const char* const LorenWho = "Lorem ipsum dolor sit amet, consectetur "
                                "adipiscing elit, sed do eiusmod tempor "
                                "incididunt ut labore et dolore magna 马aliqua." ;
   chrono::duration<short, std::milli>aMoment( 40 ) ;
   uint8_t* ubuff = NULL ;                // pointer to source byte buffer
   wchar_t* wbuff = NULL ;                // pointer to source word buffer
   gString gs( LorenWho ),                // work buffer
           gsOut, gsVal ;                 // output formatting
   int charCnt = ZERO,                    // number of characters written.
       gsCnt = gs.utfbytes() - 1,         // characters/bytes in work buffer
       trgCnt = MAX_CB_UTF8 - gsCnt - 1,  // target count
       cbBytes, cbb,                      // reported bytes on clipboard
       cbWords, cbw,                      // reported wchar_t words on clipboard
       matchCnt ;                         // comparison counter
   bool  wait4it = false ;                // set to insert a delay between calls

   //************************************
   //* Test the UTF-8 buffering limit.  *
   //************************************
   trgCnt += gsCnt ;    // attempt to overrun the WaylandCB source buffer
   ubuff = new uint8_t[MAX_CB_UTF8 + (gsCnt * 2)] ;
   do
   {
      gs.copy( &ubuff[charCnt], gs.utfbytes() ) ;
      charCnt += gsCnt ;
   }
   while ( charCnt < trgCnt ) ;
   ubuff[charCnt++] = '!' ;               // mark the end of source text
   ubuff[charCnt++] = NULLCHAR ;          // terminate the string
   gsVal.formatInt( charCnt, 6, true ) ;  // report the results
   gsOut.compose( " Write %S UTF-8 bytes to clipboard", gsVal.gstr() ) ;
   gsOut.padCols( 40 ) ; gsOut.append( L": " ) ;
   this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;

   //* Copy the data to the clipboard.                 *
   //* Because length exceeds the buffering limit, the *
   //* data should be truncated at MAX_CB_UTF8 bytes.  *
   cbBytes = this->wclip->wcbSet ( ubuff, charCnt, false ) ;
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   cbb = this->GetCbBytes ( false ) ; // verify the number of bytes actually written
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   gsVal.formatInt( cbBytes, 6, true ) ;  // report the results
   gsOut.compose( "%S bytes written. ", gsVal.gstr() ) ;
   gsVal.formatInt( cbb, 6, true ) ;
   if ( cbBytes == charCnt )
      gsOut.append( "(bytes reported:%S)\n", gsVal.gstr() ) ;
   else
      gsOut.append( "(attempted buffer overrun was prevented)\n" ) ;
   this->WriteParagraph ( gsOut.gstr(), 
                          (cbBytes == charCnt ? this->sthColor : this->apeColor), 
                          this->apnColor ) ;

   gsCnt = gs.utfbytes() - 1 ;            // valid target count
   trgCnt = MAX_CB_UTF8 - gsCnt - 1 ;
   charCnt = ZERO ;                       // reset the counter

   //***********************************************
   //* Load buffer with byte data to NEAR maximum. *
   //***********************************************
   do
   {
      gs.copy( &ubuff[charCnt], gs.utfbytes() ) ;
      charCnt += gsCnt ;
   }
   while ( charCnt < trgCnt ) ;
   ubuff[charCnt++] = '!' ;               // mark the end of source text
   ubuff[charCnt++] = NULLCHAR ;          // terminate the string
   gsVal.formatInt( charCnt, 6, true ) ;  // report the results
   gsOut.compose( " Write %S UTF-8 bytes to clipboard", gsVal.gstr() ) ;
   gsOut.padCols( 40 ) ; gsOut.append( L": " ) ;
   this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;

   //* Copy the data to the clipboard. *
   cbBytes = this->wclip->wcbSet ( ubuff, charCnt, false ) ;
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   cbb = this->GetCbBytes ( false ) ; // verify the number of bytes actually written
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   gsVal.formatInt( cbBytes, 6, true ) ;  // report the results
   gsOut.compose( "%S bytes written. ", gsVal.gstr() ) ;
   gsVal.formatInt( cbb, 6, true ) ;
   if ( cbBytes == charCnt )
      gsOut.append( "(bytes reported:%S)\n", gsVal.gstr() ) ;
   else
      gsOut.append( "(bytes reported:%S - source data truncated)\n", gsVal.gstr() ) ;
   this->WriteParagraph ( gsOut.gstr(), 
                          (cbBytes == charCnt ? this->sthColor : this->apeColor), 
                          this->apnColor ) ;

   //* Retrieve the data from the clipboard *
   //* and compare it to our source.        *
   uint8_t* cbbuff = new uint8_t[MAX_CB_UTF8] ; // allocate a buffer to receive the data
   cbb = this->wclip->wcbGet ( cbbuff, MAX_CB_UTF8, false ) ;
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   gsVal.formatInt( cbb, 6, true ) ;
   gsOut.compose( "  Read %S UTF-8 bytes from clipboard", gsVal.gstr() ) ;
   gsOut.padCols( 40 ) ; gsOut.append( L": " ) ;
   this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;

   //* If bytes read from clipboard == bytes written to clipboard *
   if ( cbb == cbBytes ) 
   {
      matchCnt = ZERO ;
      for ( short indx = ZERO ; indx < cbBytes ; ++indx ) // compare each byte of data
      {
         if ( ubuff[indx] == cbbuff[indx] )
            ++matchCnt ;
      }
      if ( matchCnt == cbBytes )
         gsOut = "retrieved data are identical.\n" ;
      else
      {
         short mm = cbBytes - matchCnt ;
         gsVal.formatInt( matchCnt, 6, true ) ;
         gsOut.compose( "matching bytes: %S and mismatch=%hd\n", gsVal.gstr(), &mm )  ;
      }
      this->WriteParagraph ( gsOut.gstr(), 
                             (matchCnt == cbBytes ? this->sthColor : this->apeColor), 
                             this->apnColor ) ;
   }
   else
   {
      gsOut.compose( "copy/paste mismatch.\n" ) ;
      this->WriteParagraph ( gsOut.gstr(), this->apeColor, this->apnColor ) ;
   }

   delete [] cbbuff ;                     // release the dynamic allocations
   delete [] ubuff ;

   //* Reset the counters *
   gsCnt = gs.gschars() - 1 ;             // characters in work buffer
   charCnt = ZERO ;                       // characters copied
   trgCnt = MAX_CB_UTF32 - gsCnt - 1 ;    // target count
   trgCnt += gsCnt ;    // attempt to overrun the WaylandCB source buffer

   //************************************
   //* Test the UTF-32 buffering limit. *
   //************************************
   wbuff = new wchar_t[MAX_CB_UTF32 + (gsCnt * 2)] ;
   do
   {
      gs.copy( &wbuff[charCnt], gs.gschars() ) ;
      charCnt += gsCnt ;
   }
   while ( charCnt < trgCnt ) ;
   wbuff[charCnt++] = L'!' ;              // mark the end of source data
   wbuff[charCnt++] = NULLCHAR ;          // terminate the string
   gsVal.formatInt( charCnt, 6, true ) ;  // report the results
   gsOut.compose( " Write %S UTF-32 chars to clipboard", gsVal.gstr() ) ;
   gsOut.padCols( 40 ) ; gsOut.append( L": " ) ;
   this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;

   //* Copy the data to the clipboard. *
   cbWords = this->wclip->wcbSet ( wbuff, charCnt, false ) ;
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   cbw = this->GetCbChars ( false ) ; // verify the number of bytes actually written
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   gsVal.formatInt( cbWords, 6, true ) ;
   gsOut.compose( "%S chars written. ", gsVal.gstr() ) ;
   gsVal.formatInt( cbw, 6, true ) ;
   if ( cbWords == charCnt )
      gsOut.append( "(chars reported:%S)\n", gsVal.gstr() ) ;
   else
      gsOut.append( "(attempted buffer overrun was prevented)\n" ) ;
   this->WriteParagraph ( gsOut.gstr(), 
                          (cbWords == charCnt ? this->sthColor : this->apeColor), 
                          this->apnColor ) ;

   gsCnt = gs.gschars() - 1 ;             // valid target count
   trgCnt = MAX_CB_UTF32 - gsCnt - 1 ;
   charCnt = ZERO ;                       // reset the counter

   //**************************************************
   //* Load buffer with wchar_t data to NEAR maximum. *
   //**************************************************
   do
   {
      gs.copy( &wbuff[charCnt], gs.gschars() ) ;
      charCnt += gsCnt ;
   }
   while ( charCnt < trgCnt ) ;
   wbuff[charCnt++] = L'!' ;              // mark the end of source data
   wbuff[charCnt++] = NULLCHAR ;          // terminate the string
   gsVal.formatInt( charCnt, 6, true ) ;  // report the results
   gsOut.compose( " Write %S UTF-32 chars to clipboard", gsVal.gstr() ) ;
   gsOut.padCols( 40 ) ; gsOut.append( L": " ) ;
   this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;

   //* Copy the data to the clipboard. *
   cbWords = this->wclip->wcbSet ( wbuff, charCnt, false ) ;
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   cbw = this->GetCbChars ( false ) ; // verify the number of characters actually written
   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   gsVal.formatInt( cbWords, 6, true ) ;
   gsOut.compose( "%S chars written. ", gsVal.gstr() ) ;
   gsVal.formatInt( cbw, 6, true ) ;
   if ( cbWords == charCnt )
      gsOut.append( "(chars reported:%S)\n", gsVal.gstr() ) ;
   else
      gsOut.append( "(chars reported:%S - source data truncated)\n", gsVal.gstr() ) ;
   this->WriteParagraph ( gsOut.gstr(), 
                          (cbWords == charCnt ? this->sthColor : this->apeColor), 
                          this->apnColor ) ;

   //* Retrieve the data from the clipboard and compare it to our source. *
   wchar_t* cwbuff = new wchar_t[MAX_CB_UTF32] ; // allocate a buffer to receive the data
   cbWords = this->wclip->wcbGet ( cwbuff, MAX_CB_UTF32, false ) ;

   if ( wait4it ) { this_thread::sleep_for( aMoment ) ; }   // wait a moment
   gsVal.formatInt( cbWords, 6, true ) ;
   gsOut.compose( "  Read %S UTF-32 chars from clipboard", gsVal.gstr() ) ;
   gsOut.padCols( 40 ) ; gsOut.append( L": " ) ;
   this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;

   //* If words read from clipboard == words written to clipboard *
   if ( cbWords == charCnt ) 
   {
      matchCnt = ZERO ;
      for ( short indx = ZERO ; indx < charCnt ; ++indx ) // compare each byte of data
      {
         if ( cwbuff[indx] == wbuff[indx] )
            ++matchCnt ;
      }
      gsVal.formatInt( matchCnt, 6, true ) ;
      gsOut.compose( "matching chars: %S ", gsVal.gstr() )  ;
      if ( matchCnt == charCnt )
         gsOut.append( "- retrieved data are identical.\n" )  ;
      else
      {
         int mm = charCnt - matchCnt ;
         gsOut.append( "and mismatch=%d\n", &mm )  ;
      }
      this->WriteParagraph ( gsOut.gstr(), 
                             (matchCnt == charCnt ? this->sthColor : this->apeColor), 
                             this->apnColor ) ;
   }
   else
   {
      gsOut.compose( "copy/paste mismatch.\n" ) ;
      this->WriteParagraph ( gsOut.gstr(), this->apeColor, this->apnColor ) ;
   }

   delete [] cwbuff ;                     // release the dynamic allocations
   delete [] wbuff ;

   #undef DEBUG_TBL_MULTI
}  //* End TestBufferingLimits() *

//*************************
//*    ReinitWaylandCB    *
//*************************
//********************************************************************************
//* Perform a full reset of the WaylandCB connection with the Wayland            *
//* clipboard.                                                                   *
//*                                                                              *
//* Note that this method ignores the 'wcbConnect' member and tries the          *
//* connection regardless.                                                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::ReinitWaylandCB ( void )
{
   bool status ;
   gString gsOut ;

   if ( (status = this->wclip->wcbReinit()) )
      gsOut = "Reinitialization successful.\n" ;
   else
      gsOut = "Reinitialization failed.\n" ;

   this->WriteParagraph ( gsOut.gstr(), this->sthColor, this->apnColor ) ;

}  //* End ReinitWaylandCB() *

//*************************
//*     ReportFormats     *
//*************************
//********************************************************************************
//* Report the available formats (MIME types) for retrieving the data on the     *
//* active clipboard. Also report the number of bytes of data currently on       *
//* the active clipboard.                                                        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::ReportFormats ( void )
{
   gString gsOut, gsTypes ;

   if ( this->wcbConnect )
   {
      //* Get the number of bytes of data on the active clipboard.*
      short cbBytes = this->GetCbBytes ( this->wcbPrimary ) ;

      if ( this->txtfmt == txtfmtUTF8 )
      {
         char buff[gsALLOCDFLT] ;
         this->wclip->wcbTypes( buff, (-1) , this->wcbPrimary ) ;
         gsTypes = buff ;
      }
      else if ( this->txtfmt == txtfmtUTF32 )
      {
         wchar_t buff[gsALLOCDFLT] ;
         this->wclip->wcbTypes( buff, (-1) , this->wcbPrimary ) ;
         gsTypes = buff ;
      }
      else  // (this->txtfmt == txtfmtGSTRING )
      {
         this->wclip->wcbTypes( gsTypes, (-1) , this->wcbPrimary ) ;
      }

      this->WriteParagraph ( L"Available Data Formats\n", 
                             this->sthColor, this->apnColor ) ;

      const short minMIMELINES = 6 ;   // threshold for reporting two types per line
      short nlIndx = -1, nlcnt = ZERO ;
      while ( (nlIndx = gsTypes.find( L'\n', nlIndx + 1 )) >= ZERO )
         ++nlcnt ;

      while ( (gsTypes.gschars()) > 1 )
      {
         gsOut = gsTypes ;
         if ( (nlIndx = gsOut.find( L'\n' )) >= ZERO )
         {
            if ( nlcnt <= minMIMELINES )
               gsOut.limitChars( nlIndx ) ;
            else
            {
               // Programmer's Note: The formatting in this section is rather 
               // kludgy. It works, but it's ugly. Sorry about that :(
               short ni, gstindx = ZERO ;
               if ( ((ni = gsOut.find( L'\n', (nlIndx + 1) )) > nlIndx) || 
                    ((gstindx = gsTypes.find( L'\n', (nlIndx + 1) )) < ZERO) )
               {
                  short i = nlIndx ;
                  gsOut.replace( L"\n", L" " ) ;
                  for ( ; i < 35 ; ++i )
                     gsOut.insert( L' ', i ) ;
                  if ( gstindx >= ZERO )
                     gsOut.limitChars( ni + (i - nlIndx) ) ;
                  nlIndx = gsTypes.find( L'\n', (nlIndx + 1) ) ;
               }
            }
         }
         gsOut.insert( L' ' ) ;
         gsOut.append( L'\n' ) ;
         this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;
         if ( nlIndx >= ZERO )
            gsTypes.shiftChars( -(nlIndx + 1) ) ;
         else
            break ;
      }
      gString gs( cbBytes, 6, true ) ;
      gsOut.compose( " Clipboard Bytes: %S\n", gs.gstr() ) ;
      short indx = gsOut.gschars() ;
      this->wclip->wcbVersion ( gs ) ;
      if ( (indx = gs.find( L'\n', false, indx )) > ZERO )
         gs.replace( L"\n", L"  wl-clipboard v:", indx ) ;
      gsOut.append( " WaylandCB v:%S", gs.gstr() ) ;
      gsOut.append( L'\n' ) ;
     this->WriteParagraph ( gsOut.gstr(), this->stnColor, this->apnColor ) ;
   }
   else
      this->WheresWayland () ;

}  //* End ReportFormats() *

//*************************
//*     WheresWayland     *
//*************************
//********************************************************************************
//* If we were unable to establish a connection with the Wayland clipboard,      *
//* disable the clipboard-option radiobuttons or alert the user.                 *
//*                                                                              *
//* Input  : verbose : (optional, 'false' by default)                            *
//*                    if 'false' write brief message                            *
//*                    if 'true', write verbose message                          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::WheresWayland ( bool verbose )
{
   if ( verbose )
   {
      const short msgCnt = 7 ;
      const wchar_t* msg[msgCnt] = 
      {
         L"\n              Warning!  Warning!",
         L"\n  Unable to establish a connection with the",
         L"\n            Wayland clipboard.",
         L"\n The most likely cause of failure would be",
         L"\n that the \"wl-copy\" and \"wl-paste\" utilities",
         L"\n are not installed or are out-of-date.  Try:",
         L"\n       sudo dnf install 'wl-clipboard'\n\n",
      } ;
      this->WriteParagraph ( msg[0], this->apeColor, this->apnColor ) ;
      this->WriteParagraph ( msg[1], this->stnColor, this->apnColor ) ;
      this->WriteParagraph ( msg[2], this->stnColor, this->apnColor ) ;
      this->WriteParagraph ( msg[3], this->stnColor, this->apnColor ) ;
      this->WriteParagraph ( msg[4], this->stnColor, this->apnColor ) ;
      this->WriteParagraph ( msg[5], this->stnColor, this->apnColor ) ;
      this->WriteParagraph ( msg[6], this->sthColor, this->apnColor ) ;
   }
   else
   {
      this->WriteParagraph ( L"Unable to connect to Wayland clipboard. Try 'R'eset.\n", 
                             this->apeColor, this->apnColor ) ;
   }

}  //* End WhereWayland() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//********************************************************************************
//* Capture user's command-line arguments.                                       *
//* Valid Arguments: see DisplayHelp() method.                                   *
//*                                                                              *
//* Input  : commArgs class object (by reference)                                *
//*                                                                              *
//* Returns: 'false' if normal startup                                           *
//*          'true'  if request for help or invalid command-line argument        *
//********************************************************************************
//* Programmer's Note: Even though it is not documented, we accept both          *
//* lowercase and uppercase option and sub-option characters where possible.     *
//* This allows us to add new, case-sensitive options at a later date.           *
//********************************************************************************

bool Wayclip::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA (0)

   //* Get the application executable's directory path.        *
   //* Save the actual, absolute path specification, replacing *
   //* symbolic links and relative path specifications.        *
   if ( (realpath ( ca.argList[0], ca.appPath )) != NULL )
   {
      gString gs( ca.appPath ) ;
      gs.limitChars( (gs.findlast( L'/' )) ) ;
      gs.copy( ca.appPath, MAX_PATH ) ;
   }

   //* If user provided command-line arguments *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
//      const char* argPtr ;       // pointer to option argument
      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;    // 'true' if concatenated options

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            #if DEBUG_GCA != 0  // debugging only
            wcout << L"TOKEN:" << i << L" '" << &ca.argList[i][j] << L"'" << endl ;
            #endif   // DEBUG_GCA

            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //* Request for application version number *
               if ( (gs.compare( L"--version" )) == ZERO )
                  ca.helpFlag = ca.verFlag = true ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  ca.helpFlag = true ;

               else  // invalid argument
               { ca.helpFlag = true ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            char argLetter = ca.argList[i][j] ;

            //* Set target clipboard to "Primary.*
            if ( argLetter == 'p' || argLetter == 'P' )
            {
               ca.primeFlag = true ;
            }

            //* Set target clipboard to "Regular.*
            else if ( argLetter == 'r' || argLetter == 'R' )
            {
               ca.primeFlag = false ;
            }

            //* Short-form request for help *
            else if ( argLetter == 'h' || argLetter == 'H' )
            { ca.helpFlag = true ; break ; }

            else  // invalid argument
            { ca.helpFlag = true ; break ; }

            //* If separate tokens have been concatenated *
            if ( ca.argList[i][j + 1] != NULLCHAR )
               multiarg = true ;
         }

         else  // invalid argument, token does not begin with a DASH character
         { ca.helpFlag = true ; break ; }

      }     // for(;;)

      #if DEBUG_GCA != 0  // debugging only
      wcout << L"commArgs Settings" << endl ;
      gs.compose( L"%hd", &ca.argCount ) ;
      wcout << L"argCount   : " << gs.gstr() << endl ;
      for ( short i = ZERO ; i < ca.argCount ; i++ )
      {
         gs.compose( L"argList[%hd] : '%s'", &i, ca.argList[i] ) ;
         wcout << gs.gstr() << endl ;
      }
      gs.compose( L"appPath    : '%s'", ca.appPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"primeFlag  : %hhd", &ca.primeFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"verFlag    : %hhd", &ca.verFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"helpFlag   : %hhd", &ca.helpFlag ) ;
      wcout << gs.gstr() << endl ;
      wcout << L"\n Press Enter" << endl ;
      getwchar () ;
      #endif   // DEBUG_GCA
   }
   return ca.helpFlag ;

   #undef DEBUG_GCA
}  //* End GetCommandLineArgs() *

//*************************
//*    WriteParagraph     *
//*************************
//********************************************************************************
//* Write a message to the display.                                              *
//*                                                                              *
//* Input  : msg    : text to be displayed                                       *
//*          bColor : ANSI color sequence to insert before message               *
//*          eColor : ANSI color sequence to append to end of message            *
//*          clear  : (optional, 'false' by default)                             *
//*                   if 'true', clear window before writing message             *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::WriteParagraph ( const wchar_t* msg, attr_t bColor, attr_t eColor, bool clear )
{
   if ( clear )
      system ( "clear" ) ;

   wcout << Ansi[bColor] ;
   wcout << msg ;
   wcout << Ansi[eColor] ;
   wcout.flush() ;

}  //* End WriteParagraph() *

//*************************
//*     DisplayTitle      *
//*************************
//********************************************************************************
//* Clear the terminal window and print the application menu.                    *
//*                                                                              *
//* Input  : clear  : (optional, 'false' by default)                             *
//*                   if 'true', clear window before writing message             *
//*          menu   : (optional, 'false' by default)                             *
//*                   if 'true', also call DisplayMenu() method                  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::DisplayTitle ( bool clear, bool menu )
{
   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) * 2), L'=' ) ;
   gsOut.append( wNEWLINE ) ;

   this->WriteParagraph ( gsOut.gstr(), this->aphColor, this->apnColor, clear ) ;

   if ( menu )
      this->DisplayMenu () ;

}  //* End DisplayTitle() *

//*************************
//*      DisplayMenu      *
//*************************
//********************************************************************************
//* Print the application menu.                                                  *
//*                                                                              *
//* Input  : clear  : (optional, 'false' by default)                             *
//*                   if 'true', clear window before writing message             *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: We trick the terminal for the line which contains a       *
//* non-default background color. Without this trick, the backgroud artifact     *
//* is left on the original line.                                                *
//********************************************************************************

void Wayclip::DisplayMenu ( bool clear )
{
   const wchar_t* const mainMenuA = 
    L"MAIN MENU: (Press ENTER key after command key)\n"
     "a : Set \"active\" clipboard. " ;
   const wchar_t* const mainMenuB = 
    L"Currently active: " ;
   const wchar_t* const mainMenuC = 
    L"c : Copy test data to active clipboard.     m : Display Main Menu\n"
     "p : Paste data from active clipboard.       r : Report available MIME types and misc. info.\n"
     "x : Clear active clipboard.                 R : Reset and re-initialize clipboard connection.\n"
     "s : Specify text to be copied to clipboard. t : Test the clipboard connection.\n"
     "S : Specify internal communications format. T : Test I/O buffer-overrun protection.\n"
     "q : Quit                                    w : Clear the terminal window.\n\n" ;

   this->WriteParagraph ( mainMenuA, this->apnColor, this->apnColor, clear ) ;
   this->WriteParagraph ( mainMenuB, this->aphColor, this->apnColor, clear ) ;
   gString gsOut( " %s ", CbNames[this->wcbPrimary] ) ;
   this->WriteParagraph ( gsOut.gstr(), this->aprColor, this->apnColor ) ;
   this->WriteParagraph ( L" \n", this->apnColor, this->apnColor ) ; // (see note above)
   this->WriteParagraph ( mainMenuC, this->apnColor, this->apnColor ) ;

}  //* End DisplayMenu() *

//*************************
//*    DisplayVersion     *
//*************************
//********************************************************************************
//* Print application version number and copyright notice.                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::DisplayVersion ( void )
{
   const short fsWidth = 68 ;
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <https://www.gnu.org/licenses/>.\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license. There is NO warranty;\n"
    "not even for merchantability or fitness for a particular purpose.\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) + fsWidth), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << wNEWLINE << freeSoftware << endl ;

}  //* End DisplayVersion() *

//*************************
//*    DisplayHelp        *
//*************************
//********************************************************************************
//* Print command-line help.                                                     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Wayclip::DisplayHelp ( void )
{
   static const char* Help = 
    //12345678901234567890123456789012345678901234567890123456789012345678901234567890
   "\nUSAGE: wayclip [OPTIONS]   Example: wayclip --version"
   "\n       Options may be specified in any order. If an option is not specified,"
   "\n       then its default value will be used."
   "\n "
   "\n -p   Set target clipboard to \"Primary\" Wayland clipboard"
   "\n -r   Set target clipboard to \"Regular\" Wayland clipboard (default)"
   "\n "
   "\n --version        Display application version number, then exit."
   "\n --help, -h, -?   Help for command line options"
   "\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) * 2), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << Help << endl ;

}  //* End DisplayHelp() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

