//******************************************************************************
//* File       : ProgbarTest.cpp                                               *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 12-May-2025                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the ProgbarTest class.                   *
//*              This class exercises the Progbar sub-dialog. The Progbar is   *
//*              an independently-defined class which implements a 'progress   *
//*              bar' widget.                                                  *
//*              It is instantiated by the Dialog4 application, Test09.        *
//*              NOTE: This test uses multi-threading to drive the automatic   *
//*                    Progbar update.                                         *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 18-Jul-2015                                                     *
//*   - Basic code adapted from RTL_ContentTest.                               *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "ProgbarTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = pbROWS ;  // display lines
static const short dialogCOLS = pbCOLS ;  // display columns

//* Module-scope access to the dialog for the non-member callback methods *
static NcDialog* pbcuPtr = NULL ;
static Progbar*  pbarPtr = NULL ;
static short pbRemoteUpdate ( short stepsRemaining ) ;
static short pbControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime ) ;

static dspinData dscData = { 1, 65, 20, dspinINTEGER },  // character cells
                 dsdData = { 1,  8,  4, dspinINTEGER } ; // divisions per cell

                 //* Data for the color-selection Dropdown control.*
static const short ddITEMS = 8 ;
static const short ddWIDTH = 27 ;
static char ddText[ddITEMS][ddWIDTH+1] = 
{
   " Red on Black     (brown ) ",
   " Yellow on Cyan   (black ) ",
   " Black on Magenta (blue  ) ",
   " Magenta on Black (yellow) ",
   " Blue on Green    (blue  ) ",
   " Green on Blue    (green ) ",
   " Green on Brown   (white ) ",
   " Blue on White    (brown ) ",
} ;
static attr_t ddAttr[ddITEMS] ;
static attr_t barAttr[ddITEMS] ;


static InitCtrl ic[pbControlsDEFINED] = 
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -   pbDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      (dialogROWS - 2),             // ulY:       upper left corner in Y
      (dialogCOLS - 21),            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  D^ONE  ",                  // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbHoriRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'Horizontal/Vertical' Radiobutton  - - - - - - - - - - -    pbHoriRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      6,                            // ulY:       upper left corner in Y
      29,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Horizonal (reset=vertical)", // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbBrdrRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'With-Border' Radiobutton  - - - - - - - - - - - - - - -    pbBrdrRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[pbHoriRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[pbHoriRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Border around progress bar",// label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbTextRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'With-Text' Radiobutton  - - - - - - - - - - - - - - - -    pbTextRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[pbBrdrRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[pbBrdrRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Add label ^text",            // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbCancRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Add-Cancel' Radiobutton - - - - - - - - - - - - - - - -    pbCancRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[pbTextRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[pbTextRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Add 'Cancel' Pushbutton",    // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbClosRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Add-Close' Radiobutton  - - - - - - - - - - - - - - - -    pbClosRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[pbCancRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[pbCancRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Add 'Close' after Completion",// label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbAutoRB]                 // nextCtrl:  link in next structure
   }, //pbAutoRB
   {  //* 'Auto-Update' Radiobutton  - - - - - - - - - - - - - - -    pbAutoRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[pbClosRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[pbClosRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Create Auto-update Thread",  // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbCellSP]                 // nextCtrl:  link in next structure
   },
   {  //* 'Character-cells' Spinner  - - - - - - - - - - - - - - -    pbCellSP *
      dctSPINNER,                   // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[pbAutoRB].ulY + 2),  // ulY:       upper left corner in Y
      ic[pbAutoRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Number of character ^cells", // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dscData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[pbDivsSP]                 // nextCtrl:  link in next structure
   },      
   {  //* 'Divisions-per-cell' Spinner   - - - - - - - - - - - - -    pbDivsSP *
      dctSPINNER,                   // type:
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[pbCellSP].ulY + 2),  // ulY:       upper left corner in Y
      ic[pbCellSP].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Divisions per step",        // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsdData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[pbAttrDD]                 // nextCtrl:  link in next structure
   },      
   { //* 'Color-Attribute' DropDown Box - - - - - - - - - - - - - -   pbAttrDD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[pbDivsSP].ulY + 3),  // ulY:
      ic[pbDivsSP].ulX,             // ulX:       upper left corner in X
      9,                            // lines:     control lines
      ddWIDTH + 2,                  // cols:      control columns
      (char*)(ddText),              // dispText:  text-data array
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Color Attribute",            // label:     label text
      3,                            // labY:      label offset
      ZERO,                         // labX       
      ddBoxUP,                      // exType:    expansion type
      ddITEMS,                      // scrItems:  number of elements data arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      ddAttr,                       // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbLaunPB]                 // nextCtrl:  link in next structure
   },
   {  //* 'LAUNCH' pushbutton - - - - - - - - - - - - - - - - - - -   pbLaunPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[pbDonePB].ulY,             // ulY:       upper left corner in Y
      short(ic[pbDonePB].ulX - 12), // ulX:       upper left corner in X
      1,                            // lines:     control lines
      10,                           // cols:      control columns
      "  L^AUNCH  ",                // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;



//*************************
//*     ~ProgbarTest      *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

ProgbarTest::~ProgbarTest ( void )
{

   if ( this->pBar != NULL )        // if Progbar open, close it
      delete this->pBar ;
   if ( this->dp != NULL )          // close the window
      delete this->dp ;

   pbcuPtr = NULL ;                 // neutralize our callback pointers
   pbarPtr = NULL ;

}  //* End ~ProgbarTest() 

//*************************
//*      ProgbarTest      *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

ProgbarTest::ProgbarTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows  = tLines ;
   this->termCols  = tCols ;
   this->ulY       = minY + 1 ;
   this->ulX       = 5 ;
   this->dp        = NULL ;
   this->dColor    = nc.gybl | ncbATTR ;
   this->bColor    = this->dColor ;
   this->pbOpen    = false ;
   this->pBar      = NULL ;
   this->pbarOpen  = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   ic[pbDonePB].nColor = ic[pbLaunPB].nColor = nc.gyR ;
   ic[pbDonePB].fColor = nc.gyre | ncbATTR ;
   ic[pbLaunPB].fColor = nc.maR | ncbATTR ;
   ic[pbHoriRB].nColor = ic[pbTextRB].nColor = ic[pbBrdrRB].nColor = 
   ic[pbCancRB].nColor = ic[pbClosRB].nColor = ic[pbAutoRB].nColor = nc.blR ;
   ic[pbHoriRB].fColor = ic[pbTextRB].fColor = ic[pbBrdrRB].fColor =
   ic[pbCancRB].fColor = ic[pbClosRB].fColor = ic[pbAutoRB].fColor = nc.rebl | ncbATTR ;
   ic[pbCellSP].nColor = ic[pbDivsSP].nColor = nc.bw ;
   ic[pbCellSP].fColor = ic[pbDivsSP].fColor = nc.gyre | ncbATTR ;
   dscData.indiAttr = dsdData.indiAttr = nc.grR ;
   ic[pbAttrDD].nColor = nc.gyR ;
   ic[pbAttrDD].fColor = nc.regy ;
   ddAttr[0] = nc.rebk ;            barAttr[0] = nc.br ;    // Red on Black @ brown
   ddAttr[1] = nc.brcy ;            barAttr[1] = nc.bw ;    // Yellow on Cyan @ black
   ddAttr[2] = nc.bkma & ~ncbATTR ; barAttr[2] = nc.bl ;    // Black on Magenta @ blue
   ddAttr[3] = nc.mabk ;            barAttr[3] = nc.brbl ;  // Magenta on Black @ yellow
   ddAttr[4] = nc.blgr & ~ncbATTR ; barAttr[4] = nc.bl ;    // Blue on Green @ blue
   ddAttr[5] = nc.grbl ;            barAttr[5] = nc.gr ;    // Green on Blue @ green
   ddAttr[6] = nc.grbr ;            barAttr[6] = nc.bwR ;   // Green on Brown @ white
   ddAttr[7] = nc.bl   ;            barAttr[7] = nc.brbk & ~ncbATTR ; // Blue on White @ brown

   if ( (this->pbOpen = this->pbOpenDialog ()) != false )
   {

   }  // OpenWindow()

}  //* End ProgbarTest() 

//*************************
//*     pbOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************
//* Note that the compiler will not allow us to use a member method as the     *
//* callback target (unless we use the -fpermissive flag), so we must give the *
//* non-member method access to our dialog. Although this is a dangerous       *
//* thing, our target method promises to behave itself :-)                     *
//******************************************************************************

bool ProgbarTest::pbOpenDialog ( void )
{
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       this->ulY,      // Y offset from upper-left of terminal 
                       this->ulX,      // X offset from upper-left of terminal 
                       "  Progress Bar Demo  ", // dialog title
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Print dialog window's static text *
      winPos wpMsg( 6, 1 ) ;
      this->dp->DrawBox ( wpMsg.ypos++, wpMsg.xpos++, 15, 27, nc.gybl ) ;
      wpMsg = this->dp->WriteParagraph ( wpMsg,
                                       "  Select configuration   \n"
                                       "settings, then launch the\n"
                                       "    the Progress Bar.    \n", nc.brbl ) ;
      wpMsg = this->dp->WriteParagraph ( wpMsg,
                                       " - - - - - - - - - - - - \n"
                                       "Mouse support is enabled.\n"
                                       "When Auto-update callback\n"
                                       "method not specified, use\n"
                                       "the mouse ScrollWheel to \n"
                                       "increase or decrease the \n"
                                       "indicator bar. When the  \n"
                                       "callback is active, bar  \n"
                                       "update is automatic.     \n",
                                       this->dColor ) ;

      this->dp->RefreshWin () ;  // make everything visible

      //* Establish a callback method.                *
      pbcuPtr = this->dp ;
      pbarPtr = NULL ;
      this->dp->EstablishCallback ( &pbControlUpdate ) ;

      success = true ;
   }
   return success ;

}  //* End pbOpenDialog() *

//*************************
//*    pbDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool ProgbarTest::pbDialogOpened ( void )
{
   return this->pbOpen ;

}  //* End pbDialogOpened() *

//*************************
//*      pbInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ProgbarTest::pbInteract ( void )
{
   if ( this->pbOpen )
   {
      //* Enable Mouse Support *
      this->dp->meEnableStableMouse ( 200 ) ;

      //* Track user's selections *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control

      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == pbDonePB )
               {
                  if ( this->pBar != NULL )  // close the progress bar
                  {
                     pbarPtr = NULL ;        // disable the callback method's pointer
                     delete this->pBar ;     // close the Progbar widget
                     this->pbarOpen = false ;// and so indicate
                     this->pBar = NULL ;     // prevent mistakes
                  }
                  done = true ;
               }
               else if ( Info.ctrlIndex == pbLaunPB )
               {
                  //* Gather our parameters and launch the Progbar.*
                  this->pbLaunchProgbar () ;
               }
            }
         }
         //*******************************************
         //* If focus is currently on a Radio button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = this->dp->EditRadiobutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               if ( Info.isSel && 
                    (Info.ctrlIndex == pbCancRB || Info.ctrlIndex == pbClosRB) )
               {  //* One or both optional Pushbuttons actions has been      *
                  //* specified. These require the secondary-thread callback.*
                  this->dp->SetRadiobuttonState ( pbAutoRB, true ) ;
                  //* If 'Close' selected, it requires 'Cancel' *
                  if ( Info.ctrlIndex == pbClosRB )
                     this->dp->SetRadiobuttonState ( pbCancRB, true ) ;
               }
               if ( ! Info.isSel && Info.ctrlIndex == pbCancRB )
               {  //* 'Close' cannot exist on its own *
                  this->dp->SetRadiobuttonState ( pbClosRB, false ) ;
               }
               if ( ! Info.isSel && Info.ctrlIndex == pbAutoRB )
               {  //* If auto-update is disabled, also disable *
                  //* the optional Pushbuttons.                *
                  this->dp->SetRadiobuttonState ( pbCancRB, false ) ;
                  this->dp->SetRadiobuttonState ( pbClosRB, false ) ;
               }
            }
         }
         //***********************************************
         //* If focus is currently on a Dropdown control *
         //***********************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {  //* If we arrived here via hotkey, then leave the data intact *
            //* as a signal that control should expand immediately.       *
            if ( Info.viaHotkey != false )
               ; /* do nothing */
            icIndex = this->dp->EditDropdown ( Info ) ;

            if ( this->pBar != NULL )  // Dropdown refreshes dialog when closed
               this->pBar->refresh() ; // so refresh the Progbar afterward
         }
         //**********************************************
         //* If focus is currently on a Spinner control *
         //**********************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            if ( Info.viaHotkey != false )
               Info.viaHotkey = false ;
            icIndex = this->dp->EditSpinner ( Info ) ;
         }
         else
         { /* no other control types defined for this method */ }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while()

      dp->meDisableMouse();
   }
   else
      { /* Caller is an idiot */ }

}  //* End pbInteract() *

//*************************
//*    pbLaunchProgbar    *
//*************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* a) If a Progbar object is already open, close it.                          *
//* b) Gather the user's settings from our user-interface controls.            *
//* c) Initialize the pbarInit structure.                                      *
//* d) Launch the Progbar widget.                                              *
//* e) If the Progbar was created with user-interface controls, then           *
//*    call 'UserInteract' method.                                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ProgbarTest::pbLaunchProgbar ( void )
{
   //* If we already have an instance open, close it.*
   if ( this->pBar != NULL )
   {
      pbarPtr = NULL ;           // disable the callback method's pointer
      delete this->pBar ;        // close the Progbar widget
      this->pbarOpen = false ;   // and so indicate
      this->pBar = NULL ;        // prevent mistakes
      this->pbClearBarArea () ;  // restore the display area
   }

   //* Gather our parameters and launch the Progbar.*
   int cellCount, divCount ;
   short colorIndex ;
   bool h,     // 'true' == horizontal layout
        t,     // 'true' == text labels
        b,     // 'true' == enable border
        canb,  // 'true; == enable 'Cancel' button
        clob,  // 'true' == enable 'Close' button
        a ;    // 'true; == enable Auto-update callback method

   this->dp->GetRadiobuttonState ( pbHoriRB, h ) ;
   this->dp->GetRadiobuttonState ( pbTextRB, t ) ;
   this->dp->GetRadiobuttonState ( pbBrdrRB, b ) ;
   this->dp->GetRadiobuttonState ( pbCancRB, canb ) ;
   this->dp->GetRadiobuttonState ( pbClosRB, clob ) ;
   this->dp->GetRadiobuttonState ( pbAutoRB, a ) ;
   this->dp->GetSpinnerValue ( pbCellSP, cellCount ) ;
   this->dp->GetSpinnerValue ( pbDivsSP, divCount ) ;
   colorIndex = this->dp->GetDropdownSelect ( pbAttrDD ) ;
   if ( ! h )
   {  //* Limit dimension of vertical Progbar to dialog size *
      if ( ! b && cellCount > 22 )            cellCount = 22 ;
      else if ( b && cellCount > 20 )         cellCount = 20 ;
      if ( t && b && cellCount > 18 )         cellCount = 18 ;
      else if ( t && ! b && cellCount > 20 )  cellCount = 20 ;
   }
   pbarInit pbi( ddAttr[colorIndex], barAttr[colorIndex], cellCount, h ) ;
   pbi.steps = (cellCount * cellDIV) / divCount ;
   pbi.border = b ;
   pbi.enableCancel = canb ;
   pbi.enableClose = clob ;
   if ( a )
   {  //* Auto-update enabled, specify callback method *
      pbi.threadCb = &pbRemoteUpdate ;
   }
   if ( t != false )
   {
      pbi.beginText = h ? "0%" : "  0%" ;
      pbi.endText   = "100%" ;
      pbi.titleText = " Progbar " ;
   }
   if ( h != false )
   {
      pbi.yOffset  = this->ulY + 1 ;
      pbi.xOffset  = this->ulX + 
                     (dialogCOLS / 2 - cellCount / 2 - 1) ;
      if ( pbi.xOffset < (this->ulX + 1) )
         pbi.xOffset = this->ulX + 1 ;
   }
   else
   {
      pbi.yOffset  = this->ulY + 1 ;
      pbi.xOffset  = this->ulX + (dialogCOLS - 4) ; // default pos: border only
      if ( !t && !b )                     pbi.xOffset += 1 ;
      else if ( t && b )                  pbi.xOffset -= 3 ;
      else if ( !b || (b && !t && canb))  pbi.xOffset -= 1 ;
   }
   // Programmer's Note: "Total Steps" report is not reliable 
   // because additional steps may be added to fill the bar.
   gString gs( "  Total Steps: %hd  ", &pbi.steps ) ;
   winPos stepMsg( short(ic[pbDivsSP].ulY + 1), short(ic[pbDivsSP].ulX + 7) ) ;
   this->dp->ClearLine ( stepMsg.ypos, false, stepMsg.xpos ) ;
   this->dp->WriteString ( stepMsg, gs, nc.grR, true ) ;

   #if 0    // DEBUG ONLY - SEE 'DEBUG_PBAR' FLAG IN Progbar.hpp
   pbi.pdp = this->dp ;
   #endif   // DEBUG ONLY

   this->pBar = new Progbar ( pbi ) ;
   if ( this->pBar != NULL )
   {
      if ( (this->pbarOpen = this->pBar->open()) )
      {
         pbarPtr = this->pBar ;
         if ( pbi.enableCancel || pbi.enableClose )
         {  //* Call the direct user-interaction method. *
            //* Returns when operation complete or when  *
            //* user aborts.                             *
            short status = this->pBar->UserInteract () ;
            if ( status == ERR )
               this->dp->UserAlert();
            chrono::duration<short, std::milli>aMoment( 800 ) ;
            this_thread::sleep_for( aMoment ) ;
            pbarPtr = NULL ;           // disable callback access
            delete this->pBar ;        // close the Progbar dialog
            this->pbarOpen = false ;   // and so indicate
            this->pBar = NULL ;        // prevent misteaks
            this->pbClearBarArea () ;  // restore the display area
         }
      }
   }
}  //* End pbLaunchProgbar() *

//*************************
//*    pbClearBarArea     *
//*************************
//******************************************************************************
//* PRIVATE METHOD.                                                            *
//* When a Progbar object is closed AND when activity has potentially taken    *
//* place in the parent dialog, we must redraw the area the Progbar previously *
//* occupied.                                                                  *
//*                                                                            *
//* Notes:                                                                     *
//* a) If the primary thread stayed with the Progbar object, OR if all         *
//*    controls in the dialog are disabled, then we could have called          *
//*    SetDialogObscured() before opening the Progbar, then called             *
//*    RefreshWin() after it closed.                                           *
//* b) We could instead have saved and image of the parent dialog to a file    *
//*    and restored it after the Progbar closed.                               *
//* c) For this test dialog, however, this is the more direct way to restore   *
//*    the display area because we 'know' where the Progbar may have been,     *
//*    so we redraw that area.                                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void ProgbarTest::pbClearBarArea ( void )
{
   //* For horizontal Progbar objects *
   this->dp->ClearArea ( 1, 1, 4, (pbCOLS - 2) ) ;

   //* For vertical Progbar objects *
   this->dp->ClearArea ( 1, (pbCOLS - 7), (pbROWS - 2), 6 ) ;

}  //* End pbClearBarArea() *

//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - Non-member Methods- - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*    pbRemoteUpdate     *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* This is a callback method used by the secondary thread which is optionally *
//* launched from within the Progbar object in order to monitor an activity's  *
//* progress and report on it.                                                 *
//*                                                                            *
//* In a real-world application, this method would monitor the progress of the *
//* activity, (number of megabytes downloaded, level of bandwidth usage for)   *
//* example and return the number of steps that the Progbar object should use  *
//* to update its display.                                                     *
//*                                                                            *
//* However, for this test application, there is no actual activity to monitor,*
//* so we simply return a single, positive step each time we are called.       *
//*                                                                            *
//*                                                                            *
//* Input  : stepsRemaing : This is the number of step remaining to completely *
//*                         fill the Progress Bar.                             *
//*                                                                            *
//* Returns: increment/decrement count                                         *
//*          returning ZERO indicates no progress                              *
//******************************************************************************

static short pbRemoteUpdate ( short stepsRemaining )
{
   short increment = 1 ;

   //* Poll the status of the activity being monitored, and select *
   //* a corresponding increment/decrement value.                  *
   {
      /* Do Stuff Here... */
   }

   return increment ;

}  //* End pbRemoteUpdate() *

//*************************
//*    pbControlUpdate    *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD:                                                         *
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. We track the (unconverted) mouse ScrollWheel events to update the     *
//*      Progress Bar.                                                         *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

static short pbControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime )
{
   if ( !firstTime  )
   { /* nothing to do */ }

   if ( pbarPtr != NULL )
   {
      if ( wkey.type == wktMOUSE )
      {
         if ( wkey.mevent.meType == metSW_U )
            pbarPtr->update() ;
         else if ( wkey.mevent.meType == metSW_D )
            pbarPtr->update( -1 ) ;
      }
   }

   return OK ;

}  //* End pbControlUpdate() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

