//******************************************************************************
//* File       : DMouseTest.cpp                                                *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2013-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 03-Jan.2020                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the DMouseTest class.                    *
//*              This class exercises the NcDialog-class mouse methods.        *
//*              It is instantiated by the Dialog4 application, Test04.        *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.8.0)                                     *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE              *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.03 01-Jan-2020                                                     *
//*    -- Output formatting of the mouse record in the Billboard control       *
//*       modified due to a change in the definition of "mmask_t" in the       *
//*       underlying ncursesw library from 'long in' to 'int'. This may be an  *
//*       artifact of the port from 32-bit to 64-bit platforms.                *
//*       In any case, "mmask_t" is a 4-byte int.                              *
//*    -- Due to a new bug (feature?) in the underlying ncurses library        *
//*       (on or prior to v:6.1.20180923), the "REPORT_MOUSE_POSITION" flag    *
//*       of the 'bstate' field is no longer reported in response to a         *
//*       scroll-wheel-up event. (The position flag is sporatically reported   *
//*       during a scroll-wheel button-up event.)                              *
//*       -- Instead, the scroll-wheel-scroll-up event is now reported through *
//*         "BUTTON5_PRESSED". This is actually a better solution, but spent   *
//*         a whole afternoon chasing down the change in functionality, and    *
//*         all of the next morning updating our code and testing the changes. *
//*         We hates meeces to pieces!                                         *
//*       -- In addition, scroll-wheel events which include the CTRL/ALT/SHIFT *
//*          modifier keys no longer report the modifier flags. This may be    *
//*          due to the Wayland mouse driver and not the ncursesw mouse code.  *
//*       NcDialog library has been updated to reflect these changes.          *
//*                                                                            *
//* v: 0.00.02 21-Jun-2015 Enhance tests to exercise mouse-event capture and   *
//*                        translation, especially ScrollWheel events.         *
//*                                                                            *
//* v: 0.00.01 03-Jun-2013 Layout based on ColorTest-class test in Dialog2.    *
//*                                                                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "DMouseTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = ndmROWS ; // display lines
static const short dialogCOLS = ndmCOLS ; // display columns

static dspinData dsData ;                 // data for the dmDelSP spinner control

//* Display data and color attributes for event-selector control.*
static const short SelectBITS = 32, SelectWIDTH = 18 ;
static char     selectText[SelectBITS][SelectWIDTH] ;
static char*    selectPntr[SelectBITS] ;
static attr_t   selectAttr[SelectBITS] ;
static attr_t   selectColor, deselectColor, noselectColor ;
attr_t   monoColor[2] = { attrDFLT, nc.cy } ;
attr_t   menuColor[2] = { attrDFLT, nc.grR } ;
static short    lastSelectable = NCURSES_MOUSE_VERSION <= 1 ? 23 : 24 ;
static ssetData setSelect( (const char**)selectPntr, selectAttr, 
                            SelectBITS, ZERO, false ) ;

////* Data for the Generic dctBILLBOARD control *
//static const char* gbbData = 
//" 我的静的花园\n"        // "In my quiet garden"
//" 鱼是很有礼貌，\n"       // "the fish are so polite,"
//" 他们很少说话\n"        // "they seldom speak"
//" 打岔我的梦想。" ;       // "to interrupt my reverie."

//* Data for the Generic dctSCROLLBOX control *
static const short scrbITEMS = 3, scrbWIDTH = 26 ;
static const char scrbData[scrbITEMS][scrbWIDTH + 5] = 
{
   " Upper Left 上左侧      ",   // "上 左侧"   // upper part - left-hand side
   " Centered 居中图像      ",    // "居中图像"   //centered image
   " Lower Right下层的右边的",      // "下层的 右边的"// bottom - right-hand side 
} ;

//* Data for the Generic dctDROPDOWN control *
static const short gddITEMS = 17, gddWIDTH = 26 ;
static char gddData[gddITEMS][gddWIDTH] = 
{
   "00 Gentle             00",
   "01  Green             01",
   "02   Grasses          02",
   "03     Grow and       03",
   "04      Giggle in     04",
   "05       Garrolous    05",
   "06        Groups      06",
   "07      - while -     07",
   "08          suoitnuoB 08",
   "09             nworB  09",
   "10            sraeB   10",
   "11          elbmuB    11",
   "12     ylehtilB       12",
   "13       tuoBa        13",
   "14    gniraeB         14",
   "15  deteksaB          15",
   "16  seirreB           16",
} ;

//* Data for independent Menuwin dmIndiMW *
static const short indiITEMS = 3, indiWIDTH = 28 ;
static const char indiData[indiITEMS][indiWIDTH-1] = 
{
   "aaaaaaaaaaaaaaaaaaaaaaaaaa",
   "bbbbbbbbbbbbbbbbbbbbbbbbbb",
   "cccccccccccccccccccccccccc",
} ;

//* Data for Menuwin Group, dmGrp1MW, dmGrp2MW, dmGrp3MW *
static const short mbITEMS = 5, mbWIDTH = 18 ;
static const char mb1Data[mbITEMS][mbWIDTH-1] = 
{
   " Aardvark       ",
   " Bonobo         ",
   " Chinchilla     ",
   " Dung Beetle    ",
   " Elephant       "
} ;
static const char mb2Data[mbITEMS][mbWIDTH-1] = 
{
   " Avocado        ",
   " Broccoli       ",
   " Cauliflower    ",
   " Dandilion      ",
   " Eggplant       "
} ;
static const char mb3Data[mbITEMS][mbWIDTH-1] = 
{
   " Amethyst       ",
   " Beryl         >",
   " Carnelian      ",
   " Dolomite       ",
   " Emerald        "
} ;
static const short mbsITEMS = 3, mbsWIDTH = 14 ;
static const char mb3SubData[mbsITEMS][mbsWIDTH - 1] = 
{
   " Aquamarine ",
   " Heliodor   ",
   " Morganite  ",
} ;
//* Data for invisible (context) Menuwin, dmConMW *
static const short ctITEMS = 3, ctWIDTH = 18 ;
static const char ctData[ctITEMS][ctWIDTH-1] = 
{
   " Leap About (23)",
   " Sing and Shout ",
   "  Work It Out!  "
} ;


static InitCtrl ic[dmControlsDEFINED] =   // control initialization structures
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -     dmDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 3),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  ^DONE  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmOutBB],                 // nextCtrl:  link in next structure
   },
   {  //* 'OUTPUT' Billboard - - - - - - - - - - - - - - - - - - -     dmOutBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      short(dialogCOLS - 72),       // ulX:       upper left corner in X
      12,                           // lines:     control lines
      71,                           // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.gybr | ncbATTR,            // nColor:    non-focus color
      nc.gybr | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Mouse Events",               // label:     
      short(ic[dmOutBB].lines),     // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[dmMaskSE],                // nextCtrl:  link in next structure
   },
   {  //* 'MASK scroll-ext control - - - - - - - - - - - - - - - - - dmMaskSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ic[dmOutBB].ulY + ic[dmOutBB].lines),// ulY:       
      short(dialogCOLS - 43),       // ulX:       upper left corner in X
      8,                            // lines:     control lines
      short(SelectWIDTH + 1),       // cols:      control columns
      NULL,                         // dispText:  n/a
      nc.gybl,                      // nColor:    non-focus border color
      nc.rebl,                      // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Event Mask Select",          // label:     
      8,                            // labY:      offset from control's ulY
      1,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmMaskTB],                // nextCtrl:  link in next structure
   },
   {  //* Mask-display text box (non-active) - - - - - - - - - -      dmMaskTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmMaskSE].ulY + 10), // ulY:       upper left corner in Y
      ic[dmMaskSE].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      41,                           // cols:      control columns
      NULL,                         // dispText:
      nc.gr,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ic[dmFilterRB],              // nextCtrl:  link in next structure
   },
   {  //* 'FILTER' radio button - - - - - - - - - - - - - - - - -   dmFilterRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[dmMaskSE].ulY + 1),  // ulY:       upper left corner in Y
      short(ic[dmMaskSE].ulX + 20), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Event Filter",               // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmWheelRB],                // nextCtrl:  link in next structure
   },
   {  //* 'ScrollWheel' radio button  - - - - - - - - - - - - - -    dmWheelRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[dmFilterRB].ulY + 1),// ulY:       upper left corner in Y
      ic[dmFilterRB].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Scroll Wheel",               // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmAutoRB],                // nextCtrl:  link in next structure
   },
   {  //* 'AUTO-CONVERT' radio button - - - - - - - - - - - - - -     dmAutoRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[dmWheelRB].ulY + 1), // ulY:       upper left corner in Y
      ic[dmWheelRB].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Auto Convert",               // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmStableRB],              // nextCtrl:  link in next structure
   },
   {  //* 'STABLE ENABLE' radio button - - - - - - - - - - - - - -  dmStableRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dmAutoRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[dmAutoRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Stable Enable",              // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmDelaySP],               // nextCtrl:  link in next structure
   },
   {  //* 'DELAY' spinner - - - - - - - - - - - - - - - - - - - - -  dmDelaySP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmStableRB].ulY + 2),// ulY:       upper left corner in Y
      short(ic[dmStableRB].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Click Delay",                // label:     
      ZERO,                         // labY:      
      9,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[dmClearPB],               // nextCtrl:  link in next structure
   },
   {  //* 'CLEAR' pushbutton  - - - - - - - - - - - - - - - - - -    dmClearPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmDelaySP].ulY + 2), // ulY:       upper left corner in Y
      ic[dmDelaySP].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      15,                           // cols:      control columns
      "  CLEAR INPUT  ",            // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.maG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmShowPB],                // nextCtrl:  link in next structure
   },
   {  //* 'ShowHide' pushbutton - - - - - - - - - - - - - - - - -     dmShowPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      28,                           // cols:      control columns
      " SHOW/HIDE CONTEXT MENU(10) ",// dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmIndiRB],                // nextCtrl:  link in next structure
   },
   {  //* 'Independent' radio button  - - - - - - - - - - - - - -     dmIndiRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dmShowPB].ulY + 2),  // ulY:       upper left corner in Y
      ic[dmShowPB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Independent Radio (11)",     // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGrp1RB],                // nextCtrl:  link in next structure
   },
   {  //* 'Group' radio button #1   - - - - - - - - - - - - - - -     dmGrp1RB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dmIndiRB].ulY + 2),  // ulY:       upper left corner in Y
      ic[dmIndiRB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Radio Button Group #1 (12)", // label:
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGrp2RB],                // nextCtrl:  link in next structure
   },
   {  //* 'Group' radio button #2   - - - - - - - - - - - - - - -     dmGrp2RB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[dmGrp1RB].ulY + 1),  // ulY:       upper left corner in Y
      ic[dmGrp1RB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Radio Button Group #2 (13)", // label:
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGrp3RB],                // nextCtrl:  link in next structure
   },
   {  //* 'Group' radio button #3   - - - - - - - - - - - - - - -     dmGrp3RB *
      dctRADIOBUTTON,               // type:      
      rbtC5,                        // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dmGrp2RB].ulY + 1),  // ulY:       upper left corner in Y
      ic[dmGrp2RB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "< # >",                      // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Radio Button Group #3 (14)", // label:
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGenTB],                 // nextCtrl:  link in next structure
   },
   {  //* 'GENERIC' text box   - - - - - - - - - - - - - - - - - -     dmGenTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmGrp3RB].ulY + 2),  // ulY:       upper left corner in Y
      ic[dmGrp3RB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      " (some text)",               // dispText:
      nc.bw,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Generic Text Box (15)",      // label:     
      ZERO,                         // labY:      
      21,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGenDD],                 // nextCtrl:  link in next structure
   },
#if 0    // OBSOLETE - WE HAD TO DISABLE THIS BECAUSE WE MAXED OUT ON CONTROLS
   {  //* 'GENERIC' Billboard  - - - - - - - - - - - - - - - - - -     dmGenBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmGenTB].ulY + 2),   // ulY:       upper left corner in Y
      short(ic[dmGenTB].ulX + 3),   // ulX:       upper left corner in X
      4,                            // lines:     control lines
      15,                           // cols:      control columns
      gbbData,                      // dispText:  initial display text
      nc.gybr | ncbATTR,            // nColor:    non-focus color
      nc.gybr | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Generic Billboard (16)",     // label:     
      ZERO,                         // labY:      
      short(ic[dmGenBB].cols + 3),  // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      true,                         // active:    view-only
      &ic[dmGenDD],                 // nextCtrl:  link in next structure
   },
#endif   // OBSOLETE - WE HAD TO DISABLE THIS BECAUSE WE MAXED OUT ON CONTROLS
   { //* 'GENERIC' DropDown Box - - - - - - - - - - - - - - - - - -    dmGenDD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmGenTB].ulY + ic[dmGenTB].lines + 2),// ulY:
      ic[dmGenTB].ulX,              // ulX:       upper left corner in X
      9,                            // lines:     control lines
      gddWIDTH,                     // cols:      control columns
      (const char*)&gddData,        // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Generic DropDown Box (16)",  // label:     label text
      -1,                           // labY:      label offset
      1,                            // labX       
      ddBoxDOWN,                    // exType:    expansion type
      gddITEMS,                     // scrItems:  number of elements data arrays
      7,                            // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGenSB],                 // nextCtrl:  link in next structure
   },
   { //* 'GENERIC' scroll box - - - - - - - - - - - - - - - -          dmGenSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmGenDD].ulY + 4),   // ulY:       upper left corner in Y
      ic[dmGenDD].ulX,              // ulX:       upper left corner in X
      short(scrbITEMS + 2),         // lines:     control lines
      short(scrbWIDTH),             // cols:      control columns
      (const char*)&scrbData,       // dispText:  text-data array
      nc.gybl,                      // nColor:    non-focus border color
      nc.rebl,                      // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Generic ScrollBox(17)",          // label:     label text
      ZERO,                         // labY:      label offset
      ZERO,                         // labX
      ddBoxTYPES,                   // exType:    (n/a)
      scrbITEMS,                    // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmIndiMW],                // nextCtrl:  link in next structure
   },
   {  //* 'Independent' Menuwin   - - - - - - - - - - - - - - - - -   dmIndiMW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmGenSB].ulY + 11),  // ulY:       upper left corner in Y
      ic[dmGenSB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      indiWIDTH,                    // cols:      control columns, 'expanded' state
      (const char*)&indiData,       // dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      "| Independent Menuwin (18) |", // label:
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      indiITEMS,                    // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  array of menu item colors
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGrp1MW],                // nextCtrl:  link in next structure
   },
   {  //* 'Menuwin Group 1' - - - - - - - - - - - - - - - - - - - -   dmGrp1MW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      21,                           // ulY:       upper left corner in Y
      31,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mbWIDTH,                      // cols:      control columns, 'expanded' state
      (const char*)&mb1Data,        // dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      "| Animal   (19)",            // label:     label text == 'collapsed state' text
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbITEMS,                      // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      menuColor,                    // scrColor:  array of menu item colors
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGrp2MW],                // nextCtrl:  link in next structure
   },
   {  //* 'Menuwin Group 2' - - - - - - - - - - - - - - - - - - - -   dmGrp2MW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[dmGrp1MW].ulY,             // ulY:       upper left corner in Y
      short(ic[dmGrp1MW].ulX + 15), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mbWIDTH,                      // cols:      control columns, 'expanded' state
      (const char*)&mb2Data,        // dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      "|Vegetable (20)",            // label:     label text == 'collapsed state' text
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbITEMS,                      // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      menuColor,                    // scrColor:  array of menu item colors
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmGrp3MW],                // nextCtrl:  link in next structure
   },
   {  //* 'Menuwin Group 3' - - - - - - - - - - - - - - - - - - - -   dmGrp3MW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[dmGrp2MW].ulY,             // ulY:       upper left corner in Y
      short(ic[dmGrp2MW].ulX + 15), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mbWIDTH,                      // cols:      control columns, 'expanded' state
      (const char*)&mb3Data,        // dispText:  array of menu items
      nc.gr,         // nColor: non-focus border color / focus 'collapsed' color
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      "|Mineral  (21)|",               // label:     label text == 'collapsed state' text
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbITEMS,                      // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      menuColor,                    // scrColor:  array of menu item colors
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dmSub3MW],                // nextCtrl:  link in next structure
   },
   {  //* Menuwin (sub-menu, invisible when collapsed)  - - - - - -   dmSub3MW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmGrp3MW].ulY + 3),  // ulY:       upper left corner in Y
      short(ic[dmGrp3MW].ulX + ic[dmGrp3MW].cols - 1),// ulX:
      1,                            // lines:     (n/a)
      mbsWIDTH,                     // cols:      control columns, 'expanded' state
      (const char*)&mb3SubData,     // dispText:  array of menu items
      nc.gr,         // nColor: (n/a) context menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     hidden menus have no label
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbsITEMS,                     // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      menuColor,                    // scrColor:  array of color attributes for menu items
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inactive
      &ic[dmConMW],                 // nextCtrl:  link in next structure
   },
   {  //* Menuwin (context menu, invisible when collapsed)  - - - -    dmConMW *
      dctMENUWIN,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dmShowPB].ulY - 1),  // ulY:       upper left corner in Y
      short(ic[dmShowPB].ulX + 28), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      ctWIDTH,                      // cols:      control columns, 'expanded' state
      (const char*)&ctData,         // dispText:  array of menu items
      nc.gr,         // nColor: (n/a) context menus are invisible when collapsed
      nc.grR,        // fColor: focus border color / non-focus 'collapsed' color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     hidden menus have no label
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      ctITEMS,                      // scrItems:  number of menu items
      ZERO,                         // scrSel:    (n/a)
      menuColor,                    // scrColor:  array of color attributes for menu items
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inactive
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Container for key-type label lookup *
   class tLab
   {
      public:
      short       type ;
      const char* label ;
   } ;

   //* Container for mouse-event-type label lookup *
   class meLab
   {
      public:
      ULONG       type ;
      const char* label ;
   } ;

   //* Types of key input *
   const short tlCount = 6 ;
   tLab typeLabels[tlCount] = 
   {
      { wktPRINT,  "wktPRINT " },
      { wktFUNKEY, "wktFUNKEY" },
      { wktMOUSE,  "wktMOUSE " },
      { wktEXTEND, "wktEXTEND" },
      { wktESCSEQ, "wktESCSEQ" },
      { wktERR,    "wktERR   " },
   } ;
   //* Mouse-event types *
   meLab meLabels[32] = 
   {
      { ZERO,                    "none:" },
      #if NCURSES_MOUSE_VERSION <= 1
      { BUTTON1_RELEASED,        "B1rel" },
      { BUTTON1_PRESSED,         "B1pre" },
      { BUTTON1_CLICKED,         "B1cl1" },
      { BUTTON1_DOUBLE_CLICKED,  "B1cl2" },
      { BUTTON1_TRIPLE_CLICKED,  "B1cl3" },
      { BUTTON1_RESERVED_EVENT,  "B1res" },

      { BUTTON2_RELEASED,        "B2rel" },
      { BUTTON2_PRESSED,         "B2pre" },
      { BUTTON2_CLICKED,         "B2cl1" },
      { BUTTON2_DOUBLE_CLICKED,  "B2cl2" },
      { BUTTON2_TRIPLE_CLICKED,  "B2cl3" },
      { BUTTON2_RESERVED_EVENT,  "B2res" },
                                 
      { BUTTON3_RELEASED,        "B3rel" },
      { BUTTON3_PRESSED,         "B3pre" },
      { BUTTON3_CLICKED,         "B3cl1" },
      { BUTTON3_DOUBLE_CLICKED,  "B3cl2" },
      { BUTTON3_TRIPLE_CLICKED,  "B3cl3" },
      { BUTTON3_RESERVED_EVENT,  "B3res" },
                                 
      { BUTTON4_RELEASED,        "B4rel" },
      { BUTTON4_PRESSED,         "B4pre" },
      { BUTTON4_CLICKED,         "B4cl1" },
      { BUTTON4_DOUBLE_CLICKED,  "B4cl2" },
      { BUTTON4_TRIPLE_CLICKED,  "B4cl3" },
      { BUTTON4_RESERVED_EVENT,  "B4res" },
      #else    // NCURSES_MOUSE_VERSION > 1
      { BUTTON1_RELEASED,        "B1rel" },
      { BUTTON1_PRESSED,         "B1pre" },
      { BUTTON1_CLICKED,         "B1cl1" },
      { BUTTON1_DOUBLE_CLICKED,  "B1cl2" },
      { BUTTON1_TRIPLE_CLICKED,  "B1cl3" },

      { BUTTON2_RELEASED,        "B2rel" },
      { BUTTON2_PRESSED,         "B2pre" },
      { BUTTON2_CLICKED,         "B2cl1" },
      { BUTTON2_DOUBLE_CLICKED,  "B2cl2" },
      { BUTTON2_TRIPLE_CLICKED,  "B2cl3" },
                                 
      { BUTTON3_RELEASED,        "B3rel" },
      { BUTTON3_PRESSED,         "B3pre" },
      { BUTTON3_CLICKED,         "B3cl1" },
      { BUTTON3_DOUBLE_CLICKED,  "B3cl2" },
      { BUTTON3_TRIPLE_CLICKED,  "B3cl3" },

      { BUTTON4_RELEASED,        "B4rel" },
      { BUTTON4_PRESSED,         "B4pre" },
      { BUTTON4_CLICKED,         "B4cl1" },
      { BUTTON4_DOUBLE_CLICKED,  "B4cl2" },
      { BUTTON4_TRIPLE_CLICKED,  "B4cl3" },

      { BUTTON5_RELEASED,        "B5rel" },
      { BUTTON5_PRESSED,         "B5pre" },
      { BUTTON5_CLICKED,         "B5cl1" },
      { BUTTON5_DOUBLE_CLICKED,  "B5cl2" },
      { BUTTON5_TRIPLE_CLICKED,  "B5cl3" },
      #endif   // NCURSES_MOUSE_VERSION > 1
   } ;

//* Module-scope access to the dialog for the non-member callback method *
static NcDialog* dmcuPtr = NULL ;
static short dmControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime ) ;


//*************************
//*     ~DMouseTest       *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DMouseTest::~DMouseTest ( void )
{

   if ( this->dp != NULL )                // close the window
   {
      //* Because one of the Menuwin controls extends BELOW THE DIALOG   *
      //* WINDOW when expanded, beautify the violated terminal window.   *
      winPos wp = this->dp->GetDialogPosition () ;

      delete ( this->dp ) ;

      wp.ypos += this->dRows ;
      nc.ClearLine ( wp.ypos++ ) ;
      nc.ClearLine ( wp.ypos++ ) ;
      nc.ClearLine ( wp.ypos++ ) ;
   }

}  //* End ~DMouseTest() 

//*************************
//*      DMouseTest       *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************
//* Programmer's Note: The maximum number of user-interface controls are       *
//* defined for this dialog. We have disabled the do-nothing dctBILLBOARD to   *
//* make room, but we have left the code in place for reference and future     *
//* testing.                                                                   *
//******************************************************************************

DMouseTest::DMouseTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->dRows    = this->dCols = ZERO ;
   this->minULY   = minY ;
   this->dp       = NULL ;
   this->dColor   = nc.gybl | ncbATTR ;
   this->bColor   = nc.brcy ;
   this->dmOpen   = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   attr_t pnColor = nc.gyR,            // pushbutton non-focus color
          pfColor = nc.gyre | ncbATTR, // pushbutton focus color
          rnColor = this->dColor,      // radio button non-focus color
          rfColor = pfColor ;          // radio button focus color
   ic[dmDonePB].nColor = ic[dmShowPB].nColor = pnColor ;
   ic[dmDonePB].fColor = ic[dmShowPB].fColor = pfColor ;
   ic[dmIndiRB].nColor = ic[dmGrp1RB].nColor = 
   ic[dmGrp2RB].nColor = ic[dmGrp3RB].nColor = rnColor ;
   ic[dmIndiRB].fColor = ic[dmGrp1RB].fColor = 
   ic[dmGrp2RB].fColor = ic[dmGrp3RB].fColor = rfColor ;

   ic[dmOutBB].nColor = ic[dmOutBB].fColor = nc.gybr | ncbATTR ;
   ic[dmMaskSE].nColor   = nc.gybl ;
   ic[dmMaskSE].fColor   = nc.rebl ;
   ic[dmMaskTB].nColor   = nc.gr ;
   ic[dmMaskTB].fColor   = nc.bw ;
   ic[dmFilterRB].nColor = ic[dmWheelRB].nColor  = 
   ic[dmAutoRB].nColor   = ic[dmStableRB].nColor = nc.gr ;
   ic[dmFilterRB].fColor = ic[dmWheelRB].fColor  =
   ic[dmAutoRB].fColor   = ic[dmStableRB].fColor = nc.re ;
   ic[dmDelaySP].nColor  = pnColor ;
   ic[dmDelaySP].fColor  = nc.bw ;
   ic[dmClearPB].nColor  = pnColor ;
   ic[dmClearPB].fColor  = nc.maG ;
   ic[dmGenTB].nColor    = nc.bw ;
   ic[dmGenTB].fColor    = nc.reB ;
//   ic[dmGenBB].nColor    = nc.bw ;
//   ic[dmGenBB].fColor    = nc.reB ;
   ic[dmGenSB].nColor    = this->dColor ;
   ic[dmGenSB].fColor    = nc.rebl ;
   ic[dmGenDD].nColor    = this->dColor ;
   ic[dmGenDD].fColor    = nc.rebl ;
   ic[dmIndiMW].nColor   = nc.cyR ;
   ic[dmIndiMW].fColor   = nc.cy ;
   ic[dmGrp1MW].nColor = ic[dmGrp2MW].nColor = ic[dmGrp3MW].nColor = 
   ic[dmSub3MW].nColor = ic[dmConMW].nColor = nc.gr ;
   ic[dmGrp1MW].fColor = ic[dmGrp2MW].fColor = ic[dmGrp3MW].fColor = 
   ic[dmSub3MW].fColor = ic[dmConMW].fColor = menuColor[1] = nc.grR ;
   monoColor[1] = nc.cy ;

   selectColor   = nc.gygr | ncbATTR ; // for dctSCROLLEXT mask selector data
   deselectColor = nc.gygr ;
   noselectColor = nc.bw ;

   //* Initialize the dctSCROLLEXT control data *
   this->dmInitSE () ;

   //* Initialize the dctSPINNER control data *
   dsData.minValue = ncmiNONE ;
   dsData.maxValue = ncmiMAX ;
   dsData.iniValue = ncmiDFLT ;
   dsData.dsFormat = dspinINTEGER ;
   dsData.indiAttr = nc.grG ;

   if ( (this->dmOpen = this->dmOpenDialog ()) != false )
   {  //* Initialize data for event selector *
      this->dp->SetScrollextText ( dmMaskSE, setSelect ) ;
      this->dmSynch2EMask () ;      // indicate which mask bits are active

   }

}  //* End DMouseTest() 

//*************************
//*     dmOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool DMouseTest::dmOpenDialog ( void )
{
   short ctrY    = this->termRows/2 + 1,     // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) > this->minULY ? 
                   (this->minULY + 1) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  NcDialog-class Mouse Methods  ", // dialog title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Remember the size of the dialog *
      this->dp->GetDialogDimensions ( this->dRows, this->dCols ) ;

      //* Create a MenuBar (a group of DialogMenuwin controls), *
      //* and attach associated sub-menu                        *
      short mwList[] = { dmGrp1MW, dmGrp2MW, dmGrp3MW, -1 } ;
      if ( (this->dp->GroupMenuwinControls ( mwList, nckC_Z )) == OK )
      {
         this->dp->WriteString ( ic[dmGrp1MW].ulY - 1, ic[dmGrp1MW].ulX + 2, 
                             "Menuwin group. Ctrl+z toggles visibility.", 
                             this->dColor ) ;
         short smList[] = { MAX_DIALOG_CONTROLS, dmSub3MW,
                            MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                            MAX_DIALOG_CONTROLS, -1 
                          } ;
         this->dp->AttachMenuwinSubmenus ( dmGrp3MW, smList ) ;
      }

      //* Enable mouse input - (Button 1,2,3 - all clicks, plus ScrollWheel) *
      mmask_t newMask = 
         BUTTON1_CLICKED | BUTTON1_DOUBLE_CLICKED | BUTTON1_TRIPLE_CLICKED |
         BUTTON2_CLICKED | BUTTON2_DOUBLE_CLICKED | BUTTON2_TRIPLE_CLICKED |
         BUTTON3_CLICKED | BUTTON3_DOUBLE_CLICKED | BUTTON3_TRIPLE_CLICKED |
         REPORT_SCROLL_WHEEL ;
      if ( (this->dp->meEnableMouse ( newMask )) == OK )
      {
         //* Set radio button group.                   *
         //* (only one can be selected at any moment). *
         short XorGroup[] = { dmGrp1RB, dmGrp2RB, dmGrp3RB, -1 } ;
         this->dp->GroupRadiobuttons ( XorGroup ) ;
// STATUS REPORT
// DO STUFF HERE
      }
      else
      {
// REPORT INITIALIZATION ERROR HERE
      }
      this->dp->RefreshWin () ;        // make everything visible

      //* Establish a callback method for display of mouse data input.    *
      //* Note that the compiler will not allow us to use a member method *
      //* as the callback target (unless we use the -fpermissive flag),   *
      //* so we must give the non-member method access to our dialog.     *
      //* Although this is a dangerous thing, our target method promises  *
      //* to behave itself :-)                                            *
      dmcuPtr = this->dp ;
      this->dp->EstablishCallback ( &dmControlUpdate ) ;

      success = true ;
   }
   return success ;

}  //* End dmOpenDialog() *

//*************************
//*    dmDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool DMouseTest::dmDialogOpened ( void )
{
   return this->dmOpen ;

}  //* End dmDialogOpened() *

//*************************
//*      dmInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Note that because a control may receive the input focus via a mouse click, *
//* the user interface loop must be aware of 'hotkey' focus change, even if    *
//* the instantiation of the control did not specify a hotkey.                 *
//*                                                                            *
//* 1) Thus, for each control type, we must check whether the input focus      *
//*    arrived at the control via TAB/ShiftTAB  OR  via hotkey.                *
//* 2) For binary-state controls such as Pushbuttons and Radiobuttons,         *
//*    selection via hotkey ALSO indicates whether a state change has occurred.*
//* 3) For more complex controls, the hotkey information is irrelevant, and    *
//*    is discarded before calling the Edit method associated with that        *
//*    control type.                                                           *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void DMouseTest::dmInteract ( void )
{
   if ( this->dmOpen )
   {
      gString gsOut, gso2 ;            // for formatting output
      winPos   wp( 2, 2 ) ;            // output position
      wkeyCode wk ;                    // key input
      bool stableMode = false ;        // track standard/stable mode

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control

      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {  //* Get user input *

            #if 0    // TESTING ONLY - ENABLE MOUSE DATA DUMP
            if ( icIndex == dmClearPB )
            { winPos wp( 3, 0 ) ; dp->Dump_uiInfo ( wp, Info, true ) ; }
            #endif   // TESTING ONLY - ENABLE MOUSE DATA DUMP
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = this->dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == dmDonePB )
               {  //* We're done *
                  done = true ;
               }
               else if ( Info.ctrlIndex == dmClearPB )
               {  //* Clear the input area *
                  this->dp->ClearBillboard ( dmOutBB ) ;
               }
               else if ( Info.ctrlIndex == dmShowPB )
               {
                  this->dp->EditMenuwin ( dmConMW, Info ) ;
               }
            }
         }     // dctPUSHBUTTON

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = this->dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == dmFilterRB )
               {  //* Enable/disable filtering *
                  this->dp->meAutoFilter ( Info.isSel ) ;
               }
               else if ( Info.ctrlIndex == dmWheelRB )
               {  //* Enable/disable ScrollWheel events *
                  this->dmSetEMask () ;
                  this->dmSynch2EMask () ;
               }
               else if ( Info.ctrlIndex == dmAutoRB )
               {  //* Enable/disable auto-conversion of mouse events *
                  this->dp->meAutoConvert ( Info.isSel ) ;
               }
               else if ( Info.ctrlIndex == dmStableRB )
               {
                  static mmask_t savedemask ;
                  static int  savedspin ;
                  static bool savedsw ;

                  if ( Info.isSel )    // enable 'stable' mode
                  {
                     //* Save our standard-mode parameters.*
                     this->dp->GetRadiobuttonState ( dmWheelRB, savedsw ) ;
                     this->dp->GetSpinnerValue ( dmDelaySP, savedspin ) ;
                     this->dp->meGetEventTypes ( savedemask ) ;
                     //* Enable stable mode, then          *
                     //* update user's display info.       *
                     short dclick = ncmiSTABLE ;
                     if ( (this->dp->meEnableStableMouse ( dclick, savedsw )) == OK )
                     {
                        //* Synch mask selector and bit display with actual *
                        this->dmSynch2EMask () ;
                        //* Update value in click-delay Spinner *
                        this->dp->SetSpinnerValue ( dmDelaySP, ncmiNONE ) ;
                        // Set state of filter and autoconvert buttons.*
                        this->dp->SetRadiobuttonState ( dmFilterRB, true ) ;
                        this->dp->SetRadiobuttonState ( dmAutoRB, true ) ;

                        stableMode = true ;
                        this->dp->WriteString (
                                 (ic[dmOutBB].ulY + ic[dmOutBB].labY + 2),
                                 ic[dmOutBB].ulX,
                                 "  Stable Mode enabled.  ",
                                 nc.reR, true ) ;
                     }
                     else
                        this->dp->WriteParagraph ( 
                                 (ic[dmOutBB].ulY + ic[dmOutBB].labY + 2),
                                 (ic[dmOutBB].ulX - 12),
                                 "  ERROR! Unable to start Stable Mode.  \n"
                                 "         Exit test and restart.        ",
                                 nc.reR, true ) ;
                  }
                  else                 // return to standard mode
                  {
                     //* Retrieve previous settings        *
                     //* and restore them. then            *
                     //* update user's display info.       *
                     stableMode = false ;
                     this->dp->meDisableMouse () ;
                     this->dp->SetRadiobuttonState ( dmWheelRB, savedsw ) ;
                     this->dp->SetSpinnerValue ( dmDelaySP, savedspin ) ;
                     this->dp->SetRadiobuttonState ( dmFilterRB, true ) ;
                     this->dp->SetRadiobuttonState ( dmAutoRB, true ) ;
                     this->dp->meEnableMouse ( savedemask, savedspin ) ;
                     this->dmSynch2EMask () ;
                     dp->WriteString (
                              (ic[dmOutBB].ulY + ic[dmOutBB].labY + 2),
                              ic[dmOutBB].ulX,
                              "                        ",
                              this->dColor, true ) ;
                  }
                  //* Go to CLEAR button *
                  icIndex = this->dp->NextControl () ;
               }
               else
                  ; /* Remaining radio buttons do nothing */
            }
         }     // dctRADIOBUTTON

         //************************************************
         //* If focus is currently on a Scrollext control *
         //************************************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            if ( Info.viaHotkey != false )   // discard hotkey info
               Info.HotData2Primary () ;
            this->dp->RefreshScrollextText ( dmMaskSE, true ) ; // enable highlight
            do
            {
               icIndex = this->dp->EditScrollext ( Info ) ;
               //* We act only if user has selected/deselected an item, *
               //* not based on the highlight position.                 *
               if ( Info.keyIn == nckENTER || Info.keyIn == nckSPACE )
                  Info.dataMod = true ;
               else
                  Info.dataMod = false ;
               if ( (Info.dataMod != false) && (stableMode == false) &&
                    (Info.selMember <= lastSelectable) )
               {  //* If user has selected/deselected an item, update the display *
                  //* text and refresh the control. Then update the event mask.   *
                  selectAttr[Info.selMember] = 
                     (selectAttr[Info.selMember] == selectColor) ?
                     deselectColor : selectColor ;
                  this->dp->MoveScrollextHighlight ( dmMaskSE, nckDOWN ) ;
                  this->dmSetEMask () ;
               }
            }
            while ( Info.dataMod != false && icIndex == dmMaskSE ) ;
            this->dp->RefreshScrollextText ( dmMaskSE, false ) ; // remove highlight
         }     // dctSCROLLEXT

         //**************************************
         //* If focus is currently on a Spinner *
         //**************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            if ( Info.viaHotkey != false )   // discard hotkey info
               Info.HotData2Primary () ;
            icIndex = this->dp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( stableMode == false )
               {
                  int newDelay ;
                  this->dp->GetSpinnerValue ( dmDelaySP, newDelay ) ;
                  this->dp->meSetClickInterval ( short(newDelay) ) ;
               }
               //* For 'Stable' mode, allow exercise of the Spinner, *
               //* but not a value change.                           *
               else
                  this->dp->SetSpinnerValue ( dmDelaySP, ncmiNONE ) ;
            }
         }     // dctSPINNER

         //****************************************
         //* If focus is currently on a Textbox   *
         //****************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( Info.viaHotkey != false )   // discard hotkey info
               Info.HotData2Primary () ;
            icIndex = this->dp->EditTextbox ( Info ) ;
         }     // dctTEXTBOX

//         //****************************************
//         //* If focus is currently on a Billboard *
//         //****************************************
//         else if ( ic[icIndex].type == dctBILLBOARD )
//         {
//            if ( Info.viaHotkey != false )   // discard hotkey info
//               Info.HotData2Primary () ;
//            icIndex = this->dp->EditBillboard ( Info ) ;
//            if ( Info.dataMod != false )
//            {
//               //* Because Billboard data are not editable, under normal    *
//               //* circumstances, we should never arrive here; but our test *
//               //* Billboard is 'active' to see if the mouse can access it. *
//            }
//         }     // dctBILLBOARD

         //****************************************
         //* If focus is currently on Scrollbox   *
         //****************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            if ( Info.viaHotkey != false )   // discard hotkey info
               Info.HotData2Primary () ;
            icIndex = this->dp->EditScrollbox ( Info ) ;
         }     // dctSCROLLBOX

         //****************************************
         //* If focus is currently on Dropbown    *
         //****************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {  //* If we arrived here via hotkey, then leave the data intact *
            //* as a signal that control should expand immediately.       *
            if ( Info.viaHotkey != false )
               ; /* do nothing */
            icIndex = this->dp->EditDropdown ( Info ) ;
         }     // dctDROPDOWN

         //****************************************
         //* If focus is currently on a Menuwin   *
         //****************************************
         else if ( ic[icIndex].type == dctMENUWIN )
         {  //* If we arrived here via hotkey, then leave the data intact *
            //* as a signal that control should expand immediately.       *
            if ( Info.viaHotkey != false )
               ; /* do nothing */
            icIndex = this->dp->EditMenuwin ( Info ) ;
            //* Because one of the Menuwin controls extends BELOW THE DIALOG   *
            //* WINDOW when expanded, beautify the violated terminal window.   *
            winPos wp = this->dp->GetDialogPosition () ;
            wp.ypos += this->dRows ;
            nc.ClearLine ( wp.ypos++ ) ;
            nc.ClearLine ( wp.ypos++ ) ;
            nc.ClearLine ( wp.ypos++ ) ;
            this->dp->RefreshWin () ;
         }     // dcrMENUWIN

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = this->dp->NextControl () ;
         }
      }  // while()
      this->dp->meDisableMouse () ;       // disable mouse input
   }
   else
      { /* Caller is an idiot */ }

}  //* End dmInteract() *

//*************************
//*       dmInitSE        *
//*************************
//******************************************************************************
//* Construct display strings for the event-selector control.                  *
//* NOTE: This is a bit clunky because we assume that the bitmask definitions  *
//*       in ncurses.h will not change. If they do, the display will be wrong. *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DMouseTest::dmInitSE ( void )
{
   #if NCURSES_MOUSE_VERSION <= 1
   const char* iStrings[SelectBITS] =        // Mouse version <=1 definitions
   { //12|4567890123456|
      " BUTTON1_RELEASE ", " BUTTON1_PRESS   ", " BUTTON1_CLICK1  ",
      " BUTTON1_CLICK2  ", " BUTTON1_CLICK3  ", " BUTTON1_RESERVE ",
      " BUTTON2_RELEASE ", " BUTTON2_PRESS   ", " BUTTON2_CLICK1  ",
      " BUTTON2_CLICK2  ", " BUTTON2_CLICK3  ", " BUTTON2_RESERVE ",
      " BUTTON3_RELEASE ", " BUTTON3_PRESS   ", " BUTTON3_CLICK1  ",
      " BUTTON3_CLICK2  ", " BUTTON3_CLICK3  ", " BUTTON3_RESERVE ",
      " BUTTON4_RELEASE ", " BUTTON4_PRESS   ", " BUTTON4_CLICK1  ",
      " BUTTON4_CLICK2  ", " BUTTON4_CLICK3  ", " BUTTON4_RESERVE ",
      " CTRL Key (read) ", " SHFT Key (read) ",
      " ALT  Key (read) ", " Position (read) ",
      " (unassigned)    ", " (unassigned)    ",
      " (unassigned)    ", " (unassigned)    ",
   } ;                                       
   #else    // NCURSES_MOUSE_VERSION >1
   const char* iStrings[SelectBITS] =        // Mouse version >1 definitions
   { //12|4567890123456|
      " BUTTON1_RELEASE ", " BUTTON1_PRESS   ", " BUTTON1_CLICK1  ",
      " BUTTON1_CLICK2  ", " BUTTON1_CLICK3  ",
      " BUTTON2_RELEASE ", " BUTTON2_PRESS   ", " BUTTON2_CLICK1  ",
      " BUTTON2_CLICK2  ", " BUTTON2_CLICK3  ",
      " BUTTON3_RELEASE ", " BUTTON3_PRESS   ", " BUTTON3_CLICK1  ",
      " BUTTON3_CLICK2  ", " BUTTON3_CLICK3  ",
      " BUTTON4_RELEASE ", " BUTTON4_PRESS   ", " BUTTON4_CLICK1  ",
      " BUTTON4_CLICK2  ", " BUTTON4_CLICK3  ",
      " BUTTON5_RELEASE ", " BUTTON5_PRESS   ", " BUTTON5_CLICK1  ",
      " BUTTON5_CLICK2  ", " BUTTON5_CLICK3  ",
      " CTRL Key (read) ", " SHFT Key (read) ",
      " ALT  Key (read) ", " Position (read) ",
      " (unassigned)    ", " (unassigned)    ", " (unassigned)    ",
   } ;
   #endif   // NCURSES_MOUSE_VERSION >1
   gString gs ;
   for ( short items = ZERO ; items < SelectBITS ; items++ )
   {
      gs = iStrings[items] ;
      gs.limitCols( SelectWIDTH - 1 ) ;
      gs.copy( selectText[items], SelectWIDTH ) ;
      selectPntr[items] = selectText[items] ;
      selectAttr[items] = items <= lastSelectable ? deselectColor : noselectColor ;
   }

}  //* End dmInitSE() *

//*************************
//*      dmSetEMask       *
//*************************
//******************************************************************************
//* Set a new value for the NCurses-class event mask based on the selections   *
//* in the cmMaskSE control.                                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if value set successfully, else 'false'                    *
//******************************************************************************

bool DMouseTest::dmSetEMask ( void )
{
   //* Set mask according to 'selectColor' *
   mmask_t  cmask = ZERO, cmark = 0x00000001 ;
   for ( short i = ZERO ; i <= lastSelectable ; i++ )
   {
      if ( setSelect.dispColor[i] == selectColor )
         cmask |= cmark ;
      cmark <<= 1 ;
   }

   //* Test whether scroll-wheel reporting is enabled *
   bool wheelFlag ;
   this->dp->GetRadiobuttonState ( dmWheelRB, wheelFlag ) ;
   if ( wheelFlag )
      cmask |= REPORT_SCROLL_WHEEL ;
   else
      cmask &= ~REPORT_SCROLL_WHEEL ;

   bool status = ((this->dp->meSetEventTypes ( cmask )) == OK) ;
   this->dmUpdateTBox ( cmask ) ;
   return status ;

}  //* End dmSetEMask() *

//*************************
//*    dmSynch2EMask      *
//*************************
//******************************************************************************
//* Synchronize the display of data in the event-mask selector with the active *
//* event mask AND update the hex value displayed.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DMouseTest::dmSynch2EMask ( void )
{
   mmask_t  cmask ;
   if ( (this->dp->meGetEventTypes ( cmask )) == OK )
   {
      short si = ZERO ;
      selectAttr[si++] = ((cmask & BUTTON1_RELEASED) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_PRESSED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      #if NCURSES_MOUSE_VERSION <= 1
      selectAttr[si++] = ((cmask & BUTTON1_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      #else    // NCURSES_MOUSE_VERSION >1
      selectAttr[si++] = ((cmask & BUTTON2_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      #endif   // NCURSES_MOUSE_VERSION >1

      this->dp->RefreshScrollextText ( dmMaskSE, false ) ;
      this->dmUpdateTBox ( cmask ) ;
   }

}  //* End dmSynch2EMask() *

//*************************
//*     dmUpdateTBox      *
//*************************
//******************************************************************************
//* Update the dispay data with current mouse-event mask.                      *
//*                                                                            *
//* Input  : cmask  : event mask                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DMouseTest::dmUpdateTBox ( mmask_t cmask )
{
   gString gs ;
   gs.compose( L" %B ", &cmask ) ;
   this->dp->SetTextboxText ( dmMaskTB, gs ) ;

}  //* End dmUpdateTBox() *



//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - Non-member Methods- - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*    cmControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. Display mouse-event data in dmBilaBB control.                         *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

static short dmControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime )
{
   if ( firstTime )
   {
      // nothing to be done
   }
   else
   {
      if ( wkey.type == wktMOUSE || wkey.type == wktERR || wkey.mevent.conv != false )
      {
         const wchar_t wStar = L'*', wSpace = L' ' ;
         const char* tlPtr ;              // pointer to key-type label
         const char* mePtr ;              // pointer to mouse-event-type label
         gString gsOut, gso2 ;            // for formatting output
         char cK, sK, aK, pF ;            // Ctrl, Shift, Alt keys, position flag
         //* If we have a mouse event, report it *
         if ( wkey.type == wktMOUSE || wkey.mevent.conv != false )
         {
            //* Report key type *
            for (short i = ZERO ; i < tlCount ; i++ )
            {
               if (wkey.type == typeLabels[i].type)
               { tlPtr = typeLabels[i].label ; break ; }
            }
            //* Report mouse-event type *
            mePtr = meLabels[0].label ;      // if no event
            for (short i = ZERO ; i < lastSelectable ; i++ )
            {
               if ( wkey.mevent.eventType & meLabels[i].type )
               { mePtr = meLabels[i].label ; break ; }
            }
            //* Report conditioning keys *
            cK = sK = aK = pF = '-' ;
            if ( wkey.mevent.cKey != false )    cK = 'C' ;
            if ( wkey.mevent.sKey != false )    sK = 'S' ;
            if ( wkey.mevent.aKey != false )    aK = 'A' ;
            if ( wkey.mevent.mPos != false )    pF = 'P' ;

            gsOut.compose( 
               L" type:%s key:%04Xh bits:%08Xh (%s %c%c%c%c)"
                " y:%-2hd x:%-3hd ", 
               tlPtr, &wkey.key, &wkey.mevent.eventType, mePtr, &cK, &sK, &aK, &pF, 
               &wkey.mevent.ypos, &wkey.mevent.xpos ) ;

            //* If mouse event occurred above a dialog control object *
            if ( wkey.mevent.cIndex >= ZERO || wkey.mevent.conv != false )
            {
               #if 0    // TEMP - TESTING ONLY
               short pings = 1 ;
               if ( wkey.mevent.meType == metB1_D )
                  ++pings ;
               dmcuPtr->UserAlert ( pings ) ;
               #endif   // TEMP - TESTING ONLY

               gso2.compose( L"(%02hd)%C", 
                             &wkey.mevent.cIndex,
                             (wchar_t*)(wkey.mevent.conv ? &wStar : &wSpace) ) ;
               gsOut.append ( gso2.gstr() ) ;
            }
            dmcuPtr->Append2Billboard ( dmOutBB, gsOut ) ;
         }
      }
   }
   return OK ;

}  //* End dmControlUpdate() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


