//******************************************************************************
//* File       : Chart.hpp                                                     *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2025 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 02-Aug-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Class definition and for the "Chart" class.                   *
//*              This is a widget built on the NcDialog API by the same author.*
//*              Draw a bar chart representing the specified data.             *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"   //* NcDialog API family of classes

//***********************************************
//* Constant Data:                              *
//***********************************************

//* Divisions per character cell (vertical OR horizontal) *
const short cellDIVS = 8 ;

//***************
//* Definitions *
//***************

//* Default character used by Cartesian chart for mapping datapoints.*
const wchar_t dblDiamond = 0x25C8 ;   // '◈'

//* Specify text justification *
enum txtJust : short
{
   tjLeft,                    // Left-justified
   tjCenter,                  // Centered
   tjRight                    // Right-justified
} ;

//* Specifies the type of chart grid used to map the data *
enum chartType : short
{
   ctLowerLeft,               // Horizontal axis at the bottom  (default type)
                              // Vertical axis on the left
   ctLowerRight,              // Horizontal axis at the bottom
                              // Vertical axis on the right
   ctUpperLeft,               // Horizontal axis at the top
                              // Vertical axis on the left
   ctUpperRight,              // Horizontal axis at the top
                              // Vertical axis on the right
   ctLowerCenter,             // Horizontal axis at the bottom
                              // Vertical axis in the center (horizontal bars only)
   ctUpperCenter,             // Horizontal axis at the top
                              // Vertical axis in the center (horizontal bars only)
   ctCenterLeft,              // Horizontal axis in the center
                              // Vertical axis on the left (vertical bars only)
   ctCenterRight,             // Horizontal axis in the center
                              // Vertical axis on the right (vertical bars only)
   ctCartesian,               // Vertical and Horizontal cross in the center
                              // (not yet implemented)
} ;

//* Supported data types. The default value is "double". Other data types *
//* may be dynamically converted to doubles for processing and display.   *
//* Note that the actual range of any data type is hardware dependent;    *
//* however, most 32-bit and 64-bit systems support the precision shown.  *
enum idataType : short
{
   idtDouble = ZERO,          // Double-precision (64 bits) floating point (default)
   idtFloat,                  // Single-precision (32 bits) floating point
   idtByte_s,                 // 8-bit, signed
   idtByte_u,                 // 8-bit, unsigned
   idtShort_s,                // 16-bit, signed
   idtShort_u,                // 16-bit, unsigned
   idtInt_s,                  // 32-bit, signed
   idtInt_u,                  // 32-bit, unsigned
   idtLong_s,                 // 64-bit, signed (64-bit systems)
   idtLong_u,                 // 64-bit, signed (64-bit systems)
   idtLongLong_s,             // 64-bit, signed
   idtLongLong_u,             // 64-bit, signed
} ;

//* Definitions for shifting data on the chart forward/backward.*
//* See ShiftData() method for detailed description.            *
enum ShiftBlock : short 
{
   sbNoShift,           // no shift, parameter ignored
   sbFirstPage,         // display data from beginning of array
   sbLastPage,          // display last page (tail) of data
   sbNextPage,          // display next page of data (shift current data out)
   sbPrevPage,          // display previous page of data (shift current data out)
} ;

//* Initialization data for call to ShiftData() method.   *
//* ShiftData() optionally accepts a pointer to an array  *
//* of ShiftDef objects indicating the types of data      *
//* shift operations the user may access and the keypress *
//* associated with each type of shift.                   *
//* See ShiftData() prototype below for more information. *
class ShiftDef
{
   public:
   //* Keycode to activate the shift. This is keycode AND key type. *
   //* See NCursesKeyDef.hpp for the complete list.                 *
   wkeyCode   wk ;
   //* Type of shift (member of enum ShiftBlock). Set to sbNoShift  *
   //* if specific shift count is specified by 'sc' member.         *
   ShiftBlock sb ;
   //* If the 'sb' member == sbNoShift, specify the number of       *
   //* elements (character cells) to shift.                         *
   //* A positive value shifts data forward (toward end of data).   *
   //* A negative value shifts data backward (toward beginning).    *
   //* If 'sb' != sbNoShift, then this value is ignored.            *
   short      sc ;
} ;

//* Calculated range for the quadrants of Cartesian *
//* charts. Used to scale the data points.          *
class CartRange
{
   public:
   double minValX ;        // X coordinate minimum value
   double maxValX ;        // X coordinate maximum value
   double minValY ;        // Y coordinate minimum value
   double maxValY ;        // Y coordinate minimum value

   double meanValX ;       // X coordinates, arithmetic mean (average value)
   double meanValY ;       // Y coordinates, arithmetic mean (average value)
   double medianValX ;     // X coordinates, median value
   double medianValY ;     // Y coordinates, median value
} ;

//* Used by the GetStats() method to return the basic *
//* statistics for the displayed dataset.             *
class ChartStats
{
   public:
   int32_t  dataItems ;    // number of data items in array
                           // (for ctCartesian, this is number of coordinate pairs)
   //* Statistics for non-Cartesian chart types *
   double   minValue ;     // minimum data value 
   double   maxValue ;     // maximum data value
   double   meanValue ;    // arithmetic mean value (average)
   double   medianValue ;  // calculated median value

   //* Statistics for Cartesian charts only *
   CartRange cartRange ;
} ;

//*****************************************
//** Initialization data for Chart-class **
//**     initialization constructor.     **
//*****************************************
class ChartDef
{
   public:
   ChartDef ( void )          // constructor
   { this->reset () ; }

   NcDialog *dPtr;            // pointer to dialog window in which to draw chart
                              // if null pointer, a new dialog will be generated
   short  ulY ;               // upper left of chart area, Y
   short  ulX ;               // upper left of chart area, X
   short  rows ;              // number of rows in chart area
   short  cols ;              // number of columns in chart area
   short  yOffset ;           // Y offset of grid within the display area (header area)
   short  xOffset ;           // X offset of grid within the display area (margin area)
   short  footRows ;          // number of rows reserved for footer area
   short  barWidth ;          // bar width in "divisions" (1-8 divisions)
                              // (zero(0) indicates that only tip of each bar is be be displayed)
   short  barSpace ;          // bar spacing 0 or 1 empty character cells between bars
   chartType chType ;         // X/Y grid type (see above)
   attr_t borderColor ;       // border color attribute
   attr_t titleColor ;        // title color attribute
   attr_t baseColor ;         // basic background color, axis label color
   attr_t textColor ;         // text color, header/footer/margin
   attr_t gridColor ;         // color attribute for X/Y grid
   attr_t barColor ;          // default color attribute for bars of bar chart
                              // (see also 'attrPtr' member)
   attr_t negColor ;          // bar color for negative values (mixed positive and negative)

   int32_t dataCount ;        // number of elements in 'dataPtr' (and 'attrPtr' if specified)
   int32_t dataOffset ;       // index into 'dataPtr' array at which to begin display
   const void* dataPtr ;      // pointer to array of data values to be charted
   idataType dataType ;       // type of data pointed to by 'dataPtr' member
   attr_t *attrPtr ;          // (optional) array of attributes, one for each bar

   wchar_t    cartChar ;      // for Cartesian charts, the character for datapoints
   ncLineType borderStyle ;   // border style
   ncLineType gridStyle ;     // grid-line style
   const char* vaxisLabel ;   // vertical-axis label
   const char* haxisLabel ;   // horizontal-axis label
   const char* titleText ;    // (optional) title of the chart (displayed in top line of area)
   const char* headText ;     // (optional) text to write above chart
   const char* footText ;     // (optional) text to write below chart (legend, footnotes, etc.)
   const char* marginText ;   // (optional) text to write in the left margin (see 'xOff')
   bool horizBars ;           // 'true' if horizontal bars, 'false' if vertical bars
                              // (for Cartesian charts: 'false'==X/Y pairs, 'true'==Y/X pairs)
   bool drawBorder ;          // if 'true' draw border around chart area

   //** Reset all members **
   void reset ( void )
   {
      this->dPtr        = NULL ;    // independent dialog window
      this->ulY         = ZERO ;    // terminal-window origin
      this->ulX         = ZERO ;
      this->rows        = 9 ;       // arbitrary-but-small window dimensions which are
      this->cols        = 28 ;      // large enough to display grid and axis labels
      this->yOffset     = 1 ;       // no header area
      this->xOffset     = ZERO ;    // no margin area
      this->footRows    = ZERO ;    // no footer area
      this->barWidth    = cellDIVS ;// full width of character cell
      this->barSpace    = ZERO ;    // no space between bars
      this->chType      = ctLowerLeft ; // default chart type
      this->borderColor = ZERO ;    // (terminal-window color attribute)
      this->titleColor  = ZERO ;
      this->baseColor   = ZERO ;
      this->textColor   = ZERO ;
      this->gridColor   = ZERO ;
      this->barColor    = ZERO ;
      this->negColor    = ZERO ;
      this->dataCount   = ZERO ;    // no data values
      this->dataOffset  = ZERO ;    // offset zero
      this->dataPtr     = NULL ;    // no data specified
      this->dataType    = idtDouble ; // double-precision floating-point data
      this->attrPtr     = NULL ;    // multi-color text not specified
      this->cartChar    = dblDiamond ; // default Cartesian datapoint character
      this->borderStyle = ncltSINGLE ; // single-line border style
      this->gridStyle   = ncltSINGLE ; // single-line axis style
      this->vaxisLabel  = "Vertical Axis Label" ;  // default axis label
      this->haxisLabel  = "Horizontal Axis Label" ;// default axis label
      this->titleText   = NULL ;    // no title text
      this->headText    = NULL ;    // no header text
      this->footText    = NULL ;    // no footer text
      this->marginText  = NULL ;    // no margin text
      this->horizBars   = false ;   // vertical bars
      this->drawBorder  = false ;   // no area border
   }  // reset

} ;   // ChartDef


//****************************
//** Chart-class Definition **
//****************************
class Chart
{
   public:

   //* Destructor. Return all resources to the system.           *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: nothing                                          *
   virtual ~Chart ( void ) ;

   //* Full-initialization constructor.                          *
   //* All data members set according to provided parameters.    *
   //*                                                           *
   //* Input  : cdef   : (by reference) a fully-initialized      *
   //*                   instance of the 'ChartDef' class.       *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', setup is performed, but     *
   //*                               display not updated         *
   //*                   if 'true',  after setup, update display *
   //*                               immediately                 *
   //* Returns: implicitly returns pointer to instantiated object*
   Chart ( const ChartDef& cdef, bool refresh = false ) ;

   //* Refresh the display. This makes any changes visible.      *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: nothing                                          *
   void RefreshDisplay ( void ) const
   { this->dp->RefreshWin () ; }

   //* Interact with user to shift the data visible in the chart *
   //* window forward and backward through the data array.       *
   //*                                                           *
   //* If all data are currently displayed, returns immediately. *
   //* Otherwise, returns when an un-handled keycode is received.*
   //*                                                           *
   //* If the 'sdPtr' argument is not provided, then the         *
   //* following default shift/keycode combinations are used.    *
   //* Shift Operation           Keycode              Key type   *
   //* -----------------------   ------------------------------- *
   //* top of data             : Home key (sbFirstPage)          *
   //* end of data             : End key  (sbLastPage)           *
   //* next page of data       : PageDown (sbNextPage)           *
   //* previous page of data   : PageUp   (sbPrevPage)           *
   //*  1 step toward end      : RightArrow           wktFUNKEY  *
   //*  1 step toward top      : LeftArrow            wktFUNKEY  *
   //*  5 steps toward end     : Shift + RightArrow   wktFUNKEY  *
   //*  5 steps toward top     : Shift + LeftArrow    wktFUNKEY  *
   //* 10 steps toward end     : Ctrl + RightArrow    wktFUNKEY  *
   //* 10 steps toward top     : Ctrl + LeftArrow     wktFUNKEY  *
   //*                                                           *
   //* Input  : wkey : (by reference) receives the unhandled     *
   //*                 keycode that triggered return             *
   //*          sdCnt: (optional, ZERO by default)               *
   //*                 If specified, this is the number of       *
   //*                 objects in the 'sdPtr' array.             *
   //*          sdPtr: (optional, NULL pointer by default)       *
   //*                 If specifed, this is a pointer to an array*
   //*                 of ShiftDef objects which define the valid*
   //*                 shift options and associated keycodes.    *
   //*                 See documentation for examples.           *
   //*                                                           *
   //* Returns: index of first currently-displayed data item     *
   //*          'wkey' contains the unhandled key input. Note:   *
   //*          if all data currently displayed, wkey.key==null  *
   int32_t ShiftData ( wkeyCode& wkey, short sdCnt = ZERO, ShiftDef *sdPtr = NULL ) ;

   //* Replace the currently-displayed dataset with a new        *
   //* dataset. Use this method to refresh the data displayed in *
   //* the Chart object when the data are dynamically changing   *
   //* in application space, or to alternately display different *
   //* datasets.                                                 *
   //*                                                           *
   //* Note that only the dataset will be updated. Header, Footer*
   //* and Margin areas are not affected. The coordinate-grid    *
   //* style and dimensions are not affected. Data-bar           *
   //* orientation is not affected.                              *
   //*                                                           *
   //* It is an application's judgement call when changing       *
   //* datasets whether to update the data in the existing Chart *
   //* object, or to close the existing Chart object and launch  *
   //* a new Chart object to display the new data.               *
   //* It is also possible to have multiple, independent Chart   *
   //* objects and simply bring them into the foregroud on a     *
   //* rotating basis. From a performance point-of-view, it is   *
   //* somewhat faster to update the data in an existing Chart   *
   //* object because it cuts the number of calculations and     *
   //* system calls by about 40%, but use your best judgement.   *
   //*                                                           *
   //* Only four(4) parameters are updated:                      *
   //*   1) the array of source data items                       *
   //*   2) the number of source data items                      *
   //*   3) the type of source data                              *
   //*   4) for Cartesian charts ONLY, indicate whether data     *
   //*      are X/Y pairs or Y/X pairs.                          *
   //*                                                           *
   //* Input  : dataPtr  : pointer to new dataset                *
   //*          dataCount: number of items in 'dataPtr' array    *
   //*          dataType : data type (member of enum idataType)  *
   //*          YX_pairs : (optional, 'false' by default)        *
   //*                     (for Cartesian chart type only)       *
   //*                     'false' == X/Y pairs                  *
   //*                     'true'  == Y/X pairs                  *
   //*                                                           *
   //* Returns: 'true' if data successfully updated              *
   //*          'false' if invalid parameter                     *
   bool ReloadDataset ( const void *dataPtr, int32_t dataCount,
                        idataType dataType, bool YX_pairs = false ) ;

   //* Get dimensions of header area.                            *
   //*                                                           *
   //* Input  : rows   : (by reference) receives number of       *
   //*                   display rows                            *
   //*          cols   : (by reference) receives number of       *
   //*                                  display columns          *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool GetHeaderDimensions ( short& rows, short& cols ) const ;

   //* Clear the header-text area and write the specified text.  *
   //*                                                           *
   //* Note: To clear the area without writing new text,         *
   //*       set 'txt' to an empty string ("").                  *
   //*                                                           *
   //* Input  : txt    : text to be written                      *
   //*          attr   : color attribute for text                *
   //*          border : (optional, 'false' by default)          *
   //*                   if 'true', draw a horizontal line across*
   //*                   the bottom row of the header area.      *
   //*                   This creates a visual division between  *
   //*                   the header and the chart grid.          *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool DrawHeaderText ( const char* txt, attr_t txtAttr, 
                         bool border = false, bool refresh = false ) ;

   //* Add text to the header area at the specified offset.      *
   //* For example, display of instructions or explanations at   *
   //* different positions or in different colors.               *
   //* -- See GetHeaderDimensions() for obtaining the dimensions *
   //*    of the header area.                                    *
   //* -- Unlike the DrawHeaderText() method above, this method  *
   //*    DOES NOT clear the header area before writing. Instead,*
   //*    the specified text is added to any existing text in    *
   //*    the header area.                                       *
   //* -- The start position (pos) is validated to be within the *
   //*    target area. If start position is out-of-range, the    *
   //*    text will not be written, and a cursor offset of 0:0   *
   //*    will be returned to indicate the error.                *
   //* -- If any portion of the source text (txt) would overrun  *
   //*    the boundaries of the target area, that portion of the *
   //*    text will be silently discarded.                       *
   //* -- If all source text is displayed successfully, the      *
   //*    offset to the space following the text will be         *
   //*    returned.                                              *
   //*    If, however, some of the text was discarded due to     *
   //*    attempted boundary violations, then the cursor offset  *
   //*    returned may not be as expected. For example, if the   *
   //*    text was truncated to avoid overrunning the right edge *
   //*    of the target area, the X offset will reference the    *
   //*    right edge of the target area. Similarly, if the text  *
   //*    was truncated to avoid overrunning the bottom edge of  *
   //*    the target area, the Y offset will reference the last  *
   //*    row of the target area.                                *
   //*                                                           *
   //* Input  : pos    : (by reference)                          *
   //*                   Y/X offset from header base position    *
   //*                   Y offset must be >= ZERO and within     *
   //*                            the header area                *
   //*                   X offset Must be >= ZERO and within     *
   //*                            the header area                *
   //*          txt    : text to be written. Line breaks are     *
   //*                   indicated by newline '\n' characters.   *
   //*          attr   : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: Y/X offset at end of text written.               *
   //*          Returns Y/X==0/0 if 'pos' is out-of-range OR     *
   //*            if end position is out-of-range (unlikely)     *
   winPos Add2HeaderText ( const winPos& pos, const char* txt,
                           attr_t txtAttr, bool refresh = false ) ;

   //* Get dimensions of footer area.                            *
   //*                                                           *
   //* Input  : rows   : (by reference) receives number of       *
   //*                   display rows                            *
   //*          cols   : (by reference) receives number of       *
   //*                   display columns                         *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool GetFooterDimensions ( short& rows, short& cols ) const ;

   //* Clear the footer-text area and write the specified text.  *
   //*                                                           *
   //* Note: To clear the area without writing new text,         *
   //*       set 'txt' to an empty string ("").                  *
   //*                                                           *
   //* Input  : txt    : text to be written                      *
   //*          attr   : color attribute for text                *
   //*          border : (optional, 'false' by default)          *
   //*                   if 'true', draw a horizontal line across*
   //*                   the top row of the footer area.         *
   //*                   This creates a visual division between  *
   //*                   the footer and the chart grid.          *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (rows == ZERO)*
   bool DrawFooterText ( const char* txt, attr_t txtAttr, 
                         bool border = false, bool refresh = false ) ;

   //* Add text to the footer area at the specified offset. For  *
   //* example, display of instructions or explanations at       *
   //* different positions or in different colors.               *
   //* -- See GetFooterDimensions() for obtaining the dimensions *
   //*    of the footer area.                                    *
   //* -- Unlike the DrawFooterText() method above, this method  *
   //*    DOES NOT clear the footer area before writing. Instead,*
   //*    the specified text is added to any existing text in    *
   //*    the footer area.                                       *
   //* -- The start position (pos) is validated to be within the *
   //*    target area. If start position is out-of-range, the    *
   //*    text will not be written, and a cursor offset of 0:0   *
   //*    will be returned to indicate the error.                *
   //* -- If any portion of the source text (txt) would overrun  *
   //*    the boundaries of the target area, that portion of the *
   //*    text will be silently discarded.                       *
   //* -- If all source text is displayed successfully, the      *
   //*    offset to the space following the text will be         *
   //*    returned.                                              *
   //*    If, however, some of the text was discarded due to     *
   //*    attempted boundary violations, then the cursor offset  *
   //*    returned may not be as expected. For example, if the   *
   //*    text was truncated to avoid overrunning the right edge *
   //*    of the target area, the X offset will reference the    *
   //*    right edge of the target area. Similarly, if the text  *
   //*    was truncated to avoid overrunning the bottom edge of  *
   //*    the target area, the Y offset will reference the last  *
   //*    row of the target area.                                *
   //*                                                           *
   //* Input  : pos    : (by reference)                          *
   //*                   Y/X offset from footer base position    *
   //*                   Y offset must be >= ZERO and within     *
   //*                            the footer area                *
   //*                   X offset Must be >= ZERO and within     *
   //*                            the footer area                *
   //*          txt    : text to be written. Line breaks are     *
   //*                   indicated by newline '\n' characters.   *
   //*          attr   : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: Y/X offset at end of text written.               *
   //*          Returns Y/X==0/0 if 'pos' is out-of-range OR     *
   //*            if end position is out-of-range (unlikely)     *
   winPos Add2FooterText ( const winPos& pos, const char* txt,
                           attr_t txtAttr, bool refresh = false ) ;

   //* Get dimensions of margin area.                            *
   //*                                                           *
   //* Input  : rows   : (by reference) receives number of       *
   //*                   display rows                            *
   //*          cols   : (by reference) receives number of       *
   //*                   display columns                         *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined (cols == ZERO)*
   bool GetMarginDimensions ( short& rows, short& cols ) const ;

   //* Clear the margin-text area and write the specified text   *
   //* to the margin area.                                       *
   //*                                                           *
   //* Note: To clear the area without writing new text,         *
   //*       set 'txt' to an empty string ("").                  *
   //*                                                           *
   //* Input  : txt    : text to be written                      *
   //*          attr   : color attribute for text                *
   //*          border : (optional, 'false' by default)          *
   //*                   draw an interior border around the      *
   //*                   Margin area.                            *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: 'true' if successful                             *
   //*          'false' if target area not defined               *
   bool DrawMarginText ( const char* txt, attr_t txtAttr, 
                         bool border = false, bool refresh = false ) ;

   //* Add text to the margin area at the specified offset.      *
   //* For example, display of instructions or explanations at   *
   //* different positions or in different colors.               *
   //* -- See GetMarginDimensions() for obtaining the dimensions *
   //*    of the margin area.                                    *
   //* -- Unlike the DrawMarginText() method above, this method  *
   //*    DOES NOT clear the margin area before writing.         *
   //*    Instead, the specified text is added to any existing   *
   //*    text in the margin area.                               *
   //* -- The start position (pos) is validated to be within the *
   //*    target area. If start position is out-of-range, the    *
   //*    text will not be written, and a cursor offset of 0:0   *
   //*    will be returned to indicate the error.                *
   //* -- If any portion of the source text (txt) would overrun  *
   //*    the boundaries of the target area, that portion of the *
   //*    text will be silently discarded.                       *
   //* -- If all source text is displayed successfully, the      *
   //*    offset to the space following the text will be         *
   //*    returned.                                              *
   //*    If, however, some of the text was discarded due to     *
   //*    attempted boundary violations, then the cursor offset  *
   //*    returned may not be as expected. For example, if the   *
   //*    text was truncated to avoid overrunning the right edge *
   //*    of the target area, the X offset will reference the    *
   //*    right edge of the target area. Similarly, if the text  *
   //*    was truncated to avoid overrunning the bottom edge of  *
   //*    the target area, the Y offset will reference the last  *
   //*    row of the target area.                                *
   //*                                                           *
   //* Input  : pos    : (by reference)                          *
   //*                   Y/X offset from margin base position    *
   //*                   Y offset must be >= ZERO and within     *
   //*                            the margin area                *
   //*                   X offset Must be >= ZERO and within     *
   //*                            the margin area                *
   //*          txt    : text to be written. Line breaks are     *
   //*                   indicated by newline '\n' characters.   *
   //*          attr   : color attribute for text                *
   //*          refresh: (optional, 'false' by default)          *
   //*                   if 'false', do not refresh the display  *
   //*                               after writing               *
   //*                   if 'true',  refresh the display         *
   //*                               (make changes visible)      *
   //* Returns: Y/X offset at end of text written.               *
   //*          Returns Y/X==0/0 if 'pos' is out-of-range OR     *
   //*            if end position is out-of-range (unlikely)     *
   winPos Add2MarginText ( const winPos& pos, const char* txt,
                           attr_t txtAttr, bool refresh ) ;

   //* Draw a horizontal line across the Footer Area.            *
   //* Line position is specified as an OFFSET, and must fall    *
   //* within the footer area.                                   *
   //* Minumum Offset: 0 == top line of footer area.             *
   //* Maximum Offset: footer_rows - 1 == bottom line of         *
   //* footer area.                                              *
   //*    See GetFooterDimensions() method.                      *
   //* The line is drawn the full width of the target area.      *
   //* If the line intersects another line (e.g. the border),    *
   //* a visual connection will be made.                         *
   //*                                                           *
   //* Input  : lineStyle: line style                            *
   //*             (member of enum ncLineType,      )            *
   //*             (excluding ncltHORIZ and ncltVERT)            *
   //*          offset   : offset from top of target area        *
   //*                     Range: >= 0 and <= last row of target *
   //*                     area. Note that if a border has been  *
   //*                     specified, the bottom border is       *
   //*                     outside the footer area.              *
   //*          refresh  : (optional, 'false' by default)        *
   //*                     if 'false', do not refresh the display*
   //*                                 after writing             *
   //*                     if 'true',  refresh the display       *
   //*                                 (make changes visible)    *
   //* Returns: 'true'  if successful                            *
   //*          'false' if 'offset' is out-of-range or if target *
   //*                  area is not defined (target rows == ZERO)*
   bool DrawDivider ( ncLineType lineStyle, short offset, 
                      attr_t lineAttr, bool refresh = false ) ;

   //* Get basic statistics on the dataset:                      *
   //*                                                           *
   //* Input  : cStats : (by reference) an instance of the       *
   //*                   ChartStats class to receive the         *
   //*                   statistical information                 *
   //*                   Note: 'dataItems' member == number of   *
   //*                         data values EXCEPT for the        *
   //*                         ctCartesian chart type where      *
   //*                         'dataItems' == number of          *
   //*                         coordinate pairs.                 *
   //* Returns: nothing                                          *
   void GetStats ( ChartStats& cStats ) const ;

   //* Enable or disable audible alert in ShiftData() method.    *
   //* If the data cannot be shifted further in the specified    *
   //* direction, for example, if already at the end of dataset, *
   //* then a beep will sound to indicate that it is not possible*
   //* to shift further in that direction.                       *
   //*                                                           *
   //* Input  : enable: 'true'  to enable alert                  *
   //*                  'false' to disable alert                 *
   //* Returns: 'true' if alert enabled, 'false' if disabled     *
   bool  AudibleShift ( bool enable ) ;

   //* Set the display character for the datapoints of a         *
   //* Cartesian chart.Any single-column, printable,             *
   //* non-whitespace character may be specified, but be aware   *
   //* that for best results, the character should relatively    *
   //* small compared to the size of the character cell, and     *
   //* should be centered in the cell.                           *
   //* By default, the '◈' character, codepoint U+25C8 is used.  *
   //* The standard bullet characters would be good alternate    *
   //* choices.                                                  *
   //*                                                           *
   //* Input  : cartChar : character to be displayed at each     *
   //*                     datapoint                             *
   //* Returns: 'true'  if success,                              *
   //*          'false' if not a 1-column, printing character    *
   bool  SetCartesianChar ( wchar_t cartChar ) ;

   //* Superimpose an additional dataset onto an existing        *
   //* Cartesian chart.                                          *
   //*                                                           *
   //* 1) 'cartData' pairs MUST be in the same order as the      *
   //*    original dataset, either X/Y pairs or Y/X pairs.       *
   //* 2) The absolute range of the data in X and Y must be less *
   //*    than or equal to the range of the original data.       *
   //*    Datapoints which fall outside the established range    *
   //*    i.e. off-the-chart, will be silently discarded.        *
   //* 3) 'cartCount' is the number of VALUES in the array, NOT  *
   //*    the number of data pairs.                              *
   //* 4) 'cartAttr' will typically be a color which contrasts   *
   //*    with the color used for the original plot so that the  *
   //*    new data may be easily identified.                     *
   //* 5) 'dType' is optional, and indicates the type of data    *
   //*    referenced by the 'cartData' parameter.                *
   //*    'dType' indicates double-precision floating-point data *
   //*    by default. If 'cartData' points to data of another    *
   //*    type, indicate the type as a member of enum hartType.  *
   //* 6) 'cartChar' is optional. By default, the character used *
   //*    for the original data will also be used for the new    *
   //*    data; however, if desired a different character may    *
   //*    be used.                                               *
   //*    (See SetCartesianChar() method above for details.)     *
   //*                                                           *
   //* Input  : cartData : pointer to array of data values to be *
   //*                     plotted, arranged as X/Y pairs        *
   //*                     (or as Y/X pairs)                     *
   //*          cartCount: number of elements in 'cartData' array*
   //*          cartAttr : color attribute for datapoint display *
   //*          dType    : (optional, idtDouble by default)      *
   //*                     type of data referenced by 'cartData' *
   //*          cartChar : (optional, "dblDiamond" (0x25C8) by   *
   //*                     default) character to be displayed    *
   //*                     at each datapoint                     *
   //*          refresh  : (optional, 'false' by default)        *
   //*                     if 'false', setup is performed, but   *
   //*                                 display not updated       *
   //*                     if 'true',  after setup, update       *
   //*                                 display immediately       *
   //* Returns: 'true'  if success,                              *
   //*          'false' if invalid parameter(s)                  *
   bool  OverlayCartesianDataset ( const void* cartData, int32_t cartCount,
                                   attr_t cartAttr, 
                                   idataType dType = idtDouble, 
                                   wchar_t cartChar = dblDiamond, 
                                   bool refresh = false ) ;

   //* Save the display data for the independent dialog window.  *
   //* This implements a call to the local NcDialog object's     *
   //* 'SetDialogObscured()' method.                             *
   //* If dialog was not created locally, this call is ignored.  *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: nothing                                          *
   void ProtectDialog ( void ) const ;

   //* Restore display data for the independent dialog window    *
   //* which was saved by a previous call to ProtectDialog().    *
   //* This implements a call to the local NcDialog object's     *
   //* 'RefreshWin()' method.                                    *
   //* If dialog was not created locally, this call is ignored.  *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: nothing                                          *
   void RestoreDialog ( void ) const ;

   //* Returns the version number of the Chart class.            *
   //*                                                           *
   //* Input  : none                                             *
   //* Returns: pointer to const string                          *
   const char* GetVersion ( void ) const ;

   //* For Debugging Only!                                       *
   //* ===================                                       *
   //* Perform a screen capture of the dialog window.            *
   //* This is a pass-through to the NcDialog method of the      *
   //* same name.                                                *
   //*                                                           *
   //* Input  : (see NcDialog.hpp)                               *
   //*                                                           *
   //* Returns: OK if successful                                 *
   //*          ERR  : a) if parameter out-of-range              *
   //*                 b) if unable to write to specified file   *
   //*                 c) if development methods are disabled    *
   short CaptureDialog ( const char* fPath, bool fhtml = false, 
                         bool timeStamp = true, const char* stylePath = NULL, 
                         short cellsPerLine = 4, bool lineComments = true, 
                         attr_t simpleColor = ZERO, bool microSpace = false ) const ;

   private:    // all other members are private

   //*******************
   //* Private Methods *
   //*******************
   Chart ( void ) ;                        //* default constructor (private)
   void reset ( void ) ;                   //* reset all data members to default values
   void Configure ( const ChartDef& cdef );//* initialize all data members
   bool ConvertData ( const void* vdPtr,   //* convert data to double-precision floating-point
                      int32_t dCount, idataType dType, const double*& trgPtr ) ;
   void SetStyle ( void ) ;                //* set the axis-line style for the chart
   void SetDimensions ( void ) ;           //* set length of X and Y axes
   void SetOrigin ( void ) ;               //* set the origin cell for the chart
   void SetRange ( void ) ;                //* establish data range
   void DrawGrid ( bool labels = false,    //* draw the vertical and horizontal axes (and labels)
                   bool refresh = false ) ;
   void MapData2Grid ( void ) ;            //* Draw the data set onto the grid
   void MapVerticalBars ( void ) ;         //* Draw the dataset as vertical bars
   void MapHorizontalBars ( void ) ;       //* Draw the dataset as horizontal bars
   void MapVerticalCentered ( void ) ;     //* Draw the dataset extending above and below horiz. axis
   void MapHorizontalCentered ( void ) ;   //* Draw the dataset extending left and right from vert. axis
   void MapCartesianPairs ( void ) ;       //* Draw the dataset as X/Y pairs
   void PlotCartPairs ( const double* dPtr, int32_t dCnt, wchar_t pntChar, 
                        attr_t pntAttr, const attr_t *atrArray = NULL, bool truncate = false ) ;
   bool shiftData ( int32_t shiftCnt, ShiftBlock shiftBlk = sbNoShift ) ;
   void DrawAxisLabel ( const char* label, //* Draw axis label text
                        const winPos& wpl, txtJust justify = tjLeft ) ;
   bool TrimText ( const char* src, gString& trg, short maxRow, short maxCol ) ;
   bool DrawDivider ( bool targArea,       //* Draw dividing line across target area
                      ncLineType lineStyle, short offset, 
                      attr_t lineAttr, bool refresh = false ) ;
   void ClearArea ( void ) ;               //* clear entire display area (border if specified)
   bool ClearArea ( const winPos& wpPos,   //* clear target area
                     short trgRows, short trgCols, attr_t trgAttr ) ;
   void ClearGrid ( void ) ;               //* clear chart grid area, re-draw the grid.

   //* Debugging Only: display data-member values.*
   void DumpCfg ( bool refresh = false ) ;
   //* Debugging Only: display area line numbers.*
   void WriteYOffsets ( const winPos gridTop, bool clear = false ) ;


   //****************
   //* Data Members *
   //****************
   NcDialog *dp ;             //* pointer to target dialog window
   winPos   wpTerm ;          //* if local dialog defined, terminal-window offset
   winPos   wpBase ;          //* display-area base (upper-left corner)
   winPos   wpOrig ;          //* origin of chart grid (where Y and X axes connect)
   winPos   hdrPos ;          //* upper-left corner of header area
   winPos   ftrPos ;          //* upper-left corner of footer area
   winPos   marPos ;          //* upper-left corner of left-margin area
   winPos   wpYlab ;          //* Y-axis label position
   winPos   wpXlab ;          //* X-axis label position
   txtJust  jYlab ;           //* text justification for Y-axis label
   txtJust  jXlab ;           //* text justification for X-axis label
   short    ldRows ;          //* if local dialog defined, dialog height
   short    ldCols ;          //* if local dialog defined, dialog width
   short    dispRows ;        //* number of rows in display area
   short    dispCols ;        //* number of columns in display area
   short    gridRows ;        //* number of rows occupied by grid (incl. axis)
   short    gridCols ;        //* number of columns occupied by grid (incl. axis)
   short    yOffset ;         //* vertical offset from top of display area to top of grid
   short    xOffset ;         //* horizontal offset from left of display area to left of grid
   short    hdrRows ;         //* header area rows
   short    hdrCols ;         //* header area columns
   short    ftrRows ;         //* footer area rows
   short    ftrCols ;         //* footer area columns
   short    marRows ;         //* margin area rows
   short    marCols ;         //* margin area columns
   short    barWidth ;        //* bar width in "divisions" (8 divisions per character cell)
   short    barSpace ;        //* bar spacing 0 or 1 empty character cells between bars
   chartType chType ;         //* X/Y grid type (see above)
   short    vCells ;          //* number of vertical character cells in grid
   short    hCells ;          //* number of horizontal character cells in grid
   attr_t   bdrColor ;        //* border color attribute
   attr_t   titColor ;        //* title text color
   attr_t   barColor ;        //* default color for chart bars (but see 'attrPtr')
   attr_t   negColor ;        //* bar color for negative values (mixed positive and negative)
   attr_t   dColor ;          //* default text color (interior foreground/background)
   attr_t   tColor ;          //* text color for header/footer/margin
   attr_t   gColor ;          //* Y/X grid-line color
   int32_t  dataCount ;       //* number of elements in 'dataPtr' and 'attrPtr'
   int32_t  dataOffset ;      //* index of first element of 'dataPtr' to be displayed
   const double *dataPtr ;    //* pointer to array of data points (data pairs for Cartesian)
   const attr_t *attrPtr ;    //* pointer to array of color-attributes (optional)
   const char *vLabel ;       //* vertical-axis label
   const char *hLabel ;       //* horizontal-axis label
   const char *titText ;      //* title for chart area
   const char *hdrText ;      //* text written above the grid (if any)
   const char *ftrText ;      //* text written below the grid (if any)
   const char *marText ;      //* text written to margin area (if any)
   ncLineType bStyle ;        //* border line type
   ncLineType gStyle ;        //* grid line type

   double minVal ;            //* minimum data value
   double maxVal ;            //* maximum data value
   double rngVal ;            //* data range (max - min)
   double meaVal ;            //* arithmetic mean data value
   double medVal ;            //* median data value (calculated on demand)
   double verDiv ;            //* total vertical-axis divisions
   double horDiv ;            //* total horizontal-axis divisions
   double minDiv ;            //* number of divisions displayed for "minimum value"
   double maxDiv ;            //* number of divisions displayed for "maximum value"
   double rngDiv ;            //* range of available divisions (maxDiv - minBAR)
   double perDiv ;            //* data units represented by one division
   CartRange cartRng ;        //* Cartesian charts only: X and Y range for dataset
   wchar_t vaxisChar ;        //* vertical-axis character
   wchar_t haxisChar ;        //* horizontal-axis character
   wchar_t axisCross ;        //* character at axis origin (connect X and Y axis)
   wchar_t axisCap ;          //* axis-cap character (denotes reserved cell)
   wchar_t cartChar ;         //* character used to map datapoints on Cartesian chart 
   bool    dynoData ;         //* 'true' if dynamic allocation for converted data
   bool    localDlg ;         //* 'true' if dialog instantiated locally (false if caller provided)
   bool    bdrFlag ;          //* 'true' == draw border, 'false' == no border
   bool    marginBdr ;        //* 'true' == Margin area interior border, 'false' == no border
   bool    hBars ;            //* 'true' == horizontal bars, 'false' == vertical bars
   bool    audible ;          //* 'true'==beep when data-shift fails, 'false'==silent
} ;   // Chart
