//******************************************************************************
//* File       : KeyTest.cpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2013-2024 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 07-Jun-2024                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: KeyTest is a simple class definition for exercising the       *
//*              NCurses/NcDialog key input methods.                           *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.8.0)                                     *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE              *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.03 13-Oct-2019 Adjust indexing corresponding to changes in the     *
//*                        lookup tables.                                      *
//*                                                                            *
//* v: 0.00.02 07-Mar-2015 Add test for nckRESIZE to refresh if term resized.  *
//*                                                                            *
//* v: 0.00.01 01-Jun-2013 Transfer functionality from Dialog2, Test05.        *
//*                                                                            *
//******************************************************************************

#include "KeyTest.hpp"

//****************
//** Local Data **
//****************
typedef struct tm Tm ;                       // Linux Time structure

static const short dialogROWS = ktdROWS ;    // display lines
static const short dialogCOLS = ktdCOLS ;    // display columns
static const short DlgStatusX = dialogCOLS - 24 ; // begin status area X
static dspinData   dsData = { 1, 1000, 500, dspinINTEGER } ;
static const char  scrextLabel[] = { " KeyType  KeyCode  Display  KeyName"
                                     "                                " } ;
static const char  lfName[] = { "KeyCapture.txt" } ;



static InitCtrl ic[ktControlsDEFINED] =    // control initialization structures
{
   {  //* 32-bit keycode width  - - - - - - - - - - - - - - -    ktKeycode32RB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      2,                            // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "32-Bit keycode",             // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktKeycode16RB]            // nextCtrl:  link in next structure
   },
   {  //* 16-bit keycode width  - - - - - - - - - - - - - - -    ktKeycode16RB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[ktKeycode32RB].ulY + 1), // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "16-Bit keycode",             // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktNotransRB]              // nextCtrl:  link in next structure
   },
   {  //* 'No Translation' input mode - - - - - - - - - - - - -    ktNotransRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[ktKeycode16RB].ulY + 3), // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "No Translation",             // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktRawRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Raw' input mode  - - - - - - - - - - - - - - - - - - - -    ktRawRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[ktNotransRB].ulY + 1), // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "'Raw'",                      // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktUnbufRB]                // nextCtrl:  link in next structure
   },
   {  //* 'Unbuffered' input mode   - - - - - - - - - - - - - - - -  ktUnbufRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[ktRawRB].ulY + 1),   // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "'Unbuffered'",               // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktCookedRB]               // nextCtrl:  link in next structure
   },
   {  //* 'Cooked' input mode   - - - - - - - - - - - - - - - - -   ktCookedRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[ktUnbufRB].ulY + 1), // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "'Cooked'",                   // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktForeverRB]              // nextCtrl:  link in next structure
   },
   {  //* 'Wait Forever' key delay  - - - - - - - - - - - - - - -  ktForeverRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[ktCookedRB].ulY + 3),// ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Wait Forever",               // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktNowaitRB]               // nextCtrl:  link in next structure
   },
   {  //* 'No Wait' key delay - - - - - - - - - - - - - - - - - -   ktNowaitRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[ktForeverRB].ulY + 1), // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "No Wait",                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktVariableRB]             // nextCtrl:  link in next structure
   },
   {  //* 'Variable' key delay  - - - - - - - - - - - - - - - - - ktVariableRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[ktNowaitRB].ulY + 1),// ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Variable Delay",             // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktDelaySP]                // nextCtrl:  link in next structure
   },
   {  //* 'Delay' spinner (4-wide, 1-999 range)  - - - - - - - - -   ktDelaySP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ktVariableRB].ulY + 1), // ulY:       upper left corner in Y
      short(DlgStatusX + 8),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "mS",                         // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      false,                        // active:    initially inactive
      &ic[ktMouseRB]                // nextCtrl:  link in next structure
   },
   {  //* 'Mouse Enable'    - - - - - - - - - - - - - - - - - - -    ktMouseRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[ktDelaySP].ulY + 2), // ulY:       upper left corner in Y
      short(DlgStatusX + 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Enable Mouse\n(stable mode)", // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktHelpPB]                 // nextCtrl:  link in next structure
   },
   {  //* 'HELP' pushbutton - - - - -- - - - - - - - - - - - - - -    ktHelpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ktMouseRB].ulY + 3), // ulY:       upper left corner in Y
      short(DlgStatusX + 4),        // ulX:       upper left corner in X
      1,                            // lines:     control lines
      16,                           // cols:      control columns
      "      HELP      ",           // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktLogPB]                  // nextCtrl:  link in next structure
   },
   {  //* 'LOG' pushbutton  - - - - -- - - - - - - - - - - - - - - -   ktLogPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ktHelpPB].ulY + 2),  // ulY:       upper left corner in Y
      short(DlgStatusX + 4),        // ulX:       upper left corner in X
      1,                            // lines:     control lines
      16,                           // cols:      control columns
      " WRITE LOG FILE ",           // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktDonePB]                 // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - -- - - - - - - - - - - - - -      ktDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ktLogPB].ulY + 2),   // ulY:       upper left corner in Y
      (short)(DlgStatusX + 9),      // ulX:        upper left corner in X
      1,                            // lines:     control lines
      6,                            // cols:      control columns
      " DONE ",                     // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ktSextKeySE]              // nextCtrl:  link in next structure
   },
   {  //* Scroll Ext control to display key input details - - - -  ktSextKeySE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      short(dialogROWS - 2),        // lines:     control lines
      short(DlgStatusX - 1) ,       // cols:      control columns
      NULL,                         // dispText:  n/a
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      scrextLabel,                  // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    (special: edit not called)
      NULL                          // nextCtrl:  link in next structure
   },
} ;   // InitCtrl array


//*************************
//*      ~KeyTest         *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

KeyTest::~KeyTest ( void )
{

   if ( this->dp != NULL )                // close the window
      delete ( this->dp ) ;

}  //* End ~KeyTest() 

//*************************
//*        KeyTest        *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

KeyTest:: KeyTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp = NULL ;
   this->ssd.dispItems = ZERO ;
   this->dColor = nc.cyR ;
   this->keyDelay = nckdFOREVER ;
   this->wideKeys = true ;
   this->dataLog = false ;
   this->ktOpen = false ;
   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   ic[ktKeycode32RB].nColor = ic[ktKeycode16RB].nColor = ic[ktNotransRB].nColor   = 
   ic[ktRawRB].nColor       = ic[ktUnbufRB].nColor     = ic[ktCookedRB].nColor    = 
   ic[ktForeverRB].nColor   = ic[ktNowaitRB].nColor    = ic[ktVariableRB].nColor  = 
   ic[ktMouseRB].nColor     = ic[ktSextKeySE].nColor   = 
                              ic[ktSextKeySE].fColor   = this->dColor ;
   dsData.indiAttr          = this->dColor | ncbATTR ;
   ic[ktDelaySP].nColor     = ic[ktHelpPB].nColor      = ic[ktLogPB].nColor       = 
                              ic[ktDonePB].nColor      = nc.gyR ;
   ic[ktHelpPB].fColor      = ic[ktLogPB].fColor = ic[ktDonePB].fColor = nc.reG ;

   //* Initialize the display-data array for the dctSCROLLEXT control *
   for ( short i = ZERO ; i < ssdMAX ; i++ )
   {
      this->ssdtPtr[i] = this->ssdText[i] ;
      this->ssdAttr[i] = nc.bl ;
   }
   this->ssdAttr[0] |= ncuATTR ;
   this->ssd.dispText  = (const char**)this->ssdtPtr ;
   this->ssd.dispColor = this->ssdAttr ;
   this->ssd.dispItems = ZERO ;
   this->ssd.hlIndex   = ZERO ;
   this->ssd.hlShow    = false ;

   // Be sure key input delay is 'forever' and key translation is 'raw' *
   nc.SetKeyDelay ( nckdFOREVER ) ;
   nc.SetKeyProcState ( nckpRAW ) ;

   if ( (this->ktOpen = this->ktOpenDialog ()) != false )
   {
      this->ktReinitSSD () ;        // clear and re-set key-input area

   }  // OpenWindow()

}  //* End KeyTest() 

//*************************
//*     ktOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool KeyTest::ktOpenDialog ( void )
{
   short ctrY    = this->termRows/2 + 1,     // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Keyboard Input and Key Translation  ", // dialog title
                       ncltSINGLE,     // border line-style
                       this->dColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Draw a line separating input data from configuration controls *
      LineDef  lDef(ncltVERT, ncltSINGLE, ZERO, DlgStatusX, dialogROWS, dColor) ;
      this->dp->DrawLine ( lDef ) ;

      //* Section headers *
      this->dp->WriteString (  1, (DlgStatusX + 3), 
                               "Keycode Width", dColor | ncbATTR ) ;
      this->dp->WriteString (  5, (DlgStatusX + 3), 
                               "Key Preprocessing", dColor | ncbATTR ) ;
      this->dp->WriteString ( 11, (DlgStatusX + 3), 
                              "Key Delay", dColor | ncbATTR ) ;

      //* Group radio buttons *
      short XorGroup1[] = { ktKeycode32RB, ktKeycode16RB, -1 } ;
      short XorGroup2[] = { ktNotransRB, ktRawRB, ktUnbufRB, ktCookedRB, -1  } ;
      short XorGroup3[] = { ktForeverRB, ktNowaitRB, ktVariableRB, -1  } ;
      this->dp->GroupRadiobuttons ( XorGroup1 ) ;
      this->dp->GroupRadiobuttons ( XorGroup2 ) ;
      this->dp->GroupRadiobuttons ( XorGroup3 ) ;

      this->dp->RefreshWin () ;
      success = true ;
   }
   return success ;

}  //* End ktOpenDialog() *

//*************************
//*    ktDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool KeyTest::ktDialogOpened ( void )
{
   return this->ktOpen ;

}  //* End ktDialogOpened() *

//*************************
//*      ktInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void KeyTest::ktInteract ( void )
{
   if ( this->ktOpen )
   {
      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      icIndex = this->dp->PrevControl () ;
      bool     done = false ;             // loop control
      while ( ! done )
      {
         //************************************************
         //* If focus is currently on a Scrollext Control *
         //************************************************
         if ( ic[icIndex].type == dctSCROLLEXT )
         {
            this->ktGetUserInput () ;

            //* In 'cooked' or 'no-trans' mode, cursor keys are not recognized *
            //* so we must return to 'raw' mode in order for the user interface*
            //* to function.                                                   *
            ncKeyProc kp = nc.GetKeyProcState () ;
            if ( kp == nckpCOOKED || kp == nckpNO_TRANSLATION )
            {
               nc.SetKeyProcState ( nckpRAW ) ;          // set key processing state
               this->dp->SetRbGroupSelection ( ktRawRB ) ;// update the button group
            }
            Info.keyIn = nckSTAB ;     // Move to previous control
            Info.viaHotkey = false ;
         }

         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         else if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = this->dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {  //* User ready to exit test *
               if ( Info.ctrlIndex == ktDonePB )
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == ktHelpPB )
               {
                  while ( (icIndex = this->dp->NextControl ()) != ktSextKeySE ) ;
                  this->ktHelp () ;
                  while ( (icIndex = this->dp->PrevControl ()) != ktHelpPB ) ;
                  this->dp->SetScrollextText ( ktSextKeySE, this->ssd ) ;
               }
               else if ( Info.ctrlIndex == ktLogPB )
               {
                  this->ktLogPrompt ( true ) ;
               }
            }
            else
            { /* No button press, so nothing to do */ }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = this->dp->EditRadiobutton ( Info ) ;

            //* If radio button group selection has changed *
            if ( Info.dataMod != false )
            {
               //* If 'selected' button is a member of a button group *
               if ( Info.selMember < MAX_DIALOG_CONTROLS )
               {
                  switch ( Info.selMember )
                  {
                     case ktKeycode32RB:
                        this->wideKeys = true ;
                        break ;
                     case ktKeycode16RB:
                        this->wideKeys = false ;
                        break ;
                     case ktNotransRB:
                        nc.SetKeyProcState ( nckpNO_TRANSLATION ) ;
                        //* Cursor keys are not functional in No_Translation   *
                        //* mode, so move focus to the key input control.      *
                        while ( icIndex != ktSextKeySE ) 
                           icIndex = this->dp->NextControl () ;
                        icIndex = this->dp->PrevControl () ;   // compensate
                        break ;
                     case ktRawRB:
                        nc.SetKeyProcState ( nckpRAW ) ;
                        break ;
                     case ktUnbufRB:
                        nc.SetKeyProcState ( nckpUNBUFFERED ) ;
                        break ;
                     case ktCookedRB:
                        nc.SetKeyProcState ( nckpCOOKED ) ;
                        //* Cursor keys are not functional in Cooked     *
                        //* mode, so move focus to the key input control.*
                        while ( icIndex != ktSextKeySE ) 
                           icIndex = this->dp->NextControl () ;
                        icIndex = this->dp->PrevControl () ;   // compensate
                        break ;
                     case ktForeverRB:
                        nc.SetKeyDelay ( nckdFOREVER ) ;
                        this->keyDelay = nckdFOREVER ;
                        dp->ControlActive ( ktDelaySP, false ) ;  // disable spinner
                        break ;
                     case ktNowaitRB: 
                        nc.SetKeyDelay ( nckdNODELAY ) ;
                        this->keyDelay = nckdNODELAY ;
                        this->dp->ControlActive ( ktDelaySP, false ) ;// disable spinner
                        break ;
                     case ktVariableRB:
                        {
                        this->dp->ControlActive ( ktDelaySP, true ) ;// enable spinner
                        // (looks odd because of data-type conversions)
                        int dVal ;
                        this->dp->GetSpinnerValue ( ktDelaySP, dVal ) ;
                        this->keyDelay = ncKeyDelay(dVal) ;
                        nc.SetKeyDelay ( this->keyDelay ) ;
                        this->keyDelay = ncKeyDelay(nc.GetKeyDelay ()) ;
                        }
                        break ;
                  }
               }
               else  // independent radio buttons, not part of a button group
               {
                  if ( Info.ctrlIndex == ktMouseRB )
                  {
                     if ( Info.isSel )       // enable mouse events
                     {  // do not convert mouse events to keycodes
                        this->dp->meEnableStableMouse () ;
                        this->dp->meAutoConvert ( false ) ;
                     }
                     else
                        this->dp->meDisableMouse () ;
                  }
               }
            }
         }

         //*******************************************
         //* If focus is currently on a Spinner      *
         //*******************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = dp->EditSpinner ( Info ) ;

            //* If spinner value has changed *
            if ( Info.dataMod != false )
            {
               int dVal ;
               this->dp->GetSpinnerValue ( ktDelaySP, dVal ) ;
               this->keyDelay = ncKeyDelay(dVal) ;
               nc.SetKeyDelay ( keyDelay ) ;
               this->keyDelay = ncKeyDelay(nc.GetKeyDelay ()) ;
            }
         }

         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while()

      //* If user has specified that data be saved to a logfile *
      this->ktLog () ;

      // Return key input delay to 'forever' and key translation to 'raw' *
      nc.SetKeyDelay ( nckdFOREVER ) ;
      nc.SetKeyProcState ( nckpRAW ) ;
   }
   else
      { /* Caller is an idiot */ }

}  //* End ktInteract() *

//*************************
//*    ktGetUserInput     *
//*************************
//******************************************************************************
//* Gather user input for the key input test and display it in the dialog's    *
//* dctSCROLLEXT control.                                                      *
//*                                                                            *
//* The special key input sequence, 'stop' will exit the input loop and return *
//* to caller.                                                                 *
//* The special key input sequence, 'clear' will clear the input area.         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void KeyTest::ktGetUserInput ( void )
{
   //* Access to debugging data for key translation *
   const char **TransTable1, **TransTable2, **TransTable3 ;
   short tt1Min, tt1Max, tt2Min, tt2Max, tt3Entries ;
   nc.GetTransTable1 ( TransTable1, tt1Min, tt1Max ) ;
   nc.GetTransTable2 ( TransTable2, tt2Min, tt2Max ) ;
   nc.GetTransTable3 ( TransTable3, tt3Entries ) ;

   const wchar_t  templatePrint[] = { L"  %S   x%06X     %lc       --  " } ;
   const wchar_t  templateFunky[] = { L"  %S   x%06X     --      %s" } ;
   const wchar_t  templateExtnd[] = { L"  %S   x%06X     --      %s - %03X %03X" } ;
   const wchar_t  templateMouse[] = { L"  %S - %08Xh %02hd %03hd %02hd" } ;
   const wchar_t  templateEscsq[] = { L"  %S -" } ;
   const wchar_t  templateError[] = { L"  %S - Key wait timed out at %04hd mS" } ;
   const wchar_t* typeNames[] = { L"PRINT ", L"FUNKEY", L"EXTEND", L"MOUSE ", L"ESCSEQ", L"ERROR " } ;
   short keyDelay = nc.GetKeyDelay () ;

   //* Gather user's key input, format it and display it *
   gString  gs ;                       // for output formatting
   wkeyCode wk ;                       // user's key input
   short    stopCount = ZERO,          // loop control
            clearCount = ZERO ;        // controls clear-input-area
   bool     done = false ;             // loop control
   while ( ! done )
   {
      if ( keyDelay != nckdFOREVER )
      {  //* If key timeout comes at a rapid interval, slow down the process *
         int waitus = 500000 - (keyDelay * 1000) ;
         if ( waitus > ZERO )
            usleep ( waitus ) ;
      }
      if ( wideKeys )
      {
         this->dp->GetKeyInput ( wk ) ;

         //* If terminal-resize signal, refresh the display *
         if ( wk.type == wktFUNKEY && wk.key == nckRESIZE )
         {
            nc.RefreshScreen () ;
            this->dp->RefreshWin () ;
         }
      }
      else
      {
         short key16bit ;
         wk.type = nc.GetKeyInput ( key16bit ) ;
         wk.key = (wchar_t)key16bit ;
      }
      switch ( wk.type )
      {
         case wktPRINT:
            gs.compose( templatePrint, typeNames[0], &wk.key, &wk.key ) ;
            break ;
         case wktFUNKEY:
            {  // fool the compiler to create a variable inside a switch statement
            const char* nPtr ;
            if ( wk.key >= tt1Min && wk.key <= tt1Max )
               nPtr = TransTable1[wk.key] ;
            else if ( wk.key >= tt2Min && wk.key <= tt2Max )
               nPtr = TransTable2[wk.key - (tt1Max + 1)] ;
            else
               nPtr = "OVERFLOW" ;
            gs.compose( templateFunky, typeNames[1], &wk.key, nPtr ) ;
            }
            break ;
         case wktEXTEND:
            {
               short nIndex, charCount ;
               nIndex = wk.key ;
               if ( nIndex < ZERO || nIndex >= tt3Entries )
                  nIndex = tt3Entries - 1 ;
               const wchar_t* wPtr = nc.GetEscapeSequence ( charCount ) ;
               gs.compose( templateExtnd, typeNames[2], &wk.key, TransTable3[nIndex], 
                           &wPtr[0], &wPtr[1] ) ;
            }
            break ;
         case wktMOUSE:
            gs.compose( templateMouse, typeNames[3], 
                        &wk.mevent.eventType, &wk.mevent.ypos, 
                        &wk.mevent.xpos, &wk.mevent.zpos ) ;
            break ;
         case wktESCSEQ:
            gs.compose( templateEscsq, typeNames[4] ) ;
            {  // fool the compiler to create a variable inside a switch statement
               gString gstmp ;
               short charCount = ZERO ;
               const wchar_t* wPtr = nc.GetEscapeSequence ( charCount ) ;
               for ( short i = ZERO ; i < charCount ; i++ )
               {
                  gstmp.compose( L" %03X", &wPtr[i] ) ;
                  gs.append( gstmp.gstr() ) ;
               }
            }
            break ;
         case wktERR:
            gs.compose( templateError, typeNames[5], &keyDelay ) ;
            break ;
      }
      if ( this->ssd.dispItems < (ssdMAX - 6) )
      {
         gs.copy( (char*)ssdText[this->ssd.dispItems++], MAX_DIALOG_WIDTH ) ;
         ++this->ssd.hlIndex ;
         this->dp->SetScrollextText ( ktSextKeySE, this->ssd ) ;
      }
      else        // data array is full, reinitialize it
      {           // if log is enabled, also update the log
         this->ktReinitSSD () ;
      }

      //* Loop Control - if user spells "stop", do so *
      if ( stopCount > ZERO && wk.type == wktPRINT )
      {
         if (    (stopCount == 1 && wk.key == 't')
              || (stopCount == 2 && wk.key == 'o')
              || (stopCount == 3 && wk.key == 'p') )
         {
            if ( ++stopCount >= 4 )
            {  //* If in 'cooked' mode, eat the Enter key *
               if ( (nc.GetKeyProcState ()) == nckpCOOKED )
                  nc.GetKeyInput ( wk ) ;
               done = true ;
            }
         }
         else
            stopCount = ZERO ;
      }
      else if ( wk.type == wktPRINT && wk.key == 's' )
         stopCount = 1 ;
      //* If user spells "clear", clear the input area *
      else if ( clearCount > ZERO && wk.type == wktPRINT )
      {
         if (    (clearCount == 1 && wk.key == 'l')
              || (clearCount == 2 && wk.key == 'e')
              || (clearCount == 3 && wk.key == 'a')
              || (clearCount == 4 && wk.key == 'r') )
         {
            if ( ++clearCount >= 5 )
            {  //* If in 'cooked' mode, eat the Enter key *
               if ( (nc.GetKeyProcState ()) == nckpCOOKED )
                  nc.GetKeyInput ( wk ) ;
               //* Reinitialize the input area. If log enabled, also update log*
               this->ktReinitSSD () ;
               clearCount = ZERO ;
            }
         }
         else
            clearCount = ZERO ;
      }
      else if ( wk.type == wktPRINT && wk.key == 'c' )
         clearCount = 1 ;
   }  // while()

}  //* End ktGetUserInput() *

//*************************
//*      ktReinitSSD      *
//*************************
//******************************************************************************
//* Reinitialize the input area.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void KeyTest::ktReinitSSD ( void )
{
   const char * clrMsg = " Type the word 'stop' to exit input mode."
                         "                            " ;

   //* If log file is enabled AND there are some data to save, then save.*
   if ( this->ssd.dispItems > 1 )
      this->ktLog () ;

   //* Re-initialize the display input area *
   gString gs( clrMsg ) ;
   gs.copy( (char*)this->ssd.dispText[ZERO], MAX_DIALOG_WIDTH ) ;
   this->ssd.dispItems = 1 ;
   this->ssd.hlIndex = ZERO ;
   this->dp->SetScrollextText ( ktSextKeySE, this->ssd ) ;

}  //* End ktReinitSSD() *

//*************************
//*      ktLogPrompt      *
//*************************
//******************************************************************************
//* Ask user whether to save the key-input data to a log file.                 *
//*  OR                                                                        *
//* Post a message that log file is being written (or error msg)               *
//*                                                                            *
//* Input  : decide  : if 'true' prompt user for a response                    *
//*                    if 'false' display a message that log was written       *
//*                                                                            *
//* Returns: 'true'  if log is to be written, else 'false'                     *
//******************************************************************************

bool KeyTest::ktLogPrompt ( bool decide )
{
   if ( decide != false )         // ask user what to do
   {
      this->dataLog = false ;// assume no log file

      short dlines = 10,
            dcols  = 46,
            yoff   = 10,
            xoff   = 14 ;
      gString gs ;
      gs.compose( L"              %s              ", lfName ) ;
      const char* mList[] = 
      { "  Save Captured Key Data To Log File  ",
        " ",
        "Captured key data will be appended to the",
        "following file in the current directory: ",
        gs.ustr(),
        "If file does not exist, it will be created.",
        "                 Continue?",
        " ",
        NULL
      } ;
      const attr_t mAttr[] = 
      {
         nc.brcy, dColor, dColor, dColor, nc.bwB, 
         dColor, dColor, dColor
      } ;
      genDialog gd( mList, dColor, dlines, dcols, yoff, xoff, mAttr ) ;
      dataLog = this->dp->DecisionDialog ( gd ) ;
   }
   else                             // display write-to-log info
   {
      // (this never happens)
   }
   return this->dataLog ;

}  //* End ktLogPrompt() *

//*************************
//*         ktLog         *
//*************************
//******************************************************************************
//* Write key input data to log file. If target already exists, new data will  *
//* be appended. If target does not exist, it will be created.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void KeyTest::ktLog ( void )
{
#if 1    // NEW METHOD
   if ( this->dataLog != false )
   {
      ofstream ofs ( lfName, ofstream::out | ofstream::app ) ;
      if ( ofs.is_open() )                // if output file open
      {
         //* Write timestamp header *
         gString  ogs( "Key-input Log: " ), tgs ;
         Tm       tm ;                    // Linux time structure
         time_t   epoch = time ( NULL ) ; // seconds since the epoch
         if ( (localtime_r ( &epoch, &tm )) != NULL )
         {
            short month = tm.tm_mon + 1,
                  date = tm.tm_mday,
                  year = tm.tm_year + 1900 ;
            tgs.compose( L"%02hd/%02hd/%04hd @ %02hd:%02hd:%02hd", 
               &month, &date, &year, &tm.tm_hour, &tm.tm_min, &tm.tm_sec ) ;
            ogs.append( tgs.gstr() ) ;
         }
         ofs << ogs.ustr() << endl ;
         //* Write all the data EXCEPT the prompt line *
         for ( short i = 1 ; i < this->ssd.dispItems ; i++ )
            ofs << this->ssd.dispText[i] << endl ;
         ofs.close() ;                             // close log file
/* TEMP */ this->dp->DebugMsg ( " Key-input logfile written. ", 2, true ) ;
      }
      else
         this->dp->DebugMsg ( " Error: Unable to open key-input logfile! ", 3, true ) ;
   }
#else    // OLD METHOD
         ofstream ofs ( lfName, ofstream::out | ofstream::app ) ;
         if ( ofs.is_open() )                // if output file open
         {
            //* Write timestamp header *
            gString  ogs( "Key-input Log: " ), tgs ;
            Tm       tm ;                    // Linux time structure
            time_t   epoch = time ( NULL ) ; // seconds since the epoch
            if ( (localtime_r ( &epoch, &tm )) != NULL )
            {
               short month = tm.tm_mon + 1,
                     date = tm.tm_mday,
                     year = tm.tm_year + 1900 ;
               tgs.compose( L"%02hd/%02hd/%04hd @ %02hd:%02hd:%02hd", 
                  &month, &date, &year, &tm.tm_hour, &tm.tm_min, &tm.tm_sec ) ;
               ogs.append( tgs.gstr() ) ;
            }
            ofs << ogs.ustr() << endl ;
            for ( short i = 1 ; i < sData.dispItems ; i++ )
               ofs << sDataText[i] << endl ;
            ofs.close() ;                             // close backup file
            dp->DebugMsg ( " Key-input logfile written. ", 2, true ) ;
         }
         else
            dp->DebugMsg ( " Error: Unable to open key-input logfile! ", 3, true ) ;
#endif   // OLD METHOD
}  //* End ktLog() *

//*************************
//*        ktHelp         *
//*************************
//******************************************************************************
//* Display help for KeyTest user interface.                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void KeyTest::ktHelp ( void )
{
const short helpITEMS = 84 ;
static const char* HelpText[helpITEMS] = 
{//---------|---------|---------|---------|---------|---------|--------||
  " (Cursor keys to view)     KEY-INPUT HELP         (TAB key to close) ",
  " Special key Sequences ",
  " There are two special sequences which can be typed in the main      ",
  " window. The reason for these sequences is to avoid assigning        ",
  " functionality to any key during input of the test data.             ",
  "      Type the word 'stop' to leave the input window                 ",
  "      Type the word 'clear' to clear the input window                ",
  "    Column Headings    ",
  " Key Type: wktPRINT  printing characters                             ",
  "           wktFUNKEY function keys, control keys, cursor keys        ",
  "           wktMOUSE  mouse events (chewing, scurrying, pooping, etc) ",
  "           wktEXTEND extensions to ncurses key definitions           ",
  "           wktESCSEQ untranslated escape sequences                   ",
  "           wktERR    no data in input stream, or system-call error   ",
  " Key Code: hexadecimal keycode representing the key                  ",
  " Display : if key code is for a printing character, display the      ",
  "           character, else '--'                                      ",
  " Key Name: for non-printing characters, the name defined for key     ",
  "           in NCursesKeyDef.hpp                                      ",
  "     Keycode Width     ",
  " Choose between standard 32-bit key input and 16-bit (truncated) key ",
  " input. Note that internally, the NcDialog family of classes uses    ",
  " 32-bit (wchar_t) key input exclusively; however you may use 16-bit  ",
  " keycodes if you 'know' that the user will not enter any character   ",
  " with a keycode greater than 16-bits (not recommended).              ",
  "   Key Preprocessing   ",
  " Both the ncurses function library and the NcDialog key-input        ",
  " methods pre-filter and translate certain key sequences according    ",
  " to the selected key input 'mode'.                                   ",
  " Three modes defined for key input: Cooked, Unbuffered, and Raw.     ",
  " - In our estimation, 'Cooked' mode is actually over-cooked and      ",
  "   should not be used in applications.                               ",
  " - Use of 'Raw' or 'Unbuffered' mode is recommended.                 ",
  " - When using 'Raw' or 'Unbuffered' mode, the ncurses library that   ",
  "   ships with Linux/Unix systems captures and translates many escape ",
  "   sequences, and the NcDialog class key input methods capture and   ",
  "   translate most of those that remain. Actual keys available to the ",
  "   application are also dependent upon system and terminal-program   ",
  "   configuration.                                                    ",
  " - The No-Translation mode is available for key mapping and          ",
  "   experimentation. This mode disables all escape-sequence           ",
  "   translation so you can see what the system's keyboard interface   ",
  "   and/or terminal emulation program is actually transmitting.       ",
  "       Key Delay       ",
  " Key-input delay is a value between 1 millisecond and nckdMAXDELAY   ",
  " -OR- a member of enum ncKeyDelay.                                   ",
  " - Wait Forever (nckdFOREVER): key input methods block (sleep)       ",
  "   until a key is pressed.                                           ",
  " - No Delay (nckdNODELAY): key input methods return immediately if   ",
  "   no data waiting in the input queue.                               ",
  " - Variable Delay: Enter a delay value in milliseconds. Key input    ",
  "   methods will return either when key input is received OR when     ",
  "   specified time interval expires without key input.                ", 
  "     Enable Mouse      ",
  " In ncurses world, if reporting of mouse events is enabled, mouse    ",
  " event data are provided to the dialog through the keyboard input    ",
  " stream.                                                             ",
  " - The need for mouse support in a text-based dialog is debatable,   ",
  "   but the ncurses library provides basic mouse support in case      ",
  "   it is needed.                                                     ",
  " - Although the ncurses library mouse functions are a bit flakey,    ",
  "   the NcDialog-class mouse interface compensates for some of the    ",
  "   library's more surprising artifacts.                              ",
  " - Within this test, only single-click mouse events are reported.    ",
  "   Comprehensive testing of mouse-event functionality is handled in  ",
  "   one of the other test applications.                               ",
 //---------|---------|--||
  "         Help          ",
 //---------|---------|---------|---------|---------|---------|--------||
  " Display help information for this test (but you already knew that~).",
  "    Write Log File     ",
  " If enabled, write the keycode data displayed in the main window to a",
  " log file:  KeyCapture.txt                                           ",
  " - Currently-displayed keycode data and any subsequent data will be  ",
  "   saved to the log file when you exit the test (or incrementally    ",
  "   when you clear the input area).                                   ",
  " - If the file does not exist, it will be created.                   ",
  "   If the file already exists, the new data will be appended.        ",
  "         Done          ",
  " Guess what this button does....                                     ",
  "",
  "=====================================================================",
  " Obviously, this simple test does not require such extensive help;   ",
  " however, this is a test-and-demonstration program, so now you know  ",
  " how to implement a scrolling information window. You win!           ",
  "",
} ;
attr_t   tColor = nc.bl, hColor = nc.gybl | ncbATTR ;
static const attr_t HelpAttr[helpITEMS+2] = 
{
   nc.blG | ncuATTR,    // Title
   hColor,              // 'Special Key Sequences' sub-heading
   tColor, tColor, tColor, tColor, tColor, 
   hColor,              // 'Column Headings' sub-heading
   tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, 
   tColor, 
   hColor,              // 'Keycode Width' sub-heading
   tColor, tColor, tColor, tColor, tColor, 
   hColor,              // 'Key Preprocessing' sub-heading
   tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, 
   tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, 
   tColor, 
   hColor,              // 'Key Delay' sub-heading
   tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor,  
   hColor,              // 'Enable Mouse' sub-heading
   tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, tColor, 
   tColor, tColor, tColor, 
   hColor,              // 'Help' sub-heading
   tColor,
   hColor,              // 'Write Log File' sub-heading
   tColor, tColor, tColor, tColor, tColor, tColor, tColor, 
   hColor,              // 'Done' sub-heading
   tColor, tColor, 
   hColor, nc.blG, nc.blG, nc.blG, nc.blG, 
   tColor, tColor, // spares
} ;
   ssetData hData( HelpText, HelpAttr, helpITEMS, ZERO, false ) ;
   if ( (this->dp->SetScrollextText ( ktSextKeySE, hData )) == OK )
   {
      uiInfo Info ;
      this->dp->ViewScrollext ( Info ) ;
   }
   else
      this->dp->DebugMsg ( "SetScrollextText == ERR", 3, true ) ;

}  //* End ktHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


