//******************************************************************************
//* File       : CapTest.cpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2024 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NCurses.hpp           *
//* Date       : 07-Jun-2024                                                   *
//* Version    : (see Dialog2.cpp)                                             *
//*                                                                            *
//* Description: This module contains various tests for the NCurses/NcWindow/  *
//*              NcDialog screen-capture methods.                              *
//*                                                                            *
//******************************************************************************


// All definitions needed for your application
#include "GlobalDef.hpp"

//* Prototypes for local methods *
static short Cap_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime ) ;
static void  Cap_Messages ( void ) ;

//* Local Data *
static NcDialog* capDlg = NULL ;

enum ccControls : short    // Controls for CapComplex()
{
   ccdPB = ZERO,  // PushButton control
   ccsTB,         // Single-line Text Box
   ccgRB,         // Radio Button for green messages
   ccmRB,         // Radio Button for magenta messages
   ccyRB,         // Radio Button for yellow messages
   ccmTB,         // Multi-line Text Box
   roBB,          // Billboard (read-only)
   ccSP,          // Spinner control
   ccSL,          // Slider control
   ccSB,          // Scrollbox control
   ccSE,          // Scrollext control
   ccDD,          // DropDown control
   ccmMW,         // Menuwin control (main menu)
   ccsMW,         // Menuwin control (sub-menu)
   ccCONTROLS     // number of controls defined
} ;
static winPos tbPos, spPos, sbPos, mwPos ;  // message positions for CapComplex()
static short foodIndex = 1, drinkIndex = ZERO ; // Menu indices


//*************************
//*       CapSimple       *
//*************************
//******************************************************************************
//* A simple dialog window with only one control, a dctPUSHBUTTON.             *
//*                                                                            *
//* This dialog exercises the NcDialog CaptureDialog() and ccapFile() methods  *
//* used to capture screen shots of dialog windows (plain text and HTML).      *
//* It also provides the full range of color combinations available for HTML   *
//* output for verification of the 'screenshot-styles.css' CSS definitions.    *
//*                                                                            *
//*                                                                            *
//* Capture:                                                                   *
//* ALT+z       == capture dialog to a plain text file: 'capturedlg.txt'.      *
//* ALT+SHIFT+z == capture dialog to an HTML file: 'capturedlg.html'.          *
//*                The HTML output requires to screenshot-styles.css file.     *
//*                                                                            *
//* Input  : wp   : dialog position in Y/X                                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CapSimple ( const winPos wp )
{
   const short dialogROWS = 32 ;    // display lines
   const short dialogCOLS  = 116 ;  // display columns
   
   attr_t dColor = nc.blR,          // dialog background color
          bColor = nc.brbl ;        // dialog border color

   enum mdControls : short
   {
      DonePB = ZERO, // PushButton
      mdCONTROLS     // number of controls defined
   } ;

   InitCtrl ic[mdCONTROLS]  // array of dialog control initialization objects
   {
      {  //* 'DONE' Pushbutton - - - - - - - - - - - - - - - - - DonePB *
         dctPUSHBUTTON,             // control type
         rbtTYPES,                  // rbSubtype: (n/a)
         false,                     // rbSelect:  (n/a)
         short(dialogROWS - 3),     // ulY:       upper-left Y
         short(dialogCOLS / 2 - 4), // ulX:       upper-left X
         1,                         // lines:     height
         8,                         // cols:      width
         "  ^DONE  ",               // dispText:  
         nc.gyR,                    // nColor:    non-focus color
         nc.reG,                    // fColor:    focus color
         tbPrint,                   // filter:    (n/a)
         NULL,                      // label:     (n/a)
         ZERO,                      // labY:      (n/a)
         ZERO,                      // labX       (n/a)
         ddBoxTYPES,                // exType:    (n/a)
         1,                         // scrItems:  (n/a)
         1,                         // scrSel:    (n/a)
         NULL,                      // scrColor:  (n/a)
         NULL,                      // spinData:  (n/a)
         true,                      // active:    can receive focus
         NULL
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( 
      dialogROWS,     // number of display lines
      dialogCOLS,     // number of display columns
      wp.ypos,        // Y offset from upper-left of terminal 
      wp.xpos,        // X offset from upper-left of terminal 
      "  NcDialog - Capture Simple Dialog  ", // dialog title
      ncltDUAL,       // border line-style
      bColor,         // border color attribute
      dColor,         // interior color attribute
      ic              // pointer to list of control definitions
      ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteParagraph ( dialogROWS - 3, 2,
                           "This color array validates the CSS definitions in\n"
                           "             'screenshot-styles.css'             ", nc.brbl ) ;
      dp->WriteParagraph ( dialogROWS - 4, dialogCOLS - 24,
                           "Capture Dialog To File\n"
                           "  ALT+z == plain text\n"
                           "  ALT+SHIFT+z == html", nc.brbl | ncbATTR ) ;
      winPos wp( 1, 3 ) ;                    // header
      dp->WriteString ( wp, "    PLAIN         BOLD        UNDERLINE      "
                            "REVERSE     Bld + Rev     Bld + Undr    "
                            "Rev + Undr   Bld+Rev+Undr", bColor ) ;
      ++wp.ypos ;                            // No attribute bits
      dp->WriteString ( wp.ypos++, wp.xpos, "  bw  Text  ", nc.bw ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  re  Text  ", nc.re ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gr  Text  ", nc.gr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  br  Text  ", nc.br ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bl  Text  ", nc.bl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  ma  Text  ", nc.ma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cy  Text  ", nc.cy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gy  Text  ", nc.gy ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bbk Text  ", nc.bbk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bre Text  ", nc.bre ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bgr Text  ", nc.bgr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bbr Text  ", nc.bbr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bbl Text  ", nc.bbl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bma Text  ", nc.bma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bcy Text  ", nc.bcy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  bgy Text  ", nc.bgy ) ;

      wp = { 2, short(wp.xpos + 14) } ;      // Bold attribute
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwB Text  ", nc.bwB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reB Text  ", nc.reB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grB Text  ", nc.grB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brB Text  ", nc.brB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blB Text  ", nc.blB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maB Text  ", nc.maB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyB Text  ", nc.cyB ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyB Text  ", nc.gyB ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkB Text  ", nc.bbk | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breB Text  ", nc.bre | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrB Text  ", nc.bgr | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrB Text  ", nc.bbr | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblB Text  ", nc.bbl | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaB Text  ", nc.bma | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyB Text  ", nc.bcy | ncbATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyB Text  ", nc.bgy | ncbATTR ) ;

      wp = { 2, short(wp.xpos + 14) } ;      // Underline attribute
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwU Text  ", nc.bwU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reU Text  ", nc.reU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grU Text  ", nc.grU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brU Text  ", nc.brU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blU Text  ", nc.blU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maU Text  ", nc.maU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyU Text  ", nc.cyU ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyU Text  ", nc.gyU ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkU Text  ", nc.bbk | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breU Text  ", nc.bre | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrU Text  ", nc.bgr | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrU Text  ", nc.bbr | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblU Text  ", nc.bbl | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaU Text  ", nc.bma | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyU Text  ", nc.bcy | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyU Text  ", nc.bgy | ncuATTR ) ;

      wp = { 2, short(wp.xpos + 14) } ;      // Reverse attribute
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwR Text  ", nc.bwR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reR Text  ", nc.reR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grR Text  ", nc.grR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brR Text  ", nc.brR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blR Text  ", nc.blR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maR Text  ", nc.maR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyR Text  ", nc.cyR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyR Text  ", nc.gyR ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkR Text  ", nc.bbk | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breR Text  ", nc.bre | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrR Text  ", nc.bgr | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrR Text  ", nc.bbr | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblR Text  ", nc.bbl | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaR Text  ", nc.bma | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyR Text  ", nc.bcy | ncrATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyR Text  ", nc.bgy | ncrATTR ) ;

      wp = { 2, short(wp.xpos + 14) } ;      // Bold + Reverse attributes
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwG Text  ", nc.bwG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reG Text  ", nc.reG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grG Text  ", nc.grG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brG Text  ", nc.brG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blG Text  ", nc.blG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maG Text  ", nc.maG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyG Text  ", nc.cyG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyG Text  ", nc.gyG ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkG Text  ", nc.bbk | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breG Text  ", nc.bre | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrG Text  ", nc.bgr | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrG Text  ", nc.bbr | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblG Text  ", nc.bbl | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaG Text  ", nc.bma | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyG Text  ", nc.bcy | ncgATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyG Text  ", nc.bgy | ncgATTR ) ;

      wp = { 2, short(wp.xpos + 14) } ;      // Bold + Underline attributes
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwX Text  ", nc.bwB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reX Text  ", nc.reB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grX Text  ", nc.grB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brX Text  ", nc.brB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blX Text  ", nc.blB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maX Text  ", nc.maB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyX Text  ", nc.cyB | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyX Text  ", nc.gyB | ncuATTR ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkX Text  ", nc.bbk | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breX Text  ", nc.bre | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrX Text  ", nc.bgr | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrX Text  ", nc.bbr | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblX Text  ", nc.bbl | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaX Text  ", nc.bma | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyX Text  ", nc.bcy | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyX Text  ", nc.bgy | ncbATTR | ncuATTR ) ;
                                                                            
      wp = { 2, short(wp.xpos + 14) } ;      // Reverse + Underline attributes
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwY Text  ", nc.bwR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reY Text  ", nc.reR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grY Text  ", nc.grR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brY Text  ", nc.brR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blY Text  ", nc.blR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maY Text  ", nc.maR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyY Text  ", nc.cyR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyY Text  ", nc.gyR | ncuATTR ) ;
      ++wp.ypos ;                                                 
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkY Text  ", nc.bbk | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breY Text  ", nc.bre | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrY Text  ", nc.bgr | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrY Text  ", nc.bbr | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblY Text  ", nc.bbl | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaY Text  ", nc.bma | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyY Text  ", nc.bcy | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyY Text  ", nc.bgy | ncrATTR | ncuATTR ) ;
                                                                  
      wp = { 2, short(wp.xpos + 14) } ;      // Bold + Reverse + Underline attributes
      dp->WriteString ( wp.ypos++, wp.xpos, "  bwZ Text  ", nc.bwG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  reZ Text  ", nc.reG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  grZ Text  ", nc.grG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  brZ Text  ", nc.brG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  blZ Text  ", nc.blG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  maZ Text  ", nc.maG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  cyZ Text  ", nc.cyG | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  gyZ Text  ", nc.gyG | ncuATTR ) ;
      ++wp.ypos ;                                                 
      dp->WriteString ( wp.ypos++, wp.xpos, " bbkZ Text  ", nc.bbk | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " breZ Text  ", nc.bre | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgrZ Text  ", nc.bgr | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bbrZ Text  ", nc.bbr | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bblZ Text  ", nc.bbl | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bmaZ Text  ", nc.bma | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bcyZ Text  ", nc.bcy | ncbATTR | ncrATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " bgyZ Text  ", nc.bgy | ncbATTR | ncrATTR | ncuATTR ) ;

      ++wp.ypos ; wp.xpos = 3 ;              // Custom Color Combinations 1
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " rebk Text  ", nc.rebk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grbk Text  ", nc.grbk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brbk Text  ", nc.brbk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blbk Text  ", nc.blbk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " mabk Text  ", nc.mabk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cybk Text  ", nc.cybk ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gybk Text  ", nc.gybk ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 2
      dp->WriteString ( wp.ypos++, wp.xpos, " bkre Text  ", nc.bkre ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grre Text  ", nc.grre ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brre Text  ", nc.brre ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blre Text  ", nc.blre ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " mare Text  ", nc.mare ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cyre Text  ", nc.cyre ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gyre Text  ", nc.gyre ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 3
      dp->WriteString ( wp.ypos++, wp.xpos, " bkgr Text  ", nc.bkgr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " regr Text  ", nc.regr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brgr Text  ", nc.brgr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blgr Text  ", nc.blgr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " magr Text  ", nc.magr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cygr Text  ", nc.cygr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gygr Text  ", nc.gygr ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 4
      dp->WriteString ( wp.ypos++, wp.xpos, " bkbr Text  ", nc.bkbr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " rebr Text  ", nc.rebr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grbr Text  ", nc.grbr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blbr Text  ", nc.blbr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " mabr Text  ", nc.mabr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cybr Text  ", nc.cybr ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gybr Text  ", nc.gybr ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 5
      dp->WriteString ( wp.ypos++, wp.xpos, " bkbl Text  ", nc.bkbl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " rebl Text  ", nc.rebl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grbl Text  ", nc.grbl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brbl Text  ", nc.brbl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " mabl Text  ", nc.mabl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cybl Text  ", nc.cybl ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gybl Text  ", nc.gybl ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 6
      dp->WriteString ( wp.ypos++, wp.xpos, " bkma Text  ", nc.bkma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " rema Text  ", nc.rema ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grma Text  ", nc.grma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brma Text  ", nc.brma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blma Text  ", nc.blma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cyma Text  ", nc.cyma ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gyma Text  ", nc.gyma ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 7
      dp->WriteString ( wp.ypos++, wp.xpos, " bkcy Text  ", nc.bkcy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " recy Text  ", nc.recy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grcy Text  ", nc.grcy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brcy Text  ", nc.brcy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blcy Text  ", nc.blcy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " macy Text  ", nc.macy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " gycy Text  ", nc.gycy ) ;

      wp.ypos -= 8; wp.xpos += 14 ;          // Custom Color Combinations 7
      dp->WriteString ( wp.ypos++, wp.xpos, " bkgy Text  ", nc.bkgy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " regy Text  ", nc.regy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " grgy Text  ", nc.grgy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " brgy Text  ", nc.brgy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " blgy Text  ", nc.blgy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " magy Text  ", nc.magy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, " cygy Text  ", nc.cygy ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "  -  -  -   ", dColor  ) ;

      #if 0    // TEMP
      int vmaj = NCURSES_VERSION_MAJOR,
          vmin = NCURSES_VERSION_MINOR,
          vpat = NCURSES_VERSION_PATCH ;
      gString gs( "ncurses version: %d.%d.%d", &vmaj, &vmin, &vpat ) ;
      dp->WriteString ( 16, 2, gs, mColor ) ;
      gs = curses_version();
      dp->WriteString ( 17, 2, gs, mColor ) ;
      #endif   // TEMP
      dp->RefreshWin () ;           // make everything visible

      //* Establish a callback method to capture the dialog *
      dp->EstablishCallback ( &Cap_ControlUpdate ) ;
      capDlg = dp ;                 // module access to dialog

      //* Interact with user *
      uiInfo  Info ;                // user interface data returned here
      short   icIndex = ZERO ;      // index of control with input focus
      bool    done = false ;        // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( !Info.viaHotkey )
               icIndex = dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;

            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == DonePB )
                  done = true ;
            }
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   else
   {  // The most likely cause of error is 
      // that the terminal window is too small.
      nc.WriteString ( wp, 
                       "Unable to open dialog window!", nc.re, true ) ;
      nckPause();
   }

   if ( dp != NULL )                // close the window
      delete ( dp ) ;

}  //* End CapSimple() *

//*************************
//*      CapComplex       *
//*************************
//******************************************************************************
//* This is a complex dialog which includes at least one instance of each      *
//* dialog control type.                                                       *
//*                                                                            *
//* This dialog exercises the NcDialog CaptureDialog() and ccapFile() methods  *
//* used to capture screen shots of dialog windows (plain text and HTML).      *
//*                                                                            *
//* Capture:                                                                   *
//* ALT+z       == capture dialog to a plain text file: 'capturedlg.txt'.      *
//* ALT+SHIFT+z == capture dialog to an HTML file: 'capturedlg.html'.          *
//*                The HTML output requires to screenshot-styles.css file.     *
//*                                                                            *
//* Input  : wp   : dialog position in Y/X                                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CapComplex ( const winPos wp )
{
   const short dialogROWS = 32 ;    // display lines
   const short dialogCOLS  = 116 ;  // display columns
   const char* bbText =
         "There once was a man from Poughkeepsie\n"
         "Who often got rather tipsy.\n"
         "He was roundly ashamed;\n"
         "Twas his wife that he blamed,\n"
         "For she was quite prone to be ditzy." ;
   const attr_t bbAttr[5] = { nc.brR, nc.cyR, nc.grR, nc.gyR, nc.maR } ;
   const char* mlText = 
         "Ilsa, I'm no good at being noble, but it doesn't take much to see "
         "that the problems of three little people don't amount to a hill of "
         "beans in this crazy world. Someday you'll understand that. Now, now...  "
         "Here's looking at you, kid! -- Rick Blaine, Casablanca (1942)" ;

   //* Data for rtScrSB, rtScrSE and rtDrpDD *
   const short ddITEMS = 16, sbITEMS = 8, sbWIDTH = 21 ;
   const char sbData[ddITEMS][sbWIDTH + 10] = 
   {  //* (8 avians and 8 mammals) *
      " goose             ",
      " chicken           ",
      " duck              ",
      " pigeon            ",
      " sparrrow          ",
      " magpie            ",
      " robin             ",
      " dove              ",
      " cow               ",
      " horse             ",
      " sheep             ",
      " hog               ",
      " goat              ",
      " rabbit            ",
      " moose             ",
      " mouse             ",
   } ;
   static const char* seData[] = 
   {
      sbData[8],  sbData[9],  sbData[10], sbData[11], 
      sbData[12], sbData[13], sbData[14], sbData[15],  
   } ;
   static attr_t seColors[sbITEMS] = 
   { nc.bw, nc.re, nc.gr, nc.br, nc.bl, nc.ma, nc.cy, nc.gy } ;

   //* Contents of Main Menu:       *
   static const short mmITEMS = 7, mmWIDTH = 30 ;
   static const char mmData[mmITEMS][mmWIDTH] = 
   {
   "  Drin^k Menu               >",
   "  Baked ^fish with onions    ",
   "  Beef ^brisket with garlic  ",
   "  Sauteed ^peas and carrots  ",
   "  ^Green beans               ",
   "  Fruit ^salad               ",
   "  Mutton ^chops              ",
   } ;
   
   //* Contents of Submenu: *
   static const short smITEMS = 4, smWIDTH = 18 ;
   static const char smData[smITEMS][smWIDTH + 8] = 
   {
      "  ^Beer         ",
      "  ^Wine         ",
      "  ^Tea          ",
      "  ^Coffee       ",
   } ;

   static attr_t monoColorB[2]  =  { attrDFLT, nc.brR } ; // menuwin data colors
   static attr_t monoColorG[2]  =  { attrDFLT, nc.grR } ; // scrollbox data colors

   //* Setup data for the Spinner control *
   const dspinData dsData( -1, 10, 2, dspinINTEGER, nc.brR ) ;

   //* Setup data for the Slider control *
   sliderData slData( 6.0, 64.0, 32.0, true, false ) ;

   attr_t dColor = nc.blR,          // dialog background color
          bColor = nc.brbl,         // dialog border color
          fColor = nc.reR,          // color of control with focus
          nColor = nc.gyR ;         // color of control without focus

   InitCtrl ic[ccCONTROLS]  // array of dialog control initialization objects
   {
      {  //* 'DONE' Pushbutton    - - - - - - - - - - - - - - - - - - -  ccdPB *
         dctPUSHBUTTON,             // control type
         rbtTYPES,                  // rbSubtype: (n/a)
         false,                     // rbSelect:  (n/a)
         short(dialogROWS - 4),     // ulY:       upper-left Y
         short(dialogCOLS / 2 - 5), // ulX:       upper-left X
         3,                         // lines:     height
         10,                        // cols:      width
         "  ^DONE  ",               // dispText:  
         nColor,                    // nColor:    non-focus color
         fColor,                    // fColor:    focus color
         tbPrint,                   // filter:    (n/a)
         NULL,                      // label:     (n/a)
         ZERO,                      // labY:      (n/a)
         ZERO,                      // labX       (n/a)
         ddBoxTYPES,                // exType:    (n/a)
         1,                         // scrItems:  (n/a)
         1,                         // scrSel:    (n/a)
         NULL,                      // scrColor:  (n/a)
         NULL,                      // spinData:  (n/a)
         true,                      // active:    can receive focus
         &ic[ccsTB]                 // nextCtrl:  next control
      },
      {  //* 'Single-line' Textbox   - - - - - - - - - - - - - - - - - - ccsTB *
         dctTEXTBOX,               // control type
         rbtTYPES,                 // rbSubtype: (n/a)
         false,                    // rbSelect:  (n/a)
         2,                        // ulY:       upper-left Y
         2,                        // ulX:       upper-left X
         1,                        // lines:     height
         27,                       // cols:      width
         "Here's lookin' at you, Kid!", // dispText:
         nc.cyR,                   // nColor:    non-focus color
         fColor,                   // fColor:    focus color
         tbPrint,                  // filter:    any printing character
         "^Write a message:",      // label:     
         -1,                       // labY:      
         ZERO,                     // labX       
         ddBoxTYPES,               // exType:    (n/a)
         1,                        // scrItems:  (n/a)
         1,                        // scrSel:    (n/a)
         NULL,                     // scrColor:  (n/a)
         NULL,                     // spinData:  (n/a)
         true,                     // active:    can receive focus
         &ic[ccgRB]                // nextCtrl:  next control
      },
      {  //* 'Green Message' Radio Button - - - - - - - - - - - - - - -  ccgRB *
         dctRADIOBUTTON,           // type:      
         rbtS3s,                   // rbSubtype: standard3
         false,                    // rbSelect:  dflt sel
         short(ic[ccsTB].ulY + 2), // ulY:       upper-left Y
         ic[ccsTB].ulX,            // ulX:       upper-left X
         1,                        // lines:     (n/a)
         0,                        // cols:      (n/a)
         NULL,                     // dispText:  (n/a)
         nc.grR,                   // nColor:    non-focus color
         fColor,                   // fColor:    focus color
         tbPrint,                  // filter:    (n/a)
         "^Green Messages",        // label:
         ZERO,                     // labY:      
         5,                        // labX       
         ddBoxTYPES,               // exType:    (n/a)
         1,                        // scrItems:  (n/a)
         1,                        // scrSel:    (n/a)
         NULL,                     // scrColor:  (n/a)
         NULL,                     // spinData:  (n/a)
         true,                     // active:    can receive focus
         &ic[ccmRB]                // nextCtrl:  next control
      },
      {  //* 'Magenta Message' Radio Button - - - - - - - - - - - - - -  ccmRB *
         dctRADIOBUTTON,           // type:      
         rbtS3s,                   // rbSubtype: standard3
         false,                    // rbSelect:  dflt sel
         short(ic[ccgRB].ulY + 1), // ulY:       upper-left Y
         ic[ccgRB].ulX,            // ulX:       upper-left X
         1,                        // lines:     (n/a)
         0,                        // cols:      (n/a)
         NULL,                     // dispText:  (n/a)
         nc.maR,                   // nColor:    non-focus color
         fColor,                   // fColor:    focus color
         tbPrint,                  // filter:    (n/a)
         "M^agenta Messages",      // label:
         ZERO,                     // labY:      
         5,                        // labX       
         ddBoxTYPES,               // exType:    (n/a)
         1,                        // scrItems:  (n/a)
         1,                        // scrSel:    (n/a)
         NULL,                     // scrColor:  (n/a)
         NULL,                     // spinData:  (n/a)
         true,                     // active:    can receive focus
         &ic[ccyRB]                // nextCtrl:  next control
      },
      {  //* 'Yellow Message' Radio Button    - - - - - - - - - - - - -  ccyRB *
         dctRADIOBUTTON,           // type:      
         rbtS3s,                   // rbSubtype: standard3
         true,                     // rbSelect:  dflt sel
         short(ic[ccmRB].ulY + 1), // ulY:       upper-left Y
         ic[ccmRB].ulX,            // ulX:       upper-left X
         1,                        // lines:     (n/a)
         0,                        // cols:      (n/a)
         NULL,                     // dispText:  (n/a)
         nc.brbk,                  // nColor:    non-focus color
         fColor,                   // fColor:    focus color
         tbPrint,                  // filter:    (n/a)
         "^Yellow Messages",       // label:
         ZERO,                     // labY:      
         5,                        // labX       
         ddBoxTYPES,               // exType:    (n/a)
         1,                        // scrItems:  (n/a)
         1,                        // scrSel:    (n/a)
         NULL,                     // scrColor:  (n/a)
         NULL,                     // spinData:  (n/a)
         true,                     // active:    can receive focus
         &ic[ccmTB]                // nextCtrl:  end of list
      },
      {  //* 'Multi-line' Textbox  - - - - - - - - - - - - - - - - - -   ccmTB *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[ccsTB].ulY,                // ulY:       upper left corner in Y
         short(ic[ccsTB].ulX + ic[ccsTB].cols + 3), // ulX:       upper left corner in X
         5,                            // lines:     control lines
         40,                           // cols:      control columns
         mlText,                       // dispText:  initial display text
         nc.cyR,                       // nColor:    non-focus color
         fColor,                       // fColor:    focus color
         tbPrint,                      // filter:    any printing character
         "Multi-Line ^Text Box (all printing chars)", // label:     
         -1,                           // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[roBB]                     // nextCtrl:  link in next structure
      },
      {  //* 'Billboard' (read-only)  - - - - - - - - - - - - - - - - - - roBB *
         dctBILLBOARD,                 // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[ccmTB].ulY,                // ulY:       upper left corner in Y
         short(ic[ccmTB].ulX + ic[ccmTB].cols + 2), // ulX:       upper left corner in X
         5,                            // lines:     control lines
         40,                           // cols:      control columns
         bbText,                       // dispText:  initial display text
         nc.brR,                       // nColor:    non-focus color
         nc.brR,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Billboard Control (read only)", // label:     control label
         -1,                           // labY:      label offset Y
         ZERO,                         // labX       label offset X
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         (attr_t*)bbAttr,              // scrColor:  if specified, attribute array
         NULL,                         // spinData:  (n/a)
         false,                        // active:    view-only
         &ic[ccSP]                     // nextCtrl:  link in next structure
      },
      { //* Spinner Control   - - - - - - - - - - - - - - - - - - - - - - ccSP *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[ccyRB].ulY + 4),     // ulY:       upper-left Y
         short(ic[ccyRB].ulX + 30),    // ulX:       upper-left X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nColor,                       // nColor:    non-focus color
         fColor,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Presidential Approval Rating", // label:     
         ZERO,                         // labY:      
         -30,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData,                      // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[ccSL]                     // nextCtrl:  link in next structure
      },
      { //* Slider Control   - - - - - - - - - - - - - - - - - - - - - -  ccSL *
         dctSLIDER,                    // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[roBB].ulY + ic[roBB].lines + 3), // ulY: upper left corner in Y
         ic[roBB].ulX,                 // ulX: upper left corner in X
         7,                            // lines:     control rows
         2,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.bkcy,                      // nColor:    non-focus color
         (nc.recy ^ ncbATTR),          // fColor:    focus color
         tbPrint,                      // filter:    any printing character
         "Drin^k Size",                // label:     label text
         ZERO,                         // labY:      label offset
         3,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         (dspinData*)&slData,          // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ccSB]                     // nextCtrl:  link in next structure
      },
      {  //* 'Avians' Scrollbox - - - - - - - - - - - - - - - - - -       ccSB *
         dctSCROLLBOX,           // type:
         rbtTYPES,               // rbSubtype: (na)
         false,                  // rbSelect:  (n/a)
         short(ic[ccyRB].ulY + 7),// ulY: upper left corner in Y
         ic[ccyRB].ulX,          // ulX: upper left corner in X
         10,                     // lines:     control lines
         sbWIDTH,                // cols:      control columns
         (char*)&sbData,         // dispText:  n/a
         nColor,                 // nColor:    non-focus border color
         fColor,                 // fColor:    focus border color
         tbPrint,                // filter:    (n/a)
         "  A^vians  ",          // label:
         ZERO,                   // labY:      offset from control's ulY
         ZERO,                   // labX       offset from control's ulX
         ddBoxTYPES,             // exType:    (n/a)
         sbITEMS,                // scrItems:  number of elements in text/color arrays
         3,                      // scrSel:    index of initial highlighted element
         monoColorG,             // scrColor:  (n/a)
         NULL,                   // spinData:  (n/a)
         true,                   // active:    allow control to gain focus
         &ic[ccSE]               // nextCtrl:  link in next structure
      },
      {  //* 'Mammals' Scrollext  - - - - - - - - - - - - - - - - -       ccSE *
         dctSCROLLEXT,                 // type:
         rbtTYPES,               // rbSubtype: (na)
         false,                  // rbSelect:  (n/a)
         ic[ccSB].ulY,           // ulY: upper left corner in Y
         short(ic[ccSB].ulX + ic[ccSB].cols + 2),// ulX: upper left corner in X
         10,                     // lines:     control lines
         sbWIDTH,                // cols:      control columns
         NULL,                   // dispText:  n/a
         nColor,                 // nColor:    non-focus border color
         fColor,                 // fColor:    focus border color
         tbPrint,                // filter:    (n/a)
         "  Mamma^ls  ",         // label:     
         ZERO,                   // labY:      offset from control's ulY
         ZERO,                   // labX       offset from control's ulX
         ddBoxTYPES,             // exType:    (n/a)
         sbITEMS,                // scrItems:  number of elements in text/color arrays
         ZERO,                   // scrSel:    index of initial highlighted element
         seColors,               // scrColor:  multi-color data
         NULL,                   // spinData:  (n/a)
         true,                   // active:    allow control to gain focus
         &ic[ccDD]               // nextCtrl:  link in next structure
      },
      {  //* 'Zoo Critters' DropDown  - - - - - - - - - - - - - - - - -   ccDD *
         dctDROPDOWN,            // type:
         rbtTYPES,               // rbSubtype: (na)
         false,                  // rbSelect:  (n/a)
         ic[ccSE].ulY,           // ulY: upper left corner in Y
         short(ic[ccSE].ulX + ic[ccSE].cols + 2), // ulX:       upper left corner in X
         ddITEMS / 2,            // lines:     control lines
         sbWIDTH,                // cols:      control columns
         (char*)&sbData,         // dispText:  n/a
         nColor,                 // nColor:    non-focus border color
         fColor,                 // fColor:    focus border color
         tbPrint,                // filter:    (n/a)
         "D^ropdown Control",    // label:     
         3,                      // labY:      offset from control's ulY
         2,                      // labX       offset from control's ulX
         ddBoxDOWN,              // exType:    (n/a)
         ddITEMS,                // scrItems:  number of elements in text/color arrays
         ZERO,                   // scrSel:    index of initial highlighted element
         monoColorG,             // scrColor:  single-color data display
         NULL,                   // spinData:  (n/a)
         true,                   // active:    allow control to gain focus
         &ic[ccmMW]              // nextCtrl:  link in next structure
      },
      {  //* 'Main Menu' Menuwin  - - - - - - - - - - - - - - - - - - -  ccmMW *
         dctMENUWIN,                   // type:
         rbtTYPES,               // rbSubtype: (n/a)
         false,                  // rbSelect:  (n/a)
         short(ic[ccDD].ulY + 5), // ulY:       upper left corner in Y
         short(ic[ccDD].ulX + 1), // ulX:  upper left corner in X
         1,                      // lines:     (n/a)
         mmWIDTH,                // cols:      control columns, 'expanded' state
         (const char*)&mmData,   // dispText:  array of menu items
         nc.reR,        // nColor: non-focus border color / focus 'collapsed' color
         nc.rebk,       // fColor: focus border color / non-focus 'collapsed' color
         tbPrint,                // filter:    (n/a)
         " Today's ^Specials ",  // label:     label text == 'collapsed state' text
         ZERO,                   // labY:      (n/a)
         ZERO,                   // labX       (n/a)
         ddBoxTYPES,             // exType:    (n/a)
         mmITEMS,                // scrItems:  number of menu items
         ZERO,                   // scrSel:    (n/a)
         monoColorB,             // scrColor:  array of color attributes for menu items
         NULL,                   // spinData:  (n/a)
         true,                   // active:    allow control to gain focus
         &ic[ccsMW]              // nextCtrl:  link in next structure
      },
      {  //* 'Sub-Menu' Menuwin - - - - - - - - - - - - - - - - - - - -  ccsMW *
         dctMENUWIN,                   // type:
         rbtTYPES,               // rbSubtype: (n/a)
         false,                  // rbSelect:  (n/a)
         short(ic[ccmMW].ulY + 2), // ulY: upper left corner in Y
         short(ic[ccmMW].ulX + ic[ccmMW].cols - 1),// ulX: upper left corner in X
         1,                      // lines:     (n/a)
         smWIDTH,                // cols:      control columns, 'expanded' state
         (const char*)&smData,   // dispText:  array of menu items
         nc.reR,        // nColor: non-focus border color / focus 'collapsed' color
         nc.rebk,       // fColor: focus border color / non-focus 'collapsed' color
         tbPrint,                // filter:    (n/a)
         NULL,                   // label:     submenu has no label
         ZERO,                   // labY:      (n/a)
         ZERO,                   // labX       (n/a)
         ddBoxTYPES,             // exType:    (n/a)
         smITEMS,                // scrItems:  number of menu items
         ZERO,                   // scrSel:    (n/a)
         monoColorB,             // scrColor:  array of color attributes for menu items
         NULL,                   // spinData:  (n/a)
         false,                  // active:    submenu initially inactive
         NULL                    // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( 
      dialogROWS,     // number of display lines
      dialogCOLS,     // number of display columns
      wp.ypos,        // Y offset from upper-left of terminal 
      wp.xpos,        // X offset from upper-left of terminal 
      "  NcDialog - Capture Dialog with Controls ", // dialog title
      ncltDUAL,       // border line-style
      bColor,         // border color attribute
      dColor,         // interior color attribute
      ic              // pointer to list of control definitions
      ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Attach sub-menu to main menu control *
      short smList[] = { ccsMW, -1 } ; 
      if ( (dp->AttachMenuwinSubmenus ( ccmMW, smList )) != OK )
         dp->UserAlert () ;

      //* Create an Exclusive-OR Radio Button group *
      short XorGroup[] = { ccgRB, ccmRB, ccyRB, -1 } ; 
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Initialize dctSCROLLEXT contents *
      ssetData ssd( seData, seColors, sbITEMS, 2, true ) ;
      dp->SetScrollextText ( ccSE, ssd ) ;

      //* Initialize the dctSLIDER control *
      dp->SliderAutoReport ( ccSL, true, 
                             (ic[ccSL].ulY + ic[ccSL].labY + 1),
                             (ic[ccSL].ulX + ic[ccSL].labX),
                             "%2.0lf Ounces", dColor, false ) ;

      dp->EnablePushbuttonBorder ( ccdPB ) ;      // Give the button a border
      dp->SetTextboxCursor ( ccmTB, tbcpAPPEND ) ;// Cursor position for edit

      dp->WriteParagraph ( dialogROWS - 4, dialogCOLS - 24,
                           "Capture Dialog To File\n"
                           "  ALT+z == plain text\n"
                           "  ALT+SHIFT+z == html", nc.brbl | ncbATTR ) ;

      dp->WriteString ( ic[ccSB].ulY + ic[ccSB].lines, ic[ccSB].ulX + 2,
                        "Scrollbox Control      Scrollext Control", 
                        dColor ) ;
      dp->WriteString ( ic[ccmMW].ulY + 1, ic[ccmMW].ulX + 1,
                        "Menuwin Control", dColor ) ;

      //* Position and caption the messages *
      capDlg = dp ;                 // module access to dialog
      tbPos = { (short)(ic[ccyRB].ulY + 2), ic[ccyRB].ulX } ;
      tbPos = dp->WriteString ( tbPos, " Your Text Message:", nc.cybr ) ;
      ++tbPos.xpos ;
      spPos = { (short)(ic[ccSP].ulY + 1), ic[ccyRB].ulX } ;
      spPos = dp->WriteString ( spPos, " Your Real Opinion:", nc.cybr ) ;
      ++spPos.xpos ;
      sbPos = { short(ic[ccmMW].ulY + 3), ic[ccmMW].ulX } ;
      sbPos = dp->WriteParagraph ( sbPos, " Flying Critters:\n\n"
                                          "Walking Critters:\n\n"
                                          "    Zoo Critters:", nc.cybr ) ;
      sbPos.ypos -= 4 ; ++sbPos.xpos ;
      mwPos = { ic[ccmMW].ulY, short(ic[ccmMW].ulX + ic[ccmMW].cols + 1) } ;
      mwPos = dp->WriteParagraph ( mwPos, "      What's For Lunch?      \n", nc.cybr ) ;
      Cap_Messages () ; // draw message(s) in specified color

      dp->RefreshWin () ;           // make everything visible

      //* Establish a callback method to capture the dialog *
      dp->EstablishCallback ( &Cap_ControlUpdate ) ;

      //* Interact with user *
      uiInfo  Info ;                // user interface data returned here
      short   icIndex = ZERO ;      // index of control with input focus
      bool    done = false ;        // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( !Info.viaHotkey )
               icIndex = dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;

            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == ccdPB )
                  done = true ;
            }
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;   // ignore via-hotkey
            icIndex = dp->EditTextbox ( Info ) ;

            if ( Info.dataMod != false )
            {
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;

            if ( Info.dataMod != false )
            {  
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         else if ( ic[icIndex].type == dctSPINNER )
         {
            Info.viaHotkey = false ;   // ignore via-hotkey
            icIndex = dp->EditSpinner ( Info ) ;

            if ( Info.dataMod != false )
            {
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         else if ( ic[icIndex].type == dctSLIDER )
         {
            Info.viaHotkey = false ;   // ignore via hotkey
            icIndex = dp->EditSlider ( Info ) ;
            if ( Info.dataMod != false )
            {
            }
         }

         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            Info.viaHotkey = false ;   // ignore via-hotkey
            icIndex = dp->EditScrollbox ( Info ) ;
            if ( Info.dataMod != false )
            {
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;   // ignore via-hotkey
            icIndex = dp->EditScrollext ( Info ) ;
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         else if ( ic[icIndex].type == dctDROPDOWN )
         {  //* If via hotkey, expand immediately. *
            icIndex = dp->EditDropdown ( Info ) ;
            if ( Info.dataMod != false )
            {
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         else if ( ic[icIndex].type == dctMENUWIN )
         {  //* If via hotkey, expand immediately. *
            icIndex = dp->EditMenuwin ( Info ) ;
            if ( Info.dataMod != false )
            {  //* Remember menu selection *
               if ( Info.ctrlIndex == ccmMW && Info.selMember > ZERO )
                  foodIndex = Info.selMember ;
               else if ( Info.ctrlIndex == ccsMW )
                  drinkIndex = Info.selMember ;
               Cap_Messages () ; // re-draw message(s) in specified color
            }
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   else
   {  // The most likely cause of error is 
      // that the terminal window is too small.
      nc.WriteString ( wp,
                       "Unable to open dialog window!", nc.re, true ) ;
      nckPause();
   }

   if ( dp != NULL )                // close the window
      delete ( dp ) ;

}  //* End capComplex() *

//*************************
//*   Cap_ControlUpdate   *
//*************************
//******************************************************************************
//* Callback method for capturing the dialog to a file.                        *
//* Used by both CapSimple() and CapComplex().                                 *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************

static short Cap_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* nothing to do *
   }

   //***********************
   //* Standard processing *
   //***********************
   //* If command to capture dialog's text/attribute data to a file.*
   //* This tests the CaptureDialog() method.                       *
   if ( (wkey.type == wktEXTEND) && 
        ((wkey.key == nckA_Z) || (wkey.key == nckAS_Z) || (wkey.key == nckAS_S)) )
   {
      const char* const capText = "./capturedlg.txt" ;
      const char* const capHtml = "./capturedlg.html" ;
      bool fmt = bool(wkey.key != nckA_Z) ;
      short capStatus ;

      //* Special, simplified HTML capture for insertion into Texinfo         *
      //* documentation. References CSS definition file 'infodoc-styles.css'. *
      if ( wkey.key == nckAS_S )
         capStatus = capDlg->CaptureDialog ( capHtml, fmt, 
                  false, "../Texinfo/infodoc-styles.css", 1, false, nc.blR ) ;

      //* Standard, plain-text or table-based HTML capture *
      else
         capStatus = capDlg->CaptureDialog ( (fmt ? capHtml : capText), fmt ) ;

      //* Report results of the capture *
      gString gs ;
      gs.compose( L"Screen Capture to: '%s'", 
                  (fmt ? capHtml : capText) ) ;
      if ( capStatus != OK )
         gs.append( L" FAILED!" ) ;
      capDlg->DebugMsg ( gs.ustr(), 2, true ) ;
   }

   return OK ;

}  //* End Cap_ControlUpdate() *

//*************************
//*     Cap_Messages      *
//*************************
//******************************************************************************
//* Called by CapComplex() to update the various messages written in the       *
//* parent window.                                                             *
//*                                                                            *
//* This method makes intimate assumptions about the layout of the dialog      *
//* window, so keep it current.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void  Cap_Messages ( void )
{
   const char* spMsg[21] = 
   {
      "You've got to be freakin' kidding me!",           // -1
      "I thought Chenny was dead.",                      //  0
      "Jeb, is that you?",                               //  1
      "About what we've come to expect.",                //  2
      "Ms. Clinton, I presume.",                         //  3
      "Nuclear war, anyone?",                            //  4
      "I guess we are luckier than we deserve.",         //  5
      "Yes, I voted for her.",                           //  6
      "I'd vote for her again.",                         //  7
      "The Once and Future King.",                       //  8
      "I'd follow him into hell.",                       //  9
      "Walks on water."                                  // 10
   } ;
   const char* sbMsg[16] = 
   {
      "Geese honk",
      "Chickens cluck",
      "Ducks quack",
      "Pigeons coo",
      "Sparrows chirp",
      "Magpies imitate",
      "Robins sing",
      "Doves chortle",
      "Cows moo",
      "Horses neigh",
      "Sheep baaa",
      "Hogs grunt",
      "Goats bleat",
      "Rabbits squeal",
      "Moose bellow",
      "Mice squeak"
   } ;
   const char* mwmMsg[] = 
   {
      " n/a ",
      "    baked fish",
      "  beef brisket",
      "peas & carrots",
      "   green beans",
      "   fruit salad",
      "  mutton chops"
   } ;
   const char* mwsMsg[] = 
   {
      "beer  ",
      "wine  ",
      "tea   ",
      "coffee"
   } ;

   gString  msgOut ;    // data capture
   int      spValue ;   // 'spinner' message index
   attr_t   mColor ;    // color attribute for writing messages
   short    cIndex,     // 'radiobutton' group selection item
            sbIndex,    // 'scrollbox' item index
            seIndex,    // 'scrollext' item index
            ddIndex ;   // 'drowdown' item index

   cIndex = capDlg->GetRbGroupSelection ( ccgRB ) ;
   mColor = cIndex == ccgRB ? nc.grR : 
            cIndex == ccmRB ? nc.maR : nc.brbl ;

   capDlg->GetTextboxText ( ccsTB, msgOut ) ;
   capDlg->ClearLine ( tbPos.ypos, false, tbPos.xpos ) ;
   if ( msgOut.gschars() > 1 )
      capDlg->WriteString ( tbPos, msgOut, mColor ) ;
   capDlg->GetSpinnerValue ( ccSP, spValue ) ;
   ++spValue ;
   capDlg->ClearArea ( spPos.ypos, spPos.xpos, 1, 50 ) ;
   capDlg->WriteString ( spPos, spMsg[spValue], mColor ) ;
   sbIndex = capDlg->GetScrollboxSelect ( ccSB ) ;
   seIndex = capDlg->GetScrollextSelect ( ccSE ) + 8 ;
   ddIndex = capDlg->GetDropdownSelect ( ccDD ) ;
   capDlg->ClearLine ( sbPos.ypos, false, sbPos.xpos ) ;
   capDlg->ClearLine ( sbPos.ypos + 2, false, sbPos.xpos ) ;
   capDlg->ClearLine ( sbPos.ypos + 4, false, sbPos.xpos ) ;
   msgOut.compose( "%s\n\n%s\n\n%s", sbMsg[sbIndex], sbMsg[seIndex], sbMsg[ddIndex] ) ;
   capDlg->WriteParagraph ( sbPos, msgOut, mColor ) ;
   //* Caller tracks the indices of of the menu controls.*
   capDlg->ClearLine ( mwPos.ypos, false, mwPos.xpos ) ;
   msgOut.compose ( "%s  -and-  %s", mwmMsg[foodIndex], mwsMsg[drinkIndex] ) ;
   capDlg->WriteString ( mwPos, msgOut, mColor ) ;

   capDlg->RefreshWin () ;    // make the changes visible

}  //* End Cap_Messages() *

