//********************************************************************************
//* File       : NcdControlSB.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 12-Feb-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogScrollbox class               *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogScrollbox object's functionality.                     *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//*********************
//* Local Definitions *
//*********************


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**           DialogScrollbox CLASS.            **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**   related to DialogScrollbox-class access.  **
      //*************************************************

//*************************
//*   DialogScrollbox     *
//*************************
//******************************************************************************
//* Constructor for DialogScrollbox class object.                              *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: constructors implicitly return a pointer to the object            *
//******************************************************************************

DialogScrollbox::DialogScrollbox ( InitCtrl* iPtr )
{
   this->type   = dctSCROLLBOX ;       // set control type
   this->ulY    = iPtr->ulY ;          // transfer the initialization data
   this->ulX    = iPtr->ulX ;
   this->lines  = iPtr->lines ;
   this->cols   = iPtr->cols ;
   this->nColor = iPtr->nColor ;       // color of border when control does not have focus
   this->fColor = iPtr->fColor ;       // color of border when control has focus
   this->labY   = iPtr->labY ;         // label offsets
   this->labX   = iPtr->labX ;
   this->bIndex = iPtr->scrSel ;       // index for initial position of highlight
   this->bItems = iPtr->scrItems ;     // number of display strings
   this->active = iPtr->active ;       // indicates whether control can be selected

   //* Copy the label text to data member, wLabel and initialize bHotkey.      *
   //* Note that if a multi-line label was specified for a label embedded      *
   //* within the control's border, we strip the newlines from the string as   *
   //* an indication to the user that he/she/it screwed up.                    *
   this->InitHotkey ( iPtr->label, bool(this->labY == ZERO && this->labX == ZERO) ) ;

   //* Default values for all other data members *
   this->groupCode = ZERO ;   // scroll box controls are not grouped (not used)
   this->sdPaint  = false ;   // selection data not yet displayed
   this->rtlContent = false ; // left-to-right language content by default
         //* NOTE: cdConnect class bConnect values are all initialized to 'false'

   //* Allocate space for strings, pointers, and color attributes.             *
   //* This is a single memory allocation, divided into sections.              *
   //* Note that our initial allocation is likely larger than we actually need,*
   //* so we give back the excess, below.                                      *
   void* blkptr ;
   char* dString ;
   int bytesNeeded = bItems * (this->cols) * 4 ;      // space for display strings
   bytesNeeded    += bItems * sizeof(char*) ;         // space for string pointers
   bytesNeeded    += bItems * sizeof(attr_t) ;        // space for color attributes
   if ( (blkptr = calloc ( bytesNeeded, 1 )) != NULL )// allocate a memory block
   {
      //* Pointer to array of menu-item pointers *
      this->bText   = (char**)blkptr ;
      //* Point to array of color attributes *
      this->bColor  = (attr_t*)((char*)this->bText + (this->bItems*sizeof(char*))) ;
      //* Point to array of display-item data strings *
      dString       = (char*)((char*)this->bColor + (this->bItems*sizeof(attr_t))) ;

      //* Copy display data and color attributes to local storage. Normalizing *
      //* the number of display columns required by each display item.         *
      const char* cPtr = iPtr->dispText ;    // pointer to source text
      short reqCols  = this->cols - 2,       // display columns available
            ubytes ;                         // for item-width calculations
      gString gs ;                           // text formatting

      //* For each display item *
      for ( short iCount = ZERO ; iCount < this->bItems ; iCount++ )
      {  //* Initialize pointer to formatted display string *
         this->bText[iCount] = dString ;

         //* Copy the display item to a formatting buffer.                     *
         //* Note that string length is limited to number of display columns.  *
         gs = cPtr ;

         //* If caller has correctly formatted the display item, just copy it. *
         if ( gs.gscols() == reqCols )
         {
            ubytes = gs.utfbytes () ;
            gs.copy( this->bText[iCount], ubytes ) ;
            cPtr += ubytes ;                 // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += ubytes ;              // point to next target buffer
         }
         //* If a display item is too short, copy what we have and pad on right*
         else if ( gs.gscols() < reqCols )
         {
            ubytes = gs.utfbytes () ;
            while ( gs.gscols() < reqCols )
               gs.append( L' ' ) ;
            gs.copy( this->bText[iCount], gs.utfbytes() ) ;
            cPtr += ubytes ;                 // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += gs.utfbytes() ;       // point to next target buffer
         }
         //* Source string is too long - truncate it to fit control window.    *
         else  // gs.gscols() > reqCols
         {
            gs.limitCols(reqCols) ;          // limit string to available width
            ubytes = gs.utfbytes () ;
            gs.copy( this->bText[iCount], ubytes ) ;
            cPtr += ubytes - 1 ;             // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += ubytes ;              // point to next target buffer
         }

         //* Establish color of display string *
         if ( iPtr->scrColor == NULL )
         {  //* If caller provided no color data, set default.*
            //*      (this prevents a segmentation fault)     *
            this->bColor[iCount] = nc.bw ;
         }
         else if ( iPtr->scrColor[0] != attrDFLT )
         {  //* Each display item has its own color attribute *
            this->bColor[iCount] = iPtr->scrColor[iCount] ;
         }
         else
         {  //* Same color attribute for all display items *
            this->bColor[iCount] = iPtr->scrColor[1] ;
         }
      }  // end for(;;)

      //* Return unneeded portion of memory allocation to the heap.            *
      // Programmer's Note: realloc() MAY copy the block to a new location,    *
      // even when shrinking block, necessitating adjustement of our pointers. *
      void* newblkPtr = realloc ( blkptr, (dString - (char*)blkptr) ) ;
      if ( newblkPtr != blkptr )
      {
         this->bText   = (char**)newblkPtr ;
         this->bColor  = (attr_t*)((char*)this->bText + (this->bItems*sizeof(char*))) ;
         dString       = (char*)((char*)this->bColor + (this->bItems*sizeof(attr_t))) ;
         for ( short iCount = ZERO ; iCount < this->bItems ; iCount++ ) // for each display item
         {
            this->bText[iCount] = dString ;
            while ( *dString != NULLCHAR )
               ++dString ;
            ++dString ;
         }
      }
   }  // calloc()

   //* Memory allocaton error will be reported in control window *
   else
   {
      //* Set all variables to 'safe' values *
      bPtr     = NULL ;
      bText    = NULL ;
      bColor   = NULL ;
      bIndex   = -1 ;
      bItems   = -1 ;
      *wLabel  = NULLCHAR ;
      labY     = ZERO ;
      labX     = ZERO ;
   }

   //* Instantiate the underlying window object *
   bPtr = new NcWindow ( lines, cols, ulY, ulX ) ;    // outer window contains border
   wPtr = new NcWindow ( lines-2, cols-2, ulY+1, ulX+1 ) ; // inner window contains data

}  //* End DialogScrollbox() *

//*************************
//*   ~DialogScrollbox    *
//*************************
//******************************************************************************
//* Destructor for DialogScrollbox class object.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogScrollbox::~DialogScrollbox ( void )
{
   //* Release dynamic allocation for display data *
   free ( (void*)bText ) ;

   //* Close the underlying windows *
   delete this->bPtr ;
   delete this->wPtr ;

}  //* End ~DialogScrollbox() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Draw a dialog scroll box control i.e. open the window previously           *
//* instantiated by a call to DialogScrollbox().                               *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogScrollbox::OpenControl ( bool hasFocus )
{
attr_t   color = hasFocus ? this->fColor : this->nColor ;
short    result = ERR ;

   //* Set interior color of the scrolling-data window.   *
   //* This is done in case the scrolling data do not fill*
   //* all the rows of the scroll window. The color used  *
   //* is the color of the first display item.            *
   this->wPtr->SetInteriorColor ( this->bColor[ZERO] ) ;

   if (   (this->bPtr->OpenWindow () == OK ) 
       && (result = this->wPtr->OpenWindow ()) == OK )
   {
      //* Draw a border around the control.                                    *
      //* If label is not in 'title' position OR no label specified, draw a    *
      //* plain border. Else draw border with specified title.                 *
      if ( this->labY != ZERO || this->labX != ZERO || *this->wLabel == NULLCHAR )
         this->bPtr->DrawBorder ( color ) ;
      else
      {
         gString gs( this->wLabel ) ;  // convert from wchar_t to UTF-8
         short xOffset = this->bPtr->DrawBorder ( color, gs.ustr(), 
                                                  ncltSINGLE, this->rtlContent ) ;
         //* If there is a hotkey in our title *
         if ( this->bHotkey.hotkey != false )
         {
            //* If a hotkey is defined AND the label is in the 'title'  *
            //* position (centered in top row of control's window),     *
            //* adjust the hotkey character's X offset, and redisplay it*
            if ( this->rtlContent )
               this->bHotkey.xoffset = xOffset - this->bHotkey.xoffset ;
            else
               this->bHotkey.xoffset += xOffset ;
            this->bPtr->WriteChar ( this->labY, this->labX+this->bHotkey.xoffset, 
                                    this->bHotkey.hotchar, color | ncuATTR ) ;
         }
      }

      //* If data strings provided, display them *
      if ( this->bText != NULL && !this->sdPaint )
      {
         this->wPtr->PaintData ( (const char**)bText, bColor, bItems, 
                                 bIndex, true, this->rtlContent ) ;
         this->sdPaint = true ;
      }
      else
         this->wPtr->WriteString ( 1, 1, "INVALID DISPLAY DATA!", fColor ) ;
      this->RefreshControl () ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data in a Scroll Box window.                                    *
//* (refreshes the control's window)                                           *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - If label is external to control, it has alreadly been drawn in the       *
//*   parent dialog's space.                                                   *
//* - If label is embedded within the control, we draw it here.                *
//* - If our 'rtlContent' member == false, then draw the label AND the         *
//*   control's contents as LTR.                                               *
//* - If our 'rtlContent' member != false, then draw the label AND the         *
//*   control's contents as RTL. This is less flexible than other controls     *
//*   whose label and contents can be separately set as LTR or RTL; however,   *
//*   we don't have access to the parent dialog's data, so we must use the     *
//*   'rtlCont' flag for both label and contents.                              *
//******************************************************************************

void DialogScrollbox::RedrawControl ( bool hasFocus )
{
attr_t   borderColor = hasFocus ? fColor : nColor ;

   //* Redraw and refresh the border (either plain border or border with title)*
   if ( this->labY != ZERO || this->labX != ZERO || *this->wLabel == NULLCHAR )
      this->bPtr->DrawBorder ( borderColor ) ;
   else
   {
      gString gs( this->wLabel ) ;  // convert from wchar_t to UTF-8
      this->bPtr->DrawBorder ( borderColor, gs.ustr(), ncltSINGLE, this->rtlContent ) ;
      if ( this->bHotkey.hotkey != false )
      {
         //* Write the hotkey character with underline *
         this->bPtr->WriteChar ( this->labY, this->labX+this->bHotkey.xoffset, 
                                 this->bHotkey.hotchar, borderColor | ncuATTR ) ;
      }
   }

   if ( this->bConnect.connection != false )
   {
      wchar_t oChar ;                  // single-character output
      oChar = wcsLTEE ;
      if ( this->bConnect.ul2Left != false )
         this->bPtr->WriteChar ( ZERO, ZERO, oChar, borderColor ) ;
      if ( this->bConnect.ll2Left != false )
         this->bPtr->WriteChar ( this->lines-1, ZERO, oChar, borderColor ) ;
      oChar = wcsRTEE ;
      if ( this->bConnect.ur2Right != false )
         this->bPtr->WriteChar ( ZERO, this->cols-1, oChar, borderColor ) ;
      if ( this->bConnect.lr2Right != false )
         this->bPtr->WriteChar ( this->lines-1, this->cols-1, oChar, borderColor ) ;
      oChar = wcsTTEE ;
      if ( this->bConnect.ul2Top != false )
         this->bPtr->WriteChar ( ZERO, ZERO, oChar, borderColor ) ;
      if ( this->bConnect.ur2Top != false )
         this->bPtr->WriteChar ( ZERO, this->cols-1, oChar, borderColor ) ;
      oChar = wcsBTEE ;
      if ( this->bConnect.ll2Bot != false )
         this->bPtr->WriteChar ( this->lines-1, ZERO, oChar, borderColor ) ;
      if ( this->bConnect.lr2Bot != false )
         this->bPtr->WriteChar ( this->lines-1, this->cols-1, oChar, borderColor ) ;
   }
   this->bPtr->RefreshWin () ;
   this->wPtr->RepaintData () ;              // redraw and refresh the scrolling data 
   this->wPtr->RefreshWin () ;

}  //* End RedrawControl()

//*************************
//*    RefreshControl     *
//*************************
//******************************************************************************
//* Refresh the display of the specified dialog control object.                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogScrollbox::RefreshControl ( void )
{

   this->bPtr->RefreshWin () ;            // Refresh the outer (border) window
   this->wPtr->RefreshWin () ;            // refresh the inner (data) window

}  //* End RefreshControl() *

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data have already been initialized                         *
//******************************************************************************
//* Formatting is done in the OpenControl() method.                            *
//******************************************************************************

short DialogScrollbox::SetOutputFormat ( bool rtlFormat )
{
   short status = ERR ;
   //* If data have not yet been initialized, set the output format.*
   if ( this->sdPaint == false )
   {
      this->rtlContent = rtlFormat ;
      status = OK ;
   }
   return status ;

}  //* End SetOutputFormat() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogDropdown-class FUNCTIONALITY.      **
      //*************************************************

//*************************
//*     EditScrollbox     *
//*************************
//******************************************************************************
//* If control with input focus == dctSCROLLBOX, call this method to get user's*
//* key input. Allows user to edit which item in the control is 'selected'.    *
//* Returns to caller when user has finished selection (selection may or may   *
//* not have changed), -OR- when control has lost focus due to a hotkey press. *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//*           (highlight IS visible on return)                                 *
//******************************************************************************

short NcDialog::EditScrollbox ( uiInfo& info )
{
DialogScrollbox* cp ;
short    origSelection ;
bool     done = false ;       // loop control

   //* Be sure the control with focus is actually a DialogScrollbox *
   if ( this->dCtrl[this->currCtrl]->type == dctSCROLLBOX )
   {
      cp = (DialogScrollbox*)this->dCtrl[this->currCtrl] ;
      origSelection = cp->bIndex ;
   }
   else                  // control is not a scroll box i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
      done = true ;     // don't enter the loop
   }

   //**************************
   //* Interact with the user *
   //**************************
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* Done scrolling. Move to next control *
      if ( (info.wk.type == wktFUNKEY) && (info.wk.key == nckTAB || info.wk.key == nckRIGHT) )
      {
         info.ctrlType = dctSCROLLBOX ;      // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.dataMod = true ;               // indicate selection change (but see below)
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // returning to caller
      }
      //* Done scrolling. Move to previous control *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckLEFT) )
      {
         info.ctrlType = dctSCROLLBOX ;      // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckSTAB ;              // focus to move backward to previous control
         info.dataMod = true ;               // indicate selection change (but see below)
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // returning to caller
      }

      //* If active 'select' of currently-highlighted item *
      else if (   (info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckENTER || info.wk.key == nckpENTER))
               || (info.wk.type == wktPRINT && info.wk.key == SPACE) )
      {
         info.ctrlType = dctSCROLLBOX ;      // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckENTER ;             // active selection, not just leaving the control
         info.dataMod = true ;               // indicate selection change (but see below)
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                 // returning to caller
      }
      
      //* Scroll up or down through list *
      // Note: nckLEFT and nckRIGHT are scroll keys also, but we eliminated 
      //       them in above tests because they don't have a function in 
      //       a vertical scrolling control.
      else if ( (cp->wPtr->IsScrollKey ( info.wk )) != false )
      {
         cp->bIndex = cp->wPtr->ScrollData ( info.wk.key ) ;
      }

      else if ( info.wk.type == wktFUNKEY && info.wk.key == nckESC )   // abort selection
      {
         //* return highlight to initial state *
         wchar_t sKey = cp->bIndex > origSelection ? nckUP : nckDOWN ;
         while ( cp->bIndex != origSelection )
            cp->bIndex = cp->wPtr->ScrollData ( sKey ) ;
         if ( ExternalControlUpdate != NULL )   // in case caller is tracking the scroll
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;

         info.ctrlType = dctSCROLLBOX ;      // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.dataMod = false ;              // data unchanged
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // returning to caller
      }

      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* Ignore hotkey selection of this scroll box,                 *
            //* else, make the indicated control the current/active control *
            if ( hotIndex != this->currCtrl )
            {
               //* Indicate the results of our edit *
               info.ctrlType = dctSCROLLBOX ;   // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.dataMod = true ;            // indicate selection change (but see below)
               info.selMember = cp->bIndex ;    // current member is the 'selected' member
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               done = true ;
            }
         }
         else
         {
            // key input value is not valid in this context, ignore it
         }
      }
      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()

   //* If 'selected' item has not changed, reset dataMod flag *
   if ( cp->bIndex == origSelection )
      info.dataMod = false ;
   return this->currCtrl;

}  //* End EditScrollbox() *

//*************************
//*  GetScrollboxSelect   *
//*************************
//******************************************************************************
//* Returns the index of the highlighted item in the specified dctSCROLLBOX    *
//* control.                                                                   *
//*                                                                            *
//* Input  : cIndex : index number of source control                           *
//*                                                                            *
//* Returns: index of 'selected' (highlighted) item                            *
//*          or ERR if:                                                        *
//*            a) specified control is not of type dctSCROLLBOX                *
//******************************************************************************

short NcDialog::GetScrollboxSelect ( short cIndex )
{
   short itemIndex = ERR ;

   //* If caller has passed a valid index number *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl)
        && (dCtrl[cIndex]->type == dctSCROLLBOX) )
   {
      DialogScrollbox* cp = (DialogScrollbox*)(dCtrl[cIndex]) ;
      itemIndex = cp->bIndex ;
   }
   return itemIndex ;

}  //* End GetScrollboxSelect() *

//*************************
//*  SetScrollboxSelect   *
//*************************
//******************************************************************************
//* Set 'selected' (highlighted) item in specified dctSCROLLBOX control.       *
//*                                                                            *
//* Input  : cIndex : index number of target control                           *
//*          selMember : index of item to be set as 'selected'                 *
//*                                                                            *
//* Returns: index of 'selected' member                                        *
//*          or ERR if:                                                        *
//*            a) invalid item index specified                                 *
//*            b) specified control currently has the input focus              *
//*            c) specified control is not of type dctSCROLLBOX                *
//******************************************************************************

short NcDialog::SetScrollboxSelect ( short cIndex, short selMember )
{
   short itemIndex = ERR ;

   //* If caller has passed a valid index number *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl)
        && (dCtrl[cIndex]->type == dctSCROLLBOX)
        && (cIndex != this->currCtrl) )
   {
      DialogScrollbox* cp = (DialogScrollbox*)(dCtrl[cIndex]) ;

      //* If a non-highlighted item has been specified for selection *
      //* move highlight to that position.                           *
      if ( selMember >= ZERO && selMember < cp->bItems )
      {
         while ( cp->bIndex != selMember )
         {
            if ( selMember > cp->bIndex )    // scroll down
               cp->bIndex = cp->wPtr->ScrollData ( nckDOWN ) ;
            else                             // scroll up
               cp->bIndex = cp->wPtr->ScrollData ( nckUP ) ;
         }
         //* Set currently-highlighted item as selection *
         itemIndex = cp->bIndex ;
      }
   }
   return itemIndex ;

}  //* End SetScrollboxSelect() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

