//********************************************************************************
//* File       : NcWindow.cpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcWindow.hpp            *
//* Date       : 21-Mar-2025                                                     *
//* Version    : (see NcWindowVersion string below)                              *
//*                                                                              *
//* Description: This class implements a companion class to the NCurses class.   *
//* This class instantiates ncurses-based windows and allows access to all       *
//* attributes of those windows while encapsulating many of the messy and        *
//* frankly primitive details of the ncurses function library.                   *
//*                                                                              *
//* Note: While it is possible to write a simple console utility using the       *
//* NCurses and NcWindow classes directly, this is not recommended.              *
//* Please see the derived class, NcDialog for a fully-realized API.             *
//*                                                                              *
//* Development Tools:                                                           *
//*   See NcDialog.cpp                                                           *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.28 20-Mar-2025                                                        *
//*   -- gString class update (v:0.0.37) uses int32_t rather than int_16 type,   *
//*      so some integer variables were promoted to silence compiler warnings.   *
//*                                                                              *
//* v: 0.0.27 12-Feb-2021                                                        *
//*    -- In Hotkey2Index() method, add support for selecting a menu item by     *
//*       number (0-9 only).                                                     *
//*                                                                              *
//* v: 0.0.26 19-Jan-2021                                                        *
//*    -- Add overload methods to the 'WriteParagraph' method. Previously, the   *
//*       WriteParagraph() method took only a 'const gString&" text parameter.   *
//*       This limited the length of the text data to gsALLOCDFLT.               *
//*       Overloads have been added to allow for 'const wchar_t*' and            *
//*       'const char*' text parameters.                                         *
//*                                                                              *
//* v: 0.0.25 02-Jan-2020                                                        *
//*    -- The mouse interface for scroll-wheel events has changed.               *
//*       The event-mask bit used to report scroll-up events has changed.        *
//*       There was no significant change within this class, but the mask        *
//*       definition changed in the NCurses class.                               *
//*    -- A bug in the ncursesw library (or inadequate support under Wayland)    *
//*       causes artifacts between the left edge of the terminal window and      *
//*       our defined window WHEN DRAWING THE WINDOW BACKGROUND (ClearWin()).    *
//*       This only happens when the specified color register pair of the        *
//*       color attribute >= 0x08.                                               *
//*       Our work-around is that IF the register pair >= 0x08, THEN we          *
//*       refresh the window after each character written. This clears           *
//*       whatever ncursesw internal tracking error that is causing the          *
//*       artifacts outside the boundaries of the defined window. There is a     *
//*       slight performance hit, but it is barely noticeable to the eye.        *
//*       -- Note also that artifacts occur ONLY the first time the window       *
//*          is drawn. Subsequent calls seem to work without error.              *
//*                                                                              *
//* v: 0.0.24 28-Aug-2017                                                        *
//*   - Bug Fix: in WriteHotString() method. For RTL output only, when the       *
//*     data are multicolumn characters or mixed single-column and multicolumn   *
//*     characters, the inter-character cursor positioning could be incorrect.   *
//*                                                                              *
//* v: 0.0.23 30-Jun-2015                                                        *
//*   - Move mouse configuration methods from NcDialog class to achieve          *
//*     thread safety. No actual functionality change except to put these        *
//*     methods under the control of the mutexes.                                *
//*   - Flush the mouse LIFO queue after each mouse event received.              *
//*     See GetKeyInput().                                                       *
//*                                                                              *
//* v: 0.0.22 25-Jun-2014                                                        *
//*   - Add member 'rtl' to the ScrollData class to indicate that data should    *
//*     be drawn or redrawn as RTL (right-to-left) data in scrolling windows.    *
//*   - Simplify the ClearWin() method: 300 bytes of codespace saved.            *
//*   - Continue development of thread safety for input and output streams.      *
//*     Protection code is complete; however, we need to watch for possible      *
//*     resource-conflict problems.                                              *
//*     - As a single, shared resource, the NCurses input stream is inherently   *
//*       thread safe, but protecting the stream-conditioning methods has        *
//*       raised our 'warm-and-fuzzy' index by a few points.                     *
//*     - Unfortunately, the cursor (output insertion point) is also a single,   *
//*       shared resource, so thread safety for output is a bit more complex.    *
//*     - The (private) WriteChar() method, the primary WriteString() method     *
//*       and the WriteParagraph() method have been encapsulated, and run        *
//*       happily in combination under several simultaneous threads.             *
//*     - Output conditioning methods are now also under access control.         *
//*     - See notes in NcwKey.cpp.                                               *
//*     - See stress testing scenario in Dialog4, Test07.                        *
//*   - Add range test for initial cursor position in WriteParagraph().          *
//*   - Add 'rtl' parameter to the ClearLine() method to simplify the call       *
//*     for use with RTL languages.                                              *
//*                                                                              *
//* v: 0.0.21 15-Jan-2014                                                        *
//*   - Add CaptureWindow() method to capture display data for the window,       *
//*     either to memory or to a file. See also the SaveWin class definition.    *
//*   - Fix bugs and improved efficiency in PaintData(), RepaintData()           *
//*     and ScrollData(). Formerly, if base color of indexed item was            *
//*     'reversed', then the highlight was not visible.                          *
//*   - Make the SaveWin class public. Although not used by the public methods,  *
//*     of the NcWindow class, it is used internally by derived classes.         *
//*   - Add an optional parameter to all WriteString() and WriteChar() methods   *
//*     to support RTL (right-to-left) written languages.                        *
//*   - Add an optional parameter to DrawBorder() and DrawBox() to allow         *
//*     writing the title (if specified) as RTL text.                            *
//*                                                                              *
//* v: 0.0.20 28-Nov-2013                                                        *
//*   - Add a SetCursor(winPos yxpos) method for convenience.                    *
//*                                                                              *
//* v: 0.0.19 02-May-2013                                                        *
//*   - Fix bug in 'Track2Item' method. Was not allowing highlight to reach      *
//*     last item in list.                                                       *
//*   - ClearLine() method now returns ERR if invalid line index specified.      *
//*   - Fix bug in primary WriteString() method to avoid modification of         *
//*     caller's text data when truncating output to window width.               *
//*   - Modify both primary WriteChar() and WriteString() methods so that if     *
//*     specified start Y/X position is invalid, cursor is set to window         *
//*     origin (ZERO/ZERO) and nothing is written. Formerly, on error we set     *
//*     Y/X to ZERO/ZERO and displayed the data anyway. In retrospect,           *
//*     that was messy.                                                          *
//*   - Rename NcScroll.cpp as NcwScroll.cpp to more clearly identify the        *
//*     class it belongs to.                                                     *
//*   - Create the NcwKey.cpp to contain the thread-safe calls to the            *
//*     NCurses-class key input methods and miscellaneous other methods.         *
//*                                                                              *
//* v: 0.0.18 18-Apr-2012                                                        *
//*   - Fix bug in cursor position returned from the WriteString() and           *
//*     WriteChar() method group.                                                *
//*   - Filter out ASCII control characters in WriteString() and WriteChar()     *
//*     group.                                                                   *
//*   - Finish implementaton of the Track2X methods for moving the highlight     *
//*     in a scrolling control without updating the display. Makes for smoother  *
//*     scrolling in large data arrays.                                          *
//*                                                                              *
//* v: 0.0.17 08-Mar-2012                                                        *
//*   - Change development platform and compiler version.                        *
//*     Fedora 15 and GNU G++ (Gcc v: 4.6.1)                                     *
//*     ncurses library v: 5.9.20110404                                          *
//*     GNOME terminal 3.0.1                                                     *
//*   - Change the way screen output methods pass data to the ncurses library    *
//*     primitives. Due to changes in the ncurses library between v:5.7 to       *
//*     v:5.9, it has become necessary to pass only one wide character (plus     *
//*     any associated mod codes) at a time to wadd_wch() via the cchar_t        *
//*     structure. Although this is less efficient than passing five characters  *
//*     per call, it is a relatively straightforward update.                     *
//*   - Convert DrawBorder(), DrawBox() and DrawLine() methods to use 'wide'     *
//*     characters rather than the Alternate Character Set (ACS).                *
//*   - Update enum ncLineType to accomodate the new wcsXXXX character           *
//*     definitions in NCurses.hpp.                                              *
//*   - Enhance the lineDef class to include a 'style' data member. Re-define    *
//*     functionality of LineDef class connection flags to 'true' by default.    *
//*                                                                              *
//* v: 0.0.16 04-Feb-2012                                                        *
//*   - Convert all color attributes from 'int' to 'attr_t'. Should have done    *
//*     this from the beginning, but type aliases can be a major pain.           *
//*   - Add new method, WriteParagraph() which writes a multi-line text string   *
//*     to the display.                                                          *
//*                                                                              *
//* v: 0.0.15 29-Aug-2011                                                        *
//*   - Convert DrawBox(), Border() and DrawLine() methods to use WriteChar()    *
//*     instead of ncurses primitives.                                           *
//*   - Enhance Border() method to accept an optional window title string.       *
//*   - Enforce minimum terminal screen size for opening a window                *
//*     (see note in OpenWindow()).                                              *
//*   - Support for UTF-8-encoded text implemented and verified.                 *
//*     (Bugs, of course, may surface...)                                        *
//*                                                                              *
//* v: 0.0.14 21-Feb-2011                                                        *
//*   - Move version history to NcWindow.cpp.                                    *
//*   - Add support for 'wide' character output and multi-byte key input.        *
//*     This is a necessary step for internationalization because the ncurses    *
//*     'narrow' library has only minimal support for wide character data.       *
//*     The technical details of UTF-8 conversion, wchar_t conversion, and       *
//*     calculation of needed display columns is encapsulated in the gString     *
//*     class. Please refer to gString.hpp and gString.cpp for more              *
//*     information.                                                             *
//*   - New methods:                                                             *
//*     1) WriteString(), WriteHotString(), and WriteChar() for wchar_t          *
//*        strings, for use if application has already converted the characters  *
//*        from their native encoding to wchar_t characters. This is not the     *
//*        preferred output method because it requires the application to use    *
//*        a lot of storage for strings AND to be aware of character width.      *
//*        We include these methods specifically for applications that use       *
//*        some regional character encoding internally, rather than UTF-8        *
//*        encoding. Using the methods that directly support UTF-8 encoding      *
//*        is preferred.                                                         *
//*     2) WriteString(), WriteHotString(), and WriteChar() which are aware of   *
//*        UTF-8 encoding, and will handle ASCII output directly AND will        *
//*        automagically convert from UTF-8 to the format needed by the          *
//*        ncursesw library without the application needing to know anything     *
//*        about the conversion process.                                         *
//*     3) ASCII-ONLY flag is available for call to WriteString() as a           *
//*        performance enhancement if the string is known to be 7-bit ASCII.     *
//*   - For performance reasons we call the mvwaddch() primitive for line        *
//*     drawing and clearing the window. However, except for the alternate       *
//*     character set, for any non-space characters, we must use the wadd_wch()  *
//*     primitive to accomodate supra-ASCII output.                              *
//*    All NcWindow class methods verified for UTF-8 output support.             *
//*    26 March, 2011.                                                           *
//*   - The WriteString(), WriteHotString(), WriteChar() and GetCursor()         *
//*     return values have been updated from a WinPos structure to a winPos      *
//*     class object, and that the WinPos structure is being phased out.         *
//*                                                                              *
//* v: 0.0.13 07-Feb-2010                                                        *
//*   - Change development platform and compiler version.                        *
//*     Fedora 12 and GNU G++ (Gcc v: 4.4.2)                                     *
//*     This neccessitated several syntax changes in class and method            *
//*     definitions to accomodate changes in the C++ definition and the GCC      *
//*     compiler's warning and error messages.                                   *
//*   - Implement DebugMsg() method.                                             *
//*   - Implement GetCursor() method.                                            *
//*   - Made CloseWin() private.                                                 *
//*   - Implement PaintMenuData() for multi-color strings.                       *
//*   - Enhanced HilightItem(). Now is sensitive to whether color data includes  *
//*     the ncrATTR bit, AND now handles strings that include hotkey             *
//*     indicator ('^').                                                         *
//*   - Obsolete HilightItemRev().                                               *
//*   - DrawLine() enhancements. Obsolete old versions.                          *
//*                                                                              *
//* v: 0.0.12 26-May-2007                                                        *
//*    - Optional 'refresh' parameter for WriteString() and WriteChar(),         *
//*      obsolete WriteStringNoRef() and WriteCharNoRef().                       *
//* v: 0.0.11 15-Apr-2007 Modify DrawBox() method to accept an optional title    *
//* v: 0.0.10 07-Apr-2007 Change name of class version string                    *
//* v: 0.0.09 18-Mar-2007 Fix bug in ClearLine(), now uses interior color        *
//* v: 0.0.08 20-Jan-2007                                                        *
//*   - Split source module NcWindow.cpp. Methods that address scrolling go      *
//*     to NcScroll.cpp.                                                         *
//* v: 0.0.07 15-Jan-2007 Fixed error in color version of PaintData().           *
//* v: 0.0.06 20-Dec-2006                                                        *
//*   - Debug PaintData() for both monochrome and multi-color scrolling data.    *
//*     (the bug fix for PaintMenuData() has not yet been tested.)               *
//* v: 0.0.05 01-Sep-2006                                                        *
//*   - Add SetInteriorColor() method. Modify ClearWin() and OpenWindow() to     *
//*     use wColor attribute.                                                    *
//*   - Modify Border() so border is drawn only if window is at least 2x2        *
//*     characters.                                                              *
//*                        Add DrawBox() method (non-border rectangles)          *
//* v: 0.0.04 14-Apr-2006 Add PaintMenuData() and ScrollMenuData() methods       *
//* v: 0.0.03 06-Jan-2006 Add ScrollView() method                                *
//* v: 0.0.02 21-Dec-2005 Add color-attribute parm to DrawLine() methods         *
//* v: 0.0.01 02-Sep-2005 Original release                                       *
//*                        Created using GNU G++ (Gcc v: 3.2.2)                  *
//*                        under RedHat Linux, kernel 2.4.20-31.9                *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCWINDOW_INCLUDED
#include "NcWindow.hpp"
#endif

//*********************
//* Local Definitions *
//*********************


//*********************
//* Local Data        *
//*********************

static const char NcWindowVersion[] = "0.0.28" ; //* Class version number

#if NCD_THREAD_SAFETY != 0
//* These mutexes control access to the input and output streams, respectively.*
//* These are not data members of the NcWindow class because they arbitrate    *
//* access to the input/output streams for all instances of the NcWindow class.*
recursive_mutex InputStream_Lock ;
recursive_mutex OutputStream_Lock ;
#endif   // NCD_THREAD_SAFETY



//*************************
//*     ~NcWindow         *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

NcWindow::~NcWindow ( void )
{
   //* If the window was previously opened, close it *
   if ( this->wp != NULL )
   {
      this->AcquireOutputLock () ;     // acquire exclusive access to output stream
      this->ClearWin ( true ) ;        // erase window from the display
      this->CloseWin () ;              // release resources of ncurses window
      this->ReleaseOutputLock () ;     // release lock on output stream
   }
   
}  //* End ~NcWindow() *

//*************************
//*     NcWindow          *
//*************************
//******************************************************************************
//* Constructor for a NcWindow object.                                         *
//*                                                                            *
//* Input  : lines   : number of lines in the window                           *
//*          columns : number of columns in the window                         *
//*          begin_y : vertical offset from upper left corner of screen        *
//*          begin_x : horizontal offset from upper left of screen             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* The ncurses engine defines a window with the following structure.          *
//* The WINDOW *wp member of the NcWindow class points to this underlying      *
//* data structure.                                                            *
//* typedef struct screen  SCREEN;                                             *
//* typedef struct _win_st WINDOW;                                             *
//* struct _win_st                                                             *
//* {                                                                          *
//*   NCURSES_SIZE_T _cury, _curx; //current cursor position                   *
//*                                                                            *
//*   //* window location and size *                                           *
//*   NCURSES_SIZE_T _maxy, _maxx; //maximums of x and y, NOT window size      *
//*   NCURSES_SIZE_T _begy, _begx; //screen coords of upper-left-hand corner   *
//*                                                                            *
//*   short   _flags;    //window state flags                                  *
//*                                                                            *
//*   //* attribute tracking *                                                 *
//*   attr_t  _attrs;	   //current attribute for non-space character         *
//*   chtype  _bkgd;       //current background char/attribute pair            *
//*                                                                            *
//*   //* option values set by user *                                          *
//*   bool  _notimeout;   //no time out on function-key entry?                 *
//*   bool  _clear;       //consider all data in the window invalid?           *
//*   bool  _leaveok;     //OK to not reset cursor on exit?                    *
//*   bool  _scroll;      //OK to scroll this window?                          *
//*   bool  _idlok;       //OK to use insert/delete line?                      *
//*   bool  _idcok;       //OK to use insert/delete char?                      *
//*   bool  _immed;       //window in immed mode? (not yet used)               *
//*   bool  _sync;        //window in sync mode?                               *
//*   bool  _use_keypad;  //process function keys into KEY_ symbols?           *
//*   int   _delay;       //0 = nodelay, <0 = blocking, >0 = delay             *
//*                                                                            *
//*   struct ldat *_line; //the actual line data                               *
//*                                                                            *
//*   //* global screen state *                                                *
//*   NCURSES_SIZE_T _regtop;     //top line of scrolling region               *
//*   NCURSES_SIZE_T _regbottom;  //bottom line of scrolling region            *
//*                                                                            *
//*   //* these are used only if this is a sub-window                          *
//*   int   _parx;      //x coordinate of this window in parent                *
//*   int   _pary;      //y coordinate of this window in parent                *
//*   WINDOW *_parent;  //pointer to parent if a sub-window                    *
//*                                                                            *
//*   //* these are used only if this is a pad *                               *
//*   struct pdat                                                              *
//*   {                                                                        *
//*      NCURSES_SIZE_T _pad_y,      _pad_x;                                   *
//*      NCURSES_SIZE_T _pad_top,    _pad_left;                                *
//*      NCURSES_SIZE_T _pad_bottom, _pad_right;                               *
//*   }  _pad;                                                                 *
//*                                                                            *
//*   NCURSES_SIZE_T _yoffset;   //real begy is _begy + _yoffset               *
//*                                                                            *
//* #ifdef _XOPEN_SOURCE_EXTENDED                                              *
//*   cchar_t  _bkgrnd;	//current background char/attribute pair               *
//* #if 0                                                                      *
//*   int   _color;     //current color-pair for non-space character           *
//* #endif                                                                     *
//* #endif                                                                     *
//* };                                                                         *
//******************************************************************************

NcWindow::NcWindow ( short lines, short columns, short startY, short startX )
{
   this->wLines = lines ;                    // save parameters for OpenWindow()
   this->wCols  = columns ;
   this->wulY = startY ;
   this->wulX = startX ;
   this->wColor = nc.bw ;
   this->wp = NULL ;                         // ncurses window pointer

   //* NOTE: These members are meaningful only for windows  *
   //* which will contain scrolling data. Ingored otherwise.*
   this->scrollData.dptr   = NULL ;          // initialize tracking variables
   this->scrollData.cptr   = NULL ;
   this->scrollData.monocolor = nc.bw ;
   this->scrollData.count  = ZERO ;
   this->scrollData.hRow   = ZERO ;
   this->scrollData.hIndex = ZERO ;
   this->scrollData.tIndex = ZERO ;
   this->scrollData.bIndex = ZERO ;
   this->scrollData.rtl    = false ;

   //* NOTE: These members are meaningful only for windows  *
   //* that are dialogs. Ingnored within component windows. *
   this->meDClick = ncmiSTABLE ; // default double-click interval for
                                 // 'stable' mouse mode
   this->meStable = false ;      // initially, mouse is disabled entirely
   this->meAuto = true ;         // initially, mouse events automagically
                                 // converted to keycodes

}  //* End NcWindow() *

//***********************
//*     OpenWindow      *
//***********************
//******************************************************************************
//* Allocate resources for the window based upon the parameters passed to      *
//* the constructor.                                                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successfully opened, else ERR                               *
//*          Note that most likely causes of error are:                        *
//*           1. terminal screen too small to hold the window plus             *
//*              offsets from upper-left corner                                *
//*           2. insufficient heap memory                                      *
//******************************************************************************
//* Programmer's Note: There is a small bug in the ncurses library (v:5.7)     *
//* which allows the window to open even if the terminal screen is one column  *
//* too narrow for the window. If terminal is two columns too narrow, newwin() *
//* fails successfully. ncurses also allows us to open a window that extends   *
//* beyond the bottom of the terminal.                                         *
//* For these reasons, we test our terminal size and window position, and      *
//* refuse to open the window if it will not fit.                              *
//*                                                                            *
//******************************************************************************

short NcWindow::OpenWindow ( void )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   short result = ERR, 
         scrLines, scrColumns ;
   bool spOk = false ;

   //* Test window size and position to determine     * 
   //* whether it will fit within the terminal screen.*
   nc.ScreenDimensions ( scrLines, scrColumns ) ;
   if ( ((this->wLines + this->wulY) <= scrLines) && 
        ((this->wCols + this->wulX) <= scrColumns) )
      spOk = true ;

   if ( spOk && ((wp = newwin ( this->wLines, this->wCols, 
                                this->wulY, this->wulX )) != NULL) )
   {  //* Initialize window-size tracking variables *
      getmaxyx ( this->wp, this->wLines, this->wCols ) ;
      keypad ( this->wp, true ) ;            // enable escape-sequence translation
      notimeout ( this->wp, true ) ;         // no delay after ESC code
      if ( this->wColor != nc.bw )           // if background is not default color
         this->ClearWin () ;                 // draw the background in specified color
      result = OK ;
   }

   this->ReleaseOutputLock () ;     // release lock on output stream
   return result ;
   
}  //* End OpenWindow() *

//*************************
//*      CloseWin         *
//*************************
//******************************************************************************
//* Destructor for primitive ncurses window. NcWindow object is not            *
//* destroyed. It is the responsibility of caller (destructor) to release the  *
//* resources of the NcWindow object.                                          *
//*                                                                            *
//* Application-level users should call the destructor directly, NOT this      *
//* this method. Only classes derived from the NcWindow class should call      *
//* this method directly.                                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::CloseWin ( void )
{
   if ( this->wp != NULL )
   {
      this->AcquireOutputLock () ;  // acquire exclusive access to output stream
      delwin ( this->wp ) ;
      this->wp = NULL ;
      this->ReleaseOutputLock () ;  // release lock on output stream
   }

}  //* End CloseWin() *

//***********************
//*      ClearWin       *
//***********************
//******************************************************************************
//* Erase the contents of the window and optionally refresh the display.       *
//*                                                                            *
//* Input  : clearAll: (optional, true by default)                             *
//*                    if false do not clear window border.                    *
//*                    else, clear entire window.                              *
//*          refresh : (optional, true by default)                             *
//*                    if true, refresh the display                            *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* Programmer's Note: Windows of one line, by definition, have no borders,    *
//* so clearAll is assumed for one-line windows.                               *
//******************************************************************************

short NcWindow::ClearWin ( bool clearAll, bool refresh )
{
   //* Work-around for ncurses bug. See notes in module header. *
   #define CLEAR_FIX (1)

   // Programmer's Note: 'wchar_t' == 'chtype' == attr_t == uint32_t.
   chtype  chattr = L' ' | this->wColor ;


   //* If this is a single-line window, force clearAll *
   if ( this->wLines == 1 )
      clearAll = true ;

   //* Initialize for clearing entire window *
   short startY = ZERO,
         startX = ZERO,
         yCount = this->wLines - 1,
         xCount = this->wCols - 1 ;
   //* Adjust if clearing interior only *
   if ( clearAll == false )
   {
      ++startY ;
      ++startX ;
      --yCount ;
      --xCount ;
   }

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   #if CLEAR_FIX != 0
   int32_t reg_pair = (this->wColor >> 8) & 0x000000FF ;
   if ( reg_pair <= 0x07 )          // fast write
   {
   #endif   // CLEAR_FIX
      for ( short line = startY ; line <= yCount ; ++line )
      {
         this->SetCursor ( line, startX ) ;
         for ( short col = startX ; col <= xCount ; ++col )
            waddch ( this->wp, chattr ) ;
      }
   #if CLEAR_FIX != 0
   }
   else                             // safe write
   {
      for ( short line = startY ; line <= yCount ; ++line )
      {
         this->SetCursor ( line, startX ) ;
         for ( short col = startX ; col <= xCount ; ++col )
            wechochar ( this->wp, chattr ) ;
      }
   }
   #endif   // CLEAR_FIX

   if ( refresh )
      wrefresh ( this->wp ) ;

   this->ReleaseOutputLock () ;     // release lock on output stream
   return OK ;

}  //* End ClearWin() *

//*************************
//*     ClearLine         *
//*************************
//******************************************************************************
//* Clear from specified cursor position to end of line. Display is refreshed. *
//*                                                                            *
//* Input  : ypos : line number (zero-based)                                   *
//*          clearAll: (optional, false by default) if 'true', clear entire    *
//*                    line, if 'false' do not clear border area               *
//*          xpos : (optional, by default, for LTR: one(1), for RTL: columns-2)*
//*                 starting column (zero-based)                               *
//*                 (ignored if clearAll != false)                             *
//*          rtl  : (optional, 'false' by default)                             *
//*                 if 'false', then clear from 'xpos' toward right            *
//*                 if 'true',  then clear from 'xpos' toward left             *
//*                 (ignored if clearAll != false)                             *
//*                                                                            *
//* Returns: OK if successful, ERR if invalid line index                       *
//******************************************************************************

short NcWindow::ClearLine ( short ypos, bool clearAll, short xpos, bool rtl )
{
   short status = ERR ;
   if ( ypos >= ZERO && ypos < this->wLines )
   {
      wchar_t chattr = L' ' | this->wColor ;
      short firstCol = xpos,           // (assume LTR output)
            lastCol  = this->wCols - 1 ;
      if ( clearAll != false )         // if clearing entire line
      {
         firstCol = ZERO ;
         ++lastCol ;
      }
      if ( rtl != false && !clearAll ) // adjust for RTL output
      {
         firstCol = 1 ;
         lastCol  = this->wCols - xpos ;
      }

      this->AcquireOutputLock () ;     // acquire exclusive access to output stream
      for ( ; firstCol < lastCol ; firstCol++ )
         mvwaddch ( this->wp, ypos, firstCol, chattr ) ;
      wrefresh ( this->wp ) ;
      this->ReleaseOutputLock () ;     // release lock on output stream
      status = OK ;
   }
   return status ;
   
}  //* End ClearLine() *

//*************************
//*      GetCursor        *
//*************************
//******************************************************************************
//* Get the window's cursor position.                                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: a winPos class object containing the Y/X cursor position          *
//******************************************************************************
//* Programmer's Note: getyx() is a macro.                                     *
//******************************************************************************

winPos NcWindow::GetCursor ( void )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   winPos wpos ;
   getyx ( this->wp, wpos.ypos, wpos.xpos ) ;

   this->ReleaseOutputLock () ;     // release lock on output stream
   return wpos ;

}  //* End GetCursor() *

//*************************
//*      SetCursor        *
//*************************
//******************************************************************************
//* Set the window's cursor posiiton.                                          *
//*                                                                            *
//*                                                                            *
//* Input  : ypos : Y offset from upper-left corner (zero-based)               *
//*          xpos : X offset from upper-left corner (zero-based)               *
//*            OR                                                              *
//*          yxpos: Y/X offset from upper-left corner (zero-based)             *
//*                                                                            *
//* Returns: OK if successful, ERR if parameter(s) out of range                *
//******************************************************************************

short NcWindow::SetCursor ( short ypos, short xpos )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   short result = OK ;

   if ( ypos >= ZERO && ypos < this->wLines && xpos >= ZERO && xpos < this->wCols )
   {
      wmove ( this->wp, ypos, xpos ) ;
   }
   else
      result = ERR ;

   this->ReleaseOutputLock () ;     // release lock on output stream
   return result ;
   
}  //* End SetCursor() *

short NcWindow::SetCursor ( winPos yxpos )
{

   return ( this->SetCursor ( yxpos.ypos, yxpos.xpos ) ) ;

}  //* End SetCursor() *

//*************************
//*     DrawBorder        *
//*************************
//******************************************************************************
//* Draw an inside border around the window.                                   *
//*                                                                            *
//* Input  : cAttr  : color attribute                                          *
//*          title  : (optional, default==NULL)                                *
//*                   if non-NULL, points to window title string               *
//*          style  : (optional, ncltSINGLE by default)                        *
//*                   member of enum ncLineType                                *
//*          rtlText: (optional, false by default)                             *
//*                   if 'title' != NULL, then:                                *
//*                   if false, draw as LTR text, if true, draw as RTL text    *
//*                                                                            *
//*                                                                            *
//* Returns: If no title string is specified, returns OK on success,           *
//*           else ERR (window too small, must be at least 2 lines by 2 cols)  *
//*          If a title string IS specified, returns the X offset              *
//*           at which the title is displayed.                                 *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - The wborder() function of the ncurses library (v:5.7) and all its        *
//*   associated macros have a bug that shows up when painting a border on a   *
//*   sub-window: the draw trashes the left side of the parent window because  *
//*   the routine draws spaces from the left edge of the parent window         *
//*   before drawing the border characters.  This bug may be masked when       *
//*   using the system default background, but try using a non-default         *
//*   background to prove it to yourself.                                      *
//*   For this reason, we draw the window manually.                            *
//*                                                                            *
//******************************************************************************

short NcWindow::DrawBorder ( attr_t cAttr, const char* title, 
                             ncLineType style, bool rtlText )
{
   short result = ERR ;
   //* Window must have at least 2 lines and 2 columns to draw a border *
   //* Note: 2 x 2 means no interior, JUST border.                      *
   if ( this->wLines > 1 && this->wCols > 1 )
   {
      result = 
      this->DrawBox ( ZERO, ZERO, this->wLines, this->wCols, cAttr, 
                      title, style, rtlText ) ;
   }
   return result ;

}  //* End DrawBorder() *

//*************************
//*       DrawBox         *
//*************************
//******************************************************************************
//* Draw a rectangle within the window.                                        *
//*                                                                            *
//* Note: For drawing a window border, use the Border() method instead.        *
//*                                                                            *
//*                                                                            *
//* Input  : startY : y offset for start of line                               *
//*          startX : x offset for start of line                               *
//*          rows   : number of rows                                           *
//*          cols   : number of columns                                        *
//*          cAttr  : color attribute                                          *
//*          title  : (optional, default==NULL)                                *
//*                   if non-NULL, points to box title string                  *
//*          style  : (optional, ncltSINGLE by default)                        *
//*          rtlText: (optional, false by default)                             *
//*                   if 'title' != NULL, then:                                *
//*                   if false, draw as LTR text, if true, draw as RTL text    *
//*                                                                            *
//* Returns: If no title string is specified, returns OK on success,           *
//*           else ERR (box bigger than parent window)                         *
//*          If a title string IS specified, returns the X offset              *
//*           at which the title is displayed.                                 *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - Note that this method uses characters from the wcsXXXX 'wide'            *
//*   line-drawing character definitions found in NCurses.hpp.                 *
//******************************************************************************

short NcWindow::DrawBox ( short startY, short startX, short rows, short cols, 
                          attr_t cAttr, const char* title, 
                          ncLineType style, bool rtlText )
{
   short y = startY,                // set the rectangle dimensions
         x = startX,
         lastRow = y + rows - 1,
         lastCol = x + cols - 1,
         result = OK ;              // return value

   //* Window must be big enough to encompass the rectangle *
   if ( (lastRow < wLines) && (lastCol < wCols ) )
   {
      wchar_t  ulCorner = wcsULs,   urCorner = wcsURs, 
               llCorner = wcsLLs,   lrCorner = wcsLRs, 
               vLine    = wcsVERTs, hLine  = wcsHORIZs ;
      if ( style == ncltSINGLEBOLD )
      {
         ulCorner = wcsULb ; urCorner = wcsURb ; 
         llCorner = wcsLLb ; lrCorner = wcsLRb ; 
         vLine  = wcsVERTb ; hLine  = wcsHORIZb ;
      }
      else if ( style == ncltDUAL )
      {
         ulCorner = wcsULd ; urCorner = wcsURd ; 
         llCorner = wcsLLd ; lrCorner = wcsLRd ; 
         vLine  = wcsVERTd ; hLine  = wcsHORIZd ;
      }
      else if ( style == ncltDASH2 )
      {
         vLine  = wcsVDASH2s ; hLine  = wcsHDASH2s ;
      }
      else if ( style == ncltDASH3 )
      {
         vLine  = wcsVDASH3s ; hLine  = wcsHDASH3s ;
      }
      else if ( style == ncltDASH4 )
      {
         vLine  = wcsVDASH4s ; hLine  = wcsHDASH4s ;
      }
      else if ( style == ncltDASH2BOLD )
      {
         ulCorner = wcsULb ; urCorner = wcsURb ; 
         llCorner = wcsLLb ; lrCorner = wcsLRb ; 
         vLine  = wcsVDASH2b ; hLine  = wcsHDASH2b ;
      }
      else if ( style == ncltDASH3BOLD )
      {
         ulCorner = wcsULb ; urCorner = wcsURb ; 
         llCorner = wcsLLb ; lrCorner = wcsLRb ; 
         vLine  = wcsVDASH3b ; hLine  = wcsHDASH3b ;
      }
      else if ( style == ncltDASH4BOLD )
      {
         ulCorner = wcsULb ; urCorner = wcsURb ; 
         llCorner = wcsLLb ; lrCorner = wcsLRb ; 
         vLine  = wcsVDASH4b ; hLine  = wcsHDASH4b ;
      }

      this->AcquireOutputLock () ;     // acquire exclusive access to output stream

      this->WriteChar ( y, x++, ulCorner, cAttr ) ;   // upper left corner
      while ( x < lastCol )                           // top border
         this->WriteChar ( y, x++, hLine, cAttr ) ;
      this->WriteChar ( y, x++, urCorner, cAttr ) ;   // upper right corner

      x = startX ;
      while ( ++y < lastRow )                         // left and right borders
      {
         this->WriteChar ( y, x, vLine, cAttr ) ;
         this->WriteChar ( y, lastCol, vLine, cAttr ) ;
      }

      this->WriteChar ( y, x++, llCorner, cAttr ) ;   // lower left corner
      while ( x < lastCol )                           // bottom border
         this->WriteChar ( y, x++, hLine, cAttr ) ;
      this->WriteChar ( y, x++, lrCorner, cAttr ) ;   // lower right corner

      //* If a title was specified *
      if ( title != NULL )
      {
         gString gs(title) ;        // copy title to gString object
         short tLen = gs.gscols() ; // get number of columns needed for display
         //* Be sure string will fit. Actually, if it's too long, it's an      *
         //* application error, but a truncated display is less confusing and  *
         //* more indicative of the error than no display at all.              *
         if ( tLen > (this->wCols - 4) )
            tLen = gs.limitCols( this->wCols - 4 ) ;
         short xpos = startX + cols,// initial X
               lPos,                // X offset of left bracket
               rPos,                // X offset of right bracket
               tPos ;               // X offset of title string
         wchar_t lChar = wcsRTEE,   // left-bracket character
                 rChar = wcsLTEE ;  // right-bracket character
         if ( style == ncltSINGLEBOLD )
         { lChar = wcsRTEEb ; rChar = wcsLTEEb ; }
         else if ( style == ncltDUAL )
         { lChar = wcsRTEEd ; rChar = wcsLTEEd ; }
      
         if ( tLen <= (cols-4) )    // if title will fit
         {
            rPos = xpos - 2 ;
            tPos = rPos - tLen ;
            lPos = tPos - 1 ;
            while ( (xpos-rPos) <= (lPos-startX) )
            {
               --lPos ;
               --tPos ;
               --rPos ;
            }
            if ( rtlText != false )
               tPos = rPos - 1 ;

            //* Draw the title *
            this->WriteChar ( startY, lPos, lChar, cAttr ) ;
            this->WriteString ( startY, tPos, gs, cAttr, false, rtlText ) ;
            this->WriteChar ( startY, rPos, rChar, cAttr ) ;
            result = tPos ;   // Return X offset of display string
         }
      }
      this->ReleaseOutputLock () ;     // release lock on output stream
   }
   else                                // rectangle won't fit in window
      result = ERR ;
   return result ;

}  //* End DrawBox() *

//*************************
//*   SetInteriorColor    *
//*************************
//******************************************************************************
//* Set a background color for the interior of the window.                     *
//* Does not redraw the window in the new color.                               *
//*                                                                            *
//* Input  : cAttr : color attribute                                           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::SetInteriorColor ( attr_t cAttr )
{

   this->wColor = cAttr ;

}  //* End SetInteriorColor() *

//*************************
//*       DrawLine        *
//*************************
//******************************************************************************
//* Draw a line of the specified type and style (see enum ncLineType).         *
//* Note that by default any contact between the new line and any existing     *
//* lines will generate a visual connection.                                   *
//* To prevent automagical line connections, please set the appropriate flag(s)*
//* (cLeft, cRight, cTop, cBot, cInsect) to 'false'.                           *
//*                                                                            *
//* Input  : lDef (by reference)                                               *
//*             fields : type  : ncltHORIZ or ncltVERT                         *
//*                      style : normal, bold, dual-line, or one of the dashed *
//*                              line types                                    *
//*                      startY: y offset for start of line                    *
//*                      startX: x offset for start of line                    *
//*                      length: number of lines (vert) or columns (horiz)     *
//*                      attr  : color attribute                               *
//*                                                                            *
//* Returns: OK if success, else ERR                                           *
//******************************************************************************

short NcWindow::DrawLine ( const LineDef& ldef )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   short    y = ldef.startY,  // write position
            x = ldef.startX ;
   wchar_t  dChar ;           // drawing character
   attr_t   cAttr ;           // color attribute

   for ( short cnt = ldef.length ; cnt > ZERO ; cnt-- )
   {
      this->dlNextChar ( ldef, y, x, dChar, cAttr ) ;
      this->WriteChar ( y, x, dChar, cAttr ) ;
      if ( ldef.type == ncltHORIZ )
         ++x ;
      else  // ldef.type == ncltVERT
         ++y ;
   }

   this->ReleaseOutputLock () ;     // release lock on output stream
   return OK ;

}  //* End DrawLine() *

//*************************
//*      dlNextChar       *
//*************************
//******************************************************************************
//* Read the character at the specified screen position, and determine whether *
//* it is a line-drawing character. If it is, determine the connection         *
//* character and attribute to be written that will join the existing line to  *
//* the line currently being drawn.                                            *
//*                                                                            *
//* Input  : ypos    : Y screen position                                       *
//*          xpos    : X screen position                                       *
//*          wkcode  : (by reference, initial values ignored)                  *
//*                    character and attribute at specified screen position    *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Returns: 'true' if character is a line drawing character, else 'false'     *
//******************************************************************************

bool NcWindow::dlNextChar ( const LineDef& ldef, short ypos, short xpos, 
                            wchar_t& dChar, attr_t& cAttr )
{
   bool ldChar = false ;            // return value

   //* Assume that screen character IS NOT a line-drawing character *
   //* and set drawing character as defined in ldef.                *
   if ( ldef.type == ncltHORIZ )
   {
      switch ( ldef.style )
      {
         case ncltSINGLE:     dChar = wcsHORIZs ;     break ;
         case ncltSINGLEBOLD: dChar = wcsHORIZb ;     break ;
         case ncltDUAL:       dChar = wcsHORIZd ;     break ;
         case ncltDASH2:      dChar = wcsHDASH2s ;    break ;
         case ncltDASH2BOLD:  dChar = wcsHDASH2b ;    break ;
         case ncltDASH3:      dChar = wcsHDASH3s ;    break ;
         case ncltDASH3BOLD:  dChar = wcsHDASH3b ;    break ;
         case ncltDASH4:      dChar = wcsHDASH4s ;    break ;
         case ncltDASH4BOLD:  dChar = wcsHDASH4b ;    break ;
         default:             dChar = wcsHORIZs ;     break ;
      }
   }
   else  // ldef.type == ncltVERT
   {
      switch ( ldef.style )
      {
         case ncltSINGLE:     dChar = wcsVERTs ;      break ;
         case ncltSINGLEBOLD: dChar = wcsVERTb ;      break ;
         case ncltDUAL:       dChar = wcsVERTd ;      break ;
         case ncltDASH2:      dChar = wcsVDASH2s ;    break ;
         case ncltDASH2BOLD:  dChar = wcsVDASH2b ;    break ;
         case ncltDASH3:      dChar = wcsVDASH3s ;    break ;
         case ncltDASH3BOLD:  dChar = wcsVDASH3b ;    break ;
         case ncltDASH4:      dChar = wcsVDASH4s ;    break ;
         case ncltDASH4BOLD:  dChar = wcsVDASH4b ;    break ;
         default:             dChar = wcsVERTs ;      break ;
      }
   }

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   cAttr = ldef.color ;             // caller's color attribute
   cchar_t  cct ;                   // for screen-data capture
   if ( (mvwin_wch (this->wp, ypos, xpos, &cct )) == OK )
   {
      //* Note that this preliminary range test is based on the distribution   *
      //* of line-drawing characters within the Unicode character set.         *
      wchar_t tChar = cct.chars[0] ;// copy of displayed character under test
      if ( tChar >= 0x002500 && tChar < 0x00257F )
      {  //* Horizontal line being drawn *
         if ( ldef.type == ncltHORIZ )
         {  //* Left end of line being drawn *
            if ( xpos == ldef.startX && ldef.cLeft != false )
            {  //* Test the captured display character *
               switch ( tChar )
               {
                  case wcsVERTs:
                  case wcsVDASH2s:
                  case wcsVDASH3s:
                  case wcsVDASH4s:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsLTEEs ;   break ;
                        case ncltSINGLEBOLD: dChar = wcsLTEEbh ;  break ;
                        case ncltDUAL:       dChar = wcsLTEEdh ;  break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                           dChar = wcsLTEEs ;                     break ;
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsLTEEbh ;                    break ;
                        default: dChar = wcsLTEEs ;               break ;
                     }
                     break ;
                  case wcsVERTb:
                  case wcsVDASH2b:
                  case wcsVDASH3b:
                  case wcsVDASH4b:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsLTEEbv ;  break ;
                        case ncltSINGLEBOLD: dChar = wcsLTEEb ;   break ;
                        case ncltDUAL:
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsLTEEb ;                     break ;
                        default: dChar = wcsLTEEbv ;              break ;
                     }
                     break ;
                  case wcsVERTd:
                     switch ( ldef.style )
                     {
                        case ncltDUAL: dChar = wcsLTEEd ;         break ;
                        default:       dChar = wcsLTEEdv ;        break ;
                     }
                     break ;
               }
            }
            //* Right end of line being drawn *
            else if ( xpos == (ldef.startX + ldef.length - 1) && ldef.cRight != false )
            {  //* Test the captured display character *
               switch ( tChar )
               {
                  case wcsVERTs:
                  case wcsVDASH2s:
                  case wcsVDASH3s:
                  case wcsVDASH4s:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsRTEEs ;   break ;
                        case ncltSINGLEBOLD: dChar = wcsRTEEbh ;  break ;
                        case ncltDUAL:       dChar = wcsRTEEdh ;  break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                           dChar = wcsRTEEs ;                     break ;
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsRTEEbh ;                    break ;
                        default: dChar = wcsRTEEs ;               break ;
                     }
                     break ;
                  case wcsVERTb:
                  case wcsVDASH2b:
                  case wcsVDASH3b:
                  case wcsVDASH4b:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsRTEEbv ;  break ;
                        case ncltSINGLEBOLD: dChar = wcsRTEEb ;   break ;
                        case ncltDUAL://       dChar = wcsRTEEdh ;  break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
//                           dChar = wcsRTEEs ;                     break ;
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsRTEEb ;                     break ;
                        default: dChar = wcsRTEEbv ;              break ;
                     }
                     break ;
                  case wcsVERTd:
                     switch ( ldef.style )
                     {
                        case ncltDUAL: dChar = wcsRTEEd ;         break ;
                        default:       dChar = wcsRTEEdv ;        break ;
                     }
                     break ;
               }
            }
            //* Mid-line intersection of line being drawn with existing line *
            else if ( ldef.cInsect != false )
            {  //* Test the captured display character *
               switch ( tChar )
               {
                  case wcsVERTs:
                  case wcsVDASH2s:  case wcsVDASH3s:  case wcsVDASH4s:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE: case ncltDASH2:
                        case ncltDASH3: case ncltDASH4:
                           dChar = wcsINSECTs ;                   break ;
                        case ncltSINGLEBOLD: case ncltDASH2BOLD:
                        case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsINSECTbh ;                  break ;
                        case ncltDUAL:
                           dChar = wcsINSECTdh ;                  break ;
                        default:    dChar = wcsINSECTs ;          break ;
                     }
                     break ;
                  case wcsVERTb:
                  case wcsVDASH2b:  case wcsVDASH3b:  case wcsVDASH4b:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE: case ncltDASH2:
                        case ncltDASH3: case ncltDASH4:
                           dChar = wcsINSECTbv ;                  break ;
                        case ncltSINGLEBOLD: case ncltDASH2BOLD:
                        case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsINSECTb ;                   break ;
                        case ncltDUAL: dChar = wcsINSECTdh ;      break ;
                        default: dChar = wcsINSECTb ;             break ;
                     }
                     break ;
                  case wcsVERTd:
                     switch ( ldef.style )
                     {
                        case ncltDUAL: dChar = wcsINSECTd ;       break ;
                        default: dChar = wcsINSECTdv ;            break ;
                     }
                     break ;
               }
            }
         }
         else  // ldef.type == ncltVERT
         {  //* Top end of line being drawn *
            if ( ypos == ldef.startY && ldef.cTop != false )
            {  //* Test the captured display character *
               switch ( tChar )
               {
                  case wcsHORIZs:
                  case wcsHDASH2s:
                  case wcsHDASH3s:
                  case wcsHDASH4s:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsTTEEs ;   break ;
                        case ncltSINGLEBOLD: dChar = wcsTTEEbv ;  break ;
                        case ncltDUAL:       dChar = wcsTTEEdv ;  break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                           dChar = wcsTTEEs ;                     break ;
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsTTEEbv ;                    break ;
                        default: dChar = wcsTTEEs ;               break ;
                     }
                     break ;
                  case wcsHORIZb:
                  case wcsHDASH2b:
                  case wcsHDASH3b:
                  case wcsHDASH4b:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsTTEEbh ;  break ;
                        case ncltSINGLEBOLD: dChar = wcsTTEEb ;   break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                           dChar = wcsTTEEbh ;                    break ;
                        case ncltDUAL:
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsTTEEb ;                     break ;
                        default: dChar = wcsTTEEbh ;              break ;
                     }
                     break ;
                  case wcsHORIZd:
                     switch ( ldef.style )
                     {
                        case ncltDUAL: dChar = wcsTTEEd ;         break ;
                        default:       dChar = wcsTTEEdh ;        break ;
                     }
                     break ;
               }
            }
            //* Bottom end of line being drawn *
            else if ( ypos == (ldef.startY + ldef.length - 1) && ldef.cBot != false )
            {  //* Test the captured display character *
               switch ( tChar )
               {
                  case wcsHORIZs:
                  case wcsHDASH2s:
                  case wcsHDASH3s:
                  case wcsHDASH4s:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsBTEEs ;   break ;
                        case ncltSINGLEBOLD: dChar = wcsBTEEbv ;  break ;
                        case ncltDUAL:       dChar = wcsBTEEdv ;  break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                           dChar = wcsBTEEs ;                     break ;
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsBTEEbv ;                    break ;
                        default: dChar = wcsBTEEs ;               break ;
                     }
                     break ;
                  case wcsHORIZb:
                  case wcsHDASH2b:
                  case wcsHDASH3b:
                  case wcsHDASH4b:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE:     dChar = wcsBTEEbh ;  break ;
                        case ncltSINGLEBOLD: dChar = wcsBTEEb ;   break ;
                        case ncltDASH2:   case ncltDASH3:   case ncltDASH4:
                           dChar = wcsBTEEbh ;                    break ;
                        case ncltDUAL:
                        case ncltDASH2BOLD:  case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsBTEEb ;                     break ;
                        default: dChar = wcsBTEEbh ;              break ;
                     }
                     break ;
                  case wcsHORIZd:
                     switch ( ldef.style )
                     {
                        case ncltDUAL: dChar = wcsBTEEd ;         break ;
                        default:       dChar = wcsBTEEdh ;        break ;
                     }
                     break ;
               }
            }
            //* Mid-line intersection of line being drawn with existing line *
            else if ( ldef.cInsect != false )
            {  //* Test the captured display character *
               switch ( tChar )
               {
                  case wcsHORIZs:
                  case wcsHDASH2s:  case wcsHDASH3s:  case wcsHDASH4s:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE: case ncltDASH2:
                        case ncltDASH3: case ncltDASH4:
                           dChar = wcsINSECTs ;                   break ;
                        case ncltSINGLEBOLD: case ncltDASH2BOLD:
                        case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsINSECTbv ;                  break ;
                        case ncltDUAL:
                           dChar = wcsINSECTdv ;                  break ;
                        default:    dChar = wcsINSECTs ;          break ;
                     }
                     break ;
                  case wcsHORIZb:
                  case wcsHDASH2b:  case wcsHDASH3b:  case wcsHDASH4b:
                     switch ( ldef.style )
                     {
                        case ncltSINGLE: case ncltDASH2:
                        case ncltDASH3: case ncltDASH4:
                           dChar = wcsINSECTbh ;                  break ;
                        case ncltSINGLEBOLD: case ncltDASH2BOLD:
                        case ncltDASH3BOLD:  case ncltDASH4BOLD:
                           dChar = wcsINSECTb ;                   break ;
                        case ncltDUAL: dChar = wcsINSECTdv ;      break ;
                        default: dChar = wcsINSECTb ;             break ;
                     }
                     break ;
                  case wcsHORIZd:
                     switch ( ldef.style )
                     {
                        case ncltDUAL: dChar = wcsINSECTd ;       break ;
                        default: dChar = wcsINSECTdh ;            break ;
                     }
                     break ;
               }
            }
         }
         ldChar = true ;
      }
   }

   this->ReleaseOutputLock () ;     // release lock on output stream
   return ldChar ;
   
}  //* End dlNextChar() *

//*************************
//*      RefreshWin       *
//*************************
//******************************************************************************
//* Refresh the entire contents of the window.                                 *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::RefreshWin ( void )
{

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream
   touchwin ( this->wp ) ;          // mark entire window as needing refreshed
   wrefresh ( this->wp ) ;          // redraw the window
   this->ReleaseOutputLock () ;     // release lock on output stream

}  //* End RefreshWin() *

//*************************
//*    WinDimensions      *
//*************************
//******************************************************************************
//* Returns the window dimensions in lines and columns.                        *
//*                                                                            *
//*                                                                            *
//* Input  : lines (by reference) receives lines in current window             *
//*          columns (by reference) receives columns in current window         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::WinDimensions ( short& lines, short& columns )
{

   lines   = this->wLines ;
   columns = this->wCols ;

}  //* End WinDimensions() *

//*************************
//*    WriteString        *
//*************************
//******************************************************************************
//* Write a character string at the specified position in the window.          *
//* Optionally, refresh the display before return to caller.                   *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                    *
//*            uStr : pointer to null-terminated UTF-8-encoded string          *
//*          OR FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wStr : pointer to null-terminated 'wide' wchar_t string         *
//*          FOR DIRECT DISPLAY OF DATA STRING IN A gString CLASS OBJECT       *
//*            gStr : a gString-class object (by reference)                    *
//*          NOTE: String length will be truncated at edge of window           *
//*          NOTE: ASCII control characters, 0x01 through 0x1F are ignored.    *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, then cursor position *
//*           returned will be the the column following the last character of  *
//*           displayed data (or the last column of the target line).          *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin (ZERO/ZERO) and no data will be written. *
//******************************************************************************
//* Note on cursor position returned: Y value will be the same as startY,      *
//* and X value will never be greater than last column in the window, nor less *
//* than ZERO.                                                                 *
//*                                                                            *
//* Note: Mixed left-to-right (LTR) and right-to-left (RTL) text output is not *
//* directly supported. To write mixed LTR/RTL text, you have two primary      *
//* choices:                                                                   *
//*  1) Write the sections individually, adjusting the cursor position after   *
//*     each section.                                                          *
//*  2) Reverse the order of characters in your data for one direction         *
//*     (usually the LTR text), and then write write the data as one block, in *
//*     the non-reversed direction (usually RTL).                              *
//* Please see the examples of RTL and mixed RTL/LTR ouput in the Dialog4 test *
//* application, Test06(): RTL_ContentTest.cpp                                 *
//* See also the Dialogw test application, Test07(): InfoDocTest.cpp           *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

winPos NcWindow::WriteString ( short startY, short startX, const gString& gStr, 
                               attr_t cAttr, bool refresh, bool rtl )
{
   //** Acquire an exclusive lock on the output stream.                       **
   //** Those desiring access to the output stream, will sleep on this mutex. **
   this->AcquireOutputLock () ;

   //* Be sure cursor position is within the window *
   if ( startY >= ZERO && startY < this->wLines && 
        startX >= ZERO && startX < this->wCols )
   {
      //* Test for possible wrap-around and truncate the output if necessary. *
      //* (Caller's data are not modified.)                                   *
      short colsAvail = rtl == false ?          // avaliable display columns
                        (this->wCols - startX) : (startX + 1),
            colsNeeded = gStr.gscols() ;        // display columns required

      gString ogs = gStr ;                      // local copy of data
      if ( colsAvail < colsNeeded )
         colsNeeded = ogs.limitCols(colsAvail) ; // truncate output data

      if ( colsAvail >= colsNeeded && colsNeeded > ZERO )
      {
         //* Output structure with color attribute initialized.*
         cchar_t oChunk = { cAttr } ;
         const wchar_t* sPtr = ogs.gstr() ;     // pointer to source text
         int   loopCount,                       // number of source characters
               cci,                             // index into oChunk.chars[]
               xoffset = startX ;               // X offset (for RTL cursor position)
         const short* cPtr = ogs.gscols( loopCount ) ; // array of column widths
         --loopCount ;                          // do not count null terminator
         bool  leadzero ;                       // 'true' if leading zero-length chars

         //* Set target cursor position *
         this->SetCursor ( startY, startX ) ;

         for ( short si = ZERO ; si < loopCount ; )
         {
            if ( sPtr[si] >= nckSPACE )         // ignore control codes
            {
               leadzero = false ;
               for ( cci = ZERO ; cci < CCHARW_MAX && sPtr[si] >= nckSPACE && 
                                  cPtr[si] == ZERO ; )
               {  //* Scan for and store leading, zero-width characters *
                  oChunk.chars[cci++] = sPtr[si++] ;
                  leadzero = true ;
               }
               if ( (cci < CCHARW_MAX) && (sPtr[si] >= nckSPACE) )
               {  //* Printing character (non-zero width) *
                  oChunk.chars[cci++] = sPtr[si++] ;
               }
               if ( !leadzero )        // trailing, zero-width characters
               {
                  while ( cci < CCHARW_MAX && 
                          sPtr[si] >= nckSPACE && cPtr[si] == ZERO )
                     oChunk.chars[cci++] = sPtr[si++] ;
               }
               while ( cci < CCHARW_MAX ) // fill remainder of array with nulls
                  oChunk.chars[cci++] = nckNULLCHAR ;

               wadd_wch ( this->wp, &oChunk ) ; // write the data

               //* If direction of text flow is left-to-right (LTR), *
               //* then cursor is automagically positioned correctly.*
               //* If direction of text flow is right-to-left (RTL), *
               //* then set cursor for the next character.           *
               if ( rtl != false )     // flow is right-to-left
               {
                  short i = si ;
                  while ( cPtr[i] == ZERO && sPtr[i] != nckNULLCHAR )
                     ++i ;
                  if ( sPtr[i] == nckNULLCHAR )    // end of string
                     --xoffset ;
                  else                             // width of next char to process
                     xoffset -= cPtr[si] ;
                  if ( xoffset < ZERO ) xoffset = ZERO ; // reached edge of window
                  this->SetCursor ( startY, xoffset ) ;
               }
            }
            else        // step over control codes
               ++si ;
         }
      }

      //* Refresh display if requested *
      if ( refresh != false )
         wrefresh ( this->wp ) ;

      //* Return current cursor position *
      winPos wpos = this->GetCursor () ;
      bool badX = bool((rtl == false && wpos.xpos < startX) ||
                       (rtl != false && wpos.xpos > startX) ||
                       (wpos.xpos < ZERO) || (wpos.xpos >= this->wCols)) ;
      if ( wpos.ypos != startY || badX )
      {
         wpos.ypos = startY ;
         if ( badX )
            wpos.xpos = rtl == false ? this->wCols - 1 : ZERO ;
         this->SetCursor ( wpos.ypos, wpos.xpos ) ;
      }
   }
   else
   {
      this->SetCursor ( ZERO, ZERO ) ;
   }
   winPos rpos = this->GetCursor () ; // retrieve current cursor position

   this->ReleaseOutputLock () ;     // release lock on output stream
   return rpos ;

}  //* End WriteString() *

winPos NcWindow::WriteString ( const winPos& startYX, const char* uStr, 
                               attr_t cAttr, bool refresh, bool rtl )
{

   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( uStr ) ;

   //* Output the supra-ASCII string *
   return ( this->WriteString ( startYX.ypos, startYX.xpos, ogs, 
                                cAttr, refresh, rtl ) ) ;

}  //* End WriteString *

winPos NcWindow::WriteString ( short startY, short startX, const char* uStr, 
                               attr_t cAttr, bool refresh, bool rtl )
{

   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( uStr ) ;

   //* Output the supra-ASCII string *
   return ( this->WriteString ( startY, startX, ogs, cAttr, refresh, rtl ) ) ;

}  //* End WriteString() *

winPos NcWindow::WriteString ( const winPos& startYX, const wchar_t* wStr, 
                               attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( wStr ) ;

   //* Output the supra-ASCII string *
   return ( this->WriteString ( startYX.ypos, startYX.xpos, ogs, 
                                cAttr, refresh, rtl ) ) ;

}  //* End WriteString() *

winPos NcWindow::WriteString ( short startY, short startX, const wchar_t* wStr, 
                               attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( wStr ) ;

   //* Output the supra-ASCII string *
   return ( this->WriteString ( startY, startX, ogs, cAttr, refresh, rtl ) ) ;

}  //* End WriteString() *

winPos NcWindow::WriteString ( const winPos& startYX, const gString& gStr, 
                               attr_t cAttr, bool refresh, bool rtl )
{

   return ( this->WriteString ( startYX.ypos, startYX.xpos, gStr, 
                                cAttr, refresh, rtl ) ) ;

}  //* End WriteString() *

//*************************
//*    WriteParagraph     *
//*************************
//******************************************************************************
//* Write a multi-line string, starting at the specified position.             *
//* A newline character '\n' signals the end of each line.                     *
//* Automagic line-wrapping is not supported. It is the caller's responsibility*
//* to correctly size each line to fit within the number of columns available  *
//* in the target window.                                                      *
//*                                                                            *
//* NOTE: Text will not be written outside window boundaries.                  *
//* NOTE: Except for the newline character '\n', ASCII control characters,     *
//*       0x01 through 0x1F are ignored.                                       *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          gStr   : a gString-class object (by reference)                    *
//*                OR                                                          *
//*          wStr   : pointer to a wchar_t string                              *
//*                OR                                                          *
//*          uStr   : pointer to a UTF-8 (char*) string                        *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, 'false' by default) refresh display window    *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: final position of cursor                                          *
//******************************************************************************

winPos NcWindow::WriteParagraph ( short startY, short startX, const gString& gStr, 
                                  attr_t cAttr, bool refresh, bool rtl )
{
   //** Acquire an exclusive lock on the output stream.                       **
   //** Those desiring access to the output stream, will sleep on this mutex. **
   this->AcquireOutputLock () ;

   winPos   rpos( startY, startX ) ;
   //* Be sure cursor position is within the window *
   if ( rpos.ypos >= ZERO && rpos.ypos < this->wLines && 
        rpos.xpos >= ZERO && rpos.xpos < this->wCols )
   {
      const wchar_t* wPtr ;
      gString  gs ;
      short    strLen = gStr.gschars() - 1,
               offset = ZERO ;
   
      while ( (offset < strLen) && (startY < this->wLines) )
      {
         gs = &gStr.gstr()[offset] ;   // copy remaining string to working buffer
         wPtr = gs.gstr() ;            // point to head of string
         if ( *wPtr != nckNEWLINE )// if more on line than just a newline character
         {
            short i = ZERO ;              // find newline character (or end of string)
            while ( wPtr[i] != nckNEWLINE && wPtr[i] != NULLCHAR )
               ++i ;
            offset += i ;
            gs.limitChars ( i ) ;      // print only the characters for this line
            if ( gs.gschars() > 1 )    // if there is something to print, print it
               rpos = this->WriteString ( startY, startX, gs, cAttr, refresh, rtl ) ;
         }
         else                    // first character on line is a newline character
         {
            ++offset ;           // step over the newline
            ++startY ;           // reference the next line
            //* If we are now at end of the string, *
            //* set cursor at start of current line.*
            if ( offset == strLen )
               rpos = { startY, startX } ;
         }
      }
      this->SetCursor ( rpos ) ;       // set the final cursor position
   
      //* Refresh display if requested *
      if ( refresh != false )
         wrefresh ( this->wp ) ;
   }
   else
   {
      this->SetCursor ( ZERO, ZERO ) ;
   }
   rpos = this->GetCursor () ;      // retrieve current cursor position

   this->ReleaseOutputLock () ;     // release lock on output stream
   return rpos ;

}  //* End WriteParagraph() *

winPos NcWindow::WriteParagraph ( const winPos& startYX, const gString& gStr, 
                                  attr_t cAttr, bool refresh, bool rtl )
{

   return ( this->WriteParagraph ( startYX.ypos, startYX.xpos, 
                                   gStr, cAttr, refresh, rtl ) ) ;

}  //* End WriteParagraph() *


winPos NcWindow::WriteParagraph ( short startY, short startX, const wchar_t* wStr, 
                                  attr_t cAttr, bool refresh, bool rtl )
{
   gString gs ;                        // work buffer
   winPos wp( startY, startX ) ;       // cursor position
   short wchars = wcslen ( wStr ) ;    // character count

   if ( wchars < gsALLOCDFLT )
   {
      gs = wStr ;
      return ( (this->WriteParagraph ( startY, startX, gs, 
                                       cAttr, refresh, rtl )) ) ;
   }
   else     // text data > size of gString object
   {
      short windx = ZERO, gindx ;
      while ( wchars > ZERO )
      {
         gs.loadChars( &wStr[windx], gsALLOCDFLT - 4 ) ;
         gindx = gs.findlast( nckNEWLINE ) + 1 ;
         if ( (gindx >= ZERO) && (gs.gstr()[gindx] != NULLCHAR) )
            gs.limitChars( gindx ) ;
         windx += gs.gschars() - 1;
         wchars -= gs.gschars() - 1 ;
         wp = this->WriteParagraph ( wp.ypos, wp.xpos, gs, cAttr, false, rtl ) ;
      }
      if ( refresh )
         this->RefreshWin () ;
   }

   return wp ;

}  //* End WriteParagraph() *

winPos NcWindow::WriteParagraph ( const winPos& startYX, const wchar_t* wStr, 
                                  attr_t cAttr, bool refresh, bool rtl )
{

   return ( (this->WriteParagraph ( startYX.ypos, startYX.xpos, wStr, 
                                    cAttr, refresh, rtl )) ) ;

}  //* End WriteParagraph() *

winPos NcWindow::WriteParagraph ( short startY, short startX, const char* uStr, 
                                  attr_t cAttr, bool refresh, bool rtl )
{
   gString gs ;                        // work buffer
   winPos wp( startY, startX ) ;       // cursor position
   short srcbytes = strlen ( uStr ) ;  // byte count

   // Programmer's Note: Buffer overflow tests are for max characters, NOT max bytes.
   if ( srcbytes < gsALLOCDFLT )
   {
      gs = uStr ;
      return ( (this->WriteParagraph ( startY, startX, 
                                      gs, cAttr, refresh, rtl )) ) ;
   }
   else     // text data > size of gString object
   {
      // Programmer's Note: Because we are working with bytes, NOT characters,
      // the parsing is inefficient. Sorry about that. 
      char ubuff[gsDFLTBYTES + 1] ;    // temp buffer
      short srcindx = ZERO,            // index into source string
            trgindx ;                  // index into target buffer

      while ( srcindx < srcbytes )
      {
         trgindx = -1 ;
         do
         { ubuff[++trgindx] = uStr[srcindx++] ; }
         while ( (ubuff[trgindx] != NEWLINE) && 
                 (ubuff[trgindx] != NULLCHAR) && (trgindx < gsALLOCDFLT) ) ;
         ubuff[++trgindx] = NULLCHAR ;    // be sure substring is terminated
         gs = ubuff ;
         wp = this->WriteParagraph ( wp.ypos, wp.xpos, gs, cAttr, false, rtl ) ;
      }
      if ( refresh )
         this->RefreshWin () ;
   }

   return wp ;

}  //* End WriteParagraph() *

winPos NcWindow::WriteParagraph ( const winPos& startYX, const char* uStr, 
                                  attr_t cAttr, bool refresh, bool rtl )
{

   return ( (this->WriteParagraph ( startYX.ypos, startYX.xpos, uStr, 
                                    cAttr, refresh, rtl )) ) ;

}  //* End WriteParagraph() *

//**************************
//*      WriteChar         *
//**************************
//******************************************************************************
//* Write a single character at the specified position.                        *
//* Optionally, refresh the display before return to caller.                   *
//*                                                                            *
//* Input  : startY = Y cursor position                                        *
//*          startX = X cursor position                                        *
//*                OR                                                          *
//*          startYX = Y/X cursor start position                               *
//*                                                                            *
//*          FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                    *
//*            uChar  = pointer to character to display                        *
//*                     (single-byte or multi-byte character)                  *
//*          OR FOR WCHAR_T CHARACTERS                                         *
//*            wChar  = character to display in wchar_t format                 *
//*          NOTE: ASCII control characters, 0x01 through 0x1F are ignored.    *
//*                                                                            *
//*          cAttr  = color attribute                                          *
//*          refresh= (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, AND the specified    *
//*           character is a printing character, then cursor position returned *
//*           will be the the column following the character (or the last      *
//*           column on the target line).                                      *
//*          If specified starting position is invalid, OR if non-printing     *
//*           character specified, then cursor position is set to window       *
//*           origin (ZERO/ZERO) and no data will be written.                  *
//******************************************************************************

winPos NcWindow::WriteChar ( short startY, short startX, const char* uChar, 
                             attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs(uChar, 1) ;

   //* Output the character * 
   return ( this->WriteChar ( startY, startX, ogs, cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

winPos NcWindow::WriteChar ( const winPos& startYX, const char* uChar, 
                             attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs(uChar, 1) ;

   //* Output the character * 
   return ( this->WriteChar ( startYX.ypos, startYX.xpos, ogs, 
                              cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

winPos NcWindow::WriteChar ( short startY, short startX, wchar_t wChar, 
                             attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( &wChar, 1 ) ;

   //* Output the character * 
   return ( this->WriteChar ( startY, startX, ogs, cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

winPos NcWindow::WriteChar ( const winPos& startYX, wchar_t wChar, 
                             attr_t cAttr, bool refresh, bool rtl )
{
   //* Convert source data to gString format (wide characters + statistics) *
   gString ogs( &wChar, 1 ) ;

   //* Output the character * 
   return ( this->WriteChar ( startYX.ypos, startYX.xpos, ogs, 
                              cAttr, refresh, rtl ) ) ;

}  //* End WriteChar() *

//**************************
//*      WriteChar         *
//**************************
//******************************************************************************
//* PRIVATE METHOD, called only by the public WriteChar() methods.             *
//* Write a single character at the specified position.                        *
//*                                                                            *
//* Input  : startY = Y cursor position                                        *
//*          startX = X cursor position                                        *
//*          ogs    = character to display in gString format                   *
//*                   (ASCII control characters, 0x01 through 0x1F are ignored)*
//*          cAttr  = color attribute                                          *
//*          refresh= (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, AND the specified    *
//*           character is a printing character, then cursor position returned *
//*           will be the the column following the character (or the last      *
//*           column on the target line).                                      *
//*          If specified starting position is invalid, OR if non-printing     *
//*           character specified, then cursor position is set to window       *
//*           origin (ZERO/ZERO) and no data will be written.                  *
//******************************************************************************
//* Note on cursor position returned: If specified cursor position is valid,   *
//* then returned position in Y will be the same as startY, and the returned   *
//* position in X will never be greater than last column in the window.        *
//*                                                                            *
//* Note that without counting columns, a two-column character positioned at   *
//* wCols-2 will display properly, while a two-column character positioned at  *
//* wCols-1, if not compensated, will wrap to start of next row to print.      *
//* While this is reasonable, it is ugly, and we wish to minimize ugliness in  *
//* the world.                                                                 *
//******************************************************************************

winPos NcWindow::WriteChar ( short startY, short startX, gString& ogs, 
                             attr_t cAttr, bool refresh, bool rtl )
{
   //** Acquire an exclusive lock on the output stream.                       **
   //** Those desiring access to the output stream, will sleep on this mutex. **
   this->AcquireOutputLock () ;

   //* Be sure cursor position is within the window *
   //* and that character is a printing character.  *
   if ( startY >= ZERO && startY < this->wLines && 
        startX >= ZERO && startX < this->wCols && 
        ogs.gschars() > 1 && *ogs.gstr() >= SPACE )
   {  //* Set target cursor position *
      this->SetCursor ( startY, startX ) ;

      //* Test for possible wrap-around and truncate the output if necessary. *
      short colsAvail = rtl == false ?          // avaliable display columns
                        (this->wCols - startX) : (startX + 1),
            colsNeeded = ogs.gscols() ;         // display columns required
      if ( colsAvail < colsNeeded )
         colsNeeded = ogs.limitCols(colsAvail) ;

      //* Output the supra-ASCII character *
      if ( colsAvail >= colsNeeded && colsNeeded > ZERO )
      {
         cchar_t oChunk = { cAttr, { wchar_t(*ogs.gstr()), wchar_t(NULLCHAR), 
                            wchar_t(NULLCHAR), wchar_t(NULLCHAR), wchar_t(NULLCHAR)} } ;
         wadd_wch ( this->wp, &oChunk ) ;

         //* If direction of text flow is left-to-right (LTR), *
         //* then cursor is automagically positioned correctly.*
         //* If direction of text flow is right-to-left (RTL), *
         //* then set cursor for the next character.           *
         if ( rtl != false )     // flow is right-to-left
            this->SetCursor ( startY, (startX - 1) ) ;
//            this->SetCursor ( startY, (startX - ogs.gscols()) ) ;
      }

      //* Refresh display if requested *
      if ( refresh != false )
         wrefresh ( this->wp ) ;

      //* Return current cursor position *
      winPos wpos = this->GetCursor () ;
      bool badX = bool((rtl == false && wpos.xpos < startX) ||
                       (rtl != false && wpos.xpos > startX) ||
                       (wpos.xpos < ZERO) || (wpos.xpos >= this->wCols)) ;
      if ( wpos.ypos != startY || badX )
      {
         wpos.ypos = startY ;
         if ( badX )
            wpos.xpos = rtl == false ? this->wCols - 1 : ZERO ;
         this->SetCursor ( wpos.ypos, wpos.xpos ) ;
      }
   }
   else
   {
      this->SetCursor ( ZERO, ZERO ) ;
   }
   winPos rpos = this->GetCursor () ; // retrieve current cursor position

   this->ReleaseOutputLock () ;     // release lock on output stream
   return rpos ;

}  //* End WriteChar() *

//**************************
//*     WriteHotString     *
//**************************
//******************************************************************************
//* Write a character string in the window.                                    *
//* String may contain a hotkey indicator ('^').  If so, the character that    *
//* follows it will be displayed with the 'underline' attribute.               *
//* Does not refresh the display.                                              *
//*              FOR UTF-8-ENCODED STRINGS (INCLUDING ASCII)                   *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*          uStr   : pointer to null-terminated UTF-8-encoded string          *
//*          cAttr  : color attribute                                          *
//*                                                                            *
//* Returns: final position of cursor                                          *
//******************************************************************************
//* NOTE: This method is 'protected' because derived classes need access to it.*
//*       We see no legitimate reason for an application to call it directly.  *
//*                                                                            *
//* Note on RTL output:                                                        *
//* 1) A sequence of calls to the WriteChar() method for RTL output may result *
//*    in incorrect cursor positioning if the source data are multi-column     *
//*    characters or a mixture of single-column and multi-column characters.   *
//* 2) This is because the WriteChar() method returns the X position as the    *
//*    column to the left of the character just written.                       *
//*    a) This is correct for single-column characters; however,               *
//*    b) if the next character in the sequence is a multi-column character,   *
//*       then it will overwrite the previously-written character unless an    *
//*       adjustment is made.                                                  *
//* 3) The X offset of the cursor is handled cleanly in the string output      *
//*    methods, but because WriteHotString() repeatedly calls WriteChar() so   *
//*    that the hotkey character will be correctly underlined, we must         *
//*    directly monitor the column count for the characters being output.      *
//******************************************************************************

winPos NcWindow::WriteHotString ( short startY, short startX, 
                                  const char* uStr, attr_t cAttr )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   //* Test range and set initial cursor positon *
   if ( startY < ZERO || startY >= this->wLines || startX < ZERO || startX >= this->wCols )
      startY = startX = this->scrollData.rtl ? (this->wCols - 1) : ZERO ;
   this->SetCursor ( startY, startX ) ;
   winPos wpos( startY, startX ) ;     // cursor position, return value

   //* Convert source data to gString format (wide characters + statistics)    *
   //* and get a pointer to the wide-character array.                          *
   gString oBuff( uStr ) ;
   int loopCount ;
   const wchar_t* wPtr = oBuff.gstr( loopCount ) ; 

   //* If the string contains a HOTCHAR ('^'), allow for an extra (virtual)    *
   //* display column. Note that we could remove the HOTCHAR, and remember the *
   //* position of the hotkey character, but that would mean reinitializing    *
   //* the gString object, which is too much processing.                       *
   short carrot = ZERO, hkIndex = ZERO, chCount = oBuff.gschars() ;
   for ( short i = ZERO ; i < chCount ; i++ )
   {
      if ( wPtr[i] == HOTCHAR && wPtr[i+1] != NULLCHAR )
      {
         carrot = 1 ;
         hkIndex = i ;
         break ;
      }
   }

   //* Test for possible wrap-around and truncate the output if necessary.     *
   // Programmer's Note: If caller has positioned the hotkey indicator or      *
   // the hotkey character itself beyond the available display area, the       *
   // call to oBuff.limitCols() will elimininate them. In this case,           *
   // 'carrot' will be returned to ZERO, disabling output of the hotkey.       *
   short colsAvail = this->scrollData.rtl ?     // display columns available
                           (startX + 1) : (this->wCols - startX),
         colsNeeded = oBuff.gscols() - carrot ; // display columns required
   if ( colsAvail < colsNeeded )
      colsNeeded = oBuff.limitCols(colsAvail + carrot) ;
   if ( colsAvail >= colsNeeded && colsNeeded > ZERO )
   {
      if ( hkIndex >= (oBuff.gschars() - 1) )   // if we truncated the caret
         carrot = ZERO ;
      int cnt ;                                 // array of column counts
      const short* cPtr = oBuff.gscols( cnt ) ;
      attr_t color ;

      //* Output the supra-ASCII string *
      for ( int index = ZERO ; index < loopCount ; index++ )
      {
         if ( carrot > ZERO && index == hkIndex )
         {
            ++index ;               // index the hotkey character
            color = cAttr | ncuATTR ;
         }
         else
            color = cAttr ;

         //* Adjust cursor position for RTL output *
         if ( this->scrollData.rtl )
            if ( cPtr[index] > 1 )
               wpos.xpos -= (cPtr[index] - 1) ;

         wpos = this->WriteChar ( wpos.ypos, wpos.xpos, wPtr[index], 
                                  color, false, this->scrollData.rtl ) ;
      }
   }
   this->ReleaseOutputLock () ;     // release lock on output stream
   return wpos ;

}  //* End WriteHotString() *

//*************************
//*     CaptureWindow     *
//*************************
//******************************************************************************
//* Capture the window's display data, text and color attributes.              *
//*                                                                            *
//* Input  : sw       : (by reference) initialized SaveWin object              *
//*                     specifies position and size of area to be captured     *
//*                     and contains memory allocated for the capture          *
//*                      sw.wRows = number of display rows to capture          *
//*                                 typically, number of window rows           *
//*                      sw.wCols = number of display columns to capture       *
//*                                 typically, number of window columns        *
//*                      sw.wulY  = offset in Y from upper left corner         *
//*                                 typically, ZERO                            *
//*                      sw.wulX  = offset in X from upper left corner         *
//*                                 typically, ZERO                            *
//*                                                                            *
//* Returns: OK if successful, or ERR if parameter out of range                *
//******************************************************************************
//* Programmer's Note:                                                         *
//* For 'wide' get-screen-data functions the infamous cchar_t structure is     *
//* used to capture the display data. Successful capture returns 'OK', while   *
//* 'ERR' is returned for NULL pointers and cursor positions outside window.   *
//*       int mvwin_wch(WINDOW *win, int y, int x, cchar_t *wcval);            *
//*                                                                            *
//*   Note that the underlying 'ncurses' C-language library contains some      *
//*   primitives and a test program in C for capturing window text and         *
//*   color attributes; however, those primitives are buggy, inflexible        *
//*   and know nothing of overlapping windows. Don't use them!                 *
//*                                                                            *
//* Cursor positioning for multi-column characters requires special handling;  *
//* otherwise, the character would be captured twice and would throw off the   *
//* saved-data index for the following line.                                   *
//*                                                                            *
//* Characters requiring meta-characters will cause errors because we capture  *
//* only one wchar_t value per character.                                      *
//******************************************************************************

short NcWindow::CaptureWindow ( SaveWin& sw )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   cchar_t cct ;                    // ncurses structure for data capture
   gString gs ;                     // for calculating character columns
   short ccols ;                    // and formatting data output
   winPos pos ( sw.wulY, sw.wulX ) ;// cursor position
   short status = ERR ;             // return value

   //* Validate the position and size of capture area
   if ( (sw.wRows > ZERO) && (sw.wCols > ZERO) && 
        ( sw.wulY >= ZERO) && (sw.wulX >= ZERO) && 
        ((sw.wulY + sw.wRows) <= this->wLines) && 
        ((sw.wulX + sw.wCols) <= this->wCols) )
   {
      status = OK ;        // hope for the best

      //* Scan and save the window object's display data *
      short ci, endCol = sw.wulX + sw.wCols ;
      for ( ci = ZERO ; pos.ypos < sw.wRows ; pos.ypos++ )
      {
         
         for ( pos.xpos = sw.wulX ; pos.xpos < endCol ; pos.xpos++ )
         {
            if ( (mvwin_wch (this->wp, pos.ypos, pos.xpos, &cct )) == OK )
            {
               sw.cData[ci] = cct.chars[ZERO] ;
               sw.aData[ci] = cct.attr ;
               if ( sw.cData[ci] > TILDE ) // multi-column character?
               {
                  sw.cData[ci+1] = NULLCHAR ;
                  gs = &sw.cData[ci] ;
                  if ( (ccols = gs.gscols()) > 1 )
                     pos.xpos += (ccols - 1) ;
               }
               ++ci ;
            }
            else
               ;  // data may be partially corrupted (unlikely)
         }
      }
      sw.cData[ci] = NULLCHAR ;  // terminate the character data (just in case)
   }
   this->ReleaseOutputLock () ;     // release lock on output stream
   return status ;

}  //* End CaptureWindow() *

//************************
//* Get_NcWindow_Version *
//************************
//******************************************************************************
//* Returns a pointer to NcWindowVersion, the NcWindow class version number.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: (const char*) pointer to version string                           *
//******************************************************************************

const char* NcWindow::Get_NcWindow_Version ( void )
{

   return NcWindowVersion ;

}  //* End Get_NcWindow_Version() *

//*************************
//*       DebugMsg        *
//*************************
//******************************************************************************
//* Write a message to the current window and refresh the window.              *
//* This method has some special properties that the WriteString() method      *
//* does not.                                                                  *
//*                                                                            *
//* Color attribute is the window default color (with Standout added)          *
//*                                                                            *
//* Input  : msg     : pointer to null-terminated string                       *
//*          startY  : Y cursor start position                                 *
//*                    (optional, default == -1, write to last line of window  *
//*          startX  : X cursor start position                                 *
//*                    (optional, default == 1, write to 2nd column of window  *
//*          pause  = (optional, default==false)                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void  NcWindow::DebugMsg ( const char* msg, short startY, short startX, bool pause )
{
   this->AcquireOutputLock () ;        // acquire exclusive access to output stream
   winPos   wp = this->GetCursor () ;  // save caller's cursor position
   if ( startY == -1 )
      startY = this->wLines -1 ;       // start line is last window line
   this->ClearLine ( startY, false, startX ) ;
   this->WriteString ( startY, startX, msg, nc.bwS, true ) ;
   if ( pause != false )
      nckPause() ;
   this->SetCursor ( wp.ypos, wp.xpos ) ; // restore caller's cursor position
   this->ReleaseOutputLock () ;     // release lock on output stream
   
}  //* End DebugMsg() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


