//********************************************************************************
//* File       : NcKey.cpp                                                       *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NCurses.hpp             *
//* Date       : 01-Jun-2024                                                     *
//* Version    : (see NCursesVersion string in NCurses.cpp)                      *
//*                                                                              *
//* Description: This module is a part of the NCurses class and contains         *
//* a majority of the methods used for accessing the user's keyboard input.      *
//*                                                                              *
//* This method also contains methods for screen-oriented mouse support.         *
//* Window-oriented mouse support is not provided here.                          *
//*                                                                              *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*  There is a polite fiction that each window has its own key I/O stream,      *
//*  but actually there is only one key I/O stream. Therefore it is not          *
//*  necessary for the caller of these methods to specify a WINDOW*. In fact,    *
//*  for NcWindow and NcDialog classes, the WINDOW* is private data and not      *
//*  available to the application programmer.  However, the wget_wch() and       *
//*  other key input functions of the underlying ncurses engine require a        *
//*  WINDOW* parameter. For this reason, we don't bother the caller with the     *
//*  WINDOW* but instead use the NCurses class private variable stdscrAddress    *
//*  for these calls.                                                            *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

//*********************
//* Local Definitions *
//*********************
//* Controls conditional compile of escape sequences in 'EscSeqTable' below.   *
//* These escape sequences are captured and translated into keycodes by the    *
//* ncurses library, and we need not translate them here. For testing purposes,*
//* however, we can disable the ncurses library keypad and translate everything*
//* locally.                                                                   *
#define ncXLATE (1)

//* The NCurses class translates some escape sequences that the ncurses        *
//* library has passed through UNLESS the 'rawData' parameter of GetKeyInput() *
//* is set. However, at the application or library level it is not convenient  *
//* to globally set this flag for testing, so for testing purposes only, we    *
//* can force the 'rawData' flag 'true' to temporarily disable NCurses-class   *
//* escape-sequence translations. Must be enabled in production builds.        *
#define LOCAL_XLATE_DISABLED (0)

//* The ncurses library is inconsistent about translating escape sequences     *
//* into keycodes for the numeric keypad keys. The key returned is dependent   *
//* upon the state of the 'Numlock' key and 'Shift key. With 'Numlock' on, or  *
//* if 'Numlock' is off but 'Shift' pressed, then ASCII characters are         *
//* returned.                                                                  *
//*                                                                            *
//* With 'Numlock' off (and no 'Shift'), ncurses fails to translate one or     *
//* more of the following:                                                     *
//*    1 (End)   == 0x01B 0x05B 0x034 0x07E                                    *
//*    5 (center)== 0x01B 0x05B 0x045                                          *
//*    7 (Home)  == 0x01B 0x05B 0x031 0x07E                                    *
//*    +         == 0x01B 0x04F 0x06B                                          *
//*    -         == 0x01B 0x04F 0x06D                                          *
//*    *         == 0x01B 0x04F 0x06A                                          *
//*    /         == 0x01B 0x04F 0x06F                                          *
//* For greater consistency, if we receive an untranslated escape sequence for *
//* any of these, we assign a keycode to the sequences (see below).            *
//*                                                                            *
//* With Numlock ON (regardless of Alt/Ctrl/Shift), as expected, we get:       *
//* 0 == 0    5 == 5    . == .    ENTER == ENTER                               *
//* 1 == 1    6 == 6    + == +                                                 *
//* 2 == 2    7 == 7    - == -                                                 *
//* 3 == 3    8 == 8    * == *                                                 *
//* 4 == 4    9 == 9    / == /                                                 *
//*                                                                            *
//* We document the escape sequences that are generated for these keys         *
//* (U.S. English keyboard, Numlock ON), and for the untranslated sequences,   *
//* assign the equivalent non-keypad value.                                    *
#define XLATE_NUMPAD (1)

//* Lookup table for:                                               *
//* ASCII Control codes, Enter, TAB, Escape and NULL                *
//* (keep table synchronized with definitions in NCursesKeyDef.hpp) *
//* Programmer's Note: For some reason, ncurses returns 0x007F      *
//* (ASCII Delete) with a key type of wktPRINT (i.e. 0).            *
#define TT1_ENTRIES (32)
#define TT1_MIN (0x0000)
#define TT1_MAX (0x007F)
const char* TransTable1[TT1_ENTRIES] = 
{
   "nckNULL  ", "nckC_A   ", "nckC_B   ", "nckC_C   ", "nckC_D   ", "nckC_E   ", 
   "nckC_F   ", "nckC_G   ", "nckC_H   ", "nckTAB   ", "nckENTER ", "nckC_K   ", 
   "nckC_L   ", "nckC_M   ", "nckC_N   ", "nckC_O   ", "nckC_P   ", "nckC_Q   ", 
   "nckC_R   ", "nckC_S   ", "nckC_T   ", "nckC_U   ", "nckC_V   ", "nckC_W   ", 
   "nckC_X   ", "nckC_Y   ", "nckC_Z   ", "nckESC   ", "nckC_4   ", "nckC_5   ", 
   "nckC_6   ", "nckC_7,  "  "asciiDEL "
} ;


//********************************************************************
//* Lookup table for:                                                *
//* Characters between TT2_MIN and TT2_MAX.                          *
//* This is the range of keycodes available to the ncurses library;  *
//* however, be aware that the library has only 64 open codes        *
//* within this range, and the remainder are left undefined,         *
//* although the system/terminal-emulator may provide translated     *
//* keycodes for additional key combinations.                        *
//* Note that the minimum keycode value in ncurses is 400 (octal)    *
//* or 0x100. Thus, we should never see a keycode between 0x07F and  *
//* 0x0FF. However, table lookup is much cleaner if we do not have   *
//* to perform offset adjustments, so we include the never-seen      *
//* keycodes.                                                        *
//*         Keep table synchronized with definitions in:             *
//*                 NCursesKeyDef_auto.hpp                           *
//*         and particularly with the section of NCursesKeyDef.hpp   *
//*         which incorporates these keycodes.                       *
//*                                                                  *
//*  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  --- *
//* IMPORTANT NOTE: A few keycodes in this table are dependent on    *
//* the ncurses library major/minor version and the patch date as    *
//* well as OS distribution and terminal-emulator definition.        *
//* For this reason the table may be dynamically generated using     *
//* system-specific information. For details, please refer to the    *
//* "Keymap" utility packaged with the NcDialogAPI distribution.     *
//*                                                                  *
//********************************************************************

#define TT2_ENTRIES (465)
#define TT2_MIN (0x0080)
#define TT2_MAX (0x0250)
const char* TransTable2[TT2_ENTRIES] = 
{
   "ascii_x80", "ascii_x81", "ascii_x82", "ascii_x83", "ascii_x84", "ascii_x85", "ascii_x86", "ascii_x87", 
   "ascii_x88", "ascii_x89", "ascii_x8A", "ascii_x8B", "ascii_x8C", "ascii_x8D", "ascii_x8E", "ascii_x8F", 
   "ascii_x90", "ascii_x91", "ascii_x92", "ascii_x93", "ascii_x94", "ascii_x95", "ascii_x96", "ascii_x97", 
   "ascii_x98", "ascii_x99", "ascii_x9A", "ascii_x9B", "ascii_x9C", "ascii_x9D", "ascii_x9E", "ascii_x9F", 
   "ascii_xA0", "ascii_xA1", "ascii_xA2", "ascii_xA3", "ascii_xA4", "ascii_xA5", "ascii_xA6", "ascii_xA7", 
   "ascii_xA8", "ascii_xA9", "ascii_xAA", "ascii_xAB", "ascii_xAC", "ascii_xAD", "ascii_xAE", "ascii_xAF", 
   "ascii_xB0", "ascii_xB1", "ascii_xB2", "ascii_xB3", "ascii_xB4", "ascii_xB5", "ascii_xB6", "ascii_xB7", 
   "ascii_xB8", "ascii_xB9", "ascii_xBA", "ascii_xBB", "ascii_xBC", "ascii_xBD", "ascii_xBE", "ascii_xBF", 
   "ascii_xC0", "ascii_xC1", "ascii_xC2", "ascii_xC3", "ascii_xC4", "ascii_xC5", "ascii_xC6", "ascii_xC7", 
   "ascii_xC8", "ascii_xC9", "ascii_xCA", "ascii_xCB", "ascii_xCC", "ascii_xCD", "ascii_xCE", "ascii_xCF", 
   "ascii_xD0", "ascii_xD1", "ascii_xD2", "ascii_xD3", "ascii_xD4", "ascii_xD5", "ascii_xD6", "ascii_xD7", 
   "ascii_xD8", "ascii_xD9", "ascii_xDA", "ascii_xDB", "ascii_xDC", "ascii_xDD", "ascii_xDE", "ascii_xDF", 
   "ascii_xE0", "ascii_xE1", "ascii_xE2", "ascii_xE3", "ascii_xE4", "ascii_xE5", "ascii_xE6", "ascii_xE7", 
   "ascii_xE8", "ascii_xE2", "ascii_xEA", "ascii_xEB", "ascii_xEC", "ascii_xED", "ascii_xEE", "ascii_xEF", 
   "ascii_xF0", "ascii_xF1", "ascii_xF2", "ascii_xF3", "ascii_xF4", "ascii_xF5", "ascii_xF6", "ascii_xF7", 
   "ascii_xF8", "ascii_xF9", "ascii_xFA", "ascii_xFB", "ascii_xFC", "ascii_xFD", "ascii_xFE", "ascii_xFF",
   "KEY_YES  ", "KEY_MIN  ", //0x100-0x101

         //***** THIS PORTION OF THE TABLE IS PROGRAMATICALLY GENERATED *****
   //*****  Generated by Keymap v:0.0.06 2024-06-01,- ncursesw v:6.4.20230520  *****
   "nckDOWN   ", "nckUP     ", "nckLEFT   ", "nckRIGHT  ", "nckHOME   ", //0x102-0x106
   "nckBKSP   ", "KEY_F0    ", "nckF01    ", "nckF02    ", "nckF03    ", //0x107-0x10B
   "nckF04    ", "nckF05    ", "nckF06    ", "nckF07    ", "nckF08    ", //0x10C-0x110
   "nckF09    ", "nckF10    ", "nckF11    ", "nckF12    ", "nckSF01   ", //0x111-0x115
   "nckSF02   ", "nckSF03   ", "nckSF04   ", "nckSF05   ", "nckSF06   ", //0x116-0x11A
   "nckSF07   ", "nckSF08   ", "nckSF09   ", "nckSF10   ", "nckSF11   ", //0x11B-0x11F
   "nckSF12   ", "nckCF01   ", "nckCF02   ", "nckCF03   ", "nckCF04   ", //0x120-0x124
   "nckCF05   ", "nckCF06   ", "nckCF07   ", "nckCF08   ", "nckCF09   ", //0x125-0x129
   "nckCF10   ", "nckCF11   ", "nckCF12   ", "nckCSF01  ", "nckCSF02  ", //0x12A-0x12E
   "nckCSF03  ", "nckCSF04  ", "nckCSF05  ", "nckCSF06  ", "nckCSF07  ", //0x12F-0x133
   "nckCSF08  ", "nckCSF09  ", "nckCSF10  ", "nckCSF11  ", "nckCSF12  ", //0x134-0x138
   "key_x139  ", "key_x13A  ", "key_x13B  ", "key_x13C  ", "key_x13D  ", //0x139-0x13D
   "key_x13E  ", "key_x13F  ", "key_x140  ", "key_x141  ", "key_x142  ", //0x13E-0x142
   "key_x143  ", "key_x144  ", "key_x145  ", "key_x146  ", "key_x147  ", //0x143-0x147
   "key_x148  ", "key_x149  ", "nckDELETE ", "nckINSERT ", "key_x14C  ", //0x148-0x14C
   "key_x14D  ", "key_x14E  ", "key_x14F  ", "nckSDOWN  ", "nckSUP    ", //0x14D-0x151
   "nckPGDOWN ", "nckPGUP   ", "key_x154  ", "key_x155  ", "key_x156  ", //0x152-0x156
   "nckpENTER ", "key_x158  ", "key_x159  ", "key_x15A  ", "key_x15B  ", //0x157-0x15B
   "key_x15C  ", "key_x15D  ", "key_x15E  ", "key_x15F  ", "key_x160  ", //0x15C-0x160
   "nckSTAB   ", "nckpCENTER", "key_x163  ", "key_x164  ", "key_x165  ", //0x161-0x165
   "key_x166  ", "key_x167  ", "nckEND    ", "key_x169  ", "key_x16A  ", //0x166-0x16A
   "key_x16B  ", "key_x16C  ", "key_x16D  ", "key_x16E  ", "key_x16F  ", //0x16B-0x16F
   "key_x170  ", "key_x171  ", "key_x172  ", "key_x173  ", "key_x174  ", //0x170-0x174
   "key_x175  ", "key_x176  ", "key_x177  ", "key_x178  ", "key_x179  ", //0x175-0x179
   "key_x17A  ", "key_x17B  ", "key_x17C  ", "key_x17D  ", "key_x17E  ", //0x17A-0x17E
   "nckSDEL   ", "key_x180  ", "key_x181  ", "nckSEND   ", "key_x183  ", //0x17F-0x183
   "key_x184  ", "key_x185  ", "key_x186  ", "nckSHOME  ", "key_x188  ", //0x184-0x188
   "nckSLEFT  ", "key_x18A  ", "key_x18B  ", "nckSPGDN  ", "key_x18D  ", //0x189-0x18D
   "nckSPGUP  ", "key_x18F  ", "key_x190  ", "key_x191  ", "nckSRIGHT ", //0x18E-0x192
   "key_x193  ", "key_x194  ", "key_x195  ", "key_x196  ", "key_x197  ", //0x193-0x197
   "key_x198  ", "KEY_MOUSE ", "nckRESIZE ", "key_x19B  ", "key_x19C  ", //0x198-0x19C
   "key_x19D  ", "key_x19E  ", "key_x19F  ", "key_x1A0  ", "key_x1A1  ", //0x19D-0x1A1
   "key_x1A2  ", "key_x1A3  ", "key_x1A4  ", "key_x1A5  ", "key_x1A6  ", //0x1A2-0x1A6
   "key_x1A7  ", "key_x1A8  ", "key_x1A9  ", "key_x1AA  ", "key_x1AB  ", //0x1A7-0x1AB
   "key_x1AC  ", "key_x1AD  ", "key_x1AE  ", "key_x1AF  ", "key_x1B0  ", //0x1AC-0x1B0
   "key_x1B1  ", "key_x1B2  ", "key_x1B3  ", "key_x1B4  ", "key_x1B5  ", //0x1B1-0x1B5
   "key_x1B6  ", "key_x1B7  ", "key_x1B8  ", "key_x1B9  ", "key_x1BA  ", //0x1B6-0x1BA
   "key_x1BB  ", "key_x1BC  ", "key_x1BD  ", "key_x1BE  ", "key_x1BF  ", //0x1BB-0x1BF
   "key_x1C0  ", "key_x1C1  ", "key_x1C2  ", "key_x1C3  ", "key_x1C4  ", //0x1C0-0x1C4
   "key_x1C5  ", "key_x1C6  ", "key_x1C7  ", "key_x1C8  ", "key_x1C9  ", //0x1C5-0x1C9
   "key_x1CA  ", "key_x1CB  ", "key_x1CC  ", "key_x1CD  ", "key_x1CE  ", //0x1CA-0x1CE
   "key_x1CF  ", "key_x1D0  ", "key_x1D1  ", "key_x1D2  ", "key_x1D3  ", //0x1CF-0x1D3
   "key_x1D4  ", "key_x1D5  ", "key_x1D6  ", "key_x1D7  ", "key_x1D8  ", //0x1D4-0x1D8
   "key_x1D9  ", "key_x1DA  ", "key_x1DB  ", "key_x1DC  ", "key_x1DD  ", //0x1D9-0x1DD
   "key_x1DE  ", "key_x1DF  ", "key_x1E0  ", "key_x1E1  ", "key_x1E2  ", //0x1DE-0x1E2
   "key_x1E3  ", "key_x1E4  ", "key_x1E5  ", "key_x1E6  ", "key_x1E7  ", //0x1E3-0x1E7
   "key_x1E8  ", "key_x1E9  ", "key_x1EA  ", "key_x1EB  ", "key_x1EC  ", //0x1E8-0x1EC
   "key_x1ED  ", "key_x1EE  ", "key_x1EF  ", "key_x1F0  ", "key_x1F1  ", //0x1ED-0x1F1
   "key_x1F2  ", "key_x1F3  ", "key_x1F4  ", "key_x1F5  ", "key_x1F6  ", //0x1F2-0x1F6
   "key_x1F7  ", "key_x1F8  ", "key_x1F9  ", "key_x1FA  ", "key_x1FB  ", //0x1F7-0x1FB
   "key_x1FC  ", "key_x1FD  ", "key_x1FE  ", "key_x1FF  ", "key_x200  ", //0x1FC-0x200
   "key_x201  ", "key_x202  ", "key_x203  ", "key_x204  ", "key_x205  ", //0x201-0x205
   "key_x206  ", "key_x207  ", "key_x208  ", "key_x209  ", "key_x20A  ", //0x206-0x20A
   "key_x20B  ", "key_x20C  ", "key_x20D  ", "nckADEL   ", "nckASDEL  ", //0x20B-0x20F
   "nckCDEL   ", "nckCSDEL  ", "nckpACDEL ", "key_x213  ", "nckADOWN  ", //0x210-0x214
   "nckASDOWN ", "nckCDOWN  ", "nckCSDOWN ", "nckACDOWN ", "nckAEND   ", //0x215-0x219
   "nckASEND  ", "nckCEND   ", "nckCSEND  ", "nckACEND  ", "nckAHOME  ", //0x21A-0x21E
   "nckASHOME ", "nckCHOME  ", "nckCSHOME ", "nckACHOME ", "nckAINSERT", //0x21F-0x223
   "key_x224  ", "key_x225  ", "key_x226  ", "key_x227  ", "nckALEFT  ", //0x224-0x228
   "nckASLEFT ", "nckCLEFT  ", "nckCSLEFT ", "nckACLEFT ", "nckAPGDN  ", //0x229-0x22D
   "nckASPGDN ", "nckCPGDN  ", "key_x230  ", "nckACPGDN ", "nckAPGUP  ", //0x22E-0x232
   "nckASPGUP ", "nckCPGUP  ", "key_x235  ", "nckACPGUP ", "nckARIGHT ", //0x233-0x237
   "nckASRIGHT", "nckCRIGHT ", "nckCSRIGHT", "nckACRIGHT", "key_x23C  ", //0x238-0x23C
   "nckAUP    ", "nckASUP   ", "nckCUP    ", "nckCSUP   ", "nckACUP   ", //0x23D-0x241
   "key_x242  ", "key_x243  ", "key_x244  ", "key_x245  ", "key_x246  ", //0x242-0x246
   "nckpPLUS  ", "key_x248  ", "nckpDIV   ", "key_x24A  ", "nckpMULT  ", //0x247-0x24B
   "nckpMINUS ", "key_x24D  ", "key_x24E  ", "key_x24F  ", "key_x250  ", //0x24C-0x250
//***** THIS PORTION OF THE TABLE IS PROGRAMATICALLY GENERATED *****
} ;

#define TT3_ENTRIES (128)
#define TT3_MIN (0x0001)
#define TT3_MAX (0x007E)
//* Lookup table for:                                               *
//* Alt + Alphabetical keys and Alt + Shift + Alphabetical keys     *
//* and Alt + Ctrl + Alphabetical keys.                             *
//* Note that these keycodes are not part of the ncurses C library. *
const char* TransTable3[TT3_ENTRIES] = 
{
   "NULLCHAR ",
   //* ALT + CTRL + Alpha character (shifted or not shifted) *
   //* (nckAC_J == nckAC_M)
   "nckAC_A  ", "nckAC_B  ", "nckAC_C  ", "nckAC_D  ", "nckAC_E  ", "nckAC_F  ", 
   "nckAC_G  ", "nckAC_H  ", "nckAC_I  ", "nckAC_J  ", "nckAC_K  ", "nckAC_L  ", 
   "nckAC_M  ", "nckAC_N  ", "nckAC_O  ", "nckAC_P  ", "nckAC_Q  ", "nckAC_R  ", 
   "nckAC_S  ", "nckAC_T  ", "nckAC_U  ", "nckAC_V  ", "nckAC_W  ", "nckAC_X  ", 
   "nckAC_Y  ", "nckAC_Z  ", 
   //* Unavailable key codes *
   "nck____1B",   // not a key
   "nck____1C",   // ALT+CTRL+4 == CTRL+4
   "nck____1D",   // ALT+CTRL+5 == CTRL+5
   "nck____1E",   // ALT+CTRL+6 == CTRL+6
   "nck____1F",   // ALT+CTRL+7 == CTRL+7
   //* ALT + Punctuation character  -OR-  ALT + SHIFT + Punctuation character *
   "nckAS_SPC",   // ALT+CTRL+SPACE == ALT+SHIFT+SPACE (available-but-ambiguous)
   "nckAS_1  ", "nckA_DQUO", "nckAS_3  ", "nckAS_4  ", "nckAS_5  ", "nckAS_7  ", 
   "nckA_SQUO", "nckAS_9  ", "nckAS_0  ", "nckAS_8  ", "nckA_PLUS", "nckA_COMM", 
   "nckA_MINU", "nckA_STOP", "nckA_FSTR", "nckA_0   ", "nckA_1   ", "nckA_2   ", 
   "nckA_3   ", "nckA_4   ", "nckA_5   ", "nckA_6   ", "nckA_7   ", "nckA_8   ", 
   "nckA_9   ", "nckA_COLO", "nckA_SEMI", "nckA_LCHV", "nckA_EQUL", "nckA_RCHV", 
   "nckA_QUES", "nckAS_2  ", 
   //* ALT + SHIFT + Alpha character *
   //* (nckAS_O == ESCAPE character)
   "nckAS_A  ", "nckAS_B  ", "nckAS_C  ", "nckAS_D  ", "nckAS_E  ", "nckAS_F  ", 
   "nckAS_G  ", "nckAS_H  ", "nckAS_I  ", "nckAS_J  ", "nckAS_K  ", "nckAS_L  ", 
   "nckAS_M  ", "nckAS_N  ", "nckAS_O  ", "nckAS_P  ", "nckAS_Q  ", "nckAS_R  ", 
   "nckAS_S  ", "nckAS_T  ", "nckAS_U  ", "nckAS_V  ", "nckAS_W  ", "nckAS_X  ", 
   "nckAS_Y  ", "nckAS_Z  ", 
   //* ALT + Punctuation character  -OR-  ALT + SHIFT + Punctuation character *
   //* Alt+[ == ESCAPE character, Alt+` (grave) is captured by system.
   "nckA_LBRK", "nckA_BSTR", "nckA_RBRK", "nckAS_6  ", "nckA_UNDR", "nckA_GRAV", 
   //* ALT + Alpha character *
   "nckA_A   ", "nckA_B   ", "nckA_C   ", "nckA_D   ", "nckA_E   ", "nckA_F   ", 
   "nckA_G   ", "nckA_H   ", "nckA_I   ", "nckA_J   ", "nckA_K   ", "nckA_L   ", 
   "nckA_M   ", "nckA_N   ", "nckA_O   ", "nckA_P   ", "nckA_Q   ", "nckA_R   ", 
   "nckA_S   ", "nckA_T   ", "nckA_U   ", "nckA_V   ", "nckA_W   ", "nckA_X   ", 
   "nckA_Y   ", "nckA_Z   ", 
   //* ALT + SHIFT + Punctuation character   *
   //* Alt+Shift+~ is captured by the system.*
   "nckA_LBRC", "nckA_VBAR", "nckA_RBRC", "nckA_TILD", 
   //* DELETE key combinations handled elsewhere.*
   "UNKNOWN  ", 
} ;

//* GetTransTable1 - Debugging Only *
void NCurses::GetTransTable1 ( const char**& tt1, short& tt1Min, short& tt1Max )
{

   tt1    = TransTable1 ;
   tt1Min = TT1_MIN ;
   tt1Max = TT1_MAX ;

}  //* End GetTransTable1() *

//* GetTransTable2 - Debugging Only *
void NCurses::GetTransTable2 ( const char**& tt2, short& tt2Min, short& tt2Max )
{

   tt2    = TransTable2 ;
   tt2Min = TT2_MIN ;
   tt2Max = TT2_MAX ;

}  //* End GetTransTable2() *

//* GetTransTable3 - Debugging Only *
void NCurses::GetTransTable3 ( const char**& tt3, short& tt3Entries )
{

   tt3    = TransTable3 ;
   tt3Entries = TT3_ENTRIES ;

}  //* End GetTransTable3() *


//* Lookup table for key input that arrives as escape sequences.
//* Note that entries in the table should be in numerical order,
//* low-to-high, in case we want to later add a hash table to 
//* the lookup process. Currently, there aren't enough entries 
//* to warrant a hash table, so we leave them in their logical groupings.
// Programmer's Note: Lookup verification has been manually performed on all key 
// sequences that are passed through to the test application (with ncurses 
// keypad() disabled). Sequences that are not visible without heavy lifting, 
// such as hooking the kernel's keyboard interrupt have not been verified. 
// In that case, it is hardly necessary to have an entry in the table for 
// a key that we will probably never see, so those entries are commented out.
struct EscSequence
{
   short    keyCode ;
   short    seqLen ;
   UCHAR    seq[MAX_SEQ] ;
} ;
EscSequence EscSeqTable[] = 
{
   //* Misc sequences *
   { nckSTAB, 3,     // Shift+TAB
      { 0x1B, 0x5B, 0x5A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckINSERT, 4,  // Insert key
      { 0x1B, 0x5B, 0x32, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckAINSERT, 6,  // Alt+Insert key
      { 0x1B, 0x5B, 0x32, 0x3B, 0x33, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckDELETE, 4,  // Delete key
      { 0x1B, 0x5B, 0x33, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSDEL, 6,  // Shift+Delete key
      { 0x1B, 0x5B, 0x33, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckADEL, 6,  // Alt+Delete key
      { 0x1B, 0x5B, 0x33, 0x3B, 0x33, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckASDEL, 6, // Alt+Delete key
      { 0x1B, 0x5B, 0x33, 0x3B, 0x34, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckCDEL, 6,  // Control+Delete key
      { 0x1B, 0x5B, 0x33, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckCSDEL, 6, // Control+Shift+Delete key
      { 0x1B, 0x5B, 0x33, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00, 0x00} },
//   { nckCADEL, 6, // Control+Alt+Delete key (Captured by system as "Logout")
//      { 0x1B, 0x5B, 0x33, 0x3B, 0x37, 0x7E, 0x00, 0x00, 0x00, 0x00} },
//   { nckCASDEL, 6,// Control+Alt+Shift+Delete key (untranslated & unassigned)
//      { 0x1B, 0x5B, 0x33, 0x3B, 0x38, 0x7E, 0x00, 0x00, 0x00, 0x00} },

   //***************************
   //* Arrow keys, cursor keys *
   //***************************
   #if ncXLATE != 0  // THIS GROUP CAPTURED AND TRANSLATED BY ncurses LIBRARY
   //* Up Arrow      *
   { nckUP, 3,     // translated by ncurses library
      { 0x1B, 0x5B, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSUP, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x32, 0x41, 0x00, 0x00, 0x00, 0x00} },
   { nckAUP, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x33, 0x41, 0x00, 0x00, 0x00, 0x00} },
   { nckASUP, 6,   // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x34, 0x41, 0x00, 0x00, 0x00, 0x00} },
   { nckCUP, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x35, 0x41, 0x00, 0x00, 0x00, 0x00} },
   { nckCSUP, 6,   // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x36, 0x41, 0x00, 0x00, 0x00, 0x00} },

   //* Down Arrow    *
   { nckDOWN, 3,   // translated by ncurses library
      { 0x1B, 0x5B, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSDOWN, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x32, 0x42, 0x00, 0x00, 0x00, 0x00} },
   { nckADOWN, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x33, 0x42, 0x00, 0x00, 0x00, 0x00} },
   { nckASDOWN, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x34, 0x42, 0x00, 0x00, 0x00, 0x00} },
   { nckCDOWN, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x35, 0x42, 0x00, 0x00, 0x00, 0x00} },
   { nckCSDOWN, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x36, 0x42, 0x00, 0x00, 0x00, 0x00} },

   //* Left Arrow    *
   { nckLEFT, 3,   // translated by ncurses library
      { 0x1B, 0x5B, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSLEFT, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x32, 0x44, 0x00, 0x00, 0x00, 0x00} },
   { nckALEFT, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x33, 0x44, 0x00, 0x00, 0x00, 0x00} },
   { nckASLEFT, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x34, 0x44, 0x00, 0x00, 0x00, 0x00} },
   { nckCLEFT, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x35, 0x44, 0x00, 0x00, 0x00, 0x00} },
   { nckCSLEFT, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x36, 0x44, 0x00, 0x00, 0x00, 0x00} },

   //* Right Arrow   *
   { nckRIGHT, 3,  // translated by ncurses library
      { 0x1B, 0x5B, 0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSRIGHT, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x32, 0x43, 0x00, 0x00, 0x00, 0x00} },
   { nckARIGHT, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x33, 0x43, 0x00, 0x00, 0x00, 0x00} },
   { nckASRIGHT, 6,// translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x34, 0x43, 0x00, 0x00, 0x00, 0x00} },
   { nckCRIGHT, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x35, 0x43, 0x00, 0x00, 0x00, 0x00} },
   { nckCSRIGHT, 6,// translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x36, 0x43, 0x00, 0x00, 0x00, 0x00} },
   #endif   // ncXLATE

   //* Page Up key  *
   #if ncXLATE != 0  // THIS GROUP CAPTURED AND TRANSLATED BY ncurses LIBRARY
   { nckPGUP, 4,     // translated by ncurses library
      { 0x1B, 0x5B, 0x35, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSPGUP, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x35, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckAPGUP, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x35, 0x3B, 0x33, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckASPGUP, 6,   // not available to terminal applications
      { 0x1B, 0x5B, 0x35, 0x3B, 0x34, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckACPGUP, 6,   // not available to terminal applications
      { 0x1B, 0x5B, 0x35, 0x3B, 0x37, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   #endif   // ncXLATE
//   { nckACSPGUP, 6,  // Alt+Ctrl+Shift+PageUp (untranslated & unassigned)
//      { 0x1B, 0x5B, 0x35, 0x3B, 0x38, 0x7E, 0x00, 0x00, 0x00, 0x00} },
//   { nckCPGUP, n,    // Terminal 'previous tab' key
//      { 0x1B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },

   //* Page Down key*
   #if ncXLATE != 0  // THIS GROUP CAPTURED AND TRANSLATED BY ncurses LIBRARY
   { nckPGDOWN, 4,   // Page Down - translated by ncurses library
      { 0x1B, 0x5B, 0x36, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSPGDN, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x36, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckAPGDN, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x36, 0x3B, 0x33, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   { nckASPGDN, 6,   // translated by ncurses library
      { 0x1B, 0x5B, 0x36, 0x3B, 0x34, 0x7E, 0x00, 0x00, 0x00, 0x00} },
   #endif   // ncXLATE
//   { nckACSPGDN, 6,  // Alt+Ctrl+Shift+PageDown (untranslated & unassigned)
//      { 0x1B, 0x5B, 0x36, 0x3B, 0x38, 0x7E, 0x00, 0x00, 0x00, 0x00} },
//   { nckCPGDOWN, n,  // Gnome 'next tab' key
//      { 0x1B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },

   //* Home key     *
   #if ncXLATE != 0  // THIS GROUP CAPTURED AND TRANSLATED BY ncurses LIBRARY
   { nckHOME, 3,   // translated by ncurses library
      { 0x1B, 0x4F, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSHOME, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x32, 0x48, 0x00, 0x00, 0x00, 0x00} },
   { nckAHOME, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x33, 0x48, 0x00, 0x00, 0x00, 0x00} },
   { nckASHOME, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x34, 0x48, 0x00, 0x00, 0x00, 0x00} },
   { nckCHOME, 6,  // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x35, 0x48, 0x00, 0x00, 0x00, 0x00} },
   { nckCSHOME, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x36, 0x48, 0x00, 0x00, 0x00, 0x00} },
   { nckACHOME, 6, // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x37, 0x48, 0x00, 0x00, 0x00, 0x00} },
   #endif   // ncXLATE
//   { nckACSHOME, 6,// Alt+Ctrl+Shift+Home (untranslated & unassigned)
//      { 0x1B, 0x5B, 0x31, 0x3B, 0x38, 0x48, 0x00, 0x00, 0x00, 0x00} },

   //* End key      *
   #if ncXLATE != 0  // THIS GROUP CAPTURED AND TRANSLATED BY ncurses LIBRARY
   { nckEND, 3,      // translated by ncurses library
      { 0x1B, 0x5B, 0x46, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckSEND, 6,     // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x32, 0x46, 0x00, 0x00, 0x00, 0x00} },
   { nckAEND, 6,     // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x33, 0x46, 0x00, 0x00, 0x00, 0x00} },
   { nckASEND, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x34, 0x46, 0x00, 0x00, 0x00, 0x00} },
   { nckCEND, 6,     // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x35, 0x46, 0x00, 0x00, 0x00, 0x00} },
   { nckCSEND, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x36, 0x46, 0x00, 0x00, 0x00, 0x00} },
   { nckACEND, 6,    // translated by ncurses library
      { 0x1B, 0x5B, 0x31, 0x3B, 0x37, 0x46, 0x00, 0x00, 0x00, 0x00} },
   #endif   // ncXLATE
//   { nckACSEND, 6, // Alt+Ctrl+Shift+End (untranslated & unassigned)
//      { 0x1B, 0x5B, 0x31, 0x3B, 0x38, 0x46, 0x00, 0x00, 0x00, 0x00} },


   //* Basic Function Key sequences *
   { nckF01, 3,    // (Terminal Help)
      { 0x1B, 0x4F, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF02, 3,
      { 0x1B, 0x4F, 0x51, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF03, 3,
      { 0x1B, 0x4F, 0x52, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF04, 3,
      { 0x1B, 0x4F, 0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF05, 5,
      { 0x1B, 0x5B, 0x31, 0x35, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF06, 5,
      { 0x1B, 0x5B, 0x31, 0x37, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF07, 5,
      { 0x1B, 0x5B, 0x31, 0x38, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF08, 5,
      { 0x1B, 0x5B, 0x31, 0x39, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF09, 5,
      { 0x1B, 0x5B, 0x32, 0x30, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF10, 5,
      { 0x1B, 0x5B, 0x32, 0x31, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF11, 5,  // Gnome maximize/restore toggle
      { 0x1B, 0x5B, 0x32, 0x33, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckF12, 5,
      { 0x1B, 0x5B, 0x32, 0x34, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00} },
   
   //* Shift+Function Key sequences *
   { nckSF01, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x32, 0x50, 0x00, 0x00, 0x00, 0x00} },
   { nckSF02, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x32, 0x51, 0x00, 0x00, 0x00, 0x00} },
   { nckSF03, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x32, 0x52, 0x00, 0x00, 0x00, 0x00} },
   { nckSF04, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x32, 0x53, 0x00, 0x00, 0x00, 0x00} },
   { nckSF05, 7, 
      { 0x1B, 0x5B, 0x31, 0x35, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF06, 7, 
      { 0x1B, 0x5B, 0x31, 0x37, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF07, 7, 
      { 0x1B, 0x5B, 0x31, 0x38, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF08, 7, 
      { 0x1B, 0x5B, 0x31, 0x39, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF09, 7, 
      { 0x1B, 0x5B, 0x32, 0x30, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF10, 7,  // Gnome context menu
      { 0x1B, 0x5B, 0x32, 0x31, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF11, 7, 
      { 0x1B, 0x5B, 0x32, 0x33, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },
   { nckSF12, 7, 
      { 0x1B, 0x5B, 0x32, 0x34, 0x3B, 0x32, 0x7E, 0x00, 0x00, 0x00} },

   //* Alt+Function Key sequences *
//   { nckAF01, 6,   // Gnome 'select-workspace' key
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x50, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF02, 6,   // Gnome 'enter-a-command' key
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x51, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF03, 6,     // AVAILABLE, BUT UNASSIGNED
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x52, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF04, 6,   // Gnome 'close-window' key
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x53, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF05, 6,   // unknown, not available to console
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x54, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF06, 6,   // Gnome does something weird to terminal window
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x55, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF07, 6,   // Gnome 'move-window' key
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x56, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF08, 6,   // Gnome 'resize-window' key
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x57, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF09, 6,   // is available, translated by ncurses library
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x58, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF10, 6,   // Gnome maximize/restore toggle
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x59, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF11, 6,   // is available, translated by ncurses library
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x5A, 0x00, 0x00, 0x00, 0x00} },
//   { nckAF12, 6,   // is available, translated by ncurses library
//      { 0x1B, 0x4F, 0x31, 0x3B, 0x33, 0x5B, 0x00, 0x00, 0x00, 0x00} },

   //* Control+Function Key sequences *
   { nckCF01, 6,  // unavailable
      { 0x1B, 0x4F, 0x31, 0x3B, 0x35, 0x50, 0x00, 0x00, 0x00, 0x00} },
   { nckCF02, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x35, 0x51, 0x00, 0x00, 0x00, 0x00} },
   { nckCF03, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x35, 0x52, 0x00, 0x00, 0x00, 0x00} },
   { nckCF04, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x35, 0x53, 0x00, 0x00, 0x00, 0x00} },
   { nckCF05, 7, 
      { 0x1B, 0x5B, 0x31, 0x35, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF06, 7, 
      { 0x1B, 0x5B, 0x31, 0x37, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF07, 7, 
      { 0x1B, 0x5B, 0x31, 0x38, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF08, 7, 
      { 0x1B, 0x5B, 0x31, 0x39, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF09, 7, 
      { 0x1B, 0x5B, 0x32, 0x30, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF10, 7, 
      { 0x1B, 0x5B, 0x32, 0x31, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF11, 7, 
      { 0x1B, 0x5B, 0x32, 0x33, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },
   { nckCF12, 7, 
      { 0x1B, 0x5B, 0x32, 0x34, 0x3B, 0x35, 0x7E, 0x00, 0x00, 0x00} },

   //* Control+Shift+Function Key sequences *
   { nckCSF01, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x36, 0x50, 0x00, 0x00, 0x00, 0x00} },
   { nckCSF02, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x36, 0x51, 0x00, 0x00, 0x00, 0x00} },
   { nckCSF03, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x36, 0x52, 0x00, 0x00, 0x00, 0x00} },
   { nckCSF04, 6, 
      { 0x1B, 0x4F, 0x31, 0x3B, 0x36, 0x53, 0x00, 0x00, 0x00, 0x00} },
   { nckCSF05, 7, 
      { 0x1B, 0x5B, 0x31, 0x35, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF06, 7, 
      { 0x1B, 0x5B, 0x31, 0x37, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF07, 7, 
      { 0x1B, 0x5B, 0x31, 0x38, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF08, 7, 
      { 0x1B, 0x5B, 0x31, 0x39, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF09, 7, 
      { 0x1B, 0x5B, 0x32, 0x30, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF10, 7, 
      { 0x1B, 0x5B, 0x32, 0x31, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF11, 7, 
      { 0x1B, 0x5B, 0x32, 0x33, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },
   { nckCSF12, 7, 
      { 0x1B, 0x5B, 0x32, 0x34, 0x3B, 0x36, 0x7E, 0x00, 0x00, 0x00} },

   //*************************************************
   //* Numeric keypad (for keyboards that have them) *
   //*        >> With 'Numlock' key OFF <<           *
   //*************************************************
   #if XLATE_NUMPAD != 0
   { nckpCENTER, 3,  // keypad '5'
      { 0x1B, 0x5B, 0x45, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckpPLUS, 3,    // keypad '+'
      { 0x1B, 0x4F, 0x6B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckpMINUS, 3,   // keypad '-'
      { 0x1B, 0x4F, 0x6D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckpMULT, 3,    // keypad '*'
      { 0x1B, 0x4F, 0x6A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckpDIV, 3,     // keypad '/'
      { 0x1B, 0x4F, 0x6F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckpHOME, 4,    // keypad 'Home'
      { 0x1B, 0x5B, 0x31, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   { nckpEND, 4,     // keypad 'End'
      { 0x1B, 0x5B, 0x34, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
   #endif   // XLATE_NUMPAD

   { -1, 0 }   // End-of-table code == ZERO sequence length
} ;


//*********************
//* Local Prototytes  *
//*********************



//*************************
//*     GetKeyInput       *
//*************************
//******************************************************************************
//* Get a keycode and keytype from the ncursesw input stream.                  *
//* This includes all supported international character codes as well as       *
//* captured-and-translated escape sequences representing function keys,       *
//* cursor keys and other special-purpose key combinations.                    *
//*                                                                            *
//* If mouse input has been enabled, mouse events will also be returned in     *
//* caller's wkeyCode-class object.                                            *
//* See also the GetMouseEvent() method for additional information.            *
//*                                                                            *
//* Input  : wKey   : wkeyCode-class object (by reference) to hold the key     *
//*                    information (initial values ignored)                    *
//*          rawData: (optional, false by default)                             *
//*                   if false, capture and translate escape sequences         *
//*                   if true, no local processing by the NCurses class, BUT   *
//*                   the ncurses library will still do its translations       *
//*                   UNLESS there has been a prior call to                    *
//*                   SetKeyProcState(nckpNO_TRANSLATION);                     *
//*                                                                            *
//* Returns: member of enum wkeyType                                           *
//*          wktPRINT  if key is a valid, printing wchar_t character code      *
//*          wktFUNKEY if key is a translated function key (cursor key, Fx,    *
//*                     etc) OR if key is (non-printing) control code < 0x0020.*
//*          wktMOUSE  if a mouse event in the input stream has been detected, *
//*                    then the mouse event will be returned in wKey.mevent.   *
//*                    (wKey.key will be set to ZERO and should be ignored.)   *
//*          wktEXTEND This is an extension to the key input captured by the   *
//*                     ncurses library primitives. It provides additional     *
//*                     keycodes to the application: Alt+'a' through Alt+'z',  *
//*                     Alt+Shift+'a' through Alt+Shift+'z' and Alt+Ctrl+'a'   *
//*                     through Alt+Ctrl+'z'. (See notes in NCursesKeyDef.hpp.)*
//*          wktESCSEQ if an untranslated escape sequence was captured.        *
//*                     The keycode (wKey.key) will be set to ZERO.            *
//*                     This should only happen if application has turned      *
//*                     off key-input translation (or in the rare case that    *
//*                     neither the ncurses library nor the NCurses class can  *
//*                     translate the sequence).                               *
//*                     The captured escape sequence is held in a static       *
//*                     buffer and can be retrieved via GetEscapeSequence()    *
//*                     before it is overwritten by the next untranslated      *
//*                     sequence.                                              *
//*          wktERR    if no key data available.                               *
//*                     (this method performs a non-blocking read if)          *
//*                     (SetKeyDelay() method not set to nckdFOREVER),         *
//*                     OR if system call returned an error.                   *
//*                     The keycode (wKey.key) will be set to ZERO.            *
//*                                                                            *
//*    IT IS STRONGLY RECOMMENDED THAT THE RETURN VALUE ALWAYS BE CHECKED!     *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* -------------------                                                        *
//* To use this method properly, the key processing state must be set to       *
//* nckpUNBUFFERED or nckpRAW. In nckpCOOKED mode, system does not return data *
//* until the Enter key is pressed, We short-circuit cooked-mode input         *
//* gathering by a direct call to the ncurses-library primitive. Note also     *
//* that mouse events cannot be detected while in cooked-mode. No developer    *
//* in his/her/its right mind would use cooked-mode input anyway.              *
//*                                                                            *
//*  The ncursesw primitives get_wch(), wget_wch(), mvget_wch(), mvwget_wch(), *
//*  unget_wch(), etc. are true int (32-bit) input and are used exclusively    *
//*  within the NCurses class and classes derived from it. The application     *
//*  programmer may still request 16-bit input, but this is not recommended    *
//*  in internationalized applications, and in any case is only a stub for     *
//*  the 32-bit method.                                                        *
//*                                                                            *
//*  FROM THE MAN PAGES:                                                       *
//*  ===================                                                       *
//*   #include <curses.h>                                                      *
//*    int get_wch(wint_t *wch);                               (macro)         *
//*    int wget_wch(WINDOW *win, wint_t *wch);                                 *
//*    int mvget_wch(int y, int x, wint_t *wch);               (macro)         *
//*    int mvwget_wch(WINDOW *win, int y, int x, wint_t *wch); (macro)         *
//*    int unget_wch(const wchar_t wch);                                       *
//*                                                                            *
//*   When  get_wch,  wget_wch,  mvget_wch, and mvwget_wch functions success-  *
//*   fully report the pressing of a function key, they return  KEY_CODE_YES.  *
//*             [ncurses bug: see below]                                       *
//*   When they successfully report a wide character, they return OK.          *
//*   Otherwise, they return ERR.                                              *
//*       [Note: OK==0, ERR==(-1), KEY_CODE_YES==wktFUNKEY==0x0400]            *
//*       [Note: what is 0x0100 ? probably an ncurses bug...     ]             *
//*                                                                            *
//*   Upon successful completion, unget_wch returns OK.  Otherwise, the        *
//*   function returns ERR.                                                    *
//*                                                                            *
//*   If keypad is enabled, these functions respond  to  the  pressing  of  a  *
//*   function key by setting the object pointed to by wch to the correspond-  *
//*   ing KEY_ value defined in <curses.h> and returning KEY_CODE_YES.  If  a  *
//*   character  (such  as  escape) that could be the beginning of a function  *
//*   key is received, curses sets a timer.  If the remainder of the sequence  *
//*   does arrive within the designated time, curses passes through the        *
//*   character; otherwise, curses returns the function key value.  For this   *
//*   reason,  many terminals experience a delay between the time a user       *
//*   presses the escape key and the time the escape is returned to the        *
//*   program.                                                                 *
//*                                                                            *
//*   The unget_wch function pushes the wide character wch back onto the head  *
//*   of the input queue, so the wide character is returned by the next call   *
//*   to get_wch.  The pushback of one character is guaranteed.  If the        *
//*   program calls unget_wch too many times without an intervening call to    *
//*   get_wch, the operation may fail.                                         *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//* IMPORTANT NOTE: Possible bug in ncurses definitions:                       *
//* - The ncurses library's 'curses.h' file defines the value KEY_CODE_YES as  *
//*   0x0400. However, somewhere within the other header files or library code *
//*   or maybe in the terminfo definitions, it gets re-defined as 0x0100.      *
//* - When the wget_wch() primitive encounters a function key it actually      *
//*   returns the re-defined KEY_CODE_YES (0x0100). This is a problem if we    *
//*   want to believe 'curses.h', but not a problem if we simply believe in    *
//*   our local definition.                                                    *
//*                                                                            *
//* IMPORTANT NOTE: If we are doing a blocking read AND if mouse input is      *
//*   enabled AND if we get a spurious mouse event, then we recurse.           *
//*   This is a simple solution, but could become a problem if we got hundreds *
//*   or thousands of sequential bad events. The mouse input queue is flushed  *
//*   on receipt of a bad event, so we should be OK, but keep an eye on it.    *
//*                                                                            *
//* IMPORTANT NOTE: For ncurses v:6.0.20160209 release (Ubuntu repository):    *
//*   -- wget_wch bug: ignores the blocking delay (-1) and returns immediately.*
//*   -- The delay is set AND reported correctly, but is apparently not        *
//*      referenced.                                                           *
//*   -- It is _possible_ that it is not ncurses itself, but the underlying    *
//*      X driver (or Wayland) that is causing the problem.                    *
//*      -- Test this by disabling Wayland completely and running pure X.      *
//*         It is possible that Ubuntu 16.04 has Wayland disabled by default.  *
//*         Fedora 25 and higher (GNOME) defaults to Wayland (assuming         *
//*         supported graphics hardware/drivers).                              *
//*   -- Temporarily, we can simply go into a loop waiting for key input, but  *
//*      this is inefficient and violates the spirit of a blocked thread.      *
//*   -- This bug has been repaired in ncursesw v:6.1; however, we will retain *
//*      the patch for a while.                                                *
//*                                                                            *
//*                                                                            *
//******************************************************************************

wkeyType NCurses::GetKeyInput ( wkeyCode& wKey, bool rawData )
{
   // NOTE: This bug seems to have been fixed in ncursesw v:6.1, 
   //       but retain this patch for a while just-in-case.
   #define LIB_BUGFIX (0)

   if ( this->keyPush.kCount == ZERO )    // if nothing in the pushback queue
   {
      wKey.mevent.reset() ;
      #if LIB_BUGFIX != 0  // compensate for ncurses lib bug
      chrono::duration<short, std::milli>aMoment( 10 ) ;
      gString gsver( this->nclibVersion ) ;
      bool ver6 = bool((gsver.find( L'6')) == ZERO) ;
      if ( ver6 && (this->keyInputDelay == nckdFOREVER) )
      {
         while ( (wKey.type = this->GetKeyInput ( wKey.key, rawData )) == wktERR )
            this_thread::sleep_for( aMoment ) ;
      }
      else
      #endif   // LIB_BUGFIX
      wKey.type = this->GetKeyInput ( wKey.key ) ; // (see note above)

      //* If a mouse input event has been detected *
      if ( (wKey.type == wktMOUSE) && (this->mouseEnabled != false) )
      {
         if ( (this->GetMouseEvent ( wKey.mevent )) != OK )
         {  //* No mouse event available, so we have received a false indicator.*
            #if LIB_BUGFIX != 0  // compensate for ncurses lib bug
            if ( this->keyInputDelay == nckdFOREVER )
            {
               if ( ver6 )
               {
                  while ( (wKey.type = this->GetKeyInput ( wKey.key, rawData )) == wktERR )
                     this_thread::sleep_for( aMoment ) ;
               }
               else
                  wKey.type = this->GetKeyInput ( wKey, rawData ) ;
            }
            #else    // PRODUCTION
            if ( this->keyInputDelay == nckdFOREVER )
               wKey.type = this->GetKeyInput ( wKey, rawData ) ;
            #endif   // PRODUCTION
            else
            {
               wKey.key  = ZERO ;
               wKey.type = wktERR ;
            }
         }
      }
      else           // be tidy
         wKey.mevent.reset() ;
   }
   else        // retrieve previously-saved key data
      this->RegetKeyInput ( wKey ) ;
   return wKey.type ;

}  //* End GetKeyInput() *

//*************************
//*      GetKeyInput      *
//*************************
//******************************************************************************
//* Get a keycode and keytype from the ncursesw input stream.                  *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* Input  : wChar  : returned key value (by reference, initial value ignored) *
//*          rawData: (optional, false by default)                             *
//*                   if false, capture and translate escape sequences         *
//*                   if true, no local processing by the NCurses class, BUT   *
//*                   the ncurses library will still do its translations       *
//*                   UNLESS there has been a prior call to                    *
//*                   SetKeyProcState(nckpNO_TRANSLATION);                     *
//*                                                                            *
//* Returns: member of enum wkeyType                                           *
//******************************************************************************

wkeyType NCurses::GetKeyInput ( wchar_t& wChar, bool rawData )
{
   #if LOCAL_XLATE_DISABLED != 0
   rawData = true ;
   #endif   // LOCAL_XLATE_DISABLED

   //* Verify the key input processing/filtering state *
   if ( this->keyProcState != nckpRAW && this->keyProcState != nckpUNBUFFERED )
   {
      //* If application wants it naked, skip translation *
      if ( this->keyProcState == nckpNO_TRANSLATION )
         rawData = true ;
      else if ( this->keyProcState == nckpCOOKED )
      {  //* Terrible things would happen if we were in 'cooked' mode for      *
         //* this method, so short-circuit the input gathering.                *
         //* Pushback queue is ignored in cooked mode.                         *
         //* Note also that mouse events ARE NOT available in 'cooked' mode.   *
         // ** NOTE: PREMATURE RETURN :NOTE **
         return (wkeyType)( wget_wch ( this->stdscrAddress, (wint_t*)&wChar ) ) ;
      }
   }

   //* Get a character or keycode *
   //* Note: The return value from the following call makes me nervous enough  *
   //*       for some defensive programming. See note above...                 *
   wchar_t  wchIn ;
   int      tRet = wget_wch ( this->stdscrAddress, (wint_t*)&wchIn ) ;
   wkeyType wchRet = (wkeyType)tRet ;  // convert to (short) enumerated type
   if ( wchRet != wktPRINT && wchRet != wktERR ) wchRet = wktFUNKEY ;

   //* If we have a valid, printing character code *
   //* OR a single-character control code          *
   //* OR an unhandled escape sequence.            *
   if ( wchRet == wktPRINT )
   {
      //* Printing characters need no further processing.   *
      if ( (iswprint ( wchIn )) != ZERO )
      {
         wChar = wchIn ;
      }

      //* Single-byte control codes (excluding nckESC) need *
      //* no further processing BUT are tagged as function  *
      //* keys here. (see note above)                       *
      else if (   (wchIn >= nckC_A && wchIn <= nckC_Z)
               || (wchIn >= nckC_4 && wchIn <= nckC_7)
               || (wchIn == NULLCHAR)
         )
      {
         wChar  = wchIn ;
         wchRet = wktFUNKEY ;
      }

      //* If is ESC, it may be the actual escape key or   *
      //* the beginning of an un-handled escape sequence. *
      else if ( wchIn == nckESC )
      {
         //* Set a 10 mSec delay to wait for key input *
         short oldDelay = this->GetKeyDelay () ;
         this->SetKeyDelay ( 10 ) ;       // set key input delay

         //* Set up sequence-capture buffer, and get the sequence (if any) *
         const wchar_t seqINIT = (-2) ;
         wchar_t seqBuff[MAX_SEQ] = 
                 { seqINIT, seqINIT, seqINIT, seqINIT, seqINIT, 
                   seqINIT, seqINIT, seqINIT, seqINIT, seqINIT } ;
         wchar_t  seqInput ;
         short    seqCount = ZERO ;
         seqBuff[seqCount++] = wchIn ;    // save the Escape key
         while ( seqCount < MAX_SEQ )
         {
            if ( (wget_wch ( this->stdscrAddress, (wint_t*)&seqInput )) == wktERR )
               break ;
            seqBuff[seqCount++] = seqInput ;
         }
         if ( seqBuff[1] == nckESC )
         {  //* Two or more escape keys in a row mean that user has had too    *
            //* much coffee, so report a single nckESC.                        *
            seqCount = 1 ;
         }

         //* If we have an actual sequence rather than just the Escape key  *
         if ( seqCount > 1 )
         {
            //* If caller does not want the raw sequence, translate it now *
            if ( rawData == false )
            {
               wchRet = wktESCSEQ ; // prepare for the worst
               wChar  = ZERO ;

               //* Check for members of the wktEXTEND escape-sequence group.   *
               //* The sequences we are interested in take the form: 0x1B 0xn  *
               //* where 'n' is the ASCII code in the range 'a'-'z' or 'A'-'Z' *
               //* OR the integers 0x01 - 0x1A.                                *
               //* OR the punctuation groups:                                  *
               //*     0x20 - 0x40 and 0x5B - 0x60 and 0x7B - 0x7E             *
               //* (This is faster than doing a lookup loop.)                  *
               if ( (seqCount == 2) && (seqBuff[0] == 0x1B) &&
                    ((seqBuff[1] >= 'a' && seqBuff[1] <= 'z') || 
                     (seqBuff[1] >= 'A' && seqBuff[1] <= 'Z') || 
                     (seqBuff[1] >= 0x0001 && seqBuff[1] <= 0x001A) ||
                     (seqBuff[1] >= 0x0020 && seqBuff[1] <= 0x0040) ||
                     (seqBuff[1] >= 0x005B && seqBuff[1] <= 0x0060) ||
                     (seqBuff[1] >= 0x007B && seqBuff[1] <= 0x007E)) )
               {
                  #if 1    // Dialog2, Test05
                  //* Save the escape sequence. This is used for testing only, *
                  //* but remains active for convenience. If it becomes a      *
                  //* problem, disable it. See note in StoreEscapeSequence().  *
                  this->StoreEscapeSequence ( seqBuff, seqCount ) ;
                  #endif   // Dialog2, Test05
                  wchRet = wktEXTEND ;    // set key type
                  wChar  = seqBuff[1] ;   // set key code
                  // NOTE: The assignment of the keycode value corresponds 
                  //       to nckA_n or nckAS_n or nckAC_n definitions.
               }
               //* Look up the code that matches the escape sequence *
               else
               {
                  for ( short esIndex = ZERO ; 
                        EscSeqTable[esIndex].seqLen > ZERO ; esIndex++ )
                  {
                     //* Verify that the captured sequence has the same *
                     //* number of elements as the table entry.         *
                     if ( EscSeqTable[esIndex].seqLen == seqCount )
                     {
                        short seqIndex ;
                        for ( seqIndex = ZERO ; seqIndex < seqCount ; seqIndex++ )
                        {
                           if ( EscSeqTable[esIndex].seq[seqIndex] != 
                                seqBuff[seqIndex] )
                              break ;  // not a match
                        }
                        //* If a match found, return the translation code *
                        if ( seqIndex == seqCount )
                        {
                           wChar = EscSeqTable[esIndex].keyCode ;
                           wchRet = wktFUNKEY ;
                           break ;
                        }
                     }
                  }
               }
            }
            //* If translation not wanted OR translation failed, *
            //* store the sequence in our static buffer.         *
            if ( rawData != false || wchRet == wktESCSEQ )
            {
               this->StoreEscapeSequence ( seqBuff, seqCount ) ;
               wChar  = ZERO ;            // no valid key data to return 
               wchRet = wktESCSEQ ;       // sequence has been saved
            }
         }
         else
         {  //* User pressed the Escape key *
            wChar = wchIn ;
            wchRet = wktFUNKEY ;
         }
         this->SetKeyDelay ( oldDelay ) ; // restore original key input delay
      }
      else
      {
         //* Else, we don't know what the hell it is. Let caller handle it.    *
         wChar = wchIn ;
      }
   }
   //* Else if we have a translated function key, or mouse event *
   else if ( wchRet == wktFUNKEY )
   {
      //* Check for a mouse event *
      if ( (this->mouseEnabled != false) && (wchIn == KEY_MOUSE) )
      {
         // Note that we cannot return the mouse event directly
         // because caller did not provide the data object. This return value
         // indicates that caller should call GetMouseEvent() directly
         wchRet = wktMOUSE ;
         wChar  = ZERO ;
      }
      else if ( wchIn == KEY_MOUSE )
      {  //* Unexpected receipt of a mouse event, so declare an error.*
         wchRet = wktERR ;
         wChar  = ZERO ;
      }
      else                 // return the translated function key
         wChar = wchIn ;
   }
   else  // wktERR return
   {
      wChar = ZERO ;
   }

   return wchRet ;

}  //* End GetKeyInput() *

//*************************
//*     GetKeyInput       *
//*************************
//******************************************************************************
//* Get a 16-bit keycode from the input stream.                                *
//*  Important Note: Internally, the NCurses family of classes uses 32-bit     *
//*  input and output exclusively. However this 16-bit method is available     *
//*  if it is known that 'wide' characters and keycodes will never be input    *
//*  by the user. Use of this method IS NOT RECOMMENDED.                       *
//*                                                                            *
//* Input  : sChar  : returned key value (by reference, initial value ignored) *
//*          rawData: (optional, false by default)                             *
//*                   if false, capture and translate escape sequences         *
//*                   if true, no local processing by the NCurses class        *
//* Returns: member of enum wkeyType   See GetKeyInput(wchar_t) for details.   *
//******************************************************************************
//* Note that if a mouse event is captured, we will discard it here, and       *
//* report it as wktERR.                                                       *
//*                                                                            *
//******************************************************************************

wkeyType NCurses::GetKeyInput ( short& sChar, bool rawData )
{
   //* Call the 32-bit input method *
   wkeyCode wk ;
   this->GetKeyInput ( wk, rawData ) ;

   //* If character fits into 16 bits, we're ok, but         *
   //* if character is larger than 16 bits, we have an error.*
   if ( (wk.type == wktPRINT || wk.type == wktFUNKEY) && !(wk.key & 0xFFFF0000) )
      sChar = short(wk.key) ;
   else
   {
      sChar = ZERO ;
      wk.type = wktERR ;
   }
   return wk.type ;

}  //* End GetKeyInput() *

//*************************
//*     GetKeyInput       *
//*************************
//******************************************************************************
//* Wait for a keystroke i.e. PAUSE.  The key input data are discarded.        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* Note that window-resize events are ignored.                                *
//******************************************************************************

short NCurses::GetKeyInput ( void )
{
   wkeyCode wKey ;
   do
   { this->GetKeyInput ( wKey ) ; }
   while ( wKey.type == wktFUNKEY && wKey.key == nckRESIZE ) ;
   return OK ;

}  //* End GetKeyInput() *

//*************************
//*    UngetKeyInput      *
//*************************
//******************************************************************************
//* Return the wchar_t keycode and its key type to the input queue.            *
//* - Note that no more than KEYPUSH_MAX keycode may be pushed back into the   *
//*   queue without an intervening call to 'GetKeyInput' method.               *
//* - Note that only valid keycodes can be pushed back into the queue.         *
//*   Escape sequences MAY NOT be pushed back into the queue.                  *
//* - Note that if a mouse event is contained in the input, it will be saved.  *
//*                                                                            *
//*                                                                            *
//* Input  : (by reference) key input to be returned to the input queue        *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if invalid key data, escape sequence or queue is full         *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The ncurses library function, 'unget_wch' is very limited.                 *
//* - First, only one keycode pushback can be guaranteed. This is usually not  *
//*   a problems, but is rather annoying, even for someone who grew up in the  *
//*   DOS, single-pushback world.                                              *
//* - Untranslated Escape Sequences consist of multiple keycodes, so ungetting *
//*   one keycode of many makes no sense.                                      *
//* - If the key to be returned to the queue is a translated Escape Sequence,  *
//*   (cursor key, function key, etc) the keycode will be correctly pushed,    *
//*   BUT when retrieved again, the key type will be wktPRINT which is         *
//*   incorrect.                                                               *
//* - Similarly, if the special function-key code KEY_MOUSE indicating a mouse *
//*   event is pushed, when retrieved again it will be seen as a simple        *
//*   wktPRINT type which is incorrect.                                        *
//* - The wktEXTEND keys that we translated locally are, of course not         *
//*   understood by the ncurses library.                                       *
//* - The 'unget_wch' function works properly only with printing characters    *
//*   and single-code function keys (Ctrl+A through Ctrl+Z).                   *
//*                                                                            *
//* For all of these reasons, we have made the design decision to emulate a    *
//* pushback within the NCurses code.                                          *
//*                                                                            *
//******************************************************************************

short NCurses::UngetKeyInput ( const wkeyCode& wKey )
{
   short status = OK ;
   if ( (wKey.type != wktERR) && (wKey.type != wktESCSEQ) && 
        (this->keyPush.kCount < KEYPUSH_MAX) )
   {
      this->keyPush.kPush[this->keyPush.kCount++] = wKey ;
   }
   else
      status = ERR ;
   return status ;

}  //* End UngetKeyInput() *

//*************************
//*     RegetKeyInput     *
//*************************
//******************************************************************************
//* If there are valid keycodes in the pushback queue, extract the oldest one. *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* Input  : wKey  : (by reference, initial contents ignored)                  *
//*                  receives data retrieved from queue                        *
//*                                                                            *
//* Returns: OK if data retrieved, or ERR if queue is empty                    *
//******************************************************************************

short NCurses::RegetKeyInput ( wkeyCode& wKey )
{
   short status = ERR ;
   if ( this->keyPush.kCount > ZERO )
   {
      wKey = this->keyPush.kPush[ZERO] ;     // oldest data in queue
      //* Shift remaining data down *
      --this->keyPush.kCount ;      // indexes newest remaining data (if any)
      short i = ZERO, j = 1 ;
      while ( j <= this->keyPush.kCount )
         this->keyPush.kPush[i++] = this->keyPush.kPush[j++] ;
   }
   return status ;
}  //* End RegetKeyInput() *

//*************************
//*       KeyPeek         *
//*************************
//******************************************************************************
//* If there is key data waiting in buffer, return a copy of it.               *
//* Does not remove key data from input stream.                                *
//*                                                                            *
//* Input  : wkeyCode-class object (by reference) to hold the key information  *
//*           (initial values ignored)                                         *
//*                                                                            *
//* Returns: member of enum wkeyType                                           *
//*          if key data waiting:                                              *
//*             wKey.type == wktPRINT, wktFUNKEY or wktMOUSE                   *
//*             wKey.key  == key code                                          *
//*          if no key data waiting (or untranslated escape sequence):         *
//*             wKey.type == wktERR                                            *
//*             wKey.key  == ZERO and should be ignored                        *
//******************************************************************************
//* Programmer's Note:                                                         *
//* This method must be aware of the pushback queue. Otherwise if it retrieves *
//* data from the queue, it would put it back into the queue in the wrong      *
//* order.                                                                     *
//******************************************************************************

wkeyType NCurses::KeyPeek ( wkeyCode& wKey )
{
   if ( this->keyPush.kCount == ZERO )    // if nothing in the pushback queue
   {
      //* Set a 10 mSec delay to wait for key input *
      short oldDelay = this->GetKeyDelay () ;
      this->SetKeyDelay ( 10 ) ;       // set key input delay

      //* Get waiting key input (if any) *
      this->GetKeyInput ( wKey ) ;

      if ( wKey.type == wktESCSEQ )
      {  //* Discard untranslated escape sequences.*
         wKey.type = wktERR ;
         wKey.key  = ZERO ;
      }
      //* If we received valid data, put it back in the stream *
      else if ( wKey.type != wktERR )
         this->UngetKeyInput ( wKey ) ;

      this->SetKeyDelay ( oldDelay ) ; // restore original key input delay
   }
   else        // get a copy of oldest data in pushback queue
   {
      wKey = this->keyPush.kPush[ZERO] ;
   }
   return wKey.type ;

}  //* End KeyPeek() *

//*************************
//*   GetEscapeSequence   *
//*************************
//******************************************************************************
//* If an escape sequence has not been translated into a keycode within a call *
//* to GetKeyInput() -- either because application has disabled translation,   *
//* or because the sequence is unknown -- GetKeyInput() returns wktESCSEQ and  *
//* the sequence is stored in a static buffer for potential retrieval.         *
//* This method returns a pointer to the most recently captured sequence.      *
//*                                                                            *
//* Input  : charCount (by reference, initial value ignored)                   *
//*                                                                            *
//* Returns: pointer to static buffer containing untranslated escape sequence. *
//*             charCount' gets number of characters in the sequence.          *
//*          OR returns a pointer to an empty string if no untranslated        *
//*            sequence has been captured and 'charCount' gets ZERO.           *
//******************************************************************************

const wchar_t* NCurses::GetEscapeSequence ( short& charCount )
{
static wchar_t escSequence[MAX_SEQ + 1] = { nckNULLCHAR } ;

   for ( charCount = ZERO ; escSequence[charCount] != nckNULLCHAR ; charCount++ ) ;

   return escSequence ;

}  //* End GetEscapeSequence() *

//*************************
//*  StoreEscapeSequence  *
//*************************
//******************************************************************************
//* Copy the raw escape-sequence input to our static buffer.                   *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* Input  : srcBuff : pointer to buffer containing the raw escape sequence    *
//*          srcCount: length of sequence                                      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* As a side note: this method is also used to store the escape sequences     *
//* related to the wktEXTEND group of codes, but wktEXTEND, rather than        *
//* wktESCSEQ is returned to caller, so this is a hidden functionality used    *
//* only for debugging. See Dialog2 test program, Test05().                    *
//******************************************************************************

void NCurses::StoreEscapeSequence ( const wchar_t* srcBuff, short srcCount )
{
   // Programmer's Note: What we're doing here is nasty: we are 
   // using a const pointer to write characters to the buffer, but
   // it keeps the buffer nicely isolated from the rest of the class.
   short i ;
   wchar_t* statBuffPtr = (wchar_t*)this->GetEscapeSequence ( i ) ;
   for ( i = ZERO ; i < srcCount ; i++ )
      statBuffPtr[i] = srcBuff[i] ;
   statBuffPtr[i] =  nckNULLCHAR ;  // just in case

}  //* End StoreEscapeSequence() *

//*************************
//*  FlushKeyInputStream  *
//*************************
//******************************************************************************
//* Flush (discard) any key input data currently in the key input stream.      *
//*                                                                            *
//* If mouse input enabled, then flush the mouse-event queue also.             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short NCurses::FlushKeyInputStream ( void )
{
   this->keyPush.kCount = ZERO ;       // clear the pushback queue

   //* Remember the current key-input delay *
   short oldDelay = this->GetKeyDelay () ;

   //* Set key-input delay to ZERO *
   this->SetKeyDelay ( nckdNODELAY ) ;

   wkeyCode wk ;
   do
   {
      wk.type = this->GetKeyInput ( wk.key ) ;
   }
   while ( wk.type != wktERR ) ;

   //* Sweep away mouse droppings *
   if ( this->mouseEnabled != false )
      this->FlushMouseEventQueue () ;

   //* Restore previous key-input delay *
   this->SetKeyDelay ( oldDelay ) ;

   return OK ;

}  //* End FlushKeyInputStream() *

//*************************
//*        KeyEcho        *
//*************************
//******************************************************************************
//* Enable or disable automatic echo of key input.                             *
//* Input echo is disabled by default, and should remain disabled at all times.*
//* - First, the cursor is not positioned where you think it is because        *
//*   the screen is continually being redrawn in the background. Thus, the     *
//*   echoed character will appear at random positions.                        *
//* - Second, you have no control over color attributes.                       *
//* - Do you really need a third reason?                                       *
//* Use with caution, if at all!                                               *
//*                                                                            *
//* Input  : enable : if 'true', then enable key input echo                    *
//*                   if 'false', then disable key input echo                  *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: This method is called by StartNCursesEngine(), but      *
//* the return value is ignored. If a system call fails, the state of this     *
//* parameter is undefined. This is quite unlikely, but possible.              *
//******************************************************************************

short NCurses::KeyEcho ( bool enable )
{
   short status ;

   if ( enable )
      status = echo () ;
   else
      status = noecho () ;
   if ( status == OK )
      this->inputEchoEnabled = enable ;

   return status ;

}  //* End KeyEcho() *

//*************************
//*   GetKeyProcState     *
//*************************
//******************************************************************************
//* Returns the current level of key-input processing done by the kernel,      *
//* the ncurses library, and the NCurses class before data are passed to the   *
//* application.                                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: member of ncKeyProc                                               *
//******************************************************************************

ncKeyProc NCurses::GetKeyProcState ( void )
{

   return keyProcState ;

}  //* End GetKeyProcState() *

//*************************
//*   SetKeyProcState     *
//*************************
//******************************************************************************
//* Set the level of key-input processing done by the kernel, the ncurses      *
//* library, and the NCurses class before data are passed to the application   *
//* code.                                                                      *
//* The default value on startup is nckpRAW because key input is fully         *
//* encapsulated in the NCurses class and the maximum number of keys are       *
//* available to the driving application. Have a good reason before modifying  *
//* this parameter.                                                            *
//* Applications should always request key data through the NCurses class and  *
//* NOT  directly from the system or through C or C++ functionality.           *
//*                                                                            *
//* Input  : member of ncKeyProc                                               *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: This method is called by StartNCursesEngine(), but      *
//* the return value is ignored. If a system call fails, the state of this     *
//* parameter is undefined. This is quite unlikely, but possible.              *
//******************************************************************************

short NCurses::SetKeyProcState ( ncKeyProc procState )
{
short    result = ERR ;    // if invalid parameter, return ERR

   this->keyPush.kCount = ZERO ;          // clear the pushback queue

   if ( procState == nckpNO_TRANSLATION )
      keypad ( stdscr, false ) ;          // turn off translation in ncurses
   else
      keypad ( stdscr, true ) ;           // turn on translation in ncurses

   if ( procState == nckpCOOKED )
   {
      if ( (noraw ()) == OK )
      {
         if ( (result = nocbreak ()) == OK )
         {
            this->keyProcState = nckpCOOKED ;
         }
      }
   }
   else if ( procState == nckpUNBUFFERED )
   {
      if ( (noraw ()) == OK )
      {
         if ( (result = cbreak ()) == OK )
            this->keyProcState = nckpUNBUFFERED ;
      }
   }
   else if ( procState == nckpRAW || procState == nckpNO_TRANSLATION )
   {
      if ( (nocbreak ()) == OK )
      {
         if ( (result = raw ()) == OK )
            this->keyProcState = procState ;
      }
   }

   return result ;
   
}  //* End SetKeyProcState() *

//*************************
//*     GetKeyDelay       *
//*************************
//******************************************************************************
//* Returns the current key-input delay in milliseconds.                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: delay in milliseconds OR nckdFOREVER OR nckdNODELAY               *
//******************************************************************************

short NCurses::GetKeyDelay ( void )
{

   return this->keyInputDelay ;
   
}  //* End GetKeyDelay() *

//*************************
//*      SetKeyDelay      *
//*************************
//******************************************************************************
//* Set the number of milliseconds to delay while waiting for key input.       *
//* If no key data arrives before the timeout, the key input methods           *
//* will return ERR and the target variable will be undefined.                 *
//*                                                                            *
//*                                                                            *
//* Input  : delay in milliseconds (1 to 1000)                                 *
//*           nckdFOREVER == wait indefinitely for input (blocking read)       *
//*           nckdNODELAY == returns immediately if no key data available      *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* From the man page:                                                         *
//*                                                                            *
//* void wtimeout(WINDOW *win, int delay);                                     *
//*                                                                            *
//* The  timeout and wtimeout routines set blocking or non-blocking read for   *
//* a given window. If delay is negative, blocking read is used (i.e., waits   *
//* indefinitely for input). If delay is zero, then non-blocking read  is      *
//* used (i.e.,  read  returns ERR if no input is waiting).  If delay is       *
//* positive, then read blocks for delay milliseconds, and returns ERR if      *
//* there is still no input.  Hence, these routines  provide  the  same        *
//* functionality  as nodelay, plus the additional capability of being able    *
//* to block for only delay milliseconds (where delay is positive).            *
//******************************************************************************

short NCurses::SetKeyDelay ( short delay )
{

   if ( delay < nckdFOREVER )                // truncate the value range
      delay = nckdFOREVER ;
   else if ( delay > nckdMAXDELAY )
      delay = nckdMAXDELAY ;

   wtimeout ( this->stdscrAddress, delay ) ; // set delay value
   this->keyInputDelay = delay ;             // and remember it

   return OK ;

}  //* End SetKeyDelay() *

//*************************
//*    SetKeycodeWidth   *
//*************************
//******************************************************************************
//* Set the number of significant bits of key input from keyboard controller.  *
//* Start-up default is nckbEIGHT.                                             *
//* IMPORTANT NOTE: UTF-8 encoding requires 8-bit width, so don't modify       *
//*                 default without a good reason.                             *
//*                                                                            *
//* Input  : member of ncKeyBits                                               *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: This method is called by StartNCursesEngine(), but      *
//* the return value is ignored. If a system call fails, the state of this     *
//* parameter is undefined. This is quite unlikely, but possible.              *
//******************************************************************************

short NCurses::SetKeycodeWidth ( ncKeyBits codeWidth )
{
short    result ;

   if ( (result = meta ( stdscr, bool(codeWidth == nckbEIGHT ? true : false) )) == OK )               
      this->keyBits = codeWidth ;

   return result ;

}  //* End SetKeycodeWidth() *

//*************************
//*   GetCursorState      *
//*************************
//******************************************************************************
//* Returns the current visibility state of the cursor.                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: current cursor state (member of enum ncCurVis)                    *
//******************************************************************************

ncCurVis NCurses::GetCursorState ( void )
{

   return this->cursorState ;

}  //* End GetCursorState() *

//*************************
//*    SetCursorState     *
//*************************
//******************************************************************************
//* Set the visibility state of the cursor.                                    *
//* Saves previous state for later call to RestoreCursorState().               *
//*                                                                            *
//* Input  : member of enum ncCurVis                                           *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* Programmer's Note: This method is called by StartNCursesEngine(), but      *
//* the return value is ignored. If a system call fails, the state of this     *
//* parameter is undefined. This is quite unlikely, but possible.              *
//******************************************************************************

short NCurses::SetCursorState ( ncCurVis state )
{

   this->cursorRestore = ncCurVis(curs_set ( this->cursorState = state )) ;
   return OK ;

}  //* End SetCursorState() *

//*************************
//*  RestoreCursorState   *
//*************************
//******************************************************************************
//* Restore the previous visibility state of the cursor.                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short NCurses::RestoreCursorState ( void )
{

   curs_set ( cursorState = cursorRestore ) ;
   return OK ;

}  //* End RestoreCursorState() *

//*************************
//*      BidiEnable       *
//*************************
//******************************************************************************
//* Disable or re-enable the terminal emulator's automatic control of BiDi     *
//* (Bi-Directional) output.                                                   *
//*                                                                            *
//* Traditionally, computer terminal software has supported only LTR           *
//* (Left-To-Right) languages such as English, French, Spanish and so on.      *
//* This means that languages which are written Right-To-Left required special *
//* hardware and/or software.                                                  *
//*                                                                            *
//* There are more that one billion speakers of RTL languages. Some of the     *
//* more common RTL languages are: Arabic, Hebrew, Yiddish, Pashto, Urdu,      *
//* Farsi, Dari, Kurdish and Uyghur. Several other languages are also written  *
//* Right-To-Left.                                                             *
//*                                                                            *
//* Since 2019 terminal emulator software (GnomeTerm, Konsole, etc.) have      *
//* begun to incorporate automatic detection and rendering of RTL languages    *
//* and bidirectional text generally. While this automatic "shuffling" of RTL  *
//* text data is a welcome enhancement for pure console applications, this     *
//* automatic shuffling of RTL text is in direct conflict with applications    *
//* using 'ncurses' and other direct-formatting tools.                         *
//*                                                                            *
//* For this reason it is necessary to disable the terminal program's automatic*
//* detection and reformatting of BiDi text when loading applications built on *
//* the NcDialog API (or any application that writes formatted text into the   *
//* terminal window).                                                          *
//* The NCurses Engine calls this method during start-up to disable the        *
//* terminal's BiDi detection, and calls the method again during the shut-down *
//* sequence to re-enable terminal control of BiDi data.                       *
//* UNFORTUNATELY, the commands to enable/disable RTL control are not uniform  *
//* among terminal-emulator implementations:                                   *
//*   GnomeTerm: Responds to the command sequences recommended by ECMA 48:     *
//*              Disable: "\e[8l"  and  Enable: "\e[8h"                        *
//*              See: https://terminal-wg.pages.freedesktop.org/bidi/          *
//*                   for additional information.                              *
//*   Konsole   : BiDi support may be enabled/disabled through a configuration *
//*               option: edit the current profile (Advanced Options).         *
//*               The method (if any) for programatically toggling BiDi support*
//*               is unknown at this time, so calling this method will have no *
//*               effect on KDE Konsole terminals.                             *
//*               It is recommended that Konsole users who use RTL languages   *
//*               create a profile specifically for applications based on the  *
//*               NcDialog API.                                                *
//*   XTerm     : XTerm is in many ways a reference standard. XTerm does not   *
//*               (and probably will not) implement BiDi support.              *
//*               (Good old XTerm :-)                                          *
//*   Guake     : Based on GTK-2, this Python app may not support BiDi.        *
//*               (untested)                                                   *
//*   Yakuake   : A KDE application emulating Guake for GNOME.                 *
//*               (untested)                                                   *
//*   Sakura    : Based on VTE so _should_ operate like GnomeTerm.             *
//*               (not tested)                                                 *
//*   xfce4 term: Based on VTE so _should_ operate like GnomeTerm.             *
//*               (not tested)                                                 *
//*   ROXterm   : Built on the GTK so _should_ operate like GnomeTerm.         *
//*               (not tested)                                                 *
//*   PuTTY     : (WE DON'T CARE. Windoze users have serious masochistic)      *
//*               (issues which we will not indulge here.               )      *
//*                                                                            *
//* Input  : enable : if 'false', disable terminal control of RTL output       *
//*                   if 'true',  return control of RTL output to terminal     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* 1) A call to this method must occur BEFORE the NCurses Engine is started,  *
//*    or after the NCurses Engine is shut down. Otherwise the underlying      *
//*    'ncursesw' code will capture the escape sequence before it reaches      *
//*    the terminal.                                                           *
//* 2) The output of the system call must not be redirected.                   *
//*    Although no VISIBLE output is produced, the non-printing output must    *
//*    be allowed to reach the terminal.                                       *
//* 3) "BiDi in Terminal Emulators"                                            *
//*     https://terminal-wg.pages.freedesktop.org/bidi/                        *
//* 4) See notes in BiDi_Support_In_Terminals.odt                              *
//******************************************************************************

void NCurses::BiDiEnable ( bool enable )
{
   const wchar_t bidiEnable = L'h' ;
   const wchar_t bidiDisable = L'l' ;
   gString gs( "printf \"\e[8%C\"", 
               (wchar_t*)(enable ? &bidiEnable : &bidiDisable) ) ;
   system ( gs.ustr() ) ;

}  //* End BiDiEnable() *


//******************************************************************************
//*                                                                            *
//*                         Mouse-event methods                                *
//*                                                                            *
//******************************************************************************

//*************************
//*   EnableMouseEvents   *
//*************************
//******************************************************************************
//* Allow mouse events to be reported                                          *
//*  indirectly: through the GetKeyInput methods and                           *
//*   directly : through GetMouseEvent method                                  *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//*                       (at least one event-type bit must be set)            *
//*                       Bit definitions are the BUTTONxxxx group of          *
//*                       defines in ncurses.h                                 *
//*                                                                            *
//* Returns: OK  if mouse enabled AND all bits of the event-type mask          *
//*              have been set successfully                                    *
//*          ERR if mouse not enabled OR actual event-type mask differs from   *
//*              that specified OR eventTypes == ZERO                          *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - The ncurses-library engine always starts with mouse events disabled and  *
//*   presumably with the mouse interface also disabled.                       *
//*   We must rely upon this becasuse there is no mechanism for requesting the *
//*   _current_ event-type mask without changing it.                           *
//* - Note the interesting test of whether the mouse is active.                *
//*   The mouse driver must have been initialized AND reporting for at least   *
//*   one mouse-event type must be available.                                  *
//* - Note the interesting test of whether the call has been successful.       *
//*   The driver must be active and ALL of the specified reporting types must  *
//*   have been enabled.                                                       *
//* - We allow the application to re-enable an already enabled mouse. It's just*
//*   easier that way. Also, it allows SetMouseEventTypes to call this method. *
//******************************************************************************

short NCurses::EnableMouseEvents ( mmask_t eventTypes )
{
   short status = ERR ;
   if ( eventTypes != ZERO )
   {
      this->mouseMask = mousemask ( eventTypes, &this->oldMouseMask ) ;
      this->mouseEnabled = (this->mouseMask != ZERO) && ((has_mouse ()) != false) ;
      status = (this->mouseEnabled && this->mouseMask == eventTypes) ? OK : ERR ;
   }
   return status ;

}  //* End EnableMouseEvents() *

//*************************
//*  SetMouseEventTypes   *
//*************************
//******************************************************************************
//* Modify the type(s) of mouse events which will be reported.                 *
//*                                                                            *
//* Input  : eventTypes: new mouse-event-type mask                             *
//*                                                                            *
//* Returns: OK  if mouse enabled AND all bits of the event-type mask          *
//*              have been set successfully                                    *
//*          ERR if mouse not enabled OR actual event-type mask differs from   *
//*              that specified OR eventTypes == ZERO                          *
//******************************************************************************

short NCurses::SetMouseEventTypes ( mmask_t eventTypes )
{
   short status = ERR ;
   if ( this->mouseEnabled != false && eventTypes != ZERO )
   {
      status = this->EnableMouseEvents ( eventTypes ) ;
   }
   return status ;

}  //* End SetMouseEventTypes() *

//*************************
//*  GetMouseEventTypes   *
//*************************
//******************************************************************************
//* Returns the current mouse-event-type bitmask.                              *
//*                                                                            *
//* Input  : eventTypes: (by reference, initial value ignored)                 *
//*                      receives the mouse-event-type mask                    *
//*                                                                            *
//* Returns: OK  if successtul                                                 *
//*          ERR if mouse-event input disabled                                 *
//******************************************************************************

short NCurses::GetMouseEventTypes ( mmask_t& eventTypes )
{
   short status = ERR ;
   if ( this->mouseEnabled != false )
   {
      eventTypes = this->mouseMask ;
      status = OK ;
   }
   else
      eventTypes = ZERO ;
   return status ;

}  //* End GetMouseEventTypes() *

//*************************
//* SetMouseClickInterval *
//*************************
//******************************************************************************
//* Specify the maximum interval for collecting a series of mouse-button-press *
//* and mouse-button-release events which will be interpreted as a 'click'     *
//* event. Examples:                                                           *
//* 1) Scanning for a 'single-click' event:                                    *
//*    One 'press' event followed by one 'release' event must be received      *
//*    within the specified interval.                                          *
//* 2) Scanning for a 'double-click' event:                                    *
//*    The event series 'press', 'release', press', 'release' must be received *
//*    within the specified interval.                                          *
//* 3) Scanning for a 'triple-click' event:                                    *
//*    The event series 'press', 'release', press', 'release', 'press',        *
//*    'release' must be received within the specified interval.               *
//*                                                                            *
//* Input  : max4Click : interval in thousandths (0.001) of a second           *
//*                      specify an interval OR a member of enum ncmInterval   *
//*          oldMax    : (optional, NULL pointer by default)                   *
//*                      pointer to variable to receive previous interval value*
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if value out-of-range OR mouse-event input disabled           *
//******************************************************************************

short NCurses::SetMouseClickInterval ( short max4Click, short* oldMax )
{
   short status = ERR ;
   if ( this->mouseEnabled && (max4Click >= ncmiNONE && max4Click <= ncmiMAX) )
   {
      short oldmax = mouseinterval ( max4Click ) ;
      if ( oldMax != NULL )
         *oldMax = oldmax ;
      if ( (mouseinterval(ncmiREPORT)) == max4Click )
         status = OK ;
   }
   return status ;

}  //* End SetMouseClickInterval() *

//*************************
//* GetMouseClickInterval *
//*************************
//******************************************************************************
//* Get the current maximum mouse-click interval in thousandths of a second.   *
//*                                                                            *
//* Input  : max4Click : (by reference, initial value ignored)                 *
//*                      receives current interval                             *
//*                                                                            *
//* Returns: OK  if successful, or ERR if mouse-event input disabled           *
//******************************************************************************

short NCurses::GetMouseClickInterval ( short& max4Click )
{
   short status = ERR ;
   if ( this->mouseEnabled != false )
   {
      max4Click = mouseinterval ( ncmiREPORT ) ;
      status = OK ;
   }
   return status ;

}  //* End GetMouseClickInterval() *

//*************************
//*    GetMouseEvent      *
//*************************
//******************************************************************************
//* Extract the newest mouse event from the event queue.                       *
//* - Events located beyond the writable area of the terminal window belong to *
//*   someone else and are therefore not available to us.                      *
//* - Some mouse events within the terminal window are captured by the GUI     *
//*   desktop, and are therefore not available to us.                          *
//*         Example: Button 3 click is the terminal's context menu.            *
//* - Some filtering of spurious events may be performed to prevent unexpected *
//*   events related to timing artifacts. See FilterSpuriousMouseEvents()      *
//*   method for more information.                                             *
//* - IT IS RECOMMENDED that all mouse-event capture be done through the       *
//*   GetKeyInput() method, and that the GetMouseEvent() and UngetMouseEvent() *
//*   methods should not be called directly.                                   *
//*                                                                            *
//* Input  : mevent : (by reference, initial value ignored)                    *
//*                   receives the mouse-event data                            *
//*                                                                            *
//* Returns: OK  if mouse event extracted successfully                         *
//*          ERR if no mouse event in the queue or mouse-event input disabled  *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Detection of an incomplete press/release pair by the ncurses C-language    *
//* library primitives during the click interval, may lead to unexpected and   *
//* surprising results; however every effort is made to filter these timing    *
//* artifacts to provide only reliable and useful data.                        *
//* For instance, we define the 'static bool corruptedEvent' flag to remember  *
//* a previous bad event, which may affect the present event. Specifically, an *
//* attempted double-click often results in two presses but only one release.  *
//*                                                                            *
//* 1) First, if we receive an event of a type not specified by the mask, we   *
//*    discard it. This is actually a likely scenario if the click interval    *
//*    encompasses unbalanced press/release pairs.                             *
//*    For instance, if scanning for both single-click and double-click events,*
//*    and 'press-release-press' is detected during the interval, we will      *
//*    likely get the single-click event as we should, BUT we may later get    *
//*    something unexpected like a position event, even though position events *
//*    have not not been enabled (and are actually not possible except as part *
//*    of a ScrollWheel event).                                                *
//* 2) If events are received from multiple buttons within the interval, the   *
//*    driver may go ape-shit and give us almost anything. This is especially  *
//*    true if 1-3-button emulation of button 2 is enabled in the xterm driver.*
//*    Because there is no way we can determine what message the user is TRYING*
//*    to send, we return any authorized event that we receive from this mess. *
//* 3) Scroll-wheel events:                                                    *
//*    a) xterm supports scroll-wheels (and button2-TrackPoint(tm) scroll      *
//*       emulation, although it may be disabled by default).                  *
//*    b) The ncurses library, however, has no way to reliably report these    *
//*       events, so expect the unexpected...                                  *
//*                                                                            *
//******************************************************************************

short NCurses::GetMouseEvent ( MouseEvent& mevent )
{
   short status = ERR ;
   static bool corruptedEvent = false ;
   MEVENT me ;
   if ( (this->mouseEnabled != false) && (getmouse ( &me )) == OK )
   {
      mevent.sypos = short(me.y) ;
      mevent.sxpos = short(me.x) ;
      mevent.szpos = short(me.z) ;
      mevent.deviceID = short(me.id) ;
      mevent.eventType = me.bstate ;

      //* Filter events that the application did not ask for.*
      if ( this->mouseFilter != false )
      {
         if ( ((mevent.eventType & this->mouseMask) == ZERO)
              || (corruptedEvent != false) )
         {
            if ( !corruptedEvent && (mevent.eventType & BUTTON1_PRESSED) )
               corruptedEvent = true ;
            else
               corruptedEvent = false ;
            mevent.reset() ;
            this->FlushMouseEventQueue () ;  // discard any lurking garbage
         }
         else
         {
            corruptedEvent = false ;
            status = OK ;
         }
      }
      else
      {
         corruptedEvent = false ;
         status = OK ;
      }
   }
   else
   {  //* No mouse event available.*
      mevent.reset() ;
   }
   return status ;

}  //* End GetMouseEvent() *

//*************************
//*    UngetMouseEvent    *
//*************************
//******************************************************************************
//* Push a previously-extracted mouse event back into the input queue.         *
//* - IT IS RECOMMENDED that all mouse-event capture be done through the       *
//*   GetKeyInput() method, and that the GetMouseEvent() and UngetMouseEvent() *
//*   methods should not be called directly.                                   *
//*                                                                            *
//* Input  : mevent : (by reference) contains the mouse-event data to be       *
//*                   pushed back into the event queue.                        *
//*                                                                            *
//* Returns: OK  if mouse event successfully re-queued                         *
//*          ERR if event queue is full OR if mouse-event input disabled       *
//******************************************************************************
//* Programmer's Note: At this time, we don't trust the ungetmouse() function. *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short NCurses::UngetMouseEvent ( MouseEvent& mevent )
{
   short status = ERR ;
   if ( this->mouseEnabled )
   {
      MEVENT me ;
      me.y      = mevent.sypos ;
      me.x      = mevent.sxpos ;
      me.z      = mevent.szpos ;
      me.id     = mevent.deviceID ;
      me.bstate = mevent.eventType ;
      status = ungetmouse ( &me ) ;
   }
   return status ;

}  //* End UngetMouseEvent() *

//*************************
//*   GetMousePosition    *
//*************************
//******************************************************************************
//* Report the current mouse position.                                         *
//*                                                                            *
//* Input  : mevent : (by reference, initial value ignored)                    *
//*                   receives the mouse-event data                            *
//*                                                                            *
//* Returns: OK  if mouse event extracted successfully                         *
//*          ERR if no mouse event in the queue or mouse-event input disabled  *
//******************************************************************************
//* NOTE: AT PRESENT THERE IS NO METHOD PROVIDED BY THE ncurses C-language     *
//*       LIBRARY FOR OBTAINING THE MOUSE-POINTER POSITION UNLESS A BUTTON     *
//*       EVENT PROVIDES US WITH IT.                                           *
//******************************************************************************

short NCurses::GetMousePosition ( MouseEvent& mpos )
{
   short status = ERR ;
   if ( this->mouseEnabled )
   {
// DO STUFF HERE
//      status = OK ;
   }
   return status ;

}  //* End GetMousePosition() *

//*************************
//* FlushMouseEventQueue  *
//*************************
//******************************************************************************
//* Remove any mouse events waiting in the mouse-event queue.                  *
//* The GetMouseEvent method returns events in LIFO (last-in-first-out) order. *
//* If you have received the event you're looking for and don't want any more  *
//* events, OR if your mouse has become tempramental and is sending undesired  *
//* or unexpected events, then this method can be used to flush the            *
//* mouse-event queue.                                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, or ERR if mouse-event input disabled            *
//******************************************************************************
//* Programmer's Note: Even though ncurses is not able to report position      *
//* events, many of our spurious events ARE position events because the mouse's*
//* ScrollWheel uses the position bit.                                         *
//******************************************************************************

short NCurses::FlushMouseEventQueue ( void )
{
   short status = ERR ;
   if ( this->mouseEnabled )
   {
      const short maxClear = 1000 ;    // prevents a neverending story
      MEVENT me ;
      //* Temporarily enable reporting of all mouse-button events *
      mmask_t origMask, tempMask = (ALL_MOUSE_EVENTS | REPORT_MOUSE_POSITION) ;
      mousemask ( tempMask, &origMask ) ;
      for ( short i = maxClear ; i > ZERO ; i-- )
      {
         if ( (getmouse ( &me )) == ERR )
            break ;     // no more events in queue
      }
      //* Re-enable application's mouse-event mask *
      mousemask ( origMask, NULL ) ;
      status = OK ;
   }
   return status ;

}  //* End FlushMouseEventQueue() *

//*****************************
//* FilterSpuriousMouseEvents *
//*****************************
//******************************************************************************
//* Enable or disable filtering of mouse events that arrive in the queue even  *
//* though events of that type have not been requested via the event mask.     *
//* - Filtering is enabled by default, and it is recommended that it remain    *
//*   enabled unless there is a specific need to see the spurious events.      *
//* While filtering is enabled:                                                *
//* - If an unrequested event is received by the GetMouseEvent method, then    *
//*   it is discarded, and reported as no event.                               *
//* - If an unrequested event is received in the keyboard-input queue via the  *
//*   GetKeyInput method, then the event is discarded AND:                     *
//*    a) if GetKeyInput is set to block, then wait for the next keycode       *
//*    b) if GetKeyInput is set for timeout, then wktERR is returned.          *
//* - In either case, if an unrequested event is received, then the mouse-input*
//*   queue is flushed.                                                        *
//*                                                                            *
//* Input  : filter : 'true' enables filtering                                 *
//*                   'false disables filtering                                *
//*                                                                            *
//* Returns: OK if successful, or ERR if mouse-event input disabled            *
//******************************************************************************

short NCurses::FilterSpuriousMouseEvents ( bool filter )
{
   short status = ERR ;
   if ( this->mouseEnabled )
   {
      this->mouseFilter = filter ;
      status = OK ;
   }
   return status ;

}  //* End FilterSpuriousMouseEvents() *

//*************************
//*  DisableMouseEvents   *
//*************************
//******************************************************************************
//* Disable reporting of mouse events.                                         *
//*                                                                            *
//* Input  : oldeventTypes : (optional, NULL pointer by default)               *
//*                          if !NULL, receives previous event-type bit mask   *
//*                                                                            *
//* Returns: OK  if mouse-event reporting disabled                             *
//*              (mouse driver MAY BE disabled also - system dependent)        *
//*          ERR if mouse-event reporting could not be disabled                *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Note the interesting test of whether mouse events have been disabled.      *
//* Setting the event-reporting mask to ZERO may, or may not actually disable  *
//* the mouse pointer (system dependent). In any case, if event reporting is   *
//* successfully disabled (mask==ZERO), then as far as we are concerned the    *
//* mouse is effectively disabled because we will not be bothered by events.   *
//*                                                                            *
//* The has_mouse() ncurses primitive returns 'true' if the mouse driver has   *
//* been initialized. This actually has nothing to do with anything. Once      *
//* initialized, always initialized, and disabling mouse events has no effect. *
//******************************************************************************

short NCurses::DisableMouseEvents ( mmask_t* oldeventTypes )
{
   this->mouseMask = mousemask ( ZERO, &this->oldMouseMask ) ;
   if ( oldeventTypes != NULL )
      *oldeventTypes = this->oldMouseMask ;
   this->mouseEnabled = (this->mouseMask != ZERO) ;
   return ( this->mouseEnabled == false ? OK : ERR ) ;

}  //* End DisableMouseEvents() *

//*************************
//*  MouseEventsEnabled   *
//*************************
//******************************************************************************
//* Reports whether mouse input is currently enabled.                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if mouse events can be detected and reported, else 'false'*
//******************************************************************************

bool NCurses::MouseEventsEnabled ( void )
{

   return this->mouseEnabled ;

}  //* End MouseEventsEnabled() *







#if 0 /* WORK IN PROGRESS */
// THIS IS A MORE C++ WAY OF DOING THINGS AND MAY BE A BETTER SOLUTION WHEN
// THE LOOKUP DATA COMES FROM A USER MAINTAINED FILE.
// STILL NEED TO MOVE THE LOOKUP DATA TO AN EXTERNAL, USER-MAINTAINED FILE.
//* String comparison table for escape sequence lookup. *
//* We make no claim to handle ALL escape sequences.    *
//* NOTE: In the final product, these data will come from a user-maintained file. *
static char EscSeqTable4F[EST4F_LINECOUNT][10] =
{
   { 0x1B, 0x4F, 0x32, 0x50 },                        // Shift+F1 
   { 0x1B, 0x4F, 0x32, 0x51 },                        // Shift+F2 
   { 0x1B, 0x4F, 0x32, 0x52 },                        // Shift+F3 
   { 0x1B, 0x4F, 0x32, 0x53 },                        // Shift+F4 
} ;
#define EST4F_LINECOUNT 8
static char EscSeqTable5B[EST5B_LINECOUNT][10] =
{
   { 0x1B, 0x5B, 0x31, 0x35, 0x3B, 0x32, 0x7E },      // Shift+F5 
   { 0x1B, 0x5B, 0x31, 0x37, 0x3B, 0x32, 0x7E },      // Shift+F6 
   { 0x1B, 0x5B, 0x31, 0x38, 0x3B, 0x32, 0x7E },      // Shift+F7 
   { 0x1B, 0x5B, 0x31, 0x39, 0x3B, 0x32, 0x7E },      // Shift+F8 
   { 0x1B, 0x5B, 0x32, 0x30, 0x3B, 0x32, 0x7E },      // Shift+F9 
   { 0x1B, 0x5B, 0x32, 0x31, 0x3B, 0x32, 0x7E },      // Shift+F10
   { 0x1B, 0x5B, 0x32, 0x33, 0x3B, 0x32, 0x7E },      // Shift+F11
   { 0x1B, 0x5B, 0x32, 0x34, 0x3B, 0x32, 0x7E },      // Shift+F12
} ;
struct EST_Data
{
   USHORT   code ;                           //* Single-value key code
   char     seq[MAX_SEQ] ;                   //* Key escape sequence (null terminated)
} ;
class EST_Table
{
public:
   EST_Table ()                              //* Constructor
   {
      code = c ;
      for ( short i = 0 ; s[i] != NULLCHAR ; i++ )
         seq[i] = s[i] ;
      seq[i] = NULLCHAR ;
   } ;
   ~EST_Table ()                             //* Destructor
   {
   } ;
   
   USHORT      count ;                       //* Number of records in array
   EST_Data*   data ;                        //* Pointer to data array
} ;
#endif  /* WORK IN PROGRESS */


//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


