//********************************************************************************
//* File       : idpp_file.cpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2014-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 02-Aug-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: This module contains methods for accessing files to be          *
//* processed, extracting information about each item and creating the           *
//* modified output.                                                             *
//*                                                                              *
//* These methods are adapted from a small subset of the FMgr class written      *
//* for the FileMangler file-management project by the same author.              *
//*                                                                              *
//*                                                                              *
//* Copyright Notice:                                                            *
//* -----------------                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version, PROVIDED THAT the copyright notices for both code and     *
//* documentation are included and unmodified.                                   *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//*         Full text of the GPL License may be found in the TexInfo             *
//*         documentation for this program under 'Copyright Notice'.             *
//********************************************************************************

//****************
//* Header Files *
//****************
#include "idpp.hpp"

//**********************
//* Non-member methods *
//**********************
static void stripTags ( gString& disp ) ;
static void copyTag ( const gString& gsSrc, short ti, gString& gsTrg ) ;
static void removeTag ( gString& gs, short ti ) ;
#if PARA_FIX != 0    // static prototype
static short stripPara ( gString& srctxt, short wi ) ;
#endif   // PARA_FIX

//****************
//* Local Data   *
//****************
//* If set, remove the useless comments inserted by the converter.*
//* It is assumed that this comment was some kind of debug marker.*
// Programmer's Note: Because few source lines contain instances of the target 
// substring (or multiple instances) enabling this flag causes a small 
// performance hit. If performance becomes an issue, set to zero (0).
#define REMOVE_STUPID_COMMENT (1)

//* Debugging output to display.                  *
//* This directive reports parsing of data blocks.*
#define DEBUG_BLOCKS (0)

//* Debugging output to display.                  *
//* This directive reports parsing of data tables,*
//* cartouche blocks and other <table> blocks.    *
#define DEBUG_TABLES (0)

//* Debugging output to display.                  *
//* This directive reports parsing of lists, both *
//* enumerate and itemized lists.                 *
#define DEBUG_LISTS (0)

//* Debugging output to display. This directive reports the major *
//* checkpoints in parsing the source file. It's purpose is to    *
//* discover any changes in HTML output formatting made by the    *
//* Texinfo group which tend to cause idpp to report syntax errors*
//* in the source data.                                           *
//* For a more granular report, see the "--book" debugging option.*
#define DEBUG_FLOW (0)

#if DEBUG_FLOW != 0 || DEBUG_BLOCKS != 0 || DEBUG_TABLES != 0 || DEBUG_LISTS != 0
gString gsdbg ; // format debugging output
#endif   // DEBUG_FLOW || DEBUG_BLOCKS || DEBUT_TABLES || DEBUG_LISTS


//*-----------------------------------------------*
//* Const strings for scanning source file blocks *
//*-----------------------------------------------*
const wchar_t* topDOCT = L"<!DOCTYPE HTML" ; // HTML5 document type
const short    topDOCT_len = 14 ;
const wchar_t* topHTML = L"<HTML" ;          // HTML3-4 document type
const short    topHTML_len = 5 ;
const wchar_t* cssCOPY = L"Copyright" ;      // Info written during a previous scan
const short    cssCOPY_len = 9 ;
const wchar_t* cssSAMU = L"The Software Samurai" ;
const short    cssSAMU_len = 20 ;
const wchar_t* cssVERS = L"Version: " ;
const short    cssVERS_len = 9 ;
const wchar_t* headTag = L"<head" ;          // open the <head> block
const short    headTag_len = 5 ;
const wchar_t* headendTag = L"</head" ;      // close the <head> block
const short    headendTag_len = 6 ;
const wchar_t* titleTag = L"<title" ;        // document title within <head> block
const short    titleTag_len = 6 ;
const wchar_t* bodyTag = L"<body" ;          // open the <body> block
const short    bodyTag_len = 5 ;
const wchar_t* bodyendTag = L"</body" ;      // close the <body> block
const short    bodyendTag_len = 6 ;
const wchar_t* divendTag = L"</div>" ;       // close a <div> block


//* First tag after body opens. Page header follows. *
const wchar_t* topTarget6 = L"<div class=\"top\" id=\"Top\">" ; // makeinfo v:6.8
const short    topTarget6_len = 27 ;
const wchar_t* topTarget7 = L"<div class=\"top-level-extent\" id=\"Top\">" ; // makeinfo v:7.0.3
const short    topTarget7_len = 38 ;

//* Page Header (navigation block)     *
const wchar_t* navBlock7a = L"<div class=\"nav-panel\">" ; // makeinfo v:7.0.3
const short    navBlock7a_len = 23 ;
const wchar_t* navBlock7b = L"Next: <a href=\"" ;
const short    navBlock7b_len = 15 ;
const wchar_t* navBlock6a = L"<div class=\"header\">" ;    // makeinfo v:6
const short    navBlock6a_len = 20 ;

//* Begin Table-Of-Contents <div> enclosure. (makeinfo v:7 and v:6)           *
//* If Table-Of-Contents not present, then the page header following where it *
//* would have been, will be found instead. See also navBlock sequence, above.*
const wchar_t* tocCont7_1 = L"<div class=\"element-contents\" id=\"SEC_Contents\">" ;
const short    tocCont7_1_len = 47 ;
const wchar_t* tocCont7_2 = L"<ul class=\"mini-toc\">" ; // if no "@contents" tag in .texi source
const short    tocCont7_2_len = 20 ;                     // (this is the same as the chapter menus)
const wchar_t* tocCont6_1 = L"<div class=\"Contents_element\" id=\"SEC_Contents\">" ;
const short    tocCont6_1_len = 47 ;
const wchar_t* tocContents2 = L"<div class=\"contents\">" ; // (same for v:6 and v:7)
const short    tocContents2_len = 21 ;
const wchar_t* tocTitle = L"<h2 class=\"contents-heading\">Table of Contents</h2>" ;
const short    tocTitle_len = 51 ; // makeinfo 6.2 and later

//* These are the title of the Index page and the enclosing block, *
//* respectively for makeinfo v:7                                  *
const wchar_t* indIndex7_1 = L"<h2 class=\"unnumbered\" id=\"Index-1\">Index</h2>" ;
const short    indIndex7_1_len = 45 ;
const wchar_t* indIndex7_2 = L"<div class=\"printindex cp-printindex\">" ;
const short    indIndex7_2_len = 37 ;
//* These are the navigation header for the index page *
//* and the index title, respectively for makeinfo v:6 *
const wchar_t* indIndex6_1 = L"<div class=\"unnumbered\" id=\"Index\">" ;
const short    indIndex6_1_len = 34 ;
const wchar_t* indIndex6_2 = L"<span id=\"Index-1\"></span>" ;
const short    indIndex6_2_len = 22 ;

const wchar_t* ulBegin = L"<ul" ;
const short    ulBegin_len = 3 ;
const wchar_t* ulEnd = L"</ul>" ;
const short    ulEnd_len = 5 ;

const wchar_t* olBegin = L"<ol" ;
const short    olBegin_len = 3 ;
const wchar_t* olEnd = L"</ol>" ;
const short    olEnd_len = 5 ;
const wchar_t* liOpen  = L"<li" ;
const short    liOpen_len = 3 ;
const wchar_t* liBegin = L"<li>" ;
const short    liBegin_len = 4 ;
const wchar_t* liLongBegin = L"</li><li>" ;
const short    liLongBegin_len = 9 ;
const wchar_t* paraEnd = L"</p>" ;
const short    paraEnd_len = 4 ;
const wchar_t* tabBegin = L"<table" ;
const short    tabBegin_len = 6 ;
const wchar_t* tabSimple = L"<table>" ;
const short    tabSimple_len = 7 ;
const char*    commBegin = "<!--" ;    // HTML comment delimiters
const char*    commEnd   = "-->" ;
const wchar_t* stupidComment = L"<!-- /@w -->" ; // meaningless random comment

//* Indented blocks *
const wchar_t* indeInherit    = L"<blockquote class=\"indentedblock\">" ;
const wchar_t* indeSmall7     = L"<blockquote class=\"indentedblock smallindentedblock\">" ;
const wchar_t* indeSmall      = L"<blockquote class=\"smallindentedblock\">" ;
const wchar_t* indeLarge      = L"<blockquote class=\"largeindentedblock\">" ;
//* Quotation blocks *
const wchar_t* quotSimple     = L"<blockquote>" ;     // makeinfo v:6 only
const wchar_t* quotInherit    = L"<blockquote class=\"quotation\">" ; // (processed, or makeinfo v:6)
const wchar_t* quotSmall7     = L"<blockquote class=\"quotation smallquotation\">" ; // makeinfo v:7
const wchar_t* quotSmall      = L"<blockquote class=\"smallquotation\">" ; // (processed, or makeinfo v:6.2 and earlier)
const wchar_t* quotLarge      = L"<blockquote class=\"largequotation\">" ; // (processed)
const wchar_t* quotEND        = L"</blockquote>" ;

const wchar_t* tagClose       = L">" ;       // tag-close character
const wchar_t* lrgCODE        = L"=l=" ;     // encoded tag appended to "large font" blocks
const wchar_t degreeSymbol = L'\x00B0' ;     // Unicode degree character (used as small bullet)
const wchar_t discBullet   = L'\x23FA' ;     // Texinfo/HTML disc bullet
const wchar_t cirtexBullet = L'\x26AC' ;     // Texinfo circle bullet
const wchar_t circleBullet = L'\x26AA' ;     // HTML circle bullet (looks bad in info)
const wchar_t squareBullet = L'\x25AA' ;     // Texinfo/HTML square bullet

//************************
//* Const output strings *
//************************
const char* stdDOCT = "<!DOCTYPE HTML>" ;
const char* stdHTML = "<html>" ;
const char* cssLINK0 = 
   "<!-- Post-processing of this document performed using %S v:%S -->" ;
const char* cssLINK1 = 
   "<meta charset=\"utf-8\" />  <!-- Import the global stylesheet -->\n"
   "<link rel=\"stylesheet\" href=\"" ;
const char* cssLINK2 = 
   "\" lang=\"en\" type=\"text/css\"/>\n" ;
const char* stdBODY = "<body>" ;
const wchar_t* begCONTAINER = L"<div class=\"infodoc_container\">" ;
const short begCONTAINER_len = 31 ;
const wchar_t* endCONTAINER = L"</div>    <!-- infodoc_container -->" ;
const short endCONTAINER_len = 36 ;
const char* tocDISC   = "<ul class=\"toc-level1\">" ;
const char* tocCIRCLE = "<ul class=\"toc-level2\">" ;
const char* tocSQUARE = "<ul class=\"toc-level3\">" ;
//const char* ulDISC   = "<ul class=\"disc-bullet\">" ;
//const char* ulCIRCLE = "<ul class=\"circle-bullet\">" ;
//const char* ulSQUARE = "<ul class=\"square-bullet\">" ;
//* Display text corresponding to enum blkType. *
//* Used for constructing  interactive prompts. *
//*    (keep synchronized with enum blkType)    *
const wchar_t* blockName[] = 
{
   L"Indentedblock", L"smallIndentedblock",  L"largeIndentedblock",
   L"Quotation",     L"smallQuotation",      L"largeQuotation",
   L"Format",        L"smallFormat",         L"largeFormat",
   L"Display",       L"smallDisplay",        L"largeDisplay",
   L"Example",       L"smallExample",        L"largeExample",
   L"Lisp",          L"smallLisp",           L"largeLisp",
   L"Verbatim",      L"smallVerbatim",       L"largeVerbatim",
   L"? ? ?" // unknown block type
} ;

//* Supported Enumeration Types - Index for CSS class name. *
//* This is the value returned from parsing user response.  *
enum enumType : short 
{
   etDecimal = ZERO,          // decimal
   etDecimalLZ,               // decimal-leading-zero
   etLowerAlpha,              // lower-case alpha
   etUpperAlpha,              // upper-case alpha
   etLowerRoman,              // lower-case Roman numerals
   etUpperRoman,              // upper-case Roman numerals
   etLowerGreek,              // lower-case Greek alpha
   etUpperGreek,              // upper-case Greek alpha
   etCJK,                     // CJK (Han informal) numbers
   etKatakana,                // Katakana (Japanese) alpha
   etHebrew,                  // Hebrew alpha
   etArabic,                  // Arabic-Indic numbers
   etCustom,                  // Custom, user-defined type (enum-custom class)
   etTYPES                    // number of enumeration types
} ;
//* Supported Enumeration Types - User response characters. *
//* This is synchronized with enumType above.               *
const wchar_t enumChar[etTYPES + 1] = 
{
   L'd', L'D',    // decimal
   L'l', L'u',    // alpha
   L'i', L'I',    // Roman
   L'g', L'G',    // Greek
   L'j', L'k',    // CJK, Katakana
   L'h', L'e',    // Hebrew, Arabic-Indic
   L'c',          // custom
   L'.'     // corresponds to etTYPES
} ;

//* Placeholder user responses. (see 'userResponse' method for details) *
const wchar_t* dToken = L"default_token" ;
const short    dToken_len = 13 ;



//*************************
//*   ppfProcessSrcHTML   *
//*************************
//********************************************************************************
//* Convert raw HTML to CSS-styled HTML.                                         *
//*                                                                              *
//* Input  : src : (by reference) path/filename of source data                   *
//*          trg : (by reference) path/filename of target                        *
//*                                                                              *
//* Returns: OK if successful, ERR processing error or user aborted process      *
//********************************************************************************
//* Programmer's Note: The TexInfo HTML converter v:7 now outputs a valid HTML5  *
//* doctype (MIME type); therefore, if any legacy HTML3/4 doctype syntax is      *
//* encountered, it is silently replaced with a modern tag.                      *
//*                                                                              *
//* The monospace font tag <tt></tt> (teletype) is obsolete. Unfortunately,      *
//* texi2any may still use it in some instances, and if found, we silently       *
//* replace all <tt>...</tt> sequences with a CSS "font-family" specification.   *
//********************************************************************************

short Idpp::ppfProcessSrcHTML ( const gString& src, const gString& trg )
{
   const wchar_t* gplBegin = L"<h3 class=\"unnumberedsec\">GNU General Public License</h3>" ;
   const wchar_t* fdlBegin = L"<h3 class=\"unnumberedsec\">GNU Free Documentation License</h3>" ;

   this->slCount = ZERO ;     // reset source-line counter (zero-based counter)
   this->tlCount = ZERO ;     // reset target-line counter (zero-based counter)
   short status = OK ;        // return value

   //* Open the source file, and                      *
   //* if processing is enabled, open the target file.*
   this->ifs.open ( src.ustr(), ifstream::in ) ;
   if ( this->no_mods == false )
      this->ofs.open ( trg.ustr(), ofstream::trunc ) ;

   //* Access to source and target? *
   if ( this->ifs.is_open() && ((this->ofs.is_open()) || this->no_mods != false) )
   {
      gString gs,             // conversion to wide text
              gsOut,          // output data for target file
              gsBook ;        // for debugging output only
      short wi ;              // index into source line

      //* Read the first source line (validates document as HTML) *
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( (gs.find( topDOCT, wi )) == wi )     // valid HTML5 doctype
            this->ppfWriteLine ( gs ) ;
         else if ( (gs.find( topHTML, wi, true )) == wi )// obsolete HTML doctype (HTML4)
         {
            gsOut = stdDOCT ;
            this->ppfWriteLine ( gsOut ) ;
         }
         else if ( (gs.find( stdHTML, wi )) == wi )// doctype omitted (HTML3)
         {
            gsOut = stdDOCT ;
            this->ppfWriteLine ( gsOut ) ;
            this->ppfWriteLine ( gs ) ;
         }
         else
            status = ERR ;
      }

      #if DEBUG_FLOW != 0
      this->textOut ( L"** Scan for <head>" ) ;
      #endif   // DEBUG_FLOW

      //* Copy source lines to target until '<head>' tag copied *
      while ( (status == OK) && ! this->abort )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            this->ppfWriteLine ( gs.ustr() ) ;
            if ( (gs.compare( headTag, false, headTag_len, wi )) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) <HEAD>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               #if DEBUG_FLOW != 0
               gsdbg.compose( "      Found <head> at:%hd", &this->slCount ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               break ;     // <head> found and copied
            }
         }
      }

      //* Process the contents of the <head> ... </head> block *
      status = this->ppfProcHEAD () ;

      if ( status == OK && this->book != false )
      {
         gsBook.compose( L"ML(%hu) </HEAD>", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }

      #if DEBUG_FLOW != 0
      gsdbg.compose( "        End <head> at:%hd (status:%hd)", &this->slCount, &status ) ;
      this->textOut ( gsdbg ) ;
      #endif   // DEBUG_FLOW

      #if DEBUG_FLOW != 0
      this->textOut ( L"** Scan for <body>" ) ;
      #endif   // DEBUG_FLOW

      //* Copy lines until we find the <body> tag *
      while ( (status == OK) && ! this->abort )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            if ( (gs.compare( bodyTag, false, bodyTag_len, wi )) == ZERO )
            {  //* <body> found, replace it with a lean <body>. *
               if ( this->no_body == false )
                  gsOut = stdBODY ;
               else     // write unmodified body tag
                  gsOut = gs ;
               this->ppfWriteLine ( gsOut ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) <BODY>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               //* Establish the container jusct inside the <body> tag. *
               if ( this->no_cont == false && this->multiPass == false )
               {
                  gsOut = begCONTAINER ;
                  this->ppfWriteLine ( gsOut ) ;
               }

               #if DEBUG_FLOW != 0
               gsdbg.compose( "      Found <body> at:%hd %S", &this->slCount, gs.gstr() ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               break ;
            }
            this->ppfWriteLine ( gs ) ;
         }
      }

      #if DEBUG_FLOW != 0
      this->textOut ( L"** Scan for top-level <div>" ) ;
      #endif   // DEBUG_FLOW

      //* Copy lines until we find the <div> at *
      //* the "Top" of the main document.       *
      short blankCount = ZERO ;
      while ( (status == OK) && ! this->abort )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            if ( gs.gschars() == 1 )
            {  //* For some reason the texi-to-HTML converter inserts 41 *
               //* blank lines into the file between the start of the    *
               //* <body> and the Table of Contents. Delete some of them.*
               if ( ++blankCount > 3 )
                  continue ;
            }

            //* If target found, write it and break.         *
            //* These tests are for source data generated by *
            //* makeinfo v:7 and makeinfo v:6, respectively. *
            if ( ((gs.compare( topTarget7, false, topTarget7_len, wi )) == ZERO) ||
                 ((gs.compare( topTarget6, false, topTarget6_len, wi )) == ZERO) )
            {
               this->ppfWriteLine ( gs ) ;

               #if DEBUG_FLOW != 0
               gsdbg.compose( "      found top level <div> at: %hd %S", &this->slCount, gs.gstr() ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               break ;
            }
            this->ppfWriteLine ( gs ) ;   // copy the line from source to target
         }
      }

      #if DEBUG_FLOW != 0
      gsdbg.compose( "** Scan for T-O-C from:%hd", &this->slCount  ) ;
      this->textOut ( gsdbg ) ;
      #endif   // DEBUG_FLOW

      //* Copy lines until we reach the Table of Contents (if any).*
      short navhdrCount = ZERO ;
      while ( (status == OK) && ! this->abort )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            //* If Table of Contents Heading found.                *
            //* Tests are for makeinfo v:7 with and without T-O-C, *
            //* and v:6, respectively.                             *
            //* Note that the source file may not include a table  *
            //* of contents, so do not rely upon its being present.*
            if ( ((gs.compare( tocCont7_1, false, tocCont7_1_len, wi )) == ZERO) ||
                 ((gs.compare( tocCont7_2, false, tocCont7_2_len, wi )) == ZERO) ||
                 ((gs.compare( tocCont6_1, false, tocCont6_1_len, wi )) == ZERO) )
            {
               #if DEBUG_FLOW != 0
               gsdbg.compose( "      Found T-O-C at:%hd %S", &this->slCount, gs.gstr() ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               status = this->ppfProcTOC ( gs ) ;

               #if DEBUG_FLOW != 0
               gsdbg.compose( "   Finished T-O-C at:%hd (status:%hd)", &this->slCount, &status ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               break ;
            }

            //* If page navigation header, it indicates that *
            //* the Table of Contents was not encountered.   *
            else if ( ((gs.compare( navBlock7a, false, navBlock7a_len, wi )) == ZERO) ||
                      ((gs.compare( navBlock6a, false, navBlock6a_len, wi )) == ZERO) )
            {
               if ( ++navhdrCount >= 2 )
               {
                  #if DEBUG_FLOW != 0
                  gsdbg.compose( "  Not Incl. T-O-C at:%hd %S", &this->slCount, gs.gstr() ) ;
                  this->textOut ( gsdbg ) ;
                  #endif   // DEBUG_FLOW

                  this->ppfWriteLine ( gs ) ;      // write the captured line

                  #if DEBUG_FLOW != 0
                  gsdbg.compose( "   Finished T-O-C at:%hd (status:%hd)", &this->slCount, &status ) ;
                  this->textOut ( gsdbg ) ;
                  #endif   // DEBUG_FLOW

                  break ;
               }
            }

            //* Test for a formatted block occurring before Table-Of-Contents. *
            //* This will be the "@copying" block, usually a copyright message.*
            //* Other blocks may also be present above Table-Of-Contents.      *
            else if ( (this->ppfFormattedBlock ( gs, status, true )) != false )
               continue ;

            //* If an indentedblock, smallindentedblock or largeindentedblock *
            else if ( (this->ppfIndentedBlock ( gs, status, true )) != false )
            { continue ; }

            //* If a quotation, smallquotation or largequotation block *
            else if ( (this->ppfQuotationBlock ( gs, status, true )) != false )
            { continue ; }

            //* If a verbatim, smallverbatim or largeverbatim block *
            else if ( (this->ppfVerbatimBlock ( gs, status, true )) != false )
               continue ;

            //* If container class was established on a previous pass, *
            //* delete the old copy to avoid duplication.              *
            else if ( (gs.compare( begCONTAINER, false, begCONTAINER_len, wi )) == ZERO )
               continue ;

            //* Copy the unidentified line from source to target *
            else
               this->ppfWriteLine ( gs ) ;
         }
      }  // Table-Of-Contents scan

      //*****************************************************************
      //* Copy lines until we reach the end of the <body></body> block. *
      //* This is the main processing loop.                             *
      //*****************************************************************
      while ( status == OK && ! this->abort )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            //* If </body> (end-of-body) tag found *
            if ( (gs.compare( bodyendTag, false, bodyendTag_len, wi )) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) </BODY>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               //* Insert close of container just above it. *
               if ( this->no_cont == false && this->multiPass == false )
               {
                  gsOut = endCONTAINER ;
                  this->ppfWriteLine ( gsOut ) ;
               }
               this->ppfWriteLine ( gs ) ;

               #if DEBUG_FLOW != 0
               gsdbg.compose( "        End <body> at:%hd", &this->slCount ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               break ;
            }

            //* Process embedded HTML comment in the source file.*
            else if ( (this->ppfTestComment ( gs )) != false )
            {
               this->ppfProcComment ( gs ) ;
               // Programmer's Note: Return value is currently ignored.
               // See notes in called method.
               status = OK ;
            }

            //* If an indentedblock, smallindentedblock or largeindentedblock *
            else if ( (this->ppfIndentedBlock ( gs, status, true )) != false )
            { continue ; }

            //* If a quotation, smallquotation or largequotation block *
            else if ( (this->ppfQuotationBlock ( gs, status, true )) != false )
            { continue ; }

            //* If a verbatim, smallverbatim or largeverbatim block *
            else if ( (this->ppfVerbatimBlock ( gs, status, true )) != false )
               continue ;

            //* If a preformatted block is detected:              *
            //* 'display, 'format', 'example', 'lisp', and their  *
            //* 'small' or 'large' counterparts.                  *
            else if ( (this->ppfFormattedBlock ( gs, status, true )) != false )
               continue ;

            //* If an Itemized List (<ul> tag) found *
            else if ( (this->ppfItemizedList ( gs, status, wi, true )) != false )
               continue ;

            //If an Enumerated List (<ol> tag) found *
            else if ( (this->ppfEnumeratedList ( gs, status, wi, true )) != false )
               continue ;

            //* If we have reached the beginning of the Index node. *
            //*              (makeinfo v:6 and v:7)                 *
            else if ( ((gs.find( indIndex7_1, ZERO, true, indIndex7_1_len )) >= ZERO) ||
                      ((gs.find( indIndex7_2, ZERO, true, indIndex7_2_len )) >= ZERO) ||
                      ((gs.find( indIndex6_1, ZERO, true, indIndex6_1_len )) >= ZERO) ||
                      ((gs.find( indIndex6_2, ZERO, true, indIndex6_2_len )) >= ZERO) )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) XBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               #if DEBUG_FLOW != 0
               gsdbg.compose( "**    Found Index at:%hd %S", &this->slCount, gs.gstr() ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               status = this->ppfProcINDEX ( gs ) ;

               #if DEBUG_FLOW != 0
               gsdbg.compose( "   Finished Index at:%hd (status:%hd)", &this->slCount, &status ) ;
               this->textOut ( gsdbg ) ;
               #endif   // DEBUG_FLOW

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) XBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If a <table> object found.                               *
            //* (NOT the document index tables which are handled above.) *
            else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
            {
               status = this->ppfProcTABLE ( gs ) ;
            }

            //* If the GNU General Public License or GNU Free  *
            //* Documentation License is detected, process it. *
            else if (   ((gs.find( gplBegin, wi, false )) >= ZERO)
                     || ((gs.find( fdlBegin, wi, false )) >= ZERO) )
            {
               this->ppfWriteLine ( gs ) ;      // write the unmodified line
               bool gpl = true ;
               if ( (gs.find( fdlBegin, wi, false )) >= ZERO )
                  gpl = false ;
               status = this->ppfProcGNU ( gpl ) ;
            }

            //* If container class was established on a previous pass, *
            //* delete the old copy to avoid duplication.              *
            else if ( (gs.find( endCONTAINER, wi, false )) >= ZERO )
            {
               /* do nothing */
            }

            else        // this line gets no processing
               this->ppfWriteLine ( gs ) ;
         }     // read_line
      }        // while()

      //* Finish copying the file *
      while ( ! this->abort )
      {
         if ( (this->ppfReadSrcLine ( gs, wi )) == OK )
            this->ppfWriteLine ( gs ) ;
         else        // last source line copied to target
            break ;
      }
   }     // source/target files opened
   if ( this->ifs.is_open() )
      this->ifs.close() ;           // close the source file
   if ( this->ofs.is_open() )
      this->ofs.close() ;           // close the target file
   return status ;

}  //* End ppfProcessSrcHTML()

//*************************
//*      ppfProcTOC       *
//*************************
//********************************************************************************
//* Called by ppfProcessSrcHTML() to process the Table Of Contents list.         *
//* The caller has identified the first line which indicates the TOC block,      *
//* but has not yet processed it (see gsBegin).                                  *
//*                                                                              *
//* 1) If 'tocDel' == false :                                                    *
//*    a) If 'tocMod' == false :                                                 *
//*       Copy Table of Contents unmodified.                                     *
//*    b) If 'tocMod' != false :                                                 *
//*       Convert the "no-bullet" classes within the Table of Contents to        *
//*       the appropriate level of bullet class.                                 *
//*       - 1st level "no-bullet" class becomes toc-level1" class                *
//*       - 2nd level "no-bullet" class becomes toc-level2" class                *
//*       - 3rd and lower levels "no-bullet" class becomes toc-level3" class     *
//* 2) If 'tocDel' != false :                                                    *
//*    Delete the entire Table of Contents.                                      *
//*                                                                              *
//* Input  : gsBegin : First line of Table Of Contents block                     *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Notes:                                                                       *
//* -- There may be a block of text after the table but BEFORE the end of the    *
//*    TOC page. Therefore, if there is no table, we will find the text block    *
//*    before reaching the end of the page.                                      *
//* ---------------------------------------------------------------------------- *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcTOC ( const gString& gsBegin )
{
   const wchar_t* tocContLevel = L"<ul class=\"toc-numbered-mark\">" ;
 
   gString gs,             // conversion to wide text
           gsOut ;         // output data for target file
   short wi,
         tocLevel = ZERO,  // nesting level for TOC line items
         trgs = ZERO,      // number of _open_ <div> tags
         status = OK ;     // return value

   if ( ((gsBegin.compare( tocCont7_1, false, tocCont7_1_len, wi )) == ZERO) ||
        ((gsBegin.compare( tocCont6_1, false, tocCont6_1_len, wi )) == ZERO) )
   { trgs = 2 ; }    // full TOC (enclosed within two(2) <div> sequences)
   else
   { trgs = 1 ; }    // mini TOC (enclosed within one(1) <div> sequence)

   //* If TOC is to be retained, write the first line *
   if ( ! this->tocDel )
      this->ppfWriteLine ( gsBegin ) ;

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( (gs.find( divendTag )) >= ZERO )
         {
            if ( ! this->tocDel )
               this->ppfWriteLine ( gs ) ;
            if ( --trgs == ZERO )
               break ;
         }
         //* The source document may, or may not have node headers.  *
         //* If it does, and if we see it here, it indicates that    *
         //* the end of T-O-C has been reached.                      *
         //* Push the navigation header back into the stream and     *
         //* return to caller.                                       *
         else if ( ((gs.compare( navBlock7a, false, navBlock7a_len, wi )) == ZERO) ||
                   ((gs.compare( navBlock6a, false, navBlock6a_len, wi )) == ZERO) )
         {
            this->ppfUnReadSrcLine ( gs ) ;
            break ;     // page header processed, end of TOC page
         }

         //* If TOC is to be retained:                            *
         //* a) If user specified that TOC should be treated as   *
         //*    an unordered list, rework the referenced classes. *
         //* b) Else TOC is copied as written.                    *
         else if ( this->tocDel == false )
         {
            if ( this->tocMod != false )
            {
               if ( (gs.find( tocContLevel, wi )) == wi )
               {
                  gs.limitChars( wi ) ;
                  if ( ++tocLevel == 1 )
                     gs.append( tocDISC ) ;
                  else if ( tocLevel == 2 )
                     gs.append( tocCIRCLE ) ;
                  else
                     gs.append( tocSQUARE ) ;
                  this->ppfWriteLine ( gs ) ;
               }
               else
               {
                  this->ppfWriteLine ( gs ) ;
                  if ( (tocLevel > ZERO) &&
                       ((gs.compare( ulEnd, false, ulEnd_len, wi )) == ZERO) )
                  {
                     --tocLevel ;
                  }
               }
            }
            else
               this->ppfWriteLine ( gs ) ;
         }

         //* Discard everything in the TOC except the page divider *
         else     // tocDel != false
         {
            if ( (gs.find( L"<hr>", wi, false, 4 )) >= ZERO )
               this->ppfWriteLine ( gs ) ;
            else
               continue ;
         }
      }
   }
   return status ;

}  //* End ppfProcTOC() *

//*************************
//*     ppfProcINDEX      *
//*************************
//********************************************************************************
//* Called by ppfProcessSrcHTML() to process the Index table.                    *
//* The caller has not processed the source line, but has sent the unprocessed   *
//* source line for analysis.                                                    *
//*                                                                              *
//* Input  : gsBegin : First line of Index block                                 *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* How to recognize the end of the Index section:                               *
//* ----------------------------------------------                               *
//*  Analyze the data sent from caller (gsBegin) and initialize 'v7format' flag. *
//*  -- If the data contains a <div> tag, increment the 'divblks' counter.       *
//*  -- Write the gsBegin data to target file.                                   *
//*                                                                              *
//*  The Index section contains three(3) tables: two summary-letter tables and   *
//*  the table which contains the actual Index data.                             *
//*  -- Increment the 'begtabs' counter for each <table> tag scanned.            *
//*  -- Increment the 'endtabs' counter for each </table> tag scanned.           *
//*                                                                              *
//*  If the source file has not been previously processed, make any necessary    *
//*  modifications.                                                              *
//*  If the source file has been previously processed, do not modify the data.   *
//*                                                                              *
//*  The Index format for makeinfo v:7 includes one(1) <div> block below the     *
//*  title of the Index page. (For v:7, the page header is not sent by caller.)  *
//*                                                                              *
//*  The Index format for makeinfo v:6 includes at least one <div> tag, and if   *
//*  caller sent us the page header, then v:6 includes two <div> tags.           *
//*                                                                              *
//* 5) There are three(3) tables which comprise the the index node:              *
//*    the "Jump to" blocks and the table containing the actual Index data.      *
//*    These are always identified and processed as borderless tables.           *
//*    ** makeinfo v:6:                                                          *
//*       tabSimple is a makeinfo v:5 and v:6 construct:                         *
//*       <table><tr><th valign="top">Jump to: &nbsp; </th> ....                 *
//*    ** makeinfo v:7:                                                          *
//*     -- Container class: <div class="printindex cp-printindex">               *
//*     -- The "Jump to" tables which provide alphabetical access to the index.  *
//*        <table class="cp-letters-header-printindex"><tr><th>Jump to: ...      *
//*        <table class="cp-letters-footer-printindex">                          *
//*     -- The main index table:                                                 *
//*        <table class="cp-entries-printindex" border="0">                      *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcINDEX ( const gString& gsBegin )
{
   #define DEBUG_INDEX (0)                // for debugging only

   const char* indexJUMPTO = " class=\"jumpto\"" ;
   const wchar_t* endTable = L"</table>" ;

   gString gs ;            // conversion to wide text
   short wi,
         begtabs = ZERO,   // loop control
         endtabs = ZERO,
         divblks = ZERO ;
   short status = OK ;     // return value
   bool  v7format = true ; // set if makeinfo format >= v:7

   #if DEBUG_INDEX != 0
   gString gsdbg, gstmp ;
   short ti, et_tmp, db_tmp ;
   gsdbg.compose( "IDX--------'%S'\n", gsBegin.gstr() ) ;
   #endif   // DEBUG_INDEX

   //* Determine whether the format is makeinfo v:6 or v:7 and if *
   //* initial tag indicates a <div> block increment the count.   *
   if ( (gsBegin.find( indIndex6_1, ZERO, false, indIndex6_1_len )) >= ZERO )
   { v7format = false ; ++divblks ; }
   else if ( (gsBegin.find( indIndex6_2, ZERO, false, indIndex6_2_len )) >= ZERO )
   { v7format = false ; }
   else if ( (gsBegin.find( indIndex7_2, ZERO, false, indIndex7_2_len )) >= ZERO )
   { ++divblks ; }

   this->ppfWriteLine ( gsBegin ) ;       // write the first line of the block

   #if DEBUG_INDEX != 0
   gsdbg.append( "-----------'%S' (db:%hd bt:%hd et:%hd)", 
                 gsBegin.gstr(), &divblks, &begtabs, &endtabs ) ;
   this->textOut ( gsdbg ) ;
   gsdbg.clear() ;
   #endif   // DEBUG_INDEX

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( v7format )   // makeinfo v:7
         {
            //* If data includes a <table> tag, determine the flavour *
            if ( (gs.find( tabBegin )) >= ZERO )
            {
               this->ppfWriteLine ( gs ) ;   // write data to target file
               ++begtabs ;                   // increment the counter

               #if DEBUG_INDEX != 0
               gstmp = gs ;
               ti = gstmp.find( tabBegin ) ;
               ti = gstmp.after( tagClose, ti ) ;
               gstmp.limitChars( ti ) ;
               gsdbg.append( "b----------'%S' (db:%hd bt:%hd et:%hd)\n", 
                             gstmp.gstr(), &divblks, &begtabs, &endtabs ) ;
               #endif   // DEBUG_INDEX
            }
            //* Else, data is either table data OR       *
            //* end-of-block (either </table> or </div>. *
            else
            {
               #if DEBUG_INDEX != 0
               et_tmp = endtabs ; db_tmp = divblks ;
               #endif   // DEBUG_INDEX

               this->ppfWriteLine ( gs ) ;
               if ( (gs.find( endTable )) >= ZERO )
                  ++endtabs ;

               else if ( (gs.compare( indIndex7_2, false, indIndex7_2_len )) == ZERO )
                  ++divblks ;

               else if ( (gs.find( divendTag )) >= ZERO )
                  --divblks ;

               #if DEBUG_INDEX != 0
               if ( (et_tmp != endtabs) || (db_tmp != divblks) )
               { gsdbg.append( "e----------'%S' (db:%hd bt:%hd et:%hd)\n", 
                                gs.gstr(), &divblks, &begtabs, &endtabs ) ; }
               #endif   // DEBUG_INDEX
            }
         }
         else              // makeinfo v:6
         {
            //* If data includes a <table> tag, determine the flavour *
            if ( (gs.find( tabBegin )) >= ZERO )
            {
               //* If unstyled "<table>" found, assign it to the "jumpto" class *
               //* Tnis includes the upper and lower summary-letter tables.     *
               if ( (gs.compare( tabSimple, false, tabSimple_len, wi )) == ZERO )
               {
                  if ( (wi = gs.find( tagClose, wi )) >= ZERO )
                     gs.insert( indexJUMPTO, wi ) ;
               }
               this->ppfWriteLine ( gs ) ;   // write data to target file
               ++begtabs ;                   // increment the counter

               #if DEBUG_INDEX != 0
               gstmp = gs ;
               ti = gstmp.find( tabBegin ) ;
               ti = gstmp.after( tagClose, ti ) ;
               gstmp.limitChars( ti ) ;
               gsdbg.append( "b6---------'%S' (db:%hd bt:%hd et:%hd)\n", 
                             gstmp.gstr(), &divblks, &begtabs, &endtabs ) ;
               #endif   // DEBUG_INDEX
            }
            //* Else, data is either table data OR       *
            //* end-of-block (either </table> or </div>. *
            else
            {
               #if DEBUG_INDEX != 0
               et_tmp = endtabs ; db_tmp = divblks ;
               #endif   // DEBUG_INDEX

               this->ppfWriteLine ( gs ) ;
               if ( (gs.find( endTable )) >= ZERO )
                  ++endtabs ;

               else if ( (gs.find( indIndex6_2, ZERO, false, indIndex6_2_len )) >= ZERO )
                  ++divblks ;

               else if ( (gs.find( divendTag )) >= ZERO )
                  --divblks ;

               #if DEBUG_INDEX != 0
               if ( (et_tmp != endtabs) || (db_tmp != divblks) )
               { gsdbg.append( "e----------'%S' (db:%hd bt:%hd et:%hd)\n", 
                                gs.gstr(), &divblks, &begtabs, &endtabs ) ; }
               #endif   // DEBUG_INDEX
            }
         }
      }
      if ( (begtabs == 3) && (endtabs == 3) && (divblks <= ZERO) )
         break ;
   }
   #if DEBUG_INDEX != 0
   this->textOut ( gsdbg ) ;
   #endif   // DEBUG_INDEX

   return status ;

}  //* End ppfProcINDEX() *

//*************************
//*     ppfProcTABLE      *
//*************************
//********************************************************************************
//* The beginning of a <table...> ... </table> sequence has been found.          *
//* Determine the TYPE of table object, and optionally modify it if it meets     *
//* the documented criteria.                                                     *
//*                                                                              *
//* Input  : gstab : (by reference) contains the first line of the <table>       *
//*                  sequence                                                    *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* 1) Only tables and cartouches within the main sequence are seen here.        *
//*    These are AFTER the Table of Contents and BEFORE the three Index tables.  *
//*    -- The Table Of Contents is handled in ppfProcTOC().                      *
//*    -- The Index group of tables is handled in ppfProcINDEX().                *
//*                                                                              *
//* 2) Command-line arguments:                                                   *
//*    a) The '--table_border' command-line option initializes the 'tabBorder'   *
//*       member to one of the following processing options:                     *
//*         cfgAuto : Apply <table class="bordered"> border style to all tables. *
//*                   (default)                                                  *
//*         cfgNone : Apply <table class="borderless"> to all tables.            *
//*         cfgSpec : For each table, prompt the user for a decision whether to  *
//*                   include a border.                                          *
//*    b) The '--cartouche' command-line option initialized the 'cartFormat'     *
//*       member to one of the following processing options:                     *
//*         cfgAuto : Apply <table class="cartouche"> to all cartouche objects.  *
//*                   (default)                                                  *
//*         cfgNone : Apply <table class="cartouche standardflow">               *
//*         cfgSpec : For each cartouche, prompt the user for a decision whether *
//*                   the text is pre-formatted or flowing.                      *
//*                                                                              *
//* 3) Interactive formatting responses:                                         *
//*    a) The multitable:                                                        *
//*       -- The multitable class specifies that tables will be bordered by      *
//*          default and with font size inherited from the parent.               *
//*       -- Interactive formatting responses:                                   *
//*          'y' = 'yes' --> bordered (default)                                  *
//*          'n' = 'no'  --> no border                                           *
//*          'A' = 'All' --> this table AND all subsequent tables will be        *
//*                          bordered                                            *
//*    b) The cartouche:                                                         *
//*       -- The cartouche class specifies that text within the block is         *
//*          pre-formatted by default and with font size inherited from parent.  *
//*       -- Interactive formatting responses:                                   *
//*          'a' = 'auto' --> pre-formatted text (default)                       *
//*          'n' = 'flow' --> flowing text                                       *
//*          'A' = 'All'  --> this cartouche AND all subsequent cartouche        *
//*                           objects will have pre-formatted text               *
//*                                                                              *
//* 4) Parsing the source data:                                                  *
//*    a) The makeinfo @multitable command generates <table...> tags:            *
//*       -- makeinfo v:6: <table>  i.e. a simple table tag.                     *
//*       -- makeinfo v:7: <table class="multitable">                            *
//*       -- The context line for the user prompt is rather complex for both     *
//*          makeinfo v6 and v:7. Example (this is a single line):               *
//*          <thead><tr><th>HTML NAME</th><th>NESTING LEVEL</th>                 *
//*                                              <th>EXAMPLE</th></tr></thead>   *
//*          (If there is no header entry, "<thead>" is "<tbody>".)              *
//*                                                                              *
//*    b) The makeinfo @cartouche command generates tags of the form:            *
//*       --  <table class="cartouche" border="1"><tr><td>                       *
//*           The generated command includes a 'border="1"' style element        *
//*           which is indended to create a one-pixel border. However, this      *
//*           looks like crap in combination with the class styling.             *
//*           That style element is removed during post-processing so that the   *
//*           cartouche class can fully control style for the block.             *
//*       -- The context line for the user prompt begins with a simple           *
//*          <p> (paragraph) tag.                                                *
//*                                                                              *
//*    c) For both multitable and cartouche the context line optionally contains *
//*       an embedded token which specifies the desired formatting options.      *
//*       The token takes the form: "=xx=" with 'xx' as the formatting options.  *
//*       -- Embedded token options for multitable:                              *
//*          First character specifies whether the table will have a border:     *
//*           'b' == bordered table (default)                                    *
//*           'n' == no border for table                                         *
//*          Second character is the font size:                                  *
//*           'i' == inherited font size (default)                               *
//*           's' == smaller font size                                           *
//*           'l' == larger font size                                            *
//*          -- For the table object, the token described may be manually        *
//*             inserted into the source document. Example:                      *
//*             Example:  @multitable {xxxxxxxxxxxx} {xxxxxxxxxxx}               *
//*                       @headitem =bs=COL#1 HEADER  @tab COL#2 HEADER          *
//*             Here the embedded token is "=bs=".                               *
//*             The embedded token is removed during post-processing.            *
//*          -- Note about invocation of the @multitable command through a macro:*
//*             Due to the complexity of the multitable syntax, it is not        *
//*             practical to invoke the object through a macro. If a smaller     *
//*             font size is desired, use either the embedded-token as described *
//*             below, or enclose the entire table within an @indentedblock.     *
//*             Example:  @smallindentedblock.                                   *
//*                       @multitable                                            *
//*                        . . .                                                 *
//*                       @end multitable                                        *
//*                       @end smallindentedblock                                *
//*             Note that the @largeindentedblock command IS a macro, and is     *
//*             therefore impractical for enclosing a @multitable construct.     *
//*                                                                              *
//*       -- Embedded token options for cartouche:                               *
//*          First character specifies text formatting:                          *
//*           'p' == pre-formated text (default)                                 *
//*           'f' == free-flowing text                                           *
//*          Second character is the font size:                                  *
//*           'i' == inherited font size (default)                               *
//*           's' == smaller font size                                           *
//*           'l' == larger font size                                            *
//*          -- For the cartouche object, text formatting and font size may      *
//*             ALSO be specified by using the macro @CartHtml with an embedded  *
//*             token of the form: "=xx=" which specifes the configuration as    *
//*             shown above.                                                     *
//*             Example : free-flow text with smaller font size.                 *
//*               @CartHtml{=fs=Hello world!                                     *
//*               It's good to see you.                                          *
//*               Are you still travelling around the sun?}                      *
//*             Here the embedded token is "=fs=".                               *
//*             The embedded token is removed during post-processing.            *
//*             (Note that macro syntax encloses the data within curly brackets.)*
//*                                                                              *
//* 5) The tables representing the auto-generated chapter menus are not modified.*
//*    -- makeinfo v:5:  <table class="menu" border="0" cellspacing="0">         *
//*    -- makeinfo v:6:  <ul class="section-toc">                                *
//*    -- makeinfo v:7:  <ul class="mini-toc">                                   *
//*    We may offer an option to modify generic menus in a future release.       *
//*                                                                              *
//* 6) All other '<table...> sequence headers are written unmodified to the      *
//*    target file.                                                              *
//*                                                                              *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//* Note that the makeinfo @table command does not generate a table.             *
//* Instead it generates a <dl> tag:                                             *
//*  ** makeinfo v:6:  <dl compact="compact">                                    *
//*  ** makeinfo v:7:  <dl class="table">                                        *
//* Either way, these constructs will not be seen here.                          *
//********************************************************************************

short Idpp::ppfProcTABLE ( const gString& gsBegin )
{
   #define  DEBUG_TOKEN (0)      // for debugging only

   //* Types of multitable classes *
   //* Note: makeinfo v:6 uses a simple "<table>" tag for multitables. *
   //* Replace the original table tag with one of the following:       *
   //* flowing text: a) standard font size, b) smaller, c) larger font *
   const wchar_t* tabMulti     = L"<table class=\"multitable\">" ;   // makeinfo v:7
   const wchar_t* tabCartouche = L"<table class=\"cartouche\"" ;     // (truncated compare)
   const wchar_t* bdrCLASS     = L"<table class=\"bordered\">" ;
   const wchar_t* bdrCLASSs    = L"<table class=\"bordered smallerflow\">" ;
   const wchar_t* bdrCLASSl    = L"<table class=\"bordered largerflow\">" ;
   const wchar_t* nobdrCLASS   = L"<table class=\"borderless\">" ;
   const wchar_t* nobdrCLASSs  = L"<table class=\"borderless smallerflow\">" ;
   const wchar_t* nobdrCLASSl  = L"<table class=\"borderless largerflow\">" ;
   const wchar_t* endTable     = L"</table>" ;   // if found, processing is complete
   const wchar_t bdrRESP       = L'b' ;  // token for bordered table
   const wchar_t nobdrRESP     = L'n' ;  // token for un-bordered table
   const wchar_t bdrallRESP    = L'A' ;  // token for All tables bordered

   //* Replace the original cartouche tag with one of the following:            *
   //* 1) pre-formatted text: a) standard font size, b) smaller, c) larger font *
   //* 2) flowing text      : a) standard font size, b) smaller, c) larger font *
   const wchar_t* preCLASSi  = L"<table class=\"cartouche\">" ; // (default)
   const wchar_t* preCLASSs  = L"<table class=\"cartouche smallerpre\">" ;
   const wchar_t* preCLASSl  = L"<table class=\"cartouche largerpre\">" ;
   const wchar_t* flowCLASS  = L"<table class=\"cartouche standardflow\">" ;
   const wchar_t* flowCLASSs = L"<table class=\"cartouche smallerflow\">" ;
   const wchar_t* flowCLASSl = L"<table class=\"cartouche largerflow\">" ;
   const wchar_t textAUTO    = L'a' ;  // user response for preformatted text
   const wchar_t textALL     = L'A' ;  // user response for All
   //const wchar_t textPREF    = L'p' ;  // token for pre-formatted text
   const wchar_t textFLOW    = L'f' ;  // user response (or token) for free-flowing text
   const wchar_t fontSMALL   = L's' ;  // user response (or token) for smaller font
   const wchar_t fontLARGE   = L'l' ;  // user response (or token) for larger font

   //* Used to format the user prompt *
   const wchar_t* paraGraph    = L"<p>" ;
   const wchar_t* headStart    = L"<thead><tr><th>" ;
   const wchar_t* headEnd      = L"</th></tr></thead>" ;
   const wchar_t* headFldStart = L"<th>" ;
   const wchar_t* headFldEnd   = L"</th>" ;
   const wchar_t* bodyStart    = L"<tbody>" ;
   const wchar_t* bodyEnd      = L"</tbody>" ;
   const wchar_t* fieldStartA  = L"<tr>" ;
   const wchar_t* fieldEndA    = L"</tr>" ;
   const wchar_t* fieldStartB  = L"<td>" ;
   const wchar_t* fieldEndB    = L"</td>" ;
   const wchar_t* spaceString  = L" " ;

   gString gsB = gsBegin,     // text formatting
           gss, gsNext,
           gsIn, gsOut, gst ; // user interface
   wchar_t usrResp = bdrRESP, // user response
           font_size = L'.' ; // font size (default or unspecified)
   short   wi, ti, ni, cti,   // character index
           status = OK ;      // return value
   bool    isTabMulti      = false,    // makeinfo v:7 multitables
           isTabSimple     = false,    // makeinfo v:6 multi-tables
           isTabBordered   = false,    // previously-processed (bordered)
           isTabUnbordered = false,    // previously-processed (un-bordered)
           isCartouche     = false,    // cartouche construct:
           isToken         = false ;   // set if embedded token found

   #if DEBUG_TABLES != 0
   gsdbg.compose( "GST at:%03hd '%S'", &this->slCount, gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_TABLES

   if ( (isTabMulti  = bool((ti = gsB.find( tabMulti )) >= ZERO)) )
   { /* do nothing */ }
   else if ( (isTabSimple = bool((ti = gsB.find( tabSimple )) >= ZERO)) )
   { /* do nothing */ }
   else if ( (isTabBordered = bool((ti = gsB.find( bdrCLASS )) >= ZERO)) )
   { /* do nothing */ }
   else if ( (isTabUnbordered = bool((ti = gsB.find( nobdrCLASS )) >= ZERO)) )
   { /* do nothing */ }
   else if ( (isCartouche = bool((cti = gsB.find( tabCartouche )) >= ZERO)) )
   {  //* (default response is based on the 'cartFormat' value) *
      usrResp = (this->cartFormat == cfgNone) ? textFLOW : textAUTO ;
   }
   else
   {
      //* Unrecognized type, assume a table and set initial user *
      //* response according to table post-processing value.     *
      usrResp = (this->tabBorder == cfgNone) ? nobdrRESP : bdrRESP ;
   }

   //* Read the next line and scan for an embedded token *
   if ( (status = this->ppfReadSrcLine ( gsNext, ni )) == OK )
   {
      const wchar_t* wbuff = gsNext.gstr() ;    // reference the text
      short tokOFF ;                            // offset to embedded token
      if ( isCartouche )
      {
         //* If an embedded token is present at the expected offset, *
         //* extract and decode it. Otherwise data will be unchanged.*
         //* The line will begin with "<p>=xx=" where "xx" is the    *
         //* two-character formatting code.                          *
         tokOFF = 3 ;                           // offset to embedded token
         if ( (wbuff[tokOFF] == L'=') && (wbuff[tokOFF + 3] == L'=') )
         {
            //* First character specifies text formatting, and       *
            //* second character is the font size. (see notes above) *
            //* Capture and range-check the token data (if any).     *
            usrResp   = tolower(wbuff[tokOFF + 1]) ;
            font_size = tolower(wbuff[tokOFF + 2]) ;
            if ( usrResp != textFLOW ) { usrResp = textAUTO ; }
            if ( (font_size != fontSMALL) && (font_size != fontLARGE) )
            { font_size = L'.' ; }
            isToken = true ;

            #if DEBUG_TABLES != 0 && DEBUG_TOKEN != 0
            gsdbg.compose( "TOKEN------'%S'\n"
                           "     ------'%C' '%C'\n", 
                           gsNext.gstr(), &usrResp, &font_size ) ;
            #endif   // DEBUG_TABLES && DEBUG_TOKEN

            gsNext.erase( tokOFF, 4 ) ;         // erase the token

            //* If there is no display text left in gsNext, *
            //* (just the <p> tag), read the next line and  *
            //* append it to gsNext. It will be pushed back *
            //* into the input stream below. This prevents  *
            //* a blank line at the top of the cartouche.   *
            if ( gsNext.gschars() <= 4 )
            {
               status = this->ppfReadSrcLine ( gst, ni ) ;
               gsNext.append( gst.gstr() ) ;
            }

            #if DEBUG_TABLES != 0 && DEBUG_TOKEN != 0
            gsdbg.append( "     ------'%S'", gsNext.gstr() ) ;
            this->textOut ( gsdbg ) ;
            #endif   // DEBUG_TABLES && DEBUG_TOKEN
         }
      }
      else     // multitable
      {
         //* If an embedded token is present at the expected offset, *
         //* extract and decode it. Otherwise data will be unchanged.*
         //* The line will begin with one of:                        *
         //*   "<thead><tr><th>=xx=" (header row)                    *
         //*   "<tbody><tr><td>=xx=" (body row, header not present)  *
         //* "xx" is the two-character formatting code.              *
         tokOFF = 15 ;                          // offset to embedded token
         if ( (wbuff[tokOFF] == L'=') && (wbuff[tokOFF + 3] == L'=') )
         {
            //* First character specified bordered ('b') vs. unbordered ('n') *
            //* table, and second character is the font size.                 *
            usrResp   = tolower(wbuff[tokOFF + 1]) ;
            font_size = tolower(wbuff[tokOFF + 2]) ;
            if ( usrResp == nobdrRESP ) { usrResp = nobdrRESP ; }
            else                        { usrResp = bdrRESP ; }
            if ( (font_size != fontSMALL) && (font_size != fontLARGE) )
            { font_size = L'.' ; }
            isToken = true ;

            #if DEBUG_TABLES != 0 && DEBUG_TOKEN != 0
            gsdbg.compose( "TOKEN------'%S'\n"
                           "     ------'%C' '%C'\n", 
                           gsNext.gstr(), &usrResp, &font_size ) ;
            #endif   // DEBUG_TABLES && DEBUG_TOKEN

            gsNext.erase( tokOFF, 4 ) ;         // erase the token

            #if DEBUG_TABLES != 0 && DEBUG_TOKEN != 0
            gsdbg.append( "     ------'%S'", gsNext.gstr() ) ;
            this->textOut ( gsdbg ) ;
            #endif   // DEBUG_TABLES && DEBUG_TOKEN
         }
      }

      this->ppfUnReadSrcLine ( gsNext ) ;   // push line back into the stream
   }
   else { return ( status ) ; }  // If a bad read from source, we're done.(unlikely)

   //* Get user's response:                                 *
   //* If a table AND not automatic table processing, or if *
   //* a cartouche AND not automatic cartouche processing.  *
   if ( (isCartouche && (this->cartFormat == cfgSpec)) ||
        (!isCartouche && (this->tabBorder == cfgSpec)) )
   {
      //* Read the next line of the table to give user visual context *
      if ( (status = this->ppfReadSrcLine ( gsNext, ni )) == OK )
      {
         //* Push the context line back into the stream      *
         //* format the text portion of the line for display.*
         this->ppfUnReadSrcLine ( gsNext ) ;

         //* Remove the HTML tag(s) from the context data to be displayed. *
         //* Programmer's Note: This is tedious because table entries have *
         //* many HTML tags before, during and after the display text.     *
         //* See the example above.                                        *
         //* The cartouche table is much simpler, containing only the      *
         //* paragraph tag ("<p>"), so that is tested first.               *
         if ( isCartouche && ((ni = gsNext.find( paraGraph )) >= ZERO) )
            removeTag ( gsNext, ni ) ;
         //* Multitable elements *
         else
         {
            if ( (ni = gsNext.find( headStart )) >= ZERO ) // table WITH header record
            {
               gsNext.erase( headStart, ni ) ;
               if ( (ni = gsNext.find( headEnd )) >= ZERO )
                  gsNext.erase( headEnd, ni ) ;
               gsNext.replace( headFldEnd,   spaceString, ZERO, false, true ) ;
               gsNext.replace( headFldStart, spaceString, ZERO, false, true ) ;
            }
            else        // (table WITHOUT header record)
            {
               if ( (ni = gsNext.find( bodyStart )) >= ZERO )
                  removeTag ( gsNext, ni ) ;
               if ( (ni = gsNext.find( bodyEnd )) >= ZERO )
                  removeTag ( gsNext, ni ) ;
            }
            //* Field markers shared between header records and plain records *
            gsNext.replace( fieldEndA,   spaceString, ZERO, false, true ) ;
            gsNext.replace( fieldStartA, spaceString, ZERO, false, true ) ;
            gsNext.replace( fieldEndB,   spaceString, ZERO, false, true ) ;
            gsNext.replace( fieldStartB, spaceString, ZERO, false, true ) ;
         }

         //* Write the user prompt *
         if ( isCartouche )
         {
            gsOut.compose( "____________________________________________\n"
                            "Cartouche found on Line:%4hu\n"
                            "First Row: %S\n"
                            "Text formatting option:\n"
                            "  with (s)=smaller text, or (l)=larger text\n"
                            "a:automatic, fixed text format (default)\n"
                            "f:flow text within field    A:All automatic\n"
                            "response format: TEXT_FORMAT[FONT_SIZE]",
                            &this->slCount, gsNext.gstr()
                         ) ;
            this->textOut ( gsOut ) ;
            this->textOut ( L"your choice: ", false ) ;
         }
         else
         {
            gsOut.compose( "____________________________________________\n"
                            "Table found on Line:%4hu\n"
                            "First Row: %S\n"
                            "  with (s)=smaller text, or (l)=larger text\n"
                            "b: add a border   n: no border   A:all\n"
                            "response format: BORDER[FONT_SIZE]",
                            &this->slCount, gsNext.gstr()
                         ) ;
            this->textOut ( gsOut ) ;
            this->textOut ( L"your choice: ", false ) ;
         }

         //* Get user's response *
         while ( ! this->abort )
         {
            this->userResponse ( gsIn ) ;    // talk with the animals

            if ( isCartouche )      // target is a cartouche
            {
               //* If abort received. Note the early return. *
               if ( this->abort )
               { return ( status = ERR ) ; }

               //* If "default_token" received *
               if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
               {
                  if ( ! isToken )        // do not override the embedded token
                     usrResp = textAUTO ; // default for cartouche is 'Automatic'
               }
               else
               {
                  usrResp = *gsIn.gstr() ;

                  //* If a secondary argument is specified, it is *
                  //* interpreted as font size.                   *
                  if ( (gsIn.gschars()) > 2 )
                     font_size = tolower(gsIn.gstr()[1]) ;
               }

               //         'auto'                  'wrap'            'All default'
               if ( usrResp == textAUTO || usrResp == textFLOW || usrResp == textALL )
               { break ; }
               else  // call user a dumbguy and ask for correct response
               { this->invalidResponse () ; }
            }
            else                    // target is a standard multitable
            {
               //* If abort received. Note the early return. *
               if ( this->abort )
               { return ( status = ERR ) ; }

               //* If "default_token" received *
               if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
               {
                  if ( ! isToken )        // do not override the embedded token
                     usrResp = bdrRESP ;  // default is 'Border'
               }
               else
               {
                  usrResp = *gsIn.gstr() ;
                  if ( (usrResp == L'y') || (usrResp == L'a') )
                  { usrResp = bdrRESP ; }
                  // Programmer's Note: In previous releases, a border was specified 
                  // by a 'y' (yes), so this test is for backward compatibility,
                  // and 'a' is silently accepted for compatibility with other prompts.

                  //* If a secondary argument is specified, it is *
                  //* interpreted as font size.                   *
                  if ( (gsIn.gschars()) > 2 )
                     font_size = tolower(gsIn.gstr()[1]) ;
               }

               //    'bordered'         'unbordered'      'All default'
               if ( usrResp == bdrRESP || usrResp == nobdrRESP || usrResp == bdrallRESP )
               { break ; }
               else  // call user a dumbguy and ask for correct response
               { this->invalidResponse () ; }
            }
         }     // input loop
      }        // read context line
   }           // request user response
   //* If skip count is active, decrement the counter *
   else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

   //* ------------------------------------- *
   //* Replace existing class with class     *
   //* corresponding to selection, and write *
   //* tag to target.                        *
   //* ------------------------------------- *

   //* If the target object is a cartouche *
   if ( isCartouche && ! this->abort )
   {
      ni = gsB.find( tabCartouche ) ;  // remove the existing tag
      removeTag ( gsB, ni ) ;

      //* Validate secondary formatting (font size) *
      if ( font_size != fontSMALL && font_size != fontLARGE ) { font_size = L'.' ; }

      #if DEBUG_TABLES != 0
      gsdbg.compose( "USR--------'%C' '%C'", &usrResp, &font_size ) ; this->textOut ( gsdbg ) ;
      #endif   // DEBUG_TABLES

      //* Insert a reference to the cartouche class *
      //* which defines text as pre-formatted.      * 
      if ( (usrResp == textAUTO) || (usrResp == textALL) )
      {
         if ( font_size == fontSMALL )          // formatted, small font
            gsB.insert( preCLASSs, ni ) ;
         else if ( font_size == fontLARGE )     // formatted, large font
            gsB.insert( preCLASSl, ni ) ;
         else                                   // formatted, standard font
            gsB.insert( preCLASSi, ni ) ;

         //* If user specified 'All', then set the global indicator *
         //* to process all remaining cartouches automagically.     *
         if ( (this->cartFormat == cfgSpec) && usrResp == L'A' )
            this->cartFormat = cfgAuto ;
      }
      //* Insert the cartouche class with  *
      //* free-flow style.  (textFLOW)     *
      else
      {
         if ( font_size == fontSMALL )          // flowing, small font
            gsB.insert( flowCLASSs, ni ) ;
         else if ( font_size == fontLARGE )     // flowing, large fone
            gsB.insert( flowCLASSl, ni ) ;
         else                                   // flowing, standard font
            gsB.insert( flowCLASS, ni ) ;

         //* Insert a hard line break _within_ the field.*
         //* This restores the top padding that was lost *
         //* when applying the new flow style.           *
         gsB.append( L"<br>" ) ;
      }
   }

   //* Target object is a multitable *
   else if ( ! this->abort )
   {
      if ( (usrResp == bdrRESP) || (usrResp == bdrallRESP) )
      {
         const wchar_t* wtarget = ((font_size == fontSMALL) ? bdrCLASSs :
                                   (font_size == fontLARGE) ? bdrCLASSl : bdrCLASS) ;
         if ( isTabMulti )
            gsB.replace( tabMulti, wtarget ) ;
         else if ( isTabSimple )
            gsB.replace( tabSimple, wtarget ) ;
         else if ( isTabBordered )
            gsB.replace( bdrCLASS, wtarget ) ;

         //* If user specified 'All', then set the global indicator *
         //* to process all remaining tables automagically.         *
         if ( (this->tabBorder == cfgSpec) && (usrResp == bdrallRESP) )
            this->tabBorder = cfgAuto ;
      }
      else  // (usrResp == nobdrRESP 'n')
      {
         const wchar_t* wtarget = ((font_size == fontSMALL) ? nobdrCLASSs :
                                   (font_size == fontLARGE) ? nobdrCLASSl : nobdrCLASS) ;
         if ( isTabMulti )
            gsB.replace( tabMulti, wtarget ) ;
         else if ( isTabSimple )
            gsB.replace( tabSimple, wtarget ) ;
         else if ( isTabBordered )
            gsB.replace( bdrCLASS, wtarget ) ;
      }

      #if DEBUG_TABLES != 0
      if ( font_size != fontSMALL && font_size != fontLARGE )
      { font_size = L'-' ; }
      gsdbg.compose( "USR--------'%C' '%C'", &usrResp, &font_size ) ; this->textOut ( gsdbg ) ;
      #endif   // DEBUG_TABLES
   }
   //* Write the updated tag to target *
   this->ppfWriteLine ( gsB ) ;

   #if DEBUG_TABLES != 0
   gsdbg.compose( "OUT------>> '%S'", gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_TABLES

   //* Copy the remainder of the object *
   while ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
   {
      this->ppfWriteLine ( gss ) ;
      if ( (gss.find( endTable )) >= ZERO )
         break ;
   }

   return status ;

   #undef  DEBUG_TOKEN
}  //* End ppfProcTABLE() *

//*************************
//*      ppfProcHEAD      *
//*************************
//********************************************************************************
//* Process the <head> ... </head> block. Caller has written the <head> tag      *
//* to the target file. We scan until the </head> tag is found (or until error   *
//* reading from the source file).                                               *
//*                                                                              *
//* Most of the data in this block is generated by the makeinfo processor, and   *
//* is rather useless. There are a few exceptions:                               *
//* 1) The <title> tag is the title of the document. This is not displayed within*
//*    the document itself, but may be scanned by the browser and displayed as   *
//*    tab text.                                                                 *
//* 2) A "content type" <meta> tag indicates the MIME type and character         *
//*    encoding; however, this is in a deprecated format, and is replaced by our *
//*    HTML5 tag.                                                                *
//* 3) A <link> to the Infodoc CSS3 stylesheet is inserted into the <head> block.*
//*    This is the definition file for CSS formatting.                           *
//* 4) Additional metadata entries will be present, but are deleted by default.  *
//*    If desired, the canonical metadata tags may be retained using the         *
//*    "--no_meta" command-line argument. The makeinfo fantasy-world tags are    *
//*    always removed.                                                           *
//* 5) Several <link> tags are generated by the makeinfo utility. These are      *
//*    completely useless, but if desired, they may be retained using the        *
//*    "--no_link" command-line argument.                                        *
//* 6) If comment(s) have been inserted into the <head> block, retain them.      *
//*    This is most likely a generated by the @insertcopying command.            *
//* 7) There is a <style> tag which includes several HTML style definitions, but *
//*    because we use CSS3 styles, this tag is always deleted.                   *
//* 8) The texinfo language includes the "@copying" command. This usually        *
//*    contains a copyright message or similar which may then be written to      *
//     multiple places in the document using the "@insert-copying" command.      *
//*    The HTML output writes a superfluous copy of this as a comment in the     *
//*    <head> block. This comment is deleted.                                    *
//*                                                                              *
//* Input  : nothing                                                             *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************

short Idpp::ppfProcHEAD ( void )
{
   //* VALID metadata that may be seen in the <head> section *
   const wchar_t* metaApp = L"<meta name=\"application-name" ;
   const short    metaApp_len = 28 ;
   const wchar_t* metaAuth = L"<meta name=\"author" ;
   const short    metaAuth_len = 18 ;
   const wchar_t* metaDesc = L"<meta name=\"description" ;
   const short    metaDesc_len = 23 ;
   const wchar_t* metaGen = L"<meta name=\"generator" ;
   const short    metaGen_len = 21 ;
   const wchar_t* metaKeys = L"<meta name=\"keywords" ;
   const short    metaKeys_len = 20 ;
   const wchar_t* metaDate = L"<meta name=\"date\"" ;
   const short    metaDate_len = 17 ;
   const wchar_t* linkBegin = L"<link" ;
   const short    linkBegin_len = 5 ;

   gString gs,             // input buffer (convert source data to wide text)
           gsOut ;         // output formatting
   short wi,               // index into source line
         status = OK ;     // return value

   //* Insert the post-processing comment and the *
   //* meta link to the CSS style sheet file.     *
   gsOut.compose( cssLINK0, AppTitle1, AppVersion ) ;
   this->ppfWriteLine ( gsOut ) ;
   gString gscf ;
   gsOut.compose( L"%s%S%s", cssLINK1, this->cssFile.gstr(), cssLINK2 ) ;
   this->ppfWriteLine ( gsOut ) ;   // write file link to target
   this->tlCount += 2 ; // this is a 3-line sequence

   while ( status == OK )
   {
      //* Read a line from the source file *
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* If <title> tag found, copy it to target *
         if ( (gs.compare( titleTag, false, titleTag_len, wi )) == ZERO )
         {
            this->ppfWriteLine ( gs ) ;      // copy the title to target
            // NOTE: We assume that the entire title is on one line.
         }

         //* If end of <head></head> block found *
         else if ( (gs.compare( headendTag, false, headTag_len, wi )) == ZERO )
         {
            //* If specified, insert user's custom metadata into the    *
            //* <head>...</head> block. We DO NOT validate these data.*
            if ( (this->userMeta.gschars()) > 1 )
            {
               if ( this->verbose != false )
               {
                  gString gsVerb( "(%4hu) Inserting custom data at line %hu.", 
                                  &this->slCount, &this->tlCount ) ;
                  this->textOut ( gsVerb ) ;
               }

               //* Copy data from user file to target *
               status = this->ppfInsertCustomData ( this->userMeta ) ;
            }

            //* Write the end-of-head tag *
            this->ppfWriteLine ( gs.ustr() ) ;

            break ;        // our <head> is now (nearly :) empty
         }

         //* If user wants to retain the VALID metadata *
         else if ( (this->no_meta != false) &&
                   (((gs.compare( metaDesc, false, metaDesc_len, wi )) == ZERO) ||
                   ((gs.compare( metaKeys, false, metaKeys_len, wi )) == ZERO)  ||
                   ((gs.compare( metaAuth, false, metaAuth_len, wi )) == ZERO)  ||
                   ((gs.compare( metaApp, false, metaApp_len, wi )) == ZERO)    ||
                   ((gs.compare( metaGen, false, metaGen_len, wi )) == ZERO)    ||
                   ((gs.compare( metaDate, false, metaDate_len, wi )) == ZERO)) )
         {
            status = this->ppfProcMETA ( gs ) ;
         }

         //* If user wants to retain the <link> entries *
         else if ( (this->no_link != false) &&
                   ((gs.compare( linkBegin, false, linkBegin_len, wi )) == ZERO) )
         {
            status = this->ppfProcMETA ( gs ) ;
         }

         //* If the HTML style block is found, delete it *
         else if ( (gs.find( L"<style" )) >= ZERO )
         {
            do
            {
               if ( (status = this->ppfReadSrcLine ( gs, wi )) != OK )
                  break ;  // read error
            }
            while ( (gs.find( L"</style>" )) < ZERO ) ;
         }
         //* Retain comments generated by the makeinfo @insertcopying command. *
         //* Discard the comment which contains the list of 
         else if ( (gs.find( commBegin )) == ZERO )
         {
            this->ppfWriteLine ( gs.ustr() ) ;
            while ( (gs.find( commEnd )) < ZERO )
            {
               if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
                  this->ppfWriteLine ( gs.ustr() ) ;
               else        // read error
                  break ;
            }
         }

         else
         { /* Discard the source line */ }
      }
   }

   return status ;

}  //* End ppfProcHEAD() *

//*************************
//*      ppfProcMETA      *
//*************************
//********************************************************************************
//* If the '--no_meta' option OR the '--no_links' option OR the '--no_head'      *
//* option has been invoked, then this method is called to retain the entry.     *
//* See ppfProcessHEAD method for element selection.                             *
//*                                                                              *
//* Although the default entries are single-line entries, IF the source          *
//* document was generated with custom data, then the entry can span multiple    *
//* lines.                                                                       *
//*                                                                              *
//* Input  : (by reference) contains the first line of the entry to be retained  *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************

short Idpp::ppfProcMETA ( const gString& gsmeta )
{
   gString gs ;            // additional source data
   short wi ;              // index into source line
   bool done = false ;     // loop control
   short status = OK ;     // return value

   //* Write the first (and possibly only) line of the entry.*
   this->ppfWriteLine ( gsmeta ) ;

   //* Index the end-of-tag character *
   if ( (wi = gsmeta.find( tagClose )) >= ZERO )
      done = true ;

   //* If metadata entry not yet terminated, scan and copy additional lines.*
   while ( ! done )
   {
      status = this->ppfReadSrcLine ( gs, wi ) ;   // read from source
      this->ppfWriteLine ( gs ) ;                  // write to target
      if ( status == OK )
      {  //* Scan the line for end-of-entry character.*
         if ( (wi = gs.find( tagClose )) >= ZERO )
            done = true ;
      }
   }     // while()
   return status ;

}  //* End ppfProcMETA() *

//**************************
//*    ppfItemizedList     *
//**************************
//********************************************************************************
//* Test whether the provided data indicates the beginning of an itemized        *
//* (<ul>) list.                                                                 *
//*                                                                              *
//* If the target block was found, process it and write it to the target file.   *
//*                                                                              *
//* Input  : gsln   : line of source data to be scanned                          *
//*          status ; (by reference) receives 'ERR' if syntax or other error,    *
//*                   else receives 'OK'                                         *
//*          wi     : index into the wide (wchar_t) source data                  *
//*          bookit : (optional, 'false' by default                              *
//*                   if 'true', AND if 'book' member set, then write the        *
//*                      bookmark messages to the display                        *
//*                   if 'false', do not write bookmark messages                 *
//*                                                                              *
//* Returns: 'true' if a preformatted block was processed                        *
//*          'false' caller is responsible for processing the specified data     *
//********************************************************************************

bool Idpp::ppfItemizedList ( const gString& gsln, short& status, short wi, bool bookit )
{
   gString gsBook ;           // text formatting
   bool isBlock = false ;     // return value

   status = OK ;              // Initialize caller's status value (hope for success)

   if ( (this->ppfTestItemizedList ( gsln, wi )) != false )
   {
      isBlock = true ;        // formatted block identified

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"ML(%hu) UBLOCK START", &this->slCount ) ;
         this->textOut ( gsBook, false ) ;
      }

      //* Process the top-level list and any nested <ul> and <ol> lists.*
      gString gs = gsln ;
      status = this->ppfProcItemizedList ( gs ) ;

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"--(%hu) UBLOCK END", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }
   }
   return isBlock ;

}  //* End ppfItemizedList() *

//*************************
//*  ppfTestItemizedList  *
//*************************
//********************************************************************************
//* Test the provided text for the beginning of an itemized list (<ul> block).   *
//*    See notes below.                                                          *
//*                                                                              *
//* Input  : gssrc : (by reference) contains the source data line being          *
//*                  processed                                                   *
//*          wi    : offset into 'gssrc' at which to begin search                *
//*                                                                              *
//* Returns: 'true'  if line contains a <ul class="no-bullet'> tag               *
//*          'false' otherwise                                                   *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* 1) Normally, a <ul...> tag is alone on a line, but in some cases it could    *
//*    be preceeded by other data.                                               *
//*                                                                              *
//* 2) We must distinguish between <ul> lists generated by the texi-to-HTML      *
//*    converter and HTML code embedded within the texi source document.         *
//*    There is no way to be 100% certain about this, but we "know" that the     *
//*    converter always ends the first line of the list immediately after        *
//*    the opening tag. Thus if there is data following the opening tag on the   *
//*    same line, we assume that the list WAS NOT generated by the converter     *
//*    (or that it has already been post-processed).                             *
//*                                                                              *
//* 3) Note that the Table Of Contents is an itemized list; however, we will     *
//*    never see it here.                                                        *
//*                                                                              *
//* 4) Note that the chapter menus are itemized lists, and they will be          *
//*    identified as such so the caller can process (or ignore) them.            *
//*                                                                              *
//********************************************************************************

bool Idpp::ppfTestItemizedList ( const gString& gssrc, short wi )
{
   bool ulListFound = false ;

   //* If source line contains '<ul>' tag AND it is last item on source line *
   if ( (wi = gssrc.find( ulBegin, wi, false, ulBegin_len )) >= ZERO )
   {
      wi = gssrc.after( tagClose, wi ) ;
      if ( (wi > ZERO) && (gssrc.gstr()[wi] == NULLCHAR) )
         ulListFound = true ;
   }

   return ulListFound ;

}  //* End ppfTestItemizedList() *

//*************************
//*  ppfProcItemizedList  *
//*************************
//********************************************************************************
//* Process itemized (<ul>) lists. Also called recursively.                      *
//*                                                                              *
//* Note that for automatic processing, only <ul> blocks declared with the       *
//* "no-bullet" class are actively processed. (<ul class="no-bullet">)           *
//* For interactive processing, any list may optionally be assigned one of the   *
//* class descriptions recognized by the browser's rendering engine.             *
//*                                                                              *
//*                                                                              *
//* Input  : gsul   : (by reference) contains the first line of the list:        *
//*                   Examples:   '<ul>'   '<ul class="no-bullet">'              *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* 1) Because lists may be embedded within other lists, this method may be      *
//*    called recursively, so be sure all data are safely on the stack.          *
//*                                                                              *
//* 2) The texi source fully supports four(4) bullet types:                      *
//*      a) "@bullet" : standard filled-disc U+2212 character                    *
//*                     (this is also the default if no character specified)     *
//*                     <ul class="itemize mark-bullet">                         *
//*      b) "@textdegree" : the degrees (temperature) U+00B0 (&deg) character    *
//*                         <ul class="itemize mark-textdegree">                 *
//*      c) "@minus"  : Unicode minus character U+2212                           *
//*                     <ul class="itemize mark-minus">                          *
//*      d) "@w{}" : no bullet  (formerly "no-bullet" class)                     *
//*                  <ul class="itemize mark-none">                              *
//*    The declared classes for these bullet types are defined in the file       *
//*    "infodoc-styles.css".                                                     *
//*    All other bullet characters generate the following construct where        *
//*    the specified character is embedded in the <ul> tag as a style element.   *
//*       <ul class="itemize" style="list-style-type: '▪'">                      *
//*       <ul class="itemize" style="list-style-type: '&gt;'"> ('>')             *
//*    The base "itemize" class is defined; however, it does not specify a       *
//*    bullet character, so the "list-style-type" controls the output.           *
//*                                                                              *
//*    The makeinfo engine defines two additional unordered lists:               *
//*      a) <ul class="toc-numbered-mark">                                       *
//*         This list contains the document's Table Of Contents, and will never  *
//*         be seen here                                                         *
//*      b) <ul class="mini-toc">                                                *
//*         This list type is used to generate chapter-heading menus.            *
//*                                                                              *
//* 3) "infodoc-styles.css" defines classes for a small number of bullet types.  *
//*    Each of the supported bullet types requires three synchronized items:     *
//*      a) A Texinfo macro defined in "texi_macros.texi".                       *
//*      b) A CSS class definition in "infodoc-styles.css".                      *
//*      c) A constant definition in this method to identify the character for   *
//*         processing: 'sDISC' through 'lDIAMOND'  (see below)                  *
//*    To modify any of these bullet types requires updating all three references*
//*    in synchronization.                                                       *
//*                                                                              *
//* 4) Because makeinfo v:7 now declares a definition for ALL unordered lists,   *
//*    the browser can no longer control the character within embedded lists.    *
//*    Some users will no doubt be relying on the browser to handle embedded     *
//*    lists, so we define a special character to re-enable browser control in   *
//*    that scenario:                                                            *
//*       a) @CIRCLESLASH macro defines the character: '⊘' (U+2298) &osol;       *
//*       b) 'cirSLASH' numeric constant defined below to identify the           *
//*                     character during post-processing.                        *
//*    When this character is identified, the tag which contains it is replaced  *
//*    by the plain <ul> tag. This returns control of bullet-character selection *
//*    to the browser.                                                           *
//*    Example: <ul class="itemize" style="list-style-type: '▪'">                *
//*             becomes: <ul>                                                    *
//*                                                                              *
//* 5) Previously-processed lists are passed through unmodified by default;      *
//*    however, if the user is running a second pass on the document, AND if     *
//*    an automatic response file is used, we must prompt for a response on this *
//*    previously-processed list to avoid getting the automatic responses        *
//*    out-of-synch. In this case, the response is ignored and the list is       *
//*    is passed through as written.                                             *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* makeinfo v:7 source:                                                         *
//* --------------------                                                         *
//* <ul class="disc-bullet">   (direct HTML markup or previously processed v:6)  *
//* <ul class="circle-bullet"> (direct HTML markup or previously processed v:6)  *
//* <ul class="square-bullet"> (direct HTML markup or previously processed v:6)  *
//* 
//* makeinfo v:6.8 itemized lists:
//* ------------------------------
//* <ul class="no-bullet">
//* <ul style="list-style-type:none;">
//* <ul style="list-style-type:disc;">
//* <ul style="list-style-type:circle;">
//* <ul style="list-style-type:square">
//* <ul class="square-bullet">
//* <ul class="section-toc">
//* <ul>
//* 
//* makeinfo v:6.6 itemized lists:
//* ------------------------------
//* <ul class="toc-level[1 | 2 | 3]">
//* 
//********************************************************************************

short Idpp::ppfProcItemizedList ( gString& gsul )
{
   #define DEBUG_ILIST (0)    // for debugging only

   //* Makeinfo v:7 defined classes and embedded CSS style *
   const wchar_t* itemBASE    = L"<ul class=\"itemize" ;
   const wchar_t* itemSTYLE   = L"<ul class=\"itemize\" style=\"list-style-type: '" ;
   const wchar_t* itemMINITOC = L"<ul class=\"mini-toc\">" ;

   //* Custom CSS classes for itemized lists. *
   const wchar_t* ul_sDISC    = L"<ul class=\"disc-small\">" ;
   const wchar_t* ul_mDISC    = L"<ul class=\"disc-medium\">" ;
   const wchar_t* ul_lDISC    = L"<ul class=\"disc-large\">" ;
   const wchar_t* ul_sCIRCLE  = L"<ul class=\"circle-small\">" ;
   const wchar_t* ul_mCIRCLE  = L"<ul class=\"circle-medium\">" ;
   const wchar_t* ul_lCIRCLE  = L"<ul class=\"circle-large\">" ;
   const wchar_t* ul_sSQUARE  = L"<ul class=\"square-small\">" ;
   const wchar_t* ul_mSQUARE  = L"<ul class=\"square-medium\">" ;
   const wchar_t* ul_lSQUARE  = L"<ul class=\"square-large\">" ;
   const wchar_t* ul_sPOINTER = L"<ul class=\"pointer-small\">" ;
   const wchar_t* ul_mPOINTER = L"<ul class=\"pointer-medium\">" ;
   const wchar_t* ul_lPOINTER = L"<ul class=\"pointer-large\">" ;
   const wchar_t* ul_sDIAMOND = L"<ul class=\"diamond-small\">" ;
   const wchar_t* ul_mDIAMOND = L"<ul class=\"diamond-medium\">" ;
   const wchar_t* ul_lDIAMOND = L"<ul class=\"diamond-large\">" ;
   const wchar_t* ul_NOBULLET = L"<ul class=\"mark-none\">" ;
   const wchar_t* ul_PLAIN    = L"<ul>" ;
   const wchar_t* fontSMALL   = L" font-small" ;
   const wchar_t* fontLARGE   = L" font-large" ;

   //* Each of these constants corresponds to a Texinfo macro *
   const wchar_t sDISC    = 0x2022 ;
   const wchar_t mDISC    = 0x25CF ;
   const wchar_t lDISC    = 0x23FA ;
   const wchar_t sCIRCLE  = 0x25E6 ;
   const wchar_t mCIRCLE  = 0x26AC ; // (this is also the legacy @BCIRCLE) (see also 0x25CB)
   const wchar_t lCIRCLE  = 0x25EF ;
   const wchar_t sSQUARE  = 0x25AB ; // HTML uses the 'square' style element
   const wchar_t mSQUARE  = 0x25AA ;
   const wchar_t lSQUARE  = 0x25FB ; // (hollow square)
   const wchar_t sPOINTER = 0x2023 ;
   const wchar_t mPOINTER = 0x25B8 ;
   const wchar_t lPOINTER = 0x25B7 ; // (hollow pointer)
   const wchar_t sDIAMOND = 0x2666 ;
   const wchar_t mDIAMOND = 0x25C6 ;
   const wchar_t lDIAMOND = 0x25C7 ; // (hollow diamond)
   const wchar_t cirSLASH = 0x2298 ; // @CIRCLESLASH macro for: '⊘' (U+2298) &osol;

   const wchar_t sFONT    = L's' ;   // "smaller" font-size token
   const wchar_t iFONT    = L'i' ;   // "inherited" (default) font-size token
   const wchar_t lFONT    = L'l' ;   // "larger" font-size token

   const wchar_t sBULL    = L's' ;   // "small" bullet size
   const wchar_t mBULL    = L'm' ;   // "medium" bullet size (default)
   const wchar_t lBULL    = L'l' ;   // "large" bullet size


   gString gs, gst,              // text analysis and formatting
           origUL,               // copy of unmodified UL tag
           fmtUL,                // formatted UL tag
           dispLI,               // stripped line item for display to user
           gsIn,                 // user response to prompt
           gsVerb ;              // for 'verbose' output
   wchar_t usrResp = L'x' ;      // user response to prompt
   wchar_t bullSize = mBULL ;    // bullet size (small, medium, large)
   wchar_t fontSize = iFONT,     // font size (small, inherit, large)
           tmpFSize ;            // temporary relative font size
   blkType bType = btNone ;      // block-type
   short   wi = gsul.scan(),     // index the first non-whitespace character
           ti,                   // claracter index
           ul,                   // index of UL tag
           lineItems = ZERO,     // count the <li> tags
           status = OK ;         // return value
   bool prompt_user = this->ulLists == cfgSpec ; // local copy of prompt flag
   bool respFile    = this->rifs.is_open() ;    // 'true' if open response file
   bool ignoreResp  = false; // 'true' if forced user response on Already Processed list

   //* Index the beginning of UL tag, and test for             *
   //* specification of class name(s) and/or style information.*
   if ( (ul = gsul.find( ulBegin, wi, false )) >= ZERO )
   {
      //* Save a copy of the original UL tag.                  *
      //* Note: We assume two things here:                     *
      //*  1. The complete tag is present in the source line.  *
      //*  2. The tag is the last thing in the source line.    *
      //*     This means that the output can be updated by     *
      //*     appending the modified tag.                      *
      copyTag ( gsul, ul, origUL ) ;   // save a copy of the original UL tag
      //* Delete the original tag from output data *
      removeTag ( gsul, ul ) ;

      #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
      gsdbg.compose( "ILIST at:%04hd tag:'%S'", &this->slCount, origUL.gstr() ) ;
      this->textOut ( gsdbg ) ;
      #endif   // DEBUG_LISTS && DEBUG_ILIST

      //* Scan the first line item for an embedded token  *
      //* signalling additional formatting options.       *
      //* If token found, it will be decoded then removed.*
      if ( (this->ppfUListToken ( tmpFSize )) )
      {
         if ( (tmpFSize == sFONT) || (tmpFSize == lFONT) )
            fontSize = tmpFSize ;
      }

      //* Test for classes declared in makeinfo v:7               *
      //* ------------------------------------------------------- *
      //* 1) Test for presence of BASE class: "itemize".          *
      //* 2) Test for a STYLEd list (embedded bullet character).  *
      //*    If a known character, replace styled tag with the    *
      //*    corresponding class.                                 *
      //* 3) Test for a Secondary CLASS name,                     *
      //*    a) a known secondary class (class is defined in CSS) *
      //*    b) a custom, user-specific class name. (unmodified)  *
      //* 4) Test for legacy (makeinfo v:6) syntax.               *
      //*    a) a known v:6 class                                 *
      //*    b) a custom (prevously processed) class name,        *
      wchar_t wchar,                      // test character
              style_char ;                // embedded style character
      short ti, sti ;                     // character indices

      //* 'ti' references the pivotal character of the sequence.  *
      if ( (ti = origUL.after( itemBASE )) > ZERO )
      {
         switch ( (wchar = origUL.gstr()[ti]) ) // extract the test character
         {
            //* Found closing quote after "itemize"<--*
            //* Style declaration may follow.         *
            case L'"':                                   // base "itemize" class
               if ( (sti = origUL.after( itemSTYLE )) > ti )
               {
                  style_char = origUL.gstr()[sti] ;

                  #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
                  gsdbg.compose( "---style_char:'%C' U+%04X ", &style_char, &style_char ) ;
                  #endif   // DEBUG_LISTS && DEBUG_ILIST

                  switch ( style_char )
                  {
                     //* These entries replace the in-line style command *
                     //* for known bullet characters, and if automatic   *
                     //* processing, will be the tags written to target. *                           *
                     case sDISC:    fmtUL = ul_sDISC ;    bullSize = sBULL ;  break ;
                     case mDISC:    fmtUL = ul_mDISC ;    bullSize = mBULL ;  break ;
                     case lDISC:    fmtUL = ul_lDISC ;    bullSize = lBULL ;  break ;
                     case sCIRCLE:  fmtUL = ul_sCIRCLE ;  bullSize = sBULL ;  break ;
                     case mCIRCLE:  fmtUL = ul_mCIRCLE ;  bullSize = mBULL ;  break ;
                     case lCIRCLE:  fmtUL = ul_lCIRCLE ;  bullSize = lBULL ;  break ;
                     case sSQUARE:  fmtUL = ul_sSQUARE ;  bullSize = sBULL ;  break ;
                     case mSQUARE:  fmtUL = ul_mSQUARE ;  bullSize = mBULL ;  break ;
                     case lSQUARE:  fmtUL = ul_lSQUARE ;  bullSize = lBULL ;  break ;
                     case sPOINTER: fmtUL = ul_sPOINTER ; bullSize = sBULL ;  break ;
                     case mPOINTER: fmtUL = ul_mPOINTER ; bullSize = mBULL ;  break ;
                     case lPOINTER: fmtUL = ul_lPOINTER ; bullSize = lBULL ;  break ;
                     case sDIAMOND: fmtUL = ul_sDIAMOND ; bullSize = sBULL ;  break ;
                     case mDIAMOND: fmtUL = ul_mDIAMOND ; bullSize = mBULL ;  break ;
                     case lDIAMOND: fmtUL = ul_lDIAMOND ; bullSize = lBULL ;  break ;
                     case cirSLASH: fmtUL = ul_PLAIN ;    bullSize = mBULL ;  break ;

                     //* None of the supported bullet characters are       *
                     //* interpreted by makeinfo as HTML-defined constant  *
                     //* elements, so if seen, they are passed through as  *
                     //* written. Exception: As a preemptive enhancement,  *
                     //* makeinfo may, in future, substitute the &osol;    *
                     //* element for a specified '⊘' (U+2298) circle-slash *
                     //* character i.e. the @CIRCLESLASH macro.            *
                     case L'&':
                        #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
                        {
                        short i = sti ;
                        gsdbg.append( L'(' ) ;
                        while ( origUL.gstr()[i] != L'\'' )
                           gsdbg.append( origUL.gstr()[i++] ) ;
                        gsdbg.append( L") " ) ;
                        }
                        #endif   // DEBUG_LISTS && DEBUG_ILIST
                        if ( (origUL.find( L"&osol;", sti )) >= sti )
                        { fmtUL = ul_PLAIN ; break ; }
                     default:
                        fmtUL = origUL ;
                        break ;
                  }
                  #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
                  gsdbg.append( fmtUL.gstr() ) ;
                  this->textOut ( gsdbg ) ;
                  #endif   // DEBUG_LISTS && DEBUG_ILIST
               }
               else  // unstyled basic "itemize" class (or unexpected syntax)
               {
                  fmtUL = origUL ;        // use default disc subclass
               }
               break ;

            //* "itemize" class with sub-class *
            case L' ':
               //* Scan for standard makeinfo secondary-class name *
               ++ti ;                                    // index the sub-class
               if ( ((origUL.find( L"mark-bullet", ti     )) >= ZERO) ||
                    ((origUL.find( L"mark-minus", ti      )) >= ZERO) ||
                    ((origUL.find( L"mark-textdegree", ti )) >= ZERO) ||
                    ((origUL.find( L"mark-none", ti       )) >= ZERO) )
               {
                  fmtUL = origUL ;
               }
               //* Unknown secondary class or unexpected syntax. *
               //* Output the tag as written.                    *
               else
               {
                  fmtUL = origUL ;
               }
               break ;

            //* Legacy list syntax *
            default:
               fmtUL = origUL ;
               break ;
         }
      }

      //* If chapter-menu list, pass it through unmodified. *
      //* NOTE: There is no user prompt for chapter menus.  *
      else if ( (origUL.find( itemMINITOC )) >= ZERO )
      {
         fmtUL = origUL ;
         prompt_user = false ;

         #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
         { this->textOut ( L"---mini-toc-------" ) ; }
         #endif   // DEBUG_LISTS && DEBUG_ILIST
      }

      //* Scan for previously-processed tag (custom classes).  *
      //* These are passed through unmodified UNLESS there is  *
      //* an active response file. See note above.             *
      else if ( ((origUL.find( ul_sDISC,    ZERO, true, 16)) == ZERO) ||
                ((origUL.find( ul_sCIRCLE,  ZERO, true, 18)) == ZERO) ||
                ((origUL.find( ul_sSQUARE,  ZERO, true, 18)) == ZERO) ||
                ((origUL.find( ul_sPOINTER, ZERO, true, 19)) == ZERO) ||
                ((origUL.find( ul_sDIAMOND, ZERO, true, 19)) == ZERO) ||
                ((origUL.find( ul_NOBULLET, ZERO, true, 22)) == ZERO) ||
                ((origUL.find( ul_PLAIN,    ZERO, true,  4)) == ZERO) )
      {
         fmtUL = origUL ;
         if ( respFile ) { prompt_user = ignoreResp = true ; }

         #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
         { 
            gsdbg.compose( "---previously-processed at:%hd", &this->slCount ) ;
            this->textOut ( gsdbg ) ;
         }
         #endif   // DEBUG_LISTS && DEBUG_ILIST
      }

      //* Scan for makeinfo v:6 legacy syntax and replace with current syntax. *
      else if ( ((origUL.find( L"bullet\">" ))   > ZERO) ||
                ((origUL.find( L"section-toc" )) > ZERO) )
      {
         if ( (origUL.find( L"<ul class=\"disc-bullet\">" )) >= ZERO )
            fmtUL = ul_sDISC ;
         else if ( (origUL.find( L"<ul class=\"circle-bullet\">" )) >= ZERO )
            fmtUL = ul_sCIRCLE ;
         else if ( (origUL.find( L"<ul class=\"square-bullet\">" )) >= ZERO )
            fmtUL = ul_sSQUARE ;
         else if ( ((origUL.find( L"<ul class=\"no-bullet\">" )) >= ZERO) ||
                   ((origUL.find( L"<ul class=\"section-toc\">" )) >= ZERO) )
         { fmtUL.compose( "%S mark-none\">", itemBASE ) ; }
         else                 // unexpected syntax, let browser handle it
            fmtUL = ul_PLAIN ;
         //* Note that v:6 chapter menus declare the "section-toc"   *
         //* class which is a no-bullet class.                       *
         //* Note that v:6 uses a plain "<ul>" for disc bullets.     *

         #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
         { 
            gsdbg.compose( "---v:6 at:%04hd '%S'", &this->slCount, origUL.gstr() ) ;
            this->textOut ( gsdbg ) ;
         }
         #endif   // DEBUG_LISTS && DEBUG_ILIST
      }
      else        // unhandled <ul> construct
      {
         fmtUL = origUL ;

         #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
         { 
            gsdbg.compose( "---Not Handled at %04hd :'%S'", 
                           &this->slCount, origUL.gstr() ) ;
            this->textOut ( gsdbg ) ;
         }
         #endif   // DEBUG_LISTS && DEBUG_ILIST
      }
   }

   if ( this->verbose )       // verbose diagnostics
   {
      short bLine = this->tlCount + 1 ;   // output line number
      gsVerb.compose( L"(%4hu) '%S' ==>> ", &bLine, fmtUL.gstr() ) ;
   }


   //******************************************
   //* If user interaction, format the prompt *
   //******************************************
   if ( prompt_user )
   {
      usrResp = L'a' ;     // initialize user response to "automatic"

      //* Read the context line (probably the first item line) *
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         this->ppfUnReadSrcLine ( gs ) ;  // push the context line back into the stream
         dispLI = gs ;
         stripTags ( dispLI ) ;

         gst.compose( L"___________________________________\n"
                       "Bullet List: (source line:%hu)\n"
                       "First Line Item: %S\n"
                       "Auto-definition: %S",
                      &this->slCount, dispLI.gstr(), fmtUL.gstr() ) ;
         this->textOut ( gst ) ;
         this->textOut ( 
            L"Choose Bullet-list type:\n"
             "  with (s)=smaller text, or (l)=larger text\n"
             "d:disc   (\x23FA)     a:automatic (default)\n"
             "c:circle (\x26AC)     A:All automatic\n"
             "s:square (\x25AA)     x:no modification\n"
             "n:no bullet\n"
             "response format: BULLET_TYPE[FONT_SIZE]" ) ;
         this->textOut ( L"your choice: ", false ) ;

         while ( ! this->abort )    // input loop
         {
            this->userResponse ( gsIn ) ;    // talk with the animals

            //* If abort received. Note the early return. *
            if ( this->abort )
            { return ( status = ERR ) ; }

            //* If user specified a default response *
            if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
               usrResp = L'a' ;
            else        // single-character response expected
            {
               usrResp = *gsIn.gstr() ;
               //* If secondary formatting option(s) specified *
               if ( gsIn.gschars() > 2 )
               {  short ci = 1 ;
                  if ( (gsIn.gstr()[ci] == sFONT) ||
                       (gsIn.gstr()[ci] == lFONT) )
                  { fontSize = gsIn.gstr()[ci] ; }
               }
            }

            //* If this list, and all remaining UL lists *
            //* are to be processed automagically        *
            if ( usrResp == L'A' )
            { usrResp = L'a' ; this->ulLists = cfgAuto ; }

            //* Test for a valid response *
            if ( (usrResp == L'a') || (usrResp == L'd') ||
                 (usrResp == L'c') || (usrResp == L's') ||
                 (usrResp == L'n') || (usrResp == L'x') )
               break ;  // good input

            //* Call user a dumbguy and ask for correct response.*
            this->invalidResponse () ;
         }     // while()

         //* If source was previously processed, it will  *
         //* contain one of the custom class designators. *
         if ( ignoreResp )
            usrResp = L'x' ;
      }  // read context line

      switch ( usrResp )
      {
         //* Bullet size is taken from the embedded token (if any); *
         //* otherwise defaults to 'medium' for all bullet types.   *
         case L'a':     // automatic selection (fmtUL was initialized above)
            break ;
         case L'd':     // disc bullet
            if ( bullSize == sBULL )     { fmtUL = ul_sDISC ; }   // small
            else if ( bullSize == lBULL) { fmtUL = ul_lDISC ; }   // large
            else                         { fmtUL = ul_mDISC ; }   // inherited
            break ;
         case L'c':     // circle bullet
            if ( bullSize == sBULL )     { fmtUL = ul_sCIRCLE ; } // small
            else if ( bullSize == lBULL) { fmtUL = ul_lCIRCLE ; } // large
            else                         { fmtUL = ul_mCIRCLE ; } // inherited
            break ;
         case L's':     // square bullet
            if ( bullSize == sBULL )     { fmtUL = ul_sSQUARE ; } // small
            else if ( bullSize == lBULL) { fmtUL = ul_lSQUARE ; } // large
            else                         { fmtUL = ul_mSQUARE ; } // inherited
            break ;
         case L'n':     // no bullet (this is simpler than the makeinfo tag)
            fmtUL = ul_NOBULLET ;
            break ;
         case L'x':        // write source to target without modification
            fmtUL = origUL ;
            break ;
         default: // Note: This will never happen, but it silences the compiler warning.
            break ;
      }
   }     // user prompt
   //* If skip count is active, decrement the counter *
   else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

   //* If non-default relative font size, insert secondary class. *
   if ( (fontSize == sFONT) || (fontSize == lFONT) )
   {
      //* Locate the insertion point for secondary class declaration.*
      ti = fmtUL.find( L"\">" ) ;
      //* Insert the appropriate secondary class name.               *
      fmtUL.insert( (fontSize == sFONT ? fontSMALL : fontLARGE), ti ) ;
   }

   //* Integrate class modifications into output buffer *
   //* and write the tag to target file.                *
   gsul.append( fmtUL.gstr() ) ;
   this->ppfWriteLine ( gsul ) ;

   #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
   gsdbg.compose( "---------------->>'%S' usrResp:%C(%C%C)\n", 
                  gsul.gstr(), &usrResp, &bullSize, &fontSize ) ;
   this->textOut ( gsdbg ) ;
   #endif   // DEBUG_LISTS && DEBUG_ILIST

   //***************************************************
   //* Copy remainder of list including embedded lists *
   //* and other objects.                              *
   //***************************************************
   while ( status == OK && ! this->abort )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* If at the end of the list, return to caller *
         if ( (gs.find( ulEnd, wi, false )) >= wi )
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
            if ( this->verbose )          // verbose diagnostics
            {
               gsVerb.append( " (%hd items)", &lineItems ) ;
               this->textOut ( gsVerb ) ;
            }
            break ;
         }     // (end-of-list)

         //* If verbose diagnostics, count line items *
         if ( verbose )
         { if ( (gs.find( liBegin )) >= ZERO  ) { ++lineItems ; } }

         //* If an Itemized List (<ul> tag) nested within current list.*
         //* Note that this may result in a recursive call.            *
         else if ( (this->ppfItemizedList ( gs, status, wi )) != false )
         { /* do nothing */ }

         //* If an Enumerated List (<ol> tag) nested within current list.*
         else if ( (this->ppfEnumeratedList ( gs, status, wi )) != false )
         { /* do nothing */ }

         //* If a preformatted block is detected:              *
         //* 'display, 'format', 'example', 'lisp', and their  *
         //* 'small' or 'large' counterparts.                  *
         else if ( (this->ppfFormattedBlock ( gs, status )) != false )
         { /* do nothing */ }

         //* If a quotation, smallquotation or largequotation block *
         else if ( (this->ppfQuotationBlock ( gs, status )) != false )
         { /* nothing to do */ }

         //* If beginning a verbatim, smallverbatim or largeverbatim block *
         else if ( (this->ppfTestVerbatimBlock ( bType, gs )) != false )
         {
            status = this->ppfProcVerbatimBlock ( bType, gs ) ;
         }

         //* If an indentedblock, smallindentedblock or largeindentedblock *
         else if ( (this->ppfIndentedBlock ( gs, status )) != false )
         { /* do nothing */ }

         //* If a <table> object found *
         else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
         {
            status = this->ppfProcTABLE ( gs ) ;
         }

         //* Else the line does not contain a control statement *
         else
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
         }
      }        // (ReadLine())
   }           // while()

   return status ;

}  //* End ppfProcItemizedList() *

//**************************
//*     ppfUListToken      *
//**************************
//********************************************************************************
//* Scan for an embedded token within the first line item of an itemized list.   *
//*                                                                              *
//*                                                                              *
//* 1) The token, if present, contains one formatting instruction.               *
//*    a) Specify if the font size should be inherited from the parent (default),*
//*       smaller, or larger than the parent container.                          *
//*                                                                              *
//* 2) No data are written to the output stream by this method.                  *
//*    If adjustments are made to the data, that data is pushed back into the    *
//*    input stream for caller to retrieve and copy to output.                   *
//*    a) A valid token will be decoded and then removed from the input stream.  *
//*    b) A malformed token will be pushed back into stream unmodified.          *
//*       This gives the user a signal that the token syntax is incorrect.       *
//*                                                                              *
//* Input  : uSize  : (by reference) relative font size                          *
//*                   'i' == inherited  's' == smaller  'l' == larger            *
//* Returns: 'true'  if token identified and removed                             *
//*                  eSize is initialized with specified VALID option.           *
//*                  If invalid option, the default value 'i' is returned.       *
//*          'false' if token not found                                          *
//*                  caller's parameters are unchanged                           *
//********************************************************************************
//* Programmer's Note: The Texinfo macro "@embedTOKEN" causes the text following *
//* the formatting token to be pushed to the next line. Therefore, the token     *
//* will be removed and the two lines will be concatenated to restore the        *
//* original text of the line item.                                              *
//*                                                                              *
//* It is recommended that the "@embedTOKEN" macro be used to specify the token. *
//* The argument to @embedTOKEN takes the form "=x=" where:                      *
//*  -- first position ('x') specifies the relative font size                    *
//*                                                                              *
//* List Invocation:                                                             *
//* ----------------                                                             *
//*  @itemize @bullet                                                            *
//*  @item @embedTOKEN{x}blah blah blah                                          *
//*  . . .                                                                       *
//*  @end itemize                                                                *
//*                                                                              *
//* Some examples are:                                                           *
//* ------------------                                                           *
//*    @embedTOKEN{s}   where 's' indicates that a 's'maller font size be used   *
//*    @embedTOKEN{l}   where 'l' indicates that a 'l'arger font size be used    *
//*                                                                              *
//********************************************************************************

bool Idpp::ppfUListToken ( wchar_t& uSize )
{
   const wchar_t* liTAG = L"<li>" ;       // begin line item
//   const wchar_t* embedTOKEN = L"<li> =" ;// comparison text (begin line item)
   const wchar_t  TOKEN_DELIM = L'=' ;    // token delimiter
   const wchar_t sizeDFLT = L'i' ;        // default font size
   const short sizeIndex = 1 ;            // offset into 'wbuff' for font-size character
   const short BUFFSIZE = 5 ;             // size of wbuff
   gString gsA, gsB ;                     // text analysis
   wchar_t wbuff[BUFFSIZE] ;              // token buffer
   short wi, ti ;                         // character index
   bool goodToken    = false,             // valid token syntax
        tokenRemoved = false ;            // return value

   //* Read a line from the source file. *
   if ( (this->ppfReadSrcLine ( gsA, wi )) == OK )
   {
      //* If the formatting token is present *
      if ( ((ti = gsA.after( liTAG )) > ZERO) &&
           ((ti = gsA.find( TOKEN_DELIM, false, ti )) > ZERO) )
      {
         //* Isolate the token for analysis (three characters).*
         gsA.substr( wbuff, ti, 3 ) ;
         if ( wbuff[2] == TOKEN_DELIM )
            goodToken = true ;

         #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
         gsdbg.compose( "---etoken---------'%S' '%S' ", gsA.gstr(), wbuff ) ;
         #endif   // DEBUG_LISTS && DEBUG_ILIST

         //* If a valid token isolated, initialize caller's arg  *
         //* and read the next line from source.                 *
         if ( goodToken )
         {
            uSize = sizeDFLT ;                  // initialize to default value
            this->ppfReadSrcLine ( gsB, wi ) ;  // read the next line
            gsA.replace( wbuff, gsB.gstr() ) ;  // replace the token with contents of 2nd line
            this->ppfUnReadSrcLine ( gsA ) ;    // push the line back into the stream
            tokenRemoved = true ;

            //** If relative font size specified. **
            if ( (wbuff[sizeIndex] == L's') || (wbuff[sizeIndex] == L'l') )
               uSize = wbuff[sizeIndex] ;

            #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
            gsdbg.append( "+ '%S' = '%S'\n"
                          "------------------uSize:%C", 
                          gsB.gstr(), gsA.gstr(), &uSize ) ;
            #endif   // DEBUG_LISTS && DEBUG_ILIST
         }
         //* Otherwise the token is malformed.             *
         //* Push the unmodified line back into the stream.*
         else
            this->ppfUnReadSrcLine ( gsA ) ;

         #if DEBUG_LISTS != 0 && DEBUG_ILIST != 0
         this->textOut ( gsdbg ) ;
         #endif   // DEBUG_LISTS && DEBUG_ILIST
      }
      else     // token not found, push the unmodified line back into the stream
         this->ppfUnReadSrcLine ( gsA ) ;
   }

   return tokenRemoved ;

   #undef DEBUG_ILIST   // this is defined in the calling method
}  //* End ppfUListToken() *

//**************************
//*   ppfEnumeratedList    *
//**************************
//********************************************************************************
//* Test whether the provided data indicates the beginning of an enumerated      *
//* (<ol>) list. or the 'small' / 'large' counterparts.                          *
//*                                                                              *
//* If the target block was found, process it and write it to the target file.   *
//*                                                                              *
//* Input  : gsln   : line of source data to be scanned                          *
//*          status ; (by reference) receives 'ERR' if syntax or other error,    *
//*                   else receives 'OK'                                         *
//*          wi     : index into the wide (wchar_t) source data                  *
//*          bookit : (optional, 'false' by default                              *
//*                   if 'true', AND if 'book' member set, then write the        *
//*                      bookmark messages to the display                        *
//*                   if 'false', do not write bookmark messages                 *
//*                                                                              *
//* Returns: 'true' if a preformatted block was processed                        *
//*          'false' caller is responsible for processing the specified data     *
//********************************************************************************

bool Idpp::ppfEnumeratedList ( const gString& gsln, short& status, short wi, bool bookit )
{
   gString gsBook ;           // text formatting
   bool isBlock = false ;     // return value

   status = OK ;              // Initialize caller's status value (hope for success)

   if ( (this->ppfTestEnumeratedList ( gsln, wi )) != false )
   {
      isBlock = true ;        // formatted block identified

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"ML(%hu) OBLOCK START", &this->slCount ) ;
         this->textOut ( gsBook, false ) ;
      }

      gString gs = gsln ;
      status = this->ppfProcEnumeratedList ( gs ) ;

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"--(%hu) OBLOCK END", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }
   }
   return isBlock ;

}  //* End ppfEnumeratedList() *

//*************************
//* ppfTestEnumeratedList *
//*************************
//********************************************************************************
//* Test the provided text for the beginning of an UNPROCESSED enumerated        *
//* list (<ol> block). See notes below.                                          *
//*                                                                              *
//* Input  : gssrc : (by reference) contains the source data line being          *
//*                  processed                                                   *
//*          wi    : offset into 'gssrc' at which to begin search                *
//*                                                                              *
//* Returns: 'true'  if line contains a plain (unprocessed) <ol> tag             *
//*          'false' otherwise                                                   *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*  1) Normally, an <ol> tag is alone on a line, but occasionally it is         *
//*     preceeded by a </p> (close paragraph) tag or a </pre> (close             *
//*     preformatted). This happens if the <ol> is immediately preceeded by      *
//*     one of the block commands.                                               *
//* 2a) For makeinfo 5.x, all <ol> tags are declared without a class or style    *
//*     parameters, so if it has a class, then it has already been processed.    *
//* 2b) For makeinfo 6.x, some <ol> tags are plain and others specify a type     *
//*     and start value. "<ol>"  vs.  "<ol type="A" start="1"> or                *
//*     "<ol start="0">                                                          *
//*  3) Also, unprocessed <ol> tags are the last thing on the source line, so    *
//*     if there is additional data AFTER the <ol> tag, then the list has        *
//*     already been processed.                                                  *
//*  4) We COULD BE fooled by HTML embedded directly into the texi source, but   *
//*     if we are fooled, processing the list will not change its appearance     *
//*     unless user selects a non-default formatting option.                     *
//********************************************************************************

bool Idpp::ppfTestEnumeratedList ( const gString& gssrc, short wi )
{
   bool olListFound = false ;

   //* If source line contains '<ol>' tag AND it is last item on source line *
   if ( (wi = gssrc.find( olBegin, wi, false, olBegin_len )) >= ZERO )
   {
      wi = gssrc.after( tagClose, wi ) ;
      if ( (wi > ZERO) && (gssrc.gstr()[wi] == NULLCHAR) )
         olListFound = true ;
   }
   return olListFound ;

}  //* End ppfTestEnumeratedList() *

//*************************
//* ppfProcEnumeratedList *
//*************************
//********************************************************************************
//* Process enumerated (<ol>) lists. Also called recursively.                    *
//*                                                                              *
//* Caller has determined that the OL tag is the last item on the source line.   *
//*                                                                              *
//*                                                                              *
//* Input  : gsol  : (by reference) contains the first line of the list:         *
//*                  '<ol>' || '</pre><ol>' || '</p><ol>' ||                     *
//*                  '<ol type="a" start="1">' or similar                        *
//*          type  : enumeration type ('1' 'a' 'A' etc)                          *
//*          start : value for first item of list ('1-999' 'a-z' 'A-Z' etc)      *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Programmer's Note:                                                           *
//* 1) Because lists may be embedded within other lists, this method may be      *
//*    called recursively, so be sure all data are safely on the stack.          *
//*                                                                              *
//* 2) Ordered list elements, in addition to the global attributes, may          *
//*    include three(3) style attributes:                                        *
//*     a) "reversed"  boolean to indicate descending sequence                   *
//*     b) "type"      enumeration type: '1', 'a', 'A', 'i', 'I'.                *
//*                    ideally, the type should be handled by the CSS            *
//*                    "list-style-type" which has many more options.            *
//*                    OL sub-classes are defined for the most common CSS        *
//*                    enumeration types.                                        *
//*     c) "start"     ordinal offset from beginning of list (one-based except   *
//*                    for decimal "type" which also accepts '0')                *
//*                                                                              *
//* 3) texi2any takes an entirely-reasonable shortcut in determining the         *
//*    "type" and "start" parameters based on the .texi source:                  *
//*    => Any single-character lower-case alpha will yield the same type:        *
//*       "@enumerate a", "@enumerate c", "@enumerate e" all yield: type="a"     *
//*    => However, these will yield: start="1", start="3" and start="5"          *
//*       respectively.                                                          *
//*    Similarly for upper-case alpha and for unsigned integer values.           *
//*       (Note that multi-character alpha e.g. "@enumerate cc" )                *
//*       (will be handled as an error and defaulted to "<ol>". )                *
//*       (Multi-digit ordinals will be encoded as expected:    )                *
//*       ( "@enumerate 25" yields <ol start="25">              )                *
//*                                                                              *
//*    While this is reasonable, it means that Roman-numeral lists cannot be     *
//*    specified directly because "@enumerate i" will yield type="a" start="9".  *
//*    In our opinion, the need for non-alpha enumerated list is significantly   *
//*    greater than the need to start lists at offsets > 1.                      *
//*    Therefore, we make a special case of certain type/start combinations.     *
//*    See below for details.                                                    *
//*    This automatic formatting may be avoided by not processing OL lists,      *
//*    or by interactively specifying the enumeration type and start value.      *
//*                                                                              *
//* 4) Two switches control processing of <ol> lists:                            *
//*    - this->olLists  : 'cfgAuto' process of OL lists automatically            *
//*                       'cfgSpec' ask user to specify list type                *
//*                       'cfgNone' do not process OL lists                      *
//*    - this->liSpec   : 'true'  perform special-case interpretation for        *
//*                               certain combinations of 'type and 'start'      *
//*                               parameters (see notes above)                   *
//*                       'false' interpret list style information literally     *
//*                                                                              *
//* 5) Note that HTML will accept alphabetical offsets of any reasonable size    *
//*    and display the sequence as: ... y, z, aa, ab, ac, ...                    *
//*                                                                              *
//* 6) GPL and FDL license text:                                                 *
//*    Please see ppfProgGNU() method for a discussion of processing these       *
//*    special enumerated lists.                                                 *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcEnumeratedList ( gString& gsol )
{
   #define DEBUG_OLIST (0)       // for debugging only

   //* List of supported OL type sub-classes *
   const wchar_t* OL_ClassList[etTYPES] = 
   {
      L"enum-decimal",           //  0 decimal ordinals
      L"enum-decimal-zero",      //  1 decimal with leading zero
      L"enum-lower-alpha",       //  2 lower-case English alpha
      L"enum-upper-alpha",       //  3 upper-case English alpha
      L"enum-lower-roman",       //  4 lower case Roman numerals
      L"enum-upper-roman",       //  5 upper-case Roman numerals
      L"enum-lower-greek",       //  6 lower-case Greek alpha
      L"enum-upper-greek",       //  7 upper-case Greek alpha
      L"enum-cjk-decimal",       //  8 CJK (Han) decimal numeric (informal)
      L"enum-katakana",          //  9 Katakana alpha (Gojūon)
      L"enum-hebrew",            // 10 Hebrew alpha
      L"enum-arabic-indic",      // 11 Arabic-Indic numerals
      L"enum-custom",            // 12 Custom support for sequence DEPRECATED
                                 // (CSS class must be modified to specify the sequence)
   } ;
   const wchar_t* olTemplate   = L"<ol class=\"%S\" start=\"%hd\">" ;
   const wchar_t* olTemplate_f = L"<ol class=\"%S %S\" start=\"%hd\">" ;
   const wchar_t* olstyleType  = L"type=\"" ;
   const wchar_t* olstyleStart = L"start=\"" ;
   const wchar_t* classSpec    = L"class=\"enum-" ;
   const wchar_t* olstypeLST   = L"list-style-type:" ;
   const wchar_t* revDIR       = L" reversed" ;
   const wchar_t* fontSMALL    = L"enum-small" ;
   const wchar_t* fontLARGE    = L"enum-large" ;
   const wchar_t chDFLT        = L'.' ;   // placeholder for default character value

   gString gs, gst,           // text formatting
           origOL,            // caller's original OL tag
           fmtOL,             // formatted OL tag
           dispLI,            // stripped line item for display to user
           gsVerb,            // for 'verbose' output
           gsIn ;             // receives user input
   wchar_t eType = L'1',      // enumeration type (default == decimal)
           usrResp = L'a',    // user response to prompt
           fontSize = L'i',   // relative font size
           eDir = 'a',        // enumeration list direction (default == ascending)
           tmpET,             // temporary character values
           tmpED,
           tmpFS ;
   short   tmpETI, tmpES,     // temporary numeric values
           etIndx = 0,        // index into OL_ClassList[] (class name written to target)
           eStart = 1,        // enumeration start value (default == 1)
           ol,                // index of <ol> tag
           wi,                // search index
           status = OK ;      // return value
   bool prompt_user = this->olLists == cfgSpec,  // local copy of prompt flag
        respFile    = this->rifs.is_open(),     // 'true' if open response file
        ignoreResp  = false,  // 'true' if forced user response on Already Processed list
        pprocClass  = false,  // 'true' if previously-processed class detected
        specCase    = false ; // 'true' if "special-case" configurations handled

   if ( this->verbose )       // verbose diagnostics
   {
      //* Index the beginning of the <ol ...> tag *
      if ( (ol = gsol.find( olBegin )) < ZERO )
         ol = ZERO ;
      short bLine = this->tlCount + 1 ;
      gsVerb.compose( L"(%4hu) '%S' ==>> ", &bLine, &gsol.gstr()[ol] ) ;
   }

   //* Index the beginning of the OL tag, and    *
   //* test for class name or style information. *
   if ( (ol = gsol.find( olBegin )) >= ZERO )   // (should always be true, but be safe)
   {
      //* Save a copy of the original OL tag.                  *
      //* Note: We assume two things here:                     *
      //*  1. The complete tag is present in the source line.  *
      //*  2. The tag is the last thing in the source line.    *
      //*     This means that the output can be updated by     *
      //*     appending the modified tag.                      *
      copyTag ( gsol, ol, origOL ) ;   // save a copy of the original UL tag
      //* Delete the original tag from output data *
      removeTag ( gsol, ol ) ;


      //* If not a plain <ol> tag, extract class/style information.     *
      //* Note that if <ol class="enumerate"> i.e. no information,      *
      //* then the default is used: eType==1, eStart==1                 *
      //* Other makeinfo types expected would be 'a' and 'A'.           *
      //* Note that makeinfo v:7 no longer uses plain <ol> tags.        *
      if ( origOL.gstr()[olBegin_len] != L'>' )
      {
         if ( (wi = origOL.after( olstyleType )) >= ZERO )    // "type" attribute
            eType = origOL.gstr()[wi] ;

         if ( (wi = origOL.after( olstyleStart )) >= ZERO )   // "start" attribute
         {
            if ( (origOL.gscanf( wi, L"%hd", &eStart )) == 1 )
            {
               //* If out-of-range, use default start value *
               if ( (eStart < ZERO) || ((eStart == ZERO) && (eType != L'1')) )
                  eStart = 1 ;
            }
            else     // if unable to scan value (be safe)
               eStart = 1 ;
         }

         //* If "reversed" (decrement) specified, remember it. *
         if ( (origOL.find( revDIR )) > ZERO )
         {
            eDir = L'd' ;
         }

         //* If in-line style "list-style-type", extract its argument *
         //* and compare it with our list of supported style classes. *
         if ( (wi = origOL.after( olstypeLST )) >= ZERO )
         {  //* Isolate the "list-style-type" argument *
            wchar_t wbuff[32] ;
            short   w = ZERO ;
            while ( (isalnum(gsol.gstr()[wi]) || (origOL.gstr()[wi] == L'-')) && (w < 31) )
               wbuff[w++] = origOL.gstr()[wi++] ;
            wbuff[w] = NULLCHAR ;   // terminate the string
            //* If isolated type name is a substring *
            //* in class name, then we have a match. *
            // Programmer's Note: If the texi-to-HTML was fully implemented, 
            // this would be enough; however, see special cases below.
            for ( short indx = ZERO ; indx < etTYPES ; ++indx )
            {
               gst = OL_ClassList[indx] ;
               if ( (gst.find( wbuff )) >= ZERO )
               {
                  etIndx = indx ;         // index the matching class name
                  eType = enumChar[etIndx] ;
                  break ;
               }
            }
         }

         //* If source specifies a class written on a previous pass, *
         //* set 'eType' and 'etIndx' to match the specified class.  *
         //* The makeinfo-generated "enumerate" class and any        *
         //* unknown class will be ignored.                          *
         if ( (wi = origOL.after( classSpec )) >= ZERO )
         {
            for ( short indx = ZERO ; indx < etTYPES ; ++indx )
            {
               if ( (origOL.find( OL_ClassList[indx] )) >= ZERO )
               {
                  etIndx = indx ;         // index the matching class name
                  eType = enumChar[etIndx] ;
                  pprocClass = true ;
                  break ;
               }
            }
         }

         //* Special Case Processing: Override certain texi2any assignments *
         //* which encode enumeration types not supported by makeinfo.      *
         if ( this->liSpec )
         {
            specCase = true ;             // provisionally handled
            if ( (eType == L'A') && (eStart == 4) )       // 'D' decimal with leading '0's
            { eType = enumChar[etIndx = etDecimalLZ] ;  eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'a') && (eStart == 9) )  // 'i' lower-case Roman numerals
            { eType = enumChar[etIndx = etLowerRoman] ; eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'A') && (eStart == 9) )  // 'I' upper-case Roman numerals
            { eType = enumChar[etIndx = etUpperRoman] ; eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'a') && (eStart == 7) )  // 'g' lower-case Greek alpha
            { eType = enumChar[etIndx = etLowerGreek] ; eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'A') && (eStart == 7) )  // 'G' upper-case Greek alpha
            { eType = enumChar[etIndx = etUpperGreek] ; eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'a') && (eStart == 10) ) // 'j' CJK numeric (informal)
            { eType = enumChar[etIndx = etCJK] ;        eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'a') && (eStart == 11) ) // 'k' Katakana alpha
            { eType = enumChar[etIndx = etKatakana] ;   eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'a') && (eStart == 8) )  // 'h' Hebrew alpha
            { eType = enumChar[etIndx = etHebrew] ;     eStart = 1 ; pprocClass = false ; }
            else if ( (eType == L'a') && (eStart == 5) )  // 'e' Eastern (Arabic-Indic)
            { eType = enumChar[etIndx = etArabic] ;     eStart = 1 ; pprocClass = false ; }
            else { specCase = false ; }
         }
      }
      else 
      { /* Special processing disabled, 'etIndx' previously initialized. */ }

      //* Non-special combinations. These are makeinfo standard type/start *
      //* combinations not handled by "special processing", above.         *
      if ( ! specCase )
      {
         if ( eType == L'1' )
         { eType = enumChar[etIndx = etDecimal] ; }
         else if ( eType == L'a' )
         { eType = enumChar[etIndx = etLowerAlpha] ; }
         else if ( eType == L'A' )
         { eType = enumChar[etIndx = etUpperAlpha] ; }
      }

      //* Scan the first line item for an embedded token *
      //* signalling additional formatting options.      *
      //* If token is present, it will be removed.       *
      if ( (this->ppfOListToken ( tmpET, tmpES, tmpFS, tmpED, tmpETI )) )
      {
         if ( tmpET != L'.' )                // enumeration type
         {
            eType = tmpET ;
            etIndx = tmpETI ;
         }
         if ( (tmpES >= ZERO) && (tmpES <= 99) )   // starting value
         {
            //* A zero value is invalid for non-decimal enumerators *
            if ( (tmpES == ZERO) && !((eType == L'd') || (eType == L'd')) )
               tmpES = 1 ;
            eStart = tmpES ;
         }
         if ( (tmpFS == L's') || (tmpFS == L'l') ) // embedded font size
            fontSize = tmpFS ;
         if ( tmpED == L'd' )                // embedded sequence direction
            eDir = tmpED ;
      }

      //* Format the OL tag based on the information gathered.*
      if ( (fontSize == L's') || (fontSize == L'l') )
      {
         fmtOL.compose( olTemplate_f, OL_ClassList[etIndx],
                        ((fontSize == L's') ? fontSMALL : fontLARGE), &eStart ) ;
      }
      else
         fmtOL.compose( olTemplate, OL_ClassList[etIndx], &eStart ) ;
      //* If reversed order specified, insert the token *
      if ( eDir == L'd' )
      { wi = fmtOL.findlast( tagClose ) ; fmtOL.insert( revDIR, wi ) ; }

      #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0
      gsdbg.compose( "OLIST at:%04hd '%S' eType:%C eStart:%hd\n"
                     "--------------'%S'", 
                     &this->slCount, origOL.gstr(), &eType, &eStart, fmtOL.gstr() ) ;
      this->textOut ( gsdbg ) ;
      #endif   // DEBUG_LISTS && DEBUG_OLIST
   }     // decode source <ol> tag

   //* If list is not to be processed, ignore the response *
   if ( this->olLists == cfgNone )
   { usrResp = L'x' ; ignoreResp = true ; }
      
   //* If a previously-processed list AND if an active response *
   //* file, force the user prompt, but ignore the response.    *
   if ( pprocClass && respFile )
   { prompt_user = ignoreResp = true ;  }

   //******************************************
   //* If user interaction, format the prompt *
   //******************************************
   if ( prompt_user )
   {
      usrResp = L'a' ;     // initialize user response to "automatic"

      //* Read the context line (probably the first item line) *
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         this->ppfUnReadSrcLine ( gs ) ;  // push the context line back into the stream
         dispLI = gs ;                    // strip out the HTML tags
         stripTags ( dispLI ) ;

         gst.compose( L"___________________________________\n"
                       "Enumerated List: (source line:%hu)\n"
                       "First Line Item: %S\n"
                       "Auto-definition: %S",
                      &this->slCount, dispLI.gstr(), fmtOL.gstr() ) ;
         this->textOut ( gst ) ;
         this->textOut ( 
            L"Choose Enumeration type:\n"
             "d:decimal            D:leading-zero decimal  j:CJK(informal)\n"
             "l:lower case alpha   u:upper case alpha      k:Katakana\n"
             "i:lower case Roman   I:upper case Roman      h:Hebrew\n"
             "g:lower case Greek   G:upper case Greek      e:Arabic-Indic\n"
             "a:automatic(default) A:All automatic         x:no modification\n"
             "response format: TYPE[START_VAL[FONT_SIZE[DIR]]]") ;
         this->textOut ( L"your choice: ", false ) ;
         while ( ! this->abort )
         {
            this->userResponse ( gsIn ) ;    // talk with the animals

            if ( this->abort )      // abort signal received (note the early return)
            { return ( status = ERR ) ; }

            //* Parse user's response *
            if ( !(this->ppfOListResponse ( gsIn, tmpET, tmpES, tmpFS, tmpED, tmpETI )) )
            {
               this->invalidResponse () ;
               continue ;
            }

            #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0
            gsdbg.compose( "---oresp------%C %02hd %C %C --class%02hd:'%S'", 
                           &tmpET, &tmpES, &tmpFS, &tmpED, &tmpETI,
                           ((tmpETI < etTYPES) ? OL_ClassList[tmpETI] : L"unk") ) ;
            this->textOut ( gsdbg ) ;
            #endif   // DEBUG_LISTS && DEBUG_OLIST

            //* During a second pass, the existing class *
            //* designation overrides user response.     *
            if ( ignoreResp )
               usrResp = L'x' ;

            //* If automatic processing for this list, *
            //* ignore remaining user parameters.      *
            else if ( tmpET == L'a' )
               usrResp = tmpET ;

            //* Else, valid (non-automatic) configuration *
            //* AND if user response is different from    *
            //* existing configuration parameters.        *
            else
            {
               bool ovrride = false ;     // set if user input overrides calculated parameters
               if ( tmpET != chDFLT )
               {
                  usrResp = eType = tmpET ; // enumeration type
                  etIndx   = tmpETI ;     // enumeration class index
                  ovrride    = true ;
               }
               if ( tmpES >= ZERO )
               {
                  eStart   = tmpES ;      // initial sequence value
                  ovrride    = true ;
               }
               if ( tmpFS != chDFLT )
               {
                  fontSize = tmpFS ;      // relative font size
                  ovrride = true ;
               }
               if ( tmpED != chDFLT )
               {
                  eDir = tmpED ;
                  ovrride = true ;
               }
               //* If user override of previously-calculated *
               //* parameters, reconstruct the <ol> tag.     *
               if ( ovrride )
               {
                  //* Format the OL tag based on the information gathered.*
                  if ( (fontSize == L's') || (fontSize == L'l') )
                  {
                     fmtOL.compose( olTemplate_f, OL_ClassList[etIndx],
                                    ((fontSize == L's') ? fontSMALL : fontLARGE), 
                                    &eStart ) ;
                  }
                  else
                  {
                     fmtOL.compose( olTemplate, OL_ClassList[etIndx], &eStart ) ;
                  }
                  //* If reversed order specified, insert the token *
                  if ( eDir == L'd' )
                  {
                     wi = fmtOL.findlast( tagClose ) ;
                     fmtOL.insert( revDIR, wi ) ;
                  }
               }  // ovrride
            }     // else(reconfigure)

            break ;     // thank you, dear user for your 'prompt' response

         }  // while()
      }     // context line read
   }        // user prompt
   //* If skip count is active, decrement the counter *
   else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

   //* Append the modified OL tag to output buffer and write to target *
   gsol.append( (wchar_t*)(usrResp == L'x' ? (origOL.gstr()) : (fmtOL.gstr())) ) ;
   this->ppfWriteLine ( gsol ) ;

   if ( this->verbose )
      gsVerb.append( fmtOL.gstr() ) ;

   #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0
   gsdbg.compose( "------------>>'%S' usrResp:%C(%02hd,%C,%C)\n", 
                  gsol.gstr(), &usrResp, &eStart, &fontSize, &eDir ) ;
   this->textOut ( gsdbg ) ;
   #endif   // DEBUG_LISTS && DEBUG_OLIST


   bool done = false ;
   do
   {  //* Read lines from source file.*
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* If at the end of the list, return to caller *
         if ( (gs.find( olEnd, ZERO, false )) >= ZERO )
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
            if ( this->verbose )          // verbose diagnostics
               this->textOut ( gsVerb ) ;
            done = true ;
            break ;
         }

         //* If an Itemized List (<ul> tag) nested within current list.*
         else if ( (this->ppfItemizedList ( gs, status, wi )) != false )
         { /* do nothing */ }

         //* If an Enumerated List (<ol> tag) nested within current list.*
         //* Note that this may result in a recursive call.              *
         else if ( (this->ppfEnumeratedList ( gs, status, wi )) != false )
         { /* do nothing */ }

         //* If a preformatted block is detected:              *
         //* 'display, 'format', 'example', 'lisp', and their  *
         //* 'small' or 'large' counterparts.                  *
         else if ( (this->ppfFormattedBlock ( gs, status )) != false )
         { /* do nothing */ }

         //* If a quotation, smallquotation or largequotation block *
         else if ( (this->ppfQuotationBlock ( gs, status )) != false )
         { /* nothing to do */ }

         //* If a verbatim, smallverbatim or largeverbatim block *
         else if ( (this->ppfVerbatimBlock ( gs, status )) != false )
         { /* do nothing */ }

         //* If an indentedblock, smallindentedblock or largeindentedblock *
         else if ( (this->ppfIndentedBlock ( gs, status )) != false )
         { /* do nothing */ }

         //* If a <table> object found *
         else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
         {
            status = this->ppfProcTABLE ( gs ) ;
         }

         //* Process line as text data *
         else
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
         }
      }
      else  done = true ;     // read error, abort the loop
   }
   while ( ! done && status == OK && ! this->abort ) ;

   return status ;

}  //* End ppfProcEnumeratedList() *

//**************************
//*     ppfOListToken      *
//**************************
//********************************************************************************
//* Scan for an embedded token within the first line item of an enumerated list. *
//*                                                                              *
//* 1) The token, if present, may contain between one and three formatting       *
//*    instructions.                                                             *
//*    a) Specify whether the enumerated list should be displayed in ascending   *
//*       order (default), or in descending order.                               *
//*    b) Specify if the font size should be inherited from the parent (default),*
//*       smaller, or larger than the parent container.                          *
//*    c) Specify the language used for the enumerated items, Roman ('i' or 'I'),*
//*       Greek ('g' or 'G'), CJK ('j'), Katakana ('k'), Hebrew ('h') or         *
//*       Arabic-Indic ('e'). Also, Custom ('c').                                *
//*       NOTE: This option is currently not implemented. The language option    *
//*             is already encoded in the 'type' and 'start' values generated    *
//*             by makeinfo. Those values are decoded by caller.                 *
//*                                                                              *
//* 2) No data are written to the output stream by this method.                  *
//*    If adjustments are made to the data, that data is pushed back into the    *
//*    input stream for caller to retrieve and copy to output.                   *
//*    a) A valid token will be decoded and then removed from the input stream.  *
//*    b) A malformed token will be pushed back into stream unmodified.          *
//*       This gives the user a signal that the token syntax is incorrect.       *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  : eType  : (by reference) enumeration type                            *
//*                   valid types:                                               *
//*                     d : decimal          D : decimal with leading zero       *
//*                     l : lower-case alpha u : upper-case alpha                *
//*                     i : lower-case Roman I : upper-case Roman                *
//*                     g : lower-case Greek G : upper-case Greek                *
//*                     c : custom enum type  (DEPRECATED)                       *
//*                     j : CJK (informal)   k : Katakana                        *
//*                     h : Hebrew           e : Arabic-Indic                    *
//*          eStart : (by reference) enumeration start offset                    *
//*                   range: 1-99 (for 'd' and 'D' types: 0 (zero) is accepted)  *
//*          eSize  : (by reference) relative font size                          *
//*                   'i' == inherited  's' == smaller  'l' == larger            *
//*          eDir   : (by reference) sequence direction                          *
//*                   'a' == descending  'd' == descending                       *
//*          etIndx : (by reference) receives class index corresponding to       *
//*                   'eType'. This is actually a member of enum enumType.       *
//*                                                                              *
//* Returns: 'true'  if token identified and removed                             *
//*                  all parameters are initialized with specified VALID options,*
//*                  else they receive default values                            *
//*          'false' if token not found                                          *
//*                  caller's parameters are unchanged                           *
//********************************************************************************
// Programmer's Note: The Texinfo macro "@embedTOKEN" causes the text following  *
// the formatting token to be pushed to the next line. Therefore, the token      *
//* will be removed and the two lines will be concatenated to restore the        *
//* original text of the line item.                                              *
//*                                                                              *
//* It is recommended that the "@embedTOKEN" macro be used to specify the token. *
//* The arguments to @embedTOKEN takes the form "xyznn" where:                   *
//*  -- first position ('x') indicate ascending or decending sequence.           *
//*  -- second position ('y') (optional) specifies the relative font size        *
//*  -- third position ('z') (optional) specifies the enumeration type, decimal, *
//*     Latin, Greek etc. (see above)                                            *
//*  -- fourth and fifth position ('nn') (optional) a numeric value indicating   *
//*     the count at which to begin the list.                                    *
//*                                                                              *
//* List Invocation:                                                             *
//* ----------------                                                             *
//*  @enumerate 1                                                                *
//*  @item @embedTOKEN{xyznn}blah blah blah                                      *
//*  . . .                                                                       *
//*  @end enumerate                                                              *
//*  -- May also be invoked using macros: @SmallEnumerate or @LargeEnumerate.    *
//*     HTML syntax will be the same either way.                                 *
//*                                                                              *
//* Some examples are:                                                           *
//* ------------------                                                           *
//*    @embedTOKEN{a}   where 'a' indicates that the list s/b in ascending order *
//*                               (this is the default)                          *
//*                           'd' indicated that the list s/b in descending order*
//*    @embedTOKEN{ds}  where 'd' indicates 'd'escending order                   *
//*                     where 's' indicates that a 's'maller font size be used   *
//*    @embedTOKEN{.s}  where '-' is a placeholder indicating an ascending       *
//*                     sequence order and                                       *
//*                     where 's' indicates that a 's'maller font size be used   *
//*    @embedTOKEN{.l}  where '.' is a placeholder indicating an ascending       *
//*                     sequence order and                                       *
//*                     where 'l' indicates that a 'l'arger font size be used    *
//*                                                                              *
//********************************************************************************

bool Idpp::ppfOListToken ( wchar_t& eType, short& eStart, wchar_t& eSize, 
                           wchar_t& eDir, short& etIndx )
{
   const wchar_t* embedTOKEN = L"<li> =" ;// comparison text (begin line item)
   const wchar_t  TOKEN_DELIM = L'=' ;    // token delimiter
   //const wchar_t seqASCEND = L'a' ;       // ascending sequence order
   const wchar_t seqDESCEND = L'd' ;      // descending sequence order
   const wchar_t chDFLT   = L'.' ;        // placeholder for default character value
   const short indxDFLT = etTYPES ;       // unknown class-name index
   const short initDFLT = -1 ;            // unknown initial sequence value

   const short typeIndex = 1 ;            // offset into 'wbuff' for enumeration type
   const short initIndex = 2 ;            // offset into 'wbuff' for initial value
   const short sizeIndex = 4 ;            // offset into 'wbuff' for font-size character
   const short dirIndex  = 5 ;            // offset into 'wbuff' for direction character
   const short BUFFSIZE  = 8 ;            // size of wbuff
   gString gsA, gsB ;                     // text analysis
   wchar_t wbuff[BUFFSIZE] ;              // token buffer
   short wi ;                             // character index
   bool goodToken    = false,             // valid token syntax
        tokenRemoved = false ;            // return value

   //* Read a line from the source file. *
   if ( (this->ppfReadSrcLine ( gsA, wi )) == OK )
   {
      //* If the formatting token is present *
      short token_index ;
      if ( (token_index = (gsA.after( embedTOKEN )) - 1) >= ZERO )
      {
         //* Isolate the token for analysis (seven characters max) *
         gsA.substr( wbuff, token_index, (BUFFSIZE - 1) ) ;
         for ( short i = 1 ; i < BUFFSIZE ; ++i )
         {
            //* If end-of-token found, fill remainder of buffer with '\0' *
            if ( wbuff[i] == TOKEN_DELIM )
            {
               while ( i < BUFFSIZE ) { wbuff[++i] = NULLCHAR ; }
               goodToken = true ;
               break ;
            }
         }

         #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0
         gsdbg.compose( "---etoken-----'%S' '%S' ", gsA.gstr(), wbuff ) ;
         #endif   // DEBUG_LISTS && DEBUG_OLIST

         //* If a valid token isolated *
         if ( goodToken )
         {
            //* Initialize caller's parameters to default values *
            eDir = eSize = eType = chDFLT ;
            etIndx = indxDFLT ;  eStart = initDFLT ;

            gsA.erase( wbuff, token_index ) ;   // delete token from source line

            //* If the token was the last data on the line, then read the  *
            //* next line and append it to the orphaned <li> tag.          *
            if ( gsA.gstr()[token_index] == NULLCHAR )
            {
               this->ppfReadSrcLine ( gsB, wi ) ;  // read the next line
               gsA.append( &gsB.gstr()[wi] ) ;     // concatenate the lines
            }
            this->ppfUnReadSrcLine ( gsA ) ;    // push the line back into the stream
            tokenRemoved = true ;               // set return-value flag

            //*********************
            //** Parse the token **
            //*********************
            //** If enumeration type specified **
            if ( (wbuff[typeIndex] != TOKEN_DELIM) && (wbuff[typeIndex] != NULLCHAR) )
            {
               wchar_t tChar = wbuff[typeIndex] ;
               switch ( tChar )
               {
                  case L'd':  etIndx = etDecimal ;    eType = tChar ; break ;
                  case L'D':  etIndx = etDecimalLZ ;  eType = tChar ; break ;
                  case L'l':  etIndx = etLowerAlpha ; eType = tChar ; break ;
                  case L'u':  etIndx = etUpperAlpha ; eType = tChar ; break ;
                  case L'i':  etIndx = etLowerRoman ; eType = tChar ; break ;
                  case L'I':  etIndx = etUpperRoman ; eType = tChar ; break ;
                  case L'g':  etIndx = etLowerGreek ; eType = tChar ; break ;
                  case L'G':  etIndx = etUpperGreek ; eType = tChar ; break ;
                  case L'c':  etIndx = etCustom ;     eType = tChar ; break ; // ('c' is DEPRECATED)
                  case L'j':  etIndx = etCJK ;        eType = tChar ; break ;
                  case L'k':  etIndx = etKatakana ;   eType = tChar ; break ;
                  case L'h':  etIndx = etHebrew ;     eType = tChar ; break ;
                  case L'e':  etIndx = etArabic ;     eType = tChar ; break ;
                  case L'.':  // use caller's default value
                  default:    // invalid option - return default values
                     break ;
               }

               //** If a non-fault initial value specified **
               if ( (wbuff[initIndex] >= L'0') && (wbuff[initIndex] <= L'9') )
               {
                  short tmpi ;
                  if ( (swscanf ( &wbuff[initIndex], L"%hd", &tmpi )) == 1 )
                     eStart = tmpi ;
                  //* If user is a dumbguy and specified only one digit *
                  //* as the start value, shift the options rightward.  *
                  if ( !((wbuff[initIndex+1] >= L'0') && (wbuff[initIndex+1] <= L'9')) )
                  {
                     wbuff[dirIndex] = wbuff[sizeIndex] ;
                     wbuff[sizeIndex] = wbuff[initIndex+1] ;
                  }
               }

               //** If a non-default relative font size specified. **
               if ( (wbuff[sizeIndex] == L's') || (wbuff[sizeIndex] == L'l') )
                  eSize = wbuff[sizeIndex] ;

               //** If non-default sequence direction specified **
               if ( wbuff[dirIndex] == seqDESCEND ) { eDir = seqDESCEND ; }
            }

            #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0
            gsdbg.append( "+ '%S' = '%S'\n"
                          "--------------------eType:%C eStart:%02hd eSize:%C eDir:%C etIndx:%02hd", 
                          gsB.gstr(), gsA.gstr(), &eType, &eStart, &eSize, &eDir, &etIndx  ) ;
            #endif   // DEBUG_LISTS && DEBUG_OLIST
         }
         //* Otherwise the token is malformed.             *
         //* Push the unmodified line back into the stream.*
         else
            this->ppfUnReadSrcLine ( gsA ) ;

         #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0
         this->textOut ( gsdbg ) ;
         #endif   // DEBUG_LISTS && DEBUG_OLIST
      }
      else     // token not found, push the unmodified line back into the stream
         this->ppfUnReadSrcLine ( gsA ) ;
   }

   return tokenRemoved ;

}  //* End ppfOListToken() *

//**************************
//*    ppfOListResponse    *
//**************************
//********************************************************************************
//* Parse user's response to the interactive prompt for configuring enumerated   *
//* lists.                                                                       *
//*                                                                              *
//* -- Enumeration type must be a valid response.                                *
//*    If not, an error status will be returned, and all other parameters        *
//*    will be ignored.                                                          *
//* -- If secondary values are valid, they will be returned.                     *
//* -- If secondary value is out-of-range, the 'no-response' value will be       *
//*    returned for that value:                                                  *
//*       Character   == L'.' (full-stop)                                        *
//*       Start value == -1                                                      *
//*                      A zero start vaulue for non-decimal enumeration         *
//*                      types will be silently corrected to 1 (one).            *
//*       Enumeration index == etTYPES                                           *
//*                                                                              *
//* Note: If user responded with the abort signal ("quit"), we won't see it here.*
//*                                                                              *
//* Input  : gsIn   : (by reference) user response text                          *
//*          eType  : (by reference) receives enumeration type                   *
//*                   valid types:                                               *
//*                     d : decimal          D : decimal with leading zero       *
//*                     l : lower-case alpha u : upper-case alpha                *
//*                     i : lower-case Roman I : upper-case Roman                *
//*                     g : lower-case Greek G : upper-case Greek                *
//*                     c : custom enum type (DEPRECATED)                        *
//*                     j : CJK (informal)   k : Katakana                        *
//*                     h : Hebrew           e : Arabic-Indic                    *
//*                     a : automatic        x : no processing                   *
//*          eStart : (by reference) enumeration start offset                    *
//*                   range: 1 - 99 (for 'd' and 'D' type: may be 0 (zero))      *
//*          fSize  : (by reference) relative font size                          *
//*                   'i' == inherited  's' == smaller  'l' == larger            *
//*          eDir   : (by reference) sequence direction                          *
//*                   'a' == descending  'd' == descending                       *
//*          etIndx : (by reference) index into list of CSS class names          *
//*                   corresponding to specified enumeration type                *
//*                                                                              *
//* Returns: 'true' if valid response, else 'false'                              *
//********************************************************************************

bool Idpp::ppfOListResponse ( const gString& gsIn, wchar_t& eType, short& eStart,
                              wchar_t& fSize, wchar_t& eDir, short& etIndx  )
{
   #define DEBUG_ORESP (0)                // for debugging only

   const wchar_t* pTemplate = L"%C %hd %C %C" ;    // template for swscanf()
   const wchar_t COMMA  = L',' ;          // delimeter character
   const wchar_t chDFLT = L'.' ;          // placeholder for default character value
   const wchar_t* sptr = gsIn.gstr() ;    // source pointer
   const short valDFLT = -1 ;             // placeholder start value 
   const short WBUFF = 16 ;               // size of work buffer
   wchar_t wbuff[WBUFF] ;                 // parsing buffer
   short wi = ZERO,                       // wbuff index
         si = ZERO,                       // source index
         capVals ;                        // number of values capture during scan
   bool  status = false ;                 // return value

   //* Initialize caller's data to indicate no values captured. *
   eType = eDir = fSize = chDFLT ;  // (indicates no value captured)
   eStart = valDFLT ;               // (indicates unknown start value)
   etIndx = etTYPES ;               // (indicates unknown type index)

   //* If default token specified, return success with 'eType' == "automatic" *
   if ( (status = bool((gsIn.compare( dToken, false, dToken_len )) == ZERO)) )
      eType = L'a' ;

   //* If not empty string and not "default_token", user has      *
   //* responded with enumeration type and optionally additional  *
   //* formatting parameters.                                     *
   if ( ! status && (gsIn.gschars() > 1) )
   {
      //* Initialize target buffer *
      for ( short i = ZERO ; i < WBUFF ; ++i ) { wbuff[i] = NULLCHAR ; }

      #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
      gsdbg.compose( "---olpr----- src:'%S' trg:'%S'\n", sptr, wbuff ) ;
      //this->textOut ( gsdbg ) ;
      #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP

      wbuff[wi++] = sptr[si++] ;          // copy enumeration type
      wbuff[wi++] = SPACE ;

      #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
      gsdbg.append( "---olpr----- src:'%S' trg:'%S'", sptr, wbuff ) ;
      this->textOut ( gsdbg ) ;
      #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP

      if ( sptr[si] == COMMA )            // step over delimeter
         ++si ;
      if ( sptr[si] != NULLCHAR )         // if not at end-of-source
      {
         if ( (sptr[si] >= L'0') && (sptr[si] <= L'9') ) // if start index specified
         {
            wbuff[wi++] = sptr[si++] ;    // first digit of start index
            if ( (sptr[si] >= L'0') && (sptr[si] <= L'9') ) // if a second digit for start value
               wbuff[wi++] = sptr[si++] ; // second digit of start index
            wbuff[wi++] = SPACE ;

            #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
            gsdbg.compose( "---olpr----- src:'%S' trg:'%S'", sptr, wbuff ) ;
            #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP
         }
         if ( sptr[si] == COMMA )         // step over delimeter
            ++si ;
         if ( sptr[si] != NULLCHAR )      // if not at end-of-source
         {
            wbuff[wi++] = sptr[si++] ;    // copy font size
            wbuff[wi++] = SPACE ;

            #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
            gsdbg.append( "\n---olpr----- src:'%S' trg:'%S'", sptr, wbuff ) ;
            #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP

            if ( sptr[si] == COMMA )      // step over delimeter
               ++si ;
            if ( sptr[si] != NULLCHAR )   // if not at end-of-source
            {
               wbuff[wi] = sptr[si] ;     // copy sequence direction

               #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
               gsdbg.append( "\n---olpr----- src:'%S' trg:'%S'", sptr, wbuff ) ;
               #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP
            }
         }

         #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
         this->textOut ( gsdbg ) ;
         #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP
      }

      //* Scan the parsed data to capture between 1 and 4 values. *
      wchar_t et = chDFLT, fs = chDFLT, ed = chDFLT ; short es = -1 ;
      capVals = swscanf ( wbuff, pTemplate, &et, &es, &fs, &ed ) ;
      if ( capVals >= 1 )                 // enumeration type captured
      {
         #if DEBUG_LISTS != 0 && DEBUG_OLIST != 0 && DEBUG_ORESP != 0
         gsdbg.compose( "---olprcap----%C %02hd %C %C (%hd items)",
                        &et, &es, &fs, &ed, &capVals ) ;
         this->textOut ( gsdbg ) ;
         #endif   // DEBUG_LISTS && DEBUG_OLIST && DEBUG_ORESP

         //* If 'a' (automatic) or 'A' (all automatic) )
         if ( et == L'A' )
         {
            this->olLists = cfgAuto ;     // set the auto-processing flag
            et = L'a' ;                   // signal automatic processing for this list
         }
         if ( et == L'a' )                // automatic processing for this list
         {
            eType = et ;
            status = true ;               // return success
         }
         else                             // non-default response
         {
            status = true ;               // provisional success
            for ( etIndx = etDecimal ; etIndx < etTYPES ; ++etIndx )
            {
               if ( et == enumChar[etIndx] ) // match found
               { eType = et ; break ; }      // return the valid type character
            }
            if ( etIndx == etTYPES )   // matching character not found
            {
               if ( et == L'x' )       // 'x' - do not modify the tag
                  eType = L'x' ;
               else                    // invalid enumeration type
                  status = false ;
            }

            //* If valid enumeration type and not 'x', *
            //* validate remaining parameters (if any).*
            if ( status && (capVals > 1) && (eType != L'x') )
            {
               //* A zero start value is valid only for decimal enumerators. *
               //* Silently set to one.                                      *
               if ( (es == ZERO) && !((eType == L'd') || (eType == L'D')) )
                  eStart = 1 ;
               else if ( (es >= 1) && (es <= 99) ) // if start value is in range
                  eStart = es ;

               if ( capVals > 2 )            // if font size specified
               {
                  if ( (fs == L's') || (fs == L'l') )
                     fSize = fs ;
                  if ( capVals > 3 )         // if sequence direction specified
                  {
                     if ( ed == L'd' )
                        eDir = ed ;
                  }
               }
            }
         }     // (non-auto)
      }        // (capVals >= 1)
   }           // (not default token)
   return status ;

   #undef DEBUG_ORESP
   #undef DEBUG_OLIST   // this is defined in the calling method
}  //* End ppfOListResponse() *

//**************************
//*   ppfFormattedBlock    *
//**************************
//********************************************************************************
//* Test whether the provided data indicates the beginning of a class block      *
//* for 'format', 'display' 'example' or 'lisp', 'verbatim' or their             *
//* 'small' / 'large' counterparts.                                              *
//*                                                                              *
//* If the target block was found, process it and write it to the target file.   *
//*                                                                              *
//* Input  : gsln   : line of source data to be scanned                          *
//*          status ; (by reference) receives 'ERR' if syntax or other error,    *
//*                   else receives 'OK'                                         *
//*          bookit : (optional, 'false' by default                              *
//*                   if 'true', AND if 'book' member set, then write the        *
//*                      bookmark messages to the display                        *
//*                   if 'false', do not write bookmark messages                 *
//*                                                                              *
//* Returns: 'true' if a preformatted block was processed                        *
//*          'false' caller is responsible for processing the specified data     *
//********************************************************************************

bool Idpp::ppfFormattedBlock ( const gString& gsln, short& status, bool bookit )
{
   gString gsBook ;           // text formatting
   blkType bType = btNone ;   // type of block found
   bool isBlock = false ;     // return value

   status = OK ;              // Initialize caller's status value (hope for success)

   if ( (this->ppfTestFormattedBlock ( bType, gsln )) != false )
   {
      isBlock = true ;        // formatted block identified

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"ML(%hu) PBLOCK START", &this->slCount ) ;
         this->textOut ( gsBook, false ) ;
      }

      //* Copy the block, optionally eliminating           *
      //* the unnecessary leading blank line.              *
      //* NOTE: Data inside these blocks ARE NOT modified. *
      status = this->ppfProcFormattedBlock ( bType, gsln ) ;

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"--(%hu) PBLOCK END", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }
   }
   return isBlock ;

}  //* End ppfFormattedBlock() *

//**************************
//* ppfTestFormattedBlock  *
//**************************
//********************************************************************************
//* Test whether the provided data indicates the beginning of a class block      *
//* for 'format', 'display' 'example' or 'lisp', 'verbatim' or their             *
//* 'small' / 'large' counterparts.                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  : bType  : (by reference) recieves indicator of block type:           *
//*                   member of enum blkType                                     *
//*          gsln   : line of source data to be scanned                          *
//*                                                                              *
//* Returns: 'true' if this is a preformatted block header, else 'false'         *
//********************************************************************************
//* Programmer's Note:                                                           *
//* 1) All formatted blocks live inside a <div>...</div> sequence, EXCEPT for    *
//*    the "verbatim" block which lives inside a <pre>...</pre> sequence.        *
//* 2) All formatted blocks are defined as being within a class.                 *
//*    -- Note that the "lisp" block is simply a special case of the "example"   *
//*       block. <div class="example lisp">                                      *
//*              <div class="example smalllisp lisp">                            *
//* 3) Note: For several iterations of makeinfo "small" versions of block        *
//*    designations were recognized, but not encoded. The "small" versions       *
//*    of formatted blocks were simply replaced by the standard class name.      *
//*    Example: The Texinfo command: @smalldisplay was rendered in the HTML as   *
//*             <div class="display">                                            *
//*    As of makeinfo v:7, the "small" versions again have actual class names:   *
//*       @smallindentedblock, @smallquotation, @smallformat, @smalldisplay,     *
//*       @smallexample and @smalllisp.                                          *
//*    --Note that makeinfo has never supported a "@smallverbatim" command, but  *
//*      we provide this as a post-processing option. (see ppfProcVerbatim).     *
//* 4) Note that makeinfo does not define the "@large...." commands, but we      *
//*    provide them as post-processing options.                                  *
//*                                                                              *
//*                                                                              *
//********************************************************************************

bool Idpp::ppfTestFormattedBlock ( blkType& bType, const gString& gsln )
{
   const wchar_t* divclassBegin = L"<div class=\"" ;
   const short    divclassBegin_len = 12 ;

   short wi ;                 // offset index
   bool fmtBlock = false ;    // return value
   bType = btNone ;           // initialize caller's variable

   if ( (wi = gsln.find( divclassBegin )) >= ZERO )
   {
      fmtBlock = true ;          // tentatively identified
      wi += divclassBegin_len ;  // step over identified partial tag

      if ( (gsln.find( L"format", wi, false )) == wi )
      {
         if ( (gsln.find( L"smallformat", wi, false )) > wi )
         {
            if ( (gsln.find( lrgCODE, wi, false )) > wi ) { bType = lrgF ; }
            else                                          { bType = smaF ; }
         }
         else { bType = stdF ; }
      }
      else if ( (gsln.find( L"display", wi, false )) == wi )
      {
         if ( (gsln.find( L"smalldisplay", wi, false )) > wi ) { bType = smaD ; }
         else { bType = stdD ; }
      }
      else if ( (gsln.find( L"example", wi, false )) == wi )
      {
         if ( (gsln.find( L"smallexample", wi, false )) > wi )           { bType = smaE ; }
         else if ( (gsln.find( L"example lisp", wi, false )) >= wi )     { bType = stdL ; }
         else if ( (gsln.find( L"example smalllisp", wi, false )) >= wi ){ bType = smaL ; }
         else { bType = stdE ; }
      }
      else if ( (gsln.find( L"small", wi, false )) == wi )
      {
         if ( (gsln.find( L"smallformat", wi, false )) >= wi )           { bType = smaF ; }
         else if ( (gsln.find( L"smalldisplay", wi, false )) >= wi )     { bType = smaD ; }
         else if ( (gsln.find( L"smallexample", wi, false )) >= wi )     { bType = smaE ; }
         else if ( (gsln.find( L"smalllisp", wi, false )) >= wi )        { bType = smaL ; }
         else { bType = btNone ; }  // invalid class
      }
      else if ( (gsln.find( L"large", wi, false )) == wi )
      {
         if ( (gsln.find( L"largeformat", wi, false )) >= wi )           { bType = lrgF ; }
         else if ( (gsln.find( L"largedisplay", wi, false )) >= wi )     { bType = lrgD ; }
         else if ( (gsln.find( L"largeexample", wi, false )) >= wi )     { bType = lrgE ; }
         else if ( (gsln.find( L"largelisp", wi, false )) >= wi )        { bType = lrgL ; }
         else { bType = btNone ; }  // invalid class
      }
      else           // line does not delimit a formatted data block
         fmtBlock = false ;
   }
   return fmtBlock ;

}  //* End ppfTestFormattedBlock() *

//**************************
//* ppfProcFormattedBlock  *
//**************************
//********************************************************************************
//* Copy the contents of a pre-formatted block from source to target.            *
//* Optionally, eliminate the extra blank line before the block by combining     *
//* the <div> and <pre> tags onto a single line.                                 *
//*                                                                              *
//* Preformatted blocks are: 'display', 'format', 'example', 'lisp' and their    *
//* small or large counterparts.                                                 *
//*                                                                              *
//*      Contents of preformatted blocks are copied as plain text,               *
//*                       EXCEPT:                                                *
//*      1) Formatted blocks within formatted blocks are processed.              *
//*      2) Beginning and end of <ol> and <ul> lists (remove inner <pre>).       *
//*         NOTE: There will be no user prompt for lists within formatted        *
//*               blocks.                                                        *
//*                                                                              *
//* Input  : bType  : indicates block type: member of enum blkType               *
//*          gsb    : (by reference) contains the first line of the block.       *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Notes:                                                                       *
//* The @group command may be used inside a formatted block (@example, @lisp,    *
//* @display, @format). It is used to keep a block all on one page in Tex output,*
//* but has no function in HTML. If a @group div is found, it will be on the     *
//* same line as the <pre> tag, and can be ignored.                              *
//*   Example: <div class="group"><pre class="display-preformatted">             *
//* The end-of-group </div> tag will be on the same line as the end-div tag for  *
//* the enclosing block, and can be ignored.                                     *
//*   Example: </pre></div></div>                                                *
//* If this syntax changes, we may need to remove the 'group' <div></div> pair.  *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcFormattedBlock ( blkType bType, const gString& gsb )
{
   const wchar_t* blockEND = L"</div>" ;
   const wchar_t* preSTART = L"<pre" ;
   const wchar_t* preEND   = L"</pre>" ;

   gString gss ;              // source line data
   blkType eType = btNone ;   // if nested block, its type
   short wi, xi,              // string index
         nestCount = 1,       // depth of nesting
         listCount = ZERO,    // embedded <ul> and <ol> lists
         status = OK ;        // return value
   bool done = false ;        // loop control

   //* If specified, this call removes the useless leading *
   //* garbage which causes an extra blank line before the *
   //* block. Prompt user for block font size.             *
   //* Note that if the "no_block"  flag if set, it        *
   //* overrides user interaction.                         *
   if ( this->no_bloc == false )
      status = this->ppfProcInnerBlock ( bType, gsb ) ;

   //* Else, copy the header line as-is *
   else
      this->ppfWriteLine ( gsb ) ;

   while ( ! done && status == OK )
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
      {
         //* If an orphaned </pre> tag and/or an embedded <pre> tag,    *
         //* discard them. Note: This discards any @exdent or similar   *
         //* tortured syntax within a preformatted block.               *
         if ( (xi = gss.find( preEND )) >= ZERO )
            removeTag ( gss, xi ) ;
         if ( (xi = gss.find( preSTART, wi, false )) >= ZERO )
            removeTag ( gss, xi ) ;

         //* If end of block, then return to caller *
         if ( (gss.find( blockEND, ZERO, false )) >= ZERO )
         {
            this->ppfWriteLine ( gss ) ;     // write line to target
            if ( --nestCount <= ZERO )
            {
               done = true ;
               break ;
            }
            else  // end of nested block found, keep processing
               continue ;
         }

         //* Test for <ol>, <ul> lists and other preformatted blocks INSIDE *
         //* this preformatted block. No intelligent user would do this,    *
         //* but if it happens, we need to identify it.                     *
         if ( ((this->ppfTestEnumeratedList ( gss, ZERO )) != false)
              ||
              ((this->ppfTestItemizedList ( gss, ZERO )) != false) )
         {
            this->ppfPFB_List ( gss ) ;      // strip the garbage
            this->ppfWriteLine ( gss ) ;     // write line to target
            ++listCount ;                    // count embedded lists
         }
         else if ( ((gss.find( ulEnd, ZERO, false )) >= ZERO)
                   ||
                   ((gss.find( olEnd, ZERO, false )) >= ZERO) )
         {
            this->ppfPFB_List ( gss ) ;      // strip the garbage
            this->ppfWriteLine ( gss ) ;     // write line to target
            //* The source line following the list unnecessarily *
            //* re-declares the containing block.                *
            this->ppfReadSrcLine ( gss, wi ) ;
            // Programmer's Note: When the source file has been previously 
            // processed, the source and target would be different without 
            // this conditional statement. If practical, we want a second pass 
            // on a previously-processed file to produce an exact copy of the 
            // source. This is not critical, but it is a matter of pride.
            if ( (gss.gschars()) > 1 )
               this->ppfPFB_List ( gss ) ;   // strip the garbage
            this->ppfWriteLine ( gss ) ;     // write line to target
            --listCount ;                    // embedded list complete
         }

         //* If start of an embedded block *
         else if ( (this->ppfTestFormattedBlock ( eType, gss )) != false )
         {
            #if DEBUG_BLOCKS
            gsdbg.compose( "EMB at:%hd '%S'", &this->slCount, gss.gstr() ) ; this->textOut ( gsdbg ) ;
            #endif   // DEBUG_BLOCKS

            this->ppfWriteLine ( gss ) ;     // write line to target
            ++nestCount ;                    // nesting level
         }
         else
         {
            //* If inside an embedded list *
            if ( listCount > ZERO )
               this->ppfPFB_List ( gss ) ;   // strip the garbage
            this->ppfWriteLine ( gss ) ;     // write the line
         }
      }
   }
   return status ;

}  //* End ppfProcFormattedBlock() *

//**************************
//*    ppfIndentedBlock    *
//**************************
//********************************************************************************
//* Test whether the provided data indicates the beginning of an 'indentedblock' *
//* or its 'small' / 'large' counterparts.                                       *
//*                                                                              *
//* If the target block was found, process it and write it to the target file.   *
//*                                                                              *
//* Input  : gsln   : line of source data to be scanned                          *
//*          status ; (by reference) receives 'ERR' if syntax or other error,    *
//*                   else receives 'OK'                                         *
//*          bookit : (optional, 'false' by default                              *
//*                   if 'true', AND if 'book' member set, then write the        *
//*                      bookmark messages to the display                        *
//*                   if 'false', do not write bookmark messages                 *
//*                                                                              *
//* Returns: 'true' if a indented block was processed                            *
//*          'false' caller is responsible for processing the specified data     *
//********************************************************************************

bool Idpp::ppfIndentedBlock ( const gString& gsln, short& status, bool bookit )
{
   gString gsBook ;           // text formatting
   blkType bType = btNone ;   // type of block found
   bool isBlock = false ;     // return value

   status = OK ;              // Initialize caller's status value (hope for success)

   if ( (this->ppfTestIndentedBlock ( bType, gsln )) != false )
   {
      isBlock = true ;        // formatted block identified

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"ML(%hu) IBLOCK START", &this->slCount ) ;
         this->textOut ( gsBook, false ) ;
      }

      status = this->ppfProcIndentedBlock ( bType, gsln ) ;

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"--(%hu) IBLOCK END", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }
   }
   return isBlock ;

}  //* End ppfIndentedBlock() *

//*************************
//* ppfTestIndentedBlock  *
//*************************
//********************************************************************************
//* Scan the source data for one of the "indentedblock" headers:                 *
//*          indentedblock                                                       *
//*          smallindentedblock                                                  *
//*          largeindentedblock                                                  *
//* Note that an unstyled <blockquote> tag indicates a "quotation" block, NOT    *
//* an indentedblock.                                                            *
//*                                                                              *
//* Input  : bType  : (by reference) recieves indicator of block type:           *
//*                   member of enum blkType                                     *
//*          gsln   : line of source data to be scanned                          *
//*                                                                              *
//* Returns: 'true'  if an "indentedblock" header ('bType' initialized)          *
//*          'false' if not indentedblock header ('bType' unchanged)             *
//********************************************************************************
//* Programmer's Note: From makeinfo v:6.6 to v:7, the "smallindentedblock"      *
//* tag was automagically converted to "indentedblock". As of makeinfo v:7,      *
//* the "smallindentedblock" is again recognized.                                *
//********************************************************************************

bool Idpp::ppfTestIndentedBlock ( blkType& bType, const gString&gsln )
{
   bool status = true ;

   if ( (gsln.find( indeInherit, ZERO, false )) >= ZERO )
      bType = stdI ;
   else if ( ((gsln.find( indeSmall7, ZERO, false )) >= ZERO) ||
             ((gsln.find( indeSmall, ZERO, false )) >= ZERO) )
      bType = smaI ;
   else if ( (gsln.find( indeLarge, ZERO, false )) >= ZERO )
      bType = lrgI ;
   else
      status = false ;

   return status ;

}  //* End ppfTestIndentedBlock() *

//*************************
//* ppfProcIndentedBlock  *
//*************************
//********************************************************************************
//* Called by ppfProcessSrcHTML() to process 'indentedblock' and                 *
//* 'smallindentedblock' sequences.                                              *
//*                                                                              *
//* Input  : bType  : indicates block type: member of enum blkType               *
//*          gsib  : (by reference) contains the first line of the block.        *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Programmer's Note: It is possible that this method will be called            *
//* recursively, so be aware.                                                    *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcIndentedBlock ( blkType bType, const gString& gsib )
{
   gString gsB = gsib,        // text formatting
           gss,               // source text
           gsNext ;           // context line
   wchar_t usrResp = L'a';    // user response to prompt
   short wi = ZERO,           // string index
         bi,                  // index of <blockquote...  tag
         status = OK ;        // return value
   bool  lrgFlag = false,     // set if large block was encoded as a macro
         customClass = false ;// 'true' if existing (non-default) class callout in source

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "GSB at:%03hd '%S'", &this->slCount, gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   //* Identify the specified block type, then erase the existing tag *
   if ( (bi = gsB.find( indeInherit )) >= ZERO )         // (default)
      gsB.erase( indeInherit, bi ) ;
   else if ( (bi = gsB.find( indeSmall )) >= ZERO )      // makeinfo v:6 (or previously processed)
   { customClass = true ; gsB.erase( indeSmall, bi ) ; }
   else if ( ((bi = gsB.find( indeSmall7 )) >= ZERO) )   // makeinfo v:7 "small" variant
   {
      customClass = true ;
      gsB.erase( indeSmall7, bi ) ;
      status = this->ppfReadSrcLine ( gsNext, wi ) ;
      if ( (gsNext.find( lrgCODE )) == 3 )
      {
         #if DEBUG_BLOCKS != 0
         gsdbg.compose( "lx---------'%S'", gsNext.gstr() ) ; this->textOut ( gsdbg ) ;
         #endif   // DEBUG_BLOCKS

         gsNext.erase( lrgCODE ) ; lrgFlag = true ; bType = lrgI ;
      }
      this->ppfUnReadSrcLine ( gsNext ) ;
   }
   else if ( (bi = gsB.find( indeLarge )) >= ZERO )      // previously processed
   { customClass = true ; gsB.erase( indeLarge, bi ) ; }

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "tx---------'%S'", gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   //* If user interaction is enabled, *
   //* prompt for user's selection.    *
   if ( this->blkFont == cfgSpec )
   {
      //* If there is display text on this line (unlikely), use it *
      if ( gsB.gstr()[bi] != NULLCHAR )
         gsNext = &gsB.gstr()[bi] ;
      //* Else read the next line to give user some context.       *
      else
      {
         status = this->ppfReadSrcLine ( gsNext, wi ) ;
         // Programmer's Note: If context line originally contained a large
         // macro token, it was removed above.

         //* Push the context line back into the stream      *
         //* format the text portion of the line for display.*
         this->ppfUnReadSrcLine ( gsNext ) ;
         gsNext.erase( L"<p>", ZERO, false ) ;
         gsNext.erase( L"<br>", ZERO, false ) ;
      }

      //* Ask the user to select a font-size option *
      usrResp = this->ppfBlockPrompt ( blockName[bType], gsNext.gstr() ) ;

      //* If abort received. Note the early return. *
      if ( this->abort )
      { return ( status = ERR ) ; }
   }
   //* If skip count is active, decrement the counter *
   else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

   //* If embedded token for a "large" block, AND    *
   //* if 'a' (automatic) selected, set large block. *
   if ( lrgFlag && (usrResp == L'a') )
      usrResp = L'l' ;

   //* Update the tag according to user's selection *
   if ( bi >= ZERO )
   {
      switch ( usrResp )
      {
         //** inherited font **
         case L'i':  gsB.insert( indeInherit, bi ) ; break ;
         //** smaller font **
         case L's':  gsB.insert( indeSmall, bi ) ;   break ;
         //** larger font **
         case L'l':  gsB.insert( indeLarge, bi ) ;   break ;
         //** Automatic assignment of class       **
         //** If font size previously specified,  **
         //** use it, else use standard font size.**
         case L'a':
         default:
            if ( bType == smaI )
               gsB.insert( indeSmall, bi ) ;
            else if ( bType == lrgI )
               gsB.insert( indeLarge, bi ) ;
            else
               gsB.insert( indeInherit, bi ) ;
            break ;
      } ;

      //* During a second pass, the existing class *
      //* designation overrides user response      *
      //* taken from a response file.              *
      if ( customClass && (this->rifs.is_open()) )
      { gsB = gsib ; }  // copy line as-written
   }
   else
   {
      gsB = gsib ;      // copy line as-written
      status = ERR ;
   }

   //* Write the header line *
   this->ppfWriteLine ( gsB ) ;

   //* Process the body of the block *
   bool done = false ;
   while ( ! done && status == OK )
   {  //* If at the end of the block, write the line and jump out
      if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
      {
         if ( (gss.find( quotEND, ZERO, false )) >= ZERO )
         {
            this->ppfWriteLine ( gss ) ;
            done = true ;

            #if DEBUG_BLOCKS != 0
            gsdbg.compose( "OUT--------'%S'\n", gss.gstr() ) ; this->textOut ( gsdbg ) ;
            #endif   // DEBUG_BLOCKS
         }

         //* If an indentedblock, smallindentedblock or largeindentedblock.*
         //* This may result in a recursive call.                          *
         else if ( (this->ppfIndentedBlock ( gss, status )) != false )
         { continue ; }

         //* If a quotation, smallquotation or largequotation block *
         else if ( (this->ppfQuotationBlock ( gss, status )) != false )
         { continue ; }

         //* If a verbatim, smallverbatim or largeverbatim block *
         else if ( (this->ppfVerbatimBlock ( gss, status )) != false )
         { continue ; }

         //* If a preformatted block is detected:              *
         //* 'display, 'format', 'example', 'lisp', and their  *
         //* 'small' or 'large' counterparts.                  *
         else if ( (this->ppfFormattedBlock ( gss, status )) != false )
         { /* do nothing */ }

         //* If an Itemized List (<ul> tag) *
         else if ( (this->ppfItemizedList ( gss, status, wi )) != false )
         { /* do nothing */ }

         //* If an Enumerated List (<ol> tag) *
         else if ( (this->ppfEnumeratedList ( gss, status, wi )) != false )
         { /* do nothing */ }

         //* If a <table> object found *
         else if ( (gss.find( tabBegin, wi, false )) >= ZERO )
         {
            status = this->ppfProcTABLE ( gss ) ;
         }

         else     // copy the line without processing
            this->ppfWriteLine ( gss ) ;
      }     // read_line
   }        // while()
   return status ;

}  //* End ppfProcIndentedBlock() *

//**************************
//*   ppfQuotationBlock    *
//**************************
//********************************************************************************
//* Test whether the provided data indicates the beginning of an 'quotation'     *
//* or its 'small' / 'large' counterparts.                                       *
//*                                                                              *
//* If the target block was found, process it and write it to the target file.   *
//*                                                                              *
//* Input  : gsln   : line of source data to be scanned                          *
//*          status ; (by reference) receives 'ERR' if syntax or other error,    *
//*                   else receives 'OK'                                         *
//*          bookit : (optional, 'false' by default                              *
//*                   if 'true', AND if 'book' member set, then write the        *
//*                      bookmark messages to the display                        *
//*                   if 'false', do not write bookmark messages                 *
//*                                                                              *
//* Returns: 'true' if a indented block was processed                            *
//*          'false' caller is responsible for processing the specified data     *
//********************************************************************************

bool Idpp::ppfQuotationBlock ( const gString& gsln, short& status, bool bookit )
{
   gString gsBook ;           // text formatting
   blkType bType = btNone ;   // type of block found
   bool isBlock = false ;     // return value

   status = OK ;              // Initialize caller's status value (hope for success)

   if ( (this->ppfTestQuotationBlock ( bType, gsln )) != false )
   {
      isBlock = true ;        // formatted block identified

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"ML(%hu) QBLOCK START", &this->slCount ) ;
         this->textOut ( gsBook, false ) ;
      }

      status = this->ppfProcQuotationBlock ( bType, gsln ) ;

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"--(%hu) QBLOCK END", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }
   }
   return isBlock ;

}  //* End ppfQuotationBlock() *

//*************************
//* ppfTestQuotationBlock *
//*************************
//********************************************************************************
//* Scan the source data for one of the "quotation" block headers:               *
//*          quotation (or an unstyled <blockquote>)                             *
//*          smallquotation                                                      *
//*          largequotation                                                      *
//*                                                                              *
//* Input  : bType  : (by reference) recieves indicator of block type:           *
//*                   member of enum blkType                                     *
//*          gsln   : line of source data to be scanned                          *
//*                                                                              *
//* Returns: 'true'  if a "quotation" block header ('bType' initialized)         *
//*          'false' if not quotation block header ('bType' unchanged)           *
//********************************************************************************

bool Idpp::ppfTestQuotationBlock ( blkType& bType, const gString&gsln )
{
   short wi ;                             // character ndex
   bool status = false ;                  // return value
   bType = btNone ;                       // initialize caller's variable

   if ( (wi = gsln.find( L"<blockquote" )) >= ZERO )
   {
      status = true ;                     // provisional success

      if ( ((gsln.find( quotInherit, wi, false )) >= ZERO) ||  // makeinfo v:7
           ((gsln.find( quotSimple,  wi, false )) >= ZERO) )   // makeinfo v:6
      { bType = stdQ ; }
      else if ( ((gsln.find( quotSmall7, wi, false )) >= ZERO) || // makeinfo v:7
                ((gsln.find( quotSmall,  wi, false )) >= ZERO) )  // makeinfo v:6
      { bType = smaQ ; }
      else if ( (gsln.find( quotLarge, wi, false )) >= ZERO )
      { bType = lrgQ ; }
      else
         status = false ;
   }

   return status ;

}  //* End ppfTestQuotationBlock() *

//*************************
//* ppfProcQuotationBlock *
//*************************
//********************************************************************************
//* Called by ppfProcessSrcHTML() to process <blockquote> sequences.             *
//* This includes:                                                               *
//*    <blockquote>  (unprocessed @quotation) PLUS:                              *
//*    <blockquote class="quotation">                                            *
//*    <blockquote class="smallquotation">                                       *
//*    <blockquote class="largequotation">                                       *
//* If we find an author's name sequence following follows the block, the        *
//* sequence will have the form:                                                 *
//*    <div align=\"center\">&mdash; <em>Author Name>/em>                        *
//*    </div>                                                                    *
//*                                                                              *
//* Input  : bType  : indicates block type: member of enum blkType               *
//*          gsbq  : (by reference) contains the first line of the block.        *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Programmer's Note: The author name sequence is moved INSIDE the block,       *
//* where we believe it should have been all along. In addition, we replace      *
//* the centered style with a simple offset style which looks much better.       *
//*                                                                              *
//* Programmer's Note: The procedure for this modification is not particularly   *
//* robust, but it handles all _reasonable_ tests well.                          *
//*                                                                              *
//* Embedded blocks: Because both "quotation" and "indentedblock" use the        *
//* <blockquote> tag, if one is embedded inside the other we need to track       *
//* the number of <blockquote>...</blockquote> pairs we see here so we know      *
//* when the main block is complete. Therefore 'bqCount' is used to track        *
//* the embedded "indentedblock" sections.                                       *
//*                                                                              *
//* However: Other blocks embedded within "quotation" blocks are merely          *
//* _identified_, NOT processed.                                                 *
//********************************************************************************

short Idpp::ppfProcQuotationBlock ( blkType bType, const gString& gsbq )
{
   const wchar_t* quotBEGIN  = L"<blockquote" ;
   const wchar_t* endDIV     = L"</div>" ;
   const wchar_t* endSPAN    = L"</span>" ;
   const wchar_t* authorCMD7 = L"<div class=\"center\">&mdash; <em class=\"emph\">" ;
   const wchar_t* authorCMD6 = L"<div align=\"center\">&mdash; <em>" ;
   const wchar_t* authorSTYLE = L"<br><span style=\"margin-left:3.2em; font-style:italic;\">&mdash; " ;
   // Possibly replace the hard-coded style with a CSS class?
   //onst wchar_t* authorClass = L"<span class=\"quote_author\">" ;
   gString gsB = gsbq,        // text formatting
           gss,               // source text
           gsNext ;           // context line
   wchar_t usrResp = L'a' ;   // user response to prompt (initialized to 'automatic)
   short wi = ZERO,           // string index
         qi,                  // index of <blockquote...  tag
         ti,                  // tag index
         bqCount = 1,         // count <blockquote...> tags within this block
         status = OK ;        // return value
   bool  customClass = false, // 'true' if existing (nondefault) class callout in source
         lrgFlag = false ;    // set if large block was encoded as a macro

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "GSB at:%03hd '%S'", &this->slCount, gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   //* Erase the existing tag *
   if ( (qi = gsB.find( quotBEGIN )) >= ZERO )
   {
      //* Identify any special cases, then delete the tag *
// Could be quotSmall7
      if ( (gsB.find( quotSmall )) >= ZERO )
         customClass = true ;
      else if ( (gsB.find( quotLarge )) >= ZERO )
         customClass = true ;

      removeTag ( gsB, qi ) ;             // delete the block-start tag

      #if DEBUG_BLOCKS != 0
      gsdbg.compose( "tx---------'%S'", gsB.gstr() ) ; this->textOut ( gsdbg ) ;
      #endif   // DEBUG_BLOCKS
   }

   //* It is assumed that there is no display text on the first     *
   //* line, only HTML tag(s). Therefore read the next line.        *
   //* If embedded "large" token, remove it and set the flag.       *
   //* If user prompt is active, prepare text for display.          *
   //* Push the (perhaps adjusted) text back into the source stream.*
   if ( (status = this->ppfReadSrcLine ( gsNext, wi )) == OK )
   {
      //* If line contains an embedded macro code *
      //* for a large block, remove it.           *
      if ( (ti = gsNext.find( lrgCODE )) >= ZERO )
      {
         #if DEBUG_BLOCKS != 0
         gsdbg.compose( "-----------'%S'", gsNext.gstr() ) ;
         #endif   // DEBUG_BLOCKS

         gsNext.erase( lrgCODE, ti ) ;    // erase the token
         bType = lrgQ ;                   // set block type to largeQuotation
         lrgFlag = true ;                 // set flag indicating "large" token found

         #if DEBUG_BLOCKS != 0
         gsdbg.append( "\nlx---------'%S'", gsNext.gstr() ) ; this->textOut ( gsdbg ) ;
         #endif   // DEBUG_BLOCKS

         //* If the embedded token was the only thing on the line, or *
         //* if the only data remaining is an HTML tag, the context   *
         //* will be on the following line.                           *
         if ( gsNext.gschars() == 1 )
            status = this->ppfReadSrcLine ( gsNext, wi ) ;
         else if ( ((gsNext.find( L"<p>" )) == ZERO) && (gsNext.gschars() == 4) )
         {
            gss.clear() ;
            status = this->ppfReadSrcLine ( gss, wi ) ;
            gsNext.append( gss.gstr() ) ;
         }
      }
      if ( status == OK )
      {
         //* Push the context line back into the stream      *
         //* format the text portion of the line for display.*
         this->ppfUnReadSrcLine ( gsNext ) ;
         if ( this->blkFont == cfgSpec )
         {
            gsNext.erase( L"<p>", ZERO, false ) ;
            gsNext.erase( L"<br>", ZERO, false ) ;
         }
      }
   }

   //* If user interaction is enabled, *
   //* prompt for user's selection.    *
   if ( this->blkFont == cfgSpec )
   {
      usrResp = this->ppfBlockPrompt ( blockName[bType], gsNext.gstr() ) ;

      if ( this->abort )            // if abort received (note the early return)
      { return ( status = ERR ) ; }
   }
   //* If skip count is active, decrement the counter *
   else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

   //* If embedded token for a "large" block, AND    *
   //* if 'a' (automatic) selected, set large block. *
   if ( lrgFlag && usrResp == L'a' )
      usrResp = L'l' ;

   //* Update the tag according to user's selection *
   if ( qi >= ZERO )
   {
      switch ( usrResp )
      {
         //** inherited font **
         case L'i':  gsB.insert( quotInherit, qi ) ; break ;
         //** smaller font **
         case L's':  gsB.insert( quotSmall, qi ) ;   break ;
         //** larger font **
         case L'l':  gsB.insert( quotLarge, qi ) ;   break ;
         //** Automatic assignment of class       **
         //** If font size previously specified,  **
         //** use it, else use standard font size.**
         case L'a':
         default:
            if ( bType == smaQ )
               gsB.insert( quotSmall, qi ) ;
            else if ( bType == lrgQ )
               gsB.insert( quotLarge, qi ) ;
            else
               gsB.insert( quotInherit, qi ) ;
            break ;
      } ;

      //* During a second pass, the existing class designation *
      //* overrides response from response file.               *
      if ( customClass && (this->rifs.is_open()) )
      { gsB = gsbq ; }  // copy line as-written
   }
   else
   {
      gsB = gsbq ;      // copy line as-written
      status = ERR ;
   }

   //* Write the header tag *
   this->ppfWriteLine ( gsB ) ;

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "OUT------>>'%S'", gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
      {
         if ( (gss.find( quotEND, ZERO, false )) >= ZERO )
         {
            if ( --bqCount <= ZERO )
               break ;     // end-tag read but not written to target
         }
         else if ( (gss.find( quotBEGIN )) >= ZERO )
            ++bqCount ;

         this->ppfWriteLine ( gss ) ;     // write to target
      }
   }

   //* The tag closing the quotation block has been captured, but has   *
   //* not yet been written. Unless the 'no_auth' flag has been set     *
   //* indicating that the "author" sequence should be written as-is,   *
   //* Scan for an "author" tag and make adjustments as necessary; then *
   //* write the author sequence (if any) followed by the closing tag.  *
   gsNext = gss ;       // save the end-of-block tag
   if ( ! this->no_auth && (status == OK) )
   {
      if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
      {
         bool hasAuthor = true ;

         //* Give the author some style *
         if ( (ti = gss.find( authorCMD7, ZERO, false )) >= ZERO )   // makeinfo v:7
            gss.replace( authorCMD7, authorSTYLE ) ;
         else if ( (ti = gss.find( authorCMD6, ZERO, false )) >= ZERO ) // makeinfo v:6
            gss.replace( authorCMD6, authorSTYLE ) ;
         else
            hasAuthor = false ;

         //* If an author entry identified and adjusted, write it to target. *
         if ( hasAuthor )
         {
            gss.append( endSPAN ) ;
            this->ppfWriteLine ( gss ) ;

            #if DEBUG_BLOCKS != 0
            gsdbg.compose( "auth-------'%S'", gss.gstr() ) ; this->textOut ( gsdbg ) ;
            #endif   // DEBUG_BLOCKS

            //* Read and discard the author sequence end-tag, *
            //* retaining any trailing data on that line      *
            //* which is pushed back into the input stream.   *
            if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
            {
               if ( (ti = gss.find( endDIV )) >= ZERO )
                  removeTag ( gss, ti ) ;
            }
            else { gss.clear() ; }  // (be safe)
         }

         //* Whatever remains is pushed back into the input stream. *
         if ( gss.gschars() > 1 )
            this->ppfUnReadSrcLine ( gss ) ;
      }
   }
   this->ppfWriteLine ( gsNext ) ;     // write the end-of-block tag

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "END--------'%S'\n", gsNext.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   return status ;

}  //* End ppfProcQuotationBlock() *

//**************************
//*    ppfVerbatimBlock    *
//**************************
//********************************************************************************
//* Scan the source data for one of the "verbatim" block headers:                *
//* 'verbatim', 'smallverbatim', 'largeverbatim'                                 *
//*                                                                              *
//* If the target block was found, process it and write it to the target file.   *
//*                                                                              *
//* Input  : gsln   : line of source data to be scanned                          *
//*          status ; (by reference) receives 'ERR' if syntax or other error,    *
//*                   else receives 'OK'                                         *
//*          bookit : (optional, 'false' by default                              *
//*                   if 'true', AND if 'book' member set, then write the        *
//*                      bookmark messages to the display                        *
//*                   if 'false', do not write bookmark messages                 *
//*                                                                              *
//* Returns: 'true' if a preformatted block was processed                        *
//*          'false' caller is responsible for processing the specified data     *
//********************************************************************************

bool Idpp::ppfVerbatimBlock ( const gString& gsln, short& status, bool bookit )
{
   gString gsBook ;           // text formatting
   blkType bType = btNone ;   // type of block found
   bool isBlock = false ;     // return value

   status = OK ;              // Initialize caller's status value (hope for success)

   if ( (this->ppfTestVerbatimBlock ( bType, gsln )) != false )
   {
      isBlock = true ;        // verbatim block identified

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"ML(%hu) VBLOCK START", &this->slCount ) ;
         this->textOut ( gsBook, false ) ;
      }

      status = this->ppfProcVerbatimBlock ( bType, gsln ) ;

      if ( (bookit != false) && (this->book != false) )
      {
         gsBook.compose( L"--(%hu) VBLOCK END", &this->slCount ) ;
         this->textOut ( gsBook ) ;
      }
   }
   return isBlock ;

}  //* End ppfVerbatimBlock() *

//*************************
//* ppfTestVerbatimBlock  *
//*************************
//********************************************************************************
//* Scan the source data for one of the "verbatim" block headers:                *
//*          verbatim                                                            *
//*          smallverbatim                                                       *
//*          largeverbatim                                                       *
//*                                                                              *
//* Input  : bType  : (by reference) receives indicator of block type:           *
//*                   member of enum blkType                                     *
//*          gsln   : line of source data to be scanned                          *
//*                                                                              *
//* Returns: 'true'  if a "verbatim" block header ('bType' initialized)          *
//*          'false' if not verbatim block header ('bType' unchanged)            *
//********************************************************************************

bool Idpp::ppfTestVerbatimBlock ( blkType& bType, const gString&gsln )
{
   const wchar_t* preclassBegin = L"<pre class=\"" ;
   const short    preclassBegin_len = 12 ;
   const wchar_t* vClass = L"verbatim\">" ;
   const short    vClass_len = 10 ;
   const wchar_t* svClass = L"smallverbatim\">" ;
   const short    svClass_len = 15 ;
   const wchar_t* lvClass = L"largeverbatim\">" ;
   const short    lvClass_len = 15 ;

   short wi ;              // text index
   bool  status = false ;  // return value

   if ( (wi = gsln.find( preclassBegin )) >= ZERO )
   {
      status = true ;
      wi += preclassBegin_len ;  // step over identified partial tag
      if ( (gsln.compare( vClass, false, vClass_len, wi )) == ZERO )
         bType = stdV ;    // 'verbatim' class block
      else if ( (gsln.compare( svClass, false, svClass_len, wi )) == ZERO )
         bType = smaV ;    // 'smallverbatim' class block
      else if ( (gsln.compare( lvClass, false, lvClass_len, wi )) == ZERO )
         bType = lrgV ;    // 'largeverbatim' class block
      else           // line does not delimit a verbatim block
         status = false ;
   }
   return status ;

}  //* End ppfTestVerbatimBlock() *

//*************************
//* ppfProcVerbatimBlock  *
//*************************
//********************************************************************************
//* Called by ppfProcFormattedBlock() method to process "verbatim" blocks.       *
//* Note that we simply copy the block, unmodified, from source to target        *
//* UNLESS the user specifies a font size for the block.                         *
//*                                                                              *
//* Input  : bType  : indicates block type: member of enum blkType               *
//*          gsb    : (by reference) contains the first line of the block.       *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Programmer's Note:                                                           *
//* Because the 'verbatim' block is a <pre>...</pre> construct, _AND_ because    *
//* texi2any combines the "</pre>" with whatever HTML follows it.                *
//* -- The HTML output doesn't care, because the data is actually written to     *
//*    the target.                                                               *
//* -- However, if the post-processor needs to see that information, it will     *
//*    get confused when it doesn't.                                             *
//* -- If a blank line follows the 'verbatim' block, then the </pre> will be     *
//*    on the line by itself, and the caller will not miss anything.             *
//* -- The problem is that the user cannot be relied upon to remember the        *
//*    trailing blank line, so we must parse the line containing the "</pre>"    *
//*    and return any unused data to the caller. The elegant way to do this is   *
//*    to push the unused data back into the input stream. Unfortunately, this   *
//*    is not supported for input-only streams. Therefore, we have implemented   *
//*    an input-stream push-back buffer 'pushBack[]' to preserve any data        *
//*    following the "</pre>" tag.                                               *
//*                                                                              *
//* -- "@smallverbatim" and "@largeverbatim" are not recognized by makeinfo,     *
//*    BUT macros for these block types are defined in the "texi_macros.texi".   *
//*    @macro smallverbatim{x}                                                   *
//*    @verbatim                                                                 *
//*    \x\                                                                       *
//*    @end verbatim                                                             *
//*    @end macro                                                                *
//*                                                                              *
//*    @macro largeverbatim{x}                                                   *
//*    @verbatim                                                                 *
//*    \x\                                                                       *
//*    @end verbatim                                                             *
//*    @end macro                                                                *
//*                                                                              *
//*    To define a block using these macros, the first three(3) characters       *
//*    signal the post-processor which font size to use. These three characters  *
//*    will be deleted by the post-processor.                                    *
//*    Example:                            Output:                               *
//*    --------                            -------                               *
//*    @smallverbatim{=s=                  Display text for the block.           *
//*    Display text for the block.         More display text for the block.      *
//*    More display text for the block.}                                         *
//*                                                                              *
//*    Example:                            Output:                               *
//*    --------                            -------                               *
//*    @largeverbatim{=l=                  Display text for the block.           *
//*    Display text for the block.         More display text for the block.      *
//*    More display text for the block.}                                         *
//*                                                                              *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcVerbatimBlock ( blkType bType, const gString& gsb )
{
   const wchar_t* smaCODE = L"=s=" ;   // encoded tag appended to "small font" blocks
   const wchar_t* verbInherit    = L"<pre class=\"verbatim\">" ;
   const wchar_t* verbSmall      = L"<pre class=\"smallverbatim\">" ;
   const wchar_t* verbLarge      = L"<pre class=\"largeverbatim\">" ;
   const wchar_t* verbEND        = L"</pre>" ;

   gString gsB = gsb,         // source line data (first line of block)
           gss,               // read from source file
           gstmp, gsIn ;      // user interaction
   wchar_t usrResp = L'a' ;   // user response to prompt
   short wi,                  // string index
         pi,                  // index of <pre...  tag
         ti,                  // index of token for embedded macro
         status = OK ;        // return value
   bool  lrgFlag = false,     // set if large block was encoded as a macro
         smaFlag = false,     // set if small block was encoded as a macro
         done = false ;       // loop control

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "GSB at:%03hd '%S'", &this->slCount, gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   //* Erase the existing tag *
   if ( (pi = gsB.find( L"<pre class=" )) >= ZERO )
   {
      removeTag ( gsB, pi ) ;

      #if DEBUG_BLOCKS != 0
      gsdbg.compose( "tx---------'%S'", gsB.gstr() ) ;
      #endif   // DEBUG_BLOCKS

      //* A bit of defensive programming: If the block contains no data,   *
      //* the closing tag </pre> will be on the same line as the opening   *
      //* tag. Example: "<pre class="verbatim"></pre><p>following_data"    *
      //* In this case, prevent the loop from executing.                   *
      if ( (gsB.find( verbEND )) >= ZERO )
         done = true ;

      //* If line contains an embedded macro code *
      //* for a small or large block, remove it.  *
      if ( (ti = gsB.find( lrgCODE )) >= ZERO )
      {
         gsB.erase( lrgCODE, ti ) ; lrgFlag = true ; bType = lrgV ;

         #if DEBUG_BLOCKS != 0
         gsdbg.append( "\nlx---------'%S'", gsB.gstr() ) ;
         #endif   // DEBUG_BLOCKS
      }
      else if ( (ti = gsB.find( smaCODE )) >= ZERO )
      {
         gsB.erase( smaCODE, ti ) ; smaFlag = true ; bType = smaV ;

         #if DEBUG_BLOCKS != 0
         gsdbg.append( "\nsx---------'%S'", gsB.gstr() ) ;
         #endif   // DEBUG_BLOCKS
      }
      #if DEBUG_BLOCKS != 0
      this->textOut ( gsdbg ) ;
      #endif   // DEBUG_BLOCKS
   }

   //* If user interaction is enabled, *
   //* prompt for user's selection.    *
   if ( this->blkFont == cfgSpec )
   {
      //* Point to the text of this line *
      if ( (wi = gsB.after( tagClose, wi )) < ZERO )
         wi = ZERO ;

      //* Ask the user to select a font-size option *
      usrResp = this->ppfBlockPrompt ( blockName[bType], &gsB.gstr()[wi] ) ;

      //* If abort received. Note the early return. *
      if ( this->abort )
      { return ( status = ERR ) ; }
   }
   //* If skip count is active, decrement the counter *
   else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

   //* If embedded token for a "large" block, AND    *
   //* if 'a' (automatic) selected, set large block. *
   if ( lrgFlag && (usrResp == L'a') )
      usrResp = L'l' ;
   else if ( smaFlag && (usrResp == L'a') )
      usrResp = L's' ;

   //* Update the tag according to user's selection *
   if ( pi >= ZERO )
   {
      switch ( usrResp )
      {
         //** inherited font **
         case L'i':  gsB.insert( verbInherit, pi ) ; break ;
         //** smaller font **
         case L's':  gsB.insert( verbSmall, pi ) ;   break ;
         //** larger font **
         case L'l':  gsB.insert( verbLarge, pi ) ;   break ;
         //** Automatic assignment of class       **
         //** If font size previously specified,  **
         //** use it, else use standard font size.**
         case L'a':
         default:
            if ( bType == smaV )
               gsB.insert( verbSmall, pi ) ;
            else if ( bType == lrgV )
               gsB.insert( verbLarge, pi ) ;
            else
               gsB.insert( verbInherit, pi ) ;
            break ;
      } ;
   }
   else
   {
      gsB = gsb ;    // copy line as-written
      status = ERR ; // This may not be necessary
   }

   #if DEBUG_BLOCKS != 0
   gsdbg.compose( "OUT------>>'%S'", gsB.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   //* Write the header line *
   this->ppfWriteLine ( gsB ) ;

   while ( ! done && status == OK )
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
      {
         //* If end of verbatim block reached, capture data after *
         //* the tag (if any), and push it back into the stream.  *
         //* Then write the end tag to target.                    *
         if ( (wi = gss.after( verbEND, wi, false )) >= ZERO )
         {
            done = true ;
            // Programmer's Note: The end-of-verbatim: '</pre>' may be followed 
            // by additional data. Usually this is just the '<br>' or <p> tag, 
            // but may include the end of some other construct such as 
            // '</ul>' or '</ol>. For this reason, we extract any trailing data 
            // and push it back into the stream before writing the verbatim end tag.
            if ( gss.gstr()[wi] != NULLCHAR ) 
            {
               gstmp = &gss.gstr()[wi] ;
               gss.limitChars( wi ) ;
               this->ppfUnReadSrcLine ( gstmp ) ;
            }

            #if DEBUG_BLOCKS != 0
            gsdbg.compose( "END--------'%S'\n", gss.gstr() ) ; this->textOut ( gsdbg ) ;
            #endif   // DEBUG_BLOCKS
         }
         this->ppfWriteLine ( gss ) ;

         // Programmer's Note: Enhancement request:
         // If target is a smallverbatim block which was generated through the 
         // @smallverbatim macro. Scan for the HTML entities: &lbrace; U+0007B 
         // and &rbrace; U+0007D and replace them with curley-brace characters: 
         // '{' and '}', respectively.
      }
   }
   return status ;

}  //* End ppfProcVerbatimBlock() *

//*************************
//*      ppfPFB_List      *
//*************************
//********************************************************************************
//* Called only by ppfProcFormattedBlock(), and ONLY if embedded lists have      *
//* been identified. Removes extra HTML garbage from the lists to make them      *
//* readable.                                                                    *
//*                                                                              *
//* Input  : gsLine : (by reference) contains the line data to be scanned        *
//*                   On return, contains the reformatted data.                  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* 1) If the list is IMMEDIATELY follows the block declaration, the block       *
//* and/or the list it contains may not be recognized. Therefore, ALWAYS place   *
//* a blank line above a list embedded within a preformatted block.              *
//* 2) The source line following the end of the list is an unnecessary           *
//*    re-declaration of the block containing the list:                          *
//*        Example:  <pre class="display">                                       *
//*    This line has no value and should also be processed here.                 *
//********************************************************************************

void Idpp::ppfPFB_List ( gString& gsLine )
{
   const wchar_t* preBEGIN = L"<pre" ;
   const wchar_t* preEND   = L"</pre>" ;

   short wi,         // data indices
         ei ;

   //* If special list processing is enabled *
   if ( this->liSpec )
   {
      if ( (wi = gsLine.compare( preEND, false, 6 )) == ZERO )
      {
         gsLine.erase( preEND, ZERO, false ) ;
      }
      if ( ((wi = gsLine.find( preBEGIN, ZERO, false )) >= ZERO) &&
           ((ei = gsLine.find( tagClose, wi )) > wi ) )
      {
         gsLine.erase( wi, (ei - wi + 1) ) ;
      }
   }
   return ;

}  //* End ppfPFB_List() *

//**************************
//*   ppfProcInnerBlock    *
//**************************
//********************************************************************************
//* Caller has found the beginning of a "pre-formatted" block.                   *
//* These blocks are enclosed within TWO <div> blocks:                           *
//* 1) The outer block declares the block type:                                  *
//*    "format", "display", "example", "lisp" or their small/large variants.     *
//* 2) The inner block declaration is a <pre class"..."> tag, and inappropriately*
//*    causes an extra blank line in the output.                                 *
//*    By removing this meaningless tag, we eliminate the extra blank line.      *
//*    -- Note that it is the caller's responsibility to delete the closing      *
//*       </pre> tag for the block.                                              *
//*                                                                              *
//* If user interaction is enabled, prompt the user for the desired font         *
//* size for the block and replace the provided <div class"..."> tag             *
//* ('gsb' parameter) with the <div class="..."> corresponding to the user's     *
//* selection.                                                                   *
//*                                                                              *
//* Input  : bType  : indicates block type: member of enum blkType               *
//*          gsb    : (by reference) contains the first line of the block.       *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************

short Idpp::ppfProcInnerBlock ( blkType bType, const gString& gsb )
{
   const wchar_t* formInherit = L"<div class=\"format\">" ;
   const wchar_t* formSmall   = L"<div class=\"smallformat\">" ;
   const wchar_t* formLarge   = L"<div class=\"largeformat\">" ;

   const wchar_t* dispInherit = L"<div class=\"display\">" ;
   const wchar_t* dispSmall   = L"<div class=\"smalldisplay\">" ;
   const wchar_t* dispLarge   = L"<div class=\"largedisplay\">" ;

   const wchar_t* examInherit = L"<div class=\"example\">" ;
   const wchar_t* examSmall   = L"<div class=\"smallexample\">" ;
   const wchar_t* examLarge   = L"<div class=\"largeexample\">" ;

   //* Note: The formatting of the <div class="lisp" tag has been    *
   //*       redefined by texi2any as: <div class="example lisp".    *
   //*       The extra "example " will be removed.                   *
   // Makeinfo v:6.1 (old format)
   const wchar_t* lispInherit = L"<div class=\"lisp\">" ;
   const wchar_t* lispSmall   = L"<div class=\"smalllisp\">" ;
   const wchar_t* lispLarge   = L"<div class=\"largelisp\">" ;

   gString gs, gss,           // source line data
           gst, gsIn,         // user interaction
           gsB = gsb,         // working copy of caller's data
           gsdisp,            // display text for user interaction
           gsOut ;            // output data for target file
   wchar_t usrResp = L'a' ;   // user response to prompt ("automatic" is the default)
   short wi,                  // source string index referencing first non-whitespace char
         hi,                  // insertion point for adjusted class tag
         pi ;                 // index of data following "<pre..." tag (if any)
   short status = OK ;        // return value
   bool  lrgFlag = false ;    // set if large block was encoded as a macro

   #if DEBUG_BLOCKS != 0
   gString gsdbg( "GSB at:%03hd '%S' ) ", &this->slCount, gsb.gstr() ) ; this->textOut ( gsdbg ) ;
   #endif   // DEBUG_BLOCKS

   //* Erase the existing tag *
   if ( (hi = gsB.find( L"<div class=" )) >= ZERO )
   {
      removeTag ( gsB, hi ) ;

      #if DEBUG_BLOCKS != 0
      gsdbg.compose( "tx---------'%S'", gsB.gstr() ) ; this->textOut ( gsdbg ) ;
      #endif   // DEBUG_BLOCKS
   }

   if ( (status = this->ppfReadSrcLine ( gss, wi )) == OK )
   {
      #if DEBUG_BLOCKS != 0
      if ( ((gss.find( L"<pre" )) >= ZERO) || ((gss.find( L"</pre" )) >= ZERO) )
      { gsdbg.compose( "GSS at:%hd '%S'", &this->slCount, gss.gstr() ) ; this->textOut ( gsdbg ) ; }
      #endif   // DEBUG_BLOCKS

      //* If line contains a <pre> tag, remove it.    *
      if ( (pi = gss.find( L"<pre", wi, false )) >= ZERO )
      {
         removeTag ( gss, pi ) ;

         #if DEBUG_BLOCKS != 0
         gsdbg.compose( "px---------'%S'", gss.gstr() ) ;
         #endif   // DEBUG_BLOCKS

         //* If line contains an embedded macro *
         //* code for large block, remove it.   *
         if ( (gss.find( lrgCODE )) == ZERO )
         {
            gss.erase( lrgCODE ) ;
            lrgFlag = true ;
            switch ( bType )
            {
               case  smaI: bType = lrgI ;    break ;
               case  smaQ: bType = lrgQ ;    break ;
               case  smaF: bType = lrgF ;    break ;
               case  smaD: bType = lrgD ;    break ;
               case  smaE: bType = lrgE ;    break ;
               case  smaL: bType = lrgL ;    break ;
               default:    bType = lrgV ;    break ;  // (this is unlikely)
            }
            #if DEBUG_BLOCKS != 0
            gsdbg.append( "\nlx---------'%S'", gss.gstr() ) ;
            #endif   // DEBUG_BLOCKS
         }

         //* If gss now contains no text, reset 'pi' *
         if ( gss.gschars() <= 1 ) { pi = -1 ; }
         #if DEBUG_BLOCKS != 0
         this->textOut ( gsdbg ) ;
         #endif   // DEBUG_BLOCKS
      }
      //* If line DOES NOT contain a <pre> tag, then  *
      //* there is nothing to delete, so push it back *
      //* into the stream for caller to process.      *
      else
      {
         this->ppfUnReadSrcLine ( gss ) ;
         gss.clear() ;
      }

      //* If user interaction is enabled, *
      //* prompt for user's selection.    *
      if ( this->blkFont == cfgSpec )
      {
         //* Isolate text of first line for display.  *
         //* If gss contains text, use it, else scan  *
         //* for text in the main tag line. If no     *
         //* display text found, indicate with '?'.   *
         if ( gss.gschars() > 1 )
            gsdisp = gss ;
         else if ( lrgFlag )
         {
            this->ppfReadSrcLine ( gsdisp, pi ) ;
            this->ppfUnReadSrcLine ( gsdisp ) ;
         }
         else
         {
            if ( ((pi = gsB.find( L"<div" )) >= ZERO) &&
                 ((pi = gsB.after( tagClose, pi )) > ZERO ) &&
                 (gsB.gstr()[pi] != NULLCHAR) )
            { gsdisp = &gsB.gstr()[pi] ; }
            else
               gsdisp = blockName[btNone] ;
         }

         //* Ask the user to select a font-size option *
         //* usrResp: a == automatic
         //*          i == inherit
         //*          s == smaller
         //*          l == larger
         usrResp = this->ppfBlockPrompt ( blockName[bType], gsdisp.gstr() ) ;

         //* If abort received. Note the early return. *
         if ( this->abort )
         { return ( status = ERR ) ; }

      }        // user prompt
      //* If skip count is active, decrement the counter *
      else if ( ! iMode && (this->skip > ZERO) ) { this->skipCounter () ; }

      //* If embedded token for a "large" block, AND    *
      //* if 'a' (automatic) selected, set large block. *
      if ( (usrResp == L'a') && lrgFlag )
         usrResp = L'l' ;

      //************************************************
      //* Update the tag according to user's selection *
      //************************************************
      bool repo = false ;     // 'true' if block repositioned

      //*********************
      //** "Format" group  **
      //*********************
      if ( (bType == stdF) || (bType == smaF) ||(bType == lrgF) )
      {
         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( formInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( formSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( formLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaF )
                     gsB.insert( formSmall, hi ) ;
                  else if ( bType == lrgF )
                     gsB.insert( formLarge, hi ) ;
                  else
                     gsB.insert( formInherit, hi ) ;
                  break ;
            } ;
         }
      }

      //*********************
      //** "Display" group **
      //*********************
      else if ( (bType == stdD) || (bType == smaD) || (bType == lrgD) )
      {
         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( dispInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( dispSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( dispLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaD )
                     gsB.insert( dispSmall, hi ) ;
                  else if ( bType == lrgD )
                     gsB.insert( dispLarge, hi ) ;
                  else
                     gsB.insert( dispInherit, hi ) ;
                  break ;
            } ;
         }
      }

      //*********************
      //** "Example" group **
      //*********************
      else if ( (bType == stdE) || (bType == smaE) || (bType == lrgE) )
      {
         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( examInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( examSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( examLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaE )
                     gsB.insert( examSmall, hi ) ;
                  else if ( bType == lrgE )
                     gsB.insert( examLarge, hi ) ;
                  else
                     gsB.insert( examInherit, hi ) ;
                  break ;
            } ;
         }
      }

      //*********************
      //** "Lisp" group    **
      //*********************
      else if ( (bType == stdL) || (bType == smaL) || (bType == lrgL) )
      {
         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( lispInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( lispSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( lispLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaL )
                     gsB.insert( lispSmall, hi ) ;
                  else if ( bType == lrgL )
                     gsB.insert( lispLarge, hi ) ;
                  else
                     gsB.insert( lispInherit, hi ) ;
                  break ;
            } ;
         }
      }
      
      else     // this is unlikely and would indicate programmer error
         status = ERR ;

      if ( status == OK )
      {
         //* Format the data for output.*
         if ( pi >= ZERO )                // index of deleted <pre> tag
         {
            gsOut.compose( L"%S%S", gsB.gstr(), gss.gstr() ) ;
            repo = true ;
         }
         else                             // unmodified source
            gsOut = gsB ;

         #if DEBUG_BLOCKS != 0
         gsdbg.compose( "OUT------>>'%S'\n", gsOut.gstr() ) ; this->textOut ( gsdbg ) ;
         #endif   // DEBUG_BLOCKS

         //* Write the re-formatted data to target *
         this->ppfWriteLine ( gsOut ) ;
         if ( repo == false )
            ++this->tlCount ; // if we have just output 2 lines in one write

         if ( this->verbose != false && repo != false )
         {
            gString gsVerb( "(%4hd) Formatted Block repositioned.", &this->tlCount ) ;
            this->textOut ( gsVerb ) ;
         }
      }
   }        // read source line
   else     // read error - save current line
      this->ppfWriteLine ( gsb ) ;
   return status ;

}  //* End ppfProcInnerBlock() *

//*************************
//*    ppfBlockPrompt     *
//*************************
//********************************************************************************
//* When processing one of the block segments AND when user interaction is       *
//* enabled, prompt the user for the font size to be applied to the block.       *
//*                                                                              *
//* If user's response is: L'A', set the 'blkFont' member to 'cfgAuto'           *
//*                              AND return L'a'.                                *
//*                                                                              *
//* Input  : blkName  : display text identifying the type of block being         *
//*                     processed                                                *
//*          firstLine:                                                          *
//*                                                                              *
//* Returns: user selection:                                                     *
//*             L'a'   automatic (default)                                       *
//*             L'i'   inherit font size from container                          *
//*             L's'   10% smaller than inherited font size                      *
//*             L'l'   10% larger than inherited font size                       *
//********************************************************************************

wchar_t Idpp::ppfBlockPrompt ( const wchar_t* blkName, const wchar_t* firstLine )
{
   gString gst,               // output formatting
           gsIn ;             // user response
   wchar_t usrResp = L'a' ;   // return value

   gst.compose( L"___________________________________\n"
                 "Block: \"%S\" (source line:%hu)\n"
                 "First Line    : %S\n"
                 "Choose font size:\n"
                 "i:inherit (std.)\n"
                 "s:smaller (-10%%)   a:automatic (default)\n"
                 "l:larger  (+10%%)   A:All automatic",
                blkName, &this->slCount, firstLine ) ;
   this->textOut ( gst ) ;
   this->textOut ( L"your choice: ", false ) ;

   while ( ! this->abort )
   {
      this->userResponse ( gsIn ) ;       // talk with the animals

      if ( this->abort )   // if abort token received
      { usrResp = L'a' ; break ; }

      //* If user specified a default response *
      if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
         usrResp = L'a' ;
      else        // single-character response expected
         usrResp = *gsIn.gstr() ;
      //* If this block, and all remaining blocks *
      //* are to be processed automagically       *
      if ( usrResp == L'A' )
      { usrResp = L'a' ; this->blkFont = cfgAuto ; }

      if ( (usrResp == L'a') || (usrResp == L'i') || 
           (usrResp == L's') || (usrResp == L'l') )
      { break ; }     // good input

      //* Call user a dumbguy and ask for correct response.*
      this->invalidResponse () ;
   }     // while()

   return usrResp ;

}  //* End ppfBlockPrompt() *

//*************************
//*      ppfProcGNU       *
//*************************
//********************************************************************************
//* Process the GNU General Public License sub-document.                         *
//*  OR                                                                          *
//* Process the GNU Free Documentation License sub-document.                     *
//*                                                                              *
//* 1) Start the top-level enumerated list at item '0' (zero).                   *
//* 2) Convert the three (3) nested lists in the GPL to 'a.' 'b.' 'c.' lists.    *
//* 3) Convert the one (1) nested list in the FDL to an 'A.' 'B.' 'C.' list.     *
//*                                                                              *
//* Input  : gpl    : 'true' if we are processing the GPL text                   *
//*                   'false' if we are processing the FDL text                  *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* NOTES:                                                                       *
//* -- GPL and FDL license text:                                                 *
//*    We silently correct the enumerated lists in these documents if detected.  *
//*    Because it is boilerplate AND because by law, we as licensees are not     *
//*    allowed to modify it, we can be reasonably confident that it can be       *
//*    correctly identified. For this reason, our tests for the tags are quite   *
//;    simple. However, past or future versions of the licenses MAY cause        *
//*    problems. Also, if the texi-to-HTML converter changes the way it formats  *
//*    <ol> lists, problems _could_ arise.                                       *
//*    -- Note that the list-formatting methods are not called from this         *
//*       method. Instead, we directly modify the list headers. This means       *
//*       that the user is not prompted for these lists.                         *
//* -- The .texi source for the GPL and FDL licenses call out "smallexample"     *
//*    blocks; however, between texi2any v:6.6 and v:7 the "@small...."          *
//*    commands were not recognized. This leaves us with a decision:             *
//*     a) Accept the (lazy-programmer) texi2any mistake and render the blocks   *
//*        in the default font size.                                             *
//*     b) Assume that the .texi source intended "@smallexample" and repair      *
//*        the HTML.                                                             *
//*    This decision is encapsulated by the "SMALL_WORLD" definition below.      *
//*    The flag which indicates the user prompt for block constructs is          *
//*    temporarily disabled, so user will not be prompted to specify these       *
//*    blocks.                                                                   *
//* -- The HTML contains a (centered) version number line, immediately followed  *
//*    by a "display" formatted block. Because we automatically correct the      *
//*    formatting for this block, we must insert a newline between the version   *
//*    number and the "display" block.                                           *
//* -- This code is based on GPL v:3 and FDL v:1.3. These are the current        *
//*    versions of the licenses (2019) and have been stable since 2008.          *
//* -- We aren't sure whether to offer the user a flag to disable these          *
//*    automatic formatting operations; so at this time, we correct the          *
//*    formatting whether the user wants it or not.                              *
//*                                                                              *
//********************************************************************************

short Idpp::ppfProcGNU ( bool gpl )
{
   const wchar_t* gnu0start = L"<ol start=\"0\">" ;
   const wchar_t* gnu0class = L"<ol class=\"enum-decimal\" start=\"0\">" ;
   const wchar_t* gnuastart = L"<ol type=\"a\" start=\"1\">" ;
   const wchar_t* gnuaclass = L"<ol class=\"enum-lower-alpha\" start=\"1\">" ;
   const wchar_t* gnuAstart = L"<ol type=\"A\" start=\"1\">" ;
   const wchar_t* gnuAclass = L"<ol class=\"enum-upper-alpha\" start=\"1\">" ;
   const wchar_t* gnuEnd    = L"The GNU General Public License does not permit "
                               "incorporating your" ;
   const wchar_t* fdlEnd    = L"If your document contains nontrivial examples "
                               "of program code," ;
   const wchar_t* gnuVersion = L"<div align=\"center\">Version" ;

   //* Compensate for texi2any error (see note above). *
   #define SMALL_WORLD (1)
   #if SMALL_WORLD != 0
   const wchar_t* examInherit = L"<div class=\"example\">" ;
   const wchar_t* examSmall   = L"<div class=\"smallexample\">" ;
   Cfg tmpblkFont = this->blkFont ; // remember the global setting
   this->blkFont = cfgAuto ;        // temporarily disable user prompt
   #endif   // SMALL_WORLD

   gString gs ;                     // source data
   short wi ;
   short status = OK ;              // return value
   blkType bType = btNone ;         // type of block found
   bool  done = false ;             // loop control

   do
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* Each license has a "display" block near the top, and *
         //* at least two(2) "example" ("smallexample") blocks.   *
         if ( (this->ppfTestFormattedBlock ( bType, gs )) != false )
         {
            #if SMALL_WORLD != 0
            //* Pretest the formatted block, and if it it is a "example" *
            //* block, convert it to "smallexample" before processing.   *
            if ( (gs.find( examInherit )) >= ZERO )
            {
               gs.replace( examInherit, examSmall ) ;
               bType = smaE ;
            }
            #endif   // SMALL_WORLD

            status = this->ppfProcFormattedBlock ( bType, gs ) ;
            continue ;
         }

         //* If end-of-license found, read through the end *
         //* of the current paragraph.                     *
         if ( ((gs.find( gnuEnd )) >= ZERO) ||
              ((gs.find( fdlEnd )) >= ZERO) )
         {
            do
            {
               this->ppfWriteLine ( gs ) ;   // write the line to target
               if ( (status = this->ppfReadSrcLine ( gs, wi )) != OK )
                  break ;
            }
            while ( (gs.find( L"</p>" )) < ZERO ) ;
            done = true ;
         }

         // Programmer's Note: This is a bit dangerous that the last paragraph 
         // of the license begins with the above text because FSF may change 
         // the text of the license, so we also scan for an "<hr>" which 
         // texi2any uses to end a chapter.
         else if ( (gs.find( L"<hr>" )) >= ZERO )
            done = true ;

         //* If zero-based decimal enumerated list *
         else if ( (wi = (gs.find( gnu0start, ZERO, false ))) >= ZERO )
            gs.replace( gnu0start, gnu0class, wi, false ) ;

         //* If lower-alpha enumerated list *
         if ( (wi = (gs.find( gnuastart, ZERO, true ))) >= ZERO )
            gs.replace( gnuastart, gnuaclass, wi ) ;

         //* If upper-alpha enumerated list *
         else if ( (wi = (gs.find( gnuAstart, ZERO, true ))) >= ZERO )
            gs.replace( gnuAstart, gnuAclass, wi ) ;

         //* If "Version" notice, append a newline (see note above).*
         else if ( ((gs.find( gnuVersion )) >= ZERO) &&
                   ((gs.find( L"<br><br>" )) < ZERO) )
            gs.append( L"<br><br>" ) ;

         this->ppfWriteLine ( gs ) ;   // write the line to target
      }
   }
   while ( !done && (status == OK) ) ;

   #if SMALL_WORLD != 0
   this->blkFont = tmpblkFont ;     // restore the global setting
   #endif   // SMALL_WORLD

   return status ;

}  //* End ppfProcGNU() *

//**************************
//*     ppfTestComment     *
//**************************
//********************************************************************************
//* Test whether the data begin with an open-comment token.                      *
//*                                                                              *
//* Input  : gsln   : (by reference) line of source data to be scanned           *
//*                                                                              *
//* Returns: 'true' if comment identified, else 'false'                          *
//********************************************************************************

bool Idpp::ppfTestComment ( const gString& gsln )
{
   short indx = gsln.scan() ;       // step over leading whitespace
   return ( bool((gsln.find( commBegin, indx )) == indx) ) ;

}  //* End ppfTestComment() *

//**************************
//*     ppfProcComment     *
//**************************
//********************************************************************************
//* Process the comment at the top of 'gsc' and return any portion of the        *
//* data which are not part of the comment.                                      *
//*                                                                              *
//* Input  : gsc    : (by reference)                                             *
//*                                                                              *
//* Returns: 'true'  if full comment processed                                   *
//*                  ('gsc' will contain any non-comment data)                   *
//*          'false' if end-of-comment not found                                 *
//*                  (comment continues on next line)                            *
//********************************************************************************
//* Programmer's Note: The texi2any converter will generate no comments as       *
//* part of the texi-2-html conversion; however, the texi source may contain     *
//* HTML comments similar to:                                                    *
//* @html                                                                        *
//* <!-- The code generated by this section may require                          *
//*      manual post-processing. -->                                             *
//* @end html                                                                    *
//*                                                                              *
//* This author often uses this method during development of the 'idpp'          *
//* post-processor to identify texi-source constructs that may not be            *
//* parsed correctly. For our purposes, all comments are contained on a single   *
//* source line, AND these comments stand alone on the line, and are not         *
//* combined with other HTML tokens.                                             *
//* Multi-line comments or mixed comment/non-comment data may not be properly    *
//* handled by 'idpp'.                                                           *
//********************************************************************************

bool Idpp::ppfProcComment ( gString& gsc )
{
   gString gsOut ;
   short indx ;
   bool  commComplete = false ;

   if ( (indx = gsc.after( commEnd )) >= ZERO )
   {
      gsc.substr( gsOut, ZERO, indx ) ;   // extract the comment substring
      this->ppfWriteLine ( gsOut ) ;      // write the comment substring
      gsc.shiftChars( -(indx) ) ;         // remove substring from source
      commComplete = true ;               // comment fully processed
   }
   else     // write the entire line
   {
      this->ppfWriteLine ( gsc ) ;
      gsc.clear() ;
   }

   return commComplete ;

}  //* End ppfProcComment() *

#if 0    // OBSOLETE - <ul> line-item processing is no longer necessary
//*************************
//*   ppfFormatLineItem   *
//*************************
//********************************************************************************
//* For line items of a UL list that has been declared with:                     *
//*             <ul class="no-bullet">                                           *
//*                                                                              *
//* Scan for the embedded bullet character, and if a recognized bullet           *
//* character is the first non-space character following the <li> tag,           *
//* process it in one of two ways:                                               *
//*  1) Adjust the indentation of the line using a CSS style element.            *
//*     This will cause the line data to be shifted left by one character cell   *
//*     which will give the appearance of a bullet list using the embedded       *
//*     bullet character.                                                        *
//*     -- The embedded bullet character (if identified) is enclosed within a    *
//*        monospace style element to ensure that it fills the entire character  *
//*        space.                                                                *
//*  2) Special processing is performed for four(4) embedded bullet              *
//*     characters:                                                              *
//*       -- the discBullet character   (\x23FA)                                 *
//*       -- the circleBullet character (\x26AC _or_ \x26AA)                     *
//*       -- the squareBullet character (\x25AA)                                 *
//*       -- the "&deg;" (degree symbol) character (\x00B0)                      *
//*     If special automatic beautification is enabled, these characters are     *
//*     deleted from the line item so that the caller may define the list        *
//*     using the appropriate bullet-class definition.                           *
//*                                                                              *
//* -- Note that only a plain "<li>" tag should arrive here.                     *
//*    An <li ...> tag with style or other information has already been          *
//*    processed, and should not be seen here.                                   *
//* -- "no-bullet" line items often contain a completely useless HTML comment.   *
//*    If such a comment is present, it is removed.                              *
//*                                                                              *
//* Input  : gsitem   : (by reference) source line item                          *
//*          liIndex  : data offset where line item <li> begins                  *
//*          eraseBull: (optional, 'false' by default)                           *
//*                     'false' erase embedded bullet characters only for the    *
//*                             special bullets                                  *
//*                     'true'  erase all embedded bullet characters             *
//*                                                                              *
//* Returns: if recognized bullet-item character, return it                      *
//*          if recognized bullet-character definition, return '&'               *
//*          if recognized bullet char not found, return space (L' ').           *
//********************************************************************************

   wchar_t ppfFormatLineItem ( gString& gsitem, short liIndex, bool eraseBull = false ) ;
wchar_t Idpp::ppfFormatLineItem ( gString& gsitem, short liIndex, bool eraseBull )
{
   const wchar_t* monoStyleBegin = L"<span style=\"font-family:monospace;\">" ;
   const wchar_t* monoStyleEnd = L"</span>" ;
   const wchar_t* liOutdent = L"<li style=\"text-indent:-1.0em;\">" ;
   const short    vbcCOUNT = 23 ;
   const wchar_t ValidBulletChars[] = 
   {
      L'\x002D',     // ASCII minus    (-)
      L'\x00B0',     // &deg;          (°)
      L'\x003E',     // &gt;           (>)
      L'\x2022',     // &bullet        (•)
      L'\x2013',     // &ndash;        (–)
      L'\x2014',     // &mdash;        (—)   (macro @EMDASH)
      L'\x2212',     // Unicode minus  (−)   (macro @UMINUS)
      L'\x2660',     // spade          (♠)
      L'\x2663',     // club           (♣)
      L'\x2665',     // heart          (♥)
      L'\x2666',     // diamond        (♦)   (macro @BDIAMOND)
      L'\x25CF',     // black-circle   (●)
      L'\x26AB',     // med-blk-circle (⚫)
      L'\x23FA',     // black-circle   (⏺)   (macro @BDISC)
      L'\x2219',     // math bullet    (∙)
      L'\x25CB',     // white-circle   (○)
      L'\x26AA',     // med-wh-circle  (⚪)
      L'\x26AC',     // med-sm wh cir  (⚬)   (macro @BCIRCLE)
      L'\x25E6',     // white bullet   (◦)
      L'\x25A0',     // black-square   (■)
      L'\x25FC',     // blk-med-square (◼)
      L'\x25FE',     // blk-med-sm-sq  (◾)
      L'\x25AA',     // blk-sm-square  (▪)   (macro @BSQUARE)
   } ;
   const short    vbdCOUNT = 5 ;
   const wchar_t* ValidBulletDefs[vbdCOUNT] =
   {
      L"&deg;",
      L"&gt;",
      L"&bullet;",
      L"&ndash;",
      L"&mdash;",
   } ;
   const short vbdLen[vbdCOUNT] = 
   { 5, 4, 8, 7, 7 } ;

   wchar_t bullChar ;   // captures embedded bullet character
   short wi = gsitem.after( tagClose, liIndex ) ; // index first char after opening tag

   //* If the stupid comment (see above) is present, delete it.*
   // Programmer's Note: If conditional compile set, then comment is removed globally.
   #if REMOVE_STUPID_COMMENT == 0
   if ( this->no_mods == false )
      gsitem.erase( stupidComment, wi ) ;
   #endif   // REMOVE_STUPID_COMMENT

   //* Step over any whitespace that separates the tag from *
   //* the text and copy the embedded bullet character.     *
   wi = gsitem.scan( wi ) ;
   bullChar = gsitem.gstr()[wi] ;

   //* Validate the bullet character
   short   cIndex = ZERO ;
   while ( cIndex < vbcCOUNT )
   {
      //* If captured bullet character is on of the     *
      //* supported characters, enclose it in monospace *
      //* style and adjust indentation of line item.    *
      if ( bullChar == ValidBulletChars[cIndex] )
      {
         if ( this->liSpec && (bullChar == discBullet) )
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = discBullet ;
         }
         else if ( this->liSpec && (bullChar == squareBullet) )
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = squareBullet ;
         }
         else if ( this->liSpec && ((bullChar == degreeSymbol) || 
                   (bullChar == cirtexBullet) || (bullChar == circleBullet)) )
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = circleBullet ;  // alert caller to set "circle-bullet" class
         }
         else if ( eraseBull )      // forced erase of embedded character
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = L' ' ;
         }
         else
         {
            if ( cIndex == ZERO )      // replace ASCII minus with Unicode minus
            {
               bullChar = L'\x2212' ;
               gsitem.replace( ValidBulletChars[cIndex], bullChar, wi ) ;
            }
            gsitem.insert( monoStyleEnd, (wi + 1) ) ;
            gsitem.insert(monoStyleBegin, wi ) ;
            gsitem.replace( liBegin, liOutdent, liIndex ) ;
         }
         break ;
      }
      ++cIndex ;
   }  // while()
   if ( cIndex >= vbcCOUNT )  // bullet character not found
   {
      //* Test for one of the defined HTML character constants *
      if ( bullChar == L'&' )
      {
         cIndex = ZERO ;
         while ( cIndex < vbdCOUNT )
         {
            if ( (gsitem.find( ValidBulletDefs[cIndex], wi )) == wi )
            {
               //* If bullet character is "&deg", erase it.         *
               //* Caller will substitute the "circle-bullet" class *
               if ( this->liSpec && cIndex == ZERO )
               {
                  gsitem.erase( ValidBulletDefs[cIndex], wi ) ;
                  bullChar = circleBullet ;
               }
               else if ( eraseBull )      // forced erase of embedded character
               {
                  gsitem.erase( ValidBulletDefs[cIndex], wi ) ;
                  bullChar = L' ' ;
               }
               else
               {
                  gsitem.insert( monoStyleEnd, (wi + vbdLen[cIndex]) ) ;
                  gsitem.insert(monoStyleBegin, wi ) ;
                  gsitem.replace( liBegin, liOutdent, liIndex ) ;
                  // (embedded bullChar identified above)
               }
               break ;
            }
            ++cIndex ;
         }
      }
      if ( cIndex >= vbdCOUNT )  // bullet character not found, set to space char
      {
         bullChar = L' ' ;
      }
   }

   return bullChar ;

}  //* End ppfFormatLineItem() *
#endif   // OBSOLETE

//*************************
//*  ppfInsertCustomData  *
//*************************
//********************************************************************************
//* Insert user-supplied data at the current position of the target document.    *
//* (existence of the data file has been previously verified)                    *
//*                                                                              *
//* Input  : fileSpec: path/filename specification of source data                *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************

bool Idpp::ppfInsertCustomData ( const gString& fileSpec )
{
   short status = ERR ;
   ifstream dataIn( fileSpec.ustr(), ifstream::in ) ;
   if ( dataIn.is_open() )
   {
      status = OK ;
      gString gsin ;
      while ( (this->ppfReadLine ( dataIn, gsin )) == OK )
         this->ppfWriteLine ( gsin ) ;

      dataIn.close() ;           // close the source file
   }
   return status ;

}  //* End ppfInsertCustomData() *

//*************************
//*    ppfReadSrcLine     *
//*************************
//********************************************************************************
//* Read one line of text from the specified source file,                        *
//* Idpp::slCount is incremented.                                                *
//*                                                                              *
//* Input  : gs     : (by reference) receives text data                          *
//*          windex : (by reference) returned with index of first (wide data)    *
//*                   non-whitespace character on line                           *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************
//* Programmer's Note: When pulling a data line from the 'fifo' object, if       *
//* that line contains only a newline character, it indicates that the           *
//* original source line was an empty line, so remove the newline character.     *
//*                                                                              *
//********************************************************************************

short Idpp::ppfReadSrcLine ( gString& gs, short& windex )
{
   char  tmp[gsDFLTBYTES] ; // UTF-8 line input from file
   short status = ERR ;    // return value
   windex = ZERO ;         // initialize caller's index

   //* If there are data in the pushback buffer, return it. *
   if ( (this->pushBack->pull( gs )) )
   {
      //* If data is only a newline character (see note above).*
      if ( (gs.gschars() == 2) && (gs.gstr()[ZERO] == L'\n') )
         gs.clear() ;
      windex = gs.scan() ;
      status = OK ;
   }

   else
   {
      this->ifs.getline ( tmp, gsDFLTBYTES, NEWLINE ) ; // read a source line
      if ( this->ifs.good() || this->ifs.gcount() > ZERO ) // if a good read
      {
         ++this->slCount ;                         // increment line counter
         gs = tmp ;                                // convert to wide data

         #if REMOVE_STUPID_COMMENT != 0
         if ( this->no_mods == false )
            while ( (gs.erase( stupidComment )) != ERR ) ;
         #endif   // REMOVE_STUPID_COMMENT

         windex = gs.scan() ;                      // ignore leading whitespace
         status = OK ;                             // return with good news

         #if PARA_FIX != 0    // call stripPara()
         if ( ! this->no_mods && ! this->no_para )
            this->paraDel += stripPara ( gs, windex ) ;
         #endif   // PARA_FIX

         //* For debugging output (super verbose) *
         if ( this->scan != false 
              && (this->slCount >= this->scan_beg && 
                  this->slCount <= this->scan_end) )
         {
            gString gss( "SCAN (s:%4hu t:%4hu) '%S'", 
                         &this->slCount, &this->tlCount, gs.gstr() ) ;
            this->textOut ( gss ) ;
         }
      }
   }
   return status ;

}  //* End ppfReadSrcLine() *

//*************************
//*   ppfUnReadSrcLine    *
//*************************
//********************************************************************************
//* Caller has read data from the source input stream that it cannot process.    *
//* Push it back into the source input stream so that it will be available       *
//* for processing at a higher level.                                            *
//*                                                                              *
//* Input  : gs   : data to be written back into the source stream               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: If caller is pushing an empty line back into the stream,  *
//* it means that the original source line contained only a newline.             *
//* Unfortunately, when pulling the line from the queue, the 'fifo' object       *
//* will interpret the empty line as no data.                                    *
//*                                                                              *
//********************************************************************************

void Idpp::ppfUnReadSrcLine ( const gString& gs )
{
   if ( gs.gschars() == 1 )      // if an empty line (see note above)
   {
      gString gstmp( L"\n" ) ;
      this->pushBack->push( gstmp ) ;
   }
   else
      this->pushBack->push( gs ) ;

}  //* End UnReadSrcLine() *

//*************************
//*      ppfReadLine      *
//*************************
//********************************************************************************
//* Read one line of text from the specified input stream.                       *
//* This is a generic ReadLine. See ppfReadSrcLine() for specialized HTML        *
//* source file read.                                                            *
//*                                                                              *
//* Input  : ifs    : open input stream (by reference)                           *
//*          gs     : (by reference) receives text data                          *
//*                                                                              *
//* Returns: OK if successful, ERR processing error                              *
//********************************************************************************

short Idpp::ppfReadLine ( ifstream& ifs, gString& gs )
{
   char  tmp[gsDFLTBYTES] ;      // UTF-8 line input from file
   short status = ERR ;          // return value

   ifs.getline ( tmp, gsDFLTBYTES, NEWLINE ) ;     // read a source line
   if ( ifs.good() || ifs.gcount() > ZERO )         // if a good read
   {
      gs = tmp ;                                // convert to wide data
      status = OK ;                             // return with good news
   }
   return status ;

}  //* End ppfReadLine() *

//*************************
//*     ppfWriteLine      *
//*************************
//********************************************************************************
//* Write a line of data to the target file. the line will be terminated with    *
//* a newline character ('\n'), and the output buffer is flushed.                *
//* Idpp::tlCount is incremented.                                                *
//*                                                                              *
//* Input  : gsOut : (by reference) text to be written.                          *
//*                                                                              *
//* Returns: OK if successful, or ERR if write error.                            *
//*            Note: currently, we don't check for output stream errors.         *
//********************************************************************************

short Idpp::ppfWriteLine ( const gString& gsOut )
{
   short status = OK ;

   if ( this->no_mods == false )
   {
      this->ofs << gsOut.ustr() << endl ;
   }
   ++this->tlCount ;    // increment the output-line counter
   return status ;

}  //* End ppfWriteLine() *

//*************************
//*      ppfScan4Src      *
//*************************
//********************************************************************************
//* Scan the target directory for HTML source documents.                         *
//* Files are selected by filename extension.                                    *
//* Valid extensions:  .html  .htm                                               *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true' if at least one valid file found                             *
//**         'false' if: a) no HTML files found                                  *
//*                      b) directory read error                                 *
//********************************************************************************

bool Idpp::ppfScan4Src ( void )
{
   const wchar_t* extA = L".html" ;    const short extA_len = 6 ;
   const wchar_t* extB = L".htm" ;     const short extB_len = 5 ;
   const wchar_t* extC = L".shtml" ;   const short extC_len = 7 ;
   const wchar_t* extD = L".shtm" ;    const short extD_len = 6 ;
   const wchar_t* extE = L".xhtml" ;   const short extE_len = 7 ;

   DIR*      dirPtr ;            // pointer to open directory file
   deStats   *destat ;           // pointer to entry read from directory file
   gString   fnPath,             // source file's path/filename
             fnName,             // source file's filename
             fnExt ;             // source file's filename extension
   FileStats rawStats ;          // source file's stat info
   bool      status = false ;    // return value

   //* Discard any filenames specified directly on *
   //* command line. This avoids duplicates.       *
   this->sfCount = ZERO ;

   if ( ((this->ppfTargetExists ( this->cwDir, true )) != false) &&
        ((dirPtr = opendir ( this->cwDir.ustr() )) != NULL) )
   {
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* '.' and '..' are not included, but hidden files are *
         if ( (destat->d_name[ZERO] == PERIOD)
              &&
              ((destat->d_name[1] == NULLCHAR) || 
               (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR)) )
         {
            continue ;
         }

         //* 'lstat' the file to determine its type 'regular' files only *
         fnName = destat->d_name ;
         this->ppfCatPathFilename ( fnPath, this->cwDir, fnName.ustr() ) ;
         if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
            {
               this->ppfExtractFileExtension ( fnExt, fnPath.ustr() ) ;
               if ( ((fnExt.gstr()[fnExt.gschars() - 1]) != L'~') && 
                    (   (fnExt.compare( extA, true, extA_len ) == ZERO)
                     || (fnExt.compare( extB, true, extB_len ) == ZERO)
                     || (fnExt.compare( extC, true, extC_len ) == ZERO)
                     || (fnExt.compare( extD, true, extD_len ) == ZERO)
                     || (fnExt.compare( extE, true, extE_len ) == ZERO))
                  )
               {
                  fnName.copy( this->srcFiles[this->sfCount++], gsALLOCDFLT ) ;
                  status = true ;   // at least one valid file found
               }
            }
            else { /* Can't access the file, so ignore it */ }
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }     // opendir()
   return status ;

}  //* End ppfScan4Src()

#if PARA_FIX != 0    // Non-member method
//*************************
//*       stripPara       *
//*************************
//********************************************************************************
//* Non-member Method - Temporary Bug Fix:                                       *
//* ======================================                                       *
//* Compensate for texi2any bug which inserts extraneous '&para;" entities       *
//* in HTML source.                                                              *
//*                                                                              *
//* 1) texi2any v:7.1 contains a bug which inserts the "&para;" entity into      *
//*    heading tags: "<h1> ... </h1>   HTML defines six(6) levels of headings.   *
//*                   <h1>, <h2>, <h3>, <h4>, <h5>. <h6>                         *
//*    and potentially any tag containing a "copiable-link".                     *
//*                                                                              *
//* 2) The 'para' token is a printing character '¶' &#182; but has no function   *
//*    in HTML headings.                                                         *
//*                                                                              *
//* 3) For this reason, we remove all instances of this token associated with    *
//*    the "copiable-link" construct from the HTML file.                         *
//*    This has no effect on the functioning of the HTML syntax; it simply       *
//*    removes the printing character, which is both unnecessary and ugly.       *
//*                                                                              *
//* Input  : srctxt : (by reference) source text line                            *
//*          wi     : (by reference) index of first (wide data) non-whitespace   *
//*                   character in srctxt.                                       *
//*                                                                              *
//*                                                                              *
//* Returns: number of tokens identified and removed (generally either 1 or zero)*
//********************************************************************************

static short stripPara ( gString& srctxt, short wi )
{
   const wchar_t* pHead  = L"<h" ;
   const wchar_t* pLink  = L"<a class=\"copiable-link" ;
   const wchar_t* pToken = L" &para;" ;

   short tokenCount = ZERO ;  // return value

   //* If a target token is identified, remove it. *
   if ( ((srctxt.find( pHead )) == wi ) &&
        ((srctxt.gstr()[wi+2] >= L'1') && (srctxt.gstr()[wi+2] <= L'6')) && 
        ((wi = srctxt.find( pToken, wi, true )) > ZERO) )
   { srctxt.erase( L" &para;", wi ) ; ++tokenCount ; }
   else if ( ((srctxt.find( pLink )) >= wi) &&
             ((wi = srctxt.find( pToken, wi, true )) > ZERO) )
   { srctxt.erase( L" &para;", wi ) ; ++tokenCount ; }

   return tokenCount ;

}  //* End stripPara() *
#endif   // PARA_FIX

//*************************
//*       ppfGetCWD       *
//*************************
//********************************************************************************
//* Returns the path of user's current working directory.                        *
//*                                                                              *
//* Input  : dPath:(by reference, initial value ignored)                         *
//*                receives path string                                          *
//*                                                                              *
//* Returns: OK if successful, ERR if path too long to fit the buffer            *
//********************************************************************************

short Idpp::ppfGetCWD ( gString& dPath )
{
char     buff[MAX_PATH] ;
short    status = OK ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
      dPath = buff ;
   else
      status = ERR ;    // (this is very unlikely)

   return status ;

}  //* End ppfGetCWD() *

//*************************
//*    ppfTargetExists    *
//*************************
//********************************************************************************
//* 'stat' the specified path/filename to see if it exists.                      *
//* Target must be a 'regular' file or optionally, a 'directory' file.           *
//*                                                                              *
//* We also do a simple test for read access: if owner, group, or other has      *
//* read access, we assume that user has read access although this may not be    *
//* true in all cases.                                                           *
//*                                                                              *
//* Input  : fPath: target path/filename                                         *
//*          isDir: (optional, false by default)                                 *
//*                 if 'true', test whether file type is 'regular'               *
//*                 if 'false', test whether file type is 'directory'            *
//*                                                                              *
//* Returns: 'true' if file exists AND is a Regular file, or optionally, a       *
//*                 'directory' file                                             *
//*          'false' otherwise                                                   *
//********************************************************************************

bool Idpp::ppfTargetExists ( const gString& fPath, bool isDir )
{
   bool exists = false ;
   FileStats rawStats ;

   //* If the file exists *
   if ( (lstat64 ( fPath.ustr(), &rawStats )) == OK )
   {  //* Simple, read-access test (see note above) *
      if ( (rawStats.st_mode & S_IRUSR) || (rawStats.st_mode & S_IRGRP) || 
           (rawStats.st_mode & S_IROTH) )
      {
         //* If testing for the existence of a directory *
         if ( isDir != false )
         {
            if ( (S_ISDIR(rawStats.st_mode)) != false )
               exists = true ;
         }
         //* We operate only on 'regular' files *
         else
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
               exists = true ;
         }
      }
   }
   return exists ;

}  //* End ppfTargetExists() *

//*************************
//*    ppfTargetIsHTML    *
//*************************
//********************************************************************************
//* Test the first line of specified file.                                       *
//* A valid HTML (v4 or v5) document must begin with "<!DOCTYPE HTML".           *
//* A valid HTML (v3x) MAY begin with "<HTML".                                   *
//* If neither, then we assume that the document is not valid HTML.              *
//*                                                                              *
//* Input  : fPath: target path/filename                                         *
//*                                                                              *
//* Returns: 'true' if file begins with <!DOCTYPE HTML" OR "<HTML"               *
//*                 (comparison IS NOT case sensitive)                           *
//*          'false' otherwise                                                   *
//********************************************************************************

bool Idpp::ppfTargetIsHTML ( const gString& fPath )
{
   bool status = false ;
   ifstream ifs ( fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      //* Read the first source line *
      gString gs ;
      if ( (status = this->ppfReadLine ( ifs, gs )) == OK )
      {
         if ( ((gs.compare( topDOCT, false, topDOCT_len )) == ZERO)
              ||
              ((gs.compare( topHTML, false, topHTML_len )) == ZERO) )
            status = true ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End ppfTargetIsHTML() *

//*************************
//*    ppfTargetIsCSS     *
//*************************
//********************************************************************************
//* Test whether the specified CSS definition file is valid.                     *
//* a) Verify that the copyright notice is intact.                               *
//* b) Extract the version number string.                                        *
//* If version string not found, then we assume that the document is             *
//* incorrectly formatted                                                        *
//*                                                                              *
//* Input  : fPath: target path/filename                                         *
//*          gsVer: receives version number string                               *
//*                                                                              *
//* Returns: 'true' if file format verified                                      *
//*          'false' otherwise                                                   *
//********************************************************************************

bool Idpp::ppfTargetIsCSS ( const gString& fPath, gString& gsVer )
{
   bool status = false ;
   ifstream ifs ( fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      gString gs ;
      short i ;
      bool  cright_found = false, done = false ;

      while ( !done )
      {
         //* Read a source line *
         if ( (status = this->ppfReadLine ( ifs, gs )) == OK )
         {
            if ( !cright_found )
            {
               for ( i = ZERO ; gs.gstr()[i] != *cssCOPY && gs.gstr()[i] != NULLCHAR ; i++ ) ;
               if ( (gs.gstr()[i] == *cssCOPY) &&
                       ((gs.compare( cssCOPY, true, cssCOPY_len, i )) == ZERO) )
               {
                  gs.shiftChars( (ZERO - i) ) ;
                  for ( i = ZERO ; gs.gstr()[i] != *cssSAMU && gs.gstr()[i] != NULLCHAR ; i++ ) ;
                  if ( (gs.gstr()[i] == *cssSAMU) && 
                       ((gs.compare( cssSAMU, true, cssSAMU_len, i )) == ZERO) )
                  {
                     cright_found = true ;
                  }
               }
            }
            else
            {
               for ( i = ZERO ; gs.gstr()[i] != *cssVERS && gs.gstr()[i] != NULLCHAR ; i++ ) ;
               if ( (gs.gstr()[i] == *cssVERS) &&
                       ((gs.compare( cssVERS, true, cssVERS_len, i )) == ZERO) )
               {
                  gs.shiftChars( (ZERO - i) ) ;
                  gs.limitChars( CSS_VER_LEN ) ;
                  gsVer = gs ;
                  done = status = true ;
               }
            }
         }
         else
            done = true ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End ppfTargetIsCSS() *

//*************************
//*  ppfCatPathFilename   *
//*************************
//********************************************************************************
//* Concatenate a path string with a filename string to create a path/filename   *
//* specification.                                                               *
//*                                                                              *
//* Input  : pgs  : gString object (by reference) to hold the path/filename      *
//*                 On return, pgs contains the path/filename string in both     *
//*                 UTF-8 and wchar_t formats.                                   *
//*          wPath: gString object (by reference) path string                    *
//*          uFile: pointer to UTF-8 filename string                             *
//*               OR                                                             *
//*          wFile: pointer to wchar_t filename string                           *
//*                                                                              *
//* Returns: OK if success                                                       *
//*          ERR if string truncated                                             *
//********************************************************************************

short Idpp::ppfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile )
{
short    success = OK ;

   pgs.compose( L"%S/%s", wPath.gstr(), uFile ) ;
   if ( pgs.gschars() >= (gsALLOCDFLT-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End ppfCatPathFilename() *

short Idpp::ppfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile )
{
short    success = OK ;

   pgs.compose( L"%S/%S", wPath.gstr(), wFile ) ;
   if ( pgs.gschars() >= (gsALLOCDFLT-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End ppfCatPathFilename() *

//*************************
//*     ppfCdTarget      *
//*************************
//********************************************************************************
//* Change the current working directory to specified target path.               *
//*                                                                              *
//* Input  : dirPath: path of new CWD (relative or absolute)                     *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//********************************************************************************

short Idpp::ppfCdTarget ( const char* dirPath )
{
   short status ;       // return value

   //* If a relative path, convert to absolute path *
   char tmpPath[gsDFLTBYTES] ;
   realpath ( dirPath, tmpPath ) ;
   gString targPath( tmpPath ) ;

   //* Change our current working directory and update CWD data member *
   if ( (status = chdir ( targPath.ustr() )) == OK )
      this->ppfGetCWD ( this->cwDir ) ;
   return status ;

}  //* End ppfCdTarget() *

//*************************
//*    ppfDeleteFile      *
//*************************
//********************************************************************************
//* Delete the specified file.                                                   *
//*                                                                              *
//* Input  : trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//********************************************************************************

short Idpp::ppfDeleteFile ( const gString& trgPath )
{

   return ( unlink ( trgPath.ustr() ) ) ;

}  //* End ppfDeleteFile() *

//*************************
//*    ppfRenameFile      *
//*************************
//********************************************************************************
//* Rename the specified file.                                                   *
//*                                                                              *
//* Input  : srcPath : full path/filename specification of source                *
//*          trgPath : full path/filename specification of target                *
//*                                                                              *
//* Returns: OK if successful, else ERR                                          *
//********************************************************************************
//* For the rename() primitive: If srcPath refers to a symbolic link the         *
//* link is renamed; if trgPath refers to a symbolic link the link will be       *
//* overwritten.                                                                 *
//********************************************************************************

short Idpp::ppfRenameFile ( const gString& srcPath, const gString& trgPath )
{

   return ( rename ( srcPath.ustr(), trgPath.ustr() ) ) ;

}  //* End ppfRenameFile() *

//*************************
//*     ppfRealpath       *
//*************************
//********************************************************************************
//* Decode the relative or aliased path for the specified target.                *
//*                                                                              *
//*                                                                              *
//* Input  : realPath : (by reference) receives decoded path specification       *
//*          rawPath  : (by reference) caller's path specification               *
//*                                                                              *
//* Returns: OK  if successful                                                   *
//*          ERR if invalid path or if system call fails ('realPath' unchanged)  *
//********************************************************************************
//* Programmer's Note: The 'realpath' library function cannot handle a           *
//* symboloc substitution such as: ${HOME}/Docs/filename.txt                     *
//* It returns: /home/Sam/Docs/(CWD)/${HOME}                                     *
//* This is a little bit nuts, but we don't have time to investigate it.         *
//*                                                                              *
//********************************************************************************

short Idpp::ppfRealpath ( gString& realPath, const gString& rawPath )
{
   short status = ERR ;

   const char* rpath ;
   if ( (rpath = realpath ( rawPath.ustr(), NULL )) != NULL )
   {
      realPath = rpath ;
      free ( (void*)rpath ) ;    // release the temp storage
      status = OK ;
   }
   return status ;

}  //* End ppfRealpath() *

//*************************
//*  ppfExtractFilename   *
//*************************
//********************************************************************************
//* Extract the filename from the path/filename provided.                        *
//*                                                                              *
//* Input  : gsName: (by reference, initial contents ignored)                    *
//*                  receives the extracted filename                             *
//*          fPath : path/filename string                                        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::ppfExtractFilename ( gString& gsName, const gString& fPath )
{
   const wchar_t* fptr = fPath.gstr() ;
   short findex = fPath.gschars() - 1 ;
   while ( findex > ZERO && fptr[findex] != SLASH )
      --findex ;
   if ( fptr[findex] == SLASH )
      ++findex ;
   gsName = &fptr[findex] ;

}  //* End ppfExtractFilename() *

//***************************
//* ppfExtractFileExtension *
//***************************
//********************************************************************************
//* Extract the filename extension (including the '.') from the path/filename    *
//* provided.                                                                    *
//*                                                                              *
//* Input  : gsName: (by reference, initial contents ignored)                    *
//*                  receives the extracted filename                             *
//*          fName : filename or path/filename string                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::ppfExtractFileExtension ( gString& gsExt, const char* fName )
{
   wchar_t tmpName[gsALLOCDFLT] ;   // convert to wide character string
   gsExt = fName ;
   gsExt.copy( tmpName, gsALLOCDFLT ) ;

   short   tpIndex = gsExt.gschars() - 1 ; // index the NULLCHAR
   while ( tpIndex > ZERO && tmpName[tpIndex] != PERIOD ) // isolate the extension
      --tpIndex ;
   if ( tmpName[tpIndex] == PERIOD )
      gsExt = &tmpName[tpIndex] ;
   else
      gsExt = L"" ;

}  //* End ppfExtractFileExtension() *

//*************************
//*       stripTags       *
//*************************
//********************************************************************************
//* Non-member method                                                            *
//* -----------------                                                            *
//*                                                                              *
//* Remove all HTML tags from the data.                                          *
//* Called by the list processing methods to create first-line display text for  *
//* the user prompt.                                                             *
//*                                                                              *
//* Input  : disp : (by reference) data to be scanned and modified               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void stripTags ( gString& disp )
{
   short bi, ei ;
   while ( (bi = disp.find( L'<' )) >= ZERO )
   {
      if ( (ei = disp.after( L'>', false, bi )) > bi )
         disp.erase( bi, (ei - bi) ) ;
   }
}  //* End stripTags() *

//*************************
//*       removeTag       *
//*************************
//********************************************************************************
//* Non-member method                                                            *
//* -----------------                                                            *
//*                                                                              *
//* Remove the HTML tag referenced by the tag-index parameter ('ti').            *
//*                                                                              *
//* Input  : gs   : (by reference) contains the text to be modified              *
//*          ti   : index of first character of the tag substring                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void removeTag ( gString& gs, short ti )
{
   short te ;                             // index end-of-tag + 1

   if ( (te = gs.after( tagClose, ti )) > ti )
      gs.erase( ti, (te - ti) ) ;

}  //* End removeTag()

//*************************
//*        copyTag        *
//*************************
//********************************************************************************
//* Non-member method                                                            *
//* -----------------                                                            *
//*                                                                              *
//* Copy the indexed HTML tag in the source string to target.                    *
//* This is used to customize tags for bullet lists, etc.                        *
//*                                                                              *
//* Input  : gsSrc : (by reference) contains source data                         *
//*          wi    : character index of first character in the tag               *
//*          gsTrg : (by reference) receives a copy of the indexed tag           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void copyTag ( const gString& gsSrc, short ti, gString& gsTrg ) 
{
   short te ;                             // index end-of-tag + 1
   if ( (te = gsSrc.after( tagClose, ti )) > ti )
      gsTrg.loadChars( &gsSrc.gstr()[ti], (te - ti + 1) ) ;
   else     // (this is unlikely)
      gsTrg.clear() ;

}  //* End copyTag() *

#undef DEBUG_FLOW // disable debugging directive
   
//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

