#!/usr/bin/perl
use strict ;
use warnings ;

# ******************************* #
# FileMangler Installation Script #
# ******************************* #
# Updated: 15-May-2025
# Author : Mahlon R. Smith
#          Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai
#              GNU GPL copyright notice located in FileMangler.hpp
# Developed under: Perl 5.18.4      Current: Perl 5.40.2
# 
# Optional command-line argument specifies alternate installation directory.
#  If no command-line arguments specified, installation directory will be:
#     ${HOME}/Apps/FileMangler
#  If valid alternate directory path specified, installation directory will be:
#     SPECIFIED_PATH/FileMangler
# Example: ./fmginstall.pl $HOME
#          yields an installation directory of: /home/username/FileMangler
# Example: ./fmginstall.pl /usr/local/utils
#          yields an installation directory of: /usr/local/utils/FileMangler
# 
# To test the compile without executing: perl -cw fmginstall.pl

# Programmer's Note: Without prototypes, the compiler complains when using #
# the "-cw" (compile-only/warnings) switches. Perl "prototypes" actually   #
# aren't; instead, they are context templates which are in most respects,  #
# pretty fracking useless.                                                 #
sub alt_install_dir () ;
sub copy_source_to_target () ;
sub identify_script_file_path () ;
sub copy_script_files_to_path ( $ ) ;
sub install_infodocs () ;
sub create_mem_function () ;
sub external_utils () ;
sub report_success () ;


my $FMGHOME = "$ENV{HOME}" ;  # Begin in user's home directory
my $FMGFMG  = "" ;            # Installation directory
my $Success = 0 ;             # Track our success
my $BRCsuccess = 0 ;          # non-zero if '.bashrc' file contains 'fmg' function
my $DOCsuccess = 0 ;          # non-zero if info docs installed
my $UTLsuccess = 0 ;          # 0==utils installed, 1==no 'wl-clipboard' -1==no 'gio'

# Declare filename of each source file #
my $fmgr  = "FileMangler" ;   # (also the application name and install dir name)
my $fmgc  = "FmConfig" ;
my $cfg   = "FileMangler.cfg" ;
my $kcfg  = "FmKeymap.cfg" ;
my $bdef  = "Backup_Def.txt" ;
my $info  = "filemangler.info" ;
my $html  = "filemangler.html" ;
my $css   = "infodoc-styles.css" ;
my $fmg   = "fmg" ;
my $fmgsh = "fmg.sh" ;

   use Cwd qw ( getcwd ) ;
   use Term::ANSIColor;
   system ( "clear" ) ;
   print   "Installing FileMangler",
         "\n----------------------" ;

   # If user specifies an alternate installation base directory, #
   # the intermediate 'Apps' directory is not created and        #
   #          $FMGFMG == alt-target-path/FileMangler             #
   if ( defined $ARGV[0] )
   {
      # Note: If user-supplied directory does not exist or is    #
      # not a directory, then exit here.                         #
      if ( (alt_install_dir ()) != 0 )
      { exit ; }
   }

   # If no alternate installation base directory specified, #
   # create the default installation path.                  #
   if ( $FMGFMG eq '' )
   {
      my $FMGAPPS    = "$FMGHOME/Apps" ;     # parent of installation directory
      if ( ! -e $FMGAPPS )
      { mkdir $FMGAPPS ; }
      $FMGFMG     = "$FMGAPPS/$fmgr" ;       # installation directory
      if ( ! -d $FMGFMG )
      { mkdir $FMGFMG ; }
   }

   # Give user a clue #
   print colored ( "\nInstallation directory: ", 'blue' ) ;
   print colored ( "$FMGFMG", 'green' ) ;

   # Verify that subdirectory tree was created #
   if ( ! -d $FMGFMG )
   {
      print colored ( "\n\nERROR! - Unable to access installation directory.\n\n", 
                      'bright_red' ) ;
      exit ;
   }

   $Success += 1 ;      # increment our success (1)
   print colored ( "\nFileMangler Installation Directory Created Successfully!\n", 
                   'blue' ) ;

   # Create target filespecs #
   my $FMGfmgr  = "$FMGFMG/$fmgr" ;
   my $FMGfmgc  = "$FMGFMG/$fmgc" ;
   my $FMGcfg   = "$FMGFMG/$cfg" ;
   my $FMGkcfg  = "$FMGFMG/$kcfg" ;
   my $FMGbdef  = "$FMGFMG/$bdef" ;
   my $FMGinfo  = "$FMGFMG/$info" ;
   my $FMGhtml  = "$FMGFMG/$html" ;
   my $FMGcss   = "$FMGFMG/$css" ;
   my $FMGfmg   = "$FMGFMG/$fmg" ;
   my $FMGfmgsh = "$FMGFMG/$fmgsh" ;

   # Copy application files from current directory #
   # to target directory.                          #
   # Existing targets will be renamed as backup.   #
   if ( (copy_source_to_target ()) != 0 )
   { exit ; }
   $Success += 1 ;      # increment our success (2)

   # Locate directory on execution path where shell scripts will be      #
   # installed, then copy the two shell scripts to the target directory. #
   # Note: If non-default install directory, scripts will be edited to   #
   # reflect actual install directory.                                   #
   my $PathTrg = identify_script_file_path () ;
   if ( $PathTrg ne "" )
   {
      if ( (copy_script_files_to_path ( $PathTrg )) != 0 )
      { exit ; }
   }
   else
   { exit ; }
   $Success += 1 ;      # increment our success (3)

   # Install the Info-reader documentation into the Info database. #
   # Note: If operation fails, it is not a fatal error.            #
   my $FMGinfodir = "" ;                     # documentation target directory
   $DOCsuccess = install_infodocs () ;
   $Success += 1 ;      # increment our success (4)

   # If .bashrc file does not contain the invocation function, #
   # then insert function at end of file.                      #
   # Note: If operation fails, it is not a fatal error;        #
   # however, if 'bash' is not the user's default shell, then  #
   # we must rely upon the user to manually enter the function.#
   my $bashrc = "$FMGHOME/.bashrc" ;
   $BRCsuccess = create_mem_function () ;
   $Success += 1 ;      # increment our success

   # Check availability of the necessary external utilities:   #
   # 1) 'gio' (Gnome Input/Output) is needed for access to     #
   #    USB buss information, access to "virtual filesystems", #
   #    i.e. MTP/GVfs smartphones, and management of           #
   #    optical-media drives (DVD/CD/Blu-ray). While the       #
   #    application will build and run without 'gio', access   #
   #    to smartphone data will be much slower and less stable.#
   #    Some minor functionality will be lost entirely.        #
   # 2) 'wl-clipboard' ('wl-copy' and 'wl-paste') are needed   #
   #    only for access to the system clipboard.               #
   #    No other functionality is affected.                    #
   $UTLsuccess = external_utils () ;
   $Success += 1 ;      # increment our success

   report_success () ;  # Report installation results.

   print "\n\n" ;       # Say Goodnight, Gracie.

#****************************************#
#*** END OF MAINLINE COMMAND SEQUENCE ***#
#****************************************#


# **************************** #
# Alternate installation       #
# directory specified          #
# **************************** #
sub alt_install_dir ()
{
   my $altTarget = $ARGV[0] ;
   if ( (! -e $altTarget || ! -d $altTarget) )
   {
      print colored ( "\nThe specified target: \"$altTarget\"", 'bright_red' ) ;
      print colored ( "\ndoes not exist or is not a directory.\n\n", 'bright_red' ) ;
      return 1 ;
   }

   # Valid target directory specified #
   $FMGFMG = "$altTarget/$fmgr" ;
   if ( ! -d $FMGFMG )
   { mkdir $FMGFMG ; }
   if ( ! -d $FMGFMG )     # if mkdir failed
   { return 1 ; }
   return 0 ;

}  # alt_install_dir()


# **************************** #
# Copy application files to    #
# target directory.            #
# **************************** #
sub copy_source_to_target ()
{
   # If targets exist, rename them as backup.   #
   # (any existing backups will be overwritten) #
   if ( -e $FMGfmgr || -e $FMGfmgc || -e $FMGcfg  || -e $FMGkcfg || -e $FMGbdef || 
        -e $FMGinfo || -e $FMGhtml || -e $FMGcss  || -e $FMGfmg  || -e $FMGfmgsh )
   {
      print colored ( "\nRename existing targets . . .", 'blue' ) ;
      use File::Copy qw(move) ;

      if ( -e $FMGfmgr )
      { move $FMGfmgr, "$FMGfmgr~" ; }
      if ( -e $FMGfmgc )
      { move $FMGfmgc, "$FMGfmgc~" ; }
      if ( -e $FMGcfg )
      { move $FMGcfg, "$FMGcfg~" ; }
      if ( -e $FMGkcfg )
      { move $FMGkcfg, "$FMGkcfg~" ; }
      if ( -e $FMGbdef )
      { move $FMGbdef, "$FMGbdef~" ; }
      if ( -e $FMGinfo )
      { move $FMGinfo, "$FMGinfo~" ; }
      if ( -e $FMGhtml )
      { move $FMGhtml, "$FMGhtml~" ; }
      if ( -e $FMGcss )
      { move $FMGcss, "$FMGcss~" ; }
      if ( -e $FMGfmg )
      { move $FMGfmg, "$FMGfmg~" ; }
      if ( -e $FMGfmgsh )
      { move $FMGfmgsh, "$FMGfmgsh~" ; }

      print colored ( " OK", 'blue' ) ;
   }

   print colored ( "\nCopying the Files . . .", 'blue' ) ;
   system ( "cp --preserve=all './$fmgr' '$FMGfmgr'" ) ;
   print "\n ==> $FMGfmgr" ;
   system ( "cp --preserve=all './$fmgc' '$FMGfmgc'" ) ;
   print "\n ==> $FMGfmgc" ;
   system ( "cp --preserve=all './$cfg' '$FMGcfg'" ) ;
   print "\n ==> $FMGcfg" ;
   system ( "cp --preserve=all './$kcfg' '$FMGkcfg'" ) ;
   print "\n ==> $FMGkcfg" ;
   system ( "cp --preserve=all './$bdef' '$FMGbdef'" ) ;
   print "\n ==> $FMGbdef" ;
   system ( "cp --preserve=all './$fmg' '$FMGfmg'" ) ;
   print "\n ==> $FMGfmg" ;
   system ( "cp --preserve=all './$fmgsh' '$FMGfmgsh'" ) ;
   print "\n ==> $FMGfmgsh" ;
   system ( "cp --preserve=all './Texinfo/$info' '$FMGinfo'" ) ;
   print "\n ==> $FMGinfo" ;
   system ( "cp --preserve=all './Texinfo/$html' '$FMGhtml'" ) ;
   print "\n ==> $FMGhtml" ;
   system ( "cp --preserve=all './Texinfo/$css' '$FMGcss'" ) ;
   print "\n ==> $FMGcss" ;

   # Verify that all files copied successfully #
   if ( -e $FMGfmgr && -e $FMGfmgc && -e $FMGcfg && -e $FMGkcfg && -e $FMGbdef && 
        -e $FMGinfo && -e $FMGhtml && -e $FMGcss && -e $FMGfmg  && -e $FMGfmgsh )
   {
      print colored ( "\nApplication Files Copied Successfully!\n", 'blue' ) ;
      return 0 ;     # return success
   }
   else
   {
      print colored ( "\n\nERROR! - One or more source files not copied to installation directory.\n\n", 'bright_red' ) ;
      return 1 ;     # return failure
   }

}  # copy_source_to_target()


# *************************************************** #
# Parse the $PATH environment variable and return the #
# path of the directory for storing the script files. #
#   1st Choice: $HOME/bin                             #
#   2nd Choice: $HOME/.local/bin                      #
#   3rd Choice: any directory below $HOME             #
# *************************************************** #
sub identify_script_file_path ()
{
   use Text::ParseWords ;
   my $ExPath = $ENV{PATH} ;
   my @PathDirs = &parse_line ( ':', 0, $ExPath ) ;
   my $HomeDir = $ENV{HOME} ;
   my $Ptrg = "" ;
   #print "\nPATH: $ExPath" ;  # FOR DEBUGGING ONLY
   #print "\nHOME: $HomeDir" ; # FOR DEBUGGING ONLY
   #print "\nPTRG: $Ptrg" ;    # FOR DEBUGGING ONLY

   foreach my $i ( 0 .. $#PathDirs )
   {
      if ( (index ( "$PathDirs[$i]", "$HomeDir" )) == 0 )
      {
         #print colored ( "\n$i: $PathDirs[$i]", 'blue' ) ;  # FOR DEBUGGING ONLY
         if ( (index ("$PathDirs[$i]", "$HomeDir/bin")) == 0 )
         {
            #print colored ( "\n   $PathDirs[$i]", 'bright_green' ) ; # FOR DEBUGGING ONLY
            if ( -e $PathDirs[$i] )
            { $Ptrg = $PathDirs[$i] ; }
         }
         elsif ( (index ("$PathDirs[$i]", "$HomeDir/.local/bin")) == 0 )
         {
            #print colored ( "\n   $PathDirs[$i]", 'bright_blue' ) ; # FOR DEBUGGING ONLY
            if ( $Ptrg eq "" && -e $PathDirs[$i] )
            { $Ptrg = $PathDirs[$i] ; }
         }
      }
      #else { print "\n$i: $PathDirs[$i]" ; } # FOR DEBUGGING ONLY
   }

   # if path contains neither preferred target
   if ( $Ptrg eq "" && -e "$HomeDir/bin" )      
   { $Ptrg = "$HomeDir/bin" ; }
   #print colored ( "\nPTRG: $Ptrg\n", 'green' ) ; # FOR DEBUGGING ONLY

   if ( $Ptrg eq "" )
   { print colored ( "\nError! Unable to parse execution path.\n\n", 'bright_red' ) ; }

   return $Ptrg ;

}  # identify_script_file_path


# ******************************** #
# Copy script files to spcified    #
# directory on the execution path. #
# ******************************** #
sub copy_script_files_to_path ( $ )
{
   # Notes: 
   # a) Caller has verified that source files  #
   #    exist in the installation directory;   #
   #    however, we verify the permission bits.#
   # b) Caller has verified that the target    #
   #    directory is on the execution path AND #
   #    that the target directory exists.      #

   my $Ptrg = shift ;
   print colored ( "\nShell scripts '$fmg' and '$fmgsh' will be copied to PATH directory:", 'blue' ) ;
   print colored ( "\n  '$Ptrg'", 'green' ) ;

   if ( -e $Ptrg )
   {
      # Set source script files as executable.#
      # Note: We ASSUME user has source write #
      # permisison and that chmod won't fail. #
      my $octal_mode = 0755 ;    # user:rwx, group:r-x, other:r-x
      my $mode = (stat ($FMGfmg))[2] & 07777 ;
      if ( $mode != $octal_mode )
      { chmod $octal_mode, $FMGfmg ; }
      $mode = (stat ($FMGfmgsh))[2] & 07777 ;
      if ( $mode != $octal_mode )
      { chmod $octal_mode, $FMGfmgsh ; }
   
      my $fmgTarget   = "$Ptrg/$fmg" ;       # 'fmg' target
      my $fmgshTarget = "$Ptrg/$fmgsh" ;     # 'fmg.sh' target
      #print "\nfmgSource: '$FMGfmg'" ;       # FOR DEBUGGING ONLY
      #print "\nfmgTarget: '$fmgTarget'" ;    # FOR DEBUGGING ONLY
      #print "\nfmgshSource: '$FMGfmgsh'" ;   # FOR DEBUGGING ONLY
      #print "\nfmgshTarget: '$fmgshTarget'" ;# FOR DEBUGGING ONLY

      # If targets exist, rename them as backup.   #
      # (any existing backups will be overwritten) #
      if ( -e $fmgTarget || -e $fmgshTarget )
      {
         print colored ( "\nRename existing targets . . .", 'blue' ) ;
         use File::Copy qw(move) ;
         if ( -e $fmgTarget )
         {
            move $fmgTarget, "$fmgTarget~" ;
            #print "\n $fmg ==> $fmg~" ;         # FOR DEBUGGING ONLY
         }
         if ( -e $fmgshTarget )
         {
            move $fmgshTarget, "$fmgshTarget~" ;
            #print "\n $fmgsh ==> $fmgsh~" ;     # FOR DEBUGGING ONLY
         }

         print colored ( " OK", 'blue' ) ;
      }

      # Copy script files to target directory #
      print colored ( "\nCopying the Files . . .", 'blue' ) ;
      system ( "cp --preserve=all $FMGfmg $fmgTarget" ) ;
      print "\n ==> $fmgTarget" ;
      system ( "cp --preserve=all $FMGfmgsh $fmgshTarget" ) ;
      print "\n ==> $fmgshTarget" ;
      if ( ! -e $fmgTarget || ! -e $fmgshTarget )
      {
         print colored ( "\n  Error! - Unable to copy script files ", 'bright_red' ) ;
         print colored ( "to target directory.\n\n", 'bright_red' ) ;
         return 1 ;
      }

      # If not default installation directory, modify   #
      # target scripts to indicate actual target.       #
      if ( "$FMGFMG" ne "$ENV{HOME}/Apps/$fmgr" )
      {
         # The salient line in the script file is the      #
         # install directory declaration and begins with:  #
         my $idHead = "export FMG_HOME=" ;
         # and ends with:                                  #
         my $idTail = "   # installation directory\n" ;

         # Edit 'fmg' script #
         my $src_fmg = "$fmgTarget\_src" ;
         use File::Copy qw(move) ;
         move $fmgTarget, $src_fmg ;

         # Open the target file, then the source file. #
         open ( my $trghandle, '>:encoding(UTF-8)', $fmgTarget ) ;
         if ( (open ( my $srchandle, '<:encoding(UTF-8)', $src_fmg )) )
         {
            while ( my $row = <$srchandle> )
            {
               if ( (index ( "$row", $idHead )) == 0 )
               {
                  $row = "$idHead'$FMGFMG'$idTail" ;
               }
               print $trghandle "$row" ;     # write line to target
            }
            close ( $srchandle ) ;
            close ( $trghandle ) ;
         }

         # Delete the obsolete source file. #
         unlink $src_fmg ;

         # Edit 'fmg.sh' script #
         my $src_fmgsh = "$fmgshTarget\_src" ;
         use File::Copy qw(move) ;
         move $fmgshTarget, $src_fmgsh ;

         # Open the target file, then the source file. #
         open ( $trghandle, '>:encoding(UTF-8)', $fmgshTarget ) ;
         if ( (open ( my $srchandle, '<:encoding(UTF-8)', $src_fmgsh )) )
         {
            while ( my $row = <$srchandle> )
            {
               if ( (index ( "$row", $idHead )) == 0 )
               {
                  $row = "$idHead'$FMGFMG'$idTail" ;
               }
               print $trghandle "$row" ;     # write line to target
            }
            close ( $srchandle ) ;
            close ( $trghandle ) ;
         }

         # Delete the obsolete source file. #
         unlink $src_fmgsh ;

         print colored ( "\nInvocation scripts updated to reflect ", 'blue' ) ;
         print colored ( "new installation directory.", 'blue' ) ;
         print colored ( "\n  '$FMGFMG'", 'green' ) ;
      }

      return 0 ;
   }
   else
   {
      print colored ( "\n  Error!: Target directory not accessible.\n\n", 'bright_red' ) ;
      return 1 ;
   }

}  # copy_script_files_to_path()


# ***************************** #
# Install Texinfo documentation #
# into the Info-reader database #
# ***************************** #
# Notes:
# From the manual:--name=TEXT
# "Specify the name portion of the menu entry.  If the TEXT does not
# start with an asterisk '*', it is presumed to be the text after the
# '*' and before the parentheses that specify the Info file. Otherwise
# TEXT is taken verbatim, and is taken as defining the text up to and 
# including the first period (a space is appended if necessary).
# 1) There are two(2) parts to an entry, the Name and the Description.
# 2) --name='FileMangler' references the entry in the source document 
#    specified by the --info-file parameter. The portion within the 
#    parentheses is the base name of the --info-file parameter.
#    The description portion of the file entry is used as written.
#    a) Duplicate entries will not be created. Only if the file's 
#       entry is _different_ from the existing entry will a second 
#       entry be created. Thus, unless we change the format of the 
#       entry in the file, there is no need to worry about duplicates.
# 3) --name='* FileMangler (fmg)." is taken as a literal value for the 
#    portion of the entry up to and including the first '.'. In this 
#    case, the left side of the entry in the file will be ignored, and 
#    only the description portion of the file entry will be used.
#    a) Note that because the "(fmg)" portion of the literal does not 
#       refer to an actual file, the --remove command will not work; 
#       neither for:  --name='FileMangler' 
#       nor for:      --name='* FileMangler (fmg)."
sub install_infodocs ()
{
   # Sequence:                                                     #
   #  a) Locate top-level Texinfo database file.                   #
   #  b) Copy FileMangler doc file to info directory.              #
   #  c) Set CWD to target dir and verify that copy was successful.#
   #  d) Back up the 'dir' database file.                          #
   #  e) Update the database file to include FileMangler docs.     #
   # Returns 1 on success, 0 if operation fails.                   #

   my $FMGlocdir  = "/usr/local/share/info" ;   # documentation target directory (local)
   my $FMGloctrg  = "$FMGlocdir/dir" ;          # documentation target database  (local)
   my $FMGglodir  = "/usr/share/info" ;         # documentation target directory (global)
   my $FMGglotrg  = "$FMGglodir/dir" ;          # documentation target database  (global)
   my $FMGinfobak = "" ;                        # documentation target database (backup)
   my $FMGinfotrg = "" ;                        # documentation target filespec

   print colored ( "\n\n** Install the Texinfo Documentation. **", 'blue' ) ;
   print colored ( "\n**  (requires super-user privelege)   **", 'blue' ) ;

   if ( -e $FMGloctrg )                      # Find the database file
   {
      $FMGinfotrg = "$FMGloctrg" ;
      $FMGinfodir = "$FMGlocdir" ;
   }
   elsif ( -e $FMGglotrg )
   {
      $FMGinfotrg = "$FMGglotrg" ;
      $FMGinfodir = "$FMGglodir" ;
   }

   if ( "$FMGinfotrg" ne "" )                # If database file found
   {
      print "\nTexinfo top-level database: $FMGinfotrg" ;

      # Copy the document file to documentation directory.        #
      # If an old copy of the file exists, it will be overwritten.#
      # Note: User will be asked for password here.               #
      system ( "sudo cp --preserve=all '$FMGinfo' '$FMGinfodir'" ) ;

      # Set our working directory to info-reader database directory.#
      use Cwd qw ( getcwd ) ;
      chdir ( "$FMGinfodir" ) ;
      my $WD=getcwd() ;

      # If CWD is target directory AND Texinfo #
      # documentation copied successfully      #
      if ( $WD eq $FMGinfodir && -e $info )
      {
         # Scan 'dir' for existing FileMangler entry.               #
         # (If present, it is assumed to be syntactically correct.) #
         my $FMGfunction = qx ( grep -hm 1 '\* $fmgr:' $FMGinfotrg 2>&1 ) ;
         if ( (index ( $FMGfunction, $fmgr )) < 0 )
         {
            # If 'dir' has not been backed up, do it now. #
            $FMGinfobak = "$FMGinfotrg~" ;
            if ( ! -e $FMGinfobak )
            {
               print "\nCreate backup copy of database: $FMGinfobak\n" ;
               system ( "sudo cp --preserve=all '$FMGinfotrg' '$FMGinfobak'" ) ;
            }

            # Update the database file to include FileMangler docs. #
            # (For explanation of command, see notes above.)        #
            system ( "sudo install-info --dir-file=dir --info-file=filemangler.info --name=FileMangler --debug" ) ;

            # Verify that 'dir' contains the FileMangler entry. #
            #   (It is assumed to be syntactically correct.)    #
            $FMGfunction = qx ( grep -hm 1 '\* $fmgr:' $FMGinfotrg 2>&1 ) ;
            if ( (index ( $FMGfunction, $fmgr )) >= 0 )
            {
               chomp ( $FMGfunction ) ;
               print colored ( "\nDatabase Entry: '", 'blue' ) ;
               print colored ( "$FMGfunction", 'green' ) ;
               print colored ( "'", 'blue' ) ;
            }
            else
            {
               print colored ( "\nWARNING! - Adding database entry failed.", 'yellow' ) ;
               return 0 ;
            }
         }
         else
         {
            print colored ( "\nTexinfo database already contains an entry ", 'blue' ) ;
            print colored ( "for this application.\nDatabase not modified.", 'blue' ) ;
         }
      }
      else     # Error installing documentation
      {
         if ( $WD ne $FMGinfodir )
         {
            print colored ( "\nWARNING! - Unable to reach '$FMGinfodir' directory.",
                            'yellow' ) ;
         }
         elsif ( ! -e $info )
         {
            print colored ( "\nWARNING! - '$info' file not found.", 'yellow' ) ;
         }
         print colored ( "\nTexinfo documentation not installed in ", 'yellow' ) ;
         print colored ( "Info-reader database.", 'yellow' ) ;
         return 0 ;
      }
   }
   else     # Database file not found
   {
      print colored ( "\nWARNING! - Info-reader database not found.", 'yellow' ) ;
      print colored ( "\nTexinfo documentation not installed.", 'yellow' ) ;
      return 0 ;
   }
   return 1 ;  # return success

}  # install_infodocs()


# **************************** #
# Create an entry in '.bashrc' #
# for a memory-resident        #
# function.                    #
# **************************** #
sub create_mem_function ()
{
   print colored ( "\n\n** Add 'fmg()' function to \".bashrc\" file. **", 'blue' ) ;

   my $FMGshell = qx ( echo '$ENV{SHELL}' 2>&1 ) ;

   if ( (index ( $FMGshell, "bash" ) >= 0) && (-e $bashrc) )
   {
      my $FMGfunction = qx ( grep -hm 1 'fmg[(][)]' $bashrc 2>&1 ) ;
      chomp ( $FMGfunction ) ;
      my $FMGindex = index ( $FMGfunction, "fmg" ) ;

      # If .bashrc already contains an fmg() function, assume it is valid.#
      # Otherwise, append the function to the target file.                #
      if ( index ( $FMGfunction, "fmg" ) == -1 )
      {
         # If '.bashrc' has not been backed up, do it now. #
         my $bashrcbak = "$bashrc~" ;
         if ( ! -e $bashrcbak )
         {
            print "\nCreate backup copy of login script: $bashrcbak" ;
            system ( "cp --preserve=all '$bashrc' '$bashrcbak'" ) ;
         }

         # Append function declaration to login script. #
         open ( my $trghandle, '>>:encoding(UTF-8)', $bashrc ) ;
         print $trghandle "\n# * Function to invoke FileMangler *",
                          "\nfmg() { source 'fmg.sh' ; }\n\n" ;
         close ( $trghandle ) ;

         # Verify that '.bashrc' now contains the function. #
         $FMGfunction = qx ( grep -hm 1 'fmg[(][)]' $bashrc 2>&1 ) ;
         if ( index ( $FMGfunction, "fmg" ) >= 0 )
         {
            print colored ( "\nMemory-resident function successfully created.",
                            'green' ) ;
         }
         else
         {
            print colored ( "\nWARNING! - Unable to add memory-resident function to $bashrc",
                            'yellow' ) ;
            return 0 ;
         }
      }
      else
      {
         print colored ( "\nYour '.bashrc' login script already contains the ", 'blue' ) ;
         print colored ( "invocation function.\n'$bashrc' not modified.", 'blue' ) ;
      }

   }
   else
   {
      print colored ( "\nWARNING! - Your shell program is not 'bash' OR", 'yellow' ) ;
      print colored ( "\n login script '$bashrc' not found.", 'yellow' ) ;
      print colored ( "\n Memory-resident function not installed.", 'yellow' ) ;
      return 0 ;
   }
   return 1 ;

}  # create_mem_function()



# **************************** #
# Test for the necessary       #
# 3rd-party utilities          #
# **************************** #
sub external_utils ()
{
   # Returns:
   # 0 == both gio and wl-clipboard available
   # 1 == wl-clipboard missing, gio available
   # 2 == gio missing, wl-clipboard available
   # 3 == both gio and wl-clipboard missing
   my $eu_status = 0 ;  # return value
   my $gio_status = 2 ; # gio availability
   my $wlc_status = 1 ; # wl-clipboard availability

   # Check for availability of 'gio' utility.                    #
   # This utility is required for full application functionality.#
   my $GioVersion = qx ( gio '--version' 2> /dev/null ) ;
   if ( (index ( $GioVersion, "2." )) == 0 )
   { $gio_status = 0 ; }

   # Check for availability of 'wl-clipboard' utilities.                 #
   # These utilities are required for Wayland (system) clipboard access. #
   # (We assume that if one is installed, they are both installed.)      #
   my $wlVersion = qx ( wl-copy '--version' 2>&1 ) ;
   chomp $wlVersion ;
   if ( (index ( $wlVersion, "wl-clipboard 2" )) == 0 )
   {
      $wlVersion = qx ( wl-paste '--version' 2>&1 ) ;
      if ( (index ( $wlVersion, "wl-clipboard 2" )) == 0 )
      { $wlc_status = 0 ; }
   }

   $eu_status = $gio_status + $wlc_status ;
   return $eu_status ;

}  # external_utils()


# **************************** #
# Report installation results. #
# **************************** #
sub report_success ()
{
   if ( $Success != 0 )
   {
      print colored ( "\n\n** Installation of FileMangler is Complete! **", 'blue' ) ;
      if ( $Success >= 1 )
      { print "\n1) Installation Directory: $FMGFMG" ; }
      if ( $Success >= 2 )
      { print "\n2) All necessary files copied to installation directory." ; }
      if ( $Success >= 3 )
      { print "\n3) Shell scripts copied to: '$PathTrg'" ; }
      if ( $Success >= 4 )
      {
         print "\n4) Info-reader documentation " ;
         if ( $DOCsuccess != 0 )
         { print "installed at: $FMGinfodir" ; }
         else
         {
            print "not installed.",
                  "\n   For manual installation of docs, see:",
                  "\n   info -f ./filemangler.info -n 'Install the Documentation'" ;
         }
      }
      if ( $Success >= 5 )
      {
         print "\n5) Memory-resident function " ;
         if ( $BRCsuccess != 0 )
         {
            print "created in '$bashrc' login script.",
                  "\n   Log out, then log in again to activate the memory-resident function." ;
         }
         else
         {
            print "was not installed in the login script (.bashrc) file.",
                  "\n Please manually edit your system's equivalent of the",
                  "\n '.bashrc' file to insert the invocation function." ;
        }
      }
      if ( $Success >= 6 )
      {
         print "\n6) Required external utilities 'gio' and 'wl-clipboard':" ;
         if ( $UTLsuccess == 0 )
         { print "\n   have been tested and are accessible." ; }
         else
         {
            if ( $UTLsuccess == 1 || $UTLsuccess == 3 )
            {
               print colored ( "\n   Warning: 'wl-clipboard' utilities required for system clipboard access.", 'red' ) ;
               print colored ( "\n            Please see \"Building from Sources\" (Item 7) of documentation.", 'red' ) ;
            }
            if ( $UTLsuccess >= 2 )
            {
               print colored ( "\n   Error! : 'gio' (Gnome Input/Output) utility is required for full   ", 'red' ) ;
               print colored ( "\n            application functionality. Please install 'gio' (or GLib) ", 'red' ) ;
               print colored ( "\n            for access to virtual filesystems such as smartphones,    ", 'red' ) ;
               print colored ( "\n            optical drives and other devices attached to the USB buss.", 'red' ) ;
               print colored ( "\n            See \"Building from Sources\" (Item 5) of documentation.  ", 'red' ) ;
            }
         }
      }
      print "\n\nTo test the installation, follow the test sequence described in the",
              "\ndocumentation:  info -f filemangler.info -n 'Testing Your Installation'" ;
   }

   return 0 ;

}  # report_success()
