//********************************************************************************
//* File       : FmCommand.cpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 10-Apr-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: This is a support module for the FileMangler application.       *
//* This module, along with FmInterface.cpp contains the non-menu (command-key)  *
//* portion of the User Interface which gathers key input and executes the       *
//* user's commands.                                                             *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//********************************************************************************

//* Header Files *
#include "FileMangler.hpp"    // FileMangler definitions and data

//*****************
//** Definitions **
//*****************

//****************
//** Prototypes **
//****************
static bool staticMount ( const gString& gsMntPt ) ;

//****************
//** Local Data **
//****************

//* Threshold for "very large" storage devices (in bytes). *
//* Used to determine whether to scan the full directory   *
//* tree or only the top-level data. See CmdMount().       *
//* This value was selected to exclude most USB flash      *
//* drives and common SD cards. Intended primarily to      *
//* identify external SATA and SSD hard drives.            *
const UINT64 HUGE_EXT_DRIVE = 64000000000 ;  // (64GB)


//***********************
//*     CmdTrash        *
//***********************
//********************************************************************************
//* Perform the specified operation on the system trashcan.                      *
//* The trashcan used is either the GNOME/KDE/ETC. Trashcan, OR AN ALTERNATE     *
//* trashcan directory specified during startup.                                 *
//*                                                                              *
//* Note that if desktop trashcan not validated AND if no alternate Trashcan     *
//* location was specified in the config file (or it is mis-configured),         *
//* then Trashcan operations will have been disabled. We check for this before   *
//* trying to trash the files and alert the user about the trashcan error.       *
//*                                                                              *
//* Input  : mCode: menu code                                                    *
//*                 mcFcmdMB_TRASH  : Move files to Trashcan.                    *
//*                 mcUtilMB_UTRASH : Restore most recent move to Trashcan       *
//*                 mcUtilMB_MTRASH : Empty the Trashcan                         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Note on sending files to trashcan:                                           *
//*   If in Dual-Window mode AND if the inactive window is anywhere on the path  *
//*   of files being deleted, then after the operation, the path of inactive     *
//*   window will no longer exist and must be updated to a directory that        *
//*   actually does exist.                                                       *
//*                                                                              *
//* NOTE: There is a certain schizophrenia about calling the Empty/Restore       *
//*       dialog for either an Empty or Restore operation. It's really the       *
//*       same call sequence, but the flag is different. However, the user       *
//*       can change his/her/its mind about the operation while in the dialog,   *
//*       and at this level, we would not know it.                               *
//********************************************************************************

void FileMangler::CmdTrash ( MenuCode mCode )
{
   //* If the CWD is the root directory, disallow file operations.*
   if ( (this->isRoot ()) )
   { this->RootWarning () ;  return ; }   // NOTE THE EARLY RETURN


   if ( this->fdPtr->TrashcanConfigured () )
   {
      //* Move selected files to Trashcan *
      if ( mCode == mcFcmdMB_TRASH )
      {
         //* If in Dual-Win mode, clear any file *
         //* selections in the inactive window   *
         if ( (this->ClearAltFileListSelections ()) == false )
            this->RestoreQuickHelpMsg () ;

         //* Move files that have been marked as 'selected' * 
         //* (or currently-highlighted file) to trashcan    *
         UINT     selFiles,            // number of 'selected' files
                  movFiles ;           // number file files successfully moved to trash
         movFiles = this->fdPtr->TrashcanSelectedFiles ( selFiles ) ;

         //* If in Dual-Win mode AND alternate window is the same as source    *
         //* dir (or an affected child dir), update the alternate window also. *
         if ( (this->DualwinMode ()) )
         {
            fmFType apType ;
            gString srcPath, trgPath ;
            this->fdPtr->GetPath ( srcPath ) ;
            this->altPtr->GetPath ( trgPath ) ;
            //* If alt window's target dir has disappeared (see note above) *
            if ( !(this->altPtr->TargetExists ( trgPath, apType )) )
            {
               trgPath = srcPath ;
               if ( (this->altPtr->SetDirectory ( trgPath )) == OK )
                  this->altPtr->RedrawCurrDir ( false ) ;
               else
               { /* (this is unlikely) */ }
            }
            else if ( trgPath == srcPath )
               this->RefreshAltFileList () ;
         }

         dtbmData msgData ;            // message display
         gString gs ;                  // output formatting
         if ( movFiles == selFiles )
            gs.compose( L"  %u files moved to trashcan", &movFiles ) ;
         else
         {
            UINT errFiles = selFiles - movFiles ;
            gs.compose( L"  %u moved successfully, but ERROR moving %u file(s)", 
                        &movFiles, &errFiles ) ;
         }
         gs.copy( msgData.textData, gsALLOCDFLT ) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
      }  // (mCode == mcFcmdMB_TRASH)

      //* Restore item(s) from Trashcan *
      else if ( mCode == mcUtilMB_UTRASH )
      {  //* Undo the most recent move-to-trash operation targeting *
         //* our trashcan. It doesn't matter if the operation was   *
         //* performed by our application or by another application.*

         //* Give user a clue *
         attr_t color = dtbmNFcolor ;
         dtbmData msgData( " Undo recent move-to-trash operation. ", &color ) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

         this->fdPtr->ManageTrashcan ( true ) ; // restore items

         //* Active window will be automagically refreshed by the call, but    *
         //* if in Dual-win Mode, we explicitly update the inactive window     *
         //* also because user may have restored files to it.                  *
         this->RefreshAltFileList ( true ) ;

      }  // (mCode == mcUtilMB_UTRASH)

      //* Empty the Trashcan *
      else if ( mCode == mcUtilMB_MTRASH )
      {
         attr_t color = dtbmFcolor ;
         dtbmData    msgData( " Manage the Trashcan ", &color ) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

         //* Active window will be automagically refreshed by the call, but    *
         //* if in Dual-win Mode, we explicitly update the inactive window     *
         //* also because user may have restored files to it.                  *
         this->fdPtr->ManageTrashcan () ;
         this->RefreshAltFileList ( true ) ;
         this->RestoreQuickHelpMsg () ;
      }  // (mCode == mcUtilMB_MTRASH)

      else
         { /* THIS WOULD BE AN APPLICATION ERROR! */ }
   }     // if(TrashcanConfigured)
   else
   {
      const char* msgText[] = 
      {
         "  ALERT! ALERT!  ",
         "Trashcan operations disabled.",
         " ",
         "Either the System Trashcan could not be found,",
         " OR",
         "specified Alt Trash location improperly configured.",
         NULL
      } ;
      this->dWin->SetDialogObscured () ;
      this->fdPtr->InfoDialog ( msgText ) ;
      this->dWin->RefreshWin () ;
   }

}  //* End CmdTrash() *

//*************************
//*     CmdFavorites      *
//*************************
//********************************************************************************
//* Invoke a dialog in which the user can select a new target directory from     *
//* the favorites list.                                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdFavorites ( void )
{
const char* dirVanished[] = 
{
   "  ALERT  ",
   "The selected target directory does not exist or is",
   "currently inaccessable. Please verify that the path",
   "was entered correctly.",
   " ",
   "If path refers to an external mountable device or",
   "to a network drive, please mount it and try again.",
   NULL
} ;

const short 
   dialogRows = (MIN_ROWS - 3),  // dialog size and position
   dialogCols = (MIN_COLS - 2),
   ulY  = (this->actualRows / 2) - (dialogRows / 2),
   ulX  = (this->actualCols / 2) - (dialogCols / 2),
   pathCols = dialogCols - 2 ;   // max length of path display string
   gString newDir ;              // new (or current) path
   attr_t dColor = this->cfgOpt.cScheme.sd ; // dialog base color attribute
   bool  newCWD = false ;        // if true, move to user-selected directory

   //* Formatting for the favorites list display data *
   gString fgs[MAX_FAVDIRS] ; 
   const char* seboxData[MAX_FAVDIRS] ;
   attr_t seboxColors[MAX_FAVDIRS] ;
   ssetData sData( seboxData, seboxColors, ZERO, ZERO, true ) ;
   for ( short i = ZERO ; i < MAX_FAVDIRS ; i++ )
   {
      fgs[i] = this->FavoriteDirs[i] ;
      this->fdPtr->TrimPathString ( fgs[i], pathCols ) ;
      seboxData[i] = fgs[i].ustr() ;
      if ( (fgs[i].gschars()) > 1 )
         ++sData.dispItems ;
      seboxColors[i] = this->cfgOpt.cScheme.tf ;
   }
   if ( sData.dispItems == ZERO )
   {
      fgs[ZERO] = "  none specified  " ;
      seboxData[ZERO] = fgs[ZERO].ustr() ;
      ++sData.dispItems ;
   }

   //* Create a list of needed controls *
   enum dpCtrls { favSE, okPB, canPB, dpcCOUNT } ;
   InitCtrl ic[dpcCOUNT] =            // control definitions
   {
   { //* 'Favorite Directories' scroll ext - - - - - - - - - - - - - -   favSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      10,                           // lines:     control lines
      dialogCols,                   // cols:      control columns
      NULL,                         // dispText:  n/a
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Favorite Directories ",     // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      MAX_FAVDIRS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      seboxColors,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[okPB],                    // nextCtrl:  link in next structure
   },
   {  //* 'OK' pushbutton   - - - - - - - - - - - - - - - - - - - - - -   okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogRows - 2),        // ulY:       upper left corner in Y
      short(dialogCols / 2 - 10),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   OK   ",                   // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[okPB].ulY,                 // ulY:       upper left corner in Y
      short(ic[okPB].ulX + 12),     // ulX:
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
   } ;

   //* Save the application's display data *
   this->dWin->SetDialogObscured () ;

   //* Initial parameters for dialog window, passed to sub-methods.*
   InitNcDialog dInit
   ( 
      dialogRows,           // number of display lines
      dialogCols,           // number of display columns
      ulY,                  // Y offset from upper-left of terminal 
      ulX,                  // X offset from upper-left of terminal 
      NULL,                 // dialog title (see below)
      ncltSINGLE,           // border line-style
      dColor,               // border color attribute
      dColor,               // interior color attribute
      ic                    // pointer to list of control definitions
   ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Make visual connections to dialog border and     *
      //* initialize display data for dctSCROLLEXT control *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ll2Left = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( favSE, cdConn ) ;
      dp->SetScrollextText ( favSE, sData ) ;

      //* Draw the static text *
      dp->WriteParagraph ( 10, 9, 
               " Highlight the desired target directory, then select 'OK'.\n"
               "     Select 'CANCEL' to remain in current directory.\n"
               "\n"
               "       Note: To edit the Favorites list, go to the\n"
               "             'Util' menu and select 'Configure'.", dColor ) ;

      dp->RefreshWin () ;           // make everything visible

      //*************************************
      //* Talk to the animals, Dr. Dolittle *
      //*************************************
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //* If focus is currently on the Scroll Box *
         if ( ic[icIndex].type == dctSCROLLEXT )
         {
            for ( short i = ZERO ; i < MAX_FAVDIRS ; i++ )
               seboxColors[i] = this->cfgOpt.cScheme.tf ;
            dp->RefreshScrollextText ( favSE ) ;

            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;

            for ( short i = ZERO ; i < MAX_FAVDIRS ; i++ )
               seboxColors[i] = this->cfgOpt.cScheme.tn ;
            dp->RefreshScrollextText ( favSE ) ;
         }

         //* If focus is currently on a Pushbutton   *
         else if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {
                  //* If synch-lock engaged, disengage it.*
                  if ( this->synchLock )
                  {
                     dp->SetDialogObscured () ;
                     this->dWin->RefreshWin () ;
                     this->CmdSynchLock () ;
                     this->dWin->SetDialogObscured () ;
                     dp->RefreshWin () ;
                  }

                  //* Get a copy of the target string *
                  newDir = this->FavoriteDirs[dp->GetScrollextSelect ( favSE )] ;

                  //* Determine whether the selected entry is actually accessible.*
                  fmFType fileType ;
                  bool fileExists = this->fdPtr->TargetExists ( newDir, fileType ) ;
                  if ( fileExists && fileType == fmDIR_TYPE )
                  {  //* Set the new target AFTER dialog has closed *
                     newCWD = done = true ;
                  }
                  else
                  {  //* Selected directory does not exist, inform user *
                     dp->SetDialogObscured () ;
                     this->fdPtr->InfoDialog ( dirVanished ) ;
                     dp->PrevControl () ;
                     dp->RefreshWin () ;
                  }
               }
               else if ( Info.ctrlIndex == canPB ) // abort selection
                  done = true ;
            }
         }

         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while(!done)
   }
   else           // unable to open window (likely memory-allocation error)
      ;

   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

   //* Restore application's display data *
   this->dWin->RefreshWin () ;

   //* If user has selected a new target directory, *
   //* set it as the new Current Working Directory. *
   if ( newCWD )
   {
      //* Get the size of the target filesystem. If target directory   *
      //* is the mountpoint for a very large storage device, read only *
      //* the top-level data.                                          *
      //* Else, assume that target is small enough to be read quickly. *
      // Programmer's Note: This test will identify the "/" (root) directory of 
      // an external drive as a mountpoint. Unfortunately, this test does not 
      // capture the "/home" directory of such a drive because it is not a mountpoint.
      bool recurse = true ;
      fileSystemStats fsStats ;
      this->fdPtr->GetFilesystemStats ( newDir, fsStats ) ;
      if ( (fsStats.isMountpoint) &&
           (fsStats.blockTotal * fsStats.blockSize) >= HUGE_EXT_DRIVE )
         recurse = false ;

      if ( (this->fdPtr->SetDirectory ( newDir, recurse )) != OK )
      {  //* Unable to move to new target directory *
         this->dWin->SetDialogObscured () ;
         this->fdPtr->InfoDialog ( dirVanished ) ;
         this->dWin->RefreshWin () ;
      }
   }

}  //* End CmdFavorites() *

//***********************
//*      CmdMount       *
//***********************
//********************************************************************************
//* Invoke a dialog in which the user can select the filesystem to be            *
//* mounted or unmounted.                                                        *
//*                                                                              *
//* Input  : mCode : menu code (either mcFileMB_MOUNT or mcFileMB_UNMOUNT)       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Mount / Unmount Sequence                                                     *
//* ========================                                                     *
//*  - Because mounting and unmounting often require superuser privlege,         *
//*    we create a small terminal window in which to launch the mount call.      *
//*                                                                              *
//*  - The recommended way to find currently-mounted filesystem is 'findmnt'.    *
//*       findmnt --kernel --real --list -o TARGET,FSTYPE,SOURCE                 *
//*    Our hack, 'SelectMount' uses this command to get the raw data, and then   *
//*    filters out the obviously-not-block-storage entries.                      *
//*                                                                              *
//*    We then use a combination of 'lsusb' and 'gio' to identify the attached   *
//*    virtual filesystems (MTP/GVfs).                                           *
//*                                                                              *
//*    So, if we combine the results of pre-defined mounts in 'fstab' with       *
//*    the list of currently-mounted and currently-attached devices, PLUS the    *
//*    mounts specified in the config file (minus duplicates), we should have    *
//*    a solid list of devices which can be mounted/unmounted.                   *
//*       findmnt --fstab -o TARGET,FSTYPE,SOURCE                                *
//*       findmnt --kernel --real --list -o TARGET,FSTYPE,SOURCE                 *
//*                                                                              *
//*  - Parsing the 'fstab' file for "known" filesystems.                         *
//*    - To find out what filesystems the system knows about, see /dev/disk.     *
//*      This directory contains three subdirectories with symbolic links        *
//*      according to:                                                           *
//*        /dev/disk/by-id            (this is pretty useless)                   *
//*        /dev/disk/by-uuid          (reports uuid and _relative_ device)       *
//*        /dev/disk/by-label         (reports label and _relative_ device)      *
//*        /dev/mapper                (members of LVM)                           *
//*      These directories contain symlinks to specific devices, (block          *
//*      special device drivers).                                                *
//*      Example: ls -l /dev/disk/by-label                                       *
//*       F20BACKUP32 -> ../../mmcblk0p1                                         *
//*       linuxboot -> ../../sda1                                                *
//*       LVM_home -> ../../dm-2                                                 *
//*       LVM_root -> ../../dm-1                                                 *
//*       LVM_swap -> ../../dm-0                                                 *
//*                                                                              *
//*    - sudo blkid           list all block-device IDs                          *
//*      blkid -L <label>     ex: blkid /LVM_home  (reports device)              *
//*      blkid -U <uuid>                                                         *
//*      blkid <device>       ex: blkid /dev/sda1  (reports all, see '-o')       *
//*      blkid -o list                             (formatted report)            *
//*      device  fs_type  label  mountpoint  UUID                                *
//*      ------------------------------------------                              *
//*      x       x        x      x           x                                   *
//*      Note that the '-o list' option doesn't necessarily list the UUID for    *
//*      all devices, but does give the mountpoint and device for all _mounted_  *
//*      filesystems.                                                            *
//*                                                                              *
//*    - The /etc/fstab file contains the rules for mounting known filesystems.  *
//*        a) UUID  : "UUID=..."                                                 *
//*        b) Label : "/dev/mapper/Fedora20-home /home ..."                      *
//*        c) Device: "/dev/sdb ..."                                             *
//*      The UUID entries are the most reliable because they are the most        *
//*      likely to uniquely identify the mount.                                  *
//*                                                                              *
//*    - If the /etc/fstab entry for a filesystem includes the word 'user',      *
//*      then it can be mounted by any user.                                     *
//*      Example: UUID=e8ccb905-969b-44de-8e84-7a3a09d121c2                      *
//*                /run/media/sam/Ubuntu16_root  ext4                            *
//*                rw,suid,dev,exec,noauto,user,async 0 0                        *
//*    - Unless the device is 'auto' mounted, the 'sudo' command may be          *
//*      required for the invocation.                                            *
//*                                                                              *
//*    - Smartphone access:                                                      *
//*      lsusb command for USB device list                                       *
//*      gio mount --list --detail for additional info                           *
//*      Additional resources:                                                   *
//*         sudo install gmtp                                                    *
//*         sudo install mtp-tools                                               *
//*         jmtpfs (FUSE - POSIX)                                                *
//*                                                                              *
//*    - Optical drive access:                                                   *
//*      gio mount --list --detail     (see DVD_DeviceStats())                   *
//*      gio info --filesystem TARGET  (seeGetFilesystemStats_aux())             *
//*                                                                              *
//*  - /etc/mtab and /proc/mounts                                                *
//*    These files contain a (mostly) up-to-date list of currently-mounted       *
//*    filesystems.                                                              *
//*                                                                              *
//*  - Commands:                                                                 *
//*    mount -t TYPE DEVICE MOUNTPOINT                                           *
//*    mount /MOUNTPOINT  _or_  mount --target /MOUNTPOINT                       *
//*          This assumes that the associated device will be found in 'fstab'.   *
//*    Note that if the target mountpoint does not exist, Nautilus will create   *
//*    it before attempting the mount. For instance, our external Ubuntu drive   *
//*    is mounted at '/run/media/sam/Ubuntu16_root' HOWEVER, that directory      *
//*    does not persist past a reboot, so it must be re-created each time the    *
//*    drive is mounted. (We are not sure how Nautilus handles this.)            *
//*                                                                              *
//*  - See '/proc/filesystems' for a list of supported filesystem types.         *
//*    The list of physical types will look something like this:                 *
//*               ext3, ext2, ext4, fuseblk, vfat                                *
//*    'tmpfs' will often be seen in the tree, but this is not an actual         *
//*    filesystem.                                                               *
//*    Other entries are probably virtual types, e.g. 'proc' 'fusectl' etc.      *
//*                                                                              *
//*  - For user convenience currently-mounted filesystems are placed at the top  *
//*    of the list. The reasoning is that most filesystems will auto-mount, so   *
//*    user will seldom need to mount a filesystem manually; however, when       *
//*    disconnecting a device, it will almost always need to be unmounted first. *
//*                                                                              *
//********************************************************************************

//* Callback method used by CmdMount().*
static short cmCallback ( const short currIndex, const wkeyCode wkey, bool firstTime = false ) ;
//* Dialog pointer used by call-back method for CmdMount().*
static NcDialog* cmcbPtr = NULL ;
//* Pointer to filesystem mount information *
static const fsInfo* cmcbFsi = NULL ;

void FileMangler::CmdMount ( MenuCode mCode )
{
   #define DEBUG_CmdMount (0)    // set to non-zero for debugging only

   const char *mntLabel  = "  MOUNT  " ;
   const short dialogRows = (MIN_ROWS - 3),  // dialog size and position
               dialogCols = (MIN_COLS - 2),
               ulY  = (this->actualRows / 2) - (dialogRows / 2),
               ulX  = (this->actualCols / 2) - (dialogCols / 2),
               pathCols = dialogCols - 21 ;  // limit display-string length
   attr_t dColor = this->cfgOpt.cScheme.sd,  // dialog base color attribute
          mntColor = nc.gr,      // color attribute of string for 'mounted'
          nmntColor = nc.br ;    // color attribute of string for 'not mounted'
   gString trgMnt ;              // target mountpoint
   short seSelect ;              // index if selected mount target
   bool    newMNT = false,       // if true, new filesystem mounted
           setCWD = false,       // if true, make target the new CWD
           superPriv = false ;   // if true, execute operation at superuser privilege

   //*Create a list of mounted (or mountable) storage devices (filesystems).*
   short fsCount = ZERO ;
   fsInfo* fsiPtr = this->cmScan4Filesystems ( fsCount ) ;
   gString* fgs = new gString[fsCount + 1] ;
   const char* seboxData[fsCount + 1] ;
   attr_t seboxColors[fsCount + 1] ;
   ssetData sData( seboxData, seboxColors, ZERO, ZERO, true ) ;


   //* Initialize the display data.*
   for ( short i = ZERO ; i < fsCount ; ++i )
   {
      fgs[i] = fsiPtr[i].fsMountpt ;
      this->fdPtr->TrimPathString ( fgs[i], pathCols ) ; // make display text fit
      seboxColors[i] = fsiPtr[i].fsMounted ? mntColor : nmntColor ;
      seboxData[i] = fgs[i].ustr() ;
      ++sData.dispItems ;        // non-null-string display items
   }

   //* If list is empty. This is unlikely, but possible. *
   if ( sData.dispItems == ZERO )
   {
      fgs[ZERO] = "  none specified  " ;
      seboxData[ZERO] = fgs[ZERO].ustr() ;
      seboxColors[ZERO] = nmntColor ;
      ++sData.dispItems ;
   }

   //* Create a list of needed controls *
   enum dpCtrls { mntSE, mntPB, canPB, cwdRB, superRB, dpcCOUNT } ;
   InitCtrl ic[dpcCOUNT] =            // control definitions
   {
   { //* 'Mount Commands' scroll ext   - - - - - - - - - - - - - - - -   mntSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      10,                           // lines:     control lines
      dialogCols,                   // cols:      control columns
      NULL,                         // dispText:  n/a
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "  Mount Filesystems  ",      // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      MAX_MNTCMDS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      seboxColors,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[mntPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'MOUNT' ('UNMOUNT') pushbutton  - - - - - - - - - - - - - - -  mntPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogRows - 4),        // ulY:       upper left corner in Y
      short(dialogCols / 2 - 12),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      9,                            // cols:      control columns
      mntLabel,                     // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[mntPB].ulY,                // ulY:       upper left corner in Y
      short(ic[mntPB].ulX + 14),    // ulX:
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cwdRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Display Mount Target' radiobutton  - - - - - - - - - - - - -  cwdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      true,                         // rbSelect:  initially set
      short(ic[mntPB].ulY + 1),     // ulY:       upper left corner in Y
      short(ic[mntPB].ulX),         // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cfgOpt.cScheme.sd,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Display mount target data immediately.", // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[superRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Superuser' radiobutton   - - - - - - - - - - - - - - - - -  superRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially reset
      short(ic[cwdRB].ulY + 1),     // ulY:       upper left corner in Y
      ic[cwdRB].ulX,                // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cfgOpt.cScheme.sd,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Execute operation with superuser privilege.", // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Save the application's display data *
   this->dWin->SetDialogObscured () ;

   //* Initial parameters for dialog window, passed to sub-methods.*
   InitNcDialog dInit
   ( 
      dialogRows,           // number of display lines
      dialogCols,           // number of display columns
      ulY,                  // Y offset from upper-left of terminal 
      ulX,                  // X offset from upper-left of terminal 
      NULL,                 // dialog title (see below)
      ncltSINGLE,           // border line-style
      dColor,               // border color attribute
      dColor,               // interior color attribute
      ic                    // pointer to list of control definitions
   ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Make visual connections to dialog border and     *
      //* initialize display data for dctSCROLLEXT control *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ll2Left = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( mntSE, cdConn ) ;
      dp->SetScrollextText ( mntSE, sData ) ;

      #if DEBUG_CmdMount != 0
      winPos wpd( 10, 1 ) ;
      gString gsx ;
      attr_t dx = nc.brR ;
      dp->PrevControl () ;
      for ( short i = ZERO ; i < fsCount ; ++i )
      {
         dp->ClearArea ( wpd.ypos, wpd.xpos, 4, dialogCols - 2, dx ) ;
         dp->SetScrollextSelect ( mntSE, i ) ;
         gsx.compose( "fsDevice:'%s'  fsType: '%s'\n"
                      "fsMounted:%hhd  fsMtp:%hhd  fsDvd:%hhd  fsDvdMedia:%hhd\n"
                      "fsMountpt: '%s'\n",
                      fsiPtr[i].fsDevice, fsiPtr[i].fsType,
                      &fsiPtr[i].fsMounted, &fsiPtr[i].fsMtp, &fsiPtr[i].fsDvd, 
                      &fsiPtr[i].fsDvdMedia, fsiPtr[i].fsMountpt ) ;
         dp->WriteParagraph ( wpd, gsx, dx, true ) ;
         dp->UserAlert () ; nckPause();
      }
      dp->ClearArea ( wpd.ypos, wpd.xpos, 4, dialogCols - 2, dColor ) ;
      dp->NextControl () ;
      #endif   // DEBUG_CmdMount

      //* Define a Pinwheel-class object for a visual indicator *
      //* that we are waiting for the system to respond.        *
      //* Single-color, tick rate == 100 mSec.                  *
      winPos pinWp( ZERO, ZERO ) ;
      Pinwheel pinwheel( dp, pwsWheelC, &this->cfgOpt.cScheme.pf, 1, 
                         pinWp, 100, 1, 1, NULL, L' ', dColor ) ;

      //* Draw the static text *
      winPos wp( 10, 2 ) ;
      dp->WriteParagraph ( wp, 
               "    Highlight the desired target filesystem, then tab to '(UN)MOUNT'.\n"
               "\n"
               "\n"
               "\n"
               "Legend:\n"
               "        mounted\n"
               "        not mounted", dColor ) ;
      dp->WriteString ( wp.ypos + 5, wp.xpos, " Green ", nc.gr ) ;
      dp->WriteString ( wp.ypos + 6, wp.xpos, " Brown ", nc.br ) ;

      //* Establish a callback to monitor user activity.*
      cmcbPtr = dp ;
      cmcbFsi = fsiPtr ;
      dp->EstablishCallback ( &cmCallback ) ;

      wp.ypos += 1 ;  wp.xpos = 2 ;
      dp->RefreshWin () ;           // make everything visible

      //*************************************
      //* Talk to the animals, Dr. Dolittle *
      //*************************************
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //* If focus is currently on the Scroll Box *
         if ( ic[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;

            //* If an MTP URI, and if device is not mounted.*
            dp->ClearLine ( wp.ypos + 1 ) ;
            if ( fsiPtr[Info.selMember].fsMtp && !fsiPtr[Info.selMember].fsMounted )
            {
               dp->WriteString ( wp.ypos + 1, wp.xpos,
                  "    Verify that screen of target device is unlocked.", 
                  this->cfgOpt.cScheme.em, true ) ;
            }
            //* If an optical drive contains no media disc.*
            else if ( fsiPtr[Info.selMember].fsDvd && !fsiPtr[Info.selMember].fsMounted
                      && !fsiPtr[Info.selMember].fsDvdMedia )
            {
               dp->WriteString ( wp.ypos + 1, wp.xpos,
                  "    Verify that media disc has been inserted.", 
                  this->cfgOpt.cScheme.em, true ) ;
            }
         }

         //* If focus is currently on a Pushbutton   *
         else if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == mntPB )
               {
                  chrono::duration<short, std::milli>aMoment( 1500 ) ;
                  winPos wpm = wp ;
                  dp->ClearLine ( wp.ypos, false ) ;

                  //* If synch-lock engaged, disengage it *
                  // Programmer's Note: If unmounting a filesystem that is not 
                  // displayed, the synch-lock COULD remain engaged, but we 
                  // think it is not worth the extra test.
                  if ( this->synchLock )
                  {
                     dp->SetDialogObscured () ;
                     this->dWin->RefreshWin () ;
                     this->CmdSynchLock () ;
                     this->dWin->SetDialogObscured () ;
                     dp->RefreshWin () ;
                  }

                  //* Get a copy of the target string *
                  seSelect = dp->GetScrollextSelect ( mntSE ) ;
                  trgMnt = fsiPtr[seSelect].fsMountpt ;
                  //* Execute with superuser privilege? *
                  dp->GetRadiobuttonState ( superRB, superPriv ) ; 

                  //**************************************
                  //* If target is not mounted, mount it.*
                  //**************************************
                  if ( fsiPtr[seSelect].fsMounted == false )
                  {
                     //* Give user a clue *
                     wpm = dp->WriteString ( wpm, " MOUNTING : ", 
                                            this->cfgOpt.cScheme.tf ) ;
                     wpm = dp->WriteString ( wpm, fgs[seSelect], 
                                            this->cfgOpt.cScheme.tf, true ) ;

                     dp->SetDialogObscured () ;    // save the dialog display

                     //* Position and launch the Pinwheel *
                     //* to indicate work-in-progress.    *
                     pinWp = { wpm.ypos, short(wpm.xpos + 1) } ;
                     pinwheel.setPosition( pinWp, L' ', dColor ) ;
                     pinwheel.launch() ;

                     //* If mounting media in an optical drive and the *
                     //* media are not visible, close the media tray.  *
                     //* It may take several seconds after the tray is *
                     //* closed before media can be mounted.           *
                     // Programmer's Note: It is likely that this will never happen
                     // because when the drive tray is open, the device has neither
                     // a URI nor a filespec. It is known only by its device driver
                     // filespec, and therefore will not be included in the list
                     // presented to the user.
                     if ( fsiPtr[seSelect].fsDvd && ! fsiPtr[seSelect].fsDvdMedia )
                        this->fdPtr->EjectMedia (fsiPtr[seSelect].fsDevice, true ) ;

                     short status = 
                        this->cmMountFilesystem ( fsiPtr[seSelect], dp, superPriv ) ;

                     pinwheel.deactivate() ;       // deactivate the Pinwheel

                     //* If a gvfs or DVD mount has failed, *
                     //* perform an automatic retry.        *
                     if ( (status == -1) && 
                          (fsiPtr[seSelect].fsMtp || fsiPtr[seSelect].fsDvd) )
                     {
                        //* Signal the retry *
                        dp->WriteString ( wpm, " -RETRY", this->cfgOpt.cScheme.em, true ) ;

                        //* Re-launch the Pinwheel *
                        pinwheel.launch() ;

                        //* For MTP devices, re-scan the target device for *
                        //* current device path: /dev/bus/usb/00n/00?      *
                        //* See notes in cmMountFilesystem().              *
                        if ( fsiPtr[seSelect].fsMtp )
                        {
                           gString gstmp ;
                           if ( (this->fdPtr->USB_DeviceID ( 
                                    fsiPtr[seSelect].fsMountpt, gstmp )) )
                           { gstmp.copy( fsiPtr[seSelect].fsDevice, FS_BYTES ) ; }
                        }
                        //* Retry the mount operation *
                        status = this->cmMountFilesystem ( fsiPtr[seSelect], 
                                                           dp, superPriv ) ;

                        pinwheel.deactivate() ;    // deactivate the Pinwheel
                     }

                     if ( status == OK )           // successfully mounted
                     {
                        dp->WriteString ( wpm, " -OK   ", 
                                          this->cfgOpt.cScheme.pf, true ) ;
                        dp->GetRadiobuttonState ( cwdRB, setCWD ) ;
                        newMNT = true ;
                     }
                     else if ( status == (-1) )    // key input - user gave up waiting
                     {
                        dp->WriteString ( wpm, " -ERROR", 
                                          this->cfgOpt.cScheme.pf, true ) ;
                     }
                     else if ( status == (-2) )    // specified target not a directory
                     {
                        dp->WriteString ( wp, 
                           " Specified Mount Point Is Not a Directory! ", 
                           this->cfgOpt.cScheme.wr, true ) ;
                     }
                  }

                  //************************************
                  //* If target is mounted, unmount it.*
                  //************************************
                  else  // (fsiPtr[seSelect].fsMounted != false)
                  {
                     //* If application's CWD is on the target device, change  *
                     //* CWD before attempting to dismount. Otherwise, dismount*
                     //* will fail because the device is "busy". The logical   *
                     //* target is the user's 'home' directory, e.g. /home/sam.*
                     //* Exception: If '/home' is the target mountpoint,       *
                     //* (unlikely), we set CWD to a universally-available     *
                     //* directory '/' (root).                                 *
                     //* If in dual-window mode, check inactive window also.   *
                     gString currPath,             // CWD
                             newPath( "%s", std::getenv( "HOME" ) ) ;
                     if ( (newPath.find( trgMnt, ZERO, true, 
                              (trgMnt.gschars() - 1) )) == ZERO )
                        newPath = "/" ;
                     gString cmpPath( trgMnt ) ;   // comparison path
                     if ( ((trgMnt.find( mtpURI )) == ZERO) ||
                          ((trgMnt.find( optiURI )) >= ZERO) )
                        this->fdPtr->Uri2Filespec ( cmpPath, trgMnt.ustr() ) ;

                     if ( (this->DualwinMode ()) )
                     {
                        this->altPtr->GetPath ( currPath ) ;
                        if ( (currPath.find( cmpPath, ZERO, true, 
                               (cmpPath.gschars() - 1))) == ZERO )
                        {
                           dp->SetDialogObscured () ;  // save our dialog
                           this->dWin->RefreshWin () ; // restore parent dialog
                           this->altPtr->SetDirectory ( newPath ) ; // scan the new CWD
                           this->altPtr->RedrawCurrDir ( false ) ;  // remove highlight
                           this->dWin->SetDialogObscured () ; // save parent dialog
                           dp->RefreshWin () ;         // restore our dialog
                        }
                     }
                     this->fdPtr->GetPath ( currPath ) ;
                     if ( (currPath.find( cmpPath, ZERO, true, 
                            (cmpPath.gschars() - 1))) == ZERO )
                     {
                        dp->SetDialogObscured () ;  // save our dialog
                        this->dWin->RefreshWin () ; // restore parent dialog
                        this->fdPtr->SetDirectory ( newPath ) ; // scan the new CWD
                        this->dWin->SetDialogObscured () ; // save parent dialog
                        dp->RefreshWin () ;         // restore our dialog
                     }

                     //* Give user a clue *
                     wpm = dp->WriteString ( wpm, " UNMOUNTING : ", 
                                            this->cfgOpt.cScheme.tf ) ;
                     wpm = dp->WriteString ( wpm, fgs[seSelect], 
                                            this->cfgOpt.cScheme.tf, true ) ;

                     //* Position and launch the Pinwheel *
                     //* to indicate work-in-progress.    *
                     pinWp = { wpm.ypos, short(wpm.xpos + 1) } ;
                     pinwheel.setPosition( pinWp, L' ', dColor ) ;
                     pinwheel.launch() ;

                     //* Perform the dismount *
                     short status = this->cmDismountFilesystem ( trgMnt, dp, superPriv ) ;

                     pinwheel.deactivate() ;       // deactivate the Pinwheel

                     if ( status == OK )           // successfully unmounted
                     {
                        dp->WriteString ( wpm, " -OK   ", 
                                          this->cfgOpt.cScheme.pf, true ) ;

                        //* If optical media dismounted, eject the media.*
                        if ( fsiPtr[seSelect].fsDvdMedia )
                           this->fdPtr->EjectMedia ( fsiPtr[seSelect].fsDevice ) ;
                     }
                     else if ( status == (-1) )    // key input - user gave up waiting
                     {
                        dp->WriteString ( wpm, " -ERROR", 
                                          this->cfgOpt.cScheme.pf, true ) ;
                     }
                  }  // (fsiPtr[seSelect].fsMounted != false)

                  //* Give user time to read message *
                  this_thread::sleep_for( aMoment ) ;
               }  // (Info.ctrlIndex == mntPB)

               done = true ;
            }     // dataMod
         }        // (ic[icIndex].type == dctPUSHBUTTON)

         //* If focus is currently on a Radio Button *
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while(!done)
   }
   else           // unable to open window (likely memory-allocation error)
      ;

   if ( dp != NULL )
      delete ( dp ) ;                  // close the window
   cmcbPtr = NULL ;                    // reset callback pointers
   cmcbFsi = NULL ;

   //* Restore application's display data *
   this->dWin->RefreshWin () ;

   //* For virtual filesystems, convert the URI to the full filespec.*
   if ( ((trgMnt.find( mtpURI )) == ZERO) || ((trgMnt.find( optiURI )) >= ZERO) )
      this->fdPtr->Uri2Filespec ( trgMnt, fsiPtr[seSelect].fsMountpt ) ;

   //* If the filesystem was mounted and it is accessible, go there *
   if ( newMNT && setCWD )
   {
      //* Get the size of the target filesystem.                       *
      //* For very large storage devices, read only the top-level data.*
      //* Else, assume that target is small enough to be read quickly. *
      bool recurse = true ;
      fileSystemStats fsStats ;
      this->fdPtr->GetFilesystemStats ( trgMnt, fsStats ) ;
      if ( (fsStats.blockTotal * fsStats.blockSize) >= HUGE_EXT_DRIVE )
         recurse = false ;

      if ( (this->fdPtr->SetDirectory ( trgMnt, recurse )) != OK )
      {  //* Unable to move to new target directory *
         this->dWin->SetDialogObscured () ;
         const char* dirVanished[] = 
         {
            "  ALERT  ",
            "File system was mounted successfully, but it is not",
            "currently accessible. Please verify that you have",
            "read-access for the target.",
            NULL
         } ;
         this->fdPtr->InfoDialog ( dirVanished ) ;
         this->dWin->RefreshWin () ;
      }
   }
   //* Dismount or mount without display update *
   else
   {
      //* If CWD was affected by the mount/dismount, re-read the CWD.*
      gString pathBuff = trgMnt,
              currPath ;
      short indx ;
      if ( (indx = pathBuff.findlast( fSLASH )) > ZERO )
         pathBuff.limitChars( indx ) ;
      this->fdPtr->GetPath ( currPath ) ;
      if ( pathBuff == currPath )
         this->fdPtr->RefreshCurrDir () ;
   }

   //* If Dual-window Mode, inactive window may need update.*
   if ( this->DualwinMode () )
   {
      //* Isolate parent directory of target directory*
      //* and compare with CWD of inactive window.    *
      //* If a match, re-read data in inactive window.*
      gString altPath ;
      short indx ;
      if ( (indx = trgMnt.findlast( L'/' )) > ZERO )
         trgMnt.limitChars( indx ) ;
      this->altPtr->GetPath ( altPath ) ;
      if ( altPath == trgMnt )
         this->RefreshAltFileList ( true ) ;
   }

   delete[]fsiPtr ;        // release dynamically-allocated data
   delete[] fgs ;

}  //* End CmdMount() *

//*************************
//*      cmCallback       *
//*************************
//********************************************************************************
//* NON-MEMBER METHOD.                                                           *
//* This is a call-back method for CmdMount.                                     *
//* It is called from within the NcDialog class during the user-input loop.      *
//*                                                                              *
//* Input  : currIndex: index of control that currently has focus                *
//*          wkey     : user's key input data                                    *
//*          firstTime: the EstablishCallback() method calls this method once    *
//*                     with firstTime==true, to perform any required            *
//*                     initialization. Subsequently, the NcDialog class         *
//*                     always calls with firstTime==false.                      *
//* Returns: OK                                                                  *
//********************************************************************************
//* Programmer's Note: This callback makes some intimate assumptions about       *
//* the target dialog. It is important to keep the reference data synchronized.  *
//*                                                                              *
//********************************************************************************

static short cmCallback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   const char *mntLabel  = "  MOUNT  ",   // pushbutton labels
              *umntLabel = " UNMOUNT " ;
   static short mntSE = ZERO,       // index of Scrollext control
                mntPB = 1 ;         // index of "Mount" Pushbutton control
   static bool prevMounted = false ;// 'true' if previous call indexed a mounted item
                                    // (initial text is "MOUNT")

   //* If first call, initialize static flag and pushbutton text.*
   if ( firstTime )
   {
      prevMounted = cmcbFsi[(cmcbPtr->GetScrollextSelect ( mntSE ))].fsMounted ;
      if ( prevMounted != false )
      {
         cmcbPtr->SetPushbuttonText ( mntPB, umntLabel ) ;
         cmcbPtr->RefreshWin () ;
      }
   }

   //* If the Scrollext control currently has focus.*
   if ( currIndex == mntSE )
   {
      short mntseIndex = cmcbPtr->GetScrollextSelect ( mntSE ) ;
      bool  isMounted = cmcbFsi[mntseIndex].fsMounted ;

      if ( isMounted != prevMounted )
      {
         cmcbPtr->SetPushbuttonText ( mntPB, (isMounted ? umntLabel : mntLabel) ) ;
         cmcbPtr->RefreshWin () ;
         prevMounted = isMounted ;
      }
   }

   return OK ;

}  //* End cmCallback() *

//*************************
//*   cmMountFilesystem   *
//*************************
//********************************************************************************
//* Called only by CmdMount(). Mount the specified filesystem.                   *
//*  1) Caller has verified that the filesystem is not mounted at the            *
//*     specified position. The target directory may, or may not exist and       *
//*     may, or may not be a directory file.                                     *
//*  2) The system commands are executed from a child terminal window.           *
//*     This window is defined and launched using the DC_Emulator class.         *
//*     The child window will close immediately after the mount command has      *
//*     been executed.                                                           *
//*  3) Some mounts require super-user privelege, but we have no idea which      *
//*     ones do, so the mount command includes the 'sudo' security which         *
//*     prompts for the user password.                                           *
//*  4) Mounting a filesystem such as a USB hard drive can take several          *
//*     seconds, so we wait for the target device to become available before     *
//*     returning to caller.                                                     *
//*  5) Timing issue with virtual (MTP/GVfs) filesystems:                        *
//*     When a phone is first attached to the USB port, there is a delay         *
//*     during which a handshake is attempted between the host and the target    *
//*     device. The device will prompt the user to acknowledge whether the       *
//*     host is allowed to access device storage.                                *
//*     If the handshake fails (or is denied), then the mountpoint directory     *
//*     may appear for a moment and then disappear. When this happens, caller    *
//*     must alert the user that the handshake has failed and that he/she        *
//*     should attempt the mount again.                                          *
//*     An interesting side issue is that when this happens, the device number   *
//*     assigned to the device may be incremented.                               *
//*           /dev/bus/usb/003/006  BECOMES  /dev/bus/usb/003/007                *
//*     While the change in device number may be transparent to the user, it     *
//*     is quite annoying to those of us who care about the underlying process.  *
//*     See the auto-retry for this situation in CmdMount() method.              *
//*  6) There is always the possibility that the mount will fail, either         *
//*     because the device is not physically connected, or because the 'fstab'   *
//*     file does not fully specify the device (or the user is just having a     *
//*     bad day. For this reason, we return a very simple set of codes to the    *
//*     caller indicating the status of the operation.                           *
//*                                                                              *
//* Input  : fsi      : information about filesystem to be mounted               *
//*          subDlg   : handle to sub-dialog window                              *
//*                     (we must refresh after launch)                           *
//*          super    : 'true' operation is executed with superuser privilege    *
//*                     'false' operation is executed with user privilege        *
//*                                                                              *
//* Returns: OK     if filesystem successfully mounted                           *
//*          -1     if user key input received                                   *
//*                   (probably mount failed and user got tired of waiting)      *
//*          -2     if target mountpoint exists but is not a directory           *
//*                                                                              *
//********************************************************************************
//* Notes for virtual filesystems:                                               *
//* 1) Mount and Unmount may be performed using the URI:                         *
//*    gio mount cdda://sr0/                                                     *
//*    gio mount --unmount cdda://sr0/                                           *
//* 2) Mount may also be performed using the device-driver pathspec:             *
//*    gio mount --device=/dev/sr0                                               *
//* 3) Unmount may also be performed using the mountpoint pathspec:              *
//*    gio mount --unmount /run/user/1000/gvfs/cdda:host=sr0                     *
//*                                                                              *
//********************************************************************************

short FileMangler::cmMountFilesystem ( const fsInfo& fsi, 
                                       NcDialog* subDlg, bool super )
{
   gString newCwd( fsi.fsMountpt ), // mountpoint path (or mountpoint directory)
           fPath,
           argList,
           tmpFile ;
   fileSystemStats fsStats ;     // determine whether the target is mounted
   fmFType fType ;               // determine whether the target exists
   short status = OK ;           // return value (see above)
   bool createMp = false ;       // 'true' if mountpoint dir does not exist
   bool mtpFilesys = bool(((newCwd.find( mtpURI )) == ZERO) ? true : false) ;
   bool dvdFilesys = bool(((newCwd.find( optiURI )) >= ZERO) ? true : false) ;

   //* Determine whether target exists AND if so, whether it is a directory.  *
   //* For virtual filesystems, the parent directory is read-only for the     *
   //* user, but mountpoint directory is automagically created.               *
   if ( ! mtpFilesys && ! dvdFilesys )
   {
      if ( !(this->fdPtr->TargetExists ( newCwd, fType )) )
         createMp = true ;
      else if ( fType != fmDIR_TYPE )
         status = (-2) ;
   }

   //* If target is a virtual (mtp/gvfs) filesystem OR *
   //* a virtual (optical-drive media) filesystem,     *
   //* mounted using the 'gio' utility.                *
   if ( mtpFilesys || dvdFilesys )
   {
      //* Convert the activation_root" to a mountpoint filespec.*
      this->fdPtr->Uri2Filespec ( newCwd, fsi.fsMountpt ) ;

      if ( super != false )   // superuser privilege
      {
         fPath = "sudo" ;
         argList.compose( "gio mount --device='%s' \"%s\"", 
                          fsi.fsDevice, fsi.fsMountpt ) ;
      }
      else                    // user privilege
      {
         fPath = "gio" ;
         argList.compose( "mount --device='%s' \"%s\"", 
                          fsi.fsDevice, fsi.fsMountpt ) ;
      }
   }

   //* Target is a "real" (physical) filesystem *
   //* to be mounted by the OS kernel.          *
   else
   {
      if ( super != false )   // superuser privilege
      {
         fPath = "sudo" ;
         argList.compose( "mount --target '%S'", newCwd.gstr() ) ;
      }
      else                    // user privilege
      {
         fPath = "mount" ;
         argList.compose( "--target '%S'", newCwd.gstr() ) ;
      }
   
      if ( createMp )
      {
         gString gs( "mkdir '%S' ; ", newCwd.gstr() ) ;
         argList.insert( gs.gstr() ) ;
      }
   }

   if ( status == OK )
   {
      this->CreateTempname ( tmpFile ) ;  // create a temp file
      DC_Emulator dce( tmpFile, fPath, argList, true ) ; // console window definition
      wkeyCode wk ;           // user key input
      //* Max iterations of the while() loop. Time elapsed:   *
      //*   four(4) seconds for "real" filesystems,           *
      //*   eight(8) seconds for virtual filesystems          *
      //*   sixteen(16) seconds for optical-drive filesystems *
      short deadman = 4 ;
      bool done = false ;     // loop control

      //* Scan the system configuration to define *
      //* the new terminal window.                *
      //* Specify window size (small is cute here)*
      dce.DefineChildEmulator ( 56, 8 ) ;

      //* Open the new terminal window *
      dce.OpenTermwin () ;

      //* Wait a moment in case child process writes to the parent window. *
      chrono::duration<short, std::milli>aMoment( 1000 ) ;
      this_thread::sleep_for( aMoment ) ;
      this->dWin->ClearTerminalWindow () ;
      this->dWin->RefreshWin () ;         // restore parent dialog
      this->dWin->SetDialogObscured () ;  // re-save parent dialog
      subDlg->RefreshWin () ;             // restore the sub-dialog

      //* Monitor the status of the target device until *
      //* the status changes OR until key input.        *
      done = false ;
      while ( ! done )
      {
         if ( (this->dWin->KeyPeek ( wk )) != ERR )
         {
            this->dWin->GetKeyInput ( wk ) ; // discard the keystroke
            status = (-1) ;      // mount _probably_ failed, user has given up
            done = true ;
         }
         else
         {
            for ( short i = (mtpFilesys ? 2 : dvdFilesys ? 4 : 1) ; i > ZERO ; --i )
               this_thread::sleep_for( aMoment ) ;
            this->fdPtr->GetFilesystemStats ( newCwd, fsStats ) ;

            if ( fsStats.isMounted )
            {
               status = OK ;     // mount was successful, alert caller to set CWD
               done = true ;
            }
            else if ( --deadman <= ZERO )
            {
               status = (-1) ;
               done = true ;
            }
         }
      }
      this->fdPtr->DeleteFile ( tmpFile ) ; // delete the temp file
   }

   return status ;

}  //* End cmMountFilesystem() *

//*************************
//* cmDismountFilesystem  *
//*************************
//********************************************************************************
//* Called only by CmdMount(). Un-mount the specified filesystem.                *
//*  1) Caller has verified that the target directory is a mountpoint AND that   *
//*     a storage device is mounted at that mountpoint.                          *
//*  2) Because a device should not be unmounted when someone is using it,       *
//*     caller has tested whether the application's CWD was on the target        *
//*     device. If it was, a new CWD has been set to ensure that there is no     *
//*     conflict.                                                                *
//*     Another application or process may be using data on the target device,   *
//*     but we can't determine that ahead of time. For this reason, we use       *
//*     the '--lazy' option which performs the unmount immediately, but waits    *
//*     to do the /etc/mtab cleanup until the filesystem is no longer busy.      *
//*                                                                              *
//* Input  : trgMnt   : filespec of mountpoint (for "real" filesystems), or      *
//*                     the URI (for virtual MTP/BVFS filesystems)               *
//*          subDlg   : handle to sub-dialog window                              *
//*                     (we must refresh after launch)                           *
//*          super    : 'true' operation is executed with superuser privilege    *
//*                     'false' operation is executed with user privilege        *
//*                                                                              *
//* Returns: OK     if filesystem successfully unmounted                         *
//*          -1     if user key input received                                   *
//*                   (probably unmount failed and user got tired of waiting)    *
//********************************************************************************
//* Programmer's Note: The sync() library function is called to ensure that      *
//* all cached data have been written to the device. Note that the 'umount'      *
//* command would handle this automagically, except that we use the "lazy"       *
//* write to avoid conflicts with other processes that might be holding the      *
//* device open which would make our application appear to hang. At the least,   *
//* we want to guarantee that all writes performed by THIS process have been     *
//* flushed to the target device. We can then continue execution while other     *
//* processes continue without interference.                                     *
//*                                                                              *
//* This issue arises when a slow (flash-memory) device is declared unmounted,   *
//* but all data have not yet been written. This is _especially_ true when       *
//* the target is a large-capacity USB flash drive or SD memory card, and        *
//* _most especially_ true if the target is a journalling filesystem i.e. a      *
//* multi-level cache.                                                           *
//********************************************************************************

short FileMangler::cmDismountFilesystem ( const gString& trgMnt, 
                                          NcDialog* subDlg, bool super )
{
   fileSystemStats fsStats ;     // determine whether the target is mounted
   short status = (-1) ;         // return value (see above)

   gString fPath,
           argList,
           tmpFile ;
   bool mtpFilesys = bool(((trgMnt.find( mtpURI )) == ZERO) ? true : false) ;
   bool dvdFilesys = bool(((trgMnt.find( optiURI )) >= ZERO) ? true : false) ;

   //* Flush the kernel buffers to ensure that all cache data have been    *
   //* written to target. (This may be very slow under some circumstances.)*
   sync () ;

   //* If a virtual filesystem URI specified *
   if ( mtpFilesys || dvdFilesys )
   {
      if ( super != false )   // superuser privilege
      {
         fPath = "sudo" ;
         argList.compose( "gio mount --unmount '%S'", trgMnt.gstr() ) ;
      }
      else                    // user privilege
      {
         fPath = "gio" ;
         argList.compose ( "mount --unmount '%S'", trgMnt.gstr() ) ;
      }
   }

   //* Else assume target is a "real" filesystem *
   else
   {
      if ( super != false )   // superuser privilege
      {
         fPath = "sudo" ;
         argList.compose( "umount --lazy '%S'", trgMnt.gstr() ) ;
      }
      else                    // user privilege
      {
         fPath = "umount" ;
         argList.compose ( "--lazy '%S'", trgMnt.gstr() ) ;
      }
   }

   this->CreateTempname ( tmpFile ) ;  // Create a temp file to capture sys info
   DC_Emulator dce( tmpFile, fPath, argList, true ) ;

   //* Scan the system configuration to define *
   //* the new terminal window.                *
   //* Specify window size (small is cute here)*
   dce.DefineChildEmulator ( 56, 8 ) ;
   this->fdPtr->DeleteFile ( tmpFile ) ; // delete the temp file

   //* Open the new terminal window *
   dce.OpenTermwin () ;

   //* Wait a moment in case child process writes to the parent window.*
   // Programmer's Note: We do not like calling the NCurses primitive here, 
   // but the just-launched terminal emulator may completely trash the display.
   chrono::duration<short, std::milli>aMoment( 1000 ) ;
   this_thread::sleep_for( aMoment ) ;
   nc.ClearScreen () ;
   this->dWin->RefreshWin () ;         // restore parent dialog
   this->dWin->SetDialogObscured () ;  // re-save parent dialog
   subDlg->RefreshWin () ;             // restore the sub-dialog

   //* Monitor the status of the target device until *
   //* the status changes OR until key input.        *
   wkeyCode wk ;
   bool done = false ;
   while ( ! done )
   {
      this_thread::sleep_for( aMoment ) ;
      if ( (this->dWin->KeyPeek ( wk )) != ERR )
      {
         this->dWin->GetKeyInput ( wk ) ; // discard the keystroke
         status = (-1) ;      // unmount _probably_ failed, user has given up
         done = true ;
      }
      else
      {
         this->fdPtr->GetFilesystemStats ( trgMnt, fsStats ) ;
         if ( ! fsStats.isMounted )
         {
            status = OK ;     // unmount was successful
            done = true ;
         }
      }
   }

   return status ;

}  //* End cmDismountFilesystem() *

//*************************
//*   Scan4Filesystems    *
//*************************
//********************************************************************************
//* Called only by CmdMount().                                                   *
//*                                                                              *
//* Scan the system for "real" (physical storage) filesystems.                   *
//* 1) Scan the system for mounted filesystems. (/proc/self/mountinfo)           *
//* 2) Scan 'fstab' for defined mountpoints.    (/etc/fstab)                     *
//* 3) Add the mountpoints from config file, then eliminate duplicate entries.   *
//* 4) Stat each entry to determine whether it is actually mounted.              *
//*                                                                              *
//* Scan the system for mtp/gvfs virtual filesystems.                            *
//* 1) Scan all USB devices and select "mtp" devices.                            *
//* 2) Gather information on each of the selected devices.                       *
//*                                                                              *
//* Input  : fsCount : (by reference, initial value ignored)                     *
//*                                                                              *
//* Returns: pointer to dynamically-allocated array of fsInfo objects            *
//*          It is caller's responsibility to free the dynamic allocation.       *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* -- "Real" filesystems are those which are physical (or logical) devices.     *
//*    We must explicitly filter out the logical devices such as 'tmpfs',        *
//*    'cgroup', 'selinuxfs' etc, as well as the '/' (root) directory which      *
//*    cannot be mounted or unmounted anyway.                                    *
//*                                                                              *
//* -- There is a bug in 'findmnt' which ignores the "--real" option when the    *
//*    "--list" formatting option is used.                                       *
//* -- Note however that the "--df" option which _implies_ both the "--list"     *
//*    and "--real" options actually does filter out the pseudo-filesystems.     *
//* -- In the first pass at this functionality, we pre-allocate a fixed number   *
//*    of 'fsInfo' objects so we don't have to scan the raw data twice. If       *
//*    this becomes a problem, we can count the records before allocating the    *
//*    'fsInfo' array.                                                           *
//*                                                                              *
//* -- When un-mounting a device, the list is sorted to place _mounted_ items    *
//*    at the top of the list. This _should_ exclude paths which are not         *
//*    detachable devices (see list in staticMount()) which are moved to the     *
//*    bottom of the list.                                                       *
//*                                                                              *
//*  ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---  *
//*                                                                              *
//* Possible Enhancements:                                                       *
//* ----------------------                                                       *
//* -- To report attached, but unmounted filesystems we could use:               *
//*       sudo blkid -o list                                                     *
//*    However, we can't run a root command from inside the application unless   *
//*    we shell out or run it in another window. In user mode:                   *
//*       blkid -o list                                                          *
//*    we get the information, but it is mixed with some garbage such as         *
//*    unallocated clusters on a hard drive, etc.                                *
//*    If the mountpoint column says "(not mounted" AND if we have at least      *
//*    _some_ of the data for fstype, label, UUID, then it's probably good:      *
//*         DEVICE     FSTYPE  LABEL     MOUNTPOINT     UUID                     *
//*         /dev/sdb1  vfat    KINGSTON  (not mounted)  6C55-431E                *
//*    To identify the target mountpoint, we would need some additional magic.   *
//*    Note that the documentation says that this option is deprecated in        *
//*    favor of 'lsblk'.                                                         *
//* -- lsblk --fs   could be used, but the information would be difficult to     *
//*    parse.                                                                    *
//*       lsblk --fs --list -o MOUNTPOINT,LABEL,UUID,FSTYPE,NAME                 *
//*    Using the above options, we would get an entry for the attached-but-      *
//*    not-mounted filesystem.                                                   *
//*    MOUNTPOINT     LABEL       UUID        FSTYPE   NAME                      *
//*    "              KINGSTON    6C55-431E   vfat     sdb1"                     *
//*    The mountpoint should be listed first, but because there is no            *
//*    mountpoint, the first field is blank. The logic would then be that        *
//*    if no mountpoint, but we DO have a filesystem type and a label and/or     *
//*    a UUID, then it is probably an attached-but-not-mounted filesystem.       *
//*    We would need a label in order to create a mountpoint.                    *
//*    Add the '--noheadings' option to simplify things.                         *
//*    Of course, this is all useless because we would have to specify a         *
//*    mountpoint, presumably: /run/media/$USER/LABEL                            *
//*    This is possible, but tricky.                                             *
//*  ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ---  *
//*                                                                              *
//********************************************************************************

fsInfo* FileMangler::cmScan4Filesystems ( short& fsCount )
{
   #define MAX_RECORDS (64)      // (see note above)
   #define VERIFY_MOUNTS (0)     // optionally verify mounts (see note below)

   fsInfo* fsiPtr = new fsInfo[MAX_RECORDS] ; // array of mountpoint records
   fsInfo  fsiSwap ;             // for swapping record pointers
   fmFType fType ;               // determine whether the target exists
   fileSystemStats fsStats ;     // determine whether the target is mounted
   gString tmpPath,              // filespec of temporary file
           cmdBuff,              // command string buffer
           gs, gsd ;             // text formatting
   char  lineBuff[gsDFLTBYTES] ; // input buffer
   short fsiIndex = ZERO,        // index into fsiPtr array
         srcIndx,                // source text index
         trgIndx ;               // target text index

         fsCount = ZERO ;        // initialize caller's variable
   bool  done = false ;          // loop control

   //*******************************************************************
   //* Place user's mountpoints from configuration file at top of list.*
   //*******************************************************************
   for ( short i = ZERO ; i < this->mntcmdCount ; ++i )
   {
      gs = this->MountCmds[i] ;
      gs.copy( fsiPtr[fsiIndex].fsMountpt, gsDFLTBYTES ) ;
      if ( (this->fdPtr->TargetExists ( fsiPtr[fsiIndex].fsMountpt, fType )) 
            && fType == fmDIR_TYPE )
      {  // Programmer's Note: The presence of the mountpoint directory is not 
         // a guarantee that the drive is actually mounted, so we dig deeper.
         if ( ((this->fdPtr->GetFilesystemStats ( gs, fsStats )) == OK)
              && fsStats.isMounted && fsStats.isMountpoint )
         {
            gs = fsStats.device ;
            gs.copy( fsiPtr[fsiIndex].fsDevice, gsDFLTBYTES ) ;
            gs = fsStats.fsType ;
            gs.copy( fsiPtr[fsiIndex].fsType, gsDFLTBYTES ) ;
            fsiPtr[fsiIndex].fsMounted = true ;
         }
      }
      ++fsiIndex ;   // increment the record count
   }

   //* Create a temporary file to capture the raw data.*
   this->CreateTempname ( tmpPath ) ;

   //**************************************************************
   //* Scan the /proc/self/mountinfo file for currently-mounted   *
   //* physical storage devices, (SATA, SSD, USB, SD cards, etc.).*
   //**************************************************************
   const char* fsTemplate = 
   "findmnt --df --noheadings -o SOURCE,FSTYPE,TARGET "
   "| grep -v '^tmpfs' "            // (exclude "tmpfs" filesystems)
   "| grep -v '[[:space:]]/sys' "   // (exculde filesystems mounted under '/sys')
   "| grep -v '[[:space:]]/dev' "   // (exculde filesystems mounted under '/dev')
   "| grep -v '[[:space:]]/var/' "  // (exculde filesystems mounted under '/var') 
   "| grep -v '[[:space:]]/$' "     // (exclude root directory)                   
   "1>>\"%s\" 2>&1" ;
   cmdBuff.compose( fsTemplate, tmpPath.ustr() ) ;
   this->Systemcall ( cmdBuff.ustr() ) ;

   //* Open the temp file and extract the interesting data.*
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      while ( ! done && (fsiIndex < MAX_RECORDS) )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
         {
            //* Extract device filespec.*
            srcIndx = trgIndx = ZERO ;
            while ( (srcIndx < gsDFLTBYTES) && (trgIndx < gsDFLTBYTES) && 
                    (lineBuff[srcIndx] != SPACE) && (lineBuff[srcIndx] != NULLCHAR) )
               fsiPtr[fsiIndex].fsDevice[trgIndx++] = lineBuff[srcIndx++] ;
            fsiPtr[fsiIndex].fsDevice[trgIndx] = NULLCHAR ;
            while ( (srcIndx < gsDFLTBYTES) && (lineBuff[srcIndx] == SPACE) )
               ++srcIndx ;
            //* Extract device type.*
            trgIndx = ZERO ;
            while ( (srcIndx < gsDFLTBYTES) && (trgIndx < gsDFLTBYTES) && 
                    (lineBuff[srcIndx] != SPACE) && (lineBuff[srcIndx] != NULLCHAR) )
               fsiPtr[fsiIndex].fsType[trgIndx++] = lineBuff[srcIndx++] ;
            fsiPtr[fsiIndex].fsType[trgIndx] = NULLCHAR ;
            while ( (srcIndx < gsDFLTBYTES) && (lineBuff[srcIndx] == SPACE) )
               ++srcIndx ;
            //* Extract mountpoint filespec.*
            trgIndx = ZERO ;
            while ( (srcIndx < gsDFLTBYTES) && (trgIndx < gsDFLTBYTES) && 
                    (lineBuff[srcIndx] != SPACE) && (lineBuff[srcIndx] != NULLCHAR) )
               fsiPtr[fsiIndex].fsMountpt[trgIndx++] = lineBuff[srcIndx++] ;
            fsiPtr[fsiIndex].fsMountpt[trgIndx] = NULLCHAR ;

            // Programmer's Note: For real (block) devices, we can assume that 
            // all listed filesystems are currently mounted because we got them 
            // from the system's list of mounted filesystems; however, we can 
            // optionally verify that they are mounted.
            #if VERIFY_MOUNTS // OPTIONAL MOUNT VERIFICATION
            if ( (this->fdPtr->TargetExists ( fsiPtr[fsiIndex].fsMountpt, fType )) 
                  && fType == fmDIR_TYPE )
            {
               if ( ((this->fdPtr->GetFilesystemStats ( 
                       fsiPtr[fsiIndex].fsMountpt, fsStats )) == OK)
                    && fsStats.isMounted && fsStats.isMountpoint )
               {
                  fsiPtr[fsiIndex].fsMounted = true ;
               }
            }
            #else    // Assume that filesystem is mounted
            fsiPtr[fsiIndex].fsMounted = true ;
            #endif   // VERIFY_MOUNTS

            ++fsiIndex ;   // increment the record count
         }
         else           // end-of-file found
            done = true ;
      }
      ifs.close() ;           // close the file
   }

   //********************************************************
   //* Scan the /etc/fstab file for mountpoint definitions, *
   //* (may, or may not be currently mounted).              *
   //********************************************************
   //* Discard current contents of temp file.*
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
      ofs.close() ;

   const char* const fstabTemplate = 
         "findmnt --fstab --noheadings -o SOURCE,FSTYPE,TARGET "
         "| grep -v 'none$' "             // (exclude swap partition)
         "| grep -v '[[:space:]]/$' "     // (exclude root directory)                   
         "1>>\"%s\" 2>>\"%s\"" ;
   cmdBuff.compose( fstabTemplate, tmpPath.ustr(), tmpPath.ustr() ) ;
   this->Systemcall ( cmdBuff.ustr() ) ;
   //* Open the temp file and extract the interesting data.*
   ifs.open( tmpPath.ustr(), ifstream::in ) ; // open the file
   if ( ifs.is_open() )                           // if file opened
   {
      done = false ;

      while ( ! done && (fsiIndex < MAX_RECORDS) )
      {
         ifs.getline ( lineBuff, gsDFLTBYTES, NEWLINE ) ;// read a source line
         if ( ifs.good() || ifs.gcount() > ZERO )        // if a good read
         {
            //* Extract device filespec. (may be a UUID or other identifier) *
            srcIndx = trgIndx = ZERO ;
            while ( (srcIndx < gsDFLTBYTES) && (trgIndx < gsDFLTBYTES) && 
                    (lineBuff[srcIndx] != SPACE) && (lineBuff[srcIndx] != NULLCHAR) )
               fsiPtr[fsiIndex].fsDevice[trgIndx++] = lineBuff[srcIndx++] ;
            fsiPtr[fsiIndex].fsDevice[trgIndx] = NULLCHAR ;
            while ( (srcIndx < gsDFLTBYTES) && (lineBuff[srcIndx] == SPACE) )
               ++srcIndx ;
            //* Extract device type.*
            trgIndx = ZERO ;
            while ( (srcIndx < gsDFLTBYTES) && (trgIndx < gsDFLTBYTES) && 
                    (lineBuff[srcIndx] != SPACE) && (lineBuff[srcIndx] != NULLCHAR) )
               fsiPtr[fsiIndex].fsType[trgIndx++] = lineBuff[srcIndx++] ;
            fsiPtr[fsiIndex].fsType[trgIndx] = NULLCHAR ;
            while ( (srcIndx < gsDFLTBYTES) && (lineBuff[srcIndx] == SPACE) )
               ++srcIndx ;
            //* Extract mountpoint filespec.*
            trgIndx = ZERO ;
            while ( (srcIndx < gsDFLTBYTES) && (trgIndx < gsDFLTBYTES) && 
                    (lineBuff[srcIndx] != SPACE) && (lineBuff[srcIndx] != NULLCHAR) )
               fsiPtr[fsiIndex].fsMountpt[trgIndx++] = lineBuff[srcIndx++] ;
            fsiPtr[fsiIndex].fsMountpt[trgIndx] = NULLCHAR ;

            //* Determine whether target filesystem is currently mounted.*
            if ( (this->fdPtr->TargetExists ( fsiPtr[fsiIndex].fsMountpt, fType )) 
                  && fType == fmDIR_TYPE )
            {
               if ( ((this->fdPtr->GetFilesystemStats ( 
                       fsiPtr[fsiIndex].fsMountpt, fsStats )) == OK)
                    && fsStats.isMounted && fsStats.isMountpoint )
               {
                  fsiPtr[fsiIndex].fsMounted = true ;
               }
            }
            ++fsiIndex ;   // increment the record count
         }
         else           // end-of-file found
            done = true ;
      }
      ifs.close() ;           // close the file
   }

   //********************************************************
   //* Scan for optical-drive (DVD/CD/Blu-Ray) filesystems. *
   //* (may, or may not contain disc media).                *
   //* (media may, or may not be currently mounted).        *
   //********************************************************
   usbDevice *usbdPtr = NULL ;
   short dvdCount = this->fdPtr->DVD_DeviceStats ( usbdPtr, true ) ;
   for ( short dvdIndex = ZERO ; dvdIndex < dvdCount ; ++dvdIndex )
   {
      //* Report optical drives which contain media disc.  *
      //* a) If access is through a URI instead of a mount *
      //*    path, URI will be listed by caller.           *
      //* b) If access is through a mountpoint filespec    *
      //*    mountpoint will be listed by caller.          *
      //* c) For a drive which has neither a URI nor a     *
      //*    filespec, it either has no media, or the tray *
      //*    is open.                                      *
      if ( usbdPtr[dvdIndex].optiDrive )
      {
         gs = usbdPtr[dvdIndex].devpath ;
         gs.copy( fsiPtr[fsiIndex].fsDevice, FS_BYTES ) ;
         gs = usbdPtr[dvdIndex].uri ;
         gs.copy( fsiPtr[fsiIndex].fsMountpt, FS_BYTES ) ;
         if ( fsiPtr[fsiIndex].fsMountpt[0] == NULLCHAR )
         {
            gs = usbdPtr[dvdIndex].mntpath ;
            gs.copy( fsiPtr[fsiIndex].fsMountpt, FS_BYTES ) ;
         }
         fsiPtr[fsiIndex].fsMounted = usbdPtr[dvdIndex].mounted ;
         gs = usbdPtr[dvdIndex].fstype ;
         gs.copy( fsiPtr[fsiIndex].fsType, FS_BYTES ) ;
         gs = usbdPtr[dvdIndex].uri ;
         gs.copy( fsiPtr[fsiIndex].fsUri, FS_BYTES ) ;
         fsiPtr[fsiIndex].fsDvdMedia = usbdPtr[dvdIndex].has_media ;
         if ( fsiPtr[fsiIndex].fsMountpt[0] != NULLCHAR )
            fsiPtr[fsiIndex].fsDvd = true ; // indicates optical-drive filesystem

         //* If either URI or mountpoint found, retain the record *
         if ( fsiPtr[fsiIndex].fsMountpt[0] != NULLCHAR )
            ++fsiIndex ;    // filesystem record initialized
      }
   }
   if ( usbdPtr != NULL )     // release the dynamic memory allocation
   { delete [] usbdPtr ; usbdPtr = NULL ; }

   //********************************************************
   //* Scan for mtp/gvfs filesystems attached via USB.      *
   //* (may, or may not be currently mounted).              *
   //********************************************************
   usbdPtr = NULL ;
   short usbCount = this->fdPtr->USB_DeviceStats ( usbdPtr, true ) ;
   for ( short usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
   {
      gs = usbdPtr[usbIndex].devpath ;
      gs.copy( fsiPtr[fsiIndex].fsDevice, FS_BYTES ) ;
      gs = usbdPtr[usbIndex].uri ;
      gs.copy( fsiPtr[fsiIndex].fsMountpt, FS_BYTES ) ;
      fsiPtr[fsiIndex].fsMounted = usbdPtr[usbIndex].mounted ;
      gs = usbdPtr[usbIndex].fstype ;
      gs.copy( fsiPtr[fsiIndex].fsType, FS_BYTES ) ;
      gs = usbdPtr[usbIndex].uri ;
      gs.copy( fsiPtr[fsiIndex].fsUri, FS_BYTES ) ;
      fsiPtr[fsiIndex].fsMtp = true ;     // indicates MTP/GVfs filesystem

      ++fsiIndex ;    // filesystem record initialized
   }
   if ( usbdPtr != NULL )     // release the dynamic memory allocation
   { delete [] usbdPtr ; usbdPtr = NULL ; }

   //********************************
   //* Eliminate duplicate entries. *
   //********************************
   fsCount = fsiIndex ;       // initialize the return value (record count)

   for ( fsiIndex = (fsCount - 1) ; fsiIndex > ZERO ; --fsiIndex )
   {
      gs  = fsiPtr[fsiIndex].fsMountpt ;
      gsd = fsiPtr[fsiIndex].fsDevice ;

      for ( short indx = ZERO ; indx < fsiIndex ; ++indx )
      {
         //* If entries reference the same media/device, delete *
         //* the later entry and shift trailing entries upward. *
         if ( ((gs.gschars() > 1) && (*fsiPtr[indx].fsMountpt != NULLCHAR) &&
               (gs.compare( fsiPtr[indx].fsMountpt )) == ZERO) ||
              ((gsd.gschars() > 1) && (*fsiPtr[indx].fsDevice != NULLCHAR) && 
               (gsd.compare( fsiPtr[indx].fsDevice )) == ZERO) )
         {
            //* Copy the fsDVD and fsDvdMedia flags to the surviving record *
            if ( fsiPtr[fsiIndex].fsDvd || fsiPtr[fsiIndex].fsDvdMedia )
            {
               fsiPtr[indx].fsDvd = true ;
               fsiPtr[indx].fsDvdMedia = fsiPtr[fsiIndex].fsDvdMedia ;
            }

            trgIndx = fsiIndex ;    // overwrite the duplicate record 
            srcIndx = trgIndx + 1 ; // with the following record (if any)
            while ( srcIndx < fsCount )
               fsiPtr[trgIndx++] = fsiPtr[srcIndx++] ;
            --fsCount ;
            break ;
         }
      }  // inner loop
   }     // outer loop

   //*****************************************************************
   //* Sort with mounted-but-unmountable filesystems at top of list, *
   //* and 'statically mounted' filesystems at bottom of list.       *
   //* Yes, this is a lot of work for a small reward, but it's fancy.*
   //*****************************************************************
   trgIndx = fsCount - 1 ;
   for ( trgIndx = fsCount - 1 ; trgIndx > ZERO ; --trgIndx )
   {
      gs  = fsiPtr[trgIndx].fsMountpt ;
      if ( !(staticMount ( gs )) )
         break ;
   }
   if ( trgIndx > ZERO )
   {
      for ( srcIndx = ZERO ; srcIndx < trgIndx ; ++srcIndx )
      {
         gs = fsiPtr[srcIndx].fsMountpt ;
         if ( (staticMount ( gs )) )
         {
            fsiSwap = fsiPtr[trgIndx] ;
            fsiPtr[trgIndx--] = fsiPtr[srcIndx] ;
            fsiPtr[srcIndx] = fsiSwap ;
         }
      }
   }
   do    // step over static mountpoints
   {
      gs = fsiPtr[trgIndx].fsMountpt ;
      if ( !(staticMount( gs )) )
         break ;
   }
   while ( --trgIndx > ZERO ) ;
   if ( trgIndx > ZERO )
   {
      srcIndx = trgIndx ;
      trgIndx = ZERO ;
      do
      {
         //* Find the next non-mounted record. *
         for ( ; trgIndx < srcIndx ; ++trgIndx )
         { if ( ! fsiPtr[trgIndx].fsMounted ) { break ; } }
         //* Find the next mounted record. *
         for ( ; srcIndx > trgIndx ; --srcIndx )
         { if ( fsiPtr[srcIndx].fsMounted ) { break ; } }
         //* Swap the records *
         if ( srcIndx > trgIndx )
         {
            fsiSwap = fsiPtr[trgIndx] ;
            fsiPtr[trgIndx++] = fsiPtr[srcIndx] ;
            fsiPtr[srcIndx--] = fsiSwap ;
         }
      }
      while ( srcIndx > trgIndx ) ;
   }

   this->fdPtr->DeleteFile ( tmpPath ) ;  // delete the temporary file

   return fsiPtr ;      // return pointer to dynamic allocation

}  //* End cmScan4Filesystems() *

//*************************
//*      staticMount      *
//*************************
//********************************************************************************
//* Non-member method:                                                           *
//* ------------------                                                           *
//* Called only by cmScan4Filesystems() to move statically mounted filesystems   *
//* i.e. filesystems which cannot be unmounted without causing system instability*
//* to the bottom of the display list. This is not a definitive list, but covers *
//* the most likey items for a 'standard' system.                                *
//*                                                                              *
//* Input  : gsMntPt : (by reference) mountpoint string for the filesystem       *
//*                    to be tested. Note that an empty string indicates that    *
//*                    the mountpoint is unknown.                                *
//*                                                                              *
//* Returns: 'true'  if filesystem mountpoint is one of those in the list        *
//*          'false' if filesystem mountpoint is not in the 'static' list        *
//********************************************************************************

static bool staticMount ( const gString& gsMntPt )
{
   const short staticFsCnt = 4 ;
   const wchar_t* staticFs[staticFsCnt] =
   {
      L"/home", L"/boot", L"/boot/efi", L"/run/user/1000/doc"
   } ;
   bool isSMnt = false ;                  // return value

   if ( (gsMntPt.gschars()) > 1 )
   {
      for ( short f = ZERO ; f < staticFsCnt ; ++f )
      { if ( (gsMntPt == staticFs[f]) ) { isSMnt = true ; break ; } }
   }
   return isSMnt ;

}  //* End staticMount() *

//*************************
//*    CmdOpticalMedia    *
//*************************
//********************************************************************************
//* Eject or close the tray for optical drives (DVD/CD/Blu-Ray).                 *
//*                                                                              *
//* Notes:                                                                       *
//* a) Not all optical drives can be closed under program control.               *
//* b) Some media will auto-mount when the tray is closed. Other media will      *
//*    need to be mounted explicitly.                                            *
//* c) Multi-disc drives are not fully supported because we do not attempt to    *
//*    select the specific disc to be ejected.                                   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdOpticalMedia ( void )
{
   gString gsTrgPath,            // mountpoint filespec of target (or "")
           gsTrgDev ;            // device driver filespec of target
   const char* devPtr = NULL ;   // pointer to target filespec
   bool closeTray = false,       // open/close flag
        toggleTray = false,      // if only one drive, toggle its state
        abort = false ;          // 'true' if user aborted operation

   //* Get stat data on all attached drives.*
   usbDevice *usbdPtr = NULL ;
   short usbCount = this->fdPtr->DVD_DeviceStats ( usbdPtr, true, true ),
         dvdCount = ZERO ;

   // Scan for all drives with "can_eject" flag set *
   //* and move them to the top of the array.       *
   for ( short srcIndex = ZERO ; srcIndex < usbCount ; ++srcIndex )
   {
      if ( usbdPtr[srcIndex].can_eject )
      {
         if ( srcIndex != dvdCount)
            usbdPtr[dvdCount] = usbdPtr[srcIndex] ;
         ++dvdCount ;
      }
   }

   //* If eject-able drive(s) identified *
   if ( dvdCount > ZERO )
   {
      const short seIndex = this->fdPtr == this->fWin ? dwFileSE : dwFileSEr,
                  dlgRows = MIN_ROWS - 7,       // dialog size and position
                  dlgCols = this->ic[seIndex].cols - 2,
                  ulY     = this->ic[seIndex].ulY + 1,
                  ulX     = this->ic[seIndex].ulX + 1,
                  pathCols = dlgCols - 2 ;      // max length of path display string
      attr_t dColor = this->cfgOpt.cScheme.sd,  // dialog base color attribute
             eColor = this->cfgOpt.cScheme.em ; // emphasized-text color

      //* Construct the display list *
      gString gsItems[dvdCount] ;
      const char* seboxData[dvdCount] ;
      attr_t seboxColors[dvdCount] ;
      ssetData sData( seboxData, seboxColors, dvdCount, ZERO, true ) ;
      short leftshift ;

      for ( short i = ZERO ; i < dvdCount ; ++i )
      {
         // If a mountpoint specified, combine it with device-driver filespec *
         if ( usbdPtr[i].mntpath[0] != NULLCHAR )
            gsItems[i].compose( "%s (%s)", usbdPtr[i].mntpath, usbdPtr[i].devpath ) ;
         //* Else, use the device-driver filespec only *
         else
            gsItems[i] = usbdPtr[i].devpath ;

         //* Adjust string width to fit the control *
         //* and include it in the display list.    *
         leftshift = pathCols - gsItems[i].gscols() ; // < ZERO == overflow
         if ( leftshift < ZERO )                // if string too wide, shift left
            gsItems[i].shiftCols( leftshift ) ;
         else                                   // else, pad to width of control
            gsItems[i].padCols( pathCols ) ;
         seboxData[i] = gsItems[i].ustr() ;
         seboxColors[i] = this->cfgOpt.cScheme.tf ;
      }

      //* Create a list of needed controls *
      enum dpCtrls { selSE, ejPB, clPB, canPB, dpcCOUNT } ;
      InitCtrl ic[dpcCOUNT] =            // control definitions
      {
      { //* 'Select' scroll ext - - - - - - - - - - - - - - - - - - - -  selSE *
         dctSCROLLEXT,                 // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (na)
         false,                        // rbSelect:  (n/a)
         2,                            // ulY:       upper left corner in Y
         ZERO,                         // ulX:       upper left corner in X
         6,                            // lines:     control lines
         dlgCols,                      // cols:      control columns
         NULL,                         // dispText:  n/a
         dColor,                       // nColor:    non-focus border color
         dColor,                       // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     
         ZERO,                         // labY:      offset from control's ulY
         ZERO,                         // labX       offset from control's ulX
         ddBoxTYPES,                   // exType:    (n/a)
         dvdCount,                     // scrItems:  number of elements in text/color arrays
         ZERO,                         // scrSel:    (n/a)
         seboxColors,                  // scrColor:  single-color data display
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ejPB],                    // nextCtrl:  link in next structure
      },
      {  //* 'EJECT' pushbutton   - - - - - - - - - - - - - - - - - - - - ejPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dlgRows - 2),           // ulY:       upper left corner in Y
         short(dlgCols / 2 - 16),      // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         9,                            // cols:      control columns
         "  ^EJECT  ",                 // dispText:  
         this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
         this->cfgOpt.cScheme.pf,      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[clPB],                    // nextCtrl:  link in next structure
      },
      {  //* 'CLOSE' pushbutton   - - - - - - - - - - - - - - - - - - -   clPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[ejPB].ulY,                 // ulY:       upper left corner in Y
         short(ic[ejPB].ulX + 12),     // ulX:
         1,                            // lines:     (n/a)
         9,                            // cols:      control columns
         "  ^CLOSE  ",                 // dispText:  
         this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
         this->cfgOpt.cScheme.pf,      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[canPB],                   // nextCtrl:  link in next structure
      },
      {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[clPB].ulY,                 // ulY:       upper left corner in Y
         short(ic[clPB].ulX + 12),     // ulX:
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " CA^NCEL ",                  // dispText:  
         this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
         this->cfgOpt.cScheme.pf,      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      },
      } ;

      //* Save the application's display data *
      this->dWin->SetDialogObscured () ;

      //* Initial parameters for dialog window, passed to sub-methods.*
      InitNcDialog dInit
      ( 
         dlgRows,              // number of display lines
         dlgCols,              // number of display columns
         ulY,                  // Y offset from upper-left of terminal 
         ulX,                  // X offset from upper-left of terminal 
         NULL,                 // dialog title (see below)
         ncltSINGLE,           // border line-style
         dColor,               // border color attribute
         dColor,               // interior color attribute
         ic                    // pointer to list of control definitions
      ) ;


      //* Instantiate the dialog window *
      NcDialog* dp = new NcDialog ( dInit ) ;

      if ( (dp->OpenWindow()) == OK )
      {
         dp->SetDialogTitle ( "  Eject Optical Media  ", eColor ) ;

         //* Make visual connections to dialog border and     *
         //* initialize display data for dctSCROLLEXT control *
         cdConnect cdConn ;
         cdConn.connection = true ;
         cdConn.ul2Left = cdConn.ur2Right = cdConn.ll2Left = cdConn.lr2Right = true ;
         dp->ConnectControl2Border ( selSE, cdConn ) ;
         dp->SetScrollextText ( selSE, sData ) ;

         //* Write static data *
         winPos wp( 8, 5 ) ;
         dp->WriteParagraph ( 1, 5, 
            "Highlight the desired target device, then select \"Eject\" or \"Close\".",
            eColor ) ;
         dp->WriteParagraph ( wp, 
            "♦ Some media will auto-mount when tray is closed, while other media\n"
            "  must be mounted as a separate operation.\n"
            "♦ Note that some drive trays cannot be closed under program control.",
            dColor ) ;

         dp->RefreshWin () ;

         uiInfo Info ;                 // user interface data returned here
         short  icIndex = ZERO ;       // index of control with input focus
         bool   done = false ;         // loop control

         //* If there is only one device listed, *
         //* set Scrollext control as inactive.  *
         if ( dvdCount == 1 )
         {
            icIndex = dp->NextControl () ;
            dp->ControlActive ( selSE, false ) ;
         }

         //************************
         //* Talk to the animals. *
         //************************
         while ( ! done )
         {
            //* If focus is currently on the Scroll Box *
            if ( ic[icIndex].type == dctSCROLLEXT )
            {
               Info.viaHotkey = false ;      // ignore hotkey data
               icIndex = dp->EditScrollext ( Info ) ;
            }

            //* If focus is currently on a Pushbutton   *
            else if ( ic[icIndex].type == dctPUSHBUTTON )
            {
               if ( Info.viaHotkey )
                  Info.HotData2Primary () ;
               else
                  icIndex = dp->EditPushbutton ( Info ) ;

               if ( Info.dataMod != false )
               {
                  //* Eject or close the tray *
                  if ( Info.ctrlIndex == ejPB || Info.ctrlIndex == clPB )
                  {
                     short seindx = dp->GetScrollextSelect ( selSE ) ;
                     gsTrgDev = usbdPtr[seindx].devpath ;
                     //* Close the tray *
                     if ( Info.ctrlIndex == clPB )
                        closeTray = true ;
                     devPtr = gsTrgDev.ustr() ;
                     gsTrgPath = usbdPtr[seindx].mntpath ;
                  }

                  else if ( Info.ctrlIndex == canPB ) // abort selection
                     abort = true ;

                  done = true ;
               }
            }

            if ( done == false && Info.viaHotkey == false )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }  // while(!done)
      }
      else           // unable to open window (likely memory-allocation error)
         ;

      if ( dp != NULL )
         delete ( dp ) ;                        // close the window
   }
   else
      abort = true ;          // no target drives found

   if ( usbdPtr != NULL )     // release the dynamic allocation
   { delete [] usbdPtr ; usbdPtr = NULL ; }

   //* Restore application's display data *
   this->dWin->RefreshWin () ;

   //* Access the target drive *
   if ( ! abort )
   {
      chrono::duration<short>aWhile( 2 ) ;

      //* When ejecting the tray, get mountpoints for active and inactive      *
      //* display controls. If either window currently contains data from the  *
      //* target path, device is "busy", so change its CWD to user's home path.*
      if ( (gsTrgPath.gschars()) > 1 )
      {
         gString homePath( "%s", std::getenv( "HOME" ) ),
                 currPath ;
         this->fdPtr->GetPath ( currPath ) ;
         if ( (currPath.find( gsTrgPath.gstr(), ZERO, true )) == ZERO )
         {
            this->fdPtr->SetDirectory ( homePath ) ;
         }
         if ( (this->DualwinMode ()) )
         {
            this->altPtr->GetPath ( currPath ) ;
            if ( (currPath.find( gsTrgPath.gstr(), ZERO, true )) == ZERO )
            {
               this->altPtr->SetDirectory ( homePath ) ;
               this->altPtr->RedrawCurrDir ( false ) ;
            }
         }
      }

      //* Note that it may take several seconds for the      *
      //* operation to complete, so display a status message.*
      attr_t monoColor[2] = { dtbmNFcolor, ZERO } ;
      dtbmData uData( " Waiting for Target Drive To Respond...", monoColor ) ;
      this->dWin->DisplayTextboxMessage ( dwMsgsTB, uData ) ;

      //* If target drive is busy when operation is executed, *
      //* system may spew crap all over the terminal window.  *
      this->dWin->SetDialogObscured () ;

      this->fdPtr->EjectMedia ( devPtr, closeTray, toggleTray ) ;

      //* Wait a moment for the disc to spin up *
      //* and/or the tray to stop moving.       *
      this_thread::sleep_for( aWhile ) ;

      //* If synch-lock is engaged, disengage it.*
      //* This refreshes the display.            *
      if ( this->synchLock )
         this->CmdSynchLock () ;
      else
         this->dWin->RefreshWin () ;

      //* If the mountpoint which has been removed is displayed   *
      //* in either window, refresh the window's contents.        *
      if ( (gsTrgPath.gschars()) > 1 )
      {
         gString homePath( "%s", std::getenv( "HOME" ) ),
                 currPath ;
         this->fdPtr->GetPath ( currPath ) ;
         if ( (gsTrgPath.compare( currPath.gstr(), true, 
                                  (currPath.gschars() - 1) )) == ZERO )
         {
            this_thread::sleep_for( aWhile ) ;
            this->fdPtr->RefreshCurrDir ( true, true ) ;
            this->dWin->DisplayTextboxMessage ( dwMsgsTB, uData ) ;
         }
         if ( (this->DualwinMode ()) )
         {
            this->altPtr->GetPath ( currPath ) ;
            if ( (gsTrgPath.compare( currPath.gstr(), true, 
                                     (currPath.gschars() - 1) )) == ZERO )
            {
               this_thread::sleep_for( aWhile ) ;
               this->altPtr->RefreshCurrDir ( false, true ) ;
               this->dWin->DisplayTextboxMessage ( dwMsgsTB, uData ) ;
            }
         }
      }

      //* If tray has been closed AND if the new disc auto-mounts,    *
      //* displayed data may be stale. Refresh the window contents.   *
      else if ( closeTray )
      {
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, uData ) ;

         //* Poll for visible media *
         for ( short rescan = 7 ; rescan > ZERO ; --rescan )
         {
            this_thread::sleep_for( (aWhile * 4) ) ;

            usbCount = this->fdPtr->DVD_DeviceStats ( usbdPtr, true ) ;
            for ( short i = ZERO ; i < usbCount ; ++i )
            {
               if ( (gsTrgDev.compare( usbdPtr[i].devpath )) == ZERO )
               {
                  if ( usbdPtr[i].has_media )
                     rescan = ZERO ;      // end outer loop
                  break ;
               }
            }

            delete [] usbdPtr ;     // release the dynamit allocation
            usbdPtr = NULL ;
         }

         this->fdPtr->RefreshCurrDir ( true, true ) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, uData ) ;
         if ( (this->DualwinMode ()) )
         {
            this->altPtr->RefreshCurrDir ( false ) ;
         }
      }

      uData.textData[0] = NULLCHAR ;   // erase the message
      this->dWin->DisplayTextboxMessage ( dwMsgsTB, uData ) ;
   }

}  //* End CmdOpticalMedia() *

//*************************
//*    CmdLocateFile      *
//*************************
//********************************************************************************
//* Given the first character of the filename, scan the list of displayed        *
//* filenames, and if a match is found, move highlight to that file.             *
//*   (Not to be confused with CmdFind() which searches the directory tree.)     *
//*                                                                              *
//* Input  : firstChar: user-provided first character of filename                *
//*                     (target may, or may not exist)                           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Concepts:                                                                    *
//* ---------                                                                    *
//* A) Be helpful, but non-obtrusive.                                            *
//*    -- If not a first-character match, return immediately.                    *
//*    -- Do not open the interactive dialog unless it is actually needed.       *
//* B) Leave non-search characters in the queue for caller.                      *
//*    -- Remove a character from the input queue only if we can use it.         *
//*    -- If input is not ours, search is terminated and caller will process     *
//*       the input.                                                             *
//*                                                                              *
//* Logic:                                                                       *
//* 1) If caller's character is a printing character i.e. a potential filename   *
//*    character, continue to step 2.                                            *
//* 2) Test caller's character against the first character of each filename      *
//*    in the list.                                                              *
//*    a) If a match is found, move the highlight to the matching file and       *
//*       go to step 3.                                                          *
//*    b) If no match, then RETURN with an audible boink but without a screen    *
//*       update. (To the user, this will look like invalid input.)              *
//* 3) If a first-character match is found, wait for the next character to       *
//*    arrive in the input queue. Peek at the character.                         *
//*    a) If it is a printing character, extract the character from the queue.   *
//*       i) If there is a two-character filename match found, move the          *
//*          highlight to the matching file and go to step 4.                    *
//*       ii) If not a two-character match, boink the user and RETURN to         *
//*           caller.                                                            *
//*    b) If not a printing character, leave it in the queue and RETURN to       *
//*       caller.                                                                *
//* 4) A two-character match has been found and highlight is on matching file.   *
//*    a) Open the interactive dialog and display the first two characters of    *
//*       user input.                                                            *
//* 5) Continue to gather input:                                                 *
//*    a) printing characters are appended to the search string                  *
//*    b) DELETE/BACKSPACE remove the most-recent character from the string.     *
//*    c) ESCAPE aborts the search. Highlight is left at its most-recent         *
//*       position                                                               *
//*    d) All other input is left in the queue, search is terminated, and        *
//*       highlight is left at its most-recent position.                         *
//* 6) When search is complete:                                                  *
//*    a) the interactive dialog is closed                                       *
//*    b) highlight is left on most recent matching filename                     *
//*    c) caller may or may not have key input waiting in the queue.             *
//*                                                                              *
//* If in Dual-window Mode AND if synch-lock is engaged, mirror the scan in      *
//* the inactive window. Only valid, _printing_ filename characters are          *
//* echoed in the inactive window.                                               *
//*                                                                              *
//*  -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   *
//* Undocumented Feature:                                                        *
//* This method is documented as a case-insensitive search; however, once        *
//* the dialog is open, the INSERT key toggles case-sensitivity for string       *
//* comparisons. This feature is not user friendly: If we have made a            *
//* case-insensitive match, and then toggle to case-sensitive, the previous      *
//* match may no longer be valid. Besides, we couldn't document this             *
//* functionality without confusing the crap out of the user.                    *
//*                                                                              *
//********************************************************************************

void FileMangler::CmdLocateFile ( wkeyCode& firstChar )
{
   if ( (firstChar.type == wktPRINT) && iswprint ( firstChar.key ) )
   {
      short currCtrl  = this->fdPtr == this->fWin ? dwFileSE : dwFileSEr,
            pathCtrl  = currCtrl == dwFileSE ? dwPathTB : dwPathTBr,
            currIndex ;
      bool  matchFound = false ;

      //* Scroll to the next file which begins with the specified character *
      gString substr ;
      substr.compose( L"%C", &firstChar.key ) ;
      matchFound = this->fdPtr->Scroll2MatchingFile ( substr, currIndex ) ;

      //* If a first-character match was found, wait for the next character.*
      if ( matchFound )
      {
         const short INTERVAL = 250 ;     // key wait interval (mS)
         chrono::duration<short, std::milli>aMoment( INTERVAL ) ;
         wkeyCode wk ;              // key input
         matchFound = false ;       // reset the match flag

         while ( true )
         {
            if ( (this->dWin->KeyPeek ( wk )) != wktERR )
            {
               if ( wk.type == wktPRINT )
               {
                  this->dWin->GetKeyInput ( wk ) ;
                  substr.append( wk.key ) ;
                  matchFound = this->fdPtr->Scroll2MatchingFile ( substr, currIndex ) ;

                  //* If synch-lock is engaged, mirror  *
                  //* the search in the inactive window.*
                  if ( this->synchLock && matchFound )
                     this->uiMonitor ( currCtrl, wk, false ) ;

                  if ( ! matchFound )
                  {
                     //* Handle SPACE character as a command key.*
                     if ( wk.key == nckSPACE )
                        this->dWin->UngetKeyInput ( wk ) ;
                     else
                        this->dWin->UserAlert () ;
                  }
               }
               else
                  ;     // Data is not for us. Leave it for caller.
               break ;
            }
            else
               this_thread::sleep_for( aMoment ) ;    // take a nap
         }

         //* If a second printing character was found AND   *
         //* if we have a two-character match, then open an *
         //* interactive sub-dialog to get additional data. *
         if ( matchFound )
         {
            attr_t   dColor = this->cfgOpt.cScheme.tf,   // text color
                     bColor = dColor | ncrATTR ;         // background color
            int  ssWidth = this->ic[pathCtrl].cols / 2,  // substring display area
                 ssOffset ;                              // substring display offset
            bool sensi = false ;                         // 'true' for case-sensitive compare
            winPos wp( ZERO, ssWidth ) ;                 // substring display position
            InitNcDialog dInit
             ( 1,                         // number of display lines
               this->ic[pathCtrl].cols,   // number of display columns
               this->ic[pathCtrl].ulY,    // Y offset
               this->ic[pathCtrl].ulX,    // X offset 
               NULL,                      // n/a (no title)
               ncltSINGLE,                // n/a (no border)
               dColor,                    // n/a (no border)tribute
               bColor,                    // interior color attribute
               NULL                       // n/a (no controls)
             ) ;

            NcDialog* dp = new NcDialog ( dInit ) ;
            if ( (dp->OpenWindow ()) == OK )
            {  //* Indicate that filename search is in progress *
               dtbmData  msgData( "  Filename Search: Printing characters, BKSP/DEL or ESCAPE keys.  " ) ;
               this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
               dp->WriteString ( wp.ypos, (wp.xpos - 10), "Filename:", bColor ) ;
               bool  done = false ;       // loop control

               while ( !done )
               {  //* Clear the display area.                        *
                  //* It is possible (though unlikely) that user has *
                  //* entered a search string that is wider than our *
                  //* display area. If so, adjust the offset.        *
                  dp->ClearLine ( ZERO, false, wp.xpos ) ;
                  ssOffset = ZERO ;
                  if ( (substr.gscols()) >= ssWidth )
                  {  // Programmer's Note: Yes, this is a lot of work for
                     // small reward, but user-friendliness R Us.
                     int swidth = substr.gscols(), ci ;
                     const short* cwidth = substr.gscols( ci ) ;
                     for ( int i = ZERO ; (swidth >= ssWidth) && i < ci ; ++i )
                     {
                        ++ssOffset ;
                        swidth -= cwidth[i] ;
                     }
                  }
                  dp->WriteString ( wp, &substr.gstr()[ssOffset], dColor, true ) ;

                  //* If data in the key-input queue *
                  if ( (dp->KeyPeek ( wk )) != wktERR )
                  {
                     //* Only printing characters, BKSP/DEL and ESCAPE are *
                     //* processed. Anything else is returned to caller.   *
                     if ( wk.type == wktPRINT )       // printing character
                     {
                        dp->GetKeyInput ( wk ) ;      // retrieve data from queue
                        substr.append( wk.key ) ;     // append to search string
                        matchFound = 
                           this->fdPtr->Scroll2MatchingFile ( substr, currIndex, sensi ) ;

                        //* If synch-lock is engaged, mirror  *
                        //* the search in the inactive window.*
                        if ( this->synchLock && matchFound )
                           this->uiMonitor ( currCtrl, wk, false ) ;

                        if ( ! matchFound )           // no match, audible boink
                        {  //* Erase most-recent character and boink the user *
                           substr.limitChars( (substr.gschars() - 2) ) ;

                           //* Handle SPACE character as a command key.*
                           if ( wk.key == nckSPACE )
                           {
                              dp->UngetKeyInput ( wk ) ;
                              done = true ;  // data belongs to caller
                           }
                           else
                              dp->UserAlert () ;
                        }
                     }
                     else if ( (wk.type == wktFUNKEY) && // one of the special keys
                               ((wk.key == nckESC) ||
                                (wk.key == nckBKSP) ||
                                (wk.key == nckDELETE) ||
                                (wk.key == nckINSERT)) )
                     {
                        dp->GetKeyInput ( wk ) ;      // retrieve data from queue
                        if ( wk.key == nckESC )       // search aborted
                           done = true ;
                        else if ( wk.key == nckINSERT )
                        {
                           sensi = sensi ? false : true ;
                           wchar_t indicator = sensi ? L'*' : L' ' ;
                           dp->WriteChar ( wp.ypos, (wp.xpos - 11), 
                                           indicator, bColor, true ) ;
                        }
                        else
                        {  //* Erase most-recent character from search string,*
                           //* and if string is now empty, we're done.        *
                           substr.limitChars( (substr.gschars() - 2) ) ;

                           //* If synch-lock is engaged, mirror  *
                           //* the search in the inactive window.*
                           if ( this->synchLock )
                           {
                              wk.key = nckDELETE ;
                              this->uiMonitor ( currCtrl, wk, false ) ;
                           }

                           if ( (substr.gschars()) > 1 )
                           {  //* Return to previous matching filename *
                              // Programmer's Note: This may be logically incorrect 
                              // because the currenty-highlighted file still matches at 
                              // chars-minus one for _most_ data, (but it does no harm).
                              this->fdPtr->Scroll2MatchingFile ( substr, currIndex, sensi ) ;
                           }
                           else     // search string is empty
                              done = true ;
                        }
                     }
                     else     // data belongs to caller
                        done = true ;
                  }
                  else
                  {
                     this_thread::sleep_for( aMoment ) ;    // take a nap
                  }
               }     //while()
            }
            if ( dp != NULL )                // close the dialog
               delete dp ;
            this->dWin->RefreshWin () ;      // restore path control
            this->RestoreQuickHelpMsg () ;   // restore application Help message
         }     // two-char match found
      }        // first char match found
      else     // no match, boink the user
         this->dWin->UserAlert () ;
   }           // first char == wktPRINT

}  //* End CmdLocateFile() *

//*************************
//*    CmdKeyBindings     *
//*************************
//********************************************************************************
//* View a list of the current key bindings.                                     *
//*                                                                              *
//* Current key bindings are listed in the format:                               *
//* CUSTOM KEY  DEFAULT KEY  DESCRIPTION                                         *
//* ============================================================================ *
//*     --      nckC_A       select/deselect All files in the current window     *
//*     --      nckC_B       unassigned                                          *
//*     --      nckC_C       Copy selected files to target directory             *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdKeyBindings ( void )
{
   const short ulY      = this->ic[dwMsgsTB].ulY,  // dialog Y position
               ulX      = ZERO,                    // dialog X position
               dlgROWS  = this->actualRows - ulY,  // dialog height
               dlgCOLS  = KEYMAP_DLGX ;            // dialog width
   const attr_t dColor = this->cfgOpt.cScheme.sd,  // dialog base color attribute
                eColor = this->cfgOpt.cScheme.em,  // emphasis color
                seFocus = this->cfgOpt.cScheme.wr; // scrollext focus color
   const char* scrollText[KEYMAP_KEYS] ;           // scrollext text data
   attr_t scrollColors[KEYMAP_KEYS] ;              // scrollext color data
   gString gs, gsEntry ;                           // text formatting and search

   enum Controls : short { dpCLOSE, dpSEARCH, dpSCROLL, dpCONTROLS } ;

   InitCtrl ic[dpCONTROLS] =          // control definitions
   {
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -  dpCLOSE *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      9,                            // cols:      control columns
      "  CLOSE  ",                  // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dpSEARCH]                 // nextCtrl:  link in next structure
   },
   {  //* 'Search' Textbox - - - - - - - - - - - - - - - - - - - -    dpSEARCH *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[dpCLOSE].ulY,              // ulY:       upper left corner in Y
      short(ic[dpCLOSE].ulX + ic[dpCLOSE].cols + 7), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      18,                           // cols:      control columns
      NULL,                         // dispText:  initially empty
      this->cfgOpt.cScheme.tn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.tf | ncrATTR, // fColor:    focus color
      tbPrint,                      // filter:    all printing characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dpSCROLL]                 // nextCtrl:  link in next structure
   },
   { //* 'Bindings' scrollext  - - - - - - - - - - - - - - - - - - -  dpSCROLL *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(dlgROWS - 3),           // lines:     control lines
      dlgCOLS,                      // cols:      control columns
      NULL,                         // dispText:  n/a
      dColor,                       // nColor:    non-focus border color
      seFocus,                      // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      KEYMAP_KEYS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      scrollColors,                 // scrColor:  array of color attributes
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
   } ;
   //* Save the application's display data *
   this->dWin->SetDialogObscured () ;

   //* If the keymap object has not yet been initialized, format    *
   //* the display text and initialize the default command keycodes.*
   //* Note: If there are user-defined keycodes, then the data      *
   //*       will have been initialized during startup.             *
   #if 1    // PRODUCTION
   if ( ! this->keyMap.kmInit )
      this->keyMap.initialize() ;
   #elif 1  // Debugging Only: re-initialize to default values
   this->keyMap.initialize() ;
   for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
      this->keyMap.keydef[i].kuser = this->keyMap.keydef[i].kdflt ;
   this->keyMap.kmInit = false ;
   this->keyMap.initialize( false, true ) ;
   #else    // Debugging Only: re-initialize with two(2) custom keycodes
   this->keyMap.keydef[2].kuser.type = wktFUNKEY ;
   this->keyMap.keydef[2].kuser.key  = nckC_B ;
   this->keyMap.keydef[11].kuser.type = wktFUNKEY ;
   this->keyMap.keydef[11].kuser.key  = nckC_P ;
   this->keyMap.kmInit = false ;
   this->keyMap.initialize( false, true ) ;
   #endif   // Debugging Only

   //* Initialize text-pointer and color attribute arrays for dctSCROLLEXT.*
   for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
   {
      scrollText[i] = this->keyMap.keydef[i].ktext ;
      scrollColors[i] = dColor ;
   }


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( "  CUSTOM KEY   - DEFAULT KEY - DESCRIPTION "
                           "------------------------------ ", eColor ) ;

      //* Make a visual connection for the dctSCROLLEXT control that *
      //* overlaps the dialog window's border.                       *
      cdConnect cdConn ;
      cdConn.ul2Left = true ;
      cdConn.ll2Left = true ;
      cdConn.ur2Right = true ;
      cdConn.lr2Right = true ;
      cdConn.connection = true ;
      dp->ConnectControl2Border ( dpSCROLL, cdConn ) ;

      //* Set the dctSCROLLEXT display data *
      ssetData sData( scrollText, scrollColors, KEYMAP_KEYS, ZERO, false ) ;
      dp->SetScrollextText ( dpSCROLL, sData ) ;

      winPos scntPos( ic[dpSEARCH].ulY, (ic[dpSEARCH].ulX - 6) ) ;
      dp->WriteString ( ic[dpSEARCH].ulY, (ic[dpSEARCH].ulX + ic[dpSEARCH].cols + 1),
                        "Enter search text or a command code.", eColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     keycodeSearch = false,     // 'true' if user entered keycode in textbox
               done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               done = true ;
            }
         }

         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->ViewScrollext ( Info ) ;
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* Manually capture user input for the 'search' textbox control.*
            //* Update the scrollext data to indicate matching entries.      *
            short match = ZERO,                       // number of matching search items
                  firstMatch = -1,                    // index of first matching search item
                  lastRow = ic[dpSCROLL].lines - 2 ;  // last visible scrollext row
            Info.dataMod = false ;                    // reset data-match flag
            while ( true )
            {
               dp->GetKeyInput ( Info.wk ) ;
               if ( (Info.wk.type == wktFUNKEY) &&
                    ((Info.wk.key == nckTAB) || (Info.wk.key == nckDOWN)
                      || (Info.wk.key == nckENTER) || (Info.wk.key == nckSTAB) 
                      || (Info.wk.key == nckUP) || (Info.wk.key == nckESC)) )
               {
                  Info.keyIn = Info.wk.key ; // indicate focus direction
                  break ;                    // edit complete
               }
               else if ( Info.wk.type == wktPRINT )
               {
                  dp->PrevControl () ;
                  dp->GetTextboxText ( dpSEARCH, gs ) ;
                  if ( keycodeSearch )
                  { gs.clear() ; keycodeSearch = false ; }
                  gs.append( Info.wk.key ) ;
                  dp->SetTextboxText ( dpSEARCH, gs ) ;
                  Info.dataMod = true ;
                  dp->NextControl () ;
               }
               else if ( (Info.wk.type == wktFUNKEY) &&
                         ((Info.wk.key == nckDELETE) || (Info.wk.key == nckBKSP)) )
               {
                  dp->PrevControl () ;
                  dp->GetTextboxText ( dpSEARCH, gs ) ;
                  if ( gs.gschars() >= 2 )
                  {
                     if ( keycodeSearch )
                     { gs.clear() ; keycodeSearch = false ; }
                     else
                        gs.limitChars( gs.gschars() - 2 ) ;
                     dp->SetTextboxText ( dpSEARCH, gs ) ;
                     Info.dataMod = true ;
                  }
                  else
                     dp->UserAlert () ;
                  dp->NextControl () ;
               }
               else
               {  //* Keycode is potentially a command-key sequence. *
                  //* Scan the list(s) of valid command keys.        *
                  //* 1) Scan user-defined list (if any).            *
                  //* 2) Scan the default command-key list.          *
                  gs = "INVALID KEYCODE!" ;
                  if ( this->keyMap.kmActive )
                  {
                     for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
                     {
                        if ( Info.wk == this->keyMap.keydef[i].kuser )
                        {
                           // Programmer's Note: Capturing the substring for which
                           // to search begins with the first character in the 
                           // display element and ends before the vertical bar.
                           // Leading and trailing spaces are discarded.
                           // Because the field is narrower than the widest 
                           // substring, it is possible, but unlikely that the 
                           // search will identify multiple matches.
                           dp->PrevControl () ;
                           gs = this->keyMap.keydef[i].ktext ;
                           short offset = gs.find( wcsVERT) ;
                           gs.limitChars( offset ) ;
                           gs.strip() ;
                           dp->SetTextboxText ( dpSEARCH, gs ) ;
                           Info.dataMod = true ;
                           keycodeSearch = true ;
                           dp->NextControl () ;
                           break ;
                        }
                     }
                  }
                  if ( ! Info.dataMod )
                  {
                     for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
                     {
                        if ( Info.wk == this->keyMap.keydef[i].kdflt )
                        {
                           // Programmer's Note: Capturing the substring for which 
                           // to search includes one leading, and one trailing space.
                           // This avoids multiple matches with the human-readable 
                           // version of the command code.
                           dp->PrevControl () ;
                           gs = this->keyMap.keydef[i].ktext ;
                           short offset = (gs.find( '|' )) + 1 ;
                           gs.shiftChars( -(offset) ) ;
                           offset = gs.find( L' ', 1 ) ;
                           gs.limitChars( offset + 1 ) ;
                           dp->SetTextboxText ( dpSEARCH, gs ) ;
                           Info.dataMod = true ;
                           keycodeSearch = true ;
                           dp->NextControl () ;
                           break ;
                        }
                     }
                     if ( ! Info.dataMod )   // user's command not found in table
                        dp->UserAlert () ;
                  }
               }

               //* Rescan the display data for matching text *
               if ( Info.dataMod != false )
               {
                  match = ZERO ;
                  for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
                  {
                     gsEntry = this->keyMap.keydef[i].ktext ;
                     if ( (gsEntry.find( gs.gstr() )) >= ZERO )
                     {
                        scrollColors[i] = seFocus ;
                        //* Be sure first matched item is visible *
                        if ( ++match == 1 )
                        {
                           firstMatch = i ;
                           if ( firstMatch < lastRow )
                              firstMatch = ZERO ;
                        }
                     }
                     else
                        scrollColors[i] = dColor ;
                  }

                  dp->WriteString ( scntPos, "     ", dColor ) ;
                  if ( match > ZERO )
                  {
                     gs.compose( "(%3hd)", &match ) ;
                     dp->WriteString ( scntPos, gs, seFocus ) ;
                  }
                  dp->SetScrollextSelect ( dpSCROLL, firstMatch ) ;
                  dp->RefreshScrollextText ( dpSCROLL, false ) ;
                  dp->RefreshWin () ;
                  Info.dataMod = false ;
               }     // dataMod != false
            }        // while()
         }           // dctTEXTBOX
         
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window
   this->dWin->RefreshWin () ;               // restore application window

}  //* CmdKeyBindings() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

