//********************************************************************************
//* File       : FileMangler.hpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 02-Apr-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: Class definition and miscellaneous constant definitions         *
//*              for the FileMangler application.                                *
//*                                                                              *
//*                                                                              *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//*         Full text of the GPL License may be found in the Texinfo             *
//*         documentation for this program under 'Copyright Notice'.             *
//********************************************************************************

//****************
//* Header Files *
//****************
//* Externally-defined class definitions *
#include "GlobalDef.hpp"               //* NcDialog family of classes
#include "FMgr.hpp"                    //* FMgr class definitions
#include "FileDlg.hpp"                 //* FileDlg class definitions
#include "FmKeymap.hpp"                //* Contains command-key mapping information
#include "FmMenu.hpp"                  //* Application's Menuing system definition

#ifndef FILEMANGLER_INCLUDED
#define FILEMANGLER_INCLUDED

//***************
//* Definitions *
//***************

//* Gather and process command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->appPath    = NULLCHAR ;
      *this->cfgPath    = NULLCHAR ;
      *this->startDir   = NULLCHAR ;
      *this->backPath   = NULLCHAR ;
      *this->localeName = NULLCHAR ;
      this->colorScheme = ncbcCOLORS ;
      this->winMode     = wmTERM_WIN ;
      this->sortOption  = fmNAME_SORT ;
      this->diagPause   = ZERO ;
      this->modeFlag   = this->sortFlag  = this->hideFlag   = this->treeFlag = 
      this->mouseFlag  = this->rootFlag  = this->schemeFlag = this->altLocale = 
      this->configFlag = this->verFlag   = this->helpFlag   = false ;
   }
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'
   char     appPath[MAX_PATH] ;  // directory of application executable
   char     cfgPath[MAX_PATH] ;  // -f option: alternate configuration file
   char     startDir[MAX_PATH] ; // -d option: startup directory
   char     backPath[MAX_PATH] ; // -b option: backup/synch definition file
   char  localeName[MAX_FNAME] ; // -a option: alternate character encoding locale name
   NcBaseColors colorScheme ;    // -c option: border color scheme
   WinMode  winMode ;            // -w option: startup mode: Single-window or Dual-window
   fmSort   sortOption ;         // -s option: sort option
   short    diagPause ;          // -p option: >0 == pause after start-up diagnostics
                                 // (undocumented) >1 == display verbose start-up diagnostics
   bool     altLocale ;          // -a option: 'true' if alternate locale specified
   bool     modeFlag ;           // -w option: 'true' if window mode specified
   bool     sortFlag ;           // -s option: 'true' if sort option specified
   bool     hideFlag ;           // -i option: 'true' if hidden files are to be displayed
   bool     treeFlag ;           // -t option: 'true' if start in 'Tree' view
   bool     mouseFlag ;          // -m option: 'true' if enable mouse support
   bool     rootFlag ;           // -r option: 'true' if full root-directory scan enabled
   bool     schemeFlag ;         // -c option: 'true' if color scheme specified
   bool     configFlag ;         // -C option: 'true' if run configuration utility
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // -H option: 'true' if display command-line help
} ;

//* Return codes from SingleWin_Mode() and DualWin_Mode() *
enum modeStat : short
{
   mstatCONFIG,      // Exit application and invoke FmConfig
   mstatCONFIGR,     // Invoke FmConfig and (potentially) return to application
   mstatEXIT,        // Exit application (to original working directory)
   mstatEXITC,       // Exit application (to current working directory)
   mstatRESTART,     // Exit application and restart
   mstatSINGLE,      // Switch to Single-Window mode
   mstatDUAL         // Switch to Dual-Window mode
} ;

//* Filesystem information for mount/unmount operations.*
const short FS_BYTES = gsDFLTBYTES + 1 ;
class fsInfo
{
   public:

   ~fsInfo ( void ) { }

   fsInfo ( void )
   {
      this->fsDevice[0]  = '\0' ;
      this->fsMountpt[0] = '\0' ;
      this->fsType[0]    = '\0' ;
      this->fsUri[0]     = '\0' ;
      this->fsMounted = this->fsMtp = this->fsDvd = this->fsDvdMedia = false ;
   }

   char fsDevice[FS_BYTES] ;     // filespec of device driver
   char fsMountpt[FS_BYTES] ;    // filespec of mountpoint
   char fsType[FS_BYTES] ;       // filesystem type
   char fsUri[FS_BYTES] ;        // Universal Resource Identifier
   bool fsMounted ;              // 'true' if filesystem currently mounted
   bool fsMtp ;                  // 'true' if virtual (MTP/GVFS) filesystem
   bool fsDvd ;                  // 'true' if virtual (optical-drive) filesystem
   bool fsDvdMedia ;             // 'true' if optical drive contains disc media
} ;


//***********************************************
//* Constant Data:                              *
//***********************************************

//* Maximum number of entries in FavoriteDirs array *
const short MAX_FAVDIRS = 24 ;

//* Maximum number of entries in MountCmds array *
const short MAX_MNTCMDS = 24 ;

//* Name of FileMangler's configuration sub-program, FmConfig.        *
//* This should be in the same location as the FileMangler executable.*
const char* const ConfigApp = "FmConfig" ;

//* Name of default configuration file *
const char* const ConfigFilename = "FileMangler.cfg" ;

//* Name of file containing FileMangler's Help Text (.info format) *
const char* const HelpFilename = "filemangler.info" ;
//* Name of file containing FileMangler's Help Text (.html format) *
const char* const HTML_Filename = "filemangler.html" ;

//* Names of environment variables passed from parent process. Keep these      *
//* synchronized with the definitions in the invocation script: 'fmg.sh'.      *
const char* const fmgHomeDir = "FMG_HOME" ;
const char* const fmgWDfile  = "FMG_WD_FILE" ;

//* Technical Support form. See DisplaySupportInfo(). *
const char* const tsFName = "FileMangler_SupportRequest.txt" ;



//********************************
//* Application class definition *
//********************************
class FileMangler
{
   public:

   virtual ~FileMangler () ;           // destructor
   FileMangler ( commArgs& clArgs ) ;  // constructor

   //* Special method for monitoring user input to dialog controls.       *
   //* (Kids, don't try this at home, it's terrible programming practice.)*
   short uiMonitor ( const short currIndex, const wkeyCode wkey, bool firstTime ) ;

   private:

   //* Application starting in Single-window mode                              *
   modeStat SingleWin_Mode ( const char* startupPath = NULL, 
                             bool treeView = false, bool rootScan = false ) ;
   //* Application starting in Dual-window mode                                *
   modeStat DualWin_Mode ( const char* startupPath = NULL, 
                           bool treeView = false, bool rootScan = false ) ;
   //* Start the external configuration process.                               *
   short Configure ( short ccCmd, const char* appPath, 
                     const char* cfgFile, bool ncEngine, bool verbose = false ) ;
   //* Create temp-file target directory                                       *
   bool  CreateTemppath ( gString& basePath ) ;
   //* Create a temp-file path/filename                                        *
   bool  CreateTempname ( gString& tmpPath ) ;
   //* Delete temp-file target directory and its contents                      *
   void  DeleteTemppath ( void ) ;
   //* Stub for C++17 function fs::temp_directory_path().                      *
   bool  GetTempdirPath ( gString& tdPath ) ;

   //*************************
   //* Menu operations group *
   //*************************
   MenuCode UserInterface ( MenuCode initialCmd = mcNO_SELECTION ) ;
   MenuCode MenuBarAccess ( short mIndex = -1 ) ;
   MenuCode MenuItem2MenuCode ( short menuIndex, short menuMember ) ;
   short SetActiveMenuItems ( dwControls mIndex ) ;
   bool  InitFmMenuData ( WinMode wMode ) ;
   void  ExecuteMenuCommand ( MenuCode menuCode ) ;

   //*************************
   //*    Utility methods    *
   //*************************
   bool  RefreshAltFileList ( bool forceRefresh = false, bool synchAlt = false ) ;
   bool  ClearAltFileListSelections ( void ) ;
   bool  DualwinMode ( void ) ;
   void  DialogTitle ( bool showMenu, short trgMenu = -1, bool establish = false ) ;
   void  RestoreQuickHelpMsg ( void ) ;
   bool  isRoot ( void ) ;
   void  RootWarning ( void ) ;
   short Systemcall ( const char* cmd ) ;

   //**********************************************
   //* Stub functions for executing menu commands *
   //**********************************************
   void  CmdENTER ( bool altEnter = false ) ;
   void  CmdParentDir ( void ) ;
   void  CmdCopy ( void ) ;
   void  CmdCut ( void ) ;
   void  CmdPaste ( void ) ;
   void  CmdPasteSp ( void ) ;
   void  CmdTrash ( MenuCode mCode ) ;
   void  CmdDelete ( void ) ;
   void  CmdRename ( void ) ;
   void  CmdTouch ( void ) ;
   void  CmdRefresh ( bool altWin = false, bool clipClear = false ) ;
   void  CmdWProtect ( bool protect ) ;
   void  CmdDialogSort ( MenuCode mCode ) ;
   void  CmdHiddenFiles ( void ) ;
   void  CmdRootScan ( void ) ;
   void  CmdContextSort ( void ) ;
   void  CmdSort ( MenuCode mCode ) ;
   void  CmdCreateDir ( void ) ;
   void  CmdSelect ( wkeyCode& wk ) ;
   void  CmdSelectAll ( void ) ;
   void  CmdFavorites ( void ) ;
   void  CmdFind ( void ) ;
   void  CmdInodes ( void ) ;
   void  CmdGrep ( void ) ;
   void  CmdMount ( MenuCode mCode ) ;
   short cmMountFilesystem ( const fsInfo& newCwd, NcDialog* subDlg, bool super ) ;
   short cmDismountFilesystem ( const gString& trgMnt, NcDialog* subDlg, bool super ) ;
   fsInfo* cmScan4Filesystems ( short& fsCount ) ;
   void  CmdView ( MenuCode mCode ) ;
   void  CmdDirTree ( void ) ;
   void  CmdCompareFiles ( void ) ;
   void  CmdShellOut ( bool prompt = false, const char* extCmd = NULL ) ;
   void  CmdAltCWD ( void ) ;
   void  CmdLocateFile ( wkeyCode& wk ) ;
   void  CmdBackup ( bool synch ) ;
   void  CmdArchive ( void ) ;
   void  CmdLockMenuBar ( bool lockVisible ) ;
   void  CmdOpticalMedia ( void ) ;
   void  CmdSynchLock ( void ) ;
   void  CmdBrowseCB ( void ) ;
   void  CmdUserInfo ( void ) ;
   void  CmdFilesysInfo ( void ) ;
   void  CmdMouse ( void ) ;
   void  CmdKeyBindings ( void ) ;
   void  CmdHelpAbout ( void ) ;
   void  CmdCallHelp ( bool html = false ) ;
   modeStat CmdExit ( void ) ;



   //* Temporary application title in console window                           *
   void  DrawTitle ( void ) ;
   //* Find out where the application executable lives                         *
   void  FindAppPath ( commArgs& ca ) ;
   //* Gather and process command-line arguments                               *
   bool  GetCommandLineArgs ( commArgs& ca ) ;
   //void  gcaCaptureName ( char* trg, const char* src, short maxBytes ) ;
   bool  gcaRealpath ( gString& rpath, const char* srcdata = NULL ) ;
   //* Initialize backup/synchronize options.                                  *
   void  Backup ( bool synch ) ;
   short Backup_Validate ( BSet& bkSet, gString& errMsg ) ;
   short Backup_Countdown ( const BSet& bkSet ) ;
   void  Backup_Summary ( const BSet& bkSet, const gString& logName, short opStatus ) ;
   short Backup_FormatLog ( const BSet& bkSet, const gString& trgLog, 
                            const gString& summary ) ;
   void  Backup_SaveLog ( const gString& srcLog, const gString& trgLog, 
                          NcDialog* dp, winPos wp, attr_t hColor, attr_t dColor ) ;
   short Backup_SaveLog ( const gString& srcLog, const gString& trgLog ) ;
   short Backup_Definition ( const char* defPath, BSet& bkSet, bool exclOnly = false ) ;
   short Backup_Exclusions ( const gString& exList, BSet& bkSet ) ;
   void  Backup_Timestamp ( gString& tStamp, const localTime& lt, 
                            bool secs = true, bool dow = false ) ;
   //* Perform a backup operation of specified directory tree.                 *
   short Backup_DirTree ( BSet& bkSet ) ;
   //* Perform a synchronize operation between specified directory trees.      *
   short Synch_DirTree ( BSet& bkSet ) ;

   //* Open an Decision Dialog asking whether we should re-start now.          *
   bool  Restart ( void ) ;
   //* Print command-line version info to console                              *
   void  DisplayVersion ( void ) ;
   //* Print command-line help to console                                      *
   void  DisplayHelp ( void ) ;
   //* Application's Help-About dialog                                         *
   void  HelpAbout ( short ctrY, short ctrX ) ;
   //* Subdialog of Help-About displays application version info               *
   bool  DisplaySupportInfo ( short ulY, short ulX, short dLines, short dCols ) ;
   //* If user==superuser, then display a friendly indicator                   *
   void  Display_SU_Flag ( void ) ;
   //* Display a diagnostic message to the console (start-up sequence only)    *
   void  DiagMsg ( const char* msg, attr_t color, bool newLine = true ) ;
   //* Ask user whether to run the configuration utility.                      *
   bool  Prompt4Config ( void ) ;
   //* Temporary 'not-implemented' message                                     *
   void  FmNotImplemented ( const char* func = NULL ) ;
   //* Debugging message - not for use in release code *
   void  FmDebugMsg ( const char msg[], short pause = 2 ) ;

   #define TEST_CS (0)     // FOR DEVELOPMENT OF COLOR SCHEME ONLY
   #if TEST_CS != 0
   void TestColorSchemes ( void ) ;
   void tcsDialog ( NcDialog* dp, winPos wp, short bRows, short bCols, 
                    const char* title, const ColorScheme& cAttr ) ;
   #endif   // TEST_CS


   //****************
   //* Data Members *
   //****************
   //* Pointer FileMangler application dialog                             *
   NcDialog* dWin ;

   //* Pointer to FileDlg class object (left side or single-window)       *
   FileDlg*  fWin ;

   //* Pointer to FileDlg class object (right side for dual-win)          *
   FileDlg*  rWin ;

   //* Pointer to FileDlg class object that currently has input focus     *
   FileDlg*  fdPtr ;

   //* Pointer to FileDlg class object that does not have input focus.    *
   //* Used for Dual-window mode only; else NULL*.                        *
   FileDlg*  altPtr ;

   //* Dialog control definitions                                         *
   InitCtrl* ic ;

   //* Configuration Options from FileMangler.cfg, command-line and       *
   //* interactive settings. Note that the FileDlg class has its own copy *
   //* of this object which actually controls everything except 'winMode'.*
   //* This start-up copy is used only during application start-up and    *
   //* dialog instantiation and is updated by the FileDlg-class methods   *
   //* when an option is changed by the user.                             *
   ConfigOptions cfgOpt ;

   //* List of user's favorite directory paths (from configuration file)  *
   char  FavoriteDirs[MAX_FAVDIRS][MAX_PATH] ;
   short favdirCount ;

   //* Key map to match user key definitions with default command keys.   *
   RemapKey keyMap ;

   //* List of user's mount command strings (from configuration file)     *
   char  MountCmds[MAX_MNTCMDS][MAX_PATH] ;
   short mntcmdCount ;

   //* Size of terminal window (set when NCurses Engine starts)           *
   //* Modified when terminal window is resized.                          *
   short termRows,
         termCols,
         actualRows,
         actualCols ;

   //* 'true' when Menu Bar locked in visible state                       *
   bool  lockMenuBar ;

   //* For Dual-window Mode, synchronize key input between windows.       *
   bool  synchLock ;

   //* 'true' if application is to exit to current-working-directory.     *
   //* 'false' if exiting to user's original invocation directory.        *
   bool  exit2cwd ;

   //* Y/X position of Menu Bar in dialog                                 *
   //* This value will change if length of title changes                  *
   winPos menuBase ;

   //* Y/X position of dialog title                                       *
   winPos titlePos ;

   //* Menu data colors (single-color menu data) *
   attr_t  monoColor[2] ;

   //* Application title color                                            *
   attr_t  titleColor ;

   //* Cursor position for writing start-up messages to console           *
   winPos suPos ;
} ;
#endif   // FILEMANGLER_INCLUDED

