//********************************************************************************
//* File       : FileMangler.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 26-Jul-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: FileMangler is a simple, small, fast file management utility.   *
//* Although FileMangler has only a small subset of the functionality provided   *
//* by GUI applications such as Nautilus, etc. it is much, much faster.          *
//* Personally, I don't have the five seconds to wait for Nautilus ('Files')     *
//* to start up nor do I have the couple of seconds required to do any           *
//* non-trivial task in such applications. GUIs are wonderful in their way,      *
//* but the time spent on drawing pretty pictures to the screen is often         *
//* better spent doing actual work.  MRS 07-Jan-2006                             *
//*                                                                              *
//*                                                                              *
//* Development Tools:                  Current:                                 *
//* ----------------------------------  ---------------------------------------- *
//*  GNU G++ v:4.8.0 or higher          G++ v:14.3.1                             *
//*  Fedora Linux v:16 or higher        Fedora v:42                              *
//*  GNOME Terminal v:3.2.1 or higher   GNOME Term v:3.54.4 (GNOME 47)           *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.40 02-Apr-2025                                                       *
//*   -- Update references to character/byte counts based upon updates to the    *
//*      gString-class definitions v:0.0.37.                                     *
//*   -- Simplify the rKey class definition by removing a patch for an error     *
//*      in an older version of the C standard library which couldn't handle     *
//*      non-ASCII text in a shared-memory space.                                *
//*   -- Remove some temporary debugging code and screenshot captures.           *
//*      No change in functionality.                                             *
//*   -- Bug Fix: In the NcDialog API, whenever a dialog or sub-dialog was       *
//*      closed, the connection with the clipboard was lost. This was not a      *
//*      problem with FileMangler itself, but a bug in the underlying NcDialog   *
//*      code (v:0.0.37). Beginning with the NcDialog API, v:0.0.38,             *
//*      disconnecting the application from the system clipboard is no longer    *
//*      embedded within the NcDialog destructor, so the clipboard connection,   *
//*      once established, remains active until explicity terminated.            *
//*   -- Update the Tree-view Mode dialog and enhance scan of the directory tree *
//*      for both speed and more robust operation, (FileDlg and FMgr classes).   *
//*   -- Documentation update.                                                   *
//*   -- Posted to website xx-xxx-2025.                                          *
//*                                                                              *
//* v: 0.00.39 01-Aug-2020                                                       *
//*   -- Bug Fix: In FMgr class, for methods which allow the user to directly    *
//*      set the target file's timestamp, the C library function occasionally    *
//*      assumes the wrong timezone which caused the timestamp to be off in      *
//*      the next/previous timezone.                                             *
//*   -- Bug Fix: In Dual-window Mode, when a filesystem is unmounted, be sure   *
//*      that the window synch-lock is disengaged.                               *
//*   -- Bug Fix: When exiting TreeView mode to the root directory, disable      *
//*      the full-scan flag ('autoRecurse'). This causes the scan to read only   *
//*      the top level of the root directory.                                    *
//*   -- In View File Contents dialog, add support for viewing ".eml"            *
//*      (exported email) files.                                                 *
//*   -- In View File Contents dialog, add support for viewing metadata for      *
//*      for WMA audio files.                                                    *
//*   -- In View File Contents dialog, when user attempts to view the contents   *
//*      of a binary file as if it were plain text, automatically set view to    *
//*      'ASCII-hex-byte' format.                                                *
//*   -- Bug Fix: Dual-window Mode only: When a "resize dialog" is invoked,      *
//*      both windows were being set to the current CWD instead of being set     *
//*      to their previous target directories. Also, focus was always set to     *
//*      left window after the resize. A "resize dialog" event now retains the   *
//*      existing CWD and focus.                                                 *
//*   -- Bug Fix: Dual-window synchlock:                                         *
//*      -- If selecting a new target directory from "Favorite Directories"      *
//*         dialog, disable the synch-lock.                                      *
//*      -- If mounting or unmounting a filesystem from "Mount Filesystems"      *
//*         dialog, disable the synch-lock.                                      *
//*      -- If synch-lock is engaged during a locate-file-by-name scan           *
//*         (CmdLocateFile() method), the scan is now echoed in the inactive     *
//*         directory window. Previously, the inactive window was not updated    *
//*         during this type of scan.                                            *
//*   -- Bug Fix: When the Mount Filesystem dialog is first opened, the          *
//*      mount/unmount toggle was sometimes initialized incorrectly. This in     *
//*      turn caused the initial text of the mount/unmount pushbutton control    *
//*      to be incorrect.                                                        *
//*   -- Bug Fix: In keycode help dialog, when a keycode was entered into the    *
//*      search textbox, the textbox was not reinitialized prior to the next     *
//*      search.                                                                 *
//*   -- Bug Fix: When mounting or unmounting a filesystem, the displayed        *
//*      file list(s) were not always updated as they should be.                 *
//*   -- Enhanced support for mount/unmount/eject of optical-drive media.        *
//*      See CmdMount() and CmdOpticalMedia() methods.                           *
//*   -- Bug Fix: When synch is engaged and user is scrolling through the file   *
//*      list, then the highlight in the inactive window was not always          *
//*      tracking matching filenames correctly.                                  *
//*   -- Add an option to the backup operation to simply _report_ the files      *
//*      that need to be updated, but do not modify any target files.            *
//*      See Backup() method dialog and the FileDlg class BackupClipboardList()  *
//*      method which directs the flow to the appropriate processing loop.       *
//*      Note that while the log file for Backup/Synch operations reports the    *
//*      _source_ files backed up, the Scan-Only operation reports the targets   *
//*      that would have been updated. Note also that because the Backup and     *
//*      Scan-Only operations do not share code, no changes to the Backup/Synch  *
//*      operation were made.                                                    *
//*   -- Update the Help About dialog to include the version number of the       *
//*      GVfs utility 'gio' (if installed).                                      *
//*      -- Also, add a message about the required ncursesw library version.     *
//*   -- When allocating shared memory during configuration, implement a         *
//*      retry loop in case the initial allocation fails.                        *
//*   -- For the mount/unmount-device dialog, the mounted devices are now        *
//*      placed at the top of the list.                                          *
//*   -- Add the option for display of line numbers in the View-File dialog.     *
//*      Valid for viewing plain text files, HTML, XML and PERL.                 *
//*   -- Bug Fix: If View-File target was one of the 'special' file types, the   *
//*      error message was incorrectly formatted, causing a seg fault.           *
//*   -- In the View-File dialog, allow Texinfo '.info' files to be viewed as    *
//*      plain text. Although 'info' files are _technically_ binary files, they  *
//*      are _almost_ text files, so the occasional control character does not   *
//*      detract very much from the display.                                     *
//*   -- Update copyright notices to reflect the new year.                       *
//*   -- Documentation update.                                                   *
//*      -- The folks who maintan the Texinfo project have made some design      *
//*         decisions which have impacted our documentation; specifically:       *
//*         1) The 'texi2any' processor now outputs an error message if a        *
//*            @xref, @pxref, @ref reference is inside a @w{} sequence.          *
//*            This does not affect the output; however, it does spew hundreds   *
//*            of warning messages across our terminal window. This has cost us  *
//*            several days to update our code.(Those guys are a laugh-riot.)    *
//*         2) The syntax following the index table has changed which causes our *
//*            'idpp' post-processing utility sees a syntax error. This does not *
//*            affect the output, but it annoys our sense of order in the        *
//*            universe.                                                         *
//*   -- No public release.                                                      *
//*                                                                              *
//* v: 0.00.38 02-Jul-2020                                                       *
//*   -- Bug Fix: Dual-window synchlock was not being released when switching    *
//*      between Dual-window and Single-window mode.                             *
//*   -- In the "Edit Mount Commands" section of the configuration utility,      *
//*      add a pushbutton option to display a list of storage devices attached   *
//*      to the system. This enables user to select a mountpoint or URI from     *
//*      the list for inclusion in the mountpoint table.                         *
//*      -- To make room for this enhancement, the partially-implemented         *
//*         "Expert Mode" (fstab editor) has been discarded. We have never       *
//*         been comfortable allowing the user to edit /etc/fstab from within    *
//*         the application, so removing that functionality seemed the wise      *
//*         choice.                                                              *
//*   -- Implement improved support for access to files on MTP/GVfs virtual      *
//*      filesystems:                                                            *
//*      -- Bug Fix: If the 'gio' utility encountered an unknown filetype, it    *
//*         would not report that field at all. To compensate, we now test for   *
//*         a missing filetype field in the stat data.                           *
//*      -- Gather more detailed stat data for both individual files and for     *
//*         the target filesystems.                                              *
//*         -- There is an interface bug between the Linux system and the        *
//*            target device: The device reports the filesystem-type code        *
//*            string as a decimal value rather than a hexidecimal value as it   *
//*            should. We have compensated, but this is an active issue.         *
//*      -- Copy/cut/paste and rename functionality is now fully functional.     *
//*         Note that the underlying tools for MTP/GVfs do not support           *
//*         recursive operations; therefore, operations on those filesystems     *
//*         are limited to the files visible in the display window, NOT THE      *
//*         CONTENTS of subdirectories.                                          *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 31-Jul-2020.                                          *
//*                                                                              *
//* v: 0.00.37 03-Nov-2018                                                       *
//*   -- Significant performance enhancement in the way the directory tree is    *
//*      scanned and formatted for display. This is accomplished by increasing   *
//*      the number of execution threads used to scan the branches of the        *
//*      directory tree. This is a major change in the directory-tree scan       *
//*      algorithm, but little or no change in the user interface.               *
//*      Please see CaptureDirTree() method group in FMgr class; specifically,   *
//*      sdtMgrthread().                                                         *
//*      -- Included as part of this enhancement we have implemented a           *
//*         single-threaded, non-recursive read of the smartphone (MTP/GVfs)     *
//*         filesystem because "smartphones" aren't. This significantly reduces  *
//*         the buffering requirements within the phone itself which reduces     *
//*         the phone's response time for directory requests although            *
//*         significant delays may still occur if the device is performing       *
//*         other tasks. Individual files and small groups of files many be      *
//*         safely and efficiently copied to and from the device; however,       *
//*         because this algorithm is non-recursive, copy/paste of directory     *
//*         trees is not supported at this time.                                 *
//*   -- Add audible alert if user tries to navigate to the parent of the        *
//*      root directory (which doesn't exist). The action was already            *
//*      disallowed, but now we audibly punish the attempt.                      *
//*   -- Add a command-line switch to start app with hidden files visible.       *
//*      "-i" == invisible files                                                 *
//*   -- Enhance the way the file backup algorithm handles symbolic link files.  *
//*      See BackupClipboardList() method.                                       *
//*   -- For setup of an "Archive" operation enhance and simplify testing for    *
//*      automatic determination of the Create/Update/Expand sub-option.         *
//*      See Archive() method.                                                   *
//*   -- Add support for MS-Office (OOXML) documents (".docx", ".xlsx", and      *
//*      ".pptx") which are structured as .ZIP archive files. Note that          *
//*      additional OOXML documents i.e. the embedded-macro versions of these    *
//*      file types: ".docm", ".xlxm", ".pptm", as well as ".ppsx" and ".ppsm"   *
//*      _may_ work also, but we have no example files to verify functionality   *
//*      for these file types.                                                   *
//*      -- view document text                                                   *
//*      -- view ZIP archive                                                     *
//*      -- expand ZIP archive                                                   *
//*      -- 'grep' the document's text data                                      *
//*   -- Completed implementation of user interruption of data backup            *
//*      operation. User can now pause, and optionally abort a backup operation  *
//*      in progress.                                                            *
//*   -- Implement ANSI color formatting for HTML source code. Tested with       *
//*      pure HTML-5 source only. NOTE: Older, ad-hoc HTML command syntax may    *
//*      not be colorized accurately.                                            *
//*   -- Simplify Tree-View mode. Use the directory-tree data from the normal    *
//*      file-view mode to create the Tree-View display data. Previously, a      *
//*      specialized tree-view scan was performed. Smaller, faster, and more     *
//*      robust.                                                                 *
//*      -- Bug Fix: If application started in Tree-View mode, the tree-view     *
//*         flag was not being reset when Tree-View closed.                      *
//*   -- Full root directory scan is now more stable and considerably faster;    *
//*      however, by default a scan of the root directory, includes ONLY the     *
//*      top layer of the directory tree.                                        *
//*      -- Add a command-line option, "-r" == root-scan, to enable a complete   *
//*         scan of the directory tree from the root downward.                   *
//*      -- Add a "Root Scan" menu item to the View Menu. Selecting this menu    *
//*         item toggles the full root-scan functionality.                       *
//*      -- To fascilitate the enhanced tree scan, we now test for externally-   *
//*         mounted filesystems encountered during the scan and do not recurse   *
//*         into external filesystems (SD cards, USB memory devices, external    *
//*         hard drives, etc.). Instead, these mountpoint directories are        *
//*         identified by appending the "(extfs)" tag after the displayed        *
//*         directory name. Tagging of external filesystems was previously       *
//*         documented but was not fully implemented.                            *
//*      -- When mounting an external storage device or when user selects the    *
//*         mountpoint of an external device from the "Favorites" list, test     *
//*         the total size of the filesystem and if "very large" scan only the   *
//*         top-level data instead of performing a full recursive scan of the    *
//*         directory tree.                                                      *
//*   -- Implement filesystem scan to report all files that share a specified    *
//*      Inode number. See FindInodes() method group in FileDlg class.           *
//*      -- Add "Inode Scan" menu item to Util Menu.                             *
//*      -- Update FmKeymap.cfg with hotkey, increment file version to 0.01.01.  *
//*   -- Update the single-file and batch-mode rename methods to allow           *
//*      filenames with Linux "special characters": ( \ ' " ? : ; & > < | * ).   *
//*      -- Test all calls to the shell to verify that these characters are      *
//*         seen as text rather than as commands to the shell.                   *
//*      -- The C-language versions of shell commands such as 'mkdir', 'rmdir',  *
//*         'chmod', 'unlink' 'rename' appear to be smart enough to do           *
//*         escaping on their own; however, they are not smart enough to leave   *
//*         pre-escaped characters alone. This is (arguably) a bug in the        *
//*         C-language library. Keep an eye on it to see if they fix it.         *
//*   -- Enhance functionality of SynchLock mode to keep highlight in active     *
//*      and inactive windows in synch. See uiMonitor() method for details.      *
//*      -- Automatic disable of SynchLock mode when a file-operation keycode    *
//*         is detected has been disabled _except_ in the case where user        *
//*         enters a child directory for which there is no corresponding         *
//*         subdirectory in the inactive window. Note: In the future, we may     *
//*         reactivate the auto-disable; however,the recent functionality        *
//*         enhancements have made it technically unnecessary.                   *
//*   -- Implement multi-file comparison. See CmdCompareFiles() and the          *
//*      FileDlg-class CompareFiles() methods. Previously, comparison could      *
//*      only be performed between two files. This enhancement allows each       *
//*      file in one group to be compared with the corresponding file in         *
//*      another group. Available in Dual-window Mode only.                      *
//*   -- Update a few places in the code to address some new warnings that       *
//*      arose in the transition from gcc v:4.8.3 to gcc v:9.2.1.                *
//*      --  -Wmaybe-unitialized                                                 *
//*           In ecoEditCS_ControlUpdate(). It was actually was an error, but    *
//*           it worked anyway because we weren't using the unitialized value.   *
//*      --  -Wdeprecated-declarations                                           *
//*           readdir64_r() is deprecated. Switch to readdir64() which is        *
//*           thread-safe under the GNU compiler.                                *
//*      No functionality change.                                                *
//*   -- In FindFiles dialog, for files that match the search criteria, along    *
//*      with the pathspec optionally display the file's timestamp. We have      *
//*      found this useful when trying to locate the newest version of a file.   *
//*   -- For all calls invoking the 'less' utility, add the '-c' (clearscreen)   *
//*      option to force 'less' to clear the terminal window on startup.         *
//*      This is to compensate for the poor formatting used by 'less' when       *
//*      displaying data that do not fill the page.                              *
//*   -- Enhance the Mount/Unmount functionality to include mountpoints defined  *
//*      in the '/proc/self/mountinfo' and '/etc/fstab' system files.            *
//*      In addition, scan for attached USB devices which use the MTP protocol   *
//*      (GVfs filesystems) which are not listed in the kernel files.            *
//*      Add two Radiobutton controls to the Mount Filesystems dialog.           *
//*      -- Increase the number of possible 'Mountpoint' entries from 16 to 24   *
//*         to accomodate mounting of additional storage devices.                *
//*   -- Add Systemcall() method to FileMangler class to consolidate top-level   *
//*      system access.                                                          *
//*   -- Enable GNOME/Wayland clipboard access for Textbox controls, IF the      *
//*      external "wl-clipboard" utilities are installed. (Otherwise, NcDialog   *
//*      local clipboard access only.) Connection established through the        *
//*      NcDialog WaylandCB-class interface. NcDialogAPI v:0.0.32 or greater.    *
//*   -- When viewing file contents, if target file has ".epub" extension,       *
//*      list the contents of the document. Note: This is a .ZIP file            *
//*      containing a series of web pages. Support for ePub document archive     *
//*      expansion is also supported.                                            *
//*   -- Implement support for reading MPEG-4 (m4a) audio file metadata.         *
//*      See FileDlgMedia.cpp for details.                                       *
//*   -- Implement ANSI color coding for viewing Perl source code files.         *
//*   -- Update the 'fminstall.pl' installation script.                          *
//*   -- Update dates of copyright notices.                                      *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 01 Jul 2020.                                          *
//*                                                                              *
//* v: 0.00.36 12-Oct-2018                                                       *
//*   -- Update formatting of ViewFile text extracted from OpenDocument files.   *
//*      As a side-effect, the grep of OpenDocuments is now also more robust.    *
//*      See odExtractOD_Text().                                                 *
//*   -- Enhance ViewFile output for XML files by applying ANSI color commands   *
//*      to highlight the elements of the source code.                           *
//*   -- Implement support for viewing .ZIP archives. This already worked by     *
//*      magic (external 'less' preprocessor), but now it works intentionally.   *
//*      See ArchiveTarget(), vfcDecodeArchive().                                *
//*   -- Bug Fix: In Backup_Validate(), auto-rename for target archive file      *
//*      was not fully functional (stupid programmers :-)                        *
//*   -- Implement create/update/expand for ZIP archives.                        *
//*      -- Add option for data backup to ZIP archive target.                    *
//*      -- Create/Update/Expand for both Tar and Zip are now fully functional.  *
//*      -- Create/Update/Expand dialog redesigned to accomodate expanded        *
//*         archiving functionality.                                             *
//*      -- Expansion of OpenDocument files as ordinary 'zip' files is under     *
//*         construction. (Not fully implemented.)                               *
//*         At this time, we see no need to support creation or update of        *
//*         OpenDocument files.                                                  *
//*   -- "Set alt directory window" default key binding reassigned to            *
//*      (ALT+SHIFT+W ) from (ALT+') This is more compatible with the ALT+W      *
//*      (WinMode) toggle.                                                       *
//*      between Single-win Mode and Dual-win Mode. See CmdAltCWD().             *
//*   -- "SynchLock toggle" default key binding reassigned to                    *
//*      nckA_SQUO (ALT+') from (ALT+SHIFT+') More intuitive now that ALT+'      *
//*      has been freed up.                                                      *
//*   -- In the main user-interface loop (UserInterface()), when user presses    *
//*      an invalid key, he/she will get an audible boink to indicate the        *
//*      invalid keypress. This previously worked only for control-key           *
//*      combinations.                                                           *
//*   -- Viewing an OpenDocument file through the ViewFile context menu:         *
//*      In the previous release, user could view the text extracted from        *
//*      supported OpenDocument files. However, because OD files are actually    *
//*      'zip' archives, we now _also_ enable the option to view an OD file's    *
//*      contents as an archive. See ViewFileContents().                         *
//*      -- As a convenience in development, OpenDocument files are now          *
//*         recognized as zip archives by the Expand Archive dialog.             *
//*         OD files _are not_ recognized as archive targets for create/update.  *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 02 Nov 2018.                                          *
//*                                                                              *
//* v: 0.00.35 26-Jan-2017                                                       *
//*   - Bug Fix: In EncodeEpochTime(), the call to the C library function to     *
//*     convert from human-readable time to epoch time is, for some timezones,   *
//*     off by exactly one hour (related to DST). We have compensated, but the   *
//*     fix is not bulletproof. See notes in the the EncodeEpochTime() header.   *
//*   - Increase the number of possible 'Favorites' entries from 16 to 24 to     *
//*     accomodate additional phones, memory sticks, external drives, etc.       *
//*   - Add hotkeys to the ViewFile dialog controls. This was implemented        *
//*     because we often view audio files as binary (hex) data, and hotkeys      *
//*     speed up the process.                                                    *
//*   - Update code for extracting metadata from audio media files. This code    *
//*     is adapted from the 'Taggit' application. Basic functionality is         *
//*     unchanged, but with more robust parsing and error checking.              *
//*     Reporting of embedded images, Popularimeter and PlayCounter are now      *
//*     also reported.                                                           *
//*   - Bug Fix: In ViewFileContents() escape single-quote characters in         *
//*     filespec. Double-quote characters, backtics and backslashes in the       *
//*     filespec will still cause the shell to parse the string incorrectly.     *
//*     On the other hand, users who name their files with these characters      *
//*     deserve the error message they will get.                                 *
//*   - Bug Fix: In FindFiles() the count of top-level directories was           *
//*     incorrect if the top level contained "hidden" directories. This caused   *
//*     the application to intermittently fail to find the searched-for item.    *
//*   - Create new source module, FileDlgContext(), for code related to the      *
//*     ViewFile context menu. This helps to reduce the size of the              *
//*     FileDlgPrompt.cpp module which was getting much too large (700KB).       *
//*   - Implement the "Execute External Program" functionality (ViewFile         *
//*     context menu) which allows user to execute binaries and scripts as well  *
//*     as invoking the system-default application for selected media files,     *
//*     documents and other common data files. This had formerly been only a     *
//*     stub program. The launch functionality was taken almost unmodified from  *
//*     our Taggit application.                                                  *
//*   - Remove the 'Transaction Log' option from the configuration file.         *
//*     It seemed like a good idea ten years ago, but we have never used it.     *
//*     The functionality of the transaction log has been largely replaced by    *
//*     the Backup/Synch/Archive log file.                                       *
//*   - Place hooks for constructing a multi-language user interface.            *
//*     See ConfigOptions::appLang.                                              *
//*   - Expand minimum application width from 78 columns to 80 columns.          *
//*     Limiting the application to 78 columns seemed like a good idea when      *
//*     we were developing on an actual Cromemco 80-column terminal attached     *
//*     to our homebrew system, but it seems a little out-of-step in the world   *
//*     of emulators.                                                            *
//*   - Add the 'Resize Dialog' item to the View Menu. This allows the user to   *
//*     re-size the application dialog when the size of the terminal window      *
//*     has changed. Command key: ALT+CTRL+Resize.                               *
//*     Formerly, it was necessary to toggle the Single/Dual-window mode to      *
//*     resize the dialog.                                                       *
//*   - When reporting filesystem stats, explicity convert an 'fsType' string    *
//*     of "fuseblk" to "ntfs". While there are multiple filesystem formats      *
//*     associated with the FUSE protocol, by far the most common is the         *
//*     Windoze NTFS format. This change makes the filesystem report less        *
//*     techie, and more user friendly.                                          *
//*   - Add the 'wr' (warning-message) member to the colorScheme class.          *
//*   - Key-command remapping:                                                   *
//*     - Keymap configuration file format definition.                           *
//*     - Keymap display via menu option: Help Menu "Key Bindings" item.         *
//*     - Interactive edit of keymap in the configuration utility.               *
//*   - Bug Fix: For backup log, the updates-only log format was incorrectly     *
//*     counting unmodified directory names as updated files, and was not        *
//*     reporting created target directories as updates.                         *
//*   - Bug Fix: A click outside the application borders was inappropriately     *
//*     activating the Menu Bar rather that being ignored as it should be.       *
//*   - Updates required by switch to G++ v:5.4.0:                               *
//*     - New warning that library functions 'tempnam' and 'tmpnam_r are         *
//*       dangerous. While this is technically true, this application uses       *
//*       them in such a way that negates the danger. In any case, silence the   *
//*       compiler warning by updating methods which create temporary files.     *
//*   - Bug Fix: In FindFiles() method, processing mouse input had a logical     *
//*     error. See note in that method.                                          *
//*   - Widen the 'Util' menu to allow more easily understood menu items.        *
//*   - FirstSelectedItem(), NextSelectedItem() and HilightItemByName() now      *
//*     return either the index of the highlighted filename or 'ERR'.            *
//*     Previously, these were 'void' (no return value).                         *
//*   - In CmdLocateFile(), handle SPACE character as a special case because     *
//*     it could be either a filename character or the 'select' command.         *
//*   - First pass at implementing 'grep' functionality is complete. Simple      *
//*     (only two user-accessible parameters), but functional, and includes      *
//*     direct scan of text in OpenDocument files.                               *
//*     In FileDlgUtil1.cpp, see GrepFiles() and the gfxxx() method group.       *
//*   - Add support for viewing the (unformatted) _text_ of an OpenDocument      *
//*     file from the ViewFile context menu.                                     *
//*   - In FMgr class, update FileSystemStats() method to include UUID, label,   *
//*     device name and mountpoint. This provides additional information to      *
//*     the user through the "File System Information" dialog.                   *
//*   - Split FmInterface.cpp creating FmCommand.cpp. The source module had      *
//*     grown to over 200,000 bytes, making development inefficient.             *
//*   - Complete the first iteration of the Mount Filesystem functionality.      *
//*     Because mounting a storage device often requires super-user privilege,   *
//*     a child terminal window is opened from which the operation is launched.  *
//*   - Add "Help Docs (HTML)" item to Help Menu. Opens HTML documentation       *
//*     in the default browser.                                                  *
//*   - Rewrite Perl installation script to simplify user's installation         *
//*     process. See notes in fmginstall.pl.                                     *
//*   - Modify configuration dialog for 'alt trashcan' so it doesn't complain    *
//*     when the specified path is an empty string.                              *
//*   - Update all copyright notices to '2018'.                                  *
//*   - Documentation update.                                                    *
//*   - Posted to website 11 Oct 2018.                                           *
//*                                                                              *
//* v: 0.00.34 19-Jan-2016                                                       *
//*   - Basic implementation of automatic Backup and Synchronize functionality.  *
//*     - Add '-b=FILENAME' command-line option to allow user to specify a       *
//*       Backup/Synch definition file.                                          *
//*     - Implement Backup_xxx() method group to interact with user in setting   *
//*       up the operation.                                                      *
//*     - See also the FileDlg class layer of Backup/Synch functionality.        *
//*   - Implement Archive functionality to create, update and expand tar         *
//*     archive files. Most archive-related functionality is in the FileDlg      *
//*     class.                                                                   *
//*   - Restructure and enhance trashcan functionality. Most trashcan-related    *
//*     functionality is in the FileDlg class.                                   *
//*   - Modify GetCommandLineArgs() method to allow arguments to be separate     *
//*     tokens. We don't necessarily think this is a good idea, but most         *
//*     command-line utilities do this, so we caved to peer pressure.            *
//*   - Implement CmdMouse() to enable/disable mouse for current session only.   *
//*   - Implement '-m' command-line option for enabling mouse support for        *
//*     current session only.                                                    *
//*   - Finish implementation of file locator. Locate a filename displayed in    *
//*     the current window by typing in part or all of its name.                 *
//*   - Implement 'Find Files' functionality to scan the directory tree for      *
//*     filenames which match the user-specified pattern.                        *
//*   - Bug Fix: in 'GetCommandLineArgs' method: 'grey' color scheme was being   *
//*     misinterpreted as 'green'.                                               *
//*   - Implement 'compare-files'. This is an enhanced and user-friendly         *
//*     implementation of the 'diff' utility's basic functionality.              *
//*   - When moving to parent directory, the highlight is now positioned on      *
//*     the name of the previous subdirectory.                                   *
//*   - The path textbox now shows the progress of reading a new CWD.            *
//*     This is functional but not elegant. Needs improvement. See Progress      *
//*     Bar implementation.                                                      *
//*   - Implement the SynchLock for Dual-window Mode. See CmdSynchLock().        *
//*   - Assign command key to:                                                   *
//*     - Display user info                                                      *
//*     - Display filesystem info                                                *
//*     - Show/hide hidden files                                                 *
//*     While this is neither necessary nor particularly useful, it does         *
//*     provide symmetry.                                                        *
//*   - Simpify the HelpAbout subdialog and provide user with an option to       *
//*     save the displayed data to a file for submitting a tech support          *
//*     request.                                                                 *
//*   - Implement reporting of media file metadata for MP3 and OGG audio files.  *
//*     This is under the 'View File Contents' context menu.                     *
//*   - Define the mechanism for setup and operation of a Progress Bar. Most     *
//*     of this functionality is in the FileDlg class.                           *
//*   - Create the basic structure for command-key remapping functionality.      *
//*   - Update all copyright notices to '2017'.                                  *
//*   - Documentation update.                                                    *
//*                                                                              *
//* v: 0.00.33 08-Jan-2014                                                       *
//*   - Trashcan access is now fully functional.                                 *
//*   - Update control definitions for dctTEXTBOX and dctPUSHBUTTON controls     *
//*     because NcDialog API now allow multiple lines for these controls.        *
//*     This primarily affects the ModifyFileStats() method where we previously  *
//*     made some assumptions about these control types.                         *
//*   - Move primary development platform from 32-bit to 64-bit system, and      *
//*     update primary compiler from G++ v:4.8.0 to G++ v:4.8.2.                 *
//*     No transition issues.                                                    *
//*     - Also updated from Fedora 16 to Fedora 20. F20 has some bugs, but       *
//*       no serious effect on FileMangler functionality.                        *
//*   - Swap elements 0 and 1 in 'View File' context menu. 'View Contents' is    *
//*     now the first item in the menu because it is by far the most             *
//*     commonly-used item.                                                      *
//*   - Bug Fix: ViewFile context menu display was corrupted by incorrect        *
//*     display-item width.                                                      *
//*   - Implement 'Find Link Target' in ViewFile context menu.                   *
//*     See FindLinkTarget() method.                                             *
//*   - Update call to nc.GetTermnalColorSupport() to match new NcDialog         *
//*     definition.                                                              *
//*   - Update get/set selection to match new NcDialog naming convention.        *
//*   - Add nckAC_J (ALT+ENTER) key to interface loop to access the ViewFile     *
//*     context menu for the highlighted file, even if it is a directory name.   *
//*   - Update all instances of the gString 'formatInt' call to match changes    *
//*     in the gString class.                                                    *
//*   - Redefine location of temp file which indicates the optional exit         *
//*     directory. Was in the application directory; now in the system's         *
//*     temporary storage directory. This was done in anticipation of moving     *
//*     all temporary files to the system's temp directory.                      *
//*   - Redefine location for creating application's temporary files.            *
//*     Formerly they lived in the installation directory for ease of            *
//*     debugging. However, the algorithm is now stable, so temp files are       *
//*     created in the system's defined temp-file directory.                     *
//*     - This includes removing the configuration option for specifying         *
//*       a temp-file directory.                                                 *
//*     - New method CreateTemppath() now does the setup for temporary files,    *
//*       and the application will not open unless this setup is successful.     *
//*     - New method DeleteTemppath() is called by the destructor to clean up    *
//*       any temp files which were not removed by the code that created them.   *
//*   - Tweak the color schemes to take advantage of 16-color terminals.         *
//*   - Minor updates to TreeView Mode to prevent recursion into external        *
//*     filesystems. This had caused a noticeable performance hit.               *
//*   - Basic implementation of the 'stable mouse' interface. The mouse          *
//*     interface is clunky, as predicted, but better than expected.             *
//*     The MenuBar and sub-dialogs work smoothly, but selection of items for    *
//*     processing is still easier via keyboard. Most of the changes required    *
//*     for mouse support were minor; however, there were over 100 of them,      *
//*     so be alert for new user-interface bugs.                                 *
//*     - Add mouse option to configuration file.                                *
//*     - Add mouse enable/disable option to Utility menu.                       *
//*     - Add interactive configuration method to set/reset the mouse flag.      *
//*     - Update primary user-interface loop to be aware of converted mouse      *
//*       events for item selection, and to interpret non-converted mouse        *
//*       events related to enabling the menu system, shifting the input focus   *
//*       and moving to parent directory.                                        *
//*     - Update all subdialogs to recognize hotkey user-input.                  *
//*     - In TreeView Mode, convert mouse events to keycodes.                    *
//*     - Update documentation.                                                  *
//*   - Consolidate all output of copyright-year range, so it only needs to      *
//*     be updated in one place. (see 'copyrightYears' constant)                 *
//*                                                                              *
//* v: 0.00.32 24-Dec-2013                                                       *
//*   - Sending files to Trashcan is now basically functional.                   *
//*   - Empty/Restore Trashcan under construction.                               *
//*   - Update instances of dtbmData class usage to reflect changes in           *
//*     NcDialog API library.                                                    *
//*                                                                              *
//* v: 0.00.31 25-Aug-2013                                                       *
//*   - Update compiler to GNU G++ 4.8.0. Minor changes needed to eliminate      *
//*     warning messages due to updated C++11 specifications.                    *
//*   - Continue implementation of 'Trashcan' functionality which uses the       *
//*     Gnome Trashcan directory and syntax. (Handling of error conditions not   *
//*     fully vetted.)                                                           *
//*   - Enhance application exit code to allow for optional exit to current      *
//*     working directory rather than original invocation directory.             *
//*     See also enhancement to 'fmg.sh' invocation script.                      *
//*   - Create placeholder 'filemangler.info' Texinfo file to be invoked when    *
//*     user requests Help. Most of the help text is not yet written.            *
//*   - Enhance MenuBarAccess() and DialogTitle() to allow user to specify       *
//*     which menu will be opened when the Menu Bar becomes visible.             *
//*     - Also implemented the ability to lock the Menu Bar in the visible       *
//*       state.                                                                 *
//*   - Implement Shift+UpArrow and Shift+DownArrow for selecting a group of     *
//*     adjacent files. See CmdSelect() method.                                  *
//*   - Basic implementation of the '-T' command-line option to start the        *
//*     application in Tree View.                                                *
//*     Note that if user starts in TreeView Mode, then CWD contents are not     *
//*     read until return from TreeView.                                         *
//*   - Add a 'refresh display' command and corresponing menu item.              *
//*     See CmdRefresh().                                                        *
//*   - Add CmdAltCWD() method to set alternate window to same CWD as active     *
//*     window. Applies to DualWin Mode only.                                    *
//*   - Add a visual indication if user == 'superuser'. See Display_SU_Flag().   *
//*                                                                              *
//* v: 0.00.30 23-Mar-2013                                                       *
//*   - Begin design of the 'Backup' and 'Synchronize' functionality.            *
//*     Added the FmBackup.cpp module for this purpose.                          *
//*   - Additional clean-up of color scheme enhancement.                         *
//*                                                                              *
//* v: 0.00.29 19-Jul-2012                                                       *
//*   - Convert main application code to the FileMangler class.                  *
//*     It's just nicer that way...                                              *
//*   - Implement key mapping for the main loop rather than static key           *
//*     assignments (conversion incomplete).                                     *
//*   - Create FmConfig.hpp and the FmConfig class. Define a shared-memory       *
//*     area for inter-process communications and implement all configuration    *
//*     operations as a seperate sub-program. See Configure().                   *
//*   - Implement undocumented switch to display verbose start-up diagnostics.   *
//*     See GetCommandLineArgs().                                                *
//*                                                                              *
//* v: 0.00.28 14-May-2012                                                       *
//*   - Implement application's user-selectable color scheme based on            *
//*     enhancements in NcDialog/NcWindow/NCurses classes.                       *
//*     (see FileDlg.hpp, class ColorScheme)                                     *
//*                                                                              *
//* v: 0.00.27 03-Oct-2011                                                       *
//*   - Continue recursive re-design.                                            *
//*   - Consolidate single-win and dual-win modules into FmInterface.cpp.        *
//*   - Redesign gathering of command-line args.                                 *
//*   - Now links with the UTF-8-aware NcDialog.a link library and with          *
//*     ncursesw, the 'wide' version of the ncurses library.                     *
//*                                                                              *
//* v: 0.00.26 25-Jul-2010                                                       *
//*   - Massive design revision in the way file data are captured and stored.    *
//*     This is necessary to support recursive copy/cut/paste/sync/backup, etc.  *
//*   - Add config option for path specification where temporary files are to    *
//*     be written.                                                              *
//*   - Add config options for Trashcan management.                              *
//*                                                                              *
//* v: 0.00.25 14-Feb-2010                                                       *
//*   - Change development platform and compiler version.                        *
//*                  Fedora 12 and GNU G++ (Gcc v: 4.4.2)                        *
//*     This neccessitated several syntax changes in class and method            *
//*     definitions to accomodate changes in the C++ definition and the GCC      *
//*     compiler's warning and error messages.                                   *
//*     Code basically re-integrated with the test programs, Cb and Dialog1.     *
//*     Header files now interchangeable, and FileMangler having the active      *
//*     copy of each .cpp source file.                                           *
//*   - Update all methods that use the NcDialog class. This class has           *
//*     undergone significant enhancement since the previous version.            *
//*   - Begin conversion of FileWin class FMgr class and FileMangler from        *
//*     NcWindow primitives to NcDialog class objects.                           *
//* v: 0.00.24 29-Mar-2007                                                       *
//*   - Add configuration option to specify the operation when copying symbolic  *
//*     links. Add new sub-menu, SymLinkCopyMenu() for user to select option.    *
//*   - Add ViewClassVersions() method.                                          *
//* v: 0.00.23 05-Feb-2007                                                       *
//*   - Implementation of PasteSpecial in FileWin object.                        *
//*   - Major overhaul of PasteClipboardList() in FileWin class.                 *
//* v: 0.00.22 26-Dec-2006 Implement Touch Files functionality.                  *
//*                        See FileWin class for details.                        *
//* v: 0.00.21 22-Dec-2006 Integrate new version of NcDialog class family.       *
//* v: 0.00.20 24-Sep-2006 Integrate completed version of NcDialog class.        *
//* v: 0.00.10 07-Jan-2006 Base code is from the ncurses test application, Cd.   *
//*                        Created using GNU G++ (Gcc v: 3.2.2)                  *
//*                        under RedHat Linux, kernel 2.4.20-31.9                *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//* -- Note on dynamic load library dependencies:                                *
//*    FileMangler requires certain system 'dynamic link library' files to be    *
//*    installed. If one or more needed libraries cannot be located, then the    *
//*    application will not start, and some messages about missing routines      *
//*    will be displayed.                                                        *
//*                                                                              *
//*    For a list of the necessary dynamic link libraries, run the 'ldd'         *
//*    command: ldd FileMangler                                                  *
//*                                                                              *
//* -- Because we sincerely dislike conditional-compile switches which place     *
//*    an unnecessary burden on all users, especially newbies, we want to        *
//*    minimize the number of conditional-compile switches.                      *
//*    The switches in this application are:                                     *
//*    -- GUI_EDITOR switch in the oepUserPrompt() method:                       *
//*       The method opens the 'Launch External Program' dialog for selecting    *
//*       launch options.                                                        *
//*       When launching the default editor through the context menu to edit a   *
//*       non-GUI executable file (shell script, etc.), the default state        *
//*       (set/reset) of the upiGED Radiobutton is controlled by this switch.    *
//*       The decision whether the application will assume that the default      *
//*       editor to be launched is a GUI application or a console application    *
//*       is determined by the state of this radio button. The default is to     *
//*       assume that a GUI editor 'Gedit', 'Kate', 'Bluefish', etc. will be     *
//*       launched. If your default script editor is a console-based             *
//*       application ('vi', 'vim', 'pico', 'jed', etc), you can define this     *
//*       switch to make the Radiobutton reset by default.                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************
//* =================                                                            *
//* == To Do List: ==                                                            *
//* =================                                                            *
//*                                                                              *
//* -- Error in clipboard when copy operation while synchLock is active and at   *
//*    other times. Item is copied to clipboard successfully, but when moving    *
//*    to the other window to paste, the clipboard is empty.                     *
//*                                                                              *
//* -- Error when trying to copy hidden files, e.g. ".bashrc".                   *
//*    This is odd and may be related to the clipboard problem described above.  *
//*                                                                              *
//* -- Error establishing clipboard connection when invoking as superuser.       *
//*    This is because the wl-clipboard utilities are not functional under       *
//*    'sudo' commands, presumably because a needed file is not on the           *
//*    superuser's $PATH. "sudo wl-paste" reports the error:                     *
//*       Failed to connect to a Wayland server: No such file or directory       *
//*       Please check whether /run/user/0/wayland-0 socket exists and is        *
//*       accessible.                                                            *
//*    Note that '/run/user/1000/wayland-0' exists BUT '/run/user/0/wayland-0'   *
//*    does not. See notes on potential fix in 'fmgr.sh'.                        *
//*                                                                              *
//* -- Error when trying to copy a file when filename includes a '?' character.  *
//*    Some filesystems will not accept '?' as a filename character.             *
//*                                                                              *
//* -- For GVFS, we need to escape '(' ')' and potentially other "special"       *
//*    characters for copy/paste operations.                                     *
//*                                                                              *
//* -- Add configuration option for searches within 'less' utility.              *
//*    case-sensitive vs. case-insensitive (default)                             *
//*    'less' defaults to case-sensitive search, unless command-line option      *
//*    specified: '-i' (--ignore-case) '-I' (--IGNORE-CASE).                     *
//*                                                                              *
//* -- Add "Edit" to the "Open" dialog. Target editor s/b specified in config    *
//*    file: either 'nano', 'Micro', 'ne' (NiceEditor), or ACEdit ('ace').       *
//*    NOT Emacs or Vim.                                                         *
//*    [ ] Edit the file using '%s' where editor name is taken from config file. *
//*    For plain-text files only:                                                *
//*      If executable: either run or edit.                                      *
//*      For non-executable files, assume edit.                                  *
//*    This is actually already functional for the "default" editor, but the     *
//*    default is a GUI editor (GNOME Text Editor). If user-provided command,    *
//*    then it should open as instructed, but prefill the user instructions      *
//*    with command-line editor from config file.                                *
//*       See radiobutton:  ( ) default editor is a GUI application.             *
//*    We should also be able to launch a console editor with superuser          *
//*    privlege.  ex: "sudo nano filename.pl"                                    *
//*                                                                              *
//* -- ShellOut() should return the 'system()' call exit code.                   *
//*    See the "et" bash script in the source code directory for one way to      *
//*    capture the return value of a system call.                                *
//*    The application launched via a system() call has an exit code, but this   *
//*    is difficult to capture. See NcDialog::ShellOut() for more information.   *
//*    -- If we have multiple threads running, when this is called, then         *
//*       ALL of them must go to sleep, or at least perform no I/O.              *
//*    -- Verify return value from Systemcall method for callers that need it.   *
//*                                                                              *
//* -- Support for additional archive types? Either full support or view-only,   *
//*    or View/Expand.                                                           *
//*    types: rpm (distro archives)                                              *
//*           rar (primarily Windoze)                                            *
//*                                                                              *
//* -- Support for viewing of CSV (Comma-Seperated-Value) files. This code is    *
//*    functional in the CsvView test application, but may be too big for        *
//*    inclusion in FileMangler.                                                 *
//*                                                                              *
//* -- Research 'source-highlight' utility for adding ANSI color to additional   *
//*    source files for ViewFile function.                                       *
//*                                                                              *
//* -- report stats on image file metadata:                                      *
//*    jpg, png, others?                                                         *
//*                                                                              *
//* -- report metadata stats for music files:                                    *
//*    mp3, ogg                    (done)                                        *
//*    m4a                         (done)                                        *
//*    mp4                                                                       *
//*    wma, wmv (ASF container)    (done)                                        *
//*    others?                                                                   *
//*                                                                              *
//* -- Configuration:                                                            *
//*    -- documentation for config (all current functionality is documented)     *
//*    -- Multi-language support: English, French, Spanish, Chinese, Japanese    *
//*                                                                              *
//* -- When confirmation dialog for permanent file delete, if there is           *
//*    only one file, then display its name. If multiple files, display count.   *
//*                                                                              *
//* -- For very large or very slow copy/backup/synch, open a secondary           *
//*    dialog to provide detailed status of the operation. Files, bytes,         *
//*    active threads, elapsed time and estimated time remaining                 *
//*                                                                              *
//* -- Be sure that the backup/synch log also records errors when reporting      *
//*    "updates only".                                                           *
//*                                                                              *
//*                                                                              *
//* -- When synch is enabled, be sure matching file (if any) in inactive         *
//*    window is actually visible.                                               *
//*                                                                              *
//*                                                                              *
//* -- When creating an archive (or other operations which use the               *
//*    "Processing..." message, replace the message with a pinwheel.             *
//*    This should appear to happen inside the same textbox that currently       *
//*    displays the message.                                                     *
//*                                                                              *
//* -- Verify the connection with system clipboard. Useful for file rename       *
//*    dialog and other operations.                                              *
//*    -- There is a difference between the local clipboard which contains a list*
//*       of filespecs, and the system clipboard which likely contains text data.*
//*    -- A paste operation to a textbox should always be data from the system   *
//*       clipboard (if available).                                              *
//*                                                                              *
//* --                                                                           *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* -- Note: When deciding where to put new methods, display source file size.   *
//*          ls -l *.cpp | sort -g --key 5,5                                     *
//********************************************************************************

//* Header Files *
#include "FileMangler.hpp"    // FileMangler definitions and data
#include "FmConfig.hpp"       // For FmConfigCommand and FmConfigStatus definitions

//* Application exit values. These are interpreted by the invocation script. *
#define FMG_ORIG_WD     (0)   // Exit to CWD where application was invoked
#define ERROR_RETURN    (1)   // One or more serious errors during startup
#define FMG_CURR_WD     (2)   // Exit to CWD most recently visited by application


//**************
//* Local data *
//**************
static int exitCode = FMG_ORIG_WD ; // indicates CWD for application exit


//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage: run FileMangler --help                                 *
//*                                                                            *
//* Command Line Arguments: See DisplayHelp()                                  *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* User may have specified one or more command-line arguments.           *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object and interact with the user.       *
   FileMangler* fmptr = new FileMangler( clArgs ) ;

   //* Before returning to the system, delete the FileMangler object.*
   //* While the class object and all its resources will be released *
   //* by the system on exit, we want to force execution of the      *
   //* destructor for an orderly termination.                        *
   delete fmptr ;

   exit ( exitCode ) ;

}  //* End main() *

//*************************
//*     ~FileMangler      *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

FileMangler::~FileMangler ( void )
{
   //* Release memory for control definitions *
   if ( this->ic != NULL )
   { delete [] this->ic ; this->ic = NULL ; }

   //* Delete an temporary files created by the application.*
   // NOTE: The exit-to-new-CWD temp file (if any) IS NOT deleted here.
   this->DeleteTemppath () ;

}  //* End ~FileMangler() *

//*************************
//*      FileMangler      *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

FileMangler::FileMangler ( commArgs& clArgs )
{
   //* Initialize our data members *
   this->dWin   = NULL ;
   this->fWin   = NULL ;
   this->rWin   = NULL ;
   this->fdPtr  = NULL ;
   this->altPtr = NULL ;
   this->ic     = NULL ;
   this->favdirCount  = ZERO ;
   this->mntcmdCount  = ZERO ;
   this->termRows     = ZERO ;
   this->termCols     = ZERO ;
   this->actualRows   = MIN_ROWS ;
   this->actualCols   = MIN_COLS ;
   this->lockMenuBar  = false ;
   this->synchLock    = false ;
   this->exit2cwd     = false ;
   this->menuBase     = { 0, 11 } ;
   this->titlePos     = { 0, 0 } ;
   this->monoColor[0] = attrDFLT ;
   this->monoColor[1] = ZERO ;
   this->titleColor   = ZERO ;
   this->suPos        = { 2, ZERO } ;

#if 0    // NOT YET IMPLEMENTED
   //* Set default key map values *
   *this->keyMap.keyFile = NULLCHAR ;
   for ( short i = ZERO ; i < VALID_CMD_KEYS ; i++ )
   {
// NOT YET IMPLEMENTED
   }
#endif   // NOT YET IMPLEMENTED

   //* See what the user has to say *
   this->GetCommandLineArgs ( clArgs ) ;

   //* Note that all members of this->cfgOpt are initialized upon *
   //* instantiation EXCEPT:                                      *
   //*   'appPath', the path of the executable                    *
   //*   'trPath',  the path to the desktop's trashcan            *
   //*   'bkPath',  the path of backup/synch definition file      *
   //*   'tfPath',  the path for creating temp files              *
   //*   'gvPath;,  the path where GVfs devices are mounted.      *
   gString gsOut ;
   gsOut = clArgs.appPath ;         // application path
   gsOut.copy( this->cfgOpt.appPath, MAX_PATH ) ;
   gsOut = ConfigDfltTrashcanPath ; // default trashcan path
   gsOut.copy( this->cfgOpt.trPath, MAX_PATH ) ;
   gsOut = clArgs.backPath ;        // backup definition path
   gsOut.copy( this->cfgOpt.bkPath, MAX_PATH ) ;
   bool badTmpDir = false ;
   if ( (this->CreateTemppath ( gsOut )) != false )   // temp-file directory
      gsOut.copy( this->cfgOpt.tfPath, MAX_PATH ) ;
   else
   {  //* We cannot continue without a valid temp-file directory. *
      clArgs.configFlag = false ;      // prevent call to configuration
      badTmpDir = true ;
   }

   //* If user is invoking interactive configuration *
   if ( clArgs.configFlag != false )
   {
      //* Construct the path/filename of configuration file, then *
      //* invoke the child process for interactive configuration. *
      //* If config file does not exist, user will be asked       *
      //* whether he/she/it wants to create it.                   *
      //* If configuration successful, continue start-up sequence.*
      if ( *clArgs.cfgPath == NULLCHAR )
      {
         gsOut.compose( L"%s/%s", clArgs.appPath, ConfigFilename ) ;
         gsOut.copy( clArgs.cfgPath, MAX_PATH ) ;
      }
      if ( (this->Configure ( ccINTERACT, clArgs.appPath, clArgs.cfgPath, false, 
            bool(clArgs.diagPause > 1 ? true : false) )) == csOK )
         clArgs.configFlag = false ;
   }

   //** Initialize the NCurses Engine                                         **
   if( (!(clArgs.helpFlag || clArgs.verFlag || badTmpDir)) && (!clArgs.configFlag) && 
       ((nc.StartNCursesEngine ()) == OK) )
   {
      //* Get the size of our playground *
      nc.ScreenDimensions ( this->termRows, this->termCols ) ;

      //* Verify that the current locale supports UTF-8 character encoding.    *
      gString envLocale( nc.GetLocale() ) ;  // name of locale from environment
      short localeSetOk = nc.VerifyLocale ( envLocale.ustr() ),// env locale supports UTF-8?
            altLocaleOk = OK ;               // alt locale supports UTF-8?
      if ( clArgs.altLocale != false )
      {
         //* If user has specified an alternate locale for interpretation of   *
         //* character output and input, it must be passed to the NCurses      *
         //* class before any I/O occurs.                                      *
         //* Note: If clArgs.localeName does not specify a locale supported by *
         //*       the terminal program, OR the locale specified is not a      *
         //*       UTF-8-encoded locale, OR the default encoding specified in  *
         //*       the environment was not UTF-8, then the results will be     *
         //*       ambiguous at best.                                          *
         altLocaleOk = nc.SetLocale ( clArgs.localeName ) ;
      }
      nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed
      this->DrawTitle () ;                   // draw app title line to console

      this->DiagMsg ( "NCurses engine initialized.", nc.bw ) ;

      //* Determine whether the hardware supports display of color text. *
      if ( nc.ColorText_Available () )
      {
         this->DiagMsg ( "Multi-color screen output supported.", nc.grB ) ;

         //* Color Engine successfully initialized? *
         if ( nc.ColorText_Initialized () )
         {
            termColorInfo tci ;
            nc.TerminalColorSupport ( tci ) ;
            gsOut.compose( L"Color Engine started. (%hd RGB Registers and "
                            "%hd fgnd/bkgnd Color Pairs)", 
                           &tci.rgb_regs, &tci.fgbg_pairs ) ;
            this->DiagMsg ( gsOut.ustr(), nc.grB ) ;
         }
         else
         {
            this->DiagMsg ( "Unable to start Color Engine.", nc.bw ) ;
         }
      }
      else
      {
         this->DiagMsg ( "Terminal does not support color output.", nc.bw ) ;
         this->DiagMsg ( " Starting in monochrome mode.", nc.bw ) ;
      }

      //* Report the results of locale settings for character-encoding *
      chrono::duration<short>aWhile( 4 ) ;   // pause interval for user to read message
      this->DiagMsg ( "Locale from environment: ", nc.grB, false ) ;

      gsOut.compose( L"\"%S\"", envLocale.gstr() ) ;
      this->DiagMsg ( gsOut.ustr(), nc.bw, false ) ;
      if ( localeSetOk == OK )
         this->DiagMsg ( "  UTF-8 encoding support verified.", nc.grB ) ;
      else
      {
         this->DiagMsg ( "  may not support UTF-8 encoding.", nc.reB ) ;
         if ( ! clArgs.altLocale )
            this_thread::sleep_for( aWhile ) ;
      }
      if ( clArgs.altLocale != false ) // alternate locale specified
      {
         this->DiagMsg ( "Alternate Locale       : ", nc.grB, false ) ;
         if ( altLocaleOk == OK )
         {
            gsOut.compose( L"\"%s\"", nc.GetLocale() ) ;
            this->DiagMsg ( gsOut.ustr(), nc.bw, false ) ;
            this->DiagMsg ( "  UTF-8 encoding support verified.", nc.grB ) ;
         }
         else
         {
            gsOut.compose( L"\"%s\"", clArgs.localeName ) ;
            this->DiagMsg ( gsOut.ustr(), nc.bw, false ) ;
            this->DiagMsg ( "  may not support UTF-8 encoding (not loaded).", nc.reB ) ;
            this_thread::sleep_for( aWhile ) ;
         }
      }

      //* Create the path/filename string identifying the configuration file.  *
      gString cfgName( ConfigFilename ) ;
      if ( *clArgs.cfgPath != NULLCHAR )  // if alternate config file specified
      {  //* User may have specified: filename, relative path/filename, *
         //* or full path/filename. Resolve to full path/filename.      *
         if ( (this->gcaRealpath ( gsOut, clArgs.cfgPath )) )
         {
            gsOut.copy( clArgs.cfgPath, MAX_PATH ) ;
            short wi = (gsOut.findlast( fSLASH )) + 1 ;
            cfgName = &gsOut.gstr()[wi] ;
         }
         else        // invalid path for alt config file, or file not found
         {
            this->DiagMsg ( "Alternate configuration file: '", nc.reB, false ) ;
            this->DiagMsg ( clArgs.cfgPath, nc.bw, false ) ;
            this->DiagMsg ( "' not found.", nc.reB ) ;
            this->DiagMsg ( "Continuing with default configuration.", nc.reB ) ;
            this_thread::sleep_for( aWhile ) ;
            *clArgs.cfgPath = NULLCHAR ;  // induce default config file
         }
      }
      if ( *clArgs.cfgPath == NULLCHAR )  // use default config file
      {
         gsOut.compose( L"%s/%s", clArgs.appPath, ConfigFilename ) ;
         gsOut.copy( clArgs.cfgPath, MAX_PATH ) ;
      }

      //* Read configuration file and set the various parameters specified     *
      //* in configuration file.                                               *
      //* Note that default configuration file lies on the application path.   *
      if ( clArgs.diagPause > 1 )   // extended-diagnostic message
      {
         this->DiagMsg ( "Application Path  : ", nc.bl, false ) ;
         this->DiagMsg ( this->cfgOpt.appPath, nc.bl ) ;
         this->DiagMsg ( "Configuration Path: ", nc.bl, false ) ;
         this->DiagMsg ( clArgs.cfgPath, nc.bl ) ;
         this->DiagMsg ( "Trashcan Path     : ", nc.bl, false ) ;
         this->DiagMsg ( this->cfgOpt.trPath, nc.bl ) ;
         this->DiagMsg ( "Tempfile Path     : ", nc.bl, false ) ;
         this->DiagMsg ( this->cfgOpt.tfPath, nc.bl ) ;
         this->DiagMsg ( "Backup Definition : ", nc.bl, false ) ;
         if ( *this->cfgOpt.bkPath != NULLCHAR )
            this->DiagMsg ( this->cfgOpt.bkPath, nc.bl ) ;
         else
            this->DiagMsg ( "(none)", nc.bl ) ;
      }
      this->DiagMsg ( "Reading configuration file: ", nc.grB, false ) ;
      this->DiagMsg ( cfgName.ustr(), nc.br ) ;

      modeStat mStat = mstatSINGLE ;   // main loop control (see below)
      ConfigStatus cfgstat ;
      if ( (cfgstat = (ConfigStatus)this->Configure ( ccREAD, clArgs.appPath, 
                           clArgs.cfgPath, true, 
                           bool(clArgs.diagPause > 1 ? true : false) )) != csOK )
      {  //* Pause so user has time to read the error message *
         if ( clArgs.diagPause == ZERO )
         {
            chrono::duration<short>aWhile( 5 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
         //* If configuration file not found, invoke       *
         //* configuration utility, so user can create one.*
         if ( cfgstat == csNOTFOUND )
            mStat = mstatCONFIG ;
         //* If memory allocation error, it probably means that user invoked   *
         //* the application before the OS was fully awake. This is common.    *
         else if ( cfgstat == csACCESS )
            mStat = mstatEXIT ;
      }

      //* Process any command-line args (overrides config file settings).*
      this->DiagMsg ( "Applying command-line args", nc.grB ) ;
      if ( clArgs.modeFlag != false )
         cfgOpt.winMode = clArgs.winMode ;
      if ( clArgs.sortFlag != false )
         cfgOpt.sortOption = clArgs.sortOption ;
      if ( clArgs.schemeFlag != false )
         cfgOpt.cScheme.scheme = clArgs.colorScheme ;
      if ( clArgs.hideFlag != false )
         this->cfgOpt.showHidden = true ;
      if ( clArgs.mouseFlag != false )
         this->cfgOpt.enableMouse = true ;
      this->DiagMsg ( "Opening application dialog window.", nc.grB ) ;
      //* If specified on the command-line  *
      //* wait for user to read diagnostics.*
      if ( clArgs.diagPause > ZERO )
      {
         if ( clArgs.diagPause == 1 )
         {
            chrono::duration<short>aWhile( 8 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
         else
         {
            this->DiagMsg ( " * Press Any Key to Continue *", nc.bl ) ;
            nckPause();
         }
      }

      //* If a Backup or Synch operation specified:   *
      //* a) force Dual-Window mode                   *
      //* b) push Backup command key into the stream. *
      if ( *this->cfgOpt.bkPath != NULLCHAR )
      {
         this->cfgOpt.winMode = wmDUAL_WIN ;
         wkeyCode wk( nckAS_B, wktEXTEND ) ;
         nc.UngetKeyInput ( wk ) ;
      }


      //**************************************
      //* Wander through the directory tree, *
      //* causing havoc at each opportunity. *
      //**************************************
      //* This loop handles the dynamic switching *
      //* between Single- and Dual-Window modes.  *
      while ( mStat != mstatCONFIG && (mStat == mstatSINGLE || mStat == mstatDUAL) )
      {  //* Get current terminal-window size *
         nc.ScreenDimensions ( this->termRows, this->termCols ) ;

         if ( this->cfgOpt.winMode == wmSINGLE_WIN 
              || (this->cfgOpt.winMode == wmTERM_WIN && 
                  this->termCols < MIN_DUALWINDOW_WIDTH) )
         {
            this->cfgOpt.winMode = wmSINGLE_WIN ; // make config setting reflect reality
            if ( (mStat = this->SingleWin_Mode ( clArgs.startDir, 
                                                 clArgs.treeFlag, 
                                                 clArgs.rootFlag )) == mstatDUAL )
               this->cfgOpt.winMode = wmDUAL_WIN ;
         }
         else
         {
            cfgOpt.winMode = wmDUAL_WIN ;   // make config setting reflect reality
            if ( (mStat = this->DualWin_Mode ( clArgs.startDir, 
                                               clArgs.treeFlag, 
                                               clArgs.rootFlag )) == mstatSINGLE )
               this->cfgOpt.winMode = wmSINGLE_WIN ;
         }
         //* Specified start directory (if any) and tree-view *
         //* flag are in effect only for first iteration.     *
         *clArgs.startDir = NULLCHAR ;
         clArgs.treeFlag = false ;

         //* The main dialog window is now closed, but the NCurses Engine *
         //* is still active. If user has requested the FileMangler       *
         //* configuration utility from the application menu, call config.*
         if ( mStat == mstatCONFIGR )
         {
            this->Configure ( ccINTERACT, clArgs.appPath, clArgs.cfgPath, true ) ;

            //* Open an Decision Dialog asking whether we should re-start now.*
            if ( this->Restart () )
               mStat = mstatRESTART ;
            else
               mStat = cfgOpt.winMode == wmDUAL_WIN ? mstatDUAL : mstatSINGLE ;
         }
      }

      nc.RestoreCursorState () ;             // make cursor visible
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* If FileMangler is incorrectly configured, invoke FmConfig.*
      if ( mStat == mstatCONFIG )
      {
         this->Configure ( ccINTERACT, clArgs.appPath, clArgs.cfgPath, false ) ;
      }
      else if ( mStat == mstatRESTART )
      {  //* Pass the same list of arguments that were passed to us *
         execl ( clArgs.argList[0],   // invocation path/filename
                 clArgs.argList[0],   // copy of invocation path/filename
                 clArgs.argCount > 1 ? clArgs.argList[1] : NULL,
                 clArgs.argCount > 2 ? clArgs.argList[2] : NULL,
                 clArgs.argCount > 3 ? clArgs.argList[3] : NULL,
                 clArgs.argCount > 4 ? clArgs.argList[4] : NULL,
                 clArgs.argCount > 5 ? clArgs.argList[5] : NULL,
                 clArgs.argCount > 6 ? clArgs.argList[6] : NULL,
                 clArgs.argCount > 7 ? clArgs.argList[7] : NULL,
                 clArgs.argCount > 8 ? clArgs.argList[8] : NULL,
                 clArgs.argCount > 9 ? clArgs.argList[9] : NULL,
                 NULL ) ;
      }
      else     // exit the application
      {  //* Either exit to the original working directory OR exit to the      *
         //* current working directory. The exit code will be interpreted by   *
         //* the invocation script ('fmg.sh' by default), which will optionally*
         //* execute a 'cd' (chdir) command constructed by the Singlewin_Mode()*
         //* or Dualwin_Mode() method when it exits.                           *
         //* Note that this will have an effect ONLY if the invocation script  *
         //* was run in the base shell process. (see memory-resident function) *
         if ( mStat == mstatEXITC )
            exitCode = FMG_CURR_WD ;
         else     // mstatEXIT
            exitCode = FMG_ORIG_WD ;
      }
   }
   else
   {
      if ( clArgs.verFlag )
         this->DisplayVersion () ;
      else if ( clArgs.helpFlag )
         this->DisplayHelp () ;
      else if ( badTmpDir )
         wcout << "SYSTEM ERROR: Unable to establish path for temporary files." << endl ;
      else if ( !clArgs.configFlag )
         wcout << "\n ERROR! Unable to initialize NCurses engine.\n" << endl ;
   }

}  //* End FileMangler() *

//*************************
//*    CreateTemppath     *
//*************************
//******************************************************************************
//* Create a unique directory in the system's temp-file directory to contain   *
//* the application's temporary files. Resulting path is returned to caller.   *
//*                                                                            *
//* This method calls 'mkdtemp' to create the unique directory name.           *
//* All temporary files created by the application will live in this directory,*
//* so clean-up on exit will be easy.                                          *
//*                                                                            *
//* Note that the temp file for exit-to-CWD is not stored in this directory    *
//* because the invocation shell script needs it _after_ exit.                 *
//*                                                                            *
//* Input  : basePath: (by reference) receives the path/dirname                *
//*                    for creation of temporary files                         *
//*                                                                            *
//* Returns: 'true'  if path/dirname created successfully                      *
//*          'false' if library call failed                                    *
//******************************************************************************
//* Programmer's Note: We do this in the base application code rather than in  *
//* the FileDlg class because it is done only once for the application, AND    *
//* because the base code needs access to the temp-file directory.             *
//******************************************************************************

bool FileMangler::CreateTemppath ( gString& basePath )
{
   char tn[gsDFLTBYTES] ;              // receives the particularized filespec
   bool  status = false ;              // return value

   //* Get path of temp directory*
   if ( (this->GetTempdirPath ( basePath )) != false )
   {
      basePath.append( "/FMG_XXXXXX" ) ;
      basePath.copy( tn, gsDFLTBYTES ) ;
      if ( (mkdtemp ( tn )) != NULL )
      {
         basePath = tn ;
         status = true ;
      }
      else
         basePath.clear() ;
   }
   return status ;

}  //* End CreateTemppath() *

//*************************
//*    CreateTempname     *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file within the previously   *
//* established application temporary directory. (See CreateTemppath())        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************

bool FileMangler::CreateTempname ( gString& tmpPath )
{
   bool  status = false ;

   char tn[gsDFLTBYTES] ;
   gString gstmp( "%s/FMG_XXXXXX", this->cfgOpt.tfPath ) ;
   gstmp.copy( tn, gsDFLTBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpPath = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpPath.clear() ;
   return status ;

}  //* End CreateTempname() *

//*************************
//*    DeleteTemppath     *
//*************************
//******************************************************************************
//* Delete the directory containing the application's temporary files.         *
//*                                                                            *
//* Called by the FileMangler destructor. At this point, the directory         *
//* _should be_ empty, but in case it is not, we first delete the contents     *
//* of the directory, and then delete the directory itself.                    *
//*                                                                            *
//* Note that the temp file for exit-to-CWD is not stored in this directory    *
//* because the invocation shell script needs it _after_ exit.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//*          (There is no point in complaining if we fail.)                    *
//******************************************************************************
//* Programmer's Note: Fortunately, we will see only non-special filenames     *
//* here in the temp directory; however, please see the FMgr-class for notes   *
//* on using 'rmdir' and 'unlink' on filenames which contain "special"         *
//* characters.                                                                *
//******************************************************************************

void FileMangler::DeleteTemppath ( void )
{
   //* First scan the target directory for its contents *
   //* and delete each remaining temp file (if any).    *
   DIR* dirPtr = NULL ;
   if ( (dirPtr = opendir ( this->cfgOpt.tfPath )) != NULL )
   {
      gString   tspec ;             // target filespec
      deStats*  destat ;            // directory-entry record
      bool      showFile ;          // 'true' if file is to be copied

      //* Read each item in the directory *
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names.              *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            //* Build full path/filename for target, then delete the file.*
            // Programmer's Note: The filetype test uses the compiler's 
            // primitive: DT_DIR: (unsigned char)(4).
            tspec.compose( "%s/%s", this->cfgOpt.tfPath, destat->d_name ) ;
            if ( destat->d_type == DT_DIR )
            {
               // Programmer's Note: At this time the application does not 
               // create any subdirectories in our tempfile directory, so this 
               // is an unnecessary step.
               rmdir ( tspec.ustr() ) ;   // remove the directory
            }
            else
               unlink ( tspec.ustr() ) ;            
         }
      }
      closedir ( dirPtr ) ;      // Close the directory

      //* Delete the directory itself. *
      rmdir ( this->cfgOpt.tfPath ) ;
   }

}  //* End DeleteTemppath() *

//*************************
//*    GetTempdirPath     *
//*************************
//******************************************************************************
//* This is a stub function which provides access to the C++17 function:       *
//*                       temp_directory_path()                                *
//*                                                                            *
//* Because most programmer's will not reach C++17 for at least several        *
//* months, we isolate the call here and implement what we expect is the way   *
//* the library function will be implemented.                                  *
//*         (This WILL NOT work under Windoze--and we don't care.)             *
//*         (Screw Windoze, and the horse it rode in on.         )             *
//* -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- *
//* From <http://en.cppreference.com>                                          *
//* ---------------------------------                                          *
//* These functions are available in C++17 (released Jan. 2018):               *
//* const char* fs::temp_directory_path ( void ) ;                             *
//* const char* fs::temp_directory_path ( std::error_code& ec ) ;              *
//*                                                                            *
//* Returns A directory suitable for temporary files. The path is guaranteed   *
//* to exist and to be a directory. The overload that takes error_code&        *
//* argument returns an empty path on error.                                   *
//*                                                                            *
//* Exceptions: The overload that does not take a std::error_code& parameter   *
//* throws filesystem_error on underlying OS API errors, constructed with path *
//* to be returned as the first argument and the OS error code as the error    *
//* code argument. std::bad_alloc may be thrown if memory allocation fails.    *
//* The overload taking a std::error_code& parameter sets it to the OS API     *
//* error code if an OS API call fails, and executes ec.clear() if no errors   *
//* occur.                                                                     *
//*                                                                            *
//* On POSIX systems, the path may be the one specified in the environment     *
//* variables TMPDIR, TMP, TEMP, TEMPDIR, and, if none of them are specified,  *
//* the path "/tmp" is returned.                                               *
//*                                                                            *
//* Input  : tdPath : (by reference) receives the path string                  *
//*                                                                            *
//* Returns: 'true'  if successful, 'false' if system error                    *
//******************************************************************************

bool FileMangler::GetTempdirPath ( gString& tdPath )
{
   //* Default path of temp directory on GNU/Linux systems. *
   //* (Used only if environment variable not set.)         *
   const char* const dfltPath = "/tmp" ;

   const char* envPath ;            // returned by getenv()
   tnFName     fStats ;             // target filestats
   bool        status = false ;     // return value

   if ( (envPath = std::getenv ( "TMPDIR" )) == NULL )
      if ( (envPath = std::getenv ( "TMP" )) == NULL )
         if ( (envPath = std::getenv ( "TEMP" )) == NULL )
            if ( (envPath = std::getenv ( "TEMPDIR" )) == NULL )
               envPath = dfltPath ;
   tdPath = envPath ;
   if ( (stat64 ( tdPath.ustr(), &fStats.rawStats )) == OK )
   {
      if ( ((S_ISDIR(fStats.rawStats.st_mode)) != false) &&
           ((access ( tdPath.ustr(), R_OK )) == ZERO) && 
           ((access ( tdPath.ustr(), W_OK )) == ZERO) )
         status = true ;
   }
   return status ;

}  //* End GetTempdirPath()

//*************************
//*     Configure         *
//*************************
//******************************************************************************
//* Invoke the configuration utility as a separate process.                    *
//* The FileMangler process goes to sleep during this activity.                *
//*                                                                            *
//* Input  : cmd    : option for configuration thread to execute               *
//*                   NOTE: To avoid including FmConfig.hpp in the whole       *
//*                         application, this is passed in as a 'short int'.   *
//*          appPath: path of FileMangler executable                           *
//*                   (FmConfig executable lives on the same path)             *
//*          cfgFile: path/filename of configuration file to read/modify       *
//*          ncEngine: 'true' if caller has started the NCurses engine         *
//*          verbose: (optional, false by default)                             *
//*                   if 'true' display extended diagnostics returned from     *
//*                   child process                                            *
//*                                                                            *
//* Returns: member of ConfigStatus (disguised as a 'short int')               *
//*          (disguised as a 'short int' so remainder of application        )  *
//*          (doesn't need to know anything about the configuration utility.)  *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - This method spawns a child process, and independent application program. *
//*   This is a lovely opportunity for bugs to hatch - beware!                 *
//*                                                                            *
//* - If a dialog window is open when this method is called, it is caller's    *
//*   responsibility to save its display data. We write directly to the        *
//*   NCurses console window.                                                  *
//*                                                                            *
//* - If 'cmd' == ccREAD, then display start-up diagnostic messages, else      *
//*   don't write messages to the console window.                              *
//******************************************************************************

short FileMangler::Configure ( short cmd, const char* appPath, 
                               const char* cfgFile, bool ncEngine, bool verbose )
{
   #define DEBUG_Configure (0)

   ConfigCommand  ccCmd = cmd == ccREADSILENT ? ccREAD : (ConfigCommand)cmd ;
   ConfigStatus   csStatus = csACCESS ;   // return value

   //* Create a new shared memory segment.            *
   //* (fail if segment already exists)               *
   //* Actual size is an even multiple of PAGE_SIZE.  *
   //* Permissions 9 LSB: 1 1011 0100 - rw- rw- r--   *
   // Programmer's Note: This memory allocation may fail if the system's memory 
   // management module has not yet been fully initialized. This is a failure at the 
   // system level, not at the application level. To compensate, we place the allocation 
   // in a loop, so a retry will occur. So far, the second try has never failed.
   int      mPerm = 0x01B4 | IPC_CREAT | IPC_EXCL ;
   size_t   mBytes = sizeof(sharedMemMap) ;  // number of bytes in block
   key_t    mKey = IPC_PRIVATE ;             // block name
   chrono::duration<short, std::milli>aWhile( 1500 ) ; // pause interval if memory-allocation failure
   int      segID ;                          // virtual address
   for ( short i = 1 ; i < 3 ; ++i )
   {
      if ( (segID = shmget ( mKey,  mBytes, mPerm )) > ZERO )
         break ;
      this_thread::sleep_for( aWhile ) ;
   }

   if ( segID > ZERO )
   {
      //* Attach memory segment to application's address space *
      //* at a position selected by the system.                *
      //* Read/write is the default.                           *
      void* segAddress = shmat ( segID, (void *)NULL, ZERO ) ;
      bool isAttached = ( segAddress != (void*)(-1) ) ;
      if ( isAttached )
      {  //* Define the layout of the shared memory area *
         //* according to the sharedMemMap class, and    *
         //* copy current data set into it.              *
         sharedMemMap* smmPtr = (sharedMemMap*)segAddress ;
         strncpy ( smmPtr->appPath, appPath, MAX_PATH ) ;
         strncpy ( smmPtr->cfgPath, cfgFile, MAX_FNAME ) ;
         *smmPtr->trashPath = NULLCHAR ;
         for ( short i = ZERO ; i < this->favdirCount ; i++ )
            strncpy ( smmPtr->favPath[i], this->FavoriteDirs[i], MAX_PATH ) ;
         smmPtr->favCount = this->favdirCount ;
         for ( short i = ZERO ; i < this->mntcmdCount ; i++ )
            strncpy ( smmPtr->mntCmd[i], this->MountCmds[i], MAX_DIALOG_WIDTH ) ;
         smmPtr->mntCount = this->mntcmdCount ;
         smmPtr->lockMb = this->lockMenuBar ;
         smmPtr->keyMap = this->keyMap ;
         smmPtr->cfgOpt = this->cfgOpt ;
         smmPtr->msgCount = ZERO ;
         smmPtr->ccCmd = ccCmd ;
         smmPtr->fcStatus = csACCESS ; // in case child process can't access shared memory
         smmPtr->verbose = verbose ;

         //* Create the command string *
         gString gsCmd ;
         gsCmd.compose( L"\"%s/%s\" %d", appPath, ConfigApp, &segID ) ;

         if ( ncEngine )
         {
            #if DEBUG_Configure == 0
            nc.Hibernate () ;             // put the NCurses engine to sleep
            #else
            this->DiagMsg ( "Starting FmConfig process", nc.bl ) ;
            this->DiagMsg ( gsCmd.ustr(), nc.bl ) ;
            #endif   // DEBUG_Configure
         }

         this->Systemcall ( gsCmd.ustr() ) ;
         // NOTE: the value returned is not necessarily the one the child sent 
         //       us, nor was the child necessarily able to place anything in 
         //       the fcStatus field. (we have compensated)
         csStatus = smmPtr->fcStatus ;

         if ( ncEngine )
         {
            #if DEBUG_Configure == 0
            nc.Wake () ;         // wake up the NCurses engine
            #else
            this->DiagMsg ( "FmConfig process completed", nc.bl ) ;
            #endif   // DEBUG_Configure
         }

         //* Display any errors that occurred in reading the config file.*
         //* Also used to display 'verbose' diagnostics.                 *
         if ( ncEngine && smmPtr->msgCount > ZERO && cmd == ccREAD )
         {
            attr_t color = nc.reB ;
            if ( smmPtr->verbose == false )
               this->DiagMsg ( "Error(s) reading configuration file:", color ) ;
            else
               color = nc.bl ;
            gString gsOut ;
            for ( short i = ZERO ; i < smmPtr->msgCount ; i++ )
            {
               gsOut.compose( L"  %s", smmPtr->msg[i] ) ;
               this->DiagMsg ( gsOut.ustr(), color ) ;
            }
            if ( smmPtr->verbose == false )
               this->DiagMsg ( "Continuing with default value(s).", color ) ;
         }
         #if 0    // DEBUGGING OF CHILD PROCESS ONLY
         else if ( !ncEngine && smmPtr->msgCount > ZERO )
         {  //* NCurses engine is not running OR is asleep *
            system ( "clear" ) ;
            wcout << L"** NCurses asleep **" << endl ;
            for ( short i = ZERO ; i < smmPtr->msgCount ; i++ )
               wcout << smmPtr->msg[i] << endl ;
            wcout << L" ** Press any Key **" << endl ;
            getchar () ;
         }
         #endif   // DEBUGGING OF CHILD PROCESS ONLY
         else if ( smmPtr->msgCount == ZERO )
         {
            switch ( csStatus )
            {
               case csOK:           // nothing to report
                  break ;
               case csACCESS:       // shared memory access error
                  if ( ncEngine && cmd == ccREAD )
                     this->DiagMsg ( "Unable to attach shared memory to child process!", nc.reB ) ;
                  break ;
               default:
                  break ;
            }
         }

         if ( cmd == ccREAD )
         {
            //* Copy the results of the configuration from the   *
            //* shared-memory segment back into our data members.*
            this->favdirCount = smmPtr->favCount ;
            for ( short i = ZERO ; i < this->favdirCount ; i++ )
               strncpy ( this->FavoriteDirs[i], smmPtr->favPath[i], MAX_PATH ) ;
            this->mntcmdCount = smmPtr->mntCount ;
            for ( short i = ZERO ; i < this->mntcmdCount ; i++ )
               strncpy ( this->MountCmds[i], smmPtr->mntCmd[i], MAX_DIALOG_WIDTH ) ;
            this->lockMenuBar = smmPtr->lockMb ;
            this->keyMap = smmPtr->keyMap ;
            this->cfgOpt = smmPtr->cfgOpt ;
         }

         //* Detach and delete the shared memory segment *
         shmdt ( segAddress ) ;
         shmctl ( segID, IPC_RMID, NULL ) ;
      }
      else if ( ncEngine && cmd == ccREAD )
      {
         this->DiagMsg ( "Error! Unable to attach shared-memory to current process", nc.reB ) ;
      }
   }
   else if ( ncEngine && cmd == ccREAD )
   {
      this->DiagMsg ( "Error! Unable to allocate shared-memory space", nc.reB ) ;
   }
   return csStatus ;

#undef DEBUG_Configure
}  //* End Configure() *

//*************************
//*      HelpAbout        *
//*************************
//******************************************************************************
//* Display the application's Help-About window.                               *
//*                                                                            *
//* Input  : ctrY : position at which to center dialog in Y                    *
//*          ctrX : position at which to center dialog in X                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::HelpAbout ( short ctrY, short ctrX )
{
//OLDconst short dialogROWS = 15 ;      // # of display lines
const short dialogROWS = 17 ;      // # of display lines
const short dialogCOLS = 70 ;      // # of display columns
enum ctrls : short { closePB = ZERO, suppPB, haCTRLS } ;
static const char* DialogText =
   "FileMangler: A file management utility for Linux.\n"
   "    Version:\n"
   "Copyright  : (c) %s Mahlon R. Smith, The Software Samurai\n"
   "                 Beijing University of Technology - Beijing, PRC\n"
   "                 马伦教授 北京工业大学 - 北京，中华人民共和国\n"
   "\n"
   "Developed under Fedora Linux, using GNU G++ (v:4.8.3 or greater)\n"
   "Requires ncursesw libraries (v:6.2 or greater)\n"
   "Software released under GNU General Public License, version 3,\n"
   "and documentation under GNU Free Documentation License, version 1.3\n"
   "               Bugs, suggestions, or possible praise?\n"
   "      Please contact the author at:  http://www.SoftwareSam.us/" ;

   short    ulY = ctrY - dialogROWS / 2,
            ulX = ctrX - dialogCOLS / 2 ;
   attr_t   dColor = this->cfgOpt.cScheme.sd,   // dialog interior color
            vColor = this->cfgOpt.cScheme.tn ;  // color for version strings
   const short dciLINES = dialogROWS - 4, // dimensions for DisplaySupportInfo dialog window
               dciCOLS  = dialogCOLS - 2,
               dciulY   = ulY + 3,
               dciulX   = ulX + 1 ;
   bool  suppFile = false ;   // 'true' if tech support request saved to file

   InitCtrl ic[haCTRLS] =        // array of dialog control info
   {
      {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - -   closePUSH *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogROWS - 2),        // ulY:       upper left corner in Y
         short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         7,                            // cols:      control columns
         " CLOSE ",                    // dispText:  
         this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
         this->cfgOpt.cScheme.pf,      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[suppPB],                  // nextCtrl:  link in next structure
      },
      {  //* 'SUPPORT INFO' pushbutton   - - - - - - - - - - - - - - -  suppPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogROWS - 2),        // ulY:       upper left corner in Y
         short(dialogCOLS - 24),       // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         21,                           // cols:      control columns
         " SUPPORT INFORMATION ",      // dispText:  
         this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
                                       // fColor:    focus color
         attr_t(this->cfgOpt.cScheme.scheme == ncbcGR ? nc.brR : nc.grR), 
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   this->dWin->SetDialogObscured () ;  // save parent dialog

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  About FileMangler  ", this->cfgOpt.cScheme.em ) ;

      //* Print dialog window's static text *
      gString gsOut( DialogText, copyrightYears ) ;
      dp->WriteParagraph ( 1, 1, gsOut, dColor ) ;

      //* Format application version string *
      gString verString( " %s ", AppVersion ) ;
      dp->WriteString ( 2, 14, verString, vColor ) ;

      dp->RefreshWin () ;                    // make the text visible

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePB )
               done = true ;
            else if ( Info.ctrlIndex == suppPB )
            {  //* Open a dialog to display interesting data needed to submit *
               //* a tech support request.                                    *
               //* Current dialog will be obscured, so save its display data. *
               dp->SetDialogObscured () ;
               suppFile = this->DisplaySupportInfo ( dciulY, dciulX, dciLINES, dciCOLS ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dWin->RefreshWin () ;               // restore parent dialog
   //* If support-request file written to CWD reread the directory *
   if ( suppFile )
   {
      this->fdPtr->RefreshCurrDir () ;
      gString tsfn( tsFName ) ;
      short indx ;
      this->fdPtr->Scroll2MatchingFile ( tsfn, indx, true ) ;
   }

}  //* End HelpAbout() *

//*************************
//*  DisplaySupportInfo   *
//*************************
//******************************************************************************
//* Display version numbers for NcDialog API, ncursesw library.                *
//* Also reports current 'locale' setting and dynamic load (shared) libraries. *
//*                                                                            *
//* Input  : ulY   : upper-left corner of dialog in Y                          *
//*          ulX   : upper-left corner of dialog in X                          *
//*          dLines: dialog lines                                              *
//*          dCols : dialog columns                                            *
//*                                                                            *
//* Returns: 'true' if support request saved to CWD, else 'false'              *
//******************************************************************************
//* Notes: For technical support requests, ask user to save the information    *
//* to a file:                                                                 *
//*                                                                            *
//*  1) Header                                                                  *
//*  2) FileMangler version                                                     *
//*  3) NcDialog API version                                                    *
//*  4) ncursesw version                                                        *
//*  5) locale                                                                  *
//*  6) Gvfs utilities version
//*  7) DLLs                                                                    *
//*  8) system info: uname -a                                                   *
//*  9) Environment variables: printenv SHELL TERM COLORTERM LANG               *
//* 10) UDC time                                                                *
//* 11) Prompt for description of issue                                         *
//*                                                                            *
//******************************************************************************

bool FileMangler::DisplaySupportInfo ( short ulY, short ulX, short dLines, short dCols )
{
   attr_t dColor = this->cfgOpt.cScheme.sd,  // dialog interior color
          vColor = this->cfgOpt.cScheme.tn,  // color for version strings
          hColor = this->cfgOpt.cScheme.em ; // highlight color
   bool tsRequestSaved = false ;             // return value

   enum ctrls : short { closePB = ZERO, savePB, controlsDEFINED } ;

   InitCtrl ic[controlsDEFINED] =      // array of dialog control info
   {
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -  closePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dLines - 2),            // ulY:       upper left corner in Y
      short(dCols / 2 - 4),         // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      " CLOSE ",                    // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[savePB]                   // nextCtrl:  link in next structure
   },
   {  //* 'SAVE' pushbutton   - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[closePB].ulY,              // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      " SAVE TO FILE ",             // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      attr_t(this->cfgOpt.cScheme.scheme == ncbcGR ? nc.brR : nc.grR),
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dLines,         // number of display lines
                       dCols,          // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  Technical Support Information  ", hColor ) ;

      localTime lt ;                      // formatted timestamp
      this->fdPtr->GetLocalTime ( lt ) ;  // get the local time
      gString gsOut( "FileMangler Tech Support Request\n"
                     "================================\n"
                     "FileMangler v:%s\n"
                     "Requested: %04hd-%02hd-%02hdT%02hd:%02hd:%02hd\n", 
                     AppVersion, &lt.year, &lt.month, &lt.date,
                     &lt.hours, &lt.minutes, &lt.seconds ) ;

      //* Query the GVfs filesystem support version *
      gString gvfsVersion ;
      if ( !(this->fWin->GetGVfsVersion ( gvfsVersion )) )
         gvfsVersion = " unknown " ;

      //* Create a temporary log *
      gString gsLog ;
      this->CreateTempname ( gsLog ) ;
      ofstream ofs( gsLog.ustr(), ofstream::out | ofstream::trunc ) ;
      ofs << gsOut.ustr() ;


      winPos wp( 1, 2 ) ;
      gsOut = "    Please include this information\n"
              "    with all tech support requests.\n" ;
      wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      ofs << '\n' << gsOut.ustr() << '\n' ;
      gsOut = "NcDialog API library : " ;
      wp = dp->WriteString ( wp, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( " %s \n\n", dp->Get_NcDialog_Version () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "    ncursesw library : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n\n", nc.Get_nclibrary_Version () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "      Locale setting : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n\n", nc.GetLocale () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "      GVfs Utilities : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n\n", gvfsVersion.ustr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      //* Display a list of shared libraries used in the application *
      gString gsTmp, gx ;
      this->CreateTempname ( gsTmp ) ;
      short libCount = ZERO ;
      wp = { 1, 43 } ;
      gsOut = "Dynamic (shared) Libs\n" ;
      wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( "ldd \"%s/FileMangler\" 1>\"%S\" 2>/dev/null", 
                     this->cfgOpt.appPath, gsTmp.gstr() ) ;
      this->Systemcall ( gsOut.ustr() ) ;
      gsOut.clear() ;
      ifstream ifs( gsTmp.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         char  lineData[gsDFLTBYTES] ;
         short sIndex ;
         bool  done = false ;

         while ( ! done )
         {
            ifs.getline( lineData, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || (ifs.gcount() > ZERO) )
            {
               ofs << &lineData[1] << '\n' ; // write all entries to log

               if ( libCount < 8 )
               {
                  gx = lineData ;
                  if ( ((sIndex = gx.find( L' ' )) > ZERO) &&
                       ((gx.find( "lib" )) == 1) ) // (line begins with "\tlib")
                  {
                     gx.limitChars( sIndex ) ;
                     gsOut.append( "%S\n", gx.gstr() ) ;
                     ++libCount ;
                  }
               }
            }
            else
               done = true ;
         }
         ifs.close() ;
      }
      if ( libCount == ZERO )
         gsOut = "(Unable to determine.)\n" ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      this->fdPtr->DeleteFile ( gsTmp ) ;

      //* Supplimentary information for the log file *
      ofs << "\nSystem Information:" << endl ;
      gsOut.compose( "printenv SHELL TERM COLORTERM LANG 1>>\"%S\" 2>/dev/null",
                     gsLog.gstr() ) ;
      this->Systemcall ( gsOut.ustr() ) ;
      gsOut.compose( "uname -srvmpio 1>>\"%S\" 2>/dev/null", gsLog.gstr() ) ;
      this->Systemcall ( gsOut.ustr() ) ;
      ofs.close() ;

      ofs.open( gsLog.ustr(), ofstream::out | ofstream::app ) ;
      if ( ofs.is_open() )
      {
         ofs << "\n------------------------------------------------------------"
                "\nPlease describe the reason for this request."
                "\nInclude: a) the operation you were attempting to complete,"
                "\n            or the feature you would like to have added."
                "\n         b) what you believe should happen"
                "\n         c) what actually happened"
                "\n         d) any other information you think would be helpful"
                "\n\n\n"
             << endl ;
         ofs.close() ;              // close the log file
      }

      dp->RefreshWin () ;        // make everything visible

      uiInfo   Info ;            // user interface data returned here
      short    icIndex = ZERO ;  // index of control with input focus
      bool     done = false ;    // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( icIndex == closePB )
                  done = true ;
               else if ( icIndex == savePB )
               {
                  this->fdPtr->GetPath ( gsOut ) ;
                  char cmd[gsALLOCDFLT * 2] ;
                  snprintf ( cmd, (gsALLOCDFLT * 2), 
                             "mv \"%s\" \"%s/%s\" 1>/dev/null 2>/dev/null", 
                             gsLog.ustr(), gsOut.ustr(), tsFName ) ;
                  this->Systemcall ( cmd ) ;
                  tsRequestSaved = true ;
                  gsOut.compose( "Saved to: %s", tsFName ) ;

                  #if 0    // SCREENSHOT FOR DOCS
                  dp->CaptureDialog ( "capturedlg.txt" ) ;
                  dp->CaptureDialog ( "capturedlg.html", true, false, 
                                      "infodoc-styles.css", 4, false, nc.blR ) ;
                  #endif   // SCREENSHOT

                  dp->WriteString ( ic[savePB].ulY - 1, ic[savePB].ulX,
                                    gsOut, hColor, true ) ;
               }
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }        // while(!done)

      this->fdPtr->DeleteFile ( gsLog ) ;    // delete temp log (if any)
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

   return tsRequestSaved ;

}  //* End DisplaySupportInfo() *

//*************************
//*       DiagMsg         *
//*************************
//******************************************************************************
//* Display start-up diagnostic message in console window.                     *
//* DO NOT call this method after dialog window has opened.                    *
//*                                                                            *
//* Input  : msg    : message to be displayed                                  *
//*          color  : color attribute for message                              *
//*          newLine: (optional, 'true' by default)                            *
//*                   if 'true'  move cursor to next message position          *
//*                   if 'false' leave cursor at end of message                *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::DiagMsg ( const char* msg, attr_t color, bool newLine )
{
   static short xCol = ZERO, xColBaseY = 11 ;

   this->suPos = nc.WriteString ( this->suPos.ypos, this->suPos.xpos, msg, color ) ;
   if ( newLine )
   {
      this->suPos = { short(this->suPos.ypos + 1), xCol  } ;
      if ( xCol == ZERO && (this->suPos.ypos >= this->termRows) )
      {
         xCol = this->termCols / 2 + 1 ;
         this->suPos = { short(xColBaseY + 1), xCol } ;
      }
   }

}  //* End DiagMsg() *

//*************************
//*      FindAppPath      *
//*************************
//******************************************************************************
//* Get the path to the application's executable file and store the string     *
//* in the commArgs class object.                                              *
//*                                                                            *
//* There are two ways to do this:                                             *
//* 1) The path/filename of the executable is passed to the application as     *
//*    argv[0].                                                                *
//* 2) The environment variable "_" (underscore) holds the most recently       *
//*    executed command (but see note below).                                  *
//*                                                                            *
//* The application path is used to locate the default configuration file      *
//* used during startup as well as the application Help files.                 *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: The path string that is received is not necessarily     *
//* the one that was executed.                                                 *
//* 1) If an environment variable such as $HOME is used in the invocation,     *
//*    that value is replaced by the VALUE of $HOME before it is passed to     *
//*    the application AND before it is stored in the "_" environment          *
//*    variable.                                                               *
//*    Example:  $HOME/Apps/FileMangler/FileMangler                            *
//*              is translated to:                                             *
//*              /home/sam/Apps/FileMangler/FileMangler                        *
//*                                                                            *
//* 2) Relative paths ARE NOT translated before being passed to the            *
//*    application and being stored at "_".                                    *
//*                                                                            *
//*    While we expect that the application will be invoked using the          *
//*    'fmg' shell script which contains an absolute path, rather than a       *
//*    relative path, we cannot rely too heavily upon this. Therefore, we      *
//*    need to convert the path we receive into an absolute path.              *
//*                                                                            *
//* 3) We are not entirely sure what would happen if this application were     *
//*    invoked from within another program using the 'exec' family of          *
//*    functions, but we hope that argv[0] would still give us something we    *
//*    can work with. (In this case, the "_" would probably have the calling   *
//*    application's path/filename, so we can't use that.                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void FileMangler::FindAppPath ( commArgs& ca )
{
   //* Save the actual, absolute path specification, replacing        *
   //* symbolic links and relative path specifications.               *
   if ( (realpath ( ca.argList[0], ca.appPath )) == NULL )
   {
      //* Unable to find the "real" path - use what we have.        *
      //* This is an unlikely scenario unless it is an invalid path,*
      //* in which case the application would never have run. DUH!  *                    
      strncpy ( ca.appPath, ca.argList[0], MAX_PATH ) ;
   }
   //* Strip the application name from the end of the string *
   short i ;
   for ( i = ZERO ; ca.appPath[i] != NULLCHAR ; i++ ) ;
   while ( (ca.appPath[i] != fSLASH) && (i > ZERO) )  --i ;
   ca.appPath[i] = NULLCHAR ;

}  //* End FindAppPath() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//* Valid Arguments: see DisplayHelp() method.                                 *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'false' if normal startup                                         *
//*          'true'  if request for help or invalid command-line argument      *
//******************************************************************************
//* Programmer's Note: Even though it is not documented, we accept both        *
//* lowercase and uppercase option and sub-option characters where possible.   *
//* This allows us to add new, case-sensitive options at a later date.         *
//******************************************************************************

bool FileMangler::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA (0)

   //* Get the application executable's directory path. *
   //* Path string will be stored in ca.appPath[]       *
   this->FindAppPath ( ca ) ;

   //* If user provided command-line arguments *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
      const char* argPtr ;       // pointer to option argument
      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            #if DEBUG_GCA != 0  // debugging only
            wcout << L"TOKEN:" << i << L" '" << &ca.argList[i][j] << L"'" << endl ;
            #endif   // DEBUG_GCA

            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //* Request for application version number *
               if ( (gs.compare( L"--version" )) == ZERO )
                  ca.helpFlag = ca.verFlag = true ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  ca.helpFlag = true ;

               else  // invalid argument
               { ca.helpFlag = true ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            char argLetter = ca.argList[i][j] ;

            //* Window Mode *
            if ( argLetter == 'w' || argLetter == 'W' )
            {
               char subchar = tolower ( ca.argList[i][++j] ) ;
               if ( subchar == 'd' )      ca.winMode = wmDUAL_WIN ;
               else if ( subchar == 's' ) ca.winMode = wmSINGLE_WIN ;
               else if ( subchar == 't' ) ca.winMode = wmTERM_WIN ;
               else
               { ca.helpFlag = true ; break ; }  // invalid argument
               ca.modeFlag = true ;
            }

            //* Sort Option *
            else if ( argLetter == 's' || argLetter == 'S' )
            {
               char subchar = ca.argList[i][++j] ;
               if ( subchar == 'D' )         // sort by modification date descending
                  ca.sortOption = fmDATEr_SORT ;
               else if ( subchar == 'd' )    // sort by modification date ascending
                  ca.sortOption = fmDATE_SORT ;
               else if ( subchar == 'S' )    // sort by file size descending
                  ca.sortOption = fmSIZEr_SORT ;
               else if ( subchar == 's' )    // sort by file size ascending
                  ca.sortOption = fmSIZE_SORT ;
               else if ( subchar == 'E' )    // sort by filename extension descending
                  ca.sortOption = fmEXTr_SORT ;
               else if ( subchar == 'e' )    // sort by filename extension ascending
                  ca.sortOption = fmEXT_SORT ;
               else if ( subchar == 'N' )    // sort by name descending
                  ca.sortOption = fmNAMEr_SORT ;
               else if ( subchar == 'T' )    // sort by file type descending
                  ca.sortOption = fmTYPEr_SORT ;
               else if ( subchar == 't' )    // sort by file type ascending
                  ca.sortOption = fmTYPE_SORT ;
               else if ( subchar == 'u' )    // unsorted data
                  ca.sortOption = fmNO_SORT ;
               else                          // sort by name ascending (default)
               {
                  ca.sortOption = fmNAME_SORT ;
                  if ( subchar != 'n' )
                  { ca.helpFlag = true ; break ; }   // invalid argument
               }
               ca.sortFlag = true ;
            }

            //* Alternate Locale Specification *
            else if ( argLetter == 'a' || argLetter == 'A' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               
               //* Capture the path and expand any environment variables. *
               //* Caller must verify that specified locale name is valid.*
               gs = argPtr ;
               gs.copy( ca.localeName, MAX_FNAME ) ;
               ca.altLocale = true ;
               continue ;     // finished with this argument
            }

            //* Directory in which to start application *
            else if ( argLetter == 'd' || argLetter == 'D' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument

               //* Capture the path and expand any environment variables. *
               //* Caller must verify that target directory is accessible.*
               this->gcaRealpath ( gs, argPtr ) ;
               gs.copy( ca.startDir, MAX_PATH ) ;
               continue ;     // finished with this argument
            }

            //* Alternate Configuration File *
            else if ( argLetter == 'f' || argLetter == 'F' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               
               //* Capture the path and expand any environment variables. *
               //* Caller must verify that target file is accessible.     *
               this->gcaRealpath ( gs, argPtr ) ;
               gs.copy( ca.cfgPath, MAX_PATH ) ;
               continue ;     // finished with this argument
            }

            //* Display invisible (hidden) files *
            else if ( argLetter == 'i' || argLetter == 'I' )
            {
               ca.hideFlag = true ;
            }

            //* Open Application in 'Tree' View *
            else if ( argLetter == 't' || argLetter == 'T' )
            {
               ca.treeFlag = true ;
            }

            //* Specify a Backup or Synch definition file.*
            else if ( argLetter == 'b' || argLetter == 'B' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               
               //* Capture the path and expand any environment variables  *
               //* and resolve real path.                                 *
               //* Caller must verify that target file is accessible.     *
               this->gcaRealpath ( gs, argPtr ) ;
               gs.copy( ca.backPath, MAX_PATH ) ;
               continue ;     // finished with this argument
            }

            //* Enable Mouse support. *
            else if ( argLetter == 'm' || argLetter == 'M' )
            {
               ca.mouseFlag = true ;
            }

            //* Enable full root-directory scan. *
            else if ( argLetter == 'r' || argLetter == 'R' )
            {
               ca.rootFlag = true ;
            }

            //* Color Scheme (window border color) *
            else if ( argLetter == 'c' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               
               gs = argPtr ;
               if ( (gs.compare( "bla", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcBK ;        // black
               else if ( (gs.compare( "red", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcRE ;        // red
               else if ( (gs.compare( "gree", false, 4 )) == ZERO )
                  ca.colorScheme = ncbcGR ;        // green
               else if ( (gs.compare( "bro", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcBR ;        // brown
               else if ( (gs.compare( "blu", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcBL ;        // blue
               else if ( (gs.compare( "mag", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcMA ;        // magenta
               else if ( (gs.compare( "cya", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcCY ;        //cyan
               else if ( (gs.compare( "grey", false, 4 )) == ZERO )
                  ca.colorScheme = ncbcGY ;        // grey
               else if ( (gs.compare( "ter", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcDEFAULT ;   // terminal default
               else if ( (gs.compare( "def", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcCOLORS ;    // application default
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               ca.schemeFlag = true ;
               continue ;     // finished with this argument
            }

            //* Pause after start-up sequence so user can see diagnostics *
            else if ( argLetter == 'p' || argLetter == 'P' )
            {
               char subchar = tolower ( ca.argList[i][j + 1] ) ;
               if ( subchar == 'v' )
               {
                  ca.diagPause = 2 ;
                  ++j ;
               }
               else
                  ca.diagPause = 1 ;
            }

            //* Configure FileMangler *
            else if ( argLetter == 'C' )
            {
               ca.configFlag = true ;
            }

            //* Else, is either 'h', 'H' or an invalid argument.*
            //* Either way, invoke help.                        *
            else
            {
               ca.helpFlag = true ;
               if ( argLetter != 'h' && argLetter != 'H' )
                  break ;
            }

            //* If separate tokens have been concatenated *
            if ( ca.argList[i][j + 1] != nckNULLCHAR )
               multiarg = true ;
         }
         else  // invalid argument, token does not begin with a DASH character
         { ca.helpFlag = true ; break ; }

      }     // for(;;)

      //* 'Help' and 'Version' requests, override everything else *
      if ( ca.helpFlag || ca.verFlag )
         ca.configFlag = false ;

      #if DEBUG_GCA != 0  // debugging only
      wcout << L"commArgs Settings" << endl ;
      gs.compose( L"%hd", &ca.argCount ) ;
      wcout << L"argCount   : " << gs.gstr() << endl ;
      for ( short i = ZERO ; i < ca.argCount ; i++ )
      {
         gs.compose( L"argList[%hd] : '%s'", &i, ca.argList[i] ) ;
         wcout << gs.gstr() << endl ;
      }
      gs.compose( L"appPath    : '%s'", ca.appPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"winMode    : %hd", &ca.winMode ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"sortOption : %hd", &ca.sortOption ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"cfgPath    : '%s'", ca.cfgPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"startDir   : '%s'", ca.startDir ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"backPath   : '%s'", ca.backPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"colorScheme: %hd", &ca.colorScheme ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"altLocale  : %hhd '%s'", &ca.altLocale, ca.localeName ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"diagPause  : %hd", &ca.diagPause ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"hideFlag   : %hhd", &ca.hideFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"treeFlag   : %hhd", &ca.treeFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"mouseFlag  : %hhd", &ca.mouseFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"rootFlag   : %hhd", &ca.rootFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"configFlag : %hhd", &ca.configFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"verFlag    : %hhd", &ca.verFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"helpFlag   : %hhd", &ca.helpFlag ) ;
      wcout << gs.gstr() << endl ;
      wcout << L"\n Press Enter" << endl ;
      getchar () ;
      #endif   // DEBUG_GCA
   }
   return ca.helpFlag ;

   #undef DEBUG_GCA
}  //* End GetCommandLineArgs() *

//*************************
//*    gcaCaptureName     *
//*************************
//******************************************************************************
//* Capture a filename or path/filename specified on command line.             *
//*                                                                            *
//* Input  : trg     : target buffer for string                                *
//*          src     : source string                                           *
//*          maxBytes: size of target buffer                                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: If user specifies a path or filename that contains      *
//* spaces, then on the command line, he/she/it must enclose the string with   *
//* quotations marks (either single or double). Otherwise part of the string   *
//* will be interpreted as a separate, probably invalid argument.              *
//*                                                                            *
//* See notes on the 'wordexp' function in gcaRealpath() method.               *
//*                                                                            *
//******************************************************************************

#if 0    // MAY BE OBSOLETE, CURRENTLY UNUSED
void FileMangler::gcaCaptureName ( char* trg, const char* src, short maxBytes )
{
   gString gsPath( src ) ;    // copy source (in case library call fails)

   //* Perform environment-variable expansion or tilde ('~') expansion.*
   wordexp_t wexp ;              // target structure
   if ( (wordexp ( gsPath.ustr(), &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         gsPath.clear() ;
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            gsPath.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc )   // re-insert stripped spaces (see note)
               gsPath.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;
   }
   gsPath.copy( trg, maxBytes ) ;

}  //* End gcaCaptureName() *
#endif   // MAY BE OBSOLETE, CURRENTLY UNUSED

//*************************
//*      gcaRealpath      *
//*************************
//******************************************************************************
//* Normalize a file specification string. Used primarily to parse             *
//* command-line arguments, but has broader use.                               *
//*                                                                            *
//* Given a filename, relative path/filename or full path/filename, with or    *
//* without environmental variables or symbolic links, resolve to an absolute  *
//* path/filename string.                                                      *
//*                                                                            *
//* Programmer's Note: If user specifies a path or filename that contains      *
//* spaces, then he/she/it must enclose the string with quotations marks       *
//* (either single or double). Otherwise part of the string will be            *
//* interpreted as a separate, probably invalid argument.                      *
//*                                                                            *
//* Input  : rpath   : on entry, contains the source data, and                 *
//*                    on return contains full filespec string                 *
//*          srcdata : (optional, NULL pointer by default)                     *
//*                    if specified, contains the source data                  *
//*                    if NOT specified, then 'rpath' contains the source data *
//*                                                                            *
//* Returns: 'true'  if successful, rpath contains converted filespec          *
//*          'false' if parsing error, rpath is unchanged                      *
//******************************************************************************
//* NOTES:                                                                     *
//* ------                                                                     *
//* 'realpath' will follow symbolic links; however, it WILL NOT do tilde ('~') *
//*            or environment-variable expansion, so do these substitutions    *
//*            before calling the 'realpath' library function.                 *
//*                                                                            *
//* 'wordexp'                                                                  *
//* The 'wordexp' function is a pretty cool, but watch out:                    *
//*  a) wordexp returns ZERO on success or WRDE_BADCHAR (2) if an invalid      *
//*     character is detected in the stream.                                   *
//*     - Note that an empty string will pass the scan, but then               *
//*       'wexp.we_wordc' will be ZERO.                                        *
//*  b) Dynamic memory allocation happens, so remember to free it.             *
//*     - If a bad character in the stream, then freeing the dynamic           *
//*       allocation will cause a segmentation fault. This is a Standard       *
//*       Library bug, so the work-around is to call 'wordfree' ONLY if        *
//*      'wordexp' returns success.                                            *
//*  c) SPACE characters delimit the parsing, so if the path contains spaces,  *
//*     then we must concatenate the resulting substrings, reinserting the     *
//*     space characters. Leading and trailing spaces are ignored.             *
//*     (We assume that a path will never contain a TAB or NEWLINE character.) *
//*  d) wordexp will choke on the following characters in the stream:          *
//*             & | ; < >  \n     (unless they are quoted)                     *
//*     - Parentheses and braces should appear ONLY as part of a token to be   *
//*       expanded (or if they are quoted).                                    *
//*     - single-quotes ''' are seen as delimiters unless they are quoted.     *
//*     - double-quotes '"' are also seen as delimiters unless they are        *
//*       quoted; however, the use of double-quotes is further complicated by  *
//*       the fact that string literals are always delimited by double-quotes, *
//*       so a double-quote within a set of double-quotes must be SENT to us   *
//*       as quoted in order for it to arrive as the double-quote character,   *
//*       which we will see as an un-quoted character.                         *
//*  e) Wildcard characters '*' and '?' are tricky, but '?' especially is a    *
//*     common filename character.                                             *
//*  f) The tokens we are most likely to see are '${HOME}' and '~'.            *
//*     These are both expanded as '/home/sam' or the equivalent.              *
//*  g) The most likely special characters (especially for song titles, book   *
//*     titles and similar) that represent themselves are:                     *
//*         single-quote '''  double-quote '"'  question mark '?'              *
//*         ampersand '&' and parentheses '(' ')'                              *
//*  h) Note that the characters: '$'  '{'  '}'  '`'  are reserved by the      *
//*     shell for expansion of environment variables, and therefore should     *
//*     not be used as filename characters; therefore, we do not quote them    *
//*     here.                                                                  *
//******************************************************************************

bool FileMangler::gcaRealpath ( gString& rpath, const char* srcdata )
{
   gString gstmp = (srcdata != NULL) ? srcdata : rpath, // working copy of source
           relpath ;
   int  wchars, index = ZERO ;
   const wchar_t* wptr = gstmp.gstr( wchars ) ;
   bool status = true ;          // return value

   //* Quote (escape) the set of special characters which ARE NOT *
   //* used for environment expansion so wordexp() will process   *
   //* them without special meaning.                              *
   while ( index < wchars )
   {
      if ( (wptr[index] == L'(') || (wptr[index] == L')')  ||
           (wptr[index] == L'"') || (wptr[index] == L'\'') ||
           (wptr[index] == L'?') || (wptr[index] == L'&')  ||
           (wptr[index] == L'<') || (wptr[index] == L'>')  ||
           (wptr[index] == L';') || (wptr[index] == L':')  ||
           (wptr[index] == L'!') || (wptr[index] == L'%')  ||
           (wptr[index] == L'=') || (wptr[index] == L'@')  ||
           (wptr[index] == L'#') || (wptr[index] == L'+')  ||
           (wptr[index] == L'[') || (wptr[index] == L']')  ||
           (wptr[index] == L'*') //|| (wptr[index] == L'\\')
         )
      {
         if ( (index == ZERO) || (wptr[index - 1] != L'\\') )
         {
            gstmp.insert( L'\\', index ) ;
            ++wchars ;
            index += 2 ;
         }
         else
            ++index ;
      }
      else
         ++index ;
   }

   //* Perform environment-variable and tilde substitutions. (see notes above) *
   wordexp_t wexp ;              // target structure
   if ( (wordexp ( gstmp.ustr(), &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         relpath.clear() ;
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            relpath.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc )      // re-insert stripped spaces (see note)
               relpath.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;
   }
   else
      status = false ;

   //* Expand relative paths and paths containing symbolic links.*
   if ( status != false )
   {
      char rtmp[MAX_PATH] ;      // temp buffer

      if ( (realpath ( relpath.ustr(), rtmp )) != NULL )
         rpath = rtmp ;
      else
         status = false ;
   }

   return status ;

}  //* End gcaRealpath() *

//*************************
//*      Systemcall       *
//*************************
//******************************************************************************
//* Call the 'system' C-language library function with the specified command.  *
//*                                                                            *
//*                                                                            *
//* Input : cmd    : command string with program name and options              *
//*                                                                            *
//* Returns: value returned by the system call (probably meaningless)          *
//******************************************************************************

short FileMangler::Systemcall ( const char* cmd )
{
   short exitCode = ZERO ;

   exitCode = system ( cmd ) ;

   return exitCode ;

}  //* End Systemcall() *

//*************************
//*     DrawTitle         *
//*************************
//******************************************************************************
//* Draw the application title in the main terminal window.                    *
//* This is a temporary title for display of startup diagnostics, or in case   *
//* NcDialog interface cannot open.                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::DrawTitle ( void )
{
   //* Construct and display the main application title line *
   gString  gs( "%s%s (c)%s                       ", 
                AppTitle1, AppVersion, copyrightYears ) ;
   nc.WriteString ( ZERO, ZERO, gs.gstr(), (nc.bw | ncuATTR) ) ;

}  //* End DrawTitle()

//*************************
//*       Restart         *
//*************************
//******************************************************************************
//* User has just run the configuration utility from inside the application.   *
//* Any changes made will take effect only on application restart.             *
//* Ask user if he/she/it wants to restart FileMangler.                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if application re-start                                   *
//*          'false' if continue main program loop                             *
//******************************************************************************

bool FileMangler::Restart ( void )
{
const char* DialogText = 
   "Any changes made during configuration will take effect on restart.\n"
   "\n"
   "                    Re-start FileMangler now?" ;
const short dialogROWS = 9 ;        // # of display lines
const short dialogCOLS = 70 ;       // # of display columns
const short ulY = 2, ulX = ZERO ;   // dialog position
attr_t dColor = this->cfgOpt.cScheme.sd ;
bool restart = false ;              // return value
enum dpCtrls : short { yesPB, noPB, dpcCOUNT } ;
InitCtrl ic[dpcCOUNT] = 
{
   {  //* 'YES' pushbutton    - - - - - - - - - - - - - - - - - - - -    yesPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 8),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      "  YES  ",                    // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[noPB],                    // nextCtrl:  link in next structure
   },
   {  //* 'YES' pushbutton    - - - - - - - - - - - - - - - - - - - -    yesPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[yesPB].ulY,                // ulY:       upper left corner in Y
      short(ic[yesPB].ulX + 9),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      "  NO  ",                     // dispText:  
      this->cfgOpt.cScheme.pn,      // nColor:    non-focus color
      this->cfgOpt.cScheme.pf,      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set dialog title *
      dp->SetDialogTitle ( "  RESTART  ", this->cfgOpt.cScheme.em ) ;
      //* Print dialog window's static text *
      dp->WriteParagraph ( 2, 2, DialogText, dColor ) ;
      dp->RefreshWin () ;                    // make the text visible

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == yesPB )
               restart = true ;
            done = true ;
         }
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }     // while()
   }        // OpenWindow()
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   return restart ;

}  //* End Restart() *

//*************************
//*    DisplayVersion     *
//*************************
//******************************************************************************
//* Print application version number and copyright notice to tty (bypassing    *
//* NCurses output).                                                           *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the version/copyright text.                         *
//*                                                                            *
//* NOTE: This method relies on the application title string being of a certain*
//*       format. If the string changes, the header output may break.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::DisplayVersion ( void )
{
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut( &AppTitle1[2] ) ;
   gsOut.limitChars( 12 ) ;
   gsOut.insert( L'\n' ) ;
   gsOut.append( "(fmg) v:%s Copyright(c) %s  The Software Samurai\n", 
                 AppVersion, copyrightYears ) ;
   short tCols = gsOut.gscols() ;
   for ( short i = ZERO ; i < tCols ; ++i )
      gsOut.append ( L'=' ) ;
   wcout << gsOut.gstr() << L'\n' << freeSoftware << endl ;

}  //* End DialpayVersion() *

//*************************
//*    DisplayHelp        *
//*************************
//******************************************************************************
//* Print command-line help to tty (bypassing NCurses output).                 *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the help text.                                      *
//*                                                                            *
//* NOTE: This method relies on the application title string being of a certain*
//*       format. If the string changes, the header output may break.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileMangler::DisplayHelp ( void )
{
static const char Help[] = 
    //12345678901234567890123456789012345678901234567890123456789012345678901234567890
   "\nUSAGE: fmg [OPTIONS]   Example: fmg -ws -sT"
   "\n       Options may be specified in any order. If an option is not specified,"
   "\n       then its setting will be taken from configuration file."
   "\n -w[s|d|t] Window display mode.  Example: -ws"
   "\n      (s)ingle-window, (d)ual-window, (t)erminal-size"
   "\n -s[n|N|d|D|s|S|e|E|t|T|u]  Sort option for directory display.   Example: -sd"
   "\n      (name,date,size,extension,type,unsorted - lowercase=ascending)"
   "\n -a=LOCALE   Alternate 'locale' (character encoding for the language/region)"
   "\n      Example: -a=wo_SN.utf8   (default locale taken from terminal environment)"
   "\n -f=FILENAME   File from which to read configuration data on startup"
   "\n      Example: -f=custom.cfg   (default: FileMangler.cfg in application dir)"
   "\n -d=DIRNAME   Directory in which to begin (default = current working directory)"
   "\n      Examples: -d=$HOME  -d=~/Downloads  -d=/usr/include  -d=../Docs"
   "\n -b=FILENAME  File from which to read a Backup or Synch definition."
   "\n      Example: -b=~/Apps/FileMangler/Backup_Def.txt"
   "\n -i   Display hidden (invisible) files."
   "\n -t   Start application in file-system 'Tree' view.  Example: -t"
   "\n -m   Enable mouse support.      Example: -m"
   "\n -r   Enable full-depth scan of directory tree from root directory."
   "\n -c=SCHEME  [black|red|green|brown|blue|magenta|cyan|grey|term|default]"
   "\n      Color scheme for application.  Example: -c=blue"
   "\n -C   Configure FileMangler and store data in FileMangler.cfg"
   "\n      (If -f option specifies, update an alternate configuration file.)"
   "\n -p   Pause to display startup diagnostics before entering main program."
   "\n --version        Display application version number, then exit."
   "\n --help, -h, -?   Help for command line options"
   "\n" ;

   gString gsOut( &AppTitle1[2] ) ;
   gsOut.limitChars( 12 ) ;
   gsOut.insert( L'\n' ) ;
   gsOut.append( "(fmg) v:%s Copyright(c) %s  The Software Samurai\n", 
                 AppVersion, copyrightYears ) ;
   short tCols = gsOut.gscols() ;
   for ( short i = ZERO ; i < tCols ; ++i )
      gsOut.append ( L'=' ) ;
   wcout << gsOut.gstr() << Help << endl ;
   
}  //* End DisplayHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

