//********************************************************************************
//* File       : FileDlgUtil2.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 19-May-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* 1) This module of the FileDlg class contains the low-level code for          *
//*    managing OpenDocument files (.odt, .ods, .odt, etc.) and Office Open      *
//*    XML files (.docx, .xlsx., .pptx, etc.). These methods support viewing     *
//*    the text contents of these documents as well as performing 'grep'         *
//*    operations on that text. To do this:                                      *
//*    a) Extract individual XML source files from the document (.zip) archive.  *
//*    b) Extract and format the text data from these XML source files.          *
//*                                                                              *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************
//*****************************************
//** Data for extracting data from       **
//** OpenDocument and Open XML documents **
//*****************************************
//* Command to extract target file from OD/OX documents.*
//* 'unzip' parameters:                                 *
//*      "-C" == case insensitive scan                  *
//*      "-o" == overwrite existing targets             *
//*      "-j" == junk (discard) subdirectory paths      *
static const char* uzTemplate = "unzip -Coj \"%S\" %s -d%S 1>/dev/null 2>/dev/null" ;
static const char* content    = "content.xml" ;        // filename of atODOC content
static const char* document   = "word/document.xml" ;  // filename of atOXML text content
static const char* sharedText = "xl/sharedStrings.xml" ; // filename of atOXML spreadsheets containing text data
static const char* sheet      = "xl/worksheets/sheet%d.xml" ; // individual worksheet filenames for atOXML spreadsheets
static const char* slide      = "ppt/slides/slide%d.xml" ;    // individual slide filenames for atOXML presentations
static const char* corruptTemplate = "Error! Office document \"%S\" appears to be corrupted." ;
//* The odpox_ScanSheetHdr() method is capable of capturing *
//* the dimensions of the worksheet; however these values   *
//* are not currently used, so the capture is disabled.     *
#define RETURN_SHEET_DIMENSIONS (0)



//********************
//* Local prototypes *
//********************
static void odParseOD_Text ( const gString& srcSpec, std::ifstream& ifs, 
                             std::ofstream& ofs, short tCols ) ;
static void odParseOX_docx ( const gString& srcSpec, std::ifstream& ifs, 
                             std::ofstream& ofs, short tCols ) ;
static bool odpox_ScanSheetHdr ( std::ifstream& ifs, 
                                 #if RETURN_SHEET_DIMENSIONS != 0
                                 int* sheetCols, int* sheetRows, 
                                 #endif   // RETURN_SHEET_DIMENSIONS
                                 std::ofstream& ofs ) ;
static bool odpox_TextRecord ( std::ifstream& ifs, gString& gsText, std::ofstream& ofs, 
                               int index = ZERO, bool init = false ) ;
static bool odpox_CaptureRow ( std::ifstream& ifs_sheet, std::ifstream& ifs_text,
                               std::ofstream& ofs, short tCols ) ;
static bool xmlGoodMIME ( std::ifstream& ifs ) ;
static bool xmlGetCmd ( std::ifstream& ifs, char lineBuff[gsDFLTBYTES] ) ;
static char xmlCharEntity ( std::ifstream& ifs, gString& ebuff ) ;



//********************************************************************************
//***                 OpenDoc and OpenXML Management Group                     ***
//********************************************************************************

//*************************
//*    odIsOpenDocFile    *
//*************************
//********************************************************************************
//* Test the filename extension to see if it is one of the supported             *
//* OpenOffice filename extensions.                                              *
//* Note that caller has verified that target is a 'Regular' file.               *
//*                                                                              *
//* Input  : fileSpec  : path/filename of file to be validated                   *
//*          all       : (optional, 'false' by default)                          *
//*                      if 'false', report success only for the                 *
//*                                  text-extraction subset of document types    *
//*                      if 'true',  report success for any known OD type        *
//*                                  (caller wants to know if target is a)       *
//*                                  (zip archive file                   )       *
//*                                                                              *
//* Returns: 'true;  if source file is a supported OpenOffice document           *
//*          'false' if source file IS NOT a supported OpenOffice document       *
//********************************************************************************
//* Open Document Filenames:                                                     *
//* ========================                                                     *
//* .odt        Text                  Only a subset of these filetypes contain   *
//* .ods        Spreadsheet           meaningful text. If 'all' == false, then   *
//* .odp        Presentation          report success only for this subset.       *
//* .sxw        Text (StarOffice)                                                *
//* .sxc        Spreadsheet (StarOffice)                                         *
//* .sxi        Presentationn (StarOffice)                                       *
//*                                                                              *
//* .ott        Text (template)                                                  *
//* .ots        Spreadsheet (template)                                           *
//* .otp        Presentation (template)                                          *
//* .odg        Drawing                                                          *
//* .odc        Chart                                                            *
//* .odf        Formula                                                          *
//* .odi        Image                                                            *
//* .odm        Master document                                                  *
//* .odb        Database                                                         *
//*                                                                              *
//* Template name formats substitute 't' for 'd'  Example: .ots                  *
//* However templates, by definition, have no content.                           *
//*                                                                              *
//* Programmer's Note:                                                           *
//* At this time, we do not know if the older StarOffice(6) (OpenOffice 1.x)     *
//* format is fully 'zip' compatible with modern OpenDocument files.             *
//* StarOffice(6) was the first version based on the XML format, so it           *
//* _should_ be compatible, although it may contain fewer files and              *
//* subdirectories than the newer OD specifications. Unfortunately, we have      *
//* converted most of our StarOffice files, and all we have left are a few       *
//* documents left over from our college days, so only the following have        *
//* been verified to have a compatible format.                                   *
//*                                                                              *
//*   '.sxw' (text)               - verified                                     *
//*   '.sxc' (spreadsheet)        - verified                                     *
//*   '.sxi' (.sxp, presentation) - verified                                     *
//*   '.sxd' (drawing)            -                                              *
//*   '.sxm' (math)               -                                              *
//*   '.sxg' (master document)    -                                              *
//*                                                                              *
//* Single-file OD documents are not supported.                                  *
//********************************************************************************

bool FileDlg::odIsOpenDocFile ( const gString& fileSpec, bool all )
{
   const short odeSUBCOUNT = 5 ;    // index indicating end of sub-count
   const short odeCOUNT = 15 ;      // item count
   const char* odExtensions[odeCOUNT] = 
   {
      ".odt",     // this group always tested
      ".ods",
      ".odp",
      ".sxw",
      ".sxc",
      ".sxi",

      ".ott",     // this group tested only if 'all' != false
      ".ots",
      ".otp",
      ".odg",
      ".odc",
      ".odf",
      ".odi",
      ".odm",
      ".odb",
   } ;
   bool status = false ;      // return value

   short indx = fileSpec.findlast( L'.' ) ;
   for ( short i = ZERO ; i < odeCOUNT ; ++i )
   {
      if ( (fileSpec.compare( odExtensions[i], false, gsALLOCDFLT, indx )) == ZERO )
      {
         status = true ;
         break ;
      }
      if ( ! all && (i >= odeSUBCOUNT) )
         break ;
   }
   return status ;

}  //* End odIsOpenDocFile() *

//*************************
//*    odIsOpenDocFile    *
//*************************
//********************************************************************************
//* Test the filename extension to see if it is a BACKUP file for one of the     *
//* supported OpenOffice filename extensions. Only current OpenDocument formats  *
//* are tested, not the older .sxw/.sxc/sxi formats.                             *
//*                                                                              *
//* This test is used by the ViewFileContents() group to determine whether the   *
//* file can be viewed as _both_ an archive file _and_ as text data.             *
//*                                                                              *
//* Input  : fileSpec  : path/filename of file to be validated                   *
//*                                                                              *
//* Returns: 'true;  if source file is a supported OpenOffice document backup    *
//*          else 'false'                                                        *
//********************************************************************************
bool FileDlg::odIsOpenDocBackup ( const gString& fileSpec )
{
   const short odeCOUNT = 3 ;       // item count
   const char* odExtensions[odeCOUNT] = 
   {
      ".odt~",
      ".ods~",
      ".odp~",
   } ;
   bool status = false ;      // return value

   short indx = fileSpec.findlast( L'.' ) ;
   for ( short i = ZERO ; i < odeCOUNT ; ++i )
   {
      if ( (fileSpec.compare( odExtensions[i], false, gsALLOCDFLT, indx )) == ZERO )
      {
         status = true ;
         break ;
      }
   }

   return status ;

}  //* End odIsOpenDocBackup() *

//*************************
//*    odIsOpenXmlFile    *
//*************************
//******************************************************************************
//* Test the filename extension to see if it is one of the supported           *
//* OpenXML ISO/IEC 29500 (OOXML - Microsloth) filename extensions.            *
//* Note that caller has verified that target is a 'Regular' file.             *
//*                                                                            *
//* Input  : fileSpec  : path/filename of file to be validated                 *
//*          all       : (optional, 'false' by default)                        *
//*                      if 'false', report success only for the               *
//*                                  text-extraction subset of document types  *
//*                      if 'true',  report success for any known OOXML type   *
//*                                                                            *
//* Returns: 'true;  if source file is a supported OpenXML document            *
//*          'false' if source file IS NOT a supported OpenXML document        *
//******************************************************************************
//* Open XML Filenames:                                                        *
//* ===================                                                        *
//* .docx       Text                          Only a subset of these filetypes *
//* .docm       Text (with macros)            contain meaningful text.         *
//* .xlsx       Spreadsheet                   If 'all' == false, then report   *
//* .xlsm       Spreadsheet (with macros)     success only for this subset.    *
//* .pptx       Presentation                                                   *
//* .pptm       Presentation (with macros)                                     *
//* .ppsx       Slideshow                                                      *
//* .ppsm       Slideshow (with macros)                                        *
//*                                                                            *
//* Template name formats substitute 't' for 'c'  Example: .docx ==> .dotx     *
//* However templates, by definition, have no meaningful text content.         *
//* .dotx                                                                      *
//* .dotm                                                                      *
//* .xltx                                                                      *
//* .xltm                                                                      *
//* .potx                                                                      *
//* .potm                                                                      *
//* .potx                                                                      *
//* .potm                                                                      *
//*                                                                            *
//* A "slide" document, is included for completeness in listing XML zip        *
//* archives, but is unlikely to contain any significant text.                 *
//* .sldx                                                                      *
//* .sldm                                                                      *
//*                                                                            *
//* .xps  is a binary print format for MS-Office documents and is not part of  *
//*       the OOXML implementation.                                            *
//*                                                                            *
//* The following indicate certain binary-format files which are not part of   *
//* the OOXML implementation.                                                  *
//* .docb       Text (binary file)                                             *
//* .xlsb       Spreadsheet (binary file)                                      *
//* .pptb       Presentation (binary file)                                     *
//*                                                                            *
//* MS-Office files created by versions prior to 2003 are not structured       *
//* according to the Open XML format. These older versions: .doc, .xls, .ppt,  *
//* etc. are 16-bit binary files. See xxx for more information.                *
//* Full implementation of the XML/ZIP archive format was not achieved until   *
//* 2007, (there is a reason their name is Microsloth) so use caution.         *
//*                                                                            *
//* https://support.office.com/en-us/article/open-xml-formats-and-file-        *
//*                       name-extensions-5200d93c-3449-4380-8e11-31ef14555b18 *
//******************************************************************************

bool FileDlg::odIsOpenXmlFile ( const gString& fileSpec, bool all )
{
   const short oxeSUBCOUNT = 8 ;
   const short oxeCOUNT = 16 ;
   const char* oxExtensions[oxeCOUNT] = 
   {
      ".docx",    // this group always tested
      ".xlsx",
      ".pptx",
      ".ppsx",
      ".docm",
      ".xlsm",
      ".pptm",
      ".ppsm",

      ".dotx",    // this group tested only if 'all' != false
      ".dotm",
      ".xltx",
      ".xltm",
      ".potx",
      ".potm",
      ".sldx",
      ".sldm",
   } ;
   bool status = false ;      // return value

   short indx = fileSpec.findlast( L'.' ) ;
   for ( short i = ZERO ; i < oxeCOUNT ; ++i )
   {
      if ( (fileSpec.compare( oxExtensions[i], false, gsALLOCDFLT, indx )) == ZERO )
      {
         status = true ;
         break ;
      }
      if ( ! all && (i >= oxeSUBCOUNT) )
         break ;
   }
   return status ;

}  //* End odIsOpenXmlFile() *

//*************************
//*    odIsBinDocFile     *
//*************************
//******************************************************************************
//* Test the filename extension to see if it is one of the supported           *
//* pre-XML Microsloth binary-format document files.                           *
//*                                                                            *
//*                                                                            *
//* Input  : fileSpec  : path/filename of file to be validated                 *
//*          all       : (optional, 'false' by default)                        *
//*                      if 'false', report success only for the               *
//*                                  text-extraction subset of document types  *
//*                      if 'true',  report success for any known BinDoc type  *
//*                                                                            *
//* Returns: 'true;  if source file is a supported BinDoc document             *
//*          'false' if source file IS NOT a supported BinDoc document         *
//******************************************************************************

bool FileDlg::odIsBinDocFile ( const gString& fileSpec, bool all )
{
   const short odocSUBCOUNT = 1 ;
   const short odocCOUNT = 3 ;
   const char* odocExtensions[odocCOUNT] = 
   {
      ".doc",     // this type is always tested
      ".xls",     // this group test only if 'all' != false
      ".ppt",
   } ;

   short indx = fileSpec.findlast( L'.' ) ;  // point to filename extension
   bool  status = false ;                    // return value

   for ( short i = ZERO ; i < odocCOUNT ; ++i )
   {
      if ( (fileSpec.compare( odocExtensions[i], false, gsALLOCDFLT, indx )) == ZERO )
      {
         status = true ;
         break ;
      }
      if ( ! all || (i >= odocSUBCOUNT) )
         break ;
   }
   return status ;

}  //* End odIsBinDocFile() *

//*************************
//*  odExtractOD_Content  *
//*************************
//******************************************************************************
//* Extract the XML sub-document containing the OpenDocument text data, the    *
//* OpenXML (.docx) text data, or the first file needed to decode OpenXML      *
//* spreadsheet documents (.xlsx) or presentation documents (.pptx).           *
//*                                                                            *
//* The file extracted is a plain-text file containing the document's text     *
//* and/or setup data, and is structured as a simple XML source file.          *
//*                                                                            *
//*                                                                            *
//* Input  : srcSpec: (by reference) filespec of source OpenDocument file      *
//*          tmpPath: (by reference) path of application's temp directory      *
//*          conSpec: (by reference) receives filespec of extracted XML file   *
//*                   containing OD text data, etc.                            *
//*          aType  : archive type, either atODOC or atOXML                    *
//*                     (or atNONE for Word binary files)                      *
//*                                                                            *
//* Returns: 'true' if operation is successful, else 'false'                   *
//******************************************************************************
//* OpenXML (MS-Office) files:                                                 *
//* -- Unlike OpenDocument files, MS-Office docs store the text in different   *
//*    files, depending on type of document.                                   *
//*    The file which contains the layout is "[Content_Types].xml". We are     *
//*    not sure how to interpret this file, but it appears to give a MIME type *
//*    for each file in the archive. Unfortunately, these are not standard     *
//*    MIME type strings, but some proprietary indicators which are not        *
//*    particularly helpful since the individual filenames provide the same    *
//*    information.                                                            *
//*    Files Containing Text:                                                  *
//*    ----------------------                                                  *
//*    -- For text documents, "word/document.xml"                              *
//*    -- For spreadsheets, multiple files: "sharedStrings.xml", then          *
//*       "worksheets/shee1.xml", etc.                                         *
//*    -- For presentations, "ppt/slides/slide1.xml" etc.  and possibly more.  *
//*                                                                            *
//*    To do a thorough job, we must concatenate all the text files for        *
//*    display. The formatting and sequencing may not be very good, but output *
//*    is basically legible and searchable.                                    *
//*                                                                            *
//******************************************************************************

bool FileDlg::odExtractOD_Content ( const gString& srcSpec, const gString& tmpPath, 
                                    gString& conSpec, arcType aType )
{
   gString fname,                // filespec of file containing the XML source text data
           escPath = srcSpec ;   // escape any "special characters" in filespec

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   if ( aType == atODOC )           // OpenOffice
      fname = content ;
   else     // aType==atOXML
   {
      gString fext ;
      this->fmPtr->ExtractExtension ( fext, srcSpec ) ;
      if ( (fext.compare( "xlsx" )) == ZERO )
      {  //* For spreadsheets, the first file needed *
         //* is "xl/sharedStrings.xml".              *
         fname = sharedText ;
      }
      else if ( (fext.compare( "pptx" )) == ZERO )
      {  //* For presentations, the first file needed *
         //* is "ppt/slides/slide1.xml".              *
         int slIndex = 1 ;
         fname.compose( slide, &slIndex ) ;
      }
      else
      {  //* For text documents, the only file needed *
         //* is "word/document.xml".                  *
         fname = document ;
      }
   }

   //* Extract the text-content file(s) from the document and *
   //* save it to the application's temp directory.           *
   // Programmer's Note: It is unlikely, but possible that the command string 
   // will be larger that gsDFLTBYTES. If this command starts to fail, 
   // look there first.
   gString gsCmd( uzTemplate, escPath.gstr(), fname.ustr(), tmpPath.gstr() ) ;
   this->fmPtr->Systemcall ( gsCmd.ustr() ) ;

   //* For OXML documents, the data are in a subdirectory within  *
   //* the archive file; however, we extract the data directly to *
   //* tmpPath directory and discard the archive's relative path. *
   if ( aType == atOXML )
   {  //* Strip the relative path. *
      short i = fname.findlast( fSLASH ) + 1 ;
      fname.shiftChars( -(i) ) ;
   }

   //* Construct filespec of extracted file *
   this->fmPtr->CatPathFname ( conSpec, tmpPath.ustr(), fname.ustr() ) ;

   fmFType fType ;   // file type code
   bool    goodTarget = this->TargetExists ( conSpec, fType ) ;

   return goodTarget ;

}  //* End odExtractOD_Content() *

//*************************
//*   odExtractOD_Text    *
//*************************
//******************************************************************************
//* Scan the XML source document, discarding the XML formatting commands and   *
//* copying the actual text data to target file.                               *
//*                                                                            *
//* Input  : srcSpec: (by reference) filespec of source document               *
//*        : conSpec: (by reference) filespec of source file containing the    *
//*                     OpenDocument text data in XML format                   *
//*          capSpec: (by reference) filespec for capture of extracted         *
//*                     plain-text data                                        *
//*          aType  : archive type, either atODOC or atOXML                    *
//*          dbPtr  : (optional, NULL pointer by default -- debug only)        *
//*          dbwp   : (optional, NULL pointer by default -- debug only)        *
//*                                                                            *
//* Returns: 'true' if operation is successful, else 'false'                   *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* Parsing of the XML could be more sophisticated.                            *
//* 1) Consider the width of the terminal window when inserting line breaks.   *
//* 2) For columnar data, keep the row together if possible.                   *
//*    For tables: <table:table-row>(stuff)</table:table-row>                  *
//* 3) Verify that UTF-8 characters will not be broken by linefeeds.           *
//* 4) OpenOffice XML files often contain the sequence 0xC2,0xA0. This is not  *
//*    a UTF-8 displayable character, and may not be UNICODE at all.           *
//* 5) Three(3) controls characters are allowed in an XML document:            *
//*    tab (0x09), carriage-return (0x0D) and linefeed (0x0A)                  *
//*    -- So far as we can tell, linefeed is used only after the MIME-type     *
//*       in the first line of the file.                                       *
//*    -- We have not encountered any tab characters in our test files         *
//*       (LibreOffice 4.2.8.2).                                               *
//*    -- A carriage-return has no place in a Linux/UNIX environment,          *
//*       so if we see it, we filter it out.                                   *
//*    -- Invalid control characters are also filtered out.                    *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool FileDlg::odExtractOD_Text ( const gString& srcSpec, const gString& conSpec, 
                                 const gString& capSpec, arcType aType, 
                                 NcDialog* dbPtr, winPos* dbwp )
{
   bool status = false ;            // return value
   short tRows, tCols ;             // dimensions of terminal window
   this->dPtr->ScreenDimensions ( tRows, tCols ) ;
   tCols -= 13 ;                    // column for line-break test

   ifstream ifs( conSpec.ustr(), ifstream::in ) ;
   ofstream ofs( capSpec.ustr(), (ofstream::out | ofstream::trunc) ) ;

   if ( (ifs.is_open()) && (ofs.is_open()) )
   {
      //* Read, verify and discard the MIME-type line.  *
      //* If test fails, source is not a valid XML file.*
      if ( (xmlGoodMIME ( ifs )) != false )
      {
         status = true ;      // declare success

         if ( aType == atODOC )
         {
            odParseOD_Text ( srcSpec, ifs, ofs, tCols ) ;
         }
         else     // (aType == atOXML)
         {
            //* Index of doc filename extension *
            short ext = srcSpec.findlast( L'.' ) + 1 ;
            //* If "xlsx" or "xlsm" *
            if ( (srcSpec.find( L"xls", false, gsALLOCDFLT, ext )) == ext )
               this->odParseOX_xlsx ( srcSpec, ifs, ofs, tCols ) ;
            //* If "pptx", "pptm", "ppsx", "ppsm" *
            else if ( (srcSpec.find( L"pp", false, gsALLOCDFLT, ext )) == ext )
            {
               ifs.close() ;     // called method re-opens the input stream
               odParseOX_pptx ( srcSpec, ofs, tCols ) ;
            }
            else  // "docx" (and anything we didn't anticipate)
               odParseOX_docx ( srcSpec, ifs, ofs, tCols ) ;
         }
      }

      //* Else, file has an invalid MIME type.*
      else
      {
         short indx = (srcSpec.findlast( fSLASH )) + 1 ;
         gString gsOut( corruptTemplate, &srcSpec.gstr()[indx] ) ;
         ofs << gsOut.ustr() << "\nContains data file(s) with invalid MIME type." << endl ;
      }
   }        // files opened

   //* Close the files.*
   ifs.close() ;
   ofs.close() ;

   return status ;

}  //* End odExtractOD_Text() *

//*************************
//*    odParseOD_Text     *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Parse the XML source data for an OpenDocument file and format it for       *
//* display.                                                                   *
//* MIME-type (first line of file) has been processed by caller.               *
//*                                                                            *
//* Input  : srcSpec  : (by reference) filespec of source OpenDocument file    *
//*                        [currently unused for OpenDocument files]           *
//*          ifs      : XML source data input stream                           *
//*          ofs      : formatted display text output stream                   *
//*          tCols    : number of text-display columns for formatted output    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void odParseOD_Text ( const gString& srcSpec, std::ifstream& ifs, 
                             std::ofstream& ofs, short tCols )
{
   const char *endPara = "</text:p",          // end-of-paragraph
              *tabRowB = "<table:table-row",  // begin table-row
              *tabRowE = "</table:table-row"; // end table-row
   const char lAngle  = '<',
              rAngle  = '>',
              ampChar = '&' ;

   gString gsEntity ;            // convert "predefined entity" characters
   char  lineBuff[gsDFLTBYTES] ; // input-stream line buffer
   short lineBytes = ZERO ;      // count chars (actually bytes) of output text
   bool  tablerow = false,       // signals that scan is within an XML <table-row>
         done = false ;          // loop control

   //* Read and discard all XML commands.   *
   //* Save non-command text to target file.*
   while ( ! done )
   {
      ifs.read( lineBuff, 1 ) ;
      if ( ifs.gcount() == 1 )
      {
         //* If at the beginning of an XML command, read and discard    *
         //* the command (with some exceptions).                        *
         if ( lineBuff[ZERO] == lAngle )
         {
            //* Capture the XML command *
            short i = ZERO ;
            while ( (ifs.gcount() == 1) && (lineBuff[i] != rAngle) )
               ifs.read( &lineBuff[++i], 1 ) ;
            lineBuff[++i] = nckNULLCHAR ;    // terminate the string
            gsEntity = lineBuff ;
            //* 1) end-of-paragraph "</text:p>" translates to NEWLINE    *
            //*    _unless_ within a table row, in which case it         *
            //*    translates to a TAB.                                  *
            if ( ((gsEntity.find( endPara )) == ZERO) )
            {
               if ( ! tablerow )
               {
                  ofs.put( NEWLINE ) ;
                  lineBytes = ZERO ;   // reset line-character counter
               }
               else
                  ofs.put( '\t' ) ;
            }
            //* 2) table-row "<table:table-row>" disables NEWLINEs.     *
            else if ( ((gsEntity.find( tabRowB )) == ZERO) )
               tablerow = true ;
            //* 3) End table-row </table:table-row> translates to NEWLINE*
            else if ( ((gsEntity.find( tabRowE )) == ZERO) )
            {
               tablerow = false ;
               ofs.put( NEWLINE ) ;
               lineBytes = ZERO ;   // reset line-character counter
            }
         }

         //* Not a left-angle-bracket character. *
         //* Assume that this is text data and   *
         //* save it to target file.             *
         else
         {
            //* Save first character of sequence.*
            if ( lineBuff[ZERO] == ampChar )
            {
               lineBuff[1] = NULLCHAR ;
               gsEntity = lineBuff ;
               xmlCharEntity ( ifs, gsEntity ) ;
               ofs.write( gsEntity.ustr(), (gsEntity.utfbytes() - 1) ) ;
               lineBytes += (gsEntity.utfbytes() - 1) ; // count # of chars written
            }
            else
            {
               ofs.put( lineBuff[ZERO] ) ;
               ++lineBytes ;        // count number of chars written
            }

            //* Copy remainder of text sequence.*
            do
            {
               ifs.read( lineBuff, 1 ) ;
               if ( ifs.gcount() == 1 )
               {
                  if ( lineBuff[ZERO] != lAngle )
                  {
                     //* Convert "predefined entities" to char.*
                     if ( lineBuff[ZERO] == ampChar )
                     {
                        lineBuff[1] = NULLCHAR ;
                        gsEntity = lineBuff ;
                        xmlCharEntity ( ifs, gsEntity ) ;
                        ofs.write( gsEntity.ustr(), ((gsEntity.utfbytes()) - 1) ) ;
                        lineBytes += (gsEntity.utfbytes() - 1) ;
                     }

                     else
                     {
                        //* If a control character found, linefeed   *
                        //* carriage-return or tab, (or invalid char)*
                        // Programmer's Note: Because lineBuff is signed 
                        // chars, we must test for >= ZERO, otherwise we 
                        // could inadvertently snag a supra-ASCII character.
                        if ( lineBuff[ZERO] >= ZERO && lineBuff[ZERO] < nckSPACE )
                        {
                           //gString gsx( "\n==>0x%hhX\n", lineBuff ) ;      // DEBUG ONLY
                           //ofs.write( gsx.ustr(), (gsx.utfbytes() - 1) ) ; // DEBUG ONLY
                           //* Tabs and Newlines are passed through *
                           //* unmodfied. All others discarded.     *
                           if ( (lineBuff[ZERO] != TAB) && 
                                (lineBuff[ZERO] != NEWLINE) )
                              continue ;
                        }

                        ofs.put( lineBuff[ZERO] ) ;
                        ++lineBytes ;  // count number of chars written

                        //* Approaching width of terminal window*
                        //* insert line break between words.    *
                        if ( (lineBytes > tCols) && (lineBuff[ZERO] == nckSPACE) )
                        {
                           ofs.put( NEWLINE ) ;
                           lineBytes = ZERO ;   // reset line-character counter
                        }
                     }
                  }

                  else
                  {
                     //* Push the lAngle back into the input stream.*
                     //* Note that this call sets gcount() to ZERO. *
                     ifs.unget() ;
                  }
               }
            }
            while ( ifs.gcount() == 1 ) ;
         }
      }
      else                 // no data read, end-of-file
         done = true ;

   }     // while()

}  //* End odParseOD_Text() *

//*************************
//*    odParseOX_docx     *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Parse the XML source data for an OpenXML (MS-Office) ".docx" file and      *
//* format it for display.                                                     *
//* MIME-type (first line of file) has been processed by caller.               *
//*                                                                            *
//* Input  : srcSpec  : (by reference) filespec of source OpenXML file         *
//*                     [not used for ".docx", all data in one xml file]       *
//*          ifs      : XML source data input stream                           *
//*          ofs      : formatted display text output stream                   *
//*          tCols    : number of text-display columns for formatted output    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* OpenXML (MS-Office) files:                                                 *
//* Note that the XML commands used are different from standard XML found in   *
//* OpenDocument text data and that the data are spread over multiple files.   *
//*                                                                            *
//* ".docx"                                                                    *
//* =======                                                                    *
//* -- The <w:t>some text</w:t> sequence seems to be the basic text            *
//*    enclosure.                                                              *
//* -- <w:p><w:r><w:t>some text</w:t></w:r></w:p>  seems to be the basic       *
//*    paragraph enclosure.                                                    *
//* -- Therefore inserting a line break after each </w:p> (if no actual        *
//*    line break) seems reasonable.                                           *
//*                                                                            *
//******************************************************************************

static void odParseOX_docx ( const gString& srcSpec, std::ifstream& ifs, 
                             std::ofstream& ofs, short tCols )
{
   const char *endPara = "</w:p>",            // end-of-paragraph
              *softBrk = "<w:br/>",           // soft line-break
              *tabTag  = "<w:tab/>",          // encoded TAB character
              *tabRowB = "<si><t>",           // begin cell
              *tabRowE = "</t></si>" ;        // end cell
   const char lAngle  = '<',
              rAngle  = '>',
              ampChar = '&' ;

   gString gsEntity ;            // convert "predefined entity" characters
   char  lineBuff[gsDFLTBYTES] ; // input-stream line buffer
   short lineBytes = ZERO ;      // count chars (actually bytes) of output text
   bool  tablerow = false,       // signals that scan is within an XML <table-row>
         done = false ;          // loop control

   //* Read and discard all XML commands.   *
   //* Save non-command text to target file.*
   while ( ! done )
   {
      ifs.read( lineBuff, 1 ) ;
      if ( ifs.gcount() == 1 )
      {
         //* If at the beginning of an XML command, read and discard    *
         //* the command (with some exceptions).                        *
         if ( lineBuff[ZERO] == lAngle )
         {
            //* Capture the XML command *
            short i = ZERO ;
            while ( (ifs.gcount() == 1) && (lineBuff[i] != rAngle) )
               ifs.read( &lineBuff[++i], 1 ) ;
            lineBuff[++i] = nckNULLCHAR ;    // terminate the string
            gsEntity = lineBuff ;

            //* 1) end-of-paragraph "</w:p>" translates to NEWLINE       *
            //*    _unless_ within a table row, in which case it         *
            //*    translates to a TAB.                                  *
            if ( ((gsEntity.find( endPara )) == ZERO) )
            {
               if ( ! tablerow )
               {
                  ofs.put( NEWLINE ) ;
                  lineBytes = ZERO ;   // reset line-character counter
               }
               else
                  ofs.put( '\t' ) ;
            }
            //* 2) Soft line-break: <w:br/>                             *
            else if ( ((gsEntity.find( softBrk )) == ZERO) )
            {
               ofs.put( NEWLINE ) ;
               lineBytes = ZERO ;   // reset line-character counter
            }
            //* 3) Encoded TAB character <w:tab/>                       *
            else if ( ((gsEntity.find( tabTag )) == ZERO) )
            {
               ofs.put( '\t' ) ;
            }
            //* 4) table-row "<table:table-row>" disables NEWLINEs.     *
            else if ( ((gsEntity.find( tabRowB )) == ZERO) )
               tablerow = true ;
            //* 5) End table-row </table:table-row> translates to NEWLINE*
            else if ( ((gsEntity.find( tabRowE )) == ZERO) )
            {
               tablerow = false ;
               ofs.put( NEWLINE ) ;
               lineBytes = ZERO ;   // reset line-character counter
            }
         }

         //* Not a left-angle-bracket character. *
         //* Assume that this is text data and   *
         //* save it to target file.             *
         else
         {
            //* Save first character of sequence.*
            if ( lineBuff[ZERO] == ampChar )
            {
               lineBuff[1] = NULLCHAR ;
               gsEntity = lineBuff ;
               xmlCharEntity ( ifs, gsEntity ) ;
               ofs.write( gsEntity.ustr(), (gsEntity.utfbytes() - 1) ) ;
               lineBytes += (gsEntity.utfbytes() - 1) ; // count # of chars written
            }
            else
            {
               ofs.put( lineBuff[ZERO] ) ;
               ++lineBytes ;        // count number of chars written
            }

            //* Copy remainder of text sequence.*
            do
            {
               ifs.read( lineBuff, 1 ) ;
               if ( ifs.gcount() == 1 )
               {
                  if ( lineBuff[ZERO] != lAngle )
                  {
                     //* Convert "predefined entities" to char.*
                     if ( lineBuff[ZERO] == ampChar )
                     {
                        lineBuff[1] = NULLCHAR ;
                        gsEntity = lineBuff ;
                        xmlCharEntity ( ifs, gsEntity ) ;
                        ofs.write( gsEntity.ustr(), ((gsEntity.utfbytes()) - 1) ) ;
                        lineBytes += (gsEntity.utfbytes() - 1) ;

                        #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
                        if ( dbPtr != NULL && dbwp != NULL ) // DEBUGGING ONLY
                           gsdb = gsEntity ;
                        #endif   // ENABLE_DEBUGGING_CODE && DB_WINDOW
                     }

                     else
                     {
                        //* If a control character found, linefeed   *
                        //* carriage-return or tab, (or invalid char)*
                        // Programmer's Note: Because lineBuff is signed 
                        // chars, we must test for >= ZERO, otherwise we 
                        // could inadvertently snag a supra-ASCII character.
                        if ( lineBuff[ZERO] >= ZERO && lineBuff[ZERO] < nckSPACE )
                        {
                           //* Tabs and Newlines are passed through *
                           //* unmodfied. All others discarded.     *
                           if ( (lineBuff[ZERO] != TAB) && 
                                (lineBuff[ZERO] != NEWLINE) )
                              continue ;
                        }

                        ofs.put( lineBuff[ZERO] ) ;
                        ++lineBytes ;  // count number of chars written

                        //* Approaching width of terminal window*
                        //* insert line break between words.    *
                        if ( (lineBytes > tCols) && (lineBuff[ZERO] == nckSPACE) )
                        {
                           ofs.put( NEWLINE ) ;
                           lineBytes = ZERO ;   // reset line-character counter
                        }

                        #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
                        if ( dbPtr != NULL && dbwp != NULL ) // DEBUGGING ONLY
                        {
                           gsdb.compose( "%c", &lineBuff[ZERO] ) ;
                           if ( lineBytes == ZERO )
                              gsdb.append( L'\n' ) ;
                        }
                        #endif   // ENABLE_DEBUGGING_CODE && DB_WINDOW
                     }

                     #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
                     if ( dbPtr != NULL && dbwp != NULL ) // DEBUGGING ONLY
                     {
                        *dbwp = dbPtr->WriteParagraph ( *dbwp, gsdb, 
                                                        nc.blR, true ) ;
                        if ( lineBytes == ZERO )
                           dbwp->xpos = 1 ;
                     }
                     #endif   // ENABLE_DEBUGGING_CODE && DB_WINDOW
                  }

                  else
                  {
                     //* Push the lAngle back into the input stream.*
                     //* Note that this call sets gcount() to ZERO. *
                     ifs.unget() ;
                  }
               }
            }
            while ( ifs.gcount() == 1 ) ;
         }
      }
      else                 // no data read, end-of-file
         done = true ;

   }     // while()

}  //* End odParseOX_docx() *

//*************************
//*    odParseOX_xlsx     *
//*************************
//******************************************************************************
//* PRIVATE METHOD                                                             *
//* --------------                                                             *
//* Parse the XML source data for an OpenXML (MS-Office) ".xlsx" file and      *
//* format it for display.                                                     *
//*                                                                            *
//* The caller has opened "sharedText.xml".                                    *
//* MIME-type (first line of file) has been processed by caller.               *
//*                                                                            *
//* Other needed from the archive are extracted here.                          *
//*                                                                            *
//* Input  : srcSpec  : (by reference) filespec of source OpenXML file         *
//*          ifs_text : XML text source data input stream (sharedText.xml)     *
//*          ofs      : formatted display text output stream                   *
//*          tCols    : number of text-display columns for formatted output    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* OpenXML (MS-Office) files:                                                 *
//* Note that the XML commands used are different from standard XML found in   *
//* OpenDocument text data and that the data are spread over multiple files.   *
//*                                                                            *
//* ".xlsx"                                                                    *
//* =======                                                                    *
//* -- xl/worksheets/sheet1.xml , xl/worksheets/sheet2.xml , etc.              *
//*    ... <dimension ref="A1:G5"/> ... shows the geometry of the active cells.*
//*   -- Each row is formatted:                                                *
//*      <row ...><c r="A1" s="2" t="s"><v>0</v></c> ... </row>                *
//*      Presumably the data in <v>data</v> indicates which record in the      *
//*      sharedStrings.xml file belongs in that cell, but it is certainly      *
//*      not that straightforward. If the referenced cell does not contain     *
//*      text then it may contain the actual cell data.                        *
//*      r="A1"     indicates the cell name                                    *
//*      s="2"      unknown meaning (values of 1, 2, 3 observed)               *
//*      t="s"      seems to indicate that value in <v>value></> is an         *
//*                 index into the sharedStrings.xml table                     *
//*      t="n"      seems to indicate that value in <v>value></> is a          *
//*                 literal value                                              *
//*      There are probably more such codes.                                   *
//*                                                                            *
//* -- sharedStrings.xml contains:                                             *
//*    -- all text fields for all sheets (not numeric data)                    *
//*    -- table is just a header:                                              *
//*       <?xml version="1.0" encoding="UTF-8" standalone="yes"?>              *
//*       <sst xmlns= ... >                                                    *
//*       followed by a series of cells:                                       *
//*           <si><t>some text data</t></si>                                   *
//*    There are no row breaks indicated.                                      *
//*                                                                            *
//* -- xl/workbook.xml contains layout data including sheet names:             *
//*    <sheets><sheet name="Name" sheetId="1" state="visible" r:id="rId2"/>    *
//*    <sheet name="Grades" sheetId="2" state="visible" r:id="rId3"/></sheets> *
//*    These are probably not worth the trouble to extract.                    *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void FileDlg::odParseOX_xlsx ( const gString& srcSpec, std::ifstream& ifs_text, 
                               std::ofstream& ofs, short tCols )
{
   gString tmpPath( this->tempDir ),// temp-file directory
           gsCmd,                   // for constructing system commands and other formatting
           fname,                   // relative path of file containing text data
           sheetSpec,               // filespec of extracted worksheets (> sheet1.xml)
           escPath = srcSpec,       // escape any "special characters" in filespec
           gsOut ;                  // formatted output
   ifstream ifs_sheet ;             // input stream for 2nd and subsequent worksheets
   fmFType ft ;                     // file type of target file
   #if RETURN_SHEET_DIMENSIONS != 0
   int   sheetCols = ZERO,          // number of active columns in current worksheet
         sheetRows = ZERO ;         // number of active rows in current worksheet
   #endif   // RETURN_SHEET_DIMENSIONS
   int   sheetnum = 1 ;             // for constructing worksheet filenames
   bool  done = false,              // loop control
         goodStringFile = false ;   // 'true' if file containing strings initialized

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   //* Scan past the text-file header, to beginning of text records.*
   goodStringFile = odpox_TextRecord ( ifs_text, gsOut, ofs, ZERO, true ) ;
   if ( !goodStringFile )
   {
      short indx = (srcSpec.findlast( fSLASH )) + 1 ;
      gsOut.compose( corruptTemplate, &srcSpec.gstr()[indx] ) ;
      ofs << gsOut.ustr() << "\n\"" << sharedText << "\" is incorrectly formatted." << endl ;
      done = true ;
   }

   while ( !done )
   {
      //* Create the filename for the target worksheet.*
      fname.compose( sheet, &sheetnum ) ;

      //* Extract the worksheet from the document.*
      gsCmd.compose( uzTemplate, escPath.gstr(), fname.ustr(), tmpPath.gstr() ) ;
      this->fmPtr->Systemcall ( gsCmd.ustr() ) ;
      fname.shiftChars( -((fname.findlast( fSLASH )) + 1) ) ; // strip the relative path
      sheetSpec.compose( "%S/%S", tmpPath.gstr(), fname.gstr() ) ;
      if ( (this->TargetExists ( sheetSpec, ft )) )
      {
         //* Open the file.                                      *
         //* Scan the worksheet header, get dimensions of        *
         //* column/row matrix and locate beginning of cell data.*
         ifs_sheet.open( sheetSpec.ustr(), ifstream::in ) ;
         if ( (ifs_sheet.is_open()) && 
              (odpox_ScanSheetHdr ( ifs_sheet, 
               #if RETURN_SHEET_DIMENSIONS != 0
               &sheetCols, &sheetRows, 
               #endif   // RETURN_SHEET_DIMENSIONS
               ofs )) && 
              goodStringFile )
         {
            //* Actual sheet name is in "xl/workbook.xml". *
            //*  (Currently, we don't bother to get it.)   *
            ofs << "Sheet " << sheetnum << "\n-------" << endl ;

            //* Capture and format each row of the first worksheet. *
            while ( (odpox_CaptureRow ( ifs_sheet, ifs_text, ofs, tCols )) != false ) ;
            ofs << endl ;                 // insert space after sheet
         }

         if ( ifs_sheet.is_open() )       // close the target file
            ifs_sheet.close() ;
         this->DeleteFile ( sheetSpec ) ; // delete the processed worksheet file
         ++sheetnum ;                     // reference next worksheet (if any)
      }
      else                 // target worksheet not found: we're done!
         done = true ;

   }     // while(!done)

}  //* End odParseOX_xlsx() *

//*************************
//*  odpox_ScanSheetHdr   *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Scan the XML data in a "SheetX.xml" file, where 'X' is the sheet number.   *
//* Locate the matrix <dimension> command and decode it to determine the       *
//* dimensions of the active-cell matrix, then continue the scan until we      *
//* reach the actual worksheet data.                                           *
//*                                                                            *
//* Input  : ifs      : XML source data input stream                           *
//*          sheetCols: (pointer to int) for sheet columns [CONDITIONAL]       *
//*          sheetRows: (pointer to int) for sheet rows    [CONDITIONAL]       *
//*                     Note: If matrix dimensions found, it is assumed that   *
//*                     one or both dimensions will be non-zero in return.     *
//*          ofs      : FOR DEBUGGING ONLY                                     *
//*                                                                            *
//* Returns: 'true'  if beginning of worksheet data found                      *
//*          'false' if EOF or other file error encountered                    *
//******************************************************************************
//* Converting matrix notation to column/row count                             *
//* ----------------------------------------------                             *
//* 1) The matrix format is:                                                   *
//*    -- "An:Nn" where "A" and "N" are alphabetic sequences denoting leftmost *
//*        and rightmost active columns. Examples:                             *
//*        "A", "B", "C" ... "Z"                                               *
//*        "AA", "AB", "AC" ... "AZ"                                           *
//*        "BA", "BB", "BC" ... "BZ", etc.                                     *
//*    -- "n" are ASCII numeric values                                         *
//* 2) The conversion is straightforward so long as column indicators are      *
//*    between 'A' and 'Z'. For multi-character column designations (e.g. "AG")*
//*    it becomes more difficult. If 'A' == 65 (0x41), and 'Z' == 90 (0x5A)    *
//*       Example: A1:G5 yields:                                               *
//*                'G' - 'A' + 1 == 71 - 65 + 1 == 7 active columns.           *
//*                5 - 1 + 1 == 5 active rows                                  *
//*       Example: A1:Z20 yields:                                              *
//*                90 - 65 + 1 == 26 active columns                            *
//*                20 - 1 + 1  == 20 active rows                               *
//*       Example: A1:AG45 yields:                                             *
//*                Column "AG": This is base26 math                            *
//*                       'A' in the second column == 'Z' - 'A' + 1 == 26      *
//*                       'G' in the first column  == 71 - 65 + 1 == 7         *
//*                       26 + 7 == 33 active columns                          *
//*                       45 - 1 + 1 == 45 active rows                         *
//*                                                                            *
//******************************************************************************

static bool odpox_ScanSheetHdr ( std::ifstream& ifs, 
                                 #if RETURN_SHEET_DIMENSIONS != 0
                                 int* sheetCols, int* sheetRows, 
                                 #endif   // RETURN_SHEET_DIMENSIONS
                                 std::ofstream& ofs )
{
   #define DEBUG_gfpoxSSH (0)    // set to non-zero to enable debugging

   const char *shData  = "<sheetData>" ;
   gString gsCmd ;
   char  lineBuff[gsDFLTBYTES] ; // input-stream line buffer
   bool  done = false,           // loop control
         status = false ;        // return value

   #if RETURN_SHEET_DIMENSIONS != 0
   const char *dimCmd  = "<dimension ref=\"" ;
   const short dimCmd_len = 16 ; // number of bytes in 'dimCmd' (not incl. nullchar)
   bool goodDimensions = false ; // 'true' if matrix dimensions captured
   *sheetCols = *sheetRows = ZERO ; // initialize caller's parameters
   #endif   // RETURN_SHEET_DIMENSIONS

   while ( !done )
   {
      if ( (xmlGetCmd ( ifs, lineBuff )) != false )
      {
         gsCmd = lineBuff ;      // prepare to parse the command

         #if RETURN_SHEET_DIMENSIONS != 0
         //* Is this the "dimension" command? *
         if ( !goodDimensions &&
              (gsCmd.utfbytes() > dimCmd_len) && 
              ((gsCmd.find( dimCmd )) == ZERO) )
         {
            //* Convert matrix notation to column/row count.*
            //* (See notes in method header.)               *
            #if DEBUG_gfpoxSSH != 0
            ofs << gsCmd.ustr() << endl ;
            #endif   // DEBUG_gfpoxSSH

            gsCmd.shiftChars( -(dimCmd_len) ) ;

            #if DEBUG_gfpoxSSH != 0
            ofs << gsCmd.ustr() << endl ;
            #endif   // DEBUG_gfpoxSSH

            const wchar_t* wPtr = gsCmd.gstr() ;
            wchar_t lowChar = L'#', highChar = L'#' ;
            int     lowRow = ZERO, highRow = ZERO ;
            if ( (wPtr[1] >= L'1') && (wPtr[1] <= L'9') )
            {
               if ( (swscanf ( wPtr, L"%C%d", &lowChar, &lowRow )) == 2 )
               {
                  while ( *wPtr != L':' )
                     gsCmd.shiftChars( -1 ) ;
                  gsCmd.shiftChars( -1 ) ;
                  if ( (wPtr[1] >= L'1') && (wPtr[1] <= L'9') )
                  {
                     if ( (swscanf ( wPtr, L"%C%d", &highChar, &highRow )) == 2 )
                     {
                        *sheetCols = highChar - lowChar + 1 ;
                        *sheetRows = highRow - lowRow + 1 ;
                        goodDimensions = true ;
                     }
                  }
                  else
                  {
                     /* NOTE: DIMENSIONING BEYOND COLUMN 'Z' *
                      * NOT YET IMPLEMENTED.                 */
                     #if DEBUG_gfpoxSSH != 0
                     ofs << "** ERROR! DIMENSIONS BEYOND COLUMN 'Z'. **" << endl ;
                     #endif   // DEBUG_gfpoxSSH
                  }
               }
            }
            else
            {
               /* NOTE: DIMENSIONING BEYOND COLUMN 'Z' *
                * NOT YET IMPLEMENTED.                 */
               #if DEBUG_gfpoxSSH != 0
               ofs << "** ERROR! DIMENSIONS BEYOND COLUMN 'Z'. **" << endl ;
               #endif   // DEBUG_gfpoxSSH
            }

            #if DEBUG_gfpoxSSH != 0
            gsCmd.compose( "lowChar:%C highChar:%C lowRow:%d highRow:%d goodDimensions:%hhd", 
                           &lowChar, &highChar, &lowRow, &highRow, &goodDimensions ) ;
            ofs << gsCmd.ustr() << endl ;
            #endif   // DEBUG_gfpoxSSH
         }
         else 
         #endif   // RETURN_SHEET_DIMENSIONS

         //* Else, is this the 'sheetData' command? *
         if ( (gsCmd.compare( shData )) == ZERO )
         {
            #if DEBUG_gfpoxSSH != 0
            ofs << gsCmd.ustr() << " done!" << endl ;
            #endif   // DEBUG_gfpoxSSH

            status = done = true ;
         }
      }
      else
         done = true ;
   }        // while(!done)

   return status ;

   #undef DEBUG_gfpoxSSH
}  //* End odpox_ScanSheetHdr() *

//*************************
//*   odpox_TextRecord    *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Scan the spreadsheet file which contains the text for each text cell.      *
//* If 'init' != false, then simply scan past the file header.                 *
//* If 'init' == false, the extract the text from the next record.             *
//*                                                                            *
//* Input  : ifs      : input stream for file containing document's text data  *
//*          gsText   : (by reference) buffer to receive string data           *
//*                     (if 'init' != false, then contents unchanged)          *
//*          ofs      : DEBUGGING ONLY                                         *
//*          index    : (optional, ZERO by default) index value in sheet cell  *
//*                     (this parameter may be removed when algorithm stable)  *
//*          init     : (optional, 'false' by default)                         *
//*                     if 'false' return the next string record               *
//*                     if 'true'  scan past the file header                   *
//*                                                                            *
//* Returns: if 'init' != false:                                               *
//*             'true'  beginning of text records found                        *
//*             'false' if end-of-file                                         *
//*          if 'init' == false:                                               *
//*             'true'  text record successfully extracted                     *
//*             'false' if end-of-data, or end-of-file                         *
//******************************************************************************

static bool odpox_TextRecord ( std::ifstream& ifs, gString& gsText, std::ofstream& ofs, 
                               int index, bool init )
{
   #define DEBUG_gfpoxSTH (0)    // set to non-zero to enable debugging
   #if DEBUG_gfpoxSTH != 0
   const char *sstEnd       = "</sst>" ;
   #endif   // DEBUG_gfpoxSTH

   const char  /*lAngle  = '<',*/
               rAngle  = '>' ;
   const char *sstStart     = "<sst xmlns=",
              *strItemStart = "<si>",
              *strItemEnd   = "</si>",
              *textStart    = "<t>",
              *textEnd      = "</t>",
              *unkString    = "#" ; // returned if unable to extract string data
   const short sstStart_len = 11,   // number of bytes in 'sstCmd' (not incl. nullchar)
               recEnd_len   = 9 ;   // combined length of "</t></si>"

   gString gsCmd ;
   short indx = ZERO ;           // index into lineBuff
   char  lineBuff[gsDFLTBYTES] ; // input-stream line buffer
   bool  done = false,           // loop control
         status = false ;        // return value

   while ( !done )
   {
      if ( (xmlGetCmd ( ifs, lineBuff )) != false )
      {
         gsCmd = lineBuff ;   // prepare to parse the command

         //* Extract a text record.                   *
         //* (Init call has scanned past the header.) *
         if ( !init )
         {
            gsText = unkString ;    // assume the worst

            // Programmer's Note: For simplicity, we (mostly) assume 
            // that the data for the cell is valid and complete.
            if ( (gsCmd.compare( strItemStart )) == ZERO )
            {
               xmlGetCmd ( ifs, lineBuff ) ;
               gsCmd.append( lineBuff ) ;
               if ( (gsCmd.find( textStart )) > ZERO )
               {
                  indx = ZERO ;
                  do
                  {
                     ifs.read ( &lineBuff[indx++], 1 ) ;
                     if ( ifs.gcount() != 1 )
                        break ;
                  }
                  while ( !((lineBuff[indx - 1] == rAngle) && (lineBuff[indx - 2] == 'i')) ) ;
                  lineBuff[indx] = NULLCHAR ;   // terminate the string
                  gsCmd.append( lineBuff ) ;

                  #if DEBUG_gfpoxSTH != 0
                  ofs << "   " << gsCmd.ustr() << endl ;
                  #endif   // DEBUG_gfpoxSTH

                  //* If end-of-record captured, extract the text.*
                  if ( (gsCmd.find( strItemEnd )) > ZERO )
                  {
                     indx = gsCmd.after( textStart ) ;
                     gsCmd.shiftChars( -(indx) ) ;
                     if ( (indx = gsCmd.find( textEnd )) > ZERO )
                     {
                        gsCmd.erase( indx, recEnd_len ) ;
                        gsText = gsCmd ;

                        #if DEBUG_gfpoxSTH != 0
                        ofs << "   " << gsText.ustr() << endl ;
                        #endif   // DEBUG_gfpoxSTH

                        status = done = true ;
                     }
                  }
               }
            }
            else        // end-of-data OR end-of-file
            {
               #if DEBUG_gfpoxSTH != 0
               if ( (gsCmd.compare( sstEnd )) == ZERO )
                  ofs << gsCmd.ustr() <<  "  done!" << endl ;
               else
                  ofs << "Error! - sharedText.xml - EOF" << endl ;
               #endif   // DEBUG_gfpoxSTH

               done = true ;
            }
         }

         //* Scan the header. (init != false) *
         else
         {
            if ( (gsCmd.utfbytes() > sstStart_len) && 
                 ((gsCmd.find( sstStart )) == ZERO) )
            {
               #if DEBUG_gfpoxSTH != 0
               ofs << gsCmd.ustr() << " done!" << endl ;
               #endif   // DEBUG_gfpoxSTH
   
               status = done = true ;
            }
         }
      }

      //* Else, end-of-file reached. Return a dummy value. *
      else
      {
         if ( !init )
            gsText = unkString ;
         done = true ;
      }
   }        // while(!done)
   return status ;

   #undef DEBUG_gfpoxSTH
}  //* End odpox_TextRecord() *

//*************************
//*   odpox_CaptureRow    *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Scan the XML data in a "SheetX.xml" file, where 'X' is the sheet number.   *
//* Extract and format for display the data for the next row of the matrix.    *
//* Note that the sheet input stream does not contain the full data for each   *
//* cell. The sheet contains cell contents for _numeric_ data, but only the    *
//* lookup indices for the text data cells. See notes in odParseOX_xlsx()      *
//* method.                                                                    *
//*                                                                            *
//* Input  : ifs_sheet: XML source data input stream                           *
//*          ifs_text : input stream for file containing document's text data  *
//*          ofs      : formatted display text output stream                   *
//*          tCols    : number of text-display columns for formatted output    *
//*                     (currently unused for spreadsheet display)             *
//*                                                                            *
//* Returns: 'true'  if row data captured                                      *
//*          'false' if sheet end-of-data reached without row capture          *
//******************************************************************************

static bool odpox_CaptureRow ( std::ifstream& ifs_sheet, std::ifstream& ifs_text,
                               std::ofstream& ofs, short tCols )
{
   #define DEBUG_gfpoxCR (0)     // set to non-zero to enable debugging

   const char *xmlStartRow   = "<row r=",
              *xmlEndRow     = "</row>",
              *xmlStartCell  = "<c r=",
              //*xmlEndCell    = "</c>",
              *xmlStartValue = "<v>",
              //*xmlEndValue   = "</v>",
              *xmlType       = "t=\"" ;
   const char  //lAngle  = '<',
               rAngle  = '>' ;

   gString gsCmd ;
   double fval ;                 // cell contents (decimal numeric value)
   int ival ;                    // cell contents (integer numeric value)
   short cells,                  // number of cells written for current row
         gsIndx,                 // index into gsCmd
         cbIndx ;                // index into 'cellBuff'
   char  lineBuff[gsDFLTBYTES],  // input-stream line buffer
         cellBuff[gsDFLTBYTES],  // raw cell data
         typeChar ;              // type of cell data: 's' string index, 'n' ASCII numeric
   bool  rowStart = false,       // 'true' if beginning of row identified
         done = false,           // loop control
         goodRow = false ;       // return value

   while ( !done )
   {
      if ( (xmlGetCmd ( ifs_sheet, lineBuff )) != false )
      {
         gsCmd = lineBuff ;      // prepare to parse the command

         //* If row processing has begun *
         if ( rowStart != false )
         {
            //* If end-of-row command *
            if ( (gsCmd.find( xmlEndRow )) == ZERO )
            {
               ofs << endl ;  // terminate the table row

               #if DEBUG_gfpoxCR != 0
               ofs << gsCmd.ustr() << endl ;
               #endif   // DEBUG_gfpoxCR

               goodRow = done = true ;
               break ;
            }

            //* If command is a cell record *
            // Programmer's Note: For simplicity, we assume that the data 
            // for the cell is valid and complete.
            if ( (gsCmd.find( xmlStartCell )) == ZERO )
            {
               gsCmd.copy( cellBuff, gsCmd.utfbytes() ) ;
               cbIndx = gsCmd.utfbytes() - 1 ;  // reference the null terminator

               xmlGetCmd ( ifs_sheet, lineBuff ) ;
               gsCmd = lineBuff ;
               gsCmd.copy( &cellBuff[cbIndx], gsCmd.utfbytes() ) ;
               cbIndx += gsCmd.utfbytes() - 1 ;

               //* The value within <v></v> is ASCII numeric.*
               //* Scan the value and the closing XML tag.   *
               do
               {
                  ifs_sheet.read( lineBuff, 1 ) ;
                  if ( ifs_sheet.gcount() != 1 )
                     break ;
                  cellBuff[cbIndx++] = lineBuff[ZERO] ;
               }
               while ( !((cellBuff[cbIndx - 1] == rAngle) && (cellBuff[cbIndx - 2] == 'c')) ) ;
               cellBuff[cbIndx] = NULLCHAR ;

               //* Decode the cell information *
               gsCmd = cellBuff ;
               if ( (gsIndx = gsCmd.after( xmlType )) < ZERO )
                  gsIndx = ZERO ;
               typeChar = gsCmd.gstr()[gsIndx] ;
               if ( (gsIndx = gsCmd.after( xmlStartValue, gsIndx )) < ZERO )
                  gsIndx = ZERO ;
               swscanf ( &gsCmd.gstr()[gsIndx], L"%lf", &fval ) ;
               ival = (int)fval ;

               #if DEBUG_gfpoxCR != 0
               gsCmd.append( "\t'%c'  %.2lf  %d", &typeChar, &fval, &ival ) ;
               ofs << "  " << gsCmd.ustr() << endl ;
               #endif   // DEBUG_gfpoxCR

               if ( typeChar == 's' )     // if cell content is a string
               {
                  odpox_TextRecord ( ifs_text, gsCmd, ofs, ival ) ;
               }
               else if ( typeChar == 'n' )   // if cell content is an ASCII integer value
               {
                  gsCmd.compose( "%d", &ival ) ;
               }
               else
               {
                  /* UNKNOWN CODE */
                  gsCmd = "###" ;
               }
               if ( cells > ZERO )
                  gsCmd.insert( L'\t' ) ;
               ofs << gsCmd.ustr() ;   // write the formatted text
               ++cells ;
            }

            //* Else, we are out-of-synch or end-of-data/end-of-file reached.*
            else
            {
               //* Try to re-synch by finding the end-of-cell tag to prepare *
               //* for the next call. If we fail, then no more data.         *
               cbIndx = ZERO ;
               do
               {
                  ifs_sheet.read( &lineBuff[cbIndx++], 1 ) ;
                  if ( ifs_sheet.gcount() != 1 )
                     done = true ;  // (probably end-of-file)
               }
               while ( !((lineBuff[cbIndx - 1] == rAngle) && (lineBuff[cbIndx - 2] == 'c')) ) ;

               gsCmd = "#" ;           // write a dummy record
               if ( cells > ZERO )
                  gsCmd.insert( L'\t' ) ;
               ofs << gsCmd.ustr() ;   // write the formatted text
               ++cells ;
            }
         }

         //* Else, beginning of row not yet identified *
         else
         {
            if ( (gsCmd.find( xmlStartRow )) == ZERO )
            {
               #if DEBUG_gfpoxCR != 0
               ofs << gsCmd.ustr() << endl ;
               #endif   // DEBUG_gfpoxCR

               cells = ZERO ;
               rowStart = true ;
            }
         }
      }
      else
         done = true ;
   }
   return goodRow ;

   #undef DEBUG_gfpoxCR
}  //* End odpox_CaptureRow() *

//*************************
//*    odParseOX_pptx     *
//*************************
//******************************************************************************
//* PRIVATE METHOD                                                             *
//* --------------                                                             *
//* Parse the XML source data for an OpenXML (MS-Office) ".pptx" file and      *
//* format it for display.                                                     *
//* MIME-type (first line of file) has been processed by caller.               *
//*                                                                            *
//* Input  : srcSpec    : (by reference) filespec of source OpenXML file       *
//*          ofs        : formatted display text output stream                 *
//*          tCols      : number of text-display columns for formatted output  *
//*                       (currently unused for slide display)                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* OpenXML (MS-Office) files:                                                 *
//* Note that the XML commands used are different from standard XML found in   *
//* OpenDocument text data and that the data are spread over multiple files.   *
//*                                                                            *
//* ".pptx"                                                                    *
//* =======                                                                    *
//* 1) Each presentation document contains one or more slide files:            *
//*      "ppt/slides/slide1.xml", ppt/slides/slide2.xml", etc.                 *
//*    So far as we can tell at this point, all display text is contained in   *
//*    these files.                                                            *
//* 2) Each text string is encapsulated with all the formatting information    *
//*    it needs; however, we are interested only in the "<a:t>some text</a:t>" *
//*    which encloses the string,                                              *
//* 3) Additional interesting tags are the paragraph: <a:p> ... </a:p>         *
//*    and line: <a:r> ... </a:r>  sequences.                                  *
//* 4) "</p:sld>" indicates the end of the slide.                              *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void FileDlg::odParseOX_pptx ( const gString& srcSpec, std::ofstream& ofs, short tCols )
{
   #define DEBUG_gfpoxPPTX (0)      // set to non-zero to enable debugging

   const char //*paraStart   = "<a:p>",
              //*paraEnd     = "</a:p>",
              //*rowStart    = "<a:r>",
              *rowEnd      = "</a:r>",
              *textStart   = "<a:t>",
              *textEnd     = "</a:t>" ;
   //const short textStart_len = 5 ;  // length of 'textStart' (not incl. nullchar)
   const short textEnd_len   = 6 ;  // length of 'textEnd' (not incl.nullchar)
   const char ampChar = '&' ;

   gString tmpPath( this->tempDir ),// temp-file directory
           gsCmd,                   // for constructing system commands and other formatting
           fname,                   // relative path of file containing text data
           slideSpec,               // filespec of extracted slide (> slide1.xml)
           gsOut,                   // formatted output
           gsEntity,                // capture XML character entities
           escPath = srcSpec ;      // escape any "special characters" in filespec
   ifstream ifs_slide ;             // input stream for 2nd and subsequent slides
   char lineBuff[gsDFLTBYTES] ;     // input-stream line buffer
   fmFType ft ;                     // file type of target file
   int   slidenum = 1 ;             // for constructing slide filenames
   short lbIndx,                    // index for 'lineBuff'
         gsIndx ;                   // index for gsCmd
   bool  done = false ;             // loop control

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   while ( !done )
   {
      //* Create the filename for the target worksheet.*
      fname.compose( slide, &slidenum ) ;

      #if DEBUG_gfpoxPPTX != 0
      ofs << "fname(1): '" << fname.ustr() << "'" << endl ;
      #endif   // DEBUG_gfpoxPPTX

      //* Except for the first slide, extract the slide file. *
      if ( slidenum > 1 )
      {
#if 1    // NEW
         gsCmd.compose( uzTemplate, escPath.gstr(), fname.ustr(), tmpPath.gstr() ) ;
#else    // OLD
         gsCmd.compose( uzTemplate, srcSpec.gstr(), fname.ustr(), tmpPath.gstr() ) ;
#endif   // OLD
         this->fmPtr->Systemcall ( gsCmd.ustr() ) ;
      }
      fname.shiftChars( -((fname.findlast( fSLASH )) + 1) ) ; // strip the relative path
      slideSpec.compose( "%S/%S", tmpPath.gstr(), fname.gstr() ) ;

      #if DEBUG_gfpoxPPTX != 0
      ofs << "fname(2): '" << fname.ustr() << "'\n" 
          << "slideSpec: '" << slideSpec.ustr() << "'" << endl ;
      #endif   // DEBUG_gfpoxPPTX

      //* If file successfully extracted from the document archive.*
      if ( (this->TargetExists ( slideSpec, ft )) )
      {
         //* Open the file and verify the file's MIME type as XML..*
         ifs_slide.open( slideSpec.ustr(), std::ifstream::in ) ;
         if ( (ifs_slide.is_open()) &&
              ((xmlGoodMIME ( ifs_slide )) != false) )
         {
            gsOut.compose( "Slide %hd\n-------", &slidenum ) ;
            ofs << gsOut.ustr() << endl ;

            //* Extract and format the text data for this slide.*
            while ( (xmlGetCmd ( ifs_slide, lineBuff )) != false )
            {
               gsCmd = lineBuff ;

               //* If begin-text tag, capture until end-text tag.*
               if ( (gsCmd.compare( textStart )) == ZERO )
               {
                  lbIndx = ZERO ;
                  do
                  {
                     ifs_slide.read( &lineBuff[lbIndx], 1 ) ;
                     if ( ifs_slide.gcount() != 1 )
                        break ;
                     if ( lineBuff[lbIndx] == ampChar )
                     {
                        lineBuff[lbIndx + 1] = NULLCHAR ;
                        gsEntity = &lineBuff[lbIndx] ;
                        xmlCharEntity ( ifs_slide, gsEntity ) ;
                        gsEntity.copy( &lineBuff[lbIndx], gsEntity.utfbytes() ) ;
                        lbIndx +=gsEntity.utfbytes() - 1 ;
                     }
                     else
                        ++lbIndx ;
                  }
                  while ( !((lineBuff[lbIndx - 1] == textEnd[5]) && 
                            (lineBuff[lbIndx - 2] == textEnd[4]))
                          && (lbIndx < (gsALLOCDFLT - 2)) ) ;
                  lineBuff[lbIndx] = NULLCHAR ;
                  gsCmd.append( lineBuff ) ;

                  #if DEBUG_gfpoxPPTX != 0
                  ofs << gsCmd.ustr() << endl ;
                  #endif   // DEBUG_gfpoxPPTX

                  //* end-lf-record captured, extract the text.*
                  if ( (gsCmd.find( textEnd )) > ZERO )
                  {
                     gsIndx = gsCmd.after( textStart ) ;
                     gsCmd.shiftChars( -(gsIndx) ) ;
                     if ( (gsIndx = gsCmd.find( textEnd )) > ZERO )
                     {
                        gsCmd.erase( gsIndx, textEnd_len ) ;
                        ofs << gsCmd.ustr() << endl ; // write text to output
                     }
                  }
               }

               //* If end-of-row or end-of-paragraph, insert linefeed.*
               else if ( /*((gsCmd.compare( paraEnd )) == ZERO) ||*/
                         ((gsCmd.compare( rowEnd )) == ZERO) )
               {
                  ofs << endl ;
               }
            }

            //* Close the source file.*
            ifs_slide.close() ;
         }

         //* Delete the source file. (Caller will delete first slide.) *
         if ( slidenum > 1 )
            this->DeleteFile ( slideSpec.ustr() ) ;

         ofs << endl ;     // space between slides
         ++slidenum ;      // index the next slide
      }     // if(!done)
      else                 // target worksheet not found: we're done!
      {
         #if DEBUG_gfpoxPPTX != 0
         ofs << "\n**End of slides **" << endl ;
         #endif   // DEBUG_gfpoxPPTX

         done = true ;
      }

   }        // while(!done)

   #undef DEBUG_gfpoxPPTX
}  //* End odParseOX_pptx() *

//*************************
//*      xmlGoodMIME      *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Read the first line of the file. If file is a valid XML file, then this    *
//* will be the MIME-type tag. Example:                                        *
//*        <?xml version="1.0" encoding="UTF-8" standalone="yes"?>             *
//*                                                                            *
//* Input  : ifs     : (by reference) open input stream                        *
//*                                                                            *
//* Returns: 'true'  if valid XML MIME type, else 'false'                      *
//******************************************************************************

static bool  xmlGoodMIME ( std::ifstream& ifs )
{
   const char* MimeTag = "<?xml " ;
   const char rAngle  = '>' ;
   char  lineBuff[gsDFLTBYTES] ; // input-stream line buffer
   bool goodMIME = false ;

   ifs.getline( lineBuff, gsALLOCDFLT, nckNEWLINE ) ;
   short indx = (ifs.gcount() - 2) ;
   if ( lineBuff[indx] == '\r' ) // protect against Windoze CR/LF combo
      --indx ;
   if ( (ifs.good() || (ifs.gcount() > ZERO)) && 
        (lineBuff[0] == MimeTag[0]) && (lineBuff[1] == MimeTag[1]) && 
        (lineBuff[2] == MimeTag[2]) && (lineBuff[3] == MimeTag[3]) && 
        (lineBuff[4] == MimeTag[4]) && (lineBuff[5] == MimeTag[5]) &&
        ((lineBuff[indx]) == rAngle) )
   {
      goodMIME = true ;
   }

   return goodMIME ;

}  //* End xmlGoodMIME() *

//*************************
//*       xmlGetCmd       *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Scan the specified input stream and capture the first complete XML         *
//* command.                                                                   *
//*                                                                            *
//* Note: If an XML command is longer than the input buffer, the entire command*
//*       will be scanned, but the tail of the command will be truncated.      *
//*       (This is unlikely because the buffer is 'gsDFLTBYTES' (4096 bytes).  *
//*                                                                            *
//* Input  : ifs     : (by reference) open input stream                        *
//*          lineBuff: receives the XML command (null terminated)              *
//*                                                                            *
//* Returns: 'true'  if XML command captured                                   *
//*          'false' if end-of-file reached before command fully captured      *
//******************************************************************************

static bool xmlGetCmd ( std::ifstream& ifs, char lineBuff[gsDFLTBYTES] )
{
   const char  lAngle  = '<',
               rAngle  = '>' ;
   short indx = ZERO ;        // index into lineBuff
   bool  done    = false,     // loop control
         goodCmd = false ;    // return value

   while ( !done )
   {
      //* Scan to beginning of XML command *
      do
      {
         ifs.read( lineBuff, 1 ) ;
         if ( ifs.gcount() != 1 )         // if end-of-file
         {
            lineBuff[ZERO] = NULLCHAR ;
            done = true ;
            break ;
         }
      }
      while ( lineBuff[ZERO] != lAngle ) ;

      //* Capture the XML command *
      if ( !done )
      {
         for ( indx = 1 ; indx < (gsDFLTBYTES - 2) ; ++indx )
         {
            ifs.read( &lineBuff[indx], 1 ) ;
            if ( ifs.gcount() == 1 )
            {
               if ( lineBuff[indx] == rAngle )
               {
                  lineBuff[indx + 1] = NULLCHAR ;
                  goodCmd = done = true ;
                  break ;
               }
            }
            else                             // if end-of-file
            {
               lineBuff[indx + 1] = NULLCHAR ;
               done = true ;
               break ;
            }
         }  // for(;;)
      }     // if(!done)
   }        // while(!done)

   return goodCmd ;

}  //* End xmlGetCmd() *

//*************************
//*     xmlCharEntity     *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* An ampersand ( '&' ) character has been read from the XML input stream.    *
//* Read the remainder of the "character entity reference" sequence            *
//* ("predefined entity" in XML-speak) and convert it to the actual character. *
//* No more than five (5) additional characters will be read from the input    *
//* stream. This covers the length of all the predefined entities.             *
//*                                                                            *
//* Note that the "XML Entity Definitions for Characters" i.e. mathematical    *
//* and scientific character conversions are not supported here. These are not *
//* single-byte characters and will probably never show up in a regexp search  *
//* expression anyway.                                                         *
//*                                                                            *
//* Input  : ifs    : (by reference) handle for open input stream              *
//*          ebuff  : (by reference) character buffer                          *
//*                   - on entry, first position contains the ampersand '&'    *
//*                                                                            *
//* Returns: first byte of ebuff.ustr()                                        *
//*                                                                            *
//*          (if malformed or unknown entity, first byte is garbage)           *
//*          If conversion successful, 'ebuff' will contain the converted      *
//*          character. Else 'ebuff' will contain the actual bytes read.       *
//*          Either way, sequence will be null-terminated.                     *
//******************************************************************************
//* Programmer's Note: This method assumes that there will be no line break    *
//* within a predefined entity.                                                *
//******************************************************************************

static char xmlCharEntity ( std::ifstream& ifs, gString& ebuff )
{
   const char SEMICOLON = ';' ;     // predefined-entity delimiter character
   const short maxINDX = 6 ;        // max character index
   char  lbuff[32] ;                // input buffer
   short indx = 1 ;
   ebuff.copy( lbuff, 32 ) ;        // working copy of user data

   do
   {
      ifs.read( &lbuff[indx], 1 ) ;
      if ( ifs.gcount() != 1 )
      {
         ++indx ;
         break ;
      }
   }
   while ( (lbuff[indx++] != SEMICOLON) && (indx < maxINDX) ) ;
   lbuff[indx] = NULLCHAR ;
   ebuff = lbuff ;   // copy raw input to caller's buffer

   if ( lbuff[indx - 1] == SEMICOLON ) // if correct format for predefined-entity
   {
      if ( (ebuff.compare( "&quot;" )) == ZERO )        // quotation
         ebuff = L"\"" ;
      else if ( (ebuff.compare( "&apos;" )) == ZERO )   // apostrophe
         ebuff = L"'" ;
      else if ( (ebuff.compare( "&amp;"  )) == ZERO )   // literal ampersand
         ebuff = L"&" ;
      else if ( (ebuff.compare( "&lt;"   )) == ZERO )   // less-than
         ebuff = L"<" ;
      else if ( (ebuff.compare( "&gt;"   )) == ZERO )   // greater-than
         ebuff = L">" ;
   }
   return ( ebuff.ustr()[ZERO] ) ;

}  //* End xmlCharEntity() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

