//********************************************************************************
//* File       : FileDlgTrash.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 08-Jun-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module contains methods related to management of data in the            *
//* 'Trashcan':  a) send files or directory trees to the Trashcan                *
//*              b) restore files or directory trees from the Trashcan           *
//*              c) empty the Trashcan (delete Trashcan contents)                *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2) and above.                            *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above  *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//********************************************************************************
//* Using the trashcan for GNOME, KDE or any trashcan (loosly) based on the      *
//* freedesktop.org trashcan specification.                                      *
//*                                                                              *
//* The trashcan for the desktop environment is located at                       *
//*  ~/.local/share/Trash                                                        *
//* and has three subdirectories                                                 *
//*   files    (deleted files)                                                   *
//*   info     (info about deleted files)                                        *
//*   expunged (not used by FileMangler)                                         *
//* Older versions of the desktop may have different locations for the trashcan  *
//* for example:  ~/.Trash                                                       *
//* Also, there are system-level directories for trash, created by Admin,        *
//* but we aren't interested in those -- only the user-local trash.              *
//*                                                                              *
//* The specification says that if the trashcan subdirectories do not exist,     *
//* they should be automagically created without delay. This is done at a        *
//* higher (application) level.                                                  *
//*                                                                              *
//* When a file is moved to the trashcan, two things happen                      *
//* 1) The file is moved to ~/.local/share/Trash/files subdirectory.             *
//*    - Note that if a file of that name already exists, then the target is     *
//*      renamed to avoid overwriting an existing file.                          *
//*      - For Nautilus, the new name will take the format originalname.2.ext    *
//*        so we follow that format.                                             *
//*                                                                              *
//* 2) A file with the extension of '.trashinfo' is created in the               *
//*    ~/.local/share/Trash/info subdirectory with a base name corresponding     *
//*    to the name of the file in the 'files' subdirectory.                      *
//*    - This file contains three entries:                                       *
//*      a) a header line                                                        *
//*      b) the full path/filename of the source                                 *
//*      c) a date/timestamp for when the file was moved to the trash            *
//*    - Note that for subdirectory trees, a '.trashinfo' file is created ONLY   *
//*      for the top-level directory name. Contents of the subdirectory are      *
//*      simply moved to the trashcan, and no associated '.trashinfo' file       *
//*      is created for them.                                                    *
//*                                                                              *
//* Example:                                                                     *
//*  - Use Nautilus to move the file 'FileDlgTrash.o' to the trashcan.           *
//*    Two files appear:                                                         *
//*      ~/.local/share/Trash/files/FileDlgTrash.o                               *
//*      ~/.local/share/Trash/info/FileDlgTrash.o.trashinfo                      *
//*    The first is the original file.                                           *
//*    The second is information about that file i.e.                            *
//*      [Trash Info]                                                            *
//*      Path=/home/sam/Documents/SoftwareDesign/FileMangler/FileDlgTrash.o      *
//*      DeletionDate=2013-04-20T10:29:43                                        *
//*                                                                              *
//*  - Recompile the source file to create another copy of FileDlgTrash.o        *
//*    then again:                                                               *
//*  - Use Nautilus to move the file 'FileDlgTrash.o' to the trashcan.           *
//*    Two additional files appear:                                              *
//*      ~/.local/share/Trash/files/FileDlgTrash.2.o                             *
//*      ~/.local/share/Trash/info/FileDlgTrash.2.o.trashinfo                    *
//*    The first is the original file.                                           *
//*    The second is information about that file i.e.                            *
//*      [Trash Info]                                                            *
//*      Path=/home/sam/Documents/SoftwareDesign/FileMangler/FileDlgTrash.o      *
//*      DeletionDate=2013-04-20T10:30:36                                        *
//*                                                                              *
//* Note: If source file is a Symbolic Link, the link, NOT link target is        *
//*       moved to the Trashcan                                                  *
//* Note: Trashcan operations for certain 'special' files, Character Device,     *
//*       Block Device, Socket, or others MAY not be supported.                  *
//*       Copy or delete of these files may require super-user privlege OR       *
//*       these operations may not be possible under normal circumstances.       *
//*       For these, the user will be alerted if the operation is not            *
//*       supported or not permitted.                                            *
//*                                                                              *
//* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *  *
//* Notes on implementing the trashcan functionality in FileMangler.             *
//* 1) On instantiation, the FileDlg class calls the VerifyTempStorage() method  *
//*    to check for the existence of the desktop trashcan directories:           *
//*         ~/.local/share/Trash/files                                           *
//*         ~/.local/share/Trash/info                                            *
//*     If the directories exist, then we set the 'trashcanOK' flag.             *
//*     If the standard desktop trashcan is unavailable, the user may have       *
//*     specified an alternate trashcan in the configuration file.               *
//*     If neither of these, then we won't be able to use the trashcan until     *
//*     the directories are properly configured.                                 *
//*                                                                              *
//* 2) We rely upon the main application code to re-verify the existence of      *
//*    these directories by an additional call to VerifyTempStorage().           *
//*    a) if they exist, we have already verified, and will use them             *
//*    b) if they do not exist, we rely upon the application to alert the user   *
//*       that the system and/or the application is incorrectly configured,      *
//*       and to properly configure the trashcan before we need to use it.       *
//*                                                                              *
//* 3) NOTE: All functionality is reasonably stable as of May 2016.              *
//*    Further optimization is possible:                                         *
//*    a) The tcDialog _could_ start in the selection Scrollext control.         *
//*       This would eliminate the 'SELECT' Pushbutton; however, it would        *
//*       require us to complain if more than one item were interactively        *
//*       selected for restoration. The algorithm could be enhanced (see         *
//*       'tcrValidateTarget' method), but then we would have to stay in the     *
//*       CWD rather that exiting to the restoration target directory.           *
//*       This is definitely possible--give it some thought...                   *
//*    b) There are too many screen updates during the intermediate processing,  *
//*       but that cannot be avoided if we want to leverage the clipboard        *
//*       algorithms for copy/cut/paste. The only reasonable solution we can     *
//*       see would be to add a class-level flag to disable screen updates.      *
//*       We aren't sure that would even work because some updates are built     *
//*       into the NcDialog API.                                                 *
//*    c) Handling moose input is fully functional but not very elegant.         *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//** Local Definitions and Data **
#if ENABLE_DEBUGGING_CODE != 0
#define enableDBWIN (0)          // set to non-zero to enable debugging window
#if enableDBWIN != 0
#define TC_MOVE (0)
#define TC_RESTORE (0)
#define TC_TRASH (0)
// NOTE: Testing must be done while running in Single-Window mode with terminal 
//       size >= 132 columns to avoid unsightly dialog overlap.
static NcDialog*  dbWin = NULL ;          // pointer to debugging window
static short      dbLines = ZERO,         // dimensions of debugging window
                  dbCols  = ZERO ;
static attr_t     dbColor = ZERO ;        // color attribute of debugging window
static winPos     dbwp( 1, 1 ) ;          // cursor position in debugging window
#if TC_MOVE != 0
static short      dbPOffset = 60 ;        // path-string offset (for beauty)
#endif // TC_MOVE
#endif   // enableDBWIN
#endif   // ENABLE_DEBUGGING_CODE

#define DEBUG_REDIRECT_TRASH2_Temp (0)    // FOR DEVELOPMENT ONLY

const char* const trashinfoHdr  = "[Trash Info]" ;
const char* const trashinfoPath = "Path=" ;
const char* const trashinfoDate = "DeletionDate=" ;
const char* const trashinfoExt  = ".trashinfo" ;
const short ddLEN = 20 ;      // length of deletion-date datestamp string

//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;

//* Control indices for tcDialog window *
enum tcDialogCtrls : short { detSE = ZERO, empPB, resPB, cloPB, dpcCOUNT } ;

//* Sort option for display of trashcan contents *
enum tcSortOption : short { tcsoDATE, tcsoNAME, tcsoSIZE } ;

//* Error type passed to tsfAlert() method *
enum tsfaCode : short
{
   tsfaDIR_WARN,           // warn user that source is a directory tree
   tsfaFILE_ERR,           // general error when moving a top-level file
   tsfaDIR_ERR,            // one or more files/dirnames in tree not trashed
   tsfaTRASH_ERR,          // Trashcan has become inaccessible (user deleted it)
   tsfaTRASH_DIR,          // if user is trying to trash files in Trashcan
   tsfaUNSUPP,             // if user requested an unsupported operation (not used)
} ;

//* Confirmation dialog size and position for Empty/Restore operations.*
const short confirmROWS = 9,
            confirm_YOFFSET = 2,
            confirm_XOFFSET = 2 ;

//* Local Prototypes *
static void tcSort ( ssetData& sData, tcDataInfo* tcdi, tcSortOption sOption ) ;
static bool tsfCreateLog ( const gString& srcPath, const gString& trgInfoPath, 
                           const gString& timeStamp ) ;
static void tcSummaryDisplay ( NcDialog* dp, winPos& wPos, UINT trashedItems, 
                               UINT trashedFiles, UINT64 trashedBytes, 
                               UINT64 trashSpace, UINT64 freeSpace, 
                               attr_t dColor, attr_t tColor ) ;
static int  tcSelectAll ( ssetData& sData, attr_t mask, bool sel ) ;
static int  tcSelectRecent ( ssetData& sData, tcDataInfo* tcdi, attr_t mask ) ;
static bool tcSameRTarget ( ssetData& sData, const tcDataInfo* tcdi, attr_t selMask ) ;
static bool tcFirstSelected ( ssetData& sData, attr_t selMask ) ;
static bool tcNextSelected ( ssetData& sData, attr_t selMask ) ;
static bool tcAlloc ( int itemCount, ssetData& sData, tcDataInfo*& tcdiPtr, 
                      char*& blkPtr, short dialogCols ) ;
static void tcFree ( ssetData& sData, tcDataInfo*& tcdiPtr, char*& blkPtr ) ;


//***********************
//* TrashcanConfigured  *
//***********************
//******************************************************************************
//* Query whether Move-to-Trashcan operations have been properly set up.       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if Trashcan operations are available                      *
//*          'false' if Trashcan operations have been disabled                 *
//******************************************************************************

bool FileDlg::TrashcanConfigured ( void )
{

   return this->trashcanOK ;

}  //* End TrashcanConfigured() *

//*************************
//* TrashcanSelectedFiles *
//*************************
//******************************************************************************
//* Move all 'selected' files in current window to the desktop environment's   *
//* Trashcan.                                                                  *
//*                                                                            *
//* Input  : (by reference, initial value ignored)                             *
//*          On return, contains number of files in selection list             *
//*                                                                            *
//* Returns: number of files moved to Trashcan                                 *
//*           If caller's 'selected' parameter == return value,                *
//*           then no errors                                                   *
//******************************************************************************
//* Programmer's Note: This is a complex operation requiring possible rename   *
//* of top-level files and directories AND creation of a '.trashinfo' file     *
//* indicating where the file originated. Contents of any subdirectory trees   *
//* are moved without rename. See module header for more detailed information. *
//******************************************************************************

UINT FileDlg::TrashcanSelectedFiles ( UINT& filesSelected )
{
   UINT  filesCanned = ZERO ;       // files successfully moved to Trashcan
   //* Number of previously-selected files *
   filesSelected = this->selInfo.Count + this->selInfo.subCount ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
   if ( dbWin == NULL )
      dbWin = this->Open_DBwindow ( dbColor, dbLines, dbCols ) ;
   dbwp = { 1, 1 } ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0
   #if DEBUG_REDIRECT_TRASH2_Temp != 0    // FOR DEBUGGING AND DEVELOPMENT ONLY
   //* For development and debug only: redirect files to our temporary *
   //* trashcan directory. Must be disabled for production build.      *
   const char* installPath = getenv ( "FMG_HOME" ) ;
   gString gs( "%s/Trash", installPath ) ;
   this->fmPtr->EnvExpansion ( gs ) ;
   gs.copy( this->trashDir, MAX_PATH ) ;
   #endif   // DEBUG_REDIRECT_TRASH2_Temp

   //* Cut 'selected' files (or file under cursor) to the clipboard *
   //* Note that source directory MAY NOT be the Trashcan itself.   *
   bool  srcEqTrashcan = this->tsfSourceEqTrashcan (),
         srcWritePerm ; // 'true' if user has write permission on source directory
   if ( this->trashcanOK && !srcEqTrashcan &&
        ((filesSelected = 
          this->MarkSelection ( this->deleteColor, srcWritePerm )) > ZERO) && 
        srcWritePerm )
   {
      //* Copy filenames of selected files to the clipboard *
      this->FillClipboard ( opCUT ) ;

      //* Update the status control to indicate captured clipboard contents *
      this->DisplayStatus () ;

      //* Test whether user has read and write permission on all the source    *
      //* files to be trashed. If the number of protected files > ZERO, then   *
      //* user will be asked whether we should proceed. If yes, attempt is made*
      //* to enable read and write access for all protected files in the list. *
      UINT  wpCount = ZERO, rpCount = ZERO ;
      bool overrideProtection = 
            this->SourcePermission ( wpCount, rpCount, true, true ) ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
      if ( dbWin != NULL )
      {
         gString gs ;
         gs.compose( L"wpCount:%u rpCount:%u ovrride:%hhd\n", 
                     &wpCount, &rpCount, &overrideProtection ) ;
         dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
      }
      #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

      if ( (wpCount == ZERO && rpCount == ZERO) || overrideProtection )
      {  //* We now have read/write access to all source files OR permission   *
         //* to do as much damage as possible. Initialize our call parameters. *
         pclParm pcl ;
         pcl.srcDir = clipBoard_srcPath ;
         this->fmPtr->CatPathFname ( pcl.trgDir, this->trashDir, trashFiles ) ;
         this->fmPtr->CatPathFname ( pcl.trgIDir, this->trashDir, trashInfo ) ;
         this->tsfConstructDeletionDate ( pcl.tStamp ) ;
         pcl.deleteSrc = true ;
         tnFName* infoPtr ;
         UINT  markedFiles, dirIndex = ZERO, fileIndex = ZERO,
               attemptedFiles = ZERO, movedFiles = ZERO, copiedFiles = ZERO ;
         bool  goodMove, ignoreErrors = false ;

         //* Walk through the list of files and directory trees, *
         //* moving each to the Trash.                           *
         for ( markedFiles = this->selInfo.Count ; markedFiles ; markedFiles-- )
         {
            //* For directory or directory tree *
            if ( this->deList[this->hIndex]->fType == fmDIR_TYPE )
            {
               infoPtr = &clipBoard[BNODE].nextLevel[dirIndex].dirStat ;
               pcl.srcTree = &clipBoard[BNODE].nextLevel[dirIndex++] ;
            }
            //* For non-directory files at top level *
            else
            {
               infoPtr = &clipBoard[BNODE].tnFiles[fileIndex++] ;
               pcl.srcTree = NULL ;
            }
            //* Disable user prompt if this is the last or only file in list,  *
            //* AND it is not a non-empty directory.                           *
            if ( (!ignoreErrors && markedFiles == 1) && 
                 ((infoPtr->fType != fmDIR_TYPE) || 
                  ((infoPtr->fType == fmDIR_TYPE) && 
                   ((pcl.srcTree != NULL) && (pcl.srcTree->totFiles == ZERO)))) )
               ignoreErrors = true ;

            //* Move the file or directory tree to trash *
            goodMove = this->tsfTopFile ( *infoPtr, pcl, ignoreErrors ) ;
            attemptedFiles += pcl.attCount ;
            copiedFiles    += pcl.cpyCount ;
            movedFiles     += pcl.delCount ;
            //* If an error has occurred AND user has aborted the operation.   *
            //* Note that application's refusal to process a 'special' file is *
            //* not considered an error.                                       *
            if ( ! goodMove && ! ignoreErrors &&
                 !(infoPtr->fType == fmCHDEV_TYPE || 
                   infoPtr->fType == fmBKDEV_TYPE || 
                   infoPtr->fType == fmSOCK_TYPE) )
               break ;
            this->NextSelectedItem () ;   // track to next 'selected' item
         }  // for(;;)

         //* Return the results to caller *
         filesCanned = movedFiles ;
      }
      else
      {  //* User has aborted the operation because one or more files is       *
         //* write-protected.                                                  *
         this->SetErrorCode ( ecSRCPROTECT, EACCES ) ;
      }
      //* Re-read directory contents (clipboard will be cleared) *
      this->RefreshCurrDir () ;
   }
   else
   {
      //* Move-to-Trashcan operations have been disabled (Trashcan not found), *
      //* in which case, user has already been told at least once so we only   *
      //* display a minor warning.    --OR--                                   *
      //* User does not have write permission on the source directory, for     *
      //* which we issue a stronger warning.  --OR--                           *
      //* User is trying to use the Trashcan as the source directory, in which *
      //* case he/she/it has already been chastized  --OR--                    *
      //* File selection failed because there are no files in this directory,  *
      //* which should be obvious even to a user.                              *
      this->DeselectFile ( true ) ; // deselect all files
      if ( ! this->trashcanOK )
         this->DisplayStatus ( ecNOTRASHCAN ) ;
      else if ( srcEqTrashcan )
         ;  // already alerted
      else if ( ! srcWritePerm )
         this->DisplayStatus ( ecREADONLY_S, EACCES ) ;
   }

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
   if ( dbWin != NULL )
   {
      gString gs ;
      dbWin->ClearLine ( dbLines - 2, false ) ;
      dbWin->WriteString ( (dbLines-2), 2, gs, dbColor ) ;
      dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                           " Press A Key ", this->cs.pf, true ) ;
      nckPause();
      delete dbWin ;
      dbWin = NULL ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

   return filesCanned ;

}  //* End TrashcanSelectedFiles () *

//***********************
//*     tsfTopFile      *
//***********************
//******************************************************************************
//* ** Private Method **                                                       *
//* Move a top-level file (and if a directory, all of its contents) to the     *
//* Trashcan.                                                                  *
//*                                                                            *
//*                                                                            *
//* Input  : srcInfo: (by reference) information on top-level source file      *
//*          pclParm structure (by reference)                                  *
//*            pcl.srcDir : base directory of source data on the clipboard     *
//*            pcl.trgDir : target trashcan directory for file                 *
//*            pcl.trgIDir: target trashcan directory for log                  *
//*            pcl.srcTree: if source is a directory name, then this points    *
//*                         to corresponding node in clipboard list,           *
//*                         else ignored                                       *
//*            pcl.tStamp : header and date-time of operation for log file     *
//*          ignoreErrors% : (by reference)                                     *
//*                         if 'true', then do not warn user about errors      *
//*                         if 'false' and an error occurs, then ask user      *
//*                          whether we should continue                        *
//*                                                                            *
//* Returns: 'true' if operation successful, else 'false'                      *
//******************************************************************************
//* Subdirectory names at the top level:                                       *
//* If top-level source directory is write accessable:                         *
//*    1) Copy the top-level directory file to 'Trash/files'.                  *
//*       a) If a file of that name already exists, copy-with-rename.          *
//*    2) Create a '.trashinfo' log file in 'Trash/info'.                      *
//*    3) If copy is successful, then copy directory contents.                 *
//*    4) If all source directory contents copied successfully, AND all source *
//*       contents deleted successfully, then delete source directory.         *
//* Else, source is unchanged and return an error.                             *
//*                                                                            *
//* Non-directory files at the top level:                                      *
//* If source file is write accessable:                                        *
//*    1) Copy the top-level non-directory file to 'Trash/files'.              *
//*       a) If a file of that name already exists, copy-with-rename.          *
//*    2) Create a '.trashinfo' log file in 'Trash/info'.                      *
//*    3) If copy is successful, then delete the source file.                  *
//* Else, source is unchanged and return an error.                             *
//******************************************************************************

bool FileDlg::tsfTopFile ( const tnFName& srcInfo, pclParm& pcl, bool ignoreErrors )
{
   //* Initialize remainder of caller's data *
   pcl.attCount = 1 ;      // attempting one, top-level file (but see below)
   pcl.cpyCount = pcl.delCount = ZERO ;

   //* If top-level file is not write accessable, OR is not read accessible,   *
   //* then return failure.                                                    *
   //* NOTE: This is an early return.                                          *
   //*       Esthetically, we prefer only one return statement in a method,    *
   //*       but in this case, the early return is cleaner.                    *
   if ( srcInfo.writeAcc == false || srcInfo.readAcc == false )
   {
      if ( srcInfo.fType == fmDIR_TYPE && pcl.srcTree != NULL )
      {  //* Count contents of directory as 'attempted' files *
         TreeNode treeAc ;    // accumulator for tree scan
         this->fmPtr->TreeNodeSummary ( pcl.srcTree, treeAc ) ;
         pcl.attCount = treeAc.totFiles ;
         #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
         if ( dbWin != NULL )
         {
            gString gs ;
            gs.compose( L"%-16s(wAcc:%hhd rAcc:%hhd):\n"
                         "  att:%2u cpy:%2u del:%2u\n",
                        pcl.srcTree->dirStat.fName, 
                        &srcInfo.writeAcc, &srcInfo.readAcc,
                        &pcl.attCount, &pcl.cpyCount, &pcl.delCount ) ;
            dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
         }
         #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0
      }
      #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
      else if ( dbWin != NULL )
      {
         gString gs ;
         gs.compose( L"%-16s(wAcc:%hhd rAcc:%hhd):\n",
                     srcInfo.fName, &srcInfo.writeAcc, &srcInfo.readAcc ) ;
         dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
      }
      #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0
      return ( false ) ;
   }  // (srcInfo.writeAcc==false || srcInfo.readAcc==false)

   gString srcPath,                 // path/filename of source file
           trgPath,                 // path/filename of target file
           trgInfoPath ;            // path/filename of target log file
   short cpyStatus = OK,            // status returned by copy operation
         delStatus = OK ;           // status returned by delete operation
   bool  logStatus = true,          // status returned by logfile creation
         movStatus = true,          // status returned by MoveDirTree method
         localAbort = false,        // if user aborted durin this iteration
         fileTrashed = false ;      // return value
   //* Create source path/filename *
   this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcInfo.fName ) ;

   //* Determine target file's name. If source name does not exist in target   *
   //* directory, then use the source filename. Otherwise, create a serialized *
   //* target name and increment it until an unused target filename is found.  *
   //* Log file shares the target filename but with an appended '.trashinfo'.  *
   fmFType  tType ;  // (file-type is ignored at this stage)
   this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.ustr(), srcInfo.fName ) ;
   bool targetExists = this->TargetExists ( trgPath, tType ) ;

   if ( targetExists )
   {
      wchar_t trgName[MAX_FNAME], trgExt[MAX_FNAME] ;
      gString gt ;
      this->fmPtr->ExtractFilename ( gt, trgPath ) ;
      gt.copy( trgName, MAX_FNAME ) ;
      short i = gt.gschars() - 1 ;
      for ( ; trgName[i] != PERIOD && trgName[i] != fSLASH && i > ZERO ; i-- ) ;
      if ( trgName[i] == PERIOD && i > ZERO )
      {
         trgName[i] = NULLCHAR ;
         this->fmPtr->ExtractExtension ( gt, trgPath ) ;
      }
      else
         gt.clear();
      gt.copy( trgExt, MAX_FNAME ) ;
      i = 2 ;                       // serialization value
      do
      {
         trgPath.compose( L"%S/%S.%hd%S", pcl.trgDir.gstr(), trgName, &i, trgExt ) ;
         targetExists = this->TargetExists ( trgPath, tType ) ;
         ++i ;
      }
      while ( targetExists ) ;
      this->fmPtr->ExtractFilename ( gt, trgPath ) ;
      this->fmPtr->CatPathFname ( trgInfoPath, pcl.trgIDir.gstr(), gt.gstr() ) ;
   }
   else
      this->fmPtr->CatPathFname ( trgInfoPath, pcl.trgIDir.ustr(), srcInfo.fName ) ;
   trgInfoPath.append( trashinfoExt ) ;
   
   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
   if ( dbWin != NULL )
   {
      gString gs ;
      gs.compose( L"Source: %S (wAcc:%hhd rAcc:%hhd)\n", 
                  &srcPath.gstr()[dbPOffset], &srcInfo.writeAcc, &srcInfo.readAcc ) ;
      dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor ) ;
      gs.compose( L"Target: %S\n", &trgPath.gstr()[29] ) ;
      dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor ) ;
      gs.compose( L"trgLog: %S\n", &trgInfoPath.gstr()[29] ) ;
      dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

   //* Subdirectory names at the top level *
   if ( srcInfo.fType == fmDIR_TYPE )
   {  //* If user has not already given permission for directory operations *
      if ( ! ignoreErrors )
      {
         ignoreErrors = this->tsfAlert ( srcInfo.fName, tsfaDIR_WARN ) ;
         if ( ! ignoreErrors )
            localAbort = true ;
      }
      
      if ( ignoreErrors != false &&
           (cpyStatus = this->CopyFile ( srcInfo, srcPath, trgPath )) == OK )
      {  //* Source successfully copied. Create log file,    *
         //* and if successful, then copy directory contents.*
         #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
         if ( dbWin != NULL )
         {
            gString gs ;
            this->fmPtr->ExtractFilename ( gs, trgPath ) ;
            gs.append( " (copied)\n" ) ;
            dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
         }
         #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

         if ( (logStatus = tsfCreateLog (srcPath, trgInfoPath, pcl.tStamp)) != false )
         {  //* Copy contents of this subdirectory to Trashcan (without rename)*
            //* and if copy to target is successful, delete the source file.   *
            #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
            if ( dbWin != NULL )
            {
               gString gs ;
               this->fmPtr->ExtractFilename ( gs, trgInfoPath ) ;
               gs.append( " (created)\n" ) ;
               dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
            }
            #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0
            --pcl.attCount ;  // avoid double count of base directory name
            pclParm pcl2 ;
            pcl2.srcDir  = srcPath ;
            pcl2.trgDir  = trgPath ;
            pcl2.srcTree = pcl.srcTree ;
            movStatus = tsfMoveDirTree ( pcl2 ) ;
            pcl.attCount += pcl2.attCount ;
            pcl.cpyCount += pcl2.cpyCount ;
            pcl.delCount += pcl2.delCount ;
            if ( movStatus != false )
               fileTrashed = true ;
            else     // error status
            {
               //* fileTrashed == false && movStatus == false *
               delStatus = ERR ;
            }
         }
      }
   }
   //* Non-directory files at the top level *
   else
   {
      if ( (cpyStatus = this->CopyFile ( srcInfo, srcPath, trgPath )) == OK )
      {  //* Source successfully copied. Create log file,   *
         //* and if successful, then delete the source file.*
         #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
         if ( dbWin != NULL )
         {
            gString gs ;
            this->fmPtr->ExtractFilename ( gs, trgPath ) ;
            gs.append( " (copied)\n" ) ;
            dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
         }
         #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

         if ( (logStatus = tsfCreateLog (srcPath, trgInfoPath, pcl.tStamp)) != false )
         {
            #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
            if ( dbWin != NULL )
            {
               gString gs ;
               this->fmPtr->ExtractFilename ( gs, trgInfoPath ) ;
               gs.append( " (created)\n" ) ;
               dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor ) ;
               gs = L"delete status: " ;
               dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
            }
            #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

            ++pcl.cpyCount ;     // copy and log creation were successful
            if ( (delStatus = this->tsfDeleteSource ( srcInfo, srcPath )) == OK )
            {
               ++pcl.delCount ;
               fileTrashed = true ;

               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
               if ( dbWin != NULL )
               {
                  gString gs ;
                  gs = L"OK\n" ;
                  dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
                  dbwp.xpos = 1 ;
               }
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0
            }
         }
         #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
         else
         {
            if ( dbWin != NULL )
            {
               gString gs ;
               gs = L"ERROR!\n" ;
               dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
            }
         }
         #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0
      }
   }
   //* Intra-operation error reporting *
   if ( ! fileTrashed && ! localAbort && ! ignoreErrors )
   {
      //* For unsupported-operation errors, user has already been warned.      *
      if ( cpyStatus != ecUNSUPP )
      {
         short errType ;
         if ( cpyStatus != OK || ! logStatus )
         {  //* If insufficient access to source file *
            if ( cpyStatus == EACCES )
               errType = tsfaFILE_ERR ;
            //* If source-to-target copy failed OR if creation of trashinfo  *
            //* file failed, then Trashcan has been compromised.             *
            //* (This is unlikely because Trashcan was verified on start-up.)*
            else
               errType = tsfaTRASH_ERR ;
         }

         //* If copy-to-target and log creation successful, but top-level      *
         //* delete failed, indicating insufficient user access, either to     *
         //* the source file OR source directory or its contents.              *
         //* Note: If movStatus == false, then a directory/contents error.     *
         else
            errType = movStatus == false ? tsfaDIR_ERR : tsfaFILE_ERR ;
         ignoreErrors = this->tsfAlert ( srcInfo.fName, errType ) ;
      }
   }
   return fileTrashed ;

}  //* End tsfTopFile() *

//***********************
//*   tsfMoveDirTree    *
//***********************
//******************************************************************************
//* ** Private Method **                                                       *
//* Copy a sub-directory tree to the Trashcan, and for each file successfully  *
//* copied, delete the source file.                                            *
//*    NOTE: This is a recursive method.                                       *
//*                                                                            *
//* Input  : pclParm structure (by reference)                                  *
//*            pcl.srcDir  : source directory path                             *
//*            pcl.trgDir  : target trashcan directory for file                *
//*            pcl.srcTree : pointer to TreeNode object describing subdir tree *
//*            pcl.attCount: (initial value ignored)                           *
//*                          receives number of move operations attempted      *
//*            pcl.cpyCount: (initial value ignored)                           *
//*                          receives number of source files copied to target  *
//*            pcl.delCount: (initial value ignored)                           *
//*                          receives number of source files deleted           *
//*                                                                            *
//* Returns: 'true' if operation successful, i.e. cpyCount==delCount==attCount *
//*          'false' if one or more copy and/or delete operations failed       *
//******************************************************************************

bool FileDlg::tsfMoveDirTree ( pclParm& pcl )
{
   bool treeMoved = false ;
   pcl.cpyCount = 1 ;               // caller copied the base directory
   pcl.attCount = ZERO ;            // no files attempted yet
   pcl.delCount = ZERO ;            // no files deleted yet
   UINT64 nodeBytes = ZERO ;        // (ignored)
   this->fmPtr->TreeNodeSummary ( pcl.srcTree, pcl.attCount, nodeBytes ) ;

   //* If files and directories within this directory, move them.*
   if ( pcl.attCount > 1 )
   {
      //* If subdirectories, travel down the tree, and move them *
      if ( pcl.srcTree->dirFiles > ZERO && pcl.srcTree->nextLevel != NULL )
      {
         pclParm pcl2 ;
         for ( UINT i = ZERO ; i < pcl.srcTree->dirFiles ; i++ )
         {  //* Create the target directories at this level, then              *
            //* Recursively call this method until we reach the lowest level   *
            //* of the TreeNode list, then process the directory contents in   *
            //* reverse order.                                                 *
            this->fmPtr->CatPathFname ( pcl2.srcDir, pcl.srcDir.ustr(), 
                                        pcl.srcTree->nextLevel[i].dirStat.fName ) ;
            this->fmPtr->CatPathFname ( pcl2.trgDir, pcl.trgDir.ustr(), 
                                        pcl.srcTree->nextLevel[i].dirStat.fName ) ;
            pcl2.srcTree = &pcl.srcTree->nextLevel[i] ;
            if ( (this->CopyFile ( pcl.srcTree->nextLevel[i].dirStat, 
                  pcl2.srcDir, pcl2.trgDir )) == OK )
            {  //* Move the directory tree *
               this->tsfMoveDirTree ( pcl2 ) ;
               pcl.cpyCount += pcl2.cpyCount ;
               pcl.delCount += pcl2.delCount ;
            }
         }
      }
      //* If non-directory files at this level, move them *
      if ( pcl.srcTree->tnFCount > ZERO && pcl.srcTree->tnFiles != NULL )
      {
         gString srcPath, trgPath ;
         tnFName* srcInfo ;
         for ( UINT i = ZERO ; i < pcl.srcTree->tnFCount ; i++ )
         {
            srcInfo = &pcl.srcTree->tnFiles[i] ;
            this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcInfo->fName ) ;
            this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.ustr(), srcInfo->fName ) ;
            if ( (this->CopyFile ( *srcInfo, srcPath, trgPath )) == OK )
            {
               ++pcl.cpyCount ;
               if ( (this->tsfDeleteSource ( *srcInfo, srcPath )) == OK )
                  ++pcl.delCount ;
            }
         }
      }
   }

   //* If source directory is now empty, delete it.*
   if ( pcl.delCount == (pcl.attCount - 1) )
   {
      if ( (this->tsfDeleteSource ( pcl.srcTree->dirStat, pcl.srcDir )) == OK )
      {
         ++pcl.delCount ;
         treeMoved = true ;
      }
   }

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_MOVE != 0
   if ( dbWin != NULL )
   {
      gString gs ;
      gs.compose( L"%-16s: att:%2u cpy:%2u del:%2u\n",
                  pcl.srcTree->dirStat.fName,
                  &pcl.attCount, &pcl.cpyCount, &pcl.delCount ) ;
      dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

   return treeMoved ;

}  //* End tsfMoveDirTree() *

//***********************
//*   tsfDeleteSource   *
//***********************
//******************************************************************************
//* Delete the specified source file.                                          *
//*                                                                            *
//*                                                                            *
//* Input  : srcInfo: information about source file                            *
//*          srcPath: path/filename of file to be deleted                      *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecUNSUPP        if operation not supported for file type.   *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) We pass the actual deletion to the dsfDeleteXX group of methods because *
//*    those methods will do customized error reporting in the form of         *
//*    info dialogs.                                                           *
//*                                                                            *
//* 2) Caller should have removed write protection if it was possible before   *
//*    calling this method. Therefore, if 'writeAcc' == false, then file is    *
//*    not deleted.                                                            *
//*                                                                            *
//* 3) Deletion of certain file types is not supported by this application,    *
//*    thus we report any attempt do do so. Note that since copying of those,  *
//*    certain file types is likewise unsupported, those should seldom if ever *
//*    arrive here.                                                            *
//*                                                                            *
//******************************************************************************

short FileDlg::tsfDeleteSource ( const tnFName& srcInfo, const gString& srcPath )
{
   short status = OK ;

   switch ( srcInfo.fType )
   {
      case fmREG_TYPE:     // regular file
      case fmLINK_TYPE:    // symbolic link file
      case fmFIFO_TYPE:    // FIFO file
         status = this->dsfDeleteRegFile ( srcInfo, srcPath ) ;
         break ;
      case fmDIR_TYPE:     // directory (empty directory only)
         status = this->dsfDeleteDirectory ( srcInfo, srcPath ) ;
         break ;
      case fmCHDEV_TYPE:   // character device file
         status = this->dsfDeleteChdevFile ( srcInfo, srcPath ) ;
         break ;
      case fmBKDEV_TYPE:   // block device file
         status = this->dsfDeleteBkdevFile ( srcInfo, srcPath ) ;
         break ;
      case fmSOCK_TYPE:    // socket file
         status = this->dsfDeleteSocketFile ( srcInfo, srcPath ) ;
         break ;
      case fmUNKNOWN_TYPE: // unknown file type
         status = this->dsfDeleteUnsuppFile ( srcInfo, srcPath ) ;
      default:
         status = ecUNSUPP ;
         break ;
   }

   return status ;

}  //* End tsfDeleteSource() *

//****************************
//* tsfConstructDeletionDate *
//****************************
//******************************************************************************
//* Construct the 'DeletionDate' entry for a '.trashinfo' log file.            *
//* ** Private Method **                                                       *
//* NOTE: If system clock is unavailable,                                      *
//*       we return the epoch date-time i.e. January 01, 1970 at 08:00:00      *
//*                                                                            *
//* Input  : timeStamp: (by reference) buffer to hold the date-time string     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::tsfConstructDeletionDate ( gString& timeStamp )
{
   localTime lt ;
   this->GetLocalTime ( lt ) ;
   timeStamp.compose( L"%s%04hu-%02hu-%02huT%02hu:%02hu:%02hu", 
                      trashinfoDate, &lt.year, &lt.month, &lt.date,
                      &lt.hours, &lt.minutes, &lt.seconds ) ;

}  //* End tsfConstructDeletionDate() *

//***********************
//* tsfSourceEqTrashcan *
//***********************
//******************************************************************************
//* Compare the current working directory with the two Trashcan directories.   *
//* If source == Trashcan, then call user a dumb-guy.                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if source==Trashcan, else, 'false'                         *
//******************************************************************************

bool FileDlg::tsfSourceEqTrashcan ( void )
{
   bool inDumpster = false ;        // return value

   gString trfDir, triDir ;
   this->fmPtr->CatPathFname ( trfDir, this->trashDir, trashFiles ) ;
   this->fmPtr->CatPathFname ( triDir, this->trashDir, trashInfo ) ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_MOVE != 0 && 0
   if ( dbWin != NULL )
   {
      gString gs ;
      gs.compose( L"trfDir:%S\ntriDir:%S\ncurDir:%s\n", 
                  trfDir.gstr(), triDir.gstr(), this->currDir ) ;
      dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_MOVE != 0

   if ( (trfDir.compare( this->currDir, true, (trfDir.gschars() - 1) ) == ZERO) ||
        (triDir.compare( this->currDir, true, (triDir.gschars() - 1) ) == ZERO) )
   {
      this->tsfAlert ( NULL, tsfaTRASH_DIR ) ;
      inDumpster = true ;
   }
   return inDumpster ;

}  //* End tsfSourceEqTrashcan() *

//***********************
//*     tsfAlert        *
//***********************
//******************************************************************************
//* Alert user of an error during move of specified file or directory tree.    *
//* Possible access violation or system error during the copy/delete process.  *
//*                                                                            *
//* Input  : srcName : pointer to name of file which was not renamed           *
//*          errType : type of error that occurred (see enum tsfaCode)         *
//*                                                                            *
//* Returns: 'true' if we are to continue processing remainder of list         *
//*          'false' if user aborted processing of file list                   *
//******************************************************************************
//* Types of errors reported:                                                  *
//* 1) Warning - directory tree as source. Always prompt for the first         *
//*    directory tree encountered by the operation. This is a friendliness     *
//*    issue.                                                                  *
//*    1a) If user wants to continue, then subsequent directory trees will be  *
//*        silently moved to the trash.                                        *
//*    1b) If user aborts, then remaining files and directories in the queue   *
//*        will not be processed.                                              *
//*                                                                            *
//* 2) Unable to access source file for copy or delete.                        *
//*    User's read/write permission on the file will have been verified before *
//*    the operation is attempted. Thus, if we get this error, it means that   *
//*    our read/write-permission test failed for some reason. This would be a  *
//*    pretty serious problem, so the first time it happens, ask user whether  *
//*    to continue with remaining files in the list.                           *
//*                                                                            *
//* 3) Operation not supported (special file). Errors of this type are         *
//*    reported at a lower level (see the CopyFile() method), so we won't see  *
//*    that here.                                                              *
//*                                                                            *
//* 4) Trashcan not properly configured. We should never see this because      *
//*    the main trashcan loop will not be executed if !this->trashcanOK        *
//*    4a) Unable to copy to target OR unable to create the '.trashinfo' log   *
//*        file. This should not happen because Trashcan configuration must    *
//*        be verified before starting the trashcan loop. If it happens, then  *
//*        we must ABORT.                                                      *
//*    4b) Source copied to trash AND '.trashinfo' file created, BUT unable    *
//*        to delete source. This would be bad because we would then have a    *
//*        spurious file in the Trashcan, which is not a devastating problem   *
//*        but is extremely untidy and we will ABORT.                          *
//*                                                                            *
//* NOTE: Directory warnings and file errors COULD BE handled with separate    *
//*       flags, but because file errors will be quite rare, we use a single   *
//*       flag i.e. the return value.                                          *
//******************************************************************************

bool FileDlg::tsfAlert ( const char* srcName, short errType )
{
   bool continueProcessing = true ;

   char  trLine2[INFODLG_WIDTH], trLine3[INFODLG_WIDTH], trLine4[INFODLG_WIDTH], 
         trLine5[INFODLG_WIDTH], trLine6[INFODLG_WIDTH], trLine7[INFODLG_WIDTH] ;
   const char* trashAlert[] = 
   { //1234567890123456789012345678901234567890123456789012 - (max line length)
      "  ALERT!  ALERT!  ",
      " ",
      trLine2,
      trLine3,
      trLine4,    // filename
      trLine5,
      trLine6,
      trLine7,
      NULL
   } ;
   attr_t trashAttr[] = 
   {
      this->cs.sd | ncbATTR, this->cs.sd, this->cs.sd, this->cs.sd,
      this->cs.em, this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd
   } ;
   const char* init_trName  = "                           " ;
   const char* spcMsg  = " " ;
   const char* dirMsg1 = "        The following file is a directory." ;
   const char* dirMsg2 = "      Are you sure that you want to move this" ;
   const char* dirMsg3 = "    directory and its contents to the Trashcan?" ;
   const char* filMsg1 = " The following file could not be moved to Trashcan!" ;
   const char* filMsg2 = "Do you want to process the remaining files in list?" ;
   const char* traMsg1 = "      Sorry, this operation is not permitted." ;
   const char* traMsg2 = "        To remove files from the Trashcan," ;
   const char* traMsg3 = "        please use the 'Empty Trash' option" ;
   const char* traMsg4 = "        or delete the files unconditionally." ;
   const char* tcfMsg1 = " System Trashcan directory has become inaccessible" ;
   //                     |..................................................|
   const char* tcfMsg2 = "            or the target disc is full." ;
   const char* tcfMsg3 = "  Operation must be terminated. Sorry about that." ;
   const char* conMsg1 = "   One or more files in the following directory:" ;
   const char* conMsg2 = "        could not be moved to the Trashcan." ;
   const char* unkMsg  = "     Unknown error(s) moving data to Trashcan." ;


   //* Set the messages according to the type of error *
   gString gs ;
   if ( errType == tsfaDIR_WARN )
   {
      gs = dirMsg1 ; gs.copy( trLine2, INFODLG_WIDTH ) ;
      gs = spcMsg ;  gs.copy( trLine3, INFODLG_WIDTH ) ; gs.copy( trLine5, INFODLG_WIDTH ) ;
      gs = dirMsg2 ; gs.copy( trLine6, INFODLG_WIDTH ) ;
      gs = dirMsg3 ; gs.copy( trLine7, INFODLG_WIDTH ) ;
   }
   else if ( errType == tsfaTRASH_DIR )
   {
      gs = traMsg1 ; gs.copy( trLine2, INFODLG_WIDTH ) ;
      gs = spcMsg ;  gs.copy( trLine3, INFODLG_WIDTH ) ;
      gs = traMsg2 ; gs.copy( trLine4, INFODLG_WIDTH ) ;
      gs = traMsg3 ; gs.copy( trLine5, INFODLG_WIDTH ) ;
      gs = traMsg4 ; gs.copy( trLine6, INFODLG_WIDTH ) ;
      trashAlert[7] = NULL ;
   }
   else if ( errType == tsfaDIR_ERR )
   {
      gs = conMsg1 ;
      gs.copy( trLine2, INFODLG_WIDTH ) ;
      gs = spcMsg ;
      gs.copy( trLine3, INFODLG_WIDTH ) ;
      gs.copy( trLine5, INFODLG_WIDTH ) ;
      gs = conMsg2 ;
      gs.copy( trLine6, INFODLG_WIDTH ) ;
      gs = filMsg2 ;
      gs.copy( trLine7, INFODLG_WIDTH ) ;
   }
   else if ( errType == tsfaTRASH_ERR )
   {  // (trash can fucked)
      gs = tcfMsg1 ;
      gs.copy( trLine2, INFODLG_WIDTH ) ;
      gs = tcfMsg2 ;
      gs.copy( trLine3, INFODLG_WIDTH ) ;
      gs = tcfMsg3 ;
      gs.copy( trLine4, INFODLG_WIDTH ) ;
      gs = spcMsg ;
      gs.copy( trLine5, INFODLG_WIDTH ) ;
      trashAlert[6] = NULL ;
   }
   else if ( errType == tsfaFILE_ERR )
   {
      gs = filMsg1 ;
      gs.copy( trLine2, INFODLG_WIDTH ) ;
      gs = spcMsg ;
      gs.copy( trLine3, INFODLG_WIDTH ) ;
      gs.copy( trLine5, INFODLG_WIDTH ) ;
      gs = filMsg2 ;
      gs.copy( trLine6, INFODLG_WIDTH ) ;
      *trLine7 = NULLCHAR ;
   }
   else  // ( errType == tsfaUNSUPP )
   {
      gs = unkMsg ;
      gs.copy( trLine2, INFODLG_WIDTH ) ;
      gs = spcMsg ;
      gs.copy( trLine3, INFODLG_WIDTH ) ;
      gs.copy( trLine5, INFODLG_WIDTH ) ;
      gs = filMsg2 ;
      gs.copy( trLine6, INFODLG_WIDTH ) ;
      *trLine7 = NULLCHAR ;
   }

   if ( errType == tsfaDIR_WARN || errType == tsfaFILE_ERR || 
        errType == tsfaDIR_ERR  || errType == tsfaUNSUPP )
   {
      //* Initialize the filename field,         *
      //* then center the filename in the field. *
      gs = srcName ;
      short totalCols = INFODLG_WIDTH - 4,
            nameCols  = gs.gscols(),
            padCols   = (totalCols - nameCols) / 2 ;
      if ( padCols < ZERO ) padCols = ZERO ;
      gs = init_trName ;
      gs.limitCols( padCols ) ;
      gs.append( srcName ) ;
      gs.copy( trLine4, totalCols ) ;

      //* Get user's response. *
      continueProcessing = this->DecisionDialog ( trashAlert, trashAttr ) ;
   }
   else
   {
      this->InfoDialog ( trashAlert ) ;
      continueProcessing = false ;
   }
   return   continueProcessing ;

}  //* End tsfAlert() *

//***********************
//*   ManageTrashcan    *
//***********************
//******************************************************************************
//* Allow user to view the contents of the Trashcan, delete (unlink) items     *
//* and restore item(s) to to their original location.                         *
//*                                                                            *
//* Input  : restore  : (optional, 'false' by default)                         *
//*                     if 'false', user will interactively manage Trashcan    *
//*                     if 'true',  restore the most recent move-to-trash      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ManageTrashcan ( bool restore )
{

   if ( this->trashcanOK )
      this->tcDialog ( restore ) ;
   else
      this->tsfAlert ( NULL, tsfaTRASH_ERR ) ;

}  //* End ManageTrashcan() *

//***********************
//*      tcDialog       *
//***********************
//******************************************************************************
//* Private Method: Called by ManageTrashcan().                                *
//* 1) Open the empty/restore trash dialog and display summary contents.       *
//* 2) User may delete items or restore items from the trashcan.               *
//*                                                                            *
//*                                                                            *
//* Input  : restore  : (optional, 'false' by default)                         *
//*                     if 'false', user will interactively manage Trashcan    *
//*                     if 'true',  restore the most recent move-to-trash      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) The CWD is temporarily changed to the Trashcan directories so we can    *
//*    access the stats for the files that live there, HOWEVER, data for those *
//*    directories is not displayed directly (or if it is, it is probably too  *
//*    fast for the user to see it.                                            *
//*                                                                            *
//* 2) The clipboard is used implicitly for data manipulation and is cleared   *
//*    before return.                                                          *
//*                                                                            *
//* 3) Items in the 'Trash/file' directory ('trashedItems') SHOULD BE equal to *
//*    items in the 'Trash/info' directory ('infoCount'). If not, it means     *
//*    that Trashcan data are partially corrupted.                             *
//*     a) For selective trash empty operation, only matching file pairs       *
//*        will be processed.                                                  *
//*     b) For restore operations, only matching file pairs will be processed. *
//*     c) However, if ALL trash is emptied, then any orphaned or corrupted    *
//*        files will be removed because ALL files in BOTH subdirectories will *
//*        be removed.                                                         *
//*                                                                            *
//*  4) Hotkey selection: Although none of the controls in this dialog have    *
//*     assigned hotkeys, if the mouse interface is active, the 'hidden'       *
//*     hotkeys are in effect. To maintain logical flow control, we have to    *
//*     manage the moose input.                                                *
//*     a) The Pushbutton controls can receive focus (and activation) via      *
//*        hotkey; however, this is inappropriate under some circumstances.    *
//*        See notes after return from 'tcScroll' about managing program flow. *
//*     b) The ScrollExt control cannot receive focus via hotkey becasuse it   *
//*        is inactive except when under edit.                                 *
//*                                                                            *
//******************************************************************************

void FileDlg::tcDialog ( bool restore )
{
   const short  MIN_TRASH_WIDTH = 78 ; // minimum dialog width
   const char* const contextHelp[dpcCOUNT] = // context-help messages
   { //---------+---------+---------+---------+-|
      "Select item(s) to be removed or restored.\n"
      "   Select: SPACE    Select All: CTRL+A   \n"
      "     Sort: CTRL+S          End: TAB      \n",
      "Delete selected items from the Trashcan.\n",
      "Restore selected items from the Trashcan\n"
      " to the original location.              ",
      "Close the Trashcan dialog.              ",
   } ;

   short    ulY = this->fulY,
            ulX = this->fulX,
            dialogRows = this->dRows - ulY,
            dialogCols = (this->fMaxX >= MIN_TRASH_WIDTH ? 
                          this->fMaxX : MIN_TRASH_WIDTH) ;
   attr_t   dColor = this->cs.sb,               // dialog interior color
            tColor = this->cs.bb & ~ncrATTR,    // status data text color
            hColor = this->cs.em ;              // context help message color

   //* Re-position the dialog if necessary *
   while ( (ulX > this->dulX) && ((ulX + dialogCols) > (this->dulX + this->dCols)) )
      --ulX ;

   InitCtrl ic[dpcCOUNT] = 
   {
   {  //* 'DETAIL' Scroll Ext control - - - - - - - - - - - - - - -      detSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      12,                           // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(dialogRows - 12),       // lines:     control lines
      dialogCols,                   // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cs.bb,                  // nColor:    non-focus border color
      this->cs.bb,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
        " Original File Location - - -[ ITEM LIST ]- - - - -  ItemSize  DeleteDate ",
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a) 
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[empPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'EMPTY TRASH' pushbutton  - - - - - - - - - - - - - - - - -    empPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      10,                           // ulY:       upper left corner in Y
      17,                           // ulX:       
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      " EMPTY TRASH ",              // dispText:
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[resPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'RESTORE ITEM' pushbutton - - - - - - - - - - - - - - - - -    resPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[empPB].ulY,                // ulY:       upper left corner in Y
      short(ic[empPB].ulX + ic[empPB].cols + 2), // ulX:       
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      " RESTORE ITEM ",             // dispText:
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cloPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -    cloPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[resPB].ulY,                // ulY:       upper left corner in Y
      short(ic[resPB].ulX + ic[resPB].cols + 2), // ulX:       
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      "    CLOSE    ",              // dispText:
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Save path of currently-display data because we will        *
   //* (probably) return. Then create path to '.trashinfo' files. *
   gString userCwd( this->currDir ),// CWD on entry
           returnCwd = userCwd,     // CWD on return to caller
           restoredItem, statusMsg, // housekeeping for 'restore' parameter
           infoPath ;
   this->fmPtr->CatPathFname ( infoPath, this->trashDir, trashInfo ) ;

   //* Scan the contents of the Trashcan *
   //* 1) summary contents               *
   //* 2) detailed contents              *
   tcSortOption sortOption = tcsoDATE ;   // display-data sort option
   tcSummaryInfo  tcSummary ;             // summary data
   this->tcSummaryScan ( tcSummary ) ;    // do summary scan
   UINT   origItems = tcSummary.items ;
   char*     blkPtr = NULL ;
   ssetData  sData( NULL, NULL, tcSummary.items, ZERO, false ) ;

   tcDataInfo* tcdi = NULL ;     // pointer to array of sorted name/date objects
   UINT  filesRemoved = ZERO,    // files deleted from Trashcan
         filesRestored = ZERO,   // files restored from Trashcan
         filesProcessed = ZERO,  // return value
         goodiCount = ZERO ;     // number of '.trashinfo' files read successfully
   int   srcItemCount = ZERO ;   // items identified for restoration
   bool  res2Cwd = false ;       // 'true' if alt target (CWD) used


   //* If Trashcan is empty, disable unneeded controls *
   if ( tcSummary.items == ZERO )
   {
      ic[empPB].active = ic[resPB].active = ic[detSE].active = false ;
   }
   //* Else, get details on each item in the Trashcan *
   //* and create a display string for each item.     *
   else
   {
      this->ClearClipboard () ;     // clear the clipboard

      goodiCount = this->tcDetailScan ( blkPtr, sData, tcdi, dialogCols, sortOption ) ;
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogRows,     // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->cs.bb,    // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Give dialog a title *
      dp->SetDialogTitle ( "  MANAGE TRASHCAN  ", this->cs.em ) ;

      //* Make a visual connection between the *
      //* dialog and the parent application.   *
      dp->WriteChar ( 0, 0, wcsLTEE, this->cs.bb ) ;
      if ( dialogCols == this->fMaxX )
         dp->WriteChar ( 0, dialogCols - 1, wcsRTEE, this->cs.bb ) ;

      winPos wPos( 2, 2 ),    // Y/X display positioning
             wpSummary, wpProcessed ;
      short  icIndex = ZERO ; // index of control with input focus

      //* Is there something in the Trashcan ? *
      if ( tcSummary.items == ZERO )
      {
         while ( (icIndex = dp->NextControl ()) != cloPB ) ;
         wPos = dp->WriteString ( wPos.ypos, dialogCols/2-15, 
                                  " Trashcan Is Currently Empty ", tColor ) ;
      }
      else
      {
         //* Make a visual connection for the dctSCROLLEXT control *
         //* that overlaps the dialog window's border.             *
         //* (if control not instantiated, this call does nothing) *
         cdConnect cdConn ;
         cdConn.ul2Left = true ;
         cdConn.ur2Right = true ;
         cdConn.connection = true ;
         dp->ConnectControl2Border ( detSE, cdConn ) ;

         //* Display the static data *
         const char* const tcHead = 
         {
            "  Trash Dir : \n"
            "      Items : \n"
            "Total Files : \n"
            "  File Size : \n"
            "Size on Disc: \n"
            "Free Space  : "
         } ;
         wPos = dp->WriteParagraph ( wPos, tcHead, dColor ) ;
         wPos.ypos = 2 ;

         //* Display the Trashcan path (trim if necessary) *
         //* and display summary statistics.               *
         gString  pgs( this->trashDir ) ;
         this->TrimPathString ( pgs, (dialogCols - 18) ) ;
         dp->WriteString ( wPos, pgs, tColor ) ;
         ++wPos.ypos ;
         wpSummary = wPos ;
         wpProcessed = wPos ; wpProcessed.ypos += 5 ;
         tcSummaryDisplay ( dp, wPos, tcSummary.items, tcSummary.files, 
                            tcSummary.bytes, tcSummary.tSpace, tcSummary.fSpace, 
                            dColor, tColor ) ;

         if ( tcSummary.items == tcSummary.iFiles && goodiCount == tcSummary.iFiles )
         {
            LineDef  lDef(ncltHORIZ, ncltDUAL, wPos.ypos - 1, wPos.xpos, 
                          40, this->cs.sd ) ;
            dp->DrawLine ( lDef ) ;
         }
         else  // alert user if Trashcan data corrupt
            dp->WriteString ( wPos.ypos - 1, wPos.xpos, 
                              "NOTE: Trashcan data partially corrupted.", dColor ) ;

         //* Initialize the data for the scrolling control *
         dp->SetScrollextText ( detSE, sData ) ;

         //* If call was to restore most recently trashed item(s):   *
         //* a) Re-position the focus to 'Restore' pushbutton.       *
         //* b) Select all items that share the newest deletion date.*
         //* c) Refresh the display.                                 *
         //* d) Push the Enter key into the input stream so 'Restore'*
         //*    button will be activated.                            *
         if ( restore != false )
         {
            while ( (icIndex = dp->NextControl ()) != resPB ) ;
            srcItemCount = tcSelectRecent ( sData, tcdi, this->selectAttr ) ;
            dp->RefreshScrollextText ( detSE, false ) ;
            wkeyCode wkpush( nckENTER, wktFUNKEY ) ;
            dp->UngetKeyInput ( wkpush ) ;
         }
      }

      //* Make everything visible *
      dp->RefreshWin () ;

      uiInfo   Info ;               // user interface data returned here
      bool     done = false ;       // loop control

      while ( ! done )
      {
         //* Display context help and process totals *
         if ( origItems > ZERO )
         {
            dp->ClearLine ( wPos.ypos, false, wPos.xpos ) ;
            dp->ClearLine ( wPos.ypos + 1, false, wPos.xpos ) ;
            dp->ClearLine ( wPos.ypos + 2, false, wPos.xpos ) ;
            dp->WriteParagraph ( wPos, contextHelp[icIndex], hColor ) ;
            if ( filesRemoved > ZERO || filesRestored > ZERO )
            {
               winPos wp = wpProcessed ;
               dp->ClearLine ( wp.ypos, false ) ;
               gString gsOut, gsi ;
               if ( filesRemoved > ZERO )
               {
                  gsi.formatInt( filesRemoved, FI_MAX_FIELDWIDTH, true ) ;
                  gsOut.compose( L"Files Removed: %S  ", gsi.gstr() ) ;
                  wp = dp->WriteString ( wp, gsOut, dColor ) ;
               }
               if ( filesRestored > ZERO )
               {
                  gsi.formatInt( filesRestored, FI_MAX_FIELDWIDTH, true ) ;
                  gsOut.compose( L"Files Restored: %S  ", gsi.gstr() ) ;
                  dp->WriteString ( wp, gsOut, dColor ) ;
               }
            }
            dp->RefreshWin () ;
         }

         //************************
         //** Pushbutton control **
         //************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //** 'EMPTY' Pushbutton **
               if ( icIndex == empPB )
               {  //* Delete 'selected' items and associated '.trashinfo' files*
                  UINT rmf = this->tcRemoveConfirm ( dp, sData, tcdi ) ;
                  filesRemoved += rmf ;
                  filesProcessed += rmf ;
               }

               //** 'RESTORE' Pushbutton **
               else if ( icIndex == resPB )
               {  //* Restore 'selected' items *
                  res2Cwd = false ;
                  UINT rsf = this->tcRestoreConfirm ( dp, sData, tcdi, srcItemCount, 
                                                      userCwd, statusMsg, &res2Cwd ) ;

                  //* If we have restored most-recent move to trash *
                  if ( restore != false )
                  {
                     //* Make the target restoration directory the new CWD.    *
                     //*        (set at the bottom of this method)             *
                     //* Note that on entry, source items are sorted in reverse*
                     //* date order, so we 'know' that the first item was      *
                     //* restored. This is not necessarily the first restored  *
                     //* item displayed because user's directory-data view     *
                     //* is sorted according to its own criteria.              *
                     if ( ! res2Cwd && rsf > ZERO )
                        this->fmPtr->ExtractPathname ( returnCwd, tcdi[ZERO].trgPath ) ;

                     //* Restored target filename to be highlighted on return *
                     if ( rsf > ZERO )
                        this->fmPtr->ExtractFilename ( restoredItem, tcdi[ZERO].trgPath ) ;
                     done = true ;
                  }
               }

               //** 'CLOSE' Pushbutton **
               else if ( icIndex == cloPB )
                  done = true ;

               //* If we aren't done yet, re-scan   *
               //* and re-display trashcan contents.*
               if ( ! done )
               {
                  //* Re-scan Trashcan contents and update display.*
                  while ( (icIndex = dp->NextControl ()) != resPB ) ;// focus to 'CLOSE'
                  wPos = wpSummary ;
                  this->tcSummaryScan ( tcSummary ) ;
                  tcSummaryDisplay ( dp, wPos, tcSummary.items, tcSummary.files, 
                                     tcSummary.bytes, tcSummary.tSpace, 
                                     tcSummary.fSpace, dColor, tColor ) ;
                  goodiCount = 
                     this->tcDetailScan ( blkPtr, sData, tcdi, dialogCols, sortOption ) ;
                  dp->SetScrollextText ( detSE, sData ) ;
                  if ( tcSummary.items == ZERO )
                  {  //* Trashcan is now empty, disable unneeded controls.*
                     while ( (icIndex = dp->NextControl ()) != cloPB ) ;
                     dp->ControlActive ( empPB, false ) ;
                     dp->ControlActive ( resPB, false ) ;
                     dp->ControlActive ( detSE, false ) ;
                  }
               }
            }
         }

         //************************
         //** Scrollext control  **
         //************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {  //* Let user select file(s) for processing *
            srcItemCount = this->tcScroll ( dp, Info, sData, tcdi, (short&)sortOption ) ;
         }

         //*************************************
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         //*************************************
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if(OpenWindow())
   tcFree ( sData, tcdi, blkPtr ) ; // release our dynamic data
   if ( dp != NULL )
      delete ( dp ) ;               // close the window

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   //* Trashcan operations may have moved the CWD.       *
   //* Set CWD to the directory from which the dialog    *
   //* was originally invoked.                           *
   //* For Undo option:                                  *
   //* a) Set CWD to target directory of Undo operation. *
   //* b) Highlight the last item restored.              *
   //* c) Display specified status message (if any).     *
   //* d) If no items restored, return to user's CWD.    *
   this->SetDirectory ( returnCwd ) ;
   if ( (restoredItem.gschars()) > 1 )
      this->HilightItemByName ( restoredItem ) ;
   if ( (restore != false) && ((statusMsg.gschars()) > 1) )
   {
      attr_t color = dtbmNFcolor ;
      dtbmData msgData( statusMsg.gstr(), &color ) ;
      this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;
   }

   //* Be sure any operation data is removed from the clipboard *
   this->ClearClipboard () ;

}  //* End tcDialog() *

//***********************
//*      tcScroll       *
//***********************
//******************************************************************************
//* Allow user to scroll through the data for the dctSCROLLEXT control and     *
//* 'select' a file or files for processing.                                   *
//*                                                                            *
//* Input  : dp      : pointer to open dialog window                           *
//*          info    : (by reference) receives results of user input           *
//*          sData   : display data                                            *
//*          tcdi    : pointer to array of tcDataInfo objects containing paths *
//*                    to source and target data                               *
//*          sOption : (by reference) display data sort option                 *
//*                    updated through user interaction                        *
//*                    Note: This is actually a member of enum tcSortOption,   *
//*                          which is defined at module scope and cannot be    *
//*                          used as a parameter for a class method.           *
//*                                                                            *
//* Returns: number of 'selected' items                                        *
//******************************************************************************

int FileDlg::tcScroll ( NcDialog* dp, uiInfo& info, 
                        ssetData& sData, tcDataInfo* tcdi, short& sOption )
{
   attr_t*  dcPtr = (attr_t*)sData.dispColor ;  // local color-attribute pointer
   int   selected = ZERO ;                // return value
   bool  done = false ;                   // loop control

   //* Refresh the display data and make the highlight visible.*
   dp->RefreshScrollextText ( detSE ) ;

   //* Count the number of 'selected' items.*
   for ( int i = ZERO ; i < sData.dispItems ; i++ )
   {
      if ( sData.dispColor[i] & this->selectAttr )
         ++selected ;
   }

   short icIndex = dp->GetCurrControl () ;
   do
   {  //* Get user input *
      icIndex = dp->EditScrollext ( info ) ;

      //* If the input focus has shifted, we're done.*
      if ( icIndex != detSE )
         break ;

      //* If command to re-sort the data *
      if ( info.wk.type == wktFUNKEY && info.wk.key == nckC_S )
      {
         switch ( sOption )
         {
            case tcsoDATE: sOption = tcsoNAME ; break ;
            case tcsoNAME: sOption = tcsoSIZE ; break ;
            case tcsoSIZE: sOption = tcsoDATE ; break ;
         } ;
         tcSort ( sData, tcdi, (tcSortOption)sOption ) ;
         dp->RefreshScrollextText ( detSE ) ;
         continue ;
      }

      if ( info.wk.type == wktFUNKEY && 
           (info.wk.key == nckTAB || info.wk.key == nckSTAB) )
      {
         done = true ;
      }
      else if ( (info.wk.type == wktPRINT && info.wk.key == nckSPACE) ||
                (info.wk.type == wktFUNKEY && 
                 (info.wk.key == nckENTER || info.wk.key == nckpENTER)) )
      {  //* If an item selected or de-selected *
         if ( !(dcPtr[info.selMember] & this->selectAttr) )
         {
            dcPtr[info.selMember] |= this->selectAttr ;
            ++selected ;
            dp->MoveScrollextHighlight ( detSE, nckDOWN ) ;
         }
         else
         {
            dcPtr[info.selMember] &= ~this->selectAttr ;
            --selected ;
            dp->MoveScrollextHighlight ( detSE, nckDOWN ) ;
         }
         dp->RefreshScrollextText ( detSE ) ;
      }
      else if ( (info.wk.type == wktFUNKEY) && (info.wk.key == nckC_A) )
      {  //* If select or de-select ALL items (multi-select only) *
         selected = tcSelectAll ( sData, this->selectAttr, bool(selected == ZERO) ) ;
         dp->RefreshScrollextText ( detSE ) ;
      }
   }
   while ( ! done ) ;
   dp->RefreshScrollextText ( detSE, false );// make highlight invisible

   return selected ;

}  //* End tcScroll() *

//***********************
//*    tuSummaryScan    *
//***********************
//******************************************************************************
//* Get summary statistics for items in the Trashcan.                          *
//*                                                                            *
//* Input  : tcSummary : (by reference) receives summary data                  *
//*                                                                            *
//* Returns: 'true'  if scan successful                                        *
//*          'false' if processing error                                       *
//******************************************************************************
//* Note on calculating disc space occupied by trashed files:                  *
//*  This is only an approximate value based on the fact that files occupy an  *
//*  integral number of logical allocation blocks on the disc. Files smaller   *
//*  than the size of a block will occupy one block. This includes all the     *
//*  'xxx.trashinfo' files and probably at least some of the trashed files.    *
//*  Files that are larger than one block, will likely waste some amount of    *
//*  space in the last block allocated to that file.                           *
//*                                                                            *
//* Our Solution:                                                              *
//*  We don't care about the exact amount, so we calculate it as:              *
//*   tcSummary.bytes +                                                        *
//*   tcSummary.files * average wasted space per file +                        *
//*   one full block for each 'xxx.trashinfo' file                             *
//******************************************************************************

bool FileDlg::tcSummaryScan ( tcSummaryInfo& tcSummary )
{
   fileSystemStats fsStats ;     // logical block size && free disc space
   gString trashPath( this->trashDir ),
           filePath, infoPath ;  // Trashcan subdir paths
   UINT64 itemSize ;
   bool  status = false ;        // return value

   tcSummary.reset() ;           // initialize caller's data

   //* 1) Create the paths to the Trashcan subdirectories.                 *
   //* 2) Get filesystem freespace and allocation-block size.              *
   //* 3) Get number of files in the 'info' subdirectory.                  *
   //* 4) Get number of items AND size of items in 'files' subdirectory.   *
   //* 5) Get number of files AND size of files in 'files' subdirectory.   *
   this->fmPtr->CatPathFname ( filePath, this->trashDir, trashFiles ) ;
   this->fmPtr->CatPathFname ( infoPath, this->trashDir, trashInfo ) ;
   if ( ((this->fmPtr->GetFilesystemStats ( trashPath, fsStats )) == OK)
        &&
        (this->fmPtr->DirectorySummary ( infoPath.ustr(), 
                                         tcSummary.iFiles, itemSize ))
        &&
        (this->fmPtr->DirectorySummary ( filePath.ustr(), 
                                         tcSummary.items, itemSize ))
        &&
        (this->fmPtr->DirectorySummary ( filePath.ustr(), 
                                         tcSummary.files, tcSummary.bytes, true ))
      )
   {
      tcSummary.fSpace = fsStats.freeBytes ;
      if ( tcSummary.bytes > ZERO && tcSummary.files > ZERO ) // (avoid math problems)
      {
         double avgfSize = (double)tcSummary.bytes / (double)tcSummary.files,
                fWaste = (avgfSize >= fsStats.fblockSize) ? 
                         (tcSummary.bytes * 0.05) : 
                         ((double)tcSummary.files * (double)fsStats.fblockSize - avgfSize) ;
         tcSummary.tSpace = (UINT64)(tcSummary.bytes + fWaste + 
                              tcSummary.iFiles * fsStats.fblockSize) ;
      }
      status = true ;
   }
   return status ;

}  //* End tcSummaryScan() *

//***********************
//*    tcDetailScan     *
//***********************
//******************************************************************************
//* Build display data for the dctSCROLLEXT control.                           *
//* 1) Release any previous dynamic-memory allocations.                        *
//* 2) Change the application's CWD to 'Trash/info' subdirectory.              *
//*    (Directory Window IS NOT updated)                                       *
//* 3) Retrieve the directory tree information about the CWD.                  *
//* 4) Allocate dynamic memory to store the formatted display data and         *
//*    tracking information.                                                   *
//* 5) Format the data.                                                        *
//*                                                                            *
//* Input  : blkPtr  : (by reference) receives a pointer to an area for        *
//*                    storing display strings                                 *
//*          sData   : (by reference) receives arrays of text pointers and     *
//*                    color attributes                                        *
//*          tcdi    : target receives a pointer to an array of                *
//*                    tcDataInfo objects for tracking item name and deletion  *
//*                    datestamp                                               *
//*          dlgCols : number of display columns in dialog                     *
//*          sOption : display data sort option                                *
//*                    Note: This is actually a member of enum tcSortOption,   *
//*                          which is defined at module scope and cannot be    *
//*                          used as a parameter for a class method.           *
//*                                                                            *
//* Returns: number of '.trashinfo' files scanned                              *
//*          s/b == to number in summary scan.                                 *
//*          if not, then Trashcan data corrupted.                             *
//******************************************************************************

UINT FileDlg::tcDetailScan ( char*& blkPtr, ssetData& sData, 
                             tcDataInfo*& tcdi, short dlgCols, short sOption )
{
   //************************************
   //** Initialize the data containers **
   //************************************
   const short maxPATHWIDTH = 52,
               fsWIDTH = 8 ;
   gString  sBuff,               // display-string construction
            iBuff,               // fixed-width integer formatting
            filePath,            // Trashcan subdir paths
            infoPath,
            infPath,             // path/filename of target '.trashinfo' file
            filPath,             // path/filename of source item in 'files' dir
            trgPath,             // receives original file location
            delDate ;            // receives deletion-date string
   this->fmPtr->CatPathFname ( filePath, this->trashDir, trashFiles ) ;
   this->fmPtr->CatPathFname ( infoPath, this->trashDir, trashInfo ) ;
   infPath = trashinfoExt ;      // provides length of filename extension
   short extLen = infPath.gschars() ;
   fmFType  trgType ;            // file type of target file
   UINT     trgFiles ;           // number of files contained in target (inclusive)
   UINT64   trgSize ;            // size of target file
   // NOTE: Although the data retrieved in these variables are not displayed at 
   //       this time, it does point to dynamically-allocated memory in the 
   //       FMgr class, so be aware.
   UINT      infoCount = ZERO ;  // number of elements in info list
   tnFName** tnfList = NULL ;    // pointer to array of pointers to file stats
   DispData dData( (this->fMaxX - 2) ) ;

   char     dirChar = 'd',
            spcChar = ' ' ;
   UINT goodiCount = ZERO ;      // return value

   //* Free any previous dynamic allocations.*
   tcFree ( sData, tcdi, blkPtr ) ;

   //* Set the CWD to 'Trash/info' directory so we can get the file data.*
   //* BUT do not display the data at this time.                         *
   if ( (this->fmPtr->CdPath ( infoPath, dData )) != false )
   {  //* Update our data members *
      this->fmPtr->GetCurrDir ( this->currDir ) ;
      infoPath = this->currDir ;
      dData.getData( this->deList, this->textData, this->colorData, 
                     this->fileCount, this->dirSize ) ;  
      this->hIndex = ZERO ;      // index first item in display list
      tnfList = this->deList ;   // local copies
      infoCount = this->fileCount ;

      //* Allocate dynamic storage for display and tracking *
      //* data and create display strings.                  *
      //*  (If no items to display, create a dummy item.)   *
      if ( (tcAlloc ( infoCount, sData, tcdi, blkPtr, dlgCols )) )
      {
         char* strPtr = blkPtr ; // local string pointer
         attr_t*  dcPtr = (attr_t*)sData.dispColor ;  // local color-attribute pointer
         if ( infoCount > ZERO )
         {
            for ( int i = ZERO ; i < sData.dispItems ; i++ )
            {
               this->fmPtr->CatPathFname ( infPath, infoPath.ustr(), tnfList[i]->fName ) ;
               if ( this->tcDecodeLog ( infPath, trgPath, delDate, 
                                        trgFiles, trgSize, trgType ) )
               {  //* Initialize tracking data *
                  infPath.copy( tcdi[i].infoPath, MAX_PATH ) ; // path of '.trashinfo' file
                  filPath.compose( "%S/%s", filePath.gstr(), tnfList[i]->fName ) ;
                  filPath.limitChars( (filPath.gschars()) - extLen ) ;
                  filPath.copy( tcdi[i].srcPath, MAX_PATH ) ;  // path of source in 'Trash/files'
                  trgPath.copy( tcdi[i].trgPath, MAX_PATH ) ;  // original-target path
                  tcdi[i].fType = trgType ;                    // file type
                  tcdi[i].files = trgFiles ;                   // files contained in item
                  tcdi[i].size = trgSize ;                     // size of item (bytes)
                  this->tcDecodeDate ( delDate, tcdi[i].tDate ) ;// item deletion date
                  //* Retain only the date: yyyy-mm-dd for display.*
                  delDate.limitChars( 10 ) ;
         
                  //* Trim the path string if necessary *
                  if ( trgPath.gscols() > maxPATHWIDTH )
                     this->TrimPathString ( trgPath, maxPATHWIDTH ) ;
         
                  //* Format the item's size *
                  iBuff.formatInt( trgSize, fsWIDTH ) ;
         
                  ++goodiCount ;
               }
               else
               {  //* If file read fails, set dummy display data (unlikely) *
                  iBuff   = L"--------" ;
                  delDate = L"----------" ;
                  trgPath = L"** corrupted file **" ;
                  //* This will force corrupted data to the bottom of display list *
                  tcdi[i].srcPath[0] = tcdi[i].trgPath[0] = 
                  tcdi[i].infoPath[0] = '~' ;
                  tcdi[i].srcPath[1] = tcdi[i].trgPath[1] = 
                  tcdi[i].infoPath[1] = NULLCHAR ;
                  tcdi[i].fType = fmREG_TYPE ;
                  tcdi[i].files = 1 ;
                  tcdi[i].size = ZERO ;
                  tcdi[i].tDate.reset() ;
               }
               sBuff.compose( L"%-52S  %S  %S %c", 
                              trgPath.gstr(), iBuff.gstr(), delDate.gstr(), 
                              (char*)(trgType == fmDIR_TYPE ? &dirChar : &spcChar)) ;
               sBuff.copy( strPtr, gsDFLTBYTES ) ;
               sData.dispText[i] = strPtr ;
               strPtr += sBuff.utfbytes() ;
               dcPtr[i] = this->ftColor[fmREG_TYPE] ;
            }
         }
         else     // create a dummy display item
         {
            sBuff = " - - - - - - - - - - - - - - - - - - - - - - -"
                    " - - - - - - - - - - - - - -   " ;
            sBuff.limitCols( this->fMaxX - 2 ) ;
            sBuff.copy( strPtr, gsDFLTBYTES ) ;
            sData.dispText[ZERO] = strPtr ;
            dcPtr[ZERO] = this->ftColor[fmREG_TYPE] ;
         }
      }

      //* Sort the resulting data.*
      tcSort ( sData, tcdi, (tcSortOption)sOption ) ;
   }
   return goodiCount ;

}  //* End tcDetailScan() *

//***********************
//*     tcDecodeLog     *
//***********************
//******************************************************************************
//* Read contents of a 'x.trashinfo' file and return its contents.             *
//* File's existance has been verified by caller.                              *
//*                                                                            *
//* Input  : srcPath  : path/filename of '.trashinfo' file to read             *
//*          trgPath  : receives path/filename of original file                *
//*          delDate  : receives string representation of deletion date/time   *
//*          trgFiles : receives file count for item ( file (not the info file)*
//*                      for non-dirs==1, or for dirs==total file count        *
//*          trgSize  : receives size of targeted item (not the info file)     *
//*                      for non-dirs==file size, or for dirs==total size      *
//*          trgType  : receives file type of target                           *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false if file not found or invalid data format                   *
//******************************************************************************

bool FileDlg::tcDecodeLog ( const gString& srcPath, gString& trgPath, 
                            gString& delDate, UINT& trgFiles,
                            UINT64& trgSize, fmFType& trgType )
{
   bool status = false ;      // return value
   //* Initialize caller's data *
   trgPath.clear() ;
   delDate.clear() ;
   trgFiles = ZERO ;
   trgSize = ZERO ;
   trgType = fmREG_TYPE ;

   //* Verify that the target matching the specified '.trashinfo' file exists. *
   //* 1) extract the filename from srcPath                                    *
   //* 2) strip the filename extension (this is tricky, beware)                *
   //* 3) construct path of matching target                                    *
   //* 4) get target's size and file type.                                     *
   gString tPath(trashinfoExt), tName ;
   this->fmPtr->ExtractFilename ( tName, srcPath ) ;
   tName.limitChars( tName.gschars() - (tPath.gschars()) ) ;
   tPath.compose( L"%s/%s/%S", this->trashDir, trashFiles, tName.gstr() ) ;
   tnFName fStats ;
   if ( (status = this->TargetExists ( tPath.ustr(), fStats )) )
   {
      trgSize = fStats.fBytes ;
      trgFiles = 1 ;
      if ( (trgType = fStats.fType) == fmDIR_TYPE )
      {  //* Get total size and file count of directory contents *
         UINT   dirCount ;
         UINT64 dirSize ;
         this->fmPtr->DirectorySummary ( tPath.ustr(), dirCount, dirSize, true ) ;
         trgFiles += dirCount ;
         trgSize += dirSize ;
      }

      //* Read the '.trashinfo' file and decode its contents *
      status = false ;
      ifstream ifs( srcPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {  //* A valid file has 3 lines, 1) header, 2) Path, 3) DelDate *
         char  lineData[gsDFLTBYTES] ;
         ifs.getline( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() && (!strncmp ( lineData, trashinfoHdr, 13 )) )
         {
            ifs.getline( lineData, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() )
            {
               trgPath = &lineData[strlen(trashinfoPath)] ;
               ifs.getline( lineData, gsDFLTBYTES, NEWLINE ) ;
               if ( ifs.good() )
               {
                  delDate = &lineData[strlen(trashinfoDate)] ;
                  status = true ;
               }
            }
         }
         ifs.close() ;
      }
   }
   return status ;

}  //* End tcDecodeLog() *

//***********************
//*    tcDecodeDate     *
//***********************
//******************************************************************************
//* Parse the deletion-date string from the '.trashinfo' file.                 *
//*                                                                            *
//* Input  : trashedDate : (by reference) original deletion-date string        *
//*          tDate       : (by reference) receives decoded datestamp           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The string SHOULD BE in the following format: yyyy-mm-ddThh:mm:ss          *
//* If it is not, we return a dummy datestamp.                                 *
//******************************************************************************

void FileDlg::tcDecodeDate ( const gString& trashedDate, localTime& tDate )
{
   int convCount = swscanf ( trashedDate.gstr(), L"%hu-%hu-%huT%hu:%hu:%hu",
                             &tDate.year, &tDate.month, &tDate.date, 
                             &tDate.hours, &tDate.minutes, &tDate.seconds ) ;
   if ( convCount == 6 )   // initialize the epoch-time variables
      this->fmPtr->EncodeEpochTime ( tDate ) ;
   else                    // invalid source data format
      tDate.reset() ;

}  //* End tcDecodeDate() *

//***********************
//*   tcRemoveConfirm   *
//***********************
//******************************************************************************
//* Ask for user confirmation, then:                                           *
//* Delete the specified item(s) from the Trashcan AND associated              *
//* '.trashinfo' files.                                                        *
//*                                                                            *
//* Input  : dpp     : pointer to open dialog window                           *
//*          sData   : (by reference) display data which includes flags on all *
//*                    'selected' items                                        *
//*          tcdi    : pointer to array of tcDataInfo objects containing paths *
//*                    to source and target data                               *
//*                                                                            *
//* Returns: number of FILES deleted                                           *
//*          (not the number of items)                                         *
//******************************************************************************

UINT FileDlg::tcRemoveConfirm ( NcDialog* dpp, 
                                const ssetData& sData, const tcDataInfo* tcdi )
{
   //* Get dimensions of caller's dialog. Set size and position of our dialog.*
   short dialogRows, dialogCols ;
   dpp->GetDialogDimensions ( dialogRows, dialogCols ) ;
   dialogRows = confirmROWS ;          // dialog rows
   dialogCols -= 4 ;                   // dialog columns
   winPos dlgOffset = dpp->GetDialogPosition () ;
   dlgOffset.ypos += confirm_YOFFSET ;
   dlgOffset.xpos += confirm_XOFFSET ;
   const attr_t dColor = this->cs.sb,  // dialog interior color
                wColor = this->cs.em ; // warning message color

   const char* const tFinalMsg = 
      "WARNING: This operation cannot be reversed!\n"
      "         Are you sure that you want to delete\n"
      "         the selected items from the Trashcan?" ;
   const char* const tNoselMsg = 
      "No items have been selected for deletion.\n"
      "Please select one or more items and try again." ;

   enum tctiControls : short
   { delPB = ZERO, canPB, dpcCOUNT } ;
   InitCtrl ic[dpcCOUNT] = 
   {
      {  //* 'EMPTY' pushbutton  - - - - - - - - - - - - - - - - - - -   delPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogRows - 2),        // ulY:       upper left corner in Y
         short(dialogCols / 2 - 10),   // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " DELETE ",                   // dispText:
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[canPB],                   // nextCtrl:  link in next structure
      },
      {  //* 'CANCEL' pushbutton   - - - - - - - - - - - - - - - - - -   canPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[delPB].ulY,                // ulY:       upper left corner in Y
         short(ic[delPB].ulX + ic[delPB].cols + 3 ),// ulX:
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " CANCEL ",                   // dispText:
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      },
   } ;
   UINT deletedFiles = ZERO ;    // return value

   //* Save the display for the parent dialog window *
   dpp->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogRows,     // number of display lines
                       dialogCols,     // number of display columns
                       dlgOffset.ypos, // Y offset from upper-left of parent dialog
                       dlgOffset.xpos, // X offset from upper-left of parent dialog
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->cs.bb,    // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Give dialog a title *
      dp->SetDialogTitle ( "  DELETE ITEMS FROM TRASHCAN  ", this->cs.em ) ;

      //* Display stats for item(s) to be deleted *
      winPos wp( 1, 2 ) ;
      UINT selItems = ZERO, selFiles = ZERO ;
      UINT64 selBytes = ZERO ;
      for ( int i = ZERO ; i < sData.dispItems ; i++ )
      {
         if ( sData.dispColor[i] & this->selectAttr )
         {
            ++selItems ;
            selBytes += tcdi[i].size ;
            selFiles += tcdi[i].files ;
         }
      }
      gString gsOut, gsItems, gsFiles, gsBytes ;
      gsItems.formatInt( selItems, 13, true ) ;
      gsFiles.formatInt( selFiles, 13, true ) ;
      gsBytes.formatInt( selBytes, 7, true ) ;
      gsOut.compose( L"Delete all 'selected' items from the Trashcan\n"
                      " %S items (%S files, %S bytes)\n", 
                      gsItems.gstr(), gsFiles.gstr(), gsBytes.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

      //* Danger! Danger, Will Robinson! *
      if ( selItems > ZERO )
         dp->WriteParagraph ( wp, tFinalMsg, wColor ) ;
      else
      {
         dp->WriteParagraph ( wp, tNoselMsg, wColor ) ;
         dp->NextControl () ;
         dp->ControlActive ( delPB, false ) ;
      }

      //* Make everything visible *
      dp->RefreshWin () ;

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == delPB )
               {
                  dp->SetDialogObscured() ;           // save dialog display

                  //* Tag user's selection of items for this view *
                  this->tcMarkSelection ( sData, tcdi, true, opDELETE ) ;

                  //* Delete selected files from 'Trash/info' directory *
                  this->tcDeleteSelectedFiles () ;

                  //* Set the CWD to 'Trash/files' directory *
                  //* so we can get the file data.           *
                  UINT selectedSourceFiles = ZERO ;
                  DispData dData( (this->fMaxX - 2) ) ;
                  gString filePath ;
                  this->fmPtr->CatPathFname ( filePath, this->trashDir, trashFiles ) ;
                  if ( (this->fmPtr->CdPath ( filePath, dData )) != false )
                  {  //* Update our data members *
                     this->fmPtr->GetCurrDir ( this->currDir ) ;
                     dData.getData( this->deList, this->textData, this->colorData, 
                                    this->fileCount, this->dirSize ) ;  
                     this->hIndex = ZERO ;      // index first item in display list
                     //* Tag user's selection of items for this view *
                     this->tcMarkSelection ( sData, tcdi, false, opDELETE ) ;
                     selectedSourceFiles = clipBoard[BNODE].totFiles ;
                     //* Delete selected items from 'Trash/files' directory *
                     //* and return number of source files deleted.         *
                     deletedFiles = this->tcDeleteSelectedFiles () ;
                  }

                  dpp->RefreshWin () ;          // refresh parent dialog
                  dpp->SetDialogObscured() ;    // re-save parent dialog
                  dp->RefreshWin () ;           // restore dialog display
                  if ( deletedFiles < selectedSourceFiles )
                  {  //* Alert user that something went wrong.*
                     //*       (this is fairly unlikely)      *
                     icIndex = dp->NextControl () ;   // focus to 'Cancel' button
                     wp = { short(dialogRows - 3), 3 } ;
                     dp->ClearLine ( wp.ypos - 3 ) ;
                     dp->ClearLine ( wp.ypos - 2 ) ;
                     dp->ClearLine ( wp.ypos - 1 ) ;
                     UINT undeletedFiles = selectedSourceFiles - deletedFiles ;
                     gsOut.compose( "ERROR! %u Trashcan files deleted, "
                                    "but unable to delete %u files", 
                                    &deletedFiles, &undeletedFiles ) ;
                     dp->WriteString ( wp, gsOut, wColor, true ) ;
                     nckPause();
                  }
               }
               done = true ;
            }
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   if ( dp != NULL )
      delete ( dp ) ;               // close the window

   //* Restore display of parent dialog window *
   dpp->RefreshWin () ;

   return deletedFiles ;

}  //* End tcRemoveConfirm() *

//*************************
//* tcDeleteSelectedFiles *
//*************************
//******************************************************************************
//* Delete from the Trashcan all files that have been marked as 'selected'.    *
//* This method is called ONLY by the 'tcRemoveConfirm' method.                *
//* It is called twice for each empty-the-trash operation: once for the        *
//* selected files in the 'Trash/info' directory, and once for the selected    *
//* items in the 'Trash/files' directory.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: returns number of files deleted                                   *
//******************************************************************************

UINT FileDlg::tcDeleteSelectedFiles ( void )
{
   UINT deletedFiles = ZERO ;

   UINT  markedFiles, dirIndex = ZERO, fileIndex = ZERO ;
   for ( markedFiles = this->selInfo.Count ; markedFiles ; markedFiles-- )
   {
      //* For directory or directory tree *
      if ( this->deList[this->hIndex]->fType == fmDIR_TYPE )
      {
         if ( clipBoard[BNODE].dirFiles > dirIndex 
              && clipBoard[BNODE].nextLevel != NULL )
         {
            TreeNode* tnPtr = &clipBoard[BNODE].nextLevel[dirIndex] ;
            if ( tnPtr != NULL )
            {
               UINT errCount = ZERO ;  // not used at this level
               deletedFiles += 
                  this->dsfDeleteDirTree ( tnPtr, clipBoard_srcPath, errCount ) ;
            }
         }
         ++dirIndex ;
      }
      //* For non-directory files at top level *
      else
      {
         deletedFiles += 
            this->dsfDeleteFileList ( &clipBoard[BNODE].tnFiles[fileIndex], 
                                      1, clipBoard_srcPath ) ;
         ++fileIndex ;
      }
      this->NextSelectedItem () ;   // track to next 'selected' item
   }  // for(;;)
   return deletedFiles ;

}  //* End tcDeleteSelectedFiles() *

//***********************
//*  tcRestoreConfirm   *
//***********************
//******************************************************************************
//* Ask for user confirmation, then:                                           *
//* Restore the specified item from the Trashcan to its original position.     *
//* This will move the source to target (with original filename)               *
//*   AND                                                                      *
//* will delete the associated '.trashinfo' file.                              *
//*                                                                            *
//* Input  : dpp     : pointer to open dialog window                           *
//*          sData   : (by reference) display data for trashcan contents with  *
//*                    'selected' items indicated                              *
//*                    Note: We modify only the highlight position.            *
//*          tcdi    : pointer to array of tcDataInfo objects containing paths *
//*                    to source and target data                               *
//*                      sData.hlIndex references the first (or only) item to  *
//*                      be restored                                           *
//*          rCount  : number of items to be restored                          *
//*          userCwd : (by reference) contains the filespec of the directory   *
//*                    the user _thinks_ is the CWD. May be used as an         *
//*                    alternate restoration target.                           *
//*                      Important Note: On entry, the _actual_ CWD is already *
//*                                      somewhere in the trashcan.            *
//*          statMsg : (by reference) receives operation status message        *
//*          res2Cwd : pointer to a flag which indicates to caller whether     *
//*                     a) item restored to its original position ('false')    *
//*                     b) item restored to path in 'userCwd' ('true')         *
//*                                                                            *
//* Returns: number of FILES restored (not the number of items)                *
//******************************************************************************
//* Notes:                                                                     *
//* -- Before restoring the item(s), we verify that:                           *
//*    a) target directory for the restore exists                              *
//*    b) user has write access to target directory                            *
//*    c) there are no existing targets that would be overwritten.             *
//* -- For multi-item restore, if one of the above errors occurs,              *
//*    then NONE of the items will be restored.                                *
//*                                                                            *
//* -- When restoring multiple items, there is a logical problem in reporting  *
//*    the restoration target directory. We can only report a single target    *
//*    directory, OR a general "many targets" message. If all targets          *
//*    originated in the same directory, then we can display the target. Else, *
//*    display the generic message.                                            *
//* -- User may optionally request that the item(s) be restored to the current *
//*    working directory (CWD) instead of being restored to the original       *
//*    position.                                                               *
//* -- Note that because we jump among the various open dialogs and move from  *
//*    one directory to another, the display may appear a bit jumpy on slow    *
//*    hardware. With a reasonably fast machine, the screen updates are almost *
//*    faster than the eye can see.                                            *
//*                                                                            *
//******************************************************************************

UINT FileDlg::tcRestoreConfirm ( NcDialog* dpp, ssetData& sData, tcDataInfo* tcdi, 
                                 int rCount, const gString& userCwd, 
                                 gString& statMsg, bool* res2Cwd )
{
   const char* const tNoselMsg = 
      "No items have been selected for restoration.\n"
      "Please select one or more items and try again." ;
   const char* multiTarget = "  Multiple Targets Specified  " ;

   //* Get dimensions of caller's dialog. Set size and position of our dialog.*
   short dialogRows, dialogCols ;
   dpp->GetDialogDimensions ( dialogRows, dialogCols ) ;
   dialogCols -= 4 ;                   // dialog columns
   dialogRows = confirmROWS ;          // dialog rows
   winPos dlgOffset = dpp->GetDialogPosition () ;
   dlgOffset.ypos += confirm_YOFFSET ;
   dlgOffset.xpos += confirm_XOFFSET ;
   const short tbWidth = dialogCols - 4 ; // columns for text box control
   const attr_t dColor = this->cs.sb,  // dialog interior color
                wColor = this->cs.em ; // context help message color
   gString  gsOut ;                    // output formatting
   UINT  itemsRestored = ZERO,         // items wholly or partially restored
         unrestoredFiles = ZERO,       // files left unrestored (still in trash)
         filesRestored = ZERO ;        // return value

   enum tcriControls : short
   { resPB = ZERO, canPB, cwdRB, trgTB, dpcCOUNT } ;
   InitCtrl ic[dpcCOUNT] = 
   {
      {  //* 'RESTORE' pushbutton  - - - - - - - - - - - - - - - - - -   resPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogRows - 2),        // ulY:       upper left corner in Y
         short(dialogCols / 2 - 10),   // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         9,                            // cols:      control columns
         " RESTORE ",                  // dispText:
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[canPB],                   // nextCtrl:  link in next structure
      },
      {  //* 'CANCEL' pushbutton   - - - - - - - - - - - - - - - - - -   canPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[resPB].ulY,                // ulY:       upper left corner in Y
         short(ic[resPB].ulX + ic[resPB].cols + 3 ),// ulX:
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " CANCEL ",                   // dispText:
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[cwdRB],                   // nextCtrl:  link in next structure
      },
      {  //* 'CWD Target' radio button    - - - - - - - - - - - - - - -  cwdRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 5-wide
         false,                        // rbSelect:  initial value
         ic[canPB].ulY,                // ulY:       upper left corner in Y
         short(ic[canPB].ulX + ic[canPB].cols + 3), // ulX:
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->cs.em,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Restore to CWD",             // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[trgTB],                   // nextCtrl:  link in next structure
      },
      {  //* 'target path' Textbox  - - - - - - - - - - - - - - - - -    trgTB *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         2,                            // ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         tbWidth,                      // cols:      control columns
         NULL,                         // dispText:  target path/filename
         this->cs.tn,                  // nColor:    non-focus color
         this->cs.tf,                  // fColor:    focus color
         #if LINUX_SPECIAL_CHARS != 0
         tbPathLinux,                  // filter: valid filespec chars (incl. Linux "special")
         #else    // BASIC FILESPEC FILTER
         tbPathName,                   // filter:    valid filespec characters
         #endif   // BASIC FILESPEC FILTER
         NULL,                         // label:     
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    read-only control
         NULL,                         // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogRows,     // number of display lines
                       dialogCols,     // number of display columns
                       dlgOffset.ypos, // Y offset from upper-left of parent dialog
                       dlgOffset.xpos, // X offset from upper-left of parent dialog
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->cs.bb,    // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
   if ( dbWin == NULL )
      dbWin = this->Open_DBwindow ( dbColor, dbLines, dbCols ) ;
   dbwp = { 1, 1 } ;
   gString dbgs ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

   if ( (dp->OpenWindow()) == OK )
   {
      //* Give dialog a title *
      dp->SetDialogTitle ( "  RESTORE ITEM FROM TRASHCAN  ", this->cs.em ) ;

      //* Display source info and target path/filename. If necessary, *
      //* shift path/filename until tail of string is displayed.      *
      winPos   wpStat( 3, 2 ),      // source stats position
               wpMsg( 4, 2 ) ;      // error message position
      gString  gsTmp,               // output formatting
               altTarget,           // alternate restore target
               errMsg ;             // validation error message
      UINT     srcFiles = ZERO ;    // number of source files (not items) to be restored
      short    icIndex = ZERO ;     // index of control with input focus

      //* Get a working copy of first 'selected' item *
      tcFirstSelected ( sData, this->selectAttr ) ;
      tcDataInfo tcItem = tcdi[sData.hlIndex] ;

      if ( rCount == 1 )      // one item being restored
      {
         //* For a single item, display the full filespec. *
         altTarget = tcItem.trgPath ;
         dp->SetTextboxText ( trgTB, altTarget ) ;
         dp->DisplayTextboxTail ( trgTB ) ;

         gsOut.compose( "Restore selected %s to:", 
                        tcItem.fType == fmDIR_TYPE ? "directory tree" : "file" ) ;
         dp->WriteString ( 1, 2, gsOut, dColor ) ;
         srcFiles = tcItem.files ;
         gsTmp.formatInt( tcItem.size, 6, true ) ;
         gsOut.compose( L" %S bytes : "
                         "trashed on %04hu-%02hu-%02hu at %02hu:%02hu:%02hu",
                         gsTmp.gstr(), &tcItem.tDate.year, &tcItem.tDate.month, 
                         &tcItem.tDate.date, &tcItem.tDate.hours, 
                         &tcItem.tDate.minutes, &tcItem.tDate.seconds ) ;

         if ( tcItem.fType == fmDIR_TYPE )
         {
            gsTmp.formatInt( tcItem.files, 5, true ) ;
            gsTmp.insert( L' ' ) ;
            gsTmp.append( " files," ) ;
            gsOut.insert( gsTmp.gstr() ) ;
         }
         dp->WriteString ( wpStat, gsOut, dColor ) ;
      }
      else                    // restoring multiple items
      {
         gsOut.compose( "Restore %d items to:", &rCount ) ;
         dp->WriteString ( 1, 2, gsOut, dColor ) ;
         if ( rCount > ZERO )
         {
            //* For multiple items with the same target, display target path.  *
            //* For multiple targets, display a generic message.               *
            if ( (tcSameRTarget ( sData, tcdi, this->selectAttr )) != false )
               this->fmPtr->ExtractPathname ( altTarget, tcItem.trgPath ) ;
            else
               altTarget = multiTarget ;
            dp->SetTextboxText ( trgTB, altTarget ) ;
            dp->DisplayTextboxTail ( trgTB ) ;

            UINT64 srcBytes = ZERO ;
            for ( int i = sData.hlIndex ; i < (sData.hlIndex + rCount) ; ++i )
            {
               srcFiles += tcdi[i].files ;
               srcBytes += tcdi[i].size ;
            }
            gString gsf( srcFiles, 5, true ) ;
            gsTmp.formatInt( srcBytes, 6, true ) ;
            gsOut.compose( " %S files, %S bytes : "
                           "trashed on %04hu-%02hu-%02hu at %02hu:%02hu:%02hu",
                           gsf.gstr(), gsTmp.gstr(), 
                           &tcItem.tDate.year, &tcItem.tDate.month, 
                           &tcItem.tDate.date, &tcItem.tDate.hours, 
                           &tcItem.tDate.minutes, &tcItem.tDate.seconds ) ;
            dp->WriteString ( wpStat, gsOut.gstr(), dColor ) ;
         }
         else
         {
            dp->WriteString ( wpStat, " 0 files, 0 bytes", dColor ) ;
            dp->WriteParagraph ( wpMsg, tNoselMsg, wColor ) ;
            dp->NextControl () ;
            dp->ControlActive ( resPB, false ) ;
            dp->ControlActive ( cwdRB, false ) ;
         }
      }

      //* Make everything visible *
      dp->RefreshWin () ;

      uiInfo   Info ;               // user interface data returned here
      bool     done = false ;       // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == resPB )
               {
                  //* Determine whether target(s) already exist, *
                  //* AND whether containing directory exists,   *
                  //* AND whether user has write access to it.   *
                  gString atPath ;     // alternate target path (if any)
                  dp->GetRadiobuttonState ( cwdRB, *res2Cwd ) ;
                  if ( *res2Cwd != false )
                     atPath = userCwd ;   // CWD is alternate target directory

                  if ( (this->tcrValidateTarget ( sData, tcdi, atPath, errMsg )) != false )
                  {
                     UINT filesDeleted = ZERO,  // number of source files deleted
                          frTemp = ZERO,        // restored item count
                          fdTemp = ZERO ;       // deleted files per source item

                     dp->SetDialogObscured () ; // save the dialog's display data

                     //* Position the highlight on first selected item, *
                     //* then process the list of selected items.       *
                     tcFirstSelected ( sData, this->selectAttr ) ;
                     do
                     {
                        #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
                        dbgs.compose( "*** hlIndex:%d\n", &sData.hlIndex ) ;
                        dbwp = dbWin->WriteParagraph ( dbwp, dbgs, dbColor, true ) ;
                        #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

                        frTemp = this->tcRestoreItem ( tcdi[sData.hlIndex], fdTemp ) ;
                        if ( frTemp > ZERO )
                           ++itemsRestored ;
                        filesRestored += frTemp ;
                        filesDeleted += fdTemp ;
                     }
                     while ( (tcNextSelected ( sData, this->selectAttr )) != false ) ;

                     dp->RefreshWin () ;     // restore saved display data

                     if ( (filesRestored != srcFiles) || 
                          (filesRestored != filesDeleted) )
                     {  //* Alert user that something went wrong.*
                        //*       (this is fairly unlikely)      *
                        icIndex = dp->NextControl () ;   // focus to 'Cancel' button
                        unrestoredFiles = srcFiles - filesRestored ;
                        UINT undeletedFiles = srcFiles - filesDeleted ;
                        if ( filesRestored == ZERO )
                        {
                           gsOut = "ERROR! Unable to restore selected item(s)." ;
                           if ( rCount > 1 )
                              gsOut.insert( L's', (gsOut.find( L'.' )) ) ;
                        }
                        else if ( filesRestored == srcFiles )
                        {  // Programmer's Note: This scenario is very unlikely.
                           // It would mean that some files in trash are write-protected.
                           gsOut.compose( "All items successfully restored, but %u "
                                          "Trashcan files not removed.", 
                                          &undeletedFiles ) ;
                        }
                        else
                        {
                           gsOut.compose( "ERROR! %u files restored, "
                                          "but %u files not restored",
                                          &filesRestored, &unrestoredFiles ) ;
                        }
                        dp->RefreshWin () ;
                        dp->WriteString ( wpMsg, gsOut, wColor, true ) ;
                        nckPause();
                     }
                  }        // if(validTarget)
                  else     // validation error
                  {  //* Display the error message, then send *
                     //* focus to 'Cancel' and wait for user  *
                     //* to finish reading the error message. *
                     dp->WriteParagraph ( wpMsg, errMsg, wColor, true ) ;
                     while ( (icIndex = dp->PrevControl ()) != canPB ) ;
                     nckPause();
                  }
                  done = true ;
               }
               else if ( Info.ctrlIndex == canPB )
                  done = true ;
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.isSel != false )    // if button state is SET
               {
                  gString tName ;
                  this->fmPtr->ExtractFilename ( tName, tcItem.trgPath ) ;
                  if ( rCount == 1 )   // single source item to alternate target
                     altTarget.compose( "%S/%S", userCwd.gstr(), tName.gstr() ) ;
                  else                 // multiple source items to user's CWD
                     altTarget = userCwd ;
               }
               else                          // if button state is RESET
               {
                  if ( rCount == 1 )
                     altTarget = tcItem.trgPath ;
                  else
                  {
                     if ( (tcSameRTarget ( sData, tcdi, this->selectAttr )) != false )
                        this->fmPtr->ExtractPathname ( altTarget, tcItem.trgPath ) ;
                     else
                        altTarget = multiTarget ;
                  }
               }
               dp->SetTextboxText ( trgTB, altTarget ) ;
               dp->DisplayTextboxTail ( trgTB ) ;
            }
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {  /* This is currently a view-only Textbox, no user access. */ }

         //* Move focus to appropriate control *
         if ( ! done )
         {
            if ( ! Info.viaHotkey )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
            dp->DisplayTextboxTail ( trgTB ) ;  // beautify path/filename display
         }
      }     // while(!done)
   }

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
   if ( dbWin != NULL )
   {
      dbWin->ClearLine ( dbLines - 2, false ) ;
      dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                           " Press A Key ", this->cs.pf, true ) ;
      nckPause();
      delete dbWin ;
      dbWin = NULL ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

   if ( dp != NULL )
      delete ( dp ) ;               // close the window

   //* Restore display of parent dialog window *
   dpp->RefreshWin () ;

   //* Return summary display data for operation results. *
   gString gsItems( itemsRestored, 5, true ),
           gsFiles( filesRestored, 5, true ) ;
   statMsg.compose( " %S items (%S files) restored.",
                  gsItems.gstr(), gsFiles.gstr() ) ;
   if ( itemsRestored < (UINT)rCount || unrestoredFiles > ZERO )
   {
      if ( itemsRestored > ZERO )
      {
         UINT unItems = rCount - itemsRestored ;
         gsItems.formatInt( unItems, 5, true ) ;
         gsFiles.formatInt( unrestoredFiles, 5, true ) ;
         statMsg.append( " (ERROR: %S items, %S files)",
                       gsItems.gstr(), gsFiles.gstr() ) ;
      }
      else
         statMsg = " Restore from Trashcan cancelled." ;
   }
   return filesRestored ;

}  //* End tcRestoreConfirm() *

//***********************
//*  tcrValidateTarget  *
//***********************
//******************************************************************************
//* Validate the pending restore-from-trash operation:                         *
//* 1) Verify that there are no existing target items. Do not allow overwrite  *
//*    of existing data.                                                       *
//* 2) Verify that user has write access to the containing directory for the   *
//*    targets.                                                                *
//* 3) If all items validated, AND if alternate target (CWD) specified:        *
//*    a) Set the 'trgPath' of each selected item to the filespec for          *
//*       restoration to the alternate target.                                 *
//*                                                                            *
//* Input  : sData   : (by reference) display data for trashcan contents with  *
//*                    'selected' items indicated                              *
//*                    Note: We modify only the highlight position.            *
//*          tcdi    : pointer to array of tcDataInfo objects containing paths *
//*                    to source and target data                               *
//*          altTrg: if specified, the alternate target directory for restoring*
//*                  the item(s). (This is the user's CWD)                     *
//*                                                                            *
//* Returns: 'true' if target area validated, else 'false'                     *
//******************************************************************************

bool FileDlg::tcrValidateTarget ( ssetData& sData, tcDataInfo* tcdi, 
                                  const gString& altTrg, gString& errMsg )
{
   gString iPath,                      // item's parent directory
           iName ;                     // item's name
   tcDataInfo trgItem ;                // copy of item info
   int  hli = sData.hlIndex,           // original position of highlight
        loopy = 2 ;                    // outer-loop counter
   bool useAltTarget = ((altTrg.gschars()) > 1), // alt target specified
        validTarget = true ;           // return value

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
   //winPos dbwp( 14, 1 ) ;
   //gString dbgs ;
   //dbgs.compose( "tcrValidateTarget\nat:%S\n", 
   //              ((altTrg.gschars() < 37) ? altTrg.gstr() : &altTrg.gstr()[37]) ) ; 
   //dbwp = dbWin->WriteParagraph ( dbwp, dbgs, dbColor, true ) ;
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

   //* Outer Loop: Process the data once. If all valid AND if  *
   //* alternate target directory specified, then process data *
   //* a second time, inserting the true target filespec.      *
   do
   {
      //* For each selected item in the list, determine whether *
      //* target already exists, AND whether target's parent    *
      //* directory exists, AND whether user has write access.  *
      if ( (tcFirstSelected ( sData, this->selectAttr )) != false )
      {
         do
         {
            trgItem = tcdi[sData.hlIndex] ;

            #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
            //dbgs.compose( "i:%2d) %s\n", &sData.hlIndex, &trgItem.trgPath[49] ) ; 
            //dbwp = dbWin->WriteParagraph ( dbwp, dbgs, dbColor, true ) ;
            #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

            if ( useAltTarget )
            {
               this->fmPtr->ExtractFilename ( iName, trgItem.trgPath ) ;
               iPath.compose( "%S/%S", altTrg.gstr(), iName.gstr() ) ;
               iPath.copy( trgItem.trgPath, MAX_PATH ) ;

               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
               //dbgs.compose( "      %s\n", &trgItem.trgPath[49] ) ; 
               //dbwp = dbWin->WriteParagraph ( dbwp, dbgs, dbColor, true ) ;
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE
            }
            if ( loopy > 1 )
            {
               if ( !(validTarget = this->tcrTargetTest ( trgItem, errMsg )) )
                  break ; // invalid target identified, return the bad news
            }
            else  // (loopy == 1) re-insert modified record into array
            {
               tcdi[sData.hlIndex] = trgItem ;
            }
         }
         while ( (tcNextSelected ( sData, this->selectAttr )) != false ) ;
      }
   }
   while ( validTarget && useAltTarget && (--loopy > ZERO) ) ;

   sData.hlIndex = hli ;
   return validTarget ;

}  //* End tcrValidateTarget() *

//***********************
//*    tcrTargetTest    *
//***********************
//******************************************************************************
//* Determine if target for restore-from-trashcan is valid, i.e.               *
//*  1) target item DOES NOT already exist                                     *
//*  2) container directory DOES exist                                         *
//*  3) container directory is write accessible                                *
//* If error, return a warning message so user can correct the error.          *
//*                                                                            *
//* Input  : tcsel : information on source and target                          *
//*                                                                            *
//* Returns: 'true' if valid target filespec, else 'false'                     *
//******************************************************************************

bool FileDlg::tcrTargetTest ( const tcDataInfo& tcItem, gString& errMsg )
{
   const char* const tExistMsg = 
      "WARNING: Cannot overwrite existing target(s)!\n"
      "         Please delete/rename existing target(s), OR\n"
      "         select an alternate path for restoring item(s)." ;
   const char* const tNoAccTemplate =
      "WARNING: Container directory  '%S'\n"
      "                              %s.\n"
      "         Please %s the target directory and try again." ;

   gString  gsTName, gsTDir ;       // output formatting
   fmFType  tType, cType ;          // file type of target
   bool     wPerm,                  // 'true' if user has write permission on target
            validTarget = false ;   // return value

   errMsg.clear() ;                 // no error message (yet)

   this->fmPtr->ExtractPathname ( gsTDir, tcItem.trgPath ) ;
   bool tExist = this->TargetExists ( tcItem.trgPath, tType ),
        cExist = this->TargetExists ( gsTDir.ustr(), cType, wPerm ) ;
   if ( ! tExist && (cExist && cType == fmDIR_TYPE && wPerm) )
   {
      validTarget = true ;
   }
   else
   {  //* Directory errors take precedence *
      if ( !cExist || !wPerm || (cType != fmDIR_TYPE) )
      {
         this->fmPtr->ExtractFilename ( gsTName, gsTDir ) ;
         errMsg.compose( tNoAccTemplate, gsTName.gstr(), 
                         (cExist ? (cType == fmDIR_TYPE ? "is write-protected" 
                                   : "is not a directory") : "does not exist"),
                         ((cType == fmDIR_TYPE) ? "enable" : "create") ) ;
      }
      else if ( tExist )   // existing targets
         errMsg = tExistMsg ;
   }
   return validTarget ;

}  //* End tcrTargetTest() *

//***********************
//*    tcRestoreItem    *
//***********************
//******************************************************************************
//* Restore the specified item from the Trashcan to its original position,     *
//* OR to the alternate position specified by tcItem.trgPath.                  *
//* This will move the source to target (with original filename)               *
//*   AND                                                                      *
//* will delete the associated '.trashinfo' file.                              *
//*                                                                            *
//* Input  : tcItem  : (by reference) description of caller's selected item    *
//*                    including source and target data                        *
//*          delFiles: (by reference) receives number of source files deleted  *
//*                    This should be the same as number of files restored,    *
//*                    so if not, caller will know there was an error.         *
//*                                                                            *
//* Returns: number of FILES restored                                          *
//*          (not the number of items, which will be either 1 or 0)            *
//******************************************************************************
//* 1) If source is a single file:                                             *
//*    a) copy source to target (with rename if necessary)                     *
//*    b) if copy to target is successful, delete source                       *
//*    c) if copy AND delete successful, delete associated '.trashinfo' file   *
//* 2) If source is a directory tree:                                          *
//*    a) copy top-level directory name to target (with rename if specified)   *
//*    b) if target directory created, copy source contents to target          *
//*    c) if copy sequence is successful, then delete source tree from trash   *
//*    d) if BOTH copy and delete sequences are successful, delete associated  *
//*       '.trashinfo' file                                                    *
//*                                                                            *
//* For a single source file, we do the copy-and-delete in the fast and simple *
//* way because we may need to rename. However, for non-empty directories, we  *
//* place the source data on the clipboard and then call the clipboard-data    *
//* copy and delete methods. This prevents code duplication.                   *
//******************************************************************************

UINT FileDlg::tcRestoreItem ( const tcDataInfo& tcItem, UINT& delFiles )
{
   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
   bool ourDBwindow = false ;
   if ( dbWin == NULL )
   {
      dbWin = this->Open_DBwindow ( dbColor, dbLines, dbCols ) ;
      dbwp = { 1, 1 } ;
      ourDBwindow = true ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

   delFiles = ZERO ;                // initialize caller's data
   UINT filesRestored = ZERO ;      // return value

   //* Set the CWD to 'Trash/files' directory *
   //* so we can get the file data.           *
   DispData dData( (this->fMaxX - 2) ) ;
   gString filePath, infoPath ;
   this->fmPtr->CatPathFname ( filePath, this->trashDir, trashFiles ) ;
   this->fmPtr->CatPathFname ( infoPath, this->trashDir, trashInfo ) ;
   if ( (this->fmPtr->CdPath ( filePath, dData )) != false )
   {  //* Update our data members *
      this->fmPtr->GetCurrDir ( this->currDir ) ;
      dData.getData( this->deList, this->textData, this->colorData, 
                     this->fileCount, this->dirSize ) ;  
      this->hIndex = ZERO ;      // index first item in display list

      gString srcPath ( tcItem.srcPath ),   // Filespec for item in trash.
              infoPath ( tcItem.infoPath ), // Filespec for trashinfo record
              trgPath ( tcItem.trgPath ) ;  // Filespec for target item

      //* Get source stats for doing the copy *
      tnFName srcStats ;         // source file stats
      this->fmPtr->GetFileStats ( srcStats, srcPath ) ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
      if ( dbWin != NULL )
      {
         gString gs ;
         gs.compose( L"srcStats:%s (r:%hhd w:%hhd t:%hd)\n"
                      "IFO:\n%S\n"
                      "SRC:\n%S\n"
                      "TRG:\n%S\n",
                     srcStats.fName,     &srcStats.readAcc, 
                     &srcStats.writeAcc, &srcStats.fType,
                     &infoPath.gstr()[25], &srcPath.gstr()[25], &trgPath.gstr()[49] ) ;
         dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
      }
      #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

      //* Copy the base file or directory name to target *
      if ( ((this->CopyFile ( srcStats, srcPath, trgPath )) == OK) )
      {
         ++filesRestored ;    // base file/dirname successfully copied
         short contentStatus = OK ;
         //* If a non-empty directory tree, copy directory contents *
         if ( srcStats.fType == fmDIR_TYPE )
         {
            //* Tag user's selected item for this view.             *
            //* Hilight (hIndex) will reference the 'selected' item.*
            this->tcMarkSelection ( tcItem, false, opCUT ) ;

            //* If a non-empty directory *
            if ( (clipBoard[BNODE].nextLevel != NULL) &&
                 (clipBoard[BNODE].nextLevel[ZERO].totFiles > ZERO) )
            {
               pclParm  pcl ;             // initializes psOption to psstSimpleCopy
                                          // and sets accumulators to ZERO
               pcl.deleteSrc = true ;     // delete source after copy
               pcl.srcTree = &clipBoard[BNODE].nextLevel[ZERO] ;
               pcl.srcDir = srcPath ;
               pcl.trgDir = trgPath ;

               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
               if ( dbWin != NULL )
               {
                  gString gs( "Move Tree Contents...\n"
                              "%S\n"
                              "%S\n",
                              &pcl.srcDir.gstr()[25], &pcl.trgDir.gstr()[49] ) ;
                  if ( pcl.srcTree == NULL )
                     gs.append( "pcl.srcTree is invalid!\n" ) ;
                  dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
               }
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

               contentStatus = ERR ;   // plan for the worst
               if ( pcl.srcTree != NULL )
               {
                  UINT pasteCount = this->pslCopyTnList ( pcl ) ;
                  if ( pasteCount == pcl.attCount && pcl.delCount == pasteCount )
                  {
                     filesRestored += pasteCount ;
                     delFiles = pcl.delCount ;
                     contentStatus = OK ;
                  }
               }
            }
            #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
            else if ( dbWin != NULL )
            {
               gString gs( "(empty directory)\n" ) ;
               dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
            }
            #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE
         }

         //* If successful, delete the base file or directory-name *
         //* source, then delete the associated '.trashinfo' file. *
         if ( contentStatus == OK )
         {
            #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
            if ( dbWin != NULL )
            {
               gString gs ;
               gs.compose( L"Delete Source Base...\n"
                            "%S\n", &srcPath.gstr()[25] ) ;
               dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
            }
            #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

            if ( (this->DeleteSourceFile ( &srcStats, srcPath )) == OK )
            {
               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
               if ( dbWin != NULL )
               {
                  gString gs ;
                  gs.compose( L"Delete Info File...\n"
                               "%S\n", &infoPath.gstr()[25] ) ;
                  dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;
               }
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

               ++delFiles ;
               if ( (this->DeleteFile ( infoPath )) != OK )
                  --delFiles ; // source gone, but error deleting '.trashinfo' file
            }
         }
      }
   }

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0  && TC_RESTORE != 0
   if ( dbWin != NULL )
   {
      gString gs ;
      gs.compose( L"filesRestored: %u delFiles: %u\n\n",
                  &filesRestored, &delFiles ) ;
      dbwp = dbWin->WriteParagraph ( dbwp, gs, dbColor, true ) ;

      if ( ourDBwindow )
      {
         dbWin->ClearLine ( dbLines - 2, false ) ;
         dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                              " Press A Key ", this->cs.pf, true ) ;
         nckPause();
         delete dbWin ;
         dbWin = NULL ;
      }
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

   return filesRestored ;

}  //* End tcRestoreItem() *

//***********************
//*   tcMarkSelection   *
//***********************
//******************************************************************************
//* 1) Display the data for main file-display window.                          *
//* 2) Scan the file-display list and mark the items corresponding to user's   *
//*    selections.                                                             *
//* 3) Finally, copy the 'selected' items to the application clipboard,        *
//*    and set the highlight to the first 'selected' item.                     *
//*                                                                            *
//* Input  : sData   : (by reference) display data which includes flags on all *
//*                    'selected' items                                        *
//*          tcdi    : pointer to array of tcDataInfo objects containing paths *
//*                    to source and target data                               *
//*          infoDir : 'true'  if CWD is 'Trash/info' directory                *
//*                    'false' if CWD is 'Trash/files' directory               *
//*          opCode  : member of enum OpPend                                   *
//*                    opDELETE == delete operation                            *
//*                    opCUT    == cut-and-paste operation                     *
//* Returns: number of selected items (not necessarily # of selected files)    *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) If fewer than ALL are to be marked as selected, the selection process   *
//*    is a bit convoluted because we want to leverage the existing            *
//*    SelectFile() method in order to properly initialize the tracking data.  *
//*                                                                            *
//* 2) The Trashcan directories may contain corrupted or orphaned files, but   *
//*    these will not be in our list because the list contains only validated  *
//*    items. For this reason, if ALL items in the display list have been      *
//*    marked as selected, we select ALL files in the directories to clear any *
//*    corrupted/orphaned data.                                                *
//*    NOTE: The actual number of items on the clipboard may be greater than   *
//*          the number of 'selected' items reported to caller if all items are*
//*          selected AND there are orphaned items in the Trashcan directories.*
//******************************************************************************

UINT FileDlg::tcMarkSelection ( const ssetData& sData, const tcDataInfo* tcdi, 
                                bool infoDir, OpPend opCode )
{
   gString gstmp( trashinfoExt ) ;
   short extLen = gstmp.gschars() ;    // length of filename extension
   int markedItems = ZERO ;            // number of selected items (return value)
   this->DisplayFiles ( false ) ;      // update application's file display
                                       // (sets highlight on first item)
   this->DeselectFile ( true ) ;       // be sure none are marked as 'selected'

   //* Count the number of items marked as 'selected'.*
   //* (see note above)                               *
   for ( int i = ZERO ; i < sData.dispItems ; i++ )
   {
      if ( sData.dispColor[i] & this->selectAttr )
         ++markedItems ;
   }
   if ( markedItems == sData.dispItems )
      this->SelectFile ( true ) ;
   else
   {
      bool sel = false ;
      for ( UINT j = ZERO ; j < this->fileCount ; j++ )
      {
         sel = false ;
         for ( int i = ZERO ; i < sData.dispItems ; i++ )
         {
            if ( sData.dispColor[i] & this->selectAttr ) // if item selected
            {  //* Isolate the target filename *
               this->fmPtr->ExtractFilename ( gstmp, tcdi[i].infoPath ) ;
               if ( ! infoDir ) gstmp.limitChars( gstmp.gschars() - extLen ) ;

               if ( (gstmp.compare( this->deList[j]->fName )) == ZERO )
               {  //* Match found. Mark highlighted file as selected. *
                  this->SelectFile () ;
                  sel = true ;
                  break ;
               }
            }
         }
         if ( ! sel )   // move highlight forward
            this->NextDisplayItem () ;
      }
   }
   //* Place selected files on clipboard and update display *
   this->FillClipboard ( opCode ) ;
   this->dPtr->RefreshScrollextText ( this->fIndex ) ;
   this->FirstSelectedItem () ;     // set highlight on first 'selected' item
   return markedItems ;

}  //* End tcMarkSelection() *

//***********************
//*   tcMarkSelection   *
//***********************
//******************************************************************************
//* 1) Display the data for main file-display window.                          *
//* 2) Scan the file-display list and mark the item corresponding to user's    *
//*    selection.                                                              *
//* 3) Finally, copy the 'selected' item to the application clipboard,         *
//*    and set the highlight on that item.                                     *
//* 4) Although this will work for any item, it is intended primarily for a    *
//*    directory tree which needs the complex cut-and-paste functionality of   *
//*    the clipboard.                                                          *
//*                                                                            *
//* Input  : tcItem  : (by reference) description of caller's selected item    *
//*                    including source and target data                        *
//*          infoDir : 'true'  if CWD is 'Trash/info' directory                *
//*                    'false' if CWD is 'Trash/files' directory               *
//*          opCode  : member of enum OpPend                                   *
//*                    opDELETE == delete operation                            *
//*                    opCUT    == cut-and-paste operation                     *
//*                                                                            *
//* Returns: number of selected items (not necessarily # of selected files)    *
//*          This SHOULD BE one(1).                                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) This method is used to mark exactly one item, and copy that item to     *
//*    the clipboard.                                                          *
//*                                                                            *
//******************************************************************************

UINT FileDlg::tcMarkSelection ( const tcDataInfo& tcItem, 
                                bool infoDir, OpPend opCode )
{
   gString gstmp( trashinfoExt ) ;
   short extLen = gstmp.gschars() ;    // length of filename extension
   int markedItems = ZERO ;            // number of selected items (return value)
   this->DisplayFiles ( false ) ;      // update application's file display
                                       // (sets highlight on first item)
   this->DeselectFile ( true ) ;       // be sure none are marked as 'selected'

   //* Isolate the target filename *
   this->fmPtr->ExtractFilename ( gstmp, tcItem.infoPath ) ;
   if ( ! infoDir ) gstmp.limitChars( gstmp.gschars() - extLen ) ;

   //* Locate the matching filename *
   for ( UINT i = ZERO ; i < this->fileCount ; ++i )
   {
      if ( (gstmp.compare( this->deList[i]->fName )) == ZERO )
      {  //* Match found. Mark highlighted file as selected. *
         this->SelectFile () ;
         break ;
      }
      this->NextDisplayItem () ;
   }

   //* Place selected files on clipboard and update display *
   this->FillClipboard ( opCode ) ;
   this->dPtr->RefreshScrollextText ( this->fIndex ) ;
   this->FirstSelectedItem () ;     // set highlight on first 'selected' item
   return markedItems ;

}  //* End tcMarkSelection() *


//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - -  Non-member methods supporting Trashcan - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//***********************
//*       tcSort        *
//***********************
//******************************************************************************
//* ** Non-member, Private Method **                                           *
//* Sort the data according to:                                                *
//*  a) deletion timestamp (descending)                                        *
//*  c) item filename (ascending, case-insensitive)                            *
//*  b) item size (descending)                                                 *
//*                                                                            *
//* Input  : sData  : display data                                             *
//*          tcdi   : pointer to array of tcDataInfo objects containing paths  *
//*                   to source and target data                                *
//*          sOption: sort option - member of enum tcSortOption                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The data are complex, so qsort is not convenient.                          *
//* We do a simple, inefficient-but-easy-to-understand cocktail sort because   *
//* we have confidence that there are a few hundred items at most.             *
//*                                                                            *
//******************************************************************************

static void tcSort ( ssetData& sData, tcDataInfo* tcdi, tcSortOption sOption )
{
   tcDataInfo  tcswap ;
   const char* sdtswap ;
   attr_t  sdaswap ;
   attr_t* sdaPtr = (attr_t*)sData.dispColor ;
   int   iCount = sData.dispItems,     // number of items to be sorted
         lIndex, gIndex,               // low and high limits search indices
         result, i ;

   //* Sort by target filename ( low-to-high ) *
   if ( sOption == tcsoNAME )
   {
      const bool caseSen = true ;
      gString gsi, gsl, gsg ;         // for comparing filenames
      short indx ;                    // filename index
      for ( lIndex = 0, gIndex = iCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            gsi  = tcdi[i].trgPath ;
            indx = (gsi.findlast( fSLASH )) + 1 ;
            gsi.shiftChars( -(indx) ) ;
            gsl  = tcdi[lIndex].trgPath ;
            indx = (gsl.findlast( fSLASH )) + 1 ;
            gsl.shiftChars( -(indx) ) ;
            result = gsi.compare( gsl, caseSen ) ;   // compare

            if ( result < ZERO )                   // i < lIndex
            {
               tcswap = tcdi[lIndex] ;             // swap compared items
               tcdi[lIndex] = tcdi[i] ;
               tcdi[i] = tcswap ;
               sdtswap = sData.dispText[lIndex] ;  // swap display data
               sData.dispText[lIndex] = sData.dispText[i] ;
               sData.dispText[i] = sdtswap ;
               sdaswap = sdaPtr[lIndex] ;
               sdaPtr[lIndex] = sdaPtr[i] ;
               sdaPtr[i] = sdaswap ;
            }
            if ( i < gIndex )
            {
               gsg  = tcdi[gIndex].trgPath ;
               indx = (gsg.findlast( fSLASH )) + 1 ;
               gsg.shiftChars( -(indx) ) ;
               result = gsg.compare( gsi, caseSen ) ;// compare

               if ( result < ZERO )                // gIndex < i
               {
                  tcswap = tcdi[gIndex] ;          // swap compared items
                  tcdi[gIndex] = tcdi[i] ;
                  tcdi[i] = tcswap ;
                  sdtswap = sData.dispText[gIndex] ; // swap display data
                  sData.dispText[gIndex] = sData.dispText[i] ;
                  sData.dispText[i] = sdtswap ;
                  sdaswap = sdaPtr[gIndex] ;
                  sdaPtr[gIndex] = sdaPtr[i] ;
                  sdaPtr[i] = sdaswap ;

                  gsi = gsg ;
                  result = gsi.compare( gsl, caseSen ) ; // compare

                  if ( result < ZERO )             // i < lIndex
                  {
                     tcswap = tcdi[lIndex] ;       // swap compared items
                     tcdi[lIndex] = tcdi[i] ;
                     tcdi[i] = tcswap ;
                     sdtswap = sData.dispText[lIndex] ; // swap display data
                     sData.dispText[lIndex] = sData.dispText[i] ;
                     sData.dispText[i] = sdtswap ;
                     sdaswap = sdaPtr[lIndex] ;
                     sdaPtr[lIndex] = sdaPtr[i] ;
                     sdaPtr[i] = sdaswap ;
                  }  // result < ZERO
               }     // result < ZERO
            }        // i < gIndex
         }           // for(;;) inner loop
      }              // for(;;) outer loop
   }

   //* Sort by deletion date OR by item size ( high-to-low ) *
   else
   {
      for ( lIndex = 0, gIndex = iCount-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = gIndex-1 ; i >= lIndex ; i-- )
         {  //* Compare the datestamps OR item sizes *
            if ( sOption == tcsoDATE )
               result = (tcdi[i].tDate.epoch < tcdi[gIndex].tDate.epoch) ? -1 : 
                        (tcdi[i].tDate.epoch > tcdi[gIndex].tDate.epoch) ? 1 : ZERO ;
            else     // sOption == tcsoSIZE
               result = (tcdi[i].size < tcdi[gIndex].size) ? -1 : 
                        (tcdi[i].size > tcdi[gIndex].size) ? 1 : ZERO ;
            if ( result < ZERO )
            {
               tcswap = tcdi[gIndex] ;            // swap compared items
               tcdi[gIndex] = tcdi[i] ;
               tcdi[i] = tcswap ;
               sdtswap = sData.dispText[gIndex] ; // swap corresponding display data
               sData.dispText[gIndex] = sData.dispText[i] ;
               sData.dispText[i] = sdtswap ;
               sdaswap = sdaPtr[gIndex] ;
               sdaPtr[gIndex] = sdaPtr[i] ;
               sdaPtr[i] = sdaswap ;
            }
            if ( i > lIndex )
            {
               if ( sOption == tcsoDATE )
                  result = (tcdi[lIndex].tDate.epoch < tcdi[i].tDate.epoch) ? -1 : 
                           (tcdi[lIndex].tDate.epoch > tcdi[i].tDate.epoch) ? 1 : ZERO ;
               else     // sOption == tcsoSIZE
                  result = (tcdi[lIndex].size < tcdi[i].size) ? -1 : 
                           (tcdi[lIndex].size > tcdi[i].size) ? 1 : ZERO ;
               if ( result < ZERO )
               {
                  tcswap = tcdi[lIndex] ;
                  tcdi[lIndex] = tcdi[i] ;
                  tcdi[i] = tcswap ;
                  sdtswap = sData.dispText[lIndex] ;
                  sData.dispText[lIndex] = sData.dispText[i] ;
                  sData.dispText[i] = sdtswap ;
                  sdaswap = sdaPtr[lIndex] ;
                  sdaPtr[lIndex] = sdaPtr[i] ;
                  sdaPtr[i] = sdaswap ;
   
                  if ( sOption == tcsoDATE )
                     result = (tcdi[i].tDate.epoch < tcdi[gIndex].tDate.epoch) ? -1 :
                              (tcdi[i].tDate.epoch > tcdi[gIndex].tDate.epoch) ? 1 : ZERO ;
                  else if ( sOption == tcsoNAME )
                  {
                  }
                  else if ( sOption == tcsoSIZE )
                     result = (tcdi[i].size < tcdi[gIndex].size) ? -1 : 
                              (tcdi[i].size > tcdi[gIndex].size) ? 1 : ZERO ;
                  if ( result < ZERO )
                  {
                     tcswap = tcdi[gIndex] ;
                     tcdi[gIndex] = tcdi[i] ;
                     tcdi[i] = tcswap ;
                     sdtswap = sData.dispText[gIndex] ;
                     sData.dispText[gIndex] = sData.dispText[i] ;
                     sData.dispText[i] = sdtswap ;
                     sdaswap = sdaPtr[gIndex] ;
                     sdaPtr[gIndex] = sdaPtr[i] ;
                     sdaPtr[i] = sdaswap ;
                  }  // if(result>0)
               }     // if(result>0)
            }        // if(i>lIndex)
         }           // for(inner)
      }              // for(outer)
   }
}  //* End tcSort() *

//***********************
//*    tsfCreateLog     *
//***********************
//******************************************************************************
//* Create a log file for the just-copied-to-trashcan file.                    *
//* ** Non-member, Private Method **                                           *
//*                                                                            *
//* Note that if there is an existing target file, then it has been orphaned,  *
//* at some previous time, so we overwrite it.                                 *
//*                                                                            *
//* Input  : srcPath    : path/filename of source                              *
//*          trgInfoPath: path/filename for log file                           *
//*          timeStamp  : deletion date entry for log file                     *
//*                                                                            *
//* Returns: 'true' if log file created, else 'false'                          *
//******************************************************************************

static bool tsfCreateLog ( const gString& srcPath, const gString& trgInfoPath, 
                           const gString& timeStamp )
{
   bool logCreated = false ;
   ofstream ofs( trgInfoPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      ofs << trashinfoHdr << '\n'
          << trashinfoPath << srcPath.ustr() << '\n'
          << timeStamp.ustr() << endl ;

      if ( ofs.is_open() )
         ofs.close() ;
      logCreated = true ;
   }
   return logCreated ;

}  //* End tsfCreateLog() *

//***********************
//*     tcSelectAll     *
//***********************
//******************************************************************************
//* Select or de-select all items in the display-data list.                    *
//* A 'selected' item is indicated by setting the ncuATTR (underline) bit of   *
//* the item's color attribute.                                                *
//* ** Non-member, Private Method **                                           *
//*                                                                            *
//* Input  : sData  : display data                                             *
//*          mask   : bit mask for visually marking 'selected' items           *
//*          sel    : if 'true'  select all items                              *
//*                   if 'false' de-select all items                           *
//*                                                                            *
//* Returns: number of 'selected' items                                        *
//******************************************************************************

static int tcSelectAll ( ssetData& sData, attr_t mask, bool sel )
{
   attr_t* dcPtr = (attr_t*)sData.dispColor ;
   for ( int i = ZERO ; i < sData.dispItems ; i++ )
   {
      if ( sel )  dcPtr[i] |= mask ;   // select item
      else        dcPtr[i] &= ~mask ;  // deselect item
   }
   return ( (sel ? sData.dispItems : ZERO) ) ;

}  //* End tcSelectAll() *

//***********************
//*   tcSelectRecent    *
//***********************
//******************************************************************************
//* Scan the list of items in the trashcan and mark as 'selected' all items    *
//* which share the most recent deletion timestamp.                            *
//*                                                                            *
//* Assumes that list is sorted in reverse deletion-date order,                *
//* (most-recently-deleted item is at the top of the list).                    *
//*                                                                            *
//* Input  : sData  : display data                                             *
//*          tcdi   : pointer to array of tcDataInfo objects containing paths  *
//*                   to source and target data                                *
//*          mask   : bit mask for visually marking 'selected' items           *
//*                                                                            *
//* Returns: number of items matching the criteria                             *
//******************************************************************************

static int tcSelectRecent ( ssetData& sData, tcDataInfo* tcdi, attr_t mask )
{
   attr_t* dcPtr = (attr_t*)sData.dispColor ;   // pointer to color-attribute array
   int selected = ZERO ;                        // return value

   //* Get deletion timestamp for first item in the list. *
   int64_t delDate = tcdi[ZERO].tDate.epoch ;

   for ( int i = ZERO ; i < sData.dispItems ; i++ )
   {
      if ( (tcdi[i].tDate.epoch) == delDate )
      {
         dcPtr[i] |= mask ;   // select item
         ++selected ;
      }
      else
         break ;     // no more matching items
   }
   return selected ;

}  //* End tcSelectRecent() *

//***********************
//*    tcSameRTarget    *
//***********************
//******************************************************************************
//* ** Non-member, Private Method **                                           *
//* For restoring files from trashcan, determine whether all source items will *
//* be restored to the same target directory.                                  *
//*                                                                            *
//* Input  : sData  : display data                                             *
//*          tcdi   : pointer to array of tcDataInfo objects containing paths  *
//*                   to source and target data                                *
//*                                                                            *
//* Returns: 'true' if all items will be restored to the same target directory *
//*          'false' if multiple target directories                            *
//******************************************************************************
//* Note: It is assumed that the data sent to us is valid.                     *
//******************************************************************************

static bool tcSameRTarget ( ssetData& sData, const tcDataInfo* tcdi, attr_t selMask )
{
   int   hli    = sData.hlIndex ;         // remember highlight position
   bool  status = true ;                  // return value

   if ( (tcFirstSelected ( sData, selMask )) != false )
   {
      gString firstTrg( tcdi[sData.hlIndex].trgPath ), nextTrg ;
      short nIndex = firstTrg.findlast( fSLASH ) ;
      firstTrg.limitChars( nIndex ) ;
//#if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
//winPos dbwp( 4, 1 ) ; gString dbgs( "ft(%d):'%S'\n", &sData.hlIndex, &firstTrg.gstr()[37] ) ;
//   dbwp = dbWin->WriteParagraph ( dbwp, dbgs, dbColor, true ) ;
//#endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE

      while ( (tcNextSelected ( sData, selMask )) != false )
      {
         nextTrg = tcdi[sData.hlIndex].trgPath ;
         nIndex = nextTrg.findlast( fSLASH ) ;
         nextTrg.limitChars( nIndex ) ;
//#if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
//dbgs.compose( "nt(%d):'%S'\n", &sData.hlIndex, &nextTrg.gstr()[37] ) ;
//   dbwp = dbWin->WriteParagraph ( dbwp, dbgs, dbColor, true ) ;
//#endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE
         if ( (nextTrg.compare( firstTrg.gstr(), (firstTrg.gschars() - 1) )) != ZERO )
         {
            status = false ;
//#if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0 && TC_RESTORE != 0
//   dbwp = dbWin->WriteParagraph ( dbwp, "NOT A MATCH\n", dbColor, true ) ;
//#endif   // ENABLE_DEBUGGING_CODE && enableDBWIN && TC_RESTORE
            break ;
         }
      }
      sData.hlIndex = hli ;      // restore original index
   }
   return status ;

}  //* End tcSameRTarget() *

//***********************
//*   tcFirstSelected   *
//***********************
//******************************************************************************
//* ** Non-member, Private Method **                                           *
//* Find the first 'selected' item in the item list and reference it using the *
//* highlight index (hlIndex).                                                 *
//*          This DOES NOT update the display, it only sets the index.         *
//*                                                                            *
//* Returns: 'true' if selection found                                         *
//*                 'hlIndex' will reference it                                *
//*          'false' no selected items                                         *
//*                 'hlIndex' unchanged                                        *
//******************************************************************************

static bool tcFirstSelected ( ssetData& sData, attr_t selMask )
{
   int  loopy  = sData.dispItems - 1 ;    // loop control
   bool status = false ;                  // return value

   sData.hlIndex = ZERO ;     // set highlight at top of array

   for ( int i = ZERO ; i < loopy ; ++i )
   {
      if ( sData.dispColor[i] & selMask )
      {
         sData.hlIndex = i ;
         status = true ;
         break ;
      }
   }
   return status ;

}  //* End tcFirstSelected() *

static bool tcNextSelected ( ssetData& sData, attr_t selMask )
{
   int  loopy  = sData.dispItems - 1 ;    // loop control
   bool status = false ;                  // return value

   if ( sData.hlIndex < loopy )
   {
      for ( int i = (sData.hlIndex + 1) ; i < loopy ; ++i )
      {
         if ( sData.dispColor[i] & selMask )
         {
            sData.hlIndex = i ;
            status = true ;
            break ;
         }
      }
   }
   return status ;

}  //* End tcFirstSelected() *

//***********************
//*  tcSummaryDisplay   *
//***********************
//******************************************************************************
//* Display the Trashcan summary information.                                  *
//* ** Non-member, Private Method **                                           *
//*                                                                            *
//* Input  : dp          : pointer to open dialog window                       *
//*          wPos        : position of first display item                      *
//*                        (on return, contains position for context-help msgs *
//*          trashedItems: number if items in Trashcan                         *
//*          trashedFiles: number of files in Trashcan                         *
//*          trashedBytes: total file size in bytes                            *
//*          trashSpace  : approximate disc space used                         *
//*          freeSpace   : disc free space available                           *
//*          dColor      : dialog background color                             *
//*          tColor      : color for formatted data display                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void tcSummaryDisplay ( NcDialog* dp, winPos& wPos, UINT trashedItems, 
                               UINT trashedFiles, UINT64 trashedBytes, 
                               UINT64 trashSpace, UINT64 freeSpace,
                               attr_t dColor, attr_t tColor )
{
   gString gsi, gsOut ;
   gsi.formatInt( trashedItems, 6 ) ;
   gsOut.compose( L" %S ", gsi.gstr() ) ;
   dp->WriteString ( wPos, gsOut, tColor ) ;
   ++wPos.ypos ;

   gsi.formatInt( trashedFiles, 6 ) ;
   gsOut.compose( L" %S ", gsi.gstr() ) ;
   dp->WriteString ( wPos, gsOut, tColor ) ;
   ++wPos.ypos ;

   gsi.formatInt( trashedBytes, 6 ) ;
   gsOut.compose( L" %S ", gsi.gstr() ) ;
   dp->WriteString ( wPos, gsOut, tColor ) ;
   ++wPos.ypos ;

   gsi.formatInt( trashSpace, 6 ) ;
   gsOut.compose( L" %S ", gsi.gstr() ) ;
   dp->WriteString ( wPos, gsOut, tColor ) ;
   ++wPos.ypos ;

   gsi.formatInt( freeSpace, 6 ) ;
   gsOut.compose( L" %S ", gsi.gstr() ) ;
   dp->WriteString ( wPos, gsOut, dColor ) ;
   //* Position for user prompt messages *
   //* (note: these are magic numbers)   *
   wPos = { short(wPos.ypos - 2), 36 } ;

}  //* End tcSummaryDisplay() *

//***********************
//*       tcAlloc       *
//***********************
//******************************************************************************
//* Allocate dynamic memory for the tcDialog() method's data.                  *
//* ** Non-member, Private Method **                                           *
//*                                                                            *
//* Input  : itemCount: number of items for which to allocate                  *
//*                     (NOTE: if 'itemCount' == ZERO, we allocate space for)  *
//*                     (      one dummy item.                              )  *
//*          sData    : (by reference) display-data control object             *
//*                                    receives pointers to display items and  *
//*                                    color-attribute data                    *
//*          tcdiPtr  : (by reference) receives pointer to array of            *
//*                                    item-tracking objects                   *
//*          blkPtr   : (by reference) receives pointer to array of raw display*
//*                                    strings                                 *
//*          dCols    : number of display columns in dialog                    *
//*                                                                            *
//* Returns: true if allocation successful, else false                         *
//*          (Note: The way the C++ 'new' method is written is not clear)      *
//*                 to us, however it _should_ return NULL on failure.  )      *
//******************************************************************************

static bool tcAlloc ( int itemCount, ssetData& sData, tcDataInfo*& tcdiPtr, 
                      char*& blkPtr, short dCols )
{
   bool status = false ;

   //* IMPORTANT NOTE:                                         *
   //* It is assumed that caller is smart, (it's me), however  *
   //* to be safe, we call tcFree() before allocating memory.  *
   tcFree ( sData, tcdiPtr, blkPtr ) ;
   sData.dispItems = itemCount > ZERO ? itemCount : 1 ;
   sData.hlIndex = ZERO ;
   sData.hlShow = false ;

   //* Space for display strings (there is extra space for each string) *
   if ( blkPtr == NULL )
   {
      for ( short i = 3 ; i > ZERO ; --i )
      { if ( (blkPtr = new (nothrow) char[sData.dispItems * dCols]) != NULL ) break ; }
   }
   //* Space for pointers to strings *
   if ( sData.dispText == NULL )
   {
      for ( short i = 3 ; i > ZERO ; --i )
      { if ( (sData.dispText  = new (nothrow) const char*[sData.dispItems]) != NULL ) break ; }
   }
   //* Space for color attributes *
   if ( sData.dispColor == NULL )
   {
      for ( short i = 3 ; i > ZERO ; --i )
      { if ( (sData.dispColor = new (nothrow) attr_t[sData.dispItems]) != NULL ) break ; }
   }
   //* Space for sorted fileID / datestamp data *
   if ( tcdiPtr == NULL )
   {
      for ( short i = 3 ; i > ZERO ; --i )
      { if ( (tcdiPtr = new (nothrow) tcDataInfo[sData.dispItems]) != NULL ) break ; }
   }

   if ( blkPtr != NULL && sData.dispText != NULL && 
        sData.dispColor != NULL && tcdiPtr != NULL )
      status = true ;

   return status ;

}  //* End tcAlloc() *

//***********************
//*       tcFree        *
//***********************
//******************************************************************************
//* Free the dynamic memory previously-allocated for the tcDialog() method.    *
//* ** Non-member, Private Method **                                           *
//*                                                                            *
//* Input  : sData   : (by reference) display-data control object              *
//*          tcdiPtr : (by reference) pointer to array of item-tracking objects*
//*          blkPtr  : (by reference) pointer to array of raw display strings  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void tcFree ( ssetData& sData, tcDataInfo*& tcdiPtr, char*& blkPtr )
{
   gString gs ;
   if ( blkPtr != NULL )
      {  delete [] blkPtr ; blkPtr = NULL ; }
   if ( sData.dispText != NULL )
      { delete [] sData.dispText ; sData.dispText = NULL ; }
   if ( sData.dispColor != NULL )
      { delete [] sData.dispColor ; sData.dispColor = NULL ; }
   sData.dispItems = sData.hlIndex = ZERO ;
   sData.hlShow = false ;
   if ( tcdiPtr != NULL )
      { delete [] tcdiPtr ; tcdiPtr = NULL ; }

}  //* End tcFree() *

//***********************
//*                     *
//***********************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

