//********************************************************************************
//* File       : FileDlgStat.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 19-Apr-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description: This module of the FileDlg class contains the sub-dialog for    *
//* display and modification of file stats:                                      *
//*                                                                              *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//********************************************************************************


//* Header Files *
#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;


//* Note: Must be in same order as enum fmFType *
static const char FTypes[][6] = 
{
   " DIR ", " REG ", "LINK ", "CHDEV", "BKDEV", "FIFO ", "SOCK ", " UNK " 
} ;
//* Column headings *
static const char Head1[] = "FTYPE  OWNER  GROUP  OTHER  sUID  sGID  STIK   INODE    hLINK   BYTES" ;
static const char Head2[] = "-----  -----  -----  -----  ----  ----  ----  --------  -----  --------" ;
static const short DIALOG_LINES = 11 ;             // display lines
static const short DIALOG_COLS = 76 ;              // display columns
static const short LPATH_MAX = DIALOG_COLS - 4 ;   // primary path/filename length max
static const short TPATH_MAX = LPATH_MAX - 8 ;     // target path/filename length max
static const short MODNOERR = 1 ;      // return value if file stats were modified with no errors
static const short MODERR   = 2 ;      // return value if file stats were modified with errors

//* dfsXXx() methods and ModStat_ControlUpdate() callback *
//* method require access to their parent dialog windows. *
static   NcDialog*   dfsdp ;        // pointer to NcDialog object
static   attr_t      dfsColor ;     // dialog window's background color

//* Pnemonics for the control-index values. Used by ModifyFileStats(), *
//* ModStat_ControlUpdate() and ModStat_ContextHelp().                 *
enum ControlIndices : short
{
   pbCANCEL,                        // index of 'CANCEL' pushbutton 
   rbUSRr, rbUSRw, rbUSRx,          // indices for USR permission controls
   rbGRPr, rbGRPw, rbGRPx,          // indices for GROUP permission controls
   rbOTHr, rbOTHw, rbOTHx,          // indices for GROUP permission controls
   pbSUID, pbSGID, pbSTIK,          // indices for executable file attribute controls
   tbMODd, tbACCd,                  // indices for date/timestamp controls
   tbOWNER, tbGROUP,                // indices for UserID and GroupID controls
   pbUPDATE,                        // index of 'UPDATE' pushbutton
   pbHELP,                          // index of 'HELP' pushbutton
   ciDEFINED                        // number of controls defined
} ;


//**************
//* Prototypes *
//**************
//static short   ModStat_ControlUpdate ( short currIndex, const wkeyCode wkey, 
//                                       bool firstTime = false ) ;
static void    ModStat_ContextHelp ( short currIndex ) ;
static void    DisplayHeaders ( short ypos, short xpos, attr_t color ) ;
static short   CompareDates ( localTime& ftA, localTime& ftB ) ;
static bool    operator==( localTime& ftA, localTime& ftB ) ;
static void    InputErrorMsg ( const char* msg, winPos ctr, short pause ) ;


//***********************
//*  DisplayFileStats   *
//***********************
//******************************************************************************
//* Display the file statistics for the currently-highlighted file.            *
//* If the file is a symbolic link, then also display the statistics for the   *
//* the target file.                                                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK     if stats successfully displayed                            *
//*          ERR    if error displaying or modifying stats                     *
//*          > ZERO if file stats were successfully modified by user           *
//******************************************************************************
//* Programmer's Note: Positioning of this sub-dialog requires that the        *
//* parent dialog be larger than the sub-dialog in both Y and X. If it is not, *
//* the sub-dialog will not open.                                              *
//******************************************************************************

short FileDlg::DisplayFileStats ( void )
{
short    status = OK ;                    // return value

   //* If there is actually a file under the highlight *
   if ( this->fileCount > ZERO )
   {
      //* Retrieve the file stat informaton *
      tnFName  fn ;        // copy of basic file info
      ExpStats es, est ;   // expanded, human-readable version of file info
      es.symbTarg = &est ; // link the data structures in case file is a symbolic link
      this->GetStats ( fn ) ;

      short dialogLines = DIALOG_LINES,
            ctrY = this->fulY + this->fMaxY/2 - 3,  // dialog position
            ctrX = this->fulX + this->fMaxX/2,
            ulY  = ctrY - DIALOG_LINES / 2,
            ulX  = ctrX - DIALOG_COLS / 2 ;
      //* If file is a symbolic link, increase size *
      //* of dialog to display target file also.    *
      if ( fn.fType == fmLINK_TYPE )
      {
         dialogLines += DIALOG_LINES - 4 ;
         ulY -= DIALOG_LINES / 4 + 1 ;
      }
      //* Stay within the parent dialog's borders *
      while ( ulY <= this->fulY )
         ++ulY ;
      while ( ulX <= this->dulX )
         ++ulX ;
      while ( (ulX > this->dulX) && ((ulX + DIALOG_COLS) >= (this->dulX + this->dCols)) )
         --ulX ;
      while ( (ulY > (this->dulY)) && ((ulY+dialogLines) >= (this->dulY+this->dRows)) )
         --ulY ;
      attr_t   dColor = this->cs.sd,            // dialog background color
               // path/filename display color
               pColor = this->cs.scheme == ncbcCOLORS ? this->cs.sd & ~ncrATTR : this->cs.tf ;
      bool     edit = false ;

enum dfsControls : short { closePush = ZERO, modPush } ;
static const short controlsDEFINED = 3 ;  // # of controls in dialog
InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - -  closePUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      22,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      9,                            // cols:      control columns
      "  CLOSE  ",                  // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'MODIFY' pushbutton   - - - - - - - - - - - -    modPUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      46,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " MODIFY ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* For files user is not allowed to modify stats for, disable 'Modify' *
//   if ( fn.fType == fm?_TYPE )
//   {
//      ic[modPush].nColor = dColor ;
//      ic[modPush].active = false ;
// NOT YET IMPLEMENTED
//   }

      //* Save the display for the parent dialog window *
      this->dPtr->SetDialogObscured () ;
   
      //* Initial parameters for dialog window *
      InitNcDialog dInit( dialogLines,    // number of display lines
                          DIALOG_COLS,    // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal 
                          NULL,           // dialog title
                          ncltSINGLE,     // border line-style
                          dColor,         // border color attribute
                          dColor,         // interior color attribute
                          ic              // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      NcDialog* dp = new NcDialog ( dInit ) ;
      dfsdp = dp ;                  // give sub-methods access to the dialog
      dfsColor = dColor ;

      if ( (dp->OpenWindow()) == OK )
      {
         //* Set dialog title *
         dp->SetDialogTitle ( "  View File Stats  ", this->cs.em ) ;
         //* Draw the static data *
         LineDef  lDefH( ncltHORIZ, ncltSINGLE, dialogLines-3, ZERO, 
                         DIALOG_COLS, dColor ) ;
         dp->DrawLine ( lDefH ) ;
         DisplayHeaders ( 2, 2, dColor ) ;
         
         //* Format the stat data for human consumption.                       *
         //* (if file is symbolic link, format data for target also)           *
         this->fmPtr->strnCpy ( es.filePath, this->currDir, MAX_PATH ) ;
         status = this->fmPtr->ExpandStats ( &fn, es ) ;
         bool  brokenLink = (fn.fType == fmLINK_TYPE && es.symbTarg->fileType == fmTYPES) ;
         if ( status == OK || brokenLink != false )
         {
            //* Construct path/filename and display it *
            winPos wPos( 1, 2 ) ;
            this->dfsDisplayFilename ( wPos, pColor, es.filePath, es.fileName, LPATH_MAX ) ;

            //* Output the data *
            wPos = { 4, 2 } ;
            this->dfsDisplayStatData ( wPos, dColor, es ) ;
            if ( fn.fType == fmLINK_TYPE )
            {
               winPos wPos( 8, 2 ) ;
               this->dfsDisplayFilename ( wPos, pColor, est.filePath, est.fileName, TPATH_MAX ) ;
               if ( status == OK )
               {
                  DisplayHeaders ( 9, 2, dColor ) ;
                  wPos = { 11, 2 } ;
                  this->dfsDisplayStatData ( wPos, dColor, est ) ;
               }
               else  // broken link
               {
                  dp->WriteString ( 10, 10, " (broken link) ", pColor ) ;
               }
            }
         }
         else
         {
            dp->WriteString ( 5, 21, "  Unable to 'stat' target file. ", 
                              this->cs.pf ) ;
            dp->ControlActive ( modPush, false ) ; // disable mod pushbutton
            status = ERR ;
         }

         dp->RefreshWin () ;                 // make the text visible

         //* Find out what the user wants to do *
         bool     done = false ;
         uiInfo   Info ;                     // user interface data returned here
         while ( ! done )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == closePush )
                  done = true ;
               else if ( Info.ctrlIndex == modPush )
               {
                  if ( status == OK )
                     edit = true ;
                  else
                  {
                     dp->WriteString ( ic[closePush].ulY, ic[closePush].ulX, 
                                       "  Unable to modify target file. ", 
                                       this->cs.pf, true ) ;
                     chrono::duration<short>aWhile( 3 ) ;
                     this_thread::sleep_for( aWhile ) ;
                     status = ERR ;
                  }
                  done = true ;
               }
            }
            if ( !done && !Info.viaHotkey )
            {
               if ( Info.keyIn == nckSTAB )
                  dp->PrevControl () ; 
               else if ( Info.keyIn != ZERO )
                  dp->NextControl () ;
            }
         }     // while()
      }
      else    // dialog did not open - alert user
      {
         dtbmData   msgData( " Unable to Open Dialog Window! " ) ;
         this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;
         chrono::duration<short>aWhile( 5 ) ;
         this_thread::sleep_for( aWhile ) ;
      }
      if ( dp != NULL )
      {
         delete ( dp ) ;            // close the window
         dfsdp = NULL ;             // tidy up
         dfsColor = ZERO ;
      }
      
      //* If user has elected to edit the stats *
      if ( edit != false )
      {
         status = dfsModifyFileStats ( ulY, ulX, es ) ;
      }
   
      //* Restore display of parent dialog window *
      this->dPtr->RefreshWin () ;
   }
   return status ;

}  //* End DisplayFileStats() *

//*************************
//*  dfsModifyFileStats   *
//*************************
//******************************************************************************
//* Interactively modify certain file stats data:                              *
//* 1. Modification timestamp                                                  *
//* 2. Access timestamp                                                        *
//* 3. Owner                                                                   *
//* 4. Group                                                                   *
//* 5. User/Group/Other permissions for read/write/execute                     *
//* 6. Effective UID                                                           *
//* 7. Effective GID                                                           *
//* 8. Sticky bit (save app image to swap space when inactive)                 *
//*                                                                            *
//* From 'info/FilePermissions/ModeStructure:                                  *
//* See chown(): set file owner and/or group,                                  *
//*              also lchown() does not follow links                           *
//*     chmod(): Read/Write/Execute                                            *
//*                                                                            *
//*     Symbolic Modes:                                                        *
//*     umask: ugoa == user/group/other/all                                    *
//*            +-=  == add/subtract permissions                                *
//*            rwxXstugo                                                       *
//*            u+s == set effective uid on execution                           *
//*            g+s == set effective gid on execution                           *
//*            o+t == set sticky bit (image to swap file)                      *
//*            '=' == set something but remove everything else ex:             *
//*                   o=t set stick bit but remove all read/write/execute      *
//*                       permissions for 'others'                             *
//*            copying permissions:                                            *
//*             o+g == add to 'other' the permissions held by 'group'          *
//*            combinations:                                                   *
//*             a+r,g+x-w == 'all' get read permission AND 'group' gets        *
//*                          execute but loses write permission                *
//*     Numeric Modes:                                                         *
//*      integer argument sets or resets the bits absolutely, but unlike       *
//*      symbolic modes, cannot take existing settings into account            *
//*        values are in octal (no leading zeros necessary)                    *
//*                                                                            *
//*        BIT                  BIT                  BIT                       *
//*         0 == other execute   3 == group execute   6 == user execute        *
//*         1 == other write     4 == group write     7 == user write          *
//*         2 == other read      5 == group read      8 == user read           *
//*                                                                            *
//*        BIT                                                                 *
//*         9 == sticky bit (image to swap space)                              *
//*        10 == set group ID on execution                                     *
//*        11 == set user ID on execution                                      *
//*                                                                            *
//* Only for executable files:                                                 *
//*     setuid(): set process effective uid upon execution                     *
//*     setgid(): set process effective gid upon execution                     *
//*     sticky bit: save text image to swap partition so it reloads faster     *
//*                 (on some systems this is used as a 'restricted             *
//*                  deletion flag)                                            *
//*                                                                            *
//* Result can be verified through ls -l:                                      *
//*   If sUID, sGID, STIK are NOT set, the format is  OWNER GROUP OTHER        *
//*                                                    rwx   rwx   rwx         *
//*   If sUID is set, Owner's x is modified:                                   *
//*     if sUID and x, 's'                                                     *
//*     if sUID and not x, 'S'                                                 *
//*   If sGID is set, Group's x is modified:                                   *
//*     if sGID and x, 's'                                                     *
//*     if sGID and not x, 'S'                                                 *
//*   If STIK is set, Other's x is modified:                                   *
//*     if STIK and x, 't'                                                     *
//*     if STIK and not x, 'T'                                                 *
//*                                                                            *
//*                                                                            *
//* Input  : ulY, ulX: X/Y position for upper left corner of dialog            *
//*          es: structure containing formatted file stat data (by reference)  *
//*                                                                            *
//* Returns: OK       if no modifications made and no errors                   *
//*          ERR      if no modifications made and errors during modify        *
//*          MODNOERR if file stats were modified with no errors               *
//*          MODERR   if file stats were modified with errors                  *
//******************************************************************************
//* Programmer's Note: Not all fields should be available to all file types.   *
//* Needs conditional inclusion of each specific control.                      *
//* Examples: 'chmod' cannot change permission bits on symbolic links.         *
//*           'suid', 'sgid', 'stik' are only for executable files             *
//*                                                                            *
//* Future enhancement: If insufficient permission for making changes, offer   *
//* user opportunity to log in as superuser.                                   *
//*                                                                            *
//* Note that only the 'UPDATE' and 'CANCEL' pushbutton controls have hotkeys  *
//* assigned to them. This is not only practical, but also good design. After  *
//* any field is modified, the user can immediately jump to 'UPDATE' or        *
//* 'CANCEL', removing the need to scroll through all the unmodified fields.   *
//******************************************************************************

short FileDlg::dfsModifyFileStats ( short ulY, short ulX, ExpStats& es )
{
#define DEBUG_MFS 0

static const char  UpdateText[] = " ^UPDATE " ;
static const char  CancelText[] = " ^CANCEL " ;
static const char  HelpText[]   = "  ^HELP  " ;
static const char  GeneralHelp[] = {"TAB to desired field(s) and make modifications." } ;
static const char  pButtons[2][5] = { { " NO " }, { "YES " } } ;
static const char  InvalidDate[] = " Invalid Date or Date Format Error!  Try again. " ;
static const char  PermCaution[] = " Caution! You you may have insufficient privilege for this operation. " ;
static const char  PermNone[]    = " You do not have permission to modify file's stats. " ;
static const char  ErrPermBits[] = "Error modifying permission bits. errno==0x" ;
static const char  ErrOwner[]    = "Error changing file owner. errno==0x" ;
static const char  ErrGroup[]    = "Error changing file group. errno==0x" ;
static const char  ErrFDate[]    = "Error changing file dates. errno==0x" ;
static const char  ErrNotSuper[] = " Only Superuser can change file ownership. " ;
static const char  ErrOwnMembr[] = " Must be Superuser OR owner AND member of target group. " ;
gString     modBuff,           // Modification-date string
            accBuff,           // Access-date string
            uidBuff,           // userID string
            gidBuff ;          // grpID string
ExpStats    esNew, *esOld ;
attr_t      dColor  = this->cs.sd,              // dialog color
            // path/filename display color
            pColor = this->cs.scheme == ncbcCOLORS ? this->cs.sd & ~ncrATTR : this->cs.tf,
            cnColor = this->cs.tn,              // controls' non-focus color
            cfColor = this->cs.pf ;             // controls' focus color
short       dialogLines = DIALOG_LINES * 2 - 4 ;// display lines for dialog window
winPos      msgCtr( (short)(dialogLines - 6), (short)(DIALOG_COLS / 2) ) ;
bool        updateStats = false ;               // true if user has requested update
short       status = OK ;


   //* Working copy of the formatted file stats *
   if ( es.fileType == fmLINK_TYPE && es.symbTarg->fileType != fmTYPES )
      esOld = es.symbTarg ;
   else
      esOld = &es ;
   esNew = *esOld ;
   this->FormatDateString ( esOld->modTime, modBuff ) ; // construct the date/time string
   this->FormatDateString ( esOld->accTime, accBuff ) ; // construct the date/time string
   uidBuff.compose( L"%lu", &esOld->userID ) ;          // construct the user ID string
   gidBuff.compose( L"%lu", &esOld->grpID ) ;           // construct the group ID string

   InitCtrl ic[ciDEFINED] =      // array of dialog control info
   {
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - -    pbCANCEL *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      46,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      CancelText,                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbUSRr]                   // nextCtrl:  link in next structure
   },

   {  //* 'USRr' radiobutton  - - - - - - - - - - - - - - - - -      rbUSRr *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->usrProt.read == 'r' ? true : false), // rbSelect:  selected?
      5,                            // ulY:       upper left corner in Y
      9,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "r",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbUSRw],                  // nextCtrl:  link in next structure
   },
   {  //* 'USRw' radiobutton  - - - - - - - - - - - - - - - - -      rbUSRw *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->usrProt.write == 'w' ? true : false), // rbSelect:  selected?
      ic[rbUSRr].ulY,               // ulY:       upper left corner in Y
      short(ic[rbUSRr].ulX + 2),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "w",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbUSRx],                  // nextCtrl:  link in next structure
   },
   {  //* 'USRx' radiobutton  - - - - - - - - - - - - - - - - -      rbUSRx *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->usrProt.exec == 'x' ? true : false), // rbSelect:  selected?
      ic[rbUSRw].ulY,               // ulY:       upper left corner in Y
      short(ic[rbUSRw].ulX + 2),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "x",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbGRPr],                  // nextCtrl:  link in next structure
   },

   {  //* 'GRPr' radiobutton  - - - - - - - - - - - - - - - - -      rbGRPr *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->grpProt.read == 'r' ? true : false), // rbSelect:  selected?
      ic[rbUSRx].ulY,               // ulY:       upper left corner in Y
      short(ic[rbUSRx].ulX + 3),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "r",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbGRPw],                  // nextCtrl:  link in next structure
   },
   {  //* 'GRPw' radiobutton  - - - - - - - - - - - - - - - - -      rbGRPw *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->grpProt.write == 'w' ? true : false), // rbSelect:  selected?
      ic[rbGRPr].ulY,               // ulY:       upper left corner in Y
      short(ic[rbGRPr].ulX + 2),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "w",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbGRPx],                  // nextCtrl:  link in next structure
   },
   {  //* 'GRPx' radiobutton  - - - - - - - - - - - - - - - - -      rbGRPx *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->grpProt.exec == 'x' ? true : false), // rbSelect:  selected?
      ic[rbGRPw].ulY,               // ulY:       upper left corner in Y
      short(ic[rbGRPw].ulX + 2),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "x",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbOTHr],                  // nextCtrl:  link in next structure
   },

   {  //* 'OTHr' radiobutton  - - - - - - - - - - - - - - - - -      rbOTHr *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->othProt.read == 'r' ? true : false), // rbSelect:  selected?
      ic[rbGRPx].ulY,               // ulY:       upper left corner in Y
      short(ic[rbGRPx].ulX + 3),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "r",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbOTHw],                  // nextCtrl:  link in next structure
   },
   {  //* 'OTHw' radiobutton  - - - - - - - - - - - - - - - - -      rbOTHw *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->othProt.write == 'w' ? true : false), // rbSelect:  selected?
      ic[rbOTHr].ulY,               // ulY:       upper left corner in Y
      short(ic[rbOTHr].ulX + 2),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "w",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[rbOTHx],                  // nextCtrl:  link in next structure
   },
   {  //* 'OTHx' radiobutton  - - - - - - - - - - - - - - - - -      rbOTHx *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      (bool)(esOld->othProt.exec == 'x' ? true : false), // rbSelect:  selected?
      ic[rbOTHw].ulY,               // ulY:       upper left corner in Y
      short(ic[rbOTHw].ulX + 2),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "x",                          // dispText:  custom content
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbSUID],                  // nextCtrl:  link in next structure
   },

   {  //* 'SUID' pushbutton   - - - - - - - - - - - - - - - - -      pbSUID *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[rbOTHx].ulY,               // ulY:       upper left corner in Y
      short(ic[rbOTHx].ulX + 3),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      4,                            // cols:      control columns
      (esOld->setUID ? pButtons[1] : pButtons[0]), // dispText:  
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbSGID]                   // nextCtrl:  link in next structure
   },
   {  //* 'SGID' pushbutton   - - - - - - - - - - - - - - - - -      pbSGID *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[pbSUID].ulY,               // ulY:       upper left corner in Y
      short(ic[pbSUID].ulX + 6),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      4,                            // cols:      control columns
      (esOld->setGID ? pButtons[1] : pButtons[0]), // dispText:  
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbSTIK]                   // nextCtrl:  link in next structure
   },
   {  //* 'STIK' pushbutton   - - - - - - - - - - - - - - - - -      pbSTIK *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[pbSGID].ulY,               // ulY:       upper left corner in Y
      short(ic[pbSGID].ulX + 6),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      4,                            // cols:      control columns
      (esOld->sticky ? pButtons[1] : pButtons[0]), // dispText:  
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tbMODd]                   // nextCtrl:  link in next structure
   },

   {  //* 'MOD Date' textbox  - - - - - - - - - - - - - - - - -      tbMODd *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[rbGRPx].ulY + 1),    // ulY:       upper left corner in Y
      ic[rbGRPx].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      modBuff.ustr(),               // dispText:  formatted date/time string
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    valid filename characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tbACCd],                  // nextCtrl:  link in next structure
   },
   {  //* 'ACC Date' textbox  - - - - - - - - - - - - - - - - -      tbACCd *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[tbMODd].ulY + 1),    // ulY:       upper left corner in Y
      ic[tbMODd].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      accBuff.ustr(),               // dispText:  formatted date/time string
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    valid filename characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tbOWNER],                 // nextCtrl:  link in next structure
   },

   {  //* 'OWNER' textbox     - - - - - - - - - - - - - - - - -     tbOWNER *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[tbMODd].ulY,               // ulY:       upper left corner in Y
      48,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      uidBuff.ustr(),               // dispText:  formatted date/time string
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbNumber,                     // filter:    valid filename characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tbGROUP],                 // nextCtrl:  link in next structure
   },
   {  //* 'GROUP' textbox     - - - - - - - - - - - - - - - - -     tbGROUP *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[tbOWNER].ulY + 1),   // ulY:       upper left corner in Y
      ic[tbOWNER].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      gidBuff.ustr(),               // dispText:  formatted date/time string
      cnColor,                      // nColor:    non-focus color
      cfColor,                      // fColor:    focus color
      tbNumber,                     // filter:    valid filename characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbUPDATE],                // nextCtrl:  link in next structure
   },
   {  //* 'UPDATE' pushbutton - - - - - - - - - - - - - - - - -    pbUPDATE *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[pbCANCEL].ulY,             // ulY:       upper left corner in Y
      22,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      UpdateText,                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pbHELP]                   // nextCtrl:  link in next structure
   },
   {  //* 'HELP' pushbutton  - - - - - - - - - - - - - - - - - - - - -  pbHELP *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[pbCANCEL].ulY,             // ulY:       upper left corner in Y
      short(DIALOG_COLS - 10),      // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      HelpText,                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.tf | ncrATTR,        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* If necessary, position the sub-dialog to stay within parent dialog *
   while ( (ulY > (this->dulY)) && ((ulY+dialogLines) >= (this->dulY+this->dRows)) )
      --ulY ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       DIALOG_COLS,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   dfsdp = dp ;                     // give callback method access to the dialog
   dfsColor = dColor ;

   if ( (dp->OpenWindow()) == OK )
   {
      LineDef  lDefH(ncltHORIZ, ncltSINGLE, dialogLines-3, ZERO, DIALOG_COLS, dColor) ;
      dp->DrawLine ( lDefH ) ;
      lDefH.startY = 9 ;
      dp->DrawLine ( lDefH ) ;
      dp->SetDialogTitle ( "  Modify File Stats  ", this->cs.em ) ;
      dp->WriteString ( 14, 13, (char*)GeneralHelp, dColor ) ;
      
      //* Right justify text in tbOWNER and tbGROUP Textboxes *
      dp->SetTextboxCursor ( tbOWNER, tbcpRIGHTJUST ) ;
      dp->SetTextboxCursor ( tbGROUP, tbcpRIGHTJUST ) ;

      //* Configure Mod Date and Access Date Textboxes as fixed-width fields *
      dp->FixedWidthTextbox ( tbMODd, false ) ;    // fixed-width data
      dp->FixedWidthTextbox ( tbACCd, false ) ;
      dp->TextboxAlert ( tbMODd, true ) ;          // complain about bad input
      dp->TextboxAlert ( tbACCd, true ) ;
      dp->SetTextboxInputMode ( true, true ) ;     // force overstrike

      //* Display the current data values *
      // Programmer's Note: The call to dfsDisplayStatData() s the same method 
      // used for static display of the data, even though in this case there 
      // are active control objects where the data are written. It works because 
      // when the display is refreshed, the controls overlay whatever was 
      // written in their space.
      winPos wPos( 1, 2 ) ;
      this->dfsDisplayFilename ( wPos, pColor, esNew.filePath, esNew.fileName, TPATH_MAX ) ;
      DisplayHeaders ( 3, 2, dColor ) ;
      wPos = { 5, 2 } ;
      this->dfsDisplayStatData ( wPos, dColor, *esOld ) ;

      dp->RefreshWin () ;                    // make the text visible

      #if 0    // NOT CURRENTLY USED
      //* Establish a call-back method that can be called from within the 
      //* NcDialog code. Called each time through the GetUserInfo() 
      //* loop so we can manually modify the controls when necessary.
      dp->EstablishCallback ( &ModStat_ControlUpdate ) ;
      #endif   // NOT CURRENTLY USED

      //* Determine whether user has permission to modify the file stats.*
      if (   this->userInfo.userID != superUID && this->userInfo.userID != esOld->userID 
          && this->userInfo.grpID != esOld->grpID )
         InputErrorMsg ( PermCaution, msgCtr, 2 ) ;

      //*******************************************
      //* Allow user to change the various fields *
      //*******************************************
      short    icIndex = ZERO ;           // control index
      bool     done = false ;
      uiInfo   Info ;                     // user interface data returned here
      while ( ! done )
      {
         ModStat_ContextHelp ( icIndex ) ;   // display context help message

         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
            }
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               short tIndex = -1 ;
               switch ( Info.ctrlIndex )
               {
                  case pbCANCEL:          // discard modifications
                     done = true ;
                     break ;
                  case pbUPDATE:          // update file stats
                     updateStats = true ;
                     done = true ;
                     break ;
                  case pbHELP:            // invoke 'info' help
                     //* a) Save our sub-dialog display data
                     //* b) Shell out to Help
                     //* c) On return, re-save the application dialog
                     //*    then restore the sub-dialog display
                     dp->SetDialogObscured () ;
                     this->CallContextHelp ( cxtMOD_STATS ) ;
                     this->dPtr->SetDialogObscured () ;
                     dp->RefreshWin () ;
                     break ;
                  case pbSUID:            // toggle SUID flag
                     esNew.setUID = (bool)(esNew.setUID ? false:true );
                     tIndex = (short)esNew.setUID ;
                     break ;
                  case pbSGID:            // toggle SGID flag
                     esNew.setGID = (bool)(esNew.setGID ? false:true );
                     tIndex = (short)esNew.setGID ;
                     break ;
                  case pbSTIK:            // toggle STIK flag
                     esNew.sticky = (bool)(esNew.sticky ? false:true );
                     tIndex = (short)esNew.sticky ;
                     break ;
                  default:                // invalid index is unlikely
                     break ;
               }
               //* Update the pushbutton display text *
               if ( tIndex >= ZERO )
                  dp->SetPushbuttonText ( icIndex, (char*)pButtons[tIndex] ) ;
            }
            else
            { /* No button press, so nothing to do */ }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input *
            if ( Info.viaHotkey == false )
               icIndex = dp->EditRadiobutton ( Info ) ;
            else
               Info.HotData2Primary () ;

            //* If a radio button state has changed *
            if ( Info.ctrlType == dctRADIOBUTTON && Info.dataMod != false )
            {
               switch ( Info.ctrlIndex )
               {
                  case rbUSRr: esNew.usrProt.read  = (char)(Info.isSel ? 'r' : '-') ; break ;
                  case rbUSRw: esNew.usrProt.write = (char)(Info.isSel ? 'w' : '-') ; break ;
                  case rbUSRx: esNew.usrProt.exec  = (char)(Info.isSel ? 'x' : '-') ; break ;
                  case rbGRPr: esNew.grpProt.read  = (char)(Info.isSel ? 'r' : '-') ; break ;
                  case rbGRPw: esNew.grpProt.write = (char)(Info.isSel ? 'w' : '-') ; break ;
                  case rbGRPx: esNew.grpProt.exec  = (char)(Info.isSel ? 'x' : '-') ; break ;
                  case rbOTHr: esNew.othProt.read  = (char)(Info.isSel ? 'r' : '-') ; break ;
                  case rbOTHw: esNew.othProt.write = (char)(Info.isSel ? 'w' : '-') ; break ;
                  case rbOTHx: esNew.othProt.exec  = (char)(Info.isSel ? 'x' : '-') ; break ;
                  default: break ;              // invalid index is unlikely
               }
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* Get user input *
            // Programmer's Note: User could set input mode to INSERT in 
            // mid-edit, which would screw up our text box data format.
            // So, just in case, we set OVERSTRIKE mode continuously 
            // within the callback method.
            Info.viaHotkey = false ;      // ignore hotkey data (if any)
            icIndex = dp->EditTextbox ( Info ) ;

            //* If text box data have changed *
            if ( Info.ctrlType == dctTEXTBOX && Info.dataMod != false )
            {
               switch ( Info.ctrlIndex )
               {
                  case tbMODd:
                     //* Validate and convert the date/time string to          *
                     //* localTime class format.                               *
                     //* See note in header of ValidateDateString() method     *
                     //* about datestamp limit.                                *
                     dp->GetTextboxText ( tbMODd, modBuff ) ;
                     if ( (this->ValidateDateString ( esNew.modTime, modBuff )) == false )
                     {
                        icIndex = dp->PrevControl () ;
                        //* Restore original date/time string, *
                        //* and give user another chance.      *
                        this->FormatDateString ( esOld->modTime, modBuff ) ;
                        dp->SetTextboxText ( tbMODd, modBuff ) ;
                        //* Give user another chance *
                        InputErrorMsg ( InvalidDate, msgCtr, 2 ) ;
                     }
                     break ;
                  case tbACCd:
                     //* Validate and convert the date/time string * 
                     //* to localTime class format                 *
                     dp->GetTextboxText ( tbACCd, accBuff ) ;
                     if ( (this->ValidateDateString ( esNew.accTime, accBuff )) == false )
                     {
                        icIndex = dp->PrevControl () ;
                        //* Restore original date/time string, *
                        //* and give user another chance.      *
                        this->FormatDateString ( esOld->accTime, accBuff ) ;
                        dp->SetTextboxText ( tbACCd, accBuff ) ;
                        InputErrorMsg ( InvalidDate, msgCtr, 2 ) ;
                     }
                     break ;
                  case tbOWNER:
                     //* Only super-user may change file ownership *
                     if ( this->userInfo.userID == superUID )
                     {
                        dp->GetTextboxText ( tbOWNER, uidBuff ) ;
                        swscanf ( uidBuff.gstr(), L"%u", &esNew.userID ) ;
                     }
                     else
                     {
                        //* Restore original text-box text for UID *
                        InputErrorMsg ( ErrNotSuper, msgCtr, 2 ) ;
                        dp->NextControl () ; // temporarily shift input focus
                        dp->SetTextboxText ( tbOWNER, uidBuff ) ;
                        dp->PrevControl () ; // restore input focus
                     }
                     break ;
                  case tbGROUP:
                     //* To change file's group ID, user must be super-user OR *
                     //* must own the file AND be a member of the target group.*
                     {     // (fool the compiler)
                     gString gidb ;
                     dp->GetTextboxText ( tbGROUP, gidb ) ;
                     ULONG   tempgid ;
                     swscanf ( gidb.gstr(), L"%lu", &tempgid ) ;
                     if ( this->userInfo.userID == superUID ||
                          (this->userInfo.userID == esOld->userID 
                           && (this->IsGroupMember (tempgid))) )
                     {
                        esNew.grpID = tempgid ;
                        gidBuff = gidb ;
                     }
                     else
                     {
                        //* Restore original text-box text for GID *
                        InputErrorMsg ( ErrOwnMembr, msgCtr, 3 ) ;
                        dp->NextControl () ; // temporarily shift input focus
                        dp->SetTextboxText ( tbGROUP, gidBuff ) ;
                        dp->PrevControl () ; // restore input focus
                     }
                     }
                     break ;
                  default:
                     break ;
               }
            }
         }

         //* Move focus to appropriate control *
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while()
      
      //* If user has selected the UPDATE pushbutton *
      if ( updateStats != false )
      {
         #if DEBUG_MFS != 0
         char chyes = 'Y', chno = 'N' ;
         gString b, md, ad, sd ;
         this->FormatDateString ( esNew.modTime, md ) ;
         this->FormatDateString ( esNew.accTime, ad ) ;
         this->FormatDateString ( esNew.staTime, sd ) ;
         b.compose( L"STATS: %c%c%c  %c%c%c  %c%c%c  %c%c%c  M:%s A:%s C:%s", 
                     &esNew.usrProt.read, &esNew.usrProt.write, &esNew.usrProt.exec,
                     &esNew.grpProt.read, &esNew.grpProt.write, &esNew.grpProt.exec,
                     &esNew.othProt.read, &esNew.othProt.write, &esNew.othProt.exec,
                     (char*)(esNew.setUID ? &chyes : &chno), 
                     (char*)(esNew.setGID ? &chyes : &chno),
                     (char*)(esNew.sticky ? &chyes : &chno), 
                     md.ustr(), ad.ustr(), sd.ustr() ) ;
         dp->DebugMsg ( b.ustr() ) ;
         #endif   // DEBUG_MFS

         //****************************************************************
         //* If values modified, and user has permission, update the file *
         //* Permissions: 1) user is superuser, OR                        *
         //*              2) user is file's owner, OR                     *
         //*              3) user is a member of the file's group         *
         //****************************************************************
         if ( updateStats != false &&
              (this->userInfo.userID == superUID || this->userInfo.userID == esOld->userID 
               || this->userInfo.grpID == esOld->grpID) ) 
         {
            //* Set read/write/execute permissions, setUID, setGID, and sticky.*
            //* If user is superuser OR user owns the file, chmod() will modify*
            //* permissions. We update the permission bits on the file even if *
            //* there have been no changes made, because this will update the  *
            //* cTime (staTime) timestamp which otherwise cannot be updated    *
            //* directly.                                                      *
            char msgBuff[64] ;
            short sfpStatus = this->SetFilePermissions ( esNew ) ;
            if ( sfpStatus == OK )
            {
               status = MODNOERR ;

               //* If difference in userID or groupID, set new user and/or group *
               if ( esNew.userID != esOld->userID || esNew.grpID != esOld->grpID )
               {
                  if ( esNew.userID != esOld->userID )
                  {
                     //* Only superuser has authority to modify file ownership *
                     if ( this->userInfo.userID == superUID )
                     {
                        sfpStatus = this->SetFileOwner ( esNew ) ;
                        if ( sfpStatus != OK )
                        {
                           snprintf ( msgBuff, 64, " %s%04hX ", ErrOwner, sfpStatus ) ;
                           InputErrorMsg ( msgBuff, msgCtr, 3 ) ;
                           status = MODERR ;
                        }
                     }
                     else
                     {
                        InputErrorMsg ( ErrNotSuper, msgCtr, 3 ) ;
                        status = MODERR ;
                     }
                  }
                  if ( status == MODNOERR && esNew.grpID != esOld->grpID )
                  {
                     //* If user is super-user OR owns the file AND is a *
                     //* member of the specified group.                  *
                     if ( this->userInfo.userID == superUID || 
                          (this->userInfo.userID == esOld->userID 
                           && (this->IsGroupMember ( esNew.grpID ))) )
                     {
                        sfpStatus = this->SetFileGroup ( esNew ) ;
                        if ( sfpStatus != OK )
                        {
                           snprintf ( msgBuff, 64, " %s%04hX ", ErrGroup, sfpStatus ) ;
                           InputErrorMsg ( msgBuff, msgCtr, 3 ) ;
                           status = MODERR ;
                        }
                     }
                  }
               }  // change in user ID or group ID
               
               //* If file mod time or access time data has been modified *
               bool  modMod   = !(esNew.modTime == esOld->modTime),
                     accMod   = !(esNew.accTime == esOld->accTime) ;
               if ( status == MODNOERR && (modMod != false || accMod != false) )
               {
                  bool sameDate = esNew.modTime == esNew.accTime ;
                  sfpStatus = this->SetFileTimestamp ( esNew, modMod, accMod, sameDate ) ;
                  if ( sfpStatus != OK )
                  {
                     snprintf ( msgBuff, 64, " %s%04hX ", ErrFDate, sfpStatus ) ;
                     InputErrorMsg ( msgBuff, msgCtr, 3 ) ;
                     status = MODERR ;
                  }
               }
            }
            else  // error in modifying permission bits 
            {     // (bits may or may not have been modified)
               snprintf ( msgBuff, 64, " %s%04hX ", ErrPermBits, sfpStatus ) ;
               InputErrorMsg ( msgBuff, msgCtr, 3 ) ;
               status = MODERR ;
            }
            //* Update the data in the file display control.           *
            //* (any clipboard data from current dir will be cleared)  *
            //* Advance highlight to the file we just modified.        *
            this->RefreshCurrDir () ;
            this->HilightItemByName ( esNew.fileName ) ;
         }
         else
         {
            InputErrorMsg ( PermNone, msgCtr, 3 ) ;
            status = ERR ;
         }
      }  // if(icIndex==pbUPDATE)
   }  // if(OpenWindow)
   else
   {
      dtbmData   msgData( " Unable to Open Dialog Window! " ) ;
      this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;
      chrono::duration<short>aWhile( 5 ) ;
      this_thread::sleep_for( aWhile ) ;
   }
   if ( dp != NULL )
   {
      delete ( dp ) ;               // close the window
      dfsdp = NULL ;                // tidy up
      dfsColor = ZERO ;
   }
   return status ;

#undef DEBUG_MFS
}  //* End dfsModifyFileStats() *

#if 0    // NOT CURRENTLY USED
//*************************
//* ModStat_ControlUpdate *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* ModifyFileStats() dialog. This method is called each time the              *
//* NcDialog::EditXXXX() methods goes through its user-input loop.             *
//*                                                                            *
//* This method handles situations that the dialog code cannot.                *
//*  - (none at this time)                                                     *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: (optional, false by default) first-time                *
//*                     processing                                             *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

static short ModStat_ControlUpdate ( short currIndex, const wkeyCode wkey, 
                                     bool firstTime )
{
   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      // no initialization currently required
   }

   //***********************
   //* Standard processing *
   //***********************

   return OK ;

}  //* End ModStat_ControlUpdate() *
#endif   // NOT CURRENTLY USED

//*************************
//*  ModStat_ContextHelp  *
//*************************
//******************************************************************************
//* Display context help for specified control of 'ModifyFileStats' dialog.    *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void ModStat_ContextHelp ( short currIndex )
{
const char* ContextClear = 
"                                                                        " ;
const char* ContextSetTog = ": Press Enter to set bit, Space to toggle." ;
const char* ContextToggle = ": Press Enter or Space to toggle." ;
const char* ContextDate   = " using specified format.(ESC discards edit)" ;
const char* ContextOwner  = ": Must have super-user access." ;
const char* ContextGroup  = ": Must have super-user access or file ownership." ;
const char* ContextHelp[] = 
{
   "Press CANCEL to discard changes.",
   "USER Read permission",
   "USER Write permission",
   "USER Execute permission",
   "GROUP Read permission",
   "GROUP Write permission",
   "GROUP Execute permission",
   "OTHER Read permission",
   "OTHER Write permission",
   "OTHER Execute permission",
   "Set effective UID on execution",
   "Set effective GID on execution",
   "Restrict access to directory",
   "Set 'Last Modified' date/time",
   "Set 'Last Accessed' date/time",
   "Set file owner ID number",
   "Set file group ID number",
   "Press UPDATE to save changes.",
   "Invoke online documentation."
} ;

winPos   wp( 10, 2 ) ;

   //* Display instructions for the dialog control that currently has focus *
   dfsdp->WriteString ( wp, ContextClear, dfsColor ) ;
   wp = dfsdp->WriteString ( wp, ContextHelp[currIndex], dfsColor ) ;
   if ( currIndex >= rbUSRr && currIndex <= rbOTHx )
      dfsdp->WriteString ( wp.ypos, wp.xpos, ContextSetTog, dfsColor ) ;
   else if ( currIndex >= pbSUID && currIndex <= pbSTIK )
      dfsdp->WriteString ( wp.ypos, wp.xpos, ContextToggle, dfsColor ) ;
   else if ( currIndex >= tbMODd && currIndex <= tbACCd )
      dfsdp->WriteString ( wp.ypos, wp.xpos, ContextDate, dfsColor ) ;
   else if ( currIndex == tbOWNER )
      dfsdp->WriteString ( wp.ypos, wp.xpos, ContextOwner, dfsColor ) ;
   else if ( currIndex == tbGROUP )
      dfsdp->WriteString ( wp.ypos, wp.xpos, ContextGroup, dfsColor ) ;
   dfsdp->RefreshWin () ;

}  //* End ModStat_ContextHelp() *

//*************************
//*  dfsDisplayFilename   *
//*************************
//******************************************************************************
//* Display path/filename + optional label (no display refresh).               *
//* If necessary, compress string to fit in alloted space.                     *
//* (used only by the DisplayFileStats() group of methods)                     *
//*                                                                            *
//* Input  : wp   : initial cursor position                                    *
//*          color: text color                                                 *
//*          fPath: pointer to path string                                     *
//*          fName: pointer to filename string                                 *
//*          maxDisp: maximum display-string columns                           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::dfsDisplayFilename ( winPos& wp, attr_t color, const char* fPath, 
                                   const char* fName, short maxDisp )
{
   gString outgs( fPath ) ;
   if ( (outgs.compare( ROOT_PATH )) == ZERO )
      outgs.append( fName ) ;
   else
      outgs.append( "/%s", fName ) ;

   //* If path too wide for dialog window, compress it before display *
   if ( outgs.gscols() > maxDisp )
      this->TrimPathString ( outgs, maxDisp ) ;

   dfsdp->WriteString ( wp.ypos, wp.xpos, outgs, color ) ;

}  //* End dfsDisplayFilename() *

//*************************
//*    DisplayHeaders     *
//*************************
//******************************************************************************
//* Display column headings (no display refresh).                              *
//*                                                                            *
//* Input  : ypos / xpos: initial cursor position                              *
//*          color: text color                                                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void DisplayHeaders ( short ypos, short xpos, attr_t color )
{

   dfsdp->WriteString ( ypos, xpos, Head1, color ) ; 
   dfsdp->WriteString ( ypos+1, xpos, Head2, color ) ;

}  //* End DisplayHeaders() *

//*************************
//*    DisplayStatData    *
//*************************
//******************************************************************************
//* Display the primary file attribute data (no display refresh).              *
//*                                                                            *
//* Input  : wp   : initial cursor position (by reference)                     *
//*          color: text color                                                 *
//*          es: structure containing formatted file stat data (by ref)        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::dfsDisplayStatData ( winPos& wp, attr_t color, ExpStats& es )
{
   const char  *ys = "YES ", *no = " NO ", *setUID, *setGID, *sticky ;
   gString outBuff, fsBuff, dateBuff ;

   //* Format filesize value to fit in field *
   fsBuff.formatInt( es.fileSize, 9 ) ;

   //* Convert bools to strings *
   setUID = es.setUID ? ys : no ;
   setGID = es.setGID ? ys : no ;
   sticky = es.sticky ? ys : no ;

   //* Output the data *
   outBuff.compose( L"%s  %c %c %c  %c %c %c  %c %c %c  %s  %s  %s%10llu  %4llu  %S", 
                  FTypes[es.fileType], 
                  &es.usrProt.read, &es.usrProt.write, &es.usrProt.exec, 
                  &es.grpProt.read, &es.grpProt.write, &es.grpProt.exec, 
                  &es.othProt.read, &es.othProt.write, &es.othProt.exec, 
                  setUID, setGID, sticky, &es.iNode, &es.hLinks, fsBuff.gstr() ) ;
   dfsdp->WriteString ( wp.ypos++, wp.xpos, outBuff, color ) ;

   //* Mod datetime *
   this->FormatDateString ( es.modTime, dateBuff ) ;
   outBuff.compose( L"Last Modified   : %S  | UID:%8u NAME: %s", 
                    dateBuff.gstr(), &es.userID, es.ownerName ) ;
   dfsdp->WriteString ( wp.ypos++, wp.xpos, outBuff, color ) ;

   //* Access datetime *
   this->FormatDateString ( es.accTime, dateBuff ) ;
   outBuff.compose( L"Last Accessed   : %S  | GID:%8u NAME: %s", 
                    dateBuff.gstr(), &es.grpID, es.groupName ) ;
   dfsdp->WriteString ( wp.ypos++, wp.xpos, outBuff, color ) ;

   //* Stat datetime *
   this->FormatDateString ( es.staTime, dateBuff ) ;
   outBuff.compose( L"Last Stat Update: %S  |", dateBuff.gstr() ) ;
   dfsdp->WriteString ( wp.ypos, wp.xpos, outBuff, color ) ;

   #if 0    // Experimental - Test maximum value that is accepted by the kernel
   outBuff.compose( L"mdate:%10lu 0x%08lX", 
                    &es.rawStats.st_mtime, &es.rawStats.st_mtime ) ;
   dfsdp->WriteString ( wp.ypos, wp.xpos+42, outBuff, color ) ;
   #endif   // Experimental

}  //* End dfsDisplayStatData() *

//*************************
//*     IsGroupMember     *
//*************************
//******************************************************************************
//* Determine whether user belongs to the specified group.                     *
//*                                                                            *
//* The full list of groups the user belongs to is stored in                   *
//* this->userInfo.suppGrp[].                                                  *
//*                                                                            *
//* Input  : group ID number to be compared                                    *
//*                                                                            *
//* Returns: true if member of group, else false                               *
//******************************************************************************
//* Note that this is not a definitive test because only the first MAX_sGID    *
//* group ID numbers are stored locally.                                       *
//******************************************************************************

bool FileDlg::IsGroupMember ( gid_t gID )
{
bool        isMember = false ;

   for ( short i = ZERO ; i < this->userInfo.suppGroups ; i++ )
   {
      if ( this->userInfo.suppGID[i] == gID )
      {
         isMember = true ;
         break ;
      }
   }
   return isMember ;

}  //* End IsGroupMember() *

//***********************
//*   InputErrorMsg     *
//***********************
//******************************************************************************
//* Display an error message, indicating the user's questionable parentage.    *
//*                                                                            *
//* Input  : msg  : message text                                               *
//*          ctr  : center of message-display area                             *
//*          pause: number of seconds to pause before erasing message          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void InputErrorMsg ( const char* msg, winPos ctr, short pause )
{
short    sLen = strlen ( msg ),
         xpos = ctr.xpos - sLen / 2 ;

   dfsdp->WriteString ( ctr.ypos, xpos, msg, (dfsColor & ~ncrATTR), true ) ;
   chrono::duration<short>aWhile( pause ) ;
   this_thread::sleep_for( aWhile ) ;
   dfsdp->ClearLine ( ctr.ypos, false ) ;

}  //* End InputErrorMsg() *

//*************************
//*  SetFilePermissions   *
//*************************
//******************************************************************************
//* Set file permissions for the specified file: USR rwx, GRP rwx, OTHER rwx,  *
//* sUID, sGID, and sticky bit.                                                *
//* (requires either superuser access or ownership of the file.)               *
//*                                                                            *
//* Input  : ExpStats structure with at least the following fields             *
//*          initialized: fileName, filePath, usrProt, grpProt, othProt,       *
//*                       setUID, setGID, sticky                               *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'.                         *
//******************************************************************************

short FileDlg::SetFilePermissions ( ExpStats& es )
{
gString  pathBuff ;
mode_t   newMode = ZERO ;
short    status ;

   //* Construct the path/filename string *
   this->fmPtr->CatPathFname ( pathBuff, es.filePath, es.fileName ) ;

   //* Construct the mode word *
   if ( es.setUID != false )        newMode |= S_ISUID ;
   if ( es.setGID != false )        newMode |= S_ISGID ;
   if ( es.sticky != false )        newMode |= S_ISVTX ;
   if ( es.usrProt.read == 'r' )    newMode |= S_IRUSR ;
   if ( es.usrProt.write == 'w' )   newMode |= S_IWUSR ;
   if ( es.usrProt.exec == 'x' )    newMode |= S_IXUSR ;
   if ( es.grpProt.read == 'r' )    newMode |= S_IRGRP ;
   if ( es.grpProt.write == 'w' )   newMode |= S_IWGRP ;
   if ( es.grpProt.exec == 'x' )    newMode |= S_IXGRP ;
   if ( es.othProt.read == 'r' )    newMode |= S_IROTH ;
   if ( es.othProt.write == 'w' )   newMode |= S_IWOTH ;
   if ( es.othProt.exec == 'x' )    newMode |= S_IXOTH ;

   //* Call the FMgr class to do the actual work *
   status = this->fmPtr->SetFilePermissions ( pathBuff.ustr(), newMode ) ;
   if ( status != OK )
      this->SetErrorCode ( ecPACCV ) ;

   return status ;

}  //* End SetFilePermissions() *

//*************************
//*     SetFileOwner      *
//*************************
//******************************************************************************
//* Set file ownership (userID).                                               *
//* (requires superuser access)                                                *
//*                                                                            *
//* Note that the setUID and setGID bits may be reset depending on Linux       *
//* kernel version.                                                            *
//*                                                                            *
//* Input  : ExpStats structure with at least the following fields initialized:*
//*            fileName, filePath, userID                                      *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'.                         *
//******************************************************************************

short FileDlg::SetFileOwner ( ExpStats& es )
{
gString  pathBuff ;
short    status ;

   //* Construct the path/filename string *
   this->fmPtr->CatPathFname ( pathBuff, es.filePath, es.fileName ) ;

   //* Call the FMgr class to do the actual work *
   status = fmPtr->SetFileOwnerGroup ( pathBuff.ustr(), es.userID, -1 ) ;

   return status ;

}  //* End SetFileOwner() *

//*************************
//*     SetFileGroup      *
//*************************
//******************************************************************************
//* Set file group (grpID).                                                    *
//* (requires either superuser access or ownership of the file)                *
//* (if non-superuser, owner must be a member of the specified group)          *
//*                                                                            *
//* Note that the setUID and setGID bits may be reset depending on Linux       *
//* kernel version and on whether the user is superuser or file owner.         *
//*                                                                            *
//*                                                                            *
//* Input  : ExpStats structure with at least the following fields             *
//*          initialized: fileName, filePath, userID, grpID                    *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'.                         *
//******************************************************************************

short FileDlg::SetFileGroup ( ExpStats& es )
{
gString  pathBuff ;
short    status ;

   //* Construct the path/filename string *
   this->fmPtr->CatPathFname ( pathBuff, es.filePath, es.fileName ) ;

   //* Call the FMgr class to do the actual work *
   status = fmPtr->SetFileOwnerGroup ( pathBuff.ustr(), -1, es.grpID ) ;

   return status ;

}  //* End SetFileGroup() *

//*************************
//*   SetFileTimestamp    *
//*************************
//******************************************************************************
//* Set file Modify and/or Access date/timestamps.                             *
//* As a side effect, cTime will be set to current system time.                *
//* (requires file write permission)                                           *
//*                                                                            *
//*                                                                            *
//* Input  : pathFilename: full path/filename specification                    *
//*          newDate     : structure must be correctly initialized             *
//*                        (except for 'day' which is ignored)                 *
//*          updateMod   : (optional) if true, update modification time        *
//*          updateAcc   : (optional) if true, update access time              *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'                          *
//******************************************************************************

short FileDlg::SetFileTimestamp ( const char* PathFilename, const localTime& newDate, 
                                  bool updateMod, bool updateAcc )
{
short status = this->fmPtr->SetFileTimestamp ( PathFilename, newDate, 
                                               updateMod, updateAcc ) ;

   return status ;

}  //* End SetFileTimestamp() *

//*************************
//*   SetFileTimestamp    *
//*************************
//******************************************************************************
//* Set file Modify and/or Access date/timestamps.                             *
//* As a side effect, cTime will be set to current system time.                *
//* (requires file write permission)                                           *
//*                                                                            *
//*                                                                            *
//* Input  : ExpStats structure with the following fields initialized:         *
//*            1. fileName == valid file name                                  *
//*            2. filePath == valid file path                                  *
//*            3. if updateMod != false, then modTime == new date/time data    *
//*                                      (except for 'day' which is ignored)   *
//*            4. if updateAcc != false, then accTime == new date/time data    *
//*                                      (except for 'day' which is ignored)   *
//*                updateMod : if true, update modification time               *
//*                updateAcc : if true, update access time                     *
//*                updateAll : (optional, false by default)                    *
//*                            if true, set BOTH Mod and Access dates using    *
//*                            data in modTime field                           *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'                          *
//******************************************************************************

short FileDlg::SetFileTimestamp ( const ExpStats& fileStats, bool updateMod, 
                                  bool updateAcc, bool updateAll )
{
gString  pathBuff ;
short    status = ZERO ;

   //* Construct the path/filename of the target *
   this->fmPtr->CatPathFname ( pathBuff, fileStats.filePath, fileStats.fileName ) ;
   if ( updateAll != false )
      status = this->fmPtr->SetFileTimestamp ( pathBuff.ustr(), 
                                               fileStats.modTime, true, true ) ;
   else
   {
      if ( updateMod != false )
         status = this->fmPtr->SetFileTimestamp ( pathBuff.ustr(), 
                                                  fileStats.modTime, true, false ) ;
      if ( status == ZERO && updateAcc != false )
         status = this->fmPtr->SetFileTimestamp ( pathBuff.ustr(), 
                                                  fileStats.accTime, false, true ) ;
   }
   return status ;

}  // End SetFileTimestamp() *

//*************************
//*     ExpandStats       *
//*************************
//******************************************************************************
//* Expand and format for human readability the 'stat' info for the file       *
//* specified by fs structure.                                                 *
//* Pass-through method to FMgr class method of the same name.                 *
//*                                                                            *
//*                                                                            *
//* Input  : fs   : pointer to FileStats structure (stat64) returned by        *
//*                 the GNU C library stat64() and lstat64() functions.        *
//*                 (Note: for displayed files and files on the    )           *
//*                 (clipboard, this structure is the 'rawStats'   )           *
//*                 (member of the tnFName class object.           )           *
//*          es   : ExpStats structure for holding expanded statistics         *
//*                 a) caller (optionally) initializes es.filePath with        *
//*                    the path string for the file (excluding filename)       *
//*                    (This is needed to 'stat' a symbolic link target.)      *
//*                 b) if the specified file is a symbolic link,               *
//*                    caller (optionally) initializes es.symbTarg             *
//*                    referencing a second ExpStats structure to              *
//*                    hold the info for the symbolic link's target.           *
//*                 (other fields of es initialized on return)                 *
//*                                                                            *
//* Return : OK if successful, else ERR                                        *
//*             ERR indicates a problem with the symbolic link target:         *
//*               a) invalid es.filePath, or                                   *
//*               b) specified file is a symbolic link, but es.symbTarg        *
//*                  does not point to an ExpStats structure to hold           *
//*                  target file's information, or                             *
//*               c) broken symbolic link                                      *
//*                  Note: if ERR returned, and cause of error was a           *
//*                   broken symbolic link, then                               *
//*                es.symbTarg->fileType == fmTYPES,                           *
//*                es.symbTarg->fileName == filename of missing target         *
//*                                         if available, else "unknown"       *
//*                es.symbTarg->filePath == path of missing target             *
//*                                         if available, else "unknown"       *
//*                   and all other fields of es.symbTarg are undefined.       *
//******************************************************************************

short FileDlg::ExpandStats ( const FileStats* fs, ExpStats& es )
{

   return this->fmPtr->ExpandStats ( fs, es ) ;

}  //* End ExpandStats *

//***********************
//* GetFilesystemStats  *
//***********************
//******************************************************************************
//* Returns a copy of the file system information for the file system          *
//* that includes the 'current' directory.                                     *
//*                                                                            *
//* This is a pass-through method for FMgr class method of the same name.      *
//*                                                                            *
//* Input  : trgPath : filespec of file or directory to be scanned             *
//*          fsStats : (by reference, initial values ignored)                  *
//*                    receives filesystem information                         *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*           All members of 'fsStats' for which the target filesystem         *
//*           provides data have been initialized.                             *
//*           Note: Some filesystems do not provide data for all fields.       *
//*          if error, 'errno' is returned (some fsStats members unreliable)   *
//******************************************************************************

short FileDlg::GetFilesystemStats ( const gString& trgPath, fileSystemStats& fsStats )
{

   return ( (this->fmPtr->GetFilesystemStats ( trgPath, fsStats )) ) ;

}  //* End GetFilesystemStats() *

//*************************
//*    USB_DeviceStats    *
//*************************
//******************************************************************************
//* Scan for devices attached to the Universal Serial Bus (USB).               *
//* Capture filesystem and support information for the devices.                *
//* This method is designed primarily for smartphones, tablets and other       *
//* virtual filesystems attached to the USB bus. Devices may, or may not be    *
//* mounted and visible to the user.                                           *
//*                                                                            *
//* Optionally, this method can report the basic bus, device, ID and           *
//* descriptive text information for each device attached to the USB bus.      *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* This is a pass-through method for FMgr class method of the same name.      *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbdPtr ;                       *
//*                                                                            *
//* Input  : usbdPtr  : (by reference, initial value ignored)                  *
//*                     receives pointer to a dynamically-allocated array of   *
//*                     usbDevice objects. Set to NULL pointer if no USB       *
//*                     devices matching the criteria are found.               *
//*          testmnt  : (optional, 'false' by default)                         *
//*                     if 'false', do not test for mounted status             *
//*                     if 'true',  for each MTP/GVFS device determine whether *
//*                                 it is currently mounted                    *
//*          all      : (optional, 'false' by default)                         *
//*                     if 'false', report only MTP/GVFS devices               *
//*                     if 'true',  summary report of all devices attached     *
//*                                 to USB bus                                 *
//*                                                                            *
//* Returns: count of USB devices captured (elements in usbDevice array)       *
//******************************************************************************

short FileDlg::USB_DeviceStats ( usbDevice*& usbdPtr, bool testmnt, bool all )
{

   return ( (this->fmPtr->USB_DeviceStats ( usbdPtr, testmnt, all )) ) ;

}  //* End USB_DeviceStats() *

//*************************
//*     USB_DeviceID      *
//*************************
//******************************************************************************
//* Determine the device-driver path associated with the specified URI.        *
//*                                                                            *
//* The system often increments the device index of attached USB devices.      *
//* This happens most often when a mount operation for a USB device fails.     *
//* When this happens, the the device path must be updated before the mount    *
//* can be retried.                                                            *
//*                                                                            *
//* Input  : uri   : (by reference) URI of target device                       *
//*          devid : (by reference) receives device-driver path                *
//*                                                                            *
//* Returns: 'true'  if device ID found                                        *
//*          'false' if no matching device found                               *
//******************************************************************************

bool FileDlg::USB_DeviceID ( const char* uri, gString& devid )
{
   gString gs( uri ) ;           // working copy of URI
   usbDevice *usbdPtr = NULL ;   // pointer to dynamic allocation
   short usbCount = ZERO ;       // number of USB mtp devices identified
   bool status = false ;         // return value
   devid.clear() ;               // initialize caller's parameter

   usbCount = this->fmPtr->USB_DeviceStats ( usbdPtr, true ) ;

   for ( short usbIndex = ZERO ; usbIndex < usbCount ; ++usbIndex )
   {
      if ( (gs.compare( usbdPtr[usbIndex].uri )) == ZERO )
      {
         devid = usbdPtr[usbIndex].devpath ;
         status = true ;
         break ;
      }
   }
   if ( usbdPtr != NULL )     // release the dynamic memory allocation
   { delete [] usbdPtr ; usbdPtr = NULL ; }

   return status ;

}  //* End USB_DeviceID() *

//*************************
//*     uri2Filespec      *
//*************************
//******************************************************************************
//* Construct the full filespec for the specified filesystem based on the      *
//* Uniform Resource Identifier (URI).                                         *
//* 1) MTP/GVfs URI:                                                           *
//*    Example URI: mtp://Android_ce012345c012345d01/                          *
//*    Filespec   : /run/user/1000/gvfs/mtp:host=Android_ce012345c012345d01    *
//* 2) Audio Disc URI:                                                         *
//*    Example URI: cdda://sr0/                                                *
//*    Filespec   : /run/user/1000/gvfs/cdda:host=sr0                          *
//*                                                                            *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : pathSpec : (by reference) receives the target filespec            *
//*          uri      : URI for an MTP/GVfs filesystem                         *
//*                                                                            *
//* Returns: 'true' if valid filespec,       (pathSpec initialized)            *
//*          'false' if invalid or unrecognized 'uri' format (pathSpec cleared)*
//******************************************************************************

bool FileDlg::Uri2Filespec ( gString& pathSpec, const char* uri )
{

   return ( (this->fmPtr->Uri2Filespec ( pathSpec, uri )) ) ;

}  //* End Uri2Filespec() *

//*************************
//*    DVD_DeviceStats    *
//*************************
//******************************************************************************
//* Scan for drives attached to the local system.                              *
//* Capture filesystem and support information for the devices.                *
//*                                                                            *
//* This method is designed primarily for locating optical drives              *
//* (DVD,CD,BLU-RAY, etc). Data discs contained in the drive may, or may not   *
//* be mounted and visible to the user.                                        *
//*                                                                            *
//* Optionally, this method can report information for all attached drives.    *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* This is a pass-through method for FMgr class method of the same name.      *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbDev ;                        *
//*                                                                            *
//* Input  : usbdPtr  : (by reference, initial value ignored)                  *
//*                     receives pointer to a dynamically-allocated array of   *
//*                     usbDevice objects. Set to NULL pointer if no optical   *
//*                     devices matching the criteria are found.               *
//*          testmnt  : (optional, 'false' by default)                         *
//*                     if 'false', do not test for mounted status             *
//*                     if 'true',  for each optical drive identified,         *
//*                                 determine whether it is currently mounted  *
//*          all      : (optional, 'false' by default)                         *
//*                     if 'false', report only optical drive devices          *
//*                     if 'true',  summary report of all drives identified    *
//*                                                                            *
//* Returns: count of optical drives captured (elements in usbDevice array)    *
//******************************************************************************

short FileDlg::DVD_DeviceStats ( usbDevice*& usbdPtr, bool testmnt, bool all )
{

   return ( (this->fmPtr->DVD_DeviceStats ( usbdPtr, testmnt, all )) ) ;

}  //* End DVD_DeviceStats() *

//*************************
//*      EjectMedia       *
//*************************
//******************************************************************************
//* Eject the removable media from the specified device, or if no device       *
//* specified, then eject the media from the first DVD/CD drive.               *
//*    Example: EjectMedia( "/dev/sr0" ) ;                                     *
//* Typical devices with ejectable media: CD-ROM, DVD-ROM, floppy disk, tape   *
//* drive, ZIP disk or USB disk.                                               *
//*                                                                            *
//* 1) Not all optical drives can be closed under software control.            *
//* 2) If target device is mounted and writable, be sure to close the output   *
//*    stream or unmount the device before ejecting the tray to avoid potential*
//*    loss of data.                                                           *
//* 3) If target device does not support removable media, the operation will   *
//*    silently fail.                                                          *
//* 4) The 'close' and 'toggle' flags are mutually exclusive. If 'toggle' is   *
//*    set, then 'close' is ignored.                                           *
//*                                                                            *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : device : (optional, NULL pointer by default)                      *
//*                   If not specified, media will be ejected from the         *
//*                    system's default DVD/CD drive: "/dev/cdrom"             *
//*                   If specified, device whose media are to be ejected.      *
//*          close  : (optional, 'false' by default)                           *
//*                   If 'false', open the tray.                               *
//*                   If 'true',  close it tray.                               *
//*          toggle : (optional, 'false' by default)                           *
//*                   If 'false', ignore.                                      *
//*                   If 'true', toggle the state of the tray.                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::EjectMedia ( const char* device, bool close, bool toggle )
{

   this->fmPtr->EjectMedia ( device, close, toggle ) ;

}  //* End EjectMedia

//*************************
//* DisplayFileSystemInfo *
//*************************
//******************************************************************************
//* Displays the file system information for the file system that includes     *
//* the 'current' directory. OR optionally, display file system info for       *
//* specified path.                                                            *
//*                                                                            *
//* Input  : trgPath : (optional, NULL pointer by default)                     *
//*                    if specified, use the path as the target for reading    *
//*                    filesystem information                                  *
//*                                                                            *
//* Returns: 'true' if information read successfully, else 'false'             *
//******************************************************************************
//* Programmer's Note: Dialog size: INFODLG_WIDTH+2 columns is the largest     *
//* width that will fit INSIDE the file-display control when in Dual-Window    *
//* mode AND terminal is at minimum width for Dual-Window mode (118 columns).  *
//*                                                                            *
//* NOTE: The optional 'trgPath' is used primarily by the user interface for   *
//*       Tree-view Mode, or to display stats for specified filesystem. If not *
//*       specified, then the filespec used is the current directory.          *
//*                                                                            *
//* NOTE: freeBytes + usedBytes != total_filesystem_bytes because of space     *
//*       reserved for superuser.                                              *
//*                                                                            *
//******************************************************************************

bool FileDlg::DisplayFilesystemInfo ( const char* trgPath )
{
   #define DEBUG_DFSI (0)                 // set non-zero for debugging only
   const short dialogROWS  = 19,          // dialog lines
               dialogCOLS  = minfitCOLS,  // dialog columns
               pathMsgWidth= dialogCOLS - 4,
               controlsDEFINED = 1 ; // number of controls defined for this dialog

   short ctrY = this->fulY + this->fMaxY/2,
         ctrX = this->fulX + this->fMaxX/2,
         ulY  = ctrY - dialogROWS / 2,
         ulX  = ctrX - dialogCOLS / 2 ;
         if ( ulY < this->fulY ) ulY = this->fulY ;   // (don't obscure path)
   attr_t dColor  = this->cs.sd,     // window background color
          tColor  = this->cs.sb,     // data-display color
          pnColor = this->cs.pn,     // pushbutton non-focus color
          pfColor = this->cs.pf ;    // pushbutton focus color
   short  fssStatus = ERR,           // status 'stat -f' of system call
          usbdStatus = ERR,          // status of GetFilesystemStats_gvfs() call
          dvddStatus = ERR ;         // status of DVD_DeviceStats() call
   bool   isgvfs = false ;           // 'true' if target filesystem is gvfs

   //* Initialize Pushbutton control *
   InitCtrl ic[controlsDEFINED] = 
   {
      {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - -  okPush *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogROWS - 2),        // ulY:       upper left corner in Y
         short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         9,                            // cols:      control columns
         "  CLOSE  ",                  // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      },
   } ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Give the dialog a title *
      dp->SetDialogTitle ( "  File System Information  ", this->cs.em ) ;

      //* Get data on the file system that contains the 'current' directory *
      fileSystemStats fsStats ;
      gString gsTrg( (trgPath != NULL ? trgPath : this->currDir) ) ;
      fssStatus = this->fmPtr->GetFilesystemStats ( gsTrg, fsStats ) ;

      //* If the target is an MTP/GVfs filesystem, OR if target *
      //* is an optical drive, get auxilliary information.      *
      usbDevice usbdStats ;
      usbDevice dvddStats ;
      if ( (isgvfs = this->fmPtr->isGvfsPath ( gsTrg )) )
      {
         usbdStatus = this->fmPtr->GetFilesystemStats_gvfs ( gsTrg.ustr(), usbdStats ) ;

         //* If target is not an MTP/GVFS device, test for optical drive.*
         if ( usbdStatus != OK )
            dvddStatus = this->fmPtr->GetFilesystemStats_opti ( gsTrg.ustr(), dvddStats ) ;
      }

      //* Display the File System data *
      winPos   wp( 1, 2 ) ;      // cursor position
      const char* msgCaption = 
            "The file system which contains the files in this view\n"
            "has the following characteristics:\n"
            "Filesystem Type:\n"
            " File System ID:\n"
            " UUID          :\n"
            " Label         :\n"
            " Mount Point   :\n"
            "\n"
            "Total Blocks:                Total Inodes:\n"
            " Free Blocks:                Avail Inodes:\n"
            "Avail Blocks:                  Block Size:\n"
            "  Used Bytes:                 FBlock Size:\n"
            "  Free Bytes:                Max Filename:\n" ;
      gString gs( msgCaption ) ;
      if ( isgvfs )
      {
         short indx = (gs.findlast ( L'\n' )) + 1 ;
         gs.limitChars( indx ) ;
      }
      dp->WriteParagraph ( wp, gs, dColor ) ;
      if ( trgPath != NULL )
      {  //* Display the target path *
         gString tpath = trgPath ;
         this->TrimPathString ( tpath, pathMsgWidth - 20 ) ;
         gs.compose( L"The file system at: %S",
                     tpath.gstr() ) ;
         dp->ClearLine ( wp.ypos ) ;
         dp->WriteString ( wp, gs, dColor ) ;
      }

      wp.ypos += 2 ;
      wp.xpos += 17 ;

      //* Mountpoint path may be too wide for the display area.*
      short fieldWidth = dialogCOLS - (wp.xpos + 2) ;
      gs = fsStats.mntPoint ;
      if ( (gs.gscols()) > fieldWidth )
      {
         this->TrimPathString ( gs, fieldWidth ) ;
         gs.copy( fsStats.mntPoint, MAX_PATH ) ;
      }
      //* Alert user if error(s) retrieving filesystem information.*
      else if ( fssStatus != OK )
      {
         gs = "Filesystem information not available." ;
         gs.copy( fsStats.mntPoint, fssLEN ) ;
      }

      //* If text fields are empty, insert a dash *
      gs.compose( "%C", &wcsHDASH3s ) ;
      if ( fsStats.uuid[0] == NULLCHAR )     gs.copy( fsStats.uuid, fssLEN ) ;
      if ( fsStats.label[0] == NULLCHAR )    gs.copy( fsStats.label, fssLEN ) ;
      if ( fsStats.mntPoint[0] == NULLCHAR ) gs.copy( fsStats.mntPoint, fssLEN ) ;
      
      // Programmer's Note: Labels on VFAT-formatted USB drives and SD cards 
      // may be formatted in non-UTF8 encoding if not all characters are ASCII, 
      // OR if the label contains spaces (0x20). (bleeping Microsloth) 
      // If that happens, the fsStats.label may print badly. Of course 
      // fsStats.mntPoint may have the same problem. Ugly, but no harm done.
      // Moral of the story: Be careful how you label mass-storage devices.
      gs.compose( L"%0X - %s\n"
                   "%llX\n"
                   "%s\n"
                   "%s\n"
                   "%s\n\n",
                  &fsStats.fsTypeCode, fsStats.fsType, &fsStats.systemID,
                  fsStats.uuid, fsStats.label, fsStats.mntPoint ) ;
      wp = dp->WriteParagraph ( wp, gs, tColor ) ;

      #if DEBUG_DFSI != 0  // FOR DEBUGGING ONLY
      gs.compose ( "Device: %s", fsStats.device ) ;
      dp->WriteString ( 8, 3, gs, nc.grbl ) ;
      gs.compose( "status:%hX mnt:%hhd mntpt:%hhd", 
                  &fssStatus, &fsStats.isMounted, &fsStats.isMountpoint ) ;
      dp->WriteString ( 17, 1, gs, nc.grbl ) ;
      #endif   // DEBUG_DFSI

      wp.xpos -= 4 ;
      gString gsfmt ;
      gs.formatInt( fsStats.blockTotal, 12 ) ;
      gs.append( L'\n' ) ;
      gsfmt.formatInt( fsStats.blockFree, 12 ) ;
      gs.append( "%S\n", gsfmt.gstr() ) ;
      gsfmt.formatInt( fsStats.blockAvail, 12 ) ;
      gs.append( "%S\n    ", gsfmt.gstr() ) ;
      gsfmt.formatInt( fsStats.usedBytes, 8 ) ;
      gs.append( "%S\n    ", gsfmt.gstr() ) ;
      gsfmt.formatInt( fsStats.freeBytes, 8 ) ;
      gs.append( gsfmt.gstr() ) ;
      dp->WriteParagraph ( wp, gs, tColor ) ;

      wp.xpos += 29 ;
      gs.formatInt( fsStats.inodeTotal, 11 ) ;
      gs.append( L'\n' ) ;
      gsfmt.formatInt( fsStats.inodeAvail, 11 ) ;
      gs.append( "%S\n", gsfmt.gstr() ) ;
      gsfmt.formatInt( (ULONG)fsStats.blockSize, 11 ) ;
      gs.append( "%S\n", gsfmt.gstr() ) ;
      gsfmt.formatInt( (ULONG)fsStats.fblockSize, 11 ) ;
      gs.append( "%S\n", gsfmt.gstr() ) ;
      gsfmt.formatInt( (ULONG)fsStats.nameLen, 11 ) ;
      gs.append( gsfmt.gstr() ) ;
      dp->WriteParagraph ( wp, gs, tColor ) ;
      wp.ypos += 5 ;
      wp.xpos  = 2 ;

      //* If target is a gvfs filesystem *
      if ( isgvfs && (usbdStatus == OK) )
      {
         wp = dp->WriteParagraph ( wp, "URI: ", dColor ) ;
         gs = usbdStats.uri ;
         gs.limitCols( dialogCOLS - 9 ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;
         wp = dp->WriteString ( (wp.ypos + 1), 2, "Bus:", dColor ) ;
         gs.compose( "%03hd", &usbdStats.bus ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;
         wp = dp->WriteString ( wp, " Dev:", dColor ) ;
         gs.compose( "%03hd", &usbdStats.dev ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;
         wp = dp->WriteString ( wp, " Device: ", dColor ) ;
         wp = dp->WriteString ( wp, usbdStats.devpath, tColor ) ;

         wp = dp->WriteString ( (wp.ypos + 1), 2, "Vendor:", dColor ) ;
         gs.compose( "%04hXh", &usbdStats.vendor ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;
         wp = dp->WriteString ( wp, " Prod:", dColor ) ;
         gs.compose( "%04hXh", &usbdStats.product ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;

         wp = dp->WriteString ( wp, " R/O:", dColor ) ;
         wp = dp->WriteString ( wp, (usbdStats.readonly ? "yes" : "no"), tColor ) ;
         wp = dp->WriteString ( wp, " Remote:", dColor ) ;
         wp = dp->WriteString ( wp, (usbdStats.remote ? "yes" : "no"), tColor ) ;

         #if DEBUG_DFSI != 0  // FOR DEBUGGING ONLY
         gs.formatInt( usbdStats.totalbytes, 6, true ) ;
         gs.append( "=" ) ;
         wp = dp->WriteString ( 17, 34, gs, nc.grbl ) ;
         gs.formatInt( usbdStats.usedbytes, 7, true ) ;
         gs.append( "+" ) ;
         wp = dp->WriteString ( wp, gs, nc.grbl ) ;
         gs.formatInt( usbdStats.freebytes, 7, true ) ;
         wp = dp->WriteString ( wp, gs, nc.grbl ) ;
         #endif   // DEBUG_DFSI
      }
      //* If target is an optical-drive filesystem *
      else if ( isgvfs && (dvddStatus == OK) )
      {
         wp = dp->WriteParagraph ( wp, "URI: ", dColor ) ;
         gs = dvddStats.uri ;
         gs.limitCols( dialogCOLS - 9 ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;
         wp = dp->WriteString ( (wp.ypos + 1), 2, "Device: ", dColor ) ;
         wp = dp->WriteString ( wp, dvddStats.devpath, tColor ) ;
         wp = dp->WriteString ( (wp.ypos + 1), 2, "R/O:", dColor ) ;
         wp = dp->WriteString ( wp, (dvddStats.readonly ? "yes" : "no"), tColor ) ;
         wp = dp->WriteString ( wp, " Remote:", dColor ) ;
         wp = dp->WriteString ( wp, (dvddStats.remote ? "yes" : "no"), tColor ) ;
         wp = dp->WriteString ( (wp.ypos + 1), 2, "Media:", dColor ) ;
         wp = dp->WriteString ( wp, (dvddStats.has_media ? "yes" : "no"), tColor ) ;
         wp = dp->WriteString ( wp, " Mounted:", dColor ) ;
         wp = dp->WriteString ( wp, (dvddStats.mounted ? "yes" : "no"), tColor ) ;
      }

      //* Display SELinux Security context string. Because string may   *
      //* be longer than width of dialog, limit its length.             *
      //* (Some systems will not report this value and will return '?') *
      else
      {
         wp = dp->WriteParagraph ( wp, "SELinux Security context:\n ", dColor ) ;
         gs = fsStats.seLinux ;
         gs.limitCols( dialogCOLS - 4 ) ;
         wp = dp->WriteString ( wp, gs, tColor ) ;
      }

      dp->RefreshWin () ;                    // make the text visible

      //******************
      //* Get user input *
      //******************
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false || Info.keyIn == nckESC )
            done = true ;
      }
   }
   else           // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return ( bool(fssStatus == OK) ) ;

}  //* End DisplayFilesystemInfo() *

//*************************
//*    GetGVfsVersion     *
//*************************
//******************************************************************************
//* Returns the version string for the GnomeVirtualFileSystem (GVfs) tools.    *
//* This is the version of the 'gio' utility if installed.                     *
//*                                                                            *
//* Input  : gvfsVersion : (by reference) receives the GVfs utilities version  *
//*                        string, or empty string if 'gio' not installed      *
//*                                                                            *
//* Returns: true if GVfs tools are accessible, else false                     *
//******************************************************************************

bool FileDlg::GetGVfsVersion ( gString& gvfsVersion )
{

   return ( (this->fmPtr->gioGetVersion ( gvfsVersion )) ) ;

}  //* End GetGVfsVersion() *

//*************************
//*     FindLinkTarget    *
//*************************
//******************************************************************************
//* Directory window's highlight is on a symbolic-link file, and user has      *
//* requested that we find the link target.                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK  if operation successful                                       *
//*          ERR if target not found or is inaccessible                        *
//******************************************************************************

short FileDlg::FindLinkTarget ( void )
{
   short status = ERR ;

   //* If there is actually a file under the highlight *
   if ( this->fileCount > ZERO )
   {
      //* Retrieve the file stat informaton *
      tnFName  fn ;        // copy of basic file info
      ExpStats es, est ;   // expanded, human-readable version of file info
      es.symbTarg = &est ; // link the data structures in case file is a symbolic link
      this->GetStats ( fn ) ;
      this->fmPtr->strnCpy ( es.filePath, this->currDir, MAX_PATH ) ;
      this->fmPtr->ExpandStats ( &fn, es ) ;

      //* If file is a symbolic link (if not, it's a program error).*
      if ( fn.fType == fmLINK_TYPE )
      {
         //* If the target exits (cannot follow a broken link).*
         if ( es.symbTarg->fileType != fmTYPES )
         {
            gString newPath( est.filePath ) ;
            if ( (status = this->SetDirectory ( newPath )) == OK )
            {  //* Highlight the target file *
               this->Scroll2MatchingFile ( est.fileName, true ) ;
               status = OK ;
            }
            else
            {  //* Unable to attach to target directory *
               this->TrimPathString ( newPath, (INFODLG_WIDTH - 5) ) ;
               newPath.insert( L"  " ) ;
               const char *msgList[] = 
               {
                  "  ALERT  ",
                  "The directory containing the symbolic-link target:",
                  newPath.ustr(),
                  "cannot be read.",
                  NULL
               } ;
               this->InfoDialog ( msgList ) ;
            }
         }
         else
         {  //* Alert user to broken link *
            gString gss( "   Link   : %s", es.fileName ) ;
            gString gst( "   Target : %s", est.fileName ) ;
            const char *msgList[] = 
            {
               "  ALERT  ",
               "The symbolic-link target was not found.",
               gss.ustr(),
               gst.ustr(),
               NULL
            } ;
            this->InfoDialog ( msgList ) ;
         }
      }
   }
   return status ;

}  //* End FindLinkTarget() *

//*************************
//*     isMountpoint      *
//*************************
//******************************************************************************
//* Determine whether the specified filespec refers to a mountpoint directory. *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : trgPath : target filespec                                         *
//*                                                                            *
//* Returns: 'true'  if target is a mountpoint directory                       *
//*          'false' if target is not a mountpoint                             *
//*                  (or is not a directory, or does not exist)                *
//******************************************************************************

bool FileDlg::isMountpoint ( const char* trgPath ) 
{

   return ( (this->fmPtr->isMountpoint ( trgPath )) ) ;

}  //* End isMountpoint() *

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Get the system timecode and convert it to localTime format.                *
//*                                                                            *
//* Input  : ft  : (by reference, initial values ignored)                      *
//*                on return, contains decoded date/time                       *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************
//* Programmer's Note: Currently we do not decode the timezone and DST fields. *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* Note that we do a bit of defensive programming in anticipation of the      *
//* dreaded year 2038 overflow of the 32-bit time_t type.                      *
//******************************************************************************

bool FileDlg::GetLocalTime ( localTime& ft )
{
   bool success = false ;                       // return value
  
   ft.reset() ;                                 // set to default values
   if ( (time ( &ft.sysepoch )) != (-1) )
   {
      if ( ft.sysepoch >= ZERO )                // reported time is AFTER the epoch
      {
         ft.epoch = (int64_t)ft.sysepoch ;      // promote to 64-bit
         //* Decode the system time *
         Tm    tm ;                             // Linux time structure
         if ( (localtime_r ( &ft.sysepoch, &tm )) != NULL )
         {
            //* Translate to localTime format *
            ft.julian   = tm.tm_yday ;          // Julian date
            ft.day      = tm.tm_wday ;          // 0 == Sunday ... 6 == Saturday
            ft.date     = tm.tm_mday ;          // today's date
            ft.month    = tm.tm_mon + 1 ;       // month
            ft.year     = tm.tm_year + 1900 ;   // year
            ft.hours    = tm.tm_hour ;          // hour
            ft.minutes  = tm.tm_min ;           // minutes
            ft.seconds  = tm.tm_sec ;           // seconds
            success = true ;
         }
      }
      else
      {  /* SYSTEM ERROR - time_t OVERFLOW */ }
   }
   return success ;

}  //* End GetLocalTime() *

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Return the current system-local epoch time.                                *
//*                                                                            *
//* Input  : epoch_time : (by reference, initial value ignored)                *
//*                       on return, contains epoch timecode                   *
//*                       (or ZERO if system call fails)                       *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************

bool FileDlg::GetLocalTime ( int64_t& epoch_time )
{
   time_t calTime ;                             // simple calendar time
   bool   success = false ;                     // return value

   epoch_time = ZERO ;                          // initialize caller's value

   if ( (calTime = time ( NULL )) != (-1) )
   {
      epoch_time = (int64_t)calTime ;
      success = true ;
   }

   return success ;

}  //* End GetLocalTime() *

//************************
//*   DecodeLocalTime    *
//************************
//******************************************************************************
//* Convert specified epoch time to human-readable time.                       *
//*                                                                            *
//* Input  : lt  : (by reference) 'epoch' member (seconds since the epoch)     *
//*                contains time to be decoded into human-readable form        *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::DecodeLocalTime ( localTime& lt )
{

   this->fmPtr->DecodeEpochTime ( lt.epoch, lt ) ;

}  //* End DecodeLocalTime() *

//************************
//*   EncodeLocalTime    *
//************************
//******************************************************************************
//* Convert specified human-readable time to epoch time.                       *
//*                                                                            *
//* Input  : lt  : (by reference) human-readable members are converted to      *
//*                'epoch' and 'sysepoch' integers (seconds since the epoch)   *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::EncodeLocalTime ( localTime& lt )
{

   lt.epoch = this->fmPtr->EncodeEpochTime ( lt ) ;

}  //* End EncodeLocalTime() *

//*************************
//*    GetNewTimestamp    *
//*************************
//******************************************************************************
//* Display modification date for specified file, and ask user if he/she       *
//* wants to modify the date.                                                  *
//*                                                                            *
//* Notes: a) By default the 'access' date will track the 'modification' date. *
//*           (unless caller sets them separately)                             *
//*        b) Stat date is always set to current date/time                     *
//*           (no program control over this)                                   *
//*                                                                            *
//* Input  : ft  : receives the new date/time                                  *
//*                                                                            *
//* Returns:  true  == new date/time specified                                 *
//*                    (ft contains new date/time data)                        *
//*           false == abort operation (ft unitialized)                        *
//******************************************************************************

bool FileDlg::GetNewTimestamp ( localTime& ft )
{
   const short dialogROWS = 10,
               dialogCOLS = 54,  // (fits in smallest dual-win window)
               ulY = this->fulY + 4,
               ulX = (this->fulX + this->fMaxX / 2) - (dialogCOLS / 2) ;
   const char* ButtonText[] = { " ACCEPT ", " CANCEL " } ;
   const char* timeFormat = "dd mmm yyyy hh:mm:ss" ;
   enum gntControls : short { accPB = ZERO, canPB, dateTB, gntCONTROLS } ;

   gString  newDate,       // working timestamp string
            origDate ;     // back-up copy
   attr_t   dColor = this->cs.sd,
            tbfColor = this->cs.scheme == ncbcCOLORS ? 
                       this->cs.tf | ncrATTR : this->cs.tf ;
   bool     newTime = false ;       // return value

   InitCtrl ic[3] =
   {
   {  //* 'ACCEPT' pushbutton - - - - - - - - - - - - - - - - - - - - -  accPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 10),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      ButtonText[0],                // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[accPB].ulY,                // ulY:       upper left corner in Y
      short(ic[accPB].ulX + ic[accPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      ButtonText[1],                // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dateTB]                   // nextCtrl:  link in next structure
   },
   {  //* 'DATE' textbox  - - - - - - - - - - - - - - - - - - - - - - - dateTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 10),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      NULL,                         // dispText:  formatted date/time string
      this->cs.tn,                  // nColor:    non-focus color
      tbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    printing characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Get the current system time *
   if ( (this->GetLocalTime ( ft )) != false )
   {
      //* Convert to display format *
      this->FormatDateString ( ft, newDate ) ;
      origDate = newDate ;
   }
   else
   {
      newDate  = "Clock Not Responding" ;
      origDate = timeFormat ;
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( "  Set File Date/Time  ", this->cs.em ) ;
      dp->WriteString ( ic[dateTB].ulY+1,  ic[dateTB].ulX, timeFormat, dColor ) ;
      dp->WriteParagraph ( ic[accPB].ulY - 2, 2, 
                           "Enter date and time according to indicated format.\n"
                           "      (default is current system local time)", dColor ) ;

      //* Configure the Textbox *
      dp->SetTextboxText ( dateTB, newDate ) ;     // initialize text
      dp->FixedWidthTextbox ( dateTB, false ) ;    // fixed-width data
      dp->SetTextboxInputMode ( true, true ) ;     // force overstrike
      dp->TextboxAlert ( dateTB, true ) ;          // complain about bad input

      dp->RefreshWin () ;                 // make the text visible

      short    icIndex = ZERO ;           // control with input focus
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == accPB )
               {
                  done = newTime = true ;
               }
               else  // (Info.ctrlIndex == canPB)
                  done = true ;
            }
            else if ( Info.keyIn == nckESC ) // user is having a panic attack
               done = true ;
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            dp->ClearLine ( (ic[dateTB].ulY + 2) ) ;  // clear the message line

            if ( Info.dataMod != false )
            {  //* Verify that user has properly formatted the date/time *
               dp->GetTextboxText ( dateTB, newDate ) ;
               if ( (this->ValidateDateString ( ft, newDate )) != false )
               {
                  origDate = newDate ;    // save a copy
               }
               else
               {
                  icIndex = dp->PrevControl () ;            // move the focus
                  dp->SetTextboxText ( dateTB, origDate ) ; // restore to valid timestamp
                  dp->WriteString ( 4, dialogCOLS / 2 - 20, // complain
                                    " Invalid Timestamp Format! - Try again. ", 
                                    this->cs.pf, true ) ;
                  Info.keyIn = nckTAB ;         // move forward to Textbox
               }
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ;
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }

   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

    //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return newTime ;

}  //* End GetNewTimestamp() *

//***********************
//* ValidateDateString  *
//***********************
//******************************************************************************
//* Scan the date/time string and initialize a localTime class object.         *
//* String must be of the format: "dd Mmm yyyy hh:mm:ss"                       *
//* Date/time string must be correctly formatted and the specified value       *
//* must be a valid date/time.                                                 *
//*                                                                            *
//* On 32-bit systems sizeof(time_t) == 4):                                    *
//*          01 Jan 1970 00:00:00  to  31 Dec 2037 23:59:59.                   *
//*          (signed 32-bit int can hold dates to 18 Jan 2038 19:14:07)        *
//* On 64-bit systems sizeof(time_t) == 8):                                    *
//*          01 Jan 1970 00:00:00 to the year 2,147,485,547                    *
//*          (We test the upper limit against to 31 Dec 9999 23:59:59 )        *
//*          (since our input field only goes to the year 9999.       )        *
//* Note: 'day' (day-of-the-week) field and 'julian' date field are not        *
//*       initialized by this method.                                          *
//*                                                                            *
//* Input  : ft  : localTime class object gets validated data (by reference)   *
//*          str : buffer containing caller's date/time string of format:      *
//*                "dd Mmm yyyy hh:mm:ss"                                      *
//*                                                                            *
//* Returns: true if successful, false if bad format or date out of range      *
//******************************************************************************
//* Wikipedia: the Gregorian calendar                                          *
//* In the Gregorian calendar, the current standard calendar in most of the    *
//* world, most years that are evenly divisible by 4 are leap years. . . .     *
//* Some exceptions to this rule are required . . .                            *
//* Years that are evenly divisible by 100 are not leap years, unless they are *
//* also evenly divisible by 400, in which case they are leap years.           *
//*                                                                            *
//* pseudocode:                                                                *
//* if year modulo 4 is 0                                                      *
//*    then                                                                    *
//*        if year modulo 100 is 0                                             *
//*            then                                                            *
//*                if year modulo 400 is 0                                     *
//*                    then                                                    *
//*                        is_leap_year                                        *
//*                else                                                        *
//*                    not_leap_year                                           *
//*        else is_leap_year                                                   *
//* else not_leap_year                                                         *
//*                                                                            *
//* Programmer's Note: At the moment, the input is a well-controlled input     *
//* string which uses only numeric values except for the 'month' string.       *
//* Currently the month string uses only English, ASCII characters for the     *
//* month abbreviations (see FMgr.hpp MonthString[]).                          *
//* Some locales may use a different language or a different character set to  *
//* represent this information, so we have built in some flexibility by        *
//* testing the month string against an array which can be modified to match   *
//* the target language/locale.                                                *
//******************************************************************************

bool FileDlg::ValidateDateString ( localTime& ft, const gString& str )
{
bool     goodDate = false ;

   if ( str.gschars() == DATE_STRING_LEN )
   {
      const wchar_t* wptr = str.gstr() ;

      //* Check for correct position of whitespace *
      if ( wptr[2] == SPACE  && wptr[6] == SPACE && wptr[11] == SPACE )
      {
         wchar_t mBuff[24] ;
         int scannedItems ;

         //* Scan the string into the 'Tm' structure *
         scannedItems = swscanf ( wptr, L"%hu %S %hu  %hu:%hu:%hu", 
                                  &ft.date, mBuff, &ft.year,
                                  &ft.hours, &ft.minutes, &ft.seconds ) ;
         ft.day = ZERO ;                     // day-of-the-week not used
         ft.julian = ZERO ;                  // Julian date not used

         if ( scannedItems == 6 )
         {
            mBuff[0] = toupper ( mBuff[0] ) ;   // encode the month
            mBuff[1] = tolower ( mBuff[1] ) ;
            mBuff[2] = tolower ( mBuff[2] ) ;
            mBuff[3] = NULLCHAR ;
            for ( short i = 1 ; i <= 12 ; i++ )
            {
               if ( !(wcsncasecmp ( mBuff, wMonthString[i], DATE_STRING_LEN )) )
               {
                  ft.month = i ;
                  goodDate = true ;          // good month
                  break ;
               }
            }
            if ( goodDate )                  // if month string matched
            {
               if (   ft.year < 1970 
                   || ft.year > ((sizeof(time_t)) == 4 ? 2037 : 9999)
                   || ft.date < 1       || ft.date > 31 
                   || ft.hours > 23 
                   || ft.minutes > 59
                   || ft.seconds > 59 )
                  goodDate = false ;
               else if ( ft.month == 2 && ft.date > 28 )
               {
                  //* Test for valid leap-day *
                  if ( ft.date == 29 )
                  {
                     if ( ((ft.year % 100) == ZERO) && !((ft.year % 400) == ZERO) )
                        goodDate = false ;
                  }
                  else
                     goodDate = false ;
               }
               else if ((   ft.month == 4 || ft.month == 6 || ft.month == 9 
                         || ft.month == 11) && ft.date > 30 )
                  goodDate = false ;
            }  // goodDate
         }     // scannedItems
      }        // whitespace check
   }           // length check
   return goodDate ;

}  //* End ValidateDateString() *

//*************************
//* AdjustPermissionBits  *
//*************************
//******************************************************************************
//* Allow user to make modifications to the default values for file permission *
//* bits before a write-enable or write-protect operation on selected files.   *
//*                                                                            *
//* Input  : newMask : (by reference, initial values ignored)                  *
//*                    receives the user's custom bitmask (expanded)           *
//*                    - if default write-protect or default write-enable, then*
//*                      newMask is unitialized                                *
//*                    - if custom mask specified, then all usrProt, grpProt   *
//*                      and othProt members are initialized to user's         *
//*                      specification                                         *
//*          xwd     : (by reference, initial value ignored)                   *
//*                    on return, indicates whether, for directory names only, *
//*                    to also set 'execute' bit IF 'read' bit is set          *
//*          protect : specifies the operation in progress:                    *
//*                    'true'  == write protect                                *
//*                    'false' == write enable                                 *
//*                                                                            *
//* Returns: 0    == if default write-protect operation                        *
//*          1    == if default write-enable operation                         *
//*          2    == if custom permission mask specified                       *
//*          ERR     if user has aborted the operation                         *
//******************************************************************************

short FileDlg::AdjustPermissionBits ( ExpStats& newMask, bool& xwd, bool protect )
{
   short opStatus = protect ? ZERO : 1 ;     // return value
   xwd = true ;   // initialize caller's data (and initial state of xwdRB)

   const char* const CustHdr = 
   "         OWNER  GROUP  OTHERS\n"
   "   Read:\n"
   "  Write:\n"
   "Execute:" ;
   const short dialogROWS = 15,                    // dialog dimensions
               dialogCOLS = INFODLG_WIDTH,
               ctrY = this->fulY + this->fMaxY/2,
               ctrX = this->fulX + this->fMaxX/2,
               ulY = (this->fMaxY > dialogROWS) ?  // dialog position
                     ctrY - dialogROWS / 2 : this->fulY,
               ulX = ctrX - dialogCOLS / 2 ;
   attr_t      bColor  = this->cs.sd,              // dialog border color
               dColor  = this->cs.sb,              // dialog interior color
               vColor  = bColor,                   // msg color
               iColor  = this->cs.scheme == ncbcCOLORS ? // 'inactive' color
                         nc.gybl : bColor ;
   bool        custEnable = false ;    // 'true' if custom controls are enabled

enum dlgControls : short { okPB, canPB, 
                           wpRB, weRB, custRB, 
                           urRB, uwRB, uxRB, 
                           grRB, gwRB, gxRB, 
                           orRB, owRB, oxRB, 
                           xwdRB, controlsDEFINED
                         } ;
InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 11),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   OK   ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[okPB].ulY,                 // ulY:       upper left corner in Y
      short(dialogCOLS / 2 + 3),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[wpRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'Write-protect' radio button  - - - - - - - - - - - - - - - -   wpRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      protect,                      // rbSelect:  initial value
      1,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Write-protect selected files",// label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[weRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'Write-enable' radio button   - - - - - - - - - - - - - - - -   weRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      !protect,                     // rbSelect:  initial value
      short(ic[wpRB].ulY + 2),      // ulY:       upper left corner in Y
      ic[wpRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Write-enable selected files",// label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[custRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Custom-mask' radio button    - - - - - - - - - - - - - - - - custRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[weRB].ulY + 2),      // ulY:       upper left corner in Y
      ic[weRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Create custom permissions for selected files",// label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[urRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'User-read'   radio button    - - - - - - - - - - - - - - - -   urRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[okPB].ulY - 4),      // ulY:       upper left corner in Y
      short(ic[okPB].ulX + 4),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[r]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[uwRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'User-write'  radio button    - - - - - - - - - - - - - - - -   uwRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[urRB].ulY +1),       // ulY:       upper left corner in Y
      ic[urRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[w]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[uxRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'User-execute' radio button   - - - - - - - - - - - - - - - -   uxRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[uwRB].ulY +1),       // ulY:       upper left corner in Y
      ic[uwRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[x]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[grRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'Group-read'  radio button    - - - - - - - - - - - - - - - -   grRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      ic[urRB].ulY,                 // ulY:       upper left corner in Y
      short(ic[urRB].ulX + 7),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[r]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[gwRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'group-write' radio button    - - - - - - - - - - - - - - - -   gwRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[grRB].ulY +1),       // ulY:       upper left corner in Y
      ic[grRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[w]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[gxRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'group-execute' radio button  - - - - - - - - - - - - - - - -   gxRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[gwRB].ulY +1),       // ulY:       upper left corner in Y
      ic[gwRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[x]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[orRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'Other-read'  radio button    - - - - - - - - - - - - - - - -   orRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      ic[grRB].ulY,                 // ulY:       upper left corner in Y
      short(ic[grRB].ulX + 7),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[r]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[owRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'Other-write' radio button    - - - - - - - - - - - - - - - -   owRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[orRB].ulY +1),       // ulY:       upper left corner in Y
      ic[orRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[w]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[oxRB],                    // nextCtrl:  link in next structure
   },
   {  //* 'Other-execute' radio button  - - - - - - - - - - - - - - - -   oxRB *
      dctRADIOBUTTON,               // type:      
      rbtC3,                        // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initial value
      short(ic[owRB].ulY +1),       // ulY:       upper left corner in Y
      ic[owRB].ulX,                 // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      "[x]",                        // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[xwdRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'execute-with-read radio button - - - - - - - - - - - - - - -  xwdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 5-wide
      xwd,                          // rbSelect:  initial value
      short(ic[uxRB].ulY + 1),      // ulY:       upper left corner in Y
      short(dialogCOLS - 5),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  custom definition
      bColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( "  Set File Permissions  ", this->cs.em ) ;
      winPos wp( ic[wpRB].ulY + ic[wpRB].labY + 1, ic[wpRB].ulX + ic[wpRB].labX + 1 ) ;
      wp = dp->WriteParagraph ( wp, 
                           "Resets 'write' bit for Owner/Group/Other.\n"
                           "\n"
                           "Sets 'write' bit for Owner and Group.\n"
                           "\n"
                           "Sets specified bits. Resets all other bits.\n"
                           "\n",
                           vColor ) ;
      dp->WriteParagraph ( wp, CustHdr, iColor ) ;
      dp->WriteString ( ic[uxRB].ulY + 1, ic[uxRB].ulX,
                        "(for dir names, set x with r) [?]", iColor ) ;

      //* Set radio buttons as an exclusive-OR group *
      //* (only one can be selected at any moment).  *
      short XorGroup[] = { wpRB, weRB, custRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;


      dp->RefreshWin () ;

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton   *
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {
                  switch ( dp->GetRbGroupSelection ( wpRB ) )
                  {
                     case wpRB:   opStatus = ZERO ; break ;
                     case weRB:   opStatus = 1 ;    break ;
                     case custRB:
                        {
                        bool state ;
                        dp->GetRadiobuttonState ( urRB, state ) ;
                        newMask.usrProt.read = state ? 'r' : '-' ;
                        dp->GetRadiobuttonState ( uwRB, state ) ;
                        newMask.usrProt.write= state ? 'w' : '-' ;
                        dp->GetRadiobuttonState ( uxRB, state ) ;
                        newMask.usrProt.exec = state ? 'x' : '-' ;

                        dp->GetRadiobuttonState ( grRB, state ) ;
                        newMask.grpProt.read = state ? 'r' : '-' ;
                        dp->GetRadiobuttonState ( gwRB, state ) ;
                        newMask.grpProt.write= state ? 'w' : '-' ;
                        dp->GetRadiobuttonState ( gxRB, state ) ;
                        newMask.grpProt.exec = state ? 'x' : '-' ;

                        dp->GetRadiobuttonState ( orRB, state ) ;
                        newMask.othProt.read = state ? 'r' : '-' ;
                        dp->GetRadiobuttonState ( owRB, state ) ;
                        newMask.othProt.write= state ? 'w' : '-' ;
                        dp->GetRadiobuttonState ( oxRB, state ) ;
                        newMask.othProt.exec = state ? 'x' : '-' ;

                        dp->GetRadiobuttonState ( xwdRB, xwd ) ;
                        }
                        opStatus = 2 ;
                        break ;
                  }
               }
               else        // user aborted the operation
                  opStatus = ERR ;
               done = true ;
            }
         }
         //* If focus is currently on a Radio Button *
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;

            //* Enable or disable user access to the 'custom' settings *
            if ( (Info.dataMod != false) && 
                 (Info.selMember == wpRB || Info.selMember == weRB || 
                  Info.selMember == custRB) )
            {
               if ( Info.selMember == custRB && ! custEnable )
               {
                  dp->ControlActive ( urRB, true ) ;
                  dp->ControlActive ( uwRB, true ) ;
                  dp->ControlActive ( uxRB, true ) ;
                  dp->ControlActive ( grRB, true ) ;
                  dp->ControlActive ( gwRB, true ) ;
                  dp->ControlActive ( gxRB, true ) ;
                  dp->ControlActive ( orRB, true ) ;
                  dp->ControlActive ( owRB, true ) ;
                  dp->ControlActive ( oxRB, true ) ;
                  dp->ControlActive ( xwdRB,true ) ;
                  dp->WriteParagraph ( wp, CustHdr, dColor, true ) ;
                  custEnable = true ;
               }
               else if ( Info.selMember != custRB && custEnable )
               {
                  dp->ControlActive ( urRB, false ) ;
                  dp->ControlActive ( uwRB, false ) ;
                  dp->ControlActive ( uxRB, false ) ;
                  dp->ControlActive ( grRB, false ) ;
                  dp->ControlActive ( gwRB, false ) ;
                  dp->ControlActive ( gxRB, false ) ;
                  dp->ControlActive ( orRB, false ) ;
                  dp->ControlActive ( owRB, false ) ;
                  dp->ControlActive ( oxRB, false ) ;
                  dp->ControlActive ( xwdRB,false ) ;
                  dp->WriteParagraph ( wp, CustHdr, iColor, true ) ;
                  custEnable = false ;
               }
            }
         }
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;
   
   return opStatus ;

}  //* End AdjustPermissionBits() *

//***********************
//* TouchSelectedFiles  *
//***********************
//******************************************************************************
//* Update the date/time stamp on selected file(s). By default the current     *
//* system timestamp is used, but user has the option of specifying any        *
//* valid timestamp. See ValidateDateString() for validation limits.           *
//*                                                                            *
//* Updates both 'modification' date and 'access' date to specified date/time  *
//* stamp.  Updates inode change time, 'ctime', to current system date/time.   *
//*                                                                            *
//* Input  : selected: receives number of updates attempted                    *
//*                                                                            *
//* Returns: number of files updated                                           *
//*           if return value == 'selected' parm, all selected files were      *
//*             updated successfully                                           *
//*           else at least one selected file was not updated                  *
//******************************************************************************

UINT FileDlg::TouchSelectedFiles ( UINT& selected )
{
   UINT  touchCount = ZERO ;        // return value - number of files updated

   //* User must have write permission on source directory in order to         *
   //* modify the source files. This is verified during selection.             *
   //* Visually mark 'selected' files for processing.                          *
   //* Re-display file list, update stats control, and move highlight to       *
   //* first 'selected' file in list. Returns ZERO if no files selected.       *
   bool srcdirWPerm ;
   selected = this->MarkSelectionNR ( this->touchColor, srcdirWPerm ) ;

   //* Note test of 'clipBoard_gvfsSrc'. Timestamp modification *
   //* is not supported for MTP/GVfs filesystems.               *
   if ( srcdirWPerm != false && selected > ZERO && clipBoard_gvfsSrc == false )
   {
      //* Test whether user has write access on *
      //* all the source files to be modified.  *
      gString gsPath ;
      UINT  wpCount = ZERO, markedFiles ;
      bool  overrideSrcWProt = false ;
      for ( markedFiles = selected ; markedFiles ; markedFiles-- )
      {
         this->fmPtr->CatPathFname ( gsPath, this->currDir, 
                                     this->deList[this->hIndex]->fName ) ;
         if ( (this->fmPtr->WritePermission ( *this->deList[this->hIndex], 
                                              gsPath )) == false )
            ++wpCount ;
         this->NextSelectedItem () ;   // track to next 'selected' item
      }
      if ( wpCount > ZERO )            // if there are write-protected files
      {
         overrideSrcWProt = this->spDecision ( wpCount, 0, opTOUCH ) ;
         if ( overrideSrcWProt != false )
         {  //* User has elected to override write protection *
            this->FirstSelectedItem () ;     // track to first 'selected' item
            wpCount = ZERO ;
            for ( markedFiles = selected ; markedFiles ; markedFiles-- )
            {
               if ( ! this->deList[this->hIndex]->writeAcc )
               {
                  this->fmPtr->CatPathFname ( gsPath, this->currDir, 
                                              this->deList[this->hIndex]->fName ) ;
                  if ( (this->fmPtr->WritePermission ( *this->deList[this->hIndex], 
                                                       gsPath, true )) == false )
                     ++wpCount ;
               }
               this->NextSelectedItem () ;   // track to next 'selected' item
            }
         }
      }

      this->FirstSelectedItem () ;     // track to first 'selected' item
      if ( wpCount == ZERO || overrideSrcWProt != false )
      {  //* We now have write access to all source files OR *
         //* permission to do as much damage as possible.    *
         //* Get new date/time from user.                    *
         localTime ft ;
         if ( (this->GetNewTimestamp ( ft )) != false )
         {  //* Touch each file which is marked as 'selected' *
            for ( markedFiles = selected ; markedFiles ; markedFiles-- )
            {
               //* We allow update of regular files, directories, *
               //* symbolic links and FIFOs without complaint,    *
               //* but we prompt user before update of character  *
               //* device, block device, socket or unknown type.  *
               bool skipFile = false ;
               if ( ! ((this->deList[this->hIndex])->fType == fmREG_TYPE 
                       || (this->deList[this->hIndex])->fType == fmLINK_TYPE 
                       || (this->deList[this->hIndex])->fType == fmDIR_TYPE 
                       || (this->deList[this->hIndex])->fType == fmFIFO_TYPE) )
               {
                  //* Warn user that he/she/it is about to screw with *
                  //* system files. (They will thank you later.)      *
                  if ( this->deList[this->hIndex]->fType == fmSOCK_TYPE )
                  {
                     if ( (this->DecisionDialog ( warnSocketMod )) == false )
                        skipFile = true ;
                  }
                  else if ( this->deList[this->hIndex]->fType == fmUNKNOWN_TYPE )
                  {
                     if ( (this->DecisionDialog ( warnUnsuppMod )) == false )
                        skipFile = true ;
                  }
                  else  // character or block device
                  {
                     if ( (this->DecisionDialog ( warnBdevCdevMod )) == false )
                        skipFile = true ;
                  }
               }
               if ( ! skipFile )
               {
                  this->fmPtr->CatPathFname ( gsPath, this->currDir, 
                                              this->deList[this->hIndex]->fName ) ;
                  if ( (this->SetFileTimestamp ( gsPath.ustr(), ft )) == OK )
                     ++touchCount ; // count of successfully-updated files
               }
               this->NextSelectedItem () ;   // track to next 'selected' item
            }
         }
         else  // operation aborted
         {
            //* Clear selections and update display *
            this->DeselectFile ( true ) ;
            touchCount = selected = ZERO ; // return successful update of ZERO files
         }
      }
      else
      {  //* User has aborted the operation because one or more files are      *
         //* write-protected.                                                  *
         this->SetErrorCode ( ecSRCPROTECT, EACCES ) ;
      }
      //* Re-read directory contents (clipboard will be cleared) *
      this->RefreshCurrDir () ;
   }
   else                             // no files processed
   {
      this->DeselectFile ( true ) ; // deselect all files
      if ( ! srcdirWPerm )          // no write permission on this directory
      {
         this->DisplayStatus ( ecREADONLY_S ) ;
      }
      else if ( clipBoard_gvfsSrc )
      {
         this->DisplayStatus ( ecNOTIMPL ) ;
      }
   }
   return touchCount ;

}  //* End TouchSelectedFiles() *

//*************************
//* ProtectSelectedFiles  *
//*************************
//******************************************************************************
//* Write-enable or write-protect selected file(s).                            *
//*   OR                                                                       *
//* Allow user to set custom permissions for selected file(s).                 *
//*                                                                            *
//* If a displayed directory name is selected, then all files in all           *
//* subdirectories it contains will ALSO be write-enabled/protected/modified.  *
//* NOTES:                                                                     *
//* 1) Write permission on the containing subdirectory IS NOT required in      *
//*    order to adjust permissions on the files it contains, BUT read          *
//*    permission on the containing subdirectory IS obvioulsy required;        *
//*    otherwise, we could not read the subdirectory contents.                 *
//* 2) If a read-protected subdirectory is in the list of selected files, then *
//*    its contents WILL NOT be on the clipboard (because we can't read them), *
//*    and thus, will not be processed during this operation.                  *
//* 3) Recursive operations on MTP/GVfs filesystems are not allowed, so warn   *
//*    user if gvfs source data.                                               *
//*                                                                            *
//* Input  : selFiles : (by reference, initial value ignored)                  *
//*                     receives number of files selected for update           *
//*          modFiles : (by reference, initial value ignored)                  *
//*                     receives number of files successfully modified         *
//*          protect  : if 'true', write-protect selected files                *
//*                     if 'false', write-enable selected files                *
//*                                                                            *
//* Returns: operation performed:                                              *
//*                     0 == write protect selected files                      *
//*                     1 == write enable selected files                       *
//*                     2 == custom permissions applied to selected files      *
//*                   ERR == if user aborts the operation or if operation      *
//*                          is not permitted                                  *
//*                if selFiles == modFiles, then all selected files were       *
//*                   updated successfully                                     *
//*                else at least one selected file was not updated             *
//******************************************************************************

short FileDlg::ProtectSelectedFiles ( UINT& selFiles, UINT& modFiles, bool protect )
{
   short opStatus = protect ? 0 : 1 ;  // return value
   ExpStats newMask ;                  // permission bitmask
   bool xwd ;                          // 
   modFiles = ZERO ;                   // initialize caller's data

   //* Visually mark 'selected' files for processing.                     *
   //* (Note that write permission on the directory IS NOT required, and) *
   //* (that read permission is assumed, else selFiles woul be == ZERO. ) *
   //* Re-display file list, update stats control, and move highlight to  *
   //* first 'selected' file in list. Returns ZERO if no files selected.  *
   bool srcdirWPerm ;
   selFiles = this->MarkSelection ( this->touchColor, srcdirWPerm ) ;
   if ( selFiles > ZERO )
   {
      //* Place selected files on clipboard *
      this->FillClipboard ( opPROTECT ) ;

      //* Note that modification of permission bits *
      //* is not supported for MTP/GVfs filesystems.*
      if ( clipBoard_gvfsSrc )
      {
         this->DisplayStatus ( ecNOTIMPL ) ;
         this->ClearClipboard () ;
         selFiles = ZERO ;
         opStatus = ERR ;
      }

      //* Open a dialog allowing user to adjust which *
      //* permission bits are to be set or reset.     *
      if ( opStatus != ERR )
         opStatus = this->AdjustPermissionBits ( newMask, xwd, protect ) ;

      //* Write-protect the selected files *
      if ( opStatus == ZERO )
      {
         modFiles = this->SourceWriteProtect ( true ) ;
      }
      //* Write-enable the selected files *
      else if ( opStatus == 1 )
      {
         modFiles = this->SourceWriteEnable ( true ) ;
      }
      //* Apply a custom bit mask to selected files *
      else if ( opStatus == 2 )
      {
         modFiles = this->SourceSetPermission ( newMask, xwd ) ;
      }
      else
      { /* operation aborted */  }

      //* Re-read directory contents (clipboard will be cleared) *
      this->RefreshCurrDir () ;
   }
   else        // no write permission on this directory
   {
      this->DeselectFile ( true ) ;       // deselect all files
      this->DisplayStatus ( ecREADONLY_S ) ;
   }
   return opStatus ;

}  //* End ProtectSelectedFiles() *

//****************************************************************************
//* Overloaded equality operator for comparing localTime class data.         *
//****************************************************************************
static bool operator==( localTime& ftA, localTime& ftB )
{
short rawStatus = CompareDates ( ftA, ftB ) ;
bool  status = false ;
   if ( rawStatus == ZERO )
      status = true ;
   return status ;

}  // End operator==() *

//************************
//*    CompareDates      *
//************************
//******************************************************************************
//* Compare the contents of two localTime class objects.                       *
//* 'day' field is ignored.                                                    *
//*                                                                            *
//* Input  : (by reference) ftA, ftB localTime structures to be compared       *
//*                                                                            *
//* ReturnS: -1 == ftA < ftB                                                   *
//*           0 == ftA == ftB                                                  *
//*           1 == fta > ftB                                                   *
//******************************************************************************

static short CompareDates ( localTime& ftA, localTime& ftB )
{
short    status = 1 ;

   if ( ftA.year >= ftB.year )
   {
      if ( ftA.year == ftB.year )
      {
         if ( ftA.month >= ftB.month )
         {
            if ( ftA.month == ftB.month )
            {
               if ( ftA.date >= ftB.date )
               {
                  if ( ftA.date == ftB.date )
                  {
                     if ( ftA.hours >= ftB.hours )
                     {
                        if ( ftA.hours == ftB.hours )
                        {
                           if ( ftA.minutes >= ftB.minutes )
                           {
                              if ( ftA.minutes == ftB.minutes )
                              {
                                 if ( ftA.seconds >= ftB.seconds )
                                 {
                                    if ( ftA.seconds == ftB.seconds )
                                       status = ZERO ;
                                 }
                                 else status = -1 ;
                              }
                           }
                           else status = -1 ;
                        }
                     }
                     else status = -1 ;
                  }
               }
               else status = -1 ;
            }
         }
         else status = -1 ;
      }
   }
   else status = -1 ;

   return status ;

}  //* End CompareDates()

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

