//********************************************************************************
//* File       : FileDlgRename..cpp                                              *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 11-Apr-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module contains methods related to the file rename operation of the     *
//* FileDlg class.                                                               *
//*                                                                              *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above  *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;

static const char*   SampleFName = "fname.fext" ;  // generic filename.extension


#if ENABLE_DEBUGGING_CODE != 0
#define enableDBWIN (0)          // set to non-zero to enable debugging window
#if enableDBWIN != 0
// NOTE: Testing must be done while running in Single-Window mode with terminal 
//       size >= 132 columns to avoid unsightly dialog overlap.
static NcDialog*  dbWin = NULL ;          // pointer to debugging window
static short      dbLines = ZERO,         // dimensions of debugging window
                  dbCols  = ZERO ;
static attr_t     dbColor = ZERO ;        // color attribute of debugging window
static winPos     dbwp( 1, 1 ) ;          // cursor position in debugging window
static short      dbPOffset = 60 ;        // path-string offset (for beauty)
#endif   // enableDBWIN
#endif   // ENABLE_DEBUGGING_CODE


//***********************
//* RenameSelectedFiles *
//***********************
//******************************************************************************
//* Interactively rename all 'selected' files in the current window.           *
//*                                                                            *
//* This is the public method for accessing the file-rename functionality.     *
//* All associated methods are private.                                        *
//*                                                                            *
//* IMPORTANT NOTE: Batch rename may be used on multiple 'selected' files, and *
//*                 directory names at the top level; however, recursive       *
//*                 rename of all files in a directory tree is not supported   *
//*                 at this time.                                              *
//*                                                                            *
//* Input  : selected: (by reference, initial value ignored)                   *
//*            On return, contains number of files in selection list           *
//*                                                                            *
//* Returns: number of files renamed                                           *
//******************************************************************************
//* Programmer's Note: The rename() library function is successful even        *
//* when the file is read-only or when user does not own the file. This is     *
//* a bug in rename() for which we compensate here.     MRS 17 Feb 2007        *
//*                                                                            *
//* Programmer's Note: On filesystems which do no distinguish between          *
//* uppercase and lowercase, the rename will fail if the only change is a      *
//* change in case. Windoze screws us again....                                *
//*                                                                            *
//******************************************************************************

UINT FileDlg::RenameSelectedFiles ( UINT& selected )
{
   UINT  renamedFiles = ZERO ;

   //* User must have write permission on source directory in order to         *
   //* modify the source files. This is verified during selection.             *
   //* Visually mark 'selected' files for processing.                          *
   //* Re-display file list, update stats control, and move highlight to       *
   //* first 'selected' file in list. Returns ZERO if no files selected.       *
   bool srcdirWPerm ;
   selected = this->MarkSelectionNR ( this->renameColor, srcdirWPerm ) ;

   if ( srcdirWPerm != false && selected > ZERO )
   {
      //* Test whether user has write access on all source files to be modified*
      gString gsPath ;
      UINT  wpCount = ZERO, markedFiles ;
      bool  overrideSrcWProt = false, ignoreRenameErrors = false ;
      for ( markedFiles = selected ; markedFiles ; markedFiles-- )
      {
         if ( this->deList[this->hIndex]->writeAcc == false )
            ++wpCount ;
         this->NextSelectedItem () ;   // track to next 'selected' item
      }

      if ( wpCount > ZERO )            // if there are write-protected files
      {
         overrideSrcWProt = this->spDecision ( wpCount, 0, opRENAME ) ;
         if ( overrideSrcWProt != false )
         {  //* User has elected to override write protection *
            this->FirstSelectedItem () ;     // track to first 'selected' item
            wpCount = ZERO ;
            for ( markedFiles = selected ; markedFiles ; markedFiles-- )
            {
               if ( ! this->deList[this->hIndex]->writeAcc )
               {
                  this->fmPtr->CatPathFname ( gsPath, this->currDir, 
                                              this->deList[this->hIndex]->fName ) ;
                  if ( (this->fmPtr->WritePermission ( *this->deList[this->hIndex], 
                                                       gsPath, true )) == false )
                     ++wpCount ;
               }
               this->NextSelectedItem () ;   // track to next 'selected' item
            }
         }
      }
      this->FirstSelectedItem () ;     // track to first 'selected' item
      if ( wpCount == ZERO || overrideSrcWProt != false )
      {
         //* Pointer to renamePattern class object. Instantiated if needed.    *
         renamePattern* rPat = NULL ;

         //* We now have write access to all source files OR permission to do  *
         //* as much damage as possible.                                       *
         //* If multiple files have been 'selected', ask user whether he/she/it*
         //* wants to do a batch rename OR rename each file individually.      *
         //* If only one filename is 'selected', batch rename does not apply.  *
         short nStatus = promptRENAME ;   // assume single file to be renamed

         if ( selected > 1 )
         {
            rPat = new renamePattern ;    // create an instance for temp use
            if ( (nStatus = rsfGetRenamePattern ( rPat )) == abortRENAME )
            {  //* If user has aborted the rename process *
               this->DeselectFile ( true ) ;
               selected = ZERO ;
            }
         }


         //**************************************************
         //* Rename each file which is marked as 'selected' *
         //**************************************************
         tnFName* tnfPtr ;
         gString newFName ;
         short rStatus ;
         for ( markedFiles = selected ; markedFiles ; markedFiles-- )
         {
            rStatus = OK ;    // initialize to 'no error'

            //* Point to existing file data *
            tnfPtr = this->deList[this->hIndex] ;

            //* Construct new filename for target *
            if ( nStatus == promptRENAME )
            {
               if ( !(this->GetNewFilename ( tnfPtr, newFName )) )
               {
                  //* If unable to get a new filename from user we will either *
                  //* skip this file and continue with the list OR abort the   *
                  //* rename operation for the remaining files in the list.    *
                  //* Prompt user if this is not the last/only file            *
                  if ( markedFiles > 1 && ! ignoreRenameErrors )
                  {
                     if ( !(ignoreRenameErrors = rsfAlert ( tnfPtr->fName )) )
                     {  //* User has specified abort of the rename process.    *
                        markedFiles = 1 ; // terminate the loop
                     }
                  }
                  rStatus = ERR ;         // don't process the current file
               }
            }
            else  // nStatus == batchRENAME
            {
               rPat->formatFName ( newFName, tnfPtr->fName ) ;
            }

            //* If no problems in obtaining new filename, then rename the file *
            if ( rStatus == OK )
            {
               switch ( tnfPtr->fType )
               {
                  case fmREG_TYPE:     // regular file
                  case fmLINK_TYPE:    // symbolic link file
                  case fmFIFO_TYPE:    // FIFO file
                  case fmDIR_TYPE:     // directory file
                     rStatus = this->rsfRenameFile ( tnfPtr, 
                                                     this->currDir, newFName ) ;
                     break ;
                  case fmCHDEV_TYPE:   // character device file
                     rStatus = this->rsfRenameChdevFile ( tnfPtr, 
                                                          this->currDir, newFName ) ;
                     break ;
                  case fmBKDEV_TYPE:   // block device file
                     rStatus = this->rsfRenameBkdevFile ( tnfPtr, 
                                                          this->currDir, newFName ) ;
                     break ;
                  case fmSOCK_TYPE:    // socket file
                     rStatus = this->rsfRenameSocketFile ( tnfPtr, 
                                                           this->currDir, newFName ) ;
                     break ;
                  case fmUNKNOWN_TYPE: // unknown file type
                     rStatus = this->rsfRenameUnsuppFile ( tnfPtr, 
                                                           this->currDir, newFName ) ;
                  default:
                     rStatus = ecUNSUPP ; // operation not supported for this file type
                     break ;              // (should never happen)
               }
               if ( rStatus == OK )    // if rename successful, add it to count
                  ++renamedFiles ;
               else if ( ! ignoreRenameErrors )
               {
                  //* Write-protect or Access error detected in FileDlg methods*
                  //* which includes user command to skip this file.           *
                  //* Otherwise, FMgr or system error encountered during rename*
                  errorCode ec ;
                  this->GetErrorCode ( ec ) ;
                  const char* errMsg = rStatus >= FIRST_ERRORCODE ? 
                                       ec.fdErrorDesc : ec.sysErrorDesc ;
                  //* Report the error, and user's continue/abort instructions *
                  if ( !(ignoreRenameErrors = rsfAlert ( tnfPtr->fName, errMsg )) )
                  {  //* User has specified to abort the rename process.       *
                     markedFiles = 1 ;    // terminate the loop
                  }
               }
            }  // if(rStatus==OK)

            this->NextSelectedItem () ;   // track to next 'selected' item
         }  // for(;;)
         //* Release dynamic allocation of renamePattern object, if any.       *
         if ( rPat != NULL )
            delete rPat ;
      }
      else
      {  //* User has aborted the operation because one or more files are      *
         //* write-protected.                                                  *
         this->SetErrorCode ( ecSRCPROTECT, EACCES ) ;
      }
      //* Re-read directory contents (clipboard will be cleared) *
      this->RefreshCurrDir () ;
   }
   else                             // no files processed
   {
      this->DeselectFile ( true ) ; // deselect all files
      if ( ! srcdirWPerm )          // no write permission on this directory
      {
         this->DisplayStatus ( ecREADONLY_S, EACCES ) ;
      }
   }
   return renamedFiles ;

}  //* End RenameSelectedFiles () *

//***********************
//*   rsfRenameFile     *
//***********************
//******************************************************************************
//* Rename the specified file. Call this method to rename a file of any type;  *
//* HOWEVER, caller must first prompt the user for confirmation before rename  *
//* of 'special' files.                                                        *
//*                                                                            *
//* Note: It is assumed that tnf->writeAcc has been correctly initialized,     *
//*       and we WILL NOT attempt to override write protection here.           *
//*                                                                            *
//* Input  : tnf  : pointer to a tnFName class object which describes the file *
//*          sPath: pointer to source path string                              *
//*          newName: pointer to new-filename string                           *
//*                                                                            *
//* Returns: OK if file successfully renamed                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecTARGEXISTS    if file with target name already exists     *
//*                ecPACCV         if user cancelled rename (dummy value)      *
//******************************************************************************

short FileDlg::rsfRenameFile ( const tnFName* tnf, const char* sPath, const gString& newFName )
{
   short status = ecTARGPROTECT ;      // return value  (operation status)

   if ( tnf->writeAcc != false )
   {
      gString  oldPath, newPath ;
      this->fmPtr->CatPathFname ( oldPath, sPath, tnf->fName ) ;
      this->fmPtr->CatPathFname ( newPath, sPath, newFName.ustr() ) ;

      fmFType fType ;                  // source filetype

      if ( ! (this->TargetExists ( newPath, fType )) )
      {
         if ( clipBoard_gvfsSrc )
            status = this->fmPtr->RenameFile_gvfs ( oldPath.ustr(), newPath.ustr() ) ;
         else
            status = this->fmPtr->RenameFile ( oldPath.ustr(), newPath.ustr() ) ;
      }
      else
      {
         status = ecTARGEXISTS ;
         this->SetErrorCode ( ecTARGEXISTS, ZERO ) ;
      }
   }
   else
      this->SetErrorCode ( ecTARGPROTECT, ZERO ) ;

   return status ;

}  //* End rsfRenameFile() *

short FileDlg::rsfRenameChdevFile ( const tnFName* tnf, const char* sPath, const gString& newFName )
{
short status = ecPACCV ;

   if ( (this->DecisionDialog ( warnBdevCdevMod )) != false )
   {
      status = this->rsfRenameFile ( tnf, sPath, newFName ) ;
   }
   else
      this->SetErrorCode ( ecPACCV, EPERM ) ;

   return status ;

}  //* End rsfRenameChdevFile() *

short FileDlg::rsfRenameBkdevFile ( const tnFName* tnf, const char* sPath, const gString& newFName )
{
short status = ecPACCV ;

   if ( (this->DecisionDialog ( warnBdevCdevMod )) != false )
   {
      status = this->rsfRenameFile ( tnf, sPath, newFName ) ;
   }
   else
      this->SetErrorCode ( ecPACCV, EPERM ) ;

   return status ;

}  //* End rsfRenameBkdevFile() *

short FileDlg::rsfRenameSocketFile ( const tnFName* tnf, const char* sPath, const gString& newFName )
{
short status = ecPACCV ;

   if ( (this->DecisionDialog ( warnSocketMod )) != false )
   {
      status = this->rsfRenameFile ( tnf, sPath, newFName ) ;
   }
   else
      this->SetErrorCode ( ecPACCV, EPERM ) ;

   return status ;

}  //* End rsfRenameSocketFile() *

short FileDlg::rsfRenameUnsuppFile ( const tnFName* tnf, const char* sPath, const gString& newFName )
{
short status = ecPACCV ;

   if ( (this->DecisionDialog ( warnUnsuppMod )) != false )
   {
      status = this->rsfRenameFile ( tnf, sPath, newFName ) ;
   }
   else
      this->SetErrorCode ( ecPACCV, EPERM ) ;

   return status ;

}  //* End rsfRenameUnsuppFile() *

//*************************
//*    GetNewFilename     *
//*************************
//******************************************************************************
//* Ask user for a new target filename. Called by RenameSelectedFiles().       *
//*                                                                            *
//* Input  : tnfPtr : pointer to tnFName class object describing the file      *
//*          newName: buffer to hold new filename                              *
//*                                                                            *
//* Returns:  true  == new target filename specified                           *
//*           false == abort rename operation                                  *
//*                    (newName == null string indicates user trying           *
//*                     to copy file onto itself)                              *
//******************************************************************************

bool FileDlg::GetNewFilename ( tnFName* tnfPtr, gString& newName )
{
   bool  success = false ;          // return value
   newName = tnfPtr->fName ;        // get a copy of original name
   if ( (success = this->GetNewFilename ( newName )) != false )
   {
      if ( (strncmp ( newName.ustr(), tnfPtr->fName, gsDFLTBYTES )) == ZERO )
      {  //* User failed to enter a new (different) filename *
         newName = "" ;
         success = false ;
      }
   }
   return success ;

}  //* End GetNewFilename() *

//*************************
//*    GetNewFilename     *
//*************************
//******************************************************************************
//* Ask user for a new target filename.                                        *
//*                                                                            *
//*                                                                            *
//* Input  : newName : old filename is passed in through this buffer           *
//*                    new name is returned in it                              *
//*          msg     : (optional, NULL* by default)                            *
//*                    If specified, references a 1- or 2-line auxilliary      *
//*                    message to be displayed. It is caller's                 *
//*                    responsibility to size the message properly to          *
//*                    avoid ugly-ness. Note that this is a dynamically-sized  *
//*                    window, and that the minimum size is dictated by the    *
//*                    width of the file-display dctSCROLLEXT control, which   *
//*                    in dual-window mode yields a minimum window width of    *
//*                    MIN_DUALWINDOW_WIDTH / 2 - 4 == 55 columns. This        *
//*                    leaves a minimum message width of 51 columns.           *
//*          suggName: (optional, NULL* by default)                            *
//*                    If specified, this is the caller's suggestion for the   *
//*                    new target filename.                                    *
//*                                                                            *
//* Returns:  true  == new target filename specified                           *
//*                    (new name MAY BE same as old name)                      *
//*           false == abort rename operation (newName unchanged)              *
//******************************************************************************
//* See notes in RenameSelectedFiles().                                        *
//* See the NcDialog-class Texbox filter for filenames.                        *
//*                                                                            *
//* Note: If the prompt shows an empty Textbox to the users, it means that     *
//*       the existing filename contains a character that did not pass the     *
//*       tbFileName filter.                                                   *
//*       -- If the filename already contains a special character, then we     *
//*          SHOULD allow it to remain, but this would create a hidden         *
//*          exception to the filter, so we do not allow it at this time.      *
//******************************************************************************

//* Prototype and dialog pointer for use by call-back method *
static short gnfControlUpdate ( const short currIndex, 
                                const wkeyCode wkey, bool firstTime = false ) ;
static NcDialog* gnfPtr = NULL ;

bool FileDlg::GetNewFilename ( gString& newName, const char* msg, const gString* suggName )
{
   const short dialogLines = short(msg == NULL ? 8 : 11),
               okPush      = ZERO,
               canPush     = 1,
               fnameTB     = 2,
//               insovrRB    = 3,
               controlsDEFINED = 4 ;
   short    dialogCols = this->fMaxX - 4,
            ctrY = this->fulY + this->fMaxY/2,
            ctrX = this->fulX + this->fMaxX/2,
            ulY  = ctrY - dialogLines / 2,
            ulX  = ctrX - dialogCols / 2 ;
   gString  origName = newName ;          // original filename
   attr_t   dColor = this->cs.sd ;        // dialog background color
   bool     success = false ;             // return value

   //* If a 'suggested filename' was provided, place in in the text box. *
   //* Else set text box text to original filename.                      *
   const char* nameString ;
   if ( suggName == NULL ) nameString = newName.ustr() ;
   else                    nameString = suggName->ustr() ;

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -   okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 2 - 11),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   OK   ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -    canPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 2 + 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* 'file name' Text Box - - - - - - - - - - - - - - - - - - -   fnameTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dialogCols - 4),        // cols:      control columns
      nameString,                   // dispText:  initially empty
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter:    valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      "Target Name:",               // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3]                        // nextCtrl:  link in next structure
   },
   {  //* 'INS/OVR' radio button (inactive) - - - - - - - - - - - -   insovrRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // sbSubtype: alt font, 5-wide
      true,                         // rbSelect:  initially 'insert' mode
      short(ic[fnameTB].ulY + 1),   // ulY:       upper left corner in Y
      short(dialogCols - 5),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      dColor,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "('Insert' toggles Ins/Overstrike mode) INSERT:", // label:
      ZERO,                         // labY:      
      -46,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   gnfPtr = dp ;     // file-scope access to the dialog (for call-back method)

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Print dialog window's static text *
      dp->SetDialogTitle ( "  Specify Target Filename  ", this->cs.em ) ;
      dp->WriteString ( 1,  2, "Source Name:", dColor ) ;
      dp->WriteString ( 1, 15, origName, dColor ) ;
      if ( msg != NULL )
      {  //* Center the message in the window *
         short msgWidth ;
         for ( msgWidth = ZERO ; 
               msg[msgWidth] != NULLCHAR && msg[msgWidth] != NEWLINE ; msgWidth++ ) ;
         winPos wp( (dialogLines - 5), (dialogCols/2 - msgWidth/2) ) ;

         dp->WriteParagraph ( wp.ypos, wp.xpos, msg, this->cs.em ) ;
      }

      //* Set 'insert' input mode (matches initial state of radiobutton) *
      dp->SetTextboxInputMode ( false ) ;

      //* Enable audible alert for invalid input *
      dp->TextboxAlert ( fnameTB, true ) ;

      //* Start with cursor at the right end of existing data.           *
      //* (note that this command won't work if textbox has input focus) *
      dp->SetTextboxCursor ( fnameTB, tbcpAPPEND ) ;
      short icIndex = dp->PrevControl () ;

      dp->RefreshWin () ;                 // make the text visible

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &gnfControlUpdate ) ;

      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* Accept edits *
               if ( Info.ctrlIndex == okPush )
               {
                  //* If edit results in something other than a NULL string *
                  if ( (dp->GetTextboxText ( fnameTB, newName )) > 1 )
                  {
                     //* Truncate the name to the POSIX max length.            *
                     //* (Few users would be sufficiently caffinated to create)*
                     //* (a filename that long, so we do it the easy way.)     *
                     newName.limitChars( MAX_FNAME - 2 ) ;

                     #if LINUX_SPECIAL_CHARS != 0
                     //* Test the data from the sample-text field. If "special"*
                     //* characters detected, ask user whether to proceed.     *
                     success = this->gnfSpecialCharsQuery ( newName, dp, fnameTB, tbFileName ) ;
                     if ( ! success )
                        icIndex = dp->NextControl () ;   // focus goes to Textbox
                     else
                     #endif   // LINUX_SPECIAL_CHARS
                        success = true ;
                     done = success ;
                  }
                  //* Invalid filename specified *
                  else
                     newName = origName ;
               }
               //* Discard edits and return original data *
               else if ( Info.ctrlIndex == canPush )
               {
                  newName = origName ;
                  done = true ;
               }
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
   {
      delete ( dp ) ;
      gnfPtr = NULL ;
   }

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return success ;

}  //* End GetNewFilename() *

//*************************
//*   gnfControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* GetNewFilename() method's dialog window.                                   *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. The Insert/Overstrike indicator is not selectable by the user;        *
//*      however, it tracks the state of the Insert key through this           *
//*      callback  method. If the NCurses engine is waiting for key input      *
//*      from the user, (and this is how the time is spent), it will monitor   *
//*      the state of this toggle because it is needed for text field input.   *
//*      Since we have the information available, why not display it!          *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control pointer, gnfPtr, at file scope above.*
//******************************************************************************

static short gnfControlUpdate ( const short currIndex, 
                                const wkeyCode wkey, bool firstTime )
{
   static bool oldOver = false ;    // state of insert/overstrike
   const short insovrRB = 3 ;       // index of inactive radio button control
   
   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize  the static variables *
      oldOver = gnfPtr->IsOverstrike () ; // initial state of insert/overstrike flag
      bool currOver ;
      gnfPtr->GetRadiobuttonState ( insovrRB, currOver ) ;
      if ( currOver != oldOver )
         gnfPtr->SetRadiobuttonState ( insovrRB, currOver ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   //* If necessary, update the non-active radio button *
   //* that indicates the insert/overstrike state       *
   bool newOver = gnfPtr->IsOverstrike () ; // current state of insert/overstrike
   if ( newOver != oldOver )
   {
      gnfPtr->SetRadiobuttonState ( insovrRB, oldOver ) ;
      oldOver = newOver ;
   }
   return OK ;
   
}  //* End gnfControlUpdate() *

//************************
//* gnfSpecialCharsQuery *
//************************
//******************************************************************************
//* Filename and pathspec Textbox controls are set so that valid filename      *
//* characters include all printing characters                                 *
//*             (except '`' backtick and '/' forward slash).                   *
//* However, not all characters which are valid under Linux/UNIX are convenient*
//* or wise; and indeed some filesystems (VFAT, NTFS) can handle all the valid *
//* Linux filename characters.                                                 *
//*                                                                            *
//* Therefore, we test the user's data against the more consertive filter      *
//* (tbFileName, or for path specs tbPathName). If the data contain characters *
//* beyond the conservative group, we ask the user whether to continue with    *
//* the extended character group is acceptable.                                *
//*                                                                            *
//* Input  : newName : data to be scanned (new filename or pathspec)           *
//*          cDialog : access to calling dialog                                *
//*          tbIndex : control index of Textbox to perform the test            *
//*          filter  : text filter to use (usually tbFileName or tbPathName)   *
//*                                                                            *
//* Returns: 'true'  : accept the special characters in filename               *
//*          'false' : reject the special characters in filename               *
//******************************************************************************

bool FileDlg::gnfSpecialCharsQuery ( const gString& newName, NcDialog *cDialog, 
                                     short tbIndex, tbChars filter )
{
   bool allowLinuxChars = true ;    // return value

   #if LINUX_SPECIAL_CHARS != 0 && LINUX_SPECIAL_QUERY != 0
   //* If the new name contains Linux special characters, *
   //* warn the user about portability issues.            *
   //* Note that the called method could theoretically    *
   //* modify the text data.                              *
   gString newNameCopy = newName ;
   if ( (cDialog->VerifyTbText ( tbIndex, newNameCopy, filter )) != OK )
   {
      //* Format the name. If necessary, truncate *
      //* the data to fit the width of the dialog.*
      gString gsMsg( " %S ", newName.gstr() ) ;
      if ( (gsMsg.gscols()) >= (INFODLG_WIDTH - 4) )
      {
         gsMsg.limitCols ( (INFODLG_WIDTH - 9) ) ;
         gsMsg.append( ".... " ) ;
      }
      const char* msgText[] = 
      {// 123456789x123456789x123456789x123456789x123456789x12
         "  WARNING!  WARNING!  ",
         gsMsg.ustr(),
         "The specified name contains one or more \"special\"",
         "characters which some non-Linux filesystems such as",
         "VFAT and NTFS do not support as filename characters.",
         "          \\  '  \"  ?  :  ;  &  >  <  |  *",
         " ",
         "             Do you want to continue?",
         NULL
      } ;
      const attr_t msgAttr[] = 
      {
         this->cs.em,
         this->cs.tf,
         this->cs.sd,
         this->cs.sd,
         this->cs.sd,
         this->cs.sd,
         this->cs.sd,
         this->cs.em,
         this->cs.bb,
         this->cs.bb,
      } ;

      cDialog->SetDialogObscured () ;        // save caller's dialog
      this->dPtr->RefreshWin () ;            // hide the caller's dialog
      this->dPtr->SetDialogObscured () ;     // save application dialog

      allowLinuxChars = this->DecisionDialog ( msgText, msgAttr ) ;

      cDialog->RefreshWin () ;               // refresh caller's dialog
   }
   #endif   // LINUX_SPECIAL_CHARS && LINUX_SPECIAL_QUERY

   return allowLinuxChars ;

}  //* End gnfSpecialCharsQuery() *

//*************************
//*  EscapeSpecialChars   *
//*************************
//******************************************************************************
//* Scan the source string (typically a filespec). If the string contains any  *
//* characters of the specified character group (enum escGroup) 'escape' them  *
//* with a backslash character.                                                *
//*                                                                            *
//* Do not use this method for URL (Universal Resource Locator) specs.         *
//* URL specifications require a specific protocol which is not handled here.  *
//*                                                                            *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : pathSpec : (by reference) source text data                        *
//*                     on return the data has been 'escaped' according to the *
//*                     specified criteria                                     *
//*          group    : (member of enum escGroup) specifies the group of       *
//*                     characters to be 'escaped'                             *
//*          wch      : (optional, NULLCHAR by default)                        *
//*                     if 'group' parameter == escWCHAR, then 'wch' specifies *
//*                     a single character to be 'escaped'                     *
//*                                                                            *
//* Returns: number of characters 'escaped'                                    *
//******************************************************************************

short FileDlg::EscapeSpecialChars ( gString& pathSpec, escGroup group, wchar_t wch )
{

   return ( (this->fmPtr->EscapeSpecialChars ( pathSpec, group, wch )) ) ;

}  //* End EscapeSpecialChars() *

//***********************
//* rsfGetRenamePattern *
//***********************
//******************************************************************************
//* Ask user to make a decision about how to rename the selected files.        *
//* If user has selected more than one file, we can either:                    *
//*  a) prompt user for a new filename for each file in the list               *
//*  b) use a regular-expression pattern to be automatically applied to rename *
//*     each file in the list                                                  *
//*                                                                            *
//* Input  : rPat      : pointer to a renamePattern object                     *
//*          copyFiles : (optional, 'false' by default)                        *
//*                      If 'true', then we were called by one of the copy-file*
//*                      methods rather than by a rename-file method. Dialog   *
//*                      title is modified to indicate this.                   *
//*          msg       : (optional, NULL* by default)                          *
//*                      If copyFiles != false, references a 1- or 2-line      *
//*                      auxilliary message to be displayed. It is caller's    *
//*                      responsibility to size the message properly to        *
//*                      avoid ugly-ness.                                      *
//*                                                                            *
//* Returns: member of enum renameOption                                       *
//******************************************************************************

renameOption FileDlg::rsfGetRenamePattern ( renamePattern* rPat, 
                                            bool copyFiles, const char* msg )
{
const short dialogRows = 12,
            dialogCols = 76 ;
enum cIndices : short { indexOK = ZERO, indexCAN, indexRBI, indexRBB, indexCOUNT } ;
short ctrY = this->fulY + this->fMaxY/2,
      ctrX = (this->dCols + this->dulX) / 2, // center across application dialog
      ulY  = ctrY - dialogRows / 2,
      ulX  = ctrX - dialogCols / 2 ;

attr_t dColor  = this->cs.sd,       // dialog background color
       pnColor = this->cs.pn,       // pushbutton non-focus color
       pfColor = this->cs.pf,       // pushbutton focus color
       rnColor = dColor,            // radiobutton non-focus color
       emColor = this->cs.em,       // emphasized text color
       rfColor = this->cs.pf ;      // radiobutton focus color
renameOption status = abortRENAME ; // return value

   //****************************************
   //* Initial parameters for dialog window *
   //****************************************
   InitCtrl ic[indexCOUNT] =                    // control initialization structures
   {
      {  //* OK pushbutton       - - - - - - - - - - - - - - - - -     indexOK *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dialogRows - 2),      // ulY:       upper left corner in Y
         (short)(dialogCols/2 - 11),   // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         "   OK   ",                   // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[1]                        // nextCtrl:  link in next structure
      },
      {  //* CANCEL pushbutton   - - - - - - - - - - - - - - - - -    indexCAN *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dialogRows - 2),      // ulY:       upper left corner in Y
         (short)(dialogCols/2 + 4),    // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " CANCEL ",                   // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[2]                        // nextCtrl:  link in next structure
      },
      {  //* INDIV radiobutton   - - - - - - - - - - - - - - - - -    indexRBI *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         true,                         // rbSelect:  default selection
         2,                            // ulY:       upper left corner in Y
         3,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Each file to be renamed individually.", // label:
         ZERO,                         // labY:      
         7,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[3]                        // nextCtrl:  link in next structure
      },
      {  //* BATCH radiobutton   - - - - - - - - - - - - - - - - -    indexRBB *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         false,                        // rbSelect:  initially reset
         4,                            // ulY:       upper left corner in Y
         3,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Batch rename all files using Pattern.", // label:
         ZERO,                         // labY:      
         7,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogRows,     // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Options for Multi-file Rename  ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Print dialog window's static text *
      dp->WriteString ( ic[indexRBI].ulY+1,  ic[indexRBI].ulX + ic[indexRBI].labX, 
         "For each selected file, prompt for new filename.", dColor ) ;
      dp->WriteString ( ic[indexRBB].ulY+1,  ic[indexRBB].ulX + ic[indexRBB].labX, 
         "For all selected files, rename according to specified pattern.", dColor ) ;
      if ( copyFiles )
      {
         dp->SetDialogTitle ( "  Options for Multi-file Copy with Rename  " ) ;
         if ( msg != NULL )
            dp->WriteParagraph ( (dialogRows - 5), 2, msg, emColor ) ;
      }

      //* Set radio buttons as an exclusive-OR group *
      //* (only one can be selected at any moment).  *
      short XorGroup[] = { indexRBI, indexRBB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      dp->RefreshWin () ;                 // make the text visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control that has focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* Accept edits *
               if ( Info.ctrlIndex == indexOK )
               {
                  //* Determine which radio button has been selected *
                  short radioSelect = dp->GetRbGroupSelection ( indexRBI ) ;
                  if ( radioSelect == indexRBI )
                     status = promptRENAME ;
                  else if ( radioSelect == indexRBB )
                     status = batchRENAME ;
               }
               //* Cancel the rename operation *
               else if ( Info.ctrlIndex == indexCAN )
               {
                  status = abortRENAME ;
               }
               done = true ;
            }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
            /* (nothing to do here) */
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* If user has elected to do a batch rename, get the rename pattern and    *
   //* place it in the rPat class object defined at the top of this module.    *
   if ( status == batchRENAME )
   {
      status = this->rsfConstructPattern ( rPat, dialogRows, dialogCols, ulY, ulX ) ;
   }

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return status ;

}  //* End rsfGetRenamePattern() *

//***********************
//* rsfConstructPattern *
//***********************
//******************************************************************************
//* Lead user through the options for creating a regular-expression pattern for*
//* performing a batch rename.                                                 *
//*                                                                            *
//* Note that we make no attempt to implement the full range of regular        *
//* expressions. That is too complex for this simple operation. Instead, we    *
//* offer the following options:                                               *
//*    1) Prepend to existing filename                                         *
//*       Example: prepend a 'BAK-' to each selected file:                     *
//*          Old Name:  fname.fext                                             *
//*          New Name:  BAK-fname.fext                                         *
//*    2) Append to existing filename                                          *
//*       Example: append a '-BAK' to each selected file:                      *
//*          Old Name:  fname.fext                                             *
//*          New Name:  fname.fext-BAK                                         *
//*    3) Insert into existing filename after the base filename but before     *
//*       filename extension (and before any sequence number - see below).     *
//*       Example: insert a '-LoRes' into each selected file:                  *
//*          Old Name:  fname.fext                                             *
//*          New Name:  fname-LoRes.fext                                       *
//*    4) Replace existing filename with a literal string + sequence number    *
//*       Example: replace existing filenames aaaaaaaaaaa.jpg                  *
//*                with Tahiti Trip nnn.jpg                                    *
//*          Old Name:  fname.fext                                             *
//*          New Name:  Tahiti Trip 001.jpg                                    *
//*    5) Replace existing filename extension.                                 *
//*       Example: replace existing filename extensions (including the         *
//*                delimiting period character '.') with .BAK                  *
//*          Old Name:  fname.txt                                              *
//*          New Name:  fname.BAK                                              *
//*    6) Insert date/time stamp into one of the above fields.                 *
//*       The date/time stamp is of the form yyyymmddhhmmss                    *
//*       It may be inserted into any field defined by the above operations    *
//*       (except that it may not replace the sequence-number).                *
//*                                                                            *
//*                                                                            *
//* Input  : rPat  : pointer to a renamePattern object                         *
//*          dLines: number of dialog lines                                    *
//*          dCols : number of dialog columns                                  *
//*          dulY  : upper left corner Y                                       *
//*          dulX  : upper left corner X                                       *
//*                                                                            *
//* Returns: member of enum renameOption                                       *
//******************************************************************************
//* NOTE: This dialog is the same size and in the same position as the         *
//* caller's dialog, so caller's dialog is hidden.                             *
//*                                                                            *
//* Note on placement of date/time stamp:                                      *
//*  If possible, place date/time stamp without bothering user; however, if    *
//*  necessary, prompt user to specify date-stamp position.                    *
//* Notes on user-friendliness:                                                *
//*  Two conditions offer un-ambiguous indication of where the date string     *
//*  should be placed:                                                         *
//*  1) If only one radiobutton is 'set' AND its associated textbox contains   *
//*     a NULL string.                                                         *
//*  2) If multiple radiobuttons are 'set' and exactly one associated textbox  *
//*     contains NULL string.                                                  *
//*  For these conditions, it is not necessary to prompt user for date string  *
//*  placement. For all other conditions, prompt user to prepare a field to    *
//*  receive date string.                                                      *
//******************************************************************************

renameOption FileDlg::rsfConstructPattern ( renamePattern* rPat, short dLines, 
                                            short dCols, short dulY, short dulX )
{
const short SAMPLE_TBW  = 51,    // width of 'Sample' textbox
            iniWIDTH    = 2,     // initial value of 'Width' spinner control
            iniFIRST    = 1,     // initial value of 'First' spinner control
            PEND_TBW    = 16,    // field width for prepend, append, insert fields
            REPLACE_TBW = (PEND_TBW * 2) ;  // field width for replace field

enum cIndices
{
   indexOK = ZERO, indexCAN, indexSAMP, indexRBP, indexTBP, indexRBI, indexTBI, 
   indexRBA, indexTBA, indexRBR, indexTBR, indexSPW, indexSPF, indexRBD, 
   indexCLR, indexHELP, indexCOUNT
} ;

// Programmer's Note: Because of the complexity of this dialog, it has been 
// exempted from following the application's Color Scheme.
attr_t   dColor  = nc.blR,                // dialog background color
         pnColor = nc.re,                 // pushbutton non-focus color
         pfColor = nc.reS,                // pushbutton focus color
         rnColor = dColor,                // radiobutton non-focus color
         rfColor = nc.reS,                // radiobutton focus color
//         stnColor = nc.cyS | ncbATTR,     // 'sample' textbox non-focus color
         stnColor = nc.bkcy,              // 'sample' textbox non-focus color
         tnaColor = nc.brR,               // non-focus color for 'A' textboxes
         tnbColor = nc.maR,               // non-focus color for 'B' textboxes
         tfColor = nc.bw,                 // textbox focus color
         hnColor = nc.gy,                 // help button non-focus color
         hfColor = nc.grS ;               // help button focus color
short    icIndex = ZERO ;                 // index of control that has focus
renameOption status = abortRENAME ;       // return value

   //* Additional parameters for dctSPINNER object initializations *
             //* Sequence field width *
   dspinData dsData1 ( 0,     5, iniWIDTH, dspinINTEGER, tnaColor ),
             //* Sequence initial value *
             dsData2 ( 0, 10000, iniFIRST, dspinINTEGER, tnaColor ) ;

   //* If caller has initialized rPat fields to non-default values *
   gString prInit, inInit, apInit, reInit, gsMsg ;
   if ( rPat->pr )   prInit = rPat->prStr ;
   if ( rPat->in )   inInit = rPat->inStr ;
   if ( rPat->ap )   apInit = rPat->apStr ;
   if ( rPat->base || rPat->ext )
   {  //* Only one of these flags may be set at a time, *
      //* and we give presedence to the extension field.*
      reInit = rPat->reStr ;
      if ( rPat->ext )
      {
         rPat->base = false ;
         rPat->seqDigits = ZERO ;
         dsData1.iniValue = ZERO ;
      }
      else
         rPat->ext = false ;
   }

   InitCtrl ic[indexCOUNT] =                    // control initialization structures
   {
      {  //* 'RENAME' pushbutton   - - - - - - - - - - - - - - - -     indexOK *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dLines - 2),          // ulY:       upper left corner in Y
         (short)(dCols/2 - 4),         // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " R^ENAME ",                   // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[1]                        // nextCtrl:  link in next structure
      },
      {  //* CANCEL pushbutton   - - - - - - - - - - - - - - - - -    indexCAN *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dLines - 2),          // ulY:       upper left corner in Y
         (short)(ic[indexOK].ulX + 11),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " ^CANCEL ",                  // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[2]                        // nextCtrl:  link in next structure
      },
      {  //* SAMPLE textbox      - - - - - - - - - - - - - - - - -   indexSAMP *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         3,                            // ulY:       upper left corner in Y
         (short)(dCols - SAMPLE_TBW - 1),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         SAMPLE_TBW,                   // cols:      control columns
         NULL,                         // dispText:  (initially none)
         stnColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    any printing character
         "Sample:",                    // label:     
         ZERO,                         // labY:      
         -8,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    cannot gain focus
         &ic[3]                        // nextCtrl:  link in next structure
      },
      {  //* PREPEND radiobutton - - - - - - - - - - - - - - - - -    indexRBP *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         rPat->pr,                     // rbSelect:  caller's option
         4,                            // ulY:       upper left corner in Y
         10,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Before:",                   // label:
         ZERO,                         // labY:      
         -7,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[4]                        // nextCtrl:  link in next structure
      },
      {  //* PREPEND textbox     - - - - - - - - - - - - - - - - -    indexTBP *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[indexRBP].ulY),    // ulY:       upper left corner in Y
         (short)(ic[indexRBP].ulX + 6),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         PEND_TBW,                     // cols:      control columns
         prInit.ustr(),                // dispText:  (initially none)
         tnaColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         #if LINUX_SPECIAL_CHARS != 0
         tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
         #else    // BASIC FILENAME FILTER
         tbFileName,                   // filter:    valid filename characters
         #endif   // BASIC FILENAME FILTER
         "",                           // label:     (none)
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    (initially disabled)
         &ic[5]                        // nextCtrl:  link in next structure
      },
      {  //* INSERT radiobutton  - - - - - - - - - - - - - - - - -    indexRBI *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         rPat->in,                     // rbSelect:  caller's option
         short(ic[indexRBP].ulY + 1),  // ulY:       upper left corner in Y
         short(ic[indexRBP].ulX),      // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "I^nsert:",                   // label:
         ZERO,                         // labY:      
         -7,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[6]                        // nextCtrl:  link in next structure
      },
      {  //* INSERT textbox      - - - - - - - - - - - - - - - - -    indexTBI *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[indexRBI].ulY),    // ulY:       upper left corner in Y
         (short)(ic[indexRBI].ulX + 6),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         PEND_TBW,                     // cols:      control columns
         inInit.ustr(),                // dispText:  (initially none)
         tnbColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         #if LINUX_SPECIAL_CHARS != 0
         tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
         #else    // BASIC FILENAME FILTER
         tbFileName,                   // filter:    valid filename characters
         #endif   // BASIC FILENAME FILTER
         "",                           // label:     (none)
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    (initially disabled)
         &ic[7]                        // nextCtrl:  link in next structure
      },
      {  //* APPEND radiobutton  - - - - - - - - - - - - - - - - -    indexRBA *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         rPat->ap,                     // rbSelect:  caller's option
         short(ic[indexRBI].ulY + 1),  // ulY:       upper left corner in Y
         short(ic[indexRBI].ulX),      // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^After :",                   // label:
         ZERO,                         // labY:      
         -7,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[8]                        // nextCtrl:  link in next structure
      },
      {  //* APPEND textbox      - - - - - - - - - - - - - - - - -    indexTBA *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[indexRBA].ulY),    // ulY:       upper left corner in Y
         (short)(ic[indexRBA].ulX + 6),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         PEND_TBW,                     // cols:      control columns
         apInit.ustr(),                // dispText:  (initially none)
         tnaColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         #if LINUX_SPECIAL_CHARS != 0
         tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
         #else    // BASIC FILENAME FILTER
         tbFileName,                   // filter:    valid filename characters
         #endif   // BASIC FILENAME FILTER
         "",                           // label:     (none)
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    (initially disabled)
         &ic[9]                        // nextCtrl:  link in next structure
      },
      {  //* REPLACE radiobutton - - - - - - - - - - - - - - - - -    indexRBR *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         bool(rPat->base || rPat->ext),// rbSelect:  initially reset
         short(ic[indexRBA].ulY + 1),  // ulY:       upper left corner in Y
         short(ic[indexRBA].ulX),      // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Replace:",                  // label:
         ZERO,                         // labY:      
         -8,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[10]                       // nextCtrl:  link in next structure
      },
      {  //* REPLACE textbox     - - - - - - - - - - - - - - - - -    indexTBR *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[indexRBR].ulY),    // ulY:       upper left corner in Y
         (short)(ic[indexRBR].ulX + 6),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         REPLACE_TBW,                  // cols:      control columns
         reInit.ustr(),                // dispText:  (initially none)
         tnbColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         #if LINUX_SPECIAL_CHARS != 0
         tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
         #else    // BASIC FILENAME FILTER
         tbFileName,                   // filter:    valid filename characters
         #endif   // BASIC FILENAME FILTER
         "",                           // label:     (none)
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    (initially disabled)
         &ic[11]                       // nextCtrl:  link in next structure
      },
      {  //* WIDTH spinner       - - - - - - - - - - - - - - - - -    indexSPW *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[indexTBR].ulY),    // ulY:       upper left corner in Y
         (short)(ic[indexTBR].ulX + ic[indexTBR].cols + 9),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         tnbColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Width ",                     // label:     
         1,                            // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData1,                     // spinData:  spinner init
         false,                        // active:    (initially disabled)
         &ic[12]                       // nextCtrl:  link in next structure
      },
      {  //* FIRST spinner       - - - - - - - - - - - - - - - - -    indexSPF *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[indexTBR].ulY),    // ulY:       upper left corner in Y
         (short)(ic[indexSPW].ulX + ic[indexSPW].cols + 3),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         tnbColor,                     // nColor:    non-focus color
         tfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "InitVal",                    // label:     
         1,                            // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData2,                     // spinData:  spinner init
         false,                        // active:    (initially disabled)
         &ic[13]                       // nextCtrl:  link in next structure
      },
      {  //* DATE radiobutton    - - - - - - - - - - - - - - - - -    indexRBD *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // sbSubtype: alt font, 5-wide
         false,                        // rbSelect:  initially reset
         10,                           // ulY:       upper left corner in Y
         9,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Date  :",                   // label:
         ZERO,                         // labY:      
         -7,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[14]                       // nextCtrl:  link in next structure
      },
      {  //* CLEAR pushbutton   - - - - - - - - - - - - - - - - -     indexCLR *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dLines - 2),          // ulY:       upper left corner in Y
         (short)(ic[indexCAN].ulX + 11),// ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         7,                            // cols:      control columns
         " C^LEAR ",                   // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[15],                      // nextCtrl:  link in next structure
      },
      {  //* HELP Pushbutton     - - - - - - - - - - - - - - - - -   indexHELP *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dLines - 2),          // ulY:       upper left corner in Y
         (short)(dCols - 10),          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         "  ^HELP  ",                  // dispText:  
         hnColor,                      // nColor:    non-focus color
         hfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dLines,         // number of display lines
                       dCols,          // number of display columns
                       dulY,           // Y offset from upper-left of terminal 
                       dulX,           // X offset from upper-left of terminal 
                       "  Construct Rename Pattern  ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Print dialog window's static text *
      dp->WriteParagraph ( 1, 2, 
         "For automated rename of selected files: select desired option(s), enter\n"
         "necessary text data, and view a sample of the resulting pattern below.", dColor ) ;
      dp->WriteString ( ic[indexTBP].ulY, (ic[indexTBP].ulX + ic[indexTBP].cols + 1), 
         "Add text to Beginning of filename.", dColor ) ;
      dp->WriteString ( ic[indexTBI].ulY, (ic[indexTBI].ulX + ic[indexTBI].cols + 1), 
         "Insert text before filename extension.", dColor ) ;
      dp->WriteString ( ic[indexTBA].ulY, (ic[indexTBA].ulX + ic[indexTBA].cols + 1), 
         "Add text to end of filename.", dColor ) ;
      dp->WriteString ( ic[indexRBR].ulY + 1, ic[indexRBR].ulX, 
         "Replace base name with text+sequence number,", dColor ) ;
      dp->WriteString ( ic[indexRBR].ulY + 2, ic[indexRBR].ulX, 
         "or if Width=0 replace extension with text.", dColor ) ;
      dp->WriteString ( ic[indexRBD].ulY, ic[indexRBD].ulX + 5, 
         "(yyyymmddhhmmss)", dColor ) ;

      dp->SetTextboxInputMode ( false ) ; // set 'Insert Mode' for text-input fields

      dp->RefreshWin () ;                 // make the text visible

      rPat->seqDigits = iniWIDTH ;        // initial values for spinner controls
      rPat->seqStart  = iniFIRST ;
      bool     tooLongMsg = false ; // prevents multiple display of error msg for single error
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         //* Update the 'Sample' pattern display control.                 *
         //* The display field is approximately 50 columns, so we have to *
         //* give a warning if the pattern cannot be fully displayed.     *
         short nameWidth = rPat->formatFName ( gsMsg, SampleFName, true ) ;
         dtbmData msgData( gsMsg.gstr() ) ;   // message display
         if ( nameWidth > SAMPLE_TBW )
         {
            if ( tooLongMsg == false )
            {
               const char* msgList[] = 
               { //2345678901234567890123456789012345678901234567890123
                  "  Caution  ",
                  "The rename pattern is valid and complete, but is too",
                  "long to be fully displayed in the 'Sample' field.",
                  NULL
               } ;
               this->InfoDialog ( msgList ) ;
               dp->RefreshWin () ;
               tooLongMsg = true ;
            }
         }
         else
            tooLongMsg = false ;
         dp->DisplayTextboxMessage ( indexSAMP, msgData ) ;

         //***************************************
         //* Focus is on a dctPUSHBUTTON control *
         //***************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               //* Accept edits *
               if ( Info.ctrlIndex == indexOK )
               {
                  //* Determine whether we have a valid rename pattern.        *
                  //* 1) Error if no radio button is set, because then no      *
                  //*    rename is possible.                                   *
                  //* 2) Error if any radio button is 'set', but its associated*
                  //*    text box contains a NULL string, it might work anyway *
                  //*    but it's illogical, so disallow it.                   *
                  //* 3) Warn if filename extension replacement does not begin *
                  //*    with a period '.'.                                    *
                  //* 4) Warn if Linux "special" characters detected.          *
                  //*                                                          *
                  //* If none of the above conditins exist, pattern is valid.  *
                  if ( ! rPat->pr && ! rPat->in && ! rPat->ap && ! rPat->base 
                                                               && ! rPat->ext )
                  {
                     const char* msgText[] = 
                     {// 123456789x123456789x123456789x123456789x123456789x12
                        "  ERROR!  ERROR!  ",
                        " ",
                        "No pattern fields have been activated.",
                        "Please activate at least one field and specify text",
                        "data for it.",
                        " ",
                        "   Select the 'HELP' button for more information.",
                        NULL,
                     } ;
                     this->InfoDialog ( msgText ) ;
                     dp->RefreshWin () ;
                  }
                  else if (   (rPat->pr && *rPat->prStr == NULLCHAR)
                           || (rPat->in && *rPat->inStr == NULLCHAR)
                           || (rPat->ap && *rPat->apStr == NULLCHAR)
                           || ((rPat->base || rPat->ext) && *rPat->reStr == NULLCHAR) )
                  {
                     const char* msgText[] = 
                     {// 123456789x123456789x123456789x123456789x123456789x12
                        "  ERROR!  ERROR!  ",
                        " ",
                        "One or more active fields has no associated text.",
                        "Each active field must contain text data.",
                        " ",
                        "   Select the 'HELP' button for more information.",
                        NULL,
                     } ;
                     this->InfoDialog ( msgText ) ;
                     dp->RefreshWin () ;
                  }
                  else if ( rPat->ext && !(*rPat->reStr == PERIOD) )
                  {
                     const char* msgText[] = 
                     {// 123456789x123456789x123456789x123456789x123456789x12
                        "  WARNING!  WARNING!  ",
                        " ",
                        "You have replaced the filename extension, but the",
                        "specified replacement text does not begin with a",
                        "period character ('.').",
                        " ",
                        "          Do you want to continue anyway?",
                        NULL
                     } ;
                     rPat->valid = this->DecisionDialog ( msgText ) ;
                     dp->RefreshWin () ;
                  }
                  else
                  {
                     rPat->valid = true ;    // hope for the best

                     #if LINUX_SPECIAL_CHARS != 0
                     //* Test the data from the sample-text field. If "special"*
                     //* characters detected, ask user whether to proceed.     *
                     rPat->valid = this->gnfSpecialCharsQuery ( gsMsg, dp, indexSAMP, tbFileName ) ;
                     if ( ! rPat->valid )
                     {  //* Move focus to a field containing text.*
                        // Programmer's Note: We assume that at at least one 
                        // textbox is active (because we have text data).
                        do
                        { icIndex = dp->PrevControl () ; }
                        while ( ic[icIndex].type != dctTEXTBOX ) ;
                     }
                     #endif   // LINUX_SPECIAL_CHARS
                  }

                  //* If user has entered valid pattern information, *
                  //* return to caller with the good news.           *
                  //* Else continue edit.                            *
                  if ( rPat->valid )
                  {
                     rPat->restartSequence () ; // restart sequence (if any)
                     status = batchRENAME ;
                     done = true ;
                  }
               }
               //* Cancel the rename operation *
               else if ( Info.ctrlIndex == indexCAN )
               {
                  status = abortRENAME ;
                  rPat->valid = false ;
                  done = true ;
               }
               else if ( Info.ctrlIndex == indexCLR )
               {  //* Set all Radio Buttons == false, and clear all Text Boxes.*
                  const wchar_t* emptyText = L"" ;
                  rPat->reinit() ;  // reset all pattern-object data fields
                  dp->ControlActive ( indexTBP, rPat->pr ) ;   // Prepend text
                  dp->SetTextboxText( indexTBP, emptyText ) ;
                  dp->SetRadiobuttonState ( indexRBP, false ) ;
                  dp->ControlActive ( indexTBI, rPat->in ) ;   // Insert text
                  dp->SetTextboxText( indexTBI, emptyText ) ;
                  dp->SetRadiobuttonState ( indexRBI, false ) ;
                  dp->ControlActive ( indexTBA, rPat->ap ) ;   // Append text
                  dp->SetTextboxText( indexTBA, emptyText ) ;
                  dp->SetRadiobuttonState ( indexRBA, false ) ;
                  dp->ControlActive ( indexTBR, rPat->base ) ; // Replace text
                  dp->SetTextboxText( indexTBR, emptyText ) ;
                  dp->SetRadiobuttonState ( indexRBR, false ) ;
                  dp->ControlActive ( indexSPW, rPat->base ) ; // Width spinner
                  dp->SetSpinnerValue ( indexSPW, iniWIDTH ) ;
                  dp->ControlActive ( indexSPF, rPat->base ) ; // Value spinner
                  dp->SetSpinnerValue ( indexSPF, iniFIRST ) ;
                  dp->NextControl () ; dp->NextControl () ; // move to 'OK'
               }
               else if ( Info.ctrlIndex == indexHELP )
               {
                  this->rsfCP_Help ( dLines, dCols, dulY, dulX ) ;
                  dp->RefreshWin () ;
                  dp->NextControl () ; // move to 'OK'
               }
            }  // if(dataMod)
         }  // if(dctPUSHBUTTON)
         //****************************************
         //* Focus is on a dctRADIOBUTTON control *
         //****************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* If no pending user input, edit control that currently has      *
            //* focus. Else, retrieve pending input for this control.          *
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;

            //* If button state has changed *
            if ( Info.dataMod != false )
            {
               switch ( Info.ctrlIndex )
               {
                  case indexRBP:          // 'Prepend' radio button
                     rPat->pr = Info.isSel ;
                     dp->ControlActive ( indexTBP, rPat->pr ) ;
                     break ;
                  case indexRBI:          // 'Insert' radio button
                     rPat->in = Info.isSel ;
                     dp->ControlActive ( indexTBI, rPat->in ) ;
                     break ;
                  case indexRBA:          // 'Append' radio button
                     rPat->ap = Info.isSel ;
                     dp->ControlActive ( indexTBA, rPat->ap ) ;
                     break ;
                  case indexRBR:          // 'Replace' radio button
                     rPat->base = rPat->ext = false ;
                     if ( Info.isSel != false )
                     {
                        if ( rPat->seqDigits > ZERO )
                           rPat->base = true ;
                        else
                           rPat->ext = true ;
                     }
                     dp->ControlActive ( indexTBR, (rPat->base | rPat->ext) ) ;
                     dp->ControlActive ( indexSPW, (rPat->base | rPat->ext) ) ;
                     dp->ControlActive ( indexSPF, (rPat->base | rPat->ext) ) ;
                     break ;
                  case indexRBD:          // 'Date' radio button
                     if ( Info.isSel )    // 'Date' button recently 'set'
                     {
                        //* Create a date string in the form: 'yyyymmddhhmmss' *
                        wchar_t dBuff[PEND_TBW] ;     // date/time stamp buffer
                        localTime ft ;                // system local time
                        this->GetLocalTime ( ft ) ;
                        swprintf ( dBuff, PEND_TBW, L"%04d%02d%02d%02d%02d%02d", 
                           ft.year, ft.month, ft.date, ft.hours, ft.minutes, ft.seconds ) ;

                        //* Insert date/time stamp into target field. Do this  *
                        //* automagically if possible, else prompt user to     *
                        //* clean up his/her act. (see note in method header)  *
                        //* First, scan the data fields for possible places to *
                        //* put the date/timestamp.                            *
                        cIndices dateTarget = indexCOUNT ;
                        wchar_t* rpatMember = NULL ;
                        short    emptyCount = ZERO ;
                        if ( rPat->pr != false && *rPat->prStr == NULLCHAR )
                           { dateTarget = indexTBP ; rpatMember = rPat->prStr ; ++emptyCount ; }
                        if ( rPat->in != false && *rPat->inStr == NULLCHAR )
                           { dateTarget = indexTBI ; rpatMember = rPat->inStr ; ++emptyCount ; }
                        if ( rPat->ap != false && *rPat->apStr == NULLCHAR )
                           { dateTarget = indexTBA ; rpatMember = rPat->apStr ; ++emptyCount ; }
                        if (  (rPat->base != false || rPat->ext != false)
                            && *rPat->reStr == NULLCHAR )
                           { dateTarget = indexTBR ; rpatMember = rPat->reStr ; ++emptyCount ; }
                        //* If target textbox control has been identified      *
                        if ( emptyCount == 1 )
                        {
                           dp->SetTextboxText ( dateTarget, dBuff ) ;   // write string to target
                           this->fmPtr->strnCpy ( rpatMember, dBuff, PEND_TBW ) ;
                           dp->SetRadiobuttonState ( indexRBD, false ) ;// reset 'Date' button
                        }
                        else
                        {
                           //* Prompt user to unambiguously specify date string*
                           //* target field.                                   *
                           const char* msgList[] = 
                           { //2345678901234567890123456789012345678901234567890123
                              "  Date/Timestamp Placement  ",
                              "Unable to determine which display field should",
                              "receive the date/time stamp message.",
                              " ",
                              "First, please set radio button for target field AND",
                              "clear any existing display text for that field.",
                              NULL
                           } ;
                           this->InfoDialog ( msgList ) ;
                           dp->RefreshWin () ;
                           dp->SetRadiobuttonState ( indexRBD, false ) ;
                        }
                     }
                     else
                        { /* User never manually resets 'Date' button */ }
                     break ;
               }
            }  // if(dataMod)
         }  // if(dctRADIOBUTTON)
         //*************************************
         //* Focus is on a dctTEXTBOX control  *
         //*************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            {  //* Modifications made to textbox control data *
               if ( Info.ctrlIndex == indexTBP )
                  dp->GetTextboxText ( indexTBP, rPat->prStr ) ;
               else if ( Info.ctrlIndex == indexTBI )
                  dp->GetTextboxText ( indexTBI, rPat->inStr ) ;
               else if ( Info.ctrlIndex == indexTBA )
                  dp->GetTextboxText ( indexTBA, rPat->apStr ) ;
               else if ( Info.ctrlIndex == indexTBR )
                  dp->GetTextboxText ( indexTBR, rPat->reStr ) ;
            }
         }
         //*************************************
         //* Focus is on a dctSPINNER control  *
         //*************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            {
               int newspVal ;
               dp->GetSpinnerValue ( Info.ctrlIndex, newspVal ) ;
               if ( Info.ctrlIndex == indexSPW )
               {
                  rPat->base = rPat->ext = false ;
                  rPat->seqDigits = newspVal ;
                  if ( rPat->seqDigits > ZERO )
                     rPat->base = true ;
                  else
                     rPat->ext = true ;
               }
               else if ( Info.ctrlIndex == indexSPF )
                  rPat->seqStart = newspVal ;
            }
         }

         //* Move focus to appropriate control *
         if ( done == false && Info.keyIn != ZERO && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   return status ;

}  //* End rsfConstructPattern() *

//***********************
//*    rsfCP_Help       *
//***********************
//******************************************************************************
//* Display a short help message to assist user in creating a batch rename     *
//* pattern. Called only by rsfConstructPattern() in response to user pressing *
//* the HELP button.                                                           *
//*                                                                            *
//* For convenience, if not efficiency, the help message is implemented as a   *
//* simple dialog with one dctSCROLLEXT control.                               *
//*                                                                            *
//*                                                                            *
//* Input  : dLines   = number of dialog lines                                 *
//*          dCols    = number of dialog columns                               *
//*          dulY     = upper left corner Y                                    *
//*          dulX     = upper left corner X                                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* NOTE: This dialog is the same size and in the same position as the         *
//* caller's dialog, so caller's dialog is hidden.                             *
//*                                                                            *
//* Programmer's Note: We will include the information here in the general     *
//* application TexInfo file, so this method may eventually become obsolete.   *
//*                                                                            *
//******************************************************************************

void FileDlg::rsfCP_Help ( short dLines, short dCols, short dulY, short dulX )
{
const short dispItems = 90 ;     // number of display strings
const char* dispData[dispItems] = 
{// xxxxxxxxxXxxxxxxxxxXxxxxxxxxxXxxxxxxxxxXxxxxxxxxxXxxxxxxxxxXxxxxxxxxxXxx0    // LINE
   "   Creating a pattern for renaming multiple files in a single operation.  ", // 000*
   " Navigate with scroll keys: Up,Down,PgUp,PgDown,Home,End. Escape to exit. ", // 001*
   "                                                                          ", // 002
   "Introduction:                                                             ", // 003*
   "   Batch rename or automatic multiple-file rename uses the so-called      ", // 004
   "\"Regular Expression\" pattern matching method. However, since few          ",// 005
   "people other than computer professionals (and not all of us) understand   ", // 006
   "RegExp, a simpler and more user-friendly set of options is provided.      ", // 007
   "                                                                          ", // 008
   "Navigating in the Dialog Window:                                          ", // 009*
   " Use Tab and Shift+Tab to move among the fields. To select a radio        ", // 010
   " button < o >  which will activate the desired field, move the highlight  ",
   " to that item and press the Space key. Press Space again to deselect.     ",
   " (Note that any text in a de-selected field will be ignored.)             ",
   " A radio button may also be selected via its defined 'hotkey' which is    ",
   " indicated by the underlined character in its description. To select a    ",
   " radio button via hotkey, use the Ctrl key + hotkey combination.          ",
   " Note that multiple fields may be activated for use within a batch        ",
   " rename operation, subject to certain limitions - see below.              ",
   " Adjusting spinner controls: by 1:Up/Down,        by 10:Shift+Up/Down,    ",
   "                             by 100:Ctrl+Up/Down, by 1000:Alt+Up/Down     ",
   " Insert/Overstrike input mode may be toggled for text box controls.       ",
   "                                                                          ",
   "Options available for renaming the list of selected files:                ", // 023*
   "*To prepend text at the beginning of the filename:                        ", // 024*
   "  a) select the 'Before' radio button control                             ",
   "  b) enter desired text in the field to the right of the radio button     ",
   "  -- Example: prepend a 'BAK-' to each selected filename:                 ",
   "       Old Name:  fname.fext                                              ",
   "       New Name:  BAK-fname.fext                                          ",
   "*To append text at the end of the filename:                               ", // 030*
   "  a) select the 'After' radio button control                              ",
   "  b) enter desired text in the field to the right of the radio button     ",
   "  -- Example: append a '-BAK' to each selected filename:                  ",
   "       Old Name:  fname.fext                                              ",
   "       New Name:  fname.fext-BAK                                          ",
   "*To insert text between the base filename and the filename extension:     ", // 036*
   "     (insertion is before sequence number, if any).                       ",
   "  a) select the 'Insert' radio button control                             ",
   "  b) enter desired text in the field to the right of the radio button     ",
   "  -- Example: insert a '-LoRes' into each selected filename:              ",
   "       Old Name:  fname.fext                                              ",
   "       New Name:  fname-LoRes.fext                                        ",
   "*To replace the existing base filename:                                   ", // 043*
   "  a) select the 'Replace' radio button control                            ",
   "  b) enter desired text in the field to the right of the radio button     ",
   "  c) specify the width for the sequence number using the 'Width' spinner  ",
   "     control. Range: 1 through 5. Be sure that width of field will        ",
   "     accomodate the expected range of sequence values.                    ",
   "  d) specify the initial value of the sequence using the 'InitVal'        ",
   "     spinner control. Range: 0 through 10000.                             ",
   "  -- Example: replace existing base filenames with Tahiti Trip nnn.jpg    ",
   "       Old Name:  fname.jpg                                               ",
   "       New Name:  Tahiti Trip 001.jpg                                     ",
   "  Note: If a specific filename has no filename extension (contains no     ",
   "        period ('.') character, not incl. first character), then the      ",
   "        entire filename will be replaced with the specified text.         ",
   "  Note: Sequence numbers are normally displayed with leading zeros to     ",
   "        fill the field width; however, to display sequence without        ",
   "        leading zeros (variable field width), specify Width = 1.          ",
   "*To replace the existing filename extension (including the period ('.'):  ", // 060*
   "  a) select the 'Replace' radio button control                            ",
   "  b) enter desired text in the field to the right of the radio button     ",
   "  c) specify a sequence width of zero (0) using the 'Width' spinner       ",
   "     control. (A width of zero disables the sequence number.)             ",
   "  Note: If a specific filename has no filename extension, (contains no    ",
   "        period ('.') character, not incl. first character), then the      ",
   "        specified replacement text will be appended to the filename.      ",
   "*To use the current local time as the text in one of the fields:          ", // 068*
   "  a) select the radio button control for the desired field                ",
   "  b) leave the text area for that field blank                             ",
   "  c) select the 'Date' radio button control                               ",
   "  The current date/time will be inserted into that field in the form:     ",
   "                         yyyymmddhhmmss.                                  ",
   "  Note: a warning message will be displayed when you select the 'Date'    ",
   "        button if you have not previously selected a target field, or if  ",
   "        the field already contains display text, or if multiple target    ",
   "        fields with no display text have been activated.                  ",
   "  Note: text in the target field may be adjusted after the date/time      ",
   "        has been inserted into the field. For instance, you may want to   ",
   "        append a dash ('-') \"20110127123154-\" or remove the time to       ",
   "        include only the date portion \"20110127\"                          ",
   "                           - - - - - - - - - -                            ",
   "When all desired fields have been selected and their text initialized,    ", // 083*
   " 'OK'     pushbutton to begin file rename operation.                      ",
   " 'CANCEL' pushbutton to discard selections and abort rename operation.    ",
   " 'CLEAR'  pushbutton to reset all input fields.                           ",
   " 'HELP'   pushbutton to display instructions (but you already knew that)  ",
   "                                                                          ",
   "                Press ESCAPE to exit this Help window.                    ",  // 089
} ;
//* Color attribute array for display strings *
// Programmer's Note: Because of the complexity of this dialog, it has been 
// exempted from following the application's Color Scheme.
attr_t dispAttr[dispItems] = 
{ 
   nc.blR,                                                                       // 000*
   nc.blR,                                                                       // 001*
   nc.bwB,                                                                       // 002
   nc.blB,                                                                       // 003*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                                       // 004-008
   nc.blB,                                                                       // 009*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,       // 010-018
   nc.bwB, nc.bwB, nc.bwB, nc.bwB,                                               // 019-022
   nc.blB,                                                                       // 023*
   nc.grB,                                                                       // 024*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                                       // 025-029
   nc.grB,                                                                       // 030*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                                       // 031-035
   nc.grB,                                                                       // 036*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                               // 037-042
   nc.grB,                                                                       // 043*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,       // 044-052
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                       // 053-059
   nc.grB,                                                                       // 060*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                       // 061-067
   nc.grB,                                                                       // 068*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,       // 069-077
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                                       // 078-082
   nc.blB,                                                                       // 083*
   nc.bwB, nc.bwB, nc.bwB, nc.bwB, nc.bwB,                                       // 084-088
   nc.reS,                                                                       // 089
} ;
attr_t   dColor = this->cs.sd,   // dialog background color
         cnColor = nc.blR,       // control non-focus color
         cfColor = nc.blR ;      // control focus color
InitCtrl ic =                    // control initialization structures
{
   //* dctSCROLLEXT control - - - - - - - - - - - - - - - - - - - - - - - - -  *
   dctSCROLLEXT,                 // type:      define a scrolling-data control
   rbtTYPES,                     // sbSubtype: (na)
   false,                        // rbSelect:  (n/a)
   ZERO,                         // ulY:       upper left corner in Y
   ZERO,                         // ulX:       upper left corner in X
   dLines,                       // lines:     control lines
   dCols,                        // cols:      control columns
   NULL,                         // dispText:  n/a
   cnColor,                      // nColor:    non-focus border color
   cfColor,                      // fColor:    focus border color
   tbPrint,                      // filter:    (n/a)
   "  Help for Batch Rename  ", // label:     
   ZERO,                         // labY:      offset from control's ulY
   ZERO,                         // labX       offset from control's ulX
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    can acquire input focus
   NULL                          // nextCtrl:  link in next structure
} ;
               
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dLines,         // number of display lines
                       dCols,          // number of display columns
                       dulY,           // Y offset from upper-left of terminal 
                       dulX,           // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       &ic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      ssetData sData( dispData, dispAttr, dispItems, 3, true ) ;
      dp->SetScrollextText ( ZERO, sData ) ;

      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         dp->EditScrollext ( Info ) ;
         if ( Info.keyIn == nckESC || Info.keyIn == nckENTER )
            done = true ;
      }     // while(!done)
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End rsfCP_Help() *

//***********************
//*     rsfAlert        *
//***********************
//******************************************************************************
//* Alert user of an error during rename of specified file.                    *
//* This could be the result of user providing a bad filename for the rename,  *
//* OR could be an access violation or system error during the rename process. *
//*                                                                            *
//* Input  : origName: pointer to name of file which was not renamed           *
//*          errMsg  : (optional, NULL by default) if not NULL, display message*
//*                    about the type of error encountered                     *
//*                                                                            *
//* Returns: 'true' if we are to continue processing remainder of list         *
//*          'false' if user aborted processing of file list                   *
//******************************************************************************

bool FileDlg::rsfAlert ( const char* origName, const char* errMsg )
{
bool  continueProcessing ;
const char* init_rnsQuery = "Do you want to process the remaining files in list?" ;
const char* init_rnsSpace = " " ;
char  rnsName[MAX_FNAME], rnsLine5[MAX_DIALOG_WIDTH], 
      rnsLine6[MAX_DIALOG_WIDTH], rnsLine7[MAX_DIALOG_WIDTH] ;
const char* renameSkip[] = 
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  ALERT!  ALERT!  ",
   " ",
   "        The following file was not renamed!",
   rnsName,
   " ",
   rnsLine5,
   rnsLine6,
   rnsLine7,
   NULL
} ;

   if ( errMsg != NULL )               // include error message (if any)
   {
      this->fmPtr->strnCpy ( rnsLine5, errMsg, MAX_DIALOG_WIDTH ) ;
      this->fmPtr->strnCpy ( rnsLine6, init_rnsSpace, MAX_DIALOG_WIDTH ) ;
      this->fmPtr->strnCpy ( rnsLine7, init_rnsQuery, MAX_DIALOG_WIDTH ) ;
   }
   else                                // no error message
   {
      this->fmPtr->strnCpy ( rnsLine5, init_rnsQuery, MAX_DIALOG_WIDTH ) ;
      rnsLine6[ZERO] = rnsLine7[ZERO] = NULLCHAR ;
      *rnsLine6 = NULLCHAR ;
   }
   gString gs( origName ) ;
   short nameWidth = gs.gscols(),
         nameOffset = ((INFODLG_WIDTH - 4) - nameWidth) / 2 ;
   if ( nameOffset > ZERO )
      gs.shiftCols( nameOffset ) ;
   gs.copy( rnsName, gs.gschars() ) ;
   continueProcessing = this->DecisionDialog ( renameSkip ) ;

   return   continueProcessing ;

}  //* End rsfAlert() *

//***********************
//*                     *
//***********************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

