//********************************************************************************
//* File       : FileDlgPrompt.cpp                                               *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2024 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 11-Apr-2024                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module of the FileDlg class contains user prompts, that is, sub-dialog  *
//* windows that prompt the user for information or ask for decisions.           *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

const char* fmtypeString[] =  // Display strings corresponding to enum fmFType
{ "DIR ", "REG ", "LINK", "CDEV", "BDEV", "FIFO", "SOCK", "UNKN", "TYPE" } ;

const char* fmtypeName[] =  // Display strings corresponding to enum fmFType
{ 
   "'Directory'", "'Regular'", "'Symbolic Link'", "'Character Device'", 
   "'Block Device'", "'FIFO'", "'Socket'", "'Unknown Type'", "'Invalid Type'",
} ;

//* If user is trying to copy files to a read-only filesystem - CD-ROM, etc.   *
const char* readonlyFileSys[] =
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  ALERT  ",
   " ",
   "        Target is a Read-Only file system.",
   NULL,
} ;

//* Alert user who is trying to paste files into clipboard source directory    *
const char* notOntoSelf[] =
{
   "  ALERT  ",
   " ",
   "       A FILE MAY NOT BE COPIED ONTO ITSELF!",
   " ",
   "  To make a copy of a file in the same directory:",
   "         use 'Paste-Special' with Rename.",
   " ",
   "      (See FileMangler Help for more details.)",
   NULL
} ;

//* Alert user that attempt to override write protection has failed.           *
const char* cannotOverrideWP[] =
{
   "  ALERT  ",
   " ",
   "   UNABLE TO OVERRIDE WRITE PROTECTION ON FILE!",
   NULL
} ;

//* Alert user that a symbolic link file may not overwrite a non-link file.    *
const char* noLinkOverwrite[] =
{
   "  ALERT  ",
   " ",
   " A Symbolic Link may not overwrite a non-Link file!",
   NULL
} ;


//* Alert user that a symbolic link file may not overwrite a non-link file.    *
const char* differentFileTypes[] =
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  ALERT  ",
   " ",
   "      Source file is of a different file type",
   "           than the existing target file.",
   "        Overwrite the existing target file?",
   NULL
} ;

//* Alert user that a 'special' file may not be deleted or overwritten.        *
const char* cannotDeleteSpecial[] =
{
   "  ALERT  ",
   " ",
   "     Deleting or overwriting a 'special' file",
   "    (Character Device, Block Device or Socket)",
   "                 is not allowed!",
   NULL
} ;

//* Alert user that a 'FIFO file may not be overwritten by a non-FIFO file.    *
const char* cannotOverwriteFifo[] =
{
   "  ALERT  ",
   " ",
   "    Overwriting a FIFO file with a non-FIFO file",
   "                 is not allowed!",
   NULL
} ;

//* Alert user that a directory file may not be overwritten by a non-directory.*
const char* cannotOverwriteDirectory[] =
{
   "  ALERT  ",
   " ",
   "        Overwriting a Directory file with a",
   "        non-directory file is not allowed!",
   NULL
} ;

//* Warn user that dinking with a 'socket' file could be hazardous to system   *
const char* warnSocketMod[] =
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  WARNING!  WARNING!  ",
   " ",
   "      You are about to modify a SOCKET file!",
   " ",
   "Sockets are system files used for inter-process",
   "communications, usually over a network.",
   "Modifying this file may cause system instability",
   "or loss of functionality. Do you want to proceed?",
   NULL
} ;

//* Warn user that dinking with a 'block device' or 'character device' file    *
//* could be hazardous to system health.                                       *
const char* warnBdevCdevMod[] =
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  WARNING!  WARNING!  ",
   "              You are about to modify",
   "      a BLOCK device or CHARACTER device file!",
   " ",
   "Block and Character device files are system files",
   "used to communicate with external hardware.",
   "Modifying this file may cause system instability",
   "or loss of functionality. Do you want to proceed?",
   NULL
} ;

//* Warn user about dinking with a filetype that we do not support.            *
const char* warnUnsuppMod[] =
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  WARNING!  WARNING!  ",
   "      You are about to modify a file of a type",
   "     that FileMangler does not directly support!",
   " ",
   "Modifying the file may have unintended consequences!",
   "              Do you want to proceed?",
   NULL
} ;

//* If user is requesting functionality that is not yet implemented.           *
const char* fdNotImplemented[] =
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  ALERT  ",
   " ",
   " Requested operation has not yet been implemented.",
   NULL,
} ;


//*************************
//*     GetErrorCode      *
//*************************
//******************************************************************************
//* Returns the code and description for the most recent FileDlg error or      *
//* file system i.e. FMgr-class error.                                         *
//* Note that the error information returned is not necessarily for the most   *
//* recent activity, but for the most recent activity that generated an error. *
//*                                                                            *
//* Input  : errorCode class object (by reference, initial contents ignored)   *
//*           (all fields initialized on return)                               *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short FileDlg::GetErrorCode ( errorCode& eCode )
{
//* FileDlg Class Error Numbers                       *
//* Keep this array synchronized with enum fdErrCode. *
static const char* ErrorCodes[] = 
{
   "No Error                           ",       // ecNOERROR
   "Memory Allocation Error            ",       // ecMEMALLOC
   "Possible access violation          ",       // ecPACCV
   "Unable to retrieve file info       ",       // ecFINFO
   "Cannot copy a file onto itself     ",       // ecFILESELF
   "Can't write target,source unchanged",       // ecNOWRITE
   "Insufficient permission for file   ",       // ecPERMISSION
   "No read access for source directory",       // ecDIRNOREAD
   "Incorrect file type for operation  ",       // ecFTYPE
   "Directory or file is in use        ",       // ecINUSE
   "Operation unsupported for file type",       // ecUNSUPP
   "Existing target file is protected  ",       // ecTARGPROTECT
   "File of target name already exists ",       // ecTARGEXISTS
   "Source write protected, not deleted",       // ecSRCPROTECT
   "Unable to create symbolic link     ",       // ecLINK
   "Unable to create 'hard' link       ",       // ecHARDLINK
   "Source directory set as read-only  ",       // ecREADONLY_S
   "Target directory set as read-only  ",       // ecREADONLY_T
   "Unable to read target directory    ",       // ecTARGDIR
   "Cannot overwrite file with sym link",       // ecLINKOVER
   "Recursive copy operation failed    ",       // ecRECURSECOPY
   "Move-to-trashcan operation disabled",       // ecNOTRASHCAN
   "Application error, please report it",       // ecAPPERROR
   "Req. operation not yet implemented ",       // ecNOTIMPL
   "Unknown error                      ",       // ecUNKERROR
   " ",                                         // ecERRORCODES
} ;

      eCode.fdErrorCode  = this->recentError ;
      if ( this->recentError >= FIRST_ERRORCODE && this->recentError < ecERRCODES )
         eCode.fdErrorDesc = ErrorCodes[this->recentError - FIRST_ERRORCODE + 1] ;
      else if ( this->recentError == ecNOERROR )
         eCode.fdErrorDesc = ErrorCodes[ecNOERROR] ;
      else
         eCode.fdErrorDesc  = ErrorCodes[ecUNKERROR - FIRST_ERRORCODE + 1] ; 
      eCode.sysErrorCode = this->recentErrno ;
      eCode.sysErrorDesc = this->fmPtr->GetErrnoMessage ( this->recentErrno ) ;
   return OK ;

}  //* End GetErrorCode() *

//*************************
//*   CreateDirectory     *
//*************************
//******************************************************************************
//* Get name for new sub-directory from user, and                              *
//* create the new sub-directory in the currently-displayed directory.         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns:  OK if directory created successfully,                            *
//*           ERR if operation aborted                                         *
//*           or 'errno' value if operation failed                             *
//******************************************************************************
//* NOTE: Filename is truncated to MAX_FNAME if necessary.                     *
//******************************************************************************

short FileDlg::CreateDirectory ( void )
{
const short dialogLines = 8,
            dnameTB     = ZERO,
            okPush      = 1,
            dummyPush   = 2,
            cancelPush  = 3,
            controlsDEFINED = 4 ;
short    dialogCols = this->fMaxX - 4,
         ctrY = this->fulY + this->fMaxY/2,
         ctrX = this->fulX + this->fMaxX/2,
         ulY  = ctrY - (dialogLines / 2 + 4),
         ulX  = ctrX - dialogCols / 2 ;
attr_t   dColor = this->cs.sd ;     // dialog background color
short    status  = ERR ;            // return value

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'dName' Text Box - - - - - - - - - - - - - - - - - - - - -   dnameTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dialogCols - 4),        // cols:      control columns
      "",                           // dispText:  initially empty
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      "Directory Name: ",           // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -   okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 2 - 11),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   OK   ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* dummy (invisible) pushbutton  - - - - - - - - - - - - - -  dummyPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      control columns
      " ",                          // dispText:  
      dColor,                       // nColor:    non-focus color
      dColor,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[3],                       // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - -   cancelPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 2 + 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Create New Directory  ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Print dialog window's static text *
      dp->WriteString ( 4, dialogCols/2-20, 
                           "('Insert' key toggles Insert/Overstrike)", dColor ) ;

      //* Enable audible alert for invalid input *
      dp->TextboxAlert ( dnameTB, true ) ;

      dp->RefreshWin () ;                 // make the text visible

      gString  newName ;                  // new directory-name string
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               //* Accept edits *
               if ( Info.ctrlIndex == okPush )
               {
                  //* If edit results in something other than a NULL string *
                  if ( (dp->GetTextboxText ( dnameTB, newName )) > 1 )
                  {
                     //* Truncate the name to the POSIX max length.            *
                     //* (Few users would be sufficiently caffinated to create)*
                     //* (a filename that long, so we do it the easy way.)     *
                     newName.limitChars( MAX_FNAME - 2 ) ;

                     #if LINUX_SPECIAL_CHARS != 0
                     if ( !(this->gnfSpecialCharsQuery ( newName, dp, dnameTB, tbFileName )) )
                        icIndex = dp->NextControl () ;
                     else
                     #endif   // LINUX_SPECIAL_CHARS
                        status = OK ;
                     if ( status == OK )
                        done = true ;
                  }
                  else
                  {
                     dp->ClearLine ( 3, false ) ;
                     dp->WriteString ( 3, (dialogCols / 2 - 21), 
                        " No directory name specified! - Try again. ", 
                        dColor | ncbATTR, true ) ;
                     dp->NextControl () ;
                  }
               }
               //* Cancel the operation *
               else if ( Info.ctrlIndex == cancelPush )
               {
                  status = ERR ;
                  done = true ;
               }
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            dp->ClearLine ( 3, false ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
      
      //* If new directory name specified, create it now. *
      if ( status == OK )
      {
         if ( (status = this->CreateDirectory ( newName )) != OK )
         {
            //* Remove input focus from 'OK' pushbutton *
            dp->ControlActive ( dummyPush, true ) ;
            icIndex = dp->NextControl () ;
            dp->ClearLine ( 3, false ) ;
            dp->ClearLine ( 4, false ) ;
            gString oBuff ;
            oBuff.compose( L" ERROR! - Unable to create directory.  errno: %hd ", &status ) ;
            dp->WriteString ( 3, (dialogCols / 2 - 25), oBuff, this->cs.tn | ncbATTR ) ;
            dp->WriteString ( 4, (dialogCols / 2 - 10), " Press Any Key . . . ", 
                                                            this->cs.pf, true ) ;
            nckPause();          // wait for user's synapses to fire
         }
      }
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;
   
   //* If the new directory was created, update the file-display list *
   dtbmData    msgData ;                     // message display
   if ( status == OK )
   {
      this->RefreshCurrDir () ;
      msgData = " New directory created." ;
   }
   else
      msgData = " Operation cancelled." ;
   this->dPtr->DisplayTextboxMessage ( this->mIndex, msgData ) ;

   return status ;

}  //* End CreateDirectory() *

//*************************
//*   ConfirmOverwrite    *
//*************************
//******************************************************************************
//* Ask user to confirm that target file should be overwritten by source       *
//* file. Call this method if cfgOptions.overWrite == owCONFIRM_ALL or         *
//* owCONFIRM_NEWER, _OR_ if existing target is write-protected.               *
//*                                                                            *
//* Caller has verified that both source and target files exist.               *
//*                                                                            *
//* Within the dialog, files are identified by Name, Modification Date and     *
//* File Size.                                                                 *
//*                                                                            *
//* Input  : srcStats : information on source file                             *
//*          trgStats : information on target file                             *
//*                                                                            *
//* Returns: member of enum corReturn                                          *
//******************************************************************************

corReturn FileDlg::ConfirmOverwrite ( const tnFName* srcStats, const tnFName* trgStats )
{
   //* Compare source and target modification dates.                    *
   //* ZERO == equal dates, > ZERO == source newer, < ZERO target newer *
   short modTimeComp = ZERO ;
   if ( srcStats->rawStats.st_mtime > trgStats->rawStats.st_mtime )
      ++modTimeComp ;
   else if ( srcStats->rawStats.st_mtime != trgStats->rawStats.st_mtime )
      --modTimeComp ;

   corReturn overWrite = this->cowPrompt ( srcStats, trgStats, modTimeComp ) ;

   return overWrite ;

}  //* End ConfirmOverwrite() *

//*************************
//*      cowPrompt        *
//*************************
//******************************************************************************
//* Open a dialog to prompt user for a decision whether to overwrite an        *
//* existing target file.                                                      *
//*                                                                            *
//* This method is called ONLY by ConfirmOverwrite().                          *
//*                                                                            *
//* Input  : srcStats: source file stats                                       *
//*          trgStats: target file stats                                       *
//*          srcNewer: comparison results of file timestamp                    *
//*                     >ZERO: source timestamp newer than target timestamp    *
//*                    ==ZERO: source timestamp == target timestamp            *
//*                     <ZERO: target timestamp newer than source timestamp    *
//*                                                                            *
//* Returns: member of enum corReturn                                          *
//******************************************************************************

corReturn FileDlg::cowPrompt ( const tnFName* srcStats, 
                               const tnFName* trgStats, short srcNewer )
{
const short replacePush = 0,                 // indices of pushbutton controls
            skipPush = 1, 
            replaceallPush = 2,
            skipallPush = 3, 
            abortPush = 4 ;
const short dialogLines = 12,                // dialog lines
            dialogCols  = INFODLG_WIDTH,     // dialog columns
            controlsDEFINED = 5 ;            // number of controls in dialog

short    ctrY = this->fulY + this->fMaxY/2,
         ctrX = this->fulX + this->fMaxX/2,
         ulY  = ctrY - dialogLines / 2,
         ulX  = ctrX - dialogCols / 2 ;
attr_t   dColor  = this->cs.sd,        // window background color
         pnColor = this->cs.pn,        // pushbutton non-focus color
         pfColor = this->cs.pf ;       // pushbutton focus color
corReturn overWrite = corProcessOnce ; // return value

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'Replace' pushbutton - - - - - - - - - - - - - - - - -   replacePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      9,                            // cols:      control columns
      " ^REPLACE ",                 // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'Skip' pushbutton - - - - - - - - - - - - - - - - - - - -   skipPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      12,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  ^SKIP  ",                  // dispText:
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* 'Replace All' pushbutton - - - - - - - - - - - - - -  replaceallPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      21,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      " REPLACE ^ALL ",             // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3],                       // nextCtrl:  link in next structure
   },
   {  //* 'Skip All' pushbutton - - - - - - - - - - - - - - - -    skipallPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      35,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      " S^KIP ALL ",             // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[4],                       // nextCtrl:  link in next structure
   },
   {  //* 'Abort' pushbutton - - - - - - - - - - - - - - - - - - -   abortPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      46,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      " A^BORT ",                   // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      winPos   wp( 2, 2 ) ;         // cursor position
      gString  outBuff,             // work buffers
               mBuff,               // date/time
               fBuff ;              // file size
      dp->SetDialogTitle ( "  Confirm File Overwrite  ", this->cs.em ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "Replace the current ", dColor | ncbATTR ) ;
      if ( trgStats->writeAcc == false )
      {
         wp = dp->WriteString ( wp.ypos, wp.xpos, "WRITE-PROTECTED", 
                                dColor | ncbATTR | ncuATTR ) ;
         ++wp.xpos ;
      }
      dp->WriteString ( wp.ypos, wp.xpos, "file:", dColor | ncbATTR ) ;
      ++wp.ypos ;
      wp.xpos = 3 ;

      outBuff.compose( L" %s ", trgStats->fName ) ;
      dp->WriteString ( wp.ypos, wp.xpos, outBuff, (dColor & !ncrATTR) ) ;
      ++wp.ypos ;
      fBuff.formatInt( trgStats->fBytes, 9 ) ; 
      this->FormatDateString ( trgStats->modTime, mBuff ) ;
      outBuff.compose( L"Modified: %S - %S bytes", mBuff.gstr(), fBuff.gstr() ) ;
      dp->WriteString ( wp.ypos, wp.xpos, outBuff, dColor ) ;
      wp.ypos += 2 ;

      fBuff.formatInt( srcStats->fBytes, 9 ) ; 
      this->FormatDateString ( srcStats->modTime, mBuff ) ;
      outBuff.compose( L"Modified: %S - %S bytes", mBuff.gstr(), fBuff.gstr() ) ;
      dp->WriteString ( wp.ypos, wp.xpos, outBuff, dColor ) ;
      ++wp.ypos ;
      wp.xpos = 2 ;

      wp = dp->WriteString ( wp.ypos, wp.xpos, "With this ", dColor | ncbATTR ) ;
      if ( srcNewer > ZERO && (trgStats->fType != fmDIR_TYPE) ) // source file is newer
         wp = dp->WriteString ( wp.ypos, wp.xpos, " newer ", dColor | ncbATTR ) ;
      else if ( srcNewer < ZERO && (trgStats->fType != fmDIR_TYPE) ) // source file is older
         wp = dp->WriteString ( wp.ypos, wp.xpos, " older ", pfColor ) ;
      else
         --wp.xpos ;
      dp->WriteString ( wp.ypos, wp.xpos, " one?", dColor | ncbATTR ) ;

      //* Place focus on appropriate pushbutton: if target's mod date *
      //* >= source's mod date, move focus to "SKIP" pushbutton.      *
      //*        (this is an exercise in user-friendliness)           *
      if ( srcNewer <= ZERO )
         dp->NextControl () ;

      dp->RefreshWin () ;              // make the text visible

      //******************
      //* Get user input *
      //******************
      uiInfo   Info ;                  // user interface data returned here
      bool     done = false ;          // loop control
      while ( ! done )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;

         if ( Info.dataMod != false )
         {
            switch ( Info.ctrlIndex )
            {
               case replacePush:    // overwrite only this file
                  if ( trgStats->writeAcc == false )
                     overWrite = corProcessOnceWP ;
                  else
                     overWrite = corProcessOnce ;
                  break ;
               case skipPush:       // do not overwrite this file
                  overWrite = corSkipOnce ;
                  break ;
               case replaceallPush: // overwrite all remaining targets in the list
                  overWrite = corProcessAll ;
                  break ;
               case skipallPush:    // do not overwrite any existing targets
                  overWrite = corSkipAll ;
                  break ;
               case abortPush:      // abort the operation in progress
                  overWrite = corAbort ;
                  break ;
            }
            //* Selection complete *
            done = true ;
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else
               dp->NextControl () ;
         }
      }     // while(!done)
   }
   else           // dialog did not open (probably memory allocation error)
   {
      this->DisplayStatus ( ecMEMALLOC ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   //* If progress bar is active, refresh it also. *
   this->wip.pbarRefresh() ;     // refresh display of progress bar

   return overWrite ;

}  //* End of cowPrompt() *

#if 0    // OBSOLETE - NOT CURRENTLY USED - MAY BE UPDATED IN A FUTURE RELEASE
//*************************
//*    CopyLinkPrompt     *
//*************************
//******************************************************************************
//* For copying symbolic links, ask user whether to copy the link, itself,     *
//* or the link target. Refer to the ConfigOptions cfgOptions.symLinkCopy      *
//* parameter.                                                                 *
//* This method should be called only when:                                    *
//*               cfgOptions.symLinkCopy == slcUSER_PROMPT.                    *
//*                                                                            *
//* Input  : srcName  : name of symbolic link to be copied                     *
//*          ltargSpec: full path/filename specification of link target        *
//*                                                                            *
//* Return Value : selected member of enum slcOptions                          *
//*                (slcCOPY_LINK, slcCOPY_TARGET_L, slcCOPY_TARGET_T)          *
//******************************************************************************

slcOptions CopyLinkPrompt ( const char* srcName, const char* ltargSpec ) ;
slcOptions FileDlg::CopyLinkPrompt ( const char* srcName, const char* ltargSpec )
{
   const char  ControlText[4][MAX_LABEL_CHARS] = 
               {
                  { "  ^OK  " },
                  { "Copy ^Symbolic Link itself" },
                  { "Copy ^Link Target, using name of link" },
                  { "Copy Link ^Target, using name of target" },
               } ;
   const short dialogRows = 15,              // dialog lines
               dialogCols  = INFODLG_WIDTH ; // dialog columns
   enum clpControls : short { pbOK = ZERO, rbSL, rbCTL, rbCTT, clpCONTROLS } ;
   short    ctrY = this->fulY + this->fMaxY/2,
            ctrX = this->fulX + this->fMaxX/2,
            ulY  = ctrY - dialogRows / 2,
            ulX  = ctrX - dialogCols / 2 ;
   attr_t   dColor  = this->cs.sd,           // window background color
            pnColor = this->cs.pn,           // pushbutton non-focus color
            pfColor = this->cs.pf,           // pushbutton focus color
            rnColor = dColor,                // radiobutton non-focus color
            rfColor = pfColor ;              // radiobutton focus color
   slcOptions selection = slcCOPY_LINK ;     // default selection

   //*******************************
   //* Controls for dialog window  *
   //*******************************
   InitCtrl ic[clpCONTROLS] =
   {
      {  //* 'OK' pushbutton     - - - - - - - - - - - - - - - - -        pbOK *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogRows - 2),        // ulY:       upper left corner in Y
         short(dialogCols / 2 - 2),    // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         (char*)ControlText[0],        // dispText:  
         pnColor,                      // nColor:    non-focus color
         pfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[rbSL]                     // nextCtrl:  link in next structure
      },
      {  //* 'COPY Link' radiobutton   - - - - - - - - - - - - - -        rbSL *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // rbSubtype: standard, 5-wide
         false,                        // rbSelect:  initial value
         short(dialogRows - 6),        // ulY:       upper left corner in Y
         2,                            // ulX:
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         ControlText[1],               // label:     
         ZERO,                         // labY:      
         7,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[rbCTL],                   // nextCtrl:  link in next structure
      },
      {  //* 'COPY Target-L' radiobutton - - - - - - - - - - - - -       rbCTL *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // rbSubtype: standard, 5-wide
         false,                        // rbSelect:  initial value
         short(ic[rbSL].ulY + 1),      // ulY:       upper left corner in Y
         ic[rbSL].ulX,                 // ulX:
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         ControlText[2],               // label:     
         ZERO,                         // labY:      
         7,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[rbCTT],                   // nextCtrl:  link in next structure
      },
      {  //* 'COPY Target-T' radiobutton - - - - - - - - - - - - -       rbCTT *
         dctRADIOBUTTON,               // type:      
         rbtS5a,                       // rbSubtype: standard, 5-wide
         true,                         // rbSelect:  initial value
         short(ic[rbCTL].ulY + 1),     // ulY:       upper left corner in Y
         ic[rbCTL].ulX,                // ulX:
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rnColor,                      // nColor:    non-focus color
         rfColor,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         ControlText[3],               // label:     
         ZERO,                         // labY:      
         7,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogRows,     // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Give dialog a title *
      dp->SetDialogTitle ( "  Symbolic Link Copy Option  ", (dColor | ncbATTR) ) ;
      
      LineDef  lDefH(ncltHORIZ, ncltSINGLE, 5, ZERO, dialogCols, dColor) ;
      dp->DrawLine ( lDefH ) ;

      //* Set radio buttons as an exclusive-OR group *
      //* (only one can be selected at any moment).  *
      short XorGroup[] = { rbSL, rbCTL, rbCTT, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Write the static text *
      const char* sText[] = 
      {
         "This source file: ",
         "is a symbolic link (shortcut file) pointing to:",
         "For this file and files like it, how do you want to",
         "handle the copy?",
      } ;
      gString  pgs ;
      winPos wp( 1, 2 ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, sText[0], dColor ) ;
      pgs.compose( L" %s ", srcName ) ;
      this->TrimPathString ( pgs, (dialogCols - 6) ) ;
      dp->WriteString ( wp.ypos++, wp.xpos+2, pgs, (dColor & !ncrATTR) ) ;

      dp->WriteString ( wp.ypos++, wp.xpos, sText[1], dColor ) ;
      pgs.compose( L" %s ", ltargSpec ) ;
      this->TrimPathString ( pgs, (dialogCols - 4) ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, pgs, (dColor & !ncrATTR) ) ;
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, sText[2], dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, sText[3], dColor ) ;

      dp->RefreshWin () ;                 // make the text visible

      short    icIndex = ZERO ;           // control with input focus
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.ctrlIndex == pbOK && Info.dataMod != false )
            {
               //* Selection complete *
               done = true ;
            }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
            //* Translate selection into a member of enum 
            switch ( Info.selMember )
            {
               case rbSL:                 // Copy Symbolic Link
                  selection = slcCOPY_LINK ;
                  break ;
               case rbCTL:                // Copy Target using name of link file
                  selection = slcCOPY_TARGET_L ;
                  break ;
               case rbCTT:                // Copy Target using name of target file
                  selection = slcCOPY_TARGET_T ;
                  break ;
            }
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else                             // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return selection ;

}  //* End CopyLinkPrompt() *
#endif   // OBSOLETE - NOT CURRENTLY USED - MAY BE UPDATED IN A FUTURE RELEASE

//*************************
//*    DisplayUserInfo    *
//*************************
//******************************************************************************
//* Open a dialog and display user's information. Primarily for debugging,     *
//* but may provide mainline code with useful functionality.                   *
//*                                                                            *
//* Note that if the supplimentary GID list is long, we don't display all of   *
//* it, only the first 15 GIDs. Also, because the actual password strings      *
//* are probably not returned by the system calls, we don't display the        *
//* dummy password strings (it would be too confusing to the user).            *
//*                                                                            *
//* Input  :none                                                               *
//*                                                                            *
//* Returns: 'true' if successful,                                             *
//*          'false' if data unavailable or memory allocation error            *
//******************************************************************************

bool FileDlg::DisplayUserInfo ( void )
{
const short dialogLines = 12,
            dialogCols  = 74,
//            okPush      = 1,
            controlsDEFINED = 1 ;
attr_t dColor = this->cs.sd,           // dialog color
       vColor = this->cs.sb ;          // data-value color
short  ulY = this->fulY + 3,           // dialog position
       ulX = this->dulX + this->dCols / 2 - dialogCols / 2 ;

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -   okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      "  OK  ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( " USER INFORMATION ", this->cs.em ) ;
      gString  oBuff ;
      winPos wp( 1, 2 ) ;
      const char* const fieldHead = "   User ID: "
                                     "                             "
                                     "Member of Additional Groups:\n"
                                    " User Acct: \n"
                                    " User Name: \n"
                                    "  Group ID: \n"
                                    "Group Name: \n"
                                    "Shell Prog: \n"
                                    "  Home Dir: " ;
      dp->WriteParagraph ( wp, fieldHead, dColor ) ;
      wp.xpos += 12 ;
      gString homePath( this->userInfo.homeDir ) ;
      this->TrimPathString ( homePath, (dialogCols - 16) ) ;
      oBuff.compose( L"%u\n"
                      "%s\n"
                      "%s\n"
                      "%u\n"
                      "%s\n"
                      "%s\n"
                      "%S",
                     &this->userInfo.userID,
                     this->userInfo.userName,
                     this->userInfo.realName,
                     &this->userInfo.grpID,
                     this->userInfo.grpName,
                     this->userInfo.shellProg,
                     homePath.gstr() ) ;
      dp->WriteParagraph ( wp, oBuff, vColor ) ;

      wp = { 2, 45 } ;
      if ( this->userInfo.suppGroups > ZERO )
      {
         for ( short i = ZERO ; i < this->userInfo.suppGroups
                             && i < uiMAX_sGID ; i++ )
         {
            oBuff.compose( L"%u", &this->userInfo.suppGID[i] ) ;
            dp->WriteString ( wp.ypos++, wp.xpos, oBuff, vColor ) ;
            if ( i == 4 )
            {
               wp.ypos = 2 ;
               wp.xpos += 8 ;
            }
            else if ( i == 9 )
            {
               wp.ypos = 2 ;
               wp.xpos += 8 ;
            }
            else if ( i >= 13 )
            {
               dp->WriteString ( wp.ypos++, wp.xpos-2, "(partial list)", dColor ) ;
               break ;
            }
         }
      }
      else
         dp->WriteString ( wp.ypos++, wp.xpos, "none specified", vColor ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      do
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;
      }
      while ( ! Info.dataMod && Info.keyIn != nckESC ) ;
   }
   else           // unable to open window (likely memory-allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return userInfo.goodData ;

}  //* End DisplayUserInfo() *

//*************************
//* ConfirmPasteToTarget  *
//*************************
//******************************************************************************
//* Available space on target file system may be insufficient for the pending  *
//* paste-from-clipboard operation. Prompt user for permission to continue.    *
//*                                                                            *
//* Input  : bNeed : approximate space needed on target file system (bytes)    *
//*          bAvail: approximate space available on target file system (bytes) *
//*                                                                            *
//* Returns: true if user has given permission to proceed with operation       *
//*          else, false == abort the operation                                *
//******************************************************************************

bool FileDlg::ConfirmPasteToTarget ( UINT64 bNeed, UINT64 bAvail )
{
const char*  Msgs[] = { "A paste-from-clipboard operation is pending,",
                        "however space on the target file system is limited.",
                        "Approximate space required : ",
                        "Approximate space available: ",
                        " bytes.",
                        "Do you want to proceed?",
                        "however the amount of free space on the target ",
                        "file system cannot be determined.",
                      } ;
const short proceedPush = 0,                 // indices of pushbutton controls
            abortPush   = 1,
            dialogLines = 11,                // dialog lines
            dialogCols  = INFODLG_WIDTH,     // dialog columns
            controlsDEFINED = 2 ;            // number of controls in dialog
short    ctrY = this->fulY + this->fMaxY / 2,
         ctrX = this->fulX + this->fMaxX / 2,
         ulY  = ctrY - dialogLines / 2 - 4,
         ulX  = ctrX - dialogCols / 2 ;
attr_t   dColor = this->cs.sd,               // window background color
         pnColor = this->cs.pn,              // pushbutton non-focus color
         pfColor = this->cs.pf ;             // pushbutton focus color
bool     confirmed = false ;                 // return value

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'Proceed' pushbutton - - - - - - - - - - - - - - - - -   proceedPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 3 - 10),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      17,                           // cols:      control columns
      "     PROCEED     ",          // dispText:
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'Abort' pushbutton - - - - - - - - - - - - - - - - - -     abortPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 3 * 2 - 7),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      17,                           // cols:      control columns
      " ABORT OPERATION ",          // dispText:
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Confirm Paste From Clipboard  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor | ncbATTR,// border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      gString  outBuff, numBuff ;
      winPos   wp( 1, 2 ) ;
      wp = dp->WriteString ( wp.ypos, 2, Msgs[0], dColor ) ;
      if ( bNeed == ZERO && bAvail == ZERO )
      {
         wp = dp->WriteString ( ++wp.ypos, 2, Msgs[6], dColor ) ;
         wp = dp->WriteString ( ++wp.ypos, 2, Msgs[7], dColor ) ;
      }
      else
      {
         wp = dp->WriteString ( ++wp.ypos, 2, Msgs[1], dColor ) ;
         wp.ypos += 2 ;
         wp = dp->WriteString ( wp.ypos, 2, Msgs[2], dColor ) ;
         numBuff.formatInt( bNeed, 7 ) ;
         outBuff.compose( L" %S ", numBuff.gstr() ) ;
         wp = dp->WriteString ( wp.ypos, wp.xpos, outBuff, this->cs.tn | ncbATTR ) ;
         wp = dp->WriteString ( wp.ypos, wp.xpos, Msgs[4], dColor ) ;
         wp = dp->WriteString ( ++wp.ypos, 2, Msgs[3], dColor ) ;
         numBuff.formatInt( bAvail, 7 ) ;
         outBuff.compose( L" %S ", numBuff.gstr() ) ;
         wp = dp->WriteString ( wp.ypos, wp.xpos, outBuff, this->cs.tn | ncbATTR ) ;
         wp = dp->WriteString ( wp.ypos, wp.xpos, Msgs[4], dColor ) ;
      }
      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, 15, Msgs[5], dColor ) ;

      dp->RefreshWin () ;                 // make the text visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == proceedPush )
                  confirmed = true ;
               else if ( Info.ctrlIndex == abortPush )
                  confirmed = false ;
               done = true ;
            }
            else if ( Info.keyIn == nckESC )
            {
               //* User is illiterate or is having a panic attack *
               confirmed = false ;    // abort
               done = true ;
            }
         }
         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ;
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else              // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

    //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

  return confirmed ;

}  //* End ConfirmPasteToTarget() *

//*************************
//*   ModifySortOptions   *
//*************************
//******************************************************************************
//* Interactively modify file-sorting options.                                 *
//*         copt.sortOption: file sort order                                   *
//*      copt.caseSensitive: whether sort is case sensitive                    *
//*         copt.showHidden: whether 'hidden' files will be displayed          *
//*                                                                            *
//* Input  : copt  : (by reference, initial contents ignored)                  *
//*                  on return, above members will reflect the current settings*
//*                                                                            *
//* Returns: 'true'  if options modified, else 'false'                         *
//*          Display list will be updated if options modified                  *
//******************************************************************************

bool FileDlg::ModifySortOptions ( ConfigOptions& copt )
{
//* Define dialog window size *
const short 
   dialogROWS = minfitROWS,  // will fit (easily) in smallest parent window
   dialogCOLS = minfitCOLS,  // will fit in smallest parent window
   dialogULY  = this->fulY + 1,
   dialogULX  = (this->fulX + this->fMaxX/2) - dialogCOLS / 2,
   sbItems    = 6,   // number of display items in scroll box
   sbItemWidth= 20 ; // number of columns in scroll box display item
const char sbText[sbItems][sbItemWidth + 1] = 
{
   " File Name          ",
   " Modification Date  ",
   " File Size          ",
   " Filename Extension ",
   " File Type          ",
   " No Sort            ",
} ;

   //* Determine the state of controls that will reflect current settings *
   //* This array maps the current sort option with the scroll-box index. *
   const short sortIndices[] = { 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5 } ;
   //* And this array maps the members of enum fmSort to the items in     *
   //* the Scroll Box. (eliminates 2 switch statements)                   *
   fmSort sortOpt[] = { fmNAME_SORT, fmNAMEr_SORT, fmDATE_SORT, fmDATEr_SORT,
                        fmSIZE_SORT, fmSIZEr_SORT, fmEXT_SORT,  fmEXTr_SORT,
                        fmTYPE_SORT, fmTYPEr_SORT, fmNO_SORT, fmNO_SORT } ;
   fmSort sbOrigOption = this->cfgOptions.sortOption ;
   short origSort = ZERO ;
   bool  origRev  = false,
         origCase = this->cfgOptions.caseSensitive,
         origHide = this->cfgOptions.showHidden ;
   for ( short i = ZERO ; i < fmSORT_OPTIONS ; i++ )
   {
      if ( sbOrigOption == sortOpt[i] )
      {
         origSort = sortIndices[i] ;
         if ( sbOrigOption == fmNAMEr_SORT || sbOrigOption == fmDATEr_SORT || 
              sbOrigOption == fmSIZEr_SORT || sbOrigOption == fmEXTr_SORT  || 
              sbOrigOption == fmTYPEr_SORT )
            origRev = true ;
         break ;
      }
   }

   //* Create a list of needed controls *
   enum dpCtrls { okPB, canPB, sortSB, revRB, caseRB, hideRB, dpcCOUNT } ;
   attr_t  singleColor[2] = { attrDFLT, this->cs.tf } ;
   InitCtrl ic[dpcCOUNT] ;          // control definitions
   ic[okPB] = 
   {  //* 'OK' pushbutton   - - - - - - - - - - - - - - - - - - - - - -   okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 10),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   OK   ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   } ;
   ic[canPB] = 
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - - -  canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[okPB].ulY,                 // ulY:       upper left corner in Y
      short(ic[okPB].ulX + 12),     // ulX:
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[sortSB],                  // nextCtrl:  link in next structure
   } ;
   ic[sortSB] = 
   { //* 'SortOption' Scroll Box - - - - - - - - - - - - - - - - - - - -sortSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      short(sbItems + 2),           // lines:     control lines
      short(sbItemWidth + 2),       // cols:      control columns
      (const char*)&sbText,         // dispText:  display items
      singleColor[1],               // nColor:    non-focus border color
      this->cs.pf,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " File-sort Option ",         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      sbItems,                      // scrItems:  number of elements in text/color arrays
      origSort,                     // scrSel:    index of initial highlighted element
      singleColor,                  // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[revRB],                   // nextCtrl:  link in next structure
   } ;
   ic[revRB] = 
   {  //* 'Reverse Sort' radio button   - - - - - - - - - - - - - - - -  revRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      origRev,                      // rbSelect:  initial value
      ic[sortSB].ulY,               // ulY:       upper left corner in Y
      short(ic[sortSB].ulX + ic[sortSB].cols + 3), // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Reverse sort order",         // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[caseRB],                  // nextCtrl:  link in next structure
   } ;
   ic[caseRB] = 
   {  //* 'Case Sensitive' radio button - - - - - - - - - - - - - - - - caseRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      origCase,                     // rbSelect:  initial value
      short(ic[revRB].ulY + 2),     // ulY:       upper left corner in Y
      ic[revRB].ulX,                // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Case-sensitive sort",        // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[hideRB],                  // nextCtrl:  link in next structure
   } ;
   ic[hideRB] = 
   {  //* 'Hidden Files' radio button   - - - - - - - - - - - - - - - - hideRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      origHide,                     // rbSelect:  initial value
      short(ic[caseRB].ulY + 2),    // ulY:       upper left corner in Y
      ic[caseRB].ulX,               // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Show hidden files",          // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   } ;

   //* Initial parameters for dialog window, passed to sub-methods.*
   InitNcDialog dInit
   ( 
      dialogROWS,           // number of display lines
      dialogCOLS,           // number of display columns
      dialogULY,            // Y offset from upper-left of terminal 
      dialogULX,            // X offset from upper-left of terminal 
      NULL,                 // dialog title
      ncltSINGLE,           // border line-style
      this->cs.sd,          // border color attribute
      this->cs.sd,          // interior color attribute
      ic                    // pointer to list of control definitions
   ) ;

   bool omods = false ;    // return vaule, 'true' if options modified

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Give the dialog a title *
      dp->SetDialogTitle ( "  Modify Sort Options  ", this->cs.em ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {  //* Get user's selections and store them in our data members *
                  short newSort = dp->GetScrollboxSelect ( sortSB ) ;
                  bool  newRev, newCase, newHide ;
                  dp->GetRadiobuttonState ( revRB, newRev ) ;
                  dp->GetRadiobuttonState ( caseRB, newCase ) ;
                  dp->GetRadiobuttonState ( hideRB, newHide ) ;
                  fmSort sbNewOption = sortOpt[newRev ? (newSort * 2 + 1) : (newSort * 2)] ;
                  if ( sbNewOption != sbOrigOption  )
                  { copt.sortOption = sbNewOption ; omods = true ; }
                  if ( newCase != origCase )
                  { copt.caseSensitive = newCase ; omods = true ; }
                  if ( newHide != origHide )
                  { copt.showHidden = newHide ; omods = true ; }
               }
               done = true ;
            }
         }
         //* If focus is currently on a Scrollext control *
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollbox ( Info ) ;
         }
         //* If focus is currently on a Radio Button *
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         //* Move focus to appropriate control *
         if ( ! done  && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }        // if((this->dPtr->OpenWindow())==OK)
   if ( dp != NULL )
      delete dp ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   //* If options modified, update file-display list *
   if ( omods )
   {
      this->UpdateSortOptions ( copt ) ;
      this->RefreshCurrDir () ;
   }

   return omods ;

}  //* End ModifySortOptions() *

//*************************
//*   FormatDateString    *
//*************************
//******************************************************************************
//* Produce a formatted output string based on data in a localTime object.     *
//*   Format:  "dd Mmm yyyy hh:mm:ss"                                          *
//*                                                                            *
//* Input  : ft  : initialized localTime structure (by reference)              *
//*                Note: 'day', the day-of-the-week field is ignored.          *
//*          str : buffer to receive formatted data                            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::FormatDateString ( const localTime& ft, gString& str )
{

   str.compose( L"%02hu %S %04hu %02hu:%02hu:%02hu",
                &ft.date, wMonthString[ft.month], &ft.year, 
                &ft.hours, &ft.minutes, &ft.seconds ) ;

}  //* End FormatDateString() *

//****************************
//* CopyOrDeleteNotSupported *
//****************************
//******************************************************************************
//* Display an information dialog with news that copy/delete operations for    *
//* specified file-type are not supported.                                     *
//*                                                                            *
//* Input  : fType   : file-type of file to be copied or deleted               *
//*          srcName : name of file to be copied or deleted                    *
//*                                                                            *
//* Returns: ecUNSUPP, operation not supported for file type                   *
//******************************************************************************

short FileDlg::CopyOrDeleteNotSupported ( fmFType fType, const char* srcName )
{
   const char* const cdnsType[] = 
   {
      "Character Device",
      "Block Device",
      "Socket",
   } ;
   short tIndex = fType == fmCHDEV_TYPE ? ZERO : fType == fmBKDEV_TYPE ? 1 : 2 ;
   gString gsTitle, gsType, gsName ;
   gsTitle.compose( L" Copy or Delete %s File ", cdnsType[tIndex] ) ;
   gsType.compose( L"is a '%s'. Copy and Delete operations", cdnsType[tIndex] ) ;
   gsName.compose( L" %s ", srcName ) ;
   const char* msg[] = 
   {
      gsTitle.ustr(),
      "The following file:",
      gsName.ustr(),
      gsType.ustr(),
      "for this file type are not supported at this time.",
      NULL,
   } ;
   attr_t msgAttr[] = { (this->cs.sd | ncbATTR), this->cs.sd, 
                        (this->cs.sd & ~ncrATTR), 
                        this->cs.sd, this->cs.sd, this->cs.sd, } ;
   this->InfoDialog ( (const char**)msg, msgAttr ) ;

   return ecUNSUPP ;

}  //* End CopyOrDeleteNotSupported() *

//*************************
//*   DisplayErrorCode    *
//*************************
//******************************************************************************
//* A processing error has just occurred.                                      *
//* Display the FMgr 'errno' error code, and the FileDlg error code, if any.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::DisplayErrorCode ( void )
{
   errorCode   ec ;
   this->GetErrorCode ( ec ) ;
   const char* ecText[] = 
   {
      "  File System Error  ",
      " ",
      "Processing Error:",
      ec.fdErrorDesc,
      " ",
      "System Error Reported ('errno'):",
      ec.sysErrorDesc,
      NULL
   } ;
   attr_t ecColor[INFODLG_MAX_MSGS] =
   {
      this->cs.sd | ncbATTR, this->cs.sd, this->cs.sd, this->cs.sd | ncbATTR, 
      this->cs.sd, this->cs.sd, this->cs.sd | ncbATTR, this->cs.sd
   } ;
   this->InfoDialog ( ecText, ecColor ) ;

}  //* End DisplayErrorCode() *

//*************************
//*      InfoDialog       *
//*************************
//******************************************************************************
//* Generic information dialog. May be used to display any list of constant    *
//* strings, then waits for user to press Enter (or ESC).                      *
//*                                                                            *
//* Dialog minimum height is 4 (top line, space above OK control, line for     *
//* OK control, and bottom line) PLUS one message line.                        *
//*                                                                            *
//* Dialog maximum height is MIN_ROWS - 9. This value represents the maximum   *
//* interior height of the file-display control, and provides space for        *
//* INFODLG_MAX_MSGS message lines including dialog title.                     *
//*                                                                            *
//* Dialog width is fixed at INFODLG_WIDTH. This value represents the minimum  *
//* interior width of the file-display control.                                *
//*                                                                            *
//* For reasons of beauty, messages begin at offset 2 in X and SHOULD end at   *
//* dialog-width minus 2.                                                      *
//*                                                                            *
//* Input  : msgList: array of char pointers to messages                       *
//*                   The first message line is interpreted as the dialog title*
//*                   " " == empty line and NULL (null pointer) == end of list *
//*          msgAttr: (optional, NULL by default)                              *
//*                   if not specified, each message line is displayed in the  *
//*                     base interior color set during FileDlg instantiation.  *
//*                   if specified, this is a pointer to an array of color     *
//*                     attributes, one for each message line in msgList.      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::InfoDialog ( const char** msgList, const attr_t* msgAttr )
{

   this->GenericDialog ( msgList, false, msgAttr ) ;

}  //* End InfoDialog() *

//*************************
//*    DecisionDialog     *
//*************************
//******************************************************************************
//* Generic decision-making dialog. Similar to InfoDialog() above, except      *
//* that user must choose between the "YES" and "NO" pushbuttons.              *
//*                                                                            *
//* For this reason, your message should be one that can be answered by        *
//* either a "yes" or "no" response.                                           *
//*                                                                            *
//* Input  : msgList: array of char pointers to messages                       *
//*                   The first message line is interpreted as the dialog title*
//*                   " " == empty line and NULL (null pointer) == end of list *
//*          msgAttr: (optional, NULL by default)                              *
//*                   if not specified, each message line is displayed in the  *
//*                     base interior color set during FileDlg instantiation.  *
//*                   if specified, this is a pointer to an array of color     *
//*                     attributes, one for each message line in msgList.      *
//*                                                                            *
//* Returns: 'true' if user selects "YES", else 'false'                        *
//******************************************************************************

bool FileDlg::DecisionDialog ( const char** msgList, const attr_t* msgAttr )
{

   return ( this->GenericDialog ( msgList, true, msgAttr ) ) ;

}  //* End DecisionDialog() *

//*************************
//*    GenericDialog      *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* Generic information OR decision-making dialog. Called by either            *
//* InfoDialog() or DecisionDialog() above.                                    *
//*                                                                            *
//* For this reason, your message should be informational only, OR should be   *
//* one that can be answered by either a "yes" or "no" response.               *
//*                                                                            *
//* Input  : msgList: array of char pointers to messages                       *
//*                   The first message line is interpreted as the dialog title*
//*                   " " == empty line and NULL (null pointer) == end of list *
//*          decision: if 'true' ask user for a yes/no decision                *
//*                    if 'false' only the OK pushbutton is displayed          *
//*          msgAttr: (optional, NULL by default)                              *
//*                   if not specified, each message line is displayed in the  *
//*                     base interior color set during FileDlg instantiation.  *
//*                   if specified, this is a pointer to an array of color     *
//*                     attributes, one for each message line in msgList.      *
//*                                                                            *
//* Returns: 'true' if 'decision' AND user selects "YES", else 'false'         *
//******************************************************************************
//* Programmer's Note: In practical terms, the color attribute list (if        *
//* specified should be limited to the value of this->cs.sd and its varients:  *
//*                 this->cs.sd | nc?ATTR  or                                  *
//*                 this->cs.sd & ~nc?ATTR.                                    *
//* Anything else looks bloody ugly.                                           *
//*                                                                            *
//* Note that we attempt to do defensive programming by checking the array of  *
//* string pointers; however, if caller has fewer that INFODLG_MAX_MSGS but no *
//* null-string to terminate the list, we will over-run the array boundary,    *
//* and there's not much we can do about it. See the NcDialog 'genDialog' class*
//* for a way to bullet-proof the code.                                        *
//******************************************************************************

bool FileDlg::GenericDialog ( const char** msgList, bool decision, const attr_t* msgAttr )
{
   const char* badArg[] = { "INVALID PARAMETER!", NULL } ;

   short dialogLines = MIN_genDialog_LINES,  // dialog lines
         dialogCols = INFODLG_WIDTH,         // dialog columns (fixed)
         msgCount = ZERO ;                   // number of message lines in msgList
         attr_t dColor = this->cs.sd,        // base dialog color
          msgColor[INFODLG_MAX_MSGS] =       // default message line color attributes
          {
             this->cs.em, dColor, dColor, dColor, dColor, dColor, dColor, dColor
          } ;
   bool  proceed = false ;                   // return value

   //* Count number of message lines and adjust dialog size accordingly.       *
   while ( msgList[msgCount] != NULL && msgCount < INFODLG_MAX_MSGS )
      ++msgCount ;
   if ( (msgCount == INFODLG_MAX_MSGS && msgList[msgCount] != NULL)
        || msgCount == ZERO )
   {
      msgList = badArg ;
      msgCount = 1 ;
   }
   dialogLines += msgCount - 1 ;

   //* If custom color attributes have been specified, re-initialize the array *
   if ( msgAttr != NULL )
   {
      for ( short i = ZERO ; i < msgCount ; i++ )
         msgColor[i] = msgAttr[i] ;
   }

   //* Offset from parent dialog's upper left corner *
   short yOff = this->fulY + this->fMaxY / 2 - dialogLines / 2,
         xOff = this->fulX + this->fMaxX / 2 - dialogCols / 2 ;

   //* Display the dialog and wait for user response *
   genDialog gd( msgList, dColor, dialogLines, dialogCols, yOff, xOff, 
                 msgColor, false, 
                 this->cfgOptions.cScheme.pn, this->cfgOptions.cScheme.pf ) ;
   if ( decision != false )
      proceed = this->dPtr->DecisionDialog ( gd ) ;
   else
      this->dPtr->InfoDialog ( gd ) ;

   return proceed ;

}  //* End GenericDialog() *

#if ENABLE_DEBUGGING_CODE != 0
//*************************
//*   Display_TreeNode    *
//*************************
//******************************************************************************
//* FOR DEBUG ONLY!                                                            *
//* Display the contents of a TreeNode object.                                 *
//*                                                                            *
//* Input  : ul   : Y/X position for upper-left corner of dialog (absolute)    *
//*        : tn   : pointer to TreeNode instance                               *
//*          stayOpen: (optional, false by default)                            *
//*                    it 'true', then return to caller without closing the    *
//*                    dialog window                                           *
//*                                                                            *
//* Returns: if stayOpen != false, then returns pointer to open dialog         *
//*             which caller must close using 'delete dnPtr'                   *
//* Returns: if stayOpen == false, then returns NULL pointer                   *
//*             dialog has been closed                                         *
//******************************************************************************

NcDialog* FileDlg::Display_TreeNode ( winPos ul, const TreeNode& tn, bool stayOpen )
{
const short dialogRows = 19,
            dialogCols = 52,
            controlsDEFINED = 1 ;
short ulY = ul.ypos,                // dialog position
      ulX = ul.xpos ;
attr_t dColor = this->cs.sd,        // dialog color
       vColor = this->cs.sd ;       // data-value color

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -   okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogRows - 2),        // ulY:       upper left corner in Y
      short(dialogCols / 2 - 7),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      "  OK  ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogRows,     // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       " TreeNode Contents ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Save the display for the parent dialog window *
   if ( stayOpen == false )
      this->dPtr->SetDialogObscured () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      LineDef  lDefH(ncltHORIZ, ncltSINGLE, 6, ZERO, dialogCols, dColor) ;
      dp->DrawLine ( lDefH ) ;
      LineDef  lDefV(ncltVERT, ncltSINGLE, 7, 27, dialogRows-8, dColor) ;
      dp->DrawLine ( lDefV ) ;

      //* Format and display the information *
      gString  outBuff, iBuff ;
      winPos   wp( 1, 2 ) ;

//      if ( tn != NULL )
//      {
      //* Directory name info *
      outBuff.compose( L"Dir Name : '%s'", tn.dirStat.fName ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "Dir Type : ", dColor ) ;
      outBuff.compose( L"%02hd", &tn.dirStat.fType ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "Dir Bytes: ", dColor ) ;
      iBuff.formatInt( tn.dirStat.fBytes, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "Dir Date : ", dColor ) ;
      this->FormatDateString ( tn.dirStat.modTime, iBuff ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "Read Acc : ", dColor ) ;
      outBuff.compose( L"%hhd", &tn.dirStat.readAcc ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, outBuff, vColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "  Write Acc: ", dColor ) ;
      outBuff.compose( L"%hhd", &tn.dirStat.writeAcc ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      //* Count and size accumulators *
      wp = { short(wp.ypos + 1), 2 } ;
      dp->WriteString ( wp.ypos++, wp.xpos, "          COUNT     BYTES", dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "totFiles : ", dColor ) ;
      iBuff.formatInt( tn.totBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.totFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "dirFiles : ", dColor ) ;
      iBuff.formatInt( tn.dirBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.dirFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "regFiles : ", dColor ) ;
      iBuff.formatInt( tn.regBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.regFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "linkFiles: ", dColor ) ;
      iBuff.formatInt( tn.linkBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.linkFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "cdevFiles: ", dColor ) ;
      iBuff.formatInt( tn.cdevBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.cdevFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "bdevFiles: ", dColor ) ;
      iBuff.formatInt( tn.bdevBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.bdevFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "fifoFiles: ", dColor ) ;
      iBuff.formatInt( tn.fifoBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.fifoFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "sockFiles: ", dColor ) ;
      iBuff.formatInt( tn.sockBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.sockFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "unkFiles : ", dColor ) ;
      iBuff.formatInt( tn.unkBytes, 9 ) ;
      outBuff.compose( L"%4u %S", &tn.unkFiles, iBuff.gstr() ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      //* Misc. stuff *
      UINT  nlnodes, tnfcount ;
      tn.GetActual ( nlnodes, tnfcount ) ;
      wp = { 7, 29 } ;

      wp = dp->WriteString ( wp.ypos, wp.xpos, "nextLevel : ", dColor ) ;
      outBuff.compose( L"%p", tn.nextLevel ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;
      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "nlnodes   : ", dColor ) ;
      iBuff.formatInt( (ULONG)nlnodes, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "tnFiles   : ", dColor ) ;
      outBuff.compose( L"%p", tn.tnFiles ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;
      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "tnFCount  : ", dColor ) ;
      iBuff.formatInt( (ULONG)tn.tnFCount, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;
      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "tnfcount  : ", dColor ) ;
      iBuff.formatInt( (ULONG)tnfcount, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "allocBytes: ", dColor ) ;
      iBuff.formatInt( (ULONG)tn.allocBytes, 9, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "wprotFiles: ", dColor ) ;
      iBuff.formatInt( (ULONG)tn.wprotFiles, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "violations: ", dColor ) ;
      iBuff.formatInt( (ULONG)tn.violations, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "prevLevel : ", dColor ) ;
      outBuff.compose( L"%p", tn.prevLevel ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=29), "level     : ", dColor ) ;
      outBuff.compose( L"%02hd", &tn.level ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

// dp->WriteString ( dialogLines-1, ZERO,
// "1--------0---------0---------0---------0---------0---------0---------0---4-----0", dColor ) ;
//      }
//      else     // invalid TreeNode reference
//      {
//         dp->WriteString( wp, "Invalid TreeNode* tn", dColor ) ;
//      }

      dp->RefreshWin () ;                 // make everything visible

      if ( stayOpen == false )
      {  //* Talk with the animals *
         uiInfo   Info ;                  // user interface data returned here
         do
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               dp->EditPushbutton ( Info ) ;
         }
         while ( ! Info.dataMod && Info.keyIn != nckESC ) ;
      }
   }
   else           // unable to open window (likely memory-allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( stayOpen == false && dp != NULL )    // close the window
   {
      delete ( dp ) ;
      dp = NULL ;
   }

   //* Restore display of parent dialog window *
   if ( stayOpen == false )
      this->dPtr->RefreshWin () ;

   //* Return a pointer to the open dialog window *
   return dp ;

}  //* End Display_TreeNode() *

//*************************
//*   Display_tnFName     *
//*************************
//******************************************************************************
//* FOR DEBUG ONLY!                                                            *
//* Display the contents of a tnFName object.                                  *
//*                                                                            *
//* Input  : ul   : Y/X position for upper-left corner of dialog (absolute)    *
//*        : tnf  : pointer to tnFName instance                                *
//*          stayOpen: (optional, false by default)                            *
//*                    it 'true', then return to caller without closing the    *
//*                    dialog window                                           *
//*                                                                            *
//* Returns: if stayOpen != false, then returns pointer to open dialog         *
//*             which caller must close using 'delete dnPtr'                   *
//*          if stayOpen == false, then returns NULL pointer                   *
//*             dialog has been closed                                         *
//******************************************************************************

NcDialog* FileDlg::Display_tnFName ( winPos ul, tnFName& tnf, bool stayOpen )
{
const short dialogLines = 19,
            dialogCols  = 52,
            controlsDEFINED = 1 ;
short ulY = ul.ypos,                // dialog position
      ulX = ul.xpos ;
attr_t dColor = this->cs.sd,        // dialog color
       vColor = this->cs.sd ;       // data-value color

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -   okPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      "  OK  ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       " tnFName Contents ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       // pointer to list of control definitions
                       (InitCtrl*)(stayOpen == false ? &ic : NULL)
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Save the display for the parent dialog window *
   if ( stayOpen == false )
      this->dPtr->SetDialogObscured () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      LineDef  lDefH(ncltHORIZ, ncltSINGLE, 6, ZERO, dialogCols, dColor) ;
      dp->DrawLine ( lDefH ) ;
      LineDef  lDefV(ncltVERT, ncltSINGLE, 8, 24, dialogLines-12, dColor) ;
      dp->DrawLine ( lDefV ) ;

      //* Format and display the information *
      gString  outBuff, iBuff ;
      winPos   wp( 1, 2 ) ;

      //* File info *
      outBuff.compose( L"Name : '%s'", tnf.fName ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "Type : ", dColor ) ;
      outBuff.compose( L"%02hd", &tnf.fType ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "Bytes: ", dColor ) ;
      iBuff.formatInt( tnf.fBytes, 5, true ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "Date : ", dColor ) ;
      this->FormatDateString ( tnf.modTime, iBuff ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, iBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "Read Acc : ", dColor ) ;
      outBuff.compose( L"%hhd", &tnf.readAcc ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, outBuff, vColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "  Write Acc: ", dColor ) ;
      outBuff.compose( L"%hhd", &tnf.writeAcc ) ;
      wp = dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp, "  fsMatch: ", dColor ) ;
      outBuff.compose( L"%hhd", &tnf.fsMatch ) ;
         dp->WriteString ( wp, outBuff, vColor ) ;


      //* Raw stat data *
      wp = { short(wp.ypos + 1), 2 } ;
      dp->WriteString ( wp.ypos++, wp.xpos, "File 'stat' Data:", dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, "st_ino  : ", dColor ) ;
      outBuff.compose( L"%llu", &tnf.rawStats.st_ino ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "st_dev  : ", dColor ) ;
      outBuff.compose( L"0x%08llx", &tnf.rawStats.st_dev ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "st_rdev : ", dColor ) ;
      outBuff.compose( L"0x%08llx", &tnf.rawStats.st_rdev ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "st_mode : ", dColor ) ;
      outBuff.compose( L"0x%08x", &tnf.rawStats.st_mode ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "st_uid  : ", dColor ) ;
      outBuff.compose( L"%u", &tnf.rawStats.st_uid ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "st_gid  : ", dColor ) ;
      outBuff.compose( L"%u", &tnf.rawStats.st_gid ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "st_nlink: ", dColor ) ;
      outBuff.compose( L"%llu", &tnf.rawStats.st_nlink ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = { 8, 26 } ;
      wp = dp->WriteString ( wp.ypos, (wp.xpos=26), "st_size   : ", dColor ) ;
      outBuff.compose( L"%llu", &tnf.rawStats.st_size ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=26), "st_blksize: ", dColor ) ;
      outBuff.compose( L"%llu", &tnf.rawStats.st_blksize ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=26), "st_blocks : ", dColor ) ;
      outBuff.compose( L"%llu", &tnf.rawStats.st_blocks ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=26), "st_atime  : ", dColor ) ;
      outBuff.compose( L"0x%08llx", &tnf.rawStats.st_atime ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=26), "st_mtime  : ", dColor ) ;
      outBuff.compose( L"0x%08llx", &tnf.rawStats.st_mtime ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      wp = dp->WriteString ( wp.ypos, (wp.xpos=26), "st_ctime  : ", dColor ) ;
      outBuff.compose( L"0x%08llx", &tnf.rawStats.st_ctime ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, outBuff, vColor ) ;

      dp->RefreshWin () ;                 // make everything visible

      if ( stayOpen == false )
      {  //* Talk with the animals *
         uiInfo   Info ;                  // user interface data returned here
         do
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               dp->EditPushbutton ( Info ) ;
         }
         while ( ! Info.dataMod && Info.keyIn != nckESC ) ;
      }
   }
   else           // unable to open window (likely memory-allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( stayOpen == false && dp != NULL )    // close the window
   {
      delete ( dp ) ;
      dp = NULL ;
   }

   //* Restore display of parent dialog window *
   if ( stayOpen == false )
      this->dPtr->RefreshWin () ;

   //* Return a pointer to the open dialog window *
   return dp ;

}  //* End Display_tnFName() *

//*************************
//*    Open_DBwindow      *
//*************************
//******************************************************************************
//* Open a dialog window in the unused portion of the terminal window.         *
//* Used for writing debugging information, specifically to track recursive    *
//* file operations.                                                           *
//*                                                                            *
//* Application must be running in Single-Window mode, with terminal window    *
//* >= 132 columns else there will be insufficient free space in the           *
//* terminal window.                                                           *
//*                                                                            *
//* Input  : dialogColor ( by reference ) receives dialog-win color attribute  *
//*          dialogLines ( by reference ) receives number of lines in dialog   *
//*          dialogCols  ( by reference ) receives number of columns in dialog *
//*                                                                            *
//* Returns: pointer to open dialog which                                      *
//*           (caller must close dialog using 'delete dnPtr'                   *
//*          return NULL* if no free space in terminal window                  *
//******************************************************************************

NcDialog* FileDlg::Open_DBwindow ( attr_t& dialogColor, short& dialogLines, short&dialogCols )
{
   //* Determine size of our window *
   short termRows, termCols,           // terminal window size
         appLines, appCols ;           // application dialog size
   this->dPtr->ScreenDimensions ( termRows, termCols ) ;
   this->dPtr->GetDialogDimensions ( appLines, appCols ) ;
   
   short ulY = ZERO,                   // dialog position
         ulX = appCols ;

   dialogLines = termRows ;            // dialog size
   dialogCols  = termCols - appCols ;
   dialogColor = this->cs.sd ;         // dialog color

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Debug Window  ",// dialog title
                       ncltSINGLE,     // border line-style
                       dialogColor,    // border color attribute
                       dialogColor,    // interior color attribute
                       NULL            // no controls defined
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->RefreshWin () ;                 // make everything visible
   }
   else     // unable to open window
   {        //(likely that terminal window too small)
      if ( dp != NULL )
      {
         delete dp ;
         dp = NULL ;
      }
      this->DisplayStatus ( ecAPPERROR ) ;
   }

   return dp ;

}  //* End Open_DBwindow() *
#endif   // ENABLE_DEBUGGING_CODE

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

