//********************************************************************************
//* File       : FileDlgContext.cpp                                              *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 25-Jul-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module of the FileDlg class is an extension of the FileDlgPrompt.cpp    *
//* module, and contains functionality to support the application 'ViewFile'     *
//* context menu.                                                                *
//*    1) View Contents:                                                         *
//*       View the contents of the selected file.                                *
//*    2) View File Stats:                                                       *
//*       View and optionally modify the file stats including dates,             *
//*       permission flags, ets.                                                 *
//*    3) Execute (Run):                                                         *
//*       a) For executable files (binaries, script files), execute the          *
//*          selected file.                                                      *
//*       b) For media file and document files, invoke the system's default      *
//*          application for the selected file.                                  *
//*    4) Find Link Target:                                                      *
//*       For symbolic link files, locate the link target and set the target's   *
//*       directory as the current working directory (CWD).                      *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************
extern const char** fdNotImplemented ;
extern const char* fmtypeName[] ;

//* Control indices for opeUserPrompt() *
enum upIndices : short
{
   upiLAUNCH = ZERO, upiCANCEL, upiGUI, upiCUR, upiNEW, upiARGS, upiGED, 
   upiCONTROLS,
   upiHEADER, upiEDIT, upiQHELP     // extra indices for various messages
} ;

#define DEBUG_base64Decode (0)      // USED FOR DEVELOPMENT ONLY
#if DEBUG_base64Decode != 0
ofstream dbgofs ;
#endif   // DEBUG_base64Decode

//******************************
//*  Local non-member methods  *
//******************************

static bool vfcIsPerlSource ( const gString& trgPath, bool strict = false ) ;
static bool vfcIsPerlCommand ( const gString& gstoken ) ;
static bool vfcGetPerlToken ( gString& gssrc, gString& gstrg ) ;
static bool oepGooeyType ( const gString& fPath, bool guiAudio ) ;
static bool oepBinexeTest ( const gString& fPath ) ;
static bool htmlExtractTag ( gString& gsIn, gString& gsOut ) ;
static bool deDecodeHeader ( const gString& gsSrc, gString& gsTrg ) ;
static bool deDecode_base64 ( const gString& gsSrc, gString& gsTrg ) ;
static bool deDecode_qprint ( const gString& gsSrc, gString& gsTrg ) ;
static short deFormatLine ( gString& gsOut, short termCols, short colOut ) ;
static short dbdFormatLine ( gString& gsOut, short termCols, short colOut ) ;


//*************************
//*   ViewFileContents    *
//*************************
//********************************************************************************
//* View contents of the currently-highlighted file.                             *
//*                                                                              *
//* Output is first formated (if necessary) and then displayed using the         *
//* 'less' command-line utility.                                                 *
//*                                                                              *
//* Input  : vfFmt : (optional, vfPrompt by default) -member of enum viewFormat  *
//*                  format in which to view file contents                       *
//*                  [ CURRENTLY IGNORED ]                                       *
//*                                                                              *
//* Returns: OK if successful,                                                   *
//*          ERR if file not found or file is not a 'regular' file               *
//********************************************************************************
//* Notes:                                                                       *
//* a) To view as text:                                                          *
//*    invoke the 'less' utility, with or without line numbers (-N option).      *
//* b) To view as hex/decimal/octal:                                             *
//*    invoke the 'hexdump' utility to format the data and pipe it to 'less'.    *
//* c) To view as binary:                                                        *
//*    Call vfcDecodeBinary() to format the source data for binary output,       *
//*    then call the 'less' utility.                                             *
//*    Note: This was formerly handled by the 'xxd' utility, which is often      *
//*          not installed by default by modern Linux distributions.             *
//* d) To view the contents of a symlink file rather than it's target:           *
//*    invoke the 'readlink' C library function to format the data, then         *
//*    call the 'less' utility.                                                  *
//* e) For archive files, note that the 'less' utility may invoke a              *
//*    preprocessor program to correctly display archive contents                *
//*    automagically. However, this is a bit TOO magical for our taste.          *
//*    On Fedora, this magic conversion done through a script specified by the   *
//*    'LESSOPEN' environment variable. This is not guaranteed to be the same    *
//*    for all systems or for all environments on a given system, so we do not   *
//*    rely upon it, and use our own decoding.                                   *
//* f) For OpenDocument files:                                                   *
//*    -- displayed as text, (.odt, .ods, .odp), the XML sub-file containing     *
//*       the document text is extracted from the document and scanned.          *
//*       The text data are formatted and copied to a temp file which is then    *
//*       displayed by the 'less' utility.                                       *
//*    -- displayed as a 'zip archive, (all document formats for which we have   *
//*       samples), run the file through the archive filter (see item 'e')       *
//*       and then display the archive contents.                                 *
//* g) For Office OpenXML files:                                                 *
//*    -- displayed as text, (.docx, pptx, xlsx), the XML sub-file containing    *
//*       the document text is extracted from the document and scanned.          *
//*       The text data are formatted and copied to a temp file which is then    *
//*       displayed by the 'less' utility.                                       *
//*    -- displayed as a 'zip archive, (all document formats for which we have   *
//*       samples), run the file through the archive filter (see item 'e')       *
//*       and then display the archive contents.                                 *
//* h) For files which contain metadata (audio, photo, video):                   *
//*    -- Extract and display the metadata.                                      *
//*                                                                              *
//* i) If the filename is too long to display in this dialog, it will result     *
//*    in a non-critical case of the uglies.                                     *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

short FileDlg::ViewFileContents ( viewFormat vfFmt )
{
   short status = ERR ;

   //* If there is actually a file under the highlight *
   if ( this->fileCount > ZERO )
   {
      const short DLG_ROWS    = 13,       // dialog lines
                  dlgCols     = minfitCOLS ; // dialog columns
      const attr_t dColor = this->cs.sd,  // dialog background color
                   hColor = this->cs.em,  // bold text
                   rbnColor = this->cs.tn,// radio-button colors
                   rbfColor = this->cs.scheme == ncbcCOLORS ? 
                                                 this->cs.tf | ncgATTR : this->cs.pf ;
      short dlgRows = DLG_ROWS ;

      enum vfcControls 
      { vfcVIEW= ZERO, vfcCANCEL, vfcTEXT, vfcHEXB, vfcHEXW, 
        vfcDECI, vfcOCT, vfcBIN, vfcLINK, vfcARCH, vfcMETA, vfcEML, vfcLNUM, 
        controlsDEFINED } ;

      //* Retrieve the file stat informaton *
      tnFName  fn, fnt ;            // copy of basic file info
      this->GetStats ( fn ) ;       // stats for currently-highlighted file
      const char* trgName = fn.fName ;// pointer to target filename
      gString trgPath ;             // target path string
      this->fmPtr->CatPathFname ( trgPath, this->currDir, fn.fName ) ;
      gString origPath = trgPath ;  // save a copy of original target
      fmFType trgType = fn.fType ;  // target's file type
      short viewFormat = vfcTEXT ;  // viewing format, initially view-as-text

      //* 'true' if source is OpenDocument file with scanable text data *
      bool  isOD_File = this->odIsOpenDocFile ( trgPath ) ;
      //* 'true' if source is OpenXML file with scanable text data *
      bool  isOX_File = this->odIsOpenXmlFile ( trgPath ) ;
      //* 'true' if source is an old-style binary document file *
      bool  isOB_File = this->odIsBinDocFile ( trgPath ) ;
      bool  isInfodoc = bool((trgPath.find( L".info" )) == (trgPath.gschars() - 6)) ;
      bool  ansiColor  = false,     // 'true' if less should interpret ANSI color
            fmtSrcCode = false,     // 'true' formatting applied to source code (XML, etc.)
            regType    = true,      // 'true' if ultimate target file == fmREG_TYPE
            lineNums   = false,     // 'true' if line-numbers control is set
            lnkType    = false,     // 'true' if original source == fmLINK_TYPE
            brokenLink = false,     // 'true' if sym-link target not found
            userCancel = false ;    // 'true' if user cancelled viewing of file

      //* If source file is a symbolic link *
      if ( fn.fType == fmLINK_TYPE )
      {  //* Get stats for sym-link target *
         lnkType = true ;
         gString fntPath ;
         if ( (this->fmPtr->GetLinkTargetStats ( trgPath, fnt, fntPath)) == OK )
         {  //* There is a valid link target. Use it as our source file *
            trgName = fnt.fName ;
            trgType = fnt.fType ;
            trgPath = fntPath ;
            regType = (bool)(fnt.fType == fmREG_TYPE ? true : false ) ;
         }
         else
         {  //* The stat failed, but traces of the broken link may exist.*
            //* Get its name && type if possible.                        *
            if ( fnt.readAcc )
            {
               trgName = fnt.fName ;
               trgType = fnt.fType ;
            }
            else
            { /* serious error */ }
            brokenLink = true ;
            regType = false ;
         }
      }
      //* Source file is neither a 'regular' file nor a link to a regular file.*
      else if ( fn.fType != fmREG_TYPE )
         regType = false ;

      //* We display contents ONLY for 'regular' files, (or as a special case, *
      //* the contents of a symlink), so if target is neither a regular file   *
      //* nor the special case, the radio-button group is not instantiated,    *
      //* and we can shrink the dialog window by an equivalent amount.         *
      if ( ! regType && ! lnkType )
         dlgRows -= 4 ;

      InitCtrl ic[controlsDEFINED] = 
      {
      {  //* 'VIEW' pushbutton - - - - - - - - - - - - - - - - - - - - vfcVIEW *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dlgRows - 5),           // ulY:       upper left corner in Y
         short(dlgCols / 3 - 5),       // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         "  ^VIEW  ",                  // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcCANCEL],               // nextCtrl:  link in next structure
      },
      {  //* 'CANCEL" pushbutton - - - - - - - - - - - - - - - - - - vfcCANCEL *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[vfcVIEW].ulY),       // ulY:       upper left corner in Y
         short(ic[vfcVIEW].ulX + ic[vfcVIEW].cols + 3), // ulX: upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         " ^CANCEL ",                  // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         ((regType || lnkType) ? &ic[vfcTEXT] : NULL),// nextCtrl:  
      },
      {  //* 'Text' radio button   - - - - - - - - - - - - - - -  vfcTEXT   *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         true,                         // rbSelect:  initially set
         3,                            // ulY:       upper left corner in Y
         3,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Text",                      // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcHEXB],                 // nextCtrl:  link in next structure
      },
      {  //* 'HexByte' radio button  - - - - - - - - - - - - - -  vfcHEXB   *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcTEXT].ulY + 1),   // ulY:       upper left corner in Y
         ic[vfcTEXT].ulX,              // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Hex (8-bit)",               // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcHEXW],                 // nextCtrl:  link in next structure
      },
      {  //* 'HexWord' radio button  - - - - - - - - - - - - - -  vfcHEXW   *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcHEXB].ulY + 1),   // ulY:       upper left corner in Y
         ic[vfcHEXB].ulX,              // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "He^x (16-bit)",              // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcDECI],                 // nextCtrl:  link in next structure
      },
      {  //* 'Decimal' radio button  - - - - - - - - - - - - - - -  vfcDECI *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         ic[vfcTEXT].ulY,              // ulY:       upper left corner in Y
         short(ic[vfcTEXT].ulX + 19),  // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Decimal",                   // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcOCT],                  // nextCtrl:  link in next structure
      },
      {  //* 'Octal' radio button    - - - - - - - - - - - - - -  vfcOCT    *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcDECI].ulY + 1),   // ulY:       upper left corner in Y
         ic[vfcDECI].ulX,              // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Octal",                     // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcBIN]                   // nextCtrl:  link in next structure
      },
      {  //* 'Binary' radio button   - - - - - - - - - - - - - - -   vfcBIN *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcOCT].ulY + 1),    // ulY:       upper left corner in Y
         ic[vfcOCT].ulX,               // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Binary",                    // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcLINK]                  // nextCtrl:  link in next structure
      },
      {  //* 'View Link' radio button  - - - - - - - - - - - - - -  vfcLINK *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         ic[vfcDECI].ulY,              // ulY:       upper left corner in Y
         short(ic[vfcDECI].ulX + 14),  // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^SymLink file",              // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcARCH]                  // nextCtrl:  link in next structure
      },
      {  //* 'List Archive' radio button  - - - - - - - - - - - - - -  vfcARCH *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcLINK].ulY + 1),   // ulY:       upper left corner in Y
         ic[vfcLINK].ulX,              // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Archive file",              // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcMETA]                  // nextCtrl:  link in next structure
      },
      {  //* 'Media Metadata' radio button   - - - - - - - - - - - - - vfcMETA *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcARCH].ulY + 1),   // ulY:       upper left corner in Y
         ic[vfcARCH].ulX,              // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "M^edia Metadata",            // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcEML]
      },
      {  //* 'Saved Email' radio button  -  - - - - - - - - - - - - - - vfcEML *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: standard, 3-wide
         false,                        // rbSelect:  initially reset
         short(ic[vfcMETA].ulY + 1),   // ulY:       upper left corner in Y
         ic[vfcMETA].ulX,              // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Email Archi^ve",             // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[vfcLNUM]
      },
      {  //* 'Line Numbers' radio button -  - - - - - - - - - - - - - vfcLNUM  *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: standard, 3-wide (angle)
         false,                        // rbSelect:  initially reset
         ic[vfcCANCEL].ulY,            // ulY:       upper left corner in Y
         short(ic[vfcCANCEL].ulX + ic[vfcCANCEL].cols + 3), // ulX: upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         rbnColor,                     // nColor:    non-focus color
         rbfColor,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Line Numbers",              // label:     
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL
      },
      } ;

      //* If file is not a symlink, disable the symlink Radiobutton *
      if ( ! lnkType )
      {
         ic[vfcLINK].active = false ;
         ic[vfcLINK].nColor = nc.gyR ;
      }
      //* If file is identified as a supported archive   *
      //* type, OR as an OpenDocument backup file  then  *
      //* set the 'Archive file' Radiobutton.            *
      arcType aType = this->ArchiveTarget ( trgPath ) ;
      if ( (aType == atNONE) && (this->odIsOpenDocBackup( trgPath)) )
         aType = atODOC ;
      if ( aType != atNONE )
      {
         //* If file is identified as an OpenDocument/OpenXML   *
         //* (zip) document, vfcARCH Radiobutton remains active *
         //* although selected viewFormat remains vfcTEXT.      *
         if ( (aType != atODOC) && (aType != atOXML) )
         {
            ic[vfcTEXT].rbSelect = false ;
            ic[vfcARCH].rbSelect = true ;
            viewFormat = vfcARCH ;
         }
      }
      //* Otherwise, target is not a recognized type *
      //* of archive, Disable archive Radiobutton.   *
      else
      {
         ic[vfcARCH].active = false ;
         ic[vfcARCH].nColor = nc.gyR ;
      }

      //* If file is identified as a supported audio file,*
      //* then set the 'Audio Metadata' Radiobutton.      *
      MediafileType mfType ;
      if ( (mfType = this->MediafileTarget ( trgPath )) != mftNONE )
      {
         ic[vfcTEXT].rbSelect = false ;
         ic[vfcMETA].rbSelect = true ;
         viewFormat = vfcMETA ;
      }
      //* If target is not a supported media file,        *
      //* disable the media-metadata Radiobutton.         *
      else
      {
         ic[vfcMETA].active = false ;
         ic[vfcMETA].nColor = nc.gyR ;
      }

      //* If file is an ".eml" (saved email) file, *
      //* then set the 'Email' Radiobutton.        *
      short extIndx = (-1) ;
      if ( ((extIndx = trgPath.findlast( L'.' )) >= ZERO) &&
           ((trgPath.compare( ".eml", false, 4, extIndx )) == ZERO) )
      {
         ic[vfcTEXT].rbSelect = false ;
         ic[vfcEML].rbSelect = true ;
         viewFormat = vfcEML ;
      }
      //* If target is not an email file, *
      //* disable the email Radiobutton.  *
      else
      {
         ic[vfcEML].active = false ;
         ic[vfcEML].nColor = nc.gyR ;
      }

      //* Line numbers are valid for text files only. *
      //* If one of the 'special' file types, disable *
      //* line number control.                        *
      if ( (viewFormat == vfcARCH) || (viewFormat == vfcMETA) || 
           (viewFormat == vfcEML) )
      {
         ic[vfcLNUM].active = false ;
         ic[vfcLNUM].nColor = nc.gyR ;
      }


      short ctrY        = this->fulY + this->fMaxY / 2,
            ctrX        = this->fulX + this->fMaxX / 2,
            ulY         = ctrY - dlgRows / 2,
            ulX         = ctrX - dlgCols / 2,
            status = ERR ;
      if ( ulY <= this->fulY ) ulY = this->fulY + 1 ; // vertical dialog position

      //* Save the display for the parent dialog window *
      this->dPtr->SetDialogObscured () ;

      //* Initial parameters for dialog window *
      InitNcDialog dInit( dlgRows,        // number of display lines
                          dlgCols,        // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal 
                          " View-File Format Options ", // dialog title
                          ncltSINGLE,     // border line-style
                          dColor,         // border color attribute
                          dColor,         // interior color attribute
                          ic              // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      NcDialog* dp = new NcDialog ( dInit ) ;

      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         gString msg ;        // message composition
         winPos wp( 2, 2 ) ;  // message position

         //* Group radio buttons *
         // Programmer's Note: Note special tests for inactive Radiobuttons.
         // This avoids a never-ending loop in the NcDialog library after item selection.
         if ( regType || lnkType )
         {
            short XorGroup1[10] = 
            { vfcTEXT, vfcHEXB, vfcHEXW, vfcDECI, vfcOCT, vfcBIN, 
              /*vfcLINK*/ -1, /*vfcARCH*/ -1, /*vfcMETA*/ -1, /*vfcEML*/ -1 } ;
              if ( ic[vfcLINK].active )
                 XorGroup1[6] = vfcLINK ;
              if ( ic[vfcARCH].active )
                 XorGroup1[(XorGroup1[6] == (-1)) ? 6 : 7] = vfcARCH ;
              if ( ic[vfcMETA].active )
                 XorGroup1[(XorGroup1[6] == (-1)) ? 6 :
                           (XorGroup1[7] == (-1)) ? 7 : 8] = vfcMETA ;
              if ( ic[vfcEML].active )
                 XorGroup1[(XorGroup1[6] == (-1)) ? 6 :
                           (XorGroup1[7] == (-1)) ? 7 :
                           (XorGroup1[8] == (-1)) ? 8 : 9] = vfcEML ;
            dp->GroupRadiobuttons ( XorGroup1 ) ;
         }

         if ( regType || lnkType )
         {
            msg.compose( L"View the file: '%s' as:", trgName ) ;
            msg.limitCols( dlgCols - 3 ) ;
            dp->WriteString ( --wp.ypos, wp.xpos, msg, hColor ) ;
            if ( fn.fType == fmLINK_TYPE )
               dp->WriteString ( ++wp.ypos, (wp.xpos + 16), "(sym-link target)", dColor ) ;
            wp = { short(dlgRows - 4), 2 } ;
            dp->WriteParagraph ( wp, 
               "  Output is first formated (if necessary) and then\n"
               "  displayed using the 'less' command-line utility.\n"
               "  (view with scroll keys, '/' to search, 'q' to quit)", dColor ) ;
         }
         else
         {
            wp = { 1, 2 } ;
            msg.compose( L"The file: '%s'", trgName ) ;
            msg.limitCols( dlgCols - 3 ) ;
            dp->WriteString ( wp.ypos++, wp.xpos, msg, dColor ) ;
            msg.compose( L"is a %s", fmtypeName[trgType] ) ;
            wp = dp->WriteString ( wp.ypos, wp.xpos, msg, dColor ) ;
            if ( brokenLink )
               dp->WriteString ( wp.ypos, wp.xpos, "  (broken link)", dColor ) ;
            else if ( fn.fType == fmLINK_TYPE )
               dp->WriteString ( wp.ypos, wp.xpos, "  (sym-link target)", dColor ) ;
            dp->WriteString ( ++wp.ypos, (wp.xpos = 2), 
                     "Only 'Regular-file' contents may be displayed.", dColor ) ;
            dp->NextControl () ; // place focus on 'Cancel' button
         }

         dp->RefreshWin () ;                 // make the text visible

         uiInfo   Info ;                     // user interface data returned here
         short    icIndex = ZERO ;           // index of control with input focus
         bool     done = false ;             // loop control
         while ( ! done )
         {
            //* If focus is currently on a Pushbutton   *
            if ( ic[icIndex].type == dctPUSHBUTTON )
            {
               if ( Info.viaHotkey )
                  Info.HotData2Primary () ;
               else
                  icIndex = dp->EditPushbutton ( Info ) ;
   
               //* If a Pushbutton was pressed *
               if ( Info.dataMod != false )
               {
                  //* If user has pressed 'VIEW' and we are viewing either *
                  //* a 'regular' file OR the contents of a 'symlink' file *
                  if ( (Info.ctrlIndex == vfcVIEW) && 
                       (regType || (lnkType && viewFormat == vfcLINK)) )
                     status = OK ;
                  else if ( Info.ctrlIndex == vfcCANCEL )
                     userCancel = true ;
                  done = true ;
               }
            }
            //* If focus is currently on a Radio Button *
            else if ( ic[icIndex].type == dctRADIOBUTTON )
            {
               if ( Info.viaHotkey )
                  Info.HotData2Primary () ;
               else
                  icIndex = dp->EditRadiobutton ( Info ) ;
               //* Get the line-number flag.                   *
               //* (this control is not a member of the group) *
               if ( icIndex == vfcLNUM )
                  lineNums = Info.h_isSel ;
               else
                  viewFormat = Info.selMember ; // 'selected' button of group
            }
            if ( ! done && ! Info.viaHotkey )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }  // while()

      }
      else              // dialog did not open (probably memory allocation error)
      {
         this->DisplayStatus ( ecMEMALLOC ) ;
      }

      if ( dp != NULL )                         // close the window
         delete ( dp ) ;

      //* Restore display of parent dialog window *
      this->dPtr->RefreshWin () ;

      //* If user has made a viewing choice *
      if ( status == OK )
      {
#if DEBUG_base64Decode != 0
gString db64Path ;
this->fmPtr->CreateTempname ( db64Path ) ;
dbgofs.open( db64Path.ustr(), (ofstream::out | ofstream::trunc) ) ;
if ( dbgofs.is_open() )
   dbgofs << "Debugging Info for \"deDecode_base64\"\n"
             "====================================\n" << endl ;
#endif   // DEBUG_base64Decode
         //* Escape any "special characters" in the target pathspec.*
         this->fmPtr->EscapeSpecialChars ( trgPath, escMIN ) ;

         gString  cmdBuff ;
         switch ( viewFormat )
         {
            case vfcHEXB:           // view as hexidecimal (byte)
               cmdBuff.compose( L"hexdump -Cv \"%S\" | less -c", trgPath.gstr() ) ;
               break ;
            case vfcHEXW:           // view as hexidecimal (word)
               cmdBuff.compose( L"hexdump -xv \"%S\" | less -c", trgPath.gstr() ) ;
               break ;
            case vfcDECI:           // view as decimal
               cmdBuff.compose( L"hexdump -dv \"%S\" | less -c", trgPath.gstr() ) ;
               break ;
            case vfcOCT:            // view as octal
               cmdBuff.compose( L"hexdump -bv \"%S\" | less -c", trgPath.gstr() ) ;
               break ;
            case vfcBIN:            // view as binary
               //* If successful, path of decoded data is returned in trgPath.*
               if ( (status = this->vfcDecodeBinary ( origPath, trgPath )) == OK )
                  cmdBuff.compose( L"less -c \"%S\"", trgPath.gstr() ) ;
               break ;
            case vfcARCH:           // view archive contents
               //* If successful, path of decoded data is returned in trgPath.*
               if ( (status = this->vfcDecodeArchive ( origPath, trgPath )) == OK )
                  cmdBuff.compose( L"less -c \"%S\"", trgPath.gstr() ) ;
               break ;
            case vfcLINK:           // view symbolic link file
               //* If successful, path of decoded data is returned in trgPath.*
               if ( (status = this->vfcDecodeSymlink ( origPath, trgPath )) == OK )
                  cmdBuff.compose( L"less -c \"%S\"", trgPath.gstr() ) ;
               break ;
            case vfcMETA:           // view media metadata
               //* If successful, path of decoded data is returned in trgPath.*
               if ( (status = this->vfcDecodeMetadata ( origPath, trgPath, mfType )) == OK )
                  cmdBuff.compose( L"less -c \"%S\"", trgPath.gstr() ) ;
               break ;
            case vfcEML:            // view ".eml" contents
               if ( (status = this->vfcDecodeEmail ( origPath, trgPath )) == OK )
               {
                  cmdBuff.compose( L"less -R -c \"%S\"", trgPath.gstr() ) ;
                  //ansiColor = true ;
               }
               break ;
            case vfcTEXT:           // view as text
            default:
               {
                  //* For HTML and XML source code, add ANSI colorization.*
                  //* (We include "backup" copies of html files.)         *
                  short scode = ZERO ;
                  if ( (trgPath.find( ".xml" )) == (trgPath.gschars() - 5) )
                     scode = 1 ;
                  else if ( ((trgPath.find( ".html" )) == (trgPath.gschars() - 6)) ||
                            ((trgPath.find( ".htm" )) == (trgPath.gschars() - 5)) ||
                            ((trgPath.find( ".html~" )) == (trgPath.gschars() - 7)) )
                     scode = 2 ;
                  // Programmer's Note: For efficiency, perform 'strict' test only 
                  // for small files on the assumption that Perl source (and other 
                  // script files) will be small, while binary executables will 
                  // almost always be larger than this threshold.
                  else if ( (vfcIsPerlSource ( trgPath, bool(fn.fBytes <= 50000) )) )
                  {
                     scode = 3 ;
                  }
                  if ( scode > ZERO )
                  {
                     gString tmpPath ;
                     this->fmPtr->CreateTempname ( tmpPath ) ;
                     bool success = false ;
                     if ( scode == 1 )
                        success = this->vfcFormatXML_Source ( origPath, tmpPath ) ;
                     else if ( scode == 2 )
                        success = this->vfcFormatHTML_Source ( origPath, tmpPath ) ;
                     else if ( scode == 3 )
                        success = this->vfcFormatPERL_Source ( origPath, tmpPath ) ;
                     if ( success )
                     {
                        trgPath = tmpPath ;
                        fmtSrcCode = ansiColor = true ;
                     }
                  }

                  //* If target is a supported OpenDocument or   *
                  //* OpenXML file, capture and decode the text  *
                  //* data from the container archive.           *
                  else if ( (aType == atODOC) || (aType == atOXML) )
                     this->vfcDecodeOpenDoc ( origPath, trgPath, aType ) ;

                  //* If old-style binary document format *
                  //* capture and decode the text data.   *
                  else if ( isOB_File )
                  {
                     //* Be sure trgPath is a null string so *
                     //* temp file will be created by call.  *
                     trgPath.clear() ;
                     this->vfcDecodeBinDoc ( origPath, trgPath ) ;
                  }

                  //* If a Texinfo (.info) document, view as plain text.*
                  else if ( isInfodoc )
                  { /* nothing to do at this time */ }

                  else
                  {
                     //* If user is trying to view a binary file as if *
                     //* it were plain text, view as hex instead.      *
                     if ( !(this->vfcIsTextfile ( trgPath )) )
                     {
                        cmdBuff.compose( L"hexdump -Cv \"%S\" | less -c", trgPath.gstr() ) ;
                        viewFormat = vfcHEXB ;
                        break ;
                     }

                     //* For other plain text files, enable decoding *
                     //* of ANSI color escape sequences (if present).*
                     //* Currently, viewing the unprocessed ANSI     *
                     //* codes as plain text is not supported.       *
                     ansiColor = true ;
                  }

                  //* If line-number control is set, but the output     *
                  //* format is not text, then disallow line numbers.   *
                  if ( lineNums && ((viewFormat != vfcTEXT) || isOB_File ||
                       (aType == atODOC) || (aType == atOXML)) )
                  { lineNums = false ; }

               }

               cmdBuff.compose( L"less -c \"%S\"", trgPath.gstr() ) ;

               //* If specified, add the "-R" parameter to 'less' *
               //* invocation, instructing it to interpret the    *
               //* ANSI color sequences rather that displaying    *
               //* them as ordinary text.                         *
               if ( ansiColor )
                  cmdBuff.insert( L" -R", 4 ) ;

               //* If specified, enable line numbers. Line numbers   *
               //* are valid for plain text, XML, HTML, or PERL.     *
               if ( lineNums )
                  cmdBuff.insert( L" -N", 4 ) ;

               break ;
         }

         if ( status == OK )
         {
            // Programmer's Note: If user exits less with a CTRL+C, then the 
            // return value shows error, even though there may be no error.
            status = this->dPtr->ShellOut ( soX, cmdBuff.ustr() ) ;
         }

         #if DEBUG_base64Decode != 0
         if ( dbgofs.is_open() )
            dbgofs.close() ;
         this->fmPtr->DeleteTempname ( db64Path ) ;
         #endif   // DEBUG_base64Decode
      }
      if ( status != OK && userCancel == false )
      {
         this->dPtr->RefreshWin () ;
         const char* sorryMsg[] = 
         {
            "  ALERT  ",
            " ",
            "Sorry, unable to view contents of specified file.",
            NULL
         } ;
         this->InfoDialog ( sorryMsg ) ;
      }

      //* If a temp file was created, delete it now.*
      fmFType ft ;
      if ( (viewFormat == vfcLINK || viewFormat == vfcBIN || viewFormat == vfcMETA ||
            viewFormat == vfcARCH || isOD_File || isOX_File || fmtSrcCode ||
            (isOB_File && viewFormat == vfcTEXT))
           && (TargetExists ( trgPath, ft )) )
      {
         this->fmPtr->DeleteTempname ( trgPath ) ;
      }
   }
   return status ;

}  //* End ViewFileContents() *

//*************************
//*   vfcDecodeSymlink    *
//*************************
//******************************************************************************
//* Read and format the contents of a symbolic link file, then write it as a   *
//* plain text temporary file. Called only by ViewFileContents() method.       *
//*                                                                            *
//* Input  : trgPath : path/filename of target                                 *
//*          tmpPath : receives path/filename of temporary file to be displayed*
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if file not found or unable to read file                      *
//******************************************************************************

short FileDlg::vfcDecodeSymlink ( const gString& trgPath, gString& tmpPath )
{
   const short hexCOLS = 49 ;
   short status = ERR ;

   tnFName slStats ;
   if ( (this->fmPtr->GetFileStats ( slStats, trgPath )) == OK )
   {
      UINT fSize = (UINT)slStats.fBytes ; // (narrowing, but OK)
      char slBuff[fSize + 2] ;
      unsigned slBytesIn = readlink ( trgPath.ustr(), slBuff, fSize ) ;
      //* If we have data, get a unique temp-file filespec.*
      if ( slBytesIn > ZERO &&
           ((this->fmPtr->CreateTempname ( tmpPath )) != false) )
      {
         ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
         if ( ofs.is_open() )
         {
            gString gsOut ;            // formatted output
            this->fmPtr->ExtractFilename ( gsOut, trgPath ) ;
            ofs << "Contents of Symbolic link file: '"
                << gsOut.ustr() << "'\n" << endl ;

            char hxBuff[32],           // hex output buffer
                 ccBuff[32] ;          // character output buffer
            unsigned slBytesOut = ZERO,// source index
                  oIndex ;             // output index
            short cols ;               // formatted columns

            while ( slBytesOut < slBytesIn )
            {
               cols = ZERO ;
               //* Initialize the output buffers *
               for ( oIndex = ZERO ; oIndex < 16 ; oIndex++ )
               {
                  hxBuff[oIndex] = ZERO ;
                  ccBuff[oIndex] = ' ' ;
               }
               //* Format and output a line of data *
               for ( short i = ZERO ; (i < 16) && (slBytesOut < slBytesIn) ; i++ )
               {
                  hxBuff[i] = slBuff[slBytesOut] ;
                  if ( isprint ( slBuff[slBytesOut] ) )
                     ccBuff[i] = slBuff[slBytesOut] ;
                  else
                     ccBuff[i] = '.' ;
                  ++slBytesOut ;
                  cols += 3 ;
                  if ( i == 8 )
                     cols += 2 ;
               }
               gsOut.compose( L"%02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX - "
                               "%02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX",
                              &hxBuff[ 0], &hxBuff[ 1], &hxBuff[ 2], &hxBuff[ 3], 
                              &hxBuff[ 4], &hxBuff[ 5], &hxBuff[ 6], &hxBuff[ 7],
                              &hxBuff[ 8], &hxBuff[ 9], &hxBuff[10], &hxBuff[11],
                              &hxBuff[12], &hxBuff[13], &hxBuff[14], &hxBuff[15] ) ;
               if ( cols < hexCOLS )      // partial (last) output line
               {
                  gsOut.limitCols( cols ) ;
                  gsOut.append( "                                               " ) ;
                  gsOut.limitCols( hexCOLS ) ;
               }
               ofs << gsOut.ustr() ;
               gsOut.compose( L" [%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c]",
                              &ccBuff[ 0], &ccBuff[ 1], &ccBuff[ 2], &ccBuff[ 3], 
                              &ccBuff[ 4], &ccBuff[ 5], &ccBuff[ 6], &ccBuff[ 7],
                              &ccBuff[ 8], &ccBuff[ 9], &ccBuff[10], &ccBuff[11],
                              &ccBuff[12], &ccBuff[13], &ccBuff[14], &ccBuff[15] ) ;
               ofs << gsOut.ustr() << endl ;
            }
            ofs << endl ;

            ofs.close() ;
            status = OK ;
         }
      }
   }
   return status ;

}  //* End vfcDecodeSymlink() *

//*************************
//*    vfcDecodeBinary    *
//*************************
//******************************************************************************
//* Read and format the contents of the target as binary output. Write the     *
//* results as a plain text temporary file. Called only by ViewFileContents(). *
//*                                                                            *
//* Input  : trgPath : path/filename of target                                 *
//*          tmpPath : receives path/filename of temporary file to be displayed*
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if file not found or unable to read file                      *
//******************************************************************************

short FileDlg::vfcDecodeBinary ( const gString& trgPath, gString& tmpPath )
{
   const char* binHeader = 
   "OFFSET BINARY BYTES                                                     ASCII\n"
   "====== ==============================================================  =======" ;
   short status = ERR ;

   //* Get a unique temp-file filespec *
   if ( (this->fmPtr->CreateTempname ( tmpPath )) != false )
   {
      //* Open the temp file for writing *
      ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         gString gsOut ;            // formatted output
         ofs << binHeader << endl ; // write the header

         //* Open the target file for reading *
         ifstream ifs ( trgPath.ustr(), ifstream::in | ifstream::binary ) ;
         if ( ifs.is_open() )
         {  //* Read the file in 7-byte chunks until EOF. *
            // Note: chunk size is chosen to keep output within 80 columns.
            ULONG readCNT = 7, cnt, byteTotal = ZERO ;
            char inBuff[readCNT * 2] ;
            const char p = '.' ;    // placeholder character
            bool done = false ;
            while ( ! done )
            {
               ifs.read ( inBuff, readCNT ) ;
               if ( (cnt = ifs.gcount()) == readCNT )
               {
                  gsOut.compose(
                     L"%06lu "
                      "%.8hhb %.8hhb %.8hhb %.8hhb %.8hhb %.8hhb %.8hhb "
                      "|%c%c%c%c%c%c%c|",
                     &byteTotal,
                     &inBuff[0], &inBuff[1], &inBuff[2], &inBuff[3], 
                     &inBuff[4], &inBuff[5], &inBuff[6],  
                     (((inBuff[0] >= SPACE && inBuff[0] <= TILDE)) ? &inBuff[0] : &p),
                     (((inBuff[1] >= SPACE && inBuff[1] <= TILDE)) ? &inBuff[1] : &p),
                     (((inBuff[2] >= SPACE && inBuff[2] <= TILDE)) ? &inBuff[2] : &p),
                     (((inBuff[3] >= SPACE && inBuff[3] <= TILDE)) ? &inBuff[3] : &p),
                     (((inBuff[4] >= SPACE && inBuff[4] <= TILDE)) ? &inBuff[4] : &p),
                     (((inBuff[5] >= SPACE && inBuff[5] <= TILDE)) ? &inBuff[5] : &p),
                     (((inBuff[6] >= SPACE && inBuff[6] <= TILDE)) ? &inBuff[6] : &p) ) ;
                  ofs << gsOut.ustr() << endl ;
                  byteTotal += cnt ;
               }
               else if ( cnt > ZERO )
               {  //* Output the partial chunk *
                  gsOut.compose(
                     L"%06lu -------- -------- -------- -------- -------- -------- -------- |",
                     &byteTotal ) ;
                  gString gsBin ;
                  wchar_t  a ;
                  ULONG indx ;
                  for ( indx = ZERO ; indx < cnt ; ++indx )
                  {  //* Format binary byte *
                     gsBin.compose( L"%.8hhb", &inBuff[indx] ) ;
                     gsOut.replace( L"--------", gsBin.gstr() ) ;
                     //* Format ASCII equivalent *
                     a = (inBuff[indx] >= SPACE && inBuff[indx] <= TILDE) ? inBuff[indx] : p ;
                     gsOut.append( a ) ;
                  }
                  //* Pad until line is complete *
                  a = p ;
                  while ( indx++ < readCNT )
                     gsOut.append( a ) ;
                  gsOut.append( L'|' ) ;
                  ofs << gsOut.ustr() << endl ;
               }
               if ( cnt < readCNT )
                  done = true ;
            }
            ifs.close () ;          // close the target file
            status = OK ;           // return the good news
         }
         ofs.close() ;
      }
   }
   return status ;

}  //* End vfcDecodeBinary() *

//*************************
//*   vfcDecodeArchive    *
//*************************
//********************************************************************************
//* Read and format the contents of:                                             *
//*   a) tar archive file,                                                       *
//*   b) zip archive file,                                                       *
//*   c) OpenDocument and OpenXML files (special case of 'zip' archive)          *
//*   d) ePub document (special case of 'zip' archive)                           *
//* then write it as a plain text temporary file.                                *
//* Called only by ViewFileContents() method.                                    *
//*                                                                              *
//* Input  : trgPath : path/filename of target                                   *
//*          tmpPath : receives path/filename of temporary file to be displayed  *
//*                                                                              *
//* Returns: OK if successful,                                                   *
//*            Note that if the target is not a valid archive, then the          *
//*            appropriate error message will be written to the temp file, so    *
//*            this will not be reported to caller as an error.                  *
//*          ERR if file not found or unable to read file                        *
//********************************************************************************

short FileDlg::vfcDecodeArchive ( const gString& trgPath, gString& tmpPath )
{
   short status = ERR ;

   if ( (this->fmPtr->CreateTempname ( tmpPath )) != false )
   {
      ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         gString gsOut ;            // formatted output
         this->fmPtr->ExtractFilename ( gsOut, trgPath ) ;
         ofs << "Contents of archive file: '"
             << gsOut.ustr() 
             << "'\n========================================"
                "====================" << endl ;
         ofs.close() ;
         char cmdBuff[gsDFLTBYTES * 3] ;
         arcType aType = this->ArchiveTarget ( trgPath ) ;

         //* Escape any "special characters" which *
         //* might be misinterpreted by the shell. *
         gString escPath = trgPath ;
         this->EscapeSpecialChars ( escPath, escMIN ) ;

         //* 'zip' archives, ePub documents and OpenDocument/OpenXML files *
         if ( (aType == atZIP) || (aType == atEPUB) || 
              (aType == atODOC) || (aType == atOXML) )
         {
            snprintf ( cmdBuff, (gsDFLTBYTES * 3), 
                       "unzip -lv \"%S\" 1>>\"%S\" 2>>\"%S\"", 
                       escPath.gstr(), tmpPath.gstr(), tmpPath.gstr() ) ;
         }

         //* 'tar' archives *
         else //if ( aType != NONE )
         {
            snprintf ( cmdBuff, (gsDFLTBYTES * 3), 
                       "tar -tvvf \"%S\" 1>>\"%S\" 2>>\"%S\"", 
                       escPath.gstr(), tmpPath.gstr(), tmpPath.gstr() ) ;
         }
         this->fmPtr->Systemcall ( cmdBuff ) ;

         status = OK ;
      }
   }
   return status ;

}  //* End vfcDecodeArchive() *

//*************************
//*   vfcDecodeOpenDoc    *
//*************************
//********************************************************************************
//* Extract the text data from an OpenDocument or OpenXML (MS-Office) file,      *
//* and format it for viewing. Caller has verified that target is a supported    *
//* OpenDocument/OpenXML file, and type is indicated by the 'aType' parameter.   *
//* Called only by ViewFileContents() method.                                    *
//*                                                                              *
//* Input  : srcPath : source filespec                                           *
//*          tmpPath : receives path/filename of temporary file to be displayed  *
//*          aType   : archive type, either atODOC or atOXML                     *
//*                                                                              *
//* Returns: OK if successful,                                                   *
//*          ERR if file not found or unable to decode file                      *
//********************************************************************************

short FileDlg::vfcDecodeOpenDoc ( const gString& srcPath, gString& tmpPath, arcType aType )
{
   gString contentSpec,             // raw XML text extracted to this file
           csPath ;                 // path of application's temp directory
   fmFType fType ;                  // file type code
   short status = ERR ;

   //* Create the target temp file *
   this->fmPtr->CreateTempname ( tmpPath ) ;
   //* Isolate the tempfile path *
   this->fmPtr->ExtractPathname ( csPath, tmpPath ) ;

   //* Extract the XML containing the OpenDocument text data *
   if ( (this->odExtractOD_Content ( srcPath, csPath, contentSpec, aType )) &&
        (this->TargetExists ( tmpPath, fType )) )
   {
      //* Scan the XML source, extract and format the text data.*
      if ( (this->odExtractOD_Text ( srcPath, contentSpec, tmpPath, aType )) != false )
         status = OK ;
   }
   this->DeleteFile ( contentSpec ) ;  // delete the XML source file

   return status ;

}  //* End vfcDecodeOpenDoc() *

//******************************************************************************
//* Formatted storage for the obsolete binary-layout Microsoft Word (.doc)     *
//* "File Identification Block" format.                                        *
//* Used to decode Word1997, Word2000, Word2002, Word2003, Word2007.           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//*                                                                            *
//*   Editorial:                                                               *
//*   I have never seen a more convoluted and unplanned pile of                *
//*   mediocrity than the "Word (.doc) Binary File Format" specification.      *
//*   Even by Microsoft's standards, this was an awful layout. A 6-year-old    *
//*   child puts more thought into a finger-painting project. -- Sam           *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*                                                                            *
//* A) Unless otherwise noted, all integer data are signed (positive)          *
//*    little-endian values. CPs are unsigned 32-bit integers.                 *
//* B) The File Information Block (FIB) gives the locations (offsets) to all   *
//*    other objects in the file.                                              *
//* C) The FibBase structure is 32 bytes.                                      *
//*    16 bits  wIdent == 0xA5EC (little-endian): ec a5 structure ID           *
//*    16 bits  nFib   == 0x00C1 (little-endian): file-format version          *
//*                       This value is obsolete. See FibRgCswNew (nFibNew)    *
//*    16 bits  reserved                                                       *
//*    16 bits  lid       language ID                                          *
//*    16 bits  pnNext    offset to AutoText items                             *
//*                       (if 0x0000, then no autotext)                        *
//*                       (if non-zero, FIB offset is pnNext+512 bytes)        *
//*    16 bits  bit field                                                      *
//*    16 bits  nFibBack reserved (0x00BF, or sometimes 0x00C1)                *
//*    32 bits  IKey    Encryption key size                                    *
//*     8 bits  reserved (0x00)                                                *
//*     8 bits  bit field                                                      *
//*    16 bits  reserved (0x0000)                                              *
//*    16 bits  reserved (0x0000)                                              *
//*    32 bits  reserved                                                       *
//*    32 bits  reserved                                                       *
//*                                                                            *
//* D) csw - 16 bits (must be 0x000E) count of 16-bit values in fibRgW         *
//* E) fibRgW 28 bytes (this is FibRgW97)                                      *
//*      reserved values: 15 16-bit values, and                                *
//*      lidFE  language code                                                  *
//* F) cslw - 16 bits (must be 0x0016) count of 32-bit values in fibRgLW       *
//* G) fibRgLw 88 bytes (this is FibRgLw97)                                    *
//*    The FibRgLw97 structure contains an array of 32-bit values indicating   *
//*    the sizes of the various structures.                                    *
//*       cbMac     size of the WordDocument stream (any additional data after *
//*                 this offset must be ignored).                              *
//*       reserved                                                             *
//*       reserved                                                             *
//*       ccpText   number of Character Points (CP) in main document           *
//*       ccpFtn    number of CPs in footnote subdocument                      *
//*       ccpHdd    number of CPs in header subdocument                        *
//*       reserved                                                             *
//*       ccpAtn    number of CPs in comment subdocument                       *
//*       ccpEdn    number of CPs in endnote subdocument                       *
//*       ccpTxbx   number of CPs in textbox subdocument                       *
//*       reserved  11 x 32 bits                                               *
//* H) cbRgFcLcb - 16 bits count of 64-bit values in fibRgFcLcbBlob            *
//*      if nFib==0x00C1, 0x005D                                               *
//*      if nFib==0x00D9, 0x006C                                               *
//*      if nFib==0x0101, 0x0088                                               *
//*      if nFib==0x010C, 0x00A4                                               *
//*      if nFib==0x0112, 0x00B7                                               *
//* I) fibRgFcLcbBlob (FibRgFcLcb)                                             *
//*    The FibRgFcLcb structure indicates the offsets and bytes counts of "Clx"*
//*    control structures in the WordDocument Stream. In turn, the Clx         *
//*    structure gives the location of various sections of the data.           *
//*    Each document version has a different FibRgFcLcb version. The version   *
//*    is indicated by the value in FIB::nFib                                  *
//*      0x00C1  fibRgFcLcb97         0x010C  fibRgFcLcb2003                   *
//*      0x00D9  fibRgFcLcb2000       0x0112  fibRgFcLcb2007                   *
//*      0x0101  fibRgFcLcb2002                                                *
//*    Each version after fibRgFcLcb97 _contains_ the previous version as the  *
//*    first element. This means that the fibRgFcLcb97 will always be first,   *
//*    with any newer-version extensions appended.                             *
//*      Each entry is organized as a pair of values:                          *
//*      fcClx   offset of a "Clx" in the Table Stream                         *
//*      lcbClx  size (in bytes) of the "Clx"                                  *
//*      Read the Clx from the Table Stream. (see below for Clx definition)    *
//* J) cswNew 16 bits count of 16-bit values in fibRgCswNew                    *
//*      if cswNew == 0x0000, then 'nFib' (above) contains file-format version.*
//*      if cswNew >  0x0000  FibRgCswNew.nFibNew contains the version.        *
//* K) fibRgCswNew (if present)                                                *
//*      cswNew   nFibNew  rgCswNewData                                        *
//*      -------  -------  ------------                                        *
//*      0x0000       (no data)                                                *
//*      0x0002   0x00D9   1 16-bit word                                       *
//*      0x0002   0x0101   1 16-bit words                                      *
//*      0x0002   0x010C   1 16-bit words                                      *
//*      0x0005   0x0112   4 16-bit words                                      *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* ?) Clx: (target of an FibRgFcLcb record)                                   *
//*    The Clx contains an array of Prc objects, followed by a Pcdt object.    *
//*    RgPrc  array of Prc objects (if empty, the first byte == 0x02)          *
//*      Prc structure:                                                        *
//*      clxt (1 byte, must be 0x01)                                           *
//*      data : a PrcData structure that contains a set of properties          *
//*         A PrcData structure is an array of Prl elements and the size of    *
//*         the array.                                                         *
//*         cbGrpPrl (16 bits) size (in bytes) of GrpPrl array                 *
//*         GrpPrl   an array of Prl elements                                  *
//*            A Prl is a property followed by an operand:                     *
//*            sprm    16 bits - property ID (Single Property Modifier)        *
//*            operand 16 bits - operand for the sprm                          *
//*    Pcdt:                                                                   *
//*       clxt     8 bits  (must be 0x02)                                      *
//*       lcb     32 bits  size (bytes) of PicPcd                              *
//*       PicPcd  (varies) PicPcd structure (this is a PLC structure)          *
//*          PicPcd: This is a PLC structure consisting of an array of CPs     *
//*          that specify the starting points for text ranges.                 *
//*          The CP array is followed by an array of Pcd objects.              *
//*          (Note: Number of CPs is one more than number of PicPcd objects.)  *
//*          -- A CP is an unsigned 32-bit integer, and acts as a zero-based   *
//*             index of a "character" in the document.                        *
//*          -- A Pcd structure specifies the location of text and additional  *
//*             text properties.                                               *
//*          16 bits  bit field:                                               *
//*             bit 00    fNoParaLast - if set, text has no paragraph mark     *
//*             bit 01    fR1         - reserved                               *
//*             bit 02    fDirty      - reserved (always false)                *
//*             bit 03-15 fR2         - reserved (13 bits)                     *
//*          32 bits      fc          - an 'FcCompressed' structure            *
//*             The FcCompressed structure specifies the location of the text  *
//*             and whether it is 8-Bit ANSI (presumably code-page 1252) text  *
//*             or UTF-16LE (no BOM) text.                                     *
//*             bit 0-29 fc (30 bits)  unsigned integer - offset into          *
//*                                    WordDocument Stream.                    *
//*             bit 30   A  (1 bit) 0 == text is UTF-16 starting at 'fc'       *
//*                                 1 == text is 8-bit ANSI starting at 'fc/2' *
//*                                      Under 8-bit ANSI, certain characters  *
//*                                      (0x82-0x9F) should be decoded as      *
//*                                      UTF-16 according to a table lookup.   *
//*             bit 31   B  (1 bit)    reserved (0)                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*          16 bits      prm         - a 'Prm' structure                      *
//*          -- A Prm structure is one of two types, indicated by bit 00       *
//*             bit 00 fComplex - 1: Prm1 structure                            *
//*                                  bit 01-15 igrpprl (index of Prc) unsigned *
//*                               0: Prm0 structure                            *
//*                                  bit 01-08 isprm (see table of values)     *
//*                                  bit 08-15 val (operand for isprm)         *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* The "WordDocument Stream" begins with an FIB at offset 0x0000.             *
//*   Scan for the FIB ID marker: 0xA5EC : EC A5                               *
//*                                                                            *
//*                                                                            *
//* -- CarriageReturns (0x0D) indicate end-of-line. (no linefeeds present)     *
//* -- Tab characters (0x09) are also present.                                 *
//* -- Other control character include:                                        *
//*    -- 0x13                                                                 *
//*    -- 0x030D combination (0d 03...)                                        *
//*    -- 0x040D combination (0d 04...)                                        *
//* -- A "Paragraph Mark" is a UTF-16 CarriageReturn: 0x000D (0d 00).          *
//*    It signals the end of the block or segment.                             *
//*    A single Paragraph Mark signals the end-of-block and is not part of     *
//*    the block.                                                              *
//*    If sequential Paragraph Marks, (UTF-16) the final one is the            *
//*    end-of-block marker, and it is not part of the block.                   *
//*                                                                            *
//* The FIB often begins at offset 000200.                                     *
//* Text _often_ (not always) begins at offset 0x000a00.                       *
//* The "Main Document begins at offset 0x0000 and is FibRgLw97.ccpText        *
//*   characters in length. (Characters are not necessarily 8 bits.)           *
//*                                                                            *
//* From the Specification: 2.4.1 Retrieving Text                              *
//* ----------------------                                                     *
//* The following algorithm specifies how to find the text at a particular     *
//* character position (cp). Negative character positions are not valid.       *
//*                                                                            *
//* 1. Read the FIB from offset zero in the WordDocument Stream.               *
//* 2. All versions of the FIB contain exactly one FibRgFcLcb97, though it     *
//*    can be nested in a larger structure. FibRgFcLcb97.fcClx specifies the   *
//*    offset in the Table Stream of a Clx. FibRgFcLcb97.lcbClx specifies the  *
//*    size, in bytes, of that Clx. Read the Clx from the Table Stream.        *
//* 3. The Clx contains a Pcdt, and the Pcdt contains a PlcPcd. Find the       *
//*    largest 'i' such that PlcPcd.aCp[i] ≤ cp.                               *
//*    As with all Plcs, the elements of PlcPcd.aCp are sorted in ascending    *
//*    order. Recall from the definition of a Plc that the aCp array has one   *
//*    more element than the aPcd array. Thus, if the last element of          *
//*    PlcPcd.aCp is less than or equal to cp, cp is outside the range of      *
//*    valid character positions in this document.                             *
//* 4. PlcPcd.aPcd[i] is a Pcd. Pcd.fc is an FcCompressed that specifies the   *
//*    location in the WordDocument Stream of the text at character position   *
//*    PlcPcd.aCp[i].                                                          *
//* 5. If FcCompressed.fCompressed is zero, the character at position cp is    *
//*    a 16-bit Unicode character at offset                                    *
//*              FcCompressed.fc + 2(cp - PlcPcd.aCp[i])                       *
//*    in the WordDocument Stream. This is to say that the text at character   *
//*    position PlcPcd.aCP[i] begins at offset FcCompressed.fc in the          *
//*    WordDocument Stream and each character occupies two bytes.              *
//* 6. If FcCompressed.fCompressed is 1, the character at position cp is an    *
//*    8-bit ANSI character at offset                                          *
//*              (FcCompressed.fc / 2) + (cp - PlcPcd.aCp[i])                  *
//*    in the WordDocument Stream, unless it is one of the special values in   *
//*    the table defined in the description of FcCompressed.fc. This is to     *
//*    say that the text at character position PlcPcd.aCP[i] begins at offset  *
//*    FcCompressed.fc / 2 in the WordDocument Stream and each character       *
//*    occupies one byte.                                                      *
//******************************************************************************

//*** Mystery structure ***
//* This is interesting information, but currently we don't know what it means.*
//* The algorithm to collect the information is fully functional, but enclosed *
//* within the conditional compilation for debugging only.                     *
class Mystery
{
   public:
   Mystery ( void )
   { this->reset () ; }
   void reset ( void )
   {
      this->name[0] = '\0' ;
      this->sVal1 = this->sVal2 = 0 ;
      for ( short i = ZERO ; i < 16 ; ++i )
         this->iVal[i] = 0 ;
      this->idByte = 0x00 ;
   }
   char     name[64] ;
   uint16_t sVal1 ;
   uint16_t sVal2 ;
   uint32_t iVal[16] ;
   uint8_t  idByte ;
} ;

class FibRgFcLcb     // An offset/size record in the FibRgFcLcb array
{                    // This record includes the referenced data structure
   public:
   FibRgFcLcb ( void )        // constructor
   {
      this->offset = ZERO ;
      this->bytes  = ZERO ;
      this->index  = ZERO ;
   }
   int32_t  offset ;          // offset into WordDocument Stream
   int32_t  bytes ;           // size of object at the offset
   short    index ;           // index into the FibRgFcLcb array
} ;

class FileID_Block
{
   public:
   ~FileID_Block ( void )     // destructor
   {
      if ( this->lcb != NULL )
      { delete [] this->lcb ; this->lcb = NULL ; }
   }
   FileID_Block ( void )      // default constructor
   {
      this->reset () ;
   }

   //***********************************************
   //* Allocate the 'lcb' array based on the value *
   //* of the 'cbRgFcLcb' member.                  *
   //* Input  : none                               *
   //* Returns: 'true'  if successful              *
   //*          'false' if array already allocated *
   //*                  or if cbRgFcLcb == ZERO    *
   //***********************************************
   bool allocateLcb ( void )
   {
      bool status = false ;
      if ( (this->lcb == NULL) && (this->cbRgFcLcb > ZERO) )
      {
         this->lcb = new FibRgFcLcb[this->cbRgFcLcb] ;
         status = true ;
      }
      return status ;
   }

   //***************************************************
   //* Sort the lcb records by offset.                 *
   //***************************************************
   void sort_lcb ( void )
   {
      FibRgFcLcb saveLow, saveHigh ;   // temp buffers
      int lIndex, gIndex,              // low and high limits search indices
          i ;

      for ( lIndex = 0, gIndex = this->cbRgFcLcb-1 ; 
                                 (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( this->lcb[i].offset < this->lcb[lIndex].offset )
            {
               saveLow = this->lcb[lIndex] ;
               this->lcb[lIndex] = this->lcb[i] ;
               this->lcb[i] = saveLow ;
            }
               
            if ( i < gIndex )
            {
               if ( this->lcb[gIndex].offset < this->lcb[i].offset )
               {
                  saveHigh = this->lcb[gIndex] ;
                  this->lcb[gIndex] = this->lcb[i] ;
                  this->lcb[i] = saveHigh ;
                  
                  if ( this->lcb[i].offset < this->lcb[lIndex].offset )
                  {
                     saveLow = this->lcb[lIndex] ;
                     this->lcb[lIndex] = this->lcb[i] ;
                     this->lcb[i] = saveLow ;
                  }
               }
            }
         }
      }
   }  //* End sort_lcb() *

   //***************************************************
   //* Convert an 8-bit ANSI string to gString format. *
   //* - Control characters are captured and converted *
   //*   to newlines or spaces.                        *
   //* - Supra-ASCII characters are converted according*
   //*   to the lookup table defined in the Microsloth *
   //*   specification.                                *
   //* Returns number of wchar_t characters in gsTrg.  *
   //***************************************************
   short ansi8Decode ( const char* rawPtr, short rawBytes, gString& gsTrg )
   {
      //* Lookup table for "special" 8-bit values *
      const short lookupCount = 24 ;
      uint8_t ansiByte[lookupCount] = 
      {
         0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 
         0x8A, 0x8B, 0x8C, 0x91, 0x92, 0x93, 0x94, 0x95, 
         0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9F, 
      } ;
      uint16_t utf16Word[lookupCount] = 
      {
         0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, 0x02C6, 0x2030, 
         0x0160, 0x2039, 0x0152, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 
         0x2013, 0x2014, 0x02DC, 0x2122, 0x0161, 0x203A, 0x0153, 0x0178, 
      } ;

      gsTrg.clear() ;                  // initialize the target buffer
      uint32_t wChar ;                 // 'wide' character
      short crCnt = ZERO ;             // count of sequential CR characters

      for ( short ti = ZERO ; ti < rawBytes ;  ++ti )
      {
         if ( rawPtr[ti] >= ' ' && rawPtr[ti] <= '~' )// ASCII characters
         {
            wChar = uint32_t(rawPtr[ti]) ;
            gsTrg.append( wChar ) ;
            crCnt = ZERO ;
         }
         else if ( rawPtr[ti] < ' ' )                 // control characters
         {
            if ( rawPtr[ti] == 0x0D )
            {
               if ( crCnt < 2 )
                  wChar = L'\n' ;
               else
                  wChar = L' ' ;
               gsTrg.append( wChar ) ;
               ++crCnt ;
               if ( rawPtr[ti + 1] == 0x03 || rawPtr[ti + 1] == 04 )
               {
                  gsTrg.append( L' ' ) ;
                  ++ti ;
               }
               else if ( rawPtr[ti + 1] == 0x00 )  // 0x000D is end-of-section marker
               {
                  // gsTrg.append( L"\n>>\n" ) ;
               }
            }
            else
            {
               crCnt = ZERO ;
               if ( rawPtr[ti] == 0x00 )        // null char ends ANSI stream
                  break ;
               else if ( rawPtr[ti] == 0x13 )   // paragraph mark (XOFF)
                  wChar = L'\n' ;
               else                             // other control characters
                  wChar = L' ' ;
               gsTrg.append( wChar ) ;
            }
         }
         else        // supra-ASCII characters (> 0x7F)
         {
            crCnt = ZERO ;
            wChar = L' ' ;       // if an invalid code, convert to space
            for ( short i = ZERO ; i < lookupCount ; ++i )
            {
               if ( rawPtr[ti] == ansiByte[i] )
               {
                  wChar = wchar_t(utf16Word[i]) ;
                  break ;
               }
            }
            gsTrg.append( wChar ) ;
         }
      }
      return gsTrg.gschars() ;

   }  //* End ansi8Decode() *

   //***************************************************
   //* Convert a Unicode-16 string (little-endian,     *
   //* no byte-order mark) to gString format.          *
   //* Source may or may not be null terminated.       *
   //* Returns number of wchar_t characters in gsTrg.  *
   //***************************************************
   short utf16Decode ( const char* rawPtr, short rawBytes, gString& gsTrg )
   {
      const UINT usxMIN20  = 0x0010000 ;  // minimum value requiring 20-bit conversion
      //const UINT usxMAX20  = 0x010FFFF ;  // maximum value for 20-bit UTF-16
      const UINT usxMSU16  = 0x000D800 ;  // MS unit mask value
                                          // and beginning of 16-bit reserved sequence
      const UINT usxMSU16e = 0x000DBFF ;  // end of MS unit range
      const UINT usxLSU16  = 0x000DC00 ;  // LS unit mask value
      const UINT usxLSU16e = 0x000DFFF ;  // end of LS unit range
      const UINT usxHIGH16 = 0x00E000 ;   // beginning of high-range 16-bit codepoints
      const UINT usxMAX16  = 0x00FFFD ;   // maximum value for 16-bit codepoint
      const UINT usxMASK10 = 0x0003FF ;   // 10-bit mask

      int ti = (*rawPtr > 0x00) ? ZERO : 1 ; // source index (ignore any leading null)
      gsTrg.clear() ;                        // initialize the target buffer
      uint32_t msUnit, lsUnit,               // for converting unit pairs
               cx ;                          // decoded 16-bit character
      short crCnt = ZERO ;                   // count of sequential CR characters

      // Programmer's Note:
      // The UTF-16 encoding used in MS-Word documents is somewhat different from 
      // standard UTF-16. Specifically, certain character sequences 
      //      (0d 00 13 00 , 0d 00 03 00 , 0d 00 04 00 and others)
      // are used for end-of-section, end-of-cell, end-of-row and other markers. 
      // If we simply see these as newline characters, the output formatting would 
      // be seriously compromised because these are not actually part of the 
      // display stream. To maintain a stable character count, we simply convert 
      // these special code sequences to spaces. All control codes except 
      // carriage-return '\r' (0x000D) are converted to spaces, carriage-returns 
      // are converted to Linux newlines ('\n'), and the number of sequential 
      // newlines is limited to preserve output formatting.
      // Note also that UTF-16 null characters (0x0000) are not used in the 
      // document text, so if a null character is encountered, we must assume 
      // that we have overrun the text block.
      while ( rawBytes > ZERO )
      {
         //* Little-endian conversion *
         cx = (uint32_t(rawPtr[ti++]) & 0x000000FF) ;
         cx |= ((uint32_t(rawPtr[ti++]) << 8) & 0x0000FF00) ;
         rawBytes -= 2 ;

         //* If the character is fully represented by 16 bits *
         //* (most characters are)                            *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {
            if ( cx == L'\0' )   // if premature null character, we're done
               break ;
            //* newline character *
            else if ( cx == 0x000D )
            {
               if ( crCnt < 2 )
                  cx = L'\n' ;
               else
                  cx = L' ' ;
               ++crCnt ;
            }
            else if ( cx < L' ' )
               cx = L' ' ;
            gsTrg.append( cx ) ;
         }

         //* Character is represented by 32 bits    *
         //* (20 bits actually used), 16 MSBs first.*
         else
         {
            msUnit = cx ;
            lsUnit = ZERO ;
            //* Little-endian conversion *
            lsUnit = (UINT(rawPtr[ti++]) & 0x000000FF) ;
            lsUnit |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
            rawBytes -= 2 ;

            //* Validate the range *
            if ( ((msUnit >= usxMSU16) && (msUnit <= usxMSU16e))
                 &&
                 ((lsUnit >= usxLSU16) && (lsUnit <= usxLSU16e)) )
            {
               cx = usxMIN20 + ((msUnit & usxMASK10) << 10) ;
               cx |= (lsUnit & usxMASK10) ;
            }
            else                 // invalid UTF-16 codepoint
               cx = L'?' ;
            gsTrg.append( cx ) ; // add the chararacter to output buffer
         }
      }
      return ( gsTrg.gschars() ) ;
   }  //* End utf16Decode() *

   //***********************************************
   //* Decode a 64-bit little-endian value         *
   //* Input  : srcPtr : pointer to binary input   *
   //* Returns: decoded value                      *
   //***********************************************
   uint64_t decode64BitLE ( const char* srcPtr )
   {
      uint64_t qword ;
      short srcindx = ZERO ;
   
      qword  =  ((uint64_t)srcPtr[srcindx++] & 0x00FF) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 24) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 32) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 40) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 48) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 56) ;
      return qword ;
   
   }  // decode64BitLE()

   //***********************************************
   //* Decode a 32-bit little-endian value         *
   //* Input  : srcPtr : pointer to binary input   *
   //* Returns: decoded value                      *
   //***********************************************
   static uint32_t decode32BitLE ( const char* srcPtr )
   {
      uint32_t dword ;
      short srcindx = ZERO ;

      dword  =  ((uint32_t)srcPtr[srcindx++] & 0x00FF) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 24) ;

      return dword ;

   }  // decode32BitLE()

   //***********************************************
   //* Decode a 16-bit little-endian value         *
   //* Input  : srcPtr : pointer to binary input   *
   //* Returns: decoded value                      *
   //***********************************************
   static uint16_t decode16BitLE ( const char* srcPtr )
   {
      uint16_t word ;
      short srcindx = ZERO ;

      word  =  ((uint16_t)srcPtr[srcindx++] & 0x00FF) ;
      word |= (((uint16_t)srcPtr[srcindx++] & 0x00FF) << 8) ;

      return word ;

   }  // decode16BitLE()

   void reset ( void )
   {
      //* Contents of FibBase *
      this->wIdent = this->nFib = this->lid = this->pnNext = this->nFibBack = ZERO ;
      this->IKey = ZERO ;
      this->fFarEast = this->fWhichTblStm = false ;
      this->base16bits = ZERO ;
      this->base8bits = ZERO ;

      //* Contents of FibRgW *
      this->csw = this->lidFE = ZERO ;

      //* Contents of FibRgFcLcb *
      this->cbRgFcLcb = ZERO ;
      this->lcb = NULL ;      // anchor for dynamic allocation

      //* Contents of FibRgLw(97) and extensions *
      this->cbMac  = this->ccpText = this->ccpFtn  = this->ccpHdd =
      this->ccpAtn = this->ccpEdn  = this->ccpTxbx = this->ccpHdrTxbx = ZERO ;

      //* Contents of
      this->cslw = this->cswNew = ZERO ;


      //* Temporary variables *
      this->temp64  = ZERO ;
      this->temp32a = this->temp32b = ZERO ;
      this->temp16  = ZERO ;
   }

   //***********************
   //* Public Data Members *
   //***********************

   //* Contents of FibBase *
   short wIdent ;       // FIB ID code : 0xA5EC (little-endian)
   short nFib ;         // version number (may be replaced by nFibNew)
   short lid ;          // language ID
   short pnNext ;       // offset into source for FIB AutoText (0x0000 if none)
   short nFibBack ;     // s/b 0x00BF (must be 0x0BF or 0x00C1)
   int32_t IKey ;       // encryption key size
   bool  fFarEast ;     // Far-East installation flag
   bool  fWhichTblStm ; // 0TableStream(reset) or 1TableStream(set) flag
   uint16_t base16bits ;// raw bit field
   uint8_t  base8bits ; // raw bit field  (all ignored)

   //* Contents of FibRgW *
   short csw ;          // count of 16-bit values in fibRgW (must be 0x000E)
   short lidFE ;        // language code (ignore if !fFarEast)

   //* Contents of FibRgFcLcb *
   short cbRgFcLcb ;    // number of offset/size records in FibRgFcLcb
   FibRgFcLcb *lcb ;    // pointer to an array of FibRgFcLcb records

   //* Contents of FibRgLw(97) and extensions *
   int32_t cbMac ;      // size of WordDocument stream
   int32_t ccpText ;    // CPs in main documents
   int32_t ccpFtn ;     // CPs in footnote
   int32_t ccpHdd ;     // CPs in header
   int32_t ccpAtn ;     // CPs in comment
   int32_t ccpEdn ;     // CPs in endnote
   int32_t ccpTxbx ;    // CPs in textbox (poilerplate)
   int32_t ccpHdrTxbx ; // CPs in header textbox

   short cslw ;         // number of elements in FibRgLw(97)
   short cswNew ;       // number of 16-bit elements in FibRgCswNew

   int64_t temp64 ;     // 64-bit temp value
   int32_t temp32a,     // 32-bit temp values
           temp32b ;
   int16_t temp16 ;     // 16-bit temp value
} ;

//*************************
//*    vfcDecodeBinDoc    *
//*************************
//********************************************************************************
//* Extract the text data from older versions of MS-Word.                        *
//*  -- Word 1997, 2000, 2002, 2003, 2007                                        *
//* Called only by ViewFileContents() method.                                    *
//*                                                                              *
//* Input  : srcPath : source filespec                                           *
//*          tmpPath : if a null string in entry, 'tmpPath' receives the         *
//*                      path/filename of temporary file to be displayed         *
//*                    if a valid path/filename on entry, use it as temp file    *
//*                                                                              *
//* Returns: OK if successful,                                                   *
//*          ERR if file not found or unable to decode file                      *
//********************************************************************************
//* Programmer's Note: We use brute-force decoding here because the Microsloth   *
//* "specification" appears to be intentionally unclear. The binary document     *
//* format is officially obsolete, this is not a serious problem;  however,      *
//8 there are still a few people who use older versions of Microsloth Office,    *
//* especially among our Chinese teachers and students who don't want to pay for *
// the updated version. (and who can blame them)                                 *
//********************************************************************************

short FileDlg::vfcDecodeBinDoc ( const gString& srcPath, gString& tmpPath )
{
   #define DEBUG_DOD (0)         // for debugging only
   #define DEBUG_DOD_VERBOSE (0)

   const short nFib_1997 = 0x00C1,     // 0x005D cbRgFcLcb elements
               nFib_2000 = 0x00D9,     // 0x006C
               nFib_2002 = 0x0101,     // 0x0088
               nFib_2003 = 0x010C,     // 0x00A4
               nFib_2007 = 0x0112 ;    // 0x00B7

   const short ibSIZE = 32 ;     // size of input buffer (for FIB scan)
   const char  fib01  = 0xEC,    // FIB ID Byte 01
               fib02  = 0xA5 ;   // FIB ID Byte 02

   #if DEBUG_DOD != 0
   gString dbg, dbgi ;
   uint64_t bytesRead = ZERO,    // source bytes read
            bOffset ;            // source offset
   #endif   // DEBUG_DOD

   //* Storage structure for File Identification Block *
   FileID_Block fib ;

   gString gsOut ;               // formatted-text output
   short   gcnt,                 // source bytes read from stream
           status = ERR ;        // return value
   char ibuff[gsDFLTBYTES] ;     // input buffer
   bool fibFound = false,        // 'true' if FIB located
        AnsiText = true,         // assume 8-bit ANSI text
        done = false ;           // loop control

   //* If existing target temp file not specified, create temp file.*
   if ( (tmpPath.gschars()) == 1 )
      this->fmPtr->CreateTempname ( tmpPath ) ;

   ifstream ifs ( srcPath.ustr(), ifstream::in | ifstream::binary ) ;
   ofstream ofs ( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ifs.is_open() && ofs.is_open() )
   {
      #if DEBUG_DOD != 0
      ofs << "=>>vfcDecodeBinDoc()\n=>>-----------------" << endl ;
      #endif   // DEBUG_DOD

      //* Locate the File Information Block (FIB) *
      while ( ! done )
      {
         ifs.read( ibuff, ibSIZE ) ;
         gcnt = ifs.gcount() - 1 ;

         #if DEBUG_DOD != 0
         bOffset = bytesRead ;
         bytesRead += ifs.gcount() ;
         #endif   // DEBUG_DOD

         for ( short i = ZERO ; i < gcnt ; ++i )
         {  // Programmer's Note: This test assumes that the FIB begins on an 
            // even byte boundary. This is true for all .DOC files tested.
            if ( (ibuff[i] == fib01) && (ibuff[i + 1] == fib02) )
            {
               fib.wIdent = fib.decode16BitLE ( ibuff ) ;
               #if DEBUG_DOD != 0
               dbgi.formatInt( bOffset, 7, true ) ;
               dbg.compose( "=>>FIB 0x%04hX found at: %S (0x%06llX)", 
                            &fib.wIdent, dbgi.gstr(), &bOffset ) ;
               ofs << dbg.ustr() << endl ;
               #endif   // DEBUG_DOD

               if ( i != ZERO )        // shift buffer if necessary
               {
                  for ( short trg = ZERO, src = i ; src <= gcnt ; ++trg, ++src )
                     ibuff[trg] = ibuff[src] ;
                  //* read remainder of FibBase structure *
                  ifs.read( &ibuff[i], (ibSIZE - i) ) ;
                  #if DEBUG_DOD != 0
                  bOffset = bytesRead ;
                  bytesRead += ifs.gcount() ;
                  #endif   // DEBUG_DOD
               }

               #if DEBUG_DOD != 0
               dbg = "=>>" ;
               for ( short i = ZERO ; i < ibSIZE ; ++i )
                  dbg.append( "%02hhX ", &ibuff[i] ) ;
               ofs << dbg.ustr() << endl ;
               #endif   // DEBUG_DOD

               done = fibFound = true ;
               break ;
            }
         }

         //* If FIB was located, get useful info *
         if ( fibFound )
         {
            //* nFib - file format version    *
            fib.nFib   = fib.decode16BitLE ( &ibuff[2] ) ;
            fib.lid = fib.decode16BitLE ( &ibuff[6] ) ;         // language ID
            fib.pnNext = fib.decode16BitLE ( &ibuff[8] ) ;      // auto-text offset
            fib.base16bits = fib.decode16BitLE ( &ibuff[10] ) ; // 16-bit bit field
            fib.nFibBack = fib.decode16BitLE ( &ibuff[12] ) ;   // reserved ID (s/b 0x00BF)
            fib.IKey = fib.decode32BitLE ( &ibuff[14] ) ;       // encryption key length
            fib.base8bits = ibuff[18] ;                         // 8-bit bit field
            // (all remaining bytes of this structure are reserved)
            // ibuff[4] reserved (16 bits)
            // ibuff[18] reserved (8 bits)
            // ibuff[19] bit field
            // ibuff[20] through ibuff[27] reserved

            //* Validate file-format *
            if ( ((fib.nFib == nFib_1997) || (fib.nFib == nFib_2000) ||
                 (fib.nFib == nFib_2002) || (fib.nFib == nFib_2003) ||
                 (fib.nFib == nFib_2007)) &&
                 ((fib.nFibBack == 0x00BF) || (fib.nFibBack == nFib_1997)) )
               status = true ;      // file format verified

            //* Decode useful bit values *
            fib.fFarEast = bool(fib.base16bits & 0x4000) ;     // bit 14
            fib.fWhichTblStm = bool(fib.base16bits & 0x0200) ; // bit 09

            #if DEBUG_DOD != 0
            dbg.compose( "=>>nFib    : 0x%04hX (status:%s)\n"
                         "=>>pnNext  : 0x%04hX (0x0000==no AutoText)\n"
                         "=>>fFarEast: %s\n"
                         "=>>TblStm  : %hhX\n"
                         "=>>raw16bit: 0x%04hX\n"
                         "=>>raw8bit : 0x%02hhX\n", 
                         &fib.nFib, (char*)(status ? "true" : "false"),
                         &fib.pnNext, (char*)(fib.fFarEast ? "true" : "false"),
                         &fib.fWhichTblStm, &fib.base16bits, &fib.base8bits) ;
            ofs << dbg.ustr() << endl    ;
            #endif   // DEBUG_DOD
         }     // fibFound
      }        // for(;;)

      //* Read 'csw' (2 bytes) and 'fibRgW' (2 x 14 bytes) *
      ifs.read( ibuff, (2 * 15) ) ;
      gcnt = ifs.gcount() ;

      #if DEBUG_DOD != 0
      bOffset = bytesRead ;
      bytesRead += ifs.gcount() ;
      dbg = "=>>fibRgW " ;
      for ( short i = ZERO ; i < (2 * 15) ; ++i )
         dbg.append( "%02hhX ", &ibuff[i] ) ;
      ofs << dbg.ustr() << endl ;
      fib.temp16 = fib.decode16BitLE ( ibuff ) ;
      dbg.compose( "=>>csw  : 0x%04hX (s/b 0x000E)", &fib.temp16 ) ;
      ofs << dbg.ustr() << endl ;
      fib.lidFE = fib.decode16BitLE ( &ibuff[2 * 14] ) ;
      // LID of stored style names
      dbg.compose( "=>>lidFE: 0x%04hX (ignore if !fFarEast)\n", &fib.lidFE ) ;
      ofs << dbg.ustr() << endl ;
      #endif   // DEBUG_DOD

      //* Read the 'cslw' which is a  16-bit value containing the *
      //* count of 32-bit values in fibRgLw. (must be 0x0016)      *
      ifs.read( ibuff, 2 ) ;
      fib.cslw = fib.decode16BitLE ( ibuff ) ;

      #if DEBUG_DOD != 0
      bOffset = bytesRead ;
      bytesRead += ifs.gcount() ;
      dbg.compose( "=>>cslw: 0x%04hX (s/b 0x0016) at 0x%06llX", &fib.cslw, &bOffset ) ;
      ofs << dbg.ustr() << endl ;
      dbg.compose( "=>>FibRgLw97 at 0x%06llX:", &bytesRead ) ;
      ofs << dbg.ustr() << endl ;
      #endif   // DEBUG_DOD

      //* FibRgLw97 consists of 22 32-bit values, 14 of which are reserved *
      for ( short i = ZERO ; i < fib.cslw ; ++i )
      {
         ifs.read( ibuff, 4 ) ;     // read a 32-bit value

         #if DEBUG_DOD != 0
         bOffset = bytesRead ;
         bytesRead += ifs.gcount() ;
         #endif   // DEBUG_DOD

         //* Capture 'cbMac', size of WordDocument stream.*
         if ( i == ZERO )        // cbMac
         {
            fib.cbMac = fib.decode32BitLE ( ibuff ) ;
            #if DEBUG_DOD != 0
            dbgi.formatInt( fib.cbMac, 13, true ) ;
            dbg.compose( "=>>cbMac: 0x%08X (%S)", &fib.cbMac, dbgi.gstr() ) ;
            ofs << dbg.ustr() << endl ;
            #endif   // DEBUG_DOD
         }
         else if ( i == 3 )      // ccpText
            fib.ccpText = fib.decode32BitLE ( ibuff ) ;
         else if ( i == 4 )      // ccpFtn
            fib.ccpFtn = fib.decode32BitLE ( ibuff ) ;
         else if ( i == 5 )      // ccpHdd
            fib.ccpHdd = fib.decode32BitLE ( ibuff ) ;
         else if ( i == 7 )      // ccpAtn
            fib.ccpAtn = fib.decode32BitLE ( ibuff ) ;
         else if ( i == 8 )      // ccpEdn
            fib.ccpEdn = fib.decode32BitLE ( ibuff ) ;
         else if ( i == 9 )      // ccpTxbx
            fib.ccpTxbx = fib.decode32BitLE ( ibuff ) ;
         else if ( i == 10 )     // ccpHdrTxbx
            fib.ccpHdrTxbx = fib.decode32BitLE ( ibuff ) ;
         else     // reserved values are not stored
            ;

         #if DEBUG_DOD != 0 && DEBUG_DOD_VERBOSE != 0
         fib.temp32a = fib.decode32BitLE ( ibuff ) ;
         dbg.compose( "=>> %2hd) 0x%08X", &i, &fib.temp32a ) ;
         if ( i == 1 || i == 2 || i == 6 || i >= 11 )
            dbg.append( " (reserved)" ) ;
         else if ( i == 0 )   dbg.append( " cbMac" ) ;   // bytes of meaningfun data
         else if ( i == 3 )   dbg.append( " ccpText" ) ; // CPs in main document
         else if ( i == 4 )   dbg.append( " ccpFtn" ) ;  // CPs in footnote
         else if ( i == 5 )   dbg.append( " ccpHdd" ) ;  // CPs in header
         else if ( i == 7 )   dbg.append( " ccpAtn" ) ;  // CPs in comment
         else if ( i == 8 )   dbg.append( " ccpEdn" ) ;  // CPs in endnote
         else if ( i == 9 )   dbg.append( " ccpTxbx" ) ; // CPs in textbox (boilerplate)
         else if ( i ==10 )   dbg.append( " ccpHdrTxbx" ) ; // CPs in header textbox
         ofs << dbg.ustr() << endl ;
         #endif   // DEBUG_DOD && DEBUG_DOD_VERBOSE
      }

      //* cbRgFcLcb 16-bit word containing count of 64-bit  *
      //* structures values in fibRgFcLcbBlob.              *
      //* Count is the size of the FibRgFcLcb structure and *
      //* depends on value of 'nFib'.                       *
      ifs.read( ibuff, 2 ) ;
      fib.cbRgFcLcb = fib.decode16BitLE ( ibuff ) ;
      fib.allocateLcb() ;     // allocate storage for FibRgFcLcb records

      #if DEBUG_DOD != 0
      bOffset = bytesRead ;
      bytesRead += ifs.gcount() ;
      dbgi.formatInt( fib.cbRgFcLcb, 7, true ) ;
      dbg.compose( "\n=>>cbRgFcLcb 0x%04hX (%S) at 0x%06llX\n"
                   "=>>FibRgFcLcb  (sorted by offset, ascending)\n" 
                   "=>>   index  offset     size\n"
                   "=>>   -----  ---------- --------", 
                   &fib.cbRgFcLcb, dbgi.gstr(), &bOffset ) ;
      ofs << dbg.ustr() << endl ;
      #endif   // DEBUG_DOD

      //* Read the FibRgFcLcbBlob array. Each 64-bit        *
      //* structure consists of a pair of 32-bit values     *
      //* which are the offset/size of the target object.   *
      //* It includes FibRgFcLcb97 and depending on document*
      //* version, FibRgFcbLcb2000, FibRgFcLcb2002,         *
      //* FibRgFcLcb2003 or FibRgFcLcb2007 extensions.      *
      //* The first pair is reserved and must be ignored.   *
      for ( short i = ZERO ; i < fib.cbRgFcLcb ; ++i )
      {
         ifs.read( ibuff, 8 ) ;     // read an offset/value pair
         fib.lcb[i].offset = fib.decode32BitLE ( ibuff ) ;
         fib.lcb[i].bytes  = fib.decode32BitLE ( &ibuff[4] ) ;
         fib.lcb[i].index  = i ;

         #if DEBUG_DOD != 0
         bOffset = bytesRead ;
         bytesRead += ifs.gcount() ;
         #endif   // DEBUG_DOD
      }
      //* Sort the FibRgFcLcb.lcb records by offset (ascending).*
      //* The first non-zero offset starts the Cix table.       *
      fib.sort_lcb() ;
// FibRgFcLcb.fcPlcffndRef (record 6): target specifies locations of footnotes

      #if DEBUG_DOD != 0
      //* Display the sorted 'lcb' array *
      for ( short i = ZERO ; i < fib.cbRgFcLcb ; ++i )
      {
         #if DEBUG_DOD_VERBOSE == 0
         if ( (fib.lcb[i].offset != ZERO) && (fib.lcb[i].bytes != ZERO) &&
              (fib.lcb[i].offset <=0x1FFFFFFF) && (fib.lcb[i].bytes <= 0x1FFFFFFF) )
         #endif   // DEBUG_DOD_VERBOSE
         {
            dbg.compose( "=>>    %3hd)  0x%08X 0x%06X",
                         &fib.lcb[i].index, &fib.lcb[i].offset, &fib.lcb[i].bytes ) ;
            ofs << dbg.ustr() << endl ;
         }
      }
      #endif   // DEBUG_DOD

      //* cswNew is a 16-bit count of of the 16-bit values in FibRgCswNew.*
      ifs.read( ibuff, 2 ) ;
      fib.cswNew = fib.decode16BitLE ( ibuff ) ;

      #if DEBUG_DOD != 0
      bOffset = bytesRead ;
      bytesRead += ifs.gcount() ;
      dbgi.formatInt( fib.cswNew, 7, true ) ;
      dbg.compose( "\n=>>cswNew: 0x%04hX (%S) at 0x%06llX (if 0x0000, then no FibRgCswNew)", 
                   &fib.cswNew, dbgi.gstr(), &bOffset ) ;
      ofs << dbg.ustr() << endl ;
      #endif   // DEBUG_DOD

      //* FibRgCswNew (optional). Present only if Fib.cswNew non-zero.   *
      //* 16 bits  indicates file version:                               *
      //*   0x00D9, 0x0101, 0x010C == fibRgCswNewData2000 (2 data bytes) *
      //*   0x0112 == fibRgCswNewData2007 (8 data bytes)                 *
      //* If file-format version is specified, save it.                  *
      //* The "quick-save" information, is not interesting to us.        *
      for ( short i = ZERO ; i < fib.cswNew ; ++i )
      {
         ifs.read( ibuff, 2 ) ;
         if ( i == ZERO )        // decode the file-format version number
            fib.nFib = fib.decode16BitLE ( ibuff ) ;
         #if DEBUG_DOD != 0
         bOffset = bytesRead ;
         bytesRead += ifs.gcount() ;
         if ( i == ZERO )
         {
            dbg.compose( "=>>nFibNew: 0x%04hX (new nFib)", &fib.nFib ) ;
            ofs << dbg.ustr() << endl ;
         }
         #endif   // DEBUG_DOD
      }
      #if DEBUG_DOD != 0
      bOffset = bytesRead - 1 ;
      dbg.compose( "File Information Block ends at: 0x%08llX\n", &bOffset ) ;
      ofs << dbg.ustr() << endl ;
      #endif   // DEBUG_DOD

      // Programmer's Note: Observationally, there are only 0x00 byte values 
      // from this point to beginning of text. This is not a reliable way 
      // to locate the text data, but it is a start.
      do
      {
         ifs.read( ibuff, 1 ) ;

         #if DEBUG_DOD != 0
         ++bytesRead ;
         #endif   // DEBUG_DOD
      } while ( ibuff[0] == 0x00 ) ;

      #if DEBUG_DOD != 0
      bOffset = (bytesRead -= 1) - bOffset  ;
      dbgi.formatInt( bOffset, 7, true ) ;
      dbg.compose( "Trailing null bytes: %S\n"
                   "Text (nominally) begins at: 0x%06llX\n"
                   "------------------------------------",
                   dbgi.gstr(), &bytesRead ) ;
      ofs << dbg.ustr() << endl ;
      ++bytesRead ;
      #endif   // DEBUG_DOD


      //*****************************
      //* Decode the document text. *
      //*****************************
      //-----------------------------------------------------------------------
      // Programmer's Note: We take a shortcut here to decide whether the 
      // text is 8-bit ANSI or UTF-16LE. If we call ansi8Decode() and the second 
      // source byte is a 0x00, then we assume that the data are UTF-16.
      // Example: In UTF-16LE,  "Abc" == 41 00 62 00 63 00 00 00
      // In that case, ansi8Decode() would return only "A" (two characters) 
      // indicating that the data are UTF-16LE.
      // If we can find the 'FcCompressed' structures, we can avoid this kludgy
      // test. Similarly, we do not know the number of data bytes in the text 
      // blocks, so we scan until a null character (not used in the text) is 
      // found. 
      //-----------------------------------------------------------------------

      //* Terminal window dimensions (used for output formatting) *
      short termRows, termCols, rowCols = ZERO ;
      nc.ScreenDimensions ( termRows, termCols ) ;

      const short rcnt = (gsALLOCDFLT - 7) ; // bytes read in each loop interation
      short convChars ;                      // number of wchar_t character converted
      ifs.read( &ibuff[1], 3 ) ;             // buffer now contains 4 bytes
      gcnt = (ifs.gcount()) + 1 ;
      #if DEBUG_DOD != 0
      bytesRead = gcnt ;
      #endif   // DEBUG_DOD

      if ( (convChars = fib.ansi8Decode ( ibuff, gcnt, gsOut )) <= 2 )
      {
         convChars = fib.utf16Decode ( ibuff, gcnt, gsOut ) ;
         AnsiText = false ;
      }
      ofs << gsOut.ustr() ;            // output the converted data
      rowCols = gsOut.gscols() ;       // track # of columns written on line
      done = false ;

      while ( ! done )
      {
         ifs.read( ibuff, rcnt ) ;
         gcnt = ifs.gcount() ;
         #if DEBUG_DOD != 0
         bytesRead += gcnt ;
         #endif   // DEBUG_DOD

         if ( gcnt == ZERO )           // unexpected EOF
         {
            break ;
         }
         if ( ibuff[ZERO] == 0x00 )    // EOT or UTF-16 out-of-synch
         {
            if ( ibuff[ZERO] == 0x00 ) // two sequential null characters
            {
               break ;
            }
         }

         if ( AnsiText )
         {
            if ( (convChars = fib.ansi8Decode ( ibuff, gcnt, gsOut )) < gcnt )
               done = true ;
         }
         else
         {
            convChars = fib.utf16Decode ( ibuff, gcnt, gsOut ) ;
            if ( convChars < (gcnt / 2) )
               done = true ;
         }
         if ( convChars > 1 )
         {
            //* Calculate number of columns to be written on current *
            //* and subsequent lines. Word-wrap if above threshold.  *
            rowCols = dbdFormatLine ( gsOut, termCols, rowCols ) ;
            ofs << gsOut.ustr() ;      // write the decoded and formatted text
         }
      }
      ofs << "\n" << endl ;     // flush the output buffer

      #if DEBUG_DOD != 0
      dbg.compose( "End-Of-Text at: 0x%06llX\n", &bytesRead ) ;
      ofs << dbg.ustr() << endl ;
      #endif   // DEBUG_DOD

      #if DEBUG_DOD != 0
      //* Scan for the stream records.                   *
      //* See the 'Mystery' class for more information.  *
      //* Read to 16-byte boundary, then read 16 bytes   *
      //* at a time until we find "Root Entry" (UTF-16). *
      const short TREX = 128 ;            // size of record
      const short iREX = 16 ;             // size of scan read
      const short tREX = TREX - iREX ;    // remainder of record
      const short nREX = 64 ;             // length of UTF-16 name (in bytes)
      bytesRead = ifs.tellg() ;
      short mod = iREX - (bytesRead % iREX) ;
      if ( mod > ZERO )
         ifs.read( ibuff, mod ) ;
      bytesRead = ifs.tellg() ;

      Mystery mys ;
      bool rootEntry = false ;
      done = false ;
      while ( ! done )
      {
         ifs.read( ibuff, iREX ) ;
         gcnt = ifs.gcount() ;
         if ( gcnt < iREX )            // premature end-of-file
            break ;

         #if DEBUG_DOD != 0
         bOffset = bytesRead ;
         bytesRead += gcnt ;
         #endif   // DEBUG_DOD

         if ( (ibuff[0] == 'R')  && (ibuff[1] == 0x00)  &&
              (ibuff[2] == 'o')  && (ibuff[3] == 0x00)  &&
              (ibuff[4] == 'o')  && (ibuff[5] == 0x00)  &&
              (ibuff[6] == 't')  && (ibuff[7] == 0x00)  &&
              (ibuff[8] == ' ')  && (ibuff[9] == 0x00)  &&
              (ibuff[10] == 'E') && (ibuff[11] == 0x00) &&
              (ibuff[12] == 'n') && (ibuff[13] == 0x00) &&
              (ibuff[14] == 't') && (ibuff[15] == 0x00) 
            )
         {
            //* Read remainder of record *
            ifs.read( &ibuff[iREX], tREX ) ;
            gcnt = ifs.gcount() ;

            #if DEBUG_DOD != 0
            fib.utf16Decode( ibuff, nREX, gsOut ) ;
            dbg.compose( "'%S' at: 0x%06llX", gsOut.gstr(), &bOffset ) ;
            ofs << dbg.ustr() << endl ;
            #endif   // DEBUG_DOD
            rootEntry = true ;
            break ;
         }
      }  // while()

      //* If 'Root Entry' found, decode it, then read   *
      //* and decode the remaining records in the array.*
      if ( rootEntry )
      {
         fib.utf16Decode( ibuff, nREX, gsOut ) ;
         gsOut.copy( mys.name, 64 ) ;
         short src = nREX ;
         mys.sVal1 = fib.decode16BitLE( &ibuff[src] ) ;
         src += 2 ;
         mys.sVal2 = fib.decode16BitLE( &ibuff[src] ) ;
         src += 2 ;
         for ( short trg = ZERO ; trg < 15 ; src += 4, ++trg )
            mys.iVal[trg] = fib.decode32BitLE( &ibuff[src] ) ;

         #if DEBUG_DOD != 0
         dbg.compose( "Record Name: '%s' at: 0x%06llX\n"
                      "idByte: 0x%02hhX  sVal1: 0x%04hX  sVal2: 0x%04hX",
                      mys.name, &bOffset, &mys.idByte, &mys.sVal1, &mys.sVal2 ) ;
         ofs << dbg.ustr() << endl ;
         bOffset = bytesRead ;   // from previous read
         bytesRead += gcnt ;
         dbg.clear() ;
         for ( src = ZERO ; src < 15 ; ++src )
         {
            if ( (src > ZERO) && ((src % 5) == ZERO) )
               dbg.append( L'\n' ) ;
            dbg.append( " 0x%08X", &mys.iVal[src] ) ;
         }
         ofs << dbg.ustr() << '\n' << endl ;
         #endif   // DEBUG_DOD

         done = false ;
         while ( ! done )
         {
            ifs.read( ibuff, TREX ) ;
            if ( (gcnt = ifs.gcount()) < TREX ) // end-of-file
               break ;

            #if DEBUG_DOD != 0
            bOffset = bytesRead ;
            bytesRead += gcnt ;
            #endif   // DEBUG_DOD

            mys.reset() ;
            fib.utf16Decode( ibuff, nREX, gsOut ) ;
            if ( ibuff[0] < ' ' )         // save idByte (if any)
            {
               mys.idByte = ibuff[0] ;
               gsOut.shiftChars( -1 ) ;
            }

            //* If no valid characters converted, either gsOut contains    *
            //* only the null char. OR contains only '?' characters.       *
            //* Note that some stream records begin with an 8-bit ID byte: *
            //* 0x03 ObjInfo        0x05 Summary Information               *
            //* 0x03 PRINT          0x05 Document Summary Information      *
            //* 0x03 EPRINT         0x06 DataSpaces      0x09 DRMContent   *
            //* Note also that the 0Table or 1Table stream may begin with  *
            //* an encryption header, but if it does, we won't be able to  *
            //* read the text anyway, so no loss.                          *
            while ( (gsOut.erase( L'?' )) >= ZERO ) ;
            if ( gsOut.gschars() == 1 )         // invalid record
            {
               mys.reset() ;
               mys.idByte = ibuff[0] ;
               gsOut = "Invalid Record" ;
            }
            gsOut.copy( mys.name, 64 ) ;
            src = nREX ;
            mys.sVal1 = fib.decode16BitLE( &ibuff[src] ) ;
            src += 2 ;
            mys.sVal2 = fib.decode16BitLE( &ibuff[src] ) ;
            src += 2 ;
            for ( short trg = ZERO ; trg < 15 ; src += 4, ++trg )
               mys.iVal[trg] = fib.decode32BitLE( &ibuff[src] ) ;

            #if DEBUG_DOD != 0
            dbg.compose( "Record Name: '%s' at: 0x%06llX\n"
                         "idByte: 0x%02hhX  sVal1: 0x%04hX  sVal2: 0x%04hX",
                         mys.name, &bOffset, &mys.idByte, &mys.sVal1, &mys.sVal2 ) ;
            ofs << dbg.ustr() << endl ;
            bOffset = bytesRead ;   // from previous read
            bytesRead += gcnt ;
            dbg.clear() ;
            for ( src = ZERO ; src < 15 ; ++src )
            {
               if ( (src > ZERO) && ((src % 5) == ZERO) )
                  dbg.append( L'\n' ) ;
               dbg.append( " 0x%08X", &mys.iVal[src] ) ;
            }
            ofs << dbg.ustr() << '\n' << endl ;
            #endif   // DEBUG_DOD
         }
      }
      #endif   // DEBUG_DOD
   }           // is_open()
   if ( ifs.is_open() )    // close the source file
      ifs.close() ;
   if ( ofs.is_open() )    // close the target file
      ofs.close() ;

   return status ;

   #undef DEBUG_DOD
   #undef DEBUG_DOD_VERBOSE
}  //* End vfcDecodeBinDoc() *

//*************************
//*     dbdFormatLine     *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Apply output formatting to decoded display text.                           *
//* Called only by vfcDecodeBinDoc().                                          *
//*                                                                            *
//* Input  : gsOut   : (by reference) data to be formatted                     *
//*          termCols: number of display columns in terminal window            *
//*          colOut  : insertion point (column) of current display line        *
//*                                                                            *
//* Returns: number of data columns on last display line                       *
//******************************************************************************

static short dbdFormatLine ( gString& gsOut, short termCols, short colOut )
{
   const short INDENT = 24 ;
   const wchar_t wNewline = L'\n' ;
   const wchar_t wSpace   = L' ' ;
   const wchar_t *wPtr = gsOut.gstr() ; // array of wchar_t characters
   int   lineChars ;                    // chars in formatted line
   //* Array of column counts in formatted line *
   const short *colPtr = gsOut.gscols( lineChars ) ;
   short breakCol = termCols - INDENT, // 'safe' line-break column
         colCount = colOut ;           // return value

   //* Format line breaks to fit the display area *
   for ( short indx = ZERO ; indx < lineChars ; ++indx )
   {
      //* If a character which requires one or more display columns *
      if ( colPtr[indx] > ZERO )
      {
         //* If within line width *
         if ( (colCount + colPtr[indx]) < breakCol )
            colCount += colPtr[indx] ;

         //* At threshold of line width *
         else
         {
            //* Scan to next space (or newline) *
            for ( short loop = INDENT ; (loop > ZERO) && (indx < lineChars) ; --loop )
            {
               if ( colPtr[indx] > ZERO )
               {
                  if ( wPtr[indx] == wSpace )
                  {
                     //* Insert a newline and refresh the tracking data *
                     ++indx ;       // set index after the space
                     gsOut.insert( wNewline, indx ) ;
                     colPtr = gsOut.gscols( lineChars ) ;
                     //++indx ;     // will be advanced by the loop
                     colCount = ZERO ;
                     break ;
                  }
                  else     // not a natural breakpoint
                     colCount += colPtr[indx++] ;
               }
               else if ( wPtr[indx] == wNewline )  // existing linebreak ;
               {
                  colCount = ZERO ;
                  //++indx ;     // will be advanced by the loop
                  break ;
               }
               else     // unknown zero-width character
                  ++indx ;
            }

            //* If no natural breakpoint found, force a line break.*
            if ( colCount >= breakCol )
            {
               gsOut.insert( wNewline, indx ) ;
               colPtr = gsOut.gscols( lineChars ) ;
               //++indx ;     // will be advanced by the loop
               colCount = ZERO ;
            }
         }
      }

      //* If newline found, reset column count *
      else if ( wPtr[indx] == wNewline )
      {
         colCount = ZERO ;
      }

      //* Other zero-width character, do nothing.*
      else
         ;
   }        // for(;;)

   return colCount ;

}  //* End dbdFormatLine() *

//*************************
//*   vfcDecodeMetadata   *
//*************************
//******************************************************************************
//* Read and format the metadata for a supported audio or video file.          *
//* Formatted metadata are written to a plain text temporary file.             *
//* Called only by ViewFileContents() method.                                  *
//*                                                                            *
//* Metadata is the description of the file's media contents which is to be    *
//* formatted in human-readable form.                                          *
//*                                                                            *
//* Input  : srcPath : source filespec                                         *
//*          tmpPath : receives path/filename of temporary file to be displayed*
//*          mfType  : target media identifier                                 *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*            Note that if the target is not a supported media filetype then  *
//*            the appropriate error message will be written to the temp file, *
//*            so this will not be reported to caller as an error.             *
//*          ERR if file not found or unable to read file                      *
//******************************************************************************

short FileDlg::vfcDecodeMetadata ( const gString& srcPath, gString& tmpPath,
                                   MediafileType mfType )
{
   short status = ERR ;

   if ( (this->fmPtr->CreateTempname ( tmpPath )) != false )
   {
      ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         ofs << "Metadata for Media File\n"
                "=======================" << endl ;

         //* Determine whether this is a supported filetype *
         if ( mfType != mftNONE )
         {
            status = OK ; // any errors encountered will be embedded in the output
            if ( mfType == mftMP3 )
               this->ExtractMetadata_MP3 ( ofs, srcPath ) ;
            else if ( mfType == mftOGG )
               this->ExtractMetadata_OGG ( ofs, srcPath ) ;
            else if ( mfType == mftM4A )
               this->ExtractMetadata_M4A ( ofs, srcPath, tmpPath ) ;
            else if ( mfType == mftASF )
               this->ExtractMetadata_ASF ( ofs, srcPath ) ;
            else if ( mfType == mftPNG )
               this->ExtractMetadata_PNG ( ofs, srcPath ) ;
            else if ( mfType == mftJPG )
               this->ExtractMetadata_JPG ( ofs, srcPath ) ;
         }

         if ( status != OK )
            ofs << "\n** Unsupported filetype OR Unable to extract metadata **\n" << endl ;

         ofs.close() ;
      }
   }
   return status ;

}  //* End vfcDecodeMetadata() *

//*************************
//*    vfcDecodeEmail     *
//*************************
//******************************************************************************
//* Read and format the text of an email file.                                 *
//*                                                                            *
//* Data reported by this scan:                                                *
//*  -- Date                                                                   *
//*  -- Sender(s)                                                              *
//*  -- Recipient(s)                                                           *
//*  -- Email text                                                             *
//*  -- Names of embedded objects (if any)                                     *
//*  -- Names of attachments (if any)                                          *
//*                                                                            *
//* Input  : srcPath : source filespec                                         *
//*          tmpPath : receives path/filename of temporary file to be displayed*
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          ERR if file not found or unable to read file                      *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//*                                                                            *
//* https://tools.ietf.org/html/rfc5322                                        *
//* https://tools.ietf.org/html/rfc6854                                        *
//* Email text format is specified by RFC-5322 or RFC-6854.                    *
//* Images, audio or other structured data are described in other parts of     *
//* the MIME document series (RFC 2045, RFC 2046, RFC 2049).                   *
//*                                                                            *
//* Library of Congress:                                                       *
//* https://www.loc.gov/preservation/digital/formats/fdd/fdd000388.shtml       *
//* https://www.loc.gov/preservation/digital/formats/fdd/fdd000393.shtml       *
//*                                                                            *
//* "The 2012 standards RFC 6532 and RFC 6531 allow the inclusion of Unicode   *
//*  characters in a header content using UTF-8 encoding, and their            *
//*  transmission via SMTP - but in practice support is only slowly rolling    *
//*  out."   https://en.wikipedia.org/wiki/International_email                 *
//*                                                                            *
//* Supported content types:                                                   *
//* ------------------------                                                   *
//* text/plain and text/html (message often in both encodings)                 *
//*   "charset" may be on the same line or next line.                          *
//*   Examples: Content-Type: text/plain; charset="UTF-8"                      *
//*             Content-Type: text/plain;                                      *
//*                     charset="iso-8859-1"                                   *
//*                                                                            *
//* charset field values:                                                      *
//*   "UTF-8"                                                                  *
//*   "iso-8859-1"    (ASCII + Windoze extended 8-bit group)                   *
//*   Note that token MAY BE enclosed within double quotations.                *
//*   Note also that token is not case sensitive.                              *
//*                                                                            *
//* Content-Type: multipart/mixed;                                             *
//*       boundary="----=_NextPart_000_0006_01CEEC81.1203D360"                 *
//* Content-Type: multipart/alternative;                                       *
//*       boundary="000000000000634c1c05ae55b2ba"                              *
//*   Note that the boundary token may be on the same line as the field label  *
//*   or on the next line.                                                     *
//*   The boundary is specified in the form:                                   *
//*     ----=_NextPart_000_0006_01CEEC81.1203D360                              *
//*       OR                                                                   *
//*     --000000000000634c1c05ae55b2ba                                         *
//*                                                                            *
//*   Multi-part messages: (thanks, Wikipedia)                                 *
//*   The MIME multipart message contains a boundary in the header field       *
//*   Content-Type:; this boundary, which must not occur in any of the parts,  *
//*   is placed between the parts, and at the beginning and end of the body    *
//*   of the message.                                                          *
//*    a) Before the first boundary is an area that is ignored by              *
//*       MIME-compliant clients. This area is generally used to put a message *
//*       to users of old non-MIME clients.                                    *
//*    b) It is up to the sending mail client to choose a boundary string that *
//*       doesn't clash with the body text. Typically this is done by inserting*
//*       a long random string.                                                *
//*    c) The last boundary must have two hyphens at the end.                  *
//*   Everything else between boundary tokens, except field identifiers, is    *
//*   assumed to be the actual message. It must be taken into account that     *
//*   some of the fields that we need to report are also within this block.    *
//*                                                                            *
//*                                                                            *
//* Not supported (obsolete):                                                  *
//* ------------------------                                                   *
//* Content-Type: message/rfc822;                                              *
//*   If a message is encoded for RFC-822, the displayed data may suffer from  *
//*   significant ugliness because it is very likely encoded, either "base64"  *
//*   or "quoted-printable", but may contain additional formatting (not HTML). *
//*                                                                            *
//* Transfer Encoding:                                                         *
//* ------------------                                                         *
//* Content-Transfer-Encoding: 7Bit                                            *
//*   This is the RFC-2045 default. Within a block defined as "7bit", the data *
//*   may be:                                                                  *
//*   a) ASCII 01h-7Eh (special rules for CR and LF)                           *
//*   b) quoted-printable encoded                                              *
//*   c) base64 encoded                                                        *
//*                                                                            *
//* Content-Transfer-Encoding: quoted-printable                                *
//*   See deDecode_qprint() method.                                            *
//*                                                                            *
//* Content-Transfer-Encoding: base64                                          *
//*   See deDecode_base64() method.                                            *
//*                                                                            *
//* Content-Transfer-Encoding: 8Bit                                            *
//*   Any 8-bit stream.                                                        *
//*                                                                            *
//* Content-Transfer-Encoding: binary                                          *
//*   This is used primarily for images, audio, etc.                           *
//*   This algorithm SHOULD skip over these blocks.                            *
//*   Field specifies attachment: "Content-Disposition: attachment;"           *
//*   Followed by:  filename"Young Frankenstein.wav"                           *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* 1) The header ends with a blank line: "\r\n"; however, some non-message    *
//*    data may be found AFTER the header.                                     *
//* 2) If the file contains an HTML-formatted message, then we can discard     *
//*    the plain-text version of that message to avoid duplication.            *
//*    Unfortunately, the plain-text copy is usually placed ABOVE the HTML     *
//*    copy, so it is necessary to pre-scan the file for a "text/html" field.  *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short FileDlg::vfcDecodeEmail ( const gString& trgPath, gString& tmpPath )
{
   #define DEBUG_DECODE_EMAIL (0)      // For Debugging Only

   const char *blColor   = "\033[0;34m",     // display text
              *grColor   = "\033[0;32m",     // field name
              //*reColor   = "\033[1;31m",     // debugging color
              //*maColor   = "\033[0;35m",     // debugging color
              *bkColor   = "\033[0;30m" ;    // default color

   const char* const ctField  = "Content-Type: " ;
   const short ctFieldLen     = 14 ;
   const char* const ctTextPlain = "text/plain;" ;
   const char* const ctTextHtml  = "text/html;" ;
   const char* const ctMultiMix  = "multipart/mixed;" ;
   const char* const ctMultiDig  = "multipart/digest;" ;
   const char* const ctMultiAlt  = "multipart/alternative;" ;
   const char* const ctRFC822    = "message/rfc822;" ; // obsolete message type
   const char* const ctMultiPart = "multipart" ;
   const char* const boundField  = "boundary=" ;
   const char* const charSetAtom = "charset=" ;
   const char* const charSetUTF8 = "UTF-8" ;
   const char* const charSet8859 = "iso-8859-1" ;
   const char* const cteField  = "Content-Transfer-Encoding: " ;
   const short cteFieldLen     = 27 ;
   const char* const cteQPrint = "quoted-printable" ;
   const char* const cteBase64 = "base64" ;
   const char* const cte7Bit   = "7bit" ;
   const char* const cte8Bit   = "8bit" ;
   const char* const cteBinary = "binary" ;

   //* Types of Content-Type supported. *
   enum ctType : short
   {
      cttTextPlain,
      cttTextHtml,
      cttMultiMix,
      cttMultiDig,
      cttMultiAlt,
      cttRFC822,     // obsolete message type
   } ;

   //* Types of "charset" supported. *
   enum charSet : short
   {
      csUTF8,     // "UTF-8"
      cs8859,     // "iso-8859-1"
   } ;

   //* Types of Content-Transfer-Encoding supported. *
   enum ctEncoding : short
   {
      cteBASE64,        // "base64" encoding
      cteQPRINT,        // "quoted-printable" encoding
      cte7BIT,          // ASCII data 01h through 7Eh (RFC-2045 default)
      cte8BIT,          // UTF-8 or other "8-bit clean" data (RFC-6152)
      cteBINARY,        // binary data (not decoded or displayed here)
   } ;

   gString gsOut,                      // output buffer
           gsIn,                       // input parsing
           gstmp,                      // temp storage
           boundaryToken ;             // for multi-part messages, boundary identifier
   ctType     cttType = cttTextPlain ; // content type for current block
   charSet    cSet    = csUTF8 ;       // character set for current block
   ctEncoding cteType = cte8BIT ;      // text-encoding type for current block
   short colOut = ZERO,                // column count of data written to current line
         termRows,                     // terminal window dimensions
         termCols,
         blCnt = ZERO,                 // blank line count
         status = ERR ;                // return value
   bool  msgbody = false,              // 'true' if message body identified
         ignoreBlock = false ;         // 'true' in ingnored space after

   #if 1    // SILENCE COMPILER WARNINGS
   //* These values are initialized, but are not currently referenced *
   //* during processing so we silence the compiler warnings here.    *
   if ( cttType == cttTextPlain ) {}
   if ( cSet == csUTF8 ) {}
   #endif   // SILENCE COMPILER WARNINGS

   //* Terminal window dimensions (used for output formatting) *
   nc.ScreenDimensions ( termRows, termCols ) ;

   if ( (this->fmPtr->CreateTempname ( tmpPath )) != false )
   {
      //* Open source and target files *
      ifstream ifs( trgPath.ustr(), ifstream::in ) ;
      ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( (ifs.is_open()) && (ofs.is_open()) )
      {
         status = OK ;           // declare success
         short indx ;            // offset into data line
         bool  done = false ;    // loop control
         while ( ! done )
         {
            if ( (this->vfcGetLine ( ifs, gsIn )) > ZERO )
            {
               //* A blank line signals the end of the header *
               if ( ! msgbody && ((gsIn.gschars()) == 1) )
               { msgbody = true ; colOut = ZERO ; continue ; }

               //* If a multipart message, test for inter-block token.*
               //* If this is the first instance of the inter-block   *
               //* token, re-enable processing. Discard the token.    *
               if ( ((boundaryToken.gschars()) > 1) &&
                    ((gsIn.find( boundaryToken.gstr() )) >= ZERO) )
               { ignoreBlock = false ; continue ; }

               //* If we are in the ignored block between the  *
               //* definition of the inter-block token and its *
               //* first instance, discard the data line.      *
               if ( ignoreBlock )
                  continue ;

               //* Write single-line fields *
               if ( ((indx = gsIn.after( L"Date: " )) == 6)      ||
                    ((indx = gsIn.after( L"Sent: " )) == 6)      ||
                    ((indx = gsIn.after( L"From: " )) == 6)      ||
                    ((indx = gsIn.after( L"Subject: " )) == 9)   ||
                    ((indx = gsIn.after( L"Reply-To: " )) == 10) ||
                    ((indx = gsIn.after( L"Sender: " )) == 8)
                  )
               {
                  //* Add ANSI color *
                  gsIn.substr( gstmp, ZERO, indx ) ;
                  gsOut.compose( "%s%S%s", grColor, gstmp.gstr(), bkColor ) ;
                  if ( colOut > ZERO )
                     gsOut.insert( L'\n' ) ;
                  ofs << gsOut.ustr() ;
                  gsIn.shiftChars( -(indx) ) ;

                  //* If byte-encoded data, decode it.*
                  deDecodeHeader ( gsIn, gsOut ) ;
                  ofs << gsOut.ustr() << "\n" ;

                  colOut = ZERO ;      // reset output-column count
               }

               //* The "To: ", "Cc: " and "Bcc: " fields may be multi-line *
               // Programmer's Note: The "From: " field technically allows 
               // multi-line data, but we do not support that because it's stupid.
               else if ( ((indx = gsIn.after( L"To: " )) == 4) ||
                         ((indx = gsIn.after( L"Cc: " )) == 4) ||
                         ((indx = gsIn.after( L"Bcc: " )) == 5) )
               {
                  //* Add ANSI color *
                  gsIn.substr( gstmp, ZERO, indx ) ;
                  gsOut.compose( "%s%S%s", grColor, gstmp.gstr(), bkColor ) ;
                  if ( colOut > ZERO )
                     gsOut.insert( L'\n' ) ;
                  ofs << gsOut.ustr() ;
                  gsIn.shiftChars( -(indx) ) ;

                  //* If byte-encoded data, decode it.*
                  deDecodeHeader ( gsIn, gsOut ) ;
                  ofs << gsOut.ustr() << "\n" ;

                  //* If recipient list continues on subsequent lines *
                  //* report all recipients.                          *
                  // Programmer's Note: Typically, entries are separated by 
                  // commas; however, for quoted-printable encoding, the '=' 
                  // character at the end of a line represents a soft line break.
                  gsOut = gsIn ;
                  gsOut.strip( false, true ) ;   // discard trailing whitespace
                  while ( ((gsOut.findlast( L',' )) == ((gsOut.gschars()) - 2)) ||
                          ((gsOut.findlast( L'=' )) == ((gsOut.gschars()) - 2)) )
                  {
                     //* Read the next line *
                     this->vfcGetLine ( ifs, gsIn ) ;

                     //* If byte-encoded data, decode it.*
                     deDecodeHeader ( gsIn, gsOut ) ;
                     ofs << gsOut.ustr() << "\n" ;
                     gsOut.strip( false, true ) ;   // discard trailing whitespace
                  }
                  colOut = ZERO ;      // reset output-column count
               }

               //* Remember the content type and charset (see notes above).   *
               else if ( (indx = gsIn.after( ctField )) == ctFieldLen )
               {
                  //* Remember the content type *
                  // Programmer's Note: For multi-part messages, the content type 
                  // is written as type/subtype, and the subtype may change 
                  // several times; however, the boundary token, if specified 
                  // is the same throughout the file.
                  if ( (gsIn.find( ctTextPlain, indx )) == indx )
                     cttType = cttTextPlain ;
                  else if ( (gsIn.find( ctTextHtml, indx )) == indx )
                     cttType = cttTextHtml ;
                  else if ( (gsIn.find( ctMultiMix, indx )) == indx )
                     cttType = cttMultiMix ;
                  else if ( (gsIn.find( ctMultiDig, indx )) == indx )
                     cttType = cttMultiDig ;
                  else if ( (gsIn.find( ctMultiAlt, indx )) == indx )
                     cttType = cttMultiAlt ;
                  else if ( (gsIn.find( ctRFC822, indx )) == indx )
                  {  // Note that the obsolete RFC-822 is always 7-bit data,
                     // so the encoding is not explicity stated. For this reason
                     // we explicitly set the encoding here.
                     cttType = cttRFC822 ; cteType = cte7BIT ;
                  }
                  else if ( (gsIn.find( ctMultiPart, indx )) == indx )
                     cttType = cttMultiMix ;    // default multipart subtype
                  else                          // default
                     cttType = cttTextPlain ;

                  //* For multi-part messages, read the "boundary" token *
                  if ( (gsIn.find( ctMultiPart, indx )) == indx )
                  {
                     //* If the boundary token is not on the current *
                     //* line, read the next line.                   *
                     if ( (gsIn.find( boundField, indx )) < ZERO )
                        this->vfcGetLine ( ifs, gsIn ) ;
                     if ( (indx = gsIn.after( boundField )) >= ZERO )
                     {
                        //* Extract the boundary token.                 *
                        //* Note that the token may or may not be       *
                        //* enclosed in double-quotes.                  *
                        if ( gsIn.gstr()[indx] == L'"' )
                           ++indx ;
                        gsIn.shiftChars( -(indx) ) ;
                        if ( (indx = gsIn.findlast( L'"' )) == ((gsIn.gschars()) - 2) )
                           gsIn.erase( L'"', indx ) ;
                        boundaryToken = gsIn ;

                        // Programmer's Note: Technically, there is an area
                        // following the boundary declaration and up to the 
                        // first use of that boundary token which is ignored
                        // by MIME-compliant servers. It is possible that 
                        // we may incorrectly report some of this as part of 
                        // the message, so we set the 'ignoreBlock' flag here.
                        ignoreBlock = true ;
                     }
                  }

                  //* Else, remember the charset *
                  else
                  {
                     //* If the charset is not specified on the current *
                     //* line, read the next line
                     if ( (gsIn.find( charSetAtom, indx )) < ZERO )
                        this->vfcGetLine ( ifs, gsIn ) ;
                     if ( (indx = gsIn.after( charSetAtom )) >= ZERO )
                     {
                        //* Type MAY BE delimited by double quotes *
                        if ( gsIn.gstr()[indx] == L'"' )
                           ++indx ;
                        if ( (gsIn.find( charSetUTF8, indx )) == indx )
                           cSet = csUTF8 ;
                        else if ( (gsIn.find( charSet8859, indx )) == indx )
                           cSet = cs8859 ;
                        else        // default
                           cSet = csUTF8 ;
                     }
                  }
                  colOut = ZERO ;      // reset output-column count
               }

               //* Remember the content-transfer-encoding *
               else if ( (indx = gsIn.after( cteField )) == cteFieldLen )
               {
                  if ( (gsIn.find( cteQPrint, indx )) == indx )
                     cteType = cteQPRINT ;
                  else if ( (gsIn.find( cteBase64, indx )) == indx )
                     cteType = cteBASE64 ;
                  else if ( (gsIn.find( cte8Bit, indx )) == indx )
                     cteType = cte8BIT ;
                  else if ( (gsIn.find( cte7Bit, indx )) == indx )
                     cteType = cte7BIT ;
                  else if ( (gsIn.find( cteBinary, indx )) == indx )
                     cteType = cteBINARY ;
                  else     // default
                     cteType = cte7BIT ;

                  colOut = ZERO ;      // reset output-column count
               }

               #if 0    // "X-YMailISG: ", "X-YMail-OSG: ", etc.
               //* For the obsolete RFC-822 email format, message data *
               //* are base64 encoded. The message begins on this line.*
               else if ( ((gsIn.find( "X-YMail" )) == ZERO) &&
                         ((indx = gsIn.after( L": " )) > ZERO) )
               {
                  gsIn.shiftChars( -(indx) ) ;
                  //gsIn.replace( L'.', L'+', ZERO, false, true ) ;
                  gsIn.replace( L'.', L'/', ZERO, false, true ) ;
                  deDecode_base64 ( gsIn, gsOut ) ;
                  ofs << reColor << gsOut.ustr() << bkColor ;
               }
               #endif   // "X-YMailISG: ", "X-YMail-OSG: ", etc.

               //* Ignored header fields *
               #if DEBUG_DECODE_EMAIL != 0   // For Debugging Only
               else if ( ((gsIn.find( L"Precedence: " )) == ZERO)     ||
                         ((gsIn.find( L"Message-ID: " )) == ZERO)     ||
                         ((gsIn.find( L"In-Reply-To: " )) == ZERO)    ||
                         ((gsIn.find( L"References: " )) == ZERO)     ||
                         ((gsIn.find( L"Archived-At: " )) == ZERO)    ||
                         ((gsIn.find( L"Received: " )) == ZERO)       ||
                         ((gsIn.find( L"Return-Path: " )) == ZERO)    ||
                         ((gsIn.find( L"Received-SPF: " )) == ZERO)   ||
                         ((gsIn.find( L"DKIM-Signature: " )) == ZERO) ||
                         ((gsIn.find( L"Auto-Submitted: " )) == ZERO) ||
                         ((gsIn.find( L"VBR-Info: " )) == ZERO)       ||
                         ((gsIn.find( L"Authentication-Results: " )) == ZERO)
                       )
               {
                  ofs << "DISCARD \"" << gsIn << "\"\n" ;
                  colOut = ZERO ;      // reset output-column count
               }
               #endif   // DEBUG_DECODE_EMAIL

               //* If message body has been identified.                     *
               //* Field-identifier lines within the message area were      *
               //* processed above. Assume that data are message text. Data *
               //* are potentially byte-encoded and may contain HTML tags.  *
               else if ( msgbody )
               {
                  //* Email files often include multiple blank lines which *
                  //* may be useful in email programs, but are just        *
                  //* annoying in this context. Removing the extra blank   *
                  //* lines here makes no difference in the output, but it *
                  //* speeds the decoding because the extra blank lines    *
                  //* will not be decoded.                                 *
                  if( (gsIn.gschars()) == 1 )
                  {
                     if ( blCnt == ZERO ) { ++blCnt ; }
                     else { blCnt = ZERO ; continue ; }
                  }

                  //* If base64-encoded data, decode it.*
                  if ( (cteType == cteBASE64) || (cteType == cte7BIT) )
                  {
                     deDecode_base64 ( gsIn, gsOut ) ;

                     //* Apply output formatting *
                     colOut = deFormatLine ( gsOut, termCols, colOut ) ;
                     ofs << blColor << gsOut.ustr() << bkColor ;
                  }

                  //* If quoted-printable-encoded data, decode it.*
                  else if ( cteType == cteQPRINT )
                  {
                     deDecode_qprint ( gsIn, gsOut ) ;

                     //* Apply output formatting *
                     colOut = deFormatLine ( gsOut, termCols, colOut ) ;
                     ofs << blColor << gsOut.ustr() << bkColor ;
                  }

                  //* If embedded binary data, silently discard it *
                  else if ( cteType == cteBINARY )
                  {
                     ofs << "\n[embedded binary data discarded]\n" ; 
                     // CZONE - NOT YET IMPLEMENTED - SEE NOTES ABOVE
                  }

                  //* Else, assume unencoded 8-bit stream (UTF-8, etc.) *
                  else
                  {
                     //* Apply output formatting *
                     colOut = deFormatLine ( gsIn, termCols, colOut ) ;
                     ofs << blColor << gsIn.ustr() << bkColor ;
                  }
               }

               //* Data that do not contain a valid field identifier, *
               //* AND are not part of the message body are ignored.  *
               else
                  ;
            }
            else
            {
               ofs << endl ;
               done = true ;
            }
         }  // while()
      }     // files open
      if ( ifs.is_open() )       // close the source file
         ifs.close() ;
      if ( ofs.is_open() )       // close the target file
         ofs.close() ;
   }

   return status ;

}  //* End vfcDecodeEmail() *

//*************************
//*    deDecodeHeader     *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Caller has identified a header-field record that is to be displayed.       *
//* The field name has been extracted, and gsSrc contains the field data.      *
//* Data may be encoded as one of the following:                               *
//* 1) 'B' encoding (base64)                                                   *
//*    Example: "=?utf-8?B?TWFobG9uIOmprOS8pg==?=" <sam@qq.com>                *
//*                                                                            *
//* 2) 'Q' encoding (quoted-printable)                                         *
//*    Example from Wikipedia: =?iso-8859-1?Q?=A1Hola,_se=F1or!?=              *
//*             written as iso-8859-1: "¡Hola, señor!"                         *
//*                                                                            *
//* Any unencoded data which follows the encoded data (e.g. email address)     *
//* will be appended to the decoded data.                                      *
//*                                                                            *
//* Input  : gsSrc  : (by reference) potentially encoded data                  *
//*          gsTrg  : (by reference, initial contents ignored                  *
//*                   receives decoded data                                    *
//*                                                                            *
//* Returns: 'true' if data decoded or if no decoding necessary                *
//*          'false' if invalid header format (data returned unmodified)       *
//******************************************************************************

static bool deDecodeHeader ( const gString& gsSrc, gString& gsTrg )
{
   const wchar_t* delimStart = L"=?" ;
   const wchar_t* delimEnd   = L"?=" ;
   const wchar_t  wDQUOTE    = L'?' ;
   const wchar_t  wB         = L'B' ;
   //const wchar_t  wQ         = L'Q' ;

   gString gssTmp, gstTmp ;
   short indx,
         qCount = 3 ;
   bool q = true,          // 'true' if 'Q' encoding, 'false' if 'B' encoding
        status = false ;   // return value

   gsTrg.clear() ;         // initialize caller's buffer
   if ( gsSrc.gstr()[0] == L'"' )  // leading double-quote?
      gsTrg = "\"" ;

   //* Index the encoding delimiter *
   indx = gsSrc.find( delimStart ) ;

   //* If delimiter begins the sequence *
   if ( (indx == ZERO) || (indx == 1) )
   {
      indx = ZERO ;
      while ( qCount > ZERO )
      {
         if ( (indx = gsSrc.after( wDQUOTE, indx )) >= ZERO )
         {
            //* Test for 'Q' encoding *
            if ( (qCount == 2) && (gsSrc.gstr()[indx] == wB) )
               q = false ;
            --qCount ;
         }
         else if ( qCount > ZERO )
            break ;
      }
      if ( qCount == ZERO )
      {
         gssTmp = &gsSrc.gstr()[indx] ;

         //* Index the end of encoded data *
         if ( (indx = gsSrc.after( delimEnd, indx )) >= ZERO )
         {
            status = true ;      // formatting verified

            //* Find the end of encoded data *
            short tindx = gssTmp.find( delimEnd ) ;
            gssTmp.limitChars( tindx ) ;

            if ( q )
               deDecode_qprint ( gssTmp, gstTmp ) ;
            else
               deDecode_base64 ( gssTmp, gstTmp ) ;
            gsTrg.append( gstTmp.gstr() ) ;

            //* Append any trailing unencoded data *
            if ( indx < (gsSrc.gschars() - 1) )
               gsTrg.append( &gsSrc.gstr()[indx] ) ;
         }
      }
   }
   //* Delimiters not found. Data are either *
   //* unencoded OR partial qprint.          *
   else
   {
      deDecode_qprint ( gsSrc, gsTrg ) ;
      status = true ;
   }

   //* If invalid header format *
   if ( status == false )
      gsTrg = gsSrc ;

   return status ;

}  //* End deDecodeHeader() *

//*************************
//*    deDecode_base64    *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Given a byte-data sequence, if "base64" encoding is detected, decode the   *
//* sequence into plain text. Any unencoded text in the sequence is written    *
//* as-is. The decoded text will most likely be ASCII or UTF-8, but other      *
//* formats are possible (UTF-16, Windoze ISO-8859-1, etc.).                   *
//*                                                                            *
//* If "base64" encoding is not detected, the data are returned unmodified.    *
//*                                                                            *
//* Input  : gsSrc  : (by reference) potentially encoded data                  *
//*          gsTrg  : (by reference, initial contents ignored                  *
//*                   receives decoded data                                    *
//*          pretest: (optional, 'false' by default)                           *
//*                   if 'true', perform a pretest of the data to determine    *
//*                         whether all data bytes are valid base64 values,    *
//*                         and if so, decode the data                         *
//*                                                                            *
//* Returns: number of bytes decoded                                           *
//******************************************************************************

static bool deDecode_base64 ( const gString& gsSrc, gString& gsTrg )
{
   //* This is the MIME "Base64" lookup table which matches six(6) bits of    *
   //* binary source data with a 7-bit ASCII character. This allows a binary  *
   //* image (or other data) to be encoded as 7-bit text for easy transmission*
   //* using 7-bit transport protocols.                                       *
   //* It is called "Base64" because the lookup table has 64 elements.        *
   //*   Editorial: We believe that this type of encoding is a huge waste     *
   //*   of time and storage space because most of the world has moved on     *
   //*   to "8-bit clean" (UTF-8, etc.) protocols.                            *
   const uint32_t bt64BYTES = 64 ;
   const uint8_t base64Table[bt64BYTES] = 
   {//CHAR    INDX  SRC BITS    
      'A', // 0x00  0000 0000
      'B', // 0x01  0000 0001
      'C', // 0x02  0000 0010
      'D', // 0x03  0000 0011
      'E', // 0x04  0000 0100
      'F', // 0x05  0000 0101
      'G', // 0x06  0000 0110
      'H', // 0x07  0000 0111
      'I', // 0x08  0000 1000
      'J', // 0x09  0000 1001
      'K', // 0x0A  0000 1010
      'L', // 0x0B  0000 1011
      'M', // 0x0C  0000 1100
      'N', // 0x0D  0000 1101
      'O', // 0x0E  0000 1110
      'P', // 0x0F  0000 1111 (Hash_P)
                                 
      'Q', // 0x10  0001 0000
      'R', // 0x11  0001 0001
      'S', // 0x12  0001 0010
      'T', // 0x13  0001 0011
      'U', // 0x14  0001 0100
      'V', // 0x15  0001 0101
      'W', // 0x16  0001 0110
      'X', // 0x17  0001 0111
      'Y', // 0x18  0001 1000
      'Z', // 0x19  0001 1001
      'a', // 0x1A  0001 1010
      'b', // 0x1B  0001 1011
      'c', // 0x1C  0001 1100
      'd', // 0x1D  0001 1101
      'e', // 0x1E  0001 1110
      'f', // 0x1F  0001 1111 (Hash_f)
      
      'g', // 0x20  0010 0000
      'h', // 0x21  0010 0001
      'i', // 0x22  0010 0010
      'j', // 0x23  0010 0011
      'k', // 0x24  0010 0100
      'l', // 0x25  0010 0101
      'm', // 0x26  0010 0110
      'n', // 0x27  0010 0111
      'o', // 0x28  0010 1000
      'p', // 0x29  0010 1001
      'q', // 0x2A  0010 1010
      'r', // 0x2B  0010 1011
      's', // 0x2C  0010 1100
      't', // 0x2D  0010 1101
      'u', // 0x2E  0010 1110
      'v', // 0x2F  0010 1111
      
      'w', // 0x30  0011 0000
      'x', // 0x31  0011 0001
      'y', // 0x32  0011 0010
      'z', // 0x33  0011 0011 (Hash_z)
      '0', // 0x34  0011 0100
      '1', // 0x35  0011 0101
      '2', // 0x36  0011 0110
      '3', // 0x37  0011 0111
      '4', // 0x38  0011 1000
      '5', // 0x39  0011 1001
      '6', // 0x3A  0011 1010
      '7', // 0x3B  0011 1011
      '8', // 0x3C  0011 1100
      '9', // 0x3D  0011 1101
      '+', // 0x3E  0011 1110
      '/', // 0x3F  0011 1111 (HashEnd)
   } ;
   //* Hash indices for reverse lookup *
   //uint8_t Hash_P = 0x0F, Hash_f = 0x1F, Hash_z = 0x33, HashEnd = (bt64BYTES - 1) ;

   //* The base64 lookup table (above) is based on groups of three(3) source   *
   //* bytes. If the final group is incomplete i.e. 1 or 2 bytes, then this    *
   //* character is used to pad the output to a multiple of four(4) characters.*
   const char base64PADCHAR = '=' ;   // (0x3D)
   const char equChar = '=' ;
   const char quoChar = '?' ;
   //const char dquChar = '"' ;

   char  trgBuff[gsDFLTBYTES] ;              // target buffer
   const char* srcPtr = gsSrc.ustr() ;       // pointer to source data
   short srcCount = gsSrc.utfbytes() - 1,    // source bytes (not incl. nullchar)
         indx     = ZERO,                    // source index
         ondx     = ZERO,                    // target index
         w = 0, x = 1, y = 2, z = 3,         // indices into source array
         convCount = ZERO ;                  // return value (bytes decoded)
   uint8_t b1, b2, b3, b4 ;                  // 6-bit to 8-bit conversion
   bool writeB2 = true, writeB3 = true,      // 'false' if invalid character identified
        cleanBase64 = false,                 // 'true' if pretest indentifies base64 data
        status = true ;                      // 'false' if invalid byte data in input stream

   //* Perform a pretest of the data to verify whether *
   //* each data byte is valid base64 data.            *
   // Programmer's Note: This is not a definitive test that the data actually 
   // are base64, but the chances of error are rather small.
   cleanBase64 = true ;       // assume good data
   for ( indx = ZERO ; indx < srcCount ; ++indx )
   {
      if ( ! (((srcPtr[indx] >= 'A') && (srcPtr[indx] <= 'Z')) ||
              ((srcPtr[indx] >= 'a') && (srcPtr[indx] <= 'z')) ||
              ((srcPtr[indx] >= '0') && (srcPtr[indx] <= '9')) ||
              (srcPtr[indx] == '+') || (srcPtr[indx] == '/')   ||
              ((srcPtr[indx] == base64PADCHAR) && (indx >= (srcCount - 2)))) )
      {
         cleanBase64 = false ;
         break ;
      }
   }
   indx = ZERO ;        // reset index to top of data

   if ( cleanBase64  )
   {
      gsTrg.clear() ;                  // initialize target buffer

      #if DEBUG_base64Decode != 0
      if ( dbgofs.is_open() ) { dbgofs << srcPtr << endl ; }
      #endif   // DEBUG_base64Decode

      //* Decode the "base64" sequence *
      while ( indx < srcCount )
      {
         //* Get the 6-bit values corresponding to the source characters.*
         w = indx ; x = w + 1 ; y = x + 1 ; z = y + 1 ;
         for ( b1 = ZERO ; b1 < bt64BYTES ; ++b1 )
            if ( base64Table[b1] == srcPtr[w] )    break ;
         for ( b2 = ZERO ; b2 < bt64BYTES ; ++b2 )
            if ( base64Table[b2] == srcPtr[x] )    break ;
         for ( b3 = ZERO ; b3 < bt64BYTES ; ++b3 )
            if ( base64Table[b3] == srcPtr[y] )    break ;
         for ( b4 = ZERO ; b4 < bt64BYTES ; ++b4 )
            if ( base64Table[b4] == srcPtr[z] )    break ;

         #if DEBUG_base64Decode != 0
         if ( dbgofs.is_open() )
         { dbgofs << srcPtr[w] << srcPtr[x] << srcPtr[y] << srcPtr[z] << "  -  " ; }
         #endif   // DEBUG_base64Decode

         //* Test for invalid characters (or padding) in the stream.*
         writeB2 = writeB3 = true ;
         if ( (b1 == bt64BYTES) || (b2 == bt64BYTES) )
            status = false ;     // invalid character code
         else if ( b3 == bt64BYTES)
         {
            if ( srcPtr[y] == base64PADCHAR )   // padding character(s)
            { b3 = b4 = ZERO ; writeB2 = writeB3 = false ; }
            else                             // invalid character code
               status = false ;
         }
         else if ( b4 == bt64BYTES )
         {
            if ( srcPtr[z] == base64PADCHAR )   // padding character
            { b4 = ZERO ; writeB3 = false ; }
            else                             // invalid character code
               status = false ;
         }

         if ( status != false )
         {
            trgBuff[ondx++] = (uint8_t)((b1 << 2) | (b2 >> 4)) ;
            ++convCount ;
            if ( writeB2 )
            {
               trgBuff[ondx++] = (uint8_t)((b2 << 4) | ((b3 & 0x3F) >> 2)) ;
               ++convCount ;
               if ( writeB3 )
               {
                  trgBuff[ondx++] = (uint8_t)((b3 << 6) | b4) ;
                  ++convCount ;
               }
            }
            indx += 4 ;    // index next quartet

            #if DEBUG_base64Decode != 0
            if ( dbgofs.is_open() )
            { trgBuff[ondx] = NULLCHAR ; dbgofs << &trgBuff[ondx-3] << endl ; }
            #endif   // DEBUG_base64Decode

            //* If end of encoded sequence *
            if ( !writeB2 || !writeB3 )
               break ;
         }
         else        // invalid character in input stream
            break ;
      }     // while()

      if ( status != false )
      {
         //* Copy converted data to target *
         trgBuff[ondx] = NULLCHAR ;
         gsTrg = trgBuff ;

         //* Discard encoding delimiter chars *
         if ( (srcPtr[indx] == quoChar) && (srcPtr[indx + 1] == equChar) )
            indx += 2 ;

         //* Append the trailing unencoded data (if any) *
         if ( indx < srcCount )
            gsTrg.append( &srcPtr[indx] ) ;
      }
      else     // invalid base64 data, return unmodified data
         gsTrg = gsSrc ;
   }
   else        // encoding not detected, return unmodified data
      gsTrg = gsSrc ;

   return convCount ;

}  //* End deDecode_base64() *

//*************************
//*    deDecode_qprint    *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Given a byte-data sequence, if "quoted-printable" encoding is detected,    *
//* decode the sequence into plain text.                                       *
//* The decoded text will most likely be ASCII or UTF-8, but other formats     *
//* are possible e.g. Windoze ISO-8859-1.                                      *
//*                                                                            *
//* Input  : gsSrc : (by reference) potentially encoded data                   *
//*          gsTrg : (by reference, initial contents ignored                   *
//*                  receives decoded data                                     *
//*                                                                            *
//* Returns: 'true' if data decoded or if no decoding necessary                *
//*          'false' if ??                                                     *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* The "quoted-printable" format allows all 8-bit data to be represented.     *
//*                                                                            *
//* From Wikipedia: <https://en.wikipedia.org/wiki/Quoted-printable>           *
//* ----------------------------------------------------------------           *
//* Quoted-Printable, or QP encoding, is a binary-to-text encoding system      *
//* using printable ASCII characters (alphanumeric and the equals sign =) to   *
//* transmit 8-bit data over a 7-bit data path or, generally, over a medium    *
//* which is not 8-bit clean. It is defined as a MIME content transfer         *
//* encoding for use in e-mail.                                                *
//* QP works by using the equals sign = as an escape character. It also limits *
//* line length to 76, as some software has limits on line length.             *
//*                                                                            *
//* Any 8-bit byte value may be encoded with 3 characters: an = followed by    *
//* two hexadecimal digits (0–9 or A–F) representing the byte's numeric value. *
//*                                                                            *
//* All printable ASCII characters i.e. '!' (21h) through '~' (7Eh) may be     *
//* represented by themselves, EXCEPT: '=' (3Dh).                              *
//*                                                                            *
//* Tab (09h) and Space (20h) may represent themselves EXCEPT when they are    *
//* at the end of the line. In that case, they must be encoded: =09  =20       *
//*                                                                            *
//* To make the encoding smaller and easier to read the underscore is used     *
//* to represent the ASCII code for space creating the side effect that        *
//* underscore cannot be represented directly. Use of encoded words in certain *
//* parts of header fields imposes further restrictions on which characters    *
//* may be represented directly.                                               *
//*                                                                            *
//* Lines of Quoted-Printable encoded data must not be longer than 76          *
//* characters. To satisfy this requirement without altering the encoded text, *
//* soft line breaks may be added as desired. A soft line break consists of    *
//* an '=' at the end of an encoded line, and does not appear as a line break  *
//* in the decoded text.                                                       *
//*                                                                            *
//* A Literal CRLF sequence ("\r\n" 0d0a hex) ends a line. (These are stripped *
//* before the data reaches this method.)                                      *
//*                                                                            *
//* If carriage-return and/or newline are used in other ways, they must be     *
//* encodes as =0D =0A respectively.                                           *
//*                                                                            *
//* Programmer's Note: If an incomplete multi-byte character is decoded, at    *
//* the end of source data, then the character will be corrupted, and the      *
//* remainder of the multi-byte character (on the next line) will also be      *
//* corrupted.                                                                 *
//*               ---  ---  ---  ---  ---  ---  ---  ---                       *
//*                                                                            *
//* Special Case: quoted-printable endoding in message headers:                *
//* From Wikipedia <https://en.wikipedia.org/wiki/MIME#Encoded-Word>           *
//* ----------------------------------------------------------------           *
//* Encoded Word:                                                              *
//* Since RFC 2822, conforming message header field names and values use ASCII *
//* characters; values that contain non-ASCII data should use the MIME         *
//* encoded-word syntax (RFC 2047) instead of a literal string.                *
//* This syntax uses a string of ASCII characters indicating both the original *
//* character encoding (the "charset") and the content-transfer-encoding used  *
//* to map the bytes of the charset into ASCII characters.                     *
//* The form is: "=?charset?encoding?encoded text?=".                          *
//* "charset" may be any character set registered with IANA. Typically it      *
//*   would be the same charset as the message body.                           *
//* "encoding" can be either "Q" denoting Q-encoding that is similar to the    *
//*   quoted-printable encoding, or "B" denoting base64 encoding.              *
//* "encoded text" is the Q-encoded or base64-encoded text.                    *
//*                                                                            *
//* An encoded-word may not be more than 75 characters long, including         *
//* charset, encoding, encoded text, and delimiters. If it is desirable to     *
//* encode more text than will fit in an encoded-word of 75 characters,        *
//* multiple encoded-words (separated by CRLF SPACE) may be used.              *
//*                                                                            *
//******************************************************************************

static bool deDecode_qprint ( const gString& gsSrc, gString& gsTrg )
{
   const char equChar = '=' ;    // this is the escape character
   const char undChar = '_' ;    // underscore characters represent spaces

   const char* srcPtr = gsSrc.ustr() ;       // source pointer
   char  trgBuff[gsDFLTBYTES] ;              // output buffer
   short srcBytes = gsSrc.utfbytes(),        // source byte count
         indx = ZERO,                        // source index
         tindx = ZERO ;                      // target index
   uint8_t charTemp,                         // 8-bit half-character code
           charCode ;                        // 8-bit character code
   bool status  = true ;                     // return value

   for ( indx = ZERO ; indx < srcBytes ; )
   {
      if ( srcPtr[indx] == equChar )
      {
         ++indx ;                            // step past '='
         charCode = '\0' ;                   // initialize character code
         if ( indx <= (srcBytes - 3) )       // prevent buffer overrun
         {
            charTemp = srcPtr[indx++] ;      // convert 1st ASCII hex
            if ( charTemp >= '0' && charTemp <= '9' )
               charTemp -= '0' ;
            else if ( charTemp >= 'a' && charTemp <= 'f' )
               charTemp = (charTemp - 'a') + 0x0A ;
            else if ( charTemp >= 'A' && charTemp <= 'F' )
               charTemp = (charTemp - 'A') + 0x0A ;
            charCode = (uint8_t)(charTemp << 4) ;

            charTemp = srcPtr[indx++] ;      // convert 2nd ASCII hex
            if ( charTemp >= '0' && charTemp <= '9' )
               charTemp -= '0' ;
            else if ( charTemp >= 'a' && charTemp <= 'f' )
               charTemp = (charTemp - 'a') + 0x0A ;
            else if ( charTemp >= 'A' && charTemp <= 'F' )
               charTemp = (charTemp - 'A') + 0x0A ;
            charCode |= (uint8_t)(charTemp) ;
         }
         else  // probably a "soft-linebreak" (or encoding error)
         { indx = srcBytes ; }
         trgBuff[tindx++] = charCode ;
      }

      //* Replace underscore characters with spaces *
      else if ( srcPtr[indx] == undChar )
      { trgBuff[tindx++] = ' ' ; ++indx ; }

      //* Else, the character represents itself *
      else
         trgBuff[tindx++] = srcPtr[indx++] ;
   }
   trgBuff[tindx] = '\0' ;    // be sure sequence is null terminated
   gsTrg = trgBuff ;

   return status ;

}  //* End deDecode_qprint() *

//*************************
//*     deFormatLine      *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Apply output formatting to decoded display text.                           *
//*                                                                            *
//* Input  : gsOut   : (by reference) data to be formatted                     *
//*          termCols: number of display columns in terminal window            *
//*          colOut  : insertion point (column) of current display line        *
//*                                                                            *
//* Returns: number of data columns on last display line                       *
//******************************************************************************

static short deFormatLine ( gString& gsOut, short termCols, short colOut )
{
   const char *blcrColor = "\n\033[0;34m" ;  // display text following newline
   const wchar_t wNewline = L'\n' ;
   const wchar_t wSpace   = L' ' ;
   const short INDENT = 24 ;

   static gString tagBuff ;            // storage for partial HTML tags
   short indx,                         // text index
         tagbeg, tagend,               // HTML tag indices
         breakCol = termCols - INDENT, // 'safe' line-break column
         colCount = colOut ;           // return value

   //* Discard all carriage return characters *
   //*         (G/D Windoze garbage)          *
   while ( (indx = gsOut.find( L'\r' )) >= ZERO )
      gsOut.erase( L'\r', indx ) ;

   //* Replace TAB chars with spaces *
   if ( (gsOut.find( L'\t' )) >= ZERO )
      gsOut.replace( L'\t', L' ', ZERO, false, true ) ;

   //* If an incomplete tag at start of data * 
   tagbeg = gsOut.find( L'<' ) ;
   tagend = gsOut.find( L'>' ) ;
   if ( (tagend >= ZERO) && ((tagend < tagbeg) || (tagbeg == (-1))) )
   {
      //* Concatenate the tag end with previously-saved beginning.*
      tagBuff.append( gsOut.gstr() ) ;
      gsOut.erase( ZERO, (tagend + 1) ) ; // discard the partial tag

      //* Both <br> (line break) and <div> cause a newline in HTML output.*
      if ( (tagBuff.find( L"<br>" )) == ZERO )
         gsOut.insert( L'\n' ) ;
      else if ( (tagBuff.find( L"<div" )) == ZERO )
         gsOut.insert( L'\n' ) ;
      tagBuff.clear() ;                   // clear the buffer
   }

   //* Delete HTML tags *
   // Programmer's Note: Tag may begin in current data or may be continued
   // from previous data line. Tag may end in current data or continue to next 
   // data line. It is assumed that tag(s) will either begin or end in the
   // current data, or both begin and end.
   while ( (tagbeg = gsOut.find( L'<' )) >= ZERO )
   {
      if ( (tagend = gsOut.after( L'>', tagbeg )) > tagbeg )
      {
         //* Both <br> (line break) and <div> cause a newline in HTML output.*
         if ( (gsOut.find( L"<br>", tagbeg )) == tagbeg )
            gsOut.replace( L"<br>", L'\n', tagbeg ) ;
         else if ( (gsOut.find( L"<div>", tagbeg )) == tagbeg )
            gsOut.replace( L"<div>", L'\n', tagbeg ) ;
         //* Discard the unneeded tag.*
         else
            gsOut.erase( tagbeg, (tagend - tagbeg) ) ;
      }
      //* Incomplete tag. Copy to temp, then erase from stream.*
      else
      {
         tagBuff = &gsOut.gstr()[tagbeg] ;
         gsOut.erase( tagbeg, gsALLOCDFLT ) ;
      }
   }

   //* Replace HTML "entities" with the actual character.*
   const char* const nbSpace = "&nbsp;" ;    // non-breaking space
   const char* const lThan   = "&lt;" ;      // ('<')
   const char* const gThan   = "&gt;" ;      // ('>')
   const char* const aSand   = "&amp;" ;     // ('&')
   const char* const dQuot   = "&quot;" ;    // ('"')
   const char* const sQuot   = "&apos;" ;    // (''')
   const char* const Cent    = "&cent;" ;    // (cent coin '¢')
   const char* const Pound   = "&pound;" ;   // (pound sterling '£')
   const char* const Yen     = "&yen;" ;     // (yen '¥' [this is also yuan])
   const char* const Euro    = "&euro;" ;    // (euro '€')
   const char* const cRight  = "&copy;" ;    // (copyright symbol '©')
   const char* const tMark   = "&reg;" ;     // (registered trademark '®')

   if ( (gsOut.find( nbSpace )) >= ZERO )
      gsOut.replace( nbSpace, L' ', ZERO, false, true ) ;
   if ( (gsOut.find( lThan )) >= ZERO )
      gsOut.replace( lThan, L'<', ZERO, false, true ) ;
   if ( (gsOut.find( gThan )) >= ZERO )
      gsOut.replace( gThan, L'>', ZERO, false, true ) ;
   if ( (gsOut.find( aSand )) >= ZERO )
      gsOut.replace( aSand, L'&', ZERO, false, true ) ;
   if ( (gsOut.find( dQuot )) >= ZERO )
      gsOut.replace( dQuot, L'"', ZERO, false, true ) ;
   if ( (gsOut.find( sQuot )) >= ZERO )
      gsOut.replace( sQuot, L'\'', ZERO, false, true ) ;
   if ( (gsOut.find( Cent )) >= ZERO )
      gsOut.replace( Cent, L'¢', ZERO, false, true ) ;
   if ( (gsOut.find( Pound )) >= ZERO )
      gsOut.replace( Pound, L'£', ZERO, false, true ) ;
   if ( (gsOut.find( Yen )) >= ZERO )
      gsOut.replace( Yen, L'¥', ZERO, false, true ) ;
   if ( (gsOut.find( Euro )) >= ZERO )
      gsOut.replace( Euro, L'€', ZERO, false, true ) ;
   if ( (gsOut.find( cRight )) >= ZERO )
      gsOut.replace( cRight, L'©', ZERO, false, true ) ;
   if ( (gsOut.find( tMark )) >= ZERO )
      gsOut.replace( tMark, L'®', ZERO, false, true ) ;

   //* Format line breaks to fit the display area *
   const wchar_t *wPtr = gsOut.gstr() ; // array of wchar_t characters
   int   lineChars ;                    // chars in formatted line
   //* Array of column counts in formatted line *
   const short *colPtr = gsOut.gscols( lineChars ) ;
   for ( indx = ZERO ; indx < lineChars ; ++indx )
   {
      //* If a character which requires one or mor display columns *
      if ( colPtr[indx] > ZERO )
      {
         //* If within line width *
         if ( (colCount + colPtr[indx]) < breakCol )
            colCount += colPtr[indx] ;

         //* At threshold of line width *
         else
         {
            //* Scan to next space (or newline) *
            for ( short loop = INDENT ; (loop > ZERO) && (indx < lineChars) ; --loop )
            {
               if ( colPtr[indx] > ZERO )
               {
                  if ( wPtr[indx] == wSpace )
                  {
                     //* Insert a newline and refresh the tracking data *
                     ++indx ;       // set index after the space
                     gsOut.insert( wNewline, indx ) ;
                     colPtr = gsOut.gscols( lineChars ) ;
                     //++indx ;     // will be advanced by the loop
                     colCount = ZERO ;
                     break ;
                  }
                  else     // not a natural breakpoint
                     colCount += colPtr[indx++] ;
               }
               else if ( wPtr[indx] == wNewline )  // existing linebreak ;
               {
                  colCount = ZERO ;
                  //++indx ;     // will be advanced by the loop
                  break ;
               }
               else     // unknown zero-width character
                  ++indx ;
            }

            //* If no natural breakpoint found, force a line break.*
            if ( colCount >= breakCol )
            {
               gsOut.insert( wNewline, indx ) ;
               colPtr = gsOut.gscols( lineChars ) ;
               //++indx ;     // will be advanced by the loop
               colCount = ZERO ;
            }
         }
      }

      //* If newline found, reset column count *
      else if ( wPtr[indx] == wNewline )
      {
         colCount = ZERO ;
      }

      //* Other zero-width character, do nothing.*
      else
         ;
   }        // for(;;)

   //* If embedded newline chars, set display color *
   //* after each newline.                          *
   if ( (gsOut.find( L'\n' )) >= ZERO )
      gsOut.replace( L'\n', blcrColor, ZERO, false, true ) ;

   return colCount ;

}  //* End deFormatLine() *

//*************************
//* vfcFormatHTML_Source  *
//*************************
//******************************************************************************
//* Format an HTML source code document for display by the 'less' utility.     *
//* Add ANSI color syntax highlighting.                                        *
//*                                                                            *
//* Input  : srcPath: (by reference) file containing original HTML source code *
//*          tmpPath: (by reference) temp file to receive formatted source code*
//*                                                                            *
//* Returns: 'true' if operation is successful, else 'false'                   *
//******************************************************************************

bool FileDlg::vfcFormatHTML_Source ( const gString& srcPath, const gString& tmpPath )
{
   const wchar_t *MIMEtype = L"<!DOCTYPE HTML",
                  dQuote   = L'"',
                 *openCom  = L"<!--",
                 *closeCom = L"-->" ;
   const short    MIMEtypeLen = 14,
                  maxINBYTES = (gsDFLTBYTES - 200) ; // max input line bytes
   const char rAngle   = '>' ;
              //* ANSI color codes *
   const char *bkColor = "\033[0;30m", // tag delimiters
              *maColor = "\033[0;35m", // HTML command
              *grColor = "\033[0;32m", // HTML command arguments
              *blColor = "\033[0;34m", // display text
              *reColor = "\033[1;31m" ;// HTML comments
   char inBuff[gsDFLTBYTES] ;          // input buffer
   gString gsOut,                      // output buffer
           gsIn ;                      // input parsing
   short indx ;                        // index for parsing and formatting data
   bool mlComment = false,             // 'true' if comment spans multiple lines
        status = false ;               // return value

   //* Open source and target files *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( (ifs.is_open()) && (ofs.is_open()) )
   {
      //* Copy the MIME-type. (If test fails, not a valid HTML file.) *
      ifs.getline( inBuff, maxINBYTES, nckNEWLINE ) ;
      indx = (ifs.gcount() - 2) ;   // protect against Windoze CR/LF combo
      if ( (indx > ZERO) && (inBuff[indx] == '\r') )
         inBuff[indx--] = NULLCHAR ;
      gsOut = inBuff ;
      if ( (ifs.good() || (ifs.gcount() > ZERO)) &&
           ((gsOut.compare( MIMEtype, false, MIMEtypeLen )) == ZERO) &&
           ((inBuff[indx]) == rAngle) )
      {
         gsOut.insert ( maColor, 2 ) ;
         indx = ZERO ;
         while ( (indx = gsOut.find( dQuote, indx )) > ZERO )
         {
            gsOut.insert( grColor, indx++ ) ;
            indx += 7 ; // step over color sequence
            if ( (indx = gsOut.after( dQuote, indx )) > ZERO )
               gsOut.insert( bkColor, indx++ ) ;
         }
         if ( (indx = gsOut.find( L'>' )) > ZERO )
            gsOut.insert( bkColor, indx ) ;
         ofs << gsOut.ustr() << '\n' ;

         // Programmer's Note: We do not test for buffer overflow during capture 
         // and colorization. No real-world line even approaches 4,000 bytes.
         bool done = false ;
         while ( ! done )
         {
            ifs.getline( inBuff, maxINBYTES, nckNEWLINE ) ;
            if ( ifs.gcount() > ZERO )
            {
               indx = (ifs.gcount() - 2) ;   // protect against Windoze CR/LF combo
               if ( (indx > ZERO) && (inBuff[indx] == '\r') )
                  inBuff[indx--] = NULLCHAR ;
               gsIn = inBuff ;

               //* If a multi-line comment continues from previous line *
               if ( mlComment )
               {
                  if ( (indx = gsIn.find( closeCom )) >= ZERO )
                  {
                     gsIn.substr( gsOut, ZERO, indx ) ;
                     gsOut.insert( reColor ) ;
                     gsOut.append( bkColor ) ;
                     ofs << gsOut.ustr() ;
                     gsIn.shiftChars( -(indx) ) ;
                     mlComment = false ;
                  }
                  else     // entire line is a comment
                  {
                     gsOut.compose( "%s%S%s", reColor, gsIn.gstr(), bkColor ) ;
                     ofs << gsOut.ustr() << endl ;
                     continue ;     // read the next source line
                  }
               }

               do
               {
                  if ( (htmlExtractTag ( gsIn, gsOut )) != false )
                  {
                     //* If tag is a comment *
                     if ( (gsOut.find( openCom )) == ZERO )
                     {
                        gsOut.insert( reColor, 4 ) ;
                        if ( (indx = gsOut.find( closeCom )) >= ZERO )
                           gsOut.insert( bkColor, indx ) ;
                        else if ( (indx = gsIn.find( closeCom )) >= ZERO )
                        {
                           for ( short i = ZERO ; i < indx ; ++i )
                              gsOut.append( "%C", &gsIn.gstr()[i] ) ;
                           gsOut.append( bkColor ) ;
                           gsIn.shiftChars( -(indx) ) ;
                        }
                        else  // comment continues to next line
                        {
                           gsOut.append( "%S%s", gsIn.gstr(), bkColor ) ;
                           gsIn.clear() ;    // terminate inner do/while() loop
                           mlComment = true ;// multi-line comment
                        }
                     }

                     //* Else tag is a command *
                     else
                     {
                        indx = 1 ;
                        if ( (gsOut.find( fSLASH )) == 1 )
                           ++indx ;
                        gsOut.insert( maColor, indx ) ;
                        indx += 7 ; // step over color sequence
   
                        while ( (indx = gsOut.find( dQuote, indx )) > ZERO )
                        {
                           gsOut.insert( grColor, indx++ ) ;
                           indx += 7 ; // step over color sequence
                           if ( (indx = gsOut.after( dQuote, indx )) > ZERO )
                              gsOut.insert( maColor, indx++ ) ;
                        }
                        if ( (indx = gsOut.find( L'>' )) > ZERO )
                        {
                           if ( gsOut.gstr()[indx - 1] == fSLASH )
                              --indx ;
                           gsOut.insert( bkColor, indx ) ;
                        }
                     }
                     ofs << gsOut.ustr() ;
                  }

                  //* If an incomplete tag OR non-tag data returned.*
                  else if ( (gsOut.gschars()) > 1 )
                  {
                     gsOut.insert( blColor ) ;
                     gsOut.append( bkColor ) ;
                     ofs << gsOut.ustr() ;
                  }
               }
               while ( (gsIn.gschars()) > 1 ) ;
               ofs << endl ;     // end-of-line
            }
            else
               done = true ;
         }  // while()
         status = true ;         // return success
      }  // valid MIME-type

      ifs.close() ;              // close the source file
      ofs.close() ;              // close the target file
   }

   return status ;

}  //* End vfcFormatHTML_Source() *

//*************************
//*    htmlExtractTag     *
//*************************
//******************************************************************************
//* Scan the raw input data and if a complete HTML tag "< ... >" is found,     *
//* move it to the output buffer.                                              *
//*                                                                            *
//* If input does not begin with a tag, then return all the data:              *
//*  a) through the end-of-tag character, or                                   *
//*  b) up to (but not including) the beginning of the next tag, or            *
//*  c) through end of input                                                   *
//*                                                                            *
//* Input  : gsIn   : (by reference) raw source input                          *
//*          gsOut  : (by reference) receives extracted tag data (if found)    *
//*                   if an HTML tag (command) is not the first thing in gsIn, *
//*                   then return the data _up to_ the start of the next tag   *
//*                                                                            *
//* Returns: 'true'  if a complete command is extracted                        *
//*          'false' if incomplete tag, no tag or no data extracted            *
//******************************************************************************

static bool htmlExtractTag ( gString& gsIn, gString& gsOut )
{
   const char lAngle   = '<',          // local char definitions
              rAngle   = '>' ;

   wchar_t w[gsALLOCDFLT] ;            // work buffer
   short wIndex = ZERO,                // index into work buffer
         iChars = gsIn.gschars() ;     // number of input characters
   bool goodTag = false ;              // return value

   gsIn.copy( w, gsALLOCDFLT ) ;       // make a working copy of input data

   //* Left angle bracket begins a tag.*
   if ( w[ZERO] == lAngle )
   {
      //* Find the bracket that ends the tag.*
      while ( (w[++wIndex] != rAngle) && (wIndex < iChars) ) ;
      if ( w[wIndex] == rAngle )
         w[++wIndex] = NULLCHAR ;
      gsOut = w ;

      if ( wIndex < iChars )     // shift the copied data out
      {
         gsIn.shiftChars( -(wIndex) ) ;
         goodTag = true ;
      }
      else                       // else, all data copied to output
         gsIn.clear() ;
   }

   //* Data does not begin with a tag.                *
   //* Scan to recognizable marker (see above notes). *
   else
   {
      while ( (w[wIndex] != lAngle) && (w[wIndex] != rAngle) && (wIndex < iChars) )
         ++wIndex ;
      if ( w[wIndex] == lAngle )
         w[wIndex] = NULLCHAR ;
      else if ( w[wIndex] == rAngle )
         w[++wIndex] = NULLCHAR ;
      else
         gsIn.clear() ;
      gsOut = w ;
      gsIn.shiftChars( -(wIndex) ) ;
   }

   return goodTag ;

}  //* End htmlExtractTag() *

//*************************
//*  vfcFormatXML_Source  *
//*************************
//******************************************************************************
//* Format an XML source code document for display by the 'less' utility.      *
//* For stand-alone XML documents (which already have linefeed formatting),    *
//* just add ANSI color-coding. However, if linefeeds are not found, add       *
//* linefeed formatting as well as ANSI color.                                 *
//*                                                                            *
//* Input  : srcPath: (by reference) file containing original XML source code  *
//*          tmpPath: (by reference) temp file to receive formatted source code*
//*                                                                            *
//* Returns: 'true' if operation is successful, else 'false'                   *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The XML file extracted from an OpenDocument file does not use newline      *
//* Special Case: The "Content.xml" file extracted from OpenDocument files     *
//* does not use newline characters beyond the MIME-type line.                 *
//* Presumably this is done to save space, but file is pretty much unreadable  *
//* without line breaks. Therefore, we must insert them into the output stream.*
//* Note that the test this is not definitive, but viewing this type of file   *
//* is primarily for development, and users will seldom want to view this kind *
//* file anyway.                                                               *
//* -- How do browsers decide where linebreaks go when displaying XML files    *
//*    that don't have them? Is there a convention about breaking at the end   *
//*    of a command? Where else?                                               *
//*    In general, each command stands alone on a line, except when the        *
//*    command sequence encloses text data. A short sequence of                *
//*    <cmd>stuff>/cmd> is sometimes all on one line.                          *
//*                                                                            *
//* Other XML source files usually DO have linefeeds to format the source.     *
//* Note that Windoze-generated XML will (stupidly) have carriage-returns also.*
//*                                                                            *
//* XML viewers, such as browsers and XML (text) editors: How they do it:      *
//* 1) Each tag is on a line by itself. (with exceptions)                      *
//*    a) The '<' or '</' is black text.                                       *
//*    b) The first token of the tag is an outstanding color                   *
//*       (blue, magenta, etc.) which differs by viewer.                       *
//*    c) Second and subsequent tokens are black, EXCEPT text enclosed within  *
//*       quotations: <style:style style:name="P1" style:family="paragraph">   *
//*       where "P1" and "paragraph" (including the quotes) are an outstanding *
//*       color _different_ from the color of the first token.                 *
//*    d) The '>' or '/>' is black text.                                       *
//*                                                                            *
//*                                                                            *
//* Possible Enhancements:                                                     *
//* 1) Colorize comments: <!--This is a comment.-->  (grey?)                   *
//* 2) Colorize C-data sequences (literal strings): <![CDATA[...]]> (red?)     *
//*                                                                            *
//*                                                                            *
//* ANSI Colors (partial list):                                                *
//* =========== Foreground  Background   ============ Foreground  Background   *
//* Black       \033[0;30m  \033[0;40m   Dark Gray    \033[1;30m  \033[1;40m   *
//* Red         \033[0;31m  \033[0;41m   Bold Red     \033[1;31m  \033[1;41m   *
//* Green       \033[0;32m  \033[0;42m   Bold Green   \033[1;32m  \033[1;42m   *
//* Yellow      \033[0;33m  \033[0;43m   Bold Yellow  \033[1;33m  \033[1;43m   *
//* Blue        \033[0;34m  \033[0;44m   Bold Blue    \033[1;34m  \033[1;44m   *
//* Purple      \033[0;35m  \033[0;45m   Bold Purple  \033[1;35m  \033[1;45m   *
//* Cyan        \033[0;36m  \033[0;46m   Bold Cyan    \033[1;36m  \033[1;46m   *
//* Light Gray  \033[0;37m  \033[0;47m   White        \033[1;37m  \033[1;47m   *
//******************************************************************************

bool FileDlg::vfcFormatXML_Source ( const gString& srcPath, const gString& tmpPath )
{
   const char lAngle   = '<',          // local char definitions
              rAngle   = '>',
              ampChar  = '&',
              semiCol  = ';',
              *bkColor = "\033[0;30m", // ANSI color codes
              *grColor = "\033[0;32m",
//              *blColor = "\033[0;34m",    // (insufficient contrast)
              *maColor = "\033[0;35m",
              *reColor = "\033[1;31m" ;
   const wchar_t *MIMEtype = L"<?xml",
                  dQuote   = L'"' ;
   const short    MIMEtypeLen = 5 ;
   char inBuff[gsDFLTBYTES] ;          // input buffer
   gString gsOut ;                     // output buffer
   short indx ;                        // index for parsing and formatting data
   bool status = false ;               // return value

   //* Special Case: The "Content.xml" file of OpenDocument files *
   //* does not use newline characters beyond the MIME-type line. *
   //* Therefore, we must insert them into the output stream.     *
   //* Note that this is not a definitive test, but viewing this  *
   //* type of file is primarily for development, not public view.*
   bool insertNewlines = false ;
   if ( ((indx = srcPath.findlast( fSLASH )) >= ZERO) &&
        ((srcPath.find( "content", ++indx)) >= ZERO) )
      insertNewlines = true ;

   //* Open source and target files *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( (ifs.is_open()) && (ofs.is_open()) )
   {
      //* Copy the MIME-type. (If test fails, not a valid XML file.) *
      ifs.getline( inBuff, gsDFLTBYTES, nckNEWLINE ) ;
      indx = (ifs.gcount() - 2) ;   // protect against Windoze CR/LF combo
      if ( (indx > ZERO) && (inBuff[indx] == '\r') )
         inBuff[indx--] = NULLCHAR ;
      gsOut = inBuff ;
      if ( (ifs.good() || (ifs.gcount() > ZERO)) &&
           ((gsOut.compare( MIMEtype, false, MIMEtypeLen )) == ZERO) && 
           ((inBuff[indx]) == rAngle) )
      {
         gsOut.insert ( maColor, 2 ) ;
         indx = ZERO ;
         while ( (indx = gsOut.find( dQuote, indx )) > ZERO )
         {
            gsOut.insert( grColor, indx++ ) ;
            indx += 7 ; // step over color sequence
            if ( (indx = gsOut.after( dQuote, indx )) > ZERO )
               gsOut.insert( bkColor, indx++ ) ;
         }
         if ( (indx = gsOut.find( L'>' )) > ZERO )
            gsOut.insert( bkColor, indx ) ;
         gsOut.append( L'\n' ) ;
         ofs.write( gsOut.ustr(), (gsOut.utfbytes() - 1) ) ;

         bool done = false ;
         while ( ! done )
         {
            indx = ZERO ;
            ifs.read( inBuff, 1 ) ;
            if ( ifs.gcount() == 1 )
            {
               //* If a tag identified, get remainder of the tag.*
               if ( inBuff[indx] == lAngle )
               {  // Programmer's Note: We do not test for buffer overflow 
                  // during capture of the tag. No real-world tag even approaches 
                  // 4,000 bytes. The longest we have seen is about 2,100 bytes 
                  // and that was in a malformed OpenDocument file.
                  while ( true )
                  {
                     ifs.read( &inBuff[++indx], 1 ) ;
                     if ( ifs.gcount() == 1 )
                     {
                        if ( inBuff[indx] == rAngle )    // end of tag
                        {  //* Complete tag captured *
                           inBuff[++indx] = NULLCHAR ;   // terminate the string
                           gsOut = inBuff ;

                           indx = ZERO ;                 // remove control chars
                           while ( (indx = gsOut.find( L'\r', indx )) >= ZERO )
                              gsOut.erase( L'\r' ) ;
                           indx = ZERO ;
                           while ( (indx = gsOut.find( L'\n', indx )) >= ZERO )
                              gsOut.erase( L'\n' ) ;

                           //* Delimit and colorize first token.*
                           //* Open tag : "<" or "</"
                           //* Token ends with: " " or ">" or "/>"
                           //* a) "<token:token " b) "<token:token>" c) "<token:token/>"
                           //* d) "</token:token " e) </token:token> f) "<
                           indx = 1 ;
                           indx = (gsOut.gstr()[1] == fSLASH) ? 2 : 1 ;
                           gsOut.insert ( maColor, indx ) ;
                           if ( (indx = gsOut.find( nckSPACE )) > ZERO )
                              gsOut.insert( bkColor, indx ) ;
                           else if ( (indx = gsOut.find( rAngle )) > ZERO )
                              gsOut.insert ( bkColor, indx ) ;

                           //* Scan for data between quotation marks.*
                           //* (quotes _should_ come in pairs)       *
                           indx = ZERO ;
                           while ( (indx = gsOut.find( dQuote, indx )) > ZERO )
                           {
                              //* Insert color sequence before opening quote.*
                              gsOut.insert( grColor, indx ) ;
                              //* Step over color sequence and the quote *
                              indx += strlen ( grColor ) + 1 ;
                              if ( (indx = gsOut.after( dQuote, indx )) > ZERO )
                                 gsOut.insert( bkColor, indx++ ) ;
                           }

                           //* Closing angle bracket is black, and if *
                           //* present, the "/>" sequence is black.   *
                           if ( (indx = gsOut.find( L'>' )) > ZERO )
                           {
                              if ( (indx > ZERO) && 
                                   ((gsOut.gstr()[indx - 1]) == fSLASH) )
                                 --indx ;
                              gsOut.insert( bkColor, indx ) ;
                           }

                           if ( insertNewlines )
                              gsOut.append( L'\n' ) ;
                           ofs.write( gsOut.ustr(), (gsOut.utfbytes() - 1) ) ;
                           break ;
                        }
                     }
                     else
                     {
                        done = true ;
                        break ;
                     }
                  }
               }

               //* Check for control characters *
               else if ( (inBuff[indx] >= ZERO) && (inBuff[indx] < nckSPACE) )
               {
                  if ( inBuff[ZERO] != '\r' )
                     ofs.put( inBuff[ZERO] ) ;
               }

               //* Else, assume text content and copy it to output.*
               else
               {
                  // Programer's Note: See note above about "predefined entities" 
                  // and literal values.
                  //* The '&' character begins "predefined entities".        *
                  //* We (foolishly) rely on the data to have correct syntax.* 
                  if ( inBuff[ZERO] == ampChar )
                  {
                     indx = ZERO ;
                     do
                     {
                        ifs.read( &inBuff[++indx], 1 ) ;
                     }
                     while ( (inBuff[indx] != semiCol) && (ifs.gcount() == 1) ) ;
                     inBuff[++indx] = NULLCHAR ;
                     gsOut = inBuff ;
                     gsOut.insert( reColor ) ;
                     gsOut.append( bkColor ) ;
                     ofs.write( gsOut.ustr(), (gsOut.utfbytes() - 1) ) ;
                  }
                  else           // not a special character
                     ofs.put( inBuff[ZERO] ) ;
               }
            }
            else
               done = true ;
         }
         status = true ;         // return success
      }

      ifs.close() ;              // close the source file
      ofs.close() ;              // close the target file
   }
   return status ;

}  //* End vfcFormatXML_Source() *

//*************************
//* vfcFormatPERL_Source  *
//*************************
//********************************************************************************
//* Format a Perl source code document with ANSI color coding for display by     *
//* the 'less' utility.                                                          *
//* (Caller has verified that target contains Perl code.)                        *
//*                                                                              *
//* Input  : srcPath: (by reference) file containing Perl source code            *
//*          tmpPath: (by reference) temp file to receive formatted source code  *
//*                                                                              *
//* Returns: 'true' if operation is successful, else 'false'                     *
//********************************************************************************
//* Notes:                                                                       *
//* 1) Perl is written primarily by people who are too lazy and undisciplined    *
//*    to care about good programming practices or even abount proper syntax.    *
//*    For this reason, it is extremely difficult to parse Perl code             *
//*    accurately; therefore, the results of our scan are not particularly       *
//*    elegant. That said, Perl is a wonderful tool in the right hands, and      *
//*    it deserves respect and some special attention. Color coding seemed       *
//*    to be a good beginning.  -- Software Sam 2020-06-01                       *
//*                                                                              *
//* 2) Colorizing of comments was lifted directly from our SourceProfiler        *
//*    utility.                                                                  *
//*                                                                              *
//* 3) Colorizing of non-comment tokens: Perl commands, operators, variable      *
//*    names, literals and other token types is modelled on the color coding     *
//*    used by the jEdit and Bluefish editors.                                   *
//*    a) Numeric constants are not identified here. Syntax for scalar values    *
//*       is simply too complex for such a simple parsing algorithm.             *
//*                                                                              *
//* 4) See list of ANSI color codes in 'vfcFormatXML_Source() above.             *
//*                                                                              *
//********************************************************************************

bool FileDlg::vfcFormatPERL_Source ( const gString& srcPath, const gString& tmpPath )
{
              //* ANSI color codes *
   const char *dfltColor = "\033[0;30m", // default color
              *commColor = "\033[0;33m", // comment color
              *cmdColor  = "\033[0;34m", // Perl command color
              *txtColor  = "\033[1;35m", // string literal color (bold purple)
//              *numColor  = "\033[1;31m", // numeric literals
              *varColor  = "\033[0;32m"; // variable name color
   const wchar_t wHASH    = L'#' ;     // hash character identifies comments
   const wchar_t wDOLLAR  = L'$' ;     // '$' indicates a variable name
   const wchar_t wAMPERE  = L'@' ;     // '@' indicates an array
   const wchar_t wDBLQ    = L'"' ;     // '"' string-data delimiter
   const wchar_t wUNSCORE = L'_' ;     // underscore character
   const wchar_t LBRACE   = (L'{') ;   // left curley brace
   const wchar_t RBRACE   = (L'}') ;   // right curley brace
   const wchar_t LBRACKET = (L'[') ;   // left curley brace
   const wchar_t RBRACKET = (L']') ;   // right curley brace
   const wchar_t LPAREN   = (L'(') ;   // left curley brace
   const wchar_t RPAREN   = (L')') ;   // right curley brace
   const wchar_t LANGLE   = (L'<') ;   // left angle bracket
   const wchar_t RANGLE   = (L'>') ;   // right angle bracket
   const wchar_t LDANGLE  = (L'«') ;   // left double-angle quotation mark
   const wchar_t RDANGLE  = (L'»') ;   // right double-angle quotation mark
   const wchar_t* OPENBLK = L"=begin comment" ;
   const wchar_t* CLOSEBLK1 = L"=end comment" ;
   const wchar_t* CLOSEBLK2 = L"=cut" ;

   char inBuff[gsDFLTBYTES] ;          // input buffer
   gString gsOut,                      // output buffer
           gsIn,                       // input parsing
           gstmp ;                     // temp storage
   wchar_t bdelim,                     // index beginning delimiter for block comment
           edelim ;                    // index ending delimiter for block comment
   short indx ;                        // index for parsing and formatting data
   bool  open_comment = false,         // 'true' if block comment is open
         open_podcomment = false,      // 'true' if POD block comment is open
         status = false ;              // return value

   //* Open source and target files *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( (ifs.is_open()) && (ofs.is_open()) )
   {
      bool done = false ;
      while ( ! done )
      {
         ifs.getline( inBuff, gsDFLTBYTES, nckNEWLINE ) ;
         if ( ifs.gcount() > ZERO )
         {
            indx = (ifs.gcount() - 2) ;   // protect against Windoze CR/LF combo
            if ( (indx > ZERO) && (inBuff[indx] == '\r') )
               inBuff[indx--] = NULLCHAR ;
            gsIn = inBuff ;

            indx = gsIn.scan() ;

            //* Blank line or whitespace only? *
            if ( gsIn.gstr()[indx] == NULLCHAR )
               ofs << gsIn.ustr() << endl ;

            //* If an open comment block *
            else if ( open_comment )
            {
               ofs << commColor << gsIn.ustr() << dfltColor << endl ;
               if ( (gsIn.find( edelim, indx )) >= ZERO )
                  open_comment = false ;
            }

            //* If an open pod-comment block *
            else if ( open_podcomment )
            {
               ofs << commColor << gsIn.ustr() << dfltColor << endl ;
               if ( ((gsIn.find( CLOSEBLK1, indx )) >= ZERO) ||
                    ((gsIn.find( CLOSEBLK2, indx )) >= ZERO) )
               { open_podcomment = false ; }
            }

            //* Whole line is a comment _OR_ begin a delimited comment block *
            else if ( ((gsIn.gstr()[indx]) == wHASH ) )
            {
               bdelim = gsIn.gstr()[indx + 1] ; // character following the '#'
               if ( bdelim == LBRACE || bdelim == LBRACKET || bdelim == LPAREN || 
                    bdelim == LANGLE || bdelim == LDANGLE )
               {
                  //* Scan to end of current line for closing delimiter.*
                  //* If not found, then comment continues on next line.*
                  switch ( bdelim )
                  {
                     case LBRACE:   edelim = RBRACE ;    break ;
                     case LBRACKET: edelim = RBRACKET ;  break ;
                     case LPAREN:   edelim = RPAREN ;    break ;
                     case LANGLE:   edelim = RANGLE ;    break ;
                     case LDANGLE:  edelim = RDANGLE ;   break ;
                  }
                  if ( (gsIn.find( edelim, (indx + 2) )) < ZERO )
                     open_comment = true ;
               }
               ofs << commColor << gsIn.ustr() << dfltColor << endl ;
            }

            //* If opening a pod-comment block *
            else if ( (gsIn.find( OPENBLK, indx )) == indx )
            {
               ofs << commColor << gsIn.ustr() << dfltColor << endl ;
               open_podcomment = true ;   // and it continues to the next line
            }

            //* Parse each token on the line *
            else
            {
               //* Output any leading whitespace *
               if ( indx > ZERO )
               {  // Programmer's Note: all leading whitespace chars converted to ' '.
                  for ( short i = ZERO ; i < indx ; ++i )
                     ofs << ' ' ;
                  gsIn.shiftChars( -(indx) ) ;
                  indx = ZERO ;
               }

               while ( (vfcGetPerlToken ( gsIn, gsOut )) )
               {
                  //* If a comment which follow code on the line *
                  if ( (indx = gsOut.gstr()[ZERO]) == wHASH )
                  {
                     ofs << commColor << gsOut.ustr() 
                         << gsIn.ustr() << dfltColor ;
                     gsIn.clear() ;    // all data for this line written
                     break ;
                  }

                  //* If a literal string, it needs further parsing *
                  //* to locate any embedded variable names.        *
                  else if ( gsOut.gstr()[ZERO] == wDBLQ )
                  {
                     if ( (gsOut.find( wDOLLAR )) > ZERO )
                     {
                        wchar_t wchar ;
                        bool    varname = false ;
                        gstmp = txtColor ;

                        while ( gsOut.gschars() > 1 )
                        {
                           //* Extract a character *
                           wchar = gsOut.gstr()[ZERO] ;
                           gsOut.shiftChars( -1 ) ;

                           //* Copy plain text character to temp buffer.*
                           if ( !varname && (wchar != wDOLLAR) )
                              gstmp.append( wchar ) ;

                           //* Variable name identified *
                           else if ( wchar == wDOLLAR )
                           {
                              if ( gstmp.gschars() > 1 )
                                 ofs << gstmp.ustr() ;
                              gstmp.compose( "%s%C", varColor, &wchar ) ;
                              varname = true ;
                           }

                           //* If continuation of variable name *
                           else if ( varname )
                           {
                              if ( (wchar >= L'A' && wchar <= L'Z') ||
                                   (wchar >= L'a' && wchar <= L'z') ||
                                   (wchar >= L'0' && wchar <= L'9') ||
                                   (wchar == wUNSCORE) )
                              {
                                 gstmp.append( wchar ) ;
                              }
                              else
                              {
                                 if ( gstmp.gschars() > 1 )
                                    ofs << gstmp.ustr() ;
                                 gstmp.compose( "%s%C", txtColor, &wchar ) ;
                                 varname = false ;
                              }
                           }
                        }
                        gstmp.append( dfltColor ) ;
                        ofs << gstmp.ustr() ;
                     }
                     else
                        ofs << txtColor << gsOut.ustr() << dfltColor ;
                  }

                  //* If Perl command word *
                  else if ( (vfcIsPerlCommand ( gsOut )) )
                  {
                     ofs << cmdColor << gsOut.ustr() << dfltColor ;
                  }

                  //* If variable name or array name *
                  else if ( (gsOut.gstr()[ZERO] == wDOLLAR) ||
                            (gsOut.gstr()[ZERO] == wAMPERE) )
                  {
                     ofs << varColor << gsOut.ustr() << dfltColor ;
                  }

                  //* Unidentified token. Output with default color.*
                  else
                  {
                     ofs << dfltColor << gsOut.ustr() ;
                  }
               }     // while()

               //* Line has been formatted and written to target.*
               //* Terminate with newline and flush buffer.      *
               ofs << endl ;
            }
         }
         else     // end-of-file
            done = true ;
      }  // while()
      status = true ;         // return success

      ifs.close() ;              // close the source file
      ofs.close() ;              // close the target file
   }     // files open
   return status ;

}  //* End vfcFormatPERL_Source() *

//*************************
//*    vfcIsPerlSource    *
//*************************
//********************************************************************************
//* Non-member Method                                                            *
//* -----------------                                                            *
//* 1) Test the filename extension of the given filespec.                        *
//*    a) If it is one of the known Perl filename extensions, assume that it     *
//*       is Perl code _unless_ 'strict' flag is set.                            *
//*    b) If no filename extension identified, first line of file determines     *
//*       whether file contains Perl code.                                       *
//* 2) Test the first line of the source file to determine whether file          *
//*    actually contains Perl code.                                              *
//*    a) If 'strict' flag is set, then regardless of filename extension,        *
//*       first line of file determines whether file is actually Perl code.      *
//*    b) If no filename extension identified, test the first line of file.      *
//*    The first line should be similar to:                                      *
//*                    #!/usr/bin/perl                                           *
//*    although the actual path may vary by installation.                        *
//*                                                                              *
//* Input  : trgPath : full filespec of target file                              *
//*          strict  : (optional, 'false' by default)                            *
//*                    if 'false', assume that a Perl filename extension         *
//*                                indicates Perl code                           *
//*                    if 'true',  if a Perl filename extension or if no         *
//*                                extension, verify that the first line of      *
//*                                the file indicates Perl code                  *
//*                                                                              *
//* Returns: 'true' if verified as Perl code, else 'false'                       *
//********************************************************************************

static bool vfcIsPerlSource ( const gString& trgPath, bool strict )
{
   char inBuff[gsDFLTBYTES] ;    // input buffer
   gString gs ;                  // string search
   bool perlext = false,         // 'true' if recognized Perl filename extension
        noext = false,           // 'true' if no filename extension
        pearl = false ;          // return value

   //* Test the filename extension *
   if ( ((trgPath.find( ".pl" )) == (trgPath.gschars() - 4)) ||
         ((trgPath.find( ".pm" )) == (trgPath.gschars() - 4)) ||
         ((trgPath.find( ".t"  )) == (trgPath.gschars() - 3)) )
   {
      perlext = true ;
      if ( ! strict )
         pearl = true ;
   }
   else if ( (trgPath.findlast( L'.' )) < (trgPath.gschars() - 5) )
   {
      noext = true ;
   }

   //* Test the first line of the target file *
   if ( strict && (perlext || noext) )
   {
      ifstream ifs( trgPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         ifs.getline( inBuff, gsDFLTBYTES, nckNEWLINE ) ;
         if ( ifs.gcount() > ZERO )
         {
            inBuff[ifs.gcount()] = NULLCHAR ;
            gs = inBuff ;
            if ( ((gs.find( "#!" )) == ZERO) && ((gs.find( "/perl" )) > ZERO) )
               pearl = true ;
         }
         ifs.close() ;
      }
   }
   return pearl ;

}  //* End isPerlSource() *

//*************************
//*    vfcGetPerlToken    *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Extract the first token from 'gssrc' and place it into 'gstrg'.            *
//* The extracted token will include the trailing space characters (if any)    *
//*                                                                            *
//* Input  : gssrc : (by reference) contains source data                       *
//*          gstrg : (by reference) receives extracted token (if any)          *
//*                                                                            *
//* Returns: 'true'  if a token extracted from source                          *
//*          'false' if no remaining source tokens                             *
//******************************************************************************

static bool vfcGetPerlToken ( gString& gssrc, gString& gstrg )
{
   const wchar_t wDBLQ     = L'"' ;             // string-data delimiter
   const wchar_t wBKSTROKE = L'\\' ;            // escape character
   const wchar_t wUNSCORE  = L'_' ;             // underscore character
   const wchar_t wDOLLAR   = L'$' ;             // dollar-sign character
   const wchar_t wAMPERE   = L'@' ;             // ampersand character
   const wchar_t *wptr = gssrc.gstr() ;         // pointer to source data
   wchar_t wbuff[gsALLOCDFLT] ;                 // temporary buffer
   short windex = ZERO ;                        // index
   wbuff[ZERO] = NULLCHAR ;                     // initialize temp buffer
   gstrg.clear() ;                              // initialize the target buffer

   while ( wptr[windex] != NULLCHAR )
   {
      wbuff[windex] = wptr[windex] ;            // copy a character

      if ( wbuff[windex] == SPACE )             // if end-of-token reached
      {  //* Copy spaces between tokens *
         while ( wptr[++windex] == SPACE )
            wbuff[windex] = wptr[windex] ;
         break ;
      }

      //* If first character indicates a variable name.        *
      //* Perl variable names use only a-z, A-Z, 0-9 and '_'   *
      else if ( (windex == ZERO) && 
                ((wbuff[windex] == wDOLLAR) || (wbuff[windex] == wAMPERE)) )
      {
         wchar_t nextwch = wptr[windex + 1] ;
         while ( ((nextwch >= L'a') && (nextwch <= L'z')) ||
                 ((nextwch >= L'A') && (nextwch <= L'Z')) ||
                 ((nextwch >= L'0') && (nextwch <= L'9')) ||
                 (nextwch == wUNSCORE) || (nextwch == SPACE) )
         {
            wbuff[windex] = wptr[windex] ;

            if ( wbuff[windex] == SPACE )             // if end-of-token reached
            {  //* Copy spaces between tokens *
               while ( wptr[++windex] == SPACE )
                  wbuff[windex] = wptr[windex] ;
               break ;
            }
            nextwch = wptr[++windex] ;
         }
         break ;
      }

      else if ( wbuff[windex] == wDBLQ )        // if a string literal, token
      {                                         // is the entire string
         do
         {
            ++windex ;
            wbuff[windex] = wptr[windex] ;
         }
         while ( ((wbuff[windex] != wDBLQ) || 
                  (wbuff[windex] == wDBLQ && wbuff[windex - 1] == wBKSTROKE)) && 
                 (wbuff[windex] != NULLCHAR) ) ;
         if ( wbuff[windex] == wDBLQ )          // copy space between tokens
         {
            ++windex ;
            while ( wptr[windex] == SPACE )
            {
               wbuff[windex] = wptr[windex] ;
               ++windex ;
            }
            wbuff[windex] = NULLCHAR ;
            break ;
         }
      }
      else if ( wbuff[windex] == NULLCHAR )     // end of source data
         break ;
      ++windex ;
   }
   wbuff[windex] = NULLCHAR ;                   // ensure string termination
   if ( windex > ZERO )                         // copy extracted data to target
      gstrg = wbuff ;
   gssrc.erase( gstrg.gstr(), ZERO, true ) ;    // remove extracted data

   return ( bool((gstrg.gschars()) > 1) ) ;

}  //* End vfcGetPerlToken() *

//*************************
//*   vfcIsPerlCommand    *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Compare the specified token to the list of known Perl function names.      *
//* Also tests for certain non-function-name reserved tokens.                  *
//*                                                                            *
//* Input  : gstoken : (by reference) contains a single token                  *
//*                                                                            *
//* Returns: 'true'  if a token is a Perl command, else 'false'                *
//******************************************************************************

static bool vfcIsPerlCommand ( const gString& gstoken )
{
   //* Perl functions list taken from:                 *
   //* "https://perldoc.perl.org/index-functions.html" *
   //* Most common functions are placed near the top of*
   //* the list for efficiency.                        *
   const short CMD_MAX = 258 ;         // number of recognized Perl commands
   const char* CmdNames[CMD_MAX] = 
   {
      "my",                // common function names
      "print",
      "printf",
      "sprintf",
      "colored",
      "if",
      "else",
      "elseif",
      "elsif",
      "for",
      "foreach",
      "while",
      "eof",
      "eq",                // comparisons
      "ge",
      "le",
      "ne",
      "gt",
      "lt",
      "getc",
      "eval",
      "evalbytes",
      "system",
      "use",
      "sub",
      "scalar",
      "grep",
      "hex",
      "index",
      "int",
      "exit",

      "AUTOLOAD",          // less common functions alphabetically
      "abs",         "accept",      "alarm",       "and",         "atan2",

      "BEGIN",       "bind",        "binmode",     "bless",       "break",

      "CHECK",       "caller",      "chdir",       "chmod",
      "chomp",       "chop",        "chown",       "chr",
      "chroot",      "close",       "closedir",    "cmp",
      "connect",     "continue",    "cos",         "crypt",

      "DESTROY",     "__DATA__",    "dbmclose",    "dbmopen",
      "default",     "defined",     "delete",      "die",
      "do",          "dump",

      "END",         "__END__",     "each",        "endgrent",
      "endhostent",  "endnetent",   "endprotoent", "endpwent",
      "endservent",  "exec",        "exists",      "exp",

      "__FILE__",    "fc",          "fcntl",       "fileno",
      "flock",       "fork",        "format",      "formline",

      "getgrent",    "getgrgid",    "getgmam",     "gethostbyaddr",
      "gethostbyname","gethostent", "getlogin",    "getnetbyaddr",
      "getnetbyname","getnetent",   "getpeername", "getpgrp",
      "getppid",     "getpriority", "getprotobyname", "getprotobynumber",
      "getprotoent", "getpwent",    "getpwnam",    "getpwuid",
      "getservbyname","getservbyport","getservent","getsockname",
      "getsockopt",   "given",      "glob",        "gmtime",      "goto",

      "INIT",        "import",      "ioctl",

      "join",

      "keys",        "kill",

      "__LINE__",    "last",        "lc",          "lcfirst",
      "length",      "link",        "listen",      "local",
      "localtime",   "lock",        "log",         "lstat",

      "m ",             // note: single-character command is dangerous
      "map",         "mkdir",       "msgctl",      "msgget",
      "msgrcv",      "msgsnd",

      "next",        "no",          "not",

      "oct",         "open",        "opendir",     "or",
      "ord",         "our",

      "__PACKAGE__", "pack",        "package",     "pipe",
      "pop",         "pos",         "prototype",   "push",

      "q ",             // note: single-character command is dangerous
      "qq",          "qr",          "quotemeta",   "qw",          "qx",

      "rand",        "read",        "readdir",     "readline",
      "readlink",    "readpipe",    "recv",        "redo",
      "ref",         "rename",      "require",     "reset",
      "return",      "reverse",     "rewinddir",   "rindex",      "rmdir",

      "__SUB__",
      "s ",             // note: single-character command is dangerous
      "say",         "seek",        "seekdir",     "select",
      "semctl",      "semget",      "semop",       "send",
      "setgrent",    "sethostent",  "setnetent",   "setpgrp",
      "setpriority", "setprotoent", "setpwent",    "setservent",
      "setsockopt",  "shift",       "shmctl",      "shmget",
      "shmread",     "shmwrite",    "shutdown",    "sin",
      "sleep",       "socket",      "socketpair",  "sort",
      "splice",      "split",       "sqrt",        "srand",
      "stat",        "state",       "study",       "substr",
      "symlink",     "syscall",     "sysopen",     "sysread",
      "sysseek",     "syswrite",

      "tell",        "telldir",     "tie",         "tied",
      "time",        "times",       "tr",          "truncate",

      "UNITCHECK",   "uc",          "ucfirst",     "umask",
      "undef",       "unless",      "unlink",      "unpack",
      "unshift",     "untie",       "until",       "utime",

      "values",      "vec",

      "wait",        "waitpid",     "wantarray",   "warn",
      "when",        "write",

      "-X",
      "x ",             // note: single-character command is dangerous
      "xor",

      "y ",             // note: single-character command is dangerous
   } ;
   const short COLOR_MAX = 16 ;
   const char* ColorGroup[COLOR_MAX] = 
   {
      "black",
      "red",
      "green",
      "yellow",
      "blue",
      "magenta",
      "cyan",
      "white",
      "bright_black",
      "bright_red",
      "bright_green",
      "bright_yellow",
      "bright_blue",
      "bright_magenta",
      "bright_cyan",
      "bright_white",
      // (Note: stand-alone "on_xxxx" varients ignored)
   } ;
   bool status = false ;

   for ( short i = ZERO ; i < CMD_MAX ; ++i )
   {
      if ( (gstoken.find( CmdNames[i] )) == ZERO )
      {
         status = true ;
         break ;
      }
   }
   if ( ! status )
   {
      short indx ;
      for ( short i = ZERO ; i < COLOR_MAX ; ++i )
      {
         indx = gstoken.find( ColorGroup[i] ) ;
         if ( (indx == ZERO) || ((indx == 1) && (gstoken.gstr()[ZERO] == L'\'')) )
         {
            status = true ;
            break ;
         }
      }
   }
   return status ;

}  //* End vfcIsPerlCommand() *

//**************************
//*  OpenExternalProgram   *
//**************************
//******************************************************************************
//* Open the currently-highlighted file. This could be a binary executable,    *
//* a script, a media file, a document or many other types.                    *
//*                                                                            *
//* Input  : cmdBuff: (optional, NULL* by default)                             *
//*                   if specified, return the command string to caller        *
//*                   instead of executing it locally. In this case, caller    *
//*                   should close the NcDialog object, stop the ncurses       *
//*                   engine, fork the process and exit the application by     *
//*                   calling 'execv' or a similar exec function.              *
//*                                                                            *
//* Returns: OK if: a) file launched successfully, or                          *
//*                 b) 'cmdBuff' contains a valid launch string                *
//*          ERR if file not found, unable to launch, or user aborted launch.  *
//******************************************************************************
//* NOTES:                                                                     *
//* ======                                                                     *
//* When the caller selects a file to be opened by the default application for *
//* that file type, there are many (hundreds) of scenarios                     *
//* for what will happen next. However, two basic scenarios present themselves:*
//* 1) An external, GUI application is invoked, in which case our application  *
//*    may continue uninterrupted.                                             *
//* 2) An external console application is invoked which wants the resources of *
//*    our terminal window, in which case we would need to:                    *
//*    a) put FileMangler into hibernation mode,                               *
//*    b) exit FileMangler and give control to the called application, or      *
//*    c) open a new terminal window in which to run the new application.      *
//*                                                                            *
//*  1. Determine whether the targeted file is 'executable'. It could be       *
//*     a compiled application, a shell script, Perl, Python, etc.             *
//*     a) If it is a GUI application, then it can be launched automatically   *
//*        as a new process without affecting our terminal window resources.   *
//*     b) If it is a script, an interpreted file or console application, we   *
//*        must ask the user whether to:                                       *
//*          i) run it in the current window, or                               *
//*         ii) open a new terminal window in which to launch it, or           *
//*        iii) if a script/interpreted file, edit the file.                   *
//*        (Unfortunately, the user may not know or may be unable to decide    *
//*        whether the file runs in a console window.)                         *
//*  2. If the target file is not an executable file, then it must be opened   *
//*     by an external application (GUI or console app).                       *
//*     a) There are certain classes of files which we have confidence will    *
//*        never be opened in a terminal window (e.g. JPEG images, videos).    *
//*        Therefore we test for images, certain media files, LibreOffice      *
//*        documents and miscellaneous other files, and if identified, we      *
//*        launch the default application for those files without the need to  *
//*        prompt the user.                                                    *
//*     b) The more difficult class of files are those which may be opened     *
//*        either by a GUI app _or_ by a console app, depending on the user's  *
//*        system configuration. Some examples would be audio files, and plain *
//*        text files (config files, source code, scripts, etc.).              *
//*        For these, we must ask the user for guidance.                       *
//*  3. Construct a command string for launching the operation, with optional  *
//*     user-supplied arguments.                                               *
//*  4. Protect our console I/O resources.                                     *
//*  5. Launch the external program.                                           *
//*  6. Continue with this application.                                        *
//*     a) Note that we _could_ offer to exit FileMangler and give the external*
//*        application control of the terminal window; however, GUI file       *
//*        managers don't do this, so we aren't sure whether it is wise to     *
//*        offer the user that choice. This would be done at a higher level by *
//*        returning the constructed command to the caller in the 'cmdBuff'    *
//*        argument.                                                           *
//*                                                                            *
//*                                                                            *
//* Execution Logic:                                                           *
//* ----------------                                                           *
//* There are three major choices for launch:                                  *
//*    a) launch from current window and continue immediately (GUI apps only)  *
//*    b) put the application to sleep and launch from current window,         *
//*      (GUI or console app)                                                  *
//*    c) launch from _new_ termwin (GUI or console app)                       *
//*                                                                            *
//* 1) If identified as GUI client, then launch as: "xdg-open client-filename" *
//* 2) If identified as non-GUI client file, then launch as:                   *
//*    "xdg-open non-gui-client-filename" from the specified terminal window.  *
//*          (cmdIndex == upiCUR) --> current termwin                          *
//*          (cmdIndex == upiNEW) --> new termwin                              *
//* 3) If binary executable (exeFlag && binFlag):                              *
//*     a) if (cmdIndex == upiGUI), then launch the application from the       *
//*        current terminal window: "xdg-open bin-exe-file"                    *
//*     b) if ( cmdIndex != upiGUI), then launch from specified terminal       *
//*        window.                                                             *
//* 4) If non-binary executable (exeFlag && !binFlag), i.e. shell scripts or   *
//*    interpreted files (Perl, Ruby, etc), it is assumed that it will execute *
//*    in a terminal window. There are two choices:                            *
//*     a) Execute in the specified terminal window.                           *
//*     b) Edit the file with the default editor for that type:                *
//*         i) (cmdIndex == upiGUI) editor is a GUI application.               *
//*        ii) (cmdIndex == upiGED) editor is a console application.           *
//*                                                                            *
//* Invoking the system's default application for the source file-type.        *
//* -------------------------------------------------------------------        *
//*  1. Media files: images, audio, video.                                     *
//*  2. Documents: pdf, odt, etc.                                              *
//*  This relies on the system script file 'xdg-open' to select the appropriate*
//*  external application. See LaunchDefaultApplication() method for details.  *
//*                                                                            *
//*  Because by default, we redirect stdout and stderr for the call to         *
//*  xdg-open, we retain control of output to the window; however, it is       *
//*  likely that a console application called from the active terminal window  *
//*  will be inaccessible to the user. While this itself is bad, what's worse  *
//*  is that we would be blamed for the mess.                                  *
//*                                                                            *
//*  To avoid this, we can offer to open a second terminal window              *
//*  (unfortuately not a new tab) from which to make the external call.        *
//*  See notes in oepNewTermwin().                                             *
//*                                                                            *
//******************************************************************************

short FileDlg::OpenExternalProgram ( char* cmdBuff )
{
   const char* dfltProg = "xdg-open" ;   // opens the default application

   gString fPath,                // target path string
           argList,              // optional arguments to launch of executable files
           gstmp ;               // general text formatting
   short cmdIndex = ERR,         // user instructions returned from setup dialog
         status = ERR ;          // return value
   bool  guiType = false,        // 'true' if extension matches a GUI type
         exeFlag = false,        // 'true' if user's 'x' bit is set
         binFlag = false,        // 'true' if a binary (compiled) executable
         autoReturn = false ;    // 'true' if close new termwin when called app closes

   //* Initialize caller's buffer *
   if ( cmdBuff != NULL )
      *cmdBuff = NULLCHAR ;

   //* If there is actually a file under the highlight *
   if ( this->fileCount > ZERO )
   {
      //* Retrieve the file stat information *
      tnFName  fStats ;             // copy of basic file info
      this->GetStats ( fStats ) ;   // stats for currently-highlighted file
      this->fmPtr->CatPathFname ( fPath, this->currDir, fStats.fName ) ;
      //* If target file is a symbolic link, check the link target *
      if ( fStats.fType == fmLINK_TYPE )
      {
         tnFName fnt ;        // filename target of symlink stats
         if ( (this->fmPtr->GetLinkTargetStats ( fPath, fnt, gstmp)) == OK )
         {
            fStats = fnt ;    // target found, copy stats and path
            fPath = gstmp ;
         }
      }

      //* Only 'Regular' files may be opened/executed *
      if ( fStats.fType != fmREG_TYPE )
      {
         const char* msgText[] = 
         {
            "  ALERT ALERT  ",
            " ",
            "Sorry, only \"Regular\" files may be opened.",
            " ",
            NULL
         } ;
         this->InfoDialog ( msgText ) ;
         //****************************
         //** Note the early return. **
         //****************************
         return status ;
      }

      //* Test whether filename extension is on our list of supported media    *
      //* and document files to be run by an external (GUI) application.       *
      if ( (guiType = oepGooeyType ( fPath, 
                               (bool)(this->cfgOptions.lCode == lcAutoAudio))) )
      {
         //* 'fPath' contains the filespec with which to *
         //* launch the default GUI application.         *
         status = OK ;
      }

      //* If target cannot be launched automagically,   *
      //* as a GUI client, test for executable filetype.*
      else  // ( ! guiType )
      {
         //* Test whether file has 'executable' permission bits set AND *
         //* whether user has permission to execute the file.           *
         exeFlag = this->fmPtr->ExecutePermission ( fPath, false ) ;
         if ( exeFlag )
            binFlag = oepBinexeTest ( fPath ) ;
      }

      //* If file is not a GUI client, and if config option  *
      //* instructs that programs should be launched from    *
      //* a new window.                                      *
      if ( ! guiType && (this->cfgOptions.lCode == lcSafeLaunch) )
      {
         cmdIndex = upiNEW ;
         status = OK ;
      }

      //* Otherwise, ask user how to proceed, either because *
      //* we can't determine what to do, _OR_ because user   *
      //* specified that we should always ask.               *
      else if ( ! guiType || (this->cfgOptions.lCode == lcManualLaunch) )
      {
         //* Value returned is index of selected Radiobutton: *
         //* (See meaning of return value in called method.)  *
         cmdIndex = this->oepUserPrompt ( fPath, argList, exeFlag, binFlag ) ;
         if ( cmdIndex >= ZERO )
            status = OK ;
      }

      //* If we have the needed information, continue. *
      if ( status == OK )
      {
         //* If opening a non-GUI, non-executable file in a terminal window.*
         //* This will invoke LaunchExternalApplication("xdg-open",filespec)*
         if ( ! exeFlag && ! guiType && (cmdIndex != upiGUI) )
         {  //* Execute 'xdg-open' with specified source file as its argument *
            argList = fPath ;
            fPath   = dfltProg ;
         }

         //* If editing a non-binary executable i.e. a script file,  *
         //* determine whether we should open the editor in a new    *
         //* terminal window.                                        *
         else if ( exeFlag && !binFlag && (cmdIndex == upiGED) )
         {  //* Execute 'xdg-open' in a new terminal window with     *
            //* specified source file as its argument.               *
            // Programmer's Note: Because we do not use a console text editor for 
            // anything, using xdg-open to invoke vim or another console editor 
            // has not been fully tested.
            argList = fPath ;
            fPath   = dfltProg ;
            cmdIndex = upiNEW ;     // force the editor to open in a new termwin
            autoReturn = true ;     // close termwin when app closes
         }

         //* If file is to be opened by the default GUI application.*
         //* This includes:                                         *
         //*  1) automatically-identified GUI client files          *
         //*  2) open any text file with the default editor for     *
         //*     that file type.                                    *
         //*  3) if EDITOR_LAUNCH==0, opening a script/interpreted  *
         //*     file with the default editor for that file type.   *
         if ( (guiType != false) ||
              ((cmdIndex == upiGUI) && (!exeFlag || (exeFlag && ! binFlag))) )
         {
            if ( (this->LaunchDefaultApplication ( fPath.ustr() )) >= ZERO )
            {
               dtbmData tbMsg( "  External Application Launched Successfully!  " ) ;
               this->dPtr->DisplayTextboxMessage ( this->mIndex, tbMsg ) ;
            }
            else  //* Report error in launching the program *
            {
               gstmp = fPath ;
               this->TrimPathString ( gstmp, (INFODLG_WIDTH - 3) ) ;
               const char* nodefaultMsg[] = 
               {
                  "  ALERT ALERT  ",
                  " ",
                  "   Unable to launch default application for:",
                  gstmp.ustr(),
                  " ",
                  NULL
                  
               } ;
               this->InfoDialog ( nodefaultMsg ) ;
               status = ERR ;       // return failure
            }
         }     // (guiType != false)

         //********************************************
         //* Launch according to user's instructions. *
         //********************************************
         else// if ( exeFlag != false )
         {
            //* Assumes GUI application which will not  *
            //* impact active terminal window.          *
            if ( cmdIndex == upiGUI )
            {
               if ( (this->LaunchExternalApplication ( fPath.ustr(), argList )) < ZERO )
               {  //* Report error in launching the program *
                  gstmp = fPath ;
                  this->TrimPathString ( gstmp, (INFODLG_WIDTH - 3) ) ;
                  argList.insert( "    " ) ;
                  const char* nolaunchMsg[] = 
                  {
                     "  ALERT ALERT  ",
                     " ",
                     "   Unable to launch external application.",
                     gstmp.ustr(),
                     argList.ustr(),
                     NULL
                  } ;
                  this->InfoDialog ( nolaunchMsg ) ;
                  status = ERR ;       // return failure
               }
            }

            //* Put FileMangler to sleep and launch     *
            //* application in current terminal window. *
            else if ( cmdIndex == upiCUR )
            {  //* If additional arguments, append them to filespec. *
               if ( argList.gschars() > 1 )
                  fPath.append( " %S", argList.gstr() ) ;
               this->dPtr->ShellOut ( soP, fPath.ustr() ) ;
            }

            //* Open a new terminal window and launch   *
            //* application in the new window.          *
            else  // ( cmdIndex == upiNEW )
            {
               this->fmPtr->CreateTempname ( gstmp ) ;
               DC_Emulator dce( gstmp, fPath, argList, autoReturn ) ;

               //* Scan the system configuration to define *
               //* the new terminal window.                *
               dce.DefineChildEmulator() ;
               this->fmPtr->DeleteTempname ( gstmp ) ; // delete the temp file

               //* Open the new terminal window *
               pid_t fpid = dce.OpenTermwin () ;

               //* The parent process continues execution here. Wait a moment *
               //* (in case the child PID writes an error message to display).*
               // Programmer's Note: We do not like calling the NCurses primitive 
               // here, but the just-launched terminal emulator may completely 
               // trash the display.
               chrono::duration<short, std::milli>aMoment( 250 ) ;
               this_thread::sleep_for( aMoment ) ;
               nc.ClearScreen () ;
               this->dPtr->RefreshWin () ;         // restore parent dialog

               dtbmData tbMsg( "  New Terminal Window Opened Successfully!  " ) ;
               if ( fpid <= ZERO )     // child window did not launch
               {
                  gstmp = "  ERROR: Unable To Open New Terminal Window!  " ;
                  gstmp.copy( tbMsg.textData, gsALLOCDFLT ) ;
               }
               this->dPtr->DisplayTextboxMessage ( this->mIndex, tbMsg ) ;
            }  // (cmdIndex == upiNEW)
         }     // (executable != false)
      }        // if(status == OK)
   }           // if(this->fileCount > ZERO)

   return status ;

}  //* OpenExternalProgram() *

//*************************
//*     oepGooeyType      *
//*************************
//********************************************************************************
//* Compare the filename extension of the specified filespec to our list of      *
//* supported client filetypes which we are 'sure' will be run by GUI            *
//* applications.                                                                *
//* -- Note that audio media files are a special case in that the audio media    *
//*    player may, or may not be a GUI application. Therefore, we test for       *
//*    audio filetypes only if caller has verified that the default audio        *
//*    media player is a GUI application (guiAudio != false).                    *
//*                                                                              *
//* If the filespec indicates a file that will be opened by a GUI application,   *
//* then the caller will not need to prompt the user for setup information.      *
//*                                                                              *
//* Input  : fPath    : filespec to be scanned                                   *
//*          guiAudio : indicates whether the default audio media player         *
//*                     is a GUI application.                                    *
//*                     'true'  if user has verified GUI audio player            *
//*                     'false' if type of audio player is a console app or if   *
//*                             the type of player is unknown                    *
//*                                                                              *
//* Returns: 'true'  if matching filename extension found                        *
//*          'false' if no match found                                           *
//********************************************************************************
//* Notes:                                                                       *
//* ======                                                                       *
//* 1) Document filename extensions:                                             *
//*    a) We include standard LibreOffice / OpenOffice extensions as well as     *
//*       legacy extensions for earlier versions of the office suite including   *
//*       the original StarOffice(tm) v:5 extensions.                            *
//*    b) As much as we detest Microsloth and all it stands for, we include      *
//*       the extensions for both the open-document-format versions and the      *
//*       proprietary version 6 extensions.                                      *
//*    c) Also included is the Portable Document Format (PDF) extension.         *
//*    d) We include the most common of the HTML extensions; HOWEVER, these      *
//*       may be removed because we don't actually know whether the user wants   *
//*       to view them or edit them. Note also that there are some legacy        *
//*       text-only browsers out there, but we ignore that possibility.          *
//*                                                                              *
//* 2) Audio and video filename extensions are taken from Widipedia's lists of   *
//*    common formats and filename extensions.                                   *
//*    a) Note that there is some overlap between audio and video formats as     *
//*       indicated by the filename extension. This is because the extension     *
//*       often indicates the _container format_ rather than the data it         *
//*       contains.                                                              *
//*    b) The intention is that for extensions that could indicate _either_      *
//*       audio or video, audio is assumed.                                      *
//*                                                                              *
//*                                                                              *
//********************************************************************************

static bool oepGooeyType ( const gString& fPath, bool guiAudio )
{
   const short docTYPES = 44 ;
   const wchar_t* DocExtensions[docTYPES]
   {  //* Document files *
      L".odt",    L".ods",    L".odp",    L".odg",    L".odf",    L".sxw",    
      L".sxc",    L".sxi",    L".sxd",    L".sxm",    L".doc",    L".docx",   
      L".docm",   L".dot",    L".dotx",   L".dotm",   L".docb",   L".ppt",    
      L".pot",    L".pps",    L".pptx",   L".pptm",   L".potx",   L".potm",   
      L".ppam",   L".ppsx",   L".ppsm",   L".sldx",   L".sldm",   L".xls",    
      L".xlt",    L".xlm",    L".xlsx",   L".xlsm",   L".xltx",   L".xltm",   
      L".xlsb",   L".xla",    L".xlam",   L".xll",    L".xlw",    L".pdf",    
      L".html",   L".htm",    // see note on HTML files above
   } ;
   const short audioTYPES = 38 ;
   const wchar_t* AudioExtensions[audioTYPES] = 
   {  //* Audio files *
      L".mp3",    L".ogg",    L".oga",    L".mogg",   L".m4a",    L".m4b",    
      L".m4p",    L".wav",    L".wma",    L".flac",   L".aiff",   L".ape",    
      L".aa",     L".aac",    L".aax",    L".act",    L".amr",    L".3gp",    
      L".au",     L".awb",    L".dct",    L".dss",    L".dvf",    L".gsm",    
      L".iklax",  L".isv",    L".mmf",    L".mpc",    L".msv",    L".opus",   
      L".ra",     L".rm",     L".raw",    L".sln",    L".tta",    L".vox",    
      L".wv",     L".8svx",       
   } ;
   const short videoTYPES = 48 ;
   const wchar_t* VideoExtensions[videoTYPES] = 
   {  //* Image and Video files *
      L".jpg",    L".jpeg",   L".jif",    L".jfif",   L".png",    L".gif",    
      L".pcd",    L".fpx",    L".jp2",    L".jpx",    L".j2k",    L".j2c",    
      L".tif",    L".tiff",   L".bmp",    L".webp",
      L".mp4",    L".m4v",    L".vob",    L".avi",    L".webm",   L".mkv",    
      L".ogv",    L".drc",    L".mov",    L".qt",     L".wmv",    L".yuv",    
      L".rmvb",   L".rm",     L".asf",    L".amv",    L".mpg",    L".mp2",    
      L".mpeg",   L".mpv",    L".m2v",    L".svi",    L".3gp",    L".3g2",    
      L".mxf",    L".roq",    L".nsv",    L".flv",    L".f4v",    L".f4p",    
      L".f4a",    L".f4b",    
   } ;


   short extIndex = fPath.findlast( L'.' ) ; // index the filename extension (if any)
   bool goodType = false ;                   // return value

   if ( extIndex >= ZERO )
   {
      for ( short i = ZERO ; i < docTYPES ; ++i )
      {
         if ( (fPath.compare( DocExtensions[i], false, gsALLOCDFLT, extIndex )) == ZERO )
         {
            goodType = true ;
            break ;
         }
      }
      if ( ! goodType && guiAudio != false )
      {
         for ( short i = ZERO ; i < audioTYPES ; ++i )
         {
            if ( (fPath.compare( AudioExtensions[i], false, gsALLOCDFLT, extIndex )) == ZERO )
            {
               goodType = true ;
               break ;
            }
         }
      }
      if ( ! goodType )
      {
         for ( short i = ZERO ; i < videoTYPES ; ++i )
         {
            if ( (fPath.compare( VideoExtensions[i], false, gsALLOCDFLT, extIndex )) == ZERO )
            {
               goodType = true ;
               break ;
            }
         }
      }
   }
   return goodType ;

}  //* End oepGooeyType() *

//*************************
//*     oepBinexeTest     *
//*************************
//******************************************************************************
//* Called only when target file's 'x' (executable) bit is set.                *
//* Test for a binary executable file according to the "ELF" signature in the  *
//* "Executable and Linkable Format" specification.                            *
//*    This signature will be: 0x7F 0x45 0x4C 0x46                             *
//*                                 'E'  'L'  'F'                              *
//*                                                                            *
//* If not a binary executable, it may be assumed that the file is a shell     *
//* script or a source file for one of the interpreted programming languages.  *
//* See below for a list of likely script and interpreted files.               *
//*                                                                            *
//* Input  : fPath    : filespec of file to be scanned                         *
//*                                                                            *
//* Returns: 'true'  if file is identified as a binary executable file         *
//*          'false' otherwise                                                 *
//******************************************************************************
//* List of filename extensions for common scripting languages and source      *
//* files for interpreted languages.                                           *
//* The list is sorted arbitrarily, high-to-low, according to our              *
//* notion of how often they are encountered in a typical system.              *
//*                                                                            *
//* const short extCOUNT = 28 ;                                                *
//* const wchar_t* extList[extCOUNT] =                                         *
//* {//EXTENSION         SIGNATURE               COMMON NAME                   *
//*  //-------------  -------------------------  --------------                *
//*    L".bash",      // #!/bin/bash             Bash                          *
//*    L".sh",        // #!/bin/sh               Bourne                        *
//*    L".csh",       // #!/bin/csh              CShell                        *
//*    L".ksh",       // #!/usr/bin/ksh          Korn shell                    *
//*    L".dash",      // #!/bin/sh               Debian ash                    *
//*    L".pl",        // #!/usr/bin/perl         Perl                          *
//*    L".pm",        //                                                       *
//*    L".t",         //                                                       *
//*    L".py",        // #!/usr/bin/env python3  Python                        *
//*    L".pyw",       //                                                       *
//*    L".rb",        // #!/usr/bin/ruby -w      Ruby                          *
//*    L".rbw",       //                                                       *
//*    L".html",      // <!DOCTYPE html>         HTML                          *
//*    L".htm",       // <html>                                                *
//*    L".xml",       //                                                       *
//*    L".xhtml",     //                                                       *
//*    L".xht",       //                                                       *
//*    L".php",       // <?php                   PHP (incl. .php[3,4,5,6,7,s]) *
//*    L".phtml",     // <!DOCTYPE html>                                       *
//*    L".zsh",       // #!/bin/sh               ZShell                        *
//*    L".tcl",       // #!/usr/bin/tclsh        Tcl script                    *
//*    L".fish",      // #!/bin/fish             Friendly Int                  *
//*    L".tcsh",      // #!/bin/tcsh             TENEX CShell                  *
//*    L".rc",        // #!/usr/bin/rsh          RC                            *
//*    L".scsh",      // #!/usr/local/bin/scsh   Scheme                        *
//*    L".ash",       // #!/bin/sh               Almquist                      *
//*    L".ch",        //                         Ch shell                      *
//*    L".eshell",    //                         Emacs                         *
//* } ;                                                                        *
//*                                                                            *
//* Scripting languages, especially shell scripts often have no filename       *
//* extension, for ease in invoking them. Most other types of executables will *
//* have valid descriptive filename extensions.                                *
//* Practically speaking, three(3) characters are enough to identify a         *
//* signature (if it is properly constructed):                                 *
//*            "#!/". This is the "shebang + path" sequence.                   *
//* Note that although HTML files are interpreted scripts, they are not marked *
//* as executable, so we will never see them here.                             *
//******************************************************************************

static bool oepBinexeTest ( const gString& fPath )
{
   bool binFlag = false ;       // return value

   //* Open the file and extract the first few characters.        *
   //* Note that there is a high likelihood that this is a binary *
   //* (compiled) executable, so we open the file in binary mode. *
   ifstream ifs ( fPath.ustr(), ifstream::in | ifstream::binary ) ;
   if ( ifs.is_open() )
   {
      char ibuff[32] ;
      ifs.read( ibuff, 4 ) ;
      if ( (ibuff[0] == 0x7F) && (ibuff[1] == 'E') && 
                (ibuff[2] == 'L')  && (ibuff[3] == 'F') )
         binFlag = true ;
      ifs.close() ;
   }
   return binFlag ;

}  //* End oepBinexeTest() *

//*************************
//*     oepUserPrompt     *
//*************************
//******************************************************************************
//* Open a dialog for user instructions on how the external file should be     *
//* launched.                                                                  *
//*                                                                            *
//* Input  : fPath    : filespec to be launched                                *
//*          argList  : receives optional, user-specified arguments for        *
//*                     launch (executable files only)                         *
//*          exeFlag  : 'true' if file is executable, else 'false'             *
//*          binFlag  : 'true' if file is identified as a binary (compiled)    *
//*                     executable                                             *
//*              Note: If exeFlag != false && binFlag == false, then file is a *
//*                    shell script or interpreted file (Perl, Python, etc.).  *
//*                                                                            *
//* Returns: selected launch code:                                             *
//*          upiGUI : launch from current terminal window                      *
//*          upiCUR : put application to sleep and launch from current window  *
//*          upiNEW : create a new terminal window from which to launch program*
//*          upiGED : edit non-GUI executable (script) in a new termial window *
//*          (returns -1 if user abort)                                        *
//******************************************************************************
//* Programmer's Note: There is a logical hole in the choices presented to the *
//* user in this method. It is assumed that a script file or interpreted file  *
//* if it is to be executed, will always be executed in a terminal window.     *
//* As a practical matter, this is not a problem, but it is _theoretically_    *
//* possible that the script or interpreted file could be executed by a GUI    *
//* program. We do not support that theoretical possibility.                   *
//*                                                                            *
//******************************************************************************

short FileDlg::oepUserPrompt ( const gString& fPath, gString& argList,
                               bool exeFlag, bool binFlag )
{
   //* Conditional-compile flag for launch of default editor for non-binary    *
   //* executable files (shell scripts, Python, Perl, etc.)                    *
   //* true  : Initialize the upiGED Radiobutton to 'true', i.e. the default   *
   //*         editor for scripts is a GUI application which will be launched  *
   //*         from the current terminal window.                               *
   //* false : Initialize the upiGED Radiobutton to 'false', i.e. the default  *
   //*         editor for scripts is assumed to be a console applicaton        *
   //*         (vi/vim, pico, jed, etc.).                                      *
   const bool GUI_EDITOR = true ;


   const char* progType[][locLang] =
   {
      {  // English
         "This is a binary executable program.",
         "This is a shell script or interpreted program.",
         "This is a non-executable data file."
      },
      {  // Espanol
         "Este es un programa ejecutable binario.",
         "Este es un script de shell o programa interpretado.",
         "Este es un archivo de datos no ejecutable."
      },
      {  // Zhongwen
         "这是一个二进制可执行程序。",
         "这是一个命令脚本或解释程序。",
         "这是一个不可执行的数据文件。"
      },
   } ;
   const char* Labels[][11] = 
   {
      {  // English
         "  L^AUNCH  ",                                              // upiLAUNCH
         "  ^CANCEL  ",                                              // upiCANCEL
         "Launch the program as a GUI application OR open file\n"    // upiGUI
         "using the default (GUI) application for this file type.\n"
         "(FileMangler continues in the current window.)",
         "Put FileMangler to sleep and launch the program\n"         // upiCUR
         "in the current window. (then type EXIT to return.)",
         "Launch the program in a new terminal window.\n"            // upiNEW
         "(FileMangler continues in the current window.)",
         "Additional launch arguments (for executable files only)",  // upiARGS
         "default editor is a GUI application",                      // upiGED
         "  Launch External Program  ",                              // dialog title
         "File: ",                                                   // upiHEADER
         "Edit the file using the default text editor.\n"            // upiEDIT
         "(FileMangler continues in the current window.)",
         "If you are unsure which option to choose, be safe, select\n" // upiQHELP
         "      \"Launch the program in a new terminal window.\"",
      },
      {  // Espanol
         "  L^ANZAR  ",                                              // upiLAUNCH
         " ^CANCELAR ",                                              // upiCANCEL
         "Abra el programa como una aplicación GUI, o abra el\n"     // upiGUI
         "archivo usando la aplicación predeterminada para este\n"
         "tipo de archivo.\n"
         "   (FileMangler continúa en la ventana actual.)",
         "Ponga FileMangler en suspensión e inicie el programa en\n" // upiCUR
         "la ventana actual. (luego escribe EXIT para regresar.)",
         "Inicie el programa en una nueva ventana de terminal.\n"    // upiNEW
         "   (FileMangler continúa en la ventana actual.)",
         "Argumentos de inicio adicionales (solo archivos ejecutables)", // upiARGS
         "editor predeterminado es una aplicación GUI",              // upiGED
         "  Lanzar Programa Externo  ",                              // dialog title
         "Archivo: ",                                                // upiHEADER
         "Edite el archivo usando el editor de texto predeter-\n"    // upiEDIT
         "minado. (FileMangler continúa en la ventana actual.)",
         "Si no está seguro de qué opción elegir, seleccione:\n" // upiQHELP
         "  \"Inicie el programa en una nueva ventana de terminal.\"",
      },
      {  // Zhongwen
         "   发起   ",                                                // upiLAUNCH
         "   取消   ",                                                // upiCANCEL
         "Launch the program as a stand-alone (GUI) application.\n"  // upiGUI
         "or Open file using default application for this file\n"
         "type. (FileMangler continues in the current window.)",
         "Put FileMangler to sleep and launch the program\n"         // upiCUR
         "in the current window. (then type EXIT to return.)",
         "Launch the program in a new terminal window.\n"            // upiNEW
         "(FileMangler continues in the current window.)",
         "Additional launch arguments (for executable files only)",  // upiARGS
         "默认编辑器是一个 GUI 应用程序",                                // upiGED
         "   启动外部程序   ",                                         // dialog title
         "文件： ",                                                   // upiHEADER
         "使用默认的文本编辑器编辑文件。\n"                               // upiEDIT
         "(FileMangler 在当前窗口中继续。)",
         "If you are unsure which option to choose, be safe, select\n" // upiQHELP
         "      \"Launch the program in a new terminal window.\"",
      },
   } ;

   const short dlgROWS = 23,
               dlgCOLS = 64 ;
   const attr_t dColor = this->cs.sd,  // dialog background color
                hColor = this->cs.em,  // bold text
                rbnColor = dColor,     // radio-button colors
                rbfColor = this->cs.pf ;
   AppLang lang = this->cfgOptions.appLang ;
   short dlgY   = this->fulY,
         dlgX   = (this->fulX + this->fMaxX / 2) - dlgCOLS / 2,
         cIndex = -1 ;    // return value - assume user abort

   // Programmer's Note: This dialog is taller than the minimum application 
   // height. Application height must be >= (dlgROWS + 1), else the dialog 
   // will not open.
   if ( this->dRows < (dlgROWS + 1) )
   {
      dtbmData tbMsg( "Unable to open dialog! Please expand terminal window and try again." ) ;
      this->dPtr->DisplayTextboxMessage ( this->mIndex, tbMsg ) ;
      this->dPtr->UserAlert () ;
      //****************************
      //** Note the early return. **
      //****************************
      return cIndex ;
   }

   //* Adjust dialog position *
   if ( this->dRows < (dlgY + dlgROWS) )
      dlgY = 1 ;
   if ( dlgX < ZERO )
      dlgX = ZERO ;
   while ( ((dlgX + dlgCOLS) > this->dCols) && (dlgX > ZERO) )
      --dlgX ;

   argList.clear() ;       // initialize optional-arguments buffer

   InitCtrl ic[upiCONTROLS] = 
   {
   {  //* 'LAUNCH' Pushbutton - - - - - - - - - - - - - - - - - - -  upiLAUNCH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 6),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 10 - 3),  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      Labels[lang][upiLAUNCH],      // dispText
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[upiCANCEL]                // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' Pushbutton - - - - - - - - - - - - - - - - - - -  upiCANCEL *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[upiLAUNCH].ulY,            // ulY:       upper left corner in Y
      short(ic[upiLAUNCH].ulX + ic[upiLAUNCH].cols + 4),// ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      Labels[lang][upiCANCEL],      // dispText
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[upiGUI]                   // nextCtrl:  
   },
   {  //* 'Launch-as-GUI' Radiobutton - - - - - - - - - - - - - - - - - upiGUI *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially set
      4,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][upiGUI],         // label
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[upiCUR]                   // nextCtrl:  link in next structure
   },
   {  //* 'Launch-in-current window' Radiobutton  - - - - - - - - - - - upiCUR *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initially set
      short(ic[upiGUI].ulY + 4),    // ulY:       upper left corner in Y
      ic[upiGUI].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][upiCUR],         // label
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[upiNEW]                   // nextCtrl:  link in next structure
   },
   {  //* 'Launch-in-new window' Radiobutton  - - - - - - - - - - - - - upiNEW *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: standard, 3-wide
      true,                         // rbSelect:  initially set
      short(ic[upiCUR].ulY + 3),    // ulY:       upper left corner in Y
      ic[upiCUR].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][upiNEW],         // label
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[upiARGS]                  // nextCtrl:  link in next structure
   },
   {  //* 'Arguments' Textbox  - - - - - - - - - - - - - - - - - - -   upiARGS *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[upiNEW].ulY + 4),    // ulY:       upper left corner in Y
      ic[upiNEW].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dlgCOLS - 4),           // cols:      control columns
      "",                           // dispText:  initially empty
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      Labels[lang][upiARGS],        // label
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'GUI Editor' Radiobutton  - - - - - - - - - - - - - - - - -   upiGED *
      dctRADIOBUTTON,               // type:      
      rbtS3p,                       // rbSubtype: parentheses, 3-wide
      GUI_EDITOR,                   // rbSelect:  initially set
      short(ic[upiARGS].ulY + 1),   // ulY:       upper left corner in Y
      short(ic[upiGUI].ulX + ic[upiGUI].labX), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][upiGED],         // label
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   if ( ! exeFlag )     // supplimentary argument are only for executable files
      ic[upiARGS].active = false ;

   //* Executable scripts and interpreted files are assumed to be non-GUI.     *
   //* (Java source could be an exception, but .jar files are not executable.) *
   //* Adjust the label for the upiGUI radiobutton to offer edit of the file.  *
   //* Activate the upiGED Radiobutton, adjusting control positions.           *
   if ( exeFlag && !binFlag )
   {
      ic[upiGUI].label = Labels[lang][upiEDIT] ;
      ic[upiGED].active = true ;
      ic[upiARGS].nextCtrl = &ic[upiGED] ;
      --ic[upiCUR].ulY ;
      --ic[upiNEW].ulY ;
      --ic[upiARGS].ulY ;
      --ic[upiGED].ulY ;
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       dlgY,           // Y offset from upper-left of terminal 
                       dlgX,           // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->cs.bb,    // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( Labels[lang][upiCONTROLS], hColor ) ;

      //* Group the radiobuttons *
      short XorGroup[4] = { upiGUI, upiCUR, upiNEW, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      //* Draw the static text *
      gString gs = fPath ;
      winPos wp( 1, 2 ) ;
      wp = dp->WriteString ( wp, Labels[lang][upiHEADER], dColor ) ;
      this->TrimPathString ( gs, (dlgCOLS - wp.xpos - 2) ) ;
      dp->WriteString ( wp, gs, hColor ) ;
      ++wp.ypos ;
      short ptIndex = exeFlag ? (binFlag ? 0 : 1) : 2 ;
      dp->WriteString ( wp, progType[lang][ptIndex], dColor ) ;

      LineDef  hLine(ncltHORIZ, ncltDUAL, (dlgROWS - 4), ZERO, dlgCOLS, this->cs.bb) ;
      dp->DrawLine ( hLine ) ;
      dp->WriteParagraph ( (hLine.startY + 1), 3, Labels[lang][upiQHELP], dColor ) ;

      dp->RefreshWin () ;                 // make the text visible

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton   *
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == upiLAUNCH )
               {
                  dp->GetTextboxText ( upiARGS, argList ) ;
                  cIndex = dp->GetRbGroupSelection ( upiGUI ) ;
                  if ( (cIndex == upiGUI) && ic[upiGED].active )
                  {
                     bool guiEditor ;
                     dp->GetRadiobuttonState ( upiGED, guiEditor ) ;
                     if ( ! guiEditor )   // signal edit with console editor
                        cIndex = upiGED ;
                  }
               }
               else if ( Info.ctrlIndex == upiCANCEL )   // abort the operation
                  cIndex = -1 ;
               done = true ;
            }
         }

         //* If focus is currently on a Radio Button *
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         //* If focus is currently on a Textbox *
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;
            icIndex = dp->EditTextbox ( Info ) ;
         }

         //* Navigate to next/previous control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore parent dialog window *
   this->dPtr->RefreshWin () ;

   return cIndex ;

}  //* End oepUserPrompt() *

//*****************************
//* LaunchExternalApplication *
//*****************************
//******************************************************************************
//* Launch a specific external application along with desired arguments.       *
//* See notes below for a detailed discussion of the arguments.                *
//*                                                                            *
//* Input  : appSpec  : filespec (or filename) of application                  *
//*        : argList  : arguments sent to target application as argv[] array   *
//*        : redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) This method requires some intelligence on the part of the caller.       *
//*    Unlike the LaunchDefaultApplication() method above, this method must    *
//*    be able to parse the individual arguments into separate substrings      *
//*    with a pointer to each. This is what the called application will see as *
//*    the "char* argv[]" array.                                               *
//* 2) The 'appSpec' parameter will contain the target application's name,     *
//*    relative path or absolute path.                                         *
//*    a) If it is known that the application in on the path, this may be the  *
//*       name of the executable file only.                                    *
//*    b) If caller is not sure whether the executable is on the path, then    *
//*       'appSpec' parameter should be the absolute or relative filespec.     *
//*    c) Path search:                                                         *
//*        i) If 'appSpec' does not contain slash ( '/' ) characters,          *
//*           then 'execvp' will search the $PATH environment variable.        *
//*       ii) If 'appSpec' DOES contain slash ( '/' ) characters,              *
//*           then 'execvp' will assume that you have provided a valid         *
//*           filespec, and will not search the $PATH.                         *
//*    d) By convention, argv[0] is the application name. This method captures *
//*       the tail of 'appSpec', stripping any path elements and inserts the   *
//*       application name as argv[0].                                         *
//* 3) The 'argList' parameter contains a list of arguments to be passed to    *
//*    target application.                                                     *
//*    a) Each argument substring, except the last, must be terminated by a    *
//*       newline '\n' character to allow reliable parsing.                    *
//*       Example: "arg01\narg02\narg03\narg04"                                *
//*    b) Any arguments which contain whitespace must be properly quoted:      *
//*       Example:  "arg01\n'Waiting for the Rain.ogs'\narg03"                 *
//*       This method WILL NOT automatically delimit the substrings.           *
//*    c) Any characters which might be incorrectly interpreted as regexp      *
//*       expressions must be properly escaped to prevent the shell from       *
//*       trying to expand them. Examples would be: \[  \]  \*  \.             *
//*       Arguments which contain _intentional_ regexp need not be escaped.    *
//*       Note however, that regexp wildcard characters ABSOLUTELY WILL NOT BE *
//*       EXPANDED. For instance: grep -n 'Taggit::' *.cpp                     *
//*       will not work because the shell will not expand the '*' character.   *                                                                        *
//* 4) One additional argument will be automatically appended to the arg list: *
//*    The NULL pointer indicates the end of the argv[] array.                 *
//* 5) Redirection of 'stdout' and 'stderr':                                   *
//*    Please see discussion of redirection in the LaunchDefaultApplication()  *
//*    method, above.                                                          *
//******************************************************************************

int FileDlg::LaunchExternalApplication ( const char* appSpec, 
                                        const gString& argList, bool redirect )
{
   const short MAX_ARGS = 48 ;         // maximum number of argument tokens (arbitrary)
   const char* argv[MAX_ARGS + 1] ;    // argv[] array of arg pointers
   char argvStr[gsDFLTBYTES] ;         // temp buffer for arguments
   char appName[gsDFLTBYTES] ;         // application filename (no path)
   gString gs ;                        // text formatting
   int soDesc, seDesc ;                // descriptors for temp files
   short wCnt = argList.gschars(),     // number of characters in source
         windx = ZERO,                 // index into wbuff
         nindx = ZERO,                 // index of newline character
         subcnt = ZERO,                // number of characters in substring
         avindx = ZERO,                // index of argvStr array
         avCount = ZERO ;              // number of arguments i.e. 'argc'

   //* Create temp filnames for redirection of stdout and stderr.   *
   //* Unfortunately, the child process will not know how to delete *
   //* the temp files, so we must rely on the application destructor*
   //* to delete the temp files (which it does by default).         *
   gString stdoutTemp, stderrTemp ;
   if ( redirect )
   {
      this->fmPtr->CreateTempname ( stdoutTemp ) ;
      this->fmPtr->CreateTempname ( stderrTemp ) ;
   }

   //* Isolate the target application's name *
   gs = appSpec ;
   short sl = gs.findlast( fSLASH ) ;
   if ( sl >= ZERO )    // if a path specified and not just a filename
      gs.shiftChars( -(sl + 1) ) ;
   gs.copy( appName, gsDFLTBYTES ) ;
   argv[avCount++] = appName ;

   //* Convert caller's arguments to an array of char pointers.*
   while ( (nindx = argList.find( L'\n', windx )) >= ZERO )
   {
      subcnt = nindx - windx ;
      argList.substr( gs, windx, subcnt ) ;
      windx = nindx + 1 ;
      argv[avCount++] = &argvStr[avindx] ;
      avindx += gs.copy( &argvStr[avindx], gsDFLTBYTES ) ;
   }
   if ( windx < (wCnt - 1) )
   {
      subcnt = (wCnt - windx) - 1 ;
      argList.substr( gs, windx, subcnt ) ;
      argv[avCount++] = &argvStr[avindx] ;
      avindx += gs.copy( &argvStr[avindx], gsDFLTBYTES ) ;
   }
   argv[avCount] = NULL ;

   pid_t fpid = vfork () ;
   if ( fpid == ZERO )
   {  //* The child process (if created) executes here. *
      if ( redirect )
      {
         //*  1) Open (create) each temp file.             *
         //*  2) Reassign (redirect) stdout and stderr     *
         //*     to the new descriptors.                   *
         //*  3) Close the temp files.                     *
         soDesc = open ( stdoutTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         seDesc = open ( stderrTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         dup2 ( soDesc, STDOUT_FILENO) ;
         dup2 ( seDesc, STDERR_FILENO) ;
         close ( soDesc ) ;
         close ( seDesc ) ;
      }

      execvp ( appSpec, (char* const*)argv ) ;

      //* In case the exec call fails: child process MUST NOT return.*
      _exit ( 0 ) ;
   }
   else
   {  //* The parent process continues execution here *
      if ( fpid > ZERO )      // child successfully launched
      {
         /* Currently, nothing to be done. */
      }
   }
   return fpid ;

}  //* End LaunchExternalApplication() *

//****************************
//* LaunchDefaultApplication *
//****************************
//******************************************************************************
//* Launch the system default application for handling the specified file type.*
//*                                                                            *
//* Input  : targFile : filespec of file to be processed by the external       *
//*                     application                                            *
//*          redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) fork a new process: vfork()                                             *
//* 2) Redirection:                                                            *
//*    a) It is possible that the called application will write to the terminal*
//*       window, either during startup or with some later status message.     *
//*    b) If the called application is a console application, this is almost   *
//*       certain; while for external GUI applications, it is still likely to  *
//*       some extent.                                                         *
//*    c) If the parent application has relinquished control of stdout and     *
//*       stderr. i.e. the NCurses engine has been put into hibernation mode,  *
//*       then the called application can display messages without conflict.   *
//*    d) If, however, the parent application is still in active control of    *
//*       stdout and stderr, then resource conflicts are likely to occur.      *
//*       In this case, it is necessary to redirect the called application's   *
//*       output. Our solution is to redirect stdout and stderr to temporary   *
//*       files by default. These files will persist until the parent          *
//*       application closes. (To prevent this, call the method with           *
//*       'redirect' set to 'false'.) The Taggit-class destructor will delete  *
//*       these temporary files during application shutdown.                   *
//* 3) Use the new process to launch the application:                          *
//*    a) This method uses the 'xdg-open' system utility (shell script) which  *
//*       launches the application associated with the file type of 'targFile'.*
//*    b) execlp searches the path looking for 'xdg-open'                      *
//*       Because we are fairly safe in assuming that 'xdg-open' actually IS   *
//*       on the path, the first argument is the name of the script (no path). *
//*    c) The second argument is argv[0] i.e. the script name again.           *
//*    d) The third argument is argv[1] i.e. the filespec passed in by caller. *
//*    e) The fourth argument is argv[2] i.e. a null pointer                   *
//*    f) Note that xdg-open passes only the specified filespec to the target  *
//*       application, and cannot pass additional configuration arguments.     *
//*       To invoke the target application will additional setup/configuration *
//*       parameters, please see the LaunchExternalApplication() method, below.*
//*    g) The child process is not directly associated with the terminal       *
//*       window from which it was launched, so the window may be closed       *
//*       asynchronously.                                                      *
//*    h) The exec functions do not return. The child process terminates when  *
//*       the external application exits.                                      *
//* 4) primary process returns to caller                                       *
//*                                                                            *
//* Programmer's Note: This sequence assumes that multi-threading is not       *
//* active. For multi-threaded applications, use pthread_fork() instead.       *
//*                                                                            *
//*  --   --   --   --   --   --   --   --   --   --   --   --   --   --   --  *
//* The xdg group of utilities and the xdg standard are maintained by          *
//* freedesktop.org. <https://specifications.freedesktop.org/                  *
//*                                    mime-apps-spec/mime-apps-spec-1.0.html> *
//* This specifies how to setup the association between an application and     *
//* the MIME types it supports.                                                *
//*                                                                            *
//*  --   --   --   --   --   --   --   --   --   --   --   --   --   --   --  *
//* How xdg-open determines which application matches each MIME type.          *
//* -----------------------------------------------------------------          *
//* <https://wiki.archlinux.org/index.php/default_applications#Desktop_entries>*
//*                                                                            *
//* Each application is associated with a ".desktop" file.                     *
//*     Example:  libreoffice-writer.desktop                                   *
//* These files list which file types may be handled by the application;       *
//* however, it would be unreasonable to scan each of these ".desktop" files   *
//* every time to find the correct application for a particular MIME type.     *
//*                                                                            *
//* Instead, xdg-open calls xdg-mime which scans the database:                 *
//*     /usr/share/applications/mimeinfo.cache                                 *
//* and returns the MIME type associated with the filetype to be opened.       *
//* For example, invoking xdg-open for an 'rtf' file:                          *
//*     xdg-open grocery_list.rtf                                              *
//* Which results in a call:                                                   *
//*     xdg-mime query filetype grocery_list.rtf                               *
//* Which yields an entry from the database:                                   *
//*     application/rtf=libreoffice-writer.desktop;                            *
//* 'sed' is used to isolate the MIME type:                                    *
//*     application/rtf                                                        *
//*                                                                            *
//* Manual Exploration:                                                        *
//* -------------------                                                        *
//* # determine a file's MIME type                                             *
//* $ xdg-mime query filetype photo.jpeg                                       *
//* image/jpeg                                                                 *
//*                                                                            *
//* # determine the default application for a MIME type                        *
//* $ xdg-mime query default image/jpeg                                        *
//* gimp.desktop                                                               *
//*                                                                            *
//* # change the default application for a MIME type                           *
//* $ xdg-mime default feh.desktop image/jpeg                                  *
//*                                                                            *
//* # open a file with its default application                                 *
//* $ xdg-open photo.jpeg                                                      *
//*                                                                            *
//* Several applications may support a particular MIME type:                   *
//*     audio/x-mp3=totem.desktop;vlc.desktop;rhythmbox.desktop;               *
//*                                                                            *
//* The default application associated with the MIME type is then executed     *
//* with the source file as its argument.                                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int FileDlg::LaunchDefaultApplication ( const char* targFile, bool redirect )
{
   gString stdoutTemp, stderrTemp ;       // stdout target file
   this->fmPtr->CreateTempname ( stdoutTemp ) ;  // stdout redirection file
   this->fmPtr->CreateTempname ( stderrTemp ) ;  // stderr redirection file
   int soDesc = ZERO,                     // stdout target descriptor
       seDesc ;                           // stderr target descriptor

   pid_t fpid = vfork () ;                // create the child process

   if ( fpid == ZERO )
   {  //* The child process (if created) executes here.                  *
      //* Create temp files as targets for redirecting stdout and stderr.*
      if ( redirect )
      {
         soDesc = open ( stdoutTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         seDesc = open ( stderrTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         dup2 ( soDesc, STDOUT_FILENO) ;
         dup2 ( seDesc, STDERR_FILENO) ;
         close ( soDesc ) ;
         close ( seDesc ) ;
      }

      execlp ( "xdg-open", "xdg-open", targFile, NULL ) ;

      //* In case the exec call fails: child process MUST NOT return.*
      _exit ( 0 ) ;
   }
   else
   {  //* The parent process continues execution here *
      if ( fpid > ZERO )      // child successfully launched
      {
         /* Currently, nothing to be done. */
      }
   }
   return fpid ;

}  //* End LaunchDefaultApplication() *

//*************************
//*     vfcIsTextfile     *
//*************************
//******************************************************************************
//* Determine whether the specified file is a text file or a binary file.      *
//* Calls the 'file' utility which scans for filesystem markers, magic numbers,*
//* and various text test (UTF-8-16-32, etc.)                                  *
//*                                                                            *
//* Input  : gsTrg : (by reference) target filespec                            *
//*                                                                            *
//* Returns: 'true' if target file contains only text,                         *
//*           else 'false' (binary data)                                       *
//******************************************************************************

bool FileDlg::vfcIsTextfile ( const gString& gsTrg )
{
   //* 'file' switches:                                               *
   //*      --brief         brief output (report only file type info) *
   //*      --mime          report canonical MIME-type string         *
   //*      --dereference   follow symbolic links                     *
   const char* cmdTemplate = 
               "file --brief --mime --dereference \"%S\" 1>\"%S\" 2>\"%S\"" ;

   gString gsTmp, gsCmd ;
   bool isText = false ;

   //* Create a temporary file *
   this->fmPtr->CreateTempname ( gsTmp ) ;

   //* Create a command which invokes 'file' utility *
   gsCmd.compose( cmdTemplate, gsTrg.gstr(), gsTmp.gstr(), gsTmp.gstr() ) ;

   //* Execute the command *
   this->dPtr->ShellOut ( soX, gsCmd.ustr() ) ;

   //* Scan the temp file. If target is a text file, *
   //* the first word in the string should be "text".*
   ifstream ifs( gsTmp.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      this->vfcGetLine ( ifs, gsCmd ) ;
      if ( (gsCmd.find( "text" )) == ZERO )
         isText = true ;
      ifs.close() ;      // close the temp file
   }
   this->fmPtr->DeleteTempname ( gsTmp ) ; // delete the temp file

   return isText ;

}  //* End vfcIsTextfile() *

//*************************
//*      vfcGetLine       *
//*************************
//******************************************************************************
//* Generic ifs.getline() method for View-File-Contents method group.          *
//* 1) Read the raw data line (discarding the newline character).              *
//* 2) If a Windoze carriage-return ('\r') character at end of line, delete it.*
//*                                                                            *
//*                                                                            *
//* Input  :  ifs     : (by reference) input data stream                       *
//*           gsIn    : (by reference) receives the stream data                *
//*                                                                            *
//* Returns: number of bytes read from the stream                              *
//*          (incl. CR if present, not incl. discarded newline)                *
//******************************************************************************

short FileDlg::vfcGetLine ( std::ifstream& ifs, gString& gsIn )
{
   char  inBuff[gsDFLTBYTES] ;      // input buffer
   short inBytes ;                  // return value

   ifs.getline( inBuff, gsDFLTBYTES, nckNEWLINE ) ;
   if ( (inBytes = ifs.gcount()) > ZERO )
   {
      short indx = (inBytes - 2) ;  // protect against Windoze CR/LF combo
      if ( (indx >= ZERO) && (inBuff[indx] == '\r') )
         inBuff[indx] = NULLCHAR ;
      gsIn = inBuff ;
   }
   return ( inBytes ) ;

}  //* End vfcGetLine() *

//*************************
//*  DefineChildEmulator  *
//*************************
//******************************************************************************
//* Member of DC_Emulator                                                      *
//* =====================                                                      *
//*                                                                            *
//* 1) Determine which terminal emulator and shell program are running in the  *
//*    current window.                                                         *
//* 2) Determine the size and position for a new terminal-emulator window.     *
//*    By default, the size and position are calculated based on the size and  *
//*    position of the parent window, but these calculations may be overridden *
//*    using the optional call parameters described below.                     *
//*                                                                            *
//* On entry, caller must have initialized the following data members:         *
//* 'tmpFile' : contains path of temporary file for collecting system info.    *
//* 'prgFile' : contains the filespec of program to be executed in to window.  *
//*             (May be initialized with the program name only if the program  *
//*              is known to be on the execution path.)                        *
//* 'argList' : contains additional arguments (if any) for program to be       *
//*             launched.                                                      *
//* 'autoReturn' : 'true' indicates that window should close immediately when  *
//*                       the launched program exits.                          *
//*                'false' indicates that window should remain open and return *
//*                       to command prompt after the launched program exits.  *
//* All other data members will be initialized here.                           *
//*                                                                            *
//* Input  : c : (optional, -1 by default)                                     *
//*              If non-default, specifies the number of columns               *
//*              for the new terminal-emulator window.                         *
//*        : r : (optional, -1 by default)                                     *
//*              If non-default, specifies the number of rows                  *
//*              for the new terminal-emulator window.                         *
//*        : x : (optional, -1 by default)                                     *
//*              If non-default, specifies the X offset (in columns)           *
//*              for the new terminal-emulator window.                         *
//*        : y : (optional, -1 by default)                                     *
//*              If non-default, specifies the Y offset (in rows)              *
//*              for the new terminal-emulator window.                         *
//*                                                                            *
//* Returns: value in dce.emuType                                              *
//*          gnomeType  : gnome-terminal (or unknown terminal)                 *
//*          konsoleType: konsole                                              *
//*          xtermType  : xterm                                                *
//******************************************************************************
//* Programmer's Note:                                                         *
//* This method is used to define a new terminal-emulator window which will be *
//* launched by a child process.                                               *
//*                                                                            *
//*         It is not necessary to understand what this method                 *
//*         is doing in order to understand the exec functions.                *
//*                                                                            *
//* 1) In previous releases, we did our best to query the X-server to determine*
//*    which terminal emulator we were running in and to launch a smaller      *
//*    version of the same emulator to gather the user's password.             *
//*    The X11 query algorithm was not foolproof; it was a simplistic way of   *
//*    identifying the terminal emulator in which we are running.              *
//*                                                                            *
//*    This method has been greatly simplified to work within the restrictions *
//*    imposed by Fedora's transition from X to the Wayland compositor.        *
//*    If you are curious about the X-server-query version of this method,     *
//*    please download FileMangler v:0.0.36, OR download our "Salmon" student  *
//*    exercise which discusses in detail how to programatically spawn a new   *
//*    terminal window.                                                        *
//*                                                                            *
//* 2) While there are at least twenty terminal emulator programs in general   *
//*    use, for our purposes we identify only three, and assume that at least  *
//*    one of them is available on any system.                                 *
//*      a) xterm                                                              *
//*      b) GNOME terminal                                                     *
//*      c) KDE konsole                                                        *
//*    If none of these emulators are installed on your system, then it is     *
//*    likely that mounting/unmounting devices requiring a password will fail. *
//*                                                                            *
//*    Also, there are many different shell programs in common use, so we      *
//*    rely on the $SHELL environment variable to tell us which one to use.    *
//*    If unable to get this information, we default to 'bash'.                *
//*                                                                            *
//* 3) Tales of Wayland the Obstructor (not exactly our favorite superhero):   *
//*    Fedora's transition from X11/Xorg to the Wayland compositor is a noble  *
//*    cause, but frustrating, annoying, buggy and often curse-worthy.         *
//*    In the interest of security, many things that are common practice in a  *
//*    standard X-windows system are forbidden under Wayland.                  *
//*    a) It is currently nearly impossible to determine what terminal emulator*
//*       our application is running under because the "ps" utility cannot get *
//*       the parent window's information ($WINDOWID is no longer an           *
//*       environment variable). The presence of other environment variables   *
//*       which might yield the needed information is similarly unreliable.    *
//*    b) Under Wayland, window positioning is forbidden, so the "geometry"    *
//*       offsets are ignored for gnome-terminal. Konsole no longer supports   *
//*       the "geometry" parameter at all, and instead requires info from a    *
//*       defined profile to indicate window size.                             *
//*    c) On the other hand, good-old xterm can still be relied upon for both  *
//*       size and position compliance. However, we must provide absolute      *
//*       positioning because Wayland will tell us neither the dimensions of   *
//*       the display nor the size or position of the parent window.           *
//*       (Note also that Xterm may no longer be installed by default on your  *
//*        system, which we consider to be the betrayal of a sacred trust!)    *
//*                                                                            *
//*                                                                            *
//* Opening a new terminal window (example):                                   *
//* ----------------------------------------                                   *
//*                                                                            *
//* gnome-terminal --geometry=80x8+400+300                                     *
//*        -e "bash -c \"grep -n \'COMPILE\' Makefile MakeChild ; exec bash\"" *
//*                                                                            *
//* a) Calling bash before and after the command will keep the terminal        *
//*    window open.                                                            *
//* b) Current-working-directory is default for GNOME --working-directory.     *
//*    Home directory is default for Konsole --workdir.                        *
//*    Current-working-directory is default for XTerm -working-directory.      *
//*    To explicity set the directory: --working-directory=$PWD                *
//*    To specify another directory  : --working-directory=$HOME/temp          *
//*    Keep in mind that the specification must be an absolute path, not a     *
//*    relative path an that the '~' shortcut character is not recognized.     *
//* c) Note that the example above describes the invocation of the Gnome       *
//*    terminal, and the bash shell. The equivalent command for Konsole or     *
//*    other terminals and for zsh and other shells may require a slightly     *
//*    modified command structure.                                             *
//*                                                                            *
//* Window dimensions and positioning:                                         *
//* ----------------------------------                                         *
//* Unfortunately, window size and position is not entirely under our control. *
//* Each terminal emulator insists of being different from its competitors,    *
//* and the environment they run in also must have its say in the proceedings. *
//* That said, here is what we would LIKE to see.                              *
//*                                                                            *
//* Position:                                                                  *
//* -- Ideally, the position will be one or two rows below and one column      *
//*    to the right of the X/Y anchor point of the current window.             *
//* -- Note that the Y offset is measured from the top of the tabs             *
//*    (if visible), otherwise the top of the user window _not_ from the top   *
//*    of the terminal software application.                                   *
//* Dimensions:                                                                *
//* -- Ideally, the width will be at least two columns less than the parent    *
//*    window width, and the height < current window height.                   *
//* -- A "standard" terminal window is 25 rows by 80 columns, so any           *
//*    spawned application should fit in a window that size.                   *
//* -- For launching a GUI application, or a console application with          *
//*    only a small amout of output, aesthetically, a smaller window           *
//*    is preferred. However, we can't determine this ahead of time.           *
//*                                                                            *
//******************************************************************************

EmuType DC_Emulator::DefineChildEmulator ( short c, short r, short x, short y )
{
   const short emuCOUNT = 3 ;
   const wchar_t* emuGroup[emuCOUNT] = 
   {
      L"xterm",
      L"gnome-terminal",
      L"konsole"
   } ;
   gString sysCmd,                     // generate system commands
           gs ;                        // text formatting
   char  inbuff[gsDFLTBYTES] ;         // input buffer
   this->emuType = xtermType ;         // return value (xterm is default)
   gs = emuGroup[ZERO] ;               // default emulator program
   gs.copy( this->emuName, enSIZE ) ;

   //* Read the name of the shell program from the terminal environment *
   sysCmd.compose( "echo ${SHELL} 1>%s", this->tmpFile ) ;
   system ( sysCmd.ustr() ) ;

   for ( short e = ZERO ; e < emuCOUNT ; ++e )
   {
      sysCmd.compose( "which %S 1>>%s 2>/dev/null", emuGroup[e], this->tmpFile ) ;
      system ( sysCmd.ustr() ) ;
   }

   //* Open the file *
   ifstream ifs ( this->tmpFile, ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      //* Read the first line in the file (shell program) *
      ifs.getline ( inbuff, gsDFLTBYTES, NEWLINE ) ;
      if ( (ifs.good()) || (ifs.gcount() > ZERO) )
      {
         //* Isolate the shell program filename *
         gs.loadChars( inbuff, ifs.gcount() ) ;
         short i = (gs.findlast( fSLASH )) + 1 ;
         if ( i > ZERO )
         {
            gs.shiftChars( -(i) ) ;
            gs.copy( this->shlName, enSIZE ) ;
         }

         //* Read as much of the file as necessary. If target  *
         //* emulator installed, response s/b something like:  *
         //*     /usr/bin/xterm                                *
         bool termFound = false ;
         for ( short e = ZERO ; (e < emuCOUNT) && !termFound ; ++e )
         {
            ifs.getline ( inbuff, gsDFLTBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount() > ZERO) )
            {
               gs.loadChars( inbuff, ifs.gcount() ) ;
               short i = (gs.findlast( fSLASH )) + 1 ;
               if ( i > ZERO )
               {
                  for ( short f = ZERO ; f < emuCOUNT ; ++f )
                  {
                     if ( (gs.compare( emuGroup[f], true, gsDFLTBYTES, i )) == ZERO )
                     {
                        gs = emuGroup[f] ;
                        gs.copy( this->emuName, enSIZE ) ;  // save emulator name
                        switch ( f )
                        {
                           case 1:  this->emuType = gnomeType ;   break ;
                           case 2:  this->emuType = konsoleType ; break ;
                           case 0:
                           default:
                              this->emuType = xtermType ;
                              break ;
                        }
                        termFound = true ;
                        break ;
                     }
                  }
               }
            }
         }
      }
      else     // file access error, return default shell
      {
         gs = "bash" ;
         gs.copy( this->shlName, enSIZE ) ;
      }
      ifs.close() ;                 // close the source file
   }

   //* Set the dimensions and position of the   *
   //* child window (geometry). See notes above.*
   if ( c == (-1) ) this->cols = 77 ;
   else             this->cols = c ;
   if ( r == (-1) ) this->rows = 23 ;
   else             this->rows = r ;
   if ( x == (-1) ) this->xpos = 150 ;
   else             this->xpos = x ;
   if ( y == (-1) ) this->ypos = 100 ;
   else             this->ypos = y ;

   //* Construct the geometry string and the command-sequence string.*
   if ( this->emuType == konsoleType )    // konsole terminal
   {
      // Programmer's Note: For newer versions of Konsole, "geometry" 
      // is no longer accepted as valid parameter.
      gs.compose( "--geometry=%dx%d+%d+%d", 
                  &this->cols, &this->rows, &this->xpos, &this->ypos ) ;
      sysCmd.compose( "%s -c \"\'%s\' %s ; exec %s\"",
         this->shlName, this->prgFile, this->argList, this->shlName ) ;
   }
   else if ( this->emuType == gnomeType ) // gnome terminal
   {
      gs.compose( "--geometry=%dx%d+%d+%d", 
                  &this->cols, &this->rows, &this->xpos, &this->ypos ) ;
      sysCmd.compose( "%s -c \"\'%s\' %s ; exec %s\"",
         this->shlName, this->prgFile, this->argList, this->shlName ) ;
   }
   else  // (this->emuType == xtermType) default
   {
      gs.compose( "%dx%d+%d+%d", 
                  &this->cols, &this->rows, &this->xpos, &this->ypos ) ;
      sysCmd.compose( "%s -c \"\'%s\' %s ; exec %s\"",
         this->shlName, this->prgFile, this->argList, this->shlName ) ;
   }
   gs.copy( this->geoData, enSIZE ) ;  // save the geometry sequence

   //* If windows is to close immediately after executing the *
   //* command sequence, remove the 'exec shell-prog' command.*
   if ( this->autoReturn )
   {
      short semiIndx = sysCmd.findlast( ';' ) - 1 ;
      sysCmd.replace( &sysCmd.gstr()[semiIndx], "\"", semiIndx ) ;
   }
   sysCmd.copy ( this->cmdData, gsDFLTBYTES ) ; // save the command sequence

   return this->emuType ;

}  //* End DefineChildEmulator() *

//*************************
//*      OpenTermwin      *
//*************************
//******************************************************************************
//* Member of DC_Emulator                                                      *
//* =====================                                                      *
//*                                                                            *
//* Spawn a new process and open the previously-defined terminal window.       *
//*   1) Create the command for opening a new terminal window, incorporating   *
//*      the caller's commands.                                                *
//*   2) Launch the child process which will control the new window.           *
//*   3) The child process will open the window.                               *
//*   4) The parent process will return to caller.                             *
//*                                                                            *
//* Input  : dce  : (by reference) contains the setup parameters for launching *
//*                 the new terminal-emulator window.                          *
//*                                                                            *
//* Returns: child process ID,                                                 *
//******************************************************************************
//* Launching a command from a new terminal window.                            *
//* -----------------------------------------------                            *
//* Because this command is rather complex, let's go through it step-by-step.  *
//* Note that each argument to the execlp() function is a string. To put it    *
//* another way, each argument is interpreted as a const char*, and arguments  *
//* are separated by commas creating an array of pointers, i.e. the argv[]     *
//* array.                                                                     *
//* The arguments described are specific to the GNOME terminal and to the      *
//* 'bash' shell program but other terminals and shells follow the same        *
//* general pattern.                                                           *
//*                                                                            *
//* 1) 'execlp'                                                                *
//*    This function carries the arguments to the system.                      *
//* 2) 'emuName'                                                               *
//*    This is the name of the terminal emulation program, which is assumed    *
//*    to be on the path where execlp() can find it.                           *
//* 3) 'emuName'                                                               *
//*    This is the path-less emulator name which constitutes argv[0].          *
//* 4) 'geoData'                                                               *
//*    This string is the geometry i.e. the size and position of the new       *
//*    window.  --geometry=COLSxROWS+XOFFSET+YOFFSET                           *
//*    Example: --geometry=65x8+400+300                                        *
//*         Note that KDE Konsole has a long-standing bug which ignores this   *
//*         command, and/or interprets the dimensions as pixels counts rather  *
//*         than as columns and rows. Newer Konsole will not accept the        *
//*         "geometry" parameter, so we have removed it from the Konsole       *
//*         invocation.                                                        *
//*    Note that XTerm options begin with a single dash and do not allow the   *
//*    '=' to join the option and its argument.                                *
//*    Example: -geometry 65x8+400+300                                         *
//*             (this is two separate arguments)                               *
//* 5) 'hide-menubar'                                                          *
//*    Does what it says, prevents the menu bar from being displayed in the    *
//*    new window.                                                             *
//* 6) '--window' (gnome-terminal only): By default, gnome-terminal opens a    *
//*    new tab in the current window, so we must explicitly specify that the   *
//*    new terminal be opened in a new window.                                 *
//* 7) '-q' switch (gnome-terminal only): silent launch i.e. child process     *
//*    will not write to the parent window. The other emulators do not offer   *
//*    this option.                                                            *
//* 8) '-e' switch: This is the terminal emulator's "Execute" command. What    *
//*    follows it is the command to be executed inside the new terminal window.*
//*    The '-e' switch is used by all three emulators: gnome-terminal, konsole *
//*    and xterm,                                                              *
//* 9) 'cmdData'                                                               *
//*    This is the main argument. These are the commands to be executed within *
//*    the new terminal window.                                                *
//*    See the templates used to create the contents of the 'cmdData' string.  *
//*    a) Note that the entire argument is enclosed in quotation marks AND     *
//*       that the part of the argument that is the command to be executed is  *
//*       ALSO enclosed in (escaped) quotation marks.                          *
//*    b) 'bash -c'                                                            *
//*       This runs a copy of the 'bash' shell program inside the new terminal *
//*       window. The '-c' argument to the shell program indicates that the    *
//*       following argument is to be executed by the shell. Most shell        *
//*       programs use the "-c" command switch for this, but if you are using  *
//*       a shell program other than bash, zsh or csh, please verify the       *
//*       switch used by your shell.                                           *
//*       -- Note that if you do not require the window to remain open after   *
//*          execution, the shell specification may be omitted.                *
//*       -- Note that the command string is preceeded by an escaped           *
//*          double-quotation mark: ( \"commands... )                          *
//*       -- The matching escaped double-quotation mark follows (see item d)   *
//*          below:  ( ...exec bash\" )                                        *
//*       Placing the backslash before the quotation marks prevents the shell  *
//*       from removing them before the command is executed, allowing it to be *
//*       seen as a single argument.                                           *
//*       Similarly, escaped single-quotation marks must surrounding any       *
//*       argument which contains whitespace. The single-quotation marks will  *
//*       not be removed by the shell. Thus the token can be interpreted as a  *
//*       single argument. For example: \"grep -i \'Copyright Notice\' ....    *
//*    c) ';'                                                                  *
//*       The semicolon character separates the the main commands from the     *
//*       'exec' call, i.e. these are SEPARATE commands.                       *
//*    d) 'exec bash'                                                          *
//*       This command _replaces_ the currently running copy of 'bash' with a  *
//*       new copy. This may seem odd, but without the 'exec' two copies of    *
//*       the shell would be left running in the window.                       *
//*       -- Note that an escaped double-quotation mark follows this command   *
//*          which closes the escaped quotation pair (see item b) above.       *
//*       -- The _effect_ of this command is to keep the new window open and   *
//*          to return to the command prompt after all the specified commands  *
//*          have finished.                                                    *
//*       -- Note that a side-effect of this replacement is that the initial   *
//*          command(s) sent to the window will not be retained in the         *
//*          window's command history.                                         *
//*       -- If you want the window to close immediately when finished, then   *
//*          omit the 'exec bash'.                                             *
//* 10) NULL                                                                   *
//*    This is the NULL pointer indicating the end of the array of pointers.   *
//*               -----  -----  -----  -----  -----  -----  -----              *
//*                                                                            *
//* Working directory: For gnome-terminal and xterm the working directory is   *
//* taken from the parent's $PWD environment variable. However, Konsole        *
//* defaults to the directory specified in the profile, so we must explicitly  *
//* redirect it to the PWD using: "--workdir=."                                *
//******************************************************************************

pid_t DC_Emulator::OpenTermwin ( void )
{
   //* - - - - - - - - - - - - -*
   //* Create the child process *
   //* - - - - - - - - - - - - -*
   pid_t fpid = vfork () ;

   //* The child process (if created) executes here. *
   //*         (see notes in method header)          *
   if ( fpid == ZERO )
   {
      switch ( this->emuType )
      {
         case 0:     // gnome-terminal
            execlp ( this->emuName, this->emuName, this->geoData, "--hide-menubar", 
                     "--window", "-q", "-e", this->cmdData, NULL ) ;
            break ;
         case 1:     // konsole
            execlp ( this->emuName, this->emuName, 
                     "--hide-menubar", "--workdir=.", 
                     "-e", this->cmdData, NULL ) ;
            break ;
         case 2:     // xterm
            execlp ( this->emuName, this->emuName, "-geometry", this->geoData, 
                     "-fa", "Monospace", "-fs", "12",
                     "-e", this->cmdData, NULL ) ;
            break ;
      } ;
   
      //* In case the exec call fails: child process MUST NOT continue.*
      _exit ( 0 ) ;
   }

   return fpid ;

}  //* End OpenTermwin() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

