//********************************************************************************
//* File       : FMgrTree.cpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 09-Jul-2025                                                     *
//* Version    : (see FMgrVersion string, in FMgr.cpp)                           *
//*                                                                              *
//* Description: This module contains methods related to recursive traversal     *
//* of a directory tree.                                                         *
//*                                                                              *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2) under Fedora Release 12,              *
//*  Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above.                          *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FMgr.cpp.                                           *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*  Because these are recursive methods AND because the author seriously        *
//*  distrusts recursion, if something breaks, look here first!                  *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#ifndef FMGR_INCLUDED
#include "FMgr.hpp"
#endif

//***********************
//** Local Definitions **
//***********************
static const USHORT MAX_SUBTHREAD = 200 ;  // sub-thread allocation limit

//****************
//** Local Data **
//****************

//* Enable/disable debugging output for multi-threaded directory-tree scan.    *
#define DEBUG_MULTITHREAD (0)
#if DEBUG_MULTITHREAD != 0
 #define DMT_TREE (0)          // for debug output of directory tree
 #define DMT_VERBOSE (0)       // for verbose debugging output

 //** Non-member methods **
 #if DMT_TREE != 0
 static void dmtTreeNodeSummary ( const TreeNode* tnPtr, TreeNode& tnAcc ) ;
 static void dmtTNSum ( const TreeNode* tnPtr, TreeNode& tnAcc ) ;
 static void dmtLogTree ( TreeNode* tn ) ;
 static void dmtLogBranch ( TreeNode* tn ) ;
 #endif   // DMT_TREE

 const wchar_t* const dmtFmt =         // verbose output
    L"%s\n"
     "  Dir:%3u Non-Dir:%3u  '%s'  (fsMatch:%hhd)" ;
 const wchar_t* const dmtFmtv =        // extremely verbose output
    L"%S\n"
     "dirStat.fName              totFiles dirFiles regFiles lnkFiles cdvFiles bdvFiles fifFiles socFiles unkFiles tnFCount\n"
     "-------------------------- -------- -------- -------- -------- -------- -------- -------- -------- -------- --------\n"
     "%-26s %8u %8u %8u %8u %8u %8u %8u %8u %8u %8u\n" ;
 const char* const dmtFilename = "dmtDebug.txt" ;
 gString dmtFilepath ;
 gString dmtOut ;
 ofstream dmtofs ;
#endif   // DEBUG_MULTITHREAD


//******************************************************************************
//*** -------------------- BEGIN TreeNode CLASS METHODS -------------------- ***
//******************************************************************************

//*************************
//*     ~TreeNode         *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//* Releases dynamic allocation for current node and associated tnFName array. *
//*      .                                                                     *
//*                                                                            *
//* NOTE: This method should only be explicitly called by the ReleaseDirTree() *
//*       method unles you 'know' that no other nodes are attached to the one  *
//*       your are trying to delete. (see notes in TreeNode constructor below).*
//*                                                                            *
//*                                                                            *
//* Input Values : none                                                        *
//*                                                                            *
//* Return Value : none                                                        *
//******************************************************************************

TreeNode::~TreeNode ( void )
{
   //* Release dynamically allocated array of tnFName class instances *
   //* associated with this node.                                     *
   if ( this->tnFiles != NULL && this->tnfcount > ZERO )
   {
      delete [] this->tnFiles ;
      this->tnFiles = NULL ;
   }

}  //* End ~TreeNode() *

//*************************
//*      TreeNode         *
//*************************
//******************************************************************************
//* Default constructor.                                                       *
//*                                                                            *
//*                                                                            *
//* Input Values : none                                                        *
//*                                                                            *
//* Return Value : none (but constructor returns a pointer to class object)    *
//******************************************************************************

TreeNode::TreeNode ( void )
{
   //* Initialize data members *
   this->ReInit () ;

}  //* End TreeNode() *

//*************************
//*       ReInit          *
//*************************
//******************************************************************************
//* Initialize all data members for the class instance.                        *
//*                                                                            *
//*                                                                            *
//* Input Values : none                                                        *
//*                                                                            *
//* Return Value : none                                                        *
//******************************************************************************

void TreeNode::ReInit ( void )
{

   //* Initialize data members *
   this->totFiles  = this->dirFiles  = this->regFiles  = this->linkFiles = ZERO ;
   this->cdevFiles = this->bdevFiles = this->fifoFiles = this->sockFiles = ZERO ;
   this->unkFiles  = ZERO ;
   this->totBytes  = this->dirBytes  = this->regBytes  = this->linkBytes = ZERO ;
   this->cdevBytes = this->bdevBytes = this->fifoBytes = this->sockBytes = ZERO ;
   this->unkBytes  = ZERO ;

   this->level      = ZERO ;
   this->wprotFiles = violations = ZERO ;
   this->allocBytes = sizeof(TreeNode) ;
   this->tnFiles    = NULL ;
   this->tnFCount   = ZERO ;
   this->nlnodes    = ZERO ;  // private member
   this->tnfcount   = ZERO ;  // private member

   this->nextLevel = NULL ;
   this->prevLevel = NULL ;

   this->dirStat.ReInit () ;  // directory-file members

}  //* End ReInit() *

//*************************
//*      GetActual        *
//*************************
//******************************************************************************
//* Returns a copy of the TreeNode private data members. FOR DEBUG ONLY!!      *
//*                                                                            *
//* Input  : nodes  (by reference): gets a copy of nlnodes                     *
//*          fcount (by reference): gets a copy of tnfcount                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void TreeNode::GetActual ( UINT& nodes, UINT& fcount ) const
{

   nodes  = this->nlnodes ;
   fcount = this->tnfcount ;

}  //* End GetActual() *

//*************************
//*   Increment_nlnodes   *
//*************************
//******************************************************************************
//* Increment the _private_ member, 'nlnodes'.                                 *
//* This allows the FileDlg class to initialize the value in the special case  *
//* of initializing the clipboard's base node.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void TreeNode::Increment_nlnodes ( void )
{

   ++this->nlnodes ;

}  //* End Increment_nlnodes() *

//*************************
//*  Increment_tnfcount   *
//*************************
//******************************************************************************
//* Increment the _private_ member, 'tnfcount'.                                *
//* This allows the FileDlg class to initialize the value in the special case  *
//* of initializing the clipboard's base node.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void TreeNode::Increment_tnfcount ( void )
{

   ++this->tnfcount ;

}  //* End Increment_nlnodes() *

//*************************
//*         +=            *
//*************************
//********************************************************************************
//* Add contents of the accumulator members of source object to the              *
//* corresponding members of this target object.                                 *
//*                                                                              *
//* Input  : src : source object (by reference)                                  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* The following code may be used to test the functionality of this method.     *
//*                                                                              *
//*   TreeNode src, trg ;                                                        *
//*   src.totFiles  = 36 ; src.totBytes  = 360 ; src.wprotFiles = 16 ;           *
//*   src.dirFiles  =  8 ; src.dirBytes  =  80 ; src.violations = 15 ;           *
//*   src.regFiles  =  7 ; src.regBytes  =  70 ; src.allocBytes = 2250 ;         *
//*   src.linkFiles =  6 ; src.linkBytes =  60 ; trg.allocBytes = ZERO ;         *
//*   src.cdevFiles =  5 ; src.cdevBytes =  50 ;                                 *
//*   src.bdevFiles =  4 ; src.bdevBytes =  40 ;                                 *
//*   src.fifoFiles =  3 ; src.fifoBytes =  30 ;                                 *
//*   src.sockFiles =  2 ; src.sockBytes =  20 ;                                 *
//*   src.unkFiles  =  1 ; src.unkBytes  =  10 ;                                 *
//*   strncpy ( src.dirStat.fName, "SOURCE", MAX_FNAME ) ;                       *
//*   strncpy ( trg.dirStat.fName, "TARGET", MAX_FNAME ) ;                       *
//*   winPos ul = { ulY, ulX } ;                                                 *
//*   this->tnDisplayNode ( ul, &src ) ;                                         *
//*   this->tnDisplayNode ( ul, &trg ) ;                                         *
//*   trg += src ;                                                               *
//*   this->tnDisplayNode ( ul, &trg ) ;                                         *
//*                                                                              *
//********************************************************************************

void TreeNode::operator += ( const TreeNode& src )
{
   this->totFiles   += src.totFiles ;
   this->totBytes   += src.totBytes ;
   this->dirFiles   += src.dirFiles ;
   this->dirBytes   += src.dirBytes ;
   this->regFiles   += src.regFiles ;
   this->regBytes   += src.regBytes ;
   this->linkFiles  += src.linkFiles ;
   this->linkBytes  += src.linkBytes ;
   this->cdevFiles  += src.cdevFiles ;
   this->cdevBytes  += src.cdevBytes ;
   this->bdevFiles  += src.bdevFiles ;
   this->bdevBytes  += src.bdevBytes ;
   this->fifoFiles  += src.fifoFiles ;
   this->fifoBytes  += src.fifoBytes ;
   this->sockFiles  += src.sockFiles ;
   this->sockBytes  += src.sockBytes ;
   this->unkFiles   += src.unkFiles ;
   this->unkBytes   += src.unkBytes ;
//   this->tnFCount   += src.tnFCount ;  // misleading and dangerous to accumulate this
   this->wprotFiles += src.wprotFiles ;
   this->violations += src.violations ;
   this->allocBytes += src.allocBytes ;
   this->nlnodes    += src.nlnodes ;
   this->tnfcount   += src.tnfcount ;
}  //* End +=() *

//******************************************************************************
//*** --------------------- END TreeNode CLASS METHODS --------------------- ***
//******************************************************************************


//*************************
//*    CaptureDirTree     *
//*************************
//********************************************************************************
//* Scan the directory tree, starting at the specified directory.                *
//* The scan proceeds from the specified directory, downward, scanning each      *
//* subdirectory in the tree and capturing data on each directory and the        *
//* files they contain. (However, see notes on exceptions, below.)               *
//*                                                                              *
//* Produce a linked list of TreeNode class objects containing a summary of      *
//* the data for each directory in the tree.                                     *
//*                                                                              *
//* This method uses a multi-threaded approach to reading the directory tree.    *
//*                                                                              *
//* IMPORTANT NOTE: If called, this method allocates persistent memory.          *
//*                 It is the caller's responsibility to release that memory     *
//*                 when it is no longer needed (see ReleaseDirTree()).          *
//*                                                                              *
//*                                                                              *
//* Input  : dPath: full path specification of top-level directory               *
//*          dData: (by reference) DispData-class object for controlling         *
//*                  access to shared data.                                      *
//*          recurse: (optional, true by default)                                *
//*                 if 'true',  scan all files and directories in the tree       *
//*                 if 'false', scan only the top-level contents of the          *
//*                             specified directory                              *
//*                      Programmer's Note: See also data member 'irRootscan'.   *
//*          dironly: (optional, default: false)                                 *
//*                 if 'false', collect data for all objects, both filenames     *
//*                             and directory names                              *
//*                 if 'true',  collect directory names only                     *
//*                             Note: A count of non-directory files is returned *
//*                                   but no detail for the individual files.    *
//*                                                                              *
//* Returns: pointer to the head of a doubly-linked list of TreeNode class       *
//*          objects containing data on each directory in the tree and the       *
//*          files contained in those directories.                               *
//*          NOTE: Returns a NULL pointer if target is not a directory name      *
//*           or target does not exist or user does not have read permission     *
//********************************************************************************
//* Because we potentially create several threads in this scan, the              *
//* opportunities for havoc are great, so pay attention.                         *
//*                                                                              *
//* 1) If CWD contains no subdirectory files, then the primary thread will       *
//*    scan all entries, and there is no need to create additional threads.      *
//* 2) If CWD contains one or more subdirectories, we launch a monitor thread    *
//*    to handle the tree scan while the primary thread returns to caller        *
//*    (i.e. the user interface layer, which will handle the progress-monitor    *
//*    bar. See the sdtMgrthread() method for additional information on how      *
//*    the manager thread works.                                                 *
//* 3) There is a point where additional threads become absurd or impossible     *
//*    due to finite system resources (stack space, etc).                        *
//*    Thus, the maximum number of threads created == the number of              *
//*    second-level subdirectories below the CWD, arbitrarily limited to         *
//*    MAX_SUBTHREAD. (Note that on the test systems (64-bit, 16Gb) the          *
//*    physical limit for creation of new threads is approximately 385, at       *
//*    which time a system exception is thrown - complaining of insufficient     *
//*    system resources.) We set the MAX_SUBTHREAD at approximately 60% of       *
//*    that maximum, which provides some confidence that there will seldom be    *
//*    a thread-allocation problem. An exception is more likely during launch    *
//*    of the subthreads where stack space, timing and other factors combine     *
//*    to potentially create system resource shortfall. For this reason, the     *
//*    code is overly careful about the thread launch process, catching any      *
//*    exceptions and performing multiple retries.                               *
//* 4) There is both time and resource overhead in creating threads, so the      *
//*    decision on thread count is not a matter of simple algebra; however,      *
//*    the measured performance of additional threads is encouraging.            *
//* 5) The speed of the disk and its ability to respond to multiple threads      *
//*    simultaneously is also a factor in the amount of time necessary to        *
//*    collect the data, but that is essentially out of our control.             *
//*                                                                              *
//* Exceptions to the Recursive algorithm:                                       *
//* --------------------------------------                                       *
//* There are certain directories that act as mountpoint directories for         *
//* external filesystems such as CD/DVD drives, Flash drives, USB hard           *
//* drives or network drives. Some of the usual suspects are:                    *
//*    a)  /run/media/${USER}                                                    *
//*    b)  /run/user/${UID}                                                      *
//*    c)  /mnt                                                                  *
//*    d)  /media                                                                *
//*    e)  /  (root directory)                                                   *
//*       The 'root' directory is of course the container for all other          *
//*       directories, and it is therefore included as one of our                *
//*       directories targeted for special processing.                           *
//*                                                                              *
//* In general, the scan of the tree structure should not recurse into           *
//* externally-mounted filesystems. To do so would not only make the scan        *
//* extremely slow, but would potentially overwhelm the system's resources.      *
//* For this reason, the scan follows a filesystem-tracking protocol which       *
//* limits the scan to the filesystem which contains the base directory          *
//* specified for the scan.                                                      *
//*                                                                              *
//* 1) Uniquely identify the filesystem of the base directory.                   *
//*    This identifier is currently the filespec of the device driver which      *
//*    controls that filesystem. Other identifiers could be used, but            *
//*    statistically, this is the least expensive way to uniquely identify       *
//*    the filesystem. See the FilesystemID(gString&, gString&) method.          *
//*    a) Because this is an expensive (unbearably slow) test, we perform it     *
//*       only when the base directory is contained within one of the            *
//*       "exceptional" paths as listed above.                                   *
//*    b) The root directory ( '/' ) contains ALL system mountpoints, AND an     *
//*       incredibly large number of files and subdirectories to be scanned.     *
//*       For this reason, the scan handles the root directory as a special      *
//*       case. See below for details.                                           *
//*    c) It is _assumed_ that there will be no external mounts in the "/home"   *
//*       filesystem, and that there will seldom be an external mount anywhere   *
//*       other than on the paths listed. If there is a device mounted at some   *
//*       unexpected position on the directory tree, then the mounted device     *
//*       will be fully scanned. If this becomes a problem for a given system,   *
//*       then we recommend that you mount the external filesystem where rules   *
//*       and convention dictate.                                                *
//*    d) NFS filesystems are filesystems mounted on a remote server.            *
//*                                                                              *
//*                                                                              *
//* 2) Within the common mountpoint paths listed above, the scan will perform    *
//*    the unique-identification test on each subdirectory encountered.          *
//*    a) If the target subdirectory is on the same filesystem as the base       *
//*       directory of the scan, then the scan will recurse into the target      *
//*       subdirectory.                                                          *
//*    b) If the target subdirectory IS NOT on the same filesystem as the base   *
//*       directory, then the directory will be flagged, and recursion for       *
//*       that node of the directory tree will stop. The recursive scan of       *
//*       other nodes will continue.                                             *
//* 3) There are four(5) data members used to track whether an external          *
//*    filesystem has been encountered.                                          *
//*    a) For the FMgr class:                                                    *
//*        irID[]:        contains the filespec of the device driver which       *
//*                       controls filesystem of the base directory.             *
//*                       Example: "/dev/mapper/sdb1"                            *
//*        irMntpath:     flag indicating whether the scan is within one of      *
//*                       the common mountpoint paths ('true' if yes)            *
//*        irVirtual:     flag indicating whether the base directory for the     *
//*                       scan resides on a virtual filesystem (MTP/GVFS).       *
//*                       ('true' if GVFS, 'false' if real (block) filesystem)   *
//*        irOnesys:      flag indicating whether the scan has encountered       *
//*                       only one filesystem ('true' if yes, 'false' if no)     *
//*    b) For the tnFName class:                                                 *
//*        fsMatch:       flag indicating whether the directory described by     *
//*                       the TreeNode::dirStat object is on the same            *
//*                       filesystem as the base directory                       *
//*                       ('true' if yes, 'false' if no)                         *
//*                                                                              *
//* Scanning from the "root" directory:                                          *
//* -----------------------------------                                          *
//* Scanning the directory tree from the root presents several challenges.       *
//*  a) speed of scan                                                            *
//*  b) required resources                                                       *
//*  c) multiple mounted filesystems (both actual and virtual)                   *
//*  d) the fact that a user will seldom perform actual file operations on       *
//*     data in the root directory                                               *
//*                                                                              *
//* 1) As a design decision, when displaying the data in the root directory,     *
//*    by default, we do not scan below the top level data. This allows the      *
//*    data to be displayed quickly and allows the user to navigate to lower     *
//*    levels of the tree without significant delay.                             *
//*    a) The data member: 'this->irRootscan' controls whether recursion will    *
//*       be performed during the root scan.                                     *
//*                                                                              *
//* 2) Some operations (e.g. the Find Files utility) require that the tree be    *
//*    scanned as fully as possible. In these cases, we scan most of the tree    *
//*    normally, but perform special processing for the nodes that commonly      *
//*    contain mountpoints for external devices (DVD-rom drives, SD cards,       *
//*    USB flash drives, phone filesystems, etc.). These are:                    *
//*      /run/media                                                              *
//*      /run/user                                                               *
//*      /mnt                                                                    *
//*      /media                                                                  *
//*    For these nodes only, we test the unique filesystem ID (the device        *
//*    driver) for each directory scanned, and if that directory is on a         *
//*    filesystem different from the base node's filesystem, then we record      *
//*    the directory's name as part of the tree but do not recurse into it.      *
//*    Example:                                                                  *
//*       "/run"                   is on a virtual filesystem: "tmpfs" driver    *
//*       "/run/media"             is on a virtual filesystem: "tmpfs" driver    *
//*       "/run/media/sam"         is on a virtual filesystem: "tmpfs" driver    *
//*       "/run/media/sam/SD_BACKUP32" is an actual block filesystem with a      *
//*                                    device driver such as "/dev/mmcblk0p1"    *
//*                                                                              *
//*           ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---              *
//*                                                                              *
//* Special note on the "/proc" directory (the procfs virtual filesystem)        *
//* ---------------------------------------------------------------------        *
//* When scanning from "/" (root directory), the "/proc" directory is handled    *
//* as a special case. See notes in called methods.                              *
//*                                                                              *
//********************************************************************************

TreeNode* FMgr::CaptureDirTree ( const gString& dPath, DispData& dData, 
                                 bool recurse, bool dironly )
{
   TreeNode*   baseNode = NULL ;       // base node for directory-tree scan
   bool mgrthreadLaunched = false ;    // 'true' if manager thread launched

   //* Reset all data members EXCEPT for 'strWidth' and 'treeView' flag.*
   dData.reset() ;
   if ( dironly ) { dData.setTreeViewFlag() ; }

   //* 'lstat' the current directory name and be sure it IS a directory.*
   FileStats   rawStats ;
   if ( ((lstat64 ( dPath.ustr(), &rawStats )) == OK) &&
        ((this->DecodeFileType ( rawStats.st_mode )) == fmDIR_TYPE) )
   {
      //* Allocate an anchor node (see notes in TreeNode constructor) *
      baseNode = this->Allocate_TreeNodes ( 1 ) ;

      //* Save information for the current directory name *
      if ( ! (this->ExtractFilename ( baseNode->dirStat.fName, dPath )) )
      {  //* Base directory is 'root' directory *
         baseNode->dirStat.fName[0] = '/' ;
         baseNode->dirStat.fName[1] = NULLCHAR ;
      } 
      baseNode->dirStat.rawStats = rawStats ;
      baseNode->dirStat.fType    = fmDIR_TYPE ;
      baseNode->dirStat.fBytes   = rawStats.st_size ;
      this->DecodeEpochTime ( (int64_t)rawStats.st_mtime, baseNode->dirStat.modTime ) ;
      baseNode->dirStat.readAcc  = 
                     this->ReadPermission ( baseNode->dirStat, dPath.ustr() ) ;
      baseNode->dirStat.writeAcc = 
                     this->WritePermission ( baseNode->dirStat, dPath.ustr() ) ;

      //* If user has read access to the base directory *
      if ( baseNode->dirStat.readAcc )
      {
         //* Initialize the data members needed to prevent the  *
         //* scan from recursing into a different filesystem.   *
         //* If scan is within one of the common mount paths,   *
         //* determine the device driver for the base           *
         //* directory. This uniquely identifies the filesystem *
         //* of the base directory from which the scan begins.  *
         // Programmer's Note: Logical filesystems such as 'tmpfs' 
         // or 'proc' do not have device drivers in the traditional 
         // sense, but in that case another identifier is returned.
         this->irID[ZERO] = NULLCHAR ;    // initialize device driver filespec
         this->irOnesys   = true ;        // assume scan is within a single filesystem
         this->irVirtual  = false ;       // assume scan is not of virtual filesystem
         if ( (this->irMntpath = this->isMountpath ( dPath )) != false )
         {
            gString devPath ;             // receives filespec of device driver
            this->FilesystemID ( dPath, devPath ) ;
            devPath.copy( this->irID, MAX_FNAME ) ;
         }

         //* Determine whether base directory is on a virtual filesystem.*
         if ( (this->irVirtual = this->isGvfsPath ( dPath )) != false )
            recurse = false ;    // disable recursion

         //* Estimate the total number of top-level files and *
         //* directories to be scanned and captured.          *
         nodeCount basecnt ( ZERO ) ;
         dPath.copy( basecnt.trgPath, MAX_PATH ) ;
         if ( ! this->irVirtual )
            this->DirectoryCount ( basecnt ) ;

         #if DEBUG_MULTITHREAD != 0
         const char* homePath = getenv ( "HOME" ) ;
         dmtFilepath.compose( "%s/Apps/FileMangler/Temp/%s", homePath, dmtFilename ) ;
         dmtofs.open( dmtFilepath.ustr(), ofstream::out | ofstream::trunc ) ;
         if ( dmtofs.is_open() )
         {
            dmtofs << "Base Path: \n" ;
            dmtofs << dPath.ustr() << endl ;

            //* Determine the (approximate) maximum number  *
            //* of threads supported by the system hardware.*
            const UINT maxTHREADS = std::thread::hardware_concurrency () ;
            dmtOut.compose( "** std::thread::hardware_concurrency( %u ) **", &maxTHREADS ) ;
            dmtofs << dmtOut.ustr() << endl ;

            //* Get the mountpoint for base directory.      *
            gString devPath ;
            this->FilesystemID ( dPath, devPath, false ) ;
            dmtOut.compose( "** Filesystem device driver: \"%s\"\n"
                            "** Filesystem mountpoint   : \"%s\"\n"
                            "** irMntpath:%hhd  irOnesys: %hhd  irVirtual: %hhd\n", 
                            this->irID, devPath.ustr(), 
                            &this->irMntpath, &this->irOnesys, &this->irVirtual ) ;
            dmtofs << dmtOut.ustr() << endl ;
         }
         #endif   // DEBUG_MULTITHREAD

         //* For "real" filesystems (handled by the system kernel), *
         //* if there are no sub-directories below base directory,  *
         //* then all the data are captured by this call.           *
         //* Otherwise, the top-level files and directory names are *
         //* captured which gives the caller something to display   *
         //* while we scan through the remainder of the tree.       *
         //*                ---  ---  ---  ---                      *
         //* Virtual filesystems do not fully support recursion, so *
         //* only the top-level data are scanned.                   *
         if ( this->irVirtual )
            this->GvfsFlatScan ( basecnt, baseNode ) ;
         else
            this->ScanDirTree ( basecnt, baseNode, false, false, &dData ) ;

         #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
         if ( dmtofs.is_open() )
         {
            dmtofs << "TOP-LEVEL PRESCAN\n-----------------\n" ;
            dmtofs << dPath.ustr() << endl ;
            for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
            {
               UINT nondir = baseNode->nextLevel[stIndx].totFiles 
                             - baseNode->nextLevel[stIndx].dirFiles ;
               dmtOut.compose( "  Dir:%3u Non-Dir:%3u  '%s'  (fsMatch:%hhd)", 
                               &baseNode->nextLevel[stIndx].dirFiles, &nondir, 
                               baseNode->nextLevel[stIndx].dirStat.fName, 
                               &baseNode->nextLevel[stIndx].dirStat.fsMatch ) ;
               dmtofs << dmtOut.ustr() << endl ;
               if ( stIndx == (baseNode->nlnodes - 1) )
                  dmtofs << endl ;
            }
         }
         #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE


         //* If we are within one of the special mountpoint paths,  *
         //* the top-level scan may have identified directories     *
         //* which are mountpoints for external filesystems. Because*
         //* the 'DirectoryCount()' method returns an _actual_      *
         //* count, dircnt.dirFiles may be too large. Therefore, to *
         //* avoid allocating second-level sub-threads for nodes    *
         //* that should not be scanned, we adjust the directory    *
         //* count here. (Yes, it's a little kludgy, sorry :-)      *
         if ( this->irMntpath && !this->irVirtual )
         {
            for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
            {
               if ( (baseNode->nextLevel[stIndx].dirStat.fsMatch == false) && 
                    (basecnt.dirCnt > ZERO) )
                  --basecnt.dirCnt ;
            }
         }

         //* If base directory (current working directory) contains*
         //* one or more subdirectories, AND if caller has         *
         //* specified recursion through the tree, then create a   *
         //* monitor thread which will analyze the targets and     *
         //* launch additional threads as needed.                  *
         if ( (basecnt.dirCnt > ZERO) && (recurse != false) )
         {
            //* Create a secondary thread to manage the     *
            //* scan while primary thread returns to caller.*
            dData.createMgrthread() ;
            if ( dData.mgrThread != NULL )
            {
               for ( short i = 3 ; i > ZERO ; --i )
               {
                  try
                  {
                     *dData.mgrThread = thread( &FMgr::sdtMgrthread, this, 
                                                dPath, basecnt, baseNode, &dData ) ;
                     mgrthreadLaunched = true ;
                     break ;
                  }
                  catch ( ... )     // manager thread created but not launched
                  {
                     #if DEBUG_MULTITHREAD != 0
                     if ( dmtofs.is_open() )
                     {
                        dmtOut.compose( "Manager thread launch error (i == %hd)", &i ) ;
                        dmtofs << dmtOut.ustr() << endl ;
                     }
                     #endif   // DEBUG_MULTITHREAD
                  }
               }
               if ( ! mgrthreadLaunched )
                  dData.setException( "Scan-manager thread not launched." ) ;
            }
            else
            {
               //* In the unlikely case that the manager thread is not created,*
               //* top-level data have been scanned, but recursion will not    *
               //* have occurred. Set exception so tree scan will be restarted.*
               dData.setException( "Scan-manager thread allocation failed." ) ;
            }
         }
         #if DEBUG_MULTITHREAD != 0
         //* If manager thread was not launched, *
         //* then close the debug file here.     *
         else if ( dmtofs.is_open() )
         {
            dmtofs << "\nbye-bye!\n" << endl ;
            dmtofs.close() ;
         }
         #endif   // DEBUG_MULTITHREAD

         //* If only directory names are to be reported, *
         //* release the non-directory arrays.           *
         //* Note that some scan threads may still be    *
         //* active, so wait for them to finish.         *
         if ( dironly != false )
         {
            chrono::duration<short, std::milli>aMoment( 50 ) ;
            do { this_thread::sleep_for( aMoment ) ; }
            while ( (dData.getActiveThreads ()) > ZERO ) ;
            this->cdtStripNonDirFiles ( baseNode ) ;
         }
      }
      else
      {  //* No read access on base directory. Directory will appear as empty.*
         this->recentErrno = EACCES ;
      }
   }
   else
   {  //* Either target is inaccessible OR it is not a directory *
      this->recentErrno = errno ;
      this->DebugLog ( "In CaptureDirTree: lstat() failed", 
                       this->recentErrno, dPath.ustr() ) ;
   }

   //* If manager thread was not launched, signal that  *
   //* the user interface threads (primary and monitor  *
   //* threads) may regain control. Note that displayed *
   //* data may be incorrect or incomplete.             *
   if ( ! mgrthreadLaunched )
      dData.unblockUI() ;

   //* Primary thread returns to the user-interface scan.*
   return baseNode ;

}  //* End CaptureDirTree() *

//*************************
//*  cdtStripNonDirFiles  *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Called by CaptureDirTree() to remove all non-directory files from the        *
//* array and to release the associated dynamically allocated memory.            *
//*                                                                              *
//* Traverse the directory tree, and for each node:                              *
//*   a) Subtract the count on non-directory files from 'totFiles' member.       *
//*   b) Release the dynamic allocation (if any) for the tnFName array.          *
//*   c) Set the tnFName pointer to null pointer.                                *
//*   d) Set the count of non-directory files ('tnfcount') to zero.              *
//*                                                                              *
//* Programmer's Note: For efficiency, accumulators for non-directory file types *
//* (regFiles, regBytes, etc.) are not modified. Caller _should not_ need to     *
//* reference these values, but if it becomes a problem, they can be zeroed out  *
//* by this method.                                                              *
//*                                                                              *
//* Input  : tnPtr : pointer to TreeNode structure                               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FMgr::cdtStripNonDirFiles ( TreeNode* tnPtr )
{
   //* Release non-directory file at lower levels.*
   if ( (tnPtr->nlnodes > ZERO) && (tnPtr->nextLevel != NULL) )
   {
      for ( UINT node = ZERO ; node < tnPtr->nlnodes ; ++node )
         this->cdtStripNonDirFiles ( &tnPtr->nextLevel[node] ) ;
   }

   //* Clear top-level files *
   if ( tnPtr->tnFiles != NULL )
   {
//      tnPtr->totFiles -= tnPtr->tnfcount ; // OBSOLETE - RETURN THE COUNT
      // Note that this call sets array pointer to NULL.*
      this->Release_tnFNames ( tnPtr->tnFiles, tnPtr->tnfcount ) ;
      tnPtr->tnfcount = ZERO ;
   }

}  //* End cdtStripNonDirFiles() *

//*************************
//*     sdtMgrthread      *
//*************************
//********************************************************************************
//* PRIVATE METHOD - called only by CaptureDirTree() above.                      *
//* When multiple execution threads are required to scan the filesystem          *
//* directory tree, the thread to manage the capture starts life here.           *
//*                                                                              *
//* Input  : dPath   : full path specification of directory to scan              *
//*          basecnt : number of subdirectories and non-dir filenames in CWD     *
//*                    (basecnt.dirCnt > ZERO)                                   *
//*          baseNode: pointer to a partially-initialized TreeNode class object  *
//*                    (see details below)                                       *
//*          dData   : pointer to DispData-class object for controlling          *
//*                    access to shared data: updateData() method, and for       *
//*                    thread management.                                        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//*                                                                              *
//* 1) Caller has scanned and recorded all the base-level subdirectory files     *
//*    and base-level non-directory files. (This method is called ONLY when      *
//*    the base directory contains at least one subdirectory.)                   *
//*                                                                              *
//* 2) The second-level data below the base directory are recorded here.         *
//*    This is the _contents_ of the subdirectories in the base directory.       *
//*    a) This second-level scan determines the number of secondary-threads      *
//*       needed to complete the full scan of the directory tree, that is, one   *
//*       secondary thread for each second-level directory.                      *
//*    b) Note that there is an arbitrary upper limit placed on the number       *
//*       of secondary threads allocated. See item 4 below for details.          *
//*    c) The scan algorithm could be more efficient, but would require a more   *
//*       sophisticated thread-management system. We choose to keep the          *
//*       algorithm simple at the expense of optimum efficiency.                 *
//*    d) Programmer's Note: If basecnt.dirCnt > MAX_SUBTHREAD, then clearly     *
//*       the second-level count will be potentially much higher.                *
//*       In this case, we skip the second-level prescan altogether.             *
//*       For example, the "/proc" directory has about 250 top-level             *
//*       directories, and over 1,700 second-level directories.                  *
//*                                                                              *
//* 3) Thread concurrency: The number of concurrent threads supported by         *
//*    a given system is variable. For modern systems, this is probably          *
//*    between 4 and 8 concurrent threads. If more threads are allocated         *
//*    and launched, the system will put them into a swap rotation. This is      *
//*    not optimal, but the implementation is simpler and more direct than       *
//*    trying to dynamically apportion the tree node scan among the              *
//*    available hardware-implemented threads. The swap rotation is made         *
//*    somewhat more efficient by the fact that each thread will block for       *
//*    some period of time while waiting for requests to be fulfilled by         *
//*    physical storage devices such as hard drives. Note that there are at      *
//*    least two threads already active when the manager thread is launched.     *
//*                                                                              *
//* 4) Thread allocation (instantiation):                                        *
//*    a) If the required number of second-level threads <= MAX_SUBTHREAD,       *
//*       allocate one sub-thread for each second-level directory. This is the   *
//*       most efficient scenario. sdtMgrthread_Level2().                        *
//*    b) Else, if the required number of top-level threads <= MAX_SUBTHREAD,    *
//*       allocate one sub-thread for each top-level directory. This is also     *
//*       efficient. sdtMgrthread_Level1().                                      *
//*    c) If the required number of threads for scenarios A and B both exceed    *
//*       MAX_SUBTHREAD, then the algorithm requiring the fewest threads is      *
//*       used. In a future release, we may create a hybrid algorithm that       *
//*       stays within the sub-thread limit while retaining efficiency.          *
//*    d) It is POSSIBLE, though unlikely that the system will not be able to    *
//*       allocate the requested number of threads. In this case, the system     *
//*       will indicate the failed allocation by returning a NULL pointer.       *
//*       See DispData::createSubthreads() method for how the exception is       *
//*       caught and handled.                                                    *
//*                                                                              *
//* 5) Launch the allocated threads, one for each target directory:              *
//*    The array of threads will capture the remaining branches below the        *
//*    target level.                                                             *
//*    a) Launching the individual sub-threads:                                  *
//*       To catch an exception use:                                             *
//*         try { /* launch thread */ } catch { /* recover and/or report */}     *
//*       The problem is that according to the C++ documentation, parameterized  *
//*       thread constructors may eat any exception caused by calling the        *
//*       target method. The biggest issue in launching multiple threads is      *
//*       timing. We have spent considerable time tweaking the timing of each    *
//*       launch, and the exception capture/retry _appears_ to work properly,    *
//*       (we haven't had an exception for over two months of intensive          *
//*       testing), but if problems arise, the thread launch is the first        *
//*       place to look.                                                         *
//*    b) Keep in mind that two display windows may be simultaneouslly active.   *
//*       This could theoretically _double_ the number of active threads;        *
//*       however, the main thread will populate each window in sequence, so     *
//*       there is little or no overlap of resource requests between windows.    *
//*                                                                              *
//* 6) Special case: "root" directory ( "/" ):                                   *
//*    If 'dPath' specifies that the base directory is "/", then caller is       *
//*    instructing that we scan the system from the root directory downward;     *
//*    otherwise, caller will have already scanned the top level of the tree     *
//*    which is as much of the root directory as it needs for display.           *
//*    a) We have observed that there are seldom more than twenty(20)            *
//*       subdirectories off the root; HOWEVER, there could easily be 500 to     *
//*       1000 second-level subdirectories off the root directory.               *
//*       Therefore, for the root directory only, we do not perform the          *
//*       second-level scan, and instead use the top-level thread launch         *
//*       algorithm. While this will likely cause the scan to be slow, it is     *
//*       less likely to overwhelm the system with memory-allocation requests,   *
//*       filesystem-access requests and other system resources needed for a     *
//*       full second-level scan of the tree from the root directory.            *
//*                                                                              *
//* 7) Special case: smartphones, tablets and other devices the system           *
//*    accesses through the MTP protocol.                                        *
//*    a) The MTP/gvfs protocol is weak at best. Even Nautilus, which _should_   *
//*       know everything there is to know about filesystems under Linux,        *
//*       continually loses the connection to such devices, or is unbearably     *
//*       slow to respond.                                                       *
//*    b) Much of the problem seems to be the inadequate size of the             *
//*       read-ahead cache. When this is coupled with the unavoidable slowness   *
//*       of SD-RAM, SD cards, and silicon discs generally, it means that        *
//*       it's no wonder that the application just gives up on trying to read    *
//*       from or write to such devices.                                         *
//*    c) When the amounts of data being moved are small (approx. 8Mb or less)   *
//*       the system seems to handle the communications adequately. Even the     *
//*       'cp' command-line copy will work if the data are relatively small.     *
//*       Listing the directory tree, however, sometimes seems to overwhelm      *
//*       the target device.                                                     *
//*    d) Because these are serial devices (or are treated as serial devices     *
//*       by MTP), it is unlikely that an operation in progress can be           *
//*       interrupted or superceeded by another command.                         *
//*    e) Given all this, it seems that the proper way to access a smartphone    *
//*       system is with a single thread which maintains an external timer       *
//*       (monitor thread) which can update the user on progress (or lack        *
//*       thereof) and give the user comfort or an opportunity to abort the      *
//*       operation.                                                             *
//*    f) GVfs filesystems are mounted according to UID by default at the        *
//*       system-defined location on the directory tree. If present, the         *
//*       $XDG_RUNTIME_DIR environment variable indicates the base directory.    *
//*             Example: XDG_RUNTIME_DIR=/run/user/1000                          *
//*       The mountpoint for the GVfs filesystem would then be:                  *
//*             /run/user/1000/gvfs/mtp:host=DEVICE_IDENTIFIER                   *
//*                                                                              *
//* 8) Hidden files and subdirectories:                                          *
//*    a) For "hidden" files and subdirectories in the base node, the display    *
//*       routine adjusts the user-visible file counts ('dirFiles' and           *
//*       'tnFCount') and file size ('totBytes' and 'xxxBytes') to match the     *
//*       actual displayed data. We still have access to the true file counts    *
//*       ('nlnodes' and 'tnfcount'), but if "hidden" data are present, the      *
//*       the file-size accumulators will be smaller than the actual totals.     *
//*    b) Hidden data below the base node will be fully reported so that it      *
//*       will be included in any file manipulation operations.                  *
//*                                                                              *
//*                                                                              *
//********************************************************************************

void FMgr::sdtMgrthread ( const gString& dPath, const nodeCount& basecnt, 
                          TreeNode* baseNode, DispData* dData )
{
   //* Accumulators:                                                      *
   //* nextTotal  : accumulated total of second-level directories         *
   //* nextCnt    : array of objects, one for each target subdirectory    *
   nodeCount nextTotal( basecnt.level + 1 ) ;
   dPath.copy( nextTotal.trgPath, MAX_PATH ) ;
   nodeCount* nextCnt = new nodeCount[basecnt.dirCnt] ;

   gString nextPath ;         // construct target paths
   USHORT  tAlloc = ZERO ;    // number of sub-threads allocated
   bool rootBase = this->isRootScan () ;  // 'true' if base directory of scan is "/"

   //* Determine the number of second-level subdirectories.*
   if ( ! this->irVirtual && ! rootBase && (basecnt.dirCnt < MAX_SUBTHREAD) )
   {
      UINT ncIndx = ZERO ; // nextCnt[] array index
      for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
      {
         //* Test whether the next-level directory is on a different *
         //* filesystem. If not, count the second-level files and    *
         //* subdirectories and add them to the second-level totals. *
         if ( baseNode->nextLevel[stIndx].dirStat.fsMatch != false )
         {
            nextCnt[ncIndx].level = basecnt.level ;        // current dir-tree level
            nextPath.compose( "%S/%s", dPath.gstr(), 
                              baseNode->nextLevel[stIndx].dirStat.fName ) ;
            nextPath.copy( nextCnt[ncIndx].trgPath, MAX_PATH ) ;
   
            this->DirectoryCount ( nextCnt[ncIndx] ) ;
            nextTotal.dirCnt += nextCnt[ncIndx].dirCnt ;
            nextTotal.regCnt += nextCnt[ncIndx].regCnt ;
            ++ncIndx ;
         }
      }     // for(;;)
   }        //  ! irVirtual && !rootBase && (basecnt.dirCnt < MAX_SUBTHREAD)

   #if DEBUG_MULTITHREAD != 0
   if ( dmtofs.is_open() )
   {
      dmtOut.compose( "** Level 1 Threads Required  : %3u vs %3hu **\n" 
                      "** Level 2 Threads Required  : %3u vs %3hu **", 
                      &basecnt.dirCnt, &MAX_SUBTHREAD,
                      &nextTotal.dirCnt, &MAX_SUBTHREAD ) ;
      dmtofs << dmtOut.ustr() << endl ;
   }
   #endif   // DEBUG_MULTITHREAD

   //* If this is a scan from the root directory *
   if ( rootBase != false )
   {
      tAlloc = this->sdtMgrthread_LevelR ( basecnt, baseNode, dData ) ;
   }

   #if 0    // OBSOLETE - NOT CURRENTLY USED
   //* Else if this is a scan of a GVFS filesystem *
   else if ( this->irVirtual != false )
   {
      tAlloc = this->sdtMgrthread_LevelG ( basecnt, baseNode, dData ) ;
   }
   #endif   // OBSOLETE - NOT CURRENTLY USED

   //* Compare second-level directory count and top-level directory  *
   //* count against sub-thread limit to determine which algorithm   *
   //* to use for launching the sub-threads. (Top-level scan is      *
   //* always used when scanning from root directory, see above.)    *
   //* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
   //* If the total number of second-level threads required for the  *
   //* scan is <= to the "safe" sub-thread maximum, _OR_ if both     *
   //* top-level and second-level requirements are greater than the  *
   //* limit, choose the smaller of the two.                         *
   else if ( (((nextTotal.dirCnt > ZERO) && (nextTotal.dirCnt <= MAX_SUBTHREAD)) || 
              ((nextTotal.dirCnt > MAX_SUBTHREAD) && (nextTotal.dirCnt < basecnt.dirCnt))) )
   {
      //* Scan and capture data for the next level of the tree.*
      UINT ncIndx = ZERO ; // nextCnt[] array index
      for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
      {
         //* If the parent of the target directory has been tagged *
         //* as a different filesystem, tag the target directory   *
         //* to prevent deeper recursion.                          *
         if ( baseNode->dirStat.fsMatch == false )
            baseNode->nextLevel[stIndx].dirStat.fsMatch = false ;

         if ( baseNode->nextLevel[stIndx].dirStat.fsMatch != false )
         {
            this->ScanDirTree ( nextCnt[ncIndx], &baseNode->nextLevel[stIndx], 
                                false, false, dData ) ;

            #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
            if ( dmtofs.is_open() )
            {
               if ( stIndx == ZERO )
                  dmtofs << "\nSECOND-LEVEL PRESCAN\n--------------------\n" ;
               UINT nondir = baseNode->nextLevel[stIndx].totFiles 
                             - baseNode->nextLevel[stIndx].dirFiles ;
               dmtOut.compose( dmtFmt, nextCnt[ncIndx].trgPath, 
                               &baseNode->nextLevel[stIndx].dirFiles, &nondir, 
                               baseNode->nextLevel[stIndx].dirStat.fName, 
                               &baseNode->nextLevel[stIndx].dirStat.fsMatch ) ;
               dmtofs << dmtOut.ustr() << endl ;
               if ( stIndx == (baseNode->nlnodes - 1) )
                  dmtofs << endl ;
            }
            #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

            ++ncIndx ;  // nextCnt[] array index
         }
      }

      //* Allocate and launch threads, one for each second-level subdirectory.*
      tAlloc = this->sdtMgrthread_Level2 ( nextTotal, baseNode, dData ) ;
   }

   //* Else:
   //* 1) top-level threads required for scan is <= "safe" sub-thread maximum *
   //* 2) top-level thread requirement < second-level thread requirement      *
   //* 3) second-level pre-scan was not performed                             *
   else// if ( basecnt.dirCnt <= MAX_SUBTHREAD )
   {
      tAlloc = this->sdtMgrthread_Level1 ( basecnt, baseNode, dData ) ;
   }

   if ( nextCnt != NULL )  // release dynamic allocation of second-level count
   { delete [] nextCnt ; nextCnt = NULL ; }

   #if DEBUG_MULTITHREAD != 0
   if ( dmtofs.is_open() )
   {  //* Report the system exception(s) in the debug log.*
      const char* exMsg = NULL ;
      USHORT sysex = dData->getExceptions( exMsg ) ;
      if ( sysex > ZERO )
      {
         dmtOut.compose( " =========================\n"
                         " = System Exceptions: %02hu =\n"
                         " =========================", &sysex ) ;
         dmtofs << dmtOut.ustr() << "\n " << exMsg << endl ;
      }
   }
   #endif   // DEBUG_MULTITHREAD

   //* Verify that all threads were launched successfully, *
   //* and if not, report number of launch errors.         *
   USHORT tLaunched = ZERO ;
   if ( (tLaunched = dData->goodSubthreads()) != (tAlloc) )
   {  //* Note: This overrides any existing exception message(s).*
      dData->resetException() ;
      USHORT badLaunch = tAlloc - tLaunched ;
      gString gs( "%hu sub-thread(s) not responding.", &badLaunch ) ;
      dData->setException( gs.ustr() ) ;
   }

   #if DEBUG_MULTITHREAD != 0
   if ( dmtofs.is_open() )
   {
      #if DMT_VERBOSE != 0
      dmtOut.compose( "External Filesystem Stats: irMntpath:%hhd  irOnesys: %hhd", 
                      &this->irMntpath, &this->irOnesys ) ;
      dmtofs << dmtOut.ustr() << endl  ;
      #endif   // DMT_VERBOSE

      #if DMT_TREE != 0
      //* Display the directory tree.*
      dmtLogTree ( baseNode ) ;

      const wchar_t* endBar = L"\x02514\x02500\x02500\x02500\x02500\x02500"
                               "\x02500\x02500\x02500" ;

      //* Summarize the contents of the tree we have just captured. *
      TreeNode tnAcc ;
      dmtTreeNodeSummary ( baseNode, tnAcc ) ;
      gString gsDirs( tnAcc.dirFiles, 13, true ),
              gsFile( tnAcc.tnFCount, 13, true ),
              gsByte( tnAcc.totBytes, 9, true, false, false, fikB ) ;
      dmtOut.compose( "%S  Totals: %S, %S, %S", endBar,
                      gsDirs.gstr(), gsFile.gstr(), gsByte.gstr() ) ;
      dmtofs << dmtOut.ustr() << endl ;
      #endif   // DMT_TREE

      dmtofs << "\nbye-bye!\n" << endl ;
      dmtofs.close() ;
   }
   #endif   // DEBUG_MULTITHREAD

   dData->completedMgrthread() ;    // signal that scan is finished

   //* IMPORTANT NOTE: When the thread leaves the scope of this method, 
   //*                 it exits (stops executing); however, its definition
   //*                 and data still exist, and must be released explicitly.

}  //* End sdtMgrthread() *

//*************************
//*  sdtMgrthread_Level2  *
//*************************
//********************************************************************************
//* PRIVATE METHOD - called only by stdMgrthread() above.                        *
//* Create and launch one execution thread for each second-level branch of the   *
//* directory tree to be scanned. Caller has determined that the number of       *
//* threads required is <= the maximum "safe" allocation limit.                  *
//* See also stdMgrthread_Level1() for top-level thread algorithm.               *
//*                                                                              *
//* Input  : nextTotal: (by reference) contains pre-scan results for thread      *
//*                     allocations (second-level directory totals)              *
//*          baseNode : pointer to partially-initialized TreeNode class object   *
//*          dData    : pointer to DispData-class object for controlling         *
//*                     access to shared data: updateData() method, and for      *
//*                     thread management.                                       *
//*                                                                              *
//* Returns: number of sub-threads allocated (and launched)                      *
//********************************************************************************
//* Programmer's Note: We launch all allocated threads, one for each node that   *
//* is not tagged as being in a different filesystem. This requires that the     *
//* caller has correctly counted the non-tagged nodes (nextTotal.dirCnt).        *
//*                                                                              *
//* While this is inefficient in some ways, it allows us to index each node      *
//* sequentially and to maintain a simple count of launched threads.             *
//*                                                                              *
//* In this way, we can handle all the details here so the user-interface        *
//* layer doesn't need to know anything except whether the node was              *
//* fully scanned.                                                               *
//********************************************************************************

UINT FMgr::sdtMgrthread_Level2 ( const nodeCount& nextTotal, TreeNode* baseNode, 
                                 DispData* dData )
{
   gString nextPath ;      // path of target directory
   USHORT tAlloc = ZERO ;  // number of sub-threads allocated (return value)

   #if DEBUG_MULTITHREAD != 0
   if ( dmtofs.is_open() && false )
   {
      dmtOut.compose( "sdtMgrthread_Level2()\n"
                      "   nextTotal.dirCnt: %u\n"
                      "   baseNode->dirStat.fName:%s\n"
                      "   baseNode->dirStat.fsMatch:%hhd\n"
                      "   baseNode->nlnodes:%u\n"
                      "   baseNode->dirFiles:%u\n",
                      &nextTotal.dirCnt, 
                      baseNode->dirStat.fName,
                      &baseNode->dirStat.fsMatch,
                      &baseNode->nlnodes,
                      &baseNode->dirFiles ) ;
      dmtofs << dmtOut.ustr() ;
      for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
      {
         dmtOut.compose( "   nextLevel[%u] : fName:%s  fsMatch:%hhd\n",
                         &stIndx, 
                         baseNode->nextLevel[stIndx].dirStat.fName, 
                         &baseNode->nextLevel[stIndx].dirStat.fsMatch ) ;
         dmtofs << dmtOut.ustr() ;
      }
      dmtofs << endl ;
   }
   #endif   // DEBUG_MULTITHREAD

   //* Instantiate the array of second-level thread objects *
   if ( (nextTotal.dirCnt > ZERO) &&
        (tAlloc = dData->createSubthreads( nextTotal.dirCnt )) == nextTotal.dirCnt )
   {
      #if DEBUG_MULTITHREAD != 0
      if ( dmtofs.is_open() )
      {
         dmtOut.compose( "** Level 2 Threads Allocated : %3hu of %3u **", 
                         &tAlloc, &nextTotal.dirCnt ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD

      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      //* Pause timer for thread launch confirmation.*
      //* For very high thread counts, we increase   *
      //* the duration by 50 percent.                *
      chrono::duration<short, std::milli>aMoment( 10 ) ;
      if ( nextTotal.dirCnt > MAX_SUBTHREAD )
         aMoment = chrono::duration<short, std::milli>(15) ;

      //* Launch the secondary threads to complete tree scan.*
      USHORT goodSubs = ZERO,    // snapshot count of threads successfully launched
             prevSubs = ZERO ;   // previous thread count snapshot
      UINT tIndx = ZERO ;        // index of thread definition
      for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
      {
         nextPath.compose( "%s/%s", nextTotal.trgPath, 
                            baseNode->nextLevel[stIndx].dirStat.fName ) ;

         for ( UINT nxtIndx = ZERO ; nxtIndx < baseNode->nextLevel[stIndx].nlnodes ; ++nxtIndx )
         {
            if ( baseNode->nextLevel[stIndx].dirStat.fsMatch != false )
            {
               dData->scanThreadPath[tIndx].compose( "%S/%s", nextPath.gstr(), 
                           baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirStat.fName ) ;

               //* Put the thread to work. *
               // Programmer's Note: This looped launch sequence may be entirely usless 
               // because the launch may never fail except for a race condition. 
               // Keep an eye on this.
               // It may be more effective to insert a small delay between each launch 
               // to be sure the previous launch is completed before starting the next one,
               // OR to use the "goodSubthreads()" method to verify one launch before 
               // starting the next one.
               // if ( baseNode->nextLevel[stIndx].dirStat.fsMatch == false )
               //    then: the launched thread should do nothing but bump the counter and exit.
               //          It is possible that 'nextLevel[]' array may not be fully initialized.
               // Signal the called sdtSubthread() that it should not process any data.
               // Either:
               // baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirStat.fsMatch == false
               //  OR
               // baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirStat.fName[0] == NULLCHAR
               //  (in which casedData->scanThreadPath[tIndx] will also be invalid.)
               for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
               {
                  try
                  {
                      dData->scanThread[tIndx] = 
                      thread( &FMgr::sdtSubthread, this, 
                              dData->scanThreadPath[tIndx], 
                              &baseNode->nextLevel[stIndx].nextLevel[nxtIndx], 
                              dData ) ;

                      //* Wait for the launched thread to respond.*
                      for ( short j = 3 ; j > ZERO ; --j )
                      {
                         this_thread::sleep_for( aMoment ) ;  // give the system some time
                         goodSubs = dData->goodSubthreads() ;
                         if ( goodSubs > prevSubs )
                         {
                            prevSubs = goodSubs ;
                            break ;
                         }

                         #if DEBUG_MULTITHREAD != 0
                         if ( dmtofs.is_open() )
                         {
                            dmtOut.compose( "Scan sub-thread not responding. (j==%hd)", &j ) ;
                            dmtofs << dmtOut.ustr() << endl ;
                         }
                         #endif   // DEBUG_MULTITHREAD

                         if ( j <= 1 )
                            dData->setException( "Scan sub-thread not responding." ) ;
                      }
                      break ;
                  }
                  catch ( ... )     // system exception
                  {
                     #if DEBUG_MULTITHREAD != 0
                     if ( dmtofs.is_open() )
                     {
                        dmtofs << dData->scanThreadPath[tIndx].ustr() 
                               << "  (" << tIndx << ")\n"
                                  " == System Exception sdtSubthread_Level2() launch. =="
                               << endl ;
                     }
                     #endif   // DEBUG_MULTITHREAD

                     //* If terminal retry failed, increment exception counter.*
                     if ( i <= 1 )
                        dData->setException( "Scan sub-thread launch failed." ) ;
                     else
                        this_thread::sleep_for( aMoment ) ;  // give the system some time
                  }
               }     // for(;;)

               #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
               if ( dmtofs.is_open() )
               {
                  // Programmer's Note: We do a two-step retrieval of the path 
                  // because otherwise, the array of gString objects will have 
                  // been deleted before we can report the last item.
                  gString gsx = dData->scanThreadPath[tIndx] ;
                  chrono::duration<short, std::milli>aMoment( 50 ) ;
                  this_thread::sleep_for( aMoment ) ;  // wait for thread to finish
                  if ( baseNode->nextLevel[stIndx].nextLevel != NULL )  // avoid accidents
                  {
                     if ( (stIndx == ZERO) && (nxtIndx == ZERO) )
                        dmtofs << "\nSECOND-LEVEL LAUNCH\n-------------------" << endl ;
                     UINT nondir = baseNode->nextLevel[stIndx].nextLevel[nxtIndx].totFiles 
                                   - baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirFiles ;
                     dmtOut.compose( dmtFmt, gsx.ustr(), 
                                     &baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirFiles, 
                                     &nondir, 
                                     baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirStat.fName, 
                                     &baseNode->nextLevel[stIndx].nextLevel[nxtIndx].dirStat.fsMatch ) ;
                                     
                     dmtofs << dmtOut.ustr() << endl ;
                  }
                  else  dmtofs << "no allocation" << endl ;
               }
               #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

               ++tIndx ;            // index the next free thread definition

            }        // fsMatch
         }           // inner loop
      }              // outer loop

      #if DEBUG_MULTITHREAD != 0
      if ( dmtofs.is_open() )
      {
         #if DMT_VERBOSE != 0
         dmtofs << endl ;
         #endif   // DMT_VERBOSE

         USHORT goodSubs = dData->goodSubthreads() ;
         dmtOut.compose( "** Level 2 Threads Launched  : %3hu of %3u **", &goodSubs, &nextTotal.dirCnt ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD
   }           // sub-threads allocated

   //* Sub-threads unneeded, OR system error in allocation of sub-threads. *
   else
   {
      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      #if DEBUG_MULTITHREAD != 0
      if ( nextTotal.dirCnt > ZERO )
      {
         if ( dmtofs.is_open() )
         {
            dmtofs << " ==========================================\n"
                      " = System Exception in CreateSubthreads() =\n"
                      " ==========================================" << endl ;
         }
      }
      #endif   // DEBUG_MULTITHREAD
   }

   return tAlloc ;

}  //* End sdtMgrthread_Level2() *

//*************************
//*  sdtMgrthread_Level1  *
//*************************
//******************************************************************************
//* PRIVATE METHOD - called only by stdMgrthread() above.                      *
//* Create and launch one execution thread for each top-level directory of the *
//* directory tree to be scanned. Caller has determined that the number of     *
//* threads required is <= the maximum "safe" allocation limit.                *
//* See also stdMgrthread_Level2() for second-level thread algorithm.          *
//*                                                                            *
//* Input  : baseCnt  : number of subdirectories and non-dir filenames in CWD  *
//*                     (basecnt.dirCnt > ZERO)                                *
//*          baseNode : pointer to partially-initialized TreeNode class object *
//*                     ( baseNode.nextLevel[] array is already initialized )  *
//*          dData    : pointer to DispData-class object for controlling       *
//*                     access to shared data: updateData() method, and for    *
//*                     thread management.                                     *
//*                                                                            *
//* Returns: number of sub-threads allocated (and launched)                    *
//******************************************************************************

UINT FMgr::sdtMgrthread_Level1 ( const nodeCount& baseCnt, TreeNode* baseNode, 
                                 DispData* dData )
{
   gString nextPath ;      // path of target directory
   USHORT tAlloc = ZERO ;  // number of sub-threads allocated (return value)

   //* Instantiate the array of top-level thread objects.*
   if ( (baseCnt.dirCnt > ZERO) &&
        (tAlloc = dData->createSubthreads( baseCnt.dirCnt )) == baseCnt.dirCnt )
   {
      #if DEBUG_MULTITHREAD != 0
      #if DMT_VERBOSE != 0
      gString gsx ;
      #endif   // DMT_VERBOSE
      if ( dmtofs.is_open() )
      {
         dmtOut.compose( "** Level 1 Threads Allocated : %3hu of %3u **", 
                         &tAlloc, &baseCnt.dirCnt ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD

      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      //* Pause timer for thread launch confirmation.*
      //* For very high thread counts, we increase   *
      //* the duration by 50 percent.                *
      chrono::duration<short, std::milli>aMoment( 10 ) ;
      if ( baseCnt.dirCnt > MAX_SUBTHREAD )
         aMoment = chrono::duration<short, std::milli>(15) ;

      //* Launch the secondary threads to complete tree scan.*
      USHORT goodSubs = ZERO,    // snapshot count of threads successfully launched
             prevSubs = ZERO ;   // previous thread count snapshot
      UINT   tIndx = ZERO ;      // index of thread definition
      for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
      {
         if ( baseNode->nextLevel[stIndx].dirStat.fsMatch != false )
         {
            //* Initialize the path to subdirectory *
            dData->scanThreadPath[tIndx].compose( "%s/%s", baseCnt.trgPath, 
                                       baseNode->nextLevel[stIndx].dirStat.fName ) ;

            //* Put the thread to work. *
            for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
            {
               try
               {
                  dData->scanThread[tIndx] = 
                     thread( &FMgr::sdtSubthread, this, dData->scanThreadPath[tIndx], 
                             &baseNode->nextLevel[stIndx], dData ) ;
   
                  //* Wait for the launched thread to respond.*
                  for ( short j = 3 ; j > ZERO ; --j )
                  {
                     this_thread::sleep_for( aMoment ) ;  // give the system some time
                     goodSubs = dData->goodSubthreads() ;
                     if ( goodSubs > prevSubs )
                     {
                        prevSubs = goodSubs ;
                        break ;
                     }

                     #if DEBUG_MULTITHREAD != 0
                     if ( dmtofs.is_open() )
                     {
                        dmtOut.compose( "Scan sub-thread not responding. (j==%hd)", &j ) ;
                        dmtofs << dmtOut.ustr() << endl ;
                     }
                     #endif   // DEBUG_MULTITHREAD
   
                     if ( j <= 1 )
                        dData->setException( "Scan sub-thread not responding." ) ;
                  }
                  break ;
               }
               catch ( ... )     // system exception
               {
                  #if DEBUG_MULTITHREAD != 0
                  if ( dmtofs.is_open() )
                  {
                     dmtofs << dData->scanThreadPath[tIndx].ustr() 
                            << "  (" << tIndx << ")\n"
                               " == System Exception sdtSubthread_Level1() launch. =="
                            << endl ;
                  }
                  #endif   // DEBUG_MULTITHREAD
   
                  //* If terminal retry failed, increment exception counter.*
                  if ( i <= 1 )
                     dData->setException( "Scan sub-thread launch failed." ) ;
                  else
                     this_thread::sleep_for( aMoment ) ;  // give the system some time
               }
            }  // for(;;)

            #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
            if ( dmtofs.is_open() )
            {
               // Programmer's Note: We do a two-step retrieval of the path 
               // because otherwise, the array of gString objects will have 
               // been deleted before we can report the last item.
               gsx = dData->scanThreadPath[tIndx] ;
               chrono::duration<short, std::milli>aWhile( 50 ) ;
               this_thread::sleep_for( aWhile ) ;   // wait for thread to finish
               if ( baseNode->nextLevel != NULL )   // avoid accidents
               {
                  if ( stIndx == ZERO )
                     dmtofs << "\nTOP-LEVEL LAUNCH\n----------------" << endl ;
                  UINT nondir = baseNode->nextLevel[stIndx].totFiles 
                                - baseNode->nextLevel[stIndx].dirFiles ;
                  dmtOut.compose( dmtFmt, gsx.ustr(),
                                  &baseNode->nextLevel[stIndx].dirFiles, 
                                  &nondir, 
                                  baseNode->nextLevel[stIndx].dirStat.fName, 
                                  &baseNode->nextLevel[stIndx].dirStat.fsMatch ) ;
                  dmtofs << dmtOut.ustr() << endl ;
               }
               else  dmtofs << "no allocation" << endl ;
            }
            #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

            ++tIndx ;            // index the next free thread definition

         }     // fsMatch
      }        // for(;;)

      #if DEBUG_MULTITHREAD != 0
      if ( dmtofs.is_open() )
      {
         #if DMT_VERBOSE != 0
         dmtofs << endl ;
         #endif   // DMT_VERBOSE

         chrono::duration<short, std::milli>aWhile( 50 ) ;
         this_thread::sleep_for( aWhile ) ;  // wait for thread to respond
         USHORT goodSubs = dData->goodSubthreads() ;
         dmtOut.compose( "** Level 1 Threads Launched  : %3hu of %3u **", 
                         &goodSubs, &baseCnt.dirCnt ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD
   }        // sub-threads allocated

   //* Sub-threads unneeded, OR system error in allocation of sub-threads. *
   else
   {
      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      #if DEBUG_MULTITHREAD != 0
      if ( baseCnt.dirCnt > ZERO )
      {
         if ( dmtofs.is_open() )
         {
            dmtofs << " ==========================================\n"
                      " = System Exception in CreateSubthreads() =\n"
                      " ==========================================" << endl ;
         }
      }
      #endif   // DEBUG_MULTITHREAD
   }

   return tAlloc ;

}  //* End sdtMgrthread_Level1() *

//*************************
//*  sdtMgrthread_LevelR  *
//*************************
//********************************************************************************
//* PRIVATE METHOD - called only by stdMgrthread() above.                        *
//*                                                                              *
//* When the base directory of the scan is the root directory, ( "/" ),          *
//* special processing is required to avoid recursion into external              *
//* filesystems. This method handles the special case of multiple filesystems,   *
//* both real (block storage devices) and virtual (dynamic system data) which    *
//* are attached to the root directory.                                          *
//*                                                                              *
//* The intent is to scan as much of the directory tree as possible, but         *
//* without severly impacting performance. For this reason, we make some         *
//* assumptions about certain nodes of the tree which commonly contain           *
//* mountpoints for externally-mounted filesystems. There are three such nodes:  *
//*   1) "/media"                                                                *
//*   2) "/mnt"                                                                  *
//*      On a standard system, the ONLY data these nodes should contain are      *
//*      externally mounted filesystems, so we scan only the top level to        *
//*      capture the names of the mountpoints (if any).                          *
//*   3) "/run"                                                                  *
//*      This node contains BOTH local system data and mountpoints for external  *
//*      filesystems. The external mounts (if any) are isolated in one of the    *
//*      following subdirectories:                                               *
//*        "/run/media/$USER"                                                    *
//*        "/run/user/$UID                                                       *
//*      For this node, the scan travels recursively downward testing each       *
//*      directory encountered to determine whether its filesystem is different  *
//*      from the root filesystem for this branch of the tree ("tmpfs").         *
//*                    --------------------------------                          *
//*                                                                              *
//* For the 'procfs' virtual filesystem, the data are located in the 'proc'      *
//* node off the root directory. There is a huge number of mostly meaningless    *
//* files in this node, so we do not perform a recursive scan on this node.      *
//* See procFlatscan() for details.                                              *
//*                                                                              *
//* For all other nodes recursion continues to the bottom of the node.           *
//*                                                                              *
//* See also sdtMgrthread_Level2() and sdtMgrthread_Level1().                    *
//*                                                                              *
//*                                                                              *
//* Input  : baseCnt  : number of subdirectories and non-dir filenames in CWD    *
//*                     (basecnt.dirCnt > ZERO)                                  *
//*          baseNode : pointer to partially-initialized TreeNode class object   *
//*                     ( baseNode.nextLevel[] array is already initialized )    *
//*          dData    : pointer to DispData-class object for controlling         *
//*                     access to shared data: updateData() method, and for      *
//*                     thread management.                                       *
//*                                                                              *
//* Returns: number of sub-threads allocated (and launched)                      *
//********************************************************************************

UINT FMgr::sdtMgrthread_LevelR ( const nodeCount& baseCnt, TreeNode* baseNode, 
                                 DispData* dData )
{
   gString nextPath,       // path of target directory
           nodeName ;      // name of target directory
   USHORT tAlloc = ZERO ;  // number of sub-threads allocated (return value)
   bool runNode = false,   // 'true' if node to be scanned is "/run", etc.
        procNode = false ; // 'true' if node to be scanned is "/proc"

   //* Instantiate the array of top-level thread objects.*
   if ( (baseCnt.dirCnt > ZERO) &&
        (tAlloc = dData->createSubthreads( baseCnt.dirCnt )) == baseCnt.dirCnt )
   {
      #if DEBUG_MULTITHREAD != 0
      if ( dmtofs.is_open() )
      {
         dmtOut.compose( "** Level R Threads Allocated : %3hu of %3u **", 
                         &tAlloc, &baseCnt.dirCnt ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD

      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      //* Pause timer for launched thread to respond *
      chrono::duration<short, std::milli>aMoment( 80 ) ;

      #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
      if ( dmtofs.is_open() )
      {
         dmtofs << "Level_R( baseCnt.trgPath: " << baseCnt.trgPath 
                << "  fsMatch:" << baseNode->dirStat.fsMatch << " )" 
                << " baseNode->nlnodes:" << baseNode->nlnodes << endl ;
      }
      #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

      //* Launch the secondary threads to complete tree scan.*
      USHORT goodSubs = ZERO,    // snapshot count of threads successfully launched
             prevSubs = ZERO ;   // previous thread count snapshot
      UINT   tIndx = ZERO ;      // index of thread definition
      for ( UINT stIndx = ZERO ; stIndx < baseNode->nlnodes ; ++stIndx )
      {
         if ( baseNode->nextLevel[stIndx].dirStat.fsMatch != false )
         {
            //* Top-level nodes which need special processing. *
            //*        (see notes in method header)            *
            nodeName = baseNode->nextLevel[stIndx].dirStat.fName ;
            if ( ((nodeName.compare( L"run" )) == ZERO) ||
                 ((nodeName.compare( L"media" )) == ZERO) ||
                 ((nodeName.compare( L"mnt" )) == ZERO) )
            {  //* These three nodes commonly contain mountpoints.   *
               //* In addition,, "/run" contains a large amount of   *
               //* local data. For this reason, we launch the threads*
               //* for these nodes into a specialized target method. *
               runNode = true ;
            }
            else if ( (nodeName.compare( L"proc" )) == ZERO )
               procNode = true ;

            //* Initialize the path to subdirectory *
            dData->scanThreadPath[tIndx].compose( "/%s", 
                                  baseNode->nextLevel[stIndx].dirStat.fName ) ;

            //* Put the thread to work. *
            for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
            {
               try
               {
                  if ( ! runNode && ! procNode ) // scan this node to the bottom
                  {
                     dData->scanThread[tIndx] = 
                        thread( &FMgr::sdtSubthread, this, dData->scanThreadPath[tIndx], 
                                &baseNode->nextLevel[stIndx], dData ) ;
                  }
                  else if ( procNode ) // flat scan of "/proc" node
                  {
                     dData->scanThread[tIndx] = 
                         thread( &FMgr::sdtSubthread_proc, this, 
                                 dData->scanThreadPath[tIndx], 
                                 &baseNode->nextLevel[stIndx], dData ) ;
                  }
                  else  // for the "/run", "/mnt" and "/media" directories only
                  {
                     dData->scanThread[tIndx] = 
                         thread( &FMgr::sdtSubthread_run, this, 
                                 dData->scanThreadPath[tIndx], 
                                 &baseNode->nextLevel[stIndx], dData ) ;
                  }

                  //* Wait for the launched thread to respond.*
                  for ( short j = 3 ; j > ZERO ; --j )
                  {
                     this_thread::sleep_for( aMoment ) ; // wait for launched thread
                     goodSubs = dData->goodSubthreads() ;
                     if ( goodSubs > prevSubs )
                     {
                        prevSubs = goodSubs ;
                        break ;
                     }
   
                     #if DEBUG_MULTITHREAD != 0
                     if ( dmtofs.is_open() )
                     {
                        dmtOut.compose( "Scan sub-thread not responding. (j==%hd)", &j ) ;
                        dmtofs << dmtOut.ustr() << endl ;
                     }
                     #endif   // DEBUG_MULTITHREAD
   
                     if ( j <= 1 )
                        dData->setException( "Scan sub-thread not responding." ) ;
                  }
                  break ;
               }
               catch ( ... )     // system exception
               {
                  #if DEBUG_MULTITHREAD != 0
                  if ( dmtofs.is_open() )
                  {
                     dmtofs << dData->scanThreadPath[tIndx].ustr() 
                            << "  (" << tIndx << ")\n"
                               " == System Exception sdtSubthread_LevelR() launch. =="
                            << endl ;
                  }
                  #endif   // DEBUG_MULTITHREAD
   
                  //* If terminal retry failed, increment exception counter.*
                  if ( i <= 1 )
                     dData->setException( "Scan sub-thread launch failed." ) ;
                  else
                     this_thread::sleep_for( aMoment ) ;  // give the system some time
               }
            }  // for(;;)

            runNode = procNode = false ; // be sure flags are reset after thread launch

            #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
            if ( dmtofs.is_open() )
            {
               // Programmer's Note: We do a two-step retrieval of the path 
               // because otherwise, the array of gString objects will have 
               // been deleted before we can report the last item.
               gString gsx = dData->scanThreadPath[tIndx] ;
               chrono::duration<short, std::milli>aWhile( 150 ) ;
               this_thread::sleep_for( aWhile ) ;   // wait for thread to finish
               if ( baseNode->nextLevel != NULL )   // avoid accidents
               {
                  if ( stIndx == ZERO )
                     dmtofs << "\nROOT-LEVEL LAUNCH\n-----------------" << endl ;
                  UINT nondir = baseNode->nextLevel[stIndx].totFiles 
                                - baseNode->nextLevel[stIndx].dirFiles ;
                  dmtOut.compose( dmtFmt, gsx.ustr(),
                                  &baseNode->nextLevel[stIndx].dirFiles, 
                                  &nondir, 
                                  baseNode->nextLevel[stIndx].dirStat.fName, 
                                  &baseNode->nextLevel[stIndx].dirStat.fsMatch ) ;
                  dmtofs << dmtOut.ustr() << endl ;
               }
               else  dmtofs << "no allocation" << endl ;
            }
            #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

            ++tIndx ;            // index the next free thread definition

         }     // fsMatch
      }        // for(;;)

      #if DEBUG_MULTITHREAD != 0
      if ( dmtofs.is_open() )
      {
         #if DMT_VERBOSE != 0
         dmtofs << endl ;
         #endif   // DMT_VERBOSE

         chrono::duration<short, std::milli>aWhile( 50 ) ;
         this_thread::sleep_for( aWhile ) ;  // wait for thread to respond
         USHORT goodSubs = dData->goodSubthreads() ;
         dmtOut.compose( "** Level R Threads Launched  : %3hu of %3u **", 
                         &goodSubs, &baseCnt.dirCnt ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
      #endif   // DEBUG_MULTITHREAD
   }        // sub-threads allocated

   //* Sub-threads unneeded, OR system error in allocation of sub-threads. *
   else
   {
      //* Signal that the user interface (progress monitor) *
      //* thread may continue.                              *
      dData->unblockUI() ;

      #if DEBUG_MULTITHREAD != 0
      if ( baseCnt.dirCnt > ZERO )
      {
         if ( dmtofs.is_open() )
         {
            dmtofs << " ==========================================\n"
                      " = System Exception in CreateSubthreads() =\n"
                      " ==========================================" << endl ;
         }
      }
      #endif   // DEBUG_MULTITHREAD
   }

   return tAlloc ;

}  //* End sdtMgrthread_LevelR() *

//*************************
//*     sdtSubthread      *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* --------------                                                               *
//* Called only by sdtMgrthread_Level2(), sdtMgrthread_Level1() and              *
//* sdtMgrthread_LevelR() above.                                                 *
//*                                                                              *
//* When multiple execution threads are created to scan the filesystem           *
//* directory tree, each one begins life in this method.                         *
//*                                                                              *
//* Each thread operates independently to scan its portion of the tree,          *
//* attaching the resulting TreeNode list to the caller's list.                  *
//*                                                                              *
//* When the thread has completed its scan, it will so indicate by a call to     *
//* dData->completedSubthread(). Then it will exit (stop execution) and wait     *
//* to be re-'join'ed with the scan-monitor thread.                              *
//*                                                                              *
//* Input  : dPath  : full path spec of directory in which to begin scan         *
//*          tnPtr  : pointer to TreeNode class object with all values set to    *
//*                   defaults (see TreeNode.ReInit()), except for the dirStat   *
//*                   structure which caller has initialized.                    *
//*          dData  : pointer to DispData-class object for controlling           *
//*                   access to shared data. This method uses only:              *
//*                   goodSubthread(),updateData() and completedSubthread()      *
//*                   methods of the DispData class.                             *
//*                                                                              *
//* Returns: nothing (does not "return" in the conventional sense)               *
//********************************************************************************

void FMgr::sdtSubthread ( const gString& dPath, TreeNode* tnPtr, DispData* dData )
{
   //* Register that thread was properly launched.*
   dData->goodSubthread() ;

   //* If the target directory is on the same filesystem as the base *
   //* directory of the scan, recursively scan the specified node of *
   //* the directory tree. Note that caller _should be_ smart enough *
   //* to call this method only for nodes in the same filesystem,    *
   //* but this test will prevent us from accessing nextLevel[]      *
   //* TreeNode objects which have not been fully initialized.       *
   if ( tnPtr->dirStat.fsMatch != false )
   {
      //* Summarize the node contents and allocate storage for each item.*
      nodeCount nodecnt ;
      dPath.copy( nodecnt.trgPath, MAX_PATH ) ;
      this->DirectoryCount ( nodecnt ) ;

      //* Scan the specified branch of the directory tree,     *
      //* creating storage nodes at each level.                *
      //* Caller's TreeNode object gets the scan-summary data. *
      this->ScanDirTree ( nodecnt, tnPtr, true, false, dData ) ;

      //* Count the number of files processed for this node.*
      UINT   fc = ZERO ;
      UINT64 bc = ZERO ;
      this->TreeNodeSummary ( tnPtr, fc, bc ) ;
      dData->updateData ( fc, bc ) ;
   }
   #if DEBUG_MULTITHREAD != 0
   else
   {
      if ( dmtofs.is_open() )
      {
         dmtOut.compose( "**No Recursion Into: %s", tnPtr->dirStat.fName ) ;
         dmtofs << dmtOut.ustr() << endl ;
      }
   }
   #endif   // DEBUG_MULTITHREAD

   //* Decrement the active-subthread counter *
   dData->completedSubthread () ;

   //* IMPORTANT NOTE: When the thread leaves the scope of this method, 
   //*                 it exits (stops executing); however, its definition
   //*                 and data still exist, and must be released separately.

}  //* End sdtSubthread() *

//*************************
//*   sdtSubthread_run    *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* --------------                                                               *
//* Called only by sdtMgrthread_LevelR() above.                                  *
//*                                                                              *
//* When an execution thread is created to scan the filesystem directory tree    *
//* starting at one of the directories which commonly contain mountpoints:       *
//* "/run", "/mnt", "/media", the thread begins life here.                       *
//*                                                                              *
//* This method is specific to these branches of the directory tree, and is      *
//* called only when the base directory of the scan is the root directory.       *
//*                                                                              *
//* When the thread has completed its scan, it will so indicate by a call to     *
//* dData->completedSubthread(). Then it will exit (stop execution) and wait     *
//* to be re-'join'ed with the scan-monitor thread.                              *
//*                                                                              *
//* Input  : dPath  : full path spec of directory in which to begin scan         *
//*          tnPtr  : pointer to TreeNode class object with all values set to    *
//*                   defaults (see TreeNode.ReInit()), except for the dirStat   *
//*                   structure which caller has initialized.                    *
//*          dData  : pointer to DispData-class object for controlling           *
//*                   access to shared data. This method uses only:              *
//*                   goodSubthread(),updateData() and completedSubthread()      *
//*                   methods of the DispData class.                             *
//*                                                                              *
//* Returns: nothing (does not "return" in the conventional sense)               *
//********************************************************************************

void FMgr::sdtSubthread_run ( const gString& dPath, TreeNode* tnPtr, DispData* dData )
{
   //* Register that thread was properly launched.*
   dData->goodSubthread() ;

   const wchar_t //*runPath   = L"/run",
                 *runmPath  = L"/run/media",
                 *runuPath  = L"/run/user",
                 *mediaPath = L"/media",
                 *mntPath   = L"/mnt" ;

   //* Summarize the node contents and allocate storage for each item.*
   nodeCount nodecnt ;
   dPath.copy( nodecnt.trgPath, MAX_PATH ) ;
   this->DirectoryCount ( nodecnt ) ;

   //* Scan the 1st-level data within the target node. *
   this->ScanDirTree ( nodecnt, tnPtr, false, false, dData ) ;

   //* If target node's unique identifier has not yet been    *
   //* captured, do it now. We will need it later. However,   *
   //* do not set the 'irMntpath' flag at this time because it*
   //* would be a significant performance hit if set too soon.*
   // Programmer's Note: Logical filesystems such as 'tmpfs' 
   // or 'proc' do not have device drivers in the traditional 
   // sense, but in that case another identifier is returned.
   if ( ! this->irMntpath )
   {
      gString devPath ;
      this->FilesystemID ( dPath, devPath ) ;
      devPath.copy( this->irID, MAX_FNAME ) ;
   }

   //* Accumulators:                                                   *
   //* nextTotal  : accumulated total of second-level directories      *
   //* nextCnt    : array of objects, one for each target subdirectory *
   nodeCount nextTotal( nodecnt.level + 1 ) ;
   dPath.copy( nextTotal.trgPath, MAX_PATH ) ;
   nodeCount* nextCnt = new nodeCount[nodecnt.dirCnt] ;
   gString nextPath ;         // construct target paths

   #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
   if ( dmtofs.is_open() )
   {
      dmtOut.compose( "       sdtSubthread_run(%s)\n"
                      "         b.trgPath: %s\n"
                      "         b.dirCnt : %u\n"
                      "         b.level  : %hd\n"
                      "         dirFiles : %u\n"
                      "         nlnodes  : %u\n"
                      "         irID     : '%s'",
                      dPath.ustr(),
                      nodecnt.trgPath,
                      &nodecnt.dirCnt,
                      &nodecnt.level,
                      &tnPtr->dirFiles,
                      &tnPtr->nlnodes,
                      this->irID
                      ) ;
      dmtofs << dmtOut.ustr() << endl ;
   }
   #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

   for ( UINT stIndx = ZERO ; stIndx < tnPtr->nlnodes ; ++stIndx )
   {
      nextCnt[stIndx].level = nodecnt.level + 1 ;
      nextPath.compose( "%s/%s", nodecnt.trgPath, 
                        tnPtr->nextLevel[stIndx].dirStat.fName ) ;
      nextPath.copy( nextCnt[stIndx].trgPath, MAX_PATH ) ;
      this->DirectoryCount ( nextCnt[stIndx] ) ;

      //* If one of the special directories, force the *
      //* filesystem comparison test which will end    *
      //* recursion if a filesystem mismatch found.    *
      if ( ((nextPath.compare( runmPath )) == ZERO) ||
           ((nextPath.compare( runuPath )) == ZERO) ||
           ((nextPath.find( mediaPath )) == ZERO) ||
           ((nextPath.find( mntPath )) == ZERO)
         )
      {
         this->ScanDirTree ( nextCnt[stIndx], &tnPtr->nextLevel[stIndx], true, true, dData ) ;
      }

      //* Otherwise, scan to the bottom of the node. *
      else
      {
         this->ScanDirTree ( nextCnt[stIndx], &tnPtr->nextLevel[stIndx], true, false, dData ) ;
      }

      //* If scan-abort flag set, terminate loop. *
      if ( (dData->getAbortFlag()) != false )
         break ;
   }

   if ( nextCnt != NULL )
   { delete [] nextCnt ; nextCnt = NULL ; }   // release the dynamic allocation

   //* Count the number of files processed for this node.*
   UINT   fc = ZERO ;
   UINT64 bc = ZERO ;
   this->TreeNodeSummary ( tnPtr, fc, bc ) ;
   dData->updateData ( fc, bc ) ;

   //* Decrement the active-subthread counter *
   dData->completedSubthread () ;

   //* IMPORTANT NOTE: When the thread leaves the scope of this method, 
   //*                 it exits (stops executing); however, its definition
   //*                 and data still exist, and must be released separately.

}  //* End sdtSubthread_run() *

//*************************
//*   sdtSubthread_proc   *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* --------------                                                               *
//* Called only by sdtMgrthread_LevelR() above and only for the "/proc" node.    *
//*                                                                              *
//* This method is specific to this branch of the directory tree, and is         *
//* called only when the base directory of the scan is the root directory.       *
//* See procFlatscan() for details.                                              *
//*                                                                              *
//* When the thread has completed its scan, it will so indicate by a call to     *
//* dData->completedSubthread(). Then it will exit (stop execution) and wait     *
//* to be re-'join'ed with the scan-monitor thread.                              *
//*                                                                              *
//* Input  : dPath  : full path spec of directory in which to begin scan         *
//*          tnPtr  : pointer to TreeNode class object with all values set to    *
//*                   defaults (see TreeNode.ReInit()), except for the dirStat   *
//*                   structure which caller has initialized.                    *
//*          dData  : pointer to DispData-class object for controlling           *
//*                   access to shared data. This method uses only:              *
//*                   goodSubthread(),updateData() and completedSubthread()      *
//*                   methods of the DispData class.                             *
//*                                                                              *
//* Returns: nothing (does not "return" in the conventional sense)               *
//********************************************************************************

void FMgr::sdtSubthread_proc ( const gString& dPath, TreeNode* tnPtr, DispData* dData )
{
   gString devPath ;                      // device name

   //* Register that thread was properly launched.*
   dData->goodSubthread() ;

   //* Summarize the node contents and allocate storage for each item.*
   nodeCount nodecnt ;
   dPath.copy( nodecnt.trgPath, MAX_PATH ) ;
   this->DirectoryCount ( nodecnt ) ;

   //* Scan the 1st-level data within the target node. *
   this->procFlatscan ( nodecnt, tnPtr ) ;

   // Programmer's Note: Logical filesystems such as 'tmpfs' 
   // or 'proc' do not have device drivers in the traditional 
   // sense, but in that case another identifier is returned.
   this->FilesystemID ( dPath, devPath ) ;
   devPath.copy( this->irID, MAX_FNAME ) ;

   //* Count the number of files processed and bytes  *
   //* count for this node and update accumulators.   *
   UINT   fc = ZERO ;
   UINT64 bc = ZERO ;
   this->TreeNodeSummary ( tnPtr, fc, bc ) ;
   dData->updateData ( fc, bc ) ;

   #if DEBUG_MULTITHREAD != 0 && DMT_VERBOSE != 0
   if ( dmtofs.is_open() )
   {
      dmtOut.compose( "       sdtSubthread_proc(%s)\n"
                      "         b.trgPath: %s\n"
                      "         b.dirCnt : %u\n"
                      "         b.level  : %hd\n"
                      "         dirFiles : %u\n"
                      "         nlnodes  : %u\n"
                      "         irID     : '%s'"
                      "         nodeFiles: %u\n"
                      "         nodeBytes: %llu",
                      dPath.ustr(),
                      nodecnt.trgPath,
                      &nodecnt.dirCnt,
                      &nodecnt.level,
                      &tnPtr->dirFiles,
                      &tnPtr->nlnodes,
                      this->irID,
                      &fc, &bc
                    ) ;
      dmtofs << dmtOut.ustr() << endl ;
   }
   #endif   // DEBUG_MULTITHREAD && DMT_VERBOSE

   //* Decrement the active-subthread counter *
   dData->completedSubthread () ;

   //* IMPORTANT NOTE: When the thread leaves the scope of this method, 
   //*                 it exits (stops executing); however, its definition
   //*                 and data still exist, and must be released separately.

}  //* End sdtSubthread_proc() *

//*************************
//*     ScanDirTree       *
//*************************
//********************************************************************************
//* Produce a list of TreeNode class objects containing a summary of             *
//* the data for each directory in the tree.                                     *
//*                                                                              *
//* PRIVATE, RECURSIVE METHOD - SEE PUBLIC METHOD CaptureDirTree() ABOVE.        *
//*                                                                              *
//* The TreeNode tree looks like this:                                           *
//*   TreeNode baseNode                                                          *
//*             |  |  |                                                          *
//*             |  |  +----tnFile (dirStat)                                      *
//*             |  |           |                                                 *
//*             |  |           +---information for the directory file itself     *
//*             |  |                                                             *
//*             |  +-----baseNode.tnFiles[]                                      *
//*             |                         |                                      *
//*             |                         + Array of tnFName objects, one for    *
//*             +--baseNode.nextLevel[]     each non-directory file contained    *
//*                                  |      in the directory named in TreeNode   *
//*                                  |      object                               *
//*                                  |                                           *
//*                                  +--Array of TreeNode objects recursively    *
//*                                     for each directory name to the bottom    *
//*                                     of the file system directory tree.       *
//*                                     (Note: for non-recursive option, the )   *
//*                                     (baseNode nextLevel array is the only)   *
//*                                     (level allocated, and this array's   )   *
//*                                     (data for lower levels is not        )   *
//*                                     (initialized.                        )   *
//*                                                                              *
//* Input  : nodecnt : number of subdirctories and non-directory filenames in    *
//*                    specified target directory                                *
//*                    'trgPath' contains path of target base directory          *
//*          tnPtr: pointer to TreeNode class object with all values set to      *
//*                 defaults (see TreeNode.ReInit()), except for the dirStat     *
//*                 structure which caller has initialized.                      *
//*          recurse: (optional, true by default)                                *
//*                 if 'true', scan all files and directories in the tree        *
//*                 if 'false', scan only the current, specified directory       *
//*          fsTest : (optional, false by default)                               *
//*                 if 'false', filesystem test is done only when                *
//*                             this->irMntpath != false                         *
//*                 if 'true',  force override of this->irMntpath and always     *
//*                             perform filesystem test                          *
//*          dData  : (optional, null pointer by default)                        *
//*                   pointer to DispData-class object for controlling           *
//*                   access to shared data. This method references ONLY two of  *
//*                   the DispData-class flags:                                  *
//*                   1) the scan-abort flag, which is used to report when user  *
//*                      has requested abort of the scan. Return to caller.      *
//*                   2) the directories-only flag, used during tree-mode scan to*
//*                      bypass decoding of entries for non-directory file types.*
//*                                                                              *
//* Returns: OK if successful, else system 'errno' code                          *
//*             If user aborts mid-scan, then no error code will be generated,   *
//*             so it is caller's responsibility to handle post-abort sequence.  *
//********************************************************************************
//* From bits/dirent.h                                                           *
//* The important point to note here is that the buffer for the filename         *
//* is the same 256 bytes we define throughout this class. The info page on      *
//* readdir warns about some systems not having enough room for the filename,    *
//* but we're OK here.                                                           *
//*                                                                              * 
//* struct dirent                   // used with readdir()                       *
//* {                                                                            *
//*    #ifndef __USE_FILE_OFFSET64                                               *
//*    __ino_t d_ino;                                                            *
//*    __off_t d_off;                                                            *
//*    #else                                                                     *
//*    __ino64_t d_ino;                                                          *
//*    __off64_t d_off;                                                          *
//*    #endif                                                                    *
//*    unsigned short int d_reclen;                                              *
//*    unsigned char d_type;                                                     *
//*    char d_name[256];                                                         *
//* } ;                                                                          *
//*                                                                              *
//* #ifdef __USE_LARGEFILE64                                                     *
//* struct dirent64                 // use with readdir64()                      *
//* {                                                                            *
//*    __ino64_t d_ino;             // file serial number                        *
//*    __off64_t d_off;             // file size in bytes                        *
//*    unsigned short int d_reclen; // size of the directory entry itself        *
//*    unsigned char d_type;        // file type                                 *
//*    char d_name[256];            // file name                                 *
//* } ;                                                                          *
//* #endif                                                                       *
//*                                                                              *
//* From dirent.h:                                                               *
//* enum   // File types for `d_type'                                            *
//* {                                                                            *
//*   DT_UNKNOWN = 0,                                                            *
//*   # define DT_UNKNOWN  DT_UNKNOWN                                            *
//*   DT_FIFO = 1,                                                               *
//*   # define DT_FIFO     DT_FIFO                                               *
//*   DT_CHR = 2,                                                                *
//*   # define DT_CHR      DT_CHR                                                *
//*   DT_DIR = 4,                                                                *
//*   # define DT_DIR      DT_DIR                                                *
//*   DT_BLK = 6,                                                                *
//*   # define DT_BLK      DT_BLK                                                *
//*   DT_REG = 8,                                                                *
//*   # define DT_REG      DT_REG                                                *
//*   DT_LNK = 10,                                                               *
//*   # define DT_LNK      DT_LNK                                                *
//*   DT_SOCK = 12,                                                              *
//*   # define DT_SOCK     DT_SOCK                                               *
//*   DT_WHT = 14                                                                *
//*   # define DT_WHT      DT_WHT                                                *
//* };                                                                           *
//*                                                                              *
//* Programmer's Note: Error reporting.                                          *
//* Errors are: Either directory could not be opened for reading (serious), or   *
//* one or more 'lstat' commands failed (less serious). If directory could not   *
//* be read, the file count will be zero for that node, so nothing bad should    *
//* happen aside from the fact that we allocated the memory for nothing.         *
//* If a file could not be included in the list because the 'lstat' failed,      *
//* fewer files will be reported than actually exist, but again, nothing bad     *
//* should happen.                                                               *
//* In either case, the baseNode->violations count will be non-zero, so that     *
//* would be a clue that caller should check the recentErrno status.             *
//* --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- --- ---  *
//* Notes on '/proc' directory:                                                  *
//* ---------------------------                                                  *
//* 1) If scanning from the root ("/") directory, the "/proc" node is handled    *
//*    as a special case. This is because the number of directories is very      *
//*    high and is in a rapid state of flux, _and_ because the procfs filesystem *
//*    which comprises this node contains little or no user-oriented data.       *
//*    a) A non-recursive (flat) scan of the node is performed.                  *
//* 2) If base directory is within the "/proc" node i.e. below root directory,   *
//*    then process the node in a normal (recursive) manner.                     *
//*                                                                              *
//* Technical Note: The state of the 'recurse' flag is ignored for the proc-node *
//* test. For technical reasons, the 'recurse' flag is always reset when scanning*
//* from the root directory ("/"). Thus, we don't know whether caller is just    *
//* scanning the top level or is actually traversing the tree. As a result,      *
//* we may sometimes scan (flat-scan) the 'proc' directory unnecessarily;        *
//* however, the result will only be a slight performance hit.                   *
//*                                                                              *
//********************************************************************************

short FMgr::ScanDirTree ( const nodeCount& nodecnt, TreeNode* tnPtr, 
                          bool recurse, bool fsTest, DispData* dData )
{
   short status = OK ;           // return value

   //* Caller has provided the top-level directory and file counts.*
   //* Allocate space for each.                                    *
   if ( (tnPtr->nlnodes = nodecnt.dirCnt) > ZERO )
   {
      tnPtr->nextLevel = this->Allocate_TreeNodes ( tnPtr->nlnodes ) ;

      for ( UINT i = ZERO ; i < tnPtr->nlnodes ; ++i )
      {
         tnPtr->nextLevel[i].level = tnPtr->level + 1 ;  // next lower level
         tnPtr->nextLevel[i].prevLevel = tnPtr ;         // point to parent
      }
      // Programmer's Note: tnPtr->dirFiles is initialized inside the loop below.
   }
   if ( (tnPtr->tnfcount = nodecnt.regCnt) > ZERO )
   {
      tnPtr->tnFiles = this->Allocate_tnFNames ( tnPtr->tnfcount ) ;
      tnPtr->tnFCount = tnPtr->tnfcount ; // user-visible file count
      tnPtr->allocBytes += sizeof(tnFName) * tnPtr->tnfcount ;
   }

   //* If this directory is on one of the "special paths",*
   //* determine whether the target directory is on the   *
   //* same filesystem as the base directory for the scan.*
   if ( (this->irMntpath != false) || (fsTest != false) )
   {
      gString gsTrg( nodecnt.trgPath ) ;
      if ( (tnPtr->dirStat.fsMatch = this->FilesystemMatch ( gsTrg )) == false )
         this->irOnesys = false ; // set flag to indicate external filesys found
   }

   //* Open the directory *
   DIR*     dirPtr ;
   if ( (tnPtr->dirStat.fsMatch != false) &&
        ((tnPtr->nlnodes > ZERO) || (tnPtr->tnfcount > ZERO)) &&
        (dirPtr = this->Safe_opendir ( nodecnt.trgPath )) != NULL )
   {
      gString   fnPath ;         // target filespec
      FileStats rawStats ;       // data from lstat
      fmFType   fileType ;       // file type
      UINT  fnIndex = ZERO, tnIndex = ZERO ; // array indices
      bool  showFile,            // set if file is to be included in display
            isProc ;             // set if directory containing 'procfs' filesystem
      deStats   *destat ;        // pointer to entry read from directory file
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* DO NOT include 'current dir' and 'parent dir' names but DO        *
         //* include 'hidden' files, i.e. filenames whose first character is   *
         //* a '.' (PERIOD).                                                   *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }
         isProc = false ;

         //* If this is a file we want to store in the list *
         if ( showFile != false )
         {
            //* 'lstat' the file *
            if ( (nodecnt.trgPath[0] == '/') && (nodecnt.trgPath[1] == NULLCHAR) )
               fnPath.compose( "/%s", destat->d_name ) ;
            else
               fnPath.compose( "%s/%s", nodecnt.trgPath, destat->d_name ) ;
            if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
            {
               //* Add size of file to running total and increment file count *
               tnPtr->totBytes += rawStats.st_size ;
               ++tnPtr->totFiles ;

               //* Find out what kind of file it is *
               fileType = this->DecodeFileType ( rawStats.st_mode ) ;

               //* Format and store the data *
               switch ( fileType )
               {
                  case fmDIR_TYPE:
                     //* Store data for directory filename *
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats = rawStats ;
                     this->strnCpy ( tnPtr->nextLevel[tnIndex].dirStat.fName, 
                                     destat->d_name, MAX_FNAME ) ;
                     this->DecodeEpochTime ( (int64_t)rawStats.st_mtime, 
                                 tnPtr->nextLevel[tnIndex].dirStat.modTime ) ;
                     tnPtr->nextLevel[tnIndex].dirStat.fType    = fileType ;
                     tnPtr->nextLevel[tnIndex].dirStat.fBytes   = rawStats.st_size ;
                     tnPtr->nextLevel[tnIndex].dirStat.readAcc  = 
                        this->ReadPermission ( tnPtr->nextLevel[tnIndex].dirStat, 
                                               fnPath.ustr() ) ;
                     tnPtr->nextLevel[tnIndex].dirStat.writeAcc =
                        this->WritePermission ( tnPtr->nextLevel[tnIndex].dirStat, 
                                                fnPath.ustr() ) ;
                     if ( ! tnPtr->nextLevel[tnIndex].dirStat.writeAcc )
                        ++tnPtr->wprotFiles ;
                     ++tnPtr->dirFiles ;
                     tnPtr->dirBytes += rawStats.st_size ;

                     //* If scan-abort flag set, terminate loop. *
                     if ( (dData != NULL) && ((dData->getAbortFlag()) != false) )
                     { break ; }

                     //* If target subdirectory is on one of the common   *
                     //* mountpoint paths, test whether it is on the same *
                     //* filesystem as the base directory for the scan.   *
                     //* If not, then reset the flag indicating that a    *
                     //* portion of the tree was not scanned.             *
                     if ( (this->irMntpath != false) || (fsTest != false) )
                     {
                        if ( (tnPtr->nextLevel[tnIndex].dirStat.fsMatch = 
                                    this->FilesystemMatch ( fnPath )) == false )
                           this->irOnesys = false ; // set flag to indicate external filesys found
                     }

                     //* When scanning from root, the '/proc' directory is    *
                     //* handled as a special case. See note in method header.*
                     //* Test whether target is _anywhere_ in the proc node.  *
                     else if ( (fnPath.compare( L"/proc", true, 5 )) == ZERO )
                     {
                        //* If target is an exact match to base of proc node, *
                        //* then the tree base is the root directory.         *
                        if ( fnPath.gschars() == 6 ) // (exact match)
                           isProc = true ;
                        // Programmer's Note: If tree base is within the "/proc" 
                        // node, i.e. below the root, then it will not be seen here.
                     }

                     if ( (recurse != false) && (isProc == false) &&
                          (tnPtr->nextLevel[tnIndex].dirStat.fsMatch != false) )
                     {
                        nodeCount ncnt ( tnPtr->nextLevel[tnIndex].level + 1 ) ;
                        fnPath.copy( ncnt.trgPath, MAX_PATH ) ;
                        this->DirectoryCount ( ncnt ) ;
                        // NOTE: This is a recursive call
                        this->ScanDirTree ( ncnt, &tnPtr->nextLevel[tnIndex], 
                                            true, fsTest, dData ) ;
                     }
                     ++tnIndex ;
                     break ;
                  case fmREG_TYPE:
                     ++tnPtr->regFiles ;
                     tnPtr->regBytes += rawStats.st_size ;
                     break ;
                  case fmLINK_TYPE:
                     ++tnPtr->linkFiles ;
                     tnPtr->linkBytes += rawStats.st_size ;
                     break ;
                  case fmCHDEV_TYPE:
                     ++tnPtr->cdevFiles ;
                     tnPtr->cdevBytes += rawStats.st_size ;
                     break ;
                  case fmBKDEV_TYPE:
                     ++tnPtr->bdevFiles ;
                     tnPtr->bdevBytes += rawStats.st_size ;
                     break ;
                  case fmFIFO_TYPE:
                     ++tnPtr->fifoFiles ;
                     tnPtr->fifoBytes += rawStats.st_size ;
                     break ;
                  case fmSOCK_TYPE:
                     ++tnPtr->sockFiles ;
                     tnPtr->sockBytes += rawStats.st_size ;
                     break ;
                  case fmUNKNOWN_TYPE:
                     ++tnPtr->unkFiles ;
                     tnPtr->unkBytes += rawStats.st_size ;
                     break ;
                  case fmTYPES:
                  default:
                     // unlikely
                     break ;
               }
               //* Store data for non-directory filenames *
               // Programmer's Note: For efficiency, if Tree-view mode scan, 
               // non-directory filetype data are not stored.
               if ( fileType != fmDIR_TYPE )
               {
                  if ( ! (dData->getTreeViewFlag()) )
                  {
                  tnPtr->tnFiles[fnIndex].rawStats = rawStats ;
                  this->strnCpy ( tnPtr->tnFiles[fnIndex].fName, destat->d_name, MAX_FNAME ) ;
                  this->DecodeEpochTime ( (int64_t)rawStats.st_mtime, 
                                             tnPtr->tnFiles[fnIndex].modTime ) ;
                  tnPtr->tnFiles[fnIndex].fType    = fileType ;
                  tnPtr->tnFiles[fnIndex].fBytes   = rawStats.st_size ;
                  tnPtr->tnFiles[fnIndex].readAcc  = 
                     this->ReadPermission ( tnPtr->tnFiles[fnIndex], fnPath.ustr() ) ;
                  tnPtr->tnFiles[fnIndex].writeAcc = 
                     this->WritePermission ( tnPtr->tnFiles[fnIndex], fnPath.ustr() ) ;
                  if ( ! tnPtr->tnFiles[fnIndex].writeAcc )
                     ++tnPtr->wprotFiles ;
                  }
                  ++fnIndex ;
               }
            }
            else     // 'lstat' system call returned an error condition
            {        // We can't access this file
               // For the temporary files (sockets, etc.) in "/proc" and other 
               // dynamic areas, the file may be deleted between reading its entry 
               // from the dir file and performing the lstat. 
               // If a directory name, the scan of this level will be completed, 
               // but this method will not recurse. 
               // Programmer's Note: If 'lstat' fails, report the information from 
               // the directory entry even though the file probably no longer exists.
               // 'nodeCount' will be too high if we don't count the failed lstats.
               if ( destat->d_type == DT_DIR )
               {
                  this->strnCpy ( tnPtr->nextLevel[tnIndex].dirStat.fName, 
                                  destat->d_name, MAX_FNAME ) ;
                  tnPtr->nextLevel[tnIndex].dirStat.fType    = fmDIR_TYPE ;
                  tnPtr->nextLevel[tnIndex].dirStat.fBytes   = ZERO ;
                  tnPtr->nextLevel[tnIndex].dirStat.readAcc  = 
                  tnPtr->nextLevel[tnIndex].dirStat.writeAcc = false ;
                  tnPtr->nextLevel[tnIndex].nextLevel = NULL ; // (be safe)
                  ++tnIndex ;
               }
               else
               {
                  if ( ! (dData->getTreeViewFlag()) )
                  {
                  this->strnCpy ( tnPtr->tnFiles[fnIndex].fName, destat->d_name, MAX_FNAME ) ;
                  tnPtr->tnFiles[fnIndex].fType    = fmREG_TYPE ;
                  tnPtr->tnFiles[fnIndex].fBytes   = ZERO ;
                  tnPtr->tnFiles[fnIndex].readAcc  = 
                  tnPtr->tnFiles[fnIndex].writeAcc = false ;
                  }
                  ++fnIndex ;
               }
               status = this->recentErrno = errno ;
               this->DebugLog ( "In ScanDirTree: lstat() failed", 
                                this->recentErrno, fnPath.ustr() ) ;
               ++tnPtr->violations ;
            }
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }
   else if ( (tnPtr->nlnodes > ZERO) || (tnPtr->tnfcount > ZERO) )
   {
      ++tnPtr->violations ;
   }

   return status ;

}  //* End ScanDirTree() *

//*************************
//*     procFlatscan      *
//*************************
//********************************************************************************
//* PRIVATE METHOD:  Called by ScanDirTree() (or an associated thread manager)   *
//* -----------------------------------------                                    *
//* Non-recursively scan a directory specified by 'trgPath'.                     *
//* This method is primarily for the "/proc" directory which lives at the        *
//* root of the filesystem and contains (the magical equivalent of) temporary    *
//* files for each system process. Most of these files contain no actual data.   *
//* This special processing is done to minimize the chance of system exceptions  *
//* caused by the fast cycling of target files and directories. See notes below. *
//*                                                                              *
//*                                                                              *
//* Input  : nodecnt : (by reference)                                            *
//*                    'trgPath' contains full path specification of directory   *
//*                              to be scanned                                   *
//*                    'dirCnt' (preliminary) count of directory names           *
//*                    'regCnt' (preliminary) count of non-directory names       *
//*                    'level' member is not referenced                          *
//*                                                                              *
//* Returns: total number of filenames scanned                                   *
//* (Totals do not include current "." or parent ".." directory shortcut names)  *
//********************************************************************************
//* Notes on the "/proc" directory (procfs filesystem)                           *
//* --------------------------------------------------                           *
//* The "/proc" directory is handled as a special case.                          *
//* 1) Even though there are dozens of sub-directories within "/proc", most      *
//*    of these directories belong to root and have restricted permissions.      *
//*    a) The top-level thread allocation requirement is approximately 247,      *
//*       which is beyond our thread limit. The second-level thread requirement  *
//*       is well over 1700 (which the system cannot handle).                    *
//*    b) The /proc directory contains an average of approximately 20 thousand   *
//*       items, only a few of which contain actual data.                        *
//*    c) For these reasons, we do not recurse into subdirectories below "/proc".*
//*       Even if we could do it safely(atomically), it would not yield any      *
//*       meaningful information because the data are not actual files in the    *
//*       traditional sense.                                                     *
//* 2) Because strictly numeric directory names contain only transient data      *
//*    which may appear/disappear at approximate 90% of the speed of light;      *
//*    recursing into those directories is both dangerous and useless.           *
//*    a) For numeric file/directory names, report the name, but do not recurse. *
//*    b) For numeric file/directory names it is better not to stat them at all, *
//*       but we do want to report what we can without creating an exception.    *
//*       All we know from the directory entry is the filename and file type.    *
//*       It is assumed that it has 'read' access because we read it.            *
//*       Other fields receive default values.                                   *
//* 3) The "/proc" directory contains approximately 50 non-numeric named files   *
//*    and a few symbolic links. Many of these have a zero size, indicating      *
//*    that they are placeholders of some kind.                                  *
//*    a) It is probably safe to lstat files with non-numeric names, but use     *
//*       caution.                                                               *
//*    b) The symbolic link files (generally) reference one of the numerically   *
//*       named subdirectories in the same directory, presumably so the system   *
//*       can find it easily.                                                    *
//*    c) Note that the file "/proc/kcore" lies to us about the size of the      *
//*       file, e.g. (140,737,477,877,760 == 1.407x10'14th == 0x7FFFFF602000).   *
//*       This is not actually the size of the file; it is an indicator of       *
//*       the size of the system's physical memory plus 4Kb padding (although    *
//*       it lies about that too).                                               *
//*       The actual size of the file changes from moment to moment, depending   *
//*       on what the system is doing. This file seems to be used to create the  *
//*       'core dump' data for when a process wanders off into the weeds. It is  *
//*       recommended that this file not be messed with (even if it was allowed).*
//*    d) Because there is a lag time between caller's count of directory        *
//*       entries and allocation of the node/file arrays, we pad the allocation  *
//*       count in case a new file is created during that time.                  *
//*                                                                              *
//* Technical Notes:                                                             *
//* ----------------                                                             *
//* Testing on multiple systems shows that there are approximately 250 strictly  *
//* numeric names plus approximately ten(10) named directories.                  *
//* There is also some number of non-directory, non-numeric filenames at the     *
//* top level.These will (probably) be either 'regular' files or symbolic links. *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

UINT FMgr::procFlatscan ( nodeCount& nodecnt, TreeNode* tnPtr )
{
   const char* kCore = "kcore" ; // special case filename (see note 3c above)

   #define DEBUG_PFS (0)      // for debugging only
   #if DEBUG_PFS != 0
   gString gsdbg ;            // output formatting
   gString logFilepath ;      // log-file spec
   #if 1    // When testing live target, create a temp file.
   this->CreateTempname ( logFilepath ) ;
   #else    // When testing static data, create temp file in 'nodecnt.trgPath' 
   logFilepath.compose( "%s/pfs.log", nodecnt.trgPath ) ;
   #endif   // create temp file
   ofstream ofs( logFilepath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      ofs << "Debug Log for ProcFlatScan\n"
             "==========================" << endl ;
      gsdbg.compose( "dirCnt :%u regCnt  :%u base_level:%hd", 
                     &nodecnt.dirCnt, &nodecnt.regCnt, &tnPtr->level ) ;
      ofs << gsdbg << endl ;
   }
   #endif   // DEBUG_PFS

   //* Caller has provided the top-level directory and file counts.*
   //* Allocate space for each.                                    *
   if ( (tnPtr->nlnodes = nodecnt.dirCnt) > ZERO )
   {
      tnPtr->nextLevel = this->Allocate_TreeNodes ( tnPtr->nlnodes ) ;

      for ( UINT i = ZERO ; i < tnPtr->nlnodes ; ++i )
      {
         tnPtr->nextLevel[i].level = tnPtr->level + 1 ;  // next lower level
         tnPtr->nextLevel[i].prevLevel = tnPtr ;         // point to parent
         tnPtr->nextLevel[i].nextLevel = NULL ;          // no recursion to lower level
      }
      // Programmer's Note: tnPtr->dirFiles is initialized inside the loop below.
   }
   if ( (tnPtr->tnfcount = nodecnt.regCnt) > ZERO )
   {
      tnPtr->tnFiles = this->Allocate_tnFNames ( tnPtr->tnfcount ) ;
      tnPtr->tnFCount = tnPtr->tnfcount ; // user-visible file count
      tnPtr->allocBytes += sizeof(tnFName) * tnPtr->tnfcount ;
   }

   #if DEBUG_PFS != 0
   if ( ofs.is_open() )
   {
      gsdbg.compose( "nlnodes:%u tnfcount:%u", &tnPtr->nlnodes, &tnPtr->tnfcount ) ;
      ofs << gsdbg << endl ;
   }
   #endif   // DEBUG_PFS

   //* Open the directory *
   DIR*     dirPtr ;
   if ( ((tnPtr->nlnodes > ZERO) || (tnPtr->tnfcount > ZERO)) &&
        ((dirPtr = this->Safe_opendir ( nodecnt.trgPath )) != NULL) )
   {
      #if DEBUG_PFS != 0
      if ( ofs.is_open() )
      {
         ofs << "dir file contents\n-----------------" << endl ;
      }
      #endif   // DEBUG_PFS

      gString   fnPath ;         // filespec for lstat() call
      FileStats rawStats ;       // data from call to lstat()
      fmFType   fileType ;       // decoded filetype
      UINT fnIndex = ZERO, tnIndex = ZERO ;
      bool      numName ;        // set if target filename is ASCII numeric
      bool      badStat ;        // set if lstat fails
      deStats   *destat ;        // pointer to entry read from directory file
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         badStat = false ;       // hope for the best
         numName = true ;        // assume ASCII-numeric filename

         //* DO NOT include 'current dir' and 'parent dir' names but DO        *
         //* include 'hidden' files, i.e. filenames whose first character is   *
         //* a '.' (PERIOD).                                                   *
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
            { continue ; }   // step over this record
         }

         //* Test for ASCII-numeric filename.*
         for ( short i = ZERO ; destat->d_name[i] != NULLCHAR ; ++i )
         {
            if ( (destat->d_name[i] > '9') || (destat->d_name[i] < '0') )
            {
               //* Test for filename: 'kcore'. The file size is incorrect       *
               //* (see note above), so process it as if it were a numeric name.*
               if ( (strncmp ( destat->d_name, kCore, 6 )) == ZERO )
                  break ;
               numName = false ;
               break ;
            }
         }

         //* If filename is not ASCII-numeric *
         if ( ! numName)
         {
            //* 'lstat' the file *
            fnPath.compose( "%s/%s", nodecnt.trgPath, destat->d_name ) ;

            #if DEBUG_PFS != 0
            if ( ofs.is_open() )
            {
               gsdbg.compose( "  '%s'", destat->d_name ) ;
               gsdbg.padCols( 17, L'-' ) ;
               gsdbg.append( L' ' ) ;
               ofs << gsdbg.ustr() ; ofs.flush() ;
            }
            #endif   // DEBUG_PFS

            if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
            {
               //* Add size of file to running total and increment file count *
               tnPtr->totBytes += rawStats.st_size ;
               ++tnPtr->totFiles ;

               //* Find out what kind of file it is *
               fileType = this->DecodeFileType ( rawStats.st_mode ) ;

               //* Format and store the data *
               switch ( fileType )
               {
                  case fmDIR_TYPE:
                     //* Store data for directory filename *
                     tnPtr->nextLevel[tnIndex].dirStat.rawStats = rawStats ;
                     this->strnCpy ( tnPtr->nextLevel[tnIndex].dirStat.fName, 
                                     destat->d_name, MAX_FNAME ) ;
                     this->DecodeEpochTime ( (int64_t)rawStats.st_mtime, 
                                 tnPtr->nextLevel[tnIndex].dirStat.modTime ) ;
                     tnPtr->nextLevel[tnIndex].dirStat.fType    = fileType ;
                     tnPtr->nextLevel[tnIndex].dirStat.fBytes   = rawStats.st_size ;
                     tnPtr->nextLevel[tnIndex].dirStat.readAcc  = 
                        this->ReadPermission ( tnPtr->nextLevel[tnIndex].dirStat, 
                                               fnPath.ustr() ) ;
                     tnPtr->nextLevel[tnIndex].dirStat.writeAcc =
                        this->WritePermission ( tnPtr->nextLevel[tnIndex].dirStat, 
                                                fnPath.ustr() ) ;
                     tnPtr->nextLevel[tnIndex].nextLevel = NULL ; // (be safe)
                     if ( ! tnPtr->nextLevel[tnIndex].dirStat.writeAcc )
                        ++tnPtr->wprotFiles ;
                     ++tnPtr->dirFiles ;
                     tnPtr->dirBytes += rawStats.st_size ;

                     #if DEBUG_PFS != 0
                     if ( ofs.is_open() )
                     {
                        wchar_t r = tnPtr->nextLevel[tnIndex].dirStat.readAcc  ? L'r' : L'-',
                                w = tnPtr->nextLevel[tnIndex].dirStat.writeAcc ? L'w' : L'-' ;
                        gsdbg.compose( "lstat: fType:%hd fBytes:%-4llu acc:%C%C",
                                       &tnPtr->nextLevel[tnIndex].dirStat.fType,
                                       &tnPtr->nextLevel[tnIndex].dirStat.fBytes,
                                       &r, &w
                                     ) ;
                        ofs << gsdbg.ustr() << endl ;
                     }
                     #endif   // DEBUG_PFS

                     ++tnIndex ;
                     break ;
                  case fmREG_TYPE:
                     ++tnPtr->regFiles ;
                     tnPtr->regBytes += rawStats.st_size ;
                     break ;
                  case fmLINK_TYPE:
                     ++tnPtr->linkFiles ;
                     tnPtr->linkBytes += rawStats.st_size ;
                     break ;
                  case fmCHDEV_TYPE:
                     ++tnPtr->cdevFiles ;
                     tnPtr->cdevBytes += rawStats.st_size ;
                     break ;
                  case fmBKDEV_TYPE:
                     ++tnPtr->bdevFiles ;
                     tnPtr->bdevBytes += rawStats.st_size ;
                     break ;
                  case fmFIFO_TYPE:
                     ++tnPtr->fifoFiles ;
                     tnPtr->fifoBytes += rawStats.st_size ;
                     break ;
                  case fmSOCK_TYPE:
                     ++tnPtr->sockFiles ;
                     tnPtr->sockBytes += rawStats.st_size ;
                     break ;
                  case fmUNKNOWN_TYPE:
                     ++tnPtr->unkFiles ;
                     tnPtr->unkBytes += rawStats.st_size ;
                     break ;
                  case fmTYPES:
                  default:
                     // unlikely
                     break ;
               }
               //* Store data for non-directory filenames *
               if ( fileType != fmDIR_TYPE )
               {
                  tnPtr->tnFiles[fnIndex].rawStats = rawStats ;
                  this->strnCpy ( tnPtr->tnFiles[fnIndex].fName, destat->d_name, MAX_FNAME ) ;
                  this->DecodeEpochTime ( (int64_t)rawStats.st_mtime, 
                                             tnPtr->tnFiles[fnIndex].modTime ) ;
                  tnPtr->tnFiles[fnIndex].fType    = fileType ;
                  tnPtr->tnFiles[fnIndex].fBytes   = rawStats.st_size ;
                  tnPtr->tnFiles[fnIndex].readAcc  = 
                     this->ReadPermission ( tnPtr->tnFiles[fnIndex], fnPath.ustr() ) ;
                  tnPtr->tnFiles[fnIndex].writeAcc = 
                     this->WritePermission ( tnPtr->tnFiles[fnIndex], fnPath.ustr() ) ;
                  if ( ! tnPtr->tnFiles[fnIndex].writeAcc )
                     ++tnPtr->wprotFiles ;

                  #if DEBUG_PFS != 0
                  if ( ofs.is_open() )
                  {
                     wchar_t r = tnPtr->tnFiles[fnIndex].readAcc  ? L'r' : L'-',
                             w = tnPtr->tnFiles[fnIndex].writeAcc ? L'w' : L'-' ;
                     gsdbg.compose( "lstat: fType:%hd fBytes:%-4llu acc:%C%C",
                                    &tnPtr->tnFiles[fnIndex].fType,
                                    &tnPtr->tnFiles[fnIndex].fBytes,
                                    &r, &w
                                  ) ;
                     ofs << gsdbg.ustr() << endl ;
                  }
                  #endif   // DEBUG_PFS

                  ++fnIndex ;
               }
            }
            else     // 'lstat' system call returned an error condition
            {        // We can't access this file
               // Programmer's Note: If 'lstat' fails, report the information from 
               // the directory entry even though the file probably no longer exists.
               // (see below). 'nodeCount' will be too high if we don't count the 
               // failed lstats.
               badStat = true ;
               ++tnPtr->violations ;
               this->recentErrno = errno ;
            }
         }

         //* If an ASCII-numeric filename or 'kcore' directory OR *
         //* lstat error, do not stat the file. (see notes above) *
         if ( numName || badStat )
         {
            if ( destat->d_type == DT_DIR )
            {
               this->strnCpy ( tnPtr->nextLevel[tnIndex].dirStat.fName, 
                               destat->d_name, MAX_FNAME ) ;
               tnPtr->nextLevel[tnIndex].dirStat.fType    = fmDIR_TYPE ;
               tnPtr->nextLevel[tnIndex].dirStat.fBytes   = ZERO ;
               tnPtr->nextLevel[tnIndex].dirStat.readAcc  = 
               tnPtr->nextLevel[tnIndex].dirStat.writeAcc = false ;
               tnPtr->nextLevel[tnIndex].nextLevel = NULL ; // (be safe)
               ++tnPtr->dirFiles ;

               #if DEBUG_PFS != 0
               if ( ofs.is_open() )
               {
                  gsdbg.compose( "  '%s' ASCII-numeric: fType:%hd fBytes:%-4llu acc:--", 
                                 destat->d_name, 
                                 &tnPtr->nextLevel[tnIndex].dirStat.fType,
                                 &tnPtr->nextLevel[tnIndex].dirStat.fBytes ) ;
                  ofs << gsdbg.ustr() << endl ;
               }
               #endif   // DEBUG_PFS

               ++tnIndex ;
            }
            else
            {
               this->strnCpy ( tnPtr->tnFiles[fnIndex].fName, destat->d_name, MAX_FNAME ) ;
               tnPtr->tnFiles[fnIndex].fType =
                  destat->d_type == DT_LNK ? fmLINK_TYPE : 
                                    DT_REG ? fmREG_TYPE : fmUNKNOWN_TYPE ;
               if ( tnPtr->tnFiles[fnIndex].fType == fmLINK_TYPE )
                  ++tnPtr->linkFiles ;
               else if ( tnPtr->tnFiles[fnIndex].fType == fmREG_TYPE )
                  ++tnPtr->regFiles ;
               else
                  ++tnPtr->unkFiles ;
               tnPtr->tnFiles[fnIndex].fBytes   = ZERO ;
               tnPtr->tnFiles[fnIndex].readAcc  = 
               tnPtr->tnFiles[fnIndex].writeAcc = false ;

               #if DEBUG_PFS != 0
               if ( ofs.is_open() )
               {
                  gsdbg.compose( "  '%s' ASCII-numeric: fType:%hd fBytes:%-4llu acc:--", 
                                 tnPtr->tnFiles[fnIndex].fName, 
                                 &tnPtr->tnFiles[fnIndex].fType,
                                 &tnPtr->tnFiles[fnIndex].fBytes ) ;
                  ofs << gsdbg.ustr() << endl ;
               }
               #endif   // DEBUG_PFS

               ++fnIndex ;
            }
            //* Increment total file count.                               *
            //* Note that size of target file is unknown (probably zero), *
            //* so 'totBytes' is not updated for this file.               *
            ++tnPtr->totFiles ;
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }     // non-empty directory

   //* If directory access error *
   else if ( (tnPtr->nlnodes > ZERO) || (tnPtr->tnfcount > ZERO) )
      ++tnPtr->violations ;

   #if DEBUG_PFS != 0
   if ( ofs.is_open() )
   {
      gsdbg.compose( "totFiles:%-2u  dirFiles:%-2u  regFiles:%-2u  linkFiles:%-2u\n"
                     "cdevFiles:%-2u bdevFiles:%-2u fifoFiles:%-2u sockFiles:%-2u unkFiles:%-2u\n"
                     "-------\nend log",
                     &tnPtr->totFiles, &tnPtr->dirFiles, &tnPtr->regFiles, &tnPtr->linkFiles, 
                     &tnPtr->cdevFiles, &tnPtr->bdevFiles, &tnPtr->fifoFiles, &tnPtr->sockFiles, 
                     &tnPtr->unkFiles
                   ) ;
      ofs << gsdbg << endl ;
      ofs.close() ;
   }
   #endif   // DEBUG_PFS

   return tnPtr->totFiles ;

   #undef DEBUG_PFS
}  //* End procFlatscan() *

//*************************
//*    ReleaseDirTree     *
//*************************
//********************************************************************************
//* Release the directory tree information allocated by ScanDirTree().           *
//* Releases the entire allocation attached to this->deHead.                     *
//*         PRIVATE METHOD - NON-MEMBERS DON'T EVEN KNOW deHead EXISTS!!         *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing,    (this->deHead is set to NULL indicating 'no data')      *
//********************************************************************************

void FMgr::ReleaseDirTree ( void )
{
   if ( this->deHead != NULL )
   {
      //* Recursively release everything in the tree except the anchor node *
      this->ReleaseDirTree ( this->deHead ) ;

      this->Release_TreeNodes ( this->deHead, 1 ) ;
   }
}  //* End ReleaseDirTree() *

//*************************
//*    ReleaseDirTree     *
//*************************
//********************************************************************************
//* Release the directory tree information list.                                 *
//* NOTE: This is a complex data allocation that cannot be released using        *
//* 'delete' [] (TreeNode*)xyz;  Prevent memory leaks, play by the rules.        *
//*                                                                              *
//* Recursively walk to the bottom of the allocation tree, then release the      *
//* tree starting with the bottom branch and working back toward the top.        *
//*                                                                              *
//* NOTE: The TreeNode object referenced by caller's pointer IS NOT deleted      *
//* because it may be a member of an array; however, all lower-level nodes ARE   *
//* deleted and the 'nextLevel' pointer which controlled them is set to NULL.    *
//*                                                                              *
//* Input  : pointer to TreeNode object                                          *
//*           Note that each object may have a list of objects attached to it.   *
//*                                                                              *
//* Returns: nothing,                                                            *
//********************************************************************************

void FMgr::ReleaseDirTree ( TreeNode* tn )
{
   if ( tn != NULL && tn->nextLevel != NULL )
   {
      for ( UINT nIndex = ZERO ; nIndex < tn->nlnodes ; nIndex++ )
      {
         this->ReleaseDirTree ( &tn->nextLevel[nIndex] ) ;
      }
      // everything attached to the nextLevel node array has been released
      //* Delete the 'nextLevel' array of TreeNode objects *
      this->Release_TreeNodes ( tn->nextLevel, tn->nlnodes ) ;
      tn->nlnodes = ZERO ;
   }

}  //* End ReleaseDirTree() *

//*************************
//*   DirectorySummary    *
//*************************
//********************************************************************************
//* Get a summary of the contents of the specified directory.                    *
//* Summary contains: the number of files and subdirectories in the              *
//* directory and the combined size (in bytes) of all files and                  *
//* subdirectories in the specified top-level directory (but see 'recurse'       *
//* parameter, below.)                                                           *
//*                                                                              *
//* Input  : dPath : full path specification of directory to be scanned          *
//*          fCount (by reference): initial value ignored                        *
//*          totBytes (by reference): initial value ignored                      *
//*          recurse (optional, 'false' by default)                              *
//*            if 'true' read specified directory and contents of all            *
//*            subdirectories it contains                                        *
//*                                                                              *
//* Returns: 'true' if directory read successfully                               *
//*                 fCount and totBytes are initialized                          *
//*          'false' if directory does not exist or no read access or path       *
//*                 links not resolved. fCount and totBytes == ZERO              *
//********************************************************************************
//* Programmer's Note: This method is primarily for sizing a recursive           *
//* copy/cut/paste operation, but it may have other uses.                        *
//*                                                                              *
//* Note that 'fCount' does not include the target directory itself,             *
//* and that 'totBytes' does not include the size of target directory file.      *
//********************************************************************************

bool FMgr::DirectorySummary ( const char* dPath, UINT& fCount, 
                              UINT64& totBytes, bool recurse )
{
DIR*     dirPtr ;
static USHORT recursionLevel = ZERO ;
bool     showFile,
         status = true ;

   //* Initialize caller's variables *
   if ( recursionLevel == ZERO )
   {
      fCount   = ZERO ;
      totBytes = ZERO ;
   }
   ++recursionLevel ;

   //* Open the directory *
   if ( (dirPtr = this->Safe_opendir ( dPath )) != NULL )
   {
      gString     fnPath ;
      FileStats   rawStats ;
      fmFType     fileType ;
      deStats   *destat ;        // pointer to entry read from directory file
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Determine whether we should include hidden files in our total     *
         //* (do not include 'current dir' and 'parent dir' names)             *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   (destat->d_name[1] == NULLCHAR) 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR)
                || (this->showHidden == false) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            //* Do an 'lstat' on the specified file for file size and type *
            this->CatPathFname ( fnPath, dPath, destat->d_name ) ;
            if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
            {
               totBytes += (UINT64)rawStats.st_size ; // update byte count
               ++fCount ;                             // and file count
               fileType = this->DecodeFileType ( rawStats.st_mode ) ;
               if ( recurse != false && fileType == fmDIR_TYPE )
               {
                  this->DirectorySummary ( fnPath.ustr(), fCount, totBytes, true ) ;
               }
            }
            else
            {
               this->recentErrno = errno ;
               this->DebugLog ( "In DirectorySummary: lstat() failed", 
                                this->recentErrno, dPath ) ;
            }
         }
      }  // while()
      closedir ( dirPtr ) ;      // Close the directory
   }
   else
   {
      status = false ;
   }
   --recursionLevel ;
   return status ;

}  //* End DirectorySummary() *

//*************************
//*    TreeNodeSummary    *
//*************************
//********************************************************************************
//* Walk through the previously-captured data for the specified TreeNode, whose  *
//* base directory is indicated by tnfPtr. Collect file count and file size      *
//* information for the entire node list, INCLUDING the directory itself.        *
//* Used to count 'selected' files.                                              *
//*                                                                              *
//* Input  : tnfPtr   : pointer to tnFName object describing base directory      *
//*          fileCount: (by reference): initial value ignored                    *
//*                     on return, contains number of files in the node          *
//*          totBytes : (by reference): initial value ignored                    *
//*                     on return, contains combined size of all files in node   *
//*                                                                              *
//* Returns: 'true' if count was successful                                      *
//*                 fileCount and totalBytes are initialized                     *
//*          'false' if specified tnFName object does not describe a directory   *
//*                 or directory data is not in displayed-file list              *
//*                 fileCount and totalBytes == ZERO                             *
//********************************************************************************

bool FMgr::TreeNodeSummary ( tnFName* tnfPtr, UINT& fileCount, UINT64& totalBytes )
{
   bool  status = false ;

   //* Initialize caller's variables *
   fileCount = ZERO ;
   totalBytes = ZERO ;

   //* Validate the index *
   if ( tnfPtr->fType == fmDIR_TYPE )
   {
      //* Locate the TreeNode object associated with this directory *
      TreeNode* rrd ;
      if ( (rrd = this->DirName2TreeNode ( tnfPtr, this->deHead )) != NULL )
      {
         //* Climb the tree root-to-leaf starting with this node *
         status = this->TreeNodeSummary ( rrd, fileCount, totalBytes ) ;
      }
   }

   return status ;

}  //* End TreeNodeSummary() *

//*************************
//*    TreeNodeSummary    *
//*************************
//********************************************************************************
//* Walk through the previously-captured data beginning at the specified         *
//* TreeNode object, and collect file count and file size information for        *
//* the entire node list.                                                        *
//*                                                                              *
//* Input  : tnPtr    : pointer to TreeNode object                               *
//*          fCount   : file-count accumulator (by reference)                    *
//*          tBytes   : file-size accumulator (by reference)                     *
//*                                                                              *
//* Returns: 'true'   (fCount and tBytes have been updated)                      *
//********************************************************************************

bool FMgr::TreeNodeSummary ( const TreeNode* tnPtr, UINT& fCount, UINT64& tBytes )
{
   bool  status = true ;

   //* Add the non-directory entries, plus the data for this directory name    *
   //* to the caller's total. (This avoids double-counting directory names.)   *
   fCount += tnPtr->totFiles - tnPtr->dirFiles + 1 ;
   tBytes += tnPtr->totBytes - tnPtr->dirBytes + tnPtr->dirStat.fBytes ;
   //* If there are subdirectories at a lower level, add them to the totals *
   if ( tnPtr->nextLevel != NULL )
   {
      for ( UINT nIndex = ZERO ; nIndex < tnPtr->nlnodes ; nIndex++ )
      {
         status = this->TreeNodeSummary ( &tnPtr->nextLevel[nIndex], fCount, tBytes ) ;
      }
   }
   return status ;

}  //* End TreeNodeSummary() *

//*************************
//*    TreeNodeSummary    *
//*************************
//********************************************************************************
//* Walk through the previously-captured data beginning at the specified         *
//* TreeNode object, and collect accumulated information for the entire          *
//* node list.                                                                   *
//*                                                                              *
//* The following data members of the accumulator node receive the totals.       *
//* (remaining data members are set to default values)                           *
//*                                                                              *
//* totFiles   totBytes  |  cdevFiles  cdevBytes  |  unkFiles    unkBytes        *
//* dirFiles   dirBytes  |  bdevFiles  bdevBytes  |  wprotFiles                  *
//* regFiles   regBytes  |  fifoFiles  fifoBytes  |  violations                  *
//* linkFiles  linkBytes |  sockFiles  sockBytes  |  allocBytes                  *
//*                                               |  nlnodes                     *
//*                                               |  tnfcount                    *
//*                                                                              *
//* Input  : tnPtr    : pointer to TreeNode object                               *
//*          tnAcc    : TreeNode instance to act as data accumulator             *
//*                                                                              *
//* Returns: 'true'   (tnAcc accumulator members have been updated)              *
//********************************************************************************

bool FMgr::TreeNodeSummary ( const TreeNode* tnPtr, TreeNode& tnAcc )
{
   tnAcc.ReInit() ;           // clear the accumulators
   tnAcc.allocBytes = ZERO ;  // don't count the size of the counter

   //* Include current directory name stats *
   ++tnAcc.totFiles ;
   tnAcc.totBytes += tnPtr->dirStat.fBytes ;
   ++tnAcc.dirFiles ;
   tnAcc.dirBytes += tnPtr->dirStat.fBytes ;
   ++tnAcc.nlnodes ;

   //* Sum the tree *
   this->TNSum ( tnPtr, tnAcc ) ;

   return true ;

}  //* End TreeNodeSummary() *

void FMgr::TNSum ( const TreeNode* tnPtr, TreeNode& tnAcc )
{
   //* Add data from current node *
   tnAcc += *tnPtr ;

   //* If there are subdirectories at a lower level, add them to the totals *
   if ( tnPtr->nlnodes > ZERO && tnPtr->nextLevel != NULL )
   {
      for ( UINT nIndex = ZERO ; nIndex < tnPtr->nlnodes ; nIndex++ )
      {
         this->TNSum ( &tnPtr->nextLevel[nIndex], tnAcc ) ;
      }
   }

}  //* End TNSum() *

//*************************
//*     CopyDirTree       *
//*************************
//********************************************************************************
//* Create a copy of the TreeNode list beginning with the base directory which   *
//* is indicated by tnfPtr.                                                      *
//*                                                                              *
//* Input  : tnfPtr : pointer to tnFName object describing base directory        *
//*          tnBase : TreeNode object (by reference) which will act as the       *
//*                   base node for the new list.                                *
//*                                                                              *
//* Returns: 'true' if count was successful                                      *
//*                 fileCount and totalBytes are initialized                     *
//*          'false' if specified tnFName object does not describe a directory   *
//*                 or directory data is not in displayed-file list or memory-   *
//*                 allocation error.                                            *
//********************************************************************************

bool FMgr::CopyDirTree ( tnFName* tnfPtr, TreeNode& tnBase )
{
   bool  status = false ;

   //* Initialize caller's base-node object *
   tnBase.ReInit () ;

   //* Validate the index *
   if ( tnfPtr->fType == fmDIR_TYPE )
   {
      //* Locate the TreeNode object associated with this directory *
      TreeNode* rrd ;
      if ( (rrd = this->DirName2TreeNode ( tnfPtr, this->deHead )) != NULL )
      {
         //* Climb the tree root-to-leaf starting with this node *
         status = this->CopyDirTree ( rrd, tnBase ) ;
      }
   }
   return status ;

}  //* End CopyDirTree() *

//*************************
//*     CopyDirTree       *
//*************************
//********************************************************************************
//* Create a copy of the specified TreeNode object and all lower-level nodes     *
//* and file arrays attached to it.                                              *
//*                                                                              *
//* Input  : tnSrc : pointer to source TreeNode object                           *
//*          tnTarg: uninitialized instance (by reference) to receive data       *
//*          nondir: (optional, 'true' by default)                               *
//*                  'true' : copy all directory AND non-directory records       *
//*                  'false': copy ONLY directory records                        *
//*                                                                              *
//* Returns: 'true' (TreeNode list has been copied)                              *
//********************************************************************************

bool FMgr::CopyDirTree ( const TreeNode* tnSrc, TreeNode& tnTarg, bool nondir )
{
   //* Copy the source node to target *
   tnTarg = *tnSrc ;

   //* If non-directory data is not required *
   if ( nondir == false )
   {
      tnTarg.totFiles = tnTarg.dirFiles ;
      tnTarg.totBytes = tnTarg.dirBytes ;
      tnTarg.tnFiles = NULL ;
      tnTarg.tnfcount  = tnTarg.tnFCount  = 
      tnTarg.regFiles  = tnTarg.linkFiles =
      tnTarg.cdevFiles = tnTarg.bdevFiles = 
      tnTarg.fifoFiles = tnTarg.sockFiles = tnTarg.unkFiles = ZERO ;

      tnTarg.regBytes  = tnTarg.linkBytes =
      tnTarg.cdevBytes = tnTarg.bdevBytes = 
      tnTarg.fifoBytes = tnTarg.sockBytes = tnTarg.unkBytes = ZERO ;
   }
   
   //* If there are lower-level nodes attached to this node *
   if ( tnSrc->nlnodes > ZERO && tnSrc->nextLevel != NULL )
   {
      //* Clone the next-level node array *
      tnTarg.nextLevel = this->Allocate_TreeNodes ( tnSrc->nlnodes ) ;

      for ( UINT i = ZERO ; i < tnSrc->nlnodes ; i++ )
      {
         this->CopyDirTree ( &tnSrc->nextLevel[i], tnTarg.nextLevel[i] ) ;
      }
   }

   //* If there are non-directory filenames attached *
   //* to this node, make a copy of them also.       *
   if ( nondir != false )
   {
      if ( tnSrc->tnfcount > ZERO && tnSrc->tnFiles != NULL )
      {
         tnTarg.tnFiles = this->Allocate_tnFNames ( tnSrc->tnfcount ) ;

         for ( UINT i = ZERO ; i < tnSrc->tnfcount ; i++ )
            tnTarg.tnFiles[i] = tnSrc->tnFiles[i] ;
      }
   }
   else
   {  //* Only the tree structure is needed.   *
      //* Non-directory files are not required.*
      tnTarg.tnFiles = NULL ;
      tnTarg.tnfcount = tnTarg.tnFCount = ZERO ;
   }

   return true ;

}  //* End CopyDirTree() *

//*************************
//*      GetBaseNode      *
//*************************
//******************************************************************************
//* Get a pointer to the base node of the captured directory tree.             *
//* Allows for filename search of the directory tree. No data may be modified. *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: const pointer to base node of TreeNode array                      *
//******************************************************************************
//* Programmer's Note: This is a semi-dangerous method in that a caller        *
//* CAN, but MUST NOT modify the private data.                                 *
//******************************************************************************

const TreeNode* FMgr::GetBaseNode ( void )
{

   return this->deHead ;

}  //* End GetBaseNode() *

//*************************
//*   DirName2TreeNode    *
//*************************
//******************************************************************************
//* Scan the TreeNode list of displayed files and return a pointer to the      *
//* TreeNode object to which the specified directory name belongs.             *
//*                                                                            *
//* Input  : pointer to tnFName object which describes a directory name in     *
//*           the list of displayed files.                                     *
//*                                                                            *
//* Returns: pointer to the TreeNode object associated with this directory.    *
//*           Returns a NULL pointer if the specified file is not a            *
//*           directory or if the tnFName object is not a member of the        *
//*           displayed-files list.                                            *
//******************************************************************************
//* Programmer's Note: This is a semi-dangerous method in that a caller        *
//* CAN, but MUST NOT modify the private data.                                 *
//******************************************************************************

const TreeNode* FMgr::DirName2TreeNode ( tnFName* dirName )
{
   TreeNode*   nodePtr = NULL ;     // return value

   if ( dirName->fType == fmDIR_TYPE )
      nodePtr = this->DirName2TreeNode ( dirName, this->deHead ) ;
   return nodePtr ;

}  //* End DirName2TreeNode() *

//*************************
//*   DirName2TreeNode    *
//*************************
//******************************************************************************
//* PRIVATE METHOD - SEE PUBLIC METHOD ABOVE                                   *
//* Locate the TreeNode object which contains the specified tnFName object.    *
//*                                                                            *
//* Input  : dirName: pointer to a tnFName object which describes a directory  *
//*                   name                                                     *
//*          topNode: pointer to position in the TreeNode list at which to     *
//*                   begin the search                                         *
//*                                                                            *
//* Returns: pointer to the corresponding TreeNode object                      *
//*          (NULL if TreeNode object not found)                               *
//******************************************************************************

TreeNode* FMgr::DirName2TreeNode ( tnFName* dirName, TreeNode* topNode )
{
   //* Compare the pointers. If we have a match, we're done.*
   //* Else, step to next level.                            *
   if ( &topNode->dirStat != dirName )
   {
      if ( topNode->nextLevel != NULL )
      {
         TreeNode* targNode ;
         for ( UINT nIndex = ZERO ; nIndex < topNode->nlnodes ; nIndex++ )
         {
            targNode = this->DirName2TreeNode ( dirName, &topNode->nextLevel[nIndex] ) ;
            if ( targNode != NULL )
               break ;
         }
         topNode = targNode ;
      }
      else
         topNode = NULL ;
   }
   return topNode ;

}  //* End DirName2TreeNode() *

//*************************
//*    DirectoryCount     *
//*************************
//********************************************************************************
//* Returns the number of subdirectory names in the specified directory.         *
//* Also initializes caller's non-directory file count.                          *
//* (Totals do not include current "." or parent ".." directory shortcut names)  *
//*                                                                              *
//* Input  : nodecnt : (by reference - initial counts ignored)                   *
//*                    'trgPath' contains full path specification of directory   *
//*                              to be scanned                                   *
//*                    'dirCnt' receives count of directory names                *
//*                    'regCnt' receives count of non-directory names            *
//*                    'level' member is not referenced                          *
//*                                                                              *
//* Returns: count of directory names found                                      *
//********************************************************************************
//* Notes                                                                        *
//* 1) Docs say that 'errno' S/B reset to zero before each call to readdir64 in  *
//*    order to distinguish between end-of-file and actual error as indicated by *
//*    the return of a null pointer. Because we end the scan on the first null   *
//*    pointer received, this is unnecessary.                                    *
//*    Note however, that if an error condition, an incorrect count (undercount) *
//*    may be returned.                                                          *
//* 2) It should not be necessary to stat each file in the directory to determine*
//*    the filetype. This info S/B available in the directory entry itself       *
//*    'destat.d_type' for _MOST_ filesystems. However, "unsigned char d_type"   *
//*    is not available for all filesystems, and its presence or absense is      *
//*    determined by the macro _DIRENT_HAVE_D_TYPE.                              *
//*    For this reason, it is safer to perform the stat for each file to         *
//*    determine its file type.                                                  *
//********************************************************************************

UINT FMgr::DirectoryCount ( nodeCount& nodecnt )
{
   nodecnt.dirCnt = nodecnt.regCnt = ZERO ;  // initialize accumulators

   DIR*     dirPtr ;                         // access to directory contents
   if ( (dirPtr = this->Safe_opendir ( nodecnt.trgPath )) != NULL )
   {
      gString   lsPath ;         // path specification for 'lstat' call
      FileStats rawStats ;       // 'lstat' data
      bool      showFile ;       // indicates whether file is included in count

      deStats   *destat ;        // pointer to entry read from directory file
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* (do not include 'current dir' and 'parent dir' names) *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   (destat->d_name[1] == NULLCHAR) 
                || ((destat->d_name[1] == PERIOD) && (destat->d_name[2] == NULLCHAR)) )
               showFile = false ;
         }

         //* Target file is to be included in the count *
         if ( showFile != false )
         {
            //* 'lstat' the file *
            this->CatPathFname ( lsPath, nodecnt.trgPath, destat->d_name ) ;
            if ( (lstat64 ( lsPath.ustr(), &rawStats )) == OK )
            {
               //* If file is of fmDIR_TYPE, increment our counter *
               //* else, non-directory file type.                  *
               if ( (this->DecodeFileType ( rawStats.st_mode )) == fmDIR_TYPE )
                  ++nodecnt.dirCnt ;
               else
                  ++nodecnt.regCnt ;
            }
            else
            {
               this->recentErrno = errno ;
            }
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }

   return nodecnt.dirCnt ;

}  //* End DirectoryCount() *

//*************************
//*     ScanFromRoot      *
//*************************
//********************************************************************************
//* Allow/disallow scan of full directory tree when CWD is the root directory.   *
//*                                                                              *
//* Input  : enable     : if 'true',  recursive tree scan from root is enabled   *
//*                       if 'false', recursive tree scan from root is disabled  *
//*          reportOnly : (optional, 'false' by default)                         *
//*                       if 'false', internal flag is set/reset as indicated    *
//*                                   by 'enable' parameter                      *
//*                       if 'true',  state of internal flag is is not           *
//*                                   modified, only reported                    *
//*                                                                              *
//* Returns: current state of recursive-scan flag                                *
//********************************************************************************

bool FMgr::ScanFromRoot ( bool enable, bool reportOnly )
{

   if ( ! reportOnly )
      this->irRootscan = enable ;
   
   return this->irRootscan ;

}  //* End ScanFromRoot() *

#if DEBUG_MULTITHREAD != 0 && DMT_TREE != 0
//*************************
//*  dmtTreeNodeSummary   *
//*************************
//******************************************************************************
//* For Debugging Only - non-member method                                     *
//* ------------------                                                         *
//*                                                                            *
//* Input  : tnPtr  : pointer to base TreeNode object for TreeNode list        *
//*          tnAcc  : (by reference, initial contents discarded)               *
//*                   accumulator node, receives accumulated totals            *
//*                    1) total files                (tnAcc.totFiles)          *
//*                    2) total directory files      (tnAcc.dirFiles)          *
//*                    3) total non-directory files  (tnAcc.tnFCount)          *
//*                    4) total bytes, all files     (tnAcc.totBytes)          *
//*                    5) total directory bytes      (tnAcc.dirBytes)          *
//*                    6) total non-directory bytes  (tnAcc.regBytes)          *
//*                   other members not used                                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dmtTreeNodeSummary ( const TreeNode* tnPtr, TreeNode& tnAcc )
{
   tnAcc.ReInit() ;           // clear the accumulators

   //* Include current directory name stats *
   ++tnAcc.totFiles ;
   tnAcc.totBytes += tnPtr->dirStat.fBytes ;
   ++tnAcc.dirFiles ;
   tnAcc.dirBytes += tnPtr->dirStat.fBytes ;

   //* Sum the tree *
   dmtTNSum ( tnPtr, tnAcc ) ;

   // Programmer's Note: We aren't exactly sure why the total bytes for "/proc" 
   // directory is so outrageously incorrect. It is presumed that because user 
   // access is very limited for this directory, the stat will likely fail.
   // In any case, we compensate here. After all, it's only a debugging method.
   
   //* If base directory is root, find the byte-total value for "/proc" node.*
   //* Subtract it from the total reported to caller.                        *
   gString gs( tnPtr->dirStat.fName ) ;
   if ( (gs.compare( L"/" )) == ZERO )
   {
      //* Get actual node count *
      UINT nodes, fcount ;
      tnPtr->GetActual ( nodes, fcount ) ;
   
      UINT64 procBytes = ZERO ;
      for ( UINT i = ZERO ; i < nodes ; ++i )
      {
         gs = tnPtr->nextLevel[i].dirStat.fName ;
         if ( (gs.compare( L"proc" )) == ZERO )
         {
            procBytes = tnPtr->nextLevel[i].totBytes ;
            break ;
         }
      }
      tnAcc.totBytes -= procBytes ;
   }

}  //* End TreeNodeSummary() *

//*************************
//*       dmtTNSum        *
//*************************
//******************************************************************************
//* For Debugging Only - non-member method                                     *
//* ------------------                                                         *
//*                                                                            *
//* Input  : tnPtr: pointer to TreeNode object                                 *
//*          tnAcc  : (by reference)                                           *
//*                   accumulator node, receives accumulated totals            *
//*                    1) add total files                (tnAcc.totFiles)      *
//*                    2) add total directory files      (tnAcc.dirFiles)      *
//*                    3) add total non-directory files  (tnAcc.tnFCount)      *
//*                    4) add total file bytes           (tnAcc.totBytes)      *
//*                   other members not modified                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) The display method, RefreshCurrDir(), subtracts "hidden" files and      *
//*    subdirectories and their file sizes from the BaseNode totals to match   *
//*    the actual data displayed. While we have access to the true file counts,*
//*    the sizes of any BaseNode hidden files are lost to us at this point.    *
//*    Therefore, the 'totBytes' accumulated total reported may be smaller     *
//*    than the actual value.                                                  *
//*    "Hidden" data at lower levels in the tree are not affected by this.     *
//* 2) If ('tnFCount' != 'tnfcount') || ('dirFiles' != 'nlnodes')              *
//*    then the byte totals will be off.                                       *
//*                                                                            *
//*                                                                            *
//******************************************************************************

static void dmtTNSum ( const TreeNode* tnPtr, TreeNode& tnAcc )
{
   //* Get actual node count *
   UINT nodes, fcount ;
   tnPtr->GetActual ( nodes, fcount ) ;

   //* Add data from current node *
   tnAcc.totFiles += nodes + fcount ;
   tnAcc.dirFiles += nodes ;
   tnAcc.tnFCount += fcount ;
   tnAcc.totBytes += tnPtr->totBytes ;
   tnAcc.dirBytes += tnPtr->dirBytes ;
   tnAcc.regBytes += tnPtr->totBytes - tnPtr->dirBytes ;

   //* If there are subdirectories at a lower level, add them to the totals *
   if ( nodes > ZERO && tnPtr->nextLevel != NULL )
   {
      for ( UINT nIndex = ZERO ; nIndex < nodes ; nIndex++ )
      {
         dmtTNSum ( &tnPtr->nextLevel[nIndex], tnAcc ) ;
      }
   }

}  //* End dmtTNSum() *

//*************************
//*      dmtLogTree       *
//*************************
//******************************************************************************
//* For Debugging Only - non-member method                                     *
//* ------------------                                                         *
//* Write the formatted directory tree to the log file.                        *
//*                                                                            *
//* Input  : tn   : pointer to base TreeNode object for TreeNode list          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dmtLogTree ( TreeNode* tn )
{
   if ( dmtofs.is_open() )    // be sure log file is open
   {
      //* Get ACTUAL directory and file counts.*
      UINT nlnodes, tnfcount ;
      tn->GetActual ( nlnodes, tnfcount ) ;

      if ( tn->level == ZERO )   // report base node directory
      {
         dmtofs << "\nDisplay TreeNode Tree"
                   "\n---------------------------------" << endl ;
         dmtOut.compose( "%C%s ([%hd] %u, %u, %llu)", &wcsUL, 
                         tn->dirStat.fName, &tn->level, 
                         &nlnodes, &tnfcount, &tn->totBytes ) ;

         //* If there are hidden files in the base node, the *
         //* byte total will be inaccurate. So indicate.     *
         if ( (tn->dirFiles != nlnodes) || (tn->tnFCount != tnfcount) )
         {
            dmtOut.insert( L'*', (dmtOut.findlast( L')' )) ) ;
         }
         dmtofs << dmtOut.ustr() << endl ;
      }

      //* Recursively report each branch of the tree.*
      for ( UINT tni = ZERO ; tni < nlnodes ; ++tni )
      {
         dmtLogBranch ( &tn->nextLevel[tni] ) ;
      }

      if ( tn->level == ZERO )   // report base node non-dir files
      {
         dmtOut.compose( "%C", &wcsVERT ) ;
         dmtofs << dmtOut.ustr() << endl ;
         for ( UINT tfi = ZERO ; tfi < tnfcount ; ++tfi )
         {
            dmtOut.compose( "%C%C%s", &wcsLTEE, &wcsHDASH4s, 
                            tn->tnFiles[tfi].fName ) ;
            dmtofs << dmtOut.ustr() << endl ;
         }
      }
   }     // log file is open

}  //* End dmtLogTree() *

//*************************
//*     dmtLogBranch      *
//*************************
//******************************************************************************
//* For Debugging Only - non-member method                                     *
//* ------------------                                                         *
//* Write the formatted directory tree to the log file.                        *
//*                                                                            *
//*                                                                            *
//* Input  : tn   : pointer to TreeNode branch                                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dmtLogBranch ( TreeNode* tn )
{
   const wchar_t *leftBar = L"\x02502 " ;
   UINT nlnodes, tnfcount ;
   tn->GetActual ( nlnodes, tnfcount ) ;
   dmtOut.compose ( "%C%C%s ([%hd] %u, %u, %llu)", &wcsLTEE, &wcsHORIZ, 
                    tn->dirStat.fName,
                    &tn->level,
                    &nlnodes,
                    &tnfcount,
                    &tn->totBytes ) ;
   if ( tn->level > 1 )
      dmtOut.insert( leftBar ) ;
   for ( short i = 2 ; i < tn->level ; ++i )
      dmtOut.insert( leftBar ) ;
   if ( tn->dirStat.fsMatch == false )
      dmtOut.append( " (extfs)" ) ;
   dmtofs << dmtOut.ustr() << endl ;

   //* If data a lower levels, recurse *
   if ( (nlnodes > ZERO) && (tn->nextLevel != NULL) )
   {
      //* For all directories at this level *
      for ( UINT txi = ZERO ; txi < nlnodes ; ++txi )
         dmtLogBranch ( &tn->nextLevel[txi] ) ;
   }

   //* For all non-directory files at this level *
   for ( UINT tfi = ZERO ; tfi < tnfcount ; ++tfi )
   {
      dmtOut.compose ( "%C %C%C%s", &wcsVERT, &wcsLTEE, &wcsHDASH4s, 
                       tn->tnFiles[tfi].fName ) ;
      for ( short i = 1 ; i < tn->level ; ++i )
         dmtOut.insert( leftBar ) ;
      dmtofs << dmtOut.ustr() << endl ;
   }

}  //* End dmtLogBranch() *
#endif   // DEBUG_MULTITHREAD && DMT_TREE

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

