//********************************************************************************
//* File       : FMgrStats.cpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 02-Apr-2025                                                     *
//* Version    : (see FMgrVersion string)                                        *
//*                                                                              *
//* Description: This is a support module for the FMgr class.                    *
//* This module contains the methods specific to the 'stat' and 'lstat'          *
//* functionality which reports full statistical information on a file, and      *
//* optionally, allows user to modify the file's stats.                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* Some functionality in this module is constructed from a class project in     *
//* CS3560, Systems Programming; however, the data storage used in that project  *
//* has been superceeded by the creation of the tnFName class and its companion  *
//* the TreeNode class.                                                          *
//*                                                                              *
//* Programmer's Notes:                                                          *
//* Because the selinux library or headers may be missing or restricted on       *
//* some systems, we have implemented a workaround in the FilesystemStats()      *
//* method which doesn't need the #include <selinux/selinux.h> include file.     *
//********************************************************************************

#include "GlobalDef.hpp"
#ifndef FMGR_INCLUDED
#include "FMgr.hpp"
#endif

//* Needed for working with socket files *
#include <stddef.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
//* Needed for the utime() function *
#include <utime.h>

//** Local Definitions **

//** Data **

//* Local prototypes *


//*************************
//*     GetFileStats      *
//*************************
//******************************************************************************
//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//* The specified file may, or may not be in the current file list.            *
//* By default the stats for the specified file are returned, but if           *
//* getTarg != false && target is a symbolic link, the symbolic link is        *
//* followed i.e. stats for link target are returned.                          *
//*                                                                            *
//* Input  : tnFile : tnFName class object (by reference) to hold 'stat' data  *
//*          fPath  : full path/filename specification                         *
//*          getTarg: (optional, false by default) if true, symbolic links are *
//*                   followed. Else, do not follow links                      *
//*                                                                            *
//* Returns: OK if successful, all data members of tnFile will be initialized  *
//*          Else returns system 'errno' value                                 *
//*           If error during system call:                                     *
//*            - tnFile.readAcc == false and all other members are undefined   *
//*             Exception:                                                     *
//*             If 'getTarg' != false AND error is a broken symbolic link      *
//*             - tnFile.fName == name of link target if known, else "unknown" *
//*             - tnFile.readAcc == false                                      *
//*             - tnFile.fType == fmTYPES                                      *
//*             - All other members of 'tnFile' are undefined                  *
//******************************************************************************
//* Note: This is the public method which tests whether the target file is on  *
//*       an MTP/GVFS virtual filesystem, and calls the appropriate private    *
//*       method.                                                              *
//******************************************************************************

short FMgr::GetFileStats ( tnFName& tnFile, const gString& fPath, bool getTarg )
{
   short success ;         // return value

   if ( (this->isGvfsPath ( fPath )) )
      success = this->GetFileStats_gvfs ( tnFile, fPath.ustr(), getTarg ) ;
   else
      success = this->GetFileStats ( tnFile, fPath.ustr(), getTarg ) ;

   return success ;

}  //* End GetFileStats() *

//*************************
//*     GetFileStats      *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* ==============                                                             *
//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//* The specified file may, or may not be in the current file list.            *
//* By default the stats for the specified file are returned, but if           *
//* getTarg != false && target is a symbolic link, the symbolic link is        *
//* followed i.e. stats for link target are returned.                          *
//*                                                                            *
//*                                                                            *
//* Input  : tnFile : tnFName class object (by reference) to hold 'stat' data  *
//*          fPath  : full path/filename specification                         *
//*          getTarg: (optional, false by default) if true, symbolic links are *
//*                   followed. Else, do not follow links                      *
//*                                                                            *
//* Returns: OK if successful, all data members of tnFile will be initialized  *
//*          Else returns system 'errno' value                                 *
//*           If error during system call:                                     *
//*            - tnFile.readAcc == false and all other members are undefined   *
//*             Exception:                                                     *
//*             If 'getTarg' != false AND error is a broken symbolic link      *
//*             - tnFile.fName == name of link target if known, else "unknown" *
//*             - tnFile.readAcc == false                                      *
//*             - tnFile.fType == fmTYPES                                      *
//*             - All other members of 'tnFile' are undefined                  *
//******************************************************************************
//* Programmer's Note: There is a bug in older versions of realpath() that may *
//* lead to buffer overflow if the true path is very large.                    *
//* This bug was fixed in libc-5.4.13.                                         *
//*                                                                            *
//******************************************************************************

short FMgr::GetFileStats ( tnFName& tnFile, const char* fPath, bool getTarg )
{
   short success ;         // return value

   //* Provides an orderly crash in case of failure *
   tnFile.fBytes = ZERO ;
   tnFile.fType = fmUNKNOWN_TYPE ;
   tnFile.readAcc = tnFile.writeAcc = false ;

   if ( (success = lstat64 ( fPath, &tnFile.rawStats )) == OK )
   {
      //* Get the file type. If a symbolic link, stat the target file *
      if ( getTarg != false 
           && (this->DecodeFileType ( tnFile.rawStats.st_mode )) == fmLINK_TYPE )
      {
         //* Follow symbolic link to stats for target file *
         char pPath[MAX_PATH] ;        // full path/filename
         if ( (success = stat64 ( fPath, &tnFile.rawStats )) == OK )
         {
            //* Get the decoded path/filename of the link-target file *
            if ( (realpath ( fPath, pPath )) != NULL )
            {
               //* Extract filename from target file's path/filename
               this->ExtractFilename ( tnFile.fName, pPath ) ;
            }
            else
            {
               this->recentErrno = success = errno ;
               this->DebugLog ( "In GetFileStats: realpath() failed", 
                                this->recentErrno, fPath ) ;
               this->BrokenLinkTargetPath ( fPath, pPath, tnFile.fName ) ;
               tnFile.fType = fmTYPES ;   // indicate broken link
               tnFile.readAcc = false ;
            }
         }
         else
         {
            //* A broken link causes 'stat' to fail, but that is not an error  *
            //* that should be reported. It is actually a successful           *
            //* determination that the link target is missing or inaccessible. *
            //* Find the path and file name of the missing target if possible. *
            this->recentErrno = errno ;
            this->BrokenLinkTargetPath ( fPath, pPath, tnFile.fName ) ;
            tnFile.fType = fmTYPES ;   // indicate broken link
            tnFile.readAcc = false ;
         }
      }
      else
      {
         //* Extract filename from caller's string *
         this->ExtractFilename ( tnFile.fName, fPath ) ;
      }
      if ( success == OK )
      {
         tnFile.fBytes = tnFile.rawStats.st_size ;    // number of bytes in file
         tnFile.fType = this->DecodeFileType ( tnFile.rawStats.st_mode ) ; // file type
         this->DecodeEpochTime ( (int64_t)tnFile.rawStats.st_mtime,  // mod time
                                 tnFile.modTime ) ;
         //* The fact that we can 'stat' the file does not necessarily mean    *
         //* that user has read access, so do the test to initialize 'readAcc'.*
         tnFile.readAcc = this->ReadPermission ( tnFile, fPath ) ;
         //* Test for write access in case caller wants to modify/delete file. *
         tnFile.writeAcc = this->WritePermission ( tnFile, fPath ) ;
      }
   }
   else
   {
      //* This method is often called to determine whether a file exists,   *
      //* and if it doesn't, it's not an error so don't display the message.*
      this->recentErrno = errno ;
   }

   return success ;

}  //* End GetFileStats() *

//*************************
//*  GetLinkTargetStats   *
//*************************
//******************************************************************************
//* For a symbolic link source file, return the stat information for the       *
//* file the sym-link points to i.e. the sym-link target.                      *
//*                                                                            *
//*                                                                            *
//* Input  : lnkPath  : full path/filename specification of symbolic link file *
//*          ltrgStats: (by reference, initial contents ignored)               *
//*                     tnFName class object (by reference) to hold 'stat' data*
//*          ltrgPath : (by reference, initial contents ignored)               *
//*                     receives absolute path/filename of link-target file    *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*            - ltrgPath gets the absolute path/filename of link target file. *
//*            - all data members of ltrgStats will be initialized             *
//*          Else if specified source file is not a symbolic link, the errno   *
//*            code EINVAL (invalid argument) is returned.                     *
//*            - ltrgPath is initialized to "unknown"                          *
//*            - ltrgStats.readAcc == false and all other members are undefined*
//*          Else system 'errno' value is returned                             *
//*            If error during system call:                                    *
//*            - ltrgPath is initialized to "unknown"                          *
//*            - ltrgStats.readAcc == false and all other members are undefined*
//*               Exception:                                                   *
//*               If the cause of the error is a broken symbolic link i.e.     *
//*               (link target file not found), then:                          *
//*               - ltrgPath == path/filename of missing link target if known, *
//*                 else "unknown"                                             *
//*               - ltrgStats.fName == name of missing link target if known,   *
//*                 else "unknown"                                             *
//*               - ltrgStats.readAcc == false                                 *
//*               - ltrgStats.fType == fmTYPES                                 *
//*               - All other members of ltrgStats are undefined               *
//******************************************************************************

short FMgr::GetLinkTargetStats ( const gString& lnkPath, 
                                 tnFName& ltrgStats, gString& ltrgPath )
{
   FileStats   lnkStats ;
   short       success ;

   //* Provides an orderly crash in case of failure *
   ltrgPath = "unknown" ;
   ltrgPath.copy( ltrgStats.fName, MAX_FNAME ) ;
   ltrgStats.fBytes = ZERO ;
   ltrgStats.fType = fmUNKNOWN_TYPE ;
   ltrgStats.readAcc = ltrgStats.writeAcc = false ;

   if ( (success = lstat64 ( lnkPath.ustr(), &lnkStats )) == OK )
   {
      //* Get the file type. If a symbolic link, stat the target file *
      if ( (this->DecodeFileType ( lnkStats.st_mode )) == fmLINK_TYPE )
      {
         char tmpPath[MAX_PATH] ;   // full path/filename of link target
         //* Follow symbolic link to stats for target file *
         if ( (success = stat64 ( lnkPath.ustr(), &ltrgStats.rawStats )) == OK )
         {
            //* Get the decoded path/filename of the link-target file *
            if ( (realpath ( lnkPath.ustr(), tmpPath )) != NULL )
            {
               ltrgPath = tmpPath ;          // store target file's path/filename
               //* Extract filename from target file's path/filename
               this->ExtractFilename ( ltrgStats.fName, ltrgPath ) ;
               //* Format the stats *
               ltrgStats.fBytes = ltrgStats.rawStats.st_size ;
               ltrgStats.fType = this->DecodeFileType ( ltrgStats.rawStats.st_mode ) ;
               this->DecodeEpochTime ( (int64_t)ltrgStats.rawStats.st_mtime, ltrgStats.modTime ) ;
               ltrgStats.readAcc = this->ReadPermission ( ltrgStats, ltrgPath.ustr() ) ;
               ltrgStats.writeAcc = this->WritePermission ( ltrgStats, ltrgPath ) ;
            }
            else
            {
               this->recentErrno = errno ;
               this->DebugLog ( "In GetLinkTargetStats: realpath() failed", 
                                this->recentErrno, lnkPath.ustr() ) ;
               this->BrokenLinkTargetPath ( lnkPath.ustr(), tmpPath, ltrgStats.fName ) ;
               //* Path/filename of target or "unknown" *
               if ( *tmpPath == '/' && (strncmp ( ltrgStats.fName, "unknown", 8 )) )
               {  //* If a real path and filename were returned *
                  this->CatPathFname ( ltrgPath, tmpPath, ltrgStats.fName ) ;
               }
               else  // path/filename "unknown"
                  ltrgPath = tmpPath ;
               ltrgStats.fType = fmTYPES ;// indicate broken link
               ltrgStats.readAcc = false ;
            }
         } 
         else  // stat on link target failed
         {
            //* A broken link causes 'stat' to fail, but that is not an error  *
            //* that should be reported. It is actually a successful           *
            //* determination that the link target is missing or inaccessible. *
            //* Find the path and file name of the missing target if possible. *
            this->recentErrno = errno ;
            this->BrokenLinkTargetPath ( lnkPath.ustr(), tmpPath, ltrgStats.fName ) ;
            ltrgStats.fType = fmTYPES ;// indicate broken link
            ltrgStats.readAcc = false ;
         }
      }
      else  // source file is not a symbolic link file
         success = EINVAL ;
   }
   else
   {
      this->recentErrno = errno ;
      this->DebugLog ( "In GetLinkTargetStats: lstat() failed", 
                       this->recentErrno, lnkPath.ustr() ) ;
   }

   return success ;

}  //* End GetLinkTargetStats() *

//***********************
//*    ExpandStats      *
//***********************
//******************************************************************************
//* Expand and format for human readability the 'stat' info for the file       *
//* specified by fs structure.                                                 *
//*                                                                            *
//* Input  : fs   : pointer to FileStats structure (stat64) returned by        *
//*                 the GNU C library stat64() and lstat64() functions.        *
//*                 (Note: for displayed files and files on the    )           *
//*                 (clipboard, this structure is the 'rawStats'   )           *
//*                 (member of the tnFName class object.           )           *
//*          es   : ExpStats structure for holding expanded statistics         *
//*                 a) caller initializes es.fileName with the name of         *
//*                    the target file.                                        *
//*                 b) caller (optionally) initializes es.filePath with        *
//*                    the path string for the file (excluding filename)       *
//*                    (This is needed to 'stat' a symbolic link target.)      *
//*                 c) if the specified file is a symbolic link,               *
//*                    caller (optionally) initializes es.symbTarg             *
//*                    referencing a second ExpStats structure to              *
//*                    hold the info for the symbolic link's target.           *
//*                 (other fields of es initialized on return)                 *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//*             ERR indicates a problem with the symbolic link target:         *
//*               a) invalid es.filePath, or                                   *
//*               b) specified file is a symbolic link, but es.symbTarg        *
//*                  does not point to an ExpStats structure to hold           *
//*                  target file's information, or                             *
//*               c) broken symbolic link                                      *
//*                  Note: if ERR returned, and cause of error was a           *
//*                   broken symbolic link, then                               *
//*                es.symbTarg->fileType == fmTYPES,                           *
//*                es.symbTarg->fileName == filename of missing target         *
//*                                         if available, else "unknown"       *
//*                es.symbTarg->filePath == path of missing target             *
//*                                         if available, else "unknown"       *
//*                   and all other fields of es.symbTarg are undefined.       *
//******************************************************************************

short FMgr::ExpandStats ( const FileStats* fs, ExpStats& es )
{
short    success = OK ;

   //* Copy the raw statistics to ExpStats structure *
   es.rawStats = *fs ;

   //* Format the expanded data *
   this->FormatExpandedStats ( es ) ;

   //* If a symbolic link, find target file *
   if ( es.fileType == fmLINK_TYPE && es.symbTarg != NULL )
   {
      tnFName  tnf ;                   // basic stats of target file
      gString  pPath ;
      this->CatPathFname ( pPath, es.filePath, es.fileName ) ;

      //* Get raw stats for the link-target file *
      //* Note: The true link-target stats and filename are returned.*
      if ( (success = this->GetFileStats ( tnf, pPath.ustr(), true )) == OK )
      {
         //* Get the decoded path/filename string for the link-target file *
         if ( (realpath ( pPath.ustr(), es.symbTarg->filePath )) != NULL )
         {
            //* Truncate string to remove filename *
            gString symtrgPath( es.symbTarg->filePath ) ;
            this->ExtractPathname ( es.symbTarg->filePath, symtrgPath ) ;
         }
         else
         {
            this->recentErrno = errno ;
            this->DebugLog ( "In ExpandStats: realpath() failed", 
                             this->recentErrno, pPath.ustr() ) ;
            this->strnCpy ( es.symbTarg->filePath, "unknown", MAX_PATH ) ;
         }

         //* Format the stats for target file *
         this->strnCpy ( es.symbTarg->fileName, tnf.fName, MAX_FNAME ) ;
         es.symbTarg->rawStats = tnf.rawStats ;
         FormatExpandedStats ( *es.symbTarg ) ;
         es.symbTarg->symbTarg = NULL ;
      }
      else
      {
         //* Symbolic Link target does not exist or is inaccessible. 'tnf' has *
         //* all the pertinent info except the path to the broken link, so     *
         //* unfortunately we must call BrokenLinkTargetPath() again to get it.*
         this->BrokenLinkTargetPath ( pPath.ustr(), es.symbTarg->filePath, 
                                      es.symbTarg->fileName ) ;
         es.symbTarg->fileType = fmTYPES ;   // indicate broken link
         es.symbTarg->fileSize = ZERO ;
      }
   }
   //* If caller has not provided a space for target-file data *
   if ( es.fileType == fmLINK_TYPE && es.symbTarg == NULL )
      success = false ;

   return success ;

}  //* End ExpandStats() *

//***********************
//*    ExpandStats      *
//***********************
//******************************************************************************
//* Expand and format for human readability the 'stat' info for the file       *
//* specified by tnFName class object.                                         *
//*                                                                            *
//* Same as above, except that source data are from an initialized tnFName     *
//* class object AND that es.fileName need not be initialized                  *
//*                                                                            *
//* Input  : tf : pointer to initialized tnFName class object.                 *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short FMgr::ExpandStats ( const tnFName* tf, ExpStats& es )
{
   //* Copy filename ExpStats structure *
   this->strnCpy ( es.fileName, tf->fName, MAX_FNAME ) ;

   return this->ExpandStats ( &tf->rawStats, es ) ;
   
}  //* End ExpandStats() *

//*************************
//*  FormatExpandedStats  *
//*************************
//******************************************************************************
//* Translate the raw directory-entry data into easily readable information.   *
//* Some information is system dependent or file-system dependent, so be       *
//* careful.                                                                   *
//*                                                                            *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* The point here is that even though the raw data are compact, the driving   *
//* application should not have to know anything about system-dependent or     *
//* encrypted data assiciated with a table entry.  We handle all of that in    *
//* this class, and this method is the first, most obvious step.               *
//*                                                                            *
//* fileName:  pointer to the filename string                                  *
//* fileSize:  number of bytes in the file                                     *
//* fileType:  file type, member of enum fmFType                               *
//* modTime :  date/time file was last modified                                *
//* accTime :  date/time file was last accessed                                *
//* staTime :  date/time file status was last updated                          *
//*                                                                            *
//* Tm:                                                                        *
//*  struct tm                                                                 *
//*  {                                                                         *
//*     int  tm_sec ;          // 0-59  (leap seconds to 61)                   *
//*     int  tm_min ;          // 0-59                                         *
//*     int  tm_hour ;         // 0-23                                         *
//*     int  tm_mday ;         // 1-31                                         *
//*     int  tm_mon ;          // 0-11                                         *
//*     int  tm_year ;         // since 1900                                   *
//*     int  tm_wday ;         // 0-6                                          *
//*     int  tm_yday ;         // 0-365                                        *
//*     int  tm_isdst ;        // >0 == is DST, 0 == not DST, <0 == unknown    *
//*  } ;                                                                       *
//*                                                                            *
//* Input  : es ExpStats object (by reference)                                 *
//*             only fileName and rawStats fields have been initialized        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* These are simplified forms of the structures defined in sys/stat.h,        *
//* bits/stat.h, time.h, types.h, etc. see 'info fstat' for more information.  *
//*                                                                            *
//* struct stat               // supports file size <= 2*31 bytes              *
//* {                                                                          *
//*    dev_t      st_dev ;    // device                            (8 bytes)   *
//*    ino_t      st_ino ;    // inode                             (4 bytes)   *
//*    mode_t     st_mode ;   // mode i.e. permissions & file type (4 bytes)   *
//*    nlink_t    st_nlink ;  // number of hard links              (4 bytes)   *
//*    uid_t      st_uid ;    // user ID of file's owner           (4 bytes)   *
//*    gid_t      st_gid ;    // group ID of file's owner          (4 bytes)   *
//*    dev_t      st_rdev ;   // device type (if an inode device)  (8 bytes)   *
//*    off_t      st_size ;   // total size, in bytes              (4 bytes)   *
//*    blksize_t  st_blksize; // blocksize for file system i/o     (4 bytes)   *
//*    blkcnt_t   st_blocks ; // number of blocks allocated        (4 bytes)   *
//*    time_t     st_atime ;  // time of last access (file opened) (4 bytes)   *
//*    time_t     st_mtime ;  // time of last modification         (4 bytes)   *
//*    time_t     st_ctime ;  // time of last change (in status)   (4 bytes)   *
//* } ;                                                                        *
//*                                                                            *
//*                                                              64-bit systems*
//* struct stat64             // supports file size > 2*31 bytes --------------*
//* {                                                                          *
//*    dev_t      st_dev ;    // device                            (8 bytes)   *
//*    ino64_t    st_ino ;    // inode  (__u_quat_t type)          (8 bytes)   *
//*    mode_t     st_mode ;   // mode i.e. permissions & file type (4 bytes)   *
//*    nlink_t    st_nlink ;  // number of hard links              (8 bytes)   *
//*    uid_t      st_uid ;    // user ID of file's owner           (4 bytes)   *
//*    gid_t      st_gid ;    // group ID of file's owner          (4 bytes)   *
//*    dev_t      st_rdev ;   // device type (if an inode device)  (8 bytes)   *
//*    off64_t    st_size ;   // total size, in bytes (__quad_t)   (8 bytes)   *
//*    blksize_t  st_blksize; // blocksize for file system i/o     (8 bytes)   *
//*    blkcnt64_t st_blocks ; // blocks allocated (__quat_t type)  (8 bytes)   *
//*    time_t     st_atime ;  // time of last access (file opened) (8 bytes)   *
//*    time_t     st_mtime ;  // time of last modification         (8 bytes)   *
//*    time_t     st_ctime ;  // time of last change (in status)   (8 bytes)   *
//*    - - -                                                                   *
//*    __ino_t    __st_ino    // legacy 32-bit inode                           *
//* } ;                                                                        *
//*                                                                            *
//******************************************************************************

void FMgr::FormatExpandedStats ( ExpStats& es )
{
   es.fileSize = es.rawStats.st_size ;
   es.fileType = DecodeFileType ( es.rawStats.st_mode ) ;
   es.iNode    = es.rawStats.st_ino ;
   es.userID   = es.rawStats.st_uid ;
   es.grpID    = es.rawStats.st_gid ;
   es.hLinks   = es.rawStats.st_nlink ;

   //* Decode the file permissions (st_mode) *
   this->DecodePermissions ( es ) ;

   this->DecodeEpochTime ( (int64_t)es.rawStats.st_mtime, es.modTime ) ;// "last modified" time
   this->DecodeEpochTime ( (int64_t)es.rawStats.st_atime, es.accTime ) ;// "last accessed" time
   this->DecodeEpochTime ( (int64_t)es.rawStats.st_ctime, es.staTime ) ;// "status last modified" time

   //* Get the name of the the file's owner *
   PwdInfo*  pw ;
   if ( (pw = getpwuid ( es.userID )) != NULL )
      this->strnCpy ( es.ownerName, pw->pw_name, USERNAME_SIZE ) ;
   else
      this->strnCpy ( es.ownerName, "unknown", USERNAME_SIZE ) ;

   //* Get the name of the the file owner's group *
   GrpInfo*  gp ;
   if ( (gp = getgrgid ( es.grpID )) != NULL )
      this->strnCpy ( es.groupName, gp->gr_name, USERNAME_SIZE ) ;
   else
      this->strnCpy ( es.groupName, "unknown", USERNAME_SIZE ) ;

}  //* End FormatExpandedStats() *

//*************************
//*    DecodeFileType     *
//*************************
//******************************************************************************
//* Extract the file type from the "st_mode" element of the stat{} structure   *
//* and return the equivalent member of enum fmFType.                          *
//*                                                                            *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* The POSIX standard defines certain macros to access the file "st_mode".    *
//* There are several layers to the macros, but for Linux EXT2, EXT3 we        *
//* finally get down to the actual bits in /usr/include/bits/stat.h:           *
//* See also page 135 of "Linux Programming by Example."                       *
//* The actual macros that use these bitmasks are used in this routine.        *
//*                                                                            *
//* Encoding of the file mode.                                                 *
//* #define	__S_IFMT	0170000	/* These bits determine file type.               *
//* File types.                                                                *
//* #define	__S_IFDIR	0040000	 Directory.                                   *
//* #define	__S_IFCHR	0020000	 Character device.                            *
//* #define	__S_IFBLK	0060000	 Block device.                                *
//* #define	__S_IFREG	0100000	 Regular file.                                *
//* #define	__S_IFIFO	0010000	 FIFO.                                        *
//* #define	__S_IFLNK	0120000	 Symbolic link.                               *
//* #define	__S_IFSOCK	0140000	 Socket.                                      *
//*                                                                            *
//* Input Values : mode: st_mode field of file's stat{} strucutre              *
//*                                                                            *
//* Return Value : member of enum fmFType                                      *
//******************************************************************************

fmFType FMgr::DecodeFileType ( UINT mode )
{
fmFType     ftype = fmUNKNOWN_TYPE ;

   if ( (S_ISREG(mode)) != false )        ftype = fmREG_TYPE ;
   else if ( (S_ISDIR(mode)) != false )   ftype = fmDIR_TYPE ;
   else if ( (S_ISLNK(mode)) != false )   ftype = fmLINK_TYPE ;
   else if ( (S_ISCHR(mode)) != false )   ftype = fmCHDEV_TYPE ;
   else if ( (S_ISBLK(mode)) != false )   ftype = fmBKDEV_TYPE ;
   else if ( (S_ISFIFO(mode)) != false )  ftype = fmFIFO_TYPE ;
   else if ( (S_ISSOCK(mode)) != false )  ftype = fmSOCK_TYPE ;
   
   return ftype ;
   
}  //* End DecodeFileType() *

//*************************
//*   DecodePermissions   *
//*************************
//******************************************************************************
//* Extract the file permissions and other info from the "st_mode" element     *
//* of the stat{} structure and initialize the corrosponding fields of the     *
//* ExpStats structure.                                                        *
//*                                                                            *
//* PRIVATE METHOD.                                                            *
//*                                                                            *
//* #define __S_ISUID   04000	 Set user ID on execution.                       *
//* #define __S_ISGID   02000	 Set group ID on execution.                      *
//* #define __S_ISVTX   01000	 Save swapped text after use (sticky).           *
//* #define S_IRUSR     00400  User read permission                            *
//* #define S_IWUSR     00200  User write permission                           *
//* #define S_IXUSR     00100  User execute permission                         *
//* #define S_IRGRP     00040  Group read permission                           *
//* #define S_IWGRP     00020  Group write permission                          *
//* #define S_IXGRP     00010  Group execute permission                        *
//* #define S_IROTH     00004  Others read permission                          *
//* #define S_IWOTH     00002  Others write permission                         *
//* #define S_IXOTH     00001  Others execute permission                       *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : es : ExpStats structure (by reference) with at least the          *
//*               es.rawStats.st_mode field initialized                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FMgr::DecodePermissions ( ExpStats& es )
{
UINT     mode = es.rawStats.st_mode ;

   es.usrProt.read  = char((mode & S_IRUSR) ? 'r' : '-') ;  // user permissions
   es.usrProt.write = char((mode & S_IWUSR) ? 'w' : '-') ;
   es.usrProt.exec  = char((mode & S_IXUSR) ? 'x' : '-') ;
   
   es.grpProt.read  = char((mode & S_IRGRP) ? 'r' : '-') ;  // group permissions
   es.grpProt.write = char((mode & S_IWGRP) ? 'w' : '-') ;
   es.grpProt.exec  = char((mode & S_IXGRP) ? 'x' : '-') ;
   
   es.othProt.read  = char((mode & S_IROTH) ? 'r' : '-') ;  // others permissions
   es.othProt.write = char((mode & S_IWOTH) ? 'w' : '-') ;
   es.othProt.exec  = char((mode & S_IXOTH) ? 'x' : '-') ;

   es.setUID  = bool((mode & S_ISUID) != false) ;
   es.setGID  = bool((mode & S_ISGID) != false) ;
   es.sticky  = bool((mode & S_ISVTX) != false) ;
   
}  //* End DecodePermissions() *

//*************************
//*  SetFilePermissions   *
//*************************
//******************************************************************************
//* Set file permissions for the specified file: USR rwx, GRP rwx, OTHER rwx,  *
//* sUID, sGID, and sticky bit.                                                *
//* (requires either superuser access or ownership of the file.)               *
//* See <sys/stat.h> and <bits/stat.h> for mask definitions.                   *
//* NOTE: For a symbolic link source file, setting or resetting permissions    *
//*       always occurs for the link target, not the link itself.              *
//*                                                                            *
//* Input  : pathBuff: full path/filename specification                        *
//*          newMode : encoded bit word (unsigned int)                         *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'.                         *
//******************************************************************************

short FMgr::SetFilePermissions ( const char* pathBuff, mode_t newPermissions )
{
   #define DEBUG_SFP 0

   short status = chmod ( pathBuff, newPermissions ) ;

   if ( status != OK )
      status = this->recentErrno = errno ;

   #if DEBUG_SFP != 0
   this->DebugLog ( "In SetFilePermissions: chmod status", status, pathBuff ) ;
   #endif   // DEBUG_SFP

   return status ;

#undef DEBUG_SFP
}  //* End SetFilePermissions() *

//*************************
//*   SetFileOwnerGroup   *
//*************************
//******************************************************************************
//* Set file owner (userID) and/or file group (grpID).                         *
//*                                                                            *
//* Requires superuser access to set owner.                                    *
//* Requires either superuser access or ownership of the file to set group.    *
//* (For owner to set group, owner must be a member of the target group.)      *
//*                                                                            *
//* As a side effect, cTime will be set to current system time.                *
//*                                                                            *
//* Input  : pathBuff: full path/filename specification                        *
//*          newOwner: new User ID   (if -1, then no change)                   *
//*          newGroup: new Group ID  (if -1, then no change)                   *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'.                         *
//******************************************************************************

short FMgr::SetFileOwnerGroup ( const char* pathBuff, uid_t newOwner, gid_t newGroup )
{
   #define DEBUG_SFOG 0

   short status = lchown ( pathBuff, newOwner, newGroup ) ;

   if ( status != OK )
      status = this->recentErrno = errno ;

   #if DEBUG_SFOG != 0
   this->DebugLog ( "In SetFileOwnerGroup: lchown status", status, pathBuff ) ;
   #endif   // DEBUG_SFOG

   return status ;

   #undef DEBUG_SFOG
}  //* End SetFileOwnerGroup() *

//*************************
//*    ReadPermission     *
//*************************
//******************************************************************************
//* Determine whether user has 'read' permission for the specified file,       *
//* and optionally enable owner/group read permission.                         *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* True read permission on a directory name actually requires BOTH 'Read' and *
//* 'Execute' permission.                                                      *
//* a) 'Execute' only, will allow us to set the directory to CWD, but          *
//*    will not allow us to read its contents.                                 *
//* b) 'Read' only, will allow us to read the NAMES of the files contained in  *
//*    the directory (but NOT the other stats), but will not allow us to set   *
//*    the directory as the CWD.                                               *
//* Therefore, if caller wants a report of read access on a directory name,    *
//* we indicate read access ONLY if user has both 'Read' and 'Execute'         *
//* permission. If caller wants to enable read access on a directory name, we  *
//* enable BOTH 'Read' and 'Execute' permissions.                              *
//*                                                                            *
//* Input : sStats: (by reference) stats of file to be tested                  *
//*                 sStats.readAcc will be set/reset according to result       *
//*                 and if 'enable', then st_mode bits will be updated,        *
//*                 other members remain unchanged                             *
//*         sPath : path/filename of file to be tested                         *
//*         enable: (optional, false by default)                               *
//*                 if 'true', enable 'owner' and 'group' read access          *
//*                 if 'false', simply report user's current read access       *
//*                                                                            *
//* Returns: 'true' if user has read permission, else 'false'                  *
//******************************************************************************
//* Programmer's Note: The 'access' library function is very good in most      *
//* respects in that it accurately mimics true admin tests of access; however, *
//* it follows symbolic links, rather than reporting the permission on the     *
//* link file itself. For this reason, IF !'enable' we do a direct check on    *
//* the permission bits of symbolic-link source files. This is necessary,      *
//* since the caller wants to know if it's ok to copy the symlink, NOT the     *
//* link target.                                                               *
//*                                                                            *
//* However, if caller wants to enable read access on a protected file, the    *
//* enable will be done on the link target, since the permission bits for the  *
//* link file are meaningless in that case.                                    *
//*                                                                            *
//******************************************************************************

bool FMgr::ReadPermission ( tnFName& sStats, const gString& sPath, bool enable )
{

   return this->ReadPermission ( sStats, sPath.ustr(), enable ) ;

}  //* End ReadPermission() *

bool FMgr::ReadPermission ( tnFName& sStats, const char* sPath, bool enable )
{
   //* Determine user's read access *
   if ( (sStats.fType != fmLINK_TYPE) || (sStats.fType == fmLINK_TYPE && enable) )
   {
      sStats.readAcc = bool((access ( sPath, R_OK )) == ZERO) ;
      if ( sStats.readAcc != false && sStats.fType == fmDIR_TYPE )
         sStats.readAcc = bool((access ( sPath, X_OK )) == ZERO) ;

      if ( !sStats.readAcc && enable != false )
      {
         mode_t sMode = sStats.rawStats.st_mode ;
         if ( sStats.fType == fmLINK_TYPE )
         {  //* Get link target info *
            tnFName ltrgStats ;
            gString ltrgPath ;
            if ( (this->GetLinkTargetStats ( sPath, ltrgStats, ltrgPath )) == OK )
               sMode = ltrgStats.rawStats.st_mode ;
            if ( ltrgStats.fType == fmDIR_TYPE )
               sMode |= (S_IXUSR | S_IXGRP) ;
         }
         else if ( sStats.fType == fmDIR_TYPE )
            sMode |= (S_IXUSR | S_IXGRP) ;
         sMode |= (S_IRUSR | S_IRGRP) ;
         if ( (this->SetFilePermissions ( sPath, sMode )) == OK )
         {
            sStats.rawStats.st_mode = sMode ;
            sStats.readAcc = true ;
         }
      }
   }
   else
   {  //* Source file is a symbolic link - report only *
      mode_t sMode = sStats.rawStats.st_mode ;
      bool   usrRE = bool((sMode & S_IRUSR) != ZERO), 
             grpRE = bool((sMode & S_IRGRP) != ZERO), 
             othRE = bool((sMode & S_IROTH) != ZERO) ;
      sStats.readAcc = false ;      // assume no read access

      //* If universal 'read' permission OR if user is Super-User *
      if ( othRE || this->userInfo.userID == superUID )
         sStats.readAcc = true ;

      //* If user is file's owner AND owner 'read' permission *
      else if ( usrRE && sStats.rawStats.st_uid == this->userInfo.userID )
         sStats.readAcc = true ;

      //* If group 'read' permission, determine  *
      //* whether user is a member of that group *
      else if ( grpRE )
      {
         gid_t grpID = sStats.rawStats.st_gid ;
         //* If file group is user's group AND group 'read' permission *
         if ( grpID == this->userInfo.grpID )
            sStats.readAcc = true ;
         //* If user is a member of the file's group * 
         else
         {
            for ( short i = ZERO ; i < this->userInfo.suppGroups ; i++ )
            {
               if ( this->userInfo.suppGID[i] == grpID )
               {
                  sStats.readAcc = true ;
                  break ;
               }
            }
         }
      }
   }
   return sStats.readAcc ;

}  //* End ReadPermission() *

//*************************
//*    WritePermission    *
//*************************
//******************************************************************************
//* Determine whether user has 'write' permission for the specified file,      *
//* and optionally enable owner/group write permission.                        *
//*                                                                            *
//* Input : sStats: (by reference) stats of file to be tested                  *
//*                 sStats.writeAcc will be set/reset according to result      *
//*                 and if 'enable', then st_mode bits will be updated,        *
//*                 other members remain unchanged                             *
//*         sPath : path/filename of file to be tested                         *
//*         enable: (optional, false by default)                               *
//*                 if 'true', enable 'owner' and 'group' write access         *
//*                 if 'false', simply report user's current write access      *
//*                                                                            *
//* Returns: 'true' if user has write permission, else 'false'                 *
//******************************************************************************
//* Programmer's Note: The 'access' library function is very good in most      *
//* respects in that it accurately mimics true admin tests of access; however, *
//* it follows symbolic links, rather than reporting the permission on the     *
//* link file itself. For this reason, IF !'enable' we do a direct check on    *
//* the permission bits of symbolic-link source files. This is necessary,      *
//* since the caller wants to know if it's ok to modify/delete the symlink,    *
//* NOT the link target.                                                       *
//*                                                                            *
//* However, if caller wants to enable write access on a protected file, the   *
//* enable will be done on the link target, since the permission bits for the  *
//* link file are meaningless in that case.                                    *
//******************************************************************************

bool FMgr::WritePermission ( tnFName& sStats, const gString& sPath, bool enable )
{

   return this->WritePermission ( sStats, sPath.ustr(), enable ) ;

}  //* End WritePermission() *

bool FMgr::WritePermission ( tnFName& sStats, const char* sPath, bool enable )
{
   //* Determine user's write access *
   if ( (sStats.fType != fmLINK_TYPE) || (sStats.fType == fmLINK_TYPE && enable) )
   {
      sStats.writeAcc = bool((access ( sPath, W_OK )) == ZERO) ;

      if ( !sStats.writeAcc && enable != false )
      {
         mode_t sMode = sStats.rawStats.st_mode ;
         if ( sStats.fType == fmLINK_TYPE )
         {  //* Get link target info *
            tnFName ltrgStats ;
            gString ltrgPath ;
            if ( (this->GetLinkTargetStats ( sPath, ltrgStats, ltrgPath )) == OK )
               sMode = ltrgStats.rawStats.st_mode ;
         }
         sMode |= (S_IWUSR | S_IWGRP) ;
         if ( (this->SetFilePermissions ( sPath, sMode )) == OK )
         {
            sStats.rawStats.st_mode = sMode ;
            sStats.writeAcc = true ;
         }
      }
   }
   else
   {  //* Source file is a symbolic link - report only *
      mode_t sMode = sStats.rawStats.st_mode ;
      bool   usrWE = bool((sMode & S_IWUSR) != ZERO), 
             grpWE = bool((sMode & S_IWGRP) != ZERO), 
             othWE = bool((sMode & S_IWOTH) != ZERO) ; 
      sStats.writeAcc = false ;     // assume no write accees

      //* If universal 'write' permission OR if user is Super-User *
      if ( othWE || this->userInfo.userID == superUID )
         sStats.writeAcc = true ;

      //* If user is file's owner AND owner 'write' permission *
      else if ( usrWE && sStats.rawStats.st_uid == this->userInfo.userID )
         sStats.writeAcc = true ;

      //* If group 'write' permission, determine *
      //* whether user is a member of that group *
      else if ( grpWE )
      {
         gid_t grpID = sStats.rawStats.st_gid ;
         //* If file group is user's group AND group 'write' permission *
         if ( grpID == this->userInfo.grpID )
            sStats.writeAcc = true ;
         //* If user is a member of the file's group * 
         else
         {
            for ( short i = ZERO ; i < this->userInfo.suppGroups ; i++ )
            {
               if ( this->userInfo.suppGID[i] == grpID )
               {
                  sStats.writeAcc = true ;
                  break ;
               }
            }
         }
      }
   }
   return sStats.writeAcc ;

}  //* End WritePermission() *

//*************************
//*  ExecutePermission    *
//*************************
//******************************************************************************
//* Determine whether user has 'execute' permission for the specified file,    *
//* and optionally set file as executable for owner/group/others.              *
//* This method follows symbolic links and reports or modifies the link-target *
//* file (if it exists).                                                       *
//*                                                                            *
//* Input : sPath : path/filename of file to be modified                       *
//*         enable: if 'true', enable 'execute' permission                     *
//*                 if 'false', simply report current 'executable' status      *
//*                                                                            *
//* Returns: 'true' if user has execute permission                             *
//*          'false' if no execute permission OR if file does not exist        *
//******************************************************************************
//* Programmer's Note: Symbolic links are executable by nature. It is the      *
//* link target's permissions that are of interest.                            *
//*                                                                            *
//******************************************************************************

bool FMgr::ExecutePermission ( const gString& sPath, bool enable )
{
   bool hasPermission = false ;
   if ( (access ( sPath.ustr(), F_OK )) == ZERO)   // if file exists
   {
      hasPermission = bool((access ( sPath.ustr(), X_OK )) == ZERO) ;

      //* If we were asked to enable exec permission,*
      //* AND user doesn't already have it           *
      if ( enable && ! hasPermission )
      {
         FileStats rawStats ;
         if ( (stat64 ( sPath.ustr(), &rawStats )) == OK )
         {
            mode_t newMode = rawStats.st_mode | (S_IXUSR | S_IXGRP | S_IXOTH) ;
            if ( (this->SetFilePermissions ( sPath.ustr(), newMode )) == OK )
               hasPermission = true ;
         }
      }
   }
   return hasPermission ;

}  //* End ExecutePermission() *

//*************************
//*     WriteProtect      *
//*************************
//******************************************************************************
//* Write-protect the specified file. Resets owner/group/others                *
//* write-permission bits.                                                     *
//*                                                                            *
//* Input : sPath : path/filename of file to be modified                       *
//*         sStats: (by reference) stats of file to be modified                *
//*                 if successful, sStats.rawStats.st_mode will be updated and *
//*                 sStats.writeAcc will be reset                              *
//*                                                                            *
//* Returns: OK if successful, else 'errno' code                               *
//******************************************************************************

short FMgr::WriteProtect ( const gString& sPath, tnFName& sStats )
{
   mode_t   newMode = sStats.rawStats.st_mode,
            mBits = (S_IWUSR | S_IWGRP | S_IWOTH) ;
   if ( sStats.fType == fmLINK_TYPE )
   {  //* Get link target info *
      tnFName ltrgStats ;
      gString ltrgPath ;
      if ( (this->GetLinkTargetStats ( sPath, ltrgStats, ltrgPath )) == OK )
         newMode = ltrgStats.rawStats.st_mode ;
   }
   short status = OK ;              // return value

   //* If file is not already write-protected, protect it now.*
   if ( ((newMode & mBits) != ZERO) || sStats.writeAcc != false )
   {
      newMode &= ~mBits ;
      if ( (status = this->SetFilePermissions ( sPath.ustr(), newMode )) == OK )
      {
         sStats.rawStats.st_mode = newMode ;
         sStats.writeAcc = false ;
      }
   }
   return status ;

}  //* End WriteProtect() *

//*************************
//*   SetFileTimestamp    *
//*************************
//******************************************************************************
//* Set file Modify and/or Access date/timestamps.                             *
//* As a side effect, cTime will be set to current system time.                *
//* (requires file write permission)                                           *
//*                                                                            *
//* Input  : pathBuff  : full path/filename specification                      *
//*          ft        : structure must be correctly initialized               *
//*                      (except for 'day' which is ignored)                   *
//*          updateMod : (optional, 'true' by default)                         *
//*                      if true, update modification time                     *
//*          updateAcc : (optional, 'true' by default)                         *
//*                      if true, update access time                           *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'                          *
//*            file does not exist, no write permission, etc.                  *
//******************************************************************************
//*                                                                            *
//* struct utimbuf        // used for call to utime()                          *
//* {                                                                          *
//*    time_t actime ;    // access time                                       *
//*    time_t modtime ;   // modification time                                 *
//* } ;                                                                        *
//*                                                                            *
//******************************************************************************

short FMgr::SetFileTimestamp ( const char* pathBuff, const localTime& ft, 
                               bool updateMod, bool updateAcc )
{
   short status = OK ;
   if ( updateMod || updateAcc )
   {
      FileStats rawStats ;
      if ( (lstat64 ( pathBuff, &rawStats )) == ZERO )
      {  //* Get the target file's current time values *
         uTime ut ;
         ut.actime  = rawStats.st_atime ;
         ut.modtime = rawStats.st_mtime ;

         //* Convert formatted time to simple epoch time *
         localTime tmp_ft = ft ;    // ('ft' is const so we need a copy)
         time_t newTime = (time_t)this->EncodeEpochTime ( tmp_ft ) ;

         if ( updateMod )  ut.modtime = newTime ;  // new modification time
         if ( updateAcc )  ut.actime = newTime ;   // new access time
         if ( (utime ( pathBuff, &ut )) != ZERO )
            status = this->recentErrno = errno ;
      }
      else
        status = this->recentErrno = errno ;
   }
   return status ;

}  //* End SetFileTimestamp() *

//*************************
//* BrokenLinkTargetPath  *
//*************************
//******************************************************************************
//* Retrieve the path/filename of missing target for a broken symbolic link.   *
//*                                                                            *
//* Input  : linkPath: full path/filename specification of symbolic link file  *
//*          trgPath : receives target path (not filename) if available        *
//*          trgName : receives target filename if available                   *
//*                                                                            *
//* Returns: none                                                              *
//*          (if unable to retrieve the data, then both buffers get "unknown") *
//******************************************************************************
//* Programmer's Note: Contents of link file may be relative path OR absolute  *
//* path OR upgefuched. We return the absolute path if possible. If 'realpath' *
//* returns NULL, we may still have the path IF errno == EACCES ||             *
//* errno == ENOENT. If contents of link file are corrupted, then we will      *
//* unfortunately return garbage.                                              *
//******************************************************************************

void FMgr::BrokenLinkTargetPath ( const char* linkPath, 
                                  char trgPath[MAX_PATH], char trgName[MAX_FNAME] )
{
   static const char* cantDoIt = "unknown" ;
   char tmpPath[MAX_PATH] ;
   short status = ERR ;

   short bytesIn = readlink ( linkPath, tmpPath, (MAX_PATH-1) ) ;
   if ( bytesIn > ZERO )
   {
      tmpPath[bytesIn] = NULLCHAR ;    // be sure string is terminated
      const char* rpRet = realpath ( tmpPath, trgPath ) ;
      short rpErrno = errno ;
      if ( rpRet != NULL || (rpErrno == EACCES || rpErrno == ENOENT) )
      {
         gString gs( trgPath ) ;
         this->ExtractPathname ( trgPath, gs ) ;
         this->ExtractFilename (trgName, gs ) ;
         if ( *trgPath != NULLCHAR && *trgName != NULLCHAR )
            status = OK ;
      }
   }
   if ( status != OK )
   {  //* Path retrieval unsuccessful, use default strings *
      this->strnCpy ( trgPath, cantDoIt, MAX_PATH ) ;
      this->strnCpy ( trgName, cantDoIt, MAX_FNAME ) ;
   }

}  //* End BrokenLinkTargetPath() *

//*************************
//*  GetFilesystemStats   *
//*************************
//******************************************************************************
//* Read file system stats for the filesystem containing CURRENT directory.    *
//*                                                                            *
//* Input  : fsStats: (by reference) unitialized instance of                   *
//*                   fileSystemStats class: receives stat data                *
//*                                                                            *
//* Returns: OK if successful or 'errno' value if stat failed                  *
//*           All members of 'fsStats' for which the target filesystem         *
//*           provides data have been initialized.                             *
//*           Note: Some filesystems do not provide data for all fields.       *
//******************************************************************************
//* References the 'irVirtual' member to determine whether current base        *
//* directory lies within an MTP/GVFS filesystem, then calls the appropriate   *
//* private method.                                                            *
//******************************************************************************

short FMgr::GetFilesystemStats ( fileSystemStats& fsStats )
{

   return ( this->GetFilesystemStats ( this->currDir, fsStats ) ) ;

}  //* GetFilesystemStats() *

//*************************
//*  GetFilesystemStats   *
//*************************
//******************************************************************************
//* Read file system stats for filesystem containing specified path/filename.  *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*          fsStats: (by reference) unitialized instance of                   *
//*                   fileSystemStats class: receives stat data                *
//*                                                                            *
//* Returns: OK if successful or 'errno' value if stat failed                  *
//*           All members of 'fsStats' for which the target filesystem         *
//*           provides data have been initialized.                             *
//*           Note: Some filesystems do not provide data for all fields.       *
//******************************************************************************
//* 'stat' utility, valid format sequences for file systems:                   *
//*  %a   Free blocks available to non-superuser                               *
//*  %b   Total data blocks in file system                                     *
//*  %c   Total file nodes in file system                                      *
//*  %d   Free file nodes in file system                                       *
//*  %f   Free blocks in file system                                           *
//*  %C   SELinux security context string                                      *
//*  %i   File System ID in hex                                                *
//*  %l   Maximum length of filenames                                          *
//*  %n   File name                                                            *
//*  %s   Block size (for faster transfers)                                    *
//*  %S   Fundamental block size (for block counts)                            *
//*  %t   Type in hex                                                          *
//*  %T   Type in human readable form (not very accurate, ignored)             *
//*                                                                            *
//* See 'info stat' and 'stat --help' for more information.                    *
//* See also 'info ls --context' to get seLinux context strings if not         *
//*                    available from 'stat' command. This avoids dependency   *
//*                    on the seLinux library which may be unavailable or      *
//*                    restricted on some systems. (see info lgetfilecon)      *
//*                                                                            *
//* See 'info df'. 'df' command is used to retrieve the block-device driver    *
//*                     name, given the target path.                           *
//*                                                                            *
//* See 'info lsblk'. 'lsblk' returns various information on the specified     *
//*                   block storage device. In this method we query for the    *
//*                   UUID (Universally Unique IDentifier), filesystem type    *
//*                   string, label (if any) and mountpoint (path/dirname).    *
//*                                                                            *
//* Programmer's Notes:                                                        *
//*  1) The 'df' call determines how much information will be retrieved:       *
//*     a) If a valid device name (path) is returned, then the 'lsblk' call is *
//*        likely to succeed and indicates that the block device IS mounted.   *
//*     b) If a device name of "tmpfs" is returned, then 'trgPath' is not on a *
//*        mounted filesystem, and the 'lsblk' call will fail.                 *
//*     c) If a device name of "gvfsd-fuse" is returned, then 'trgPath' is     *
//*        within a smartphone filesystem and requires the MTP tools for full  *
//*        data access. (example trgPath: /run/user/1000/mtp:host=....)        *
//*     d) If any other device name which does not begin with "/dev/", then    *
//*        the 'lsblk' call will fail indicating either that the block device  *
//*        is not mounted, or an unknown special case has been encountered.    *
//*     e) Note that the gString class will not allow most control codes, but  *
//*        that the smartphone virtual path WILL contain escape sequences      *
//*        and/or literal characters:                                          *
//*           example: mtp:host=%5Busb%3A001%2C007%5D                          *
//*        where '%' represents the escape character, 0x5B is the left-square- *
//*        bracket ( '[' ), 0x3A is a colon ( ':' ), 0x2C is a comma ( ',' )   *
//*        and 0x5D is the right-square-bracket ( ']' ).                       *
//*     f) Note that the 'stat' call will almost always succeed, but the data  *
//*        returned may be unreliable because it may not reference a physical  *
//*        block storage device.                                               *
//*                                                                            *
//*  2) 'isMountpoint' member is initialized by comparing 'mntPoint' to        *
//*     'trgPath'*. This information is useful when deciding whether to        *
//*     recurse into a subdirectory. Of course, if the device is not mounted,  *
//*     then there is insufficient information to make the determination.      *
//*                                                                            *
//*  3) The algorithm for parsing the string data is a bit convoluted because  *
//*     we parse it as wchar_t (wide) data as much as possible. This is        *
//*     because it is unclear whether non-ASCII data will be present in any    *
//*     given field, therefore it is dangerous to assume that one byte equals  *
//*     one character.                                                         *
//*                                                                            *
//* "Special Characters"                                                       *
//* --------------------                                                       *
//* Shell utilities "stat,", "df," and "ls" as well as the shell itself may    *
//* misinterpret some filename characters unless they are "escaped".           *
//*                                                                            *
//*                 ----  ----  ----  ----  ----  ----                         *
//* Programmer's Note: There is a bug in the 'stat' utility which for GVFS     *
//* filesystem only, incorrectly reports the filesystem type code 'fsTypeCode' *
//* (%t parameter). This is because 'stat' is expecting a hexidecimal value    *
//* (ASCII hex string), but the gvfs device returns a decimal value instead.   *
//* We compensate for this by substituting the intended hex value.             *
//******************************************************************************

short FMgr::GetFilesystemStats ( const char* trgPath, fileSystemStats& fsStats )
{
   const char fssCmd[] = 
         "stat -f --printf=\"%b %f %a %c %d %s %S %l %i %t %C \n\"" ;
   const char selCmd[] = "ls -Zd" ;
   const uint32_t GVFS_FSTYPE = 0x3EB0B7A,   // compensate for bug (see above)
                  GVFS_FSTYPE_ERR = 65735546 ;
   //const uint32_t EXFAT_FSTYPE = 0x2011BAB0 ; // exFAT (Microsloth proprietary filesystem for flash drives)
   //const uint32_t TMPFS_FSTYPE = 0x1021994 ; // type code for 'tmpfs' filesystems
   //const uint32_t PROC_FSTYPE  = 0x09fA0 ;   // type code for 'proc' filesystems
   gString escPath( trgPath ),tmpPath ;
   char    cmdBuff[MAX_PATH*3] ;
   short   status ;                       // return value

   this->CreateTempname ( tmpPath ) ;     // create a temp file

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   //* Construct the system command.*
   snprintf ( cmdBuff, (MAX_PATH*3), "%s \"%s\" 1>\"%s\" 2>/dev/null", 
              fssCmd, escPath.ustr(), tmpPath.ustr() ) ;

   fsStats.reset() ;    // re-initialize caller's data

   if ( (status = this->Systemcall ( cmdBuff )) == OK )
   {
      //* Read the only line in the file *
      ifstream ifsIn ( tmpPath.ustr(), ifstream::in ) ;
      ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
      ifsIn.close () ;     // close the temp file

      gString fsData( cmdBuff ) ;
      fsData.gscanf( L"%lu %lu %lu %lu %lu %u %u %hu %llX %x %63s",
         &fsStats.blockTotal, &fsStats.blockFree,  &fsStats.blockAvail, 
         &fsStats.inodeTotal, &fsStats.inodeAvail,
         &fsStats.blockSize,  &fsStats.fblockSize, &fsStats.nameLen,
         &fsStats.systemID,   &fsStats.fsTypeCode, fsStats.seLinux ) ;

      fsStats.freeBytes = (UINT64)fsStats.fblockSize * fsStats.blockAvail ;
      fsStats.usedBytes = (UINT64)fsStats.fblockSize * 
                                  (fsStats.blockTotal - fsStats.blockFree) ;

      //* Call the 'df' command to obtain the device name.*
      const char* dfTemplate = 
               "df --output=source \"%s\" 1>\"%s\" 2>/dev/null" ;
      snprintf ( cmdBuff, (MAX_PATH*3), dfTemplate, 
                 escPath.ustr(), tmpPath.ustr() ) ;
      fsStats.device[0] = NULLCHAR ;   // initialize the target field
      if ( (status = this->Systemcall ( cmdBuff )) == OK )
      {
         ifsIn.open( tmpPath.ustr(), ifstream::in ) ;
         if ( ifsIn.is_open() )
         {  //* Read and discard the header line.*
            ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;

            //* Read and scan the data line.*
            ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
            if ( ifsIn.gcount() > ZERO )
            {
               fsData = cmdBuff ;
               fsData.copy( fsStats.device, fssLEN ) ;
            }
            ifsIn.close () ;     // close the temp file
         }
      }

      if ( fsStats.device[0] != NULLCHAR )
      {
         //* If device is a virtual (gvfs) filesystem *
         if ( (fsData.find( gvfsDriver )) == ZERO )
         {
            usbDevice usbDev ;
            if ( (this->GetFilesystemStats_gvfs ( trgPath, usbDev )) == OK )
            {
               fsData = usbDev.uri ;
               short indx = (fsData.findlast( L'_' )) + 1 ;
               if ( indx >= ZERO )
                  fsData.gscanf( indx, "%64[^/]", fsStats.uuid ) ;
               fsData.compose( "[%04hx:%04hx]", &usbDev.vendor, &usbDev.product ) ;
               fsData.copy( fsStats.label, fssLEN ) ;
               fsData = usbDev.fstype ;
               fsData.copy( fsStats.fsType, fssLEN ) ;
               fsData = usbDev.mntpath ;
               fsData.copy( fsStats.mntPoint, MAX_PATH ) ;
               fsData = usbDev.devpath ;
               fsData.copy( fsStats.device, fssLEN ) ;
               
               //* Determine whether caller's target filespec is a mountpoint.*
               //* Referenced when mounting/unmounting a block storage device.*
               fsStats.isMountpoint = bool((fsData.compare( trgPath )) == ZERO) ;
               fsStats.isMounted = usbDev.mounted ;
            }
            else 
            {
               //* If device is an optical (DVD/CD/Blu-Ray) drive *
               usbDevice usbDev ;
               if ( (this->GetFilesystemStats_opti ( trgPath, usbDev )) == OK )
               {
                  // Programmer's Note: uuid may not be available for optical 
                  // drives which are not mounted as data drives.
                  fsData = usbDev.uuid ;
                  fsData.copy( fsStats.uuid, fssLEN ) ;
                  fsData = usbDev.label ;
                  fsData.copy( fsStats.label, fssLEN ) ;
                  fsData = usbDev.fstype ;
                  fsData.copy( fsStats.fsType, fssLEN ) ;
                  fsData = usbDev.mntpath ;
                  fsData.copy( fsStats.mntPoint, MAX_PATH ) ;
                  fsData = usbDev.devpath ;
                  fsData.copy( fsStats.device, fssLEN ) ;
                  fsStats.isMountpoint = bool((fsData.compare( trgPath )) == ZERO) ;
                  fsStats.isMounted = usbDev.mounted ;
               }
               //* Neither mtp nor optical drive.      *
               //* Probably 'tmpfs' logical filesystem.*
               else
               {
                  fsData = gvfsDriver ;
                  fsData.copy( fsStats.fsType, fssLEN ) ;
               }
            }

            //* Compensate for gvfs (Android) bug. (see above) *
            if ( fsStats.fsTypeCode > GVFS_FSTYPE_ERR )
               fsStats.fsTypeCode = GVFS_FSTYPE ;
         }

         //* Logical filesystems are neither block devices nor virtual  *
         //* filesystems, and therefore contain only sparse information.*
         else if ( ((fsData.find( "tmpfs" )) == ZERO) ||
                   ((fsData.find( "proc" )) == ZERO) )
         {
            fsData.copy( fsStats.fsType, fssLEN ) ;
            // UUID, Mountpoint and Label are not applicable.
            // 'stat' and 'df' commands above may have captured some info.
         }

         //* Else, assume a block device filesystem.*
         else
         {
            //* Call the 'lsblk' command to obtain UUID, filesystem type,   *
            //* label and mountpoint strings.                               *
            //* Note that the mountpoint string may include whitespace, but *
            //* we assume that the UUID, filesystem type and label do not.  *
            const char* lsblkTemplate = 
                  "lsblk --output=UUID,FSTYPE,LABEL,MOUNTPOINT \"%s\" 1>\"%s\" 2>/dev/null" ;
            snprintf ( cmdBuff, (MAX_PATH*3), lsblkTemplate, 
                       fsStats.device, tmpPath.ustr() ) ;
            //* Initialize the target fields ('fsType' previously initialized).*
            fsStats.uuid[0]     =
            fsStats.label[0]    =
            fsStats.mntPoint[0] = NULLCHAR ;

            if ( (status = this->Systemcall ( cmdBuff )) == OK )
            {
               ifsIn.open( tmpPath.ustr(), ifstream::in ) ;
               if ( ifsIn.is_open() )
               {  //* Read and discard the header line.*
                  ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;

                  //* Read and scan the data line.*
                  ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
                  fsData = cmdBuff ;
                  short indx = fsData.find( L' ' ) ;
                  gString gs( fsData.gstr(), indx ) ;
                  gs.copy( fsStats.uuid, fssLEN ) ;   // extract the UUID
                  indx = fsData.scan( indx ) ;        // discard UUID from source buffer
                  fsData.shiftChars( -indx ) ;
                  indx = fsData.find( L' ' ) ;
                  gs.loadChars( fsData.gstr(), indx ) ;
                  gs.copy( fsStats.fsType, fssLEN ) ; // extract filesystem type
                  indx = fsData.scan( indx ) ;        // discard fstype from source buffer
                  fsData.shiftChars( -indx ) ;
                  indx = fsData.find( fSLASH ) ;
                  if ( indx > ZERO )                  // extract the label (if any)
                  {
                     gs.loadChars( fsData.gstr(), (indx - 1) ) ;
                     gs.copy( fsStats.label, fssLEN ) ;
                     fsData.shiftChars( -(indx) ) ;   // discard label
                  }
                  fsData.copy( fsStats.mntPoint, MAX_PATH ) ;

                  //* Determine whether caller's target filespec is a mountpoint.*
                  //* Referenced when mounting/unmounting a block storage device.*
                  fsStats.isMountpoint = bool((fsData.compare( trgPath )) == ZERO) ;

                  //* Because good data were returned by the system call, it   *
                  //* is assumed that the device is mounted. (see notes above).*
                  fsStats.isMounted = true ;

                  ifsIn.close () ;     // close the temp file
               }
            }
         }
      }

      //* If we got nothing from the SELinux Security Context field *
      if ( fsStats.seLinux[0] == '?' || fsStats.seLinux[0] == NULLCHAR )
      {
         snprintf ( cmdBuff, (MAX_PATH*3), "%s \"%s\" 1>\"%s\" 2>/dev/null", 
                    selCmd, escPath.ustr(), tmpPath.ustr() ) ;
         if ( (this->Systemcall ( cmdBuff )) == OK )
         {
            //* Read the only line in the file *
            ifsIn.open ( tmpPath.ustr(), ifstream::in ) ;
            if ( ifsIn.is_open() )
            {
               ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
               ifsIn.close () ;     // close the temp file
               char bitBucket[fssLEN] ;
               fsData = cmdBuff ;
               swscanf ( fsData.gstr(), L"%63s %63s %63s %63s", 
                         bitBucket, bitBucket, bitBucket, fsStats.seLinux ) ;
            }
         }
      }
   }
   else     // system call unsuccessful, send up a flare
   {
      status = this->recentErrno = errno ;
      this->DebugLog ( "In FilesystemStats: stat of file system failed", 
                       this->recentErrno, trgPath ) ;
   }
   this->DeleteFile ( tmpPath.ustr() ) ;  // delete the temp file
   return status ;

}  //* GetFilesystemStats() *

short FMgr::GetFilesystemStats ( const gString& trgPath, fileSystemStats& fsStats )
{

   return ( (this->GetFilesystemStats ( trgPath.ustr(), fsStats )) ) ;

}  //* End GetFilesystemStats() *

//*************************
//*     FilesystemID      *
//*************************
//******************************************************************************
//* Read the ID of the filesystem containing specified path/filename.          *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*          trgID  : (by reference) receives filesystem ID  (hex)             *
//*          trgType: (by reference) receives filesystem type  (hex)           *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: As mentioned elsewhere, the C library stat() function   *
//* cannot return filesystem information. We call the coreutils 'stat' utility *
//* to get the info. Unfortunately, this is rather slow. If we can find a way  *
//* around this, we will implement it in a later release.                      *
//*                                                                            *
//* '%i' reports the filesystem ID (hex value)                                 *
//*      Unfortunately, all logical and virtual filesystems report zero value. *
//* '%t' reports the filesystem type (hex value)                               *
//******************************************************************************

short FMgr::FilesystemID ( const char* trgPath, UINT64& trgID, UINT& trgType )
{
   static const char fssCmd[] = "stat -f --printf=\"%i %t \n\"" ;
   gString tmpPath ;
   char    cmdBuff[MAX_PATH*2] ;
   short   status ;

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   gString escPath( trgPath ) ;
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   this->CreateTempname ( tmpPath ) ;
   snprintf ( cmdBuff, (MAX_PATH*2), "%s \"%s\" 1>\"%s\" 2>/dev/null", 
              fssCmd, escPath.ustr(), tmpPath.ustr() );
   if ( (status = this->Systemcall ( cmdBuff )) == OK )
   {
      //* Read the only line in the file *
      ifstream ifsIn ( tmpPath.ustr(), ifstream::in ) ;
      ifsIn.getline ( cmdBuff, (MAX_PATH*2) ) ;
      ifsIn.close () ;     // close the temp file

      gString fsData( cmdBuff ) ;
      if ( (fsData.gscanf( L"%LX %X", &trgID, &trgType )) != 2 )
         status = ERR ;
   }
   if ( status != OK )
   {
      trgID   = UINT64(-1) ;
      trgType = UINT(-1) ;
   }
   this->DeleteFile ( tmpPath.ustr() ) ;  // delete the temp file
   return status ;

}  //* End FilesystemID() *

//*************************
//*     FilesystemID      *
//*************************
//******************************************************************************
//* Returns the path of the device driver or other unique identifier           *
//* associated with the filesystem which contains the specified file.          *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*                   (normally the filespec of a directory)                   *
//*          devPath: (by reference) receives the requested filespec:          *
//*                   source: filespec of associated device driver             *
//*                   target: filespec of mountpoint                           *
//*          source : (optional, 'true' by default)                            *
//*                   'true' i.e. "source"  : return the device driver         *
//*                         associated with the filesystem containing the      *
//*                         specified target path                              *
//*                   'false' i.e. "target" : return the mountpoint of the     *
//*                         filesystem which contains the target path          *
//*                                                                            *
//* Returns: OK if 'devPath' contains:                                         *
//*             - device driver for block device                               *
//*             - device driver for virtual (MTP/GVFS) device                  *
//*             - inode number for logical filesystems (tmpfs, proc, etc.)
//*             - mountpoint filespec (source == false)                        *
//*          ERR if system call fails (unlikely)                               *
//******************************************************************************
//* Note: For non-block-device drivers, when 'source != false':                *
//*       'df' may not report a unique identifier.                             *
//*  1) 'df' will report a virtual device driver token: e.g. "gvfs-fuse"       *
//*     For gvfs virtual filesystems we must do a more expensive test to       *
//*     obtain the actual device driver filespec.                              *
//*  2) 'df' will report a logical device driver token: e.g. "tmpfs"           *
//*     However, logical devices are placeholders, and are not associated      *
//*     with a real device driver. Because these filesystems are typically a   *
//*     single directory name (NOT a tree structure), the inode number of      *
//*     the target is returned. While it is not always true that a tmpfs       *
//*     filesystem is a single directory name (/run, /run/user and /run/media  *
//*     appear to be the same filesystem), these are not places to store data, *
//*     but only placeholder directories for mounting other filesystems.       *
//******************************************************************************

short FMgr::FilesystemID ( const gString& trgPath, gString& devPath, bool source )
{
   const char *dfTemplate = "df --output=%s \"%s\" 1>\"%s\" 2>/dev/null",
              *src = "source",
              *trg = "target" ;
   gString escPath = trgPath, tmpPath ;
   char    cmdBuff[MAX_PATH*3] ;
   short   status = ERR ;

   devPath.clear() ;       // clear caller's buffer

   //* Create a temp file to capture the output of the system call.*
   this->CreateTempname ( tmpPath ) ;

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   //* Construct the system command.*
   snprintf ( cmdBuff, (MAX_PATH*3), dfTemplate, (source ? src : trg), 
              escPath.ustr(), tmpPath.ustr() ) ;

   //* Execute the system command.*
   if ( (status = this->Systemcall ( cmdBuff )) == OK )
   {
      //* Open the temp file, then read *
      //* and discard the header line.  *
      ifstream ifsIn ( tmpPath.ustr(), ifstream::in ) ;
      if ( ifsIn.is_open() )
      {
         ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;

         //* Read and scan the data line.*
         ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
         if ( ifsIn.gcount() > ZERO )
         {
            devPath = cmdBuff ;
            if ( source )        // device driver reported
            {
               //* If block-device driver specified *
               if ( (devPath.find( "/dev" )) == ZERO )
                  status = OK ;
               //* If target is a virtual (gvfs) filesystem *
               else if ( (devPath.compare( gvfsDriver )) == ZERO )
               {
                  usbDevice usbDev ;
                  if ( (this->GetFilesystemStats_gvfs ( trgPath.ustr(), usbDev )) == OK )
                  {
                     devPath = usbDev.devpath ;
                     status = OK ;
                  }
               }
               //* Logical filesystem (e.g. 'tmpfs' 'proc') or unknown type.*
               // Both major and minor device type are zero (0), but        *
               //* The inode number _should be_ unique for these.           *
               else
               {
                  tnFName tnf ;
                  this->GetFileStats ( tnf, trgPath.ustr() ) ;
                  devPath.compose( "%llu", &tnf.rawStats.st_ino ) ;
                     status = OK ;
               }
            }
            else                 // mountpoint reported
               status = OK ;
         }
         ifsIn.close () ;     // close the temp file
      }
   }

   this->DeleteTempname ( tmpPath ) ;  // delete the temp file

   return status ;

}  //* End FilesystemID() *

//*************************
//*    FilesystemMatch    *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* ---------------                                                            *
//* Determine whether the specified directory is on the same filesystem as the *
//* base directory of the current scan.                                        *
//* Compares the device driver for the filesystem which contains the specified *
//* directory with the device driver for the base directory for the current    *
//* scan.                                                                      *
//*                                                                            *
//* Should be called only for target directories on one of the "special paths".*
//* (see notes below)                                                          *
//*                                                                            *
//* Note: All data under the "/home" mountpoint is assumed to belong to the    *
//*       same filesystem (no external filesystems mounted within the "/home"  *
//*       filesystem. This design decision is made in the interest of speed.   *
//*       Not only is this where most file manipulation takes place, only a    *
//*       non-standard installation would mount anything under "/home".        *
//*                                                                            *
//* Input  : trgPath: full path/filename of directory under test               *
//*                                                                            *
//* Returns: 'true'  if same filesystem                                        *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* Because the filesystem tests are very expensive, we perform them only for  *
//* directory paths that are likely to contain mountpoints. While a filesystem *
//* may be mounted anywhere on the the directory tree, there are certain       *
//* places that are conventionally designated as mountpoints.                  *
//*                                                                            *
//* Directories identified for special processing:  (see "/proc/mounts")       *
//* ----------------------------------------------                             *
//* / (root)                                                                   *
//*    All filesystems are mounted relative to the root directory; however,    *
//*    the root's filesystem contains not only a majority of the basic system  *
//*    files, it also hosts the mountpoints for a number of special-purpose    *
//*    filesystems which are not physically block storage devices.             *
//*    These are designated as "tmpfs", or some other special filesystem.      *
//*      Examples:      Filesystem               Mountpoint                    *
//*      ---------      ----------               ----------                    *
//*      /run           "tmpfs"                  /run                          *
//*      /dev           "devtmpfs"               /dev                          *
//*      /proc          "proc"                   /proc                         *
//*      /sys           "sysfs"                  /sys                          *
//*      /tmp           "tmpfs"                  /tmp                          *
//*                                                                            *
//*    Many other subdirectories in the root directory are actually mounted    *
//*    within the root filesystem.                                             *
//*      Examples:      Filesystem               Mountpoint                    *
//*      ---------      ----------               ----------                    *
//*      /etc           "/dev/mapper/LVM-root"   /                             *
//*      /lost+found    "/dev/mapper/LVM-root"   /                             *
//*      /media         "/dev/mapper/LVM-root"   /                             *
//*      /mnt           "/dev/mapper/LVM-root"   /                             *
//*      /opt           "/dev/mapper/LVM-root"   /                             *
//*      /root          "/dev/mapper/LVM-root"   /                             *
//*      /srv           "/dev/mapper/LVM-root"   /                             *
//*      /usr           "/dev/mapper/LVM-root"   /                             *
//*      /var           "/dev/mapper/LVM-root"   /                             *
//*                                                                            *
//*      The filesystems in the root directory which are independently-mounted *
//*      filesystems include:                                                  *
//*      /boot          "/dev/sda1"              /boot                         *
//*      /home          "/dev/mapper/LVM-home"   /home                         *
//*      Most other mounted filesystems are to be found at one of the          *
//*      conventional mountpoint directories described below                   *
//*                                                                            *
//* /run                                                                       *
//*    This directory is intended for transient information that is added      *
//*    AFTER the system is booted. It is emptied during the boot process,      *
//*    then programs and processes add things to it. User-level accounts       *
//*    cannot write into this directory except in user-specific directories    *
//*    such as:  /run/media/${USER}                                            *
//*                                                                            *
//* /run/media/${USER}                                                         *
//*    This directory will contain most of the locally-mounted i.e. user-owned *
//*    drives (SD-cards, USB FLASH drives, external hard drives, etc.)         *
//*    GNOME: The "udisks" (udisks2) management software directs mounts        *
//*    to this directory instead of the older target: "/var/run/...".          *
//*                                                                            *
//* /run/user/${UID}/gvfs/...   (smartphones)                                  *
//*    User-id will be 1000 on any fairly recent single user system.           *
//*    The GNOME Virtual Filesystem (GVFS) is used to manage mountable local   *
//*    and network filesystems. Access requires both GVfs and a runtime library*
//*    (or 'gio' utility) and can use the FUSE interface for direct mounting.  *
//*    Some older systems still use "/var/run/user/$UID/gvfs" as the smartphone*
//*    mountpoint. A given system may use either mount target but not both.    *
//*                                                                            *
//*    Examples:                         Filesystem      Mountpoint            *
//*    ---------                         ----------      ----------            *
//*    /run/user/1000                    "tmpfs"        /run                   *
//*    /run/user/1000/gvfs               "gvfsd-fuse"   /run/user/1000/gvfs    *
//*    /run/user/1000/gvfs/mtp:host=...  "gvfsd-fuse"   /run/user/1000/gvfs    *
//*                                                                            *
//*    Please note that devices mounted under "/run/user/$UID/gvfs" are        *
//*    named dynamically: mtp:host=mfgr_productA_productB_serialnumber         *
//*                                                                            *
//* /media                                                                     *
//*    This directory may contain multiple, dynamically-allocated sub-         *
//*    directories such as "/media/floppy", "/media/cdrom", "/media/zip".      *
//*                                                                            *
//* /mnt                                                                       *
//*    On older systems, this was the standard place to mount external devices.*
//*    For instance: "/mnt/cdrom" or "/mnt/floppy".                            *
//*    However, according to the Filesystem Hierarchy Standard (FSSTND),       *
//*    this directory is now reserved for use by the sys admin for temporarily *
//*    mounting a SINGLE filesystem while performing maintenance.              *
//*                                                                            *
//* -  -  -  -  -                                                              *
//* See the Linux Documentation Project:                                       *
//*      <http://www.tldp.org/LDP/Linux-Filesystem-Hierarchy/html/>            *
//* for more information about how the system directory tree is organized.     *
//* See also draft standard v:3.0 at:                                          *
//*      <http://www.linuxbase.org/betaspecs/fhs/fhs.html>                     *
//*                                                                            *
//******************************************************************************

bool FMgr::FilesystemMatch ( const gString& trgPath )
{
   const char *devDriver    = "/dev/",
              *homePath     = "/home" ;
   gString devPath ;
   bool match = true ;        // return value

   //* Assume that no external filesystems are mounted    *
   //* within the /home filesystem. If they are, they     *
   //* will be local physical drives, slow but manageable.*
   if ( (trgPath.find( homePath )) != ZERO )
   {
      //* Get device driver for target.*
      if ( (this->FilesystemID ( trgPath, devPath )) == OK )
      {
         //* If device drivers are identical, we're done.*
         match = bool((devPath.compare( this->irID )) == ZERO) ;

         //* Else, if neither the base nor the target     *
         //* are associated with an actual device driver, *
         //* then declare a match which allows the scan   *
         //* recursion to continue through the various    *
         //* special-purpose logical filesystems.         *
         if ( match == false )
         {
            if ( (devPath.find( devDriver )) != ZERO )
            {
               devPath = this->irID ;
               if ( (devPath.find( devDriver )) != ZERO )
                  match = true ;
            }
         }
      }
   }
   return match ;

}  //* End FilesystemMatch() *

//*****************************
//* TargetFilesystemFreespace *
//*****************************
//******************************************************************************
//* Returns the amount of free storage space on the file system which contains *
//* the file or directory specified.                                           *
//*                                                                            *
//* Input  : targPath : path specification to any file/directory on            *
//*                     target file system                                     *
//*          fsysID   : (initial value ignored)                                *
//*                     on return, contains file system ID code                *
//*          freeBytes: (initial value ignored)                                *
//*                     on return, contains number of free storage bytes       *
//*                                                                            *
//* Returns: OK if successful or system's error status if stat failed          *
//*           if not 'OK', fsysID and freeBytes are undefined                  *
//******************************************************************************
//* Programmer's Note: Some removable media are slow to respond or have other  *
//* problems reporting what we request of them, so we include a retry loop to  *
//* to increase the odds of a successful read.                                 *
//*                                                                            *
//* Programmer's Note: It might be more efficient and/or more accurate to use  *
//* the 'df' or 'du' command. Think about this...                              *
//* 'df' actually does work:                                                   *
//*  a) report used and free blocks                                            *
//*  b) report used and free MB, GB, etc.                                      *
//*  c) report used and free MiB, GiB, etc.                                    *
//* However, it does not report block size, and if not in bytes, the resolution*
//* is minimal.                                                                *
//******************************************************************************

short FMgr::TargetFilesystemFreespace ( const char* targPath, 
                                        UINT64& fsysID, UINT64& freeBytes )
{
   static const char fssCmd[] = "stat -f --printf=\"%a %S %i \n\"" ;
   gString tmpPath ;
   char  cmdBuff[MAX_PATH*2] ;
   short status, retry = 3 ;

   //* Escape any "special characters" which *
   //* might be misinterpreted by the shell. *
   gString escPath( targPath ) ;
   this->EscapeSpecialChars ( escPath, escMIN ) ;

   //* Redirect standard-out (sdtout) to a temporary file. *
   this->CreateTempname ( tmpPath ) ;  // (it is unlikely but possible that this will fail)
   snprintf ( cmdBuff, (MAX_PATH*2), "%s \"%s\" 1>\"%s\" 2>/dev/null", 
             fssCmd, escPath.ustr(), tmpPath.ustr() );

   do
   { status = this->Systemcall ( cmdBuff ) ; }
   while ( (status != OK) && (--retry > ZERO) ) ;

   if ( status == OK )
   {
      //* Read the only line in the file *
      ifstream ifsIn ( tmpPath.ustr(), ifstream::in ) ;
      ifsIn.getline ( cmdBuff, (MAX_PATH*4) ) ;
      ifsIn.close () ;     // close the temp file

      UINT64 blockAvail, fblockSize ;
      gString fsData( cmdBuff ) ;
      swscanf ( fsData.gstr(), L"%llu %llu %llX", 
         &blockAvail, &fblockSize, &fsysID ) ;
      freeBytes = fblockSize * blockAvail ;
   }
   else     // system call unsuccessful, send up a flare
   {
      this->DebugLog ( "TargetFilesystemFreespace: stat of file system failed", 
                       status, targPath ) ;
   }
   this->DeleteFile ( tmpPath.ustr() ) ;  // delete the temp file
   return status ;

}  //* End TargetFilesystemFreespace() *

//*************************
//*     isMountpoint      *
//*************************
//******************************************************************************
//* Determine whether the specified filespec refers to a mountpoint directory. *
//* This method uses the 'mountpoint' system utility.                          *
//*                                                                            *
//* Input  : trgPath : target filespec                                         *
//*                                                                            *
//* Returns: 'true'  if target is a mountpoint directory                       *
//*          'false' if target is not a mountpoint                             *
//*                  (or is not a directory, or does not exist)                *
//******************************************************************************
//* Programmer's Note: In non-English locales, the 'mountpoint' utility may    *
//* return data in another language; in which case, this method would always   *
//* return 'false'. (This has not been tested.)                                *
//*                                                                            *
//* Programmer's Note: The 'mountpoint' utility will fail when testing for an  *
//* MTP mountpoint. For this reason, we call the private gvfs-specific         *
//* method for MTP/GVFS mounts.                                                *
//******************************************************************************

bool FMgr::isMountpoint ( const char* trgPath ) 
{
   tnFName fStats ;              // stats of trgPath
   bool status = false ;         // return value

   //* If the target is on an MTP/GVFS filesystem, call *
   //* the gvfs-specific method. Note the early return. *
   gString gsTrg( trgPath ) ;
   if ( (this->isGvfsPath ( gsTrg )) )
      return ( (this->isMountpoint_gvfs ( gsTrg )) ) ;


   if ( ((this->GetFileStats ( fStats, trgPath )) == OK)
         && (fStats.fType == fmDIR_TYPE) )
   {
      gString tmpPath ;
      this->CreateTempname ( tmpPath ) ;  // create a temporary file

      //* Escape any "special characters" which *
      //* might be misinterpreted by the shell. *
      gString escPath( trgPath ) ;
      this->EscapeSpecialChars ( escPath, escMIN ) ;

      //* Create and execute the system command.*
      char cmdBuff[gsDFLTBYTES] ;
      snprintf ( cmdBuff, gsDFLTBYTES, "mountpoint \"%s\" 1>\"%s\" 2>/dev/null",
                 escPath.ustr(), tmpPath.ustr() ) ;
      if ( (this->Systemcall ( cmdBuff, true )) == OK )
      {
         //* Read the only line in the file *
         ifstream ifs ( tmpPath.ustr(), ifstream::in ) ;
         ifs.getline ( cmdBuff, gsDFLTBYTES ) ;
         ifs.close () ;                   // close the temp file

         gString gs( cmdBuff ) ;
         if ( (gs.find( L"is a mountpoint" )) >= ZERO )
            status = true ;
      }
      this->DeleteTempname ( tmpPath ) ;  // delete the temp file
   }
   return status ;

}  //* End isMountpoint() *

//*************************
//*      isMountpath      *
//*************************
//******************************************************************************
//* PRIVATE METHOD                                                             *
//* --------------                                                             *
//* Determine whether the specified filespec lies on one of the common         *
//* mountpoint paths. Used to determine whether to perform filesystem tests    *
//* during directory-tree scans.                                               *
//*                                                                            *
//* Note that the test returns 'true' only if 'trgPath' references a directory *
//* that is ABOVE the actual mountpoint of an external device. If the entire   *
//* scan is within the external device's filesystem, then there is no need to  *
//* perform additional filesystem tests during the scan.                       *
//*                                                                            *
//* Note that whereas, technically a filesystem may be mounted ANYWHERE at or  *
//* below the root directory, we test only for the common directory paths on   *
//* which filesystems are mounted. The list of potential mount paths is taken  *
//* from the Filesystem Hierarchy Standard (FSSTND) maintained at:             *
//*           <http://www.linuxbase.org/betaspecs/fhs/fhs.html>                *
//* For more details, see method header for FilesysteMatch() (above).          *
//*                                                                            *
//* See also the public method, isMountpoint() which determines whether        *
//* specified filespec IS a mountpoint.                                        *
//*                                                                            *
//* Input  : trgPath : target filespec                                         *
//*                                                                            *
//* Returns: 'true'  if target is on one of the common mount-directory paths   *
//*          'false' if target is not on one of the mount paths                *
//******************************************************************************

bool FMgr::isMountpath ( const gString& trgPath ) 
{
   const char *runPath   = "/run",
              *runmPath  = "/run/media",
              *runuPath  = "/run/user",
              *mediaPath = "/media",
              *mntPath   = "/mnt" ;
   bool status = false ;         // return value

   if ( (trgPath.find( runPath )) == ZERO )
   {
      if ( ((trgPath.compare( runPath ))  == ZERO) ||
           ((trgPath.compare( runmPath )) == ZERO) ||
           ((trgPath.compare( runuPath )) == ZERO) )
      {
         status = true ;
      }
      else if ( (trgPath.find( runmPath )) == ZERO )
      {
         gString unameBase( "%s/%s", runmPath, this->userInfo.userName ) ;
         if ( trgPath == unameBase )
            status = true ;
      }
      else if ( (trgPath.find( runuPath )) == ZERO )
      {
         gString gvfsBase( "%s/%u", runuPath, &this->userInfo.userID ) ;
         if ( trgPath == gvfsBase )
            status = true ;
         else
         {
            gvfsBase.append( L"/gvfs" ) ;
            if ( trgPath == gvfsBase )
               status = true ;
         }
      }
   }
   else if ( ((trgPath.compare( mediaPath )) == ZERO) ||
             ((trgPath.compare( mntPath )) == ZERO) )
   {
      status = true ;
   }
   return status ;

}  //* End isMountpath() *

//*************************
//*      isRootScan       *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* When scanning the directory tree, the root directory is a special case.    *
//* Test whether the current-working-directory (base directory of tree scan)   *
//* is the root directory.                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if current-working-directory is the "root" directory       *
//*          'false' otherwise                                                 *
//******************************************************************************

bool FMgr::isRootScan ( void )
{

   return ( bool((this->currDir[0] == '/') && (this->currDir[1] == NULLCHAR)) ) ;

}  //* End isRootScan()

//*************************
//*      Systemcall       *
//*************************
//******************************************************************************
//* Call the 'system' C-language library function with the specified command.  *
//*                                                                            *
//* This method (optionally) captures the actual exit code of the called       *
//* program and returns it to caller. This is done through two layers of       *
//* indirection which allows us to retrieve the exit code in our own           *
//* environment.                                                               *
//* The capture of the exit code is optional because if the caller doesn't     *
//* need the exit code, then we can omit one layer of indirection and used the *
//* 'system' library call instead.                                             *
//*                                                                            *
//* Input : cmd    : command string with program name and options              *
//*         retExit: (optional, 'false' by default)                            *
//*                  if 'false', do not capture the exit code of the called    *
//*                              process.                                      *
//*                                                                            *
//* Returns: if specified, the exit code of called external program            *
//*          Else, OK (ZERO)                                                   *
//*          [UNTIL FULLY IMPLEMENTED, RETURNS VALUE FROM system()]            *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* -- The system() function is quite useful, but doesn't give us the exit     *
//*    code of the called program. Instead, it gives us the termination code   *
//*    of the child process which executes the program--which is useless       *
//*    unless 'sh' was not found (unlikely), or if the process was killed      *
//*    before completion.                                                      *
//*    -- 'system()' returns the status of the shell process:                  *
//*       '-1' == error, and errno contains the exit code of the child process.*
//*       Other exit codes are related to the termination of the child process,*
//*       and ARE NOT related to the exit code of the program running in that  *
//*       process.                                                             *
//* -- Environment:                                                            *
//*    -- A child process typically inherits a COPY of the parent's            *
//*       environment, meaning that it is difficult for the child process to   *
//*       return information to the parent. For an example of this, see what   *
//*       we had to do to get the application to exit to a directory other     *
//*       than the one it started in. (painful)                                *
//*    -- The 'source' builtin command is used to execute the desired program  *
//*       within the CURRENT environment, which means that the exit code of    *
//*       the target program may be stored in the environment and retrieved    *
//*       by the process which called the external process.                    *
//*          SYSCALL_EXITCODE=source 'cmd'                                     *
//*       or more formally:                                                    *
//*          SYSCALL_EXITCODE="$?"  where $? is the environment variable which *
//*          contains the exit code of the most recently exited program.       *
//*       Note that 'source' cannot execute a binary file directly. Instead    *
//*       it executes a script which can contain invocations of binary file(s).*
//*       The last program executed leaves its exit code in "$?" where the     *
//*       script can find it.                                                  *
//*    -- Numeric values stored in the environment are unsigned 8-bit values;  *
//*       therefore, program exit codes of -1, -257 etc. will each yield 255.  *
//*       Program exit codes of 0, 256 etc. will each yield 0.                 *
//*       Thus, even though C/C++ applications exit with an int value AND      *
//*       'errno' is an int value, the environment will muck it up unless we   *
//*       store the value as a string.                                         *
//*                                                                            *
//* -- We don't want to use one of the 'exec' functions because it requires    *
//*    extra high-level work to format the arguments. Unfortunately, we may    *
//*    have to do it because system() doesn't give us much control.            *
//*                                                                            *
//* -- See also the 'popen()' and 'pclose()' functions which create pipes to   *
//*    the called program for stdin and stdout. Although this _seems_ useful,  *
//*    it really isn't because it's hard to predict what might come through    *
//*    those streams.                                                          *
//*                                                                            *
//* Who Calls This Method?                                                     *
//* ======================                                                     *
//* CopyFile()                'cp' command                 FMgr.cpp            *
//* FilesystemStats()         'stat' command               FMgrStats.cpp       *
//*                           'df' command                                     *
//*                           'lsblk' command                                  *
//*                           'ls' command                                     *
//* FilesystemID()            'stat' command                                   *
//* TargetFilesysFreespace()  'stat' command                                   *
//* isMountpoint()            'mountpoint' command                             *
//* tnAllocLogEntry()          write to debug log        FMgrTree.cpp          *
//* odExtractOD_Content()     'unzip' command            FileDlgUtil1.cpp      *
//* gfIsRegTextfile()         'file' command                                   *
//* gfConstructFilenameList() 'ls' command                                     *
//* gsScan()                  'grep' command                                   *
//* cfPretest()               'diff' command                                   *
//* CompareFiles()            'diff' command                                   *
//* vfcDecodeArchive()        'tar' command              FileDlgContext.cpp    *
//* acCreateTarArchive()      'tar' command              FileDlgBackup.cpp     *
//* acCreateZipArchive()      'zip' command                                    *
//* aeExpandTarArchive()      'tar' command                                    *
//* aeExpandZipArchive()      'unzip' command                                  *
//* ArchiveSummary()          'unzip' command                                  *
//*                           'tar' command                                    *
//*                                                                            *
//* For most of these calls, if it fails, the called utility produces an error,*
//* the caller will see it while parsing the data captured through the call.   *
//*                                                                            *
//******************************************************************************

short FMgr::Systemcall ( const char* cmd, bool retExit )
{
   short exitCode = ZERO ;

   exitCode = system ( cmd ) ;

   return exitCode ;

}  //* End Systemcall() *

//*************************
//*  EscapeSpecialChars   *
//*************************
//******************************************************************************
//* Scan the source string (typically a filespec). If the string contains any  *
//* characters of the specified character group (enum escGroup) 'escape' them  *
//* with a backslash character.                                                *
//*                                                                            *
//* Do not use this method for URL (Universal Resource Locator) specs.         *
//* URL specifications require a specific protocol which is not handled here.  *
//*                                                                            *
//* Input  : pathSpec : (by reference) source text data                        *
//*                     on return the data has been 'escaped' accoring to the  *
//*                     specified criteria                                     *
//*          group    : (member of enum escGroup) specifies the group of       *
//*                     characters to be 'escaped'                             *
//*          wch      : (optional, NULLCHAR by default)                        *
//*                     if 'group' parameter == escWCHAR, then 'wch' specifies *
//*                     a single character (NOT nullchar) to be 'escaped'      *
//*                                                                            *
//* Returns: number of characters 'escaped'                                    *
//******************************************************************************
//* Special Case Testing for '\' as the special character:                     *
//* 1) If the incomming data contains a '\', we must test whether:             *
//*    a) The '\' itself has already been escaped,                             *
//*    b) The '\' is escaping some other special character,                    *
//*    c) The '\' is a special character which needs to be escaped.            *
//* 2) If data contains an escaped special character, but caller intended      *
//*    for it to be two SEPERATE special characters, caller will be            *
//*    disappointed that we could not read his/her/its mind. (sorry about that)*
//*                                                                            *
//* -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -  *
//* For more information, see "Quoting Special Characters in BASH.odt".        *
//******************************************************************************

short FMgr::EscapeSpecialChars ( gString& pathSpec, escGroup group, wchar_t wch )
{
   //* List of critical "special" characters.*
   const short   minSpecialCount = 4 ;
   const wchar_t minSpecialChars[minSpecialCount] = 
   { L'\\', L'"', L'`', L'$' } ;

   //* List of Linux/UNIX shell "special" characters.*
   const short   linuxSpecialCount = 13 ;
   const wchar_t linuxSpecialChars[linuxSpecialCount] = 
   { L'\\', L'"', L'\'', L'?', L':', L';', L'&', L'>', L'<', L'|', L'*', L'`', L'$' } ;

   //* List of all known "special" characters which might be interpreted *
   //* by the shell or shell programs as commands rather than plain text.*
   const short   maxSpecialCount = 27 ;
   const wchar_t maxSpecialChars[maxSpecialCount] = 
   { L'\\', L'"', L'\'', L'?', L':', L';', L'&', L'>', L'<', L'|', L'*', L'`', L'$',
     L'#', L'+', L'=', L'%', L'!', L'~', L'{', L'}', L'@', L'(', L')', L'[', L']', L' ' } ;

   const wchar_t wDQ = L'"',
                 wBS = L'\\' ;

   const wchar_t *wPtr ;            // pointer to wchar_t array
   short wCount,                    // number of elements in wchar_t array
         indx,                      // text index
         specChars = ZERO ;         // return value

   //* Point to the reference array and establish number of array elements.*
   if ( group == escMIN )
   { wPtr = minSpecialChars ;   wCount = minSpecialCount ; }
   else if ( group == escLINUX )
   { wPtr = linuxSpecialChars ; wCount = linuxSpecialCount ; }
   else if ( group == escMAX )
   { wPtr = maxSpecialChars ;   wCount = maxSpecialCount ; }
   else if ( group == escDQ )
   { wPtr = &wDQ ;              wCount = 1 ; }
   else if ( group == escWCHAR )
   {  //* Do not allow escaping of null terminator.*
      wPtr = &wch ;
      wCount = (wch != NULLCHAR) ? 1 : 0 ;
   }

   for ( short specIndx = ZERO ; specIndx < wCount ; ++specIndx )
   {
      indx = ZERO ;
      do
      {
         if ( (indx = pathSpec.find( wPtr[specIndx], indx )) >= ZERO )
         {
            //* Special character == '\'. (see note above) *
            if ( (pathSpec.gstr()[indx]) == wBS )
            {
               bool insert = true ;
               //* If escape character is itself escaped *
               if ( (indx > ZERO) && ((pathSpec.gstr()[indx - 1]) == wBS) )
               { ++indx ; insert = false ; }
               //* If '\' is being used as an escape character *
               if ( insert )
               {
                  for ( short i = ZERO ; i < maxSpecialCount && insert ; ++i )
                  {
                     if ( (pathSpec.gstr()[indx + 1]) == maxSpecialChars[i] )
                     { indx += 2 ; insert = false ; }
                  }
               }

               if ( insert )  // escape the escape character
               {
                  pathSpec.insert( wBS, indx ) ;
                  ++specChars ;
                  indx += 2 ;
               }
               else
                  ++indx ;
            }
            else
            {
               if ( indx == ZERO )       // first character in array
               {
                  pathSpec.insert( wBS, indx ) ;
                  ++specChars ;
                  indx += 2 ;
               }
               else // (indx > ZERO), second or later character in array
               {
                  if ( (pathSpec.gstr()[indx - 1]) != wBS )
                  {
                     pathSpec.insert( wBS, indx ) ;
                     ++specChars ;
                     indx += 2 ;
                  }
                  else
                     ++indx ;
               }
            }
         }
      }
      while ( (indx >= ZERO) && (pathSpec.gstr()[indx] != NULLCHAR) ) ;
   }

   return specChars ;

}  //* End EscapeSpecialChars() *

//*************************
//*   RemoveCharEscapes   *
//*************************
//******************************************************************************
//* Scan the source string (typically a filespec). If the string contains any  *
//* "escaped" characters i.e. preceed by a backstroke ('\'), remove the        *
//* backstrokes.                                                               *
//*                                                                            *
//* Escaped characters are typically regexp or shell "special characters".     *
//* This includes the backstroke character itself.                             *
//*                                                                            *
//* Input  : gsSrc    : (by reference) contains source text to be scanned      *
//*          retainDQ : (optional, 'false' by default)                         *
//*                     if 'true', then remove the backstroke for all escaped  *
//*                     characters EXCEPT the double-quote ( " ) character     *
//*                                                                            *
//* Returns: number of characters for which the escaping was removed           *
//******************************************************************************

short FMgr::RemoveCharEscapes ( gString& gsSrc, bool retainDQ )
{
   const wchar_t wBS = L'\\',
                 wDQ = L'"' ;
   short bsCount = ZERO ;        // return value

   if ( (gsSrc.find( wBS )) >= ZERO )
   {
      short i = ZERO ;
      while ( (i = gsSrc.find( wBS, i )) >= ZERO )
      {
         if ( retainDQ && (gsSrc.gstr()[i+1]) == wDQ ) // if an escaped '"'
         { i += 2 ; continue ; }
         else if ( (gsSrc.gstr()[i+1]) == wBS )        // if an escaped '\'
            ++i ;
         gsSrc.erase( wBS, i ) ;
         ++bsCount ;
      }
   }
   return bsCount ;

}  //* End RemoveCharEscapes() *

//*************************
//*     EnvExpansion      *
//*************************
//******************************************************************************
//* Perform environment-variable expansion or tilde ('~') expansion on the     *
//* specified path string.                                                     *
//*                                                                            *
//* Input  : gsPath  : (by reference) contains the string to be scanned        *
//*                                                                            *
//* Returns: 'true'  if expansion successful (or no expansion needed)          *
//*          'false' if unable to expand                                       *
//******************************************************************************
//* Notes on 'wordexp':                                                        *
//* The 'wordexp' function is a pretty cool, but watch out:                    *
//*  a) wordexp returns ZERO on success or WRDE_BADCHAR (2) if an invalid      *
//*     character is detected in the stream.                                   *
//*     - Note that an empty string will pass the scan, but then               *
//*       'wexp.we_wordc' will be ZERO.                                        *
//*  b) Dynamic memory allocation happens, so remember to free it.             *
//*     - If a bad character in the stream, then freeing the dynamic           *
//*       allocation will cause a segmentation fault. This is a Standard       *
//*       Library bug, so the work-around is to call 'wordfree' ONLY if        *
//*      'wordexp' returns success.                                            *
//*  c) SPACE characters delimit the parsing, so if the path contains spaces,  *
//*     then we must concatenate the resulting substrings, reinserting the     *
//*     space characters. Leading and trailing spaces are ignored.             *
//*     (We assume that a path will never contain a TAB character.)            *
//*  d) wordexp will choke on the following characters in the stream:          *
//*             & | ; < >  \n     (unless they are quoted)                     *
//*     - Parentheses and braces should appear ONLY as part of a token to be   *
//*       expanded (or if they are quoted).                                    *
//*  e) The tokens we are most likely to see are '${HOME}' and '~'.            *
//*     These are both expanded as '/home/sam' or the equivalent.              *
//******************************************************************************

bool FMgr::EnvExpansion ( gString& gsPath )
{
   bool status = false ;         // return value

   wordexp_t wexp ;              // target structure
   if ( (wordexp ( gsPath.ustr(), &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         gsPath.clear() ;
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            gsPath.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc ) // re-insert stripped spaces (see note 'c')
               gsPath.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;
      status = true ;
   }
   return status ;

}  //* End EnvExpansion() *

//*************************
//*    CreateTempname     *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file.                        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The application's temporary files live in a uniquely-named subdirectory    *
//* within the system's temporary-file directory. The path to this directory   *
//* is located in our 'tfPath' data member.                                    *
//*                                                                            *
//* The unique filename is appended to 'tfPath' and returned to caller.        *
//*                                                                            *
//******************************************************************************

bool FMgr::CreateTempname ( gString& tmpPath )
{
   bool  status = false ;

   char tn[gsDFLTBYTES] ;
   gString gstmp( "%s/FMG_XXXXXX", this->tfPath ) ;
   gstmp.copy( tn, gsDFLTBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpPath = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpPath.clear() ;
   return status ;

}  //* End CreateTempname() *

//*************************
//*    DeleteTempname     *
//*************************
//******************************************************************************
//* Delete the specified temporary file.                                       *
//*                                                                            *
//* Input  : tmpPath : full path/filename specification of source              *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
//******************************************************************************

short FMgr::DeleteTempname ( const gString& tmpPath )
{

   return ( this->DeleteFile ( tmpPath.ustr() ) ) ;

}  //* End DeleteTempname() *

//*************************
//*        strnCpy        *
//*************************
//******************************************************************************
//* Efficiently copy a UTF-8 string or a wchar_t (wide) string from source     *
//* to target.                                                                 *
//*                                                                            *
//* Replaces the C library strncpy() and wcsncpy() functions, which are safer  *
//* than strcpy() and wcscpy() BUT are grossly inefficient for copying         *
//* NULL-terminated strings which are shorter than the target buffer.          *
//*                                                                            *
//* Behavior is undefined if source and target overlap. DON'T DO IT!           *
//*                                                                            *
//* Input  : trg   : pointer to target buffer (caller must verify that target  *
//*                  target is large enough to hold source data.)              *
//*          src   : pointer to source data to be copied                       *
//*          size  : maximum number of bytes (char*) or characters (wchar_t*)  *
//*                  to copy INCLUDING the NULL terminator                     *
//*                  NOTE: Target will always be NULL terminated.              *
//*                                                                            *
//* Returns: index into 'trg' of first free position after NULL terminator     *
//******************************************************************************
//* Programmer's Note: One byte DOES NOT EQUAL one character!!                 *
//* If caller specifies a byte copy limit, then there is a chance that the     *
//* last character before the NULL terminator will be corrupted. Tough luck.   *
//*                                                                            *
//* Programmer's Note: The C library functions are probably implemented in     *
//* assembly language, so the performance increase here may not be as great    *
//* as might be expected.                                                      *
//*                                                                            *
//* Return value: In the (very unlikely) event that caller actually wants to   *
//* use the return value from this method, note that it IS NOT the same as     *
//* the C library function. The following example works BUT does not check     *
//* for buffer overrun.                                                        *
//*                                                                            *
//*   char    cbuff[132] ;                                                     *
//*   int i = strnCpy ( cbuff, "Rabbits do it underground", 132 ) ;            *
//*   int j = strnCpy ( &cbuff[--i], ", but ", 132 ) ;                         *
//*   strnCpy ( &cbuff[i + (--j)], "elephants do it very carefully.", 132 ) ;  *
//*   Yields:                                                                  *
//*     "Rabbits do it underground, but elephants do it very carefully."       *
//******************************************************************************

int FMgr::strnCpy ( wchar_t* trg, const wchar_t* src, int size )
{
   int sindx = ZERO, tindx = ZERO ;

   do
   {
      if ( tindx < size )
         trg[tindx] = src[sindx++] ;
      else
         trg[--tindx] = NULLCHAR ;
   }
   while ( trg[tindx++] != NULLCHAR ) ;

   return tindx ;

}  //* End wcsnCpy() *

int FMgr::strnCpy ( char* trg, const char* src, int size )
{
   int sindx = ZERO, tindx = ZERO ;

   do
   {
      if ( tindx < size )
         trg[tindx] = src[sindx++] ;
      else
         trg[--tindx] = NULLCHAR ;
   }
   while ( trg[tindx++] != NULLCHAR ) ;

   return tindx ;

}  //* End wcsnCpy() *

#if 0    // SOCKET CREATION NOT YET SUPPORTED
//*************************
//*   CreateSocketFile    *
//*************************
//******************************************************************************
//* Used by CopySocket() to create a duplicate of an existing socket file.     *
//*                                                                            *
//*                                                                            *
//* Input :                                                                    *
//*                                                                            *
//* Returns: Returns OK if successful, or                                      *
//*          ERR if sockPath too long, socket() call fails,                    *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The sockaddr_un structure is a union with a rather fluid definition        *
//* (see socket.h, bits/sockaddr.h, sys/un.h).                                 *
//* This for use with the AF_LOCAL (PF_LOCAL) address domain.                  *
//* In a simplified way it looks like this:                                    *
//* struct sockaddr_un                                                         *
//* {                                                                          *
//*   unsigned short sun_family ;                                              *
//*   char sun_path[108] ;                                                     *
//* } ;                                                                        *
//*                                                                            *
//* The sockaddr_in structure is similarly weird:                              *
//* struct sockaddr_in                                                         *
//* {                                                                          *
//*   unsigned short sin_family ;     (16 bits)                                *
//*   unsigned short sin_port ;       (16 bits)                                *
//*   unsigned int sin_addr ;         (32 bits)                                *
//* } ;                                                                        *
//*                                                                            *
//* int socket ( int NAMESPACE, int STYLE, int PROTOCOL ) ;                    *
//* int bind ( int SOCKET, struct sockaddr *ADDR, socklen_t LENGTH ) ;         *
//*                                                                            *
//*                                                                            *
//* NOTE: The magic number '108' for socket name length is a legacy from       *
//*       somewhere. Presumably, it is intended for either the filename alone, *
//*       or something like '/dev/filename'. Caller should verify that the     *
//*       path fits.                                                           *
//*                                                                            *
//* To get more information about a socket file, see: 'netstat', 'lsof',       *
//* 'fuser', or 'socat'.                                                       *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short FMgr::CreateSocketFile ( char sockPath[108], int sockFamily, int sockType )
{
short status = ERR ;

   //* Read command-line args *
   if ( (strlen ( sockPath )) < 108 )
   {
      //* Create the socket *
      int sockHandle ;
      if ( (sockHandle = socket ( sockFamily, sockType, 0 )) != (-1) )
      {
         //* Bind the socket to the filename *
         struct sockaddr_un sName ;
         int sockaddrLen ;
         sName.sun_family = sockFamily ;        // socket format
         this->strnCpy ( sName.sun_path, sockPath, 108 ) ;// socket filename
         sockaddrLen = SUN_LEN ( &sName ) ;     // socket address length
         if ( (bind ( sockHandle, (const struct sockaddr *)&sName, sockaddrLen )) >= 0 )
            status = OK ;
         //* Close the socket *
         close ( sockHandle ) ;
      }
      else
      {
         this->recentErrno = errno ;
         this->DebugLog ( "In CreateSocketFile: socket() failed", 
                          this->recentErrno, sockPath ) ;
      }
   }
   return status ;
   
}  //* End CreateSocketFile() *
#endif   // U/C

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

