//********************************************************************************
//* File       : CxFile.cpp                                                      *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2023-2024 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice in AcEdit.cpp.                      *
//* Date       : 01-Oct-2024                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: File-access support for CommExt class.                          *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "CommExtract.hpp"       //* Application-class definition

//***************
//* Definitions *
//***************

//**************
//* Local data *
//**************

//********************
//* Local prototypes *
//********************
static bool extractFilename ( gString& nameBuff, const gString& pfSource ) ;
static fmFType decodeFileType ( uint32_t mode ) ;
static void decodeEpochTime ( int64_t eTime, localTime& ftTime ) ;
bool getLocalTime ( localTime& lt, gString& ts ) ;


//*************************
//*     getFileStats      *
//*************************
//********************************************************************************
//* Perform a 'stat' ('lstat') on the file specified by the full path string.    *
//* By default the stats for the specified file are returned, but if target is   *
//* a symbolic link, the symbolic link is followed i.e. stats for link target    *
//* are returned.                                                                *
//*                                                                              *
//*                                                                              *
//* Input  : tnFile : tnFName class object (by reference) to hold 'stat' data    *
//*          fPath  : full or partial path/filename specification                *
//*          getTarg: (optional, false by default) if true, symbolic links are   *
//*                   followed. Else, do not follow links                        *
//*                                                                              *
//* Returns: ZERO if successful, all data members of tnFile will be initialized  *
//*          Else returns system 'errno' value                                   *
//*            From 'realpath':                                                  *
//*            If errno == EACCES, at least one path component is not readable   *
//*            If errno == ENOENT, at least one path component does not exist    *
//*            From 'stat' i.e. 'stat64':                                        *
//*            If errno == ENOENT, file does not exist                           *
//*                                (or link target does not exist)               *
//*            If errno == EACCES, the process lacks permission for a container  *
//*                                directory                                     *
//*            If errno == ENOTDIR, an element of the path s/b a directory,      *
//*                                 but is not (unlikely)                        *
//*                                                                              *
//*           If error during system call, on return:                            *
//*            - tnFile.readAcc == false and all other members are undefined     *
//*             Exception:                                                       *
//*             If error is a broken symbolic link:                              *
//*             - tnFile.fName == name of link target if known, else "unknown"   *
//*             - tnFile.fSpec == full filespec if known, else "unknown"         *
//*             - tnFile.readAcc == false                                        *
//*             - tnFile.fType == fmTYPES                                        *
//*             - All other members of 'tnFile' are undefined                    *
//********************************************************************************

short CommExt::getFileStats ( tnFName& tnFile, const gString& fPath )
{
   char fullPath[MAX_PATH] ;        // full path/filename
   gString gsfp, gsfn ;             // text management
   short status = ZERO ;            // return value

   //* Provides an orderly crash in case of failure *
   tnFile.fBytes = ZERO ;
   tnFile.fType = fmUNKNOWN_TYPE ;
   tnFile.readAcc = tnFile.writeAcc = false ;

   //* Convert the provided filename string to a full filespec. *
   //* Symbolic links _are_followed.                            *
   if ( (realpath ( fPath.ustr(), fullPath )) != NULL )
   {
      gsfp = fullPath ;
      gsfp.copy( tnFile.fSpec, MAX_PATH ) ;     // save the full filespec
      extractFilename ( gsfn, gsfp ) ;          // extract filename from filespec
      gsfn.copy( tnFile.fName, MAX_FNAME ) ;    // save the target filename

      //* If successful 'stat' of the target, decode the raw stats. *
      if ( (status = stat64 ( tnFile.fSpec, &tnFile.rawStats )) == ZERO )
      {
         tnFile.fType = decodeFileType ( tnFile.rawStats.st_mode ) ; // file type
         tnFile.fBytes = tnFile.rawStats.st_size ;    // number of bytes in file
         decodeEpochTime ( (int64_t)tnFile.rawStats.st_mtime,  // mod time
                           tnFile.modTime ) ;
         //* Extract the read and write access bits. *
         tnFile.readAcc  = bool((access ( tnFile.fSpec, R_OK )) == ZERO) ;
         tnFile.writeAcc = bool((access ( tnFile.fSpec, W_OK )) == ZERO) ;
      }
      else     // stat failed. return the 'errno' value (see note above)
         status = errno ;

   }     // realpath()

   //* 'realpath' failed. Either file does not exist, or *
   //* target is a broken symbolic link.                 *
   else
      status = errno ;

   return status ;

}  //* End getFileStats() *

//*************************
//*    extractFilename    *
//*************************
//********************************************************************************
//* Given a string that may contain one of the following:                        *
//*  - a filename                                                                *
//*  - a relative path/filename                                                  *
//*  - an absolute path/filename                                                 *
//* extract the filename string from the input string and return it to caller.   *
//*                                                                              *
//* Input  : nameBuff: buffer to hold filename                                   *
//*          pfSource: source filename or path/filename                          *
//*                                                                              *
//* Returns: 'true' if filename successfully isolated                            *
//*          'false' in special case of pfSource=="/" (at root directory)        *
//*                  OR if invalid input                                         *
//********************************************************************************
//* Programmer's Note: It is assumed that caller has sent us a valid filespec    *
//* in pfSource. If not, the source will be returned in the target.              *
//********************************************************************************

static bool extractFilename ( gString& nameBuff, const gString& pfSource )
{
   bool status = false ;      // return value

   short fIndx = (pfSource.findlast( fSLASH )) + 1 ;
   nameBuff = &pfSource.gstr()[fIndx] ;
   if ( ((nameBuff.gschars()) > 2) ||
        (((nameBuff.gschars()) == 2) && (*nameBuff.gstr() != fSLASH)) )
   {
      status = true ;
   }
   return status ;

}  //* End extractFilename() *

//*************************
//*    decodeFileType     *
//*************************
//********************************************************************************
//* Extract the file type from the "st_mode" element of the stat{} structure     *
//* and return the equivalent member of enum fmFType.                            *
//*                                                                              *
//* PRIVATE METHOD.                                                              *
//*                                                                              *
//* The POSIX standard defines certain macros to access the file "st_mode".      *
//* There are several layers to the macros, but for Linux EXT2, EXT3 we          *
//* finally get down to the actual bits in /usr/include/bits/stat.h:             *
//* See also page 135 of "Linux Programming by Example."                         *
//* The actual macros that use these bitmasks are used in this routine.          *
//*                                                                              *
//* Encoding of the file mode.                                                   *
//* #define	__S_IFMT	0170000	/* These bits determine file type.                 *
//* File types.                                                                  *
//* #define	__S_IFDIR	0040000	 Directory.                                     *
//* #define	__S_IFCHR	0020000	 Character device.                              *
//* #define	__S_IFBLK	0060000	 Block device.                                  *
//* #define	__S_IFREG	0100000	 Regular file.                                  *
//* #define	__S_IFIFO	0010000	 FIFO.                                          *
//* #define	__S_IFLNK	0120000	 Symbolic link.                                 *
//* #define	__S_IFSOCK	0140000	 Socket.                                        *
//*                                                                              *
//* Input Values : mode: st_mode field of file's stat{} strucutre                *
//*                                                                              *
//* Return Value : member of enum fmFType                                        *
//********************************************************************************

static fmFType decodeFileType ( uint32_t mode )
{
   fmFType  ftype = fmUNKNOWN_TYPE ;

   if ( (S_ISREG(mode)) != false )        ftype = fmREG_TYPE ;
   else if ( (S_ISDIR(mode)) != false )   ftype = fmDIR_TYPE ;
   else if ( (S_ISLNK(mode)) != false )   ftype = fmLINK_TYPE ;
   else if ( (S_ISCHR(mode)) != false )   ftype = fmCHDEV_TYPE ;
   else if ( (S_ISBLK(mode)) != false )   ftype = fmBKDEV_TYPE ;
   else if ( (S_ISFIFO(mode)) != false )  ftype = fmFIFO_TYPE ;
   else if ( (S_ISSOCK(mode)) != false )  ftype = fmSOCK_TYPE ;
   
   return ftype ;
   
}  //* End decodeFileType() *

//************************
//*    getLocalTime      *
//************************
//********************************************************************************
//* Get the system timecode and convert it to localTime format.                  *
//*                                                                              *
//* Input  : ft  : (by reference, initial values ignored)                        *
//*                on return, contains decoded date/time                         *
//*          ts  : (by reference, initial value ignored)                         *
//*                on return, contains the canonical timestamp string            *
//*                                                                              *
//* Returns: true if successful, false if system call fails                      *
//********************************************************************************
//* Programmer's Note: Currently we do not decode the timezone and DST fields.   *
//*                                                                              *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//* Note that we do a bit of defensive programming in anticipation of the        *
//* dreaded year 2038 overflow of the 32-bit time_t type.                        *
//********************************************************************************

bool getLocalTime ( localTime& lt, gString& ts )
{
   bool success = false ;                       // return value

   ts.clear() ;                                 // clear timestamp buffer
   lt.reset() ;                                 // set to default values

   if ( (time ( &lt.sysepoch )) != (-1) )       // system epoch time
   {
      if ( lt.sysepoch >= ZERO )                // reported time is AFTER the epoch
      {
         lt.epoch = (int64_t)lt.sysepoch ;      // promote to 64-bit
         //* Decode the system time *
         Tm    tm ;                             // Linux time structure
         if ( (localtime_r ( &lt.sysepoch, &tm )) != NULL )
         {
            //* Translate to localTime format *
            lt.julian   = tm.tm_yday ;          // Julian date
            lt.day      = tm.tm_wday ;          // 0 == Sunday ... 6 == Saturday
            lt.date     = tm.tm_mday ;          // today's date
            lt.month    = tm.tm_mon + 1 ;       // month
            lt.year     = tm.tm_year + 1900 ;   // year
            lt.hours    = tm.tm_hour ;          // hour
            lt.minutes  = tm.tm_min ;           // minutes
            lt.seconds  = tm.tm_sec ;           // seconds
            success = true ;
         }

         //* Construct the human-readable timestamp. *
         ts.compose( "%04hd-%02hd-%02hdT%02hd:%02hd:%02hd",
                     &lt.year, &lt.month, &lt.date,
                     &lt.hours, &lt.minutes, &lt.seconds ) ;
      }
      else
      {  /* SYSTEM ERROR - time_t OVERFLOW */ }
   }
   return success ;

}  //* End getLocalTime() *

//*************************
//*    decodeEpochTime    *
//*************************
//********************************************************************************
//* Convert GNU/UNIX epoch time code (seconds since the epoch: Jan. 01, 1970)    *
//* to local time in a structured format (class localTime).                      *
//*                                                                              *
//* See the info page for the 'stat' command for more information.               *
//* See also the notes in DirEntryDisplayFormat() above.                         *
//*                                                                              *
//* Input  : eTime  : epoch time (this is a signed, 64-bit value)                *
//*                   (See note in definition of localTime class.)               *
//*          ftTime : (by reference, initial values ignored)                     *
//*                                                                              *
//* Returns: nothing, but all members of ftTime will be initialized              *
//********************************************************************************
//* System conversion from time_t code to local time is returned in this         *
//* structure.                                                                   *
//*                                                                              *
//*  struct tm                                                                   *
//*  {                                                                           *
//*     int  tm_sec ;          // 0-59  (leap seconds to 61)                     *
//*     int  tm_min ;          // 0-59                                           *
//*     int  tm_hour ;         // 0-23                                           *
//*     int  tm_mday ;         // 1-31                                           *
//*     int  tm_mon ;          // 0-11                                           *
//*     int  tm_year ;         // since 1900                                     *
//*     int  tm_wday ;         // 0-6                                            *
//*     int  tm_yday ;         // 0-365                                          *
//*     int  tm_isdst ;        // >0 == is DST, 0 == not DST, <0 == unknown      *
//*  } ;                                                                         *
//*                                                                              *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//* Note that we do a bit of defensive programming in anticipation of the        *
//* dreaded year 2038 overflow of the 32-bit time_t type.                        *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//* Daylight Savings Time:                                                       *
//* ----------------------                                                       *
//* Because we use this method to convert file timestamps, the DST calculation   *
//* done by 'localtime_r' throws the time off by an hour during the DST          *
//* part of the year. Unfortunately it does not do this uniformly. Files on      *
//* Linux filesystems drop back an hour (3,600 seconds), while files on          *
//* non-Linux filesystem such as VFAT do not drop back. What's up with that?     *
//* Presumably because VFAT does not include TZ data?                            *
//*                                                                              *
//* It's actually much worse than that. The system dynamically MODIFIES the      *
//* reported mod and stat timestamps for all files based on DST.                 *
//* This means that localtime_r is doing its job properly, but is working with   *
//* bad data sent from the kernel for filesystems that have DST offsets.         *
//* We're pretty sure this isn't the whole story but we don't have enough        *
//* information at this time to solve it. See the 'zdump' utility which lists    *
//* the offsets in a zoneinfo file:                                              *
//*         zdump -v /etc/localtime                                              *
//*         zdump -v /usr/share/zoneinfo/America/New_York | grep '2024 E.T'      *
//* This yields:                                                                 *
//* Sun Mar 10 06:59:59 2024 UT = Sun Mar 10 01:59:59 2024                       *
//*                                                  EST isdst=0 gmtoff=-18000   *
//* Sun Mar 10 07:00:00 2024 UT = Sun Mar 10 03:00:00 2024                       *
//*                                                  EDT isdst=1 gmtoff=-14400   *
//* Sun Nov  3 05:59:59 2024 UT = Sun Nov  3 01:59:59 2024                       *
//*                                                  EDT isdst=1 gmtoff=-14400   *
//* Sun Nov  3 06:00:00 2024 UT = Sun Nov  3 01:00:00 2024                       *
//*                                                  EST isdst=0 gmtoff=-18000   *
//*                                                                              *
//* It seems that file timestamps are in UTC, and the system applies the offset  *
//* listed in the specified zoneinfo file (/etc/localtime). If this is true,     *
//* then why is one offset applied to ext4, and a different offset applied       *
//* to VFAT filesystem (SD card)?                                                *
//* What EXACTLY is the difference that causes the DST offset to be applied in   *
//* one case and the standard time offset to be applied in the other?            *
//*  -- the 'tm_isdst' value seems to be > ZERO for both                         *
//*  -- 'tm_gmtoff' value ?                                                      *
//*                                                                              *
//* "To disable DST changes, link your /etc/localtime file to one of zoneinfo    *
//* files placed under the folder /usr/share/zoneinfo/Etc/"                      *
//* "Zones like Etc/GMT+6 are intentionally reversed for backwards               *
//* compatibility with POSIX standards"                                          *
//*   Example command:                                                           *
//*         ln -s /usr/share/zoneinfo/Etc/GMT+3 /etc/localtime                   *
//* Note that the 'GMT' files do not contain DST offsets. Instead, they contain  *
//* two epoch values with the second value exactly 24 hours (24*60*60 seconds)   *
//* ahead of the first. What's up with that?                                     *
//*                                                                              *
//* Because this development machine is on New York time (-5:00 i.e. 5 hours     *
//* west of UTC, -4:00DST), and because the GMT files are "intentionally         *
//* reversed," our setting would be:                                             *
//*         ln -s /usr/share/zoneinfo/Etc/GMT+5 /etc/localtime                   *
//* Be aware, however, that this doesn't actually resolve the issue. It simply   *
//* sets BOTH vfat and ext4 timestamps back an _additional_ hour.                *
//*                                                                              *
//* If the kernel would tell us the truth, we would consider using ctime_r       *
//* which converts the time_t value to a string, which we would then scan for    *
//* the values, but since the kernel is a lying SOS, this won't help us.         *
//*                                                                              *
//********************************************************************************

static void decodeEpochTime ( int64_t eTime, localTime& ftTime )
{
   ftTime.reset() ;           // initialize caller's data
   if ( eTime >= ZERO )       // defend against stupidity
   {  //* Width of time_t is system dependent *
      ftTime.epoch = eTime ;
      ftTime.sysepoch = (time_t)eTime ;   // (possible narrowing)

      //* Receives the broken-down time.              *
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &ftTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format *
         ftTime.date    = bdt.tm_mday ;         // today's date
         ftTime.month   = bdt.tm_mon + 1 ;      // month
         ftTime.year    = bdt.tm_year + 1900 ;  // year
         ftTime.hours   = bdt.tm_hour ;         // hour
         ftTime.minutes = bdt.tm_min ;          // minutes
         ftTime.seconds = bdt.tm_sec ;          // seconds
         ftTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         ftTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)

         #if DECODE_TZ != 0   // time-zone data
         ftTime.gmtoffset = bdt.tm_gmtoff ;     // offset as number of seconds
         ftTime.dst = (bdt.tm_isdst > ZERO ) ? true : false ; // DST in effect

         #if 1    // additional time zone data (not very useful) *
         gString gs( bdt.tm_zone ) ;
         gs.copy( ftTime.timezone, ltFMT_LEN ) ;
         int absOffset = abs(ftTime.gmtoffset) ;
         int hOffset = absOffset / (60*60),     // convert offset seconds to 
             mOffset = absOffset % (60*60) ;    // hours and minutes
         if ( ftTime.gmtoffset < ZERO )
            hOffset = -(hOffset) ;
         gs.compose( L"UTC%+d:%02d", &hOffset, &mOffset ) ;
         gs.copy( ftTime.utc_zone, ltFMT_LEN ) ;
         #endif   // additional time zone data
         #endif   // DECODE_TZ
      }
   }

}  //* End decodeEpochTime() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

