//********************************************************************************
//* File       : CommExtract.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2024 Mahlon R. Smith, The Software Samurai        *
//*                 GNU GPL copyright notice below                               *
//* Date       : 01-Oct-2024                                                     *
//* Version    : (see AppVersion string, below)                                  *
//*                                                                              *
//* Description: Extract C++ comment lines from a source file, then remove       *
//* the comment token (//) and leading/trailing whitespace, to create plain text *
//* which can then be inserted into the documentation for that project.          *
//*                                                                              *
//* The author has found this to be a useful tool, in that his development       *
//* style for a new project is to write the comments first. The code is then     *
//* written to implement the project described in those comments.                *
//* This is ALSO the approach all of our students are encouraged to use for      *
//* their class projects.                                                        *
//*                                                                              *
//* The practical result of this approach is that the project documentation      *
//* essentially writes itself. The bulk of the comments can be moved to a        *
//* ReadMe file or to a Texinfo documentation file. However, it is rather        *
//* tedious to manually remove the C++ comment tokens, so this simple utility    *
//* performs the task automagically.                                             *
//*                                                                              *
//*                                                                              *
//* Development Tools:                  Current:                                 *
//* ----------------------------------  ---------------------------------------- *
//*  GNU G++ v:4.8.0 or higher (C++11)  G++ v:13.3.1 20240913                    *
//*  Fedora Linux v:16 or higher        Fedora v:39                              *
//*  GNOME Terminal v:3.2.1 or higher   GNOME Term v:3.50.1 (GNOME 45)           *
//*                                                                              *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//********************************************************************************
//* Version History:                                                             *
//* v: 0.00.02 30-Sep-2024                                                       *
//*    -- Errors identified by beta-tester corrected 30-Sep (Thanks Lucy!)       *
//*    -- Added timestamp to output (because it was fun).                        *
//*    -- Created documentation in the form of a readme file.                    *
//*       Note that creating this file took about twenty minutes, and            *
//*       served as a real-world test of application functionality.              *
//*    -- Wrote Perl script for creating distribution archive.                   *
//*    -- Posted to website: 01-Oct-2024                                         *
//*                                                                              *
//* v: 0.00.01 26-Sep-2024                                                       *
//*    -- First Effort.                                                          *
//*       - Define the user interface (command-line arguments).                  *
//*       - File management utilities adapted (in simplified form) from          *
//*         our AcEdit application.                                              *
//*       - First beta release 29-Sep-2024.                                      *
//*                                                                              *
//*                                                                              *
//********************************************************************************
//* Programmer's Notes: See README file.                                         *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "CommExtract.hpp"       //* Application-class definition

//**************
//* Local data *
//**************
static const wchar_t* const appVersion = L"0.0.02" ;
static const wchar_t* const crYears    = L"2024-____" ;
static const wchar_t* const appName1   = L"CommExtract (comex)" ;
static const char*    const titleTemplate =  
   "%S v:%S Copyright(c) %S  The Software Samurai\n"
   "=========================================================================" ;

static const char* const dfltTargetFile = "./xcom.txt" ;
static const char* const headerTemplate = 
   "**comex: Reformat comments from '%s' - lines: %04hd - %04hd\n" ;
static const char* const footerTemplate =
   "**comex: %hd source lines reformatted" ;

//********************
//* Local prototypes *
//********************
extern bool getLocalTime ( localTime& lt, gString& ts ) ;


//*************************
//*        main           *
//*************************
//********************************************************************************
//* Program entry point.                                                         *
//*                                                                              *
//********************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* For capture and parsing of command-line arguments *
   commArgs ca( argc, argv, argenv ) ;

   //* Instantiate the application class. *
   CommExt *sm = new CommExt ( ca ) ;

   delete sm ;                            // delete the application class object

   exit ( ZERO ) ;                        // exit, returning control to the shell

}  //* End main() *

//*************************
//*       ~CommExt        *
//*************************
//********************************************************************************
//* CommExt destructor.                                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

CommExt::~CommExt ( void )
{
   //* If a locale object has been allocated, delete it now. *
   if ( this->ioLocale != NULL )
   {
      delete ( this->ioLocale ) ;
      this->ioLocale = NULL ;
   }

}  //* End ~CommExt() *

//*************************
//*        CommExt        *
//*************************
//********************************************************************************
//* CommExt constructor.                                                         *
//*                                                                              *
//* Input  : ca : (by reference) copy of command-line arguments                  *
//*                                                                              *
//* Returns: implicitly returns pointer to class                                 *
//********************************************************************************
//* Notes on terminal configuration:                                             *
//* Instantiation of The CommExt class performs important configuration          *
//* operations which should be completed before the application writes to        *
//* stdout/stderr and before the application reads from stdin.                   *
//*                                                                              *
//* This is because the application's first access to stdout/stdin/stderr        *
//* will cause certain configuration options to be set for the duration of the   *
//* session. This automatic configuration may cause the application to act in    *
//* unexpected ways. For this reason, it is important to EXPLICITLY set the      *
//* terminal configuration options as early in the application as possible.      *
//* This is done during instantiation of the CommExt class.                      *
//* The two most critical configuration options are described briefly here.      *
//*                                                                              *
//* 1) The width of the I/O streams are set the first time they are used.        *
//*    The wide input/output streams are accessed through the 'wcin' and         *
//*    'wcout' classs, respectively.                                             *
//*    The narrow input/output streams are accessed through the 'cin' and        *
//*    'cout' classes, respectively. Narrow streams are not used in this         *
//*    application.                                                              *
//*                                                                              *
//* 2) The terminal's "locale" determines the character encoding used, and the   *
//*    primary language expected. This determines the way the terminal will      *
//*    handle certain text formatting -- for instance, currency symbols          *
//*    and number formats.                                                       *
//*    It is important to understand that if the application does not            *
//*    explicitly set the desired locale, the so-called "C-local" is used.       *
//*    The C-locale restricts the character set to an ASCII superset (7-8-bit    *
//*    characters), and is unacceptable for any serious application.             *
//*                                                                              *
//********************************************************************************

CommExt::CommExt ( commArgs& ca )
{
   gString gsOut ;                        // text formatting

   //* Initialize all class data members.*
   this->reset () ;

   //* Parse command-line arguments *
   clArgs userOption = this->getCommandLineArgs ( ca ) ;

   if ( userOption == clExtract )
   {
      system ( "clear" ) ;                // clear the terminal window
      this->composeTitle ( gsOut, true ) ;// Display app title
      wcout << gsOut.gstr() ;

      //* Set locale from the environment. *
      this->setLocale () ;

      //* If all parameters validated, scan the source file,   *
      //* extract and reformat the data in the specified range.*
      if ( (this->validateArgs ( ca )) )
      {
         this->scanSourceData () ;
      }
      else     // report the validation error and exit
      { gsOut = ca.parmErr ; wcout << gsOut << endl ; }
   }
   else
   {
      //* Display copyright and version information, then exit *
      if ( userOption == clVersion )
      {
         static const wchar_t* const freeSoftware = 
         L"License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
          "This is free software: you are free to modify and/or redistribute it\n"
          "under the terms set out in the license.\n"
          "There is NO WARRANTY, to the extent permitted by law.\n" ;

         this->composeTitle ( gsOut, true ) ;
         wcout << gsOut.gstr() ;
         wcout << freeSoftware << endl ;
      }

      //* Display command-line help *
      else if ( userOption == clHelp )
      {
         const wchar_t* const Help1 =
         L"Convert a C++ style comment block to plain text.\n"
          "The leading \"//* \" or \"// \" sequence is removed; along with\n"
          "any trailing '*' and whitespace. One trailing space character is retained.\n"
          "Mandatory Arguments:\n"
          " --src=FNAME : Source filename, the name of a file in the current directory.\n"
          " --beg=LINE# : First source line for the scan.\n"
          " --end OR --cnt (but not both)\n"
          " --end=LINE# : Last source line for the scan (inclusive).\n"
          "       or\n"
          " --cnt=COUNT : Number of source lines to be scanned.\n"
          "Example: comex --src=MySource.cpp --beg=1 --end=24\n"
          "         comex MySource.cpp --beg=1 --cnt=24\n"
          "Optional Arguments:\n"
          " --trg=FNAME : Target filename, the name of the file to which the formatted\n"
          "               data will be written. If not specified, \"xcom.txt\" is the target file.\n"
          "               To avoid loss of data, if the target file already exists, the scanned\n"
          "               data will be APPENDED to the existing file by default.\n"
          " --deltrg    : Delete existing target file (if any) before writing output.\n"
          " --wspall    : Remove all leading whitespace following comment token.\n"
          " --echo      : Echo the reformatted data to the display.\n"
          " --help (-h) : Display application help.\n"
          " --version   : Display application version number.\n"
          ;

         system ( "clear" ) ;             // clear the terminal window
         this->composeTitle ( gsOut, true ) ;
         wcout << gsOut.gstr() ;
         wcout << Help1 << endl ;
      }

      //* userOption == clQHelp (no command-line options specified), *
      //* OR misc. user command syntax error.                        *
      else
      {
         this->composeTitle ( gsOut, true ) ;
         wcout << gsOut.gstr() ;
         if ( *ca.parmErr != NULLCHAR )
         { gsOut = ca.parmErr ; wcout << gsOut << endl ; }
         gsOut = L"Please specify source filename and line range, "
                  "or type: \"comex --help\"\n" ;
         wcout << gsOut.gstr() << endl ;
      }
   }

}  //* End CommExt() *

//*************************
//*    scanSourceData     *
//*************************
//********************************************************************************
//* Scan the source file, extract the data within the specified range, reformat  *
//* it according to the specified criteria and write it to the target file.      *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true'  if data successfully scanned and reformatted.               *
//*          'false' if I/O error                                                *
//********************************************************************************
//* Processing Logic:                                                            *
//* -----------------                                                            *
//* For each line in the specified range:                                        *
//* 1) Scan over leading whitespace.                                             *
//*    If first non-whitespace characters are not "//", skip the line.           *
//* 2) If first non-whitespace is "//" C++ comment:                              *
//*    a) If "//" is followed by '*', include it in sequence to be deleted.      *
//*    b) If '//' or '//*' is followed by a space, include it in sequence        *
//*       to be deleted, (2, 3 or 4 characters, total)                           *
//* 3) If the line ends with a trailing '*', remove it.                          *
//* 4) Strip whitespace:                                                         *
//*    a) If 'wspAll' flag is set, strip both leading and trailing whitespace.   *
//*    b) Else, strip trailing whitespace only.                                  *
//*       (maintains vertical column alignment)                                  *
//* 5) Append a single space to the end of the line.                             *
//* 6) Write the reformatted text to the output file.                            *
//*                                                                              *
//*                                                                              *
//********************************************************************************

bool CommExt::scanSourceData ( void )
{
   #define DEBUG_SSD (0)

   ifstream ifs ;                         // input file access
   ofstream ofs ;                         // output file access
   char    ibuff[gsMAXBYTES] ;            // input buffer
   gString gs, gst ;                      // text analysis
   localTime lt ;                         // system local time
   short   icnt = 1,                      // count source lines
           ocnt = ZERO ;                  // number of target lines written
   bool status = false ;                  // return value

   //* Open the source file and scan forward to the first target line. *
   ifs.open( this->srctnf.fSpec, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      status = true ;                     // hope for the best
      while ( icnt < this->lnBeg )
      {
         ifs.getline( ibuff, gsMAXBYTES ) ;
         if ( ifs.good() || (ifs.gcount() > ZERO) )
            ++icnt ;
         else           // read error (or EOF)
         { status = false ; break ; }
      }

      //* If pre-scan of source file was successful *
      if ( status )
      {
         //* Open the target file and write the output header. *
         ofs.open( this->trgtnf.fSpec, (ofstream::out | ofstream::app) ) ;
         if ( ofs.is_open() )
         {
            short lcnt = this->lnBeg + this->lnCnt, indx ;
            gs.compose( headerTemplate, this->srctnf.fName, &this->lnBeg, &lcnt ) ;
            indx = gs.after( SPACE ) ;
            wcout << &gs.gstr()[indx] ; wcout.flush() ;
            ofs << gs.ustr() << endl ;

            //* Process the source lines for the specified range. *
            for ( icnt = ZERO ; icnt < this->lnCnt ; ++icnt )
            {
               ifs.getline( ibuff, gsMAXBYTES ) ;
               if ( ifs.good() || (ifs.gcount() > ZERO) )
               {
                  if ( ibuff[0] != NULLCHAR )
                  {
                     gs = ibuff ;
                     if ( (indx = gs.scan()) < (gs.gschars() - 1) )
                     {
                        if ( (gs.find( CCOMM, indx )) == indx )
                        {
                           indx += 2 ;
                           if ( gs.gstr()[indx] == STAR )
                              ++indx ;
                           if ( gs.gstr()[indx] == SPACE )
                              ++indx ;
                           gs.shiftChars( -(indx) ) ;
                           if ( (gs.findlast( STAR )) == (gs.gschars() - 2) )
                              gs.limitChars( gs.gschars() - 2 ) ;
                           if ( this->wspAll )
                              gs.strip( true, true ) ;
                           else
                              gs.strip( false, true ) ;
                           gs.append( SPACE ) ;
                           ofs << gs.ustr() << endl ;
                           ++ocnt ;

                           //* If specified echo data to the display. *
                           if ( this->trmEcho )
                              wcout << gs << endl ;

                           #if DEBUG_SSD != 0
                           gs.replace( SPACE, L'_', ZERO, false, true ) ;
                           wcout << gs << endl ;
                           #endif   // DEBUG_SSD
                        }
                     }
                  }
               }
               else           // read error (or EOF)
                  break ;
            }

            //* Write the footer and close the output file. *
            getLocalTime ( lt, gst ) ;
            gs.compose( footerTemplate, &ocnt ) ;
            ofs << gs.ustr() << " - " << gst.ustr() << "\n" << endl ;
            ofs.close() ;                 // close the target file

            gs.append( " and written to: '%s'\n", this->trgtnf.fName ) ;
            wcout << gs << endl ;
         }
      }
      ifs.close() ;                       // close the source file
   }

   return status ;

   #undef DEBUG_SSD
}  //* End scanSourceData() *

//*************************
//*     validateArgs      *
//*************************
//********************************************************************************
//* Validate command-line arguments.                                             *
//* Caller has verified that all required arguments were provided.               *
//*  1) Verify that specified source file exists, that is is a 'regular' file,   *
//*     and that owner (not necessarily the user) has read access.               *
//*  2) Verify that beginning line number is a positive value.                   *
//*  3) Ending line number:                                                      *
//*     a) If ending line number specified, verify that it is larger than        *
//*        beginning line number; then convert it to a line count i.e. the       *
//*        difference between begin and end.                                     *
//*     b) If line count specified, verify that it is a positive value.          *
//*  4) Determine whether target file exists, and if it does, verify that it     *
//*     is a 'regular' file and that owner has both read and write access.       *
//*     To simplify caller's life:                                               *
//*     a) If target-file exists AND if target deletion specified, open the      *
//*        file with 'truncate', and then close the file.                        *
//*     b) If target file does not exist, create it, and close it, leaving       *
//*        it empty.                                                             *
//*                                                                              *
//* Input  : ca : (by reference) copy of command-line arguments                  *
//*                                                                              *
//* Returns: 'true' if all parameters validated, else 'false'                    *
//********************************************************************************

bool CommExt::validateArgs ( commArgs& ca )
{
   #define DEBUG_VARGS (0)
   #if DEBUG_VARGS != 0
   gString gsdbg ;
   #endif   // DEBUG_VARGS

   gString gs( ca.sFile ),                // text management
           gsErr( "OK" ) ;                // status messages
   ofstream ofs ;                         // access target file
   bool status = false ;                  // return value

   *ca.parmErr = NULLCHAR ;               // clear the error-message buffer
   this->wspAll = ca.strAll ;             // copy the strip-all-leading-whitespace flag
   this->trmEcho = ca.trmEcho ;           // copy the echo flag

   //* Verify that file exists and get its stats. *
   if ( ((this->getFileStats ( this->srctnf, gs )) == ZERO) &&
        (this->srctnf.fType == fmREG_TYPE) && this->srctnf.readAcc )
   {
      #if DEBUG_VARGS != 0
      gsdbg.compose(
         "  source : '%s'\n"
         "filename : '%s'\n"
         "filespec : '%s'\n"
         "   fType : %hd readAcc:%hhd writeAcc:%hhd\n",
         ca.sFile, this->srctnf.fName, this->srctnf.fSpec,
         &this->srctnf.fType, &this->srctnf.readAcc, &this->srctnf.writeAcc ) ;
      wcout << gsdbg ;
      #endif   // DEBUG_VARGS

      //* Validate the scan range. *
      if ( ca.lnBeg > ZERO )
      {
         this->lnBeg = ca.lnBeg ;
         if ( ca.lnCnt > ZERO )
            this->lnCnt = ca.lnCnt ;
         else if ( ca.lnEnd > this->lnBeg )
            this->lnCnt = ca.lnEnd - this->lnBeg + 1 ; // (inclusive)
         else
            gsErr.compose( "Invalid line number range: begin:%hd end:%hd", &this->lnBeg, &ca.lnEnd ) ;

         #if DEBUG_VARGS != 0
         gsdbg.compose( "   lnBeg : %hd\n   lnCnt : %hd\n", &this->lnBeg, &this->lnCnt ) ;
         wcout << gsdbg ;
         #endif   // DEBUG_VARGS

         //* If valid line range, scan for target file. *
         if ( this->lnCnt > ZERO )
         {
            gs = ca.tFile ;
            //* If target exists and is accessible *
            if ( (this->getFileStats ( this->trgtnf, gs )) == ZERO ) 
            {
               if ( (this->trgtnf.fType == fmREG_TYPE) && 
                    this->trgtnf.readAcc && this->trgtnf.writeAcc )
               {
                  if ( ca.delTrg )           // if specified, delete (empty) the target file
                  {
                     ofs.open( this->trgtnf.fSpec, (ofstream::out | ofstream::trunc) ) ;
                     if ( ofs.is_open() )
                     { ofs.close() ; status = true ; }
                  }
                  else
                     status = true ;
               }
               if ( ! status )
               {
                  gsErr.compose( "Target file '%s' access error: read:%hhd write:%hhd",
                                 this->trgtnf.fName, &this->trgtnf.readAcc, & this->trgtnf.writeAcc ) ;
               }
            }
            //* Target does not exist, so create it. *
            else
            {
               gs = ca.tFile ;
               ofs.open( gs.ustr(), (ofstream::out | ofstream::trunc) ) ;
               if ( ofs.is_open() )
               {
                  ofs.close() ;
                  status = bool((this->getFileStats ( this->trgtnf, gs )) == ZERO) ;
               }
               else
                  gsErr.compose( "Error creating target file: '%s'", ca.tFile ) ;
            }

            #if DEBUG_VARGS != 0
            gsdbg.compose(
              "  target : '%s'\n"
              "filename : '%s'\n"
              "filespec : '%s'\n"
              "   fType : %hd readAcc:%hhd writeAcc:%hhd fBytes:%Lu\n",
              ca.tFile, this->trgtnf.fName, this->trgtnf.fSpec, &this->trgtnf.fType, 
              &this->trgtnf.readAcc, &this->trgtnf.writeAcc, &this->trgtnf.fBytes ) ;
            wcout << gsdbg ;
            #endif   // DEBUG_VARGS
         }
      }
      else
         gsErr = "Initial line number (--beg) must be a positive value." ;
   }
   else
      gsErr.compose( "Source file: '%s' not found, or no read access.", ca.sFile ) ;

   if ( status == false )     // return status message to caller
      gsErr.copy( ca.parmErr, MAX_FNAME ) ;

   return status ;

   #undef DEBUG_VARGS
}  //* End validateArgs() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//********************************************************************************
//* Capture user's command-line arguments.                                       *
//* Valid Arguments: see help, above.                                            *
//*                                                                              *
//* Input  : ca : (by reference) copy of command-line arguments                  *
//*               argCount : number of elements in 'argList' array               *
//*               argList  : array of command-line arguments                     *
//*               argEnv   : array of environment variables                      *
//*                                                                              *
//* Returns: member of enum clArgs                                               *
//********************************************************************************

clArgs CommExt::getCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCLA (0)
   #if DEBUG_GCLA != 0
   gString gsdbg ;
   bool dbgReport = false ;
   #endif   // DEBUG_GCLA

   clArgs userOption = clQHelp ;       // return value (default == short help)

   //* If user provided command-line arguments, parse them. *
   //* Note that first argument is application name.        *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
      short j = ZERO ;           // index into token

      for ( short i = 1 ; i < ca.argCount ; ++i )
      {
         #if DEBUG_GCLA != 0
         if ( dbgReport )
            wcout << gsdbg.gstr() << endl ;
         gsdbg.compose( "%hd) '%s'", &i, ca.argList[i] ) ;
         dbgReport = true ;
         #endif   // DEBUG_GCLA

         j = ZERO ;

         //* If a command-line switch identified.              *
         //* Note: For first argument only (source filename),  *
         //* allow name without an argument specifier.         *
         if ( ca.argList[i][j] == DASH || (i == 1) )
         {
            ++j ;

            if ( *ca.argList[i] != DASH )
               gs.compose( "--src=%s", ca.argList[i] ) ;
            else
               gs = ca.argList[i] ;

            //** -------------------------- **
            //** Options without arguments: **
            //** -------------------------- **
            //* Request for version number and copyright information. *
            //* Note: Full option: "--version", but only six(6)   *
            //*       characters are required.                    *
            if ( (gs.compare( L"--vers", true, 6 )) == ZERO )
            {  // Programmer's Note: Request for version number overrides
               // all other options and error conditions.
               ca.verFlag = true ;        // set the flag
               userOption = clVersion ;   // report application version
               break ;                    // ignore any remaining arguments
            }

            //* Invoke command-line help *
            else if ( ((gs.compare( L"--help", true, 6 )) == ZERO) ||
                      ((gs.compare( L"-h", false )) == ZERO) )
            {  // Programmer's Note: Request for help overrides all options 
               // except '--version'. Note however that parsing of remaining 
               // arguments continues after receipt of a help request.
               ca.helpFlag = true ;       // set the flag
               userOption = clHelp ;      // display application help
               continue ;                 // finished with this option
            }

            //* Delete all leading whitespace *
            //* Note: Full option: "--wspall", but only six(6)    *
            //*       characters are required.                    *
            else if ( (gs.compare( L"--wspa", true, 6 )) == ZERO )
            {
               ca.strAll = true ;         // set the flag
               continue ;                 // finished with this option
            }

            //* Delete Existing Target File *
            //* Note: Full option: "--deltrg", but only six(6)    *
            //*       characters are required.                    *
            else if ( (gs.compare( L"--delt", true, 6 )) == ZERO )
            {
               ca.delTrg = true ;         // set the flag
               continue ;                 // finished with this option
            }

            //* In addition to writing the reformatted data to    *
            //* the target file, echo the data to terminal window.*
            else if ( (gs.compare( L"--echo", true, 6 )) == ZERO )
            {
               ca.trmEcho = true ;        // set the flag
               continue ;                 // finished with this option
            }

            //** ----------------------- **
            //** Options with arguments: **
            //** ----------------------- **
            //* Capture the option switch and its arguments.            *
            //* If option and argument are not separated by a '=',      *
            //* append the next token which SHOULD contain the argument.*
            if ( ((gs.find( L'=' )) < ZERO) && ((i + 1) < ca.argCount) )
            {
               gs.append( "=%s", ca.argList[++i] ) ;

               #if DEBUG_GCLA != 0
               gsdbg.compose( "%hd) '%S'", &i, gs.gstr() ) ;
               #endif   // DEBUG_GCLA
            }

            //* Source Filename *
            if ( (gs.compare( L"--src", true, 5 )) == ZERO )
            {
               if ( (j = gs.after( L'=' )) > ZERO )
               {
                  gs.shiftChars( -j ) ;   // isolate the argument
                  gs.copy( ca.sFile, MAX_FNAME ) ;
               }
               else { userOption = clQHelp ; break ; } // syntax error
            }

            //* Target Filename *
            else if ( (gs.compare( L"--trg", true, 5 )) == ZERO )
            {
               if ( (j = gs.after( L'=' )) > ZERO )
               {
                  gs.shiftChars( -j ) ;   // isolate the argument
                  gs.copy( ca.tFile, MAX_FNAME ) ;
               }
               else { userOption = clQHelp ; break ; } // syntax error
            }

            //* Beginning Source Line *
            else if ( (gs.compare( L"--beg", true, 5 )) == ZERO )
            {
               if ( (j = gs.after( L'=' )) > ZERO )
               {
                  gs.shiftChars( -j ) ;   // isolate the argument
                  if ( (gs.gscanf( "%hd", &ca.lnBeg )) < 1 )
                  { userOption = clQHelp ; break ; }     // syntax error
               }
               else { userOption = clQHelp ; break ; }   // syntax error
            }

            //* Ending Source Line *
            else if ( (gs.compare( L"--end", true, 5 )) == ZERO )
            {
               if ( (j = gs.after( L'=' )) > ZERO )
               {
                  gs.shiftChars( -j ) ;   // isolate the argument
                  if ( (gs.gscanf( "%hd", &ca.lnEnd )) == 1 )
                     ca.lnCnt = ZERO ;                   // reset 'lnCnt'
                  else
                  { userOption = clQHelp ; break ; }     // syntax error
               }
               else { userOption = clQHelp ; break ; }   // syntax error
            }

            //* Source Line Count *
            else if ( (gs.compare( L"--cnt", true, 5 )) == ZERO )
            {
               if ( (j = gs.after( L'=' )) > ZERO )
               {
                  gs.shiftChars( -j ) ;   // isolate the argument
                  if ( (gs.gscanf( "%hd", &ca.lnCnt )) == 1 )
                     ca.lnEnd = ZERO ;                   // reset 'lnEnd'
                  else
                  { userOption = clQHelp ; break ; }     // syntax error
               }
               else { userOption = clQHelp ; break ; }   // syntax error
            }

            //* Invalid command-line switch *
            else
            {
               #if DEBUG_GCLA != 0
               gsdbg.compose( "**Error: '%S'", gs.gstr() ) ;
               #endif   // DEBUG_GCLA

               userOption = clQHelp ;
               gs.insert( "**Parameter Error: " ) ;
               gs.copy( ca.parmErr, MAX_FNAME ) ;
               break ;
            }
         }

         //* An argument was found that neither begins with a dash ('-') *
         //* nor is it associated with a previous argument.              *
         else
         {
            #if DEBUG_GCLA != 0
            gsdbg.compose( "**Error: '%S'", ca.argList[i] ) ;
            #endif   // DEBUG_GCLA

            userOption = clQHelp ;
            gs.compose( "**Parameter Error: %s", ca.argList[i] ) ;
            gs.copy( ca.parmErr, MAX_FNAME ) ;
            break ;
         }
      }     // for(;;)

      //* If not 'help' and not 'version', validate basic info.*
      //* Source file must be specified.                       *
      //* Beginning line must be specified.                    *
      //* Either ending line OR line count must be specified.  *
      //* The data are not necessarily valid, but they must    *
      //* be present.                                          *
      if ( (*ca.parmErr == NULLCHAR) && !ca.helpFlag && !ca.verFlag )
      {
         if ( (*ca.sFile != NULLCHAR) && (ca.lnBeg > ZERO) &&
              ((ca.lnEnd > ZERO) || (ca.lnCnt > ZERO)) )
         {
            if ( *ca.tFile == NULLCHAR )
            { gs = dfltTargetFile ; gs.copy( ca.tFile, MAX_FNAME ) ; }
            userOption = clExtract ;
         }
      }

      #if DEBUG_GCLA != 0
      if ( dbgReport )     // report last item in list
         wcout << gsdbg.gstr() << endl ;

      wchar_t wbuff[16] ;  // summary results of parameter capture
      gsdbg.compose( "sFile : '%s'\n"
                     "tFile : '%s'\n"
                     "lnBeg :%hd lnEnd:%hd lnCnt:%hd\n"
                     "strAll:%hhd delTrg:%hhd helpFlag:%hhd verFlag:%hhd\n"
                     "Press Enter...",
                     ca.sFile, ca.tFile, &ca.lnBeg, &ca.lnEnd, &ca.lnCnt,
                     &ca.strAll, &ca.delTrg, &ca.helpFlag, &ca.verFlag
                   ) ;
      wcout << gsdbg << endl ;
      wcin.getline( wbuff, 16 ) ;
      #endif   // DEBUG_GCLA

   }
   return userOption ;

   #undef DEBUG_GCLA
}  //* End getCommandLineArgs() *

//*************************
//*       setLocale       *
//*************************
//********************************************************************************
//* Set the "locale" for the application.                                        *
//* By default, the application locale is taken from terminal environment.       *
//* (Type: echo $LANG   at the command line to get the terminal's locale.        *
//* An alternate locale may be specified.                                        *
//*                                                                              *
//* Input  : (optional, null pointer by default)                                 *
//*          If specified, pointer to name of UTF-8 compliant locale filename.   *
//*          If null pointer (or empty string: "\0"), set locale from terminal   *
//*          environment.                                                        *
//*                                                                              *
//* Returns: 'true'                                                              *
//*          Currently, we do not verify that 'localeName' is a valid UTF-8      *
//*          locale filename.                                                    *
//********************************************************************************

bool CommExt::setLocale ( const char * localeName )
{
   //* Create a locale object if it does not exist *
   if ( this->ioLocale == NULL )
      this->ioLocale = new locale("") ;// get default locale from environment

   //* If a locale name (filename) is specified, overwrite *
   //* the default locale with the specified locale.       *
   if ( (localeName != NULL) && (localeName[0] != NULLCHAR) )
   {
      locale tmp(localeName) ;   // create a locale object using specified file
      *this->ioLocale = tmp ;    // copy it to our data member
   }

   //* Give the locale "global" (application/process) scope *
   this->ioLocale->global( *this->ioLocale ) ;

   return true ;

}  //* End setLocale() *

//*************************
//*     composeTitle      *
//*************************
//********************************************************************************
//* Create a text string including the application title, application version    *
//* number and copywrite years.                                                  *
//*                                                                              *
//* Input  : gs     : (by reference) receives the formatting data                *
//*          tailNL : (optional, 'false' by default)                             *
//*                   if 'true',  append a newline to the end of the string      *
//*                   if 'false', no trailing newline                            *
//*          headNL : (optional, 'false' by default)                             *
//*                   if 'true',  insert a newline at the beginning of string    *
//*                   if 'false', no leading newline                             *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void CommExt::composeTitle ( gString& gs, bool tailNL, bool headNL )
{
   gs.compose( titleTemplate, appName1, appVersion, crYears ) ;
   if ( tailNL )
      gs.append( NEWLINE ) ;
   if ( headNL )
      gs.insert( NEWLINE ) ;

}  //* End composeTitle() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

